import { UserState } from "./UserState";
/**
 * @interface
 * @category SDK
 * @subcategory Internal
 * @property {string=} id - The UUID of the active passcode.
 * @property {number=} ttl - Timestamp until when the passcode is valid in seconds (since January 1, 1970 00:00:00 UTC).
 * @property {number=} resendAfter - Seconds until a passcode can be resent.
 * @property {emailID=} emailID - The email address ID.
 */
export interface LocalStoragePasscode {
    id?: string;
    ttl?: number;
    resendAfter?: number;
    emailID?: string;
}
/**
 * A class that manages passcodes via local storage.
 *
 * @extends UserState
 * @category SDK
 * @subcategory Internal
 */
declare class PasscodeState extends UserState {
    /**
     * Get the passcode state.
     *
     * @private
     * @param {string} userID - The UUID of the user.
     * @return {LocalStoragePasscode}
     */
    private getState;
    /**
     * Reads the current state.
     *
     * @public
     * @return {PasscodeState}
     */
    read(): PasscodeState;
    /**
     * Gets the UUID of the active passcode.
     *
     * @param {string} userID - The UUID of the user.
     * @return {string}
     */
    getActiveID(userID: string): string;
    /**
     * Sets the UUID of the active passcode.
     *
     * @param {string} userID - The UUID of the user.
     * @param {string} passcodeID - The UUID of the passcode to be set as active.
     * @return {PasscodeState}
     */
    setActiveID(userID: string, passcodeID: string): PasscodeState;
    /**
     * Gets the UUID of the email address.
     *
     * @param {string} userID - The UUID of the user.
     * @return {string}
     */
    getEmailID(userID: string): string;
    /**
     * Sets the UUID of the email address.
     *
     * @param {string} userID - The UUID of the user.
     * @param {string} emailID - The UUID of the email address.
     * @return {PasscodeState}
     */
    setEmailID(userID: string, emailID: string): PasscodeState;
    /**
     * Removes the active passcode.
     *
     * @param {string} userID - The UUID of the user.
     * @return {PasscodeState}
     */
    reset(userID: string): PasscodeState;
    /**
     * Gets the TTL in seconds. When the seconds expire, the code is invalid.
     *
     * @param {string} userID - The UUID of the user.
     * @return {number}
     */
    getTTL(userID: string): number;
    /**
     * Sets the passcode's TTL and stores it to the local storage.
     *
     * @param {string} userID - The UUID of the user.
     * @param {string} seconds - Number of seconds the passcode is valid for.
     * @return {PasscodeState}
     */
    setTTL(userID: string, seconds: number): PasscodeState;
    /**
     * Gets the number of seconds until when the next passcode can be sent.
     *
     * @param {string} userID - The UUID of the user.
     * @return {number}
     */
    getResendAfter(userID: string): number;
    /**
     * Sets the number of seconds until a new passcode can be sent.
     *
     * @param {string} userID - The UUID of the user.
     * @param {number} seconds - Number of seconds the passcode is valid for.
     * @return {PasscodeState}
     */
    setResendAfter(userID: string, seconds: number): PasscodeState;
}
export { PasscodeState };
