/**
 * Every error thrown in the SDK is an instance of 'HankoError'. The value of the 'code' property is eligible to
 * translate the error into an error message.
 *
 * @extends {Error}
 * @category SDK
 * @subcategory Errors
 * @param code {string} - An error code that refers to the error instance.
 * @param cause {Error=} - The original error
 */
declare abstract class HankoError extends Error {
    code: string;
    cause?: Error;
    protected constructor(message: string, code: string, cause?: Error);
}
/**
 * Every error that doesn't need to be handled in a special way is a 'TechnicalError'. Whenever you catch one, there is
 * usually nothing you can do but present an error to the user, e.g. "Something went wrong".
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class TechnicalError extends HankoError {
    constructor(cause?: Error);
}
/**
 * Attempting to create a resource that already exists results in a 'ConflictError'.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class ConflictError extends HankoError {
    constructor(userID?: string, cause?: Error);
}
/**
 * A 'RequestTimeoutError' occurs when the specified timeout has been reached.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class RequestTimeoutError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'WebauthnRequestCancelledError' occurs during WebAuthn authentication or registration, when the WebAuthn API throws
 * an error. In most cases, this happens when the user cancels the browser's WebAuthn dialog.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class WebauthnRequestCancelledError extends HankoError {
    constructor(cause?: Error);
}
/**
 * An 'InvalidPasswordError' occurs when invalid credentials are provided when logging in with a password.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class InvalidPasswordError extends HankoError {
    constructor(cause?: Error);
}
/**
 * An 'InvalidPasswordError' occurs when an incorrect code is entered when logging in with a passcode.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class InvalidPasscodeError extends HankoError {
    constructor(cause?: Error);
}
/**
 * An 'InvalidWebauthnCredentialError' occurs if invalid credentials were used when logging in with WebAuthn.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class InvalidWebauthnCredentialError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'PasscodeExpiredError' occurs when the passcode has expired.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class PasscodeExpiredError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'MaxNumOfPasscodeAttemptsReachedError' occurs when an incorrect passcode is provided too many times.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class MaxNumOfPasscodeAttemptsReachedError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'NotFoundError' occurs when the requested resource was not found.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class NotFoundError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'TooManyRequestsError' occurs due to rate limiting when too many requests are made.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class TooManyRequestsError extends HankoError {
    retryAfter?: number;
    constructor(retryAfter?: number, cause?: Error);
}
/**
 * An 'UnauthorizedError' occurs when the user is not authorized to access the resource.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class UnauthorizedError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'ForbiddenError' occurs when the user is not allowed to perform the requested action.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class ForbiddenError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'UserVerificationError' occurs when the user verification requirements
 * for a WebAuthn ceremony are not met.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class UserVerificationError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A 'MaxNumOfEmailAddressesReachedError' occurs when the user tries to add a new email address while the maximum number
 * of email addresses (see backend configuration) equals the number of email addresses already registered.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class MaxNumOfEmailAddressesReachedError extends HankoError {
    constructor(cause?: Error);
}
/**
 * An 'EmailAddressAlreadyExistsError' occurs when the user tries to add a new email address which already exists.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class EmailAddressAlreadyExistsError extends HankoError {
    constructor(cause?: Error);
}
/**
 * A `ThirdPartyError` may occur during a sign in/sign up with a third party
 * provider.
 *
 * @category SDK
 * @subcategory Errors
 * @extends {HankoError}
 */
declare class ThirdPartyError extends HankoError {
    constructor(code: string, cause?: Error);
}
export { HankoError, TechnicalError, ConflictError, RequestTimeoutError, WebauthnRequestCancelledError, InvalidPasswordError, InvalidPasscodeError, InvalidWebauthnCredentialError, PasscodeExpiredError, MaxNumOfPasscodeAttemptsReachedError, NotFoundError, TooManyRequestsError, UnauthorizedError, ForbiddenError, UserVerificationError, MaxNumOfEmailAddressesReachedError, EmailAddressAlreadyExistsError, ThirdPartyError, };
