import { SessionState } from "../state/session/SessionState";
import { Dispatcher } from "../events/Dispatcher";
import { Cookie } from "../Cookie";
/**
 * This class wraps an XMLHttpRequest to maintain compatibility with the fetch API.
 *
 * @category SDK
 * @subcategory Internal
 * @param {XMLHttpRequest} xhr - The request to be wrapped.
 * @see HttpClient
 */
declare class Headers {
    _xhr: XMLHttpRequest;
    constructor(xhr: XMLHttpRequest);
    /**
     * Returns the response header with the given name.
     *
     * @param {string} name
     * @return {string}
     */
    getResponseHeader(name: string): string;
}
/**
 * This class wraps an XMLHttpRequest to maintain compatibility with the fetch API.
 *
 * @category SDK
 * @subcategory Internal
 * @param {XMLHttpRequest} xhr - The request to be wrapped.
 * @see HttpClient
 */
declare class Response {
    headers: Headers;
    ok: boolean;
    status: number;
    statusText: string;
    url: string;
    _decodedJSON: any;
    xhr: XMLHttpRequest;
    constructor(xhr: XMLHttpRequest);
    /**
     * Returns the JSON decoded response.
     *
     * @return {any}
     */
    json(): any;
    /**
     * Returns the response header value with the given `name` as a number. When the value is not a number the return
     * value will be 0.
     *
     * @param {string} name - The name of the header field
     * @return {number}
     */
    parseNumericHeader(name: string): number;
}
/**
 * Options for the HttpClient
 *
 * @category SDK
 * @subcategory Internal
 * @property {number} timeout - The http request timeout in milliseconds.
 * @property {string} cookieName - The name of the session cookie set from the SDK.
 * @property {string=} cookieDomain - The domain where cookie set from the SDK is available. Defaults to the domain of the page where the cookie was created.
 * @property {string} localStorageKey - The prefix / name of the local storage keys.
 */
export interface HttpClientOptions {
    timeout: number;
    cookieName: string;
    cookieDomain?: string;
    localStorageKey: string;
}
/**
 * Internally used for communication with the Hanko API. It also handles authorization tokens to enable authorized
 * requests.
 *
 * Currently, there is an issue with Safari and on iOS 15 devices where decoding a JSON response via the fetch API
 * breaks the user gesture and the user is not able to use the authenticator. Therefore, this class uses XMLHttpRequests
 * instead of the fetch API, but maintains compatibility by wrapping the XMLHttpRequests. So, if the issues are fixed,
 * we can easily return to the fetch API.
 *
 * @category SDK
 * @subcategory Internal
 * @param {string} api - The URL of your Hanko API instance
 * @param {HttpClientOptions} options - The options the HttpClient must be provided
 */
declare class HttpClient {
    timeout: number;
    api: string;
    sessionState: SessionState;
    dispatcher: Dispatcher;
    cookie: Cookie;
    constructor(api: string, options: HttpClientOptions);
    _fetch(path: string, options: RequestInit, xhr?: XMLHttpRequest): Promise<Response>;
    /**
     * Processes the response headers on login and extracts the JWT and expiration time.
     *
     * @param {XMLHttpRequest} xhr - The xhr object.
     */
    processHeaders(xhr: XMLHttpRequest): void;
    /**
     * Performs a GET request.
     *
     * @param {string} path - The path to the requested resource.
     * @return {Promise<Response>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     */
    get(path: string): Promise<Response>;
    /**
     * Performs a POST request.
     *
     * @param {string} path - The path to the requested resource.
     * @param {any=} body - The request body.
     * @return {Promise<Response>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     */
    post(path: string, body?: any): Promise<Response>;
    /**
     * Performs a PUT request.
     *
     * @param {string} path - The path to the requested resource.
     * @param {any=} body - The request body.
     * @return {Promise<Response>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     */
    put(path: string, body?: any): Promise<Response>;
    /**
     * Performs a PATCH request.
     *
     * @param {string} path - The path to the requested resource.
     * @param {any=} body - The request body.
     * @return {Promise<Response>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     */
    patch(path: string, body?: any): Promise<Response>;
    /**
     * Performs a DELETE request.
     *
     * @param {string} path - The path to the requested resource.
     * @return {Promise<Response>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     */
    delete(path: string): Promise<Response>;
}
export { Headers, Response, HttpClient };
