import { Throttle } from "../Throttle";
import { SessionDetail } from "./CustomEvents";
/**
 * A callback function to be executed when an event is triggered.
 *
 * @alias CallbackFunc
 * @typedef {function} CallbackFunc
 * @memberOf Listener
 */
type CallbackFunc<T> = (detail: T) => any;
/**
 * A function returned when adding an event listener. The function can be called to remove the corresponding event
 * listener.
 *
 * @alias CleanupFunc
 * @typedef {function} CleanupFunc
 * @memberOf Listener
 */
type CleanupFunc = () => void;
/**
 * A class to bind event listener for custom events.
 *
 * @category SDK
 * @subcategory Events
 */
export declare class Listener {
    throttleLimit: number;
    _addEventListener: any;
    _removeEventListener: any;
    _throttle: typeof Throttle.throttle;
    /**
     * Wraps the given callback.
     *
     * @param callback
     * @param throttle
     * @private
     * @return {WrappedCallback}
     */
    private wrapCallback;
    /**
     * Adds an event listener with the specified type, callback function, and options.
     *
     * @private
     * @param {EventListenerWithTypeParams<T>} params - The parameters for the event listener.
     * @returns {CleanupFunc} This function can be called to remove the event listener.
     */
    private addEventListenerWithType;
    /**
     * Maps the parameters for an event listener to the `EventListenerWithTypeParams` interface.
     *
     * @static
     * @private
     * @param {string} type - The type of the event.
     * @param {EventListenerParams<T>} params - The parameters for the event listener.
     * @param {boolean} [throttle=false] - Whether the event listener should be throttled.
     * @returns {EventListenerWithTypeParams<T>}
     **/
    private static mapAddEventListenerParams;
    /**
     * Adds an event listener with the specified type, callback function, and options.
     *
     * @private
     * @param {string} type - The type of the event.
     * @param {EventListenerParams<T>} params - The parameters for the event listener.
     * @param {boolean=} throttle - Whether the event listener should be throttled.
     * @returns {CleanupFunc} This function can be called to remove the event listener.
     */
    private addEventListener;
    /**
     * Adds an event listener for "hanko-session-created" events. Will be triggered across all browser windows, when the user
     * logs in, or when the page has been loaded or refreshed and there is a valid session.
     *
     * @param {CallbackFunc<SessionDetail>} callback - The function to be called when the event is triggered.
     * @param {boolean=} once - Whether the event listener should be removed after being called once.
     * @returns {CleanupFunc} This function can be called to remove the event listener.
     */
    onSessionCreated(callback: CallbackFunc<SessionDetail>, once?: boolean): CleanupFunc;
    /**
     * Adds an event listener for "hanko-session-expired" events. The event will be triggered across all browser windows
     * as soon as the current JWT expires or the user logs out. It also triggers, when the user deletes the account in
     * another window.
     *
     * @param {CallbackFunc<null>} callback - The function to be called when the event is triggered.
     * @param {boolean=} once - Whether the event listener should be removed after being called once.
     * @returns {CleanupFunc} This function can be called to remove the event listener.
     */
    onSessionExpired(callback: CallbackFunc<null>, once?: boolean): CleanupFunc;
    /**
     * Adds an event listener for hanko-user-deleted events. The event triggers, when the user has deleted the account in
     * the browser window where the deletion happened.
     *
     * @param {CallbackFunc<null>} callback - The function to be called when the event is triggered.
     * @param {boolean=} once - Whether the event listener should be removed after being called once.
     * @returns {CleanupFunc} This function can be called to remove the event listener.
     */
    onUserLoggedOut(callback: CallbackFunc<null>, once?: boolean): CleanupFunc;
    /**
     * Adds an event listener for hanko-user-deleted events. The event triggers, when the user has deleted the account.
     *
     * @param {CallbackFunc<null>} callback - The function to be called when the event is triggered.
     * @param {boolean=} once - Whether the event listener should be removed after being called once.
     * @returns {CleanupFunc} This function can be called to remove the event listener.
     */
    onUserDeleted(callback: CallbackFunc<null>, once?: boolean): CleanupFunc;
}
export {};
