"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tf = require("@tensorflow/tfjs-core");
var mobileNet100Architecture = [
    ['conv2d', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1]
];
var mobileNet75Architecture = [
    ['conv2d', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1]
];
var mobileNet50Architecture = [
    ['conv2d', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 2],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1],
    ['separableConv', 1]
];
var mobileNet25Architecture = mobileNet50Architecture;
var VALID_OUTPUT_STRIDES = [8, 16, 32];
function assertValidOutputStride(outputStride) {
    tf.util.assert(typeof outputStride === 'number', function () { return 'outputStride is not a number'; });
    tf.util.assert(VALID_OUTPUT_STRIDES.indexOf(outputStride) >= 0, function () { return "outputStride of " + outputStride + " is invalid. " +
        "It must be either 8, 16, or 32"; });
}
exports.assertValidOutputStride = assertValidOutputStride;
function assertValidResolution(resolution, outputStride) {
    tf.util.assert(typeof resolution === 'number', function () { return 'resolution is not a number'; });
    tf.util.assert((resolution - 1) % outputStride === 0, function () { return "resolution of " + resolution + " is invalid for output stride " +
        (outputStride + "."); });
}
exports.assertValidResolution = assertValidResolution;
exports.mobileNetArchitectures = {
    100: mobileNet100Architecture,
    75: mobileNet75Architecture,
    50: mobileNet50Architecture,
    25: mobileNet25Architecture
};
function toOutputStridedLayers(convolutionDefinition, outputStride) {
    var currentStride = 1;
    var rate = 1;
    return convolutionDefinition.map(function (_a, blockId) {
        var convType = _a[0], stride = _a[1];
        var layerStride, layerRate;
        if (currentStride === outputStride) {
            layerStride = 1;
            layerRate = rate;
            rate *= stride;
        }
        else {
            layerStride = stride;
            layerRate = 1;
            currentStride *= stride;
        }
        return {
            blockId: blockId,
            convType: convType,
            stride: layerStride,
            rate: layerRate,
            outputStride: currentStride
        };
    });
}
var MobileNet = (function () {
    function MobileNet(modelWeights, convolutionDefinitions) {
        this.PREPROCESS_DIVISOR = tf.scalar(255.0 / 2);
        this.ONE = tf.scalar(1.0);
        this.modelWeights = modelWeights;
        this.convolutionDefinitions = convolutionDefinitions;
    }
    MobileNet.prototype.predict = function (input, outputStride) {
        var _this = this;
        var normalized = tf.div(input.toFloat(), this.PREPROCESS_DIVISOR);
        var preprocessedInput = tf.sub(normalized, this.ONE);
        var layers = toOutputStridedLayers(this.convolutionDefinitions, outputStride);
        return layers.reduce(function (previousLayer, _a) {
            var blockId = _a.blockId, stride = _a.stride, convType = _a.convType, rate = _a.rate;
            if (convType === 'conv2d') {
                return _this.conv(previousLayer, stride, blockId);
            }
            else if (convType === 'separableConv') {
                return _this.separableConv(previousLayer, stride, blockId, rate);
            }
            else {
                throw Error("Unknown conv type of " + convType);
            }
        }, preprocessedInput);
    };
    MobileNet.prototype.convToOutput = function (mobileNetOutput, outputLayerName) {
        return mobileNetOutput.conv2d(this.weights(outputLayerName), 1, 'same')
            .add(this.convBias(outputLayerName, false));
    };
    MobileNet.prototype.conv = function (inputs, stride, blockId) {
        var weights = this.weights("Conv2d_" + String(blockId));
        var a = inputs.conv2d(weights, stride, 'same');
        var b = a.add(this.convBias("Conv2d_" + String(blockId)));
        return b.clipByValue(0, 6);
    };
    MobileNet.prototype.separableConv = function (inputs, stride, blockID, dilations) {
        if (dilations === void 0) { dilations = 1; }
        var dwLayer = "Conv2d_" + String(blockID) + "_depthwise";
        var pwLayer = "Conv2d_" + String(blockID) + "_pointwise";
        var x1 = inputs
            .depthwiseConv2D(this.depthwiseWeights(dwLayer), stride, 'same', 'NHWC', dilations)
            .add(this.depthwiseBias(dwLayer))
            .clipByValue(0, 6);
        var x2 = x1.conv2d(this.weights(pwLayer), [1, 1], 'same')
            .add(this.convBias(pwLayer))
            .clipByValue(0, 6);
        return x2;
    };
    MobileNet.prototype.weights = function (layerName) {
        return this.modelWeights.weights(layerName);
    };
    MobileNet.prototype.convBias = function (layerName, doublePrefix) {
        if (doublePrefix === void 0) { doublePrefix = true; }
        return this.modelWeights.convBias(layerName, doublePrefix);
    };
    MobileNet.prototype.depthwiseBias = function (layerName) {
        return this.modelWeights.depthwiseBias(layerName);
    };
    MobileNet.prototype.depthwiseWeights = function (layerName) {
        return this.modelWeights.depthwiseWeights(layerName);
    };
    MobileNet.prototype.dispose = function () {
        this.modelWeights.dispose();
    };
    return MobileNet;
}());
exports.MobileNet = MobileNet;
//# sourceMappingURL=mobilenet.js.map