/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '@tensorflow/tfjs-core';
import { backend_util, BackendTimingInfo, DataStorage, DataType, KernelBackend, Rank, Scalar, ShapeMap, Tensor, Tensor1D, Tensor2D, Tensor4D, Tensor5D, TensorInfo } from '@tensorflow/tfjs-core';
interface DataId {
}
export interface TensorData<D extends DataType> {
    values?: backend_util.BackendValues;
    dtype: D;
    complexTensorInfos?: {
        real: TensorInfo;
        imag: TensorInfo;
    };
    refCount: number;
}
export declare class MathBackendCPU extends KernelBackend {
    blockSize: number;
    data: DataStorage<TensorData<DataType>>;
    private firstUse;
    constructor();
    write(values: backend_util.BackendValues, shape: number[], dtype: DataType): DataId;
    /**
     * Create a data bucket in cpu backend.
     * @param shape Shape of the `TensorInfo`.
     * @param dtype DType of the `TensorInfo`.
     * @param values The value of the `TensorInfo` stored as a flattened array.
     */
    makeTensorInfo(shape: number[], dtype: DataType, values?: backend_util.BackendValues | string[]): TensorInfo;
    /** Increase refCount of a `TensorData`. */
    incRef(dataId: DataId): void;
    /** Decrease refCount of a `TensorData`. */
    decRef(dataId: DataId): void;
    move(dataId: DataId, values: backend_util.BackendValues, shape: number[], dtype: DataType): void;
    numDataIds(): number;
    read(dataId: DataId): Promise<backend_util.BackendValues>;
    readSync(dataId: DataId): backend_util.BackendValues;
    private bufferSync;
    makeOutput<T extends Tensor>(values: backend_util.BackendValues, shape: number[], dtype: DataType): T;
    disposeData(dataId: DataId): void;
    disposeIntermediateTensorInfo(tensorInfo: TensorInfo): void;
    time(f: () => void): Promise<BackendTimingInfo>;
    memory(): {
        unreliable: boolean;
        reasons: string[];
    };
    stridedSlice<T extends Tensor>(x: T, begin: number[], end: number[], strides: number[]): T;
    diag(x: Tensor): Tensor;
    unstack(x: Tensor, axis: number): Tensor[];
    reverse<T extends Tensor>(x: T, axis: number[]): T;
    neg<T extends Tensor>(x: T): T;
    addN<T extends Tensor>(tensors: T[]): T;
    softmax<T extends Tensor>(logits: T, dim: number): T;
    pow<T extends Tensor>(a: T, b: Tensor): T;
    floorDiv(a: Tensor, b: Tensor): Tensor;
    sum(x: Tensor, axes: number[]): Tensor;
    prod(x: Tensor, axes: number[]): Tensor;
    unsortedSegmentSum<T extends Tensor>(x: T, segmentIds: Tensor1D, numSegments: number): Tensor;
    argMin(x: Tensor, axis: number): Tensor;
    argMax(x: Tensor, axis: number): Tensor;
    cumsum(x: Tensor, axis: number, exclusive: boolean, reverse: boolean): Tensor;
    equal(a: Tensor, b: Tensor): Tensor;
    notEqual(a: Tensor, b: Tensor): Tensor;
    less(a: Tensor, b: Tensor): Tensor;
    lessEqual(a: Tensor, b: Tensor): Tensor;
    greater(a: Tensor, b: Tensor): Tensor;
    greaterEqual(a: Tensor, b: Tensor): Tensor;
    logicalAnd(a: Tensor, b: Tensor): Tensor;
    logicalOr(a: Tensor, b: Tensor): Tensor;
    select(condition: Tensor, a: Tensor, b: Tensor): Tensor;
    where(condition: Tensor): Tensor2D;
    topk<T extends Tensor>(x: T, k: number, sorted: boolean): [T, T];
    min(x: Tensor, axes: number[]): Tensor;
    minimum(a: Tensor, b: Tensor): Tensor;
    mod(a: Tensor, b: Tensor): Tensor;
    maximum(a: Tensor, b: Tensor): Tensor;
    all(x: Tensor, axes: number[]): Tensor;
    any(x: Tensor, axes: number[]): Tensor;
    squaredDifference(a: Tensor, b: Tensor): Tensor;
    eluDer<T extends Tensor>(dy: T, y: T): T;
    atan2<T extends Tensor>(a: T, b: T): T;
    tile<T extends Tensor>(x: T, reps: number[]): T;
    gather<T extends Tensor>(x: T, indices: Tensor1D, axis: number): T;
    batchToSpaceND<T extends Tensor>(x: T, blockShape: number[], crops: number[][]): T;
    private pool3d;
    avgPool3d(x: Tensor5D, convInfo: backend_util.Conv3DInfo): Tensor5D;
    avgPool3dBackprop(dy: Tensor5D, x: Tensor5D, convInfo: backend_util.Conv3DInfo): Tensor5D;
    maxPool3d(x: Tensor5D, convInfo: backend_util.Conv3DInfo): Tensor5D;
    private maxPool3dPositions;
    maxPool3dBackprop(dy: Tensor5D, x: Tensor5D, y: Tensor5D, convInfo: backend_util.Conv3DInfo): Tensor5D;
    resizeBilinear(x: Tensor4D, newHeight: number, newWidth: number, alignCorners: boolean): Tensor4D;
    resizeBilinearBackprop(dy: Tensor4D, x: Tensor4D, alignCorners: boolean): tf.Tensor<tf.Rank.R4>;
    resizeNearestNeighbor(x: Tensor4D, newHeight: number, newWidth: number, alignCorners: boolean): Tensor4D;
    resizeNearestNeighborBackprop(dy: Tensor4D, x: Tensor4D, alignCorners: boolean): tf.Tensor<tf.Rank.R4>;
    localResponseNormalization4D(x: Tensor4D, depthRadius: number, bias: number, alpha: number, beta: number): Tensor4D;
    LRNGrad(dy: Tensor4D, inputImage: Tensor4D, outputImage: Tensor4D, depthRadius: number, bias: number, alpha: number, beta: number): Tensor4D;
    multinomial(logits: Tensor2D, normalized: boolean, numSamples: number, seed: number): Tensor2D;
    oneHot(indices: Tensor1D, depth: number, onValue: number, offValue: number): Tensor2D;
    nonMaxSuppression(boxes: Tensor2D, scores: Tensor1D, maxOutputSize: number, iouThreshold: number, scoreThreshold: number): Tensor1D;
    depthToSpace(x: Tensor4D, blockSize: number, dataFormat: 'NHWC' | 'NCHW'): Tensor4D;
    private broadcastedBinaryOp;
    split<T extends Tensor>(x: T, sizeSplits: number[], axis: number): T[];
    dispose(): void;
    floatPrecision(): 16 | 32;
    /** Returns the smallest representable number.  */
    epsilon(): number;
    cropAndResize(images: Tensor4D, boxes: Tensor2D, boxIndex: Tensor1D, cropSize: [number, number], method: string, extrapolationValue: number): tf.Tensor<tf.Rank.R4>;
    sparseToDense<R extends Rank>(sparseIndices: Tensor, sparseValues: Tensor, outputShape: ShapeMap[R], defaultValue: Scalar): Tensor<R>;
    gatherND(x: Tensor, indices: Tensor): Tensor;
    scatterND<R extends Rank>(indices: Tensor, updates: Tensor, shape: ShapeMap[R]): Tensor<R>;
    onesLike<R extends Rank>(x: Tensor<R>): Tensor<R>;
    zerosLike<R extends Rank>(x: Tensor<R>): Tensor<R>;
    linspace(start: number, stop: number, num: number): Tensor1D;
    private scatter;
}
export {};
