/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { buffer } from '@tensorflow/tfjs-core';
export function scatterImpl(indices, updates, shape, outputSize, sliceSize, numUpdates, sliceRank, strides, defaultValue, sumDupeIndices) {
    const flattenShape = [outputSize / sliceSize, sliceSize];
    const indicesData = indices.values;
    const updatesData = updates.values;
    if (outputSize === 0) {
        return buffer(shape, updates.dtype);
    }
    const outBuf = buffer(flattenShape, updates.dtype);
    outBuf.values.fill(defaultValue);
    for (let i = 0; i < numUpdates; i++) {
        const index = [];
        let flattenIndex = 0;
        for (let j = 0; j < sliceRank; j++) {
            const dim = indicesData[i * sliceRank + j];
            index.push(dim);
            flattenIndex += dim * strides[j];
        }
        if (flattenIndex < 0 || flattenIndex >= outputSize / sliceSize) {
            throw new Error(`Invalid indices: ${index} does not index into ${shape}`);
        }
        for (let k = 0; k < sliceSize; k++) {
            if (sumDupeIndices) {
                outBuf.values[flattenIndex * sliceSize + k] +=
                    updatesData[i * sliceSize + k];
            }
            else {
                outBuf.values[flattenIndex * sliceSize + k] = updates.rank === 0 ?
                    updatesData[0] :
                    updatesData[i * sliceSize + k];
            }
        }
    }
    return outBuf;
}
//# sourceMappingURL=data:application/json;base64,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