/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { BatchMatMul, buffer, util } from '@tensorflow/tfjs-core';
import { assertNotComplex } from '../cpu_util';
import { reshape } from './Reshape';
export function batchMatMul(args) {
    const { inputs, backend, attrs } = args;
    const { a, b } = inputs;
    const { transposeA, transposeB } = attrs;
    assertNotComplex([a, b], 'matMul');
    const aRank = a.shape.length;
    const bRank = b.shape.length;
    const innerShapeA = transposeA ? a.shape[aRank - 2] : a.shape[aRank - 1];
    const innerShapeB = transposeB ? b.shape[bRank - 1] : b.shape[bRank - 2];
    const outerShapeA = transposeA ? a.shape[aRank - 1] : a.shape[aRank - 2];
    const outerShapeB = transposeB ? b.shape[bRank - 2] : b.shape[bRank - 1];
    const outerDimsA = a.shape.slice(0, -2);
    const outerDimsB = b.shape.slice(0, -2);
    const batchDimA = util.sizeFromShape(outerDimsA);
    const batchDimB = util.sizeFromShape(outerDimsB);
    const batchDimsCompatible = batchDimA === batchDimB || batchDimA === 1 || batchDimB === 1;
    util.assert(aRank >= 2 && bRank >= 2 && batchDimsCompatible, () => `Error in matMul: the input batch dimensions must either be the ` +
        `same or at least one input batch dimension must be 1. Got input ` +
        `batch dimensions of (${outerDimsA}) and (${outerDimsB}).`);
    const outShapeOuterDims = batchDimA > batchDimB ? a.shape.slice(0, -2) : b.shape.slice(0, -2);
    const outShape = outShapeOuterDims.concat([outerShapeA, outerShapeB]);
    util.assert(innerShapeA === innerShapeB, () => `Error in matMul: inner shapes (${innerShapeA}) and (` +
        `${innerShapeB}) of Tensors with shapes ${a.shape} and ` +
        `${b.shape} and transposeA=${transposeA}` +
        ` and transposeB=${transposeB} must match.`);
    const a3dShape = transposeA ? [batchDimA, innerShapeA, outerShapeA] :
        [batchDimA, outerShapeA, innerShapeA];
    const b3dShape = transposeB ? [batchDimB, outerShapeB, innerShapeB] :
        [batchDimB, innerShapeB, outerShapeB];
    // The rest of the implementation is designed to operate on rank-3 tensors
    const a3d = reshape({ inputs: { x: a }, backend, attrs: { shape: a3dShape } });
    const b3d = reshape({ inputs: { x: b }, backend, attrs: { shape: b3dShape } });
    const sharedDim = transposeA ? a3d.shape[1] : a3d.shape[2];
    const leftDim = transposeA ? a3d.shape[2] : a3d.shape[1];
    const rightDim = transposeB ? b3d.shape[1] : b3d.shape[2];
    const batchDim = Math.max(batchDimA, batchDimB);
    const a3dValues = backend.data.get(a3d.dataId).values;
    const b3dValues = backend.data.get(b3d.dataId).values;
    const a3dStrides = util.computeStrides(a3d.shape);
    const b3dStrides = util.computeStrides(b3d.shape);
    const [aBatch, aOuterStep, aInnerStep] = transposeA ?
        [a3dStrides[0], 1, a3dStrides[1]] :
        [a3dStrides[0], a3dStrides[1], 1];
    const [bInnerStep, bOuterStep, bBatch] = transposeB ?
        [1, b3dStrides[1], b3dStrides[0]] :
        [b3dStrides[1], 1, b3dStrides[0]];
    const size = leftDim * rightDim;
    const result = buffer([batchDim, leftDim, rightDim], a3d.dtype);
    const resVals = result.values;
    const blockSize = backend.blockSize;
    for (let bi = 0; bi < batchDim; bi++) {
        for (let i0 = 0; i0 < leftDim; i0 += blockSize) {
            for (let j0 = 0; j0 < rightDim; j0 += blockSize) {
                for (let k0 = 0; k0 < sharedDim; k0 += blockSize) {
                    // for when blockSize doesn't evenly divide the input
                    const iBlock = Math.min(i0 + blockSize, leftDim);
                    const jBlock = Math.min(j0 + blockSize, rightDim);
                    const kBlock = Math.min(k0 + blockSize, sharedDim);
                    for (let i = i0; i < iBlock; i++) {
                        for (let j = j0; j < jBlock; j++) {
                            let sum = 0.0;
                            for (let k = k0; k < kBlock; k++) {
                                const batchOffsetA = Math.min(bi, batchDimA - 1) * aBatch;
                                const batchOffsetB = Math.min(bi, batchDimB - 1) * bBatch;
                                const aVal = a3dValues[batchOffsetA + i * aOuterStep + k * aInnerStep];
                                const bVal = b3dValues[k * bInnerStep + j * bOuterStep + batchOffsetB];
                                sum += aVal * bVal;
                            }
                            resVals[bi * size + (i * rightDim + j)] += sum;
                        }
                    }
                }
            }
        }
    }
    backend.disposeIntermediateTensorInfo(a3d);
    backend.disposeIntermediateTensorInfo(b3d);
    // set correct shape on output.
    return backend.makeTensorInfo(outShape, result.dtype, result.values);
}
export const batchMatMulConfig = {
    kernelName: BatchMatMul,
    backendName: 'cpu',
    kernelFunc: batchMatMul,
};
//# sourceMappingURL=data:application/json;base64,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