/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { DepthToSpace, util } from '@tensorflow/tfjs-core';
export function depthToSpace(args) {
    const { inputs, backend, attrs } = args;
    const { x } = inputs;
    const { blockSize, dataFormat } = attrs;
    util.assert(dataFormat === 'NHWC', () => `Only NHWC dataFormat supported on CPU for depthToSpace. Got ${dataFormat}`);
    util.assert(blockSize > 1, () => `blockSize should be > 1 for depthToSpace, but was: ${blockSize}`);
    const batchSize = x.shape[0];
    const inputHeight = x.shape[1];
    const inputWidth = x.shape[2];
    const inputDepth = x.shape[3];
    const outputHeight = inputHeight * blockSize;
    const outputWidth = inputWidth * blockSize;
    const outputDepth = inputDepth / (blockSize * blockSize);
    const xValues = backend.data.get(x.dataId).values;
    const result = new Float32Array(batchSize * outputHeight * outputWidth * outputDepth);
    let outputIdx = 0;
    for (let b = 0; b < batchSize; ++b) {
        for (let h = 0; h < outputHeight; ++h) {
            const inH = Math.floor(h / blockSize);
            const offsetH = (h % blockSize);
            for (let w = 0; w < outputWidth; ++w) {
                const inW = Math.floor(w / blockSize);
                const offsetW = (w % blockSize);
                const offsetD = (offsetH * blockSize + offsetW) * outputDepth;
                for (let d = 0; d < outputDepth; ++d) {
                    const inD = d + offsetD;
                    const inputIdx = inD + inputDepth * (inW + inputWidth * (inH + inputHeight * b));
                    result[outputIdx++] = xValues[inputIdx];
                }
            }
        }
    }
    return backend.makeTensorInfo([batchSize, outputHeight, outputWidth, outputDepth], x.dtype, result);
}
export const depthToSpaceConfig = {
    kernelName: DepthToSpace,
    backendName: 'cpu',
    kernelFunc: depthToSpace
};
//# sourceMappingURL=data:application/json;base64,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