/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { util } from '@tensorflow/tfjs-core';
export function sparseFillEmptyRowsImpl(indices, indicesShape, indicesDType, values, valuesDType, denseShape, defaultValue) {
    const indicesCount = indicesShape[0];
    const denseRows = denseShape[0];
    const emptyRowIndicator = new Array(denseRows);
    const reverseIndexMap = new Array(indicesCount);
    const rank = indicesShape[1];
    if (denseRows === 0) {
        if (indicesCount !== 0) {
            throw new Error(`Received SparseTensor with denseShape[0] = 0 but
         indices.shape[0] = ${indicesCount}`);
        }
        const outputIndices = util.getArrayFromDType(indicesDType, 0);
        const outputValues = util.getArrayFromDType(valuesDType, 0);
        return [
            outputIndices, [0, rank], outputValues, emptyRowIndicator, reverseIndexMap
        ];
    }
    let rowsAreOrdered = true;
    let lastIndicesRow = 0;
    const csrOffset = new Array(denseRows).fill(0);
    for (let i = 0; i < indicesCount; ++i) {
        // indices is a 2d tensor with shape of [N, rank]
        const row = indices[i * rank];
        if (row < 0) {
            throw new Error(`indices(${i}, 0) is invalid: ${row} < 0`);
        }
        if (row >= denseRows) {
            throw new Error(`indices(${i}, 0) is invalid: ${row} >= ${denseRows}`);
        }
        ++csrOffset[row];
        rowsAreOrdered = rowsAreOrdered && (row >= lastIndicesRow);
        lastIndicesRow = row;
    }
    let allRowsFull = true;
    for (let row = 0; row < denseRows; ++row) {
        // csrOffset here describes the number of elements in this dense row
        const rowEmpty = (csrOffset[row] === 0);
        emptyRowIndicator[row] = rowEmpty;
        allRowsFull = allRowsFull && !rowEmpty;
        // In filled version, each row has at least one element.
        csrOffset[row] = Math.max(csrOffset[row], 1);
        // Update csrOffset to represent the number of elements up to and
        // including denseRows + 1:
        //  csrOffset[0] == #{elements of row 0}
        //  csrOffset[1] == #{elements of row 1} + #{elements of row 0}
        //  ..
        //  csrOffset[i] == starting index for elements in row i + 1.
        if (row > 0) {
            csrOffset[row] += csrOffset[row - 1];
        }
    }
    if (allRowsFull && rowsAreOrdered) {
        const outputIndices = indices;
        const outputValues = values;
        for (let i = 0; i < indicesCount; ++i) {
            reverseIndexMap[i] = i;
        }
        return [
            outputIndices, [indicesCount, rank], outputValues, emptyRowIndicator,
            reverseIndexMap
        ];
    }
    else {
        const fullIndicesCount = csrOffset[denseRows - 1];
        const outputIndices = util.getArrayFromDType(indicesDType, fullIndicesCount * rank);
        const outputValues = util.getArrayFromDType(valuesDType, fullIndicesCount);
        const filledCount = new Array(denseRows).fill(0);
        // Fill in values for rows that are not missing
        for (let i = 0; i < indicesCount; ++i) {
            // indices is a 2d tensor with shape of [N, rank]
            const row = indices[i * rank];
            const offset = filledCount[row];
            const outputI = ((row === 0) ? 0 : csrOffset[row - 1]) + offset;
            filledCount[row]++; // Increment the filled count for this row.
            for (let j = 0; j < rank; ++j) {
                // indices and outputIndices are 2d tensors with shape of [N, rank]
                outputIndices[outputI * rank + j] = indices[i * rank + j];
            }
            outputValues[outputI] = values[i];
            // We'll need this reverse index map to backprop correctly.
            reverseIndexMap[i] = outputI;
        }
        // Fill in values for rows that are missing
        for (let row = 0; row < denseRows; ++row) {
            const rowCount = filledCount[row];
            if (rowCount === 0) { // We haven't filled this row
                const startingIndex = (row === 0) ? 0 : csrOffset[row - 1];
                // Remaining index values were set to zero already.
                // Just need to set the row index in the right location.
                // outputIndices is a 2d tensor with shape of [N, rank]
                outputIndices[startingIndex * rank + 0] = row;
                for (let col = 1; col < rank; ++col) {
                    outputIndices[startingIndex * rank + col] = 0;
                }
                outputValues[startingIndex] = defaultValue;
            }
        }
        return [
            outputIndices, [fullIndicesCount, rank], outputValues, emptyRowIndicator,
            reverseIndexMap
        ];
    }
}
//# sourceMappingURL=data:application/json;base64,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