/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { util } from '@tensorflow/tfjs-core';
export function sparseReshapeImpl(inputIndices, inputIndicesShape, inputDType, inputShape, targetShape) {
    const denseSize = util.sizeFromShape(inputShape);
    const nnz = inputIndicesShape[0];
    const outputRank = targetShape.length;
    // Compute the output shape. Determine product of specified dimensions, and
    // find the index of the unspecified one.
    const outputShape = [];
    let product = 1;
    let unknownIndex = -1;
    for (let d = 0; d < outputRank; ++d) {
        const size = targetShape[d];
        if (size === -1) {
            if (unknownIndex !== -1) {
                throw new Error(`only one output dimension may be -1, not both ${unknownIndex} and ${d}`);
            }
            unknownIndex = d;
            outputShape.push(1);
        }
        else {
            if (size < 0) {
                throw new Error(`size ${d} must be non-negative, not ${size}`);
            }
            product *= size;
            outputShape.push(size);
        }
    }
    if (unknownIndex !== -1) {
        if (product <= 0) {
            throw new Error('reshape cannot infer the missing ' +
                'input size for an empty tensor unless all ' +
                'specified input sizes are non-zero');
        }
        const missing = Math.trunc(denseSize / product);
        if (product * missing !== denseSize) {
            throw new Error(`Input to reshape is a SparseTensor with ${denseSize}
          dense values, but the requested shape requires a multiple of ${product}. inputShape=${inputShape} outputShape= ${outputShape}`);
        }
        outputShape[unknownIndex] = missing;
    }
    const outputSize = util.sizeFromShape(outputShape);
    if (outputSize !== denseSize) {
        throw new Error(`Input to reshape is a tensor with ${denseSize} dense values, but the requested shape has ${outputSize}. inputShape=${inputShape} outputShape=${outputShape}`);
    }
    const inputRank = inputShape.length;
    const inputStrides = [];
    if (inputRank > 0) {
        inputStrides[inputRank - 1] = 1;
        for (let d = inputRank - 2; d >= 0; --d) {
            inputStrides[d] = inputStrides[d + 1] * inputShape[d + 1];
        }
    }
    const outputStrides = [];
    if (outputRank > 0) {
        outputStrides[outputRank - 1] = 1;
        for (let d = outputRank - 2; d >= 0; --d) {
            outputStrides[d] = outputStrides[d + 1] * outputShape[d + 1];
        }
    }
    const newIndices = util.getArrayFromDType(inputDType, nnz * outputRank);
    for (let i = 0; i < nnz; ++i) {
        let id = 0;
        for (let j = 0; j < inputRank; ++j) {
            // inputIndices is a 2d tensor with shape of [nnz, inputRank]
            id += inputIndices[i * inputRank + j] * inputStrides[j];
        }
        for (let j = 0; j < outputRank; ++j) {
            // newIndices is a 2d tensor with shape of [nnz, outputRank]
            newIndices[i * outputRank + j] = Math.trunc(id / outputStrides[j]);
            id %= outputStrides[j];
        }
    }
    return [newIndices, [nnz, outputRank], outputShape];
}
//# sourceMappingURL=data:application/json;base64,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