/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { util } from '@tensorflow/tfjs-core';
export function sparseSegmentReductionImpl(input, inputShape, inputDType, indices, segmentIds, isMean = false, defaultValue = 0) {
    const numIndices = indices.length;
    if (numIndices !== segmentIds.length) {
        throw new Error(`segmentIds and indices should have same size.`);
    }
    // Flatten the array to two dimensions
    const inputFlat = [inputShape[0], input.length / inputShape[0]];
    const numCol = inputFlat[1];
    // Note that the current implementation assumes that segmentIds values are
    // sorted.
    const lastSegmentIdPlusOne = numIndices > 0 ? segmentIds[numIndices - 1] + 1 : 0;
    const outputRows = lastSegmentIdPlusOne;
    if (outputRows < 0) {
        throw new Error(`segment ids must be >= 0`);
    }
    const outputShape = inputShape.slice();
    outputShape[0] = outputRows;
    const outputLength = outputShape.reduce((product, value) => product * value, 1);
    // Output array is initialized with the value 0 by default.
    const output = util.getArrayFromDType(inputDType, outputLength);
    // Note that we do not initialize the output buffer with a default value, so
    // we need to explicitly set missing indices to the default value.
    if (numIndices === 0) {
        if (outputRows > 0) {
            output.fill(defaultValue);
        }
        return [output, outputShape];
    }
    if (outputRows <= 0) {
        throw new Error(`segment ids must be >= 0`);
    }
    let start = 0, end = 1;
    // Index from which the output is not initialized.
    let uninitializedIndex = 0;
    let outIndex = segmentIds[start];
    while (true) {
        // We initialize nextIndex to 0 to avoid may be uninitialized warning
        let nextIndex = 0;
        if (end < numIndices) {
            nextIndex = segmentIds[end];
            if (outIndex === nextIndex) {
                ++end;
                continue;
            }
            // We have a new segment here.  Verify that the segment ids are growing.
            if (outIndex >= nextIndex) {
                throw new Error(`segment ids are not increasing`);
            }
        }
        if (outIndex < 0 || outIndex >= outputRows) {
            throw new Error(`Segment id ${outIndex} out of range [0, ${outputRows}), possibly because segmentIds input is not sorted.`);
        }
        // If there is a gap between two indices, we need to set that gap to the
        // default value.
        if (outIndex > uninitializedIndex) {
            output.fill(defaultValue, uninitializedIndex * numCol, outIndex * numCol);
        }
        for (let i = start; i < end; ++i) {
            const index = indices[i];
            if (index < 0 || index >= inputFlat[0]) {
                throw new Error(`Bad: indices[${i}] == ${indices[i]} out of range [0, ${inputFlat[0]})`);
            }
            for (let j = 0; j < numCol; j++) {
                output[outIndex * numCol + j] += input[index * numCol + j];
            }
        }
        if (isMean) {
            for (let j = 0; j < numCol; j++) {
                output[outIndex * numCol + j] /= end - start;
            }
        }
        start = end;
        ++end;
        uninitializedIndex = outIndex + 1;
        outIndex = nextIndex;
        if (end > numIndices) {
            break;
        }
    }
    // Fill the gap at the end with the default value.
    if (uninitializedIndex < outputRows) {
        output.fill(defaultValue, uninitializedIndex * numCol, outputRows * numCol);
    }
    return [output, outputShape];
}
//# sourceMappingURL=data:application/json;base64,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