/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
// Import webgl flags.
import './flags_webgl';
import { backend_util, buffer, DataStorage, engine, env, kernel_impls, KernelBackend, scalar, tidy, util } from '@tensorflow/tfjs-core';
import { getWebGLContext } from './canvas_util';
import { DecodeMatrixProgram } from './decode_matrix_gpu';
import { DecodeMatrixPackedProgram } from './decode_matrix_packed_gpu';
import { EncodeFloatProgram } from './encode_float_gpu';
import { EncodeFloatPackedProgram } from './encode_float_packed_gpu';
import { EncodeMatrixProgram } from './encode_matrix_gpu';
import { EncodeMatrixPackedProgram } from './encode_matrix_packed_gpu';
import { GPGPUContext } from './gpgpu_context';
import * as gpgpu_math from './gpgpu_math';
import { simpleAbsImplCPU } from './kernel_utils/shared';
import { PackProgram } from './pack_gpu';
import { ReshapePackedProgram } from './reshape_packed_gpu';
import * as tex_util from './tex_util';
import { TextureUsage } from './tex_util';
import { TextureManager } from './texture_manager';
import * as unary_op from './unaryop_gpu';
import { UnaryOpProgram } from './unaryop_gpu';
import { UnaryOpPackedProgram } from './unaryop_packed_gpu';
import { UnpackProgram } from './unpack_gpu';
import * as webgl_util from './webgl_util';
const whereImpl = kernel_impls.whereImpl;
export const EPSILON_FLOAT32 = 1e-7;
export const EPSILON_FLOAT16 = 1e-4;
const binaryCaches = {};
export function getBinaryCache(webGLVersion) {
    if (webGLVersion in binaryCaches) {
        return binaryCaches[webGLVersion];
    }
    binaryCaches[webGLVersion] = {};
    return binaryCaches[webGLVersion];
}
// Empirically determined constant used to determine size threshold for handing
// off execution to the CPU.
const CPU_HANDOFF_SIZE_THRESHOLD = env().getNumber('CPU_HANDOFF_SIZE_THRESHOLD');
// Empirically determined constant used to decide the number of MB on GPU
// before we warn about high memory use. The MB are this constant * screen area
// * dpi / 1024 / 1024.
const BEFORE_PAGING_CONSTANT = 600;
function numMBBeforeWarning() {
    if (env().global.screen == null) {
        return 1024; // 1 GB.
    }
    return (env().global.screen.height * env().global.screen.width *
        window.devicePixelRatio) *
        BEFORE_PAGING_CONSTANT / 1024 / 1024;
}
export class MathBackendWebGL extends KernelBackend {
    constructor(gpuResource) {
        super();
        // Maps data ids that have a pending read operation, to list of subscribers.
        this.pendingRead = new WeakMap();
        // List of data ids that are scheduled for disposal, but are waiting on a
        // pending read operation.
        this.pendingDisposal = new WeakSet();
        // Used to count the number of 'shallow' sliced tensors that point to the
        // same data id.
        this.dataRefCount = new WeakMap();
        this.numBytesInGPU = 0;
        // Accumulated time spent (including blocking) in uploading data to webgl.
        this.uploadWaitMs = 0;
        // Accumulated time spent (including blocking in downloading data from webgl.
        this.downloadWaitMs = 0;
        // record the last manual GL Flush time.
        this.lastGlFlushTime = 0;
        this.warnedAboutMemory = false;
        this.pendingDeletes = 0;
        this.disposed = false;
        if (!env().getBool('HAS_WEBGL')) {
            throw new Error('WebGL is not supported on this device');
        }
        let newGPGPU;
        if (gpuResource != null) {
            if (gpuResource instanceof GPGPUContext) {
                newGPGPU = gpuResource;
            }
            else {
                const gl = getWebGLContext(env().getNumber('WEBGL_VERSION'), gpuResource);
                newGPGPU = new GPGPUContext(gl);
            }
            this.binaryCache = {};
            this.gpgpuCreatedLocally = false;
        }
        else {
            const gl = getWebGLContext(env().getNumber('WEBGL_VERSION'));
            newGPGPU = new GPGPUContext(gl);
            this.binaryCache = getBinaryCache(env().getNumber('WEBGL_VERSION'));
            this.gpgpuCreatedLocally = true;
        }
        this.gpgpu = newGPGPU;
        this.canvas = this.gpgpu.gl.canvas;
        this.textureManager = new TextureManager(this.gpgpu);
        this.numMBBeforeWarning = numMBBeforeWarning();
        this.texData = new DataStorage(this, engine());
    }
    nextDataId() {
        return MathBackendWebGL.nextDataId++;
    }
    numDataIds() {
        return this.texData.numDataIds() - this.pendingDeletes;
    }
    write(values, shape, dtype) {
        if (env().getBool('WEBGL_CHECK_NUMERICAL_PROBLEMS') ||
            env().getBool('DEBUG')) {
            this.checkNumericalProblems(values);
        }
        if (dtype === 'complex64' && values != null) {
            throw new Error(`Cannot write to a complex64 dtype. ` +
                `Please use tf.complex(real, imag).`);
        }
        const dataId = { id: this.nextDataId() };
        this.texData.set(dataId, { shape, dtype, values, usage: TextureUsage.UPLOAD, refCount: 1 });
        return dataId;
    }
    /** Return refCount of a `TensorData`. */
    refCount(dataId) {
        if (this.texData.has(dataId)) {
            const tensorData = this.texData.get(dataId);
            return tensorData.refCount;
        }
        return 0;
    }
    /** Increase refCount of a `TextureData`. */
    incRef(dataId) {
        const texData = this.texData.get(dataId);
        texData.refCount++;
    }
    /** Decrease refCount of a `TextureData`. */
    decRef(dataId) {
        if (this.texData.has(dataId)) {
            const texData = this.texData.get(dataId);
            texData.refCount--;
        }
    }
    move(dataId, values, shape, dtype, refCount) {
        if (env().getBool('DEBUG')) {
            this.checkNumericalProblems(values);
        }
        if (dtype === 'complex64') {
            throw new Error(`Cannot write to a complex64 dtype. ` +
                `Please use tf.complex(real, imag).`);
        }
        this.texData.set(dataId, { shape, dtype, values, usage: TextureUsage.UPLOAD, refCount });
    }
    disposeIntermediateTensorInfo(tensorInfo) {
        this.disposeData(tensorInfo.dataId);
    }
    readSync(dataId) {
        const texData = this.texData.get(dataId);
        const { values, dtype, complexTensorInfos, slice, shape, isPacked } = texData;
        // The presence of `slice` indicates this tensor is a shallow slice of a
        // different tensor, and is using that original tensor's texture. Run
        // `clone` in order to copy that texture and read from it.
        if (slice != null) {
            let program;
            if (isPacked) {
                program = new UnaryOpPackedProgram(shape, unary_op.CLONE);
            }
            else {
                program = new UnaryOpProgram(shape, unary_op.CLONE);
            }
            const res = this.runWebGLProgram(program, [{ dataId, shape, dtype }], dtype);
            const data = this.readSync(res.dataId);
            this.disposeIntermediateTensorInfo(res);
            return data;
        }
        if (values != null) {
            return this.convertAndCacheOnCPU(dataId);
        }
        if (dtype === 'string') {
            return values;
        }
        const shouldTimeProgram = this.activeTimers != null;
        let start;
        if (shouldTimeProgram) {
            start = util.now();
        }
        let result;
        if (dtype === 'complex64') {
            const realValues = this.readSync(complexTensorInfos.real.dataId);
            const imagValues = this.readSync(complexTensorInfos.imag.dataId);
            result = backend_util.mergeRealAndImagArrays(realValues, imagValues);
        }
        else {
            result = this.getValuesFromTexture(dataId);
        }
        if (shouldTimeProgram) {
            this.downloadWaitMs += util.now() - start;
        }
        return this.convertAndCacheOnCPU(dataId, result);
    }
    async read(dataId) {
        if (this.pendingRead.has(dataId)) {
            const subscribers = this.pendingRead.get(dataId);
            return new Promise(resolve => subscribers.push(resolve));
        }
        const texData = this.texData.get(dataId);
        const { values, shape, slice, dtype, complexTensorInfos, isPacked } = texData;
        // The presence of `slice` indicates this tensor is a shallow slice of a
        // different tensor, and is using that original tensor's texture. Run
        // `clone` in order to copy that texture and read from it.
        if (slice != null) {
            let program;
            if (isPacked) {
                program = new UnaryOpPackedProgram(shape, unary_op.CLONE);
            }
            else {
                program = new UnaryOpProgram(shape, unary_op.CLONE);
            }
            const res = this.runWebGLProgram(program, [{ dataId, shape, dtype }], dtype);
            const data = this.read(res.dataId);
            this.disposeIntermediateTensorInfo(res);
            return data;
        }
        if (values != null) {
            return this.convertAndCacheOnCPU(dataId);
        }
        if (env().getBool('DEBUG')) {
            // getBool('WEBGL_DOWNLOAD_FLOAT_ENABLED') caused a blocking GPU call.
            // For performance reason, only check it for debugging. In production,
            // it doesn't handle this use case anyway, so behavior is not changed.
            if (!env().getBool('WEBGL_DOWNLOAD_FLOAT_ENABLED') &&
                env().getNumber('WEBGL_VERSION') === 2) {
                throw new Error(`tensor.data() with WEBGL_DOWNLOAD_FLOAT_ENABLED=false and ` +
                    `WEBGL_VERSION=2 not yet supported.`);
            }
        }
        let buffer = null;
        let tmpDownloadTarget;
        if (dtype !== 'complex64' && env().get('WEBGL_BUFFER_SUPPORTED')) {
            // Possibly copy the texture into a buffer before inserting a fence.
            tmpDownloadTarget = this.decode(dataId);
            const tmpData = this.texData.get(tmpDownloadTarget.dataId);
            buffer = this.gpgpu.createBufferFromTexture(tmpData.texture.texture, ...tex_util.getDenseTexShape(shape));
        }
        this.pendingRead.set(dataId, []);
        if (dtype !== 'complex64') {
            // Create a fence and wait for it to resolve.
            await this.gpgpu.createAndWaitForFence();
        }
        // Download the values from the GPU.
        let vals;
        if (dtype === 'complex64') {
            const ps = await Promise.all([
                this.read(complexTensorInfos.real.dataId),
                this.read(complexTensorInfos.imag.dataId)
            ]);
            const realValues = ps[0];
            const imagValues = ps[1];
            vals = backend_util.mergeRealAndImagArrays(realValues, imagValues);
        }
        else if (buffer == null) {
            vals = this.getValuesFromTexture(dataId);
        }
        else {
            const size = util.sizeFromShape(shape);
            vals = this.gpgpu.downloadFloat32MatrixFromBuffer(buffer, size);
        }
        if (tmpDownloadTarget != null) {
            this.disposeIntermediateTensorInfo(tmpDownloadTarget);
        }
        if (buffer != null) {
            const gl = this.gpgpu.gl;
            webgl_util.callAndCheck(gl, () => gl.deleteBuffer(buffer));
        }
        const dTypeVals = this.convertAndCacheOnCPU(dataId, vals);
        const subscribers = this.pendingRead.get(dataId);
        this.pendingRead.delete(dataId);
        // Notify all pending reads.
        subscribers.forEach(resolve => resolve(dTypeVals));
        if (this.pendingDisposal.has(dataId)) {
            this.pendingDisposal.delete(dataId);
            if (this.disposeData(dataId)) {
                engine().removeDataId(dataId, this);
            }
            this.pendingDeletes--;
        }
        return dTypeVals;
    }
    /**
     * Read tensor to a new texture that is densely packed for ease of use.
     * @param dataId The source tensor.
     * @param options
     *     customTexShape: Optional. If set, will use the user defined texture
     *     shape to create the texture.
     */
    readToGPU(dataId, options = {}) {
        const texData = this.texData.get(dataId);
        const { values, shape, slice, dtype, isPacked, texture } = texData;
        if (dtype === 'complex64') {
            throw new Error('Does not support reading texture for complex64 dtype.');
        }
        // The presence of `slice` indicates this tensor is a shallow slice of a
        // different tensor, and is using that original tensor's texture. Run
        // `clone` in order to copy that texture and read from it.
        if (slice != null) {
            let program;
            if (isPacked) {
                program = new UnaryOpPackedProgram(shape, unary_op.CLONE);
            }
            else {
                program = new UnaryOpProgram(shape, unary_op.CLONE);
            }
            const res = this.runWebGLProgram(program, [{ dataId, shape, dtype }], dtype);
            const gpuResouorce = this.readToGPU(res, options);
            this.disposeIntermediateTensorInfo(res);
            return gpuResouorce;
        }
        if (texture == null) {
            if (values != null) {
                throw new Error('Data is not on GPU but on CPU.');
            }
            else {
                throw new Error('There is no data on GPU or CPU.');
            }
        }
        // Decode the texture so that it is stored densely (using four channels).
        const tmpTarget = this.decode(dataId, options.customTexShape);
        // Make engine track this tensor, so that we can dispose it later.
        const tensorRef = engine().makeTensorFromDataId(tmpTarget.dataId, tmpTarget.shape, tmpTarget.dtype);
        const tmpData = this.texData.get(tmpTarget.dataId);
        return Object.assign({ tensorRef }, tmpData.texture);
    }
    bufferSync(t) {
        const data = this.readSync(t.dataId);
        let decodedData = data;
        if (t.dtype === 'string') {
            try {
                // Decode the bytes into string.
                decodedData = data.map(d => util.decodeString(d));
            }
            catch (_a) {
                throw new Error('Failed to decode encoded string bytes into utf-8');
            }
        }
        return buffer(t.shape, t.dtype, decodedData);
    }
    checkNumericalProblems(values) {
        if (values == null) {
            return;
        }
        for (let i = 0; i < values.length; i++) {
            const num = values[i];
            if (!webgl_util.canBeRepresented(num)) {
                if (env().getBool('WEBGL_RENDER_FLOAT32_CAPABLE')) {
                    throw Error(`The value ${num} cannot be represented with your ` +
                        `current settings. Consider enabling float32 rendering: ` +
                        `'tf.env().set('WEBGL_RENDER_FLOAT32_ENABLED', true);'`);
                }
                throw Error(`The value ${num} cannot be represented on this device.`);
            }
        }
    }
    getValuesFromTexture(dataId) {
        const { shape, dtype, isPacked } = this.texData.get(dataId);
        const size = util.sizeFromShape(shape);
        if (env().getBool('WEBGL_DOWNLOAD_FLOAT_ENABLED')) {
            const tmpTarget = this.decode(dataId);
            const tmpData = this.texData.get(tmpTarget.dataId);
            const vals = this.gpgpu
                .downloadMatrixFromPackedTexture(tmpData.texture.texture, ...tex_util.getDenseTexShape(shape))
                .subarray(0, size);
            this.disposeIntermediateTensorInfo(tmpTarget);
            return vals;
        }
        const shouldUsePackedProgram = env().getBool('WEBGL_PACK') && isPacked === true;
        const outputShape = shouldUsePackedProgram ? webgl_util.getShapeAs3D(shape) : shape;
        const program = shouldUsePackedProgram ?
            new EncodeFloatPackedProgram(outputShape) :
            new EncodeFloatProgram(outputShape);
        const output = this.runWebGLProgram(program, [{ shape: outputShape, dtype, dataId }], 'float32');
        const tmpData = this.texData.get(output.dataId);
        const vals = this.gpgpu
            .downloadByteEncodedFloatMatrixFromOutputTexture(tmpData.texture.texture, tmpData.texShape[0], tmpData.texShape[1])
            .subarray(0, size);
        this.disposeIntermediateTensorInfo(output);
        return vals;
    }
    timerAvailable() {
        return env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0;
    }
    time(f) {
        const oldActiveTimers = this.activeTimers;
        const newActiveTimers = [];
        let outerMostTime = false;
        if (this.programTimersStack == null) {
            this.programTimersStack = newActiveTimers;
            outerMostTime = true;
        }
        else {
            this.activeTimers.push(newActiveTimers);
        }
        this.activeTimers = newActiveTimers;
        f();
        // needing to split these up because util.flatten only accepts certain types
        const flattenedActiveTimerQueries = util.flatten(this.activeTimers.map((d) => d.query))
            .filter(d => d != null);
        const flattenedActiveTimerNames = util.flatten(this.activeTimers.map((d) => d.name))
            .filter(d => d != null);
        this.activeTimers = oldActiveTimers;
        if (outerMostTime) {
            this.programTimersStack = null;
        }
        const res = {
            uploadWaitMs: this.uploadWaitMs,
            downloadWaitMs: this.downloadWaitMs,
            kernelMs: null,
            wallMs: null // will be filled by the engine
        };
        return (async () => {
            if (env()
                .getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
                const kernelMs = await Promise.all(flattenedActiveTimerQueries);
                res['kernelMs'] = util.sum(kernelMs);
                res['getExtraProfileInfo'] = () => kernelMs.map((d, i) => ({ name: flattenedActiveTimerNames[i], ms: d }))
                    .map(d => `${d.name}: ${d.ms}`)
                    .join(', ');
            }
            else {
                res['kernelMs'] = {
                    error: 'WebGL query timers are not supported in this environment.'
                };
            }
            this.uploadWaitMs = 0;
            this.downloadWaitMs = 0;
            return res;
        })();
    }
    memory() {
        return {
            unreliable: false,
            numBytesInGPU: this.numBytesInGPU,
            numBytesInGPUAllocated: this.textureManager.numBytesAllocated,
            numBytesInGPUFree: this.textureManager.numBytesFree
        };
    }
    startTimer() {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
            return this.gpgpu.beginQuery();
        }
        return { startMs: util.now(), endMs: null };
    }
    endTimer(query) {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
            this.gpgpu.endQuery();
            return query;
        }
        query.endMs = util.now();
        return query;
    }
    async getQueryTime(query) {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
            return this.gpgpu.waitForQueryAndGetTime(query);
        }
        const timerQuery = query;
        return timerQuery.endMs - timerQuery.startMs;
    }
    /**
     * Decrease the RefCount on the dataId and dispose the memory if the dataId
     * has 0 refCount. If there are pending read on the data, the disposal would
     * added to the pending delete queue. Return true if the dataId is removed
     * from backend or the backend does not contain the dataId, false if the
     * dataId is not removed. Memory may or may not be released even when dataId
     * is removed, which also depends on dataRefCount, see `releaseGPU`.
     * @param dataId
     * @oaram force Optional, remove the data regardless of refCount
     */
    disposeData(dataId, force = false) {
        if (this.pendingDisposal.has(dataId)) {
            return false;
        }
        // No-op if already disposed.
        if (!this.texData.has(dataId)) {
            return true;
        }
        // if force flag is set, change refCount to 0, this would ensure disposal
        // when added to the pendingDisposal queue. Memory may or may not be
        // released, which also depends on dataRefCount, see `releaseGPU`.
        if (force) {
            this.texData.get(dataId).refCount = 0;
        }
        else {
            this.texData.get(dataId).refCount--;
        }
        if (!force && this.texData.get(dataId).refCount > 0) {
            return false;
        }
        if (this.pendingRead.has(dataId)) {
            this.pendingDisposal.add(dataId);
            this.pendingDeletes++;
            return false;
        }
        this.releaseGPUData(dataId);
        const { complexTensorInfos } = this.texData.get(dataId);
        if (complexTensorInfos != null) {
            this.disposeData(complexTensorInfos.real.dataId, force);
            this.disposeData(complexTensorInfos.imag.dataId, force);
        }
        this.texData.delete(dataId);
        return true;
    }
    releaseGPUData(dataId) {
        const { texture, dtype, texShape, usage, isPacked, slice } = this.texData.get(dataId);
        const key = slice && slice.origDataId || dataId;
        const refCount = this.dataRefCount.get(key);
        if (refCount > 1) {
            this.dataRefCount.set(key, refCount - 1);
        }
        else {
            this.dataRefCount.delete(key);
            if (texture != null) {
                this.numBytesInGPU -= this.computeBytes(texShape, dtype);
                this.textureManager.releaseTexture(texture, texShape, usage, isPacked);
            }
        }
        const texData = this.texData.get(dataId);
        texData.texture = null;
        texData.texShape = null;
        texData.isPacked = false;
        texData.slice = null;
    }
    getTexture(dataId) {
        this.uploadToGPU(dataId);
        return this.texData.get(dataId).texture.texture;
    }
    /**
     * Returns internal information for the specific data bucket. Used in unit
     * tests.
     */
    getDataInfo(dataId) {
        return this.texData.get(dataId);
    }
    /*
    Tests whether all the inputs to an op are small and on the CPU. This heuristic
    determines when it would be faster to execute a kernel on the CPU. WebGL
    kernels opt into running this check and forwarding when appropriate.
    TODO(https://github.com/tensorflow/tfjs/issues/872): Develop a more
    sustainable strategy for optimizing backend execution of ops.
     */
    shouldExecuteOnCPU(inputs, sizeThreshold = CPU_HANDOFF_SIZE_THRESHOLD) {
        return env().getBool('WEBGL_CPU_FORWARD') &&
            inputs.every(input => this.texData.get(input.dataId).texture == null &&
                util.sizeFromShape(input.shape) < sizeThreshold);
    }
    getGPGPUContext() {
        return this.gpgpu;
    }
    where(condition) {
        backend_util.warn('tf.where() in webgl locks the UI thread. ' +
            'Call tf.whereAsync() instead');
        const condVals = condition.dataSync();
        return whereImpl(condition.shape, condVals);
    }
    packedUnaryOp(x, op, dtype) {
        const program = new UnaryOpPackedProgram(x.shape, op);
        const outInfo = this.compileAndRun(program, [x], dtype);
        return engine().makeTensorFromDataId(outInfo.dataId, outInfo.shape, outInfo.dtype);
    }
    // TODO(msoulanille) remove this once the backend has been modularized
    // a copy is needed here to break a circular dependency.
    // Also remove the op from unary_op.
    abs(x) {
        // TODO: handle cases when x is complex.
        if (this.shouldExecuteOnCPU([x]) && x.dtype !== 'complex64') {
            const outValues = simpleAbsImplCPU(this.texData.get(x.dataId).values);
            return this.makeOutput(x.shape, x.dtype, outValues);
        }
        if (env().getBool('WEBGL_PACK_UNARY_OPERATIONS')) {
            return this.packedUnaryOp(x, unary_op.ABS, x.dtype);
        }
        const program = new UnaryOpProgram(x.shape, unary_op.ABS);
        const outInfo = this.compileAndRun(program, [x]);
        return engine().makeTensorFromDataId(outInfo.dataId, outInfo.shape, outInfo.dtype);
    }
    makeTensorInfo(shape, dtype, values) {
        let dataId;
        if (dtype === 'string' && values != null && values.length > 0 &&
            util.isString(values[0])) {
            const encodedValues = values.map(d => util.encodeString(d));
            dataId = this.write(encodedValues, shape, dtype);
        }
        else {
            dataId = this.write(values, shape, dtype);
        }
        this.texData.get(dataId).usage = null;
        return { dataId, shape, dtype };
    }
    makeOutput(shape, dtype, values) {
        const { dataId } = this.makeTensorInfo(shape, dtype, values);
        return engine().makeTensorFromDataId(dataId, shape, dtype, this);
    }
    unpackTensor(input) {
        const program = new UnpackProgram(input.shape);
        return this.runWebGLProgram(program, [input], input.dtype);
    }
    packTensor(input) {
        const program = new PackProgram(input.shape);
        const preventEagerUnpackingOutput = true;
        return this.runWebGLProgram(program, [input], input.dtype, null /* customUniformValues */, preventEagerUnpackingOutput);
    }
    packedReshape(input, afterShape) {
        const input3DShape = [
            webgl_util.getBatchDim(input.shape),
            ...webgl_util.getRowsCols(input.shape)
        ];
        const input3D = {
            dtype: input.dtype,
            shape: input3DShape,
            dataId: input.dataId
        };
        const afterShapeAs3D = [
            webgl_util.getBatchDim(afterShape), ...webgl_util.getRowsCols(afterShape)
        ];
        const program = new ReshapePackedProgram(afterShapeAs3D, input3DShape);
        const preventEagerUnpackingOfOutput = true;
        const customValues = [input3DShape];
        const output = this.runWebGLProgram(program, [input3D], input.dtype, customValues, preventEagerUnpackingOfOutput);
        return { dataId: output.dataId, shape: afterShape, dtype: output.dtype };
    }
    decode(dataId, customTexShape) {
        const texData = this.texData.get(dataId);
        const { isPacked, shape, dtype } = texData;
        if (customTexShape != null) {
            const size = util.sizeFromShape(shape);
            const texSize = customTexShape[0] * customTexShape[1] * 4;
            util.assert(size <= texSize, () => 'customTexShape is too small. ' +
                'Row * Column * 4 should be equal or larger than the ' +
                'size of the tensor data.');
        }
        const shapeAs3D = webgl_util.getShapeAs3D(shape);
        let program;
        if (isPacked) {
            program = new DecodeMatrixPackedProgram(shapeAs3D);
        }
        else {
            program = new DecodeMatrixProgram(shapeAs3D);
        }
        const preventEagerUnpackingOfOutput = true;
        const customValues = [customTexShape != null ? customTexShape :
                tex_util.getDenseTexShape(shapeAs3D)];
        const out = this.runWebGLProgram(program, [{ shape: shapeAs3D, dtype, dataId }], dtype, customValues, preventEagerUnpackingOfOutput, customTexShape);
        return { dtype, shape, dataId: out.dataId };
    }
    runWebGLProgram(program, inputs, outputDtype, customUniformValues, preventEagerUnpackingOfOutput = false, customTexShape) {
        const output = this.makeTensorInfo(program.outputShape, outputDtype);
        const outData = this.texData.get(output.dataId);
        if (program.packedOutput) {
            outData.isPacked = true;
        }
        if (program.outPackingScheme === tex_util.PackingScheme.DENSE) {
            const texelShape = customTexShape != null ?
                customTexShape :
                tex_util.getDenseTexShape(program.outputShape);
            // For a densely packed output, we explicitly set texShape
            // so it doesn't get assigned later according to our typical packing
            // scheme wherein a single texel can only contain values from adjacent
            // rows/cols.
            outData.texShape = texelShape.map(d => d * 2);
        }
        if (program.outTexUsage != null) {
            outData.usage = program.outTexUsage;
        }
        if (util.sizeFromShape(output.shape) === 0) {
            // Short-circuit the computation since the result is empty (has 0 in its
            // shape).
            outData.values =
                util.getTypedArrayFromDType(output.dtype, 0);
            return output;
        }
        const dataToDispose = [];
        const inputsData = inputs.map(input => {
            if (input.dtype === 'complex64') {
                throw new Error(`GPGPUProgram does not support complex64 input. For complex64 ` +
                    `dtypes, please separate the program into real and imaginary ` +
                    `parts.`);
            }
            let texData = this.texData.get(input.dataId);
            if (texData.texture == null) {
                if (!program.packedInputs &&
                    util.sizeFromShape(input.shape) <=
                        env().getNumber('WEBGL_SIZE_UPLOAD_UNIFORM')) {
                    // Upload small tensors that live on the CPU as uniforms, not as
                    // textures. Do this only when the environment supports 32bit floats
                    // due to problems when comparing 16bit floats with 32bit floats.
                    // TODO(https://github.com/tensorflow/tfjs/issues/821): Make it
                    // possible for packed shaders to sample from uniforms.
                    return {
                        shape: input.shape,
                        texData: null,
                        isUniform: true,
                        uniformValues: texData.values
                    };
                }
                // This ensures that if a packed program's inputs have not yet been
                // uploaded to the GPU, they get uploaded as packed right off the bat.
                if (program.packedInputs) {
                    texData.isPacked = true;
                    texData.shape = input.shape;
                }
            }
            this.uploadToGPU(input.dataId);
            if (!!texData.isPacked !== !!program.packedInputs) {
                input = texData.isPacked ? this.unpackTensor(input) :
                    this.packTensor(input);
                dataToDispose.push(input);
                texData = this.texData.get(input.dataId);
            }
            else if (texData.isPacked &&
                !webgl_util.isReshapeFree(texData.shape, input.shape)) {
                // This is a special case where a texture exists for a tensor
                // but the shapes are incompatible (due to packing constraints) because
                // the tensor did not have a chance to go through the packed reshape
                // shader. This only happens when we reshape the *same* tensor to form
                // *distinct* inputs to an op, e.g. dotting a vector with itself. This
                // case will disappear once packed uploading is the default.
                const savedInput = input;
                const targetShape = input.shape;
                input.shape = texData.shape;
                input = this.packedReshape(input, targetShape);
                dataToDispose.push(input);
                texData = this.texData.get(input.dataId);
                savedInput.shape = targetShape;
            }
            return { shape: input.shape, texData, isUniform: false };
        });
        this.uploadToGPU(output.dataId);
        const outputData = { shape: output.shape, texData: outData, isUniform: false };
        const key = gpgpu_math.makeShaderKey(program, inputsData, outputData);
        const binary = this.getAndSaveBinary(key, () => {
            return gpgpu_math.compileProgram(this.gpgpu, program, inputsData, outputData);
        });
        const shouldTimeProgram = this.activeTimers != null;
        let query;
        if (shouldTimeProgram) {
            query = this.startTimer();
        }
        gpgpu_math.runProgram(this.gpgpu, binary, inputsData, outputData, customUniformValues);
        dataToDispose.forEach(info => this.disposeIntermediateTensorInfo(info));
        if (shouldTimeProgram) {
            query = this.endTimer(query);
            this.activeTimers.push({ name: program.constructor.name, query: this.getQueryTime(query) });
        }
        const glFlushThreshold = env().get('WEBGL_FLUSH_THRESHOLD');
        // Manually GL flush requested
        if (glFlushThreshold > 0) {
            const time = util.now();
            if ((time - this.lastGlFlushTime) > glFlushThreshold) {
                this.gpgpu.gl.flush();
                this.lastGlFlushTime = time;
            }
        }
        if (!env().getBool('WEBGL_LAZILY_UNPACK') && outData.isPacked &&
            preventEagerUnpackingOfOutput === false) {
            const unpacked = this.unpackTensor(output);
            this.disposeIntermediateTensorInfo(output);
            return unpacked;
        }
        return output;
    }
    compileAndRun(program, inputs, outputDtype, customUniformValues, preventEagerUnpackingOfOutput = false) {
        outputDtype = outputDtype || inputs[0].dtype;
        const outInfo = this.runWebGLProgram(program, inputs, outputDtype, customUniformValues, preventEagerUnpackingOfOutput);
        return outInfo;
    }
    getAndSaveBinary(key, getBinary) {
        if (!(key in this.binaryCache)) {
            this.binaryCache[key] = getBinary();
        }
        return this.binaryCache[key];
    }
    getTextureManager() {
        return this.textureManager;
    }
    dispose() {
        if (this.disposed) {
            return;
        }
        // Avoid disposing the compiled webgl programs during unit testing because
        // it slows down test execution.
        if (!env().getBool('IS_TEST')) {
            const allKeys = Object.keys(this.binaryCache);
            allKeys.forEach(key => {
                this.gpgpu.deleteProgram(this.binaryCache[key].webGLProgram);
                delete this.binaryCache[key];
            });
        }
        this.textureManager.dispose();
        if (this.canvas != null &&
            (typeof (HTMLCanvasElement) !== 'undefined' &&
                this.canvas instanceof HTMLCanvasElement)) {
            this.canvas.remove();
        }
        else {
            this.canvas = null;
        }
        if (this.gpgpuCreatedLocally) {
            this.gpgpu.program = null;
            this.gpgpu.dispose();
        }
        this.disposed = true;
    }
    floatPrecision() {
        if (this.floatPrecisionValue == null) {
            this.floatPrecisionValue = tidy(() => {
                if (!env().get('WEBGL_RENDER_FLOAT32_ENABLED')) {
                    // Momentarily switching DEBUG flag to false so we don't throw an
                    // error trying to upload a small value.
                    const debugFlag = env().getBool('DEBUG');
                    env().set('DEBUG', false);
                    const underflowCheckValue = this.abs(scalar(1e-8)).dataSync()[0];
                    env().set('DEBUG', debugFlag);
                    if (underflowCheckValue > 0) {
                        return 32;
                    }
                }
                return 16;
            });
        }
        return this.floatPrecisionValue;
    }
    /** Returns the smallest representable number.  */
    epsilon() {
        return this.floatPrecision() === 32 ? EPSILON_FLOAT32 : EPSILON_FLOAT16;
    }
    uploadToGPU(dataId) {
        const texData = this.texData.get(dataId);
        const { shape, dtype, values, texture, usage, isPacked } = texData;
        if (texture != null) {
            // Array is already on GPU. No-op.
            return;
        }
        const shouldTimeProgram = this.activeTimers != null;
        let start;
        if (shouldTimeProgram) {
            start = util.now();
        }
        let texShape = texData.texShape;
        if (texShape == null) {
            // This texShape may not be the final texture shape. For packed or dense
            // textures, the texShape will be changed when textures are created.
            texShape = webgl_util.getTextureShapeFromLogicalShape(shape, isPacked);
            texData.texShape = texShape;
        }
        if (values != null) {
            const shapeAs3D = webgl_util.getShapeAs3D(shape);
            let program;
            let width = texShape[1], height = texShape[0];
            const isByteArray = values instanceof Uint8Array || values instanceof Uint8ClampedArray;
            // texture for float array is PhysicalTextureType.PACKED_2X2_FLOAT32, we
            // need to make sure the upload uses the same packed size
            if (isPacked || !isByteArray) {
                [width, height] = tex_util.getPackedMatrixTextureShapeWidthHeight(texShape[0], texShape[1]);
            }
            if (isPacked) {
                program = new EncodeMatrixPackedProgram(shapeAs3D, isByteArray);
            }
            else {
                program = new EncodeMatrixProgram(shapeAs3D, isByteArray);
            }
            // TexShape for float array needs to be the original shape, which byte
            // array needs to be packed size. This allow the data upload shape to be
            // matched with texture creation logic.
            const tempDenseInputTexShape = isByteArray ? [height, width] : texShape;
            const tempDenseInputHandle = this.makeTensorInfo(tempDenseInputTexShape, dtype);
            const tempDenseInputTexData = this.texData.get(tempDenseInputHandle.dataId);
            if (isByteArray) {
                tempDenseInputTexData.usage = TextureUsage.PIXELS;
            }
            else {
                tempDenseInputTexData.usage = TextureUsage.UPLOAD;
            }
            tempDenseInputTexData.texShape = tempDenseInputTexShape;
            this.gpgpu.uploadDenseMatrixToTexture(this.getTexture(tempDenseInputHandle.dataId), width, height, values);
            const customValues = [[height, width]];
            // We want the output to remain packed regardless of the value of
            // WEBGL_PACK.
            const preventEagerUnpacking = true;
            const encodedOutputTarget = this.runWebGLProgram(program, [tempDenseInputHandle], dtype, customValues, preventEagerUnpacking);
            // Have the original texture assume the identity of the encoded output.
            const outputTexData = this.texData.get(encodedOutputTarget.dataId);
            texData.texture = outputTexData.texture;
            texData.texShape = outputTexData.texShape;
            texData.isPacked = outputTexData.isPacked;
            texData.usage = outputTexData.usage;
            this.disposeIntermediateTensorInfo(tempDenseInputHandle);
            this.texData.delete(encodedOutputTarget.dataId);
            // Once uploaded, don't store the values on cpu.
            texData.values = null;
            if (shouldTimeProgram) {
                this.uploadWaitMs += util.now() - start;
            }
        }
        else {
            const newTexture = this.acquireTexture(texShape, usage, dtype, isPacked);
            texData.texture = newTexture;
        }
    }
    convertAndCacheOnCPU(dataId, float32Values) {
        const texData = this.texData.get(dataId);
        const { dtype } = texData;
        this.releaseGPUData(dataId);
        if (float32Values != null) {
            texData.values = float32ToTypedArray(float32Values, dtype);
        }
        return texData.values;
    }
    acquireTexture(texShape, texType, dtype, isPacked) {
        this.numBytesInGPU += this.computeBytes(texShape, dtype);
        if (!this.warnedAboutMemory &&
            this.numBytesInGPU > this.numMBBeforeWarning * 1024 * 1024) {
            const mb = (this.numBytesInGPU / 1024 / 1024).toFixed(2);
            this.warnedAboutMemory = true;
            console.warn(`High memory usage in GPU: ${mb} MB, ` +
                `most likely due to a memory leak`);
        }
        return this.textureManager.acquireTexture(texShape, texType, isPacked);
    }
    computeBytes(shape, dtype) {
        return shape[0] * shape[1] * util.bytesPerElement(dtype);
    }
}
MathBackendWebGL.nextDataId = 0;
function float32ToTypedArray(a, dtype) {
    if (dtype === 'float32' || dtype === 'complex64') {
        return a;
    }
    else if (dtype === 'int32' || dtype === 'bool') {
        const result = (dtype === 'int32') ? new Int32Array(a.length) :
            new Uint8Array(a.length);
        for (let i = 0; i < result.length; ++i) {
            result[i] = Math.round(a[i]);
        }
        return result;
    }
    else {
        throw new Error(`Unknown dtype ${dtype}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFja2VuZF93ZWJnbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uL3RmanMtYmFja2VuZC13ZWJnbC9zcmMvYmFja2VuZF93ZWJnbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFFSCxzQkFBc0I7QUFDdEIsT0FBTyxlQUFlLENBQUM7QUFHdkIsT0FBTyxFQUFDLFlBQVksRUFBaUIsTUFBTSxFQUFVLFdBQVcsRUFBOEMsTUFBTSxFQUFFLEdBQUcsRUFBVyxZQUFZLEVBQUUsYUFBYSxFQUFxRCxNQUFNLEVBQXdELElBQUksRUFBMEIsSUFBSSxFQUFDLE1BQU0sdUJBQXVCLENBQUM7QUFFblYsT0FBTyxFQUFDLGVBQWUsRUFBQyxNQUFNLGVBQWUsQ0FBQztBQUM5QyxPQUFPLEVBQUMsbUJBQW1CLEVBQUMsTUFBTSxxQkFBcUIsQ0FBQztBQUN4RCxPQUFPLEVBQUMseUJBQXlCLEVBQUMsTUFBTSw0QkFBNEIsQ0FBQztBQUNyRSxPQUFPLEVBQUMsa0JBQWtCLEVBQUMsTUFBTSxvQkFBb0IsQ0FBQztBQUN0RCxPQUFPLEVBQUMsd0JBQXdCLEVBQUMsTUFBTSwyQkFBMkIsQ0FBQztBQUNuRSxPQUFPLEVBQUMsbUJBQW1CLEVBQUMsTUFBTSxxQkFBcUIsQ0FBQztBQUN4RCxPQUFPLEVBQUMseUJBQXlCLEVBQUMsTUFBTSw0QkFBNEIsQ0FBQztBQUNyRSxPQUFPLEVBQUMsWUFBWSxFQUFDLE1BQU0saUJBQWlCLENBQUM7QUFDN0MsT0FBTyxLQUFLLFVBQVUsTUFBTSxjQUFjLENBQUM7QUFFM0MsT0FBTyxFQUFDLGdCQUFnQixFQUFDLE1BQU0sdUJBQXVCLENBQUM7QUFDdkQsT0FBTyxFQUFDLFdBQVcsRUFBQyxNQUFNLFlBQVksQ0FBQztBQUN2QyxPQUFPLEVBQUMsb0JBQW9CLEVBQUMsTUFBTSxzQkFBc0IsQ0FBQztBQUMxRCxPQUFPLEtBQUssUUFBUSxNQUFNLFlBQVksQ0FBQztBQUN2QyxPQUFPLEVBQXVCLFlBQVksRUFBQyxNQUFNLFlBQVksQ0FBQztBQUM5RCxPQUFPLEVBQUMsY0FBYyxFQUFDLE1BQU0sbUJBQW1CLENBQUM7QUFDakQsT0FBTyxLQUFLLFFBQVEsTUFBTSxlQUFlLENBQUM7QUFDMUMsT0FBTyxFQUFDLGNBQWMsRUFBQyxNQUFNLGVBQWUsQ0FBQztBQUM3QyxPQUFPLEVBQUMsb0JBQW9CLEVBQUMsTUFBTSxzQkFBc0IsQ0FBQztBQUMxRCxPQUFPLEVBQUMsYUFBYSxFQUFDLE1BQU0sY0FBYyxDQUFDO0FBQzNDLE9BQU8sS0FBSyxVQUFVLE1BQU0sY0FBYyxDQUFDO0FBRTNDLE1BQU0sU0FBUyxHQUFHLFlBQVksQ0FBQyxTQUFTLENBQUM7QUFFekMsTUFBTSxDQUFDLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQztBQUNwQyxNQUFNLENBQUMsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDO0FBNEJwQyxNQUFNLFlBQVksR0FBMkQsRUFBRSxDQUFDO0FBRWhGLE1BQU0sVUFBVSxjQUFjLENBQUMsWUFBb0I7SUFDakQsSUFBSSxZQUFZLElBQUksWUFBWSxFQUFFO1FBQ2hDLE9BQU8sWUFBWSxDQUFDLFlBQVksQ0FBQyxDQUFDO0tBQ25DO0lBQ0QsWUFBWSxDQUFDLFlBQVksQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUNoQyxPQUFPLFlBQVksQ0FBQyxZQUFZLENBQUMsQ0FBQztBQUNwQyxDQUFDO0FBRUQsK0VBQStFO0FBQy9FLDRCQUE0QjtBQUM1QixNQUFNLDBCQUEwQixHQUM1QixHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsNEJBQTRCLENBQUMsQ0FBQztBQUVsRCx5RUFBeUU7QUFDekUsK0VBQStFO0FBQy9FLHVCQUF1QjtBQUN2QixNQUFNLHNCQUFzQixHQUFHLEdBQUcsQ0FBQztBQUNuQyxTQUFTLGtCQUFrQjtJQUN6QixJQUFJLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFO1FBQy9CLE9BQU8sSUFBSSxDQUFDLENBQUUsUUFBUTtLQUN2QjtJQUNELE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQUs7UUFDdEQsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1FBQzVCLHNCQUFzQixHQUFHLElBQUksR0FBRyxJQUFJLENBQUM7QUFDM0MsQ0FBQztBQUVELE1BQU0sT0FBTyxnQkFBaUIsU0FBUSxhQUFhO0lBd0NqRCxZQUFZLFdBQTREO1FBQ3RFLEtBQUssRUFBRSxDQUFDO1FBakNWLDRFQUE0RTtRQUNwRSxnQkFBVyxHQUFHLElBQUksT0FBTyxFQUE0QyxDQUFDO1FBQzlFLHlFQUF5RTtRQUN6RSwwQkFBMEI7UUFDbEIsb0JBQWUsR0FBRyxJQUFJLE9BQU8sRUFBVSxDQUFDO1FBRWhELHlFQUF5RTtRQUN6RSxnQkFBZ0I7UUFDaEIsaUJBQVksR0FBRyxJQUFJLE9BQU8sRUFBa0IsQ0FBQztRQUNyQyxrQkFBYSxHQUFHLENBQUMsQ0FBQztRQU0xQiwwRUFBMEU7UUFDbEUsaUJBQVksR0FBRyxDQUFDLENBQUM7UUFDekIsNkVBQTZFO1FBQ3JFLG1CQUFjLEdBQUcsQ0FBQyxDQUFDO1FBRTNCLHdDQUF3QztRQUNoQyxvQkFBZSxHQUFHLENBQUMsQ0FBQztRQVNwQixzQkFBaUIsR0FBRyxLQUFLLENBQUM7UUE4YzFCLG1CQUFjLEdBQUcsQ0FBQyxDQUFDO1FBZ1puQixhQUFRLEdBQUcsS0FBSyxDQUFDO1FBMTFCdkIsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUMvQixNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7U0FDMUQ7UUFFRCxJQUFJLFFBQVEsQ0FBQztRQUNiLElBQUksV0FBVyxJQUFJLElBQUksRUFBRTtZQUN2QixJQUFJLFdBQVcsWUFBWSxZQUFZLEVBQUU7Z0JBQ3ZDLFFBQVEsR0FBRyxXQUFXLENBQUM7YUFDeEI7aUJBQU07Z0JBQ0wsTUFBTSxFQUFFLEdBQ0osZUFBZSxDQUFDLEdBQUcsRUFBRSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsRUFBRSxXQUFXLENBQUMsQ0FBQztnQkFDbkUsUUFBUSxHQUFHLElBQUksWUFBWSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQ2pDO1lBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRyxFQUFFLENBQUM7WUFDdEIsSUFBSSxDQUFDLG1CQUFtQixHQUFHLEtBQUssQ0FBQztTQUNsQzthQUFNO1lBQ0wsTUFBTSxFQUFFLEdBQUcsZUFBZSxDQUFDLEdBQUcsRUFBRSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO1lBQzdELFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNoQyxJQUFJLENBQUMsV0FBVyxHQUFHLGNBQWMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQztZQUNwRSxJQUFJLENBQUMsbUJBQW1CLEdBQUcsSUFBSSxDQUFDO1NBQ2pDO1FBRUQsSUFBSSxDQUFDLEtBQUssR0FBRyxRQUFRLENBQUM7UUFDdEIsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUM7UUFDbkMsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDckQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLGtCQUFrQixFQUFFLENBQUM7UUFDL0MsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLFdBQVcsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBaEVPLFVBQVU7UUFDaEIsT0FBTyxnQkFBZ0IsQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUN2QyxDQUFDO0lBZ0VELFVBQVU7UUFDUixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQztJQUN6RCxDQUFDO0lBRUQsS0FBSyxDQUFDLE1BQXFCLEVBQUUsS0FBZSxFQUFFLEtBQWU7UUFDM0QsSUFBSSxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsZ0NBQWdDLENBQUM7WUFDL0MsR0FBRyxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzFCLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUNyQztRQUNELElBQUksS0FBSyxLQUFLLFdBQVcsSUFBSSxNQUFNLElBQUksSUFBSSxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQ1gscUNBQXFDO2dCQUNyQyxvQ0FBb0MsQ0FBQyxDQUFDO1NBQzNDO1FBQ0QsTUFBTSxNQUFNLEdBQUcsRUFBQyxFQUFFLEVBQUUsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQ1osTUFBTSxFQUNOLEVBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLFlBQVksQ0FBQyxNQUFNLEVBQUUsUUFBUSxFQUFFLENBQUMsRUFBQyxDQUFDLENBQUM7UUFDckUsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUVELHlDQUF5QztJQUN6QyxRQUFRLENBQUMsTUFBYztRQUNyQixJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQzVCLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzVDLE9BQU8sVUFBVSxDQUFDLFFBQVEsQ0FBQztTQUM1QjtRQUNELE9BQU8sQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUVELDRDQUE0QztJQUM1QyxNQUFNLENBQUMsTUFBYztRQUNuQixNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxPQUFPLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDckIsQ0FBQztJQUVELDRDQUE0QztJQUM1QyxNQUFNLENBQUMsTUFBYztRQUNuQixJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQzVCLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3pDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUNwQjtJQUNILENBQUM7SUFFRCxJQUFJLENBQ0EsTUFBYyxFQUFFLE1BQXFCLEVBQUUsS0FBZSxFQUFFLEtBQWUsRUFDdkUsUUFBZ0I7UUFDbEIsSUFBSSxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDMUIsSUFBSSxDQUFDLHNCQUFzQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3JDO1FBQ0QsSUFBSSxLQUFLLEtBQUssV0FBVyxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQ1gscUNBQXFDO2dCQUNyQyxvQ0FBb0MsQ0FBQyxDQUFDO1NBQzNDO1FBQ0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQ1osTUFBTSxFQUFFLEVBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLFlBQVksQ0FBQyxNQUFNLEVBQUUsUUFBUSxFQUFDLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQsNkJBQTZCLENBQUMsVUFBc0I7UUFDbEQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVELFFBQVEsQ0FBQyxNQUFjO1FBQ3JCLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pDLE1BQU0sRUFBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLGtCQUFrQixFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFDLEdBQUcsT0FBTyxDQUFDO1FBRTVFLHdFQUF3RTtRQUN4RSxxRUFBcUU7UUFDckUsMERBQTBEO1FBQzFELElBQUksS0FBSyxJQUFJLElBQUksRUFBRTtZQUNqQixJQUFJLE9BQU8sQ0FBQztZQUNaLElBQUksUUFBUSxFQUFFO2dCQUNaLE9BQU8sR0FBRyxJQUFJLG9CQUFvQixDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDM0Q7aUJBQU07Z0JBQ0wsT0FBTyxHQUFHLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDckQ7WUFDRCxNQUFNLEdBQUcsR0FDTCxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLEVBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ25FLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3ZDLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUN4QyxPQUFPLElBQUksQ0FBQztTQUNiO1FBQ0QsSUFBSSxNQUFNLElBQUksSUFBSSxFQUFFO1lBQ2xCLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzFDO1FBQ0QsSUFBSSxLQUFLLEtBQUssUUFBUSxFQUFFO1lBQ3RCLE9BQU8sTUFBTSxDQUFDO1NBQ2Y7UUFDRCxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDO1FBQ3BELElBQUksS0FBYSxDQUFDO1FBQ2xCLElBQUksaUJBQWlCLEVBQUU7WUFDckIsS0FBSyxHQUFHLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQztTQUNwQjtRQUVELElBQUksTUFBb0IsQ0FBQztRQUN6QixJQUFJLEtBQUssS0FBSyxXQUFXLEVBQUU7WUFDekIsTUFBTSxVQUFVLEdBQ1osSUFBSSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFpQixDQUFDO1lBQ2xFLE1BQU0sVUFBVSxHQUNaLElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBaUIsQ0FBQztZQUNsRSxNQUFNLEdBQUcsWUFBWSxDQUFDLHNCQUFzQixDQUFDLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztTQUN0RTthQUFNO1lBQ0wsTUFBTSxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUM1QztRQUVELElBQUksaUJBQWlCLEVBQUU7WUFDckIsSUFBSSxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsR0FBRyxFQUFFLEdBQUcsS0FBSyxDQUFDO1NBQzNDO1FBQ0QsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRCxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQWM7UUFDdkIsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNoQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNqRCxPQUFPLElBQUksT0FBTyxDQUFhLE9BQU8sQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1NBQ3RFO1FBQ0QsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDekMsTUFBTSxFQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxRQUFRLEVBQUMsR0FBRyxPQUFPLENBQUM7UUFFNUUsd0VBQXdFO1FBQ3hFLHFFQUFxRTtRQUNyRSwwREFBMEQ7UUFDMUQsSUFBSSxLQUFLLElBQUksSUFBSSxFQUFFO1lBQ2pCLElBQUksT0FBTyxDQUFDO1lBQ1osSUFBSSxRQUFRLEVBQUU7Z0JBQ1osT0FBTyxHQUFHLElBQUksb0JBQW9CLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUMzRDtpQkFBTTtnQkFDTCxPQUFPLEdBQUcsSUFBSSxjQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUNyRDtZQUNELE1BQU0sR0FBRyxHQUNMLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxFQUFFLENBQUMsRUFBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDbkUsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDbkMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ3hDLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFFRCxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDbEIsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDMUM7UUFFRCxJQUFJLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMxQixzRUFBc0U7WUFDdEUsc0VBQXNFO1lBQ3RFLHNFQUFzRTtZQUN0RSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsT0FBTyxDQUFDLDhCQUE4QixDQUFDO2dCQUM5QyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUMxQyxNQUFNLElBQUksS0FBSyxDQUNYLDREQUE0RDtvQkFDNUQsb0NBQW9DLENBQUMsQ0FBQzthQUMzQztTQUNGO1FBRUQsSUFBSSxNQUFNLEdBQWdCLElBQUksQ0FBQztRQUMvQixJQUFJLGlCQUE2QixDQUFDO1FBRWxDLElBQUksS0FBSyxLQUFLLFdBQVcsSUFBSSxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsRUFBRTtZQUNoRSxvRUFBb0U7WUFDcEUsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN4QyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUUzRCxNQUFNLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsQ0FDdkMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztTQUNuRTtRQUVELElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQyxJQUFJLEtBQUssS0FBSyxXQUFXLEVBQUU7WUFDekIsNkNBQTZDO1lBQzdDLE1BQU0sSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsRUFBRSxDQUFDO1NBQzFDO1FBRUQsb0NBQW9DO1FBQ3BDLElBQUksSUFBa0IsQ0FBQztRQUN2QixJQUFJLEtBQUssS0FBSyxXQUFXLEVBQUU7WUFDekIsTUFBTSxFQUFFLEdBQUcsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDO2dCQUMzQixJQUFJLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7Z0JBQ3pDLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQzthQUMxQyxDQUFDLENBQUM7WUFFSCxNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDekIsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pCLElBQUksR0FBRyxZQUFZLENBQUMsc0JBQXNCLENBQ3RDLFVBQTBCLEVBQUUsVUFBMEIsQ0FBQyxDQUFDO1NBQzdEO2FBQU0sSUFBSSxNQUFNLElBQUksSUFBSSxFQUFFO1lBQ3pCLElBQUksR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDMUM7YUFBTTtZQUNMLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdkMsSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsK0JBQStCLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQ2pFO1FBQ0QsSUFBSSxpQkFBaUIsSUFBSSxJQUFJLEVBQUU7WUFDN0IsSUFBSSxDQUFDLDZCQUE2QixDQUFDLGlCQUFpQixDQUFDLENBQUM7U0FDdkQ7UUFDRCxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDbEIsTUFBTSxFQUFFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDekIsVUFBVSxDQUFDLFlBQVksQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1NBQzVEO1FBQ0QsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztRQUUxRCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVoQyw0QkFBNEI7UUFDNUIsV0FBVyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ25ELElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDcEMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDcEMsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUM1QixNQUFNLEVBQUUsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO2FBQ3JDO1lBQ0QsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO1NBQ3ZCO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNILFNBQVMsQ0FBQyxNQUFjLEVBQUUsVUFBZ0MsRUFBRTtRQUMxRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxNQUFNLEVBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQUMsR0FBRyxPQUFPLENBQUM7UUFFakUsSUFBSSxLQUFLLEtBQUssV0FBVyxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQztTQUMxRTtRQUVELHdFQUF3RTtRQUN4RSxxRUFBcUU7UUFDckUsMERBQTBEO1FBQzFELElBQUksS0FBSyxJQUFJLElBQUksRUFBRTtZQUNqQixJQUFJLE9BQU8sQ0FBQztZQUNaLElBQUksUUFBUSxFQUFFO2dCQUNaLE9BQU8sR0FBRyxJQUFJLG9CQUFvQixDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDM0Q7aUJBQU07Z0JBQ0wsT0FBTyxHQUFHLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDckQ7WUFDRCxNQUFNLEdBQUcsR0FDTCxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLEVBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ25FLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2xELElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUN4QyxPQUFPLFlBQVksQ0FBQztTQUNyQjtRQUVELElBQUksT0FBTyxJQUFJLElBQUksRUFBRTtZQUNuQixJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7Z0JBQ2xCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0NBQWdDLENBQUMsQ0FBQzthQUNuRDtpQkFBTTtnQkFDTCxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7YUFDcEQ7U0FDRjtRQUVELHlFQUF5RTtRQUN6RSxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFOUQsa0VBQWtFO1FBQ2xFLE1BQU0sU0FBUyxHQUFHLE1BQU0sRUFBRSxDQUFDLG9CQUFvQixDQUMzQyxTQUFTLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXhELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNuRCx1QkFBUSxTQUFTLElBQUssT0FBTyxDQUFDLE9BQU8sRUFBRTtJQUN6QyxDQUFDO0lBRUQsVUFBVSxDQUFpQixDQUFhO1FBQ3RDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3JDLElBQUksV0FBVyxHQUFHLElBQWtCLENBQUM7UUFDckMsSUFBSSxDQUFDLENBQUMsS0FBSyxLQUFLLFFBQVEsRUFBRTtZQUN4QixJQUFJO2dCQUNGLGdDQUFnQztnQkFDaEMsV0FBVyxHQUFJLElBQXFCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ3JFO1lBQUMsV0FBTTtnQkFDTixNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7YUFDckU7U0FDRjtRQUNELE9BQU8sTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFvQixFQUFFLENBQUMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUN2QyxDQUFDO0lBQ3RCLENBQUM7SUFFTyxzQkFBc0IsQ0FBQyxNQUFxQjtRQUNsRCxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDbEIsT0FBTztTQUNSO1FBQ0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDdEMsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBVyxDQUFDO1lBQ2hDLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQ3JDLElBQUksR0FBRyxFQUFFLENBQUMsT0FBTyxDQUFDLDhCQUE4QixDQUFDLEVBQUU7b0JBQ2pELE1BQU0sS0FBSyxDQUNQLGFBQWEsR0FBRyxtQ0FBbUM7d0JBQ25ELHlEQUF5RDt3QkFDekQsdURBQXVELENBQUMsQ0FBQztpQkFDOUQ7Z0JBQ0QsTUFBTSxLQUFLLENBQUMsYUFBYSxHQUFHLHdDQUF3QyxDQUFDLENBQUM7YUFDdkU7U0FDRjtJQUNILENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxNQUFjO1FBQ3pDLE1BQU0sRUFBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdkMsSUFBSSxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsOEJBQThCLENBQUMsRUFBRTtZQUNqRCxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3RDLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNuRCxNQUFNLElBQUksR0FDTixJQUFJLENBQUMsS0FBSztpQkFDTCwrQkFBK0IsQ0FDNUIsT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7aUJBQ2hFLFFBQVEsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFFM0IsSUFBSSxDQUFDLDZCQUE2QixDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBRTlDLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFFRCxNQUFNLHNCQUFzQixHQUN4QixHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLElBQUksUUFBUSxLQUFLLElBQUksQ0FBQztRQUNyRCxNQUFNLFdBQVcsR0FDYixzQkFBc0IsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1FBQ3BFLE1BQU0sT0FBTyxHQUFHLHNCQUFzQixDQUFDLENBQUM7WUFDcEMsSUFBSSx3QkFBd0IsQ0FBQyxXQUF1QyxDQUFDLENBQUMsQ0FBQztZQUN2RSxJQUFJLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQy9CLE9BQU8sRUFBRSxDQUFDLEVBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFDLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUMvRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDaEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUs7YUFDTCwrQ0FBK0MsQ0FDNUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFDNUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUN2QixRQUFRLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRCxjQUFjO1FBQ1osT0FBTyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsK0NBQStDLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDOUUsQ0FBQztJQUVELElBQUksQ0FBQyxDQUFhO1FBQ2hCLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUM7UUFDMUMsTUFBTSxlQUFlLEdBQWdCLEVBQUUsQ0FBQztRQUV4QyxJQUFJLGFBQWEsR0FBRyxLQUFLLENBQUM7UUFDMUIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLElBQUksSUFBSSxFQUFFO1lBQ25DLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxlQUFlLENBQUM7WUFDMUMsYUFBYSxHQUFHLElBQUksQ0FBQztTQUN0QjthQUFNO1lBQ0wsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDekM7UUFDRCxJQUFJLENBQUMsWUFBWSxHQUFHLGVBQWUsQ0FBQztRQUVwQyxDQUFDLEVBQUUsQ0FBQztRQUVKLDRFQUE0RTtRQUM1RSxNQUFNLDJCQUEyQixHQUM3QixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBYSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDMUQsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDO1FBQ2hDLE1BQU0seUJBQXlCLEdBQzNCLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFhLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUN6RCxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLENBQUM7UUFFaEMsSUFBSSxDQUFDLFlBQVksR0FBRyxlQUFlLENBQUM7UUFFcEMsSUFBSSxhQUFhLEVBQUU7WUFDakIsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztTQUNoQztRQUVELE1BQU0sR0FBRyxHQUFvQjtZQUMzQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO1lBQ25DLFFBQVEsRUFBRSxJQUFJO1lBQ2QsTUFBTSxFQUFFLElBQUksQ0FBRSwrQkFBK0I7U0FDOUMsQ0FBQztRQUVGLE9BQU8sQ0FBQyxLQUFLLElBQUksRUFBRTtZQUNqQixJQUFJLEdBQUcsRUFBRTtpQkFDTixTQUFTLENBQUMsK0NBQStDLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQ2pFLE1BQU0sUUFBUSxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO2dCQUVoRSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDckMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLEdBQUcsR0FBRyxFQUFFLENBQ2hDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUMsQ0FBQyxDQUFDO3FCQUNsRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDO3FCQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDakI7aUJBQU07Z0JBQ0wsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHO29CQUNoQixLQUFLLEVBQUUsMkRBQTJEO2lCQUNuRSxDQUFDO2FBQ0g7WUFFRCxJQUFJLENBQUMsWUFBWSxHQUFHLENBQUMsQ0FBQztZQUN0QixJQUFJLENBQUMsY0FBYyxHQUFHLENBQUMsQ0FBQztZQUN4QixPQUFPLEdBQUcsQ0FBQztRQUNiLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDUCxDQUFDO0lBQ0QsTUFBTTtRQUNKLE9BQU87WUFDTCxVQUFVLEVBQUUsS0FBSztZQUNqQixhQUFhLEVBQUUsSUFBSSxDQUFDLGFBQWE7WUFDakMsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxpQkFBaUI7WUFDN0QsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZO1NBQ2pDLENBQUM7SUFDdkIsQ0FBQztJQUVPLFVBQVU7UUFDaEIsSUFBSSxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsK0NBQStDLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDeEUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDO1NBQ2hDO1FBQ0QsT0FBTyxFQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBQyxDQUFDO0lBQzVDLENBQUM7SUFFTyxRQUFRLENBQUMsS0FBK0I7UUFDOUMsSUFBSSxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsK0NBQStDLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDeEUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUN0QixPQUFPLEtBQUssQ0FBQztTQUNkO1FBQ0EsS0FBdUIsQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVDLE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVPLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBK0I7UUFDeEQsSUFBSSxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsK0NBQStDLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDeEUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLEtBQW1CLENBQUMsQ0FBQztTQUMvRDtRQUNELE1BQU0sVUFBVSxHQUFHLEtBQXNCLENBQUM7UUFDMUMsT0FBTyxVQUFVLENBQUMsS0FBSyxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUM7SUFDL0MsQ0FBQztJQUlEOzs7Ozs7Ozs7T0FTRztJQUNILFdBQVcsQ0FBQyxNQUFjLEVBQUUsS0FBSyxHQUFHLEtBQUs7UUFDdkMsSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNwQyxPQUFPLEtBQUssQ0FBQztTQUNkO1FBRUQsNkJBQTZCO1FBQzdCLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUM3QixPQUFPLElBQUksQ0FBQztTQUNiO1FBRUQseUVBQXlFO1FBQ3pFLG9FQUFvRTtRQUNwRSxrRUFBa0U7UUFDbEUsSUFBSSxLQUFLLEVBQUU7WUFDVCxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxRQUFRLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZDO2FBQU07WUFDTCxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUNyQztRQUVELElBQUksQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsUUFBUSxHQUFHLENBQUMsRUFBRTtZQUNuRCxPQUFPLEtBQUssQ0FBQztTQUNkO1FBRUQsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNoQyxJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNqQyxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7WUFDdEIsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUVELElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDNUIsTUFBTSxFQUFDLGtCQUFrQixFQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDdEQsSUFBSSxrQkFBa0IsSUFBSSxJQUFJLEVBQUU7WUFDOUIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3hELElBQUksQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztTQUN6RDtRQUVELElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRTVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVPLGNBQWMsQ0FBQyxNQUFjO1FBQ25DLE1BQU0sRUFBQyxPQUFPLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBQyxHQUNwRCxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM3QixNQUFNLEdBQUcsR0FBRyxLQUFLLElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxNQUFNLENBQUM7UUFDaEQsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFNUMsSUFBSSxRQUFRLEdBQUcsQ0FBQyxFQUFFO1lBQ2hCLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLEdBQUcsRUFBRSxRQUFRLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDMUM7YUFBTTtZQUNMLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzlCLElBQUksT0FBTyxJQUFJLElBQUksRUFBRTtnQkFDbkIsSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDekQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7YUFDeEU7U0FDRjtRQUVELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDO1FBQ3ZCLE9BQU8sQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDO1FBQ3hCLE9BQU8sQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO1FBQ3pCLE9BQU8sQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDO0lBQ3ZCLENBQUM7SUFFRCxVQUFVLENBQUMsTUFBYztRQUN2QixJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQztJQUNsRCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsV0FBVyxDQUFDLE1BQWM7UUFDeEIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsa0JBQWtCLENBQ2QsTUFBb0IsRUFDcEIsYUFBYSxHQUFHLDBCQUEwQjtRQUM1QyxPQUFPLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQztZQUNyQyxNQUFNLENBQUMsS0FBSyxDQUNSLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sSUFBSSxJQUFJO2dCQUNuRCxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsR0FBRyxhQUFhLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQsZUFBZTtRQUNiLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQztJQUNwQixDQUFDO0lBRUQsS0FBSyxDQUFDLFNBQWlCO1FBQ3JCLFlBQVksQ0FBQyxJQUFJLENBQ2IsMkNBQTJDO1lBQzNDLDhCQUE4QixDQUFDLENBQUM7UUFDcEMsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ3RDLE9BQU8sU0FBUyxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVPLGFBQWEsQ0FBQyxDQUFhLEVBQUUsRUFBVSxFQUFFLEtBQWU7UUFDOUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEQsT0FBTyxNQUFNLEVBQUUsQ0FBQyxvQkFBb0IsQ0FDaEMsT0FBTyxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQsc0VBQXNFO0lBQ3RFLHdEQUF3RDtJQUN4RCxvQ0FBb0M7SUFDcEMsR0FBRyxDQUFtQixDQUFJO1FBQ3hCLHdDQUF3QztRQUN4QyxJQUFJLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssS0FBSyxXQUFXLEVBQUU7WUFDM0QsTUFBTSxTQUFTLEdBQ1gsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQW9CLENBQUMsQ0FBQztZQUN0RSxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsNkJBQTZCLENBQUMsRUFBRTtZQUNoRCxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBTSxDQUFDO1NBQzFEO1FBRUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxjQUFjLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDMUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sTUFBTSxFQUFFLENBQUMsb0JBQW9CLENBQ3pCLE9BQU8sQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsS0FBSyxDQUFNLENBQUM7SUFDaEUsQ0FBQztJQUVELGNBQWMsQ0FDVixLQUFlLEVBQUUsS0FBZSxFQUNoQyxNQUErQjtRQUNqQyxJQUFJLE1BQU0sQ0FBQztRQUNYLElBQUksS0FBSyxLQUFLLFFBQVEsSUFBSSxNQUFNLElBQUksSUFBSSxJQUFJLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQztZQUN6RCxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzVCLE1BQU0sYUFBYSxHQUNkLE1BQXlCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTlELE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7U0FDbEQ7YUFBTTtZQUNMLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQW9CLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQztRQUN0QyxPQUFPLEVBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRU8sVUFBVSxDQUNkLEtBQWUsRUFBRSxLQUFlLEVBQUUsTUFBc0I7UUFDMUQsTUFBTSxFQUFDLE1BQU0sRUFBQyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUMzRCxPQUFPLE1BQU0sRUFBRSxDQUFDLG9CQUFvQixDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBTSxDQUFDO0lBQ3hFLENBQUM7SUFFRCxZQUFZLENBQUMsS0FBaUI7UUFDNUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxhQUFhLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMsRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0QsQ0FBQztJQUVELFVBQVUsQ0FBQyxLQUFpQjtRQUMxQixNQUFNLE9BQU8sR0FBRyxJQUFJLFdBQVcsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDN0MsTUFBTSwyQkFBMkIsR0FBRyxJQUFJLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUN2QixPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMsRUFBRSxLQUFLLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyx5QkFBeUIsRUFDN0QsMkJBQTJCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRU8sYUFBYSxDQUFDLEtBQWlCLEVBQUUsVUFBb0I7UUFDM0QsTUFBTSxZQUFZLEdBQUc7WUFDbkIsVUFBVSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDO1lBQ25DLEdBQUcsVUFBVSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDO1NBQ1gsQ0FBQztRQUM5QixNQUFNLE9BQU8sR0FBZTtZQUMxQixLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7WUFDbEIsS0FBSyxFQUFFLFlBQVk7WUFDbkIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1NBQ3JCLENBQUM7UUFDRixNQUFNLGNBQWMsR0FBRztZQUNyQixVQUFVLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxFQUFFLEdBQUcsVUFBVSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUM7U0FDOUMsQ0FBQztRQUU5QixNQUFNLE9BQU8sR0FBRyxJQUFJLG9CQUFvQixDQUFDLGNBQWMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUN2RSxNQUFNLDZCQUE2QixHQUFHLElBQUksQ0FBQztRQUMzQyxNQUFNLFlBQVksR0FBRyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3BDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQy9CLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEtBQUssQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUM3Qyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sRUFBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsS0FBSyxFQUFDLENBQUM7SUFDekUsQ0FBQztJQUVPLE1BQU0sQ0FBQyxNQUFjLEVBQUUsY0FBaUM7UUFFOUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDekMsTUFBTSxFQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFDLEdBQUcsT0FBTyxDQUFDO1FBQ3pDLElBQUksY0FBYyxJQUFJLElBQUksRUFBRTtZQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3ZDLE1BQU0sT0FBTyxHQUFHLGNBQWMsQ0FBQyxDQUFDLENBQUMsR0FBRyxjQUFjLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzFELElBQUksQ0FBQyxNQUFNLENBQ1AsSUFBSSxJQUFJLE9BQU8sRUFDZixHQUFHLEVBQUUsQ0FBQywrQkFBK0I7Z0JBQ2pDLHNEQUFzRDtnQkFDdEQsMEJBQTBCLENBQUMsQ0FBQztTQUNyQztRQUNELE1BQU0sU0FBUyxHQUNYLFVBQVUsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUE2QixDQUFDO1FBQy9ELElBQUksT0FBTyxDQUFDO1FBQ1osSUFBSSxRQUFRLEVBQUU7WUFDWixPQUFPLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUNwRDthQUFNO1lBQ0wsT0FBTyxHQUFHLElBQUksbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDOUM7UUFDRCxNQUFNLDZCQUE2QixHQUFHLElBQUksQ0FBQztRQUMzQyxNQUFNLFlBQVksR0FDZCxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDO2dCQUNoQixRQUFRLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUNwRSxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUM1QixPQUFPLEVBQUUsQ0FBQyxFQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFDakUsNkJBQTZCLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDbkQsT0FBTyxFQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLEVBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQsZUFBZSxDQUNYLE9BQXFCLEVBQUUsTUFBb0IsRUFBRSxXQUFxQixFQUNsRSxtQkFBZ0MsRUFBRSw2QkFBNkIsR0FBRyxLQUFLLEVBQ3ZFLGNBQWlDO1FBQ25DLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNyRSxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDaEQsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO1lBQ3hCLE9BQU8sQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDO1NBQ3pCO1FBQ0QsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLEtBQUssUUFBUSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUU7WUFDN0QsTUFBTSxVQUFVLEdBQUcsY0FBYyxJQUFJLElBQUksQ0FBQyxDQUFDO2dCQUN2QyxjQUFjLENBQUMsQ0FBQztnQkFDaEIsUUFBUSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUNuRCwwREFBMEQ7WUFDMUQsb0VBQW9FO1lBQ3BFLHNFQUFzRTtZQUN0RSxhQUFhO1lBQ2IsT0FBTyxDQUFDLFFBQVEsR0FBRyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBcUIsQ0FBQztTQUNuRTtRQUNELElBQUksT0FBTyxDQUFDLFdBQVcsSUFBSSxJQUFJLEVBQUU7WUFDL0IsT0FBTyxDQUFDLEtBQUssR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO1NBQ3JDO1FBRUQsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDMUMsd0VBQXdFO1lBQ3hFLFVBQVU7WUFDVixPQUFPLENBQUMsTUFBTTtnQkFDVixJQUFJLENBQUMsc0JBQXNCLENBQUMsTUFBTSxDQUFDLEtBQWtCLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDOUQsT0FBTyxNQUFNLENBQUM7U0FDZjtRQUVELE1BQU0sYUFBYSxHQUFpQixFQUFFLENBQUM7UUFDdkMsTUFBTSxVQUFVLEdBQWlCLE1BQU0sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDbEQsSUFBSSxLQUFLLENBQUMsS0FBSyxLQUFLLFdBQVcsRUFBRTtnQkFDL0IsTUFBTSxJQUFJLEtBQUssQ0FDWCwrREFBK0Q7b0JBQy9ELDhEQUE4RDtvQkFDOUQsUUFBUSxDQUFDLENBQUM7YUFDZjtZQUVELElBQUksT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUU3QyxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxFQUFFO2dCQUMzQixJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVk7b0JBQ3JCLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQzt3QkFDM0IsR0FBRyxFQUFFLENBQUMsU0FBUyxDQUFDLDJCQUEyQixDQUFDLEVBQUU7b0JBQ3BELGdFQUFnRTtvQkFDaEUsb0VBQW9FO29CQUNwRSxpRUFBaUU7b0JBQ2pFLCtEQUErRDtvQkFDL0QsdURBQXVEO29CQUN2RCxPQUFPO3dCQUNMLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSzt3QkFDbEIsT0FBTyxFQUFFLElBQUk7d0JBQ2IsU0FBUyxFQUFFLElBQUk7d0JBQ2YsYUFBYSxFQUFFLE9BQU8sQ0FBQyxNQUFvQjtxQkFDNUMsQ0FBQztpQkFDSDtnQkFFRCxtRUFBbUU7Z0JBQ25FLHNFQUFzRTtnQkFDdEUsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO29CQUN4QixPQUFPLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQztvQkFDeEIsT0FBTyxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO2lCQUM3QjthQUNGO1lBRUQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDL0IsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLFFBQVEsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRTtnQkFDakQsS0FBSyxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztvQkFDMUIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDbEQsYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDMUIsT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQzthQUMxQztpQkFBTSxJQUNILE9BQU8sQ0FBQyxRQUFRO2dCQUNoQixDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ3pELDZEQUE2RDtnQkFDN0QsdUVBQXVFO2dCQUN2RSxvRUFBb0U7Z0JBQ3BFLHNFQUFzRTtnQkFDdEUsc0VBQXNFO2dCQUN0RSw0REFBNEQ7Z0JBRTVELE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQztnQkFDekIsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztnQkFFaEMsS0FBSyxDQUFDLEtBQUssR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDO2dCQUM1QixLQUFLLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFlLEVBQUUsV0FBVyxDQUFDLENBQUM7Z0JBQ3pELGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQzFCLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBRXpDLFVBQVUsQ0FBQyxLQUFLLEdBQUcsV0FBVyxDQUFDO2FBQ2hDO1lBRUQsT0FBTyxFQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFDLENBQUM7UUFDekQsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNoQyxNQUFNLFVBQVUsR0FDQyxFQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBQyxDQUFDO1FBQzNFLE1BQU0sR0FBRyxHQUFHLFVBQVUsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUN0RSxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRTtZQUM3QyxPQUFPLFVBQVUsQ0FBQyxjQUFjLENBQzVCLElBQUksQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUNuRCxDQUFDLENBQUMsQ0FBQztRQUNILE1BQU0saUJBQWlCLEdBQUcsSUFBSSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUM7UUFDcEQsSUFBSSxLQUErQixDQUFDO1FBQ3BDLElBQUksaUJBQWlCLEVBQUU7WUFDckIsS0FBSyxHQUFHLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztTQUMzQjtRQUVELFVBQVUsQ0FBQyxVQUFVLENBQ2pCLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxVQUFVLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUVyRSxhQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFFeEUsSUFBSSxpQkFBaUIsRUFBRTtZQUNyQixLQUFLLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUM3QixJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FDbEIsRUFBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUMsQ0FBQyxDQUFDO1NBQ3hFO1FBRUQsTUFBTSxnQkFBZ0IsR0FBRyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUM1RCw4QkFBOEI7UUFDOUIsSUFBSSxnQkFBZ0IsR0FBRyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ3hCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLGdCQUFnQixFQUFFO2dCQUNwRCxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDdEIsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUM7YUFDN0I7U0FDRjtRQUVELElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMscUJBQXFCLENBQUMsSUFBSSxPQUFPLENBQUMsUUFBUTtZQUN6RCw2QkFBNkIsS0FBSyxLQUFLLEVBQUU7WUFDM0MsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUMzQyxJQUFJLENBQUMsNkJBQTZCLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDM0MsT0FBTyxRQUFRLENBQUM7U0FDakI7UUFDRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRUQsYUFBYSxDQUNULE9BQXFCLEVBQUUsTUFBb0IsRUFBRSxXQUFzQixFQUNuRSxtQkFBZ0MsRUFDaEMsNkJBQTZCLEdBQUcsS0FBSztRQUN2QyxXQUFXLEdBQUcsV0FBVyxJQUFJLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7UUFDN0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FDaEMsT0FBTyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsbUJBQW1CLEVBQ2pELDZCQUE2QixDQUFDLENBQUM7UUFDbkMsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVPLGdCQUFnQixDQUFDLEdBQVcsRUFBRSxTQUE0QjtRQUVoRSxJQUFJLENBQUMsQ0FBQyxHQUFHLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQzlCLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLEdBQUcsU0FBUyxFQUFFLENBQUM7U0FDckM7UUFDRCxPQUFPLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVELGlCQUFpQjtRQUNmLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQztJQUM3QixDQUFDO0lBSUQsT0FBTztRQUNMLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNqQixPQUFPO1NBQ1I7UUFDRCwwRUFBMEU7UUFDMUUsZ0NBQWdDO1FBQ2hDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDN0IsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDOUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDcEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDN0QsT0FBTyxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQy9CLENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLENBQUMsY0FBYyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQzlCLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxJQUFJO1lBQ25CLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEtBQUssV0FBVztnQkFDMUMsSUFBSSxDQUFDLE1BQU0sWUFBWSxpQkFBaUIsQ0FBQyxFQUFFO1lBQzlDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUM7U0FDdEI7YUFBTTtZQUNMLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDO1NBQ3BCO1FBQ0QsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDNUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDO1lBQzFCLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUM7U0FDdEI7UUFDRCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQztJQUN2QixDQUFDO0lBRUQsY0FBYztRQUNaLElBQUksSUFBSSxDQUFDLG1CQUFtQixJQUFJLElBQUksRUFBRTtZQUNwQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsSUFBSSxDQUFDLEdBQUcsRUFBRTtnQkFDbkMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxFQUFFO29CQUM5QyxpRUFBaUU7b0JBQ2pFLHdDQUF3QztvQkFDeEMsTUFBTSxTQUFTLEdBQUcsR0FBRyxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO29CQUN6QyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO29CQUMxQixNQUFNLG1CQUFtQixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ2pFLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLENBQUM7b0JBRTlCLElBQUksbUJBQW1CLEdBQUcsQ0FBQyxFQUFFO3dCQUMzQixPQUFPLEVBQUUsQ0FBQztxQkFDWDtpQkFDRjtnQkFDRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztJQUNsQyxDQUFDO0lBRUQsa0RBQWtEO0lBQ2xELE9BQU87UUFDTCxPQUFPLElBQUksQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO0lBQzFFLENBQUM7SUFFRCxXQUFXLENBQUMsTUFBYztRQUN4QixNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxNQUFNLEVBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUMsR0FBRyxPQUFPLENBQUM7UUFFakUsSUFBSSxPQUFPLElBQUksSUFBSSxFQUFFO1lBQ25CLGtDQUFrQztZQUNsQyxPQUFPO1NBQ1I7UUFDRCxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDO1FBQ3BELElBQUksS0FBYSxDQUFDO1FBQ2xCLElBQUksaUJBQWlCLEVBQUU7WUFDckIsS0FBSyxHQUFHLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQztTQUNwQjtRQUVELElBQUksUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUM7UUFDaEMsSUFBSSxRQUFRLElBQUksSUFBSSxFQUFFO1lBQ3BCLHdFQUF3RTtZQUN4RSxvRUFBb0U7WUFDcEUsUUFBUSxHQUFHLFVBQVUsQ0FBQywrQkFBK0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDdkUsT0FBTyxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7U0FDN0I7UUFFRCxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDbEIsTUFBTSxTQUFTLEdBQUcsVUFBVSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUVqRCxJQUFJLE9BQU8sQ0FBQztZQUNaLElBQUksS0FBSyxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFBRSxNQUFNLEdBQUcsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlDLE1BQU0sV0FBVyxHQUNiLE1BQU0sWUFBWSxVQUFVLElBQUksTUFBTSxZQUFZLGlCQUFpQixDQUFDO1lBRXhFLHdFQUF3RTtZQUN4RSx5REFBeUQ7WUFDekQsSUFBSSxRQUFRLElBQUksQ0FBQyxXQUFXLEVBQUU7Z0JBQzVCLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxHQUFHLFFBQVEsQ0FBQyxzQ0FBc0MsQ0FDN0QsUUFBUSxDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQy9CO1lBRUQsSUFBSSxRQUFRLEVBQUU7Z0JBQ1osT0FBTyxHQUFHLElBQUkseUJBQXlCLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxDQUFDO2FBQ2pFO2lCQUFNO2dCQUNMLE9BQU8sR0FBRyxJQUFJLG1CQUFtQixDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQzthQUMzRDtZQUVELHNFQUFzRTtZQUN0RSx3RUFBd0U7WUFDeEUsdUNBQXVDO1lBQ3ZDLE1BQU0sc0JBQXNCLEdBQ3hCLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztZQUM3QyxNQUFNLG9CQUFvQixHQUN0QixJQUFJLENBQUMsY0FBYyxDQUFDLHNCQUFzQixFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQ3ZELE1BQU0scUJBQXFCLEdBQ3ZCLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ2xELElBQUksV0FBVyxFQUFFO2dCQUNmLHFCQUFxQixDQUFDLEtBQUssR0FBRyxZQUFZLENBQUMsTUFBTSxDQUFDO2FBQ25EO2lCQUFNO2dCQUNMLHFCQUFxQixDQUFDLEtBQUssR0FBRyxZQUFZLENBQUMsTUFBTSxDQUFDO2FBQ25EO1lBQ0QscUJBQXFCLENBQUMsUUFBUSxHQUFHLHNCQUFzQixDQUFDO1lBQ3hELElBQUksQ0FBQyxLQUFLLENBQUMsMEJBQTBCLENBQ2pDLElBQUksQ0FBQyxVQUFVLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFDM0QsTUFBb0IsQ0FBQyxDQUFDO1lBRTFCLE1BQU0sWUFBWSxHQUFHLENBQUMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUN2QyxpRUFBaUU7WUFDakUsY0FBYztZQUNkLE1BQU0scUJBQXFCLEdBQUcsSUFBSSxDQUFDO1lBQ25DLE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FDNUMsT0FBTyxFQUFFLENBQUMsb0JBQW9CLENBQUMsRUFBRSxLQUFLLEVBQUUsWUFBWSxFQUNwRCxxQkFBcUIsQ0FBQyxDQUFDO1lBRTNCLHVFQUF1RTtZQUN2RSxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNuRSxPQUFPLENBQUMsT0FBTyxHQUFHLGFBQWEsQ0FBQyxPQUFPLENBQUM7WUFDeEMsT0FBTyxDQUFDLFFBQVEsR0FBRyxhQUFhLENBQUMsUUFBUSxDQUFDO1lBQzFDLE9BQU8sQ0FBQyxRQUFRLEdBQUcsYUFBYSxDQUFDLFFBQVEsQ0FBQztZQUMxQyxPQUFPLENBQUMsS0FBSyxHQUFHLGFBQWEsQ0FBQyxLQUFLLENBQUM7WUFFcEMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLG9CQUFvQixDQUFDLENBQUM7WUFDekQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFaEQsZ0RBQWdEO1lBQ2hELE9BQU8sQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDO1lBQ3RCLElBQUksaUJBQWlCLEVBQUU7Z0JBQ3JCLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLEtBQUssQ0FBQzthQUN6QztTQUNGO2FBQU07WUFDTCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3pFLE9BQU8sQ0FBQyxPQUFPLEdBQUcsVUFBVSxDQUFDO1NBQzlCO0lBQ0gsQ0FBQztJQUVPLG9CQUFvQixDQUFDLE1BQWMsRUFBRSxhQUE0QjtRQUV2RSxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxNQUFNLEVBQUMsS0FBSyxFQUFDLEdBQUcsT0FBTyxDQUFDO1FBRXhCLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFNUIsSUFBSSxhQUFhLElBQUksSUFBSSxFQUFFO1lBQ3pCLE9BQU8sQ0FBQyxNQUFNLEdBQUcsbUJBQW1CLENBQUMsYUFBYSxFQUFFLEtBQWtCLENBQUMsQ0FBQztTQUN6RTtRQUNELE9BQU8sT0FBTyxDQUFDLE1BQW9CLENBQUM7SUFDdEMsQ0FBQztJQUVPLGNBQWMsQ0FDbEIsUUFBMEIsRUFBRSxPQUFxQixFQUFFLEtBQWUsRUFDbEUsUUFBaUI7UUFDbkIsSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQjtZQUN2QixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLEdBQUcsSUFBSSxFQUFFO1lBQzlELE1BQU0sRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUM7WUFDOUIsT0FBTyxDQUFDLElBQUksQ0FDUiw2QkFBNkIsRUFBRSxPQUFPO2dCQUN0QyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQ3pDO1FBQ0QsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFTyxZQUFZLENBQUMsS0FBdUIsRUFBRSxLQUFlO1FBQzNELE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNELENBQUM7O0FBampDYywyQkFBVSxHQUFHLENBQUMsQ0FBQztBQW9qQ2hDLFNBQVMsbUJBQW1CLENBQ3hCLENBQWUsRUFBRSxLQUFRO0lBQzNCLElBQUksS0FBSyxLQUFLLFNBQVMsSUFBSSxLQUFLLEtBQUssV0FBVyxFQUFFO1FBQ2hELE9BQU8sQ0FBc0IsQ0FBQztLQUMvQjtTQUFNLElBQUksS0FBSyxLQUFLLE9BQU8sSUFBSSxLQUFLLEtBQUssTUFBTSxFQUFFO1FBQ2hELE1BQU0sTUFBTSxHQUFHLENBQUMsS0FBSyxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUMxQixJQUFJLFVBQVUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDOUQsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLEVBQUU7WUFDdEMsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDOUI7UUFDRCxPQUFPLE1BQTJCLENBQUM7S0FDcEM7U0FBTTtRQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsaUJBQWlCLEtBQUssRUFBRSxDQUFDLENBQUM7S0FDM0M7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IDIwMTcgR29vZ2xlIExMQy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIik7XG4gKiB5b3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gKiBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcbiAqXG4gKiBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiBVbmxlc3MgcmVxdWlyZWQgYnkgYXBwbGljYWJsZSBsYXcgb3IgYWdyZWVkIHRvIGluIHdyaXRpbmcsIHNvZnR3YXJlXG4gKiBkaXN0cmlidXRlZCB1bmRlciB0aGUgTGljZW5zZSBpcyBkaXN0cmlidXRlZCBvbiBhbiBcIkFTIElTXCIgQkFTSVMsXG4gKiBXSVRIT1VUIFdBUlJBTlRJRVMgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZWl0aGVyIGV4cHJlc3Mgb3IgaW1wbGllZC5cbiAqIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9ucyBhbmRcbiAqIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICogPT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT1cbiAqL1xuXG4vLyBJbXBvcnQgd2ViZ2wgZmxhZ3MuXG5pbXBvcnQgJy4vZmxhZ3Nfd2ViZ2wnO1xuXG5pbXBvcnQgKiBhcyB0ZiBmcm9tICdAdGVuc29yZmxvdy90ZmpzLWNvcmUnO1xuaW1wb3J0IHtiYWNrZW5kX3V0aWwsIEJhY2tlbmRWYWx1ZXMsIGJ1ZmZlciwgRGF0YUlkLCBEYXRhU3RvcmFnZSwgRGF0YVRvR1BVV2ViR0xPcHRpb24sIERhdGFUeXBlLCBEYXRhVmFsdWVzLCBlbmdpbmUsIGVudiwgR1BVRGF0YSwga2VybmVsX2ltcGxzLCBLZXJuZWxCYWNrZW5kLCBNZW1vcnlJbmZvLCBOdW1lcmljRGF0YVR5cGUsIFJhbmssIFJlY3Vyc2l2ZUFycmF5LCBzY2FsYXIsIFNoYXBlTWFwLCBUZW5zb3IsIFRlbnNvcjJELCBUZW5zb3JCdWZmZXIsIFRlbnNvckluZm8sIHRpZHksIFRpbWluZ0luZm8sIFR5cGVkQXJyYXksIHV0aWx9IGZyb20gJ0B0ZW5zb3JmbG93L3RmanMtY29yZSc7XG5cbmltcG9ydCB7Z2V0V2ViR0xDb250ZXh0fSBmcm9tICcuL2NhbnZhc191dGlsJztcbmltcG9ydCB7RGVjb2RlTWF0cml4UHJvZ3JhbX0gZnJvbSAnLi9kZWNvZGVfbWF0cml4X2dwdSc7XG5pbXBvcnQge0RlY29kZU1hdHJpeFBhY2tlZFByb2dyYW19IGZyb20gJy4vZGVjb2RlX21hdHJpeF9wYWNrZWRfZ3B1JztcbmltcG9ydCB7RW5jb2RlRmxvYXRQcm9ncmFtfSBmcm9tICcuL2VuY29kZV9mbG9hdF9ncHUnO1xuaW1wb3J0IHtFbmNvZGVGbG9hdFBhY2tlZFByb2dyYW19IGZyb20gJy4vZW5jb2RlX2Zsb2F0X3BhY2tlZF9ncHUnO1xuaW1wb3J0IHtFbmNvZGVNYXRyaXhQcm9ncmFtfSBmcm9tICcuL2VuY29kZV9tYXRyaXhfZ3B1JztcbmltcG9ydCB7RW5jb2RlTWF0cml4UGFja2VkUHJvZ3JhbX0gZnJvbSAnLi9lbmNvZGVfbWF0cml4X3BhY2tlZF9ncHUnO1xuaW1wb3J0IHtHUEdQVUNvbnRleHR9IGZyb20gJy4vZ3BncHVfY29udGV4dCc7XG5pbXBvcnQgKiBhcyBncGdwdV9tYXRoIGZyb20gJy4vZ3BncHVfbWF0aCc7XG5pbXBvcnQge0dQR1BVQmluYXJ5LCBHUEdQVVByb2dyYW0sIFRlbnNvckRhdGF9IGZyb20gJy4vZ3BncHVfbWF0aCc7XG5pbXBvcnQge3NpbXBsZUFic0ltcGxDUFV9IGZyb20gJy4va2VybmVsX3V0aWxzL3NoYXJlZCc7XG5pbXBvcnQge1BhY2tQcm9ncmFtfSBmcm9tICcuL3BhY2tfZ3B1JztcbmltcG9ydCB7UmVzaGFwZVBhY2tlZFByb2dyYW19IGZyb20gJy4vcmVzaGFwZV9wYWNrZWRfZ3B1JztcbmltcG9ydCAqIGFzIHRleF91dGlsIGZyb20gJy4vdGV4X3V0aWwnO1xuaW1wb3J0IHtUZXh0dXJlLCBUZXh0dXJlRGF0YSwgVGV4dHVyZVVzYWdlfSBmcm9tICcuL3RleF91dGlsJztcbmltcG9ydCB7VGV4dHVyZU1hbmFnZXJ9IGZyb20gJy4vdGV4dHVyZV9tYW5hZ2VyJztcbmltcG9ydCAqIGFzIHVuYXJ5X29wIGZyb20gJy4vdW5hcnlvcF9ncHUnO1xuaW1wb3J0IHtVbmFyeU9wUHJvZ3JhbX0gZnJvbSAnLi91bmFyeW9wX2dwdSc7XG5pbXBvcnQge1VuYXJ5T3BQYWNrZWRQcm9ncmFtfSBmcm9tICcuL3VuYXJ5b3BfcGFja2VkX2dwdSc7XG5pbXBvcnQge1VucGFja1Byb2dyYW19IGZyb20gJy4vdW5wYWNrX2dwdSc7XG5pbXBvcnQgKiBhcyB3ZWJnbF91dGlsIGZyb20gJy4vd2ViZ2xfdXRpbCc7XG5cbmNvbnN0IHdoZXJlSW1wbCA9IGtlcm5lbF9pbXBscy53aGVyZUltcGw7XG5cbmV4cG9ydCBjb25zdCBFUFNJTE9OX0ZMT0FUMzIgPSAxZS03O1xuZXhwb3J0IGNvbnN0IEVQU0lMT05fRkxPQVQxNiA9IDFlLTQ7XG5cbnR5cGUgS2VybmVsSW5mbyA9IHtcbiAgbmFtZTogc3RyaW5nOyBxdWVyeTogUHJvbWlzZTxudW1iZXI+O1xufTtcblxuZXhwb3J0IHR5cGUgVGltZXJOb2RlID0gUmVjdXJzaXZlQXJyYXk8S2VybmVsSW5mbz58S2VybmVsSW5mbztcbmV4cG9ydCBpbnRlcmZhY2UgQ1BVVGltZXJRdWVyeSB7XG4gIHN0YXJ0TXM6IG51bWJlcjtcbiAgZW5kTXM/OiBudW1iZXI7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgV2ViR0xNZW1vcnlJbmZvIGV4dGVuZHMgTWVtb3J5SW5mbyB7XG4gIG51bUJ5dGVzSW5HUFU6IG51bWJlcjtcbiAgLy8gVHJhY2tzIHRoZSB0b3RhbCBudW1iZXIgb2YgYnl0ZXMgYWxsb2NhdGVkIG9uIHRoZSBHUFUsIGFjY291bnRpbmcgZm9yIHRoZVxuICAvLyBwaHlzaWNhbCB0ZXh0dXJlIHR5cGUuXG4gIG51bUJ5dGVzSW5HUFVBbGxvY2F0ZWQ6IG51bWJlcjtcbiAgLy8gVHJhY2tzIGJ5dGUgc2l6ZSBvZiB0ZXh0dXJlcyB0aGF0IHdlcmUgY3JlYXRlZCBhbmQgdGhlbiBtYWRlIGF2YWlsYWJsZSBmb3JcbiAgLy8gcmV1c2UgKGRpc3Bvc2VkKS5cbiAgbnVtQnl0ZXNJbkdQVUZyZWU6IG51bWJlcjtcbiAgdW5yZWxpYWJsZTogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBXZWJHTFRpbWluZ0luZm8gZXh0ZW5kcyBUaW1pbmdJbmZvIHtcbiAgdXBsb2FkV2FpdE1zOiBudW1iZXI7XG4gIGRvd25sb2FkV2FpdE1zOiBudW1iZXI7XG59XG5cbmNvbnN0IGJpbmFyeUNhY2hlczoge1t3ZWJHTFZlcnNpb246IHN0cmluZ106IHtba2V5OiBzdHJpbmddOiBHUEdQVUJpbmFyeX19ID0ge307XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRCaW5hcnlDYWNoZSh3ZWJHTFZlcnNpb246IG51bWJlcikge1xuICBpZiAod2ViR0xWZXJzaW9uIGluIGJpbmFyeUNhY2hlcykge1xuICAgIHJldHVybiBiaW5hcnlDYWNoZXNbd2ViR0xWZXJzaW9uXTtcbiAgfVxuICBiaW5hcnlDYWNoZXNbd2ViR0xWZXJzaW9uXSA9IHt9O1xuICByZXR1cm4gYmluYXJ5Q2FjaGVzW3dlYkdMVmVyc2lvbl07XG59XG5cbi8vIEVtcGlyaWNhbGx5IGRldGVybWluZWQgY29uc3RhbnQgdXNlZCB0byBkZXRlcm1pbmUgc2l6ZSB0aHJlc2hvbGQgZm9yIGhhbmRpbmdcbi8vIG9mZiBleGVjdXRpb24gdG8gdGhlIENQVS5cbmNvbnN0IENQVV9IQU5ET0ZGX1NJWkVfVEhSRVNIT0xEID1cbiAgICBlbnYoKS5nZXROdW1iZXIoJ0NQVV9IQU5ET0ZGX1NJWkVfVEhSRVNIT0xEJyk7XG5cbi8vIEVtcGlyaWNhbGx5IGRldGVybWluZWQgY29uc3RhbnQgdXNlZCB0byBkZWNpZGUgdGhlIG51bWJlciBvZiBNQiBvbiBHUFVcbi8vIGJlZm9yZSB3ZSB3YXJuIGFib3V0IGhpZ2ggbWVtb3J5IHVzZS4gVGhlIE1CIGFyZSB0aGlzIGNvbnN0YW50ICogc2NyZWVuIGFyZWFcbi8vICogZHBpIC8gMTAyNCAvIDEwMjQuXG5jb25zdCBCRUZPUkVfUEFHSU5HX0NPTlNUQU5UID0gNjAwO1xuZnVuY3Rpb24gbnVtTUJCZWZvcmVXYXJuaW5nKCk6IG51bWJlciB7XG4gIGlmIChlbnYoKS5nbG9iYWwuc2NyZWVuID09IG51bGwpIHtcbiAgICByZXR1cm4gMTAyNDsgIC8vIDEgR0IuXG4gIH1cbiAgcmV0dXJuIChlbnYoKS5nbG9iYWwuc2NyZWVuLmhlaWdodCAqIGVudigpLmdsb2JhbC5zY3JlZW4ud2lkdGggKlxuICAgICAgICAgIHdpbmRvdy5kZXZpY2VQaXhlbFJhdGlvKSAqXG4gICAgICBCRUZPUkVfUEFHSU5HX0NPTlNUQU5UIC8gMTAyNCAvIDEwMjQ7XG59XG5cbmV4cG9ydCBjbGFzcyBNYXRoQmFja2VuZFdlYkdMIGV4dGVuZHMgS2VybmVsQmFja2VuZCB7XG4gIHRleERhdGE6IERhdGFTdG9yYWdlPFRleHR1cmVEYXRhPjtcbiAgZ3BncHU6IEdQR1BVQ29udGV4dDtcblxuICBwcml2YXRlIHN0YXRpYyBuZXh0RGF0YUlkID0gMDtcbiAgcHJpdmF0ZSBuZXh0RGF0YUlkKCk6IG51bWJlciB7XG4gICAgcmV0dXJuIE1hdGhCYWNrZW5kV2ViR0wubmV4dERhdGFJZCsrO1xuICB9XG4gIC8vIE1hcHMgZGF0YSBpZHMgdGhhdCBoYXZlIGEgcGVuZGluZyByZWFkIG9wZXJhdGlvbiwgdG8gbGlzdCBvZiBzdWJzY3JpYmVycy5cbiAgcHJpdmF0ZSBwZW5kaW5nUmVhZCA9IG5ldyBXZWFrTWFwPERhdGFJZCwgQXJyYXk8KGFycjogVHlwZWRBcnJheSkgPT4gdm9pZD4+KCk7XG4gIC8vIExpc3Qgb2YgZGF0YSBpZHMgdGhhdCBhcmUgc2NoZWR1bGVkIGZvciBkaXNwb3NhbCwgYnV0IGFyZSB3YWl0aW5nIG9uIGFcbiAgLy8gcGVuZGluZyByZWFkIG9wZXJhdGlvbi5cbiAgcHJpdmF0ZSBwZW5kaW5nRGlzcG9zYWwgPSBuZXcgV2Vha1NldDxEYXRhSWQ+KCk7XG5cbiAgLy8gVXNlZCB0byBjb3VudCB0aGUgbnVtYmVyIG9mICdzaGFsbG93JyBzbGljZWQgdGVuc29ycyB0aGF0IHBvaW50IHRvIHRoZVxuICAvLyBzYW1lIGRhdGEgaWQuXG4gIGRhdGFSZWZDb3VudCA9IG5ldyBXZWFrTWFwPERhdGFJZCwgbnVtYmVyPigpO1xuICBwcml2YXRlIG51bUJ5dGVzSW5HUFUgPSAwO1xuXG4gIHByaXZhdGUgY2FudmFzOiBIVE1MQ2FudmFzRWxlbWVudHxPZmZzY3JlZW5DYW52YXM7XG5cbiAgcHJpdmF0ZSBwcm9ncmFtVGltZXJzU3RhY2s6IFRpbWVyTm9kZVtdO1xuICBwcml2YXRlIGFjdGl2ZVRpbWVyczogVGltZXJOb2RlW107XG4gIC8vIEFjY3VtdWxhdGVkIHRpbWUgc3BlbnQgKGluY2x1ZGluZyBibG9ja2luZykgaW4gdXBsb2FkaW5nIGRhdGEgdG8gd2ViZ2wuXG4gIHByaXZhdGUgdXBsb2FkV2FpdE1zID0gMDtcbiAgLy8gQWNjdW11bGF0ZWQgdGltZSBzcGVudCAoaW5jbHVkaW5nIGJsb2NraW5nIGluIGRvd25sb2FkaW5nIGRhdGEgZnJvbSB3ZWJnbC5cbiAgcHJpdmF0ZSBkb3dubG9hZFdhaXRNcyA9IDA7XG5cbiAgLy8gcmVjb3JkIHRoZSBsYXN0IG1hbnVhbCBHTCBGbHVzaCB0aW1lLlxuICBwcml2YXRlIGxhc3RHbEZsdXNoVGltZSA9IDA7XG5cbiAgLy8gTnVtYmVyIG9mIGJpdHMgb2YgcHJlY2lzaW9uIG9mIHRoaXMgYmFja2VuZC5cbiAgcHJpdmF0ZSBmbG9hdFByZWNpc2lvblZhbHVlOiAzMnwxNjtcblxuICBwcml2YXRlIHRleHR1cmVNYW5hZ2VyOiBUZXh0dXJlTWFuYWdlcjtcbiAgcHJpdmF0ZSBiaW5hcnlDYWNoZToge1trZXk6IHN0cmluZ106IEdQR1BVQmluYXJ5fTtcbiAgcHJpdmF0ZSBncGdwdUNyZWF0ZWRMb2NhbGx5OiBib29sZWFuO1xuICBwcml2YXRlIG51bU1CQmVmb3JlV2FybmluZzogbnVtYmVyO1xuICBwcml2YXRlIHdhcm5lZEFib3V0TWVtb3J5ID0gZmFsc2U7XG5cbiAgY29uc3RydWN0b3IoZ3B1UmVzb3VyY2U/OiBHUEdQVUNvbnRleHR8SFRNTENhbnZhc0VsZW1lbnR8T2Zmc2NyZWVuQ2FudmFzKSB7XG4gICAgc3VwZXIoKTtcbiAgICBpZiAoIWVudigpLmdldEJvb2woJ0hBU19XRUJHTCcpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1dlYkdMIGlzIG5vdCBzdXBwb3J0ZWQgb24gdGhpcyBkZXZpY2UnKTtcbiAgICB9XG5cbiAgICBsZXQgbmV3R1BHUFU7XG4gICAgaWYgKGdwdVJlc291cmNlICE9IG51bGwpIHtcbiAgICAgIGlmIChncHVSZXNvdXJjZSBpbnN0YW5jZW9mIEdQR1BVQ29udGV4dCkge1xuICAgICAgICBuZXdHUEdQVSA9IGdwdVJlc291cmNlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgY29uc3QgZ2wgPVxuICAgICAgICAgICAgZ2V0V2ViR0xDb250ZXh0KGVudigpLmdldE51bWJlcignV0VCR0xfVkVSU0lPTicpLCBncHVSZXNvdXJjZSk7XG4gICAgICAgIG5ld0dQR1BVID0gbmV3IEdQR1BVQ29udGV4dChnbCk7XG4gICAgICB9XG4gICAgICB0aGlzLmJpbmFyeUNhY2hlID0ge307XG4gICAgICB0aGlzLmdwZ3B1Q3JlYXRlZExvY2FsbHkgPSBmYWxzZTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgZ2wgPSBnZXRXZWJHTENvbnRleHQoZW52KCkuZ2V0TnVtYmVyKCdXRUJHTF9WRVJTSU9OJykpO1xuICAgICAgbmV3R1BHUFUgPSBuZXcgR1BHUFVDb250ZXh0KGdsKTtcbiAgICAgIHRoaXMuYmluYXJ5Q2FjaGUgPSBnZXRCaW5hcnlDYWNoZShlbnYoKS5nZXROdW1iZXIoJ1dFQkdMX1ZFUlNJT04nKSk7XG4gICAgICB0aGlzLmdwZ3B1Q3JlYXRlZExvY2FsbHkgPSB0cnVlO1xuICAgIH1cblxuICAgIHRoaXMuZ3BncHUgPSBuZXdHUEdQVTtcbiAgICB0aGlzLmNhbnZhcyA9IHRoaXMuZ3BncHUuZ2wuY2FudmFzO1xuICAgIHRoaXMudGV4dHVyZU1hbmFnZXIgPSBuZXcgVGV4dHVyZU1hbmFnZXIodGhpcy5ncGdwdSk7XG4gICAgdGhpcy5udW1NQkJlZm9yZVdhcm5pbmcgPSBudW1NQkJlZm9yZVdhcm5pbmcoKTtcbiAgICB0aGlzLnRleERhdGEgPSBuZXcgRGF0YVN0b3JhZ2UodGhpcywgZW5naW5lKCkpO1xuICB9XG5cbiAgbnVtRGF0YUlkcygpIHtcbiAgICByZXR1cm4gdGhpcy50ZXhEYXRhLm51bURhdGFJZHMoKSAtIHRoaXMucGVuZGluZ0RlbGV0ZXM7XG4gIH1cblxuICB3cml0ZSh2YWx1ZXM6IEJhY2tlbmRWYWx1ZXMsIHNoYXBlOiBudW1iZXJbXSwgZHR5cGU6IERhdGFUeXBlKTogRGF0YUlkIHtcbiAgICBpZiAoZW52KCkuZ2V0Qm9vbCgnV0VCR0xfQ0hFQ0tfTlVNRVJJQ0FMX1BST0JMRU1TJykgfHxcbiAgICAgICAgZW52KCkuZ2V0Qm9vbCgnREVCVUcnKSkge1xuICAgICAgdGhpcy5jaGVja051bWVyaWNhbFByb2JsZW1zKHZhbHVlcyk7XG4gICAgfVxuICAgIGlmIChkdHlwZSA9PT0gJ2NvbXBsZXg2NCcgJiYgdmFsdWVzICE9IG51bGwpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgQ2Fubm90IHdyaXRlIHRvIGEgY29tcGxleDY0IGR0eXBlLiBgICtcbiAgICAgICAgICBgUGxlYXNlIHVzZSB0Zi5jb21wbGV4KHJlYWwsIGltYWcpLmApO1xuICAgIH1cbiAgICBjb25zdCBkYXRhSWQgPSB7aWQ6IHRoaXMubmV4dERhdGFJZCgpfTtcbiAgICB0aGlzLnRleERhdGEuc2V0KFxuICAgICAgICBkYXRhSWQsXG4gICAgICAgIHtzaGFwZSwgZHR5cGUsIHZhbHVlcywgdXNhZ2U6IFRleHR1cmVVc2FnZS5VUExPQUQsIHJlZkNvdW50OiAxfSk7XG4gICAgcmV0dXJuIGRhdGFJZDtcbiAgfVxuXG4gIC8qKiBSZXR1cm4gcmVmQ291bnQgb2YgYSBgVGVuc29yRGF0YWAuICovXG4gIHJlZkNvdW50KGRhdGFJZDogRGF0YUlkKTogbnVtYmVyIHtcbiAgICBpZiAodGhpcy50ZXhEYXRhLmhhcyhkYXRhSWQpKSB7XG4gICAgICBjb25zdCB0ZW5zb3JEYXRhID0gdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpO1xuICAgICAgcmV0dXJuIHRlbnNvckRhdGEucmVmQ291bnQ7XG4gICAgfVxuICAgIHJldHVybiAwO1xuICB9XG5cbiAgLyoqIEluY3JlYXNlIHJlZkNvdW50IG9mIGEgYFRleHR1cmVEYXRhYC4gKi9cbiAgaW5jUmVmKGRhdGFJZDogRGF0YUlkKTogdm9pZCB7XG4gICAgY29uc3QgdGV4RGF0YSA9IHRoaXMudGV4RGF0YS5nZXQoZGF0YUlkKTtcbiAgICB0ZXhEYXRhLnJlZkNvdW50Kys7XG4gIH1cblxuICAvKiogRGVjcmVhc2UgcmVmQ291bnQgb2YgYSBgVGV4dHVyZURhdGFgLiAqL1xuICBkZWNSZWYoZGF0YUlkOiBEYXRhSWQpOiB2b2lkIHtcbiAgICBpZiAodGhpcy50ZXhEYXRhLmhhcyhkYXRhSWQpKSB7XG4gICAgICBjb25zdCB0ZXhEYXRhID0gdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpO1xuICAgICAgdGV4RGF0YS5yZWZDb3VudC0tO1xuICAgIH1cbiAgfVxuXG4gIG1vdmUoXG4gICAgICBkYXRhSWQ6IERhdGFJZCwgdmFsdWVzOiBCYWNrZW5kVmFsdWVzLCBzaGFwZTogbnVtYmVyW10sIGR0eXBlOiBEYXRhVHlwZSxcbiAgICAgIHJlZkNvdW50OiBudW1iZXIpOiB2b2lkIHtcbiAgICBpZiAoZW52KCkuZ2V0Qm9vbCgnREVCVUcnKSkge1xuICAgICAgdGhpcy5jaGVja051bWVyaWNhbFByb2JsZW1zKHZhbHVlcyk7XG4gICAgfVxuICAgIGlmIChkdHlwZSA9PT0gJ2NvbXBsZXg2NCcpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgQ2Fubm90IHdyaXRlIHRvIGEgY29tcGxleDY0IGR0eXBlLiBgICtcbiAgICAgICAgICBgUGxlYXNlIHVzZSB0Zi5jb21wbGV4KHJlYWwsIGltYWcpLmApO1xuICAgIH1cbiAgICB0aGlzLnRleERhdGEuc2V0KFxuICAgICAgICBkYXRhSWQsIHtzaGFwZSwgZHR5cGUsIHZhbHVlcywgdXNhZ2U6IFRleHR1cmVVc2FnZS5VUExPQUQsIHJlZkNvdW50fSk7XG4gIH1cblxuICBkaXNwb3NlSW50ZXJtZWRpYXRlVGVuc29ySW5mbyh0ZW5zb3JJbmZvOiBUZW5zb3JJbmZvKTogdm9pZCB7XG4gICAgdGhpcy5kaXNwb3NlRGF0YSh0ZW5zb3JJbmZvLmRhdGFJZCk7XG4gIH1cblxuICByZWFkU3luYyhkYXRhSWQ6IERhdGFJZCk6IEJhY2tlbmRWYWx1ZXMge1xuICAgIGNvbnN0IHRleERhdGEgPSB0aGlzLnRleERhdGEuZ2V0KGRhdGFJZCk7XG4gICAgY29uc3Qge3ZhbHVlcywgZHR5cGUsIGNvbXBsZXhUZW5zb3JJbmZvcywgc2xpY2UsIHNoYXBlLCBpc1BhY2tlZH0gPSB0ZXhEYXRhO1xuXG4gICAgLy8gVGhlIHByZXNlbmNlIG9mIGBzbGljZWAgaW5kaWNhdGVzIHRoaXMgdGVuc29yIGlzIGEgc2hhbGxvdyBzbGljZSBvZiBhXG4gICAgLy8gZGlmZmVyZW50IHRlbnNvciwgYW5kIGlzIHVzaW5nIHRoYXQgb3JpZ2luYWwgdGVuc29yJ3MgdGV4dHVyZS4gUnVuXG4gICAgLy8gYGNsb25lYCBpbiBvcmRlciB0byBjb3B5IHRoYXQgdGV4dHVyZSBhbmQgcmVhZCBmcm9tIGl0LlxuICAgIGlmIChzbGljZSAhPSBudWxsKSB7XG4gICAgICBsZXQgcHJvZ3JhbTtcbiAgICAgIGlmIChpc1BhY2tlZCkge1xuICAgICAgICBwcm9ncmFtID0gbmV3IFVuYXJ5T3BQYWNrZWRQcm9ncmFtKHNoYXBlLCB1bmFyeV9vcC5DTE9ORSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBwcm9ncmFtID0gbmV3IFVuYXJ5T3BQcm9ncmFtKHNoYXBlLCB1bmFyeV9vcC5DTE9ORSk7XG4gICAgICB9XG4gICAgICBjb25zdCByZXMgPVxuICAgICAgICAgIHRoaXMucnVuV2ViR0xQcm9ncmFtKHByb2dyYW0sIFt7ZGF0YUlkLCBzaGFwZSwgZHR5cGV9XSwgZHR5cGUpO1xuICAgICAgY29uc3QgZGF0YSA9IHRoaXMucmVhZFN5bmMocmVzLmRhdGFJZCk7XG4gICAgICB0aGlzLmRpc3Bvc2VJbnRlcm1lZGlhdGVUZW5zb3JJbmZvKHJlcyk7XG4gICAgICByZXR1cm4gZGF0YTtcbiAgICB9XG4gICAgaWYgKHZhbHVlcyAhPSBudWxsKSB7XG4gICAgICByZXR1cm4gdGhpcy5jb252ZXJ0QW5kQ2FjaGVPbkNQVShkYXRhSWQpO1xuICAgIH1cbiAgICBpZiAoZHR5cGUgPT09ICdzdHJpbmcnKSB7XG4gICAgICByZXR1cm4gdmFsdWVzO1xuICAgIH1cbiAgICBjb25zdCBzaG91bGRUaW1lUHJvZ3JhbSA9IHRoaXMuYWN0aXZlVGltZXJzICE9IG51bGw7XG4gICAgbGV0IHN0YXJ0OiBudW1iZXI7XG4gICAgaWYgKHNob3VsZFRpbWVQcm9ncmFtKSB7XG4gICAgICBzdGFydCA9IHV0aWwubm93KCk7XG4gICAgfVxuXG4gICAgbGV0IHJlc3VsdDogRmxvYXQzMkFycmF5O1xuICAgIGlmIChkdHlwZSA9PT0gJ2NvbXBsZXg2NCcpIHtcbiAgICAgIGNvbnN0IHJlYWxWYWx1ZXMgPVxuICAgICAgICAgIHRoaXMucmVhZFN5bmMoY29tcGxleFRlbnNvckluZm9zLnJlYWwuZGF0YUlkKSBhcyBGbG9hdDMyQXJyYXk7XG4gICAgICBjb25zdCBpbWFnVmFsdWVzID1cbiAgICAgICAgICB0aGlzLnJlYWRTeW5jKGNvbXBsZXhUZW5zb3JJbmZvcy5pbWFnLmRhdGFJZCkgYXMgRmxvYXQzMkFycmF5O1xuICAgICAgcmVzdWx0ID0gYmFja2VuZF91dGlsLm1lcmdlUmVhbEFuZEltYWdBcnJheXMocmVhbFZhbHVlcywgaW1hZ1ZhbHVlcyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJlc3VsdCA9IHRoaXMuZ2V0VmFsdWVzRnJvbVRleHR1cmUoZGF0YUlkKTtcbiAgICB9XG5cbiAgICBpZiAoc2hvdWxkVGltZVByb2dyYW0pIHtcbiAgICAgIHRoaXMuZG93bmxvYWRXYWl0TXMgKz0gdXRpbC5ub3coKSAtIHN0YXJ0O1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5jb252ZXJ0QW5kQ2FjaGVPbkNQVShkYXRhSWQsIHJlc3VsdCk7XG4gIH1cblxuICBhc3luYyByZWFkKGRhdGFJZDogRGF0YUlkKTogUHJvbWlzZTxCYWNrZW5kVmFsdWVzPiB7XG4gICAgaWYgKHRoaXMucGVuZGluZ1JlYWQuaGFzKGRhdGFJZCkpIHtcbiAgICAgIGNvbnN0IHN1YnNjcmliZXJzID0gdGhpcy5wZW5kaW5nUmVhZC5nZXQoZGF0YUlkKTtcbiAgICAgIHJldHVybiBuZXcgUHJvbWlzZTxUeXBlZEFycmF5PihyZXNvbHZlID0+IHN1YnNjcmliZXJzLnB1c2gocmVzb2x2ZSkpO1xuICAgIH1cbiAgICBjb25zdCB0ZXhEYXRhID0gdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpO1xuICAgIGNvbnN0IHt2YWx1ZXMsIHNoYXBlLCBzbGljZSwgZHR5cGUsIGNvbXBsZXhUZW5zb3JJbmZvcywgaXNQYWNrZWR9ID0gdGV4RGF0YTtcblxuICAgIC8vIFRoZSBwcmVzZW5jZSBvZiBgc2xpY2VgIGluZGljYXRlcyB0aGlzIHRlbnNvciBpcyBhIHNoYWxsb3cgc2xpY2Ugb2YgYVxuICAgIC8vIGRpZmZlcmVudCB0ZW5zb3IsIGFuZCBpcyB1c2luZyB0aGF0IG9yaWdpbmFsIHRlbnNvcidzIHRleHR1cmUuIFJ1blxuICAgIC8vIGBjbG9uZWAgaW4gb3JkZXIgdG8gY29weSB0aGF0IHRleHR1cmUgYW5kIHJlYWQgZnJvbSBpdC5cbiAgICBpZiAoc2xpY2UgIT0gbnVsbCkge1xuICAgICAgbGV0IHByb2dyYW07XG4gICAgICBpZiAoaXNQYWNrZWQpIHtcbiAgICAgICAgcHJvZ3JhbSA9IG5ldyBVbmFyeU9wUGFja2VkUHJvZ3JhbShzaGFwZSwgdW5hcnlfb3AuQ0xPTkUpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcHJvZ3JhbSA9IG5ldyBVbmFyeU9wUHJvZ3JhbShzaGFwZSwgdW5hcnlfb3AuQ0xPTkUpO1xuICAgICAgfVxuICAgICAgY29uc3QgcmVzID1cbiAgICAgICAgICB0aGlzLnJ1bldlYkdMUHJvZ3JhbShwcm9ncmFtLCBbe2RhdGFJZCwgc2hhcGUsIGR0eXBlfV0sIGR0eXBlKTtcbiAgICAgIGNvbnN0IGRhdGEgPSB0aGlzLnJlYWQocmVzLmRhdGFJZCk7XG4gICAgICB0aGlzLmRpc3Bvc2VJbnRlcm1lZGlhdGVUZW5zb3JJbmZvKHJlcyk7XG4gICAgICByZXR1cm4gZGF0YTtcbiAgICB9XG5cbiAgICBpZiAodmFsdWVzICE9IG51bGwpIHtcbiAgICAgIHJldHVybiB0aGlzLmNvbnZlcnRBbmRDYWNoZU9uQ1BVKGRhdGFJZCk7XG4gICAgfVxuXG4gICAgaWYgKGVudigpLmdldEJvb2woJ0RFQlVHJykpIHtcbiAgICAgIC8vIGdldEJvb2woJ1dFQkdMX0RPV05MT0FEX0ZMT0FUX0VOQUJMRUQnKSBjYXVzZWQgYSBibG9ja2luZyBHUFUgY2FsbC5cbiAgICAgIC8vIEZvciBwZXJmb3JtYW5jZSByZWFzb24sIG9ubHkgY2hlY2sgaXQgZm9yIGRlYnVnZ2luZy4gSW4gcHJvZHVjdGlvbixcbiAgICAgIC8vIGl0IGRvZXNuJ3QgaGFuZGxlIHRoaXMgdXNlIGNhc2UgYW55d2F5LCBzbyBiZWhhdmlvciBpcyBub3QgY2hhbmdlZC5cbiAgICAgIGlmICghZW52KCkuZ2V0Qm9vbCgnV0VCR0xfRE9XTkxPQURfRkxPQVRfRU5BQkxFRCcpICYmXG4gICAgICAgICAgZW52KCkuZ2V0TnVtYmVyKCdXRUJHTF9WRVJTSU9OJykgPT09IDIpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgYHRlbnNvci5kYXRhKCkgd2l0aCBXRUJHTF9ET1dOTE9BRF9GTE9BVF9FTkFCTEVEPWZhbHNlIGFuZCBgICtcbiAgICAgICAgICAgIGBXRUJHTF9WRVJTSU9OPTIgbm90IHlldCBzdXBwb3J0ZWQuYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgbGV0IGJ1ZmZlcjogV2ViR0xCdWZmZXIgPSBudWxsO1xuICAgIGxldCB0bXBEb3dubG9hZFRhcmdldDogVGVuc29ySW5mbztcblxuICAgIGlmIChkdHlwZSAhPT0gJ2NvbXBsZXg2NCcgJiYgZW52KCkuZ2V0KCdXRUJHTF9CVUZGRVJfU1VQUE9SVEVEJykpIHtcbiAgICAgIC8vIFBvc3NpYmx5IGNvcHkgdGhlIHRleHR1cmUgaW50byBhIGJ1ZmZlciBiZWZvcmUgaW5zZXJ0aW5nIGEgZmVuY2UuXG4gICAgICB0bXBEb3dubG9hZFRhcmdldCA9IHRoaXMuZGVjb2RlKGRhdGFJZCk7XG4gICAgICBjb25zdCB0bXBEYXRhID0gdGhpcy50ZXhEYXRhLmdldCh0bXBEb3dubG9hZFRhcmdldC5kYXRhSWQpO1xuXG4gICAgICBidWZmZXIgPSB0aGlzLmdwZ3B1LmNyZWF0ZUJ1ZmZlckZyb21UZXh0dXJlKFxuICAgICAgICAgIHRtcERhdGEudGV4dHVyZS50ZXh0dXJlLCAuLi50ZXhfdXRpbC5nZXREZW5zZVRleFNoYXBlKHNoYXBlKSk7XG4gICAgfVxuXG4gICAgdGhpcy5wZW5kaW5nUmVhZC5zZXQoZGF0YUlkLCBbXSk7XG5cbiAgICBpZiAoZHR5cGUgIT09ICdjb21wbGV4NjQnKSB7XG4gICAgICAvLyBDcmVhdGUgYSBmZW5jZSBhbmQgd2FpdCBmb3IgaXQgdG8gcmVzb2x2ZS5cbiAgICAgIGF3YWl0IHRoaXMuZ3BncHUuY3JlYXRlQW5kV2FpdEZvckZlbmNlKCk7XG4gICAgfVxuXG4gICAgLy8gRG93bmxvYWQgdGhlIHZhbHVlcyBmcm9tIHRoZSBHUFUuXG4gICAgbGV0IHZhbHM6IEZsb2F0MzJBcnJheTtcbiAgICBpZiAoZHR5cGUgPT09ICdjb21wbGV4NjQnKSB7XG4gICAgICBjb25zdCBwcyA9IGF3YWl0IFByb21pc2UuYWxsKFtcbiAgICAgICAgdGhpcy5yZWFkKGNvbXBsZXhUZW5zb3JJbmZvcy5yZWFsLmRhdGFJZCksXG4gICAgICAgIHRoaXMucmVhZChjb21wbGV4VGVuc29ySW5mb3MuaW1hZy5kYXRhSWQpXG4gICAgICBdKTtcblxuICAgICAgY29uc3QgcmVhbFZhbHVlcyA9IHBzWzBdO1xuICAgICAgY29uc3QgaW1hZ1ZhbHVlcyA9IHBzWzFdO1xuICAgICAgdmFscyA9IGJhY2tlbmRfdXRpbC5tZXJnZVJlYWxBbmRJbWFnQXJyYXlzKFxuICAgICAgICAgIHJlYWxWYWx1ZXMgYXMgRmxvYXQzMkFycmF5LCBpbWFnVmFsdWVzIGFzIEZsb2F0MzJBcnJheSk7XG4gICAgfSBlbHNlIGlmIChidWZmZXIgPT0gbnVsbCkge1xuICAgICAgdmFscyA9IHRoaXMuZ2V0VmFsdWVzRnJvbVRleHR1cmUoZGF0YUlkKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3Qgc2l6ZSA9IHV0aWwuc2l6ZUZyb21TaGFwZShzaGFwZSk7XG4gICAgICB2YWxzID0gdGhpcy5ncGdwdS5kb3dubG9hZEZsb2F0MzJNYXRyaXhGcm9tQnVmZmVyKGJ1ZmZlciwgc2l6ZSk7XG4gICAgfVxuICAgIGlmICh0bXBEb3dubG9hZFRhcmdldCAhPSBudWxsKSB7XG4gICAgICB0aGlzLmRpc3Bvc2VJbnRlcm1lZGlhdGVUZW5zb3JJbmZvKHRtcERvd25sb2FkVGFyZ2V0KTtcbiAgICB9XG4gICAgaWYgKGJ1ZmZlciAhPSBudWxsKSB7XG4gICAgICBjb25zdCBnbCA9IHRoaXMuZ3BncHUuZ2w7XG4gICAgICB3ZWJnbF91dGlsLmNhbGxBbmRDaGVjayhnbCwgKCkgPT4gZ2wuZGVsZXRlQnVmZmVyKGJ1ZmZlcikpO1xuICAgIH1cbiAgICBjb25zdCBkVHlwZVZhbHMgPSB0aGlzLmNvbnZlcnRBbmRDYWNoZU9uQ1BVKGRhdGFJZCwgdmFscyk7XG5cbiAgICBjb25zdCBzdWJzY3JpYmVycyA9IHRoaXMucGVuZGluZ1JlYWQuZ2V0KGRhdGFJZCk7XG4gICAgdGhpcy5wZW5kaW5nUmVhZC5kZWxldGUoZGF0YUlkKTtcblxuICAgIC8vIE5vdGlmeSBhbGwgcGVuZGluZyByZWFkcy5cbiAgICBzdWJzY3JpYmVycy5mb3JFYWNoKHJlc29sdmUgPT4gcmVzb2x2ZShkVHlwZVZhbHMpKTtcbiAgICBpZiAodGhpcy5wZW5kaW5nRGlzcG9zYWwuaGFzKGRhdGFJZCkpIHtcbiAgICAgIHRoaXMucGVuZGluZ0Rpc3Bvc2FsLmRlbGV0ZShkYXRhSWQpO1xuICAgICAgaWYgKHRoaXMuZGlzcG9zZURhdGEoZGF0YUlkKSkge1xuICAgICAgICBlbmdpbmUoKS5yZW1vdmVEYXRhSWQoZGF0YUlkLCB0aGlzKTtcbiAgICAgIH1cbiAgICAgIHRoaXMucGVuZGluZ0RlbGV0ZXMtLTtcbiAgICB9XG4gICAgcmV0dXJuIGRUeXBlVmFscztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWFkIHRlbnNvciB0byBhIG5ldyB0ZXh0dXJlIHRoYXQgaXMgZGVuc2VseSBwYWNrZWQgZm9yIGVhc2Ugb2YgdXNlLlxuICAgKiBAcGFyYW0gZGF0YUlkIFRoZSBzb3VyY2UgdGVuc29yLlxuICAgKiBAcGFyYW0gb3B0aW9uc1xuICAgKiAgICAgY3VzdG9tVGV4U2hhcGU6IE9wdGlvbmFsLiBJZiBzZXQsIHdpbGwgdXNlIHRoZSB1c2VyIGRlZmluZWQgdGV4dHVyZVxuICAgKiAgICAgc2hhcGUgdG8gY3JlYXRlIHRoZSB0ZXh0dXJlLlxuICAgKi9cbiAgcmVhZFRvR1BVKGRhdGFJZDogRGF0YUlkLCBvcHRpb25zOiBEYXRhVG9HUFVXZWJHTE9wdGlvbiA9IHt9KTogR1BVRGF0YSB7XG4gICAgY29uc3QgdGV4RGF0YSA9IHRoaXMudGV4RGF0YS5nZXQoZGF0YUlkKTtcbiAgICBjb25zdCB7dmFsdWVzLCBzaGFwZSwgc2xpY2UsIGR0eXBlLCBpc1BhY2tlZCwgdGV4dHVyZX0gPSB0ZXhEYXRhO1xuXG4gICAgaWYgKGR0eXBlID09PSAnY29tcGxleDY0Jykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdEb2VzIG5vdCBzdXBwb3J0IHJlYWRpbmcgdGV4dHVyZSBmb3IgY29tcGxleDY0IGR0eXBlLicpO1xuICAgIH1cblxuICAgIC8vIFRoZSBwcmVzZW5jZSBvZiBgc2xpY2VgIGluZGljYXRlcyB0aGlzIHRlbnNvciBpcyBhIHNoYWxsb3cgc2xpY2Ugb2YgYVxuICAgIC8vIGRpZmZlcmVudCB0ZW5zb3IsIGFuZCBpcyB1c2luZyB0aGF0IG9yaWdpbmFsIHRlbnNvcidzIHRleHR1cmUuIFJ1blxuICAgIC8vIGBjbG9uZWAgaW4gb3JkZXIgdG8gY29weSB0aGF0IHRleHR1cmUgYW5kIHJlYWQgZnJvbSBpdC5cbiAgICBpZiAoc2xpY2UgIT0gbnVsbCkge1xuICAgICAgbGV0IHByb2dyYW07XG4gICAgICBpZiAoaXNQYWNrZWQpIHtcbiAgICAgICAgcHJvZ3JhbSA9IG5ldyBVbmFyeU9wUGFja2VkUHJvZ3JhbShzaGFwZSwgdW5hcnlfb3AuQ0xPTkUpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcHJvZ3JhbSA9IG5ldyBVbmFyeU9wUHJvZ3JhbShzaGFwZSwgdW5hcnlfb3AuQ0xPTkUpO1xuICAgICAgfVxuICAgICAgY29uc3QgcmVzID1cbiAgICAgICAgICB0aGlzLnJ1bldlYkdMUHJvZ3JhbShwcm9ncmFtLCBbe2RhdGFJZCwgc2hhcGUsIGR0eXBlfV0sIGR0eXBlKTtcbiAgICAgIGNvbnN0IGdwdVJlc291b3JjZSA9IHRoaXMucmVhZFRvR1BVKHJlcywgb3B0aW9ucyk7XG4gICAgICB0aGlzLmRpc3Bvc2VJbnRlcm1lZGlhdGVUZW5zb3JJbmZvKHJlcyk7XG4gICAgICByZXR1cm4gZ3B1UmVzb3VvcmNlO1xuICAgIH1cblxuICAgIGlmICh0ZXh0dXJlID09IG51bGwpIHtcbiAgICAgIGlmICh2YWx1ZXMgIT0gbnVsbCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0RhdGEgaXMgbm90IG9uIEdQVSBidXQgb24gQ1BVLicpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGVyZSBpcyBubyBkYXRhIG9uIEdQVSBvciBDUFUuJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gRGVjb2RlIHRoZSB0ZXh0dXJlIHNvIHRoYXQgaXQgaXMgc3RvcmVkIGRlbnNlbHkgKHVzaW5nIGZvdXIgY2hhbm5lbHMpLlxuICAgIGNvbnN0IHRtcFRhcmdldCA9IHRoaXMuZGVjb2RlKGRhdGFJZCwgb3B0aW9ucy5jdXN0b21UZXhTaGFwZSk7XG5cbiAgICAvLyBNYWtlIGVuZ2luZSB0cmFjayB0aGlzIHRlbnNvciwgc28gdGhhdCB3ZSBjYW4gZGlzcG9zZSBpdCBsYXRlci5cbiAgICBjb25zdCB0ZW5zb3JSZWYgPSBlbmdpbmUoKS5tYWtlVGVuc29yRnJvbURhdGFJZChcbiAgICAgICAgdG1wVGFyZ2V0LmRhdGFJZCwgdG1wVGFyZ2V0LnNoYXBlLCB0bXBUYXJnZXQuZHR5cGUpO1xuXG4gICAgY29uc3QgdG1wRGF0YSA9IHRoaXMudGV4RGF0YS5nZXQodG1wVGFyZ2V0LmRhdGFJZCk7XG4gICAgcmV0dXJuIHt0ZW5zb3JSZWYsIC4uLnRtcERhdGEudGV4dHVyZX07XG4gIH1cblxuICBidWZmZXJTeW5jPFIgZXh0ZW5kcyBSYW5rPih0OiBUZW5zb3JJbmZvKTogVGVuc29yQnVmZmVyPFI+IHtcbiAgICBjb25zdCBkYXRhID0gdGhpcy5yZWFkU3luYyh0LmRhdGFJZCk7XG4gICAgbGV0IGRlY29kZWREYXRhID0gZGF0YSBhcyBEYXRhVmFsdWVzO1xuICAgIGlmICh0LmR0eXBlID09PSAnc3RyaW5nJykge1xuICAgICAgdHJ5IHtcbiAgICAgICAgLy8gRGVjb2RlIHRoZSBieXRlcyBpbnRvIHN0cmluZy5cbiAgICAgICAgZGVjb2RlZERhdGEgPSAoZGF0YSBhcyBVaW50OEFycmF5W10pLm1hcChkID0+IHV0aWwuZGVjb2RlU3RyaW5nKGQpKTtcbiAgICAgIH0gY2F0Y2gge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ZhaWxlZCB0byBkZWNvZGUgZW5jb2RlZCBzdHJpbmcgYnl0ZXMgaW50byB1dGYtOCcpO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gYnVmZmVyKHQuc2hhcGUgYXMgU2hhcGVNYXBbUl0sIHQuZHR5cGUsIGRlY29kZWREYXRhKSBhc1xuICAgICAgICBUZW5zb3JCdWZmZXI8Uj47XG4gIH1cblxuICBwcml2YXRlIGNoZWNrTnVtZXJpY2FsUHJvYmxlbXModmFsdWVzOiBCYWNrZW5kVmFsdWVzKTogdm9pZCB7XG4gICAgaWYgKHZhbHVlcyA9PSBudWxsKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdmFsdWVzLmxlbmd0aDsgaSsrKSB7XG4gICAgICBjb25zdCBudW0gPSB2YWx1ZXNbaV0gYXMgbnVtYmVyO1xuICAgICAgaWYgKCF3ZWJnbF91dGlsLmNhbkJlUmVwcmVzZW50ZWQobnVtKSkge1xuICAgICAgICBpZiAoZW52KCkuZ2V0Qm9vbCgnV0VCR0xfUkVOREVSX0ZMT0FUMzJfQ0FQQUJMRScpKSB7XG4gICAgICAgICAgdGhyb3cgRXJyb3IoXG4gICAgICAgICAgICAgIGBUaGUgdmFsdWUgJHtudW19IGNhbm5vdCBiZSByZXByZXNlbnRlZCB3aXRoIHlvdXIgYCArXG4gICAgICAgICAgICAgIGBjdXJyZW50IHNldHRpbmdzLiBDb25zaWRlciBlbmFibGluZyBmbG9hdDMyIHJlbmRlcmluZzogYCArXG4gICAgICAgICAgICAgIGAndGYuZW52KCkuc2V0KCdXRUJHTF9SRU5ERVJfRkxPQVQzMl9FTkFCTEVEJywgdHJ1ZSk7J2ApO1xuICAgICAgICB9XG4gICAgICAgIHRocm93IEVycm9yKGBUaGUgdmFsdWUgJHtudW19IGNhbm5vdCBiZSByZXByZXNlbnRlZCBvbiB0aGlzIGRldmljZS5gKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGdldFZhbHVlc0Zyb21UZXh0dXJlKGRhdGFJZDogRGF0YUlkKTogRmxvYXQzMkFycmF5IHtcbiAgICBjb25zdCB7c2hhcGUsIGR0eXBlLCBpc1BhY2tlZH0gPSB0aGlzLnRleERhdGEuZ2V0KGRhdGFJZCk7XG4gICAgY29uc3Qgc2l6ZSA9IHV0aWwuc2l6ZUZyb21TaGFwZShzaGFwZSk7XG4gICAgaWYgKGVudigpLmdldEJvb2woJ1dFQkdMX0RPV05MT0FEX0ZMT0FUX0VOQUJMRUQnKSkge1xuICAgICAgY29uc3QgdG1wVGFyZ2V0ID0gdGhpcy5kZWNvZGUoZGF0YUlkKTtcbiAgICAgIGNvbnN0IHRtcERhdGEgPSB0aGlzLnRleERhdGEuZ2V0KHRtcFRhcmdldC5kYXRhSWQpO1xuICAgICAgY29uc3QgdmFscyA9XG4gICAgICAgICAgdGhpcy5ncGdwdVxuICAgICAgICAgICAgICAuZG93bmxvYWRNYXRyaXhGcm9tUGFja2VkVGV4dHVyZShcbiAgICAgICAgICAgICAgICAgIHRtcERhdGEudGV4dHVyZS50ZXh0dXJlLCAuLi50ZXhfdXRpbC5nZXREZW5zZVRleFNoYXBlKHNoYXBlKSlcbiAgICAgICAgICAgICAgLnN1YmFycmF5KDAsIHNpemUpO1xuXG4gICAgICB0aGlzLmRpc3Bvc2VJbnRlcm1lZGlhdGVUZW5zb3JJbmZvKHRtcFRhcmdldCk7XG5cbiAgICAgIHJldHVybiB2YWxzO1xuICAgIH1cblxuICAgIGNvbnN0IHNob3VsZFVzZVBhY2tlZFByb2dyYW0gPVxuICAgICAgICBlbnYoKS5nZXRCb29sKCdXRUJHTF9QQUNLJykgJiYgaXNQYWNrZWQgPT09IHRydWU7XG4gICAgY29uc3Qgb3V0cHV0U2hhcGUgPVxuICAgICAgICBzaG91bGRVc2VQYWNrZWRQcm9ncmFtID8gd2ViZ2xfdXRpbC5nZXRTaGFwZUFzM0Qoc2hhcGUpIDogc2hhcGU7XG4gICAgY29uc3QgcHJvZ3JhbSA9IHNob3VsZFVzZVBhY2tlZFByb2dyYW0gP1xuICAgICAgICBuZXcgRW5jb2RlRmxvYXRQYWNrZWRQcm9ncmFtKG91dHB1dFNoYXBlIGFzIFtudW1iZXIsIG51bWJlciwgbnVtYmVyXSkgOlxuICAgICAgICBuZXcgRW5jb2RlRmxvYXRQcm9ncmFtKG91dHB1dFNoYXBlKTtcbiAgICBjb25zdCBvdXRwdXQgPSB0aGlzLnJ1bldlYkdMUHJvZ3JhbShcbiAgICAgICAgcHJvZ3JhbSwgW3tzaGFwZTogb3V0cHV0U2hhcGUsIGR0eXBlLCBkYXRhSWR9XSwgJ2Zsb2F0MzInKTtcbiAgICBjb25zdCB0bXBEYXRhID0gdGhpcy50ZXhEYXRhLmdldChvdXRwdXQuZGF0YUlkKTtcbiAgICBjb25zdCB2YWxzID0gdGhpcy5ncGdwdVxuICAgICAgICAgICAgICAgICAgICAgLmRvd25sb2FkQnl0ZUVuY29kZWRGbG9hdE1hdHJpeEZyb21PdXRwdXRUZXh0dXJlKFxuICAgICAgICAgICAgICAgICAgICAgICAgIHRtcERhdGEudGV4dHVyZS50ZXh0dXJlLCB0bXBEYXRhLnRleFNoYXBlWzBdLFxuICAgICAgICAgICAgICAgICAgICAgICAgIHRtcERhdGEudGV4U2hhcGVbMV0pXG4gICAgICAgICAgICAgICAgICAgICAuc3ViYXJyYXkoMCwgc2l6ZSk7XG4gICAgdGhpcy5kaXNwb3NlSW50ZXJtZWRpYXRlVGVuc29ySW5mbyhvdXRwdXQpO1xuXG4gICAgcmV0dXJuIHZhbHM7XG4gIH1cblxuICB0aW1lckF2YWlsYWJsZSgpOiBib29sZWFuIHtcbiAgICByZXR1cm4gZW52KCkuZ2V0TnVtYmVyKCdXRUJHTF9ESVNKT0lOVF9RVUVSWV9USU1FUl9FWFRFTlNJT05fUkVMSUFCTEUnKSA+IDA7XG4gIH1cblxuICB0aW1lKGY6ICgpID0+IHZvaWQpOiBQcm9taXNlPFdlYkdMVGltaW5nSW5mbz4ge1xuICAgIGNvbnN0IG9sZEFjdGl2ZVRpbWVycyA9IHRoaXMuYWN0aXZlVGltZXJzO1xuICAgIGNvbnN0IG5ld0FjdGl2ZVRpbWVyczogVGltZXJOb2RlW10gPSBbXTtcblxuICAgIGxldCBvdXRlck1vc3RUaW1lID0gZmFsc2U7XG4gICAgaWYgKHRoaXMucHJvZ3JhbVRpbWVyc1N0YWNrID09IG51bGwpIHtcbiAgICAgIHRoaXMucHJvZ3JhbVRpbWVyc1N0YWNrID0gbmV3QWN0aXZlVGltZXJzO1xuICAgICAgb3V0ZXJNb3N0VGltZSA9IHRydWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuYWN0aXZlVGltZXJzLnB1c2gobmV3QWN0aXZlVGltZXJzKTtcbiAgICB9XG4gICAgdGhpcy5hY3RpdmVUaW1lcnMgPSBuZXdBY3RpdmVUaW1lcnM7XG5cbiAgICBmKCk7XG5cbiAgICAvLyBuZWVkaW5nIHRvIHNwbGl0IHRoZXNlIHVwIGJlY2F1c2UgdXRpbC5mbGF0dGVuIG9ubHkgYWNjZXB0cyBjZXJ0YWluIHR5cGVzXG4gICAgY29uc3QgZmxhdHRlbmVkQWN0aXZlVGltZXJRdWVyaWVzID1cbiAgICAgICAgdXRpbC5mbGF0dGVuKHRoaXMuYWN0aXZlVGltZXJzLm1hcCgoZDogS2VybmVsSW5mbykgPT4gZC5xdWVyeSkpXG4gICAgICAgICAgICAuZmlsdGVyKGQgPT4gZCAhPSBudWxsKTtcbiAgICBjb25zdCBmbGF0dGVuZWRBY3RpdmVUaW1lck5hbWVzID1cbiAgICAgICAgdXRpbC5mbGF0dGVuKHRoaXMuYWN0aXZlVGltZXJzLm1hcCgoZDogS2VybmVsSW5mbykgPT4gZC5uYW1lKSlcbiAgICAgICAgICAgIC5maWx0ZXIoZCA9PiBkICE9IG51bGwpO1xuXG4gICAgdGhpcy5hY3RpdmVUaW1lcnMgPSBvbGRBY3RpdmVUaW1lcnM7XG5cbiAgICBpZiAob3V0ZXJNb3N0VGltZSkge1xuICAgICAgdGhpcy5wcm9ncmFtVGltZXJzU3RhY2sgPSBudWxsO1xuICAgIH1cblxuICAgIGNvbnN0IHJlczogV2ViR0xUaW1pbmdJbmZvID0ge1xuICAgICAgdXBsb2FkV2FpdE1zOiB0aGlzLnVwbG9hZFdhaXRNcyxcbiAgICAgIGRvd25sb2FkV2FpdE1zOiB0aGlzLmRvd25sb2FkV2FpdE1zLFxuICAgICAga2VybmVsTXM6IG51bGwsXG4gICAgICB3YWxsTXM6IG51bGwgIC8vIHdpbGwgYmUgZmlsbGVkIGJ5IHRoZSBlbmdpbmVcbiAgICB9O1xuXG4gICAgcmV0dXJuIChhc3luYyAoKSA9PiB7XG4gICAgICBpZiAoZW52KClcbiAgICAgICAgLmdldE51bWJlcignV0VCR0xfRElTSk9JTlRfUVVFUllfVElNRVJfRVhURU5TSU9OX1JFTElBQkxFJykgPiAwKSB7XG4gICAgICAgIGNvbnN0IGtlcm5lbE1zID0gYXdhaXQgUHJvbWlzZS5hbGwoZmxhdHRlbmVkQWN0aXZlVGltZXJRdWVyaWVzKTtcblxuICAgICAgICByZXNbJ2tlcm5lbE1zJ10gPSB1dGlsLnN1bShrZXJuZWxNcyk7XG4gICAgICAgIHJlc1snZ2V0RXh0cmFQcm9maWxlSW5mbyddID0gKCkgPT5cbiAgICAgICAgICBrZXJuZWxNcy5tYXAoKGQsIGkpID0+ICh7bmFtZTogZmxhdHRlbmVkQWN0aXZlVGltZXJOYW1lc1tpXSwgbXM6IGR9KSlcbiAgICAgICAgICAgIC5tYXAoZCA9PiBgJHtkLm5hbWV9OiAke2QubXN9YClcbiAgICAgICAgICAgIC5qb2luKCcsICcpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmVzWydrZXJuZWxNcyddID0ge1xuICAgICAgICAgIGVycm9yOiAnV2ViR0wgcXVlcnkgdGltZXJzIGFyZSBub3Qgc3VwcG9ydGVkIGluIHRoaXMgZW52aXJvbm1lbnQuJ1xuICAgICAgICB9O1xuICAgICAgfVxuXG4gICAgICB0aGlzLnVwbG9hZFdhaXRNcyA9IDA7XG4gICAgICB0aGlzLmRvd25sb2FkV2FpdE1zID0gMDtcbiAgICAgIHJldHVybiByZXM7XG4gICAgfSkoKTtcbiAgfVxuICBtZW1vcnkoKTogV2ViR0xNZW1vcnlJbmZvIHtcbiAgICByZXR1cm4ge1xuICAgICAgdW5yZWxpYWJsZTogZmFsc2UsXG4gICAgICBudW1CeXRlc0luR1BVOiB0aGlzLm51bUJ5dGVzSW5HUFUsXG4gICAgICBudW1CeXRlc0luR1BVQWxsb2NhdGVkOiB0aGlzLnRleHR1cmVNYW5hZ2VyLm51bUJ5dGVzQWxsb2NhdGVkLFxuICAgICAgbnVtQnl0ZXNJbkdQVUZyZWU6IHRoaXMudGV4dHVyZU1hbmFnZXIubnVtQnl0ZXNGcmVlXG4gICAgfSBhcyBXZWJHTE1lbW9yeUluZm87XG4gIH1cblxuICBwcml2YXRlIHN0YXJ0VGltZXIoKTogV2ViR0xRdWVyeXxDUFVUaW1lclF1ZXJ5IHtcbiAgICBpZiAoZW52KCkuZ2V0TnVtYmVyKCdXRUJHTF9ESVNKT0lOVF9RVUVSWV9USU1FUl9FWFRFTlNJT05fUkVMSUFCTEUnKSA+IDApIHtcbiAgICAgIHJldHVybiB0aGlzLmdwZ3B1LmJlZ2luUXVlcnkoKTtcbiAgICB9XG4gICAgcmV0dXJuIHtzdGFydE1zOiB1dGlsLm5vdygpLCBlbmRNczogbnVsbH07XG4gIH1cblxuICBwcml2YXRlIGVuZFRpbWVyKHF1ZXJ5OiBXZWJHTFF1ZXJ5fENQVVRpbWVyUXVlcnkpOiBXZWJHTFF1ZXJ5fENQVVRpbWVyUXVlcnkge1xuICAgIGlmIChlbnYoKS5nZXROdW1iZXIoJ1dFQkdMX0RJU0pPSU5UX1FVRVJZX1RJTUVSX0VYVEVOU0lPTl9SRUxJQUJMRScpID4gMCkge1xuICAgICAgdGhpcy5ncGdwdS5lbmRRdWVyeSgpO1xuICAgICAgcmV0dXJuIHF1ZXJ5O1xuICAgIH1cbiAgICAocXVlcnkgYXMgQ1BVVGltZXJRdWVyeSkuZW5kTXMgPSB1dGlsLm5vdygpO1xuICAgIHJldHVybiBxdWVyeTtcbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgZ2V0UXVlcnlUaW1lKHF1ZXJ5OiBXZWJHTFF1ZXJ5fENQVVRpbWVyUXVlcnkpOiBQcm9taXNlPG51bWJlcj4ge1xuICAgIGlmIChlbnYoKS5nZXROdW1iZXIoJ1dFQkdMX0RJU0pPSU5UX1FVRVJZX1RJTUVSX0VYVEVOU0lPTl9SRUxJQUJMRScpID4gMCkge1xuICAgICAgcmV0dXJuIHRoaXMuZ3BncHUud2FpdEZvclF1ZXJ5QW5kR2V0VGltZShxdWVyeSBhcyBXZWJHTFF1ZXJ5KTtcbiAgICB9XG4gICAgY29uc3QgdGltZXJRdWVyeSA9IHF1ZXJ5IGFzIENQVVRpbWVyUXVlcnk7XG4gICAgcmV0dXJuIHRpbWVyUXVlcnkuZW5kTXMgLSB0aW1lclF1ZXJ5LnN0YXJ0TXM7XG4gIH1cblxuICBwcml2YXRlIHBlbmRpbmdEZWxldGVzID0gMDtcblxuICAvKipcbiAgICogRGVjcmVhc2UgdGhlIFJlZkNvdW50IG9uIHRoZSBkYXRhSWQgYW5kIGRpc3Bvc2UgdGhlIG1lbW9yeSBpZiB0aGUgZGF0YUlkXG4gICAqIGhhcyAwIHJlZkNvdW50LiBJZiB0aGVyZSBhcmUgcGVuZGluZyByZWFkIG9uIHRoZSBkYXRhLCB0aGUgZGlzcG9zYWwgd291bGRcbiAgICogYWRkZWQgdG8gdGhlIHBlbmRpbmcgZGVsZXRlIHF1ZXVlLiBSZXR1cm4gdHJ1ZSBpZiB0aGUgZGF0YUlkIGlzIHJlbW92ZWRcbiAgICogZnJvbSBiYWNrZW5kIG9yIHRoZSBiYWNrZW5kIGRvZXMgbm90IGNvbnRhaW4gdGhlIGRhdGFJZCwgZmFsc2UgaWYgdGhlXG4gICAqIGRhdGFJZCBpcyBub3QgcmVtb3ZlZC4gTWVtb3J5IG1heSBvciBtYXkgbm90IGJlIHJlbGVhc2VkIGV2ZW4gd2hlbiBkYXRhSWRcbiAgICogaXMgcmVtb3ZlZCwgd2hpY2ggYWxzbyBkZXBlbmRzIG9uIGRhdGFSZWZDb3VudCwgc2VlIGByZWxlYXNlR1BVYC5cbiAgICogQHBhcmFtIGRhdGFJZFxuICAgKiBAb2FyYW0gZm9yY2UgT3B0aW9uYWwsIHJlbW92ZSB0aGUgZGF0YSByZWdhcmRsZXNzIG9mIHJlZkNvdW50XG4gICAqL1xuICBkaXNwb3NlRGF0YShkYXRhSWQ6IERhdGFJZCwgZm9yY2UgPSBmYWxzZSk6IGJvb2xlYW4ge1xuICAgIGlmICh0aGlzLnBlbmRpbmdEaXNwb3NhbC5oYXMoZGF0YUlkKSkge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cblxuICAgIC8vIE5vLW9wIGlmIGFscmVhZHkgZGlzcG9zZWQuXG4gICAgaWYgKCF0aGlzLnRleERhdGEuaGFzKGRhdGFJZCkpIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIC8vIGlmIGZvcmNlIGZsYWcgaXMgc2V0LCBjaGFuZ2UgcmVmQ291bnQgdG8gMCwgdGhpcyB3b3VsZCBlbnN1cmUgZGlzcG9zYWxcbiAgICAvLyB3aGVuIGFkZGVkIHRvIHRoZSBwZW5kaW5nRGlzcG9zYWwgcXVldWUuIE1lbW9yeSBtYXkgb3IgbWF5IG5vdCBiZVxuICAgIC8vIHJlbGVhc2VkLCB3aGljaCBhbHNvIGRlcGVuZHMgb24gZGF0YVJlZkNvdW50LCBzZWUgYHJlbGVhc2VHUFVgLlxuICAgIGlmIChmb3JjZSkge1xuICAgICAgdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpLnJlZkNvdW50ID0gMDtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpLnJlZkNvdW50LS07XG4gICAgfVxuXG4gICAgaWYgKCFmb3JjZSAmJiB0aGlzLnRleERhdGEuZ2V0KGRhdGFJZCkucmVmQ291bnQgPiAwKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucGVuZGluZ1JlYWQuaGFzKGRhdGFJZCkpIHtcbiAgICAgIHRoaXMucGVuZGluZ0Rpc3Bvc2FsLmFkZChkYXRhSWQpO1xuICAgICAgdGhpcy5wZW5kaW5nRGVsZXRlcysrO1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cblxuICAgIHRoaXMucmVsZWFzZUdQVURhdGEoZGF0YUlkKTtcbiAgICBjb25zdCB7Y29tcGxleFRlbnNvckluZm9zfSA9IHRoaXMudGV4RGF0YS5nZXQoZGF0YUlkKTtcbiAgICBpZiAoY29tcGxleFRlbnNvckluZm9zICE9IG51bGwpIHtcbiAgICAgIHRoaXMuZGlzcG9zZURhdGEoY29tcGxleFRlbnNvckluZm9zLnJlYWwuZGF0YUlkLCBmb3JjZSk7XG4gICAgICB0aGlzLmRpc3Bvc2VEYXRhKGNvbXBsZXhUZW5zb3JJbmZvcy5pbWFnLmRhdGFJZCwgZm9yY2UpO1xuICAgIH1cblxuICAgIHRoaXMudGV4RGF0YS5kZWxldGUoZGF0YUlkKTtcblxuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgcHJpdmF0ZSByZWxlYXNlR1BVRGF0YShkYXRhSWQ6IERhdGFJZCk6IHZvaWQge1xuICAgIGNvbnN0IHt0ZXh0dXJlLCBkdHlwZSwgdGV4U2hhcGUsIHVzYWdlLCBpc1BhY2tlZCwgc2xpY2V9ID1cbiAgICAgICAgdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpO1xuICAgIGNvbnN0IGtleSA9IHNsaWNlICYmIHNsaWNlLm9yaWdEYXRhSWQgfHwgZGF0YUlkO1xuICAgIGNvbnN0IHJlZkNvdW50ID0gdGhpcy5kYXRhUmVmQ291bnQuZ2V0KGtleSk7XG5cbiAgICBpZiAocmVmQ291bnQgPiAxKSB7XG4gICAgICB0aGlzLmRhdGFSZWZDb3VudC5zZXQoa2V5LCByZWZDb3VudCAtIDEpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLmRhdGFSZWZDb3VudC5kZWxldGUoa2V5KTtcbiAgICAgIGlmICh0ZXh0dXJlICE9IG51bGwpIHtcbiAgICAgICAgdGhpcy5udW1CeXRlc0luR1BVIC09IHRoaXMuY29tcHV0ZUJ5dGVzKHRleFNoYXBlLCBkdHlwZSk7XG4gICAgICAgIHRoaXMudGV4dHVyZU1hbmFnZXIucmVsZWFzZVRleHR1cmUodGV4dHVyZSwgdGV4U2hhcGUsIHVzYWdlLCBpc1BhY2tlZCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgdGV4RGF0YSA9IHRoaXMudGV4RGF0YS5nZXQoZGF0YUlkKTtcbiAgICB0ZXhEYXRhLnRleHR1cmUgPSBudWxsO1xuICAgIHRleERhdGEudGV4U2hhcGUgPSBudWxsO1xuICAgIHRleERhdGEuaXNQYWNrZWQgPSBmYWxzZTtcbiAgICB0ZXhEYXRhLnNsaWNlID0gbnVsbDtcbiAgfVxuXG4gIGdldFRleHR1cmUoZGF0YUlkOiBEYXRhSWQpOiBXZWJHTFRleHR1cmUge1xuICAgIHRoaXMudXBsb2FkVG9HUFUoZGF0YUlkKTtcbiAgICByZXR1cm4gdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpLnRleHR1cmUudGV4dHVyZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGludGVybmFsIGluZm9ybWF0aW9uIGZvciB0aGUgc3BlY2lmaWMgZGF0YSBidWNrZXQuIFVzZWQgaW4gdW5pdFxuICAgKiB0ZXN0cy5cbiAgICovXG4gIGdldERhdGFJbmZvKGRhdGFJZDogRGF0YUlkKTogVGV4dHVyZURhdGEge1xuICAgIHJldHVybiB0aGlzLnRleERhdGEuZ2V0KGRhdGFJZCk7XG4gIH1cblxuICAvKlxuICBUZXN0cyB3aGV0aGVyIGFsbCB0aGUgaW5wdXRzIHRvIGFuIG9wIGFyZSBzbWFsbCBhbmQgb24gdGhlIENQVS4gVGhpcyBoZXVyaXN0aWNcbiAgZGV0ZXJtaW5lcyB3aGVuIGl0IHdvdWxkIGJlIGZhc3RlciB0byBleGVjdXRlIGEga2VybmVsIG9uIHRoZSBDUFUuIFdlYkdMXG4gIGtlcm5lbHMgb3B0IGludG8gcnVubmluZyB0aGlzIGNoZWNrIGFuZCBmb3J3YXJkaW5nIHdoZW4gYXBwcm9wcmlhdGUuXG4gIFRPRE8oaHR0cHM6Ly9naXRodWIuY29tL3RlbnNvcmZsb3cvdGZqcy9pc3N1ZXMvODcyKTogRGV2ZWxvcCBhIG1vcmVcbiAgc3VzdGFpbmFibGUgc3RyYXRlZ3kgZm9yIG9wdGltaXppbmcgYmFja2VuZCBleGVjdXRpb24gb2Ygb3BzLlxuICAgKi9cbiAgc2hvdWxkRXhlY3V0ZU9uQ1BVKFxuICAgICAgaW5wdXRzOiBUZW5zb3JJbmZvW10sXG4gICAgICBzaXplVGhyZXNob2xkID0gQ1BVX0hBTkRPRkZfU0laRV9USFJFU0hPTEQpOiBib29sZWFuIHtcbiAgICByZXR1cm4gZW52KCkuZ2V0Qm9vbCgnV0VCR0xfQ1BVX0ZPUldBUkQnKSAmJlxuICAgICAgICBpbnB1dHMuZXZlcnkoXG4gICAgICAgICAgICBpbnB1dCA9PiB0aGlzLnRleERhdGEuZ2V0KGlucHV0LmRhdGFJZCkudGV4dHVyZSA9PSBudWxsICYmXG4gICAgICAgICAgICAgICAgdXRpbC5zaXplRnJvbVNoYXBlKGlucHV0LnNoYXBlKSA8IHNpemVUaHJlc2hvbGQpO1xuICB9XG5cbiAgZ2V0R1BHUFVDb250ZXh0KCk6IEdQR1BVQ29udGV4dCB7XG4gICAgcmV0dXJuIHRoaXMuZ3BncHU7XG4gIH1cblxuICB3aGVyZShjb25kaXRpb246IFRlbnNvcik6IFRlbnNvcjJEIHtcbiAgICBiYWNrZW5kX3V0aWwud2FybihcbiAgICAgICAgJ3RmLndoZXJlKCkgaW4gd2ViZ2wgbG9ja3MgdGhlIFVJIHRocmVhZC4gJyArXG4gICAgICAgICdDYWxsIHRmLndoZXJlQXN5bmMoKSBpbnN0ZWFkJyk7XG4gICAgY29uc3QgY29uZFZhbHMgPSBjb25kaXRpb24uZGF0YVN5bmMoKTtcbiAgICByZXR1cm4gd2hlcmVJbXBsKGNvbmRpdGlvbi5zaGFwZSwgY29uZFZhbHMpO1xuICB9XG5cbiAgcHJpdmF0ZSBwYWNrZWRVbmFyeU9wKHg6IFRlbnNvckluZm8sIG9wOiBzdHJpbmcsIGR0eXBlOiBEYXRhVHlwZSkge1xuICAgIGNvbnN0IHByb2dyYW0gPSBuZXcgVW5hcnlPcFBhY2tlZFByb2dyYW0oeC5zaGFwZSwgb3ApO1xuICAgIGNvbnN0IG91dEluZm8gPSB0aGlzLmNvbXBpbGVBbmRSdW4ocHJvZ3JhbSwgW3hdLCBkdHlwZSk7XG4gICAgcmV0dXJuIGVuZ2luZSgpLm1ha2VUZW5zb3JGcm9tRGF0YUlkKFxuICAgICAgICBvdXRJbmZvLmRhdGFJZCwgb3V0SW5mby5zaGFwZSwgb3V0SW5mby5kdHlwZSk7XG4gIH1cblxuICAvLyBUT0RPKG1zb3VsYW5pbGxlKSByZW1vdmUgdGhpcyBvbmNlIHRoZSBiYWNrZW5kIGhhcyBiZWVuIG1vZHVsYXJpemVkXG4gIC8vIGEgY29weSBpcyBuZWVkZWQgaGVyZSB0byBicmVhayBhIGNpcmN1bGFyIGRlcGVuZGVuY3kuXG4gIC8vIEFsc28gcmVtb3ZlIHRoZSBvcCBmcm9tIHVuYXJ5X29wLlxuICBhYnM8VCBleHRlbmRzIFRlbnNvcj4oeDogVCk6IFQge1xuICAgIC8vIFRPRE86IGhhbmRsZSBjYXNlcyB3aGVuIHggaXMgY29tcGxleC5cbiAgICBpZiAodGhpcy5zaG91bGRFeGVjdXRlT25DUFUoW3hdKSAmJiB4LmR0eXBlICE9PSAnY29tcGxleDY0Jykge1xuICAgICAgY29uc3Qgb3V0VmFsdWVzID1cbiAgICAgICAgICBzaW1wbGVBYnNJbXBsQ1BVKHRoaXMudGV4RGF0YS5nZXQoeC5kYXRhSWQpLnZhbHVlcyBhcyBUeXBlZEFycmF5KTtcbiAgICAgIHJldHVybiB0aGlzLm1ha2VPdXRwdXQoeC5zaGFwZSwgeC5kdHlwZSwgb3V0VmFsdWVzKTtcbiAgICB9XG5cbiAgICBpZiAoZW52KCkuZ2V0Qm9vbCgnV0VCR0xfUEFDS19VTkFSWV9PUEVSQVRJT05TJykpIHtcbiAgICAgIHJldHVybiB0aGlzLnBhY2tlZFVuYXJ5T3AoeCwgdW5hcnlfb3AuQUJTLCB4LmR0eXBlKSBhcyBUO1xuICAgIH1cblxuICAgIGNvbnN0IHByb2dyYW0gPSBuZXcgVW5hcnlPcFByb2dyYW0oeC5zaGFwZSwgdW5hcnlfb3AuQUJTKTtcbiAgICBjb25zdCBvdXRJbmZvID0gdGhpcy5jb21waWxlQW5kUnVuKHByb2dyYW0sIFt4XSk7XG4gICAgcmV0dXJuIGVuZ2luZSgpLm1ha2VUZW5zb3JGcm9tRGF0YUlkKFxuICAgICAgICAgICAgICAgb3V0SW5mby5kYXRhSWQsIG91dEluZm8uc2hhcGUsIG91dEluZm8uZHR5cGUpIGFzIFQ7XG4gIH1cblxuICBtYWtlVGVuc29ySW5mbyhcbiAgICAgIHNoYXBlOiBudW1iZXJbXSwgZHR5cGU6IERhdGFUeXBlLFxuICAgICAgdmFsdWVzPzogQmFja2VuZFZhbHVlc3xzdHJpbmdbXSk6IFRlbnNvckluZm8ge1xuICAgIGxldCBkYXRhSWQ7XG4gICAgaWYgKGR0eXBlID09PSAnc3RyaW5nJyAmJiB2YWx1ZXMgIT0gbnVsbCAmJiB2YWx1ZXMubGVuZ3RoID4gMCAmJlxuICAgICAgICB1dGlsLmlzU3RyaW5nKHZhbHVlc1swXSkpIHtcbiAgICAgIGNvbnN0IGVuY29kZWRWYWx1ZXMgPVxuICAgICAgICAgICh2YWx1ZXMgYXMge30gYXMgc3RyaW5nW10pLm1hcChkID0+IHV0aWwuZW5jb2RlU3RyaW5nKGQpKTtcblxuICAgICAgZGF0YUlkID0gdGhpcy53cml0ZShlbmNvZGVkVmFsdWVzLCBzaGFwZSwgZHR5cGUpO1xuICAgIH0gZWxzZSB7XG4gICAgICBkYXRhSWQgPSB0aGlzLndyaXRlKHZhbHVlcyBhcyBUeXBlZEFycmF5LCBzaGFwZSwgZHR5cGUpO1xuICAgIH1cblxuICAgIHRoaXMudGV4RGF0YS5nZXQoZGF0YUlkKS51c2FnZSA9IG51bGw7XG4gICAgcmV0dXJuIHtkYXRhSWQsIHNoYXBlLCBkdHlwZX07XG4gIH1cblxuICBwcml2YXRlIG1ha2VPdXRwdXQ8VCBleHRlbmRzIFRlbnNvcj4oXG4gICAgICBzaGFwZTogbnVtYmVyW10sIGR0eXBlOiBEYXRhVHlwZSwgdmFsdWVzPzogQmFja2VuZFZhbHVlcyk6IFQge1xuICAgIGNvbnN0IHtkYXRhSWR9ID0gdGhpcy5tYWtlVGVuc29ySW5mbyhzaGFwZSwgZHR5cGUsIHZhbHVlcyk7XG4gICAgcmV0dXJuIGVuZ2luZSgpLm1ha2VUZW5zb3JGcm9tRGF0YUlkKGRhdGFJZCwgc2hhcGUsIGR0eXBlLCB0aGlzKSBhcyBUO1xuICB9XG5cbiAgdW5wYWNrVGVuc29yKGlucHV0OiBUZW5zb3JJbmZvKTogVGVuc29ySW5mbyB7XG4gICAgY29uc3QgcHJvZ3JhbSA9IG5ldyBVbnBhY2tQcm9ncmFtKGlucHV0LnNoYXBlKTtcbiAgICByZXR1cm4gdGhpcy5ydW5XZWJHTFByb2dyYW0ocHJvZ3JhbSwgW2lucHV0XSwgaW5wdXQuZHR5cGUpO1xuICB9XG5cbiAgcGFja1RlbnNvcihpbnB1dDogVGVuc29ySW5mbyk6IFRlbnNvckluZm8ge1xuICAgIGNvbnN0IHByb2dyYW0gPSBuZXcgUGFja1Byb2dyYW0oaW5wdXQuc2hhcGUpO1xuICAgIGNvbnN0IHByZXZlbnRFYWdlclVucGFja2luZ091dHB1dCA9IHRydWU7XG4gICAgcmV0dXJuIHRoaXMucnVuV2ViR0xQcm9ncmFtKFxuICAgICAgICBwcm9ncmFtLCBbaW5wdXRdLCBpbnB1dC5kdHlwZSwgbnVsbCAvKiBjdXN0b21Vbmlmb3JtVmFsdWVzICovLFxuICAgICAgICBwcmV2ZW50RWFnZXJVbnBhY2tpbmdPdXRwdXQpO1xuICB9XG5cbiAgcHJpdmF0ZSBwYWNrZWRSZXNoYXBlKGlucHV0OiBUZW5zb3JJbmZvLCBhZnRlclNoYXBlOiBudW1iZXJbXSk6IFRlbnNvckluZm8ge1xuICAgIGNvbnN0IGlucHV0M0RTaGFwZSA9IFtcbiAgICAgIHdlYmdsX3V0aWwuZ2V0QmF0Y2hEaW0oaW5wdXQuc2hhcGUpLFxuICAgICAgLi4ud2ViZ2xfdXRpbC5nZXRSb3dzQ29scyhpbnB1dC5zaGFwZSlcbiAgICBdIGFzIFtudW1iZXIsIG51bWJlciwgbnVtYmVyXTtcbiAgICBjb25zdCBpbnB1dDNEOiBUZW5zb3JJbmZvID0ge1xuICAgICAgZHR5cGU6IGlucHV0LmR0eXBlLFxuICAgICAgc2hhcGU6IGlucHV0M0RTaGFwZSxcbiAgICAgIGRhdGFJZDogaW5wdXQuZGF0YUlkXG4gICAgfTtcbiAgICBjb25zdCBhZnRlclNoYXBlQXMzRCA9IFtcbiAgICAgIHdlYmdsX3V0aWwuZ2V0QmF0Y2hEaW0oYWZ0ZXJTaGFwZSksIC4uLndlYmdsX3V0aWwuZ2V0Um93c0NvbHMoYWZ0ZXJTaGFwZSlcbiAgICBdIGFzIFtudW1iZXIsIG51bWJlciwgbnVtYmVyXTtcblxuICAgIGNvbnN0IHByb2dyYW0gPSBuZXcgUmVzaGFwZVBhY2tlZFByb2dyYW0oYWZ0ZXJTaGFwZUFzM0QsIGlucHV0M0RTaGFwZSk7XG4gICAgY29uc3QgcHJldmVudEVhZ2VyVW5wYWNraW5nT2ZPdXRwdXQgPSB0cnVlO1xuICAgIGNvbnN0IGN1c3RvbVZhbHVlcyA9IFtpbnB1dDNEU2hhcGVdO1xuICAgIGNvbnN0IG91dHB1dCA9IHRoaXMucnVuV2ViR0xQcm9ncmFtKFxuICAgICAgICBwcm9ncmFtLCBbaW5wdXQzRF0sIGlucHV0LmR0eXBlLCBjdXN0b21WYWx1ZXMsXG4gICAgICAgIHByZXZlbnRFYWdlclVucGFja2luZ09mT3V0cHV0KTtcbiAgICByZXR1cm4ge2RhdGFJZDogb3V0cHV0LmRhdGFJZCwgc2hhcGU6IGFmdGVyU2hhcGUsIGR0eXBlOiBvdXRwdXQuZHR5cGV9O1xuICB9XG5cbiAgcHJpdmF0ZSBkZWNvZGUoZGF0YUlkOiBEYXRhSWQsIGN1c3RvbVRleFNoYXBlPzogW251bWJlciwgbnVtYmVyXSk6XG4gICAgICBUZW5zb3JJbmZvIHtcbiAgICBjb25zdCB0ZXhEYXRhID0gdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpO1xuICAgIGNvbnN0IHtpc1BhY2tlZCwgc2hhcGUsIGR0eXBlfSA9IHRleERhdGE7XG4gICAgaWYgKGN1c3RvbVRleFNoYXBlICE9IG51bGwpIHtcbiAgICAgIGNvbnN0IHNpemUgPSB1dGlsLnNpemVGcm9tU2hhcGUoc2hhcGUpO1xuICAgICAgY29uc3QgdGV4U2l6ZSA9IGN1c3RvbVRleFNoYXBlWzBdICogY3VzdG9tVGV4U2hhcGVbMV0gKiA0O1xuICAgICAgdXRpbC5hc3NlcnQoXG4gICAgICAgICAgc2l6ZSA8PSB0ZXhTaXplLFxuICAgICAgICAgICgpID0+ICdjdXN0b21UZXhTaGFwZSBpcyB0b28gc21hbGwuICcgK1xuICAgICAgICAgICAgICAnUm93ICogQ29sdW1uICogNCBzaG91bGQgYmUgZXF1YWwgb3IgbGFyZ2VyIHRoYW4gdGhlICcgK1xuICAgICAgICAgICAgICAnc2l6ZSBvZiB0aGUgdGVuc29yIGRhdGEuJyk7XG4gICAgfVxuICAgIGNvbnN0IHNoYXBlQXMzRCA9XG4gICAgICAgIHdlYmdsX3V0aWwuZ2V0U2hhcGVBczNEKHNoYXBlKSBhcyBbbnVtYmVyLCBudW1iZXIsIG51bWJlcl07XG4gICAgbGV0IHByb2dyYW07XG4gICAgaWYgKGlzUGFja2VkKSB7XG4gICAgICBwcm9ncmFtID0gbmV3IERlY29kZU1hdHJpeFBhY2tlZFByb2dyYW0oc2hhcGVBczNEKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcHJvZ3JhbSA9IG5ldyBEZWNvZGVNYXRyaXhQcm9ncmFtKHNoYXBlQXMzRCk7XG4gICAgfVxuICAgIGNvbnN0IHByZXZlbnRFYWdlclVucGFja2luZ09mT3V0cHV0ID0gdHJ1ZTtcbiAgICBjb25zdCBjdXN0b21WYWx1ZXMgPVxuICAgICAgICBbY3VzdG9tVGV4U2hhcGUgIT0gbnVsbCA/IGN1c3RvbVRleFNoYXBlIDpcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB0ZXhfdXRpbC5nZXREZW5zZVRleFNoYXBlKHNoYXBlQXMzRCldO1xuICAgIGNvbnN0IG91dCA9IHRoaXMucnVuV2ViR0xQcm9ncmFtKFxuICAgICAgICBwcm9ncmFtLCBbe3NoYXBlOiBzaGFwZUFzM0QsIGR0eXBlLCBkYXRhSWR9XSwgZHR5cGUsIGN1c3RvbVZhbHVlcyxcbiAgICAgICAgcHJldmVudEVhZ2VyVW5wYWNraW5nT2ZPdXRwdXQsIGN1c3RvbVRleFNoYXBlKTtcbiAgICByZXR1cm4ge2R0eXBlLCBzaGFwZSwgZGF0YUlkOiBvdXQuZGF0YUlkfTtcbiAgfVxuXG4gIHJ1bldlYkdMUHJvZ3JhbShcbiAgICAgIHByb2dyYW06IEdQR1BVUHJvZ3JhbSwgaW5wdXRzOiBUZW5zb3JJbmZvW10sIG91dHB1dER0eXBlOiBEYXRhVHlwZSxcbiAgICAgIGN1c3RvbVVuaWZvcm1WYWx1ZXM/OiBudW1iZXJbXVtdLCBwcmV2ZW50RWFnZXJVbnBhY2tpbmdPZk91dHB1dCA9IGZhbHNlLFxuICAgICAgY3VzdG9tVGV4U2hhcGU/OiBbbnVtYmVyLCBudW1iZXJdKTogVGVuc29ySW5mbyB7XG4gICAgY29uc3Qgb3V0cHV0ID0gdGhpcy5tYWtlVGVuc29ySW5mbyhwcm9ncmFtLm91dHB1dFNoYXBlLCBvdXRwdXREdHlwZSk7XG4gICAgY29uc3Qgb3V0RGF0YSA9IHRoaXMudGV4RGF0YS5nZXQob3V0cHV0LmRhdGFJZCk7XG4gICAgaWYgKHByb2dyYW0ucGFja2VkT3V0cHV0KSB7XG4gICAgICBvdXREYXRhLmlzUGFja2VkID0gdHJ1ZTtcbiAgICB9XG4gICAgaWYgKHByb2dyYW0ub3V0UGFja2luZ1NjaGVtZSA9PT0gdGV4X3V0aWwuUGFja2luZ1NjaGVtZS5ERU5TRSkge1xuICAgICAgY29uc3QgdGV4ZWxTaGFwZSA9IGN1c3RvbVRleFNoYXBlICE9IG51bGwgP1xuICAgICAgICAgIGN1c3RvbVRleFNoYXBlIDpcbiAgICAgICAgICB0ZXhfdXRpbC5nZXREZW5zZVRleFNoYXBlKHByb2dyYW0ub3V0cHV0U2hhcGUpO1xuICAgICAgLy8gRm9yIGEgZGVuc2VseSBwYWNrZWQgb3V0cHV0LCB3ZSBleHBsaWNpdGx5IHNldCB0ZXhTaGFwZVxuICAgICAgLy8gc28gaXQgZG9lc24ndCBnZXQgYXNzaWduZWQgbGF0ZXIgYWNjb3JkaW5nIHRvIG91ciB0eXBpY2FsIHBhY2tpbmdcbiAgICAgIC8vIHNjaGVtZSB3aGVyZWluIGEgc2luZ2xlIHRleGVsIGNhbiBvbmx5IGNvbnRhaW4gdmFsdWVzIGZyb20gYWRqYWNlbnRcbiAgICAgIC8vIHJvd3MvY29scy5cbiAgICAgIG91dERhdGEudGV4U2hhcGUgPSB0ZXhlbFNoYXBlLm1hcChkID0+IGQgKiAyKSBhcyBbbnVtYmVyLCBudW1iZXJdO1xuICAgIH1cbiAgICBpZiAocHJvZ3JhbS5vdXRUZXhVc2FnZSAhPSBudWxsKSB7XG4gICAgICBvdXREYXRhLnVzYWdlID0gcHJvZ3JhbS5vdXRUZXhVc2FnZTtcbiAgICB9XG5cbiAgICBpZiAodXRpbC5zaXplRnJvbVNoYXBlKG91dHB1dC5zaGFwZSkgPT09IDApIHtcbiAgICAgIC8vIFNob3J0LWNpcmN1aXQgdGhlIGNvbXB1dGF0aW9uIHNpbmNlIHRoZSByZXN1bHQgaXMgZW1wdHkgKGhhcyAwIGluIGl0c1xuICAgICAgLy8gc2hhcGUpLlxuICAgICAgb3V0RGF0YS52YWx1ZXMgPVxuICAgICAgICAgIHV0aWwuZ2V0VHlwZWRBcnJheUZyb21EVHlwZShvdXRwdXQuZHR5cGUgYXMgJ2Zsb2F0MzInLCAwKTtcbiAgICAgIHJldHVybiBvdXRwdXQ7XG4gICAgfVxuXG4gICAgY29uc3QgZGF0YVRvRGlzcG9zZTogVGVuc29ySW5mb1tdID0gW107XG4gICAgY29uc3QgaW5wdXRzRGF0YTogVGVuc29yRGF0YVtdID0gaW5wdXRzLm1hcChpbnB1dCA9PiB7XG4gICAgICBpZiAoaW5wdXQuZHR5cGUgPT09ICdjb21wbGV4NjQnKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgIGBHUEdQVVByb2dyYW0gZG9lcyBub3Qgc3VwcG9ydCBjb21wbGV4NjQgaW5wdXQuIEZvciBjb21wbGV4NjQgYCArXG4gICAgICAgICAgICBgZHR5cGVzLCBwbGVhc2Ugc2VwYXJhdGUgdGhlIHByb2dyYW0gaW50byByZWFsIGFuZCBpbWFnaW5hcnkgYCArXG4gICAgICAgICAgICBgcGFydHMuYCk7XG4gICAgICB9XG5cbiAgICAgIGxldCB0ZXhEYXRhID0gdGhpcy50ZXhEYXRhLmdldChpbnB1dC5kYXRhSWQpO1xuXG4gICAgICBpZiAodGV4RGF0YS50ZXh0dXJlID09IG51bGwpIHtcbiAgICAgICAgaWYgKCFwcm9ncmFtLnBhY2tlZElucHV0cyAmJlxuICAgICAgICAgICAgdXRpbC5zaXplRnJvbVNoYXBlKGlucHV0LnNoYXBlKSA8PVxuICAgICAgICAgICAgICAgIGVudigpLmdldE51bWJlcignV0VCR0xfU0laRV9VUExPQURfVU5JRk9STScpKSB7XG4gICAgICAgICAgLy8gVXBsb2FkIHNtYWxsIHRlbnNvcnMgdGhhdCBsaXZlIG9uIHRoZSBDUFUgYXMgdW5pZm9ybXMsIG5vdCBhc1xuICAgICAgICAgIC8vIHRleHR1cmVzLiBEbyB0aGlzIG9ubHkgd2hlbiB0aGUgZW52aXJvbm1lbnQgc3VwcG9ydHMgMzJiaXQgZmxvYXRzXG4gICAgICAgICAgLy8gZHVlIHRvIHByb2JsZW1zIHdoZW4gY29tcGFyaW5nIDE2Yml0IGZsb2F0cyB3aXRoIDMyYml0IGZsb2F0cy5cbiAgICAgICAgICAvLyBUT0RPKGh0dHBzOi8vZ2l0aHViLmNvbS90ZW5zb3JmbG93L3RmanMvaXNzdWVzLzgyMSk6IE1ha2UgaXRcbiAgICAgICAgICAvLyBwb3NzaWJsZSBmb3IgcGFja2VkIHNoYWRlcnMgdG8gc2FtcGxlIGZyb20gdW5pZm9ybXMuXG4gICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIHNoYXBlOiBpbnB1dC5zaGFwZSxcbiAgICAgICAgICAgIHRleERhdGE6IG51bGwsXG4gICAgICAgICAgICBpc1VuaWZvcm06IHRydWUsXG4gICAgICAgICAgICB1bmlmb3JtVmFsdWVzOiB0ZXhEYXRhLnZhbHVlcyBhcyBUeXBlZEFycmF5XG4gICAgICAgICAgfTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFRoaXMgZW5zdXJlcyB0aGF0IGlmIGEgcGFja2VkIHByb2dyYW0ncyBpbnB1dHMgaGF2ZSBub3QgeWV0IGJlZW5cbiAgICAgICAgLy8gdXBsb2FkZWQgdG8gdGhlIEdQVSwgdGhleSBnZXQgdXBsb2FkZWQgYXMgcGFja2VkIHJpZ2h0IG9mZiB0aGUgYmF0LlxuICAgICAgICBpZiAocHJvZ3JhbS5wYWNrZWRJbnB1dHMpIHtcbiAgICAgICAgICB0ZXhEYXRhLmlzUGFja2VkID0gdHJ1ZTtcbiAgICAgICAgICB0ZXhEYXRhLnNoYXBlID0gaW5wdXQuc2hhcGU7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgdGhpcy51cGxvYWRUb0dQVShpbnB1dC5kYXRhSWQpO1xuICAgICAgaWYgKCEhdGV4RGF0YS5pc1BhY2tlZCAhPT0gISFwcm9ncmFtLnBhY2tlZElucHV0cykge1xuICAgICAgICBpbnB1dCA9IHRleERhdGEuaXNQYWNrZWQgPyB0aGlzLnVucGFja1RlbnNvcihpbnB1dCkgOlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB0aGlzLnBhY2tUZW5zb3IoaW5wdXQpO1xuICAgICAgICBkYXRhVG9EaXNwb3NlLnB1c2goaW5wdXQpO1xuICAgICAgICB0ZXhEYXRhID0gdGhpcy50ZXhEYXRhLmdldChpbnB1dC5kYXRhSWQpO1xuICAgICAgfSBlbHNlIGlmIChcbiAgICAgICAgICB0ZXhEYXRhLmlzUGFja2VkICYmXG4gICAgICAgICAgIXdlYmdsX3V0aWwuaXNSZXNoYXBlRnJlZSh0ZXhEYXRhLnNoYXBlLCBpbnB1dC5zaGFwZSkpIHtcbiAgICAgICAgLy8gVGhpcyBpcyBhIHNwZWNpYWwgY2FzZSB3aGVyZSBhIHRleHR1cmUgZXhpc3RzIGZvciBhIHRlbnNvclxuICAgICAgICAvLyBidXQgdGhlIHNoYXBlcyBhcmUgaW5jb21wYXRpYmxlIChkdWUgdG8gcGFja2luZyBjb25zdHJhaW50cykgYmVjYXVzZVxuICAgICAgICAvLyB0aGUgdGVuc29yIGRpZCBub3QgaGF2ZSBhIGNoYW5jZSB0byBnbyB0aHJvdWdoIHRoZSBwYWNrZWQgcmVzaGFwZVxuICAgICAgICAvLyBzaGFkZXIuIFRoaXMgb25seSBoYXBwZW5zIHdoZW4gd2UgcmVzaGFwZSB0aGUgKnNhbWUqIHRlbnNvciB0byBmb3JtXG4gICAgICAgIC8vICpkaXN0aW5jdCogaW5wdXRzIHRvIGFuIG9wLCBlLmcuIGRvdHRpbmcgYSB2ZWN0b3Igd2l0aCBpdHNlbGYuIFRoaXNcbiAgICAgICAgLy8gY2FzZSB3aWxsIGRpc2FwcGVhciBvbmNlIHBhY2tlZCB1cGxvYWRpbmcgaXMgdGhlIGRlZmF1bHQuXG5cbiAgICAgICAgY29uc3Qgc2F2ZWRJbnB1dCA9IGlucHV0O1xuICAgICAgICBjb25zdCB0YXJnZXRTaGFwZSA9IGlucHV0LnNoYXBlO1xuXG4gICAgICAgIGlucHV0LnNoYXBlID0gdGV4RGF0YS5zaGFwZTtcbiAgICAgICAgaW5wdXQgPSB0aGlzLnBhY2tlZFJlc2hhcGUoaW5wdXQgYXMgVGVuc29yLCB0YXJnZXRTaGFwZSk7XG4gICAgICAgIGRhdGFUb0Rpc3Bvc2UucHVzaChpbnB1dCk7XG4gICAgICAgIHRleERhdGEgPSB0aGlzLnRleERhdGEuZ2V0KGlucHV0LmRhdGFJZCk7XG5cbiAgICAgICAgc2F2ZWRJbnB1dC5zaGFwZSA9IHRhcmdldFNoYXBlO1xuICAgICAgfVxuXG4gICAgICByZXR1cm4ge3NoYXBlOiBpbnB1dC5zaGFwZSwgdGV4RGF0YSwgaXNVbmlmb3JtOiBmYWxzZX07XG4gICAgfSk7XG5cbiAgICB0aGlzLnVwbG9hZFRvR1BVKG91dHB1dC5kYXRhSWQpO1xuICAgIGNvbnN0IG91dHB1dERhdGE6XG4gICAgICAgIFRlbnNvckRhdGEgPSB7c2hhcGU6IG91dHB1dC5zaGFwZSwgdGV4RGF0YTogb3V0RGF0YSwgaXNVbmlmb3JtOiBmYWxzZX07XG4gICAgY29uc3Qga2V5ID0gZ3BncHVfbWF0aC5tYWtlU2hhZGVyS2V5KHByb2dyYW0sIGlucHV0c0RhdGEsIG91dHB1dERhdGEpO1xuICAgIGNvbnN0IGJpbmFyeSA9IHRoaXMuZ2V0QW5kU2F2ZUJpbmFyeShrZXksICgpID0+IHtcbiAgICAgIHJldHVybiBncGdwdV9tYXRoLmNvbXBpbGVQcm9ncmFtKFxuICAgICAgICAgIHRoaXMuZ3BncHUsIHByb2dyYW0sIGlucHV0c0RhdGEsIG91dHB1dERhdGEpO1xuICAgIH0pO1xuICAgIGNvbnN0IHNob3VsZFRpbWVQcm9ncmFtID0gdGhpcy5hY3RpdmVUaW1lcnMgIT0gbnVsbDtcbiAgICBsZXQgcXVlcnk6IFdlYkdMUXVlcnl8Q1BVVGltZXJRdWVyeTtcbiAgICBpZiAoc2hvdWxkVGltZVByb2dyYW0pIHtcbiAgICAgIHF1ZXJ5ID0gdGhpcy5zdGFydFRpbWVyKCk7XG4gICAgfVxuXG4gICAgZ3BncHVfbWF0aC5ydW5Qcm9ncmFtKFxuICAgICAgICB0aGlzLmdwZ3B1LCBiaW5hcnksIGlucHV0c0RhdGEsIG91dHB1dERhdGEsIGN1c3RvbVVuaWZvcm1WYWx1ZXMpO1xuXG4gICAgZGF0YVRvRGlzcG9zZS5mb3JFYWNoKGluZm8gPT4gdGhpcy5kaXNwb3NlSW50ZXJtZWRpYXRlVGVuc29ySW5mbyhpbmZvKSk7XG5cbiAgICBpZiAoc2hvdWxkVGltZVByb2dyYW0pIHtcbiAgICAgIHF1ZXJ5ID0gdGhpcy5lbmRUaW1lcihxdWVyeSk7XG4gICAgICB0aGlzLmFjdGl2ZVRpbWVycy5wdXNoKFxuICAgICAgICAgIHtuYW1lOiBwcm9ncmFtLmNvbnN0cnVjdG9yLm5hbWUsIHF1ZXJ5OiB0aGlzLmdldFF1ZXJ5VGltZShxdWVyeSl9KTtcbiAgICB9XG5cbiAgICBjb25zdCBnbEZsdXNoVGhyZXNob2xkID0gZW52KCkuZ2V0KCdXRUJHTF9GTFVTSF9USFJFU0hPTEQnKTtcbiAgICAvLyBNYW51YWxseSBHTCBmbHVzaCByZXF1ZXN0ZWRcbiAgICBpZiAoZ2xGbHVzaFRocmVzaG9sZCA+IDApIHtcbiAgICAgIGNvbnN0IHRpbWUgPSB1dGlsLm5vdygpO1xuICAgICAgaWYgKCh0aW1lIC0gdGhpcy5sYXN0R2xGbHVzaFRpbWUpID4gZ2xGbHVzaFRocmVzaG9sZCkge1xuICAgICAgICB0aGlzLmdwZ3B1LmdsLmZsdXNoKCk7XG4gICAgICAgIHRoaXMubGFzdEdsRmx1c2hUaW1lID0gdGltZTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoIWVudigpLmdldEJvb2woJ1dFQkdMX0xBWklMWV9VTlBBQ0snKSAmJiBvdXREYXRhLmlzUGFja2VkICYmXG4gICAgICAgIHByZXZlbnRFYWdlclVucGFja2luZ09mT3V0cHV0ID09PSBmYWxzZSkge1xuICAgICAgY29uc3QgdW5wYWNrZWQgPSB0aGlzLnVucGFja1RlbnNvcihvdXRwdXQpO1xuICAgICAgdGhpcy5kaXNwb3NlSW50ZXJtZWRpYXRlVGVuc29ySW5mbyhvdXRwdXQpO1xuICAgICAgcmV0dXJuIHVucGFja2VkO1xuICAgIH1cbiAgICByZXR1cm4gb3V0cHV0O1xuICB9XG5cbiAgY29tcGlsZUFuZFJ1bihcbiAgICAgIHByb2dyYW06IEdQR1BVUHJvZ3JhbSwgaW5wdXRzOiBUZW5zb3JJbmZvW10sIG91dHB1dER0eXBlPzogRGF0YVR5cGUsXG4gICAgICBjdXN0b21Vbmlmb3JtVmFsdWVzPzogbnVtYmVyW11bXSxcbiAgICAgIHByZXZlbnRFYWdlclVucGFja2luZ09mT3V0cHV0ID0gZmFsc2UpOiBUZW5zb3JJbmZvIHtcbiAgICBvdXRwdXREdHlwZSA9IG91dHB1dER0eXBlIHx8IGlucHV0c1swXS5kdHlwZTtcbiAgICBjb25zdCBvdXRJbmZvID0gdGhpcy5ydW5XZWJHTFByb2dyYW0oXG4gICAgICAgIHByb2dyYW0sIGlucHV0cywgb3V0cHV0RHR5cGUsIGN1c3RvbVVuaWZvcm1WYWx1ZXMsXG4gICAgICAgIHByZXZlbnRFYWdlclVucGFja2luZ09mT3V0cHV0KTtcbiAgICByZXR1cm4gb3V0SW5mbztcbiAgfVxuXG4gIHByaXZhdGUgZ2V0QW5kU2F2ZUJpbmFyeShrZXk6IHN0cmluZywgZ2V0QmluYXJ5OiAoKSA9PiBHUEdQVUJpbmFyeSk6XG4gICAgICBHUEdQVUJpbmFyeSB7XG4gICAgaWYgKCEoa2V5IGluIHRoaXMuYmluYXJ5Q2FjaGUpKSB7XG4gICAgICB0aGlzLmJpbmFyeUNhY2hlW2tleV0gPSBnZXRCaW5hcnkoKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuYmluYXJ5Q2FjaGVba2V5XTtcbiAgfVxuXG4gIGdldFRleHR1cmVNYW5hZ2VyKCk6IFRleHR1cmVNYW5hZ2VyIHtcbiAgICByZXR1cm4gdGhpcy50ZXh0dXJlTWFuYWdlcjtcbiAgfVxuXG4gIHByaXZhdGUgZGlzcG9zZWQgPSBmYWxzZTtcblxuICBkaXNwb3NlKCkge1xuICAgIGlmICh0aGlzLmRpc3Bvc2VkKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vIEF2b2lkIGRpc3Bvc2luZyB0aGUgY29tcGlsZWQgd2ViZ2wgcHJvZ3JhbXMgZHVyaW5nIHVuaXQgdGVzdGluZyBiZWNhdXNlXG4gICAgLy8gaXQgc2xvd3MgZG93biB0ZXN0IGV4ZWN1dGlvbi5cbiAgICBpZiAoIWVudigpLmdldEJvb2woJ0lTX1RFU1QnKSkge1xuICAgICAgY29uc3QgYWxsS2V5cyA9IE9iamVjdC5rZXlzKHRoaXMuYmluYXJ5Q2FjaGUpO1xuICAgICAgYWxsS2V5cy5mb3JFYWNoKGtleSA9PiB7XG4gICAgICAgIHRoaXMuZ3BncHUuZGVsZXRlUHJvZ3JhbSh0aGlzLmJpbmFyeUNhY2hlW2tleV0ud2ViR0xQcm9ncmFtKTtcbiAgICAgICAgZGVsZXRlIHRoaXMuYmluYXJ5Q2FjaGVba2V5XTtcbiAgICAgIH0pO1xuICAgIH1cbiAgICB0aGlzLnRleHR1cmVNYW5hZ2VyLmRpc3Bvc2UoKTtcbiAgICBpZiAodGhpcy5jYW52YXMgIT0gbnVsbCAmJlxuICAgICAgICAodHlwZW9mIChIVE1MQ2FudmFzRWxlbWVudCkgIT09ICd1bmRlZmluZWQnICYmXG4gICAgICAgICB0aGlzLmNhbnZhcyBpbnN0YW5jZW9mIEhUTUxDYW52YXNFbGVtZW50KSkge1xuICAgICAgdGhpcy5jYW52YXMucmVtb3ZlKCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuY2FudmFzID0gbnVsbDtcbiAgICB9XG4gICAgaWYgKHRoaXMuZ3BncHVDcmVhdGVkTG9jYWxseSkge1xuICAgICAgdGhpcy5ncGdwdS5wcm9ncmFtID0gbnVsbDtcbiAgICAgIHRoaXMuZ3BncHUuZGlzcG9zZSgpO1xuICAgIH1cbiAgICB0aGlzLmRpc3Bvc2VkID0gdHJ1ZTtcbiAgfVxuXG4gIGZsb2F0UHJlY2lzaW9uKCk6IDE2fDMyIHtcbiAgICBpZiAodGhpcy5mbG9hdFByZWNpc2lvblZhbHVlID09IG51bGwpIHtcbiAgICAgIHRoaXMuZmxvYXRQcmVjaXNpb25WYWx1ZSA9IHRpZHkoKCkgPT4ge1xuICAgICAgICBpZiAoIWVudigpLmdldCgnV0VCR0xfUkVOREVSX0ZMT0FUMzJfRU5BQkxFRCcpKSB7XG4gICAgICAgICAgLy8gTW9tZW50YXJpbHkgc3dpdGNoaW5nIERFQlVHIGZsYWcgdG8gZmFsc2Ugc28gd2UgZG9uJ3QgdGhyb3cgYW5cbiAgICAgICAgICAvLyBlcnJvciB0cnlpbmcgdG8gdXBsb2FkIGEgc21hbGwgdmFsdWUuXG4gICAgICAgICAgY29uc3QgZGVidWdGbGFnID0gZW52KCkuZ2V0Qm9vbCgnREVCVUcnKTtcbiAgICAgICAgICBlbnYoKS5zZXQoJ0RFQlVHJywgZmFsc2UpO1xuICAgICAgICAgIGNvbnN0IHVuZGVyZmxvd0NoZWNrVmFsdWUgPSB0aGlzLmFicyhzY2FsYXIoMWUtOCkpLmRhdGFTeW5jKClbMF07XG4gICAgICAgICAgZW52KCkuc2V0KCdERUJVRycsIGRlYnVnRmxhZyk7XG5cbiAgICAgICAgICBpZiAodW5kZXJmbG93Q2hlY2tWYWx1ZSA+IDApIHtcbiAgICAgICAgICAgIHJldHVybiAzMjtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIDE2O1xuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmZsb2F0UHJlY2lzaW9uVmFsdWU7XG4gIH1cblxuICAvKiogUmV0dXJucyB0aGUgc21hbGxlc3QgcmVwcmVzZW50YWJsZSBudW1iZXIuICAqL1xuICBlcHNpbG9uKCk6IG51bWJlciB7XG4gICAgcmV0dXJuIHRoaXMuZmxvYXRQcmVjaXNpb24oKSA9PT0gMzIgPyBFUFNJTE9OX0ZMT0FUMzIgOiBFUFNJTE9OX0ZMT0FUMTY7XG4gIH1cblxuICB1cGxvYWRUb0dQVShkYXRhSWQ6IERhdGFJZCk6IHZvaWQge1xuICAgIGNvbnN0IHRleERhdGEgPSB0aGlzLnRleERhdGEuZ2V0KGRhdGFJZCk7XG4gICAgY29uc3Qge3NoYXBlLCBkdHlwZSwgdmFsdWVzLCB0ZXh0dXJlLCB1c2FnZSwgaXNQYWNrZWR9ID0gdGV4RGF0YTtcblxuICAgIGlmICh0ZXh0dXJlICE9IG51bGwpIHtcbiAgICAgIC8vIEFycmF5IGlzIGFscmVhZHkgb24gR1BVLiBOby1vcC5cbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3Qgc2hvdWxkVGltZVByb2dyYW0gPSB0aGlzLmFjdGl2ZVRpbWVycyAhPSBudWxsO1xuICAgIGxldCBzdGFydDogbnVtYmVyO1xuICAgIGlmIChzaG91bGRUaW1lUHJvZ3JhbSkge1xuICAgICAgc3RhcnQgPSB1dGlsLm5vdygpO1xuICAgIH1cblxuICAgIGxldCB0ZXhTaGFwZSA9IHRleERhdGEudGV4U2hhcGU7XG4gICAgaWYgKHRleFNoYXBlID09IG51bGwpIHtcbiAgICAgIC8vIFRoaXMgdGV4U2hhcGUgbWF5IG5vdCBiZSB0aGUgZmluYWwgdGV4dHVyZSBzaGFwZS4gRm9yIHBhY2tlZCBvciBkZW5zZVxuICAgICAgLy8gdGV4dHVyZXMsIHRoZSB0ZXhTaGFwZSB3aWxsIGJlIGNoYW5nZWQgd2hlbiB0ZXh0dXJlcyBhcmUgY3JlYXRlZC5cbiAgICAgIHRleFNoYXBlID0gd2ViZ2xfdXRpbC5nZXRUZXh0dXJlU2hhcGVGcm9tTG9naWNhbFNoYXBlKHNoYXBlLCBpc1BhY2tlZCk7XG4gICAgICB0ZXhEYXRhLnRleFNoYXBlID0gdGV4U2hhcGU7XG4gICAgfVxuXG4gICAgaWYgKHZhbHVlcyAhPSBudWxsKSB7XG4gICAgICBjb25zdCBzaGFwZUFzM0QgPSB3ZWJnbF91dGlsLmdldFNoYXBlQXMzRChzaGFwZSk7XG5cbiAgICAgIGxldCBwcm9ncmFtO1xuICAgICAgbGV0IHdpZHRoID0gdGV4U2hhcGVbMV0sIGhlaWdodCA9IHRleFNoYXBlWzBdO1xuICAgICAgY29uc3QgaXNCeXRlQXJyYXkgPVxuICAgICAgICAgIHZhbHVlcyBpbnN0YW5jZW9mIFVpbnQ4QXJyYXkgfHwgdmFsdWVzIGluc3RhbmNlb2YgVWludDhDbGFtcGVkQXJyYXk7XG5cbiAgICAgIC8vIHRleHR1cmUgZm9yIGZsb2F0IGFycmF5IGlzIFBoeXNpY2FsVGV4dHVyZVR5cGUuUEFDS0VEXzJYMl9GTE9BVDMyLCB3ZVxuICAgICAgLy8gbmVlZCB0byBtYWtlIHN1cmUgdGhlIHVwbG9hZCB1c2VzIHRoZSBzYW1lIHBhY2tlZCBzaXplXG4gICAgICBpZiAoaXNQYWNrZWQgfHwgIWlzQnl0ZUFycmF5KSB7XG4gICAgICAgIFt3aWR0aCwgaGVpZ2h0XSA9IHRleF91dGlsLmdldFBhY2tlZE1hdHJpeFRleHR1cmVTaGFwZVdpZHRoSGVpZ2h0KFxuICAgICAgICAgICAgdGV4U2hhcGVbMF0sIHRleFNoYXBlWzFdKTtcbiAgICAgIH1cblxuICAgICAgaWYgKGlzUGFja2VkKSB7XG4gICAgICAgIHByb2dyYW0gPSBuZXcgRW5jb2RlTWF0cml4UGFja2VkUHJvZ3JhbShzaGFwZUFzM0QsIGlzQnl0ZUFycmF5KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHByb2dyYW0gPSBuZXcgRW5jb2RlTWF0cml4UHJvZ3JhbShzaGFwZUFzM0QsIGlzQnl0ZUFycmF5KTtcbiAgICAgIH1cblxuICAgICAgLy8gVGV4U2hhcGUgZm9yIGZsb2F0IGFycmF5IG5lZWRzIHRvIGJlIHRoZSBvcmlnaW5hbCBzaGFwZSwgd2hpY2ggYnl0ZVxuICAgICAgLy8gYXJyYXkgbmVlZHMgdG8gYmUgcGFja2VkIHNpemUuIFRoaXMgYWxsb3cgdGhlIGRhdGEgdXBsb2FkIHNoYXBlIHRvIGJlXG4gICAgICAvLyBtYXRjaGVkIHdpdGggdGV4dHVyZSBjcmVhdGlvbiBsb2dpYy5cbiAgICAgIGNvbnN0IHRlbXBEZW5zZUlucHV0VGV4U2hhcGU6IFtudW1iZXIsIG51bWJlcl0gPVxuICAgICAgICAgIGlzQnl0ZUFycmF5ID8gW2hlaWdodCwgd2lkdGhdIDogdGV4U2hhcGU7XG4gICAgICBjb25zdCB0ZW1wRGVuc2VJbnB1dEhhbmRsZSA9XG4gICAgICAgICAgdGhpcy5tYWtlVGVuc29ySW5mbyh0ZW1wRGVuc2VJbnB1dFRleFNoYXBlLCBkdHlwZSk7XG4gICAgICBjb25zdCB0ZW1wRGVuc2VJbnB1dFRleERhdGEgPVxuICAgICAgICAgIHRoaXMudGV4RGF0YS5nZXQodGVtcERlbnNlSW5wdXRIYW5kbGUuZGF0YUlkKTtcbiAgICAgIGlmIChpc0J5dGVBcnJheSkge1xuICAgICAgICB0ZW1wRGVuc2VJbnB1dFRleERhdGEudXNhZ2UgPSBUZXh0dXJlVXNhZ2UuUElYRUxTO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGVtcERlbnNlSW5wdXRUZXhEYXRhLnVzYWdlID0gVGV4dHVyZVVzYWdlLlVQTE9BRDtcbiAgICAgIH1cbiAgICAgIHRlbXBEZW5zZUlucHV0VGV4RGF0YS50ZXhTaGFwZSA9IHRlbXBEZW5zZUlucHV0VGV4U2hhcGU7XG4gICAgICB0aGlzLmdwZ3B1LnVwbG9hZERlbnNlTWF0cml4VG9UZXh0dXJlKFxuICAgICAgICAgIHRoaXMuZ2V0VGV4dHVyZSh0ZW1wRGVuc2VJbnB1dEhhbmRsZS5kYXRhSWQpLCB3aWR0aCwgaGVpZ2h0LFxuICAgICAgICAgIHZhbHVlcyBhcyBUeXBlZEFycmF5KTtcblxuICAgICAgY29uc3QgY3VzdG9tVmFsdWVzID0gW1toZWlnaHQsIHdpZHRoXV07XG4gICAgICAvLyBXZSB3YW50IHRoZSBvdXRwdXQgdG8gcmVtYWluIHBhY2tlZCByZWdhcmRsZXNzIG9mIHRoZSB2YWx1ZSBvZlxuICAgICAgLy8gV0VCR0xfUEFDSy5cbiAgICAgIGNvbnN0IHByZXZlbnRFYWdlclVucGFja2luZyA9IHRydWU7XG4gICAgICBjb25zdCBlbmNvZGVkT3V0cHV0VGFyZ2V0ID0gdGhpcy5ydW5XZWJHTFByb2dyYW0oXG4gICAgICAgICAgcHJvZ3JhbSwgW3RlbXBEZW5zZUlucHV0SGFuZGxlXSwgZHR5cGUsIGN1c3RvbVZhbHVlcyxcbiAgICAgICAgICBwcmV2ZW50RWFnZXJVbnBhY2tpbmcpO1xuXG4gICAgICAvLyBIYXZlIHRoZSBvcmlnaW5hbCB0ZXh0dXJlIGFzc3VtZSB0aGUgaWRlbnRpdHkgb2YgdGhlIGVuY29kZWQgb3V0cHV0LlxuICAgICAgY29uc3Qgb3V0cHV0VGV4RGF0YSA9IHRoaXMudGV4RGF0YS5nZXQoZW5jb2RlZE91dHB1dFRhcmdldC5kYXRhSWQpO1xuICAgICAgdGV4RGF0YS50ZXh0dXJlID0gb3V0cHV0VGV4RGF0YS50ZXh0dXJlO1xuICAgICAgdGV4RGF0YS50ZXhTaGFwZSA9IG91dHB1dFRleERhdGEudGV4U2hhcGU7XG4gICAgICB0ZXhEYXRhLmlzUGFja2VkID0gb3V0cHV0VGV4RGF0YS5pc1BhY2tlZDtcbiAgICAgIHRleERhdGEudXNhZ2UgPSBvdXRwdXRUZXhEYXRhLnVzYWdlO1xuXG4gICAgICB0aGlzLmRpc3Bvc2VJbnRlcm1lZGlhdGVUZW5zb3JJbmZvKHRlbXBEZW5zZUlucHV0SGFuZGxlKTtcbiAgICAgIHRoaXMudGV4RGF0YS5kZWxldGUoZW5jb2RlZE91dHB1dFRhcmdldC5kYXRhSWQpO1xuXG4gICAgICAvLyBPbmNlIHVwbG9hZGVkLCBkb24ndCBzdG9yZSB0aGUgdmFsdWVzIG9uIGNwdS5cbiAgICAgIHRleERhdGEudmFsdWVzID0gbnVsbDtcbiAgICAgIGlmIChzaG91bGRUaW1lUHJvZ3JhbSkge1xuICAgICAgICB0aGlzLnVwbG9hZFdhaXRNcyArPSB1dGlsLm5vdygpIC0gc3RhcnQ7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IG5ld1RleHR1cmUgPSB0aGlzLmFjcXVpcmVUZXh0dXJlKHRleFNoYXBlLCB1c2FnZSwgZHR5cGUsIGlzUGFja2VkKTtcbiAgICAgIHRleERhdGEudGV4dHVyZSA9IG5ld1RleHR1cmU7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjb252ZXJ0QW5kQ2FjaGVPbkNQVShkYXRhSWQ6IERhdGFJZCwgZmxvYXQzMlZhbHVlcz86IEZsb2F0MzJBcnJheSk6XG4gICAgICBUeXBlZEFycmF5IHtcbiAgICBjb25zdCB0ZXhEYXRhID0gdGhpcy50ZXhEYXRhLmdldChkYXRhSWQpO1xuICAgIGNvbnN0IHtkdHlwZX0gPSB0ZXhEYXRhO1xuXG4gICAgdGhpcy5yZWxlYXNlR1BVRGF0YShkYXRhSWQpO1xuXG4gICAgaWYgKGZsb2F0MzJWYWx1ZXMgIT0gbnVsbCkge1xuICAgICAgdGV4RGF0YS52YWx1ZXMgPSBmbG9hdDMyVG9UeXBlZEFycmF5KGZsb2F0MzJWYWx1ZXMsIGR0eXBlIGFzICdmbG9hdDMyJyk7XG4gICAgfVxuICAgIHJldHVybiB0ZXhEYXRhLnZhbHVlcyBhcyBUeXBlZEFycmF5O1xuICB9XG5cbiAgcHJpdmF0ZSBhY3F1aXJlVGV4dHVyZShcbiAgICAgIHRleFNoYXBlOiBbbnVtYmVyLCBudW1iZXJdLCB0ZXhUeXBlOiBUZXh0dXJlVXNhZ2UsIGR0eXBlOiBEYXRhVHlwZSxcbiAgICAgIGlzUGFja2VkOiBib29sZWFuKTogVGV4dHVyZSB7XG4gICAgdGhpcy5udW1CeXRlc0luR1BVICs9IHRoaXMuY29tcHV0ZUJ5dGVzKHRleFNoYXBlLCBkdHlwZSk7XG4gICAgaWYgKCF0aGlzLndhcm5lZEFib3V0TWVtb3J5ICYmXG4gICAgICAgIHRoaXMubnVtQnl0ZXNJbkdQVSA+IHRoaXMubnVtTUJCZWZvcmVXYXJuaW5nICogMTAyNCAqIDEwMjQpIHtcbiAgICAgIGNvbnN0IG1iID0gKHRoaXMubnVtQnl0ZXNJbkdQVSAvIDEwMjQgLyAxMDI0KS50b0ZpeGVkKDIpO1xuICAgICAgdGhpcy53YXJuZWRBYm91dE1lbW9yeSA9IHRydWU7XG4gICAgICBjb25zb2xlLndhcm4oXG4gICAgICAgICAgYEhpZ2ggbWVtb3J5IHVzYWdlIGluIEdQVTogJHttYn0gTUIsIGAgK1xuICAgICAgICAgIGBtb3N0IGxpa2VseSBkdWUgdG8gYSBtZW1vcnkgbGVha2ApO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy50ZXh0dXJlTWFuYWdlci5hY3F1aXJlVGV4dHVyZSh0ZXhTaGFwZSwgdGV4VHlwZSwgaXNQYWNrZWQpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb21wdXRlQnl0ZXMoc2hhcGU6IFtudW1iZXIsIG51bWJlcl0sIGR0eXBlOiBEYXRhVHlwZSkge1xuICAgIHJldHVybiBzaGFwZVswXSAqIHNoYXBlWzFdICogdXRpbC5ieXRlc1BlckVsZW1lbnQoZHR5cGUpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGZsb2F0MzJUb1R5cGVkQXJyYXk8RCBleHRlbmRzIE51bWVyaWNEYXRhVHlwZT4oXG4gICAgYTogRmxvYXQzMkFycmF5LCBkdHlwZTogRCk6IHRmLkRhdGFUeXBlTWFwW0RdIHtcbiAgaWYgKGR0eXBlID09PSAnZmxvYXQzMicgfHwgZHR5cGUgPT09ICdjb21wbGV4NjQnKSB7XG4gICAgcmV0dXJuIGEgYXMgdGYuRGF0YVR5cGVNYXBbRF07XG4gIH0gZWxzZSBpZiAoZHR5cGUgPT09ICdpbnQzMicgfHwgZHR5cGUgPT09ICdib29sJykge1xuICAgIGNvbnN0IHJlc3VsdCA9IChkdHlwZSA9PT0gJ2ludDMyJykgPyBuZXcgSW50MzJBcnJheShhLmxlbmd0aCkgOlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBuZXcgVWludDhBcnJheShhLmxlbmd0aCk7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCByZXN1bHQubGVuZ3RoOyArK2kpIHtcbiAgICAgIHJlc3VsdFtpXSA9IE1hdGgucm91bmQoYVtpXSk7XG4gICAgfVxuICAgIHJldHVybiByZXN1bHQgYXMgdGYuRGF0YVR5cGVNYXBbRF07XG4gIH0gZWxzZSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIGR0eXBlICR7ZHR5cGV9YCk7XG4gIH1cbn1cbiJdfQ==