/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
const contexts = {};
const WEBGL_ATTRIBUTES = {
    alpha: false,
    antialias: false,
    premultipliedAlpha: false,
    preserveDrawingBuffer: false,
    depth: false,
    stencil: false,
    failIfMajorPerformanceCaveat: true
};
export function clearWebGLContext(webGLVersion) {
    delete contexts[webGLVersion];
}
export function setWebGLContext(webGLVersion, gl) {
    contexts[webGLVersion] = gl;
}
export function getWebGLContext(webGLVersion, customCanvas) {
    if (!(webGLVersion in contexts)) {
        const newCtx = getWebGLRenderingContext(webGLVersion, customCanvas);
        if (newCtx !== null) {
            contexts[webGLVersion] = newCtx;
        }
        else {
            console.log('Could not get context for WebGL version', webGLVersion);
            return null;
        }
    }
    const gl = contexts[webGLVersion];
    if (gl == null || gl.isContextLost()) {
        delete contexts[webGLVersion];
        return getWebGLContext(webGLVersion);
    }
    gl.disable(gl.DEPTH_TEST);
    gl.disable(gl.STENCIL_TEST);
    gl.disable(gl.BLEND);
    gl.disable(gl.DITHER);
    gl.disable(gl.POLYGON_OFFSET_FILL);
    gl.disable(gl.SAMPLE_COVERAGE);
    gl.enable(gl.SCISSOR_TEST);
    gl.enable(gl.CULL_FACE);
    gl.cullFace(gl.BACK);
    return contexts[webGLVersion];
}
function createCanvas(webGLVersion) {
    if (typeof OffscreenCanvas !== 'undefined' && webGLVersion === 2) {
        return new OffscreenCanvas(300, 150);
    }
    else if (typeof document !== 'undefined') {
        return document.createElement('canvas');
    }
    else {
        throw new Error('Cannot create a canvas in this context');
    }
}
function getWebGLRenderingContext(webGLVersion, customCanvas) {
    if (webGLVersion !== 1 && webGLVersion !== 2) {
        throw new Error('Cannot get WebGL rendering context, WebGL is disabled.');
    }
    const canvas = customCanvas == null ? createCanvas(webGLVersion) : customCanvas;
    canvas.addEventListener('webglcontextlost', (ev) => {
        ev.preventDefault();
        delete contexts[webGLVersion];
    }, false);
    if (webGLVersion === 1) {
        return (canvas.getContext('webgl', WEBGL_ATTRIBUTES) ||
            canvas.getContext('experimental-webgl', WEBGL_ATTRIBUTES));
    }
    return canvas.getContext('webgl2', WEBGL_ATTRIBUTES);
}
//# sourceMappingURL=data:application/json;base64,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