/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { backend_util, env, util } from '@tensorflow/tfjs-core';
import { ConcatProgram } from '../concat_gpu';
import { ConcatPackedProgram } from '../concat_packed_gpu';
import { concatImplCPU } from '../kernel_utils/shared';
import { complex } from './Complex';
import { imag } from './Imag';
import { real } from './Real';
import { reshape } from './Reshape';
export function concatImpl(inputs, axis, backend) {
    const dtype = inputs[0].dtype;
    if (dtype === 'complex64') {
        const reals = inputs.map((t) => real({ inputs: { input: t }, backend }));
        const imags = inputs.map((t) => imag({ inputs: { input: t }, backend }));
        const realConcated = concatImpl(reals, axis, backend);
        const imagConcated = concatImpl(imags, axis, backend);
        const result = complex({ inputs: { real: realConcated, imag: imagConcated }, backend });
        reals.forEach(r => backend.disposeIntermediateTensorInfo(r));
        imags.forEach(i => backend.disposeIntermediateTensorInfo(i));
        backend.disposeIntermediateTensorInfo(realConcated);
        backend.disposeIntermediateTensorInfo(imagConcated);
        return result;
    }
    let runOnCpu = backend.shouldExecuteOnCPU(inputs);
    // Run on cpu if dtype is string. For string, the backend represents it
    // as Uint8Array[], where each Uint8Array is a character. Given that the
    // computation is only on the outer array, uploading the whole data onto
    // gpu is wasteful. Also, currently webgl doesn't have a design to
    // upload and retrieve Uint8Array[] between cpu and gpu. Therefore, we
    // just run the kernel on cpu if dtype is string.
    if (dtype === 'string') {
        runOnCpu = true;
    }
    if (runOnCpu) {
        // Any concat of n-dimensional tensors across any axis can be reduced to
        // a concatenation of two-dimensional tensors across the axis 1 by first
        // partitioning the axes of the original tensors into those less than the
        // axis to be concatenated and the rest. Then reshape the tensors
        // into a two-dimensional tensor by collapsing these two sets of axes and
        // concatenate the resulting matrices across the axis 1, finally reshaping
        // the result to have the proper shape.
        const tensors2D = inputs.map(t => {
            const innerSize = util.sizeFromShape(t.shape.slice(axis));
            const shape = [-1, innerSize];
            return reshape({ inputs: { x: t }, backend, attrs: { shape } });
        });
        const inputsValShapes = tensors2D.map(t => {
            return { vals: backend.readSync(t.dataId), shape: t.shape };
        });
        // Concats 2d tensors along axis=1.
        const outShape = backend_util.computeOutShape(tensors2D.map(t => t.shape), 1 /* axis */);
        const simplyConcat = tensors2D[0].shape[0] === 1;
        const outVals = concatImplCPU(inputsValShapes, outShape, dtype, simplyConcat);
        const finalOutShape = backend_util.computeOutShape(inputs.map(t => t.shape), axis);
        const outInfo = backend.makeTensorInfo(finalOutShape, dtype, outVals);
        tensors2D.forEach(t => backend.disposeIntermediateTensorInfo(t));
        return outInfo;
    }
    if (inputs.length > env().getNumber('WEBGL_MAX_TEXTURES_IN_SHADER')) {
        const midIndex = Math.floor(inputs.length / 2);
        const leftSide = concatImpl(inputs.slice(0, midIndex), axis, backend);
        const rightSide = concatImpl(inputs.slice(midIndex), axis, backend);
        const result = concatImpl([leftSide, rightSide], axis, backend);
        backend.disposeIntermediateTensorInfo(leftSide);
        backend.disposeIntermediateTensorInfo(rightSide);
        return result;
    }
    if (env().getBool('WEBGL_PACK_ARRAY_OPERATIONS') &&
        inputs[0].shape.length > 1) {
        const program = new ConcatPackedProgram(inputs.map(t => t.shape), axis);
        return backend.runWebGLProgram(program, inputs, dtype);
    }
    const { tensors2D, outShape } = computeTensors2D(inputs, axis, backend);
    const program = new ConcatProgram(tensors2D.map(t => t.shape));
    const result = backend.runWebGLProgram(program, tensors2D, dtype);
    tensors2D.forEach(r => backend.disposeIntermediateTensorInfo(r));
    const reshapedResult = reshape({ inputs: { x: result }, attrs: { shape: outShape }, backend });
    backend.disposeIntermediateTensorInfo(result);
    return reshapedResult;
}
function computeTensors2D(inputs, axis, backend) {
    // Any concat of n-dimensional tensors across any axis can be reduced to
    // a concatenation of two-dimensional tensors across the axis 1 by first
    // partitioning the axes of the original tensors into those less than the
    // axis to be concatenated and the rest. Then reshape the tensors
    // into a two-dimensional tensor by collapsing these two sets of axes and
    // concatenate the resulting matrices across the axis 1, finally reshaping
    // the result to have the proper shape.
    const outShape = backend_util.computeOutShape(inputs.map(t => t.shape), axis);
    const tensors2D = inputs.map(x => reshape({
        inputs: { x },
        attrs: { shape: [-1, util.sizeFromShape(x.shape.slice(axis))] },
        backend
    }));
    return { tensors2D, outShape };
}
//# sourceMappingURL=data:application/json;base64,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