/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { backend_util, env, FusedConv2D, util } from '@tensorflow/tfjs-core';
import { Conv2DProgram } from '../conv_gpu';
import { mapActivationToShaderProgram } from '../kernel_utils/kernel_funcs_utils';
import { conv2dByMatMul, conv2dWithIm2Row } from './Conv2D_impl';
import { reshape } from './Reshape';
export function fusedConv2d(args) {
    const { inputs, backend, attrs } = args;
    const { x, filter, bias, preluActivationWeights } = inputs;
    const { strides, pad, dataFormat, dilations, dimRoundingMode, activation, leakyreluAlpha } = attrs;
    const $dataFormat = backend_util.convertConv2DDataFormat(dataFormat);
    const convInfo = backend_util.computeConv2DInfo(x.shape, filter.shape, strides, dilations, pad, dimRoundingMode, false /* depthwise */, $dataFormat);
    let out;
    const intermediates = [];
    if (convInfo.filterHeight === 1 && convInfo.filterWidth === 1 &&
        convInfo.dilationHeight === 1 && convInfo.dilationWidth === 1 &&
        convInfo.strideHeight === 1 && convInfo.strideWidth === 1 &&
        (convInfo.padInfo.type === 'SAME' || convInfo.padInfo.type === 'VALID')) {
        out = conv2dByMatMul({
            x,
            filter,
            convInfo,
            backend,
            bias,
            activation,
            preluActivationWeights,
            leakyreluAlpha
        });
    }
    else if (env().getBool('WEBGL_CONV_IM2COL') && x.shape[0] === 1) {
        out = conv2dWithIm2Row({
            x,
            filter,
            convInfo,
            backend,
            bias,
            activation,
            preluActivationWeights,
            leakyreluAlpha
        });
    }
    else {
        const hasBias = bias != null;
        const hasPreluActivationWeights = preluActivationWeights != null;
        const hasLeakyreluAlpha = activation === 'leakyrelu';
        const fusedActivation = activation ? mapActivationToShaderProgram(activation, false) : null;
        const program = new Conv2DProgram(convInfo, hasBias, fusedActivation, hasPreluActivationWeights, hasLeakyreluAlpha);
        const inputs = [x, filter];
        if (bias) {
            inputs.push(bias);
        }
        if (preluActivationWeights) {
            inputs.push(preluActivationWeights);
        }
        if (hasLeakyreluAlpha) {
            const $leakyreluAlpha = backend.makeTensorInfo([], 'float32', util.createScalarValue(leakyreluAlpha, 'float32'));
            inputs.push($leakyreluAlpha);
            intermediates.push($leakyreluAlpha);
        }
        out = backend.runWebGLProgram(program, inputs, 'float32');
    }
    const outReshaped = reshape({ inputs: { x: out }, backend, attrs: { shape: convInfo.outShape } });
    intermediates.push(out);
    intermediates.forEach(t => backend.disposeIntermediateTensorInfo(t));
    return outReshaped;
}
export const fusedConv2DConfig = {
    kernelName: FusedConv2D,
    backendName: 'webgl',
    kernelFunc: fusedConv2d,
};
//# sourceMappingURL=data:application/json;base64,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