/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
// Import webgl flags.
import './flags_webgl';
import { backend_util, buffer, DataStorage, engine, env, kernel_impls, KernelBackend, nextFrame, scalar, tidy, util } from '@tensorflow/tfjs-core';
import { getWebGLContext } from './canvas_util';
import { DecodeMatrixProgram } from './decode_matrix_gpu';
import { DecodeMatrixPackedProgram } from './decode_matrix_packed_gpu';
import { EncodeFloatProgram } from './encode_float_gpu';
import { EncodeFloatPackedProgram } from './encode_float_packed_gpu';
import { EncodeMatrixProgram } from './encode_matrix_gpu';
import { EncodeMatrixPackedProgram } from './encode_matrix_packed_gpu';
import { GPGPUContext } from './gpgpu_context';
import * as gpgpu_math from './gpgpu_math';
import { getUniformLocations } from './gpgpu_math';
import { simpleAbsImplCPU } from './kernel_utils/shared';
import { PackProgram } from './pack_gpu';
import { ReshapePackedProgram } from './reshape_packed_gpu';
import * as tex_util from './tex_util';
import { TextureUsage } from './tex_util';
import { TextureManager } from './texture_manager';
import * as unary_op from './unaryop_gpu';
import { UnaryOpProgram } from './unaryop_gpu';
import { UnaryOpPackedProgram } from './unaryop_packed_gpu';
import { UnpackProgram } from './unpack_gpu';
import * as webgl_util from './webgl_util';
const whereImpl = kernel_impls.whereImpl;
export const EPSILON_FLOAT32 = 1e-7;
export const EPSILON_FLOAT16 = 1e-4;
const binaryCaches = {};
export function getBinaryCache(webGLVersion) {
    if (webGLVersion in binaryCaches) {
        return binaryCaches[webGLVersion];
    }
    binaryCaches[webGLVersion] = {};
    return binaryCaches[webGLVersion];
}
// Empirically determined constant used to determine size threshold for handing
// off execution to the CPU.
const CPU_HANDOFF_SIZE_THRESHOLD = env().getNumber('CPU_HANDOFF_SIZE_THRESHOLD');
// Empirically determined constant used to decide the number of MB on GPU
// before we warn about high memory use. The MB are this constant * screen area
// * dpi / 1024 / 1024.
const BEFORE_PAGING_CONSTANT = 600;
function numMBBeforeWarning() {
    if (env().global.screen == null) {
        return 1024; // 1 GB.
    }
    return (env().global.screen.height * env().global.screen.width *
        window.devicePixelRatio) *
        BEFORE_PAGING_CONSTANT / 1024 / 1024;
}
export class MathBackendWebGL extends KernelBackend {
    constructor(gpuResource) {
        super();
        // Maps data ids that have a pending read operation, to list of subscribers.
        this.pendingRead = new WeakMap();
        // List of data ids that are scheduled for disposal, but are waiting on a
        // pending read operation.
        this.pendingDisposal = new WeakSet();
        // Used to count the number of 'shallow' sliced tensors that point to the
        // same data id.
        this.dataRefCount = new WeakMap();
        this.numBytesInGPU = 0;
        // Accumulated time spent (including blocking) in uploading data to webgl.
        this.uploadWaitMs = 0;
        // Accumulated time spent (including blocking in downloading data from webgl.
        this.downloadWaitMs = 0;
        // record the last manual GL Flush time.
        this.lastGlFlushTime = 0;
        this.warnedAboutMemory = false;
        this.pendingDeletes = 0;
        this.disposed = false;
        if (!env().getBool('HAS_WEBGL')) {
            throw new Error('WebGL is not supported on this device');
        }
        let newGPGPU;
        if (gpuResource != null) {
            if (gpuResource instanceof GPGPUContext) {
                newGPGPU = gpuResource;
            }
            else {
                const gl = getWebGLContext(env().getNumber('WEBGL_VERSION'), gpuResource);
                newGPGPU = new GPGPUContext(gl);
            }
            this.binaryCache = {};
            this.gpgpuCreatedLocally = false;
        }
        else {
            const gl = getWebGLContext(env().getNumber('WEBGL_VERSION'));
            newGPGPU = new GPGPUContext(gl);
            this.binaryCache = getBinaryCache(env().getNumber('WEBGL_VERSION'));
            this.gpgpuCreatedLocally = true;
        }
        this.gpgpu = newGPGPU;
        this.canvas = this.gpgpu.gl.canvas;
        this.textureManager = new TextureManager(this.gpgpu);
        this.numMBBeforeWarning = numMBBeforeWarning();
        this.texData = new DataStorage(this, engine());
    }
    nextDataId() {
        return MathBackendWebGL.nextDataId++;
    }
    numDataIds() {
        return this.texData.numDataIds() - this.pendingDeletes;
    }
    // Writes a new entry to the data store with a WebGL texture, and registers it
    // to the texture manager.
    writeTexture(texture, shape, dtype, texHeight, texWidth, channels) {
        // Temporarily create an tensor info to make the texture compatible with
        // the runWebGLProgram's input.
        const input = this.makeTensorInfo(shape, dtype);
        const inData = this.texData.get(input.dataId);
        // Even though the input texture could be unpacked or dense packed, it is
        // always considered as unpacked for EncodeMatrixProgram.
        inData.isPacked = false;
        // Bind texture to the input tensor.
        inData.texture = { texture, texShape: [texHeight, texWidth] };
        inData.texShape = [texHeight, texWidth];
        const shapeAs3D = webgl_util.getShapeAs3D(shape);
        const program = new EncodeMatrixProgram(shapeAs3D, false /* isByteArray */, channels);
        const output = this.runWebGLProgram(program, [input], dtype, [[texHeight, texWidth]]);
        output.shape = shape;
        // Unbind the texture from the input tensor to avoid the texture being
        // released.
        inData.texture = null;
        this.disposeIntermediateTensorInfo(input);
        return output.dataId;
    }
    write(values, shape, dtype) {
        if (env().getBool('WEBGL_CHECK_NUMERICAL_PROBLEMS') ||
            env().getBool('DEBUG')) {
            this.checkNumericalProblems(values);
        }
        if (dtype === 'complex64' && values != null) {
            throw new Error(`Cannot write to a complex64 dtype. ` +
                `Please use tf.complex(real, imag).`);
        }
        const dataId = { id: this.nextDataId() };
        this.texData.set(dataId, { shape, dtype, values, usage: TextureUsage.UPLOAD, refCount: 1 });
        return dataId;
    }
    /** Return refCount of a `TensorData`. */
    refCount(dataId) {
        if (this.texData.has(dataId)) {
            const tensorData = this.texData.get(dataId);
            return tensorData.refCount;
        }
        return 0;
    }
    /** Increase refCount of a `TextureData`. */
    incRef(dataId) {
        const texData = this.texData.get(dataId);
        texData.refCount++;
    }
    /** Decrease refCount of a `TextureData`. */
    decRef(dataId) {
        if (this.texData.has(dataId)) {
            const texData = this.texData.get(dataId);
            texData.refCount--;
        }
    }
    move(dataId, values, shape, dtype, refCount) {
        if (env().getBool('DEBUG')) {
            this.checkNumericalProblems(values);
        }
        if (dtype === 'complex64') {
            throw new Error(`Cannot write to a complex64 dtype. ` +
                `Please use tf.complex(real, imag).`);
        }
        this.texData.set(dataId, { shape, dtype, values, usage: TextureUsage.UPLOAD, refCount });
    }
    disposeIntermediateTensorInfo(tensorInfo) {
        this.disposeData(tensorInfo.dataId);
    }
    readSync(dataId) {
        const texData = this.texData.get(dataId);
        const { values, dtype, complexTensorInfos, slice, shape, isPacked } = texData;
        // The presence of `slice` indicates this tensor is a shallow slice of a
        // different tensor, and is using that original tensor's texture. Run
        // `clone` in order to copy that texture and read from it.
        if (slice != null) {
            let program;
            if (isPacked) {
                program = new UnaryOpPackedProgram(shape, unary_op.CLONE);
            }
            else {
                program = new UnaryOpProgram(shape, unary_op.CLONE);
            }
            const res = this.runWebGLProgram(program, [{ dataId, shape, dtype }], dtype);
            const data = this.readSync(res.dataId);
            this.disposeIntermediateTensorInfo(res);
            return data;
        }
        if (values != null) {
            return this.convertAndCacheOnCPU(dataId);
        }
        if (dtype === 'string') {
            return values;
        }
        const shouldTimeProgram = this.activeTimers != null;
        let start;
        if (shouldTimeProgram) {
            start = util.now();
        }
        let result;
        if (dtype === 'complex64') {
            const realValues = this.readSync(complexTensorInfos.real.dataId);
            const imagValues = this.readSync(complexTensorInfos.imag.dataId);
            result = backend_util.mergeRealAndImagArrays(realValues, imagValues);
        }
        else {
            result = this.getValuesFromTexture(dataId);
        }
        if (shouldTimeProgram) {
            this.downloadWaitMs += util.now() - start;
        }
        return this.convertAndCacheOnCPU(dataId, result);
    }
    async read(dataId) {
        if (this.pendingRead.has(dataId)) {
            const subscribers = this.pendingRead.get(dataId);
            return new Promise(resolve => subscribers.push(resolve));
        }
        const texData = this.texData.get(dataId);
        const { values, shape, slice, dtype, complexTensorInfos, isPacked } = texData;
        // The presence of `slice` indicates this tensor is a shallow slice of a
        // different tensor, and is using that original tensor's texture. Run
        // `clone` in order to copy that texture and read from it.
        if (slice != null) {
            let program;
            if (isPacked) {
                program = new UnaryOpPackedProgram(shape, unary_op.CLONE);
            }
            else {
                program = new UnaryOpProgram(shape, unary_op.CLONE);
            }
            const res = this.runWebGLProgram(program, [{ dataId, shape, dtype }], dtype);
            const data = this.read(res.dataId);
            this.disposeIntermediateTensorInfo(res);
            return data;
        }
        if (values != null) {
            return this.convertAndCacheOnCPU(dataId);
        }
        if (env().getBool('DEBUG')) {
            // getBool('WEBGL_DOWNLOAD_FLOAT_ENABLED') caused a blocking GPU call.
            // For performance reason, only check it for debugging. In production,
            // it doesn't handle this use case anyway, so behavior is not changed.
            if (!env().getBool('WEBGL_DOWNLOAD_FLOAT_ENABLED') &&
                env().getNumber('WEBGL_VERSION') === 2) {
                throw new Error(`tensor.data() with WEBGL_DOWNLOAD_FLOAT_ENABLED=false and ` +
                    `WEBGL_VERSION=2 not yet supported.`);
            }
        }
        let buffer = null;
        let tmpDownloadTarget;
        if (dtype !== 'complex64' && env().get('WEBGL_BUFFER_SUPPORTED')) {
            // Possibly copy the texture into a buffer before inserting a fence.
            tmpDownloadTarget = this.decode(dataId);
            const tmpData = this.texData.get(tmpDownloadTarget.dataId);
            buffer = this.gpgpu.createBufferFromTexture(tmpData.texture.texture, ...tex_util.getDenseTexShape(shape));
        }
        this.pendingRead.set(dataId, []);
        if (dtype !== 'complex64') {
            // Create a fence and wait for it to resolve.
            await this.gpgpu.createAndWaitForFence();
        }
        // Download the values from the GPU.
        let vals;
        if (dtype === 'complex64') {
            const ps = await Promise.all([
                this.read(complexTensorInfos.real.dataId),
                this.read(complexTensorInfos.imag.dataId)
            ]);
            const realValues = ps[0];
            const imagValues = ps[1];
            vals = backend_util.mergeRealAndImagArrays(realValues, imagValues);
        }
        else if (buffer == null) {
            vals = this.getValuesFromTexture(dataId);
        }
        else {
            const size = util.sizeFromShape(shape);
            vals = this.gpgpu.downloadFloat32MatrixFromBuffer(buffer, size);
        }
        if (tmpDownloadTarget != null) {
            this.disposeIntermediateTensorInfo(tmpDownloadTarget);
        }
        if (buffer != null) {
            const gl = this.gpgpu.gl;
            webgl_util.callAndCheck(gl, () => gl.deleteBuffer(buffer));
        }
        const dTypeVals = this.convertAndCacheOnCPU(dataId, vals);
        const subscribers = this.pendingRead.get(dataId);
        this.pendingRead.delete(dataId);
        // Notify all pending reads.
        subscribers.forEach(resolve => resolve(dTypeVals));
        if (this.pendingDisposal.has(dataId)) {
            this.pendingDisposal.delete(dataId);
            if (this.disposeData(dataId)) {
                engine().removeDataId(dataId, this);
            }
            this.pendingDeletes--;
        }
        return dTypeVals;
    }
    /**
     * Read tensor to a new texture that is densely packed for ease of use.
     * @param dataId The source tensor.
     * @param options
     *     customTexShape: Optional. If set, will use the user defined texture
     *     shape to create the texture.
     */
    readToGPU(dataId, options = {}) {
        const texData = this.texData.get(dataId);
        const { values, shape, slice, dtype, isPacked, texture } = texData;
        if (dtype === 'complex64') {
            throw new Error('Does not support reading texture for complex64 dtype.');
        }
        // The presence of `slice` indicates this tensor is a shallow slice of a
        // different tensor, and is using that original tensor's texture. Run
        // `clone` in order to copy that texture and read from it.
        if (slice != null) {
            let program;
            if (isPacked) {
                program = new UnaryOpPackedProgram(shape, unary_op.CLONE);
            }
            else {
                program = new UnaryOpProgram(shape, unary_op.CLONE);
            }
            const res = this.runWebGLProgram(program, [{ dataId, shape, dtype }], dtype);
            const gpuResouorce = this.readToGPU(res, options);
            this.disposeIntermediateTensorInfo(res);
            return gpuResouorce;
        }
        if (texture == null) {
            if (values != null) {
                throw new Error('Data is not on GPU but on CPU.');
            }
            else {
                throw new Error('There is no data on GPU or CPU.');
            }
        }
        // Decode the texture so that it is stored densely (using four channels).
        const tmpTarget = this.decode(dataId, options.customTexShape);
        // Make engine track this tensor, so that we can dispose it later.
        const tensorRef = engine().makeTensorFromTensorInfo(tmpTarget);
        const tmpData = this.texData.get(tmpTarget.dataId);
        return Object.assign({ tensorRef }, tmpData.texture);
    }
    bufferSync(t) {
        const data = this.readSync(t.dataId);
        if (t.dtype === 'string') {
            try {
                // Decode the bytes into string.
                const strings = data.map(d => util.decodeString(d));
                return buffer(t.shape, t.dtype, strings);
            }
            catch (_a) {
                throw new Error('Failed to decode encoded string bytes into utf-8');
            }
        }
        return buffer(t.shape, t.dtype, data);
    }
    checkNumericalProblems(values) {
        if (values == null) {
            return;
        }
        for (let i = 0; i < values.length; i++) {
            const num = values[i];
            if (!webgl_util.canBeRepresented(num)) {
                if (env().getBool('WEBGL_RENDER_FLOAT32_CAPABLE')) {
                    throw Error(`The value ${num} cannot be represented with your ` +
                        `current settings. Consider enabling float32 rendering: ` +
                        `'tf.env().set('WEBGL_RENDER_FLOAT32_ENABLED', true);'`);
                }
                throw Error(`The value ${num} cannot be represented on this device.`);
            }
        }
    }
    getValuesFromTexture(dataId) {
        const { shape, dtype, isPacked } = this.texData.get(dataId);
        const size = util.sizeFromShape(shape);
        if (env().getBool('WEBGL_DOWNLOAD_FLOAT_ENABLED')) {
            const tmpTarget = this.decode(dataId);
            const tmpData = this.texData.get(tmpTarget.dataId);
            const vals = this.gpgpu
                .downloadMatrixFromPackedTexture(tmpData.texture.texture, ...tex_util.getDenseTexShape(shape))
                .subarray(0, size);
            this.disposeIntermediateTensorInfo(tmpTarget);
            return vals;
        }
        const shouldUsePackedProgram = env().getBool('WEBGL_PACK') && isPacked === true;
        const outputShape = shouldUsePackedProgram ? webgl_util.getShapeAs3D(shape) : shape;
        const program = shouldUsePackedProgram ?
            new EncodeFloatPackedProgram(outputShape) :
            new EncodeFloatProgram(outputShape);
        const output = this.runWebGLProgram(program, [{ shape: outputShape, dtype, dataId }], 'float32');
        const tmpData = this.texData.get(output.dataId);
        const vals = this.gpgpu
            .downloadByteEncodedFloatMatrixFromOutputTexture(tmpData.texture.texture, tmpData.texShape[0], tmpData.texShape[1])
            .subarray(0, size);
        this.disposeIntermediateTensorInfo(output);
        return vals;
    }
    timerAvailable() {
        return env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0;
    }
    time(f) {
        const oldActiveTimers = this.activeTimers;
        const newActiveTimers = [];
        let outerMostTime = false;
        if (this.programTimersStack == null) {
            this.programTimersStack = newActiveTimers;
            outerMostTime = true;
        }
        else {
            this.activeTimers.push(newActiveTimers);
        }
        this.activeTimers = newActiveTimers;
        f();
        // needing to split these up because util.flatten only accepts certain types
        const flattenedActiveTimerQueries = util.flatten(this.activeTimers.map((d) => d.query))
            .filter(d => d != null);
        const flattenedActiveTimerNames = util.flatten(this.activeTimers.map((d) => d.name))
            .filter(d => d != null);
        this.activeTimers = oldActiveTimers;
        if (outerMostTime) {
            this.programTimersStack = null;
        }
        const res = {
            uploadWaitMs: this.uploadWaitMs,
            downloadWaitMs: this.downloadWaitMs,
            kernelMs: null,
            wallMs: null // will be filled by the engine
        };
        return (async () => {
            if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') >
                0) {
                const kernelMs = await Promise.all(flattenedActiveTimerQueries);
                res['kernelMs'] = util.sum(kernelMs);
                res['getExtraProfileInfo'] = () => kernelMs
                    .map((d, i) => ({ name: flattenedActiveTimerNames[i], ms: d }))
                    .map(d => `${d.name}: ${d.ms}`)
                    .join(', ');
            }
            else {
                res['kernelMs'] = {
                    error: 'WebGL query timers are not supported in this environment.'
                };
            }
            this.uploadWaitMs = 0;
            this.downloadWaitMs = 0;
            return res;
        })();
    }
    memory() {
        return {
            unreliable: false,
            numBytesInGPU: this.numBytesInGPU,
            numBytesInGPUAllocated: this.textureManager.numBytesAllocated,
            numBytesInGPUFree: this.textureManager.numBytesFree
        };
    }
    startTimer() {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
            return this.gpgpu.beginQuery();
        }
        return { startMs: util.now(), endMs: null };
    }
    endTimer(query) {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
            this.gpgpu.endQuery();
            return query;
        }
        query.endMs = util.now();
        return query;
    }
    async getQueryTime(query) {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE') > 0) {
            return this.gpgpu.waitForQueryAndGetTime(query);
        }
        const timerQuery = query;
        return timerQuery.endMs - timerQuery.startMs;
    }
    /**
     * Decrease the RefCount on the dataId and dispose the memory if the dataId
     * has 0 refCount. If there are pending read on the data, the disposal would
     * added to the pending delete queue. Return true if the dataId is removed
     * from backend or the backend does not contain the dataId, false if the
     * dataId is not removed. Memory may or may not be released even when dataId
     * is removed, which also depends on dataRefCount, see `releaseGPU`.
     * @param dataId
     * @oaram force Optional, remove the data regardless of refCount
     */
    disposeData(dataId, force = false) {
        if (this.pendingDisposal.has(dataId)) {
            return false;
        }
        // No-op if already disposed.
        if (!this.texData.has(dataId)) {
            return true;
        }
        // if force flag is set, change refCount to 0, this would ensure disposal
        // when added to the pendingDisposal queue. Memory may or may not be
        // released, which also depends on dataRefCount, see `releaseGPU`.
        if (force) {
            this.texData.get(dataId).refCount = 0;
        }
        else {
            this.texData.get(dataId).refCount--;
        }
        if (!force && this.texData.get(dataId).refCount > 0) {
            return false;
        }
        if (this.pendingRead.has(dataId)) {
            this.pendingDisposal.add(dataId);
            this.pendingDeletes++;
            return false;
        }
        this.releaseGPUData(dataId);
        const { complexTensorInfos } = this.texData.get(dataId);
        if (complexTensorInfos != null) {
            this.disposeData(complexTensorInfos.real.dataId, force);
            this.disposeData(complexTensorInfos.imag.dataId, force);
        }
        this.texData.delete(dataId);
        return true;
    }
    releaseGPUData(dataId) {
        const { texture, dtype, texShape, usage, isPacked, slice } = this.texData.get(dataId);
        const key = slice && slice.origDataId || dataId;
        const refCount = this.dataRefCount.get(key);
        if (refCount > 1) {
            this.dataRefCount.set(key, refCount - 1);
        }
        else {
            this.dataRefCount.delete(key);
            if (texture != null) {
                this.numBytesInGPU -= this.computeBytes(texShape, dtype);
                this.textureManager.releaseTexture(texture, texShape, usage, isPacked);
            }
        }
        const texData = this.texData.get(dataId);
        texData.texture = null;
        texData.texShape = null;
        texData.isPacked = false;
        texData.slice = null;
    }
    getTexture(dataId) {
        this.uploadToGPU(dataId);
        return this.texData.get(dataId).texture.texture;
    }
    /**
     * Returns internal information for the specific data bucket. Used in unit
     * tests.
     */
    getDataInfo(dataId) {
        return this.texData.get(dataId);
    }
    /*
    Tests whether all the inputs to an op are small and on the CPU. This heuristic
    determines when it would be faster to execute a kernel on the CPU. WebGL
    kernels opt into running this check and forwarding when appropriate.
    TODO(https://github.com/tensorflow/tfjs/issues/872): Develop a more
    sustainable strategy for optimizing backend execution of ops.
     */
    shouldExecuteOnCPU(inputs, sizeThreshold = CPU_HANDOFF_SIZE_THRESHOLD) {
        return env().getBool('WEBGL_CPU_FORWARD') &&
            inputs.every(input => this.texData.get(input.dataId).texture == null &&
                util.sizeFromShape(input.shape) < sizeThreshold);
    }
    getGPGPUContext() {
        return this.gpgpu;
    }
    where(condition) {
        backend_util.warn('tf.where() in webgl locks the UI thread. ' +
            'Call tf.whereAsync() instead');
        const condVals = condition.dataSync();
        return whereImpl(condition.shape, condVals);
    }
    packedUnaryOp(x, op, dtype) {
        const program = new UnaryOpPackedProgram(x.shape, op);
        const outInfo = this.compileAndRun(program, [x], dtype);
        return engine().makeTensorFromTensorInfo(outInfo);
    }
    // TODO(msoulanille) remove this once the backend has been modularized
    // a copy is needed here to break a circular dependency.
    // Also remove the op from unary_op.
    abs(x) {
        // TODO: handle cases when x is complex.
        if (this.shouldExecuteOnCPU([x]) && x.dtype !== 'complex64') {
            const outValues = simpleAbsImplCPU(this.texData.get(x.dataId).values);
            return this.makeOutput(x.shape, x.dtype, outValues);
        }
        if (env().getBool('WEBGL_PACK_UNARY_OPERATIONS')) {
            return this.packedUnaryOp(x, unary_op.ABS, x.dtype);
        }
        const program = new UnaryOpProgram(x.shape, unary_op.ABS);
        const outInfo = this.compileAndRun(program, [x]);
        return engine().makeTensorFromTensorInfo(outInfo);
    }
    makeTensorInfo(shape, dtype, values) {
        let dataId;
        if (dtype === 'string' && values != null && values.length > 0 &&
            util.isString(values[0])) {
            const encodedValues = values.map(d => util.encodeString(d));
            dataId = this.write(encodedValues, shape, dtype);
        }
        else {
            dataId = this.write(values, shape, dtype);
        }
        this.texData.get(dataId).usage = null;
        return { dataId, shape, dtype };
    }
    makeOutput(shape, dtype, values) {
        return engine().makeTensorFromTensorInfo(this.makeTensorInfo(shape, dtype, values), this);
    }
    unpackTensor(input) {
        const program = new UnpackProgram(input.shape);
        return this.runWebGLProgram(program, [input], input.dtype);
    }
    packTensor(input) {
        const program = new PackProgram(input.shape);
        const preventEagerUnpackingOutput = true;
        return this.runWebGLProgram(program, [input], input.dtype, null /* customUniformValues */, preventEagerUnpackingOutput);
    }
    packedReshape(input, afterShape) {
        const input3DShape = [
            webgl_util.getBatchDim(input.shape),
            ...webgl_util.getRowsCols(input.shape)
        ];
        const input3D = {
            dtype: input.dtype,
            shape: input3DShape,
            dataId: input.dataId
        };
        const afterShapeAs3D = [
            webgl_util.getBatchDim(afterShape), ...webgl_util.getRowsCols(afterShape)
        ];
        const program = new ReshapePackedProgram(afterShapeAs3D, input3DShape);
        const preventEagerUnpackingOfOutput = true;
        const customValues = [input3DShape];
        const output = this.runWebGLProgram(program, [input3D], input.dtype, customValues, preventEagerUnpackingOfOutput);
        return { dataId: output.dataId, shape: afterShape, dtype: output.dtype };
    }
    decode(dataId, customTexShape) {
        const texData = this.texData.get(dataId);
        const { isPacked, shape, dtype } = texData;
        if (customTexShape != null) {
            const size = util.sizeFromShape(shape);
            const texSize = customTexShape[0] * customTexShape[1] * 4;
            util.assert(size <= texSize, () => 'customTexShape is too small. ' +
                'Row * Column * 4 should be equal or larger than the ' +
                'size of the tensor data.');
        }
        const shapeAs3D = webgl_util.getShapeAs3D(shape);
        let program;
        if (isPacked) {
            program = new DecodeMatrixPackedProgram(shapeAs3D);
        }
        else {
            program = new DecodeMatrixProgram(shapeAs3D);
        }
        const preventEagerUnpackingOfOutput = true;
        const customValues = [customTexShape != null ? customTexShape :
                tex_util.getDenseTexShape(shapeAs3D)];
        const out = this.runWebGLProgram(program, [{ shape: shapeAs3D, dtype, dataId }], dtype, customValues, preventEagerUnpackingOfOutput, customTexShape);
        return { dtype, shape, dataId: out.dataId };
    }
    runWebGLProgram(program, inputs, outputDtype, customUniformValues, preventEagerUnpackingOfOutput = false, customTexShape) {
        const output = this.makeTensorInfo(program.outputShape, outputDtype);
        const outData = this.texData.get(output.dataId);
        if (program.packedOutput) {
            outData.isPacked = true;
        }
        if (program.outPackingScheme === tex_util.PackingScheme.DENSE) {
            const texelShape = customTexShape != null ?
                customTexShape :
                tex_util.getDenseTexShape(program.outputShape);
            // For a densely packed output, we explicitly set texShape
            // so it doesn't get assigned later according to our typical packing
            // scheme wherein a single texel can only contain values from adjacent
            // rows/cols.
            outData.texShape = texelShape.map(d => d * 2);
        }
        if (program.outTexUsage != null) {
            outData.usage = program.outTexUsage;
        }
        if (util.sizeFromShape(output.shape) === 0) {
            // Short-circuit the computation since the result is empty (has 0 in its
            // shape).
            outData.values =
                util.getTypedArrayFromDType(output.dtype, 0);
            return output;
        }
        const dataToDispose = [];
        const inputsData = inputs.map(input => {
            if (input.dtype === 'complex64') {
                throw new Error(`GPGPUProgram does not support complex64 input. For complex64 ` +
                    `dtypes, please separate the program into real and imaginary ` +
                    `parts.`);
            }
            let texData = this.texData.get(input.dataId);
            if (texData.texture == null) {
                if (!program.packedInputs &&
                    util.sizeFromShape(input.shape) <=
                        env().getNumber('WEBGL_SIZE_UPLOAD_UNIFORM')) {
                    // Upload small tensors that live on the CPU as uniforms, not as
                    // textures. Do this only when the environment supports 32bit floats
                    // due to problems when comparing 16bit floats with 32bit floats.
                    // TODO(https://github.com/tensorflow/tfjs/issues/821): Make it
                    // possible for packed shaders to sample from uniforms.
                    return {
                        shape: input.shape,
                        texData: null,
                        isUniform: true,
                        uniformValues: texData.values
                    };
                }
                // This ensures that if a packed program's inputs have not yet been
                // uploaded to the GPU, they get uploaded as packed right off the bat.
                if (program.packedInputs) {
                    texData.isPacked = true;
                    texData.shape = input.shape;
                }
            }
            this.uploadToGPU(input.dataId);
            if (!!texData.isPacked !== !!program.packedInputs) {
                input = texData.isPacked ? this.unpackTensor(input) :
                    this.packTensor(input);
                dataToDispose.push(input);
                texData = this.texData.get(input.dataId);
            }
            else if (texData.isPacked &&
                !webgl_util.isReshapeFree(texData.shape, input.shape)) {
                // This is a special case where a texture exists for a tensor
                // but the shapes are incompatible (due to packing constraints) because
                // the tensor did not have a chance to go through the packed reshape
                // shader. This only happens when we reshape the *same* tensor to form
                // *distinct* inputs to an op, e.g. dotting a vector with itself. This
                // case will disappear once packed uploading is the default.
                const savedInput = input;
                const targetShape = input.shape;
                input.shape = texData.shape;
                input = this.packedReshape(input, targetShape);
                dataToDispose.push(input);
                texData = this.texData.get(input.dataId);
                savedInput.shape = targetShape;
            }
            return { shape: input.shape, texData, isUniform: false };
        });
        this.uploadToGPU(output.dataId);
        const outputData = { shape: output.shape, texData: outData, isUniform: false };
        const key = gpgpu_math.makeShaderKey(program, inputsData, outputData);
        const binary = this.getAndSaveBinary(key, () => {
            return gpgpu_math.compileProgram(this.gpgpu, program, inputsData, outputData);
        });
        const shouldTimeProgram = this.activeTimers != null;
        let query;
        if (shouldTimeProgram) {
            query = this.startTimer();
        }
        if (!env().get('ENGINE_COMPILE_ONLY')) {
            gpgpu_math.runProgram(this.gpgpu, binary, inputsData, outputData, customUniformValues);
        }
        dataToDispose.forEach(info => this.disposeIntermediateTensorInfo(info));
        if (shouldTimeProgram) {
            query = this.endTimer(query);
            this.activeTimers.push({ name: program.constructor.name, query: this.getQueryTime(query) });
        }
        const glFlushThreshold = env().get('WEBGL_FLUSH_THRESHOLD');
        // Manually GL flush requested
        if (glFlushThreshold > 0) {
            const time = util.now();
            if ((time - this.lastGlFlushTime) > glFlushThreshold) {
                this.gpgpu.gl.flush();
                this.lastGlFlushTime = time;
            }
        }
        if (!env().getBool('WEBGL_LAZILY_UNPACK') && outData.isPacked &&
            preventEagerUnpackingOfOutput === false) {
            const unpacked = this.unpackTensor(output);
            this.disposeIntermediateTensorInfo(output);
            return unpacked;
        }
        return output;
    }
    compileAndRun(program, inputs, outputDtype, customUniformValues, preventEagerUnpackingOfOutput = false) {
        outputDtype = outputDtype || inputs[0].dtype;
        const outInfo = this.runWebGLProgram(program, inputs, outputDtype, customUniformValues, preventEagerUnpackingOfOutput);
        return outInfo;
    }
    getAndSaveBinary(key, getBinary) {
        if (!(key in this.binaryCache)) {
            this.binaryCache[key] = getBinary();
        }
        return this.binaryCache[key];
    }
    getTextureManager() {
        return this.textureManager;
    }
    dispose() {
        if (this.disposed) {
            return;
        }
        // Avoid disposing the compiled webgl programs during unit testing because
        // it slows down test execution.
        if (!env().getBool('IS_TEST')) {
            const allKeys = Object.keys(this.binaryCache);
            allKeys.forEach(key => {
                this.gpgpu.deleteProgram(this.binaryCache[key].webGLProgram);
                delete this.binaryCache[key];
            });
        }
        this.textureManager.dispose();
        if (this.canvas != null &&
            (typeof (HTMLCanvasElement) !== 'undefined' &&
                this.canvas instanceof HTMLCanvasElement)) {
            this.canvas.remove();
        }
        else {
            this.canvas = null;
        }
        if (this.gpgpuCreatedLocally) {
            this.gpgpu.program = null;
            this.gpgpu.dispose();
        }
        this.disposed = true;
    }
    floatPrecision() {
        if (this.floatPrecisionValue == null) {
            this.floatPrecisionValue = tidy(() => {
                if (!env().get('WEBGL_RENDER_FLOAT32_ENABLED')) {
                    // Momentarily switching DEBUG flag to false so we don't throw an
                    // error trying to upload a small value.
                    const debugFlag = env().getBool('DEBUG');
                    env().set('DEBUG', false);
                    const underflowCheckValue = this.abs(scalar(1e-8)).dataSync()[0];
                    env().set('DEBUG', debugFlag);
                    if (underflowCheckValue > 0) {
                        return 32;
                    }
                }
                return 16;
            });
        }
        return this.floatPrecisionValue;
    }
    /** Returns the smallest representable number.  */
    epsilon() {
        return this.floatPrecision() === 32 ? EPSILON_FLOAT32 : EPSILON_FLOAT16;
    }
    uploadToGPU(dataId) {
        const texData = this.texData.get(dataId);
        const { shape, dtype, values, texture, usage, isPacked } = texData;
        if (texture != null) {
            // Array is already on GPU. No-op.
            return;
        }
        const shouldTimeProgram = this.activeTimers != null;
        let start;
        if (shouldTimeProgram) {
            start = util.now();
        }
        let texShape = texData.texShape;
        if (texShape == null) {
            // This texShape may not be the final texture shape. For packed or dense
            // textures, the texShape will be changed when textures are created.
            texShape = webgl_util.getTextureShapeFromLogicalShape(shape, isPacked);
            texData.texShape = texShape;
        }
        if (values != null) {
            const shapeAs3D = webgl_util.getShapeAs3D(shape);
            let program;
            let width = texShape[1], height = texShape[0];
            const isByteArray = values instanceof Uint8Array || values instanceof Uint8ClampedArray;
            // texture for float array is PhysicalTextureType.PACKED_2X2_FLOAT32, we
            // need to make sure the upload uses the same packed size
            if (isPacked || !isByteArray) {
                [width, height] = tex_util.getPackedMatrixTextureShapeWidthHeight(texShape[0], texShape[1]);
            }
            if (isPacked) {
                program = new EncodeMatrixPackedProgram(shapeAs3D, isByteArray);
            }
            else {
                program = new EncodeMatrixProgram(shapeAs3D, isByteArray);
            }
            // TexShape for float array needs to be the original shape, which byte
            // array needs to be packed size. This allow the data upload shape to be
            // matched with texture creation logic.
            const tempDenseInputTexShape = isByteArray ? [height, width] : texShape;
            const tempDenseInputHandle = this.makeTensorInfo(tempDenseInputTexShape, dtype);
            const tempDenseInputTexData = this.texData.get(tempDenseInputHandle.dataId);
            if (isByteArray) {
                tempDenseInputTexData.usage = TextureUsage.PIXELS;
            }
            else {
                tempDenseInputTexData.usage = TextureUsage.UPLOAD;
            }
            tempDenseInputTexData.texShape = tempDenseInputTexShape;
            this.gpgpu.uploadDenseMatrixToTexture(this.getTexture(tempDenseInputHandle.dataId), width, height, values);
            const customValues = [[height, width]];
            // We want the output to remain packed regardless of the value of
            // WEBGL_PACK.
            const preventEagerUnpacking = true;
            const encodedOutputTarget = this.runWebGLProgram(program, [tempDenseInputHandle], dtype, customValues, preventEagerUnpacking);
            // Have the original texture assume the identity of the encoded output.
            const outputTexData = this.texData.get(encodedOutputTarget.dataId);
            texData.texShape = outputTexData.texShape;
            texData.isPacked = outputTexData.isPacked;
            texData.usage = outputTexData.usage;
            if (!env().get('ENGINE_COMPILE_ONLY')) {
                texData.texture = outputTexData.texture;
                // Once uploaded, don't store the values on cpu.
                texData.values = null;
                this.texData.delete(encodedOutputTarget.dataId);
            }
            else {
                this.disposeData(encodedOutputTarget.dataId);
            }
            this.disposeIntermediateTensorInfo(tempDenseInputHandle);
            if (shouldTimeProgram) {
                this.uploadWaitMs += util.now() - start;
            }
        }
        else {
            const newTexture = this.acquireTexture(texShape, usage, dtype, isPacked);
            texData.texture = newTexture;
        }
    }
    convertAndCacheOnCPU(dataId, float32Values) {
        const texData = this.texData.get(dataId);
        const { dtype } = texData;
        this.releaseGPUData(dataId);
        if (float32Values != null) {
            texData.values = float32ToTypedArray(float32Values, dtype);
        }
        return texData.values;
    }
    acquireTexture(texShape, texType, dtype, isPacked) {
        this.numBytesInGPU += this.computeBytes(texShape, dtype);
        if (!this.warnedAboutMemory &&
            this.numBytesInGPU > this.numMBBeforeWarning * 1024 * 1024) {
            const mb = (this.numBytesInGPU / 1024 / 1024).toFixed(2);
            this.warnedAboutMemory = true;
            console.warn(`High memory usage in GPU: ${mb} MB, ` +
                `most likely due to a memory leak`);
        }
        return this.textureManager.acquireTexture(texShape, texType, isPacked);
    }
    computeBytes(shape, dtype) {
        return shape[0] * shape[1] * util.bytesPerElement(dtype);
    }
    checkCompileCompletion() {
        for (const [, binary] of Object.entries(this.binaryCache)) {
            this.checkCompletion_(binary);
        }
    }
    async checkCompileCompletionAsync() {
        const ps = [];
        if (this.gpgpu.parallelCompilationExtension) {
            for (const [, binary] of Object.entries(this.binaryCache)) {
                ps.push(this.checkCompletionAsync_(binary));
            }
            return Promise.all(ps);
        }
        else {
            for (const [, binary] of Object.entries(this.binaryCache)) {
                const p = new Promise((resolve) => {
                    try {
                        this.checkCompletion_(binary);
                        resolve(true);
                    }
                    catch (error) {
                        throw error;
                    }
                });
                ps.push(p);
            }
            return Promise.all(ps);
        }
    }
    async checkCompletionAsync_(binary) {
        if (this.gpgpu.gl.getProgramParameter(binary.webGLProgram, this.gpgpu.parallelCompilationExtension.COMPLETION_STATUS_KHR)) {
            return this.checkCompletion_(binary);
        }
        else {
            await nextFrame();
            return this.checkCompletionAsync_(binary);
        }
    }
    checkCompletion_(binary) {
        if (this.gpgpu.gl.getProgramParameter(binary.webGLProgram, this.gpgpu.gl.LINK_STATUS) === false) {
            console.log(this.gpgpu.gl.getProgramInfoLog(binary.webGLProgram));
            if (this.gpgpu.gl.getShaderParameter(binary.fragmentShader, this.gpgpu.gl.COMPILE_STATUS) === false) {
                webgl_util.logShaderSourceAndInfoLog(binary.source, this.gpgpu.gl.getShaderInfoLog(binary.fragmentShader));
                throw new Error('Failed to compile fragment shader.');
            }
            throw new Error('Failed to link vertex and fragment shaders.');
        }
        return true;
    }
    getUniformLocations() {
        for (const [, binary] of Object.entries(this.binaryCache)) {
            const { uniformLocations, customUniformLocations, infLoc, nanLoc, inShapesLocations, inTexShapesLocations, outShapeLocation, outShapeStridesLocation, outTexShapeLocation } = getUniformLocations(this.gpgpu, binary.program, binary.webGLProgram);
            binary.uniformLocations = uniformLocations;
            binary.customUniformLocations = customUniformLocations;
            binary.infLoc = infLoc;
            binary.nanLoc = nanLoc;
            binary.inShapesLocations = inShapesLocations;
            binary.inTexShapesLocations = inTexShapesLocations;
            binary.outShapeLocation = outShapeLocation;
            binary.outShapeStridesLocation = outShapeStridesLocation;
            binary.outTexShapeLocation = outTexShapeLocation;
        }
    }
    /**
     * Create a TF.js tensor out of an existing WebGL texture. A new texture will
     * be created.
     */
    createTensorFromTexture(values, shape, dtype) {
        const { texture, height, width, channels } = values;
        const backend = engine().backend;
        // Have to throw an error, otherwise WebGL just warns and returns wrong
        // values.
        if (!backend.gpgpu.gl.isTexture(texture)) {
            throw new Error(`The texture is invalid. Also, please make sure the texture and ` +
                `the TFJS WebGL backend are using the same canvas. If you want to ` +
                `use your own custom canvas, you have to create and use the custom ` +
                `TFJS WebGL backend created from the canvas through ` +
                `'new tf.MathBackendWebGL(customCanvas)'.`);
        }
        const dataId = backend.writeTexture(texture, shape, dtype, height, width, channels);
        return engine().makeTensorFromDataId(dataId, shape, dtype, backend);
    }
}
MathBackendWebGL.nextDataId = 0;
function float32ToTypedArray(a, dtype) {
    if (dtype === 'float32' || dtype === 'complex64') {
        return a;
    }
    else if (dtype === 'int32' || dtype === 'bool') {
        const result = (dtype === 'int32') ? new Int32Array(a.length) :
            new Uint8Array(a.length);
        for (let i = 0; i < result.length; ++i) {
            result[i] = Math.round(a[i]);
        }
        return result;
    }
    else {
        throw new Error(`Unknown dtype ${dtype}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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