/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env } from '@tensorflow/tfjs-core';
const contexts = {};
const WEBGL_ATTRIBUTES = {
    alpha: false,
    antialias: false,
    premultipliedAlpha: false,
    preserveDrawingBuffer: false,
    depth: false,
    stencil: false,
    failIfMajorPerformanceCaveat: true
};
export function clearWebGLContext(webGLVersion) {
    delete contexts[webGLVersion];
}
export function setWebGLContext(webGLVersion, gl) {
    contexts[webGLVersion] = gl;
}
export function getWebGLContext(webGLVersion, customCanvas) {
    if (!(webGLVersion in contexts) || customCanvas != null) {
        const newCtx = getWebGLRenderingContext(webGLVersion, customCanvas);
        if (newCtx !== null) {
            contexts[webGLVersion] = newCtx;
        }
        else {
            console.log('Could not get context for WebGL version', webGLVersion);
            return null;
        }
    }
    const gl = contexts[webGLVersion];
    if (gl == null || gl.isContextLost()) {
        delete contexts[webGLVersion];
        return getWebGLContext(webGLVersion);
    }
    gl.disable(gl.DEPTH_TEST);
    gl.disable(gl.STENCIL_TEST);
    gl.disable(gl.BLEND);
    gl.disable(gl.DITHER);
    gl.disable(gl.POLYGON_OFFSET_FILL);
    gl.disable(gl.SAMPLE_COVERAGE);
    gl.enable(gl.SCISSOR_TEST);
    gl.enable(gl.CULL_FACE);
    gl.cullFace(gl.BACK);
    return contexts[webGLVersion];
}
function createCanvas(webGLVersion) {
    if (typeof OffscreenCanvas !== 'undefined' && webGLVersion === 2) {
        return new OffscreenCanvas(300, 150);
    }
    else if (typeof document !== 'undefined') {
        return document.createElement('canvas');
    }
    else {
        throw new Error('Cannot create a canvas in this context');
    }
}
function getWebGLRenderingContext(webGLVersion, customCanvas) {
    if (webGLVersion !== 1 && webGLVersion !== 2) {
        throw new Error('Cannot get WebGL rendering context, WebGL is disabled.');
    }
    const canvas = customCanvas == null ? createCanvas(webGLVersion) : customCanvas;
    canvas.addEventListener('webglcontextlost', (ev) => {
        ev.preventDefault();
        delete contexts[webGLVersion];
    }, false);
    if (env().getBool('SOFTWARE_WEBGL_ENABLED')) {
        WEBGL_ATTRIBUTES.failIfMajorPerformanceCaveat = false;
    }
    if (webGLVersion === 1) {
        return (canvas.getContext('webgl', WEBGL_ATTRIBUTES) ||
            canvas
                .getContext('experimental-webgl', WEBGL_ATTRIBUTES));
    }
    return canvas.getContext('webgl2', WEBGL_ATTRIBUTES);
}
//# sourceMappingURL=data:application/json;base64,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