/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { device_util, env } from '@tensorflow/tfjs-core';
import { getMaxTexturesInShader, getWebGLDisjointQueryTimerVersion, getWebGLMaxTextureSize, isCapableOfRenderingToFloatTexture, isDownloadFloatTextureEnabled, isWebGLFenceEnabled, isWebGLVersionEnabled } from './webgl_util';
const ENV = env();
/**
 * This file contains WebGL-specific flag registrations.
 */
/**
 * True if WebGL is supported.
 */
ENV.registerFlag('HAS_WEBGL', () => ENV.getNumber('WEBGL_VERSION') > 0);
/** 0: No WebGL, 1: WebGL 1.0, 2: WebGL 2.0. */
ENV.registerFlag('WEBGL_VERSION', () => {
    if (isWebGLVersionEnabled(2)) {
        return 2;
    }
    else if (isWebGLVersionEnabled(1)) {
        return 1;
    }
    return 0;
});
/** Whether to check for numerical representation problems. */
ENV.registerFlag('WEBGL_CHECK_NUMERICAL_PROBLEMS', () => false);
ENV.registerFlag('WEBGL_BUFFER_SUPPORTED', () => ENV.get('WEBGL_VERSION') === 2);
/** Whether the WebGL backend will sometimes forward ops to the CPU. */
ENV.registerFlag('WEBGL_CPU_FORWARD', () => true);
/** Whether the WebGL backend will always use f16 textures for rendering. */
ENV.registerFlag('WEBGL_FORCE_F16_TEXTURES', () => false);
/** Whether to turn all packing related flags on. */
ENV.registerFlag('WEBGL_PACK', () => ENV.getBool('HAS_WEBGL'));
/** Whether we will pack the batchnormalization op. */
ENV.registerFlag('WEBGL_PACK_NORMALIZATION', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack the clip op. */
ENV.registerFlag('WEBGL_PACK_CLIP', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack the depthwise conv op. */
ENV.registerFlag('WEBGL_PACK_DEPTHWISECONV', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack binary ops. */
ENV.registerFlag('WEBGL_PACK_BINARY_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack unary ops. */
ENV.registerFlag('WEBGL_PACK_UNARY_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack array ops. */
ENV.registerFlag('WEBGL_PACK_ARRAY_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack image ops. */
ENV.registerFlag('WEBGL_PACK_IMAGE_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack reduce ops. */
ENV.registerFlag('WEBGL_PACK_REDUCE', () => ENV.getBool('WEBGL_PACK'));
/** Whether packed WebGL kernels lazily unpack their outputs. */
ENV.registerFlag('WEBGL_LAZILY_UNPACK', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will use the im2col algorithm to speed up convolutions. */
ENV.registerFlag('WEBGL_CONV_IM2COL', () => ENV.getBool('WEBGL_PACK'));
/** The maximum texture dimension. */
ENV.registerFlag('WEBGL_MAX_TEXTURE_SIZE', () => getWebGLMaxTextureSize(ENV.getNumber('WEBGL_VERSION')));
/** The maximum texture dimension. */
ENV.registerFlag('WEBGL_MAX_TEXTURES_IN_SHADER', () => getMaxTexturesInShader(ENV.getNumber('WEBGL_VERSION')));
/**
 * The disjoint_query_timer extension version.
 * 0: disabled, 1: EXT_disjoint_timer_query, 2:
 * EXT_disjoint_timer_query_webgl2.
 * In Firefox with WebGL 2.0,
 * EXT_disjoint_timer_query_webgl2 is not available, so we must use the
 * WebGL 1.0 extension.
 */
ENV.registerFlag('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION', () => {
    const webGLVersion = ENV.getNumber('WEBGL_VERSION');
    if (webGLVersion === 0) {
        return 0;
    }
    return getWebGLDisjointQueryTimerVersion(webGLVersion);
});
/**
 * Whether the timer object from the disjoint_query_timer extension gives
 * timing information that is reliable.
 */
ENV.registerFlag('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE', () => ENV.getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION') > 0 &&
    !device_util.isMobile());
/**
 * Whether the device is physically capable of rendering to float32 textures.
 */
ENV.registerFlag('WEBGL_RENDER_FLOAT32_CAPABLE', () => isCapableOfRenderingToFloatTexture(ENV.getNumber('WEBGL_VERSION')));
/**
 * Whether rendering to float32 textures is enabled. If disabled, renders to
 * float16 textures.
 */
ENV.registerFlag('WEBGL_RENDER_FLOAT32_ENABLED', () => {
    return ENV.getBool('WEBGL_FORCE_F16_TEXTURES') ?
        false :
        ENV.getBool('WEBGL_RENDER_FLOAT32_CAPABLE');
});
/**
 * Whether downloading float textures is enabled (16 or 32 bit). If disabled,
 * uses IEEE 754 encoding of the float32 values to 4 uint8 when downloading.
 */
ENV.registerFlag('WEBGL_DOWNLOAD_FLOAT_ENABLED', () => isDownloadFloatTextureEnabled(ENV.getNumber('WEBGL_VERSION')));
/** Whether the fence API is available. */
ENV.registerFlag('WEBGL_FENCE_API_ENABLED', () => isWebGLFenceEnabled(ENV.getNumber('WEBGL_VERSION')));
/**
 * Tensors with size <= than this will be uploaded as uniforms, not textures.
 */
ENV.registerFlag('WEBGL_SIZE_UPLOAD_UNIFORM', () => {
    // Use uniform uploads only when 32bit floats are supported. In
    // 16bit
    // environments there are problems with comparing a 16bit texture value
    // with a 32bit uniform value.
    const useUniforms = ENV.getBool('WEBGL_RENDER_FLOAT32_ENABLED');
    return useUniforms ? 4 : 0;
});
/**
 * If the total number of bytes allocated on the GPU is greater than this
 * number, we will aggressively delete textures upon disposal with
 * gl.deleteMatrixTexture, rather than making them available for reuse.
 *
 * Default value -1 indicates that we will never aggressively delete textures.
 */
ENV.registerFlag('WEBGL_DELETE_TEXTURE_THRESHOLD', () => {
    return -1;
}, threshold => {
    if (threshold < 0 && threshold !== -1) {
        throw new Error(`WEBGL_DELETE_TEXTURE_THRESHOLD must be -1 (indicating never ` +
            `delete) or at least 0, but got ${threshold}.`);
    }
});
/**
 * Trigger a manual GL command flush if the threshold of time has passed since
 * previous Kernel execution. This can be useful for Andorid device where GL
 * command flush are delayed un til the end of javascript task. This value is
 * measured in millisecond. Typically you want to set this value to close to 1.
 *
 * Default value 1 for mobile chrome, and -1 for rest cases. -1 indicates that
 * we will not enforce manual flush and depend on system default flush schedule.
 */
ENV.registerFlag('WEBGL_FLUSH_THRESHOLD', () => {
    return device_util.isMobile() ? 1 : -1;
}, threshold => {
    if (threshold < 0 && threshold !== -1) {
        throw new Error(`WEBGL_FLUSH_THRESHOLD must be -1 (indicating never ` +
            `manual flush) or at least 0, but got ${threshold}.`);
    }
});
/**
 * Threshold for input tensor size that determines whether WebGL backend will
 * delegate computation to CPU.
 *
 * Default value is 128.
 */
ENV.registerFlag('CPU_HANDOFF_SIZE_THRESHOLD', () => 128);
/** Whether we will use shapes uniforms. */
ENV.registerFlag('WEBGL_USE_SHAPES_UNIFORMS', () => false);
/**
 * Threshold for last dimension of input tensor that determines whether
 * WebGL backend for the Top K op will delegate computation to CPU. If input
 * is smaller than threshold then CPU will be used
 *
 * Default value is 100000.
 */
ENV.registerFlag('TOPK_LAST_DIM_CPU_HANDOFF_SIZE_THRESHOLD', () => 100000);
/**
 * Threshold for K that determines whether
 * WebGL backend for the Top K op will delegate computation to CPU. If k
 * is larger than threshold then CPU will be used
 *
 * Default value is 128.
 */
ENV.registerFlag('TOPK_K_CPU_HANDOFF_THRESHOLD', () => 128);
/** Whether we will use the experimental conv op. */
ENV.registerFlag('WEBGL_EXP_CONV', () => false);
/**
 * If the device performance is low or if no hardware GPU is available, whether
 * software WebGL will be used.
 */
ENV.registerFlag('SOFTWARE_WEBGL_ENABLED', () => ENV.getBool('IS_TEST'));
/**
 * For narrow texture (physical height or physical width is 1), if the length of
 * any texture edges exceed the threshold, the texture will be reshaped to be
 * more squarish.
 *
 * This flag is used to help some GPUs that could not provide correct
 * interpolations for long skinny triangles. We found Mali GPU probably has this
 * problem: https://github.com/tensorflow/tfjs/issues/6775.
 */
ENV.registerFlag('WEBGL_MAX_SIZE_FOR_NARROW_TEXTURE', () => Infinity);
/**
 * If the flag is set to true, the max size of the narrow texture will be auto
 * computed and it will be considerred as a threshold to reshape the narrow
 * texture to be more squarish.
 *
 * This flag is used to help some GPUs that could not provide correct
 * interpolations for long skinny triangles. We found Mali GPU probably has this
 * problem: https://github.com/tensorflow/tfjs/issues/6775.
 */
ENV.registerFlag('WEBGL_AUTO_SQUARIFY_NARROW_TEXTURE_SHAPE', () => false);
/**
 * Whether to use the customized isnan. It's only useful for webgl2 since webgl1
 * doesn't have the builtin isnan.
 */
ENV.registerFlag('WEBGL2_ISNAN_CUSTOM', () => false);
//# sourceMappingURL=data:application/json;base64,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