/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env, util } from '@tensorflow/tfjs-core';
import { getWebGLContext, setWebGLContext } from './canvas_util';
import * as gpgpu_util from './gpgpu_util';
import * as tex_util from './tex_util';
import * as webgl_util from './webgl_util';
export class GPGPUContext {
    constructor(gl) {
        this.outputTexture = null;
        this.program = null;
        this.disposed = false;
        this.vertexAttrsAreBound = false;
        this.itemsToPoll = [];
        const glVersion = env().getNumber('WEBGL_VERSION');
        if (gl != null) {
            this.gl = gl;
            setWebGLContext(glVersion, gl);
        }
        else {
            this.gl = getWebGLContext(glVersion);
        }
        // WebGL 2.0 enables texture floats without an extension.
        let COLOR_BUFFER_FLOAT = 'WEBGL_color_buffer_float';
        const COLOR_BUFFER_HALF_FLOAT = 'EXT_color_buffer_half_float';
        this.parallelCompilationExtension =
            this.gl.getExtension('KHR_parallel_shader_compile');
        if (env().getNumber('WEBGL_VERSION') === 1) {
            const TEXTURE_FLOAT = 'OES_texture_float';
            const TEXTURE_HALF_FLOAT = 'OES_texture_half_float';
            this.textureFloatExtension =
                webgl_util.getExtensionOrThrow(this.gl, TEXTURE_FLOAT);
            if (webgl_util.hasExtension(this.gl, TEXTURE_HALF_FLOAT)) {
                this.textureHalfFloatExtension =
                    webgl_util.getExtensionOrThrow(this.gl, TEXTURE_HALF_FLOAT);
            }
            else if (env().get('WEBGL_FORCE_F16_TEXTURES')) {
                throw new Error('GL context does not support half float textures, yet the ' +
                    'environment flag WEBGL_FORCE_F16_TEXTURES is set to true.');
            }
            this.colorBufferFloatExtension = this.gl.getExtension(COLOR_BUFFER_FLOAT);
            if (webgl_util.hasExtension(this.gl, COLOR_BUFFER_HALF_FLOAT)) {
                this.colorBufferHalfFloatExtension =
                    webgl_util.getExtensionOrThrow(this.gl, COLOR_BUFFER_HALF_FLOAT);
            }
            else if (env().get('WEBGL_FORCE_F16_TEXTURES')) {
                throw new Error('GL context does not support color renderable half floats, yet ' +
                    'the environment flag WEBGL_FORCE_F16_TEXTURES is set to true.');
            }
        }
        else {
            COLOR_BUFFER_FLOAT = 'EXT_color_buffer_float';
            if (webgl_util.hasExtension(this.gl, COLOR_BUFFER_FLOAT)) {
                this.colorBufferFloatExtension =
                    this.gl.getExtension(COLOR_BUFFER_FLOAT);
            }
            else if (webgl_util.hasExtension(this.gl, COLOR_BUFFER_HALF_FLOAT)) {
                this.colorBufferHalfFloatExtension =
                    this.gl.getExtension(COLOR_BUFFER_HALF_FLOAT);
            }
            else {
                throw new Error('GL context does not support color renderable floats');
            }
        }
        this.vertexBuffer = gpgpu_util.createVertexBuffer(this.gl);
        this.indexBuffer = gpgpu_util.createIndexBuffer(this.gl);
        this.framebuffer = webgl_util.createFramebuffer(this.gl);
        this.textureConfig =
            tex_util.getTextureConfig(this.gl, this.textureHalfFloatExtension);
    }
    get debug() {
        return env().getBool('DEBUG');
    }
    dispose() {
        if (this.disposed) {
            return;
        }
        if (this.program != null) {
            console.warn('Disposing a GPGPUContext that still has a bound WebGLProgram.' +
                ' This is probably a resource leak, delete the program with ' +
                'GPGPUContext.deleteProgram before disposing.');
        }
        if (this.outputTexture != null) {
            console.warn('Disposing a GPGPUContext that still has a bound output matrix ' +
                'texture.  This is probably a resource leak, delete the output ' +
                'matrix texture with GPGPUContext.deleteMatrixTexture before ' +
                'disposing.');
        }
        const gl = this.gl;
        webgl_util.callAndCheck(gl, () => gl.finish());
        webgl_util.callAndCheck(gl, () => gl.bindFramebuffer(gl.FRAMEBUFFER, null));
        webgl_util.callAndCheck(gl, () => gl.deleteFramebuffer(this.framebuffer));
        webgl_util.callAndCheck(gl, () => gl.bindBuffer(gl.ARRAY_BUFFER, null));
        webgl_util.callAndCheck(gl, () => gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null));
        webgl_util.callAndCheck(gl, () => gl.deleteBuffer(this.indexBuffer));
        this.disposed = true;
    }
    createFloat32MatrixTexture(rows, columns) {
        this.throwIfDisposed();
        return gpgpu_util.createFloat32MatrixTexture(this.gl, rows, columns, this.textureConfig);
    }
    createFloat16MatrixTexture(rows, columns) {
        this.throwIfDisposed();
        return gpgpu_util.createFloat16MatrixTexture(this.gl, rows, columns, this.textureConfig);
    }
    createUnsignedBytesMatrixTexture(rows, columns) {
        this.throwIfDisposed();
        return gpgpu_util.createUnsignedBytesMatrixTexture(this.gl, rows, columns, this.textureConfig);
    }
    uploadPixelDataToTexture(texture, pixels) {
        this.throwIfDisposed();
        gpgpu_util.uploadPixelDataToTexture(this.gl, texture, pixels);
    }
    uploadDenseMatrixToTexture(texture, width, height, data) {
        this.throwIfDisposed();
        gpgpu_util.uploadDenseMatrixToTexture(this.gl, texture, width, height, data, this.textureConfig);
    }
    createFloat16PackedMatrixTexture(rows, columns) {
        this.throwIfDisposed();
        return gpgpu_util.createFloat16PackedMatrixTexture(this.gl, rows, columns, this.textureConfig);
    }
    createPackedMatrixTexture(rows, columns) {
        this.throwIfDisposed();
        return gpgpu_util.createPackedMatrixTexture(this.gl, rows, columns, this.textureConfig);
    }
    deleteMatrixTexture(texture) {
        this.throwIfDisposed();
        if (this.outputTexture === texture) {
            webgl_util.unbindColorTextureFromFramebuffer(this.gl, this.framebuffer);
            this.outputTexture = null;
        }
        webgl_util.callAndCheck(this.gl, () => this.gl.deleteTexture(texture));
    }
    downloadByteEncodedFloatMatrixFromOutputTexture(texture, rows, columns) {
        return this.downloadMatrixDriver(texture, () => gpgpu_util.downloadByteEncodedFloatMatrixFromOutputTexture(this.gl, rows, columns, this.textureConfig));
    }
    downloadPackedMatrixFromBuffer(buffer, batch, rows, columns, physicalRows, physicalCols) {
        return gpgpu_util.downloadPackedMatrixFromBuffer(this.gl, buffer, batch, rows, columns, physicalRows, physicalCols, this.textureConfig);
    }
    downloadFloat32MatrixFromBuffer(buffer, size) {
        return gpgpu_util.downloadFloat32MatrixFromBuffer(this.gl, buffer, size);
    }
    createBufferFromTexture(texture, rows, columns) {
        this.bindTextureToFrameBuffer(texture);
        const result = gpgpu_util.createBufferFromOutputTexture(this.gl, rows, columns, this.textureConfig);
        this.unbindTextureToFrameBuffer();
        return result;
    }
    createAndWaitForFence() {
        const fenceContext = this.createFence(this.gl);
        return this.pollFence(fenceContext);
    }
    createFence(gl) {
        let query;
        let isFencePassed;
        if (env().getBool('WEBGL_FENCE_API_ENABLED')) {
            const gl2 = gl;
            const sync = gl2.fenceSync(gl2.SYNC_GPU_COMMANDS_COMPLETE, 0);
            gl.flush();
            isFencePassed = () => {
                const status = gl2.clientWaitSync(sync, 0, 0);
                return status === gl2.ALREADY_SIGNALED ||
                    status === gl2.CONDITION_SATISFIED;
            };
            query = sync;
        }
        else if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION') > 0) {
            query = this.beginQuery();
            this.endQuery();
            isFencePassed = () => this.isQueryAvailable(query, env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION'));
        }
        else {
            // If we have no way to fence, return true immediately. This will fire in
            // WebGL 1.0 when there is no disjoint query timer. In this case, because
            // the fence passes immediately, we'll immediately ask for a download of
            // the texture, which will cause the UI thread to hang.
            isFencePassed = () => true;
        }
        return { query, isFencePassed };
    }
    downloadMatrixFromPackedTexture(texture, physicalRows, physicalCols) {
        return this.downloadMatrixDriver(texture, () => gpgpu_util.downloadMatrixFromPackedOutputTexture(this.gl, physicalRows, physicalCols));
    }
    createProgram(fragmentShader) {
        this.throwIfDisposed();
        const gl = this.gl;
        if (this.vertexShader == null) {
            this.vertexShader = gpgpu_util.createVertexShader(gl);
        }
        const program = webgl_util.createProgram(gl);
        webgl_util.callAndCheck(gl, () => gl.attachShader(program, this.vertexShader));
        webgl_util.callAndCheck(gl, () => gl.attachShader(program, fragmentShader));
        webgl_util.linkProgram(gl, program);
        if (this.debug) {
            webgl_util.validateProgram(gl, program);
        }
        if (!this.vertexAttrsAreBound) {
            this.setProgram(program);
            this.vertexAttrsAreBound = gpgpu_util.bindVertexProgramAttributeStreams(gl, this.program, this.vertexBuffer);
        }
        return program;
    }
    deleteProgram(program) {
        this.throwIfDisposed();
        if (program === this.program) {
            this.program = null;
        }
        if (program != null) {
            webgl_util.callAndCheck(this.gl, () => this.gl.deleteProgram(program));
        }
    }
    setProgram(program) {
        this.throwIfDisposed();
        this.program = program;
        if ((this.program != null) && this.debug) {
            webgl_util.validateProgram(this.gl, this.program);
        }
        webgl_util.callAndCheck(this.gl, () => this.gl.useProgram(program));
    }
    getUniformLocation(program, uniformName, shouldThrow = true) {
        this.throwIfDisposed();
        if (shouldThrow) {
            return webgl_util.getProgramUniformLocationOrThrow(this.gl, program, uniformName);
        }
        else {
            return webgl_util.getProgramUniformLocation(this.gl, program, uniformName);
        }
    }
    getAttributeLocation(program, attribute) {
        this.throwIfDisposed();
        return webgl_util.callAndCheck(this.gl, () => this.gl.getAttribLocation(program, attribute));
    }
    getUniformLocationNoThrow(program, uniformName) {
        this.throwIfDisposed();
        return this.gl.getUniformLocation(program, uniformName);
    }
    setInputMatrixTexture(inputMatrixTexture, uniformLocation, textureUnit) {
        this.throwIfDisposed();
        this.throwIfNoProgram();
        webgl_util.bindTextureToProgramUniformSampler(this.gl, inputMatrixTexture, uniformLocation, textureUnit);
    }
    setOutputMatrixTexture(outputMatrixTexture, rows, columns) {
        this.setOutputMatrixTextureDriver(outputMatrixTexture, columns, rows);
    }
    setOutputPackedMatrixTexture(outputPackedMatrixTexture, rows, columns) {
        this.throwIfDisposed();
        const [width, height] = tex_util.getPackedMatrixTextureShapeWidthHeight(rows, columns);
        this.setOutputMatrixTextureDriver(outputPackedMatrixTexture, width, height);
    }
    setOutputMatrixWriteRegion(startRow, numRows, startColumn, numColumns) {
        this.setOutputMatrixWriteRegionDriver(startColumn, startRow, numColumns, numRows);
    }
    setOutputPackedMatrixWriteRegion(startRow, numRows, startColumn, numColumns) {
        throw new Error('setOutputPackedMatrixWriteRegion not implemented.');
    }
    debugValidate() {
        if (this.program != null) {
            webgl_util.validateProgram(this.gl, this.program);
        }
        webgl_util.validateFramebuffer(this.gl);
    }
    executeProgram() {
        this.throwIfDisposed();
        this.throwIfNoProgram();
        const gl = this.gl;
        if (this.debug) {
            this.debugValidate();
        }
        webgl_util.callAndCheck(gl, () => gl.drawElements(gl.TRIANGLES, 6, gl.UNSIGNED_SHORT, 0));
    }
    blockUntilAllProgramsCompleted() {
        this.throwIfDisposed();
        webgl_util.callAndCheck(this.gl, () => this.gl.finish());
    }
    getQueryTimerExtension() {
        if (this.disjointQueryTimerExtension == null) {
            this.disjointQueryTimerExtension =
                webgl_util.getExtensionOrThrow(this.gl, env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION') === 2 ?
                    'EXT_disjoint_timer_query_webgl2' :
                    'EXT_disjoint_timer_query');
        }
        return this.disjointQueryTimerExtension;
    }
    getQueryTimerExtensionWebGL2() {
        return this.getQueryTimerExtension();
    }
    getQueryTimerExtensionWebGL1() {
        return this.getQueryTimerExtension();
    }
    beginQuery() {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION') === 2) {
            const gl2 = this.gl;
            const ext = this.getQueryTimerExtensionWebGL2();
            const query = gl2.createQuery();
            gl2.beginQuery(ext.TIME_ELAPSED_EXT, query);
            return query;
        }
        const ext = this.getQueryTimerExtensionWebGL1();
        const query = ext.createQueryEXT();
        ext.beginQueryEXT(ext.TIME_ELAPSED_EXT, query);
        return query;
    }
    endQuery() {
        if (env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION') === 2) {
            const gl2 = this.gl;
            const ext = this.getQueryTimerExtensionWebGL2();
            gl2.endQuery(ext.TIME_ELAPSED_EXT);
            return;
        }
        const ext = this.getQueryTimerExtensionWebGL1();
        ext.endQueryEXT(ext.TIME_ELAPSED_EXT);
    }
    async waitForQueryAndGetTime(query) {
        await util.repeatedTry(() => this.disposed || // while testing contexts are created / disposed
            // in rapid succession, so without this check we
            // may poll for the query timer indefinitely
            this.isQueryAvailable(query, env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION')));
        return this.getQueryTime(query, env().getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION'));
    }
    getQueryTime(query, queryTimerVersion) {
        if (queryTimerVersion === 0) {
            return null;
        }
        if (queryTimerVersion === 2) {
            const gl2 = this.gl;
            const timeElapsedNanos = gl2.getQueryParameter(query, gl2.QUERY_RESULT);
            // Return milliseconds.
            return timeElapsedNanos / 1000000;
        }
        else {
            const ext = this.getQueryTimerExtensionWebGL1();
            const timeElapsedNanos = ext.getQueryObjectEXT(query, ext.QUERY_RESULT_EXT);
            // Return milliseconds.
            return timeElapsedNanos / 1000000;
        }
    }
    isQueryAvailable(query, queryTimerVersion) {
        if (queryTimerVersion === 0) {
            return true;
        }
        if (queryTimerVersion === 2) {
            const gl2 = this.gl;
            const ext = this.getQueryTimerExtensionWebGL2();
            const available = gl2.getQueryParameter(query, gl2.QUERY_RESULT_AVAILABLE);
            if (this.disjoint == null) {
                this.disjoint = this.gl.getParameter(ext.GPU_DISJOINT_EXT);
            }
            return available && !this.disjoint;
        }
        else {
            const ext = this.getQueryTimerExtensionWebGL1();
            const available = ext.getQueryObjectEXT(query, ext.QUERY_RESULT_AVAILABLE_EXT);
            if (this.disjoint == null) {
                this.disjoint = this.gl.getParameter(ext.GPU_DISJOINT_EXT);
            }
            return available && !this.disjoint;
        }
    }
    pollFence(fenceContext) {
        return new Promise(resolve => {
            this.addItemToPoll(() => fenceContext.isFencePassed(), () => resolve());
        });
    }
    pollItems() {
        // Find the last query that has finished.
        const index = linearSearchLastTrue(this.itemsToPoll.map(x => x.isDoneFn));
        for (let i = 0; i <= index; ++i) {
            const { resolveFn } = this.itemsToPoll[i];
            resolveFn();
        }
        this.itemsToPoll = this.itemsToPoll.slice(index + 1);
    }
    addItemToPoll(isDoneFn, resolveFn) {
        this.itemsToPoll.push({ isDoneFn, resolveFn });
        if (this.itemsToPoll.length > 1) {
            // We already have a running loop that polls.
            return;
        }
        // Start a new loop that polls.
        let scheduleFn = undefined;
        if ('setTimeoutCustom' in env().platform) {
            scheduleFn = env().platform.setTimeoutCustom.bind(env().platform);
        }
        util.repeatedTry(() => {
            this.pollItems();
            // End the loop if no more items to poll.
            return this.itemsToPoll.length === 0;
        }, () => 0, null, scheduleFn);
    }
    bindTextureToFrameBuffer(texture) {
        this.throwIfDisposed();
        webgl_util.bindColorTextureToFramebuffer(this.gl, texture, this.framebuffer);
        if (this.debug) {
            webgl_util.validateFramebuffer(this.gl);
        }
    }
    unbindTextureToFrameBuffer() {
        if (this.outputTexture != null) {
            webgl_util.bindColorTextureToFramebuffer(this.gl, this.outputTexture, this.framebuffer);
            if (this.debug) {
                webgl_util.validateFramebuffer(this.gl);
            }
        }
        else {
            webgl_util.unbindColorTextureFromFramebuffer(this.gl, this.framebuffer);
        }
    }
    downloadMatrixDriver(texture, downloadAndDecode) {
        this.bindTextureToFrameBuffer(texture);
        const result = downloadAndDecode();
        this.unbindTextureToFrameBuffer();
        return result;
    }
    setOutputMatrixTextureDriver(outputMatrixTextureMaybePacked, width, height) {
        this.throwIfDisposed();
        const gl = this.gl;
        webgl_util.bindColorTextureToFramebuffer(gl, outputMatrixTextureMaybePacked, this.framebuffer);
        if (this.debug) {
            webgl_util.validateFramebuffer(gl);
        }
        this.outputTexture = outputMatrixTextureMaybePacked;
        webgl_util.callAndCheck(gl, () => gl.viewport(0, 0, width, height));
        webgl_util.callAndCheck(gl, () => gl.scissor(0, 0, width, height));
    }
    setOutputMatrixWriteRegionDriver(x, y, width, height) {
        this.throwIfDisposed();
        webgl_util.callAndCheck(this.gl, () => this.gl.scissor(x, y, width, height));
    }
    throwIfDisposed() {
        if (this.disposed) {
            throw new Error('Attempted to use disposed GPGPUContext.');
        }
    }
    throwIfNoProgram() {
        if (this.program == null) {
            throw new Error('No GPU program is currently set.');
        }
    }
}
/**
 * Finds the index of the last true element using linear search.
 * Note: We can't do binary search because Chrome expects us to explicitly
 * test all fences before download:
 * https://github.com/tensorflow/tfjs/issues/1145
 */
export function linearSearchLastTrue(arr) {
    let i = 0;
    for (; i < arr.length; ++i) {
        const isDone = arr[i]();
        if (!isDone) {
            break;
        }
    }
    return i - 1;
}
//# sourceMappingURL=data:application/json;base64,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