/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { backend_util, env, upcastType } from '@tensorflow/tfjs-core';
import { BinaryOpProgram } from '../binaryop_gpu';
import { BinaryOpPackedProgram } from '../binaryop_packed_gpu';
import { complex } from '../kernels/Complex';
import { LEAKYRELU, LEAKYRELU_PACKED } from '../kernels/LeakyRelu';
import { PRELU, PRELU_PACKED } from '../kernels/Prelu';
import * as unary_op from '../unaryop_gpu';
import { UnaryOpProgram } from '../unaryop_gpu';
import * as unary_packed_op from '../unaryop_packed_gpu';
import { UnaryOpPackedProgram } from '../unaryop_packed_gpu';
export const CHECK_NAN_SNIPPET_UNARY = `if (isnan(x)) return x;`;
/**
 * Template that creates a `KernelFunc` for unary ops.
 * @param opSnippet Op snippet to create `UnaryOpProgram`.
 * @param packedOpSnippet Op snippet to create `UnaryOpPackedProgram`.
 * @param dtype Optional. If set, the result has this dtype. Otherwise, the
 *     result has the same dtype as the first input. This is mainly used in
 *     comparison kernels, such as Equal, Less, Greater, etc.
 */
export function unaryKernelFunc({ opSnippet, packedOpSnippet, cpuKernelImpl, dtype }) {
    return ({ inputs, backend }) => {
        const { x } = inputs;
        const webglBackend = backend;
        const $dtype = dtype || x.dtype;
        if (webglBackend.shouldExecuteOnCPU([x]) && cpuKernelImpl != null) {
            const xData = webglBackend.texData.get(x.dataId);
            const outValues = cpuKernelImpl(xData.values, $dtype);
            return webglBackend.makeTensorInfo(x.shape, $dtype, outValues);
        }
        const shouldUsePackedProgram = env().getBool('WEBGL_PACK_UNARY_OPERATIONS') && packedOpSnippet != null;
        let program;
        if (shouldUsePackedProgram) {
            program = new UnaryOpPackedProgram(x.shape, packedOpSnippet);
        }
        else {
            program = new UnaryOpProgram(x.shape, opSnippet);
        }
        return webglBackend.runWebGLProgram(program, [x], $dtype);
    };
}
/**
 * Template that creates a `KernelFunc` for binary ops.
 * @param opSnippet Op snippet to create `BinaryOpProgram`.
 * @param packedOpSnippet Op snippet to create `BinaryOpPackedProgram`.
 * @param checkOutOfBoundsForPackedProgram Whether to set checkOutOfBounds=true
 *     when creating BinaryOpPackedProgram.
 * @param dtype Optional. If set, the result has this dtype. Otherwise, the
 *     result has the same dtype as the first input. This is mainly used in
 *     comparison kernels, such as Equal, Less, Greater, etc.
 */
export function binaryKernelFunc({ opSnippet, packedOpSnippet, checkOutOfBounds = false, supportsComplex = false, cpuKernelImpl, dtype }) {
    return ({ inputs, backend }) => {
        const { a, b } = inputs;
        const webglBackend = backend;
        if (supportsComplex && a.dtype === 'complex64') {
            const aData = webglBackend.texData.get(a.dataId);
            const bData = webglBackend.texData.get(b.dataId);
            const [real, imag] = [
                [aData.complexTensorInfos.real, bData.complexTensorInfos.real],
                [aData.complexTensorInfos.imag, bData.complexTensorInfos.imag]
            ].map(complexParts => {
                const [aPart, bPart] = complexParts;
                const aHandle = {
                    dataId: aPart.dataId,
                    dtype: aPart.dtype,
                    shape: a.shape
                };
                const bHandle = {
                    dataId: bPart.dataId,
                    dtype: bPart.dtype,
                    shape: b.shape
                };
                const program = new BinaryOpProgram(opSnippet, a.shape, b.shape);
                return webglBackend.runWebGLProgram(program, [aHandle, bHandle], upcastType(aPart.dtype, bPart.dtype));
            });
            const complexOutput = complex({ inputs: { real, imag }, backend: webglBackend });
            webglBackend.disposeIntermediateTensorInfo(real);
            webglBackend.disposeIntermediateTensorInfo(imag);
            // TODO(annxingyuan): Implement CPU forwarding for complex inputs.
            return complexOutput;
        }
        const $dtype = dtype || upcastType(a.dtype, b.dtype);
        if ((a.dtype === 'string' || b.dtype === 'string' ||
            webglBackend.shouldExecuteOnCPU([a, b])) &&
            cpuKernelImpl != null) {
            const aVals = webglBackend.texData.get(a.dataId).values;
            const bVals = webglBackend.texData.get(b.dataId).values;
            const decodedAVals = a.dtype === 'string' ?
                // tslint:disable-next-line: no-any
                backend_util.fromUint8ToStringArray(aVals) :
                aVals;
            const decodedBVals = a.dtype === 'string' ?
                // tslint:disable-next-line: no-any
                backend_util.fromUint8ToStringArray(bVals) :
                bVals;
            const [outValues, outShape] = cpuKernelImpl(a.shape, b.shape, decodedAVals, decodedBVals, $dtype);
            const out = webglBackend.makeTensorInfo(outShape, $dtype);
            const outData = webglBackend.texData.get(out.dataId);
            outData.values = outValues;
            return out;
        }
        const shouldUsePackedProgram = env().getBool('WEBGL_PACK_BINARY_OPERATIONS') &&
            packedOpSnippet != null;
        let program;
        if (shouldUsePackedProgram) {
            program = new BinaryOpPackedProgram(packedOpSnippet, a.shape, b.shape, checkOutOfBounds);
        }
        else {
            program = new BinaryOpProgram(opSnippet, a.shape, b.shape);
        }
        return webglBackend.runWebGLProgram(program, [a, b], $dtype);
    };
}
export function mapActivationToShaderProgram(activation, packed = false) {
    if (activation === 'linear') {
        if (packed) {
            return unary_packed_op.LINEAR;
        }
        return unary_op.LINEAR;
    }
    else if (activation === 'relu') {
        if (packed) {
            return unary_packed_op.RELU;
        }
        return unary_op.RELU;
    }
    else if (activation === 'elu') {
        if (packed) {
            return unary_packed_op.ELU;
        }
        return unary_op.ELU;
    }
    else if (activation === 'relu6') {
        if (packed) {
            return unary_packed_op.RELU6;
        }
        return unary_op.RELU6;
    }
    else if (activation === 'prelu') {
        if (packed) {
            return PRELU_PACKED;
        }
        return PRELU;
    }
    else if (activation === 'leakyrelu') {
        if (packed) {
            return LEAKYRELU_PACKED;
        }
        return LEAKYRELU;
    }
    else if (activation === 'sigmoid') {
        if (packed) {
            return unary_packed_op.SIGMOID;
        }
        return unary_op.SIGMOID;
    }
    throw new Error(`Activation ${activation} has not been implemented for the WebGL backend.`);
}
//# sourceMappingURL=data:application/json;base64,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