/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env, Slice, slice_util, util } from '@tensorflow/tfjs-core';
import { sliceImplCPU } from '../kernel_utils/shared';
import { SliceProgram } from '../slice_gpu';
import { SlicePackedProgram } from '../slice_packed_gpu';
function shallowSlice(x, begin, size, backend) {
    const xTexData = backend.texData.get(x.dataId);
    const t = backend.makeTensorInfo(size, x.dtype);
    const newTexData = backend.texData.get(t.dataId);
    // Copy texture data from the original tensor.
    Object.assign(newTexData, xTexData);
    newTexData.refCount = 1;
    newTexData.shape = size;
    newTexData.dtype = x.dtype;
    let flatOffset = slice_util.computeFlatOffset(begin, util.computeStrides(x.shape));
    if (xTexData.slice) {
        // We are slicing an already sliced tensor, so we have to accumulate
        // the offset.
        flatOffset += xTexData.slice.flatOffset;
    }
    newTexData.slice = {
        flatOffset,
        // Point to the original dataId, which is used to do ref counting.
        origDataId: xTexData.slice && xTexData.slice.origDataId || x.dataId
    };
    // Increase the ref count for that data bucket.
    const refCount = backend.dataRefCount.get(newTexData.slice.origDataId) || 1;
    backend.dataRefCount.set(newTexData.slice.origDataId, refCount + 1);
    return t;
}
export function slice(args) {
    const { inputs, backend, attrs } = args;
    const { x } = inputs;
    const { begin, size } = attrs;
    const [$begin, $size] = slice_util.parseSliceParams(x, begin, size);
    slice_util.assertParamsValid(x, $begin, $size);
    if (util.sizeFromShape($size) === 0) {
        return backend.makeTensorInfo($size, x.dtype, []);
    }
    // Run on cpu if dtype is string. For string, the backend represents it
    // as Uint8Array[], where each Uint8Array is a character. Given that the
    // computation is only on the outer array, uploading the whole data onto
    // gpu is wasteful. Also, currently webgl doesn't have a design to
    // upload and retrieve Uint8Array[] between cpu and gpu. Therefore, we
    // just run the kernel on cpu if dtype is string.
    if (backend.shouldExecuteOnCPU([x]) || x.dtype === 'string') {
        const xTexData = backend.texData.get(x.dataId);
        const outValues = sliceImplCPU(xTexData.values, $begin, $size, x.shape, x.dtype);
        return backend.makeTensorInfo($size, x.dtype, outValues);
    }
    const { isPacked } = backend.texData.get(x.dataId);
    const isContinous = slice_util.isSliceContinous(x.shape, $begin, $size);
    if (isPacked || !isContinous) {
        const program = env().getBool('WEBGL_PACK_ARRAY_OPERATIONS') ?
            new SlicePackedProgram($size) :
            new SliceProgram($size);
        const customValues = [$begin];
        return backend.runWebGLProgram(program, [x], x.dtype, customValues);
    }
    backend.uploadToGPU(x.dataId);
    return shallowSlice(x, $begin, $size, backend);
}
export const sliceConfig = {
    kernelName: Slice,
    backendName: 'webgl',
    kernelFunc: slice
};
//# sourceMappingURL=data:application/json;base64,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