/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * =============================================================================
 */
import { env, tensor, util } from '@tensorflow/tfjs-core';
import { LazyIterator } from './lazy_iterator';
/**
 * Provide a stream of tensors from microphone audio stream. The tensors are
 * representing audio data as frequency-domain spectrogram generated with
 * browser's native FFT. Tensors representing time-domain waveform is available
 * based on configuration. Only works in browser environment.
 */
export class MicrophoneIterator extends LazyIterator {
    constructor(microphoneConfig) {
        super();
        this.microphoneConfig = microphoneConfig;
        this.isClosed = false;
        this.fftSize = microphoneConfig.fftSize || 1024;
        const fftSizeLog2 = Math.log2(this.fftSize);
        if (this.fftSize < 0 || fftSizeLog2 < 4 || fftSizeLog2 > 14 ||
            !Number.isInteger(fftSizeLog2)) {
            throw new Error(`Invalid fftSize: it must be a power of 2 between ` +
                `2 to 4 and 2 to 14, but got ${this.fftSize}`);
        }
        this.numFrames = microphoneConfig.numFramesPerSpectrogram || 43;
        this.sampleRateHz = microphoneConfig.sampleRateHz;
        this.columnTruncateLength =
            microphoneConfig.columnTruncateLength || this.fftSize;
        this.audioTrackConstraints = microphoneConfig.audioTrackConstraints;
        this.smoothingTimeConstant = microphoneConfig.smoothingTimeConstant || 0;
        this.includeSpectrogram =
            microphoneConfig.includeSpectrogram === false ? false : true;
        this.includeWaveform =
            microphoneConfig.includeWaveform === true ? true : false;
        if (!this.includeSpectrogram && !this.includeWaveform) {
            throw new Error('Both includeSpectrogram and includeWaveform are false. ' +
                'At least one type of data should be returned.');
        }
    }
    summary() {
        return `microphone`;
    }
    // Construct a MicrophoneIterator and start the audio stream.
    static async create(microphoneConfig = {}) {
        if (env().get('IS_NODE')) {
            throw new Error('microphone API is only supported in browser environment.');
        }
        const microphoneIterator = new MicrophoneIterator(microphoneConfig);
        // Call async function start() to initialize the audio stream.
        await microphoneIterator.start();
        return microphoneIterator;
    }
    // Start the audio stream and FFT.
    async start() {
        try {
            this.stream = await navigator.mediaDevices.getUserMedia({
                audio: this.audioTrackConstraints == null ? true :
                    this.audioTrackConstraints,
                video: false
            });
        }
        catch (e) {
            throw new Error(`Error thrown while initializing video stream: ${e.message}`);
        }
        if (!this.stream) {
            throw new Error('Could not obtain audio from microphone.');
        }
        const ctxConstructor = 
        // tslint:disable-next-line:no-any
        window.AudioContext || window.webkitAudioContext;
        this.audioContext = new ctxConstructor();
        if (!this.sampleRateHz) {
            // If sample rate is not provided, use the available sample rate on
            // device.
            this.sampleRateHz = this.audioContext.sampleRate;
        }
        else if (this.audioContext.sampleRate !== this.sampleRateHz) {
            throw new Error(`Mismatch in sampling rate: ` +
                `Expected: ${this.sampleRateHz}; ` +
                `Actual: ${this.audioContext.sampleRate}`);
        }
        const streamSource = this.audioContext.createMediaStreamSource(this.stream);
        this.analyser = this.audioContext.createAnalyser();
        this.analyser.fftSize = this.fftSize * 2;
        this.analyser.smoothingTimeConstant = this.smoothingTimeConstant;
        streamSource.connect(this.analyser);
        this.freqData = new Float32Array(this.fftSize);
        this.timeData = new Float32Array(this.fftSize);
        return;
    }
    async next() {
        if (this.isClosed) {
            return { value: null, done: true };
        }
        let spectrogramTensor;
        let waveformTensor;
        const audioDataQueue = await this.getAudioData();
        if (this.includeSpectrogram) {
            const freqData = this.flattenQueue(audioDataQueue.freqDataQueue);
            spectrogramTensor = this.getTensorFromAudioDataArray(freqData, [this.numFrames, this.columnTruncateLength, 1]);
        }
        if (this.includeWaveform) {
            const timeData = this.flattenQueue(audioDataQueue.timeDataQueue);
            waveformTensor = this.getTensorFromAudioDataArray(timeData, [this.numFrames * this.fftSize, 1]);
        }
        return {
            value: { 'spectrogram': spectrogramTensor, 'waveform': waveformTensor },
            done: false
        };
    }
    // Capture one result from the audio stream, and extract the value from
    // iterator.next() result.
    async capture() {
        return (await this.next()).value;
    }
    async getAudioData() {
        const freqDataQueue = [];
        const timeDataQueue = [];
        let currentFrames = 0;
        return new Promise(resolve => {
            const intervalID = setInterval(() => {
                if (this.includeSpectrogram) {
                    this.analyser.getFloatFrequencyData(this.freqData);
                    // If the audio stream is initializing, return empty queue.
                    if (this.freqData[0] === -Infinity) {
                        resolve({ freqDataQueue, timeDataQueue });
                    }
                    freqDataQueue.push(this.freqData.slice(0, this.columnTruncateLength));
                }
                if (this.includeWaveform) {
                    this.analyser.getFloatTimeDomainData(this.timeData);
                    timeDataQueue.push(this.timeData.slice());
                }
                // Clean interval and return when all frames have been collected
                if (++currentFrames === this.numFrames) {
                    clearInterval(intervalID);
                    resolve({ freqDataQueue, timeDataQueue });
                }
            }, this.fftSize / this.sampleRateHz * 1e3);
        });
    }
    // Stop the audio stream and pause the iterator.
    stop() {
        if (!this.isClosed) {
            this.isClosed = true;
            this.analyser.disconnect();
            this.audioContext.close();
            if (this.stream != null && this.stream.getTracks().length > 0) {
                this.stream.getTracks()[0].stop();
            }
        }
    }
    // Override toArray() function to prevent collecting.
    toArray() {
        throw new Error('Can not convert infinite audio stream to array.');
    }
    // Return audio sampling rate in Hz
    getSampleRate() {
        return this.sampleRateHz;
    }
    flattenQueue(queue) {
        const frameSize = queue[0].length;
        const freqData = new Float32Array(queue.length * frameSize);
        queue.forEach((data, i) => freqData.set(data, i * frameSize));
        return freqData;
    }
    getTensorFromAudioDataArray(freqData, shape) {
        const vals = new Float32Array(util.sizeFromShape(shape));
        // If the data is less than the output shape, the rest is padded with zeros.
        vals.set(freqData, vals.length - freqData.length);
        return tensor(vals, shape);
    }
}
//# sourceMappingURL=data:application/json;base64,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