/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * =============================================================================
 */
import { browser, cast, env, expandDims, image, reshape, tensor1d, tensor2d, tidy, util } from '@tensorflow/tfjs-core';
import { LazyIterator } from './lazy_iterator';
/**
 * Provide a stream of image tensors from webcam video stream. Only works in
 * browser environment.
 */
export class WebcamIterator extends LazyIterator {
    constructor(webcamVideoElement, webcamConfig) {
        super();
        this.webcamVideoElement = webcamVideoElement;
        this.webcamConfig = webcamConfig;
        this.isClosed = true;
        this.resize = false;
        if (this.needToResize()) {
            this.resize = true;
            this.cropSize =
                [this.webcamConfig.resizeHeight, this.webcamConfig.resizeWidth];
            this.cropBoxInd = tensor1d([0], 'int32');
            if (this.webcamConfig.centerCrop) {
                // Calculate the box based on resizing shape.
                const widthCroppingRatio = this.webcamConfig.resizeWidth * 1.0 / this.webcamVideoElement.width;
                const heightCroppingRatio = this.webcamConfig.resizeHeight * 1.0 /
                    this.webcamVideoElement.height;
                const widthCropStart = (1 - widthCroppingRatio) / 2;
                const heightCropStart = (1 - heightCroppingRatio) / 2;
                const widthCropEnd = widthCropStart + widthCroppingRatio;
                const heightCropEnd = heightCroppingRatio + heightCropStart;
                this.cropBox = tensor2d([heightCropStart, widthCropStart, heightCropEnd, widthCropEnd], [1, 4]);
            }
            else {
                this.cropBox = tensor2d([0, 0, 1, 1], [1, 4]);
            }
        }
    }
    summary() {
        return `webcam`;
    }
    // Construct a WebcamIterator and start it's video stream.
    static async create(webcamVideoElement, webcamConfig = {}) {
        if (env().get('IS_NODE')) {
            throw new Error('tf.data.webcam is only supported in browser environment.');
        }
        if (!webcamVideoElement) {
            // If webcam video element is not provided, create a hidden video element
            // with provided width and height.
            webcamVideoElement = document.createElement('video');
            if (!webcamConfig.resizeWidth || !webcamConfig.resizeHeight) {
                throw new Error('Please provide webcam video element, or resizeWidth and ' +
                    'resizeHeight to create a hidden video element.');
            }
            webcamVideoElement.width = webcamConfig.resizeWidth;
            webcamVideoElement.height = webcamConfig.resizeHeight;
        }
        const webcamIterator = new WebcamIterator(webcamVideoElement, webcamConfig);
        // Call async function to initialize the video stream.
        await webcamIterator.start();
        return webcamIterator;
    }
    // Async function to start video stream.
    async start() {
        if (this.webcamConfig.facingMode) {
            util.assert((this.webcamConfig.facingMode === 'user') ||
                (this.webcamConfig.facingMode === 'environment'), () => `Invalid webcam facing mode: ${this.webcamConfig.facingMode}. ` +
                `Please provide 'user' or 'environment'`);
        }
        try {
            this.stream = await navigator.mediaDevices.getUserMedia({
                video: {
                    deviceId: this.webcamConfig.deviceId,
                    facingMode: this.webcamConfig.facingMode ?
                        this.webcamConfig.facingMode :
                        'user',
                    width: this.webcamVideoElement.width,
                    height: this.webcamVideoElement.height
                }
            });
        }
        catch (e) {
            // Modify the error message but leave the stack trace intact
            e.message = `Error thrown while initializing video stream: ${e.message}`;
            throw e;
        }
        if (!this.stream) {
            throw new Error('Could not obtain video from webcam.');
        }
        // Older browsers may not have srcObject
        try {
            this.webcamVideoElement.srcObject = this.stream;
        }
        catch (error) {
            console.log(error);
            this.webcamVideoElement.src = window.URL.createObjectURL(this.stream);
        }
        // Start the webcam video stream
        this.webcamVideoElement.play();
        this.isClosed = false;
        return new Promise(resolve => {
            // Add event listener to make sure the webcam has been fully initialized.
            this.webcamVideoElement.onloadedmetadata = () => {
                resolve();
            };
        });
    }
    async next() {
        if (this.isClosed) {
            return { value: null, done: true };
        }
        let img;
        try {
            img = browser.fromPixels(this.webcamVideoElement);
        }
        catch (e) {
            throw new Error(`Error thrown converting video to pixels: ${JSON.stringify(e)}`);
        }
        if (this.resize) {
            try {
                return { value: this.cropAndResizeFrame(img), done: false };
            }
            catch (e) {
                throw new Error(`Error thrown cropping the video: ${e.message}`);
            }
            finally {
                img.dispose();
            }
        }
        else {
            return { value: img, done: false };
        }
    }
    needToResize() {
        // If resizeWidth and resizeHeight are provided, and different from the
        // width and height of original HTMLVideoElement, then resizing and cropping
        // is required.
        if (this.webcamConfig.resizeWidth && this.webcamConfig.resizeHeight &&
            (this.webcamVideoElement.width !== this.webcamConfig.resizeWidth ||
                this.webcamVideoElement.height !== this.webcamConfig.resizeHeight)) {
            return true;
        }
        return false;
    }
    // Cropping and resizing each frame based on config
    cropAndResizeFrame(img) {
        return tidy(() => {
            const expandedImage = expandDims(cast(img, 'float32'), (0));
            let resizedImage;
            resizedImage = image.cropAndResize(expandedImage, this.cropBox, this.cropBoxInd, this.cropSize, 'bilinear');
            // Extract image from batch cropping.
            const shape = resizedImage.shape;
            return reshape(resizedImage, shape.slice(1));
        });
    }
    // Capture one frame from the video stream, and extract the value from
    // iterator.next() result.
    async capture() {
        return (await this.next()).value;
    }
    // Stop the video stream and pause webcam iterator.
    stop() {
        const tracks = this.stream.getTracks();
        tracks.forEach(track => track.stop());
        try {
            this.webcamVideoElement.srcObject = null;
        }
        catch (error) {
            console.log(error);
            this.webcamVideoElement.src = null;
        }
        this.isClosed = true;
    }
    // Override toArray() function to prevent collecting.
    toArray() {
        throw new Error('Can not convert infinite video stream to array.');
    }
}
//# sourceMappingURL=data:application/json;base64,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