/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * =============================================================================
 */
// tslint:disable-next-line: no-imports-from-dist
import { ALL_ENVS, BROWSER_ENVS, describeWithFlags, NODE_ENVS, registerTestEnv } from '@tensorflow/tfjs-core/dist/jasmine_util';
// Provide fake video stream
export function setupFakeVideoStream() {
    const width = 100;
    const height = 200;
    const canvasElement = document.createElement('canvas');
    const ctx = canvasElement.getContext('2d');
    ctx.fillStyle = 'rgb(1,2,3)';
    ctx.fillRect(0, 0, width, height);
    // tslint:disable-next-line:no-any
    const stream = canvasElement.captureStream(60);
    navigator.mediaDevices.getUserMedia = async () => {
        return stream;
    };
}
export async function replaceHTMLVideoElementSource(videoElement) {
    const source = document.createElement('source');
    // tslint:disable:max-line-length
    source.src =
        'data:video/mp4;base64,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';
    source.type = 'video/mp4';
    videoElement.srcObject = null;
    videoElement.appendChild(source);
    videoElement.play();
    if (videoElement.readyState < 2) {
        await new Promise(resolve => {
            videoElement.addEventListener('loadeddata', () => resolve());
        });
    }
}
// Register backends.
registerTestEnv({ name: 'cpu', backendName: 'cpu' });
registerTestEnv({
    name: 'webgl2',
    backendName: 'webgl',
    flags: {
        'WEBGL_VERSION': 2,
        'WEBGL_CPU_FORWARD': false,
        'WEBGL_SIZE_UPLOAD_UNIFORM': 0
    }
});
export function describeAllEnvs(testName, tests) {
    describeWithFlags(testName, ALL_ENVS, () => {
        tests();
    });
}
export function describeBrowserEnvs(testName, tests) {
    describeWithFlags(testName, BROWSER_ENVS, () => {
        tests();
    });
}
export function describeNodeEnvs(testName, tests) {
    describeWithFlags(testName, NODE_ENVS, () => {
        tests();
    });
}
/**
 * Testing Utilities for browser audio stream.
 */
export function setupFakeAudioStream() {
    navigator.mediaDevices.getUserMedia = async () => {
        const stream = new MediaStream();
        return stream;
    };
    // tslint:disable-next-line:no-any
    window.AudioContext = FakeAudioContext;
}
export class FakeAudioContext {
    constructor() {
        this.sampleRate = 44100;
    }
    static createInstance() {
        return new FakeAudioContext();
    }
    createMediaStreamSource() {
        return new FakeMediaStreamAudioSourceNode();
    }
    createAnalyser() {
        return new FakeAnalyser();
    }
    close() { }
}
export class FakeAudioMediaStream {
    constructor() { }
    getTracks() {
        return [];
    }
}
class FakeMediaStreamAudioSourceNode {
    constructor() { }
    connect(node) { }
}
class FakeAnalyser {
    constructor() {
        this.x = 0;
    }
    getFloatFrequencyData(data) {
        const xs = [];
        for (let i = 0; i < this.fftSize / 2; ++i) {
            xs.push(this.x++);
        }
        data.set(new Float32Array(xs));
    }
    getFloatTimeDomainData(data) {
        const xs = [];
        for (let i = 0; i < this.fftSize / 2; ++i) {
            xs.push(-(this.x++));
        }
        data.set(new Float32Array(xs));
    }
    disconnect() { }
}
//# sourceMappingURL=data:application/json;base64,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