/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * =============================================================================
 */
import * as tf from '@tensorflow/tfjs-core';
import * as seedrandom from 'seedrandom';
import { iteratorFromConcatenated, iteratorFromFunction, iteratorFromItems, iteratorFromZipped, ZipMismatchMode } from './iterators/lazy_iterator';
import { canTensorify, deepMapAndAwaitAll, isIterable } from './util/deep_map';
// TODO(soergel): consider vectorized operations within the pipeline.
/**
 * Represents a potentially large list of independent data elements (typically
 * 'samples' or 'examples').
 *
 * A 'data example' may be a primitive, an array, a map from string keys to
 * values, or any nested structure of these.
 *
 * A `Dataset` represents an ordered collection of elements, together with a
 * chain of transformations to be performed on those elements. Each
 * transformation is a method of `Dataset` that returns another `Dataset`, so
 * these may be chained, e.g.
 * `const processedDataset = rawDataset.filter(...).map(...).batch(...)`.
 *
 * Data loading and transformation is done in a lazy, streaming fashion.  The
 * dataset may be iterated over multiple times; each iteration starts the data
 * loading anew and recapitulates the transformations.
 *
 * A `Dataset` is typically processed as a stream of unbatched examples -- i.e.,
 * its transformations are applied one example at a time. Batching produces a
 * new `Dataset` where each element is a batch. Batching should usually come
 * last in a pipeline, because data transformations are easier to express on a
 * per-example basis than on a per-batch basis.
 *
 * The following code examples are calling `await dataset.forEachAsync(...)` to
 * iterate once over the entire dataset in order to print out the data.
 *
 * @doc {heading: 'Data', subheading: 'Classes', namespace: 'data'}
 */
export class Dataset {
    constructor() {
        this.size = null;
    }
    // TODO(soergel): Make Datasets report whether repeated iterator() calls
    // produce the same result (e.g., reading from a file) or different results
    // (e.g., from the webcam).  Currently we don't make this distinction but it
    // could be important for the user to know.
    // abstract isDeterministic(): boolean;
    /**
     * Groups elements into batches.
     *
     * It is assumed that each of the incoming dataset elements has the same
     * structure -- i.e. the same set of keys at each location in an object
     * hierarchy.  For each key, the resulting `Dataset` provides a batched
     * element collecting all of the incoming values for that key.
     *
     *  * Incoming primitives are grouped into a 1-D Tensor.
     *  * Incoming Tensors are grouped into a new Tensor where the 0th axis is
     *    the batch dimension.
     *  * Incoming arrays are converted to Tensor and then batched.
     *  * A nested array is interpreted as an n-D Tensor, so the batched result
     *    has n+1 dimensions.
     *  * An array that cannot be converted to Tensor produces an error.
     *
     * If an array should not be batched as a unit, it should first be converted
     * to an object with integer keys.
     *
     * Here are a few examples:
     *
     * Batch a dataset of numbers:
     * ```js
     * const a = tf.data.array([1, 2, 3, 4, 5, 6, 7, 8]).batch(4);
     * await a.forEachAsync(e => e.print());
     * ```
     *
     * Batch a dataset of arrays:
     * ```js
     * const b = tf.data.array([[1], [2], [3], [4], [5], [6], [7], [8]]).batch(4);
     * await b.forEachAsync(e => e.print());
     * ```
     *
     * Batch a dataset of objects:
     * ```js
     * const c = tf.data.array([{a: 1, b: 11}, {a: 2, b: 12}, {a: 3, b: 13},
     *   {a: 4, b: 14}, {a: 5, b: 15}, {a: 6, b: 16}, {a: 7, b: 17},
     *   {a: 8, b: 18}]).batch(4);
     * await c.forEachAsync(e => {
     *   console.log('{');
     *   for(var key in e) {
     *     console.log(key+':');
     *     e[key].print();
     *   }
     *   console.log('}');
     * })
     * ```
     *
     * @param batchSize The number of elements desired per batch.
     * @param smallLastBatch Whether to emit the final batch when it has fewer
     *   than batchSize elements. Default true.
     * @returns A `Dataset`, from which a stream of batches can be obtained.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    batch(batchSize, smallLastBatch = true) {
        const base = this;
        tf.util.assert(batchSize > 0, () => `batchSize needs to be positive, but it is
      ${batchSize}`);
        let size;
        if (this.size === Infinity || this.size == null) {
            // If the size of this dataset is infinity or null, the new size keeps the
            // same.
            size = this.size;
        }
        else if (smallLastBatch) {
            // If the size of this dataset is known and include small last batch, the
            // new size is full batch count plus last batch.
            size = Math.ceil(this.size / batchSize);
        }
        else {
            // If the size of this dataset is known and not include small last batch,
            // the new size is full batch count.
            size = Math.floor(this.size / batchSize);
        }
        return datasetFromIteratorFn(async () => {
            return (await base.iterator())
                .columnMajorBatch(batchSize, smallLastBatch, deepBatchConcat);
        }, size);
    }
    /**
     * Concatenates this `Dataset` with another.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3]);
     * const b = tf.data.array([4, 5, 6]);
     * const c = a.concatenate(b);
     * await c.forEachAsync(e => console.log(e));
     * ```
     *
     * @param dataset A `Dataset` to be concatenated onto this one.
     * @returns A `Dataset`.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    concatenate(dataset) {
        const base = this;
        let size;
        if (this.size === Infinity || dataset.size === Infinity) {
            // If the size of any of these two dataset is infinity, new size is
            // infinity.
            size = Infinity;
        }
        else if (this.size != null && dataset.size != null) {
            // If the size of both datasets are known and not infinity, new size is
            // sum the size of these two datasets.
            size = this.size + dataset.size;
        }
        else {
            // If neither of these two datasets has infinite size and any of these two
            // datasets' size is null, the new size is null.
            size = null;
        }
        return datasetFromIteratorFn(async () => (await base.iterator()).concatenate(await dataset.iterator()), size);
    }
    /**
     * Filters this dataset according to `predicate`.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3, 4, 5, 6, 7, 8, 9, 10])
     *   .filter(x => x%2 === 0);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param predicate A function mapping a dataset element to a boolean or a
     * `Promise` for one.
     *
     * @returns A `Dataset` of elements for which the predicate was true.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    filter(predicate) {
        const base = this;
        let size;
        if (this.size === Infinity) {
            // If the size of this dataset is infinity, new size is infinity
            size = Infinity;
        }
        else {
            // If this dataset has limited elements, new size is null because it might
            // exhausted randomly.
            size = null;
        }
        return datasetFromIteratorFn(async () => {
            return (await base.iterator()).filter(x => tf.tidy(() => predicate(x)));
        }, size);
    }
    /**
     * Apply a function to every element of the dataset.
     *
     * After the function is applied to a dataset element, any Tensors contained
     * within that element are disposed.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3]);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param f A function to apply to each dataset element.
     * @returns A `Promise` that resolves after all elements have been processed.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    async forEachAsync(f) {
        return (await this.iterator()).forEachAsync(f);
    }
    /**
     * Maps this dataset through a 1-to-1 transform.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3]).map(x => x*x);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param transform A function mapping a dataset element to a transformed
     *   dataset element.
     *
     * @returns A `Dataset` of transformed elements.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    map(transform) {
        const base = this;
        return datasetFromIteratorFn(async () => {
            return (await base.iterator()).map(x => tf.tidy(() => transform(x)));
        }, this.size);
    }
    /**
     * Maps this dataset through an async 1-to-1 transform.
     *
     * ```js
     * const a =
     *  tf.data.array([1, 2, 3]).mapAsync(x => new Promise(function(resolve){
     *    setTimeout(() => {
     *      resolve(x * x);
     *    }, Math.random()*1000 + 500);
     *  }));
     * console.log(await a.toArray());
     * ```
     *
     * @param transform A function mapping a dataset element to a `Promise` for a
     *   transformed dataset element.  This transform is responsible for disposing
     *   any intermediate `Tensor`s, i.e. by wrapping its computation in
     *   `tf.tidy()`; that cannot be automated here (as it is in the synchronous
     *   `map()` case).
     *
     * @returns A `Dataset` of transformed elements.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    mapAsync(transform) {
        const base = this;
        return datasetFromIteratorFn(async () => {
            return (await base.iterator()).mapAsync(transform);
        }, this.size);
    }
    /**
     *  Creates a `Dataset` that prefetches elements from this dataset.
     *
     * @param bufferSize: An integer specifying the number of elements to be
     *   prefetched.
     * @returns A `Dataset`.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    prefetch(bufferSize) {
        if (bufferSize == null) {
            throw new RangeError('`Dataset.prefetch()` requires bufferSize to be specified.');
        }
        const base = this;
        return datasetFromIteratorFn(async () => (await base.iterator()).prefetch(bufferSize), this.size);
    }
    /**
     * Repeats this dataset `count` times.
     *
     * NOTE: If this dataset is a function of global state (e.g. a random number
     * generator), then different repetitions may produce different elements.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3]).repeat(3);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param count: (Optional) An integer, representing the number of times
     *   the dataset should be repeated. The default behavior (if `count` is
     *   `undefined` or negative) is for the dataset be repeated indefinitely.
     * @returns A `Dataset`.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    repeat(count) {
        const base = this;
        let size;
        if (this.size != null && count > 0) {
            // If this dataset has size and count is positive, new size is current
            // size multiply count. This also covers the case that current size is
            // infinity.
            size = this.size * count;
        }
        else if (count === 0) {
            // If count is 0, new size is 0.
            size = 0;
        }
        else if (this.size != null && (count === undefined || count < 0)) {
            // If this dataset has size and count is undefined or negative, the
            // dataset will be repeated indefinitely and new size is infinity.
            size = Infinity;
        }
        else {
            // If the size of this dataset is null, the new dataset's size is null.
            size = null;
        }
        return datasetFromIteratorFn(async () => {
            const iteratorIterator = iteratorFromFunction(async () => ({ value: await base.iterator(), done: false }));
            return iteratorFromConcatenated(iteratorIterator.take(count));
        }, size);
    }
    /**
     * Creates a `Dataset` that skips `count` initial elements from this dataset.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3, 4, 5, 6]).skip(3);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param count: The number of elements of this dataset that should be skipped
     *   to form the new dataset.  If `count` is greater than the size of this
     *   dataset, the new dataset will contain no elements.  If `count`
     *   is `undefined` or negative, skips the entire dataset.
     *
     * @returns A `Dataset`.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    skip(count) {
        const base = this;
        let size;
        if (this.size != null && count >= 0 && this.size >= count) {
            // If the size of this dataset is greater than count, the new dataset's
            // size is current size minus skipped size.This also covers the case that
            // current size is infinity.
            size = this.size - count;
        }
        else if (this.size != null &&
            (this.size < count || count === undefined || count < 0)) {
            // If the size of this dataset is smaller than count, or count is
            // undefined or negative, skips the entire dataset and the new size is 0.
            size = 0;
        }
        else {
            // If the size of this dataset is null, the new dataset's size is null.
            size = null;
        }
        return datasetFromIteratorFn(async () => (await base.iterator()).skip(count), size);
    }
    /**
     * Pseudorandomly shuffles the elements of this dataset. This is done in a
     * streaming manner, by sampling from a given number of prefetched elements.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3, 4, 5, 6]).shuffle(3);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param bufferSize: An integer specifying the number of elements from this
     *   dataset from which the new dataset will sample.
     * @param seed: (Optional) An integer specifying the random seed that will
     *   be used to create the distribution.
     * @param reshuffleEachIteration: (Optional) A boolean, which if true
     *   indicates that the dataset should be pseudorandomly reshuffled each time
     *   it is iterated over. If false, elements will be returned in the same
     *   shuffled order on each iteration. (Defaults to `true`.)
     * @returns A `Dataset`.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    shuffle(bufferSize, seed, reshuffleEachIteration = true) {
        if (bufferSize == null || bufferSize < 0) {
            if (this.size == null) {
                throw new RangeError('`Dataset.shuffle()` requires bufferSize to be specified.');
            }
            else {
                throw new RangeError('`Dataset.shuffle()` requires bufferSize to be specified.  ' +
                    'If your data fits in main memory (for regular JS objects), ' +
                    'and/or GPU memory (for `tf.Tensor`s), consider setting ' +
                    `bufferSize to the dataset size (${this.size} elements)`);
            }
        }
        const base = this;
        const random = seedrandom.alea(seed || tf.util.now().toString());
        return datasetFromIteratorFn(async () => {
            let seed2 = random.int32();
            if (reshuffleEachIteration) {
                seed2 += random.int32();
            }
            return (await base.iterator()).shuffle(bufferSize, seed2.toString());
        }, this.size);
    }
    /**
     * Creates a `Dataset` with at most `count` initial elements from this
     * dataset.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3, 4, 5, 6]).take(3);
     * await a.forEachAsync(e => console.log(e));
     * ```
     *
     * @param count: The number of elements of this dataset that should be taken
     *   to form the new dataset.  If `count` is `undefined` or negative, or if
     *   `count` is greater than the size of this dataset, the new dataset will
     *   contain all elements of this dataset.
     * @returns A `Dataset`.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    take(count) {
        const base = this;
        let size;
        if (this.size != null && this.size > count) {
            // If the size of this dataset is greater than count, the new dataset's
            // size is count.
            size = count;
        }
        else if (this.size != null && this.size <= count) {
            // If the size of this dataset is equal or smaller than count, the new
            // dataset's size is the size of this dataset.
            size = this.size;
        }
        else {
            // If the size of this dataset is null, the new dataset's size is null.
            size = null;
        }
        return datasetFromIteratorFn(async () => (await base.iterator()).take(count), size);
    }
    /**
     * Collect all elements of this dataset into an array.
     *
     * Obviously this will succeed only for small datasets that fit in memory.
     * Useful for testing and generally should be avoided if possible.
     *
     * ```js
     * const a = tf.data.array([1, 2, 3, 4, 5, 6]);
     * console.log(await a.toArray());
     * ```
     *
     * @returns A Promise for an array of elements, which will resolve
     *   when a new stream has been obtained and fully consumed.
     *
     * @doc {heading: 'Data', subheading: 'Classes'}
     */
    async toArray() {
        if (this.size === Infinity) {
            throw new Error('Can not convert infinite data stream to array.');
        }
        return (await this.iterator()).toArray();
    }
    /**
     * Collect all elements of this dataset into an array with prefetching 100
     * elements. This is useful for testing, because the prefetch changes the
     * order in which the Promises are resolved along the processing pipeline.
     * This may help expose bugs where results are dependent on the order of
     * Promise resolution rather than on the logical order of the stream (i.e.,
     * due to hidden mutable state).
     *
     * @returns A Promise for an array of elements, which will resolve
     *   when a new stream has been obtained and fully consumed.
     */
    async toArrayForTest() {
        if (this.size === Infinity) {
            throw new Error('Can not convert infinite data stream to array.');
        }
        return (await this.iterator()).toArrayForTest();
    }
}
// TODO(soergel): deep sharded shuffle, where supported
Dataset.MAX_BUFFER_SIZE = 10000;
/**
 * Create a `Dataset` defined by a provided iterator() function.
 *
 * ```js
 * let i = -1;
 * const func = () =>
 *    ++i < 5 ? {value: i, done: false} : {value: null, done: true};
 * const iter = tf.data.iteratorFromFunction(func);
 * const ds = tf.data.datasetFromIteratorFn(iter);
 * await ds.forEachAsync(e => console.log(e));
 * ```
 */
export function datasetFromIteratorFn(iteratorFn, size = null) {
    return new class extends Dataset {
        constructor() {
            super(...arguments);
            this.size = size;
        }
        /*
         * Provide a new stream of elements.  Note this will also start new streams
         * from any underlying `Dataset`s.
         */
        async iterator() {
            return iteratorFn();
        }
    }();
}
/**
 * Create a `Dataset` from an array of elements.
 *
 * Create a Dataset from an array of objects:
 * ```js
 * const a = tf.data.array([{'item': 1}, {'item': 2}, {'item': 3}]);
 * await a.forEachAsync(e => console.log(e));
 * ```
 *
 * Create a Dataset from an array of numbers:
 * ```js
 * const a = tf.data.array([4, 5, 6]);
 * await a.forEachAsync(e => console.log(e));
 * ```
 * @param items An array of elements that will be parsed as items in a dataset.
 *
 * @doc {heading: 'Data', subheading: 'Creation', namespace: 'data'}
 */
export function array(items) {
    return datasetFromIteratorFn(async () => iteratorFromItems(items), items.length);
}
/**
 * Create a `Dataset` by zipping together an array, dict, or nested
 * structure of `Dataset`s (and perhaps additional constants).
 * The underlying datasets must provide elements in a consistent order such that
 * they correspond.
 *
 * The number of elements in the resulting dataset is the same as the size of
 * the smallest dataset in datasets.
 *
 * The nested structure of the `datasets` argument determines the
 * structure of elements in the resulting iterator.
 *
 * Note this means that, given an array of two datasets that produce dict
 * elements, the result is a dataset that produces elements that are arrays
 * of two dicts:
 *
 * Zip an array of datasets:
 * ```js
 * console.log('Zip two datasets of objects:');
 * const ds1 = tf.data.array([{a: 1}, {a: 2}, {a: 3}]);
 * const ds2 = tf.data.array([{b: 4}, {b: 5}, {b: 6}]);
 * const ds3 = tf.data.zip([ds1, ds2]);
 * await ds3.forEachAsync(e => console.log(JSON.stringify(e)));
 *
 * // If the goal is to merge the dicts in order to produce elements like
 * // {a: ..., b: ...}, this requires a second step such as:
 * console.log('Merge the objects:');
 * const ds4 = ds3.map(x => {return {a: x[0].a, b: x[1].b}});
 * await ds4.forEachAsync(e => console.log(e));
 * ```
 *
 * Zip a dict of datasets:
 * ```js
 * const a = tf.data.array([{a: 1}, {a: 2}, {a: 3}]);
 * const b = tf.data.array([{b: 4}, {b: 5}, {b: 6}]);
 * const c = tf.data.zip({c: a, d: b});
 * await c.forEachAsync(e => console.log(JSON.stringify(e)));
 * ```
 *
 * @doc {heading: 'Data', subheading: 'Operations', namespace: 'data'}
 */
export function zip(datasets) {
    // manually type-check the argument for JS users
    if (!isIterable(datasets)) {
        throw new Error('The argument to zip() must be an object or array.');
    }
    let size;
    if (Array.isArray(datasets)) {
        for (let i = 0; i < datasets.length; i++) {
            size = size == null ? datasets[i].size :
                Math.min(size, datasets[i].size);
        }
    }
    else if (datasets instanceof Object) {
        for (const ds in datasets) {
            size = size == null ? datasets[ds].size :
                Math.min(size, datasets[ds].size);
        }
    }
    return datasetFromIteratorFn(async () => {
        const streams = await deepMapAndAwaitAll(datasets, d => {
            if (d instanceof Dataset) {
                return { value: d.iterator(), recurse: false };
            }
            else if (isIterable(d)) {
                return { value: null, recurse: true };
            }
            else {
                throw new Error('Leaves of the structure passed to zip() must be Datasets, ' +
                    'not primitives.');
            }
        });
        return iteratorFromZipped(streams, ZipMismatchMode.SHORTEST);
    }, size);
}
/**
 * A zip function for use with deepZip, passed via the columnMajorBatch call.
 *
 * Accepts an array of identically-structured nested elements and either batches
 * them (if they are primitives, numeric arrays, or Tensors) or requests
 * recursion (if not).
 */
// tslint:disable-next-line:no-any
function deepBatchConcat(rows) {
    if (rows === null) {
        return null;
    }
    // use the first item to decide whether to recurse or batch here.
    const exampleRow = rows[0];
    if (canTensorify(exampleRow)) {
        // rows is an array of primitives, Tensors, or arrays.  Batch them.
        const value = batchConcat(rows);
        return { value, recurse: false };
    }
    // the example row is an object, so recurse into it.
    return { value: null, recurse: true };
}
/**
 * Assembles a list of same-shaped numbers, number arrays, or Tensors
 * into a single new Tensor where axis 0 is the batch dimension.
 */
function batchConcat(arrays) {
    if (arrays.length === 0) {
        // We can't return an empty Tensor because we don't know the element shape.
        throw new Error('Can\'t make a batch of zero elements.');
    }
    if (arrays[0] instanceof tf.Tensor) {
        // Input is an array of Tensors
        return tf.stack(arrays);
    }
    else {
        // Input is a possibly-nested array of numbers.
        return tf.tensor(arrays);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YXNldC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uL3RmanMtZGF0YS9zcmMvZGF0YXNldC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7Ozs7Ozs7Ozs7OztHQWdCRztBQUVILE9BQU8sS0FBSyxFQUFFLE1BQU0sdUJBQXVCLENBQUM7QUFFNUMsT0FBTyxLQUFLLFVBQVUsTUFBTSxZQUFZLENBQUM7QUFFekMsT0FBTyxFQUFDLHdCQUF3QixFQUFFLG9CQUFvQixFQUFFLGlCQUFpQixFQUFFLGtCQUFrQixFQUFnQixlQUFlLEVBQUMsTUFBTSwyQkFBMkIsQ0FBQztBQUUvSixPQUFPLEVBQUMsWUFBWSxFQUFFLGtCQUFrQixFQUFpQixVQUFVLEVBQUMsTUFBTSxpQkFBaUIsQ0FBQztBQU81RixxRUFBcUU7QUFFckU7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQTJCRztBQUNILE1BQU0sT0FBZ0IsT0FBTztJQUE3QjtRQVdXLFNBQUksR0FBVyxJQUFJLENBQUM7SUEyYy9CLENBQUM7SUF6Y0Msd0VBQXdFO0lBQ3hFLDJFQUEyRTtJQUMzRSw0RUFBNEU7SUFDNUUsMkNBQTJDO0lBQzNDLHVDQUF1QztJQUV2Qzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0RHO0lBQ0gsS0FBSyxDQUFDLFNBQWlCLEVBQUUsY0FBYyxHQUFHLElBQUk7UUFDNUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUNWLFNBQVMsR0FBRyxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUM7UUFDckIsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLElBQUksQ0FBQztRQUNULElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxRQUFRLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUU7WUFDL0MsMEVBQTBFO1lBQzFFLFFBQVE7WUFDUixJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztTQUNsQjthQUFNLElBQUksY0FBYyxFQUFFO1lBQ3pCLHlFQUF5RTtZQUN6RSxnREFBZ0Q7WUFDaEQsSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxTQUFTLENBQUMsQ0FBQztTQUN6QzthQUFNO1lBQ0wseUVBQXlFO1lBQ3pFLG9DQUFvQztZQUNwQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLFNBQVMsQ0FBQyxDQUFDO1NBQzFDO1FBQ0QsT0FBTyxxQkFBcUIsQ0FBQyxLQUFLLElBQUksRUFBRTtZQUN0QyxPQUFPLENBQUMsTUFBTSxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7aUJBQ3pCLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxjQUFjLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDcEUsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0gsV0FBVyxDQUFDLE9BQW1CO1FBQzdCLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixJQUFJLElBQUksQ0FBQztRQUNULElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxRQUFRLElBQUksT0FBTyxDQUFDLElBQUksS0FBSyxRQUFRLEVBQUU7WUFDdkQsbUVBQW1FO1lBQ25FLFlBQVk7WUFDWixJQUFJLEdBQUcsUUFBUSxDQUFDO1NBQ2pCO2FBQU0sSUFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsSUFBSSxJQUFJLElBQUksRUFBRTtZQUNwRCx1RUFBdUU7WUFDdkUsc0NBQXNDO1lBQ3RDLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUM7U0FDakM7YUFBTTtZQUNMLDBFQUEwRTtZQUMxRSxnREFBZ0Q7WUFDaEQsSUFBSSxHQUFHLElBQUksQ0FBQztTQUNiO1FBQ0QsT0FBTyxxQkFBcUIsQ0FDeEIsS0FBSyxJQUFJLEVBQUUsQ0FDUCxDQUFDLE1BQU0sSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsV0FBVyxDQUFDLE1BQU0sT0FBTyxDQUFDLFFBQVEsRUFBRSxDQUFDLEVBQ2pFLElBQUksQ0FBQyxDQUFDO0lBQ1osQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNILE1BQU0sQ0FBQyxTQUFnQztRQUNyQyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsSUFBSSxJQUFJLENBQUM7UUFDVCxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssUUFBUSxFQUFFO1lBQzFCLGdFQUFnRTtZQUNoRSxJQUFJLEdBQUcsUUFBUSxDQUFDO1NBQ2pCO2FBQU07WUFDTCwwRUFBMEU7WUFDMUUsc0JBQXNCO1lBQ3RCLElBQUksR0FBRyxJQUFJLENBQUM7U0FDYjtRQUNELE9BQU8scUJBQXFCLENBQUMsS0FBSyxJQUFJLEVBQUU7WUFDdEMsT0FBTyxDQUFDLE1BQU0sSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzFFLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNYLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSCxLQUFLLENBQUMsWUFBWSxDQUFDLENBQXFCO1FBQ3RDLE9BQU8sQ0FBQyxNQUFNLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSCxHQUFHLENBQStCLFNBQTBCO1FBQzFELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixPQUFPLHFCQUFxQixDQUFDLEtBQUssSUFBSSxFQUFFO1lBQ3RDLE9BQU8sQ0FBQyxNQUFNLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2RSxDQUFDLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXNCRztJQUNILFFBQVEsQ0FBK0IsU0FBbUM7UUFFeEUsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLE9BQU8scUJBQXFCLENBQUMsS0FBSyxJQUFJLEVBQUU7WUFDdEMsT0FBTyxDQUFDLE1BQU0sSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3JELENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDaEIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0gsUUFBUSxDQUFDLFVBQWtCO1FBQ3pCLElBQUksVUFBVSxJQUFJLElBQUksRUFBRTtZQUN0QixNQUFNLElBQUksVUFBVSxDQUNoQiwyREFBMkQsQ0FBQyxDQUFDO1NBQ2xFO1FBRUQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLE9BQU8scUJBQXFCLENBQ3hCLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNILE1BQU0sQ0FBQyxLQUFjO1FBQ25CLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixJQUFJLElBQUksQ0FBQztRQUNULElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLElBQUksS0FBSyxHQUFHLENBQUMsRUFBRTtZQUNsQyxzRUFBc0U7WUFDdEUsc0VBQXNFO1lBQ3RFLFlBQVk7WUFDWixJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUM7U0FDMUI7YUFBTSxJQUFJLEtBQUssS0FBSyxDQUFDLEVBQUU7WUFDdEIsZ0NBQWdDO1lBQ2hDLElBQUksR0FBRyxDQUFDLENBQUM7U0FDVjthQUFNLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLElBQUksQ0FBQyxLQUFLLEtBQUssU0FBUyxJQUFJLEtBQUssR0FBRyxDQUFDLENBQUMsRUFBRTtZQUNsRSxtRUFBbUU7WUFDbkUsa0VBQWtFO1lBQ2xFLElBQUksR0FBRyxRQUFRLENBQUM7U0FDakI7YUFBTTtZQUNMLHVFQUF1RTtZQUN2RSxJQUFJLEdBQUcsSUFBSSxDQUFDO1NBQ2I7UUFDRCxPQUFPLHFCQUFxQixDQUFDLEtBQUssSUFBSSxFQUFFO1lBQ3RDLE1BQU0sZ0JBQWdCLEdBQUcsb0JBQW9CLENBQ3pDLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFDLEtBQUssRUFBRSxNQUFNLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFDLENBQUMsQ0FBQyxDQUFDO1lBQy9ELE9BQU8sd0JBQXdCLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDaEUsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7O09BZ0JHO0lBQ0gsSUFBSSxDQUFDLEtBQWE7UUFDaEIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLElBQUksSUFBSSxDQUFDO1FBQ1QsSUFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksSUFBSSxLQUFLLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLElBQUksS0FBSyxFQUFFO1lBQ3pELHVFQUF1RTtZQUN2RSx5RUFBeUU7WUFDekUsNEJBQTRCO1lBQzVCLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQztTQUMxQjthQUFNLElBQ0gsSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJO1lBQ2pCLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLElBQUksS0FBSyxLQUFLLFNBQVMsSUFBSSxLQUFLLEdBQUcsQ0FBQyxDQUFDLEVBQUU7WUFDM0QsaUVBQWlFO1lBQ2pFLHlFQUF5RTtZQUN6RSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQ1Y7YUFBTTtZQUNMLHVFQUF1RTtZQUN2RSxJQUFJLEdBQUcsSUFBSSxDQUFDO1NBQ2I7UUFDRCxPQUFPLHFCQUFxQixDQUN4QixLQUFLLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDN0QsQ0FBQztJQU1EOzs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW9CRztJQUNILE9BQU8sQ0FBQyxVQUFrQixFQUFFLElBQWEsRUFBRSxzQkFBc0IsR0FBRyxJQUFJO1FBRXRFLElBQUksVUFBVSxJQUFJLElBQUksSUFBSSxVQUFVLEdBQUcsQ0FBQyxFQUFFO1lBQ3hDLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUU7Z0JBQ3JCLE1BQU0sSUFBSSxVQUFVLENBQ2hCLDBEQUEwRCxDQUFDLENBQUM7YUFDakU7aUJBQU07Z0JBQ0wsTUFBTSxJQUFJLFVBQVUsQ0FDaEIsNERBQTREO29CQUM1RCw2REFBNkQ7b0JBQzdELHlEQUF5RDtvQkFDekQsbUNBQW1DLElBQUksQ0FBQyxJQUFJLFlBQVksQ0FBQyxDQUFDO2FBQy9EO1NBQ0Y7UUFDRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxNQUFNLEdBQUcsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLE9BQU8scUJBQXFCLENBQUMsS0FBSyxJQUFJLEVBQUU7WUFDdEMsSUFBSSxLQUFLLEdBQUcsTUFBTSxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzNCLElBQUksc0JBQXNCLEVBQUU7Z0JBQzFCLEtBQUssSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLENBQUM7YUFDekI7WUFDRCxPQUFPLENBQUMsTUFBTSxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZFLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDaEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7O09BZ0JHO0lBQ0gsSUFBSSxDQUFDLEtBQWE7UUFDaEIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLElBQUksSUFBSSxDQUFDO1FBQ1QsSUFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksSUFBSSxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssRUFBRTtZQUMxQyx1RUFBdUU7WUFDdkUsaUJBQWlCO1lBQ2pCLElBQUksR0FBRyxLQUFLLENBQUM7U0FDZDthQUFNLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxLQUFLLEVBQUU7WUFDbEQsc0VBQXNFO1lBQ3RFLDhDQUE4QztZQUM5QyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztTQUNsQjthQUFNO1lBQ0wsdUVBQXVFO1lBQ3ZFLElBQUksR0FBRyxJQUFJLENBQUM7U0FDYjtRQUNELE9BQU8scUJBQXFCLENBQ3hCLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0gsS0FBSyxDQUFDLE9BQU87UUFDWCxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssUUFBUSxFQUFFO1lBQzFCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztTQUNuRTtRQUNELE9BQU8sQ0FBQyxNQUFNLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0gsS0FBSyxDQUFDLGNBQWM7UUFDbEIsSUFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLFFBQVEsRUFBRTtZQUMxQixNQUFNLElBQUksS0FBSyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7U0FDbkU7UUFDRCxPQUFPLENBQUMsTUFBTSxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUNsRCxDQUFDOztBQTdIRCx1REFBdUQ7QUFFdkMsdUJBQWUsR0FBRyxLQUFLLENBQUM7QUE4SDFDOzs7Ozs7Ozs7OztHQVdHO0FBQ0gsTUFBTSxVQUFVLHFCQUFxQixDQUNqQyxVQUEwQyxFQUMxQyxPQUFlLElBQUk7SUFDckIsT0FBTyxJQUFJLEtBQU0sU0FBUSxPQUFVO1FBQXhCOztZQUNULFNBQUksR0FBRyxJQUFJLENBQUM7UUFTZCxDQUFDO1FBUEM7OztXQUdHO1FBQ0gsS0FBSyxDQUFDLFFBQVE7WUFDWixPQUFPLFVBQVUsRUFBRSxDQUFDO1FBQ3RCLENBQUM7S0FDRixFQUNDLENBQUM7QUFDTCxDQUFDO0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBaUJHO0FBQ0gsTUFBTSxVQUFVLEtBQUssQ0FBK0IsS0FBVTtJQUM1RCxPQUFPLHFCQUFxQixDQUN4QixLQUFLLElBQUksRUFBRSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUMxRCxDQUFDO0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0F3Q0c7QUFDSCxNQUFNLFVBQVUsR0FBRyxDQUErQixRQUEwQjtJQUUxRSxnREFBZ0Q7SUFDaEQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRTtRQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxDQUFDLENBQUM7S0FDdEU7SUFDRCxJQUFJLElBQUksQ0FBQztJQUNULElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRTtRQUMzQixLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUN4QyxJQUFJLEdBQUcsSUFBSSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUUsUUFBUSxDQUFDLENBQUMsQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUcsUUFBUSxDQUFDLENBQUMsQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN4RTtLQUNGO1NBQU0sSUFBSSxRQUFRLFlBQVksTUFBTSxFQUFFO1FBQ3JDLEtBQUssTUFBTSxFQUFFLElBQUksUUFBUSxFQUFFO1lBQ3pCLElBQUksR0FBRyxJQUFJLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBRSxRQUFRLENBQUMsRUFBRSxDQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRyxRQUFRLENBQUMsRUFBRSxDQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3pFO0tBQ0Y7SUFDRCxPQUFPLHFCQUFxQixDQUFJLEtBQUssSUFBSSxFQUFFO1FBQ3pDLE1BQU0sT0FBTyxHQUFHLE1BQU0sa0JBQWtCLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxFQUFFO1lBQ3JELElBQUksQ0FBQyxZQUFZLE9BQU8sRUFBRTtnQkFDeEIsT0FBTyxFQUFDLEtBQUssRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBQyxDQUFDO2FBQzlDO2lCQUFNLElBQUksVUFBVSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUN4QixPQUFPLEVBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFDLENBQUM7YUFDckM7aUJBQU07Z0JBQ0wsTUFBTSxJQUFJLEtBQUssQ0FDWCw0REFBNEQ7b0JBQzVELGlCQUFpQixDQUFDLENBQUM7YUFDeEI7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILE9BQU8sa0JBQWtCLENBQUksT0FBTyxFQUFFLGVBQWUsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNsRSxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDWCxDQUFDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsa0NBQWtDO0FBQ2xDLFNBQVMsZUFBZSxDQUFDLElBQVc7SUFDbEMsSUFBSSxJQUFJLEtBQUssSUFBSSxFQUFFO1FBQ2pCLE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFFRCxpRUFBaUU7SUFDakUsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRTNCLElBQUksWUFBWSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1FBQzVCLG1FQUFtRTtRQUNuRSxNQUFNLEtBQUssR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDaEMsT0FBTyxFQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFDLENBQUM7S0FDaEM7SUFFRCxvREFBb0Q7SUFDcEQsT0FBTyxFQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBQyxDQUFDO0FBQ3RDLENBQUM7QUFFRDs7O0dBR0c7QUFDSCxTQUFTLFdBQVcsQ0FBb0MsTUFBVztJQUVqRSxJQUFJLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3ZCLDJFQUEyRTtRQUMzRSxNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7S0FDMUQ7SUFFRCxJQUFJLE1BQU0sQ0FBQyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUMsTUFBTSxFQUFFO1FBQ2xDLCtCQUErQjtRQUMvQixPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBcUIsQ0FBQyxDQUFDO0tBQ3hDO1NBQU07UUFDTCwrQ0FBK0M7UUFDL0MsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQW9CLENBQUMsQ0FBQztLQUN4QztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgMjAxOCBHb29nbGUgTExDLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKTtcbiAqIHlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiAqIFlvdSBtYXkgb2J0YWluIGEgY29weSBvZiB0aGUgTGljZW5zZSBhdFxuICpcbiAqIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbiAqIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMgSVNcIiBCQVNJUyxcbiAqIFdJVEhPVVQgV0FSUkFOVElFUyBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBlaXRoZXIgZXhwcmVzcyBvciBpbXBsaWVkLlxuICogU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuICogbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKlxuICogPT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT1cbiAqL1xuXG5pbXBvcnQgKiBhcyB0ZiBmcm9tICdAdGVuc29yZmxvdy90ZmpzLWNvcmUnO1xuaW1wb3J0IHtUZW5zb3JDb250YWluZXIsIFRlbnNvckxpa2V9IGZyb20gJ0B0ZW5zb3JmbG93L3RmanMtY29yZSc7XG5pbXBvcnQgKiBhcyBzZWVkcmFuZG9tIGZyb20gJ3NlZWRyYW5kb20nO1xuXG5pbXBvcnQge2l0ZXJhdG9yRnJvbUNvbmNhdGVuYXRlZCwgaXRlcmF0b3JGcm9tRnVuY3Rpb24sIGl0ZXJhdG9yRnJvbUl0ZW1zLCBpdGVyYXRvckZyb21aaXBwZWQsIExhenlJdGVyYXRvciwgWmlwTWlzbWF0Y2hNb2RlfSBmcm9tICcuL2l0ZXJhdG9ycy9sYXp5X2l0ZXJhdG9yJztcbmltcG9ydCB7Q29udGFpbmVyfSBmcm9tICcuL3R5cGVzJztcbmltcG9ydCB7Y2FuVGVuc29yaWZ5LCBkZWVwTWFwQW5kQXdhaXRBbGwsIERlZXBNYXBSZXN1bHQsIGlzSXRlcmFibGV9IGZyb20gJy4vdXRpbC9kZWVwX21hcCc7XG5cbi8qKlxuICogQSBuZXN0ZWQgc3RydWN0dXJlIG9mIERhdGFzZXRzLCB1c2VkIGFzIHRoZSBpbnB1dCB0byB6aXAoKS5cbiAqL1xuZXhwb3J0IHR5cGUgRGF0YXNldENvbnRhaW5lciA9IENvbnRhaW5lcjxEYXRhc2V0PFRlbnNvckNvbnRhaW5lcj4+O1xuXG4vLyBUT0RPKHNvZXJnZWwpOiBjb25zaWRlciB2ZWN0b3JpemVkIG9wZXJhdGlvbnMgd2l0aGluIHRoZSBwaXBlbGluZS5cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgcG90ZW50aWFsbHkgbGFyZ2UgbGlzdCBvZiBpbmRlcGVuZGVudCBkYXRhIGVsZW1lbnRzICh0eXBpY2FsbHlcbiAqICdzYW1wbGVzJyBvciAnZXhhbXBsZXMnKS5cbiAqXG4gKiBBICdkYXRhIGV4YW1wbGUnIG1heSBiZSBhIHByaW1pdGl2ZSwgYW4gYXJyYXksIGEgbWFwIGZyb20gc3RyaW5nIGtleXMgdG9cbiAqIHZhbHVlcywgb3IgYW55IG5lc3RlZCBzdHJ1Y3R1cmUgb2YgdGhlc2UuXG4gKlxuICogQSBgRGF0YXNldGAgcmVwcmVzZW50cyBhbiBvcmRlcmVkIGNvbGxlY3Rpb24gb2YgZWxlbWVudHMsIHRvZ2V0aGVyIHdpdGggYVxuICogY2hhaW4gb2YgdHJhbnNmb3JtYXRpb25zIHRvIGJlIHBlcmZvcm1lZCBvbiB0aG9zZSBlbGVtZW50cy4gRWFjaFxuICogdHJhbnNmb3JtYXRpb24gaXMgYSBtZXRob2Qgb2YgYERhdGFzZXRgIHRoYXQgcmV0dXJucyBhbm90aGVyIGBEYXRhc2V0YCwgc29cbiAqIHRoZXNlIG1heSBiZSBjaGFpbmVkLCBlLmcuXG4gKiBgY29uc3QgcHJvY2Vzc2VkRGF0YXNldCA9IHJhd0RhdGFzZXQuZmlsdGVyKC4uLikubWFwKC4uLikuYmF0Y2goLi4uKWAuXG4gKlxuICogRGF0YSBsb2FkaW5nIGFuZCB0cmFuc2Zvcm1hdGlvbiBpcyBkb25lIGluIGEgbGF6eSwgc3RyZWFtaW5nIGZhc2hpb24uICBUaGVcbiAqIGRhdGFzZXQgbWF5IGJlIGl0ZXJhdGVkIG92ZXIgbXVsdGlwbGUgdGltZXM7IGVhY2ggaXRlcmF0aW9uIHN0YXJ0cyB0aGUgZGF0YVxuICogbG9hZGluZyBhbmV3IGFuZCByZWNhcGl0dWxhdGVzIHRoZSB0cmFuc2Zvcm1hdGlvbnMuXG4gKlxuICogQSBgRGF0YXNldGAgaXMgdHlwaWNhbGx5IHByb2Nlc3NlZCBhcyBhIHN0cmVhbSBvZiB1bmJhdGNoZWQgZXhhbXBsZXMgLS0gaS5lLixcbiAqIGl0cyB0cmFuc2Zvcm1hdGlvbnMgYXJlIGFwcGxpZWQgb25lIGV4YW1wbGUgYXQgYSB0aW1lLiBCYXRjaGluZyBwcm9kdWNlcyBhXG4gKiBuZXcgYERhdGFzZXRgIHdoZXJlIGVhY2ggZWxlbWVudCBpcyBhIGJhdGNoLiBCYXRjaGluZyBzaG91bGQgdXN1YWxseSBjb21lXG4gKiBsYXN0IGluIGEgcGlwZWxpbmUsIGJlY2F1c2UgZGF0YSB0cmFuc2Zvcm1hdGlvbnMgYXJlIGVhc2llciB0byBleHByZXNzIG9uIGFcbiAqIHBlci1leGFtcGxlIGJhc2lzIHRoYW4gb24gYSBwZXItYmF0Y2ggYmFzaXMuXG4gKlxuICogVGhlIGZvbGxvd2luZyBjb2RlIGV4YW1wbGVzIGFyZSBjYWxsaW5nIGBhd2FpdCBkYXRhc2V0LmZvckVhY2hBc3luYyguLi4pYCB0b1xuICogaXRlcmF0ZSBvbmNlIG92ZXIgdGhlIGVudGlyZSBkYXRhc2V0IGluIG9yZGVyIHRvIHByaW50IG91dCB0aGUgZGF0YS5cbiAqXG4gKiBAZG9jIHtoZWFkaW5nOiAnRGF0YScsIHN1YmhlYWRpbmc6ICdDbGFzc2VzJywgbmFtZXNwYWNlOiAnZGF0YSd9XG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBEYXRhc2V0PFQgZXh0ZW5kcyB0Zi5UZW5zb3JDb250YWluZXI+IHtcbiAgLypcbiAgICogUHJvdmlkZSBhIG5ldyBzdHJlYW0gb2YgZWxlbWVudHMuICBOb3RlIHRoaXMgd2lsbCBhbHNvIHN0YXJ0IG5ldyBzdHJlYW1zXG4gICAqIGZyb20gYW55IHVuZGVybHlpbmcgYERhdGFzZXRgcy5cbiAgICpcbiAgICogQ0FVVElPTjogQW55IFRlbnNvcnMgY29udGFpbmVkIHdpdGhpbiB0aGUgZWxlbWVudHMgcmV0dXJuZWQgZnJvbVxuICAgKiB0aGlzIHN0cmVhbSAqbXVzdCogYmUgbWFudWFsbHkgZGlzcG9zZWQgdG8gYXZvaWQgYSBHUFUgbWVtb3J5IGxlYWsuXG4gICAqIFRoZSB0Zi50aWR5KCkgYXBwcm9hY2ggY2Fubm90IGJlIHVzZWQgaW4gYW4gYXN5bmNocm9ub3VzIGNvbnRleHQuXG4gICAqL1xuICBhYnN0cmFjdCBpdGVyYXRvcigpOiBQcm9taXNlPExhenlJdGVyYXRvcjxUPj47XG5cbiAgcmVhZG9ubHkgc2l6ZTogbnVtYmVyID0gbnVsbDtcblxuICAvLyBUT0RPKHNvZXJnZWwpOiBNYWtlIERhdGFzZXRzIHJlcG9ydCB3aGV0aGVyIHJlcGVhdGVkIGl0ZXJhdG9yKCkgY2FsbHNcbiAgLy8gcHJvZHVjZSB0aGUgc2FtZSByZXN1bHQgKGUuZy4sIHJlYWRpbmcgZnJvbSBhIGZpbGUpIG9yIGRpZmZlcmVudCByZXN1bHRzXG4gIC8vIChlLmcuLCBmcm9tIHRoZSB3ZWJjYW0pLiAgQ3VycmVudGx5IHdlIGRvbid0IG1ha2UgdGhpcyBkaXN0aW5jdGlvbiBidXQgaXRcbiAgLy8gY291bGQgYmUgaW1wb3J0YW50IGZvciB0aGUgdXNlciB0byBrbm93LlxuICAvLyBhYnN0cmFjdCBpc0RldGVybWluaXN0aWMoKTogYm9vbGVhbjtcblxuICAvKipcbiAgICogR3JvdXBzIGVsZW1lbnRzIGludG8gYmF0Y2hlcy5cbiAgICpcbiAgICogSXQgaXMgYXNzdW1lZCB0aGF0IGVhY2ggb2YgdGhlIGluY29taW5nIGRhdGFzZXQgZWxlbWVudHMgaGFzIHRoZSBzYW1lXG4gICAqIHN0cnVjdHVyZSAtLSBpLmUuIHRoZSBzYW1lIHNldCBvZiBrZXlzIGF0IGVhY2ggbG9jYXRpb24gaW4gYW4gb2JqZWN0XG4gICAqIGhpZXJhcmNoeS4gIEZvciBlYWNoIGtleSwgdGhlIHJlc3VsdGluZyBgRGF0YXNldGAgcHJvdmlkZXMgYSBiYXRjaGVkXG4gICAqIGVsZW1lbnQgY29sbGVjdGluZyBhbGwgb2YgdGhlIGluY29taW5nIHZhbHVlcyBmb3IgdGhhdCBrZXkuXG4gICAqXG4gICAqICAqIEluY29taW5nIHByaW1pdGl2ZXMgYXJlIGdyb3VwZWQgaW50byBhIDEtRCBUZW5zb3IuXG4gICAqICAqIEluY29taW5nIFRlbnNvcnMgYXJlIGdyb3VwZWQgaW50byBhIG5ldyBUZW5zb3Igd2hlcmUgdGhlIDB0aCBheGlzIGlzXG4gICAqICAgIHRoZSBiYXRjaCBkaW1lbnNpb24uXG4gICAqICAqIEluY29taW5nIGFycmF5cyBhcmUgY29udmVydGVkIHRvIFRlbnNvciBhbmQgdGhlbiBiYXRjaGVkLlxuICAgKiAgKiBBIG5lc3RlZCBhcnJheSBpcyBpbnRlcnByZXRlZCBhcyBhbiBuLUQgVGVuc29yLCBzbyB0aGUgYmF0Y2hlZCByZXN1bHRcbiAgICogICAgaGFzIG4rMSBkaW1lbnNpb25zLlxuICAgKiAgKiBBbiBhcnJheSB0aGF0IGNhbm5vdCBiZSBjb252ZXJ0ZWQgdG8gVGVuc29yIHByb2R1Y2VzIGFuIGVycm9yLlxuICAgKlxuICAgKiBJZiBhbiBhcnJheSBzaG91bGQgbm90IGJlIGJhdGNoZWQgYXMgYSB1bml0LCBpdCBzaG91bGQgZmlyc3QgYmUgY29udmVydGVkXG4gICAqIHRvIGFuIG9iamVjdCB3aXRoIGludGVnZXIga2V5cy5cbiAgICpcbiAgICogSGVyZSBhcmUgYSBmZXcgZXhhbXBsZXM6XG4gICAqXG4gICAqIEJhdGNoIGEgZGF0YXNldCBvZiBudW1iZXJzOlxuICAgKiBgYGBqc1xuICAgKiBjb25zdCBhID0gdGYuZGF0YS5hcnJheShbMSwgMiwgMywgNCwgNSwgNiwgNywgOF0pLmJhdGNoKDQpO1xuICAgKiBhd2FpdCBhLmZvckVhY2hBc3luYyhlID0+IGUucHJpbnQoKSk7XG4gICAqIGBgYFxuICAgKlxuICAgKiBCYXRjaCBhIGRhdGFzZXQgb2YgYXJyYXlzOlxuICAgKiBgYGBqc1xuICAgKiBjb25zdCBiID0gdGYuZGF0YS5hcnJheShbWzFdLCBbMl0sIFszXSwgWzRdLCBbNV0sIFs2XSwgWzddLCBbOF1dKS5iYXRjaCg0KTtcbiAgICogYXdhaXQgYi5mb3JFYWNoQXN5bmMoZSA9PiBlLnByaW50KCkpO1xuICAgKiBgYGBcbiAgICpcbiAgICogQmF0Y2ggYSBkYXRhc2V0IG9mIG9iamVjdHM6XG4gICAqIGBgYGpzXG4gICAqIGNvbnN0IGMgPSB0Zi5kYXRhLmFycmF5KFt7YTogMSwgYjogMTF9LCB7YTogMiwgYjogMTJ9LCB7YTogMywgYjogMTN9LFxuICAgKiAgIHthOiA0LCBiOiAxNH0sIHthOiA1LCBiOiAxNX0sIHthOiA2LCBiOiAxNn0sIHthOiA3LCBiOiAxN30sXG4gICAqICAge2E6IDgsIGI6IDE4fV0pLmJhdGNoKDQpO1xuICAgKiBhd2FpdCBjLmZvckVhY2hBc3luYyhlID0+IHtcbiAgICogICBjb25zb2xlLmxvZygneycpO1xuICAgKiAgIGZvcih2YXIga2V5IGluIGUpIHtcbiAgICogICAgIGNvbnNvbGUubG9nKGtleSsnOicpO1xuICAgKiAgICAgZVtrZXldLnByaW50KCk7XG4gICAqICAgfVxuICAgKiAgIGNvbnNvbGUubG9nKCd9Jyk7XG4gICAqIH0pXG4gICAqIGBgYFxuICAgKlxuICAgKiBAcGFyYW0gYmF0Y2hTaXplIFRoZSBudW1iZXIgb2YgZWxlbWVudHMgZGVzaXJlZCBwZXIgYmF0Y2guXG4gICAqIEBwYXJhbSBzbWFsbExhc3RCYXRjaCBXaGV0aGVyIHRvIGVtaXQgdGhlIGZpbmFsIGJhdGNoIHdoZW4gaXQgaGFzIGZld2VyXG4gICAqICAgdGhhbiBiYXRjaFNpemUgZWxlbWVudHMuIERlZmF1bHQgdHJ1ZS5cbiAgICogQHJldHVybnMgQSBgRGF0YXNldGAsIGZyb20gd2hpY2ggYSBzdHJlYW0gb2YgYmF0Y2hlcyBjYW4gYmUgb2J0YWluZWQuXG4gICAqXG4gICAqIEBkb2Mge2hlYWRpbmc6ICdEYXRhJywgc3ViaGVhZGluZzogJ0NsYXNzZXMnfVxuICAgKi9cbiAgYmF0Y2goYmF0Y2hTaXplOiBudW1iZXIsIHNtYWxsTGFzdEJhdGNoID0gdHJ1ZSk6IERhdGFzZXQ8dGYuVGVuc29yQ29udGFpbmVyPiB7XG4gICAgY29uc3QgYmFzZSA9IHRoaXM7XG4gICAgdGYudXRpbC5hc3NlcnQoXG4gICAgICAgIGJhdGNoU2l6ZSA+IDAsICgpID0+IGBiYXRjaFNpemUgbmVlZHMgdG8gYmUgcG9zaXRpdmUsIGJ1dCBpdCBpc1xuICAgICAgJHtiYXRjaFNpemV9YCk7XG4gICAgbGV0IHNpemU7XG4gICAgaWYgKHRoaXMuc2l6ZSA9PT0gSW5maW5pdHkgfHwgdGhpcy5zaXplID09IG51bGwpIHtcbiAgICAgIC8vIElmIHRoZSBzaXplIG9mIHRoaXMgZGF0YXNldCBpcyBpbmZpbml0eSBvciBudWxsLCB0aGUgbmV3IHNpemUga2VlcHMgdGhlXG4gICAgICAvLyBzYW1lLlxuICAgICAgc2l6ZSA9IHRoaXMuc2l6ZTtcbiAgICB9IGVsc2UgaWYgKHNtYWxsTGFzdEJhdGNoKSB7XG4gICAgICAvLyBJZiB0aGUgc2l6ZSBvZiB0aGlzIGRhdGFzZXQgaXMga25vd24gYW5kIGluY2x1ZGUgc21hbGwgbGFzdCBiYXRjaCwgdGhlXG4gICAgICAvLyBuZXcgc2l6ZSBpcyBmdWxsIGJhdGNoIGNvdW50IHBsdXMgbGFzdCBiYXRjaC5cbiAgICAgIHNpemUgPSBNYXRoLmNlaWwodGhpcy5zaXplIC8gYmF0Y2hTaXplKTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gSWYgdGhlIHNpemUgb2YgdGhpcyBkYXRhc2V0IGlzIGtub3duIGFuZCBub3QgaW5jbHVkZSBzbWFsbCBsYXN0IGJhdGNoLFxuICAgICAgLy8gdGhlIG5ldyBzaXplIGlzIGZ1bGwgYmF0Y2ggY291bnQuXG4gICAgICBzaXplID0gTWF0aC5mbG9vcih0aGlzLnNpemUgLyBiYXRjaFNpemUpO1xuICAgIH1cbiAgICByZXR1cm4gZGF0YXNldEZyb21JdGVyYXRvckZuKGFzeW5jICgpID0+IHtcbiAgICAgIHJldHVybiAoYXdhaXQgYmFzZS5pdGVyYXRvcigpKVxuICAgICAgICAgIC5jb2x1bW5NYWpvckJhdGNoKGJhdGNoU2l6ZSwgc21hbGxMYXN0QmF0Y2gsIGRlZXBCYXRjaENvbmNhdCk7XG4gICAgfSwgc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogQ29uY2F0ZW5hdGVzIHRoaXMgYERhdGFzZXRgIHdpdGggYW5vdGhlci5cbiAgICpcbiAgICogYGBganNcbiAgICogY29uc3QgYSA9IHRmLmRhdGEuYXJyYXkoWzEsIDIsIDNdKTtcbiAgICogY29uc3QgYiA9IHRmLmRhdGEuYXJyYXkoWzQsIDUsIDZdKTtcbiAgICogY29uc3QgYyA9IGEuY29uY2F0ZW5hdGUoYik7XG4gICAqIGF3YWl0IGMuZm9yRWFjaEFzeW5jKGUgPT4gY29uc29sZS5sb2coZSkpO1xuICAgKiBgYGBcbiAgICpcbiAgICogQHBhcmFtIGRhdGFzZXQgQSBgRGF0YXNldGAgdG8gYmUgY29uY2F0ZW5hdGVkIG9udG8gdGhpcyBvbmUuXG4gICAqIEByZXR1cm5zIEEgYERhdGFzZXRgLlxuICAgKlxuICAgKiBAZG9jIHtoZWFkaW5nOiAnRGF0YScsIHN1YmhlYWRpbmc6ICdDbGFzc2VzJ31cbiAgICovXG4gIGNvbmNhdGVuYXRlKGRhdGFzZXQ6IERhdGFzZXQ8VD4pOiBEYXRhc2V0PFQ+IHtcbiAgICBjb25zdCBiYXNlID0gdGhpcztcbiAgICBsZXQgc2l6ZTtcbiAgICBpZiAodGhpcy5zaXplID09PSBJbmZpbml0eSB8fCBkYXRhc2V0LnNpemUgPT09IEluZmluaXR5KSB7XG4gICAgICAvLyBJZiB0aGUgc2l6ZSBvZiBhbnkgb2YgdGhlc2UgdHdvIGRhdGFzZXQgaXMgaW5maW5pdHksIG5ldyBzaXplIGlzXG4gICAgICAvLyBpbmZpbml0eS5cbiAgICAgIHNpemUgPSBJbmZpbml0eTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuc2l6ZSAhPSBudWxsICYmIGRhdGFzZXQuc2l6ZSAhPSBudWxsKSB7XG4gICAgICAvLyBJZiB0aGUgc2l6ZSBvZiBib3RoIGRhdGFzZXRzIGFyZSBrbm93biBhbmQgbm90IGluZmluaXR5LCBuZXcgc2l6ZSBpc1xuICAgICAgLy8gc3VtIHRoZSBzaXplIG9mIHRoZXNlIHR3byBkYXRhc2V0cy5cbiAgICAgIHNpemUgPSB0aGlzLnNpemUgKyBkYXRhc2V0LnNpemU7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIElmIG5laXRoZXIgb2YgdGhlc2UgdHdvIGRhdGFzZXRzIGhhcyBpbmZpbml0ZSBzaXplIGFuZCBhbnkgb2YgdGhlc2UgdHdvXG4gICAgICAvLyBkYXRhc2V0cycgc2l6ZSBpcyBudWxsLCB0aGUgbmV3IHNpemUgaXMgbnVsbC5cbiAgICAgIHNpemUgPSBudWxsO1xuICAgIH1cbiAgICByZXR1cm4gZGF0YXNldEZyb21JdGVyYXRvckZuKFxuICAgICAgICBhc3luYyAoKSA9PlxuICAgICAgICAgICAgKGF3YWl0IGJhc2UuaXRlcmF0b3IoKSkuY29uY2F0ZW5hdGUoYXdhaXQgZGF0YXNldC5pdGVyYXRvcigpKSxcbiAgICAgICAgc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyB0aGlzIGRhdGFzZXQgYWNjb3JkaW5nIHRvIGBwcmVkaWNhdGVgLlxuICAgKlxuICAgKiBgYGBqc1xuICAgKiBjb25zdCBhID0gdGYuZGF0YS5hcnJheShbMSwgMiwgMywgNCwgNSwgNiwgNywgOCwgOSwgMTBdKVxuICAgKiAgIC5maWx0ZXIoeCA9PiB4JTIgPT09IDApO1xuICAgKiBhd2FpdCBhLmZvckVhY2hBc3luYyhlID0+IGNvbnNvbGUubG9nKGUpKTtcbiAgICogYGBgXG4gICAqXG4gICAqIEBwYXJhbSBwcmVkaWNhdGUgQSBmdW5jdGlvbiBtYXBwaW5nIGEgZGF0YXNldCBlbGVtZW50IHRvIGEgYm9vbGVhbiBvciBhXG4gICAqIGBQcm9taXNlYCBmb3Igb25lLlxuICAgKlxuICAgKiBAcmV0dXJucyBBIGBEYXRhc2V0YCBvZiBlbGVtZW50cyBmb3Igd2hpY2ggdGhlIHByZWRpY2F0ZSB3YXMgdHJ1ZS5cbiAgICpcbiAgICogQGRvYyB7aGVhZGluZzogJ0RhdGEnLCBzdWJoZWFkaW5nOiAnQ2xhc3Nlcyd9XG4gICAqL1xuICBmaWx0ZXIocHJlZGljYXRlOiAodmFsdWU6IFQpID0+IGJvb2xlYW4pOiBEYXRhc2V0PFQ+IHtcbiAgICBjb25zdCBiYXNlID0gdGhpcztcbiAgICBsZXQgc2l6ZTtcbiAgICBpZiAodGhpcy5zaXplID09PSBJbmZpbml0eSkge1xuICAgICAgLy8gSWYgdGhlIHNpemUgb2YgdGhpcyBkYXRhc2V0IGlzIGluZmluaXR5LCBuZXcgc2l6ZSBpcyBpbmZpbml0eVxuICAgICAgc2l6ZSA9IEluZmluaXR5O1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBJZiB0aGlzIGRhdGFzZXQgaGFzIGxpbWl0ZWQgZWxlbWVudHMsIG5ldyBzaXplIGlzIG51bGwgYmVjYXVzZSBpdCBtaWdodFxuICAgICAgLy8gZXhoYXVzdGVkIHJhbmRvbWx5LlxuICAgICAgc2l6ZSA9IG51bGw7XG4gICAgfVxuICAgIHJldHVybiBkYXRhc2V0RnJvbUl0ZXJhdG9yRm4oYXN5bmMgKCkgPT4ge1xuICAgICAgcmV0dXJuIChhd2FpdCBiYXNlLml0ZXJhdG9yKCkpLmZpbHRlcih4ID0+IHRmLnRpZHkoKCkgPT4gcHJlZGljYXRlKHgpKSk7XG4gICAgfSwgc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogQXBwbHkgYSBmdW5jdGlvbiB0byBldmVyeSBlbGVtZW50IG9mIHRoZSBkYXRhc2V0LlxuICAgKlxuICAgKiBBZnRlciB0aGUgZnVuY3Rpb24gaXMgYXBwbGllZCB0byBhIGRhdGFzZXQgZWxlbWVudCwgYW55IFRlbnNvcnMgY29udGFpbmVkXG4gICAqIHdpdGhpbiB0aGF0IGVsZW1lbnQgYXJlIGRpc3Bvc2VkLlxuICAgKlxuICAgKiBgYGBqc1xuICAgKiBjb25zdCBhID0gdGYuZGF0YS5hcnJheShbMSwgMiwgM10pO1xuICAgKiBhd2FpdCBhLmZvckVhY2hBc3luYyhlID0+IGNvbnNvbGUubG9nKGUpKTtcbiAgICogYGBgXG4gICAqXG4gICAqIEBwYXJhbSBmIEEgZnVuY3Rpb24gdG8gYXBwbHkgdG8gZWFjaCBkYXRhc2V0IGVsZW1lbnQuXG4gICAqIEByZXR1cm5zIEEgYFByb21pc2VgIHRoYXQgcmVzb2x2ZXMgYWZ0ZXIgYWxsIGVsZW1lbnRzIGhhdmUgYmVlbiBwcm9jZXNzZWQuXG4gICAqXG4gICAqIEBkb2Mge2hlYWRpbmc6ICdEYXRhJywgc3ViaGVhZGluZzogJ0NsYXNzZXMnfVxuICAgKi9cbiAgYXN5bmMgZm9yRWFjaEFzeW5jKGY6IChpbnB1dDogVCkgPT4gdm9pZCk6IFByb21pc2U8dm9pZD4ge1xuICAgIHJldHVybiAoYXdhaXQgdGhpcy5pdGVyYXRvcigpKS5mb3JFYWNoQXN5bmMoZik7XG4gIH1cblxuICAvKipcbiAgICogTWFwcyB0aGlzIGRhdGFzZXQgdGhyb3VnaCBhIDEtdG8tMSB0cmFuc2Zvcm0uXG4gICAqXG4gICAqIGBgYGpzXG4gICAqIGNvbnN0IGEgPSB0Zi5kYXRhLmFycmF5KFsxLCAyLCAzXSkubWFwKHggPT4geCp4KTtcbiAgICogYXdhaXQgYS5mb3JFYWNoQXN5bmMoZSA9PiBjb25zb2xlLmxvZyhlKSk7XG4gICAqIGBgYFxuICAgKlxuICAgKiBAcGFyYW0gdHJhbnNmb3JtIEEgZnVuY3Rpb24gbWFwcGluZyBhIGRhdGFzZXQgZWxlbWVudCB0byBhIHRyYW5zZm9ybWVkXG4gICAqICAgZGF0YXNldCBlbGVtZW50LlxuICAgKlxuICAgKiBAcmV0dXJucyBBIGBEYXRhc2V0YCBvZiB0cmFuc2Zvcm1lZCBlbGVtZW50cy5cbiAgICpcbiAgICogQGRvYyB7aGVhZGluZzogJ0RhdGEnLCBzdWJoZWFkaW5nOiAnQ2xhc3Nlcyd9XG4gICAqL1xuICBtYXA8TyBleHRlbmRzIHRmLlRlbnNvckNvbnRhaW5lcj4odHJhbnNmb3JtOiAodmFsdWU6IFQpID0+IE8pOiBEYXRhc2V0PE8+IHtcbiAgICBjb25zdCBiYXNlID0gdGhpcztcbiAgICByZXR1cm4gZGF0YXNldEZyb21JdGVyYXRvckZuKGFzeW5jICgpID0+IHtcbiAgICAgIHJldHVybiAoYXdhaXQgYmFzZS5pdGVyYXRvcigpKS5tYXAoeCA9PiB0Zi50aWR5KCgpID0+IHRyYW5zZm9ybSh4KSkpO1xuICAgIH0sIHRoaXMuc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogTWFwcyB0aGlzIGRhdGFzZXQgdGhyb3VnaCBhbiBhc3luYyAxLXRvLTEgdHJhbnNmb3JtLlxuICAgKlxuICAgKiBgYGBqc1xuICAgKiBjb25zdCBhID1cbiAgICogIHRmLmRhdGEuYXJyYXkoWzEsIDIsIDNdKS5tYXBBc3luYyh4ID0+IG5ldyBQcm9taXNlKGZ1bmN0aW9uKHJlc29sdmUpe1xuICAgKiAgICBzZXRUaW1lb3V0KCgpID0+IHtcbiAgICogICAgICByZXNvbHZlKHggKiB4KTtcbiAgICogICAgfSwgTWF0aC5yYW5kb20oKSoxMDAwICsgNTAwKTtcbiAgICogIH0pKTtcbiAgICogY29uc29sZS5sb2coYXdhaXQgYS50b0FycmF5KCkpO1xuICAgKiBgYGBcbiAgICpcbiAgICogQHBhcmFtIHRyYW5zZm9ybSBBIGZ1bmN0aW9uIG1hcHBpbmcgYSBkYXRhc2V0IGVsZW1lbnQgdG8gYSBgUHJvbWlzZWAgZm9yIGFcbiAgICogICB0cmFuc2Zvcm1lZCBkYXRhc2V0IGVsZW1lbnQuICBUaGlzIHRyYW5zZm9ybSBpcyByZXNwb25zaWJsZSBmb3IgZGlzcG9zaW5nXG4gICAqICAgYW55IGludGVybWVkaWF0ZSBgVGVuc29yYHMsIGkuZS4gYnkgd3JhcHBpbmcgaXRzIGNvbXB1dGF0aW9uIGluXG4gICAqICAgYHRmLnRpZHkoKWA7IHRoYXQgY2Fubm90IGJlIGF1dG9tYXRlZCBoZXJlIChhcyBpdCBpcyBpbiB0aGUgc3luY2hyb25vdXNcbiAgICogICBgbWFwKClgIGNhc2UpLlxuICAgKlxuICAgKiBAcmV0dXJucyBBIGBEYXRhc2V0YCBvZiB0cmFuc2Zvcm1lZCBlbGVtZW50cy5cbiAgICpcbiAgICogQGRvYyB7aGVhZGluZzogJ0RhdGEnLCBzdWJoZWFkaW5nOiAnQ2xhc3Nlcyd9XG4gICAqL1xuICBtYXBBc3luYzxPIGV4dGVuZHMgdGYuVGVuc29yQ29udGFpbmVyPih0cmFuc2Zvcm06ICh2YWx1ZTogVCkgPT4gUHJvbWlzZTxPPik6XG4gICAgICBEYXRhc2V0PE8+IHtcbiAgICBjb25zdCBiYXNlID0gdGhpcztcbiAgICByZXR1cm4gZGF0YXNldEZyb21JdGVyYXRvckZuKGFzeW5jICgpID0+IHtcbiAgICAgIHJldHVybiAoYXdhaXQgYmFzZS5pdGVyYXRvcigpKS5tYXBBc3luYyh0cmFuc2Zvcm0pO1xuICAgIH0sIHRoaXMuc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogIENyZWF0ZXMgYSBgRGF0YXNldGAgdGhhdCBwcmVmZXRjaGVzIGVsZW1lbnRzIGZyb20gdGhpcyBkYXRhc2V0LlxuICAgKlxuICAgKiBAcGFyYW0gYnVmZmVyU2l6ZTogQW4gaW50ZWdlciBzcGVjaWZ5aW5nIHRoZSBudW1iZXIgb2YgZWxlbWVudHMgdG8gYmVcbiAgICogICBwcmVmZXRjaGVkLlxuICAgKiBAcmV0dXJucyBBIGBEYXRhc2V0YC5cbiAgICpcbiAgICogQGRvYyB7aGVhZGluZzogJ0RhdGEnLCBzdWJoZWFkaW5nOiAnQ2xhc3Nlcyd9XG4gICAqL1xuICBwcmVmZXRjaChidWZmZXJTaXplOiBudW1iZXIpOiBEYXRhc2V0PFQ+IHtcbiAgICBpZiAoYnVmZmVyU2l6ZSA9PSBudWxsKSB7XG4gICAgICB0aHJvdyBuZXcgUmFuZ2VFcnJvcihcbiAgICAgICAgICAnYERhdGFzZXQucHJlZmV0Y2goKWAgcmVxdWlyZXMgYnVmZmVyU2l6ZSB0byBiZSBzcGVjaWZpZWQuJyk7XG4gICAgfVxuXG4gICAgY29uc3QgYmFzZSA9IHRoaXM7XG4gICAgcmV0dXJuIGRhdGFzZXRGcm9tSXRlcmF0b3JGbihcbiAgICAgICAgYXN5bmMgKCkgPT4gKGF3YWl0IGJhc2UuaXRlcmF0b3IoKSkucHJlZmV0Y2goYnVmZmVyU2l6ZSksIHRoaXMuc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogUmVwZWF0cyB0aGlzIGRhdGFzZXQgYGNvdW50YCB0aW1lcy5cbiAgICpcbiAgICogTk9URTogSWYgdGhpcyBkYXRhc2V0IGlzIGEgZnVuY3Rpb24gb2YgZ2xvYmFsIHN0YXRlIChlLmcuIGEgcmFuZG9tIG51bWJlclxuICAgKiBnZW5lcmF0b3IpLCB0aGVuIGRpZmZlcmVudCByZXBldGl0aW9ucyBtYXkgcHJvZHVjZSBkaWZmZXJlbnQgZWxlbWVudHMuXG4gICAqXG4gICAqIGBgYGpzXG4gICAqIGNvbnN0IGEgPSB0Zi5kYXRhLmFycmF5KFsxLCAyLCAzXSkucmVwZWF0KDMpO1xuICAgKiBhd2FpdCBhLmZvckVhY2hBc3luYyhlID0+IGNvbnNvbGUubG9nKGUpKTtcbiAgICogYGBgXG4gICAqXG4gICAqIEBwYXJhbSBjb3VudDogKE9wdGlvbmFsKSBBbiBpbnRlZ2VyLCByZXByZXNlbnRpbmcgdGhlIG51bWJlciBvZiB0aW1lc1xuICAgKiAgIHRoZSBkYXRhc2V0IHNob3VsZCBiZSByZXBlYXRlZC4gVGhlIGRlZmF1bHQgYmVoYXZpb3IgKGlmIGBjb3VudGAgaXNcbiAgICogICBgdW5kZWZpbmVkYCBvciBuZWdhdGl2ZSkgaXMgZm9yIHRoZSBkYXRhc2V0IGJlIHJlcGVhdGVkIGluZGVmaW5pdGVseS5cbiAgICogQHJldHVybnMgQSBgRGF0YXNldGAuXG4gICAqXG4gICAqIEBkb2Mge2hlYWRpbmc6ICdEYXRhJywgc3ViaGVhZGluZzogJ0NsYXNzZXMnfVxuICAgKi9cbiAgcmVwZWF0KGNvdW50PzogbnVtYmVyKTogRGF0YXNldDxUPiB7XG4gICAgY29uc3QgYmFzZSA9IHRoaXM7XG4gICAgbGV0IHNpemU7XG4gICAgaWYgKHRoaXMuc2l6ZSAhPSBudWxsICYmIGNvdW50ID4gMCkge1xuICAgICAgLy8gSWYgdGhpcyBkYXRhc2V0IGhhcyBzaXplIGFuZCBjb3VudCBpcyBwb3NpdGl2ZSwgbmV3IHNpemUgaXMgY3VycmVudFxuICAgICAgLy8gc2l6ZSBtdWx0aXBseSBjb3VudC4gVGhpcyBhbHNvIGNvdmVycyB0aGUgY2FzZSB0aGF0IGN1cnJlbnQgc2l6ZSBpc1xuICAgICAgLy8gaW5maW5pdHkuXG4gICAgICBzaXplID0gdGhpcy5zaXplICogY291bnQ7XG4gICAgfSBlbHNlIGlmIChjb3VudCA9PT0gMCkge1xuICAgICAgLy8gSWYgY291bnQgaXMgMCwgbmV3IHNpemUgaXMgMC5cbiAgICAgIHNpemUgPSAwO1xuICAgIH0gZWxzZSBpZiAodGhpcy5zaXplICE9IG51bGwgJiYgKGNvdW50ID09PSB1bmRlZmluZWQgfHwgY291bnQgPCAwKSkge1xuICAgICAgLy8gSWYgdGhpcyBkYXRhc2V0IGhhcyBzaXplIGFuZCBjb3VudCBpcyB1bmRlZmluZWQgb3IgbmVnYXRpdmUsIHRoZVxuICAgICAgLy8gZGF0YXNldCB3aWxsIGJlIHJlcGVhdGVkIGluZGVmaW5pdGVseSBhbmQgbmV3IHNpemUgaXMgaW5maW5pdHkuXG4gICAgICBzaXplID0gSW5maW5pdHk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIElmIHRoZSBzaXplIG9mIHRoaXMgZGF0YXNldCBpcyBudWxsLCB0aGUgbmV3IGRhdGFzZXQncyBzaXplIGlzIG51bGwuXG4gICAgICBzaXplID0gbnVsbDtcbiAgICB9XG4gICAgcmV0dXJuIGRhdGFzZXRGcm9tSXRlcmF0b3JGbihhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCBpdGVyYXRvckl0ZXJhdG9yID0gaXRlcmF0b3JGcm9tRnVuY3Rpb24oXG4gICAgICAgICAgYXN5bmMgKCkgPT4gKHt2YWx1ZTogYXdhaXQgYmFzZS5pdGVyYXRvcigpLCBkb25lOiBmYWxzZX0pKTtcbiAgICAgIHJldHVybiBpdGVyYXRvckZyb21Db25jYXRlbmF0ZWQoaXRlcmF0b3JJdGVyYXRvci50YWtlKGNvdW50KSk7XG4gICAgfSwgc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIGBEYXRhc2V0YCB0aGF0IHNraXBzIGBjb3VudGAgaW5pdGlhbCBlbGVtZW50cyBmcm9tIHRoaXMgZGF0YXNldC5cbiAgICpcbiAgICogYGBganNcbiAgICogY29uc3QgYSA9IHRmLmRhdGEuYXJyYXkoWzEsIDIsIDMsIDQsIDUsIDZdKS5za2lwKDMpO1xuICAgKiBhd2FpdCBhLmZvckVhY2hBc3luYyhlID0+IGNvbnNvbGUubG9nKGUpKTtcbiAgICogYGBgXG4gICAqXG4gICAqIEBwYXJhbSBjb3VudDogVGhlIG51bWJlciBvZiBlbGVtZW50cyBvZiB0aGlzIGRhdGFzZXQgdGhhdCBzaG91bGQgYmUgc2tpcHBlZFxuICAgKiAgIHRvIGZvcm0gdGhlIG5ldyBkYXRhc2V0LiAgSWYgYGNvdW50YCBpcyBncmVhdGVyIHRoYW4gdGhlIHNpemUgb2YgdGhpc1xuICAgKiAgIGRhdGFzZXQsIHRoZSBuZXcgZGF0YXNldCB3aWxsIGNvbnRhaW4gbm8gZWxlbWVudHMuICBJZiBgY291bnRgXG4gICAqICAgaXMgYHVuZGVmaW5lZGAgb3IgbmVnYXRpdmUsIHNraXBzIHRoZSBlbnRpcmUgZGF0YXNldC5cbiAgICpcbiAgICogQHJldHVybnMgQSBgRGF0YXNldGAuXG4gICAqXG4gICAqIEBkb2Mge2hlYWRpbmc6ICdEYXRhJywgc3ViaGVhZGluZzogJ0NsYXNzZXMnfVxuICAgKi9cbiAgc2tpcChjb3VudDogbnVtYmVyKTogRGF0YXNldDxUPiB7XG4gICAgY29uc3QgYmFzZSA9IHRoaXM7XG4gICAgbGV0IHNpemU7XG4gICAgaWYgKHRoaXMuc2l6ZSAhPSBudWxsICYmIGNvdW50ID49IDAgJiYgdGhpcy5zaXplID49IGNvdW50KSB7XG4gICAgICAvLyBJZiB0aGUgc2l6ZSBvZiB0aGlzIGRhdGFzZXQgaXMgZ3JlYXRlciB0aGFuIGNvdW50LCB0aGUgbmV3IGRhdGFzZXQnc1xuICAgICAgLy8gc2l6ZSBpcyBjdXJyZW50IHNpemUgbWludXMgc2tpcHBlZCBzaXplLlRoaXMgYWxzbyBjb3ZlcnMgdGhlIGNhc2UgdGhhdFxuICAgICAgLy8gY3VycmVudCBzaXplIGlzIGluZmluaXR5LlxuICAgICAgc2l6ZSA9IHRoaXMuc2l6ZSAtIGNvdW50O1xuICAgIH0gZWxzZSBpZiAoXG4gICAgICAgIHRoaXMuc2l6ZSAhPSBudWxsICYmXG4gICAgICAgICh0aGlzLnNpemUgPCBjb3VudCB8fCBjb3VudCA9PT0gdW5kZWZpbmVkIHx8IGNvdW50IDwgMCkpIHtcbiAgICAgIC8vIElmIHRoZSBzaXplIG9mIHRoaXMgZGF0YXNldCBpcyBzbWFsbGVyIHRoYW4gY291bnQsIG9yIGNvdW50IGlzXG4gICAgICAvLyB1bmRlZmluZWQgb3IgbmVnYXRpdmUsIHNraXBzIHRoZSBlbnRpcmUgZGF0YXNldCBhbmQgdGhlIG5ldyBzaXplIGlzIDAuXG4gICAgICBzaXplID0gMDtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gSWYgdGhlIHNpemUgb2YgdGhpcyBkYXRhc2V0IGlzIG51bGwsIHRoZSBuZXcgZGF0YXNldCdzIHNpemUgaXMgbnVsbC5cbiAgICAgIHNpemUgPSBudWxsO1xuICAgIH1cbiAgICByZXR1cm4gZGF0YXNldEZyb21JdGVyYXRvckZuKFxuICAgICAgICBhc3luYyAoKSA9PiAoYXdhaXQgYmFzZS5pdGVyYXRvcigpKS5za2lwKGNvdW50KSwgc2l6ZSk7XG4gIH1cblxuICAvLyBUT0RPKHNvZXJnZWwpOiBkZWVwIHNoYXJkZWQgc2h1ZmZsZSwgd2hlcmUgc3VwcG9ydGVkXG5cbiAgc3RhdGljIHJlYWRvbmx5IE1BWF9CVUZGRVJfU0laRSA9IDEwMDAwO1xuXG4gIC8qKlxuICAgKiBQc2V1ZG9yYW5kb21seSBzaHVmZmxlcyB0aGUgZWxlbWVudHMgb2YgdGhpcyBkYXRhc2V0LiBUaGlzIGlzIGRvbmUgaW4gYVxuICAgKiBzdHJlYW1pbmcgbWFubmVyLCBieSBzYW1wbGluZyBmcm9tIGEgZ2l2ZW4gbnVtYmVyIG9mIHByZWZldGNoZWQgZWxlbWVudHMuXG4gICAqXG4gICAqIGBgYGpzXG4gICAqIGNvbnN0IGEgPSB0Zi5kYXRhLmFycmF5KFsxLCAyLCAzLCA0LCA1LCA2XSkuc2h1ZmZsZSgzKTtcbiAgICogYXdhaXQgYS5mb3JFYWNoQXN5bmMoZSA9PiBjb25zb2xlLmxvZyhlKSk7XG4gICAqIGBgYFxuICAgKlxuICAgKiBAcGFyYW0gYnVmZmVyU2l6ZTogQW4gaW50ZWdlciBzcGVjaWZ5aW5nIHRoZSBudW1iZXIgb2YgZWxlbWVudHMgZnJvbSB0aGlzXG4gICAqICAgZGF0YXNldCBmcm9tIHdoaWNoIHRoZSBuZXcgZGF0YXNldCB3aWxsIHNhbXBsZS5cbiAgICogQHBhcmFtIHNlZWQ6IChPcHRpb25hbCkgQW4gaW50ZWdlciBzcGVjaWZ5aW5nIHRoZSByYW5kb20gc2VlZCB0aGF0IHdpbGxcbiAgICogICBiZSB1c2VkIHRvIGNyZWF0ZSB0aGUgZGlzdHJpYnV0aW9uLlxuICAgKiBAcGFyYW0gcmVzaHVmZmxlRWFjaEl0ZXJhdGlvbjogKE9wdGlvbmFsKSBBIGJvb2xlYW4sIHdoaWNoIGlmIHRydWVcbiAgICogICBpbmRpY2F0ZXMgdGhhdCB0aGUgZGF0YXNldCBzaG91bGQgYmUgcHNldWRvcmFuZG9tbHkgcmVzaHVmZmxlZCBlYWNoIHRpbWVcbiAgICogICBpdCBpcyBpdGVyYXRlZCBvdmVyLiBJZiBmYWxzZSwgZWxlbWVudHMgd2lsbCBiZSByZXR1cm5lZCBpbiB0aGUgc2FtZVxuICAgKiAgIHNodWZmbGVkIG9yZGVyIG9uIGVhY2ggaXRlcmF0aW9uLiAoRGVmYXVsdHMgdG8gYHRydWVgLilcbiAgICogQHJldHVybnMgQSBgRGF0YXNldGAuXG4gICAqXG4gICAqIEBkb2Mge2hlYWRpbmc6ICdEYXRhJywgc3ViaGVhZGluZzogJ0NsYXNzZXMnfVxuICAgKi9cbiAgc2h1ZmZsZShidWZmZXJTaXplOiBudW1iZXIsIHNlZWQ/OiBzdHJpbmcsIHJlc2h1ZmZsZUVhY2hJdGVyYXRpb24gPSB0cnVlKTpcbiAgICAgIERhdGFzZXQ8VD4ge1xuICAgIGlmIChidWZmZXJTaXplID09IG51bGwgfHwgYnVmZmVyU2l6ZSA8IDApIHtcbiAgICAgIGlmICh0aGlzLnNpemUgPT0gbnVsbCkge1xuICAgICAgICB0aHJvdyBuZXcgUmFuZ2VFcnJvcihcbiAgICAgICAgICAgICdgRGF0YXNldC5zaHVmZmxlKClgIHJlcXVpcmVzIGJ1ZmZlclNpemUgdG8gYmUgc3BlY2lmaWVkLicpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgbmV3IFJhbmdlRXJyb3IoXG4gICAgICAgICAgICAnYERhdGFzZXQuc2h1ZmZsZSgpYCByZXF1aXJlcyBidWZmZXJTaXplIHRvIGJlIHNwZWNpZmllZC4gICcgK1xuICAgICAgICAgICAgJ0lmIHlvdXIgZGF0YSBmaXRzIGluIG1haW4gbWVtb3J5IChmb3IgcmVndWxhciBKUyBvYmplY3RzKSwgJyArXG4gICAgICAgICAgICAnYW5kL29yIEdQVSBtZW1vcnkgKGZvciBgdGYuVGVuc29yYHMpLCBjb25zaWRlciBzZXR0aW5nICcgK1xuICAgICAgICAgICAgYGJ1ZmZlclNpemUgdG8gdGhlIGRhdGFzZXQgc2l6ZSAoJHt0aGlzLnNpemV9IGVsZW1lbnRzKWApO1xuICAgICAgfVxuICAgIH1cbiAgICBjb25zdCBiYXNlID0gdGhpcztcbiAgICBjb25zdCByYW5kb20gPSBzZWVkcmFuZG9tLmFsZWEoc2VlZCB8fCB0Zi51dGlsLm5vdygpLnRvU3RyaW5nKCkpO1xuICAgIHJldHVybiBkYXRhc2V0RnJvbUl0ZXJhdG9yRm4oYXN5bmMgKCkgPT4ge1xuICAgICAgbGV0IHNlZWQyID0gcmFuZG9tLmludDMyKCk7XG4gICAgICBpZiAocmVzaHVmZmxlRWFjaEl0ZXJhdGlvbikge1xuICAgICAgICBzZWVkMiArPSByYW5kb20uaW50MzIoKTtcbiAgICAgIH1cbiAgICAgIHJldHVybiAoYXdhaXQgYmFzZS5pdGVyYXRvcigpKS5zaHVmZmxlKGJ1ZmZlclNpemUsIHNlZWQyLnRvU3RyaW5nKCkpO1xuICAgIH0sIHRoaXMuc2l6ZSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIGBEYXRhc2V0YCB3aXRoIGF0IG1vc3QgYGNvdW50YCBpbml0aWFsIGVsZW1lbnRzIGZyb20gdGhpc1xuICAgKiBkYXRhc2V0LlxuICAgKlxuICAgKiBgYGBqc1xuICAgKiBjb25zdCBhID0gdGYuZGF0YS5hcnJheShbMSwgMiwgMywgNCwgNSwgNl0pLnRha2UoMyk7XG4gICAqIGF3YWl0IGEuZm9yRWFjaEFzeW5jKGUgPT4gY29uc29sZS5sb2coZSkpO1xuICAgKiBgYGBcbiAgICpcbiAgICogQHBhcmFtIGNvdW50OiBUaGUgbnVtYmVyIG9mIGVsZW1lbnRzIG9mIHRoaXMgZGF0YXNldCB0aGF0IHNob3VsZCBiZSB0YWtlblxuICAgKiAgIHRvIGZvcm0gdGhlIG5ldyBkYXRhc2V0LiAgSWYgYGNvdW50YCBpcyBgdW5kZWZpbmVkYCBvciBuZWdhdGl2ZSwgb3IgaWZcbiAgICogICBgY291bnRgIGlzIGdyZWF0ZXIgdGhhbiB0aGUgc2l6ZSBvZiB0aGlzIGRhdGFzZXQsIHRoZSBuZXcgZGF0YXNldCB3aWxsXG4gICAqICAgY29udGFpbiBhbGwgZWxlbWVudHMgb2YgdGhpcyBkYXRhc2V0LlxuICAgKiBAcmV0dXJucyBBIGBEYXRhc2V0YC5cbiAgICpcbiAgICogQGRvYyB7aGVhZGluZzogJ0RhdGEnLCBzdWJoZWFkaW5nOiAnQ2xhc3Nlcyd9XG4gICAqL1xuICB0YWtlKGNvdW50OiBudW1iZXIpOiBEYXRhc2V0PFQ+IHtcbiAgICBjb25zdCBiYXNlID0gdGhpcztcbiAgICBsZXQgc2l6ZTtcbiAgICBpZiAodGhpcy5zaXplICE9IG51bGwgJiYgdGhpcy5zaXplID4gY291bnQpIHtcbiAgICAgIC8vIElmIHRoZSBzaXplIG9mIHRoaXMgZGF0YXNldCBpcyBncmVhdGVyIHRoYW4gY291bnQsIHRoZSBuZXcgZGF0YXNldCdzXG4gICAgICAvLyBzaXplIGlzIGNvdW50LlxuICAgICAgc2l6ZSA9IGNvdW50O1xuICAgIH0gZWxzZSBpZiAodGhpcy5zaXplICE9IG51bGwgJiYgdGhpcy5zaXplIDw9IGNvdW50KSB7XG4gICAgICAvLyBJZiB0aGUgc2l6ZSBvZiB0aGlzIGRhdGFzZXQgaXMgZXF1YWwgb3Igc21hbGxlciB0aGFuIGNvdW50LCB0aGUgbmV3XG4gICAgICAvLyBkYXRhc2V0J3Mgc2l6ZSBpcyB0aGUgc2l6ZSBvZiB0aGlzIGRhdGFzZXQuXG4gICAgICBzaXplID0gdGhpcy5zaXplO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBJZiB0aGUgc2l6ZSBvZiB0aGlzIGRhdGFzZXQgaXMgbnVsbCwgdGhlIG5ldyBkYXRhc2V0J3Mgc2l6ZSBpcyBudWxsLlxuICAgICAgc2l6ZSA9IG51bGw7XG4gICAgfVxuICAgIHJldHVybiBkYXRhc2V0RnJvbUl0ZXJhdG9yRm4oXG4gICAgICAgIGFzeW5jICgpID0+IChhd2FpdCBiYXNlLml0ZXJhdG9yKCkpLnRha2UoY291bnQpLCBzaXplKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb2xsZWN0IGFsbCBlbGVtZW50cyBvZiB0aGlzIGRhdGFzZXQgaW50byBhbiBhcnJheS5cbiAgICpcbiAgICogT2J2aW91c2x5IHRoaXMgd2lsbCBzdWNjZWVkIG9ubHkgZm9yIHNtYWxsIGRhdGFzZXRzIHRoYXQgZml0IGluIG1lbW9yeS5cbiAgICogVXNlZnVsIGZvciB0ZXN0aW5nIGFuZCBnZW5lcmFsbHkgc2hvdWxkIGJlIGF2b2lkZWQgaWYgcG9zc2libGUuXG4gICAqXG4gICAqIGBgYGpzXG4gICAqIGNvbnN0IGEgPSB0Zi5kYXRhLmFycmF5KFsxLCAyLCAzLCA0LCA1LCA2XSk7XG4gICAqIGNvbnNvbGUubG9nKGF3YWl0IGEudG9BcnJheSgpKTtcbiAgICogYGBgXG4gICAqXG4gICAqIEByZXR1cm5zIEEgUHJvbWlzZSBmb3IgYW4gYXJyYXkgb2YgZWxlbWVudHMsIHdoaWNoIHdpbGwgcmVzb2x2ZVxuICAgKiAgIHdoZW4gYSBuZXcgc3RyZWFtIGhhcyBiZWVuIG9idGFpbmVkIGFuZCBmdWxseSBjb25zdW1lZC5cbiAgICpcbiAgICogQGRvYyB7aGVhZGluZzogJ0RhdGEnLCBzdWJoZWFkaW5nOiAnQ2xhc3Nlcyd9XG4gICAqL1xuICBhc3luYyB0b0FycmF5KCkge1xuICAgIGlmICh0aGlzLnNpemUgPT09IEluZmluaXR5KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NhbiBub3QgY29udmVydCBpbmZpbml0ZSBkYXRhIHN0cmVhbSB0byBhcnJheS4nKTtcbiAgICB9XG4gICAgcmV0dXJuIChhd2FpdCB0aGlzLml0ZXJhdG9yKCkpLnRvQXJyYXkoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb2xsZWN0IGFsbCBlbGVtZW50cyBvZiB0aGlzIGRhdGFzZXQgaW50byBhbiBhcnJheSB3aXRoIHByZWZldGNoaW5nIDEwMFxuICAgKiBlbGVtZW50cy4gVGhpcyBpcyB1c2VmdWwgZm9yIHRlc3RpbmcsIGJlY2F1c2UgdGhlIHByZWZldGNoIGNoYW5nZXMgdGhlXG4gICAqIG9yZGVyIGluIHdoaWNoIHRoZSBQcm9taXNlcyBhcmUgcmVzb2x2ZWQgYWxvbmcgdGhlIHByb2Nlc3NpbmcgcGlwZWxpbmUuXG4gICAqIFRoaXMgbWF5IGhlbHAgZXhwb3NlIGJ1Z3Mgd2hlcmUgcmVzdWx0cyBhcmUgZGVwZW5kZW50IG9uIHRoZSBvcmRlciBvZlxuICAgKiBQcm9taXNlIHJlc29sdXRpb24gcmF0aGVyIHRoYW4gb24gdGhlIGxvZ2ljYWwgb3JkZXIgb2YgdGhlIHN0cmVhbSAoaS5lLixcbiAgICogZHVlIHRvIGhpZGRlbiBtdXRhYmxlIHN0YXRlKS5cbiAgICpcbiAgICogQHJldHVybnMgQSBQcm9taXNlIGZvciBhbiBhcnJheSBvZiBlbGVtZW50cywgd2hpY2ggd2lsbCByZXNvbHZlXG4gICAqICAgd2hlbiBhIG5ldyBzdHJlYW0gaGFzIGJlZW4gb2J0YWluZWQgYW5kIGZ1bGx5IGNvbnN1bWVkLlxuICAgKi9cbiAgYXN5bmMgdG9BcnJheUZvclRlc3QoKSB7XG4gICAgaWYgKHRoaXMuc2l6ZSA9PT0gSW5maW5pdHkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuIG5vdCBjb252ZXJ0IGluZmluaXRlIGRhdGEgc3RyZWFtIHRvIGFycmF5LicpO1xuICAgIH1cbiAgICByZXR1cm4gKGF3YWl0IHRoaXMuaXRlcmF0b3IoKSkudG9BcnJheUZvclRlc3QoKTtcbiAgfVxufVxuXG4vKipcbiAqIENyZWF0ZSBhIGBEYXRhc2V0YCBkZWZpbmVkIGJ5IGEgcHJvdmlkZWQgaXRlcmF0b3IoKSBmdW5jdGlvbi5cbiAqXG4gKiBgYGBqc1xuICogbGV0IGkgPSAtMTtcbiAqIGNvbnN0IGZ1bmMgPSAoKSA9PlxuICogICAgKytpIDwgNSA/IHt2YWx1ZTogaSwgZG9uZTogZmFsc2V9IDoge3ZhbHVlOiBudWxsLCBkb25lOiB0cnVlfTtcbiAqIGNvbnN0IGl0ZXIgPSB0Zi5kYXRhLml0ZXJhdG9yRnJvbUZ1bmN0aW9uKGZ1bmMpO1xuICogY29uc3QgZHMgPSB0Zi5kYXRhLmRhdGFzZXRGcm9tSXRlcmF0b3JGbihpdGVyKTtcbiAqIGF3YWl0IGRzLmZvckVhY2hBc3luYyhlID0+IGNvbnNvbGUubG9nKGUpKTtcbiAqIGBgYFxuICovXG5leHBvcnQgZnVuY3Rpb24gZGF0YXNldEZyb21JdGVyYXRvckZuPFQgZXh0ZW5kcyB0Zi5UZW5zb3JDb250YWluZXI+KFxuICAgIGl0ZXJhdG9yRm46ICgpID0+IFByb21pc2U8TGF6eUl0ZXJhdG9yPFQ+PixcbiAgICBzaXplOiBudW1iZXIgPSBudWxsKTogRGF0YXNldDxUPiB7XG4gIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBEYXRhc2V0PFQ+IHtcbiAgICBzaXplID0gc2l6ZTtcblxuICAgIC8qXG4gICAgICogUHJvdmlkZSBhIG5ldyBzdHJlYW0gb2YgZWxlbWVudHMuICBOb3RlIHRoaXMgd2lsbCBhbHNvIHN0YXJ0IG5ldyBzdHJlYW1zXG4gICAgICogZnJvbSBhbnkgdW5kZXJseWluZyBgRGF0YXNldGBzLlxuICAgICAqL1xuICAgIGFzeW5jIGl0ZXJhdG9yKCk6IFByb21pc2U8TGF6eUl0ZXJhdG9yPFQ+PiB7XG4gICAgICByZXR1cm4gaXRlcmF0b3JGbigpO1xuICAgIH1cbiAgfVxuICAoKTtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBgRGF0YXNldGAgZnJvbSBhbiBhcnJheSBvZiBlbGVtZW50cy5cbiAqXG4gKiBDcmVhdGUgYSBEYXRhc2V0IGZyb20gYW4gYXJyYXkgb2Ygb2JqZWN0czpcbiAqIGBgYGpzXG4gKiBjb25zdCBhID0gdGYuZGF0YS5hcnJheShbeydpdGVtJzogMX0sIHsnaXRlbSc6IDJ9LCB7J2l0ZW0nOiAzfV0pO1xuICogYXdhaXQgYS5mb3JFYWNoQXN5bmMoZSA9PiBjb25zb2xlLmxvZyhlKSk7XG4gKiBgYGBcbiAqXG4gKiBDcmVhdGUgYSBEYXRhc2V0IGZyb20gYW4gYXJyYXkgb2YgbnVtYmVyczpcbiAqIGBgYGpzXG4gKiBjb25zdCBhID0gdGYuZGF0YS5hcnJheShbNCwgNSwgNl0pO1xuICogYXdhaXQgYS5mb3JFYWNoQXN5bmMoZSA9PiBjb25zb2xlLmxvZyhlKSk7XG4gKiBgYGBcbiAqIEBwYXJhbSBpdGVtcyBBbiBhcnJheSBvZiBlbGVtZW50cyB0aGF0IHdpbGwgYmUgcGFyc2VkIGFzIGl0ZW1zIGluIGEgZGF0YXNldC5cbiAqXG4gKiBAZG9jIHtoZWFkaW5nOiAnRGF0YScsIHN1YmhlYWRpbmc6ICdDcmVhdGlvbicsIG5hbWVzcGFjZTogJ2RhdGEnfVxuICovXG5leHBvcnQgZnVuY3Rpb24gYXJyYXk8VCBleHRlbmRzIHRmLlRlbnNvckNvbnRhaW5lcj4oaXRlbXM6IFRbXSk6IERhdGFzZXQ8VD4ge1xuICByZXR1cm4gZGF0YXNldEZyb21JdGVyYXRvckZuKFxuICAgICAgYXN5bmMgKCkgPT4gaXRlcmF0b3JGcm9tSXRlbXMoaXRlbXMpLCBpdGVtcy5sZW5ndGgpO1xufVxuXG4vKipcbiAqIENyZWF0ZSBhIGBEYXRhc2V0YCBieSB6aXBwaW5nIHRvZ2V0aGVyIGFuIGFycmF5LCBkaWN0LCBvciBuZXN0ZWRcbiAqIHN0cnVjdHVyZSBvZiBgRGF0YXNldGBzIChhbmQgcGVyaGFwcyBhZGRpdGlvbmFsIGNvbnN0YW50cykuXG4gKiBUaGUgdW5kZXJseWluZyBkYXRhc2V0cyBtdXN0IHByb3ZpZGUgZWxlbWVudHMgaW4gYSBjb25zaXN0ZW50IG9yZGVyIHN1Y2ggdGhhdFxuICogdGhleSBjb3JyZXNwb25kLlxuICpcbiAqIFRoZSBudW1iZXIgb2YgZWxlbWVudHMgaW4gdGhlIHJlc3VsdGluZyBkYXRhc2V0IGlzIHRoZSBzYW1lIGFzIHRoZSBzaXplIG9mXG4gKiB0aGUgc21hbGxlc3QgZGF0YXNldCBpbiBkYXRhc2V0cy5cbiAqXG4gKiBUaGUgbmVzdGVkIHN0cnVjdHVyZSBvZiB0aGUgYGRhdGFzZXRzYCBhcmd1bWVudCBkZXRlcm1pbmVzIHRoZVxuICogc3RydWN0dXJlIG9mIGVsZW1lbnRzIGluIHRoZSByZXN1bHRpbmcgaXRlcmF0b3IuXG4gKlxuICogTm90ZSB0aGlzIG1lYW5zIHRoYXQsIGdpdmVuIGFuIGFycmF5IG9mIHR3byBkYXRhc2V0cyB0aGF0IHByb2R1Y2UgZGljdFxuICogZWxlbWVudHMsIHRoZSByZXN1bHQgaXMgYSBkYXRhc2V0IHRoYXQgcHJvZHVjZXMgZWxlbWVudHMgdGhhdCBhcmUgYXJyYXlzXG4gKiBvZiB0d28gZGljdHM6XG4gKlxuICogWmlwIGFuIGFycmF5IG9mIGRhdGFzZXRzOlxuICogYGBganNcbiAqIGNvbnNvbGUubG9nKCdaaXAgdHdvIGRhdGFzZXRzIG9mIG9iamVjdHM6Jyk7XG4gKiBjb25zdCBkczEgPSB0Zi5kYXRhLmFycmF5KFt7YTogMX0sIHthOiAyfSwge2E6IDN9XSk7XG4gKiBjb25zdCBkczIgPSB0Zi5kYXRhLmFycmF5KFt7YjogNH0sIHtiOiA1fSwge2I6IDZ9XSk7XG4gKiBjb25zdCBkczMgPSB0Zi5kYXRhLnppcChbZHMxLCBkczJdKTtcbiAqIGF3YWl0IGRzMy5mb3JFYWNoQXN5bmMoZSA9PiBjb25zb2xlLmxvZyhKU09OLnN0cmluZ2lmeShlKSkpO1xuICpcbiAqIC8vIElmIHRoZSBnb2FsIGlzIHRvIG1lcmdlIHRoZSBkaWN0cyBpbiBvcmRlciB0byBwcm9kdWNlIGVsZW1lbnRzIGxpa2VcbiAqIC8vIHthOiAuLi4sIGI6IC4uLn0sIHRoaXMgcmVxdWlyZXMgYSBzZWNvbmQgc3RlcCBzdWNoIGFzOlxuICogY29uc29sZS5sb2coJ01lcmdlIHRoZSBvYmplY3RzOicpO1xuICogY29uc3QgZHM0ID0gZHMzLm1hcCh4ID0+IHtyZXR1cm4ge2E6IHhbMF0uYSwgYjogeFsxXS5ifX0pO1xuICogYXdhaXQgZHM0LmZvckVhY2hBc3luYyhlID0+IGNvbnNvbGUubG9nKGUpKTtcbiAqIGBgYFxuICpcbiAqIFppcCBhIGRpY3Qgb2YgZGF0YXNldHM6XG4gKiBgYGBqc1xuICogY29uc3QgYSA9IHRmLmRhdGEuYXJyYXkoW3thOiAxfSwge2E6IDJ9LCB7YTogM31dKTtcbiAqIGNvbnN0IGIgPSB0Zi5kYXRhLmFycmF5KFt7YjogNH0sIHtiOiA1fSwge2I6IDZ9XSk7XG4gKiBjb25zdCBjID0gdGYuZGF0YS56aXAoe2M6IGEsIGQ6IGJ9KTtcbiAqIGF3YWl0IGMuZm9yRWFjaEFzeW5jKGUgPT4gY29uc29sZS5sb2coSlNPTi5zdHJpbmdpZnkoZSkpKTtcbiAqIGBgYFxuICpcbiAqIEBkb2Mge2hlYWRpbmc6ICdEYXRhJywgc3ViaGVhZGluZzogJ09wZXJhdGlvbnMnLCBuYW1lc3BhY2U6ICdkYXRhJ31cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHppcDxPIGV4dGVuZHMgdGYuVGVuc29yQ29udGFpbmVyPihkYXRhc2V0czogRGF0YXNldENvbnRhaW5lcik6XG4gICAgRGF0YXNldDxPPiB7XG4gIC8vIG1hbnVhbGx5IHR5cGUtY2hlY2sgdGhlIGFyZ3VtZW50IGZvciBKUyB1c2Vyc1xuICBpZiAoIWlzSXRlcmFibGUoZGF0YXNldHMpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgYXJndW1lbnQgdG8gemlwKCkgbXVzdCBiZSBhbiBvYmplY3Qgb3IgYXJyYXkuJyk7XG4gIH1cbiAgbGV0IHNpemU7XG4gIGlmIChBcnJheS5pc0FycmF5KGRhdGFzZXRzKSkge1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgZGF0YXNldHMubGVuZ3RoOyBpKyspIHtcbiAgICAgIHNpemUgPSBzaXplID09IG51bGwgPyAoZGF0YXNldHNbaV0gYXMgRGF0YXNldDxPPikuc2l6ZSA6XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgTWF0aC5taW4oc2l6ZSwgKGRhdGFzZXRzW2ldIGFzIERhdGFzZXQ8Tz4pLnNpemUpO1xuICAgIH1cbiAgfSBlbHNlIGlmIChkYXRhc2V0cyBpbnN0YW5jZW9mIE9iamVjdCkge1xuICAgIGZvciAoY29uc3QgZHMgaW4gZGF0YXNldHMpIHtcbiAgICAgIHNpemUgPSBzaXplID09IG51bGwgPyAoZGF0YXNldHNbZHNdIGFzIERhdGFzZXQ8Tz4pLnNpemUgOlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIE1hdGgubWluKHNpemUsIChkYXRhc2V0c1tkc10gYXMgRGF0YXNldDxPPikuc2l6ZSk7XG4gICAgfVxuICB9XG4gIHJldHVybiBkYXRhc2V0RnJvbUl0ZXJhdG9yRm48Tz4oYXN5bmMgKCkgPT4ge1xuICAgIGNvbnN0IHN0cmVhbXMgPSBhd2FpdCBkZWVwTWFwQW5kQXdhaXRBbGwoZGF0YXNldHMsIGQgPT4ge1xuICAgICAgaWYgKGQgaW5zdGFuY2VvZiBEYXRhc2V0KSB7XG4gICAgICAgIHJldHVybiB7dmFsdWU6IGQuaXRlcmF0b3IoKSwgcmVjdXJzZTogZmFsc2V9O1xuICAgICAgfSBlbHNlIGlmIChpc0l0ZXJhYmxlKGQpKSB7XG4gICAgICAgIHJldHVybiB7dmFsdWU6IG51bGwsIHJlY3Vyc2U6IHRydWV9O1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgJ0xlYXZlcyBvZiB0aGUgc3RydWN0dXJlIHBhc3NlZCB0byB6aXAoKSBtdXN0IGJlIERhdGFzZXRzLCAnICtcbiAgICAgICAgICAgICdub3QgcHJpbWl0aXZlcy4nKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgICByZXR1cm4gaXRlcmF0b3JGcm9tWmlwcGVkPE8+KHN0cmVhbXMsIFppcE1pc21hdGNoTW9kZS5TSE9SVEVTVCk7XG4gIH0sIHNpemUpO1xufVxuXG4vKipcbiAqIEEgemlwIGZ1bmN0aW9uIGZvciB1c2Ugd2l0aCBkZWVwWmlwLCBwYXNzZWQgdmlhIHRoZSBjb2x1bW5NYWpvckJhdGNoIGNhbGwuXG4gKlxuICogQWNjZXB0cyBhbiBhcnJheSBvZiBpZGVudGljYWxseS1zdHJ1Y3R1cmVkIG5lc3RlZCBlbGVtZW50cyBhbmQgZWl0aGVyIGJhdGNoZXNcbiAqIHRoZW0gKGlmIHRoZXkgYXJlIHByaW1pdGl2ZXMsIG51bWVyaWMgYXJyYXlzLCBvciBUZW5zb3JzKSBvciByZXF1ZXN0c1xuICogcmVjdXJzaW9uIChpZiBub3QpLlxuICovXG4vLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bm8tYW55XG5mdW5jdGlvbiBkZWVwQmF0Y2hDb25jYXQocm93czogYW55W10pOiBEZWVwTWFwUmVzdWx0IHtcbiAgaWYgKHJvd3MgPT09IG51bGwpIHtcbiAgICByZXR1cm4gbnVsbDtcbiAgfVxuXG4gIC8vIHVzZSB0aGUgZmlyc3QgaXRlbSB0byBkZWNpZGUgd2hldGhlciB0byByZWN1cnNlIG9yIGJhdGNoIGhlcmUuXG4gIGNvbnN0IGV4YW1wbGVSb3cgPSByb3dzWzBdO1xuXG4gIGlmIChjYW5UZW5zb3JpZnkoZXhhbXBsZVJvdykpIHtcbiAgICAvLyByb3dzIGlzIGFuIGFycmF5IG9mIHByaW1pdGl2ZXMsIFRlbnNvcnMsIG9yIGFycmF5cy4gIEJhdGNoIHRoZW0uXG4gICAgY29uc3QgdmFsdWUgPSBiYXRjaENvbmNhdChyb3dzKTtcbiAgICByZXR1cm4ge3ZhbHVlLCByZWN1cnNlOiBmYWxzZX07XG4gIH1cblxuICAvLyB0aGUgZXhhbXBsZSByb3cgaXMgYW4gb2JqZWN0LCBzbyByZWN1cnNlIGludG8gaXQuXG4gIHJldHVybiB7dmFsdWU6IG51bGwsIHJlY3Vyc2U6IHRydWV9O1xufVxuXG4vKipcbiAqIEFzc2VtYmxlcyBhIGxpc3Qgb2Ygc2FtZS1zaGFwZWQgbnVtYmVycywgbnVtYmVyIGFycmF5cywgb3IgVGVuc29yc1xuICogaW50byBhIHNpbmdsZSBuZXcgVGVuc29yIHdoZXJlIGF4aXMgMCBpcyB0aGUgYmF0Y2ggZGltZW5zaW9uLlxuICovXG5mdW5jdGlvbiBiYXRjaENvbmNhdDxUIGV4dGVuZHMoVGVuc29yTGlrZSB8IHRmLlRlbnNvcik+KGFycmF5czogVFtdKTpcbiAgICB0Zi5UZW5zb3Ige1xuICBpZiAoYXJyYXlzLmxlbmd0aCA9PT0gMCkge1xuICAgIC8vIFdlIGNhbid0IHJldHVybiBhbiBlbXB0eSBUZW5zb3IgYmVjYXVzZSB3ZSBkb24ndCBrbm93IHRoZSBlbGVtZW50IHNoYXBlLlxuICAgIHRocm93IG5ldyBFcnJvcignQ2FuXFwndCBtYWtlIGEgYmF0Y2ggb2YgemVybyBlbGVtZW50cy4nKTtcbiAgfVxuXG4gIGlmIChhcnJheXNbMF0gaW5zdGFuY2VvZiB0Zi5UZW5zb3IpIHtcbiAgICAvLyBJbnB1dCBpcyBhbiBhcnJheSBvZiBUZW5zb3JzXG4gICAgcmV0dXJuIHRmLnN0YWNrKGFycmF5cyBhcyB0Zi5UZW5zb3JbXSk7XG4gIH0gZWxzZSB7XG4gICAgLy8gSW5wdXQgaXMgYSBwb3NzaWJseS1uZXN0ZWQgYXJyYXkgb2YgbnVtYmVycy5cbiAgICByZXR1cm4gdGYudGVuc29yKGFycmF5cyBhcyBUZW5zb3JMaWtlKTtcbiAgfVxufVxuIl19