/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/* Original source: keras/contraints.py */
import * as tfc from '@tensorflow/tfjs-core';
import { serialization, tidy } from '@tensorflow/tfjs-core';
import { epsilon } from './backend/common';
import { deserializeKerasObject, serializeKerasObject } from './utils/generic_utils';
/**
 * Helper function used by many of the Constraints to find the L2Norms.
 */
function calcL2Norms(w, axis) {
    return tidy(() => tfc.sqrt(tfc.sum(tfc.mul(w, w), axis, true)));
}
/**
 * Base class for functions that impose constraints on weight values
 *
 * @doc {
 *   heading: 'Constraints',
 *   subheading: 'Classes',
 *   namespace: 'constraints'
 * }
 */
export class Constraint extends serialization.Serializable {
    getConfig() {
        return {};
    }
}
export class MaxNorm extends Constraint {
    constructor(args) {
        super();
        this.defaultMaxValue = 2;
        this.defaultAxis = 0;
        this.maxValue =
            args.maxValue != null ? args.maxValue : this.defaultMaxValue;
        this.axis = args.axis != null ? args.axis : this.defaultAxis;
    }
    apply(w) {
        return tidy(() => {
            const norms = calcL2Norms(w, this.axis);
            const desired = tfc.clipByValue(norms, 0, this.maxValue);
            return tfc.mul(w, tfc.div(desired, tfc.add(epsilon(), norms)));
        });
    }
    getConfig() {
        return { maxValue: this.maxValue, axis: this.axis };
    }
}
/** @nocollapse */
MaxNorm.className = 'MaxNorm';
serialization.registerClass(MaxNorm);
export class UnitNorm extends Constraint {
    constructor(args) {
        super();
        this.defaultAxis = 0;
        this.axis = args.axis != null ? args.axis : this.defaultAxis;
    }
    apply(w) {
        return tidy(() => tfc.div(w, tfc.add(epsilon(), calcL2Norms(w, this.axis))));
    }
    getConfig() {
        return { axis: this.axis };
    }
}
/** @nocollapse */
UnitNorm.className = 'UnitNorm';
serialization.registerClass(UnitNorm);
export class NonNeg extends Constraint {
    apply(w) {
        return tfc.relu(w);
    }
}
/** @nocollapse */
NonNeg.className = 'NonNeg';
serialization.registerClass(NonNeg);
export class MinMaxNorm extends Constraint {
    constructor(args) {
        super();
        this.defaultMinValue = 0.0;
        this.defaultMaxValue = 1.0;
        this.defaultRate = 1.0;
        this.defaultAxis = 0;
        this.minValue =
            args.minValue != null ? args.minValue : this.defaultMinValue;
        this.maxValue =
            args.maxValue != null ? args.maxValue : this.defaultMaxValue;
        this.rate = args.rate != null ? args.rate : this.defaultRate;
        this.axis = args.axis != null ? args.axis : this.defaultAxis;
    }
    apply(w) {
        return tidy(() => {
            const norms = calcL2Norms(w, this.axis);
            const desired = tfc.add(tfc.mul(this.rate, tfc.clipByValue(norms, this.minValue, this.maxValue)), tfc.mul(1.0 - this.rate, norms));
            return tfc.mul(w, tfc.div(desired, tfc.add(epsilon(), norms)));
        });
    }
    getConfig() {
        return {
            minValue: this.minValue,
            maxValue: this.maxValue,
            rate: this.rate,
            axis: this.axis
        };
    }
}
/** @nocollapse */
MinMaxNorm.className = 'MinMaxNorm';
serialization.registerClass(MinMaxNorm);
// Maps the JavaScript-like identifier keys to the corresponding registry
// symbols.
export const CONSTRAINT_IDENTIFIER_REGISTRY_SYMBOL_MAP = {
    'maxNorm': 'MaxNorm',
    'minMaxNorm': 'MinMaxNorm',
    'nonNeg': 'NonNeg',
    'unitNorm': 'UnitNorm'
};
export function serializeConstraint(constraint) {
    return serializeKerasObject(constraint);
}
export function deserializeConstraint(config, customObjects = {}) {
    return deserializeKerasObject(config, serialization.SerializationMap.getMap().classNameMap, customObjects, 'constraint');
}
export function getConstraint(identifier) {
    if (identifier == null) {
        return null;
    }
    if (typeof identifier === 'string') {
        const className = identifier in CONSTRAINT_IDENTIFIER_REGISTRY_SYMBOL_MAP ?
            CONSTRAINT_IDENTIFIER_REGISTRY_SYMBOL_MAP[identifier] :
            identifier;
        const config = { className, config: {} };
        return deserializeConstraint(config);
    }
    else if (identifier instanceof Constraint) {
        return identifier;
    }
    else {
        return deserializeConstraint(identifier);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uc3RyYWludHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi90ZmpzLWxheWVycy9zcmMvY29uc3RyYWludHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7Ozs7O0dBUUc7QUFFSCwwQ0FBMEM7QUFFMUMsT0FBTyxLQUFLLEdBQUcsTUFBTSx1QkFBdUIsQ0FBQztBQUM3QyxPQUFPLEVBQUMsYUFBYSxFQUFVLElBQUksRUFBQyxNQUFNLHVCQUF1QixDQUFDO0FBQ2xFLE9BQU8sRUFBQyxPQUFPLEVBQUMsTUFBTSxrQkFBa0IsQ0FBQztBQUN6QyxPQUFPLEVBQUMsc0JBQXNCLEVBQUUsb0JBQW9CLEVBQUMsTUFBTSx1QkFBdUIsQ0FBQztBQUVuRjs7R0FFRztBQUNILFNBQVMsV0FBVyxDQUFDLENBQVMsRUFBRSxJQUFZO0lBQzFDLE9BQU8sSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ2xFLENBQUM7QUFFRDs7Ozs7Ozs7R0FRRztBQUNILE1BQU0sT0FBZ0IsVUFBVyxTQUFRLGFBQWEsQ0FBQyxZQUFZO0lBR2pFLFNBQVM7UUFDUCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7Q0FDRjtBQXdCRCxNQUFNLE9BQU8sT0FBUSxTQUFRLFVBQVU7SUFRckMsWUFBWSxJQUFpQjtRQUMzQixLQUFLLEVBQUUsQ0FBQztRQUpPLG9CQUFlLEdBQUcsQ0FBQyxDQUFDO1FBQ3BCLGdCQUFXLEdBQUcsQ0FBQyxDQUFDO1FBSS9CLElBQUksQ0FBQyxRQUFRO1lBQ1QsSUFBSSxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUM7UUFDakUsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQztJQUMvRCxDQUFDO0lBRUQsS0FBSyxDQUFDLENBQVM7UUFDYixPQUFPLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDZixNQUFNLEtBQUssR0FBRyxXQUFXLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN4QyxNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxDQUFDLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3pELE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakUsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsU0FBUztRQUNQLE9BQU8sRUFBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBQyxDQUFDO0lBQ3BELENBQUM7O0FBeEJELGtCQUFrQjtBQUNGLGlCQUFTLEdBQUcsU0FBUyxDQUFDO0FBeUJ4QyxhQUFhLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0FBb0JyQyxNQUFNLE9BQU8sUUFBUyxTQUFRLFVBQVU7SUFLdEMsWUFBWSxJQUFrQjtRQUM1QixLQUFLLEVBQUUsQ0FBQztRQUZPLGdCQUFXLEdBQUcsQ0FBQyxDQUFDO1FBRy9CLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7SUFDL0QsQ0FBQztJQUVELEtBQUssQ0FBQyxDQUFTO1FBQ2IsT0FBTyxJQUFJLENBQ1AsR0FBRyxFQUFFLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsRUFBRSxXQUFXLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBRUQsU0FBUztRQUNQLE9BQU8sRUFBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBQyxDQUFDO0lBQzNCLENBQUM7O0FBaEJELGtCQUFrQjtBQUNGLGtCQUFTLEdBQUcsVUFBVSxDQUFDO0FBaUJ6QyxhQUFhLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBRXRDLE1BQU0sT0FBTyxNQUFPLFNBQVEsVUFBVTtJQUlwQyxLQUFLLENBQUMsQ0FBUztRQUNiLE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNyQixDQUFDOztBQUxELGtCQUFrQjtBQUNGLGdCQUFTLEdBQUcsUUFBUSxDQUFDO0FBTXZDLGFBQWEsQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7QUFvQ3BDLE1BQU0sT0FBTyxVQUFXLFNBQVEsVUFBVTtJQVl4QyxZQUFZLElBQW9CO1FBQzlCLEtBQUssRUFBRSxDQUFDO1FBTk8sb0JBQWUsR0FBRyxHQUFHLENBQUM7UUFDdEIsb0JBQWUsR0FBRyxHQUFHLENBQUM7UUFDdEIsZ0JBQVcsR0FBRyxHQUFHLENBQUM7UUFDbEIsZ0JBQVcsR0FBRyxDQUFDLENBQUM7UUFJL0IsSUFBSSxDQUFDLFFBQVE7WUFDVCxJQUFJLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQztRQUNqRSxJQUFJLENBQUMsUUFBUTtZQUNULElBQUksQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDO1FBQ2pFLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7UUFDN0QsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQztJQUMvRCxDQUFDO0lBRUQsS0FBSyxDQUFDLENBQVM7UUFDYixPQUFPLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDZixNQUFNLEtBQUssR0FBRyxXQUFXLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN4QyxNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsR0FBRyxDQUNuQixHQUFHLENBQUMsR0FBRyxDQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsRUFDcEUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ3JDLE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakUsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsU0FBUztRQUNQLE9BQU87WUFDTCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNmLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtTQUNoQixDQUFDO0lBQ0osQ0FBQzs7QUF2Q0Qsa0JBQWtCO0FBQ0Ysb0JBQVMsR0FBRyxZQUFZLENBQUM7QUF3QzNDLGFBQWEsQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLENBQUM7QUFNeEMseUVBQXlFO0FBQ3pFLFdBQVc7QUFDWCxNQUFNLENBQUMsTUFBTSx5Q0FBeUMsR0FDRDtJQUMvQyxTQUFTLEVBQUUsU0FBUztJQUNwQixZQUFZLEVBQUUsWUFBWTtJQUMxQixRQUFRLEVBQUUsUUFBUTtJQUNsQixVQUFVLEVBQUUsVUFBVTtDQUN2QixDQUFDO0FBRU4sTUFBTSxVQUFVLG1CQUFtQixDQUFDLFVBQXNCO0lBRXhELE9BQU8sb0JBQW9CLENBQUMsVUFBVSxDQUFDLENBQUM7QUFDMUMsQ0FBQztBQUVELE1BQU0sVUFBVSxxQkFBcUIsQ0FDakMsTUFBZ0MsRUFDaEMsZ0JBQTBDLEVBQUU7SUFDOUMsT0FBTyxzQkFBc0IsQ0FDekIsTUFBTSxFQUFFLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxZQUFZLEVBQzVELGFBQWEsRUFBRSxZQUFZLENBQUMsQ0FBQztBQUNuQyxDQUFDO0FBRUQsTUFBTSxVQUFVLGFBQWEsQ0FBQyxVQUNtQztJQUMvRCxJQUFJLFVBQVUsSUFBSSxJQUFJLEVBQUU7UUFDdEIsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUNELElBQUksT0FBTyxVQUFVLEtBQUssUUFBUSxFQUFFO1FBQ2xDLE1BQU0sU0FBUyxHQUFHLFVBQVUsSUFBSSx5Q0FBeUMsQ0FBQyxDQUFDO1lBQ3ZFLHlDQUF5QyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDdkQsVUFBVSxDQUFDO1FBQ2YsTUFBTSxNQUFNLEdBQUcsRUFBQyxTQUFTLEVBQUUsTUFBTSxFQUFFLEVBQUUsRUFBQyxDQUFDO1FBQ3ZDLE9BQU8scUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUM7S0FDdEM7U0FBTSxJQUFJLFVBQVUsWUFBWSxVQUFVLEVBQUU7UUFDM0MsT0FBTyxVQUFVLENBQUM7S0FDbkI7U0FBTTtRQUNMLE9BQU8scUJBQXFCLENBQUMsVUFBVSxDQUFDLENBQUM7S0FDMUM7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IDIwMTggR29vZ2xlIExMQ1xuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZVxuICogbGljZW5zZSB0aGF0IGNhbiBiZSBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIG9yIGF0XG4gKiBodHRwczovL29wZW5zb3VyY2Uub3JnL2xpY2Vuc2VzL01JVC5cbiAqID09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09XG4gKi9cblxuLyogT3JpZ2luYWwgc291cmNlOiBrZXJhcy9jb250cmFpbnRzLnB5ICovXG5cbmltcG9ydCAqIGFzIHRmYyBmcm9tICdAdGVuc29yZmxvdy90ZmpzLWNvcmUnO1xuaW1wb3J0IHtzZXJpYWxpemF0aW9uLCBUZW5zb3IsIHRpZHl9IGZyb20gJ0B0ZW5zb3JmbG93L3RmanMtY29yZSc7XG5pbXBvcnQge2Vwc2lsb259IGZyb20gJy4vYmFja2VuZC9jb21tb24nO1xuaW1wb3J0IHtkZXNlcmlhbGl6ZUtlcmFzT2JqZWN0LCBzZXJpYWxpemVLZXJhc09iamVjdH0gZnJvbSAnLi91dGlscy9nZW5lcmljX3V0aWxzJztcblxuLyoqXG4gKiBIZWxwZXIgZnVuY3Rpb24gdXNlZCBieSBtYW55IG9mIHRoZSBDb25zdHJhaW50cyB0byBmaW5kIHRoZSBMMk5vcm1zLlxuICovXG5mdW5jdGlvbiBjYWxjTDJOb3Jtcyh3OiBUZW5zb3IsIGF4aXM6IG51bWJlcik6IFRlbnNvciB7XG4gIHJldHVybiB0aWR5KCgpID0+IHRmYy5zcXJ0KHRmYy5zdW0odGZjLm11bCh3LCB3KSwgYXhpcywgdHJ1ZSkpKTtcbn1cblxuLyoqXG4gKiBCYXNlIGNsYXNzIGZvciBmdW5jdGlvbnMgdGhhdCBpbXBvc2UgY29uc3RyYWludHMgb24gd2VpZ2h0IHZhbHVlc1xuICpcbiAqIEBkb2Mge1xuICogICBoZWFkaW5nOiAnQ29uc3RyYWludHMnLFxuICogICBzdWJoZWFkaW5nOiAnQ2xhc3NlcycsXG4gKiAgIG5hbWVzcGFjZTogJ2NvbnN0cmFpbnRzJ1xuICogfVxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQ29uc3RyYWludCBleHRlbmRzIHNlcmlhbGl6YXRpb24uU2VyaWFsaXphYmxlIHtcbiAgLyogUG9ydGluZyBub3RlOiB3YXMgX19jYWxsX18sIGFwcGx5IGNob3NlbiB0byBtYXRjaCBvdGhlciBzaW1pbGFyIGNob2ljZXMgKi9cbiAgYWJzdHJhY3QgYXBwbHkodzogVGVuc29yKTogVGVuc29yO1xuICBnZXRDb25maWcoKTogc2VyaWFsaXphdGlvbi5Db25maWdEaWN0IHtcbiAgICByZXR1cm4ge307XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBNYXhOb3JtQXJncyB7XG4gIC8qKlxuICAgKiBNYXhpbXVtIG5vcm0gZm9yIGluY29taW5nIHdlaWdodHNcbiAgICovXG4gIG1heFZhbHVlPzogbnVtYmVyO1xuICAvKipcbiAgICogQXhpcyBhbG9uZyB3aGljaCB0byBjYWxjdWxhdGUgbm9ybXMuXG4gICAqXG4gICAqICBGb3IgaW5zdGFuY2UsIGluIGEgYERlbnNlYCBsYXllciB0aGUgd2VpZ2h0IG1hdHJpeFxuICAgKiAgaGFzIHNoYXBlIGBbaW5wdXREaW0sIG91dHB1dERpbV1gLFxuICAgKiAgc2V0IGBheGlzYCB0byBgMGAgdG8gY29uc3RyYWluIGVhY2ggd2VpZ2h0IHZlY3RvclxuICAgKiAgb2YgbGVuZ3RoIGBbaW5wdXREaW0sXWAuXG4gICAqICBJbiBhIGBDb252MkRgIGxheWVyIHdpdGggYGRhdGFGb3JtYXQ9XCJjaGFubmVsc19sYXN0XCJgLFxuICAgKiAgdGhlIHdlaWdodCB0ZW5zb3IgaGFzIHNoYXBlXG4gICAqICBgW3Jvd3MsIGNvbHMsIGlucHV0RGVwdGgsIG91dHB1dERlcHRoXWAsXG4gICAqICBzZXQgYGF4aXNgIHRvIGBbMCwgMSwgMl1gXG4gICAqICB0byBjb25zdHJhaW4gdGhlIHdlaWdodHMgb2YgZWFjaCBmaWx0ZXIgdGVuc29yIG9mIHNpemVcbiAgICogIGBbcm93cywgY29scywgaW5wdXREZXB0aF1gLlxuICAgKi9cbiAgYXhpcz86IG51bWJlcjtcbn1cblxuZXhwb3J0IGNsYXNzIE1heE5vcm0gZXh0ZW5kcyBDb25zdHJhaW50IHtcbiAgLyoqIEBub2NvbGxhcHNlICovXG4gIHN0YXRpYyByZWFkb25seSBjbGFzc05hbWUgPSAnTWF4Tm9ybSc7XG4gIHByaXZhdGUgbWF4VmFsdWU6IG51bWJlcjtcbiAgcHJpdmF0ZSBheGlzOiBudW1iZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgZGVmYXVsdE1heFZhbHVlID0gMjtcbiAgcHJpdmF0ZSByZWFkb25seSBkZWZhdWx0QXhpcyA9IDA7XG5cbiAgY29uc3RydWN0b3IoYXJnczogTWF4Tm9ybUFyZ3MpIHtcbiAgICBzdXBlcigpO1xuICAgIHRoaXMubWF4VmFsdWUgPVxuICAgICAgICBhcmdzLm1heFZhbHVlICE9IG51bGwgPyBhcmdzLm1heFZhbHVlIDogdGhpcy5kZWZhdWx0TWF4VmFsdWU7XG4gICAgdGhpcy5heGlzID0gYXJncy5heGlzICE9IG51bGwgPyBhcmdzLmF4aXMgOiB0aGlzLmRlZmF1bHRBeGlzO1xuICB9XG5cbiAgYXBwbHkodzogVGVuc29yKTogVGVuc29yIHtcbiAgICByZXR1cm4gdGlkeSgoKSA9PiB7XG4gICAgICBjb25zdCBub3JtcyA9IGNhbGNMMk5vcm1zKHcsIHRoaXMuYXhpcyk7XG4gICAgICBjb25zdCBkZXNpcmVkID0gdGZjLmNsaXBCeVZhbHVlKG5vcm1zLCAwLCB0aGlzLm1heFZhbHVlKTtcbiAgICAgIHJldHVybiB0ZmMubXVsKHcsIHRmYy5kaXYoZGVzaXJlZCwgdGZjLmFkZChlcHNpbG9uKCksIG5vcm1zKSkpO1xuICAgIH0pO1xuICB9XG5cbiAgZ2V0Q29uZmlnKCk6IHNlcmlhbGl6YXRpb24uQ29uZmlnRGljdCB7XG4gICAgcmV0dXJuIHttYXhWYWx1ZTogdGhpcy5tYXhWYWx1ZSwgYXhpczogdGhpcy5heGlzfTtcbiAgfVxufVxuc2VyaWFsaXphdGlvbi5yZWdpc3RlckNsYXNzKE1heE5vcm0pO1xuXG5leHBvcnQgaW50ZXJmYWNlIFVuaXROb3JtQXJncyB7XG4gIC8qKlxuICAgKiBBeGlzIGFsb25nIHdoaWNoIHRvIGNhbGN1bGF0ZSBub3Jtcy5cbiAgICpcbiAgICogRm9yIGluc3RhbmNlLCBpbiBhIGBEZW5zZWAgbGF5ZXIgdGhlIHdlaWdodCBtYXRyaXhcbiAgICogaGFzIHNoYXBlIGBbaW5wdXREaW0sIG91dHB1dERpbV1gLFxuICAgKiBzZXQgYGF4aXNgIHRvIGAwYCB0byBjb25zdHJhaW4gZWFjaCB3ZWlnaHQgdmVjdG9yXG4gICAqIG9mIGxlbmd0aCBgW2lucHV0RGltLF1gLlxuICAgKiBJbiBhIGBDb252MkRgIGxheWVyIHdpdGggYGRhdGFGb3JtYXQ9XCJjaGFubmVsc19sYXN0XCJgLFxuICAgKiB0aGUgd2VpZ2h0IHRlbnNvciBoYXMgc2hhcGVcbiAgICogYFtyb3dzLCBjb2xzLCBpbnB1dERlcHRoLCBvdXRwdXREZXB0aF1gLFxuICAgKiBzZXQgYGF4aXNgIHRvIGBbMCwgMSwgMl1gXG4gICAqIHRvIGNvbnN0cmFpbiB0aGUgd2VpZ2h0cyBvZiBlYWNoIGZpbHRlciB0ZW5zb3Igb2Ygc2l6ZVxuICAgKiBgW3Jvd3MsIGNvbHMsIGlucHV0RGVwdGhdYC5cbiAgICovXG4gIGF4aXM/OiBudW1iZXI7XG59XG5cbmV4cG9ydCBjbGFzcyBVbml0Tm9ybSBleHRlbmRzIENvbnN0cmFpbnQge1xuICAvKiogQG5vY29sbGFwc2UgKi9cbiAgc3RhdGljIHJlYWRvbmx5IGNsYXNzTmFtZSA9ICdVbml0Tm9ybSc7XG4gIHByaXZhdGUgYXhpczogbnVtYmVyO1xuICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRBeGlzID0gMDtcbiAgY29uc3RydWN0b3IoYXJnczogVW5pdE5vcm1BcmdzKSB7XG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLmF4aXMgPSBhcmdzLmF4aXMgIT0gbnVsbCA/IGFyZ3MuYXhpcyA6IHRoaXMuZGVmYXVsdEF4aXM7XG4gIH1cblxuICBhcHBseSh3OiBUZW5zb3IpOiBUZW5zb3Ige1xuICAgIHJldHVybiB0aWR5KFxuICAgICAgICAoKSA9PiB0ZmMuZGl2KHcsIHRmYy5hZGQoZXBzaWxvbigpLCBjYWxjTDJOb3Jtcyh3LCB0aGlzLmF4aXMpKSkpO1xuICB9XG5cbiAgZ2V0Q29uZmlnKCk6IHNlcmlhbGl6YXRpb24uQ29uZmlnRGljdCB7XG4gICAgcmV0dXJuIHtheGlzOiB0aGlzLmF4aXN9O1xuICB9XG59XG5zZXJpYWxpemF0aW9uLnJlZ2lzdGVyQ2xhc3MoVW5pdE5vcm0pO1xuXG5leHBvcnQgY2xhc3MgTm9uTmVnIGV4dGVuZHMgQ29uc3RyYWludCB7XG4gIC8qKiBAbm9jb2xsYXBzZSAqL1xuICBzdGF0aWMgcmVhZG9ubHkgY2xhc3NOYW1lID0gJ05vbk5lZyc7XG5cbiAgYXBwbHkodzogVGVuc29yKTogVGVuc29yIHtcbiAgICByZXR1cm4gdGZjLnJlbHUodyk7XG4gIH1cbn1cbnNlcmlhbGl6YXRpb24ucmVnaXN0ZXJDbGFzcyhOb25OZWcpO1xuXG5leHBvcnQgaW50ZXJmYWNlIE1pbk1heE5vcm1BcmdzIHtcbiAgLyoqXG4gICAqIE1pbmltdW0gbm9ybSBmb3IgaW5jb21pbmcgd2VpZ2h0c1xuICAgKi9cbiAgbWluVmFsdWU/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBNYXhpbXVtIG5vcm0gZm9yIGluY29taW5nIHdlaWdodHNcbiAgICovXG4gIG1heFZhbHVlPzogbnVtYmVyO1xuICAvKipcbiAgICogQXhpcyBhbG9uZyB3aGljaCB0byBjYWxjdWxhdGUgbm9ybXMuXG4gICAqIEZvciBpbnN0YW5jZSwgaW4gYSBgRGVuc2VgIGxheWVyIHRoZSB3ZWlnaHQgbWF0cml4XG4gICAqIGhhcyBzaGFwZSBgW2lucHV0RGltLCBvdXRwdXREaW1dYCxcbiAgICogc2V0IGBheGlzYCB0byBgMGAgdG8gY29uc3RyYWluIGVhY2ggd2VpZ2h0IHZlY3RvclxuICAgKiBvZiBsZW5ndGggYFtpbnB1dERpbSxdYC5cbiAgICogSW4gYSBgQ29udjJEYCBsYXllciB3aXRoIGBkYXRhRm9ybWF0PVwiY2hhbm5lbHNfbGFzdFwiYCxcbiAgICogdGhlIHdlaWdodCB0ZW5zb3IgaGFzIHNoYXBlXG4gICAqIGBbcm93cywgY29scywgaW5wdXREZXB0aCwgb3V0cHV0RGVwdGhdYCxcbiAgICogc2V0IGBheGlzYCB0byBgWzAsIDEsIDJdYFxuICAgKiB0byBjb25zdHJhaW4gdGhlIHdlaWdodHMgb2YgZWFjaCBmaWx0ZXIgdGVuc29yIG9mIHNpemVcbiAgICogYFtyb3dzLCBjb2xzLCBpbnB1dERlcHRoXWAuXG4gICAqL1xuICBheGlzPzogbnVtYmVyO1xuICAvKipcbiAgICogUmF0ZSBmb3IgZW5mb3JjaW5nIHRoZSBjb25zdHJhaW50OiB3ZWlnaHRzIHdpbGwgYmUgcmVzY2FsZWQgdG8geWllbGQ6XG4gICAqIGAoMSAtIHJhdGUpICogbm9ybSArIHJhdGUgKiBub3JtLmNsaXAobWluVmFsdWUsIG1heFZhbHVlKWAuXG4gICAqIEVmZmVjdGl2ZWx5LCB0aGlzIG1lYW5zIHRoYXQgcmF0ZT0xLjAgc3RhbmRzIGZvciBzdHJpY3RcbiAgICogZW5mb3JjZW1lbnQgb2YgdGhlIGNvbnN0cmFpbnQsIHdoaWxlIHJhdGU8MS4wIG1lYW5zIHRoYXRcbiAgICogd2VpZ2h0cyB3aWxsIGJlIHJlc2NhbGVkIGF0IGVhY2ggc3RlcCB0byBzbG93bHkgbW92ZVxuICAgKiB0b3dhcmRzIGEgdmFsdWUgaW5zaWRlIHRoZSBkZXNpcmVkIGludGVydmFsLlxuICAgKi9cbiAgcmF0ZT86IG51bWJlcjtcbn1cblxuZXhwb3J0IGNsYXNzIE1pbk1heE5vcm0gZXh0ZW5kcyBDb25zdHJhaW50IHtcbiAgLyoqIEBub2NvbGxhcHNlICovXG4gIHN0YXRpYyByZWFkb25seSBjbGFzc05hbWUgPSAnTWluTWF4Tm9ybSc7XG4gIHByaXZhdGUgbWluVmFsdWU6IG51bWJlcjtcbiAgcHJpdmF0ZSBtYXhWYWx1ZTogbnVtYmVyO1xuICBwcml2YXRlIHJhdGU6IG51bWJlcjtcbiAgcHJpdmF0ZSBheGlzOiBudW1iZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgZGVmYXVsdE1pblZhbHVlID0gMC4wO1xuICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRNYXhWYWx1ZSA9IDEuMDtcbiAgcHJpdmF0ZSByZWFkb25seSBkZWZhdWx0UmF0ZSA9IDEuMDtcbiAgcHJpdmF0ZSByZWFkb25seSBkZWZhdWx0QXhpcyA9IDA7XG5cbiAgY29uc3RydWN0b3IoYXJnczogTWluTWF4Tm9ybUFyZ3MpIHtcbiAgICBzdXBlcigpO1xuICAgIHRoaXMubWluVmFsdWUgPVxuICAgICAgICBhcmdzLm1pblZhbHVlICE9IG51bGwgPyBhcmdzLm1pblZhbHVlIDogdGhpcy5kZWZhdWx0TWluVmFsdWU7XG4gICAgdGhpcy5tYXhWYWx1ZSA9XG4gICAgICAgIGFyZ3MubWF4VmFsdWUgIT0gbnVsbCA/IGFyZ3MubWF4VmFsdWUgOiB0aGlzLmRlZmF1bHRNYXhWYWx1ZTtcbiAgICB0aGlzLnJhdGUgPSBhcmdzLnJhdGUgIT0gbnVsbCA/IGFyZ3MucmF0ZSA6IHRoaXMuZGVmYXVsdFJhdGU7XG4gICAgdGhpcy5heGlzID0gYXJncy5heGlzICE9IG51bGwgPyBhcmdzLmF4aXMgOiB0aGlzLmRlZmF1bHRBeGlzO1xuICB9XG5cbiAgYXBwbHkodzogVGVuc29yKTogVGVuc29yIHtcbiAgICByZXR1cm4gdGlkeSgoKSA9PiB7XG4gICAgICBjb25zdCBub3JtcyA9IGNhbGNMMk5vcm1zKHcsIHRoaXMuYXhpcyk7XG4gICAgICBjb25zdCBkZXNpcmVkID0gdGZjLmFkZChcbiAgICAgICAgICB0ZmMubXVsKFxuICAgICAgICAgICAgICB0aGlzLnJhdGUsIHRmYy5jbGlwQnlWYWx1ZShub3JtcywgdGhpcy5taW5WYWx1ZSwgdGhpcy5tYXhWYWx1ZSkpLFxuICAgICAgICAgIHRmYy5tdWwoMS4wIC0gdGhpcy5yYXRlLCBub3JtcykpO1xuICAgICAgcmV0dXJuIHRmYy5tdWwodywgdGZjLmRpdihkZXNpcmVkLCB0ZmMuYWRkKGVwc2lsb24oKSwgbm9ybXMpKSk7XG4gICAgfSk7XG4gIH1cblxuICBnZXRDb25maWcoKTogc2VyaWFsaXphdGlvbi5Db25maWdEaWN0IHtcbiAgICByZXR1cm4ge1xuICAgICAgbWluVmFsdWU6IHRoaXMubWluVmFsdWUsXG4gICAgICBtYXhWYWx1ZTogdGhpcy5tYXhWYWx1ZSxcbiAgICAgIHJhdGU6IHRoaXMucmF0ZSxcbiAgICAgIGF4aXM6IHRoaXMuYXhpc1xuICAgIH07XG4gIH1cbn1cbnNlcmlhbGl6YXRpb24ucmVnaXN0ZXJDbGFzcyhNaW5NYXhOb3JtKTtcblxuLyoqIEBkb2NpbmxpbmUgKi9cbmV4cG9ydCB0eXBlIENvbnN0cmFpbnRJZGVudGlmaWVyID1cbiAgICAnbWF4Tm9ybSd8J21pbk1heE5vcm0nfCdub25OZWcnfCd1bml0Tm9ybSd8c3RyaW5nO1xuXG4vLyBNYXBzIHRoZSBKYXZhU2NyaXB0LWxpa2UgaWRlbnRpZmllciBrZXlzIHRvIHRoZSBjb3JyZXNwb25kaW5nIHJlZ2lzdHJ5XG4vLyBzeW1ib2xzLlxuZXhwb3J0IGNvbnN0IENPTlNUUkFJTlRfSURFTlRJRklFUl9SRUdJU1RSWV9TWU1CT0xfTUFQOlxuICAgIHtbaWRlbnRpZmllciBpbiBDb25zdHJhaW50SWRlbnRpZmllcl06IHN0cmluZ30gPSB7XG4gICAgICAnbWF4Tm9ybSc6ICdNYXhOb3JtJyxcbiAgICAgICdtaW5NYXhOb3JtJzogJ01pbk1heE5vcm0nLFxuICAgICAgJ25vbk5lZyc6ICdOb25OZWcnLFxuICAgICAgJ3VuaXROb3JtJzogJ1VuaXROb3JtJ1xuICAgIH07XG5cbmV4cG9ydCBmdW5jdGlvbiBzZXJpYWxpemVDb25zdHJhaW50KGNvbnN0cmFpbnQ6IENvbnN0cmFpbnQpOlxuICAgIHNlcmlhbGl6YXRpb24uQ29uZmlnRGljdFZhbHVlIHtcbiAgcmV0dXJuIHNlcmlhbGl6ZUtlcmFzT2JqZWN0KGNvbnN0cmFpbnQpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZGVzZXJpYWxpemVDb25zdHJhaW50KFxuICAgIGNvbmZpZzogc2VyaWFsaXphdGlvbi5Db25maWdEaWN0LFxuICAgIGN1c3RvbU9iamVjdHM6IHNlcmlhbGl6YXRpb24uQ29uZmlnRGljdCA9IHt9KTogQ29uc3RyYWludCB7XG4gIHJldHVybiBkZXNlcmlhbGl6ZUtlcmFzT2JqZWN0KFxuICAgICAgY29uZmlnLCBzZXJpYWxpemF0aW9uLlNlcmlhbGl6YXRpb25NYXAuZ2V0TWFwKCkuY2xhc3NOYW1lTWFwLFxuICAgICAgY3VzdG9tT2JqZWN0cywgJ2NvbnN0cmFpbnQnKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldENvbnN0cmFpbnQoaWRlbnRpZmllcjogQ29uc3RyYWludElkZW50aWZpZXJ8XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICBzZXJpYWxpemF0aW9uLkNvbmZpZ0RpY3R8Q29uc3RyYWludCk6IENvbnN0cmFpbnQge1xuICBpZiAoaWRlbnRpZmllciA9PSBudWxsKSB7XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cbiAgaWYgKHR5cGVvZiBpZGVudGlmaWVyID09PSAnc3RyaW5nJykge1xuICAgIGNvbnN0IGNsYXNzTmFtZSA9IGlkZW50aWZpZXIgaW4gQ09OU1RSQUlOVF9JREVOVElGSUVSX1JFR0lTVFJZX1NZTUJPTF9NQVAgP1xuICAgICAgICBDT05TVFJBSU5UX0lERU5USUZJRVJfUkVHSVNUUllfU1lNQk9MX01BUFtpZGVudGlmaWVyXSA6XG4gICAgICAgIGlkZW50aWZpZXI7XG4gICAgY29uc3QgY29uZmlnID0ge2NsYXNzTmFtZSwgY29uZmlnOiB7fX07XG4gICAgcmV0dXJuIGRlc2VyaWFsaXplQ29uc3RyYWludChjb25maWcpO1xuICB9IGVsc2UgaWYgKGlkZW50aWZpZXIgaW5zdGFuY2VvZiBDb25zdHJhaW50KSB7XG4gICAgcmV0dXJuIGlkZW50aWZpZXI7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIGRlc2VyaWFsaXplQ29uc3RyYWludChpZGVudGlmaWVyKTtcbiAgfVxufVxuIl19