/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/**
 * Interfaces and methods for training models using tf.Tensor objects.
 */
import * as tfc from '@tensorflow/tfjs-core';
import { Tensor, tensor1d, util } from '@tensorflow/tfjs-core';
import { expandDims, gather, sliceAlongFirstAxis } from '../backend/tfjs_backend';
import { configureCallbacks, standardizeCallbacks } from '../base_callbacks';
import { NotImplementedError, ValueError } from '../errors';
import { disposeTensorsInLogs } from '../logs';
import { range } from '../utils/math_utils';
export function checkBatchSize(batchSize) {
    tfc.util.assert(batchSize > 0 && Number.isInteger(batchSize), () => `batchSize is required to be a positive integer, but got ${batchSize}`);
}
/**
 * Slice a Tensor or an Array of Tensors, by start and stop indices.
 *
 * Porting Note: The `_slice_arrays` function in PyKeras is covered by this
 *   function and `sliceArraysByIndices()` together.
 *
 * @param arrays: the input.
 * @param start: the starting index (inclusive).
 * @param stop: the stopping index (exclusive).
 * @returns The result of the slicing. If `arrays` is an `Array` of
 *   `tf.Tensor`s, the slicing will be applied to all elements of the `Array`
 *   in the same way.
 */
export function sliceArrays(arrays, start, stop) {
    if (arrays == null) {
        return [null];
    }
    else if (Array.isArray(arrays)) {
        return arrays.map(array => sliceAlongFirstAxis(array, start, stop - start));
    }
    else { // Tensor.
        return sliceAlongFirstAxis(arrays, start, stop - start);
    }
}
/**
 * Slice a Tensor or an Array of Tensors, by random-order indices.
 *
 * Porting Note: The `_slice_arrays` function in PyKeras is covered by this
 *   function and `sliceArrays()` together.
 *
 * @param arrays The input `tf.Tensor` or `Array` of `tf.Tensor`s to slice.
 *   If an `Array` of `tf.Tensor`s, all `tf.Tensor`s will be sliced in the
 *   same fashion.
 * @param indices The indices to use for slicing along the first (batch)
 *   dimension.
 * @returns Result(s) of the slicing.
 */
export function sliceArraysByIndices(arrays, indices) {
    return tfc.tidy(() => {
        if (arrays == null) {
            return null;
        }
        else if (Array.isArray(arrays)) {
            return arrays.map(array => sliceArraysByIndices(array, indices));
        }
        else {
            // TODO(cais): indices should be a pre-constructed Tensor1D to avoid
            //   tensor1d() calls.
            return gather(arrays, indices.dtype === 'int32' ? indices : tfc.cast(indices, 'int32'));
        }
    });
}
/**
 * Returns a list of batch indices (tuples of indices).
 * @param size: Integer, total size of the data to slice into batches.
 * @param batchSize: Integer, batch size.
 * @returns An Array of [batchStart, batchEnd] tuples. batchStart is
 *   inclusive; batchEnd is exclusive. I.e., each batch consists of indices x
 *   that satisfy batchStart <= x < batchEnd.
 */
export function makeBatches(size, batchSize) {
    const output = [];
    let batchStart = 0;
    let batchEnd = null;
    while (batchStart < size) {
        batchEnd = batchStart + batchSize;
        if (batchEnd >= size) {
            batchEnd = size;
        }
        output.push([batchStart, batchEnd]);
        batchStart = batchEnd;
    }
    return output;
}
/**
 * Abstract fit function for `f(ins)`.
 * @param f A Function returning a list of tensors. For training, this
 *   function is expected to perform the updates to the variables.
 * @param ins List of tensors to be fed to `f`.
 * @param outLabels List of strings, display names of the outputs of `f`.
 * @param batchSize Integer batch size or `== null` if unknown. Default : 32.
 * @param epochs Number of times to iterate over the data. Default : 1.
 * @param verbose Verbosity mode: 0, 1, or 2. Default: 1.
 * @param callbacks List of callbacks to be called during training.
 * @param valF Function to call for validation.
 * @param valIns List of tensors to be fed to `valF`.
 * @param shuffle Whether to shuffle the data at the beginning of every
 * epoch. Default : true.
 * @param callbackMetrics List of strings, the display names of the metrics
 *   passed to the callbacks. They should be the concatenation of the
 *   display names of the outputs of `f` and the list of display names
 *   of the outputs of `valF`.
 * @param initialEpoch Epoch at which to start training (useful for
 *   resuming a previous training run). Default : 0.
 * @param stepsPerEpoch Total number of steps (batches on samples) before
 *   declaring one epoch finished and starting the next epoch. Ignored with
 *   the default value of `undefined` or `null`.
 * @param validationSteps Number of steps to run validation for (only if
 *   doing validation from data tensors). Not applicable for tfjs-layers.
 * @returns A `History` object.
 */
async function fitLoop(
// Type `model` as `any` here to avoid circular dependency w/ training.ts.
// tslint:disable-next-line:no-any
model, f, ins, outLabels, batchSize, epochs, verbose, callbacks, valF, valIns, shuffle, callbackMetrics, initialEpoch, stepsPerEpoch, validationSteps) {
    if (batchSize == null) {
        batchSize = 32;
    }
    if (epochs == null) {
        epochs = 1;
    }
    if (shuffle == null) {
        shuffle = true;
    }
    if (initialEpoch == null) {
        initialEpoch = 0;
    }
    // TODO(cais): Change const to let below when implementing validation.
    let doValidation = false;
    if (valF != null && valIns != null) {
        doValidation = true;
        // TODO(cais): verbose message.
    }
    if (validationSteps != null) {
        doValidation = true;
        if (stepsPerEpoch == null) {
            throw new ValueError('Can only use `validationSteps` when doing step-wise training, ' +
                'i.e., `stepsPerEpoch` must be set.');
        }
    }
    const numTrainSamples = model.checkNumSamples(ins, batchSize, stepsPerEpoch, 'steps_per_epoch');
    let indexArray;
    if (numTrainSamples != null) {
        indexArray = range(0, numTrainSamples);
    }
    if (verbose == null) {
        verbose = 1;
    }
    const { callbackList, history } = configureCallbacks(callbacks, verbose, epochs, initialEpoch, numTrainSamples, stepsPerEpoch, batchSize, doValidation, callbackMetrics);
    callbackList.setModel(model);
    model.history = history;
    await callbackList.onTrainBegin();
    model.stopTraining_ = false;
    // TODO(cais): Take care of callbacks.validation_data as in PyKeras.
    // TODO(cais): Pre-convert feeds for performance as in PyKeras.
    for (let epoch = initialEpoch; epoch < epochs; ++epoch) {
        await callbackList.onEpochBegin(epoch);
        const epochLogs = {};
        if (stepsPerEpoch != null) {
            throw new NotImplementedError('stepsPerEpoch mode is not implemented yet.');
        }
        else {
            if (shuffle === 'batch') {
                throw new NotImplementedError('batch shuffling is not implemneted yet');
            }
            else if (shuffle) {
                util.shuffle(indexArray);
            }
            // Convert the potentially shuffled indices to Tensor1D, to avoid the
            // cost of repeated creation of Array1Ds later on.
            const epochIndexArray1D = tensor1d(indexArray);
            const batches = makeBatches(numTrainSamples, batchSize);
            for (let batchIndex = 0; batchIndex < batches.length; ++batchIndex) {
                const batchLogs = {};
                await callbackList.onBatchBegin(batchIndex, batchLogs);
                tfc.tidy(() => {
                    const batchStart = batches[batchIndex][0];
                    const batchEnd = batches[batchIndex][1];
                    const batchIds = sliceAlongFirstAxis(epochIndexArray1D, batchStart, batchEnd - batchStart);
                    batchLogs['batch'] = batchIndex;
                    batchLogs['size'] = batchEnd - batchStart;
                    // TODO(cais): In ins, train flag can be a number, instead of an
                    //   Tensor? Do we need to handle this in tfjs-layers?
                    const insBatch = sliceArraysByIndices(ins, batchIds);
                    const outs = f(insBatch);
                    for (let i = 0; i < outLabels.length; ++i) {
                        const label = outLabels[i];
                        const out = outs[i];
                        batchLogs[label] = out;
                        tfc.keep(out);
                        // TODO(cais): Use scope() to avoid ownership.
                    }
                    if (batchIndex === batches.length - 1) { // Last batch.
                        if (doValidation) {
                            const valOuts = model.testLoop(valF, valIns, batchSize);
                            // Porting Notes: In tfjs-layers, valOuts is always an Array.
                            for (let i = 0; i < outLabels.length; ++i) {
                                const label = outLabels[i];
                                const out = valOuts[i];
                                tfc.keep(out);
                                // TODO(cais): Use scope() to avoid ownership.
                                epochLogs['val_' + label] = out;
                            }
                        }
                    }
                });
                await callbackList.onBatchEnd(batchIndex, batchLogs);
                disposeTensorsInLogs(batchLogs);
                if (model.stopTraining_) {
                    break;
                }
                // TODO(cais): return outs as list of Tensor.
            }
            epochIndexArray1D.dispose();
        }
        // TODO(cais): Run validation at the end of the epoch.
        await callbackList.onEpochEnd(epoch, epochLogs);
        if (model.stopTraining_) {
            break;
        }
    }
    await callbackList.onTrainEnd();
    await model.history.syncData();
    return model.history;
}
export async function fitTensors(
// Type `model` as `any` here to avoid circular dependency w/ training.ts.
// tslint:disable-next-line:no-any
model, x, y, args = {}) {
    if (model.isTraining) {
        throw new Error('Cannot start training because another fit() call is ongoing.');
    }
    model.isTraining = true;
    let inputs;
    let targets;
    let originalInputs;
    let originalTargets;
    let inputValX;
    let inputValY;
    let valX;
    let valY;
    let sampleWeights;
    try {
        const batchSize = args.batchSize == null ? 32 : args.batchSize;
        checkBatchSize(batchSize);
        // Validate user data.
        // TODO(cais): Support sampleWeight.
        const checkBatchAxis = false;
        const standardizedOuts = await model.standardizeUserData(x, y, args.sampleWeight, args.classWeight, checkBatchAxis, batchSize);
        inputs = standardizedOuts[0];
        targets = standardizedOuts[1];
        sampleWeights = standardizedOuts[2];
        // Prepare validation data.
        let doValidation = false;
        let valIns;
        if (args.validationData != null && args.validationData.length > 0) {
            doValidation = true;
            if (args.validationData.length === 2) {
                // config.validationData consists of valX and valY.
                inputValX = args.validationData[0];
                inputValY = args.validationData[1];
            }
            else if (args.validationData.length === 3) {
                throw new NotImplementedError('validationData including sample weights is not supported yet.');
            }
            else {
                throw new ValueError(`When passing validation data, it must contain 2 (valX, valY) ` +
                    `or 3 (valX, valY, valSampleWeight) items; ` +
                    `${args.validationData} is invalid.`);
            }
            const checkBatchAxis = true;
            const valStandardized = await model.standardizeUserData(inputValX, inputValY, null, /** Unused sample weights. */ null, /** Unused class weights. */ checkBatchAxis, batchSize);
            valX = valStandardized[0];
            valY = valStandardized[1];
            valIns = valX.concat(valY);
            // TODO(cais): Add useLearningPhase data properly.
        }
        else if (args.validationSplit != null && args.validationSplit > 0 &&
            args.validationSplit < 1) {
            doValidation = true;
            // Porting Note: In tfjs-layers, inputs[0] is always a Tensor.
            const splitAt = Math.floor(inputs[0].shape[0] * (1 - args.validationSplit));
            const originalBatchSize = inputs[0].shape[0];
            valX = sliceArrays(inputs, splitAt, originalBatchSize);
            originalInputs = inputs;
            inputs = sliceArrays(inputs, 0, splitAt);
            valY = sliceArrays(targets, splitAt, originalBatchSize);
            originalTargets = targets;
            targets = sliceArrays(targets, 0, splitAt);
            // TODO(cais): Once sampleWeights becomes available, slice it to get
            //   valSampleWeights.
            valIns = valX.concat(valY);
            // TODO(cais): Add useLearningPhase data properly.
        }
        else if (args.validationSteps != null) {
            doValidation = true;
            // TODO(cais): Add useLearningPhase.
        }
        const ins = inputs.concat(targets).concat(sampleWeights);
        model.checkTrainableWeightsConsistency();
        // TODO(cais): Handle use_learning_phase and learning_phase?
        // Porting Note: Here we see a key deviation of tfjs-layers from
        // Keras.
        //  Due to the imperative nature of tfjs-layers' backend (tfjs-core),
        //  we do not construct symbolic computation graphs to embody the
        //  training process. Instead, we define a function that performs the
        //  training action. In PyKeras, the data (inputs and targets) are fed
        //  through graph placeholders. In tfjs-layers, the data are fed as
        //  function arguments. Since the function are defined below in the
        //  scope, we don't have equivalents of PyKeras's
        //  `_make_train_funciton`.
        const trainFunction = model.makeTrainFunction();
        const outLabels = model.getDedupedMetricsNames();
        let valFunction;
        let callbackMetrics;
        if (doValidation) {
            model.makeTestFunction();
            valFunction = model.testFunction;
            callbackMetrics =
                outLabels.slice().concat(outLabels.map(n => 'val_' + n));
        }
        else {
            valFunction = null;
            valIns = [];
            callbackMetrics = outLabels.slice();
        }
        const callbacks = standardizeCallbacks(args.callbacks, args.yieldEvery);
        const out = await fitLoop(model, trainFunction, ins, outLabels, batchSize, args.epochs, args.verbose, callbacks, valFunction, valIns, args.shuffle, callbackMetrics, args.initialEpoch, null, null);
        return out;
    }
    finally {
        model.isTraining = false;
        // Memory clean up.
        disposeNewTensors(inputs, x);
        disposeNewTensors(targets, y);
        disposeNewTensors(originalInputs, x);
        disposeNewTensors(originalTargets, y);
        disposeNewTensors(valX, inputValX);
        disposeNewTensors(valY, inputValY);
        if (sampleWeights != null) {
            tfc.dispose(sampleWeights);
        }
    }
    // TODO(cais): Add value to outLabels.
}
/**
 * Ensure tensors all have a rank of at least 2.
 *
 * If a tensor has a rank of 1, it is dimension-expanded to rank 2.
 * If any tensor has a rank of 0 (i.e., is a scalar), an error will be thrown.
 */
export function ensureTensorsRank2OrHigher(tensors) {
    const outs = [];
    if (tensors instanceof Tensor) {
        tensors = [tensors];
    }
    // Make Tensors at least 2D.
    for (let i = 0; i < tensors.length; ++i) {
        const tensor = tensors[i];
        if (tensor.rank === 1) {
            outs.push(expandDims(tensor, 1));
        }
        else if (tensor.rank === 0) {
            throw new Error('Expected tensor to be at least 1D, but received a 0D tensor ' +
                '(scalar).');
        }
        else {
            outs.push(tensor);
        }
    }
    return outs;
}
/**
 * Compare a set of tensors with a reference (old) set, discard the ones
 * in the new set that are not present in the reference set.
 *
 * This method is used for memory clenaup during calls such as
 * LayersModel.fit().
 *
 * @param tensors New set which may contain Tensors not present in
 *   `refTensors`.
 * @param refTensors Reference Tensor set.
 */
// TODO(cais, kangyizhang): Deduplicate with tfjs-data.
export function disposeNewTensors(tensors, refTensors) {
    if (tensors == null) {
        return;
    }
    const oldTensorIds = [];
    if (refTensors instanceof Tensor) {
        oldTensorIds.push(refTensors.id);
    }
    else if (Array.isArray(refTensors)) {
        refTensors.forEach(t => oldTensorIds.push(t.id));
    }
    else if (refTensors != null) {
        // `oldTensors` is a map from string name to Tensor.
        for (const name in refTensors) {
            const oldTensor = refTensors[name];
            oldTensorIds.push(oldTensor.id);
        }
    }
    const tensorsToDispose = [];
    if (tensors instanceof Tensor) {
        if (oldTensorIds.indexOf(tensors.id) === -1) {
            tensorsToDispose.push(tensors);
        }
    }
    else if (Array.isArray(tensors)) {
        tensors.forEach(t => {
            if (oldTensorIds.indexOf(t.id) === -1) {
                tensorsToDispose.push(t);
            }
        });
    }
    else if (tensors != null) {
        // `oldTensors` is a map from string name to Tensor.
        for (const name in tensors) {
            const tensor = tensors[name];
            if (oldTensorIds.indexOf(tensor.id) === -1) {
                tensorsToDispose.push(tensor);
            }
        }
    }
    tensorsToDispose.forEach(t => {
        if (!t.isDisposed) {
            t.dispose();
        }
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHJhaW5pbmdfdGVuc29ycy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uL3RmanMtbGF5ZXJzL3NyYy9lbmdpbmUvdHJhaW5pbmdfdGVuc29ycy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7Ozs7R0FRRztBQUVIOztHQUVHO0FBRUgsT0FBTyxLQUFLLEdBQUcsTUFBTSx1QkFBdUIsQ0FBQztBQUM3QyxPQUFPLEVBQVMsTUFBTSxFQUFZLFFBQVEsRUFBRSxJQUFJLEVBQUMsTUFBTSx1QkFBdUIsQ0FBQztBQUUvRSxPQUFPLEVBQUMsVUFBVSxFQUFFLE1BQU0sRUFBRSxtQkFBbUIsRUFBQyxNQUFNLHlCQUF5QixDQUFDO0FBQ2hGLE9BQU8sRUFBZSxrQkFBa0IsRUFBc0Qsb0JBQW9CLEVBQW9CLE1BQU0sbUJBQW1CLENBQUM7QUFDaEssT0FBTyxFQUFDLG1CQUFtQixFQUFFLFVBQVUsRUFBQyxNQUFNLFdBQVcsQ0FBQztBQUMxRCxPQUFPLEVBQUMsb0JBQW9CLEVBQWlCLE1BQU0sU0FBUyxDQUFDO0FBQzdELE9BQU8sRUFBQyxLQUFLLEVBQUMsTUFBTSxxQkFBcUIsQ0FBQztBQTRJMUMsTUFBTSxVQUFVLGNBQWMsQ0FBQyxTQUFpQjtJQUM5QyxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FDWCxTQUFTLEdBQUcsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLEVBQzVDLEdBQUcsRUFBRSxDQUFDLDJEQUNGLFNBQVMsRUFBRSxDQUFDLENBQUM7QUFDdkIsQ0FBQztBQUVEOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILE1BQU0sVUFBVSxXQUFXLENBQ3ZCLE1BQXVCLEVBQUUsS0FBYSxFQUFFLElBQVk7SUFDdEQsSUFBSSxNQUFNLElBQUksSUFBSSxFQUFFO1FBQ2xCLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUNmO1NBQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1FBQ2hDLE9BQU8sTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsSUFBSSxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUM7S0FDN0U7U0FBTSxFQUFHLFVBQVU7UUFDbEIsT0FBTyxtQkFBbUIsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLElBQUksR0FBRyxLQUFLLENBQUMsQ0FBQztLQUN6RDtBQUNILENBQUM7QUFFRDs7Ozs7Ozs7Ozs7O0dBWUc7QUFDSCxNQUFNLFVBQVUsb0JBQW9CLENBQ2hDLE1BQXVCLEVBQUUsT0FBaUI7SUFDNUMsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRTtRQUNuQixJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDbEIsT0FBTyxJQUFJLENBQUM7U0FDYjthQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNoQyxPQUFPLE1BQU0sQ0FBQyxHQUFHLENBQ2IsS0FBSyxDQUFDLEVBQUUsQ0FBRSxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFZLENBQUMsQ0FBQztTQUNoRTthQUFNO1lBQ0wsb0VBQW9FO1lBQ3BFLHNCQUFzQjtZQUN0QixPQUFPLE1BQU0sQ0FDVCxNQUFNLEVBQ04sT0FBTyxDQUFDLEtBQUssS0FBSyxPQUFPLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUN2RTtJQUNILENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVEOzs7Ozs7O0dBT0c7QUFDSCxNQUFNLFVBQVUsV0FBVyxDQUN2QixJQUFZLEVBQUUsU0FBaUI7SUFDakMsTUFBTSxNQUFNLEdBQTRCLEVBQUUsQ0FBQztJQUMzQyxJQUFJLFVBQVUsR0FBRyxDQUFDLENBQUM7SUFDbkIsSUFBSSxRQUFRLEdBQVcsSUFBSSxDQUFDO0lBQzVCLE9BQU8sVUFBVSxHQUFHLElBQUksRUFBRTtRQUN4QixRQUFRLEdBQUcsVUFBVSxHQUFHLFNBQVMsQ0FBQztRQUNsQyxJQUFJLFFBQVEsSUFBSSxJQUFJLEVBQUU7WUFDcEIsUUFBUSxHQUFHLElBQUksQ0FBQztTQUNqQjtRQUNELE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUNwQyxVQUFVLEdBQUcsUUFBUSxDQUFDO0tBQ3ZCO0lBQ0QsT0FBTyxNQUFNLENBQUM7QUFDaEIsQ0FBQztBQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQTBCRztBQUNILEtBQUssVUFBVSxPQUFPO0FBQ2xCLDBFQUEwRTtBQUMxRSxrQ0FBa0M7QUFDbEMsS0FBVSxFQUFFLENBQStCLEVBQUUsR0FBYSxFQUMxRCxTQUFvQixFQUFFLFNBQWtCLEVBQUUsTUFBZSxFQUFFLE9BQWdCLEVBQzNFLFNBQTBCLEVBQUUsSUFBbUMsRUFDL0QsTUFBaUIsRUFBRSxPQUF3QixFQUFFLGVBQTBCLEVBQ3ZFLFlBQXFCLEVBQUUsYUFBc0IsRUFDN0MsZUFBd0I7SUFDMUIsSUFBSSxTQUFTLElBQUksSUFBSSxFQUFFO1FBQ3JCLFNBQVMsR0FBRyxFQUFFLENBQUM7S0FDaEI7SUFDRCxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7UUFDbEIsTUFBTSxHQUFHLENBQUMsQ0FBQztLQUNaO0lBQ0QsSUFBSSxPQUFPLElBQUksSUFBSSxFQUFFO1FBQ25CLE9BQU8sR0FBRyxJQUFJLENBQUM7S0FDaEI7SUFDRCxJQUFJLFlBQVksSUFBSSxJQUFJLEVBQUU7UUFDeEIsWUFBWSxHQUFHLENBQUMsQ0FBQztLQUNsQjtJQUVELHNFQUFzRTtJQUN0RSxJQUFJLFlBQVksR0FBRyxLQUFLLENBQUM7SUFDekIsSUFBSSxJQUFJLElBQUksSUFBSSxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7UUFDbEMsWUFBWSxHQUFHLElBQUksQ0FBQztRQUNwQiwrQkFBK0I7S0FDaEM7SUFDRCxJQUFJLGVBQWUsSUFBSSxJQUFJLEVBQUU7UUFDM0IsWUFBWSxHQUFHLElBQUksQ0FBQztRQUNwQixJQUFJLGFBQWEsSUFBSSxJQUFJLEVBQUU7WUFDekIsTUFBTSxJQUFJLFVBQVUsQ0FDaEIsZ0VBQWdFO2dCQUNoRSxvQ0FBb0MsQ0FBQyxDQUFDO1NBQzNDO0tBQ0Y7SUFFRCxNQUFNLGVBQWUsR0FDakIsS0FBSyxDQUFDLGVBQWUsQ0FBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBQzVFLElBQUksVUFBb0IsQ0FBQztJQUN6QixJQUFJLGVBQWUsSUFBSSxJQUFJLEVBQUU7UUFDM0IsVUFBVSxHQUFHLEtBQUssQ0FBQyxDQUFDLEVBQUUsZUFBZSxDQUFDLENBQUM7S0FDeEM7SUFFRCxJQUFJLE9BQU8sSUFBSSxJQUFJLEVBQUU7UUFDbkIsT0FBTyxHQUFHLENBQUMsQ0FBQztLQUNiO0lBRUQsTUFBTSxFQUFDLFlBQVksRUFBRSxPQUFPLEVBQUMsR0FBRyxrQkFBa0IsQ0FDOUMsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFLGVBQWUsRUFBRSxhQUFhLEVBQ3hFLFNBQVMsRUFBRSxZQUFZLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDOUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM3QixLQUFLLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztJQUN4QixNQUFNLFlBQVksQ0FBQyxZQUFZLEVBQUUsQ0FBQztJQUNsQyxLQUFLLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQztJQUM1QixvRUFBb0U7SUFDcEUsK0RBQStEO0lBRS9ELEtBQUssSUFBSSxLQUFLLEdBQUcsWUFBWSxFQUFFLEtBQUssR0FBRyxNQUFNLEVBQUUsRUFBRSxLQUFLLEVBQUU7UUFDdEQsTUFBTSxZQUFZLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3ZDLE1BQU0sU0FBUyxHQUFtQixFQUFFLENBQUM7UUFDckMsSUFBSSxhQUFhLElBQUksSUFBSSxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxtQkFBbUIsQ0FDekIsNENBQTRDLENBQUMsQ0FBQztTQUNuRDthQUFNO1lBQ0wsSUFBSSxPQUFPLEtBQUssT0FBTyxFQUFFO2dCQUN2QixNQUFNLElBQUksbUJBQW1CLENBQUMsd0NBQXdDLENBQUMsQ0FBQzthQUN6RTtpQkFBTSxJQUFJLE9BQU8sRUFBRTtnQkFDbEIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQzthQUMxQjtZQUNELHFFQUFxRTtZQUNyRSxrREFBa0Q7WUFDbEQsTUFBTSxpQkFBaUIsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7WUFFL0MsTUFBTSxPQUFPLEdBQUcsV0FBVyxDQUFDLGVBQWUsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUN4RCxLQUFLLElBQUksVUFBVSxHQUFHLENBQUMsRUFBRSxVQUFVLEdBQUcsT0FBTyxDQUFDLE1BQU0sRUFBRSxFQUFFLFVBQVUsRUFBRTtnQkFDbEUsTUFBTSxTQUFTLEdBQW1CLEVBQUUsQ0FBQztnQkFDckMsTUFBTSxZQUFZLENBQUMsWUFBWSxDQUFDLFVBQVUsRUFBRSxTQUFTLENBQUMsQ0FBQztnQkFFdkQsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7b0JBQ1osTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO29CQUMxQyxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ3hDLE1BQU0sUUFBUSxHQUFHLG1CQUFtQixDQUNmLGlCQUFpQixFQUFFLFVBQVUsRUFDN0IsUUFBUSxHQUFHLFVBQVUsQ0FBYSxDQUFDO29CQUN4RCxTQUFTLENBQUMsT0FBTyxDQUFDLEdBQUcsVUFBVSxDQUFDO29CQUNoQyxTQUFTLENBQUMsTUFBTSxDQUFDLEdBQUcsUUFBUSxHQUFHLFVBQVUsQ0FBQztvQkFFMUMsZ0VBQWdFO29CQUNoRSxzREFBc0Q7b0JBQ3RELE1BQU0sUUFBUSxHQUFHLG9CQUFvQixDQUFDLEdBQUcsRUFBRSxRQUFRLENBQWEsQ0FBQztvQkFDakUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO29CQUN6QixLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsU0FBUyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsRUFBRTt3QkFDekMsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO3dCQUMzQixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7d0JBQ3BCLFNBQVMsQ0FBQyxLQUFLLENBQUMsR0FBRyxHQUFHLENBQUM7d0JBQ3ZCLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7d0JBQ2QsOENBQThDO3FCQUMvQztvQkFFRCxJQUFJLFVBQVUsS0FBSyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFHLGNBQWM7d0JBQ3RELElBQUksWUFBWSxFQUFFOzRCQUNoQixNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUUsU0FBUyxDQUFDLENBQUM7NEJBQ3hELDZEQUE2RDs0QkFDN0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLFNBQVMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLEVBQUU7Z0NBQ3pDLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQ0FDM0IsTUFBTSxHQUFHLEdBQUcsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dDQUN2QixHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dDQUNkLDhDQUE4QztnQ0FDOUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsR0FBRyxHQUFHLENBQUM7NkJBQ2pDO3lCQUNGO3FCQUNGO2dCQUNILENBQUMsQ0FBQyxDQUFDO2dCQUVILE1BQU0sWUFBWSxDQUFDLFVBQVUsQ0FBQyxVQUFVLEVBQUUsU0FBUyxDQUFDLENBQUM7Z0JBQ3JELG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUVoQyxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7b0JBQ3ZCLE1BQU07aUJBQ1A7Z0JBQ0QsNkNBQTZDO2FBQzlDO1lBRUQsaUJBQWlCLENBQUMsT0FBTyxFQUFFLENBQUM7U0FDN0I7UUFDRCxzREFBc0Q7UUFDdEQsTUFBTSxZQUFZLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDdkIsTUFBTTtTQUNQO0tBQ0Y7SUFDRCxNQUFNLFlBQVksQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUVoQyxNQUFNLEtBQUssQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDL0IsT0FBTyxLQUFLLENBQUMsT0FBTyxDQUFDO0FBQ3ZCLENBQUM7QUFFRCxNQUFNLENBQUMsS0FBSyxVQUFVLFVBQVU7QUFDNUIsMEVBQTBFO0FBQzFFLGtDQUFrQztBQUNsQyxLQUFVLEVBQUUsQ0FBZ0QsRUFDNUQsQ0FBZ0QsRUFDaEQsT0FBcUIsRUFBRTtJQUN6QixJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7UUFDcEIsTUFBTSxJQUFJLEtBQUssQ0FDWCw4REFBOEQsQ0FBQyxDQUFDO0tBQ3JFO0lBQ0QsS0FBSyxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUM7SUFDeEIsSUFBSSxNQUFnQixDQUFDO0lBQ3JCLElBQUksT0FBaUIsQ0FBQztJQUN0QixJQUFJLGNBQXdCLENBQUM7SUFDN0IsSUFBSSxlQUF5QixDQUFDO0lBQzlCLElBQUksU0FBMEIsQ0FBQztJQUMvQixJQUFJLFNBQTBCLENBQUM7SUFDL0IsSUFBSSxJQUFxQixDQUFDO0lBQzFCLElBQUksSUFBcUIsQ0FBQztJQUMxQixJQUFJLGFBQXVCLENBQUM7SUFDNUIsSUFBSTtRQUNGLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7UUFDL0QsY0FBYyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRTFCLHNCQUFzQjtRQUN0QixvQ0FBb0M7UUFDcEMsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDO1FBQzdCLE1BQU0sZ0JBQWdCLEdBQ2xCLE1BQU0sS0FBSyxDQUFDLG1CQUFtQixDQUMzQixDQUFDLEVBQUUsQ0FBQyxFQUFFLElBQUksQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxjQUFjLEVBQ3pELFNBQVMsQ0FBbUMsQ0FBQztRQUNyRCxNQUFNLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0IsT0FBTyxHQUFHLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlCLGFBQWEsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVwQywyQkFBMkI7UUFDM0IsSUFBSSxZQUFZLEdBQUcsS0FBSyxDQUFDO1FBQ3pCLElBQUksTUFBZ0IsQ0FBQztRQUNyQixJQUFJLElBQUksQ0FBQyxjQUFjLElBQUksSUFBSSxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNqRSxZQUFZLEdBQUcsSUFBSSxDQUFDO1lBQ3BCLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUNwQyxtREFBbUQ7Z0JBQ25ELFNBQVMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNuQyxTQUFTLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUNwQztpQkFBTSxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtnQkFDM0MsTUFBTSxJQUFJLG1CQUFtQixDQUN6QiwrREFBK0QsQ0FBQyxDQUFDO2FBQ3RFO2lCQUFNO2dCQUNMLE1BQU0sSUFBSSxVQUFVLENBQ2hCLCtEQUErRDtvQkFDL0QsNENBQTRDO29CQUM1QyxHQUFHLElBQUksQ0FBQyxjQUFjLGNBQWMsQ0FBQyxDQUFDO2FBQzNDO1lBRUQsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDO1lBQzVCLE1BQU0sZUFBZSxHQUNqQixNQUFNLEtBQUssQ0FBQyxtQkFBbUIsQ0FDM0IsU0FBUyxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsNkJBQTZCLENBQ3pELElBQUksRUFBd0IsNEJBQTRCLENBQ3hELGNBQWMsRUFBRSxTQUFTLENBQW1DLENBQUM7WUFDckUsSUFBSSxHQUFHLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMxQixJQUFJLEdBQUcsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzFCLE1BQU0sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzNCLGtEQUFrRDtTQUNuRDthQUFNLElBQ0gsSUFBSSxDQUFDLGVBQWUsSUFBSSxJQUFJLElBQUksSUFBSSxDQUFDLGVBQWUsR0FBRyxDQUFDO1lBQ3hELElBQUksQ0FBQyxlQUFlLEdBQUcsQ0FBQyxFQUFFO1lBQzVCLFlBQVksR0FBRyxJQUFJLENBQUM7WUFDcEIsOERBQThEO1lBQzlELE1BQU0sT0FBTyxHQUNULElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQztZQUNoRSxNQUFNLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDN0MsSUFBSSxHQUFHLFdBQVcsQ0FBQyxNQUFNLEVBQUUsT0FBTyxFQUFFLGlCQUFpQixDQUFhLENBQUM7WUFDbkUsY0FBYyxHQUFHLE1BQU0sQ0FBQztZQUN4QixNQUFNLEdBQUcsV0FBVyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsT0FBTyxDQUFhLENBQUM7WUFDckQsSUFBSSxHQUFHLFdBQVcsQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLGlCQUFpQixDQUFhLENBQUM7WUFDcEUsZUFBZSxHQUFHLE9BQU8sQ0FBQztZQUMxQixPQUFPLEdBQUcsV0FBVyxDQUFDLE9BQU8sRUFBRSxDQUFDLEVBQUUsT0FBTyxDQUFhLENBQUM7WUFDdkQsb0VBQW9FO1lBQ3BFLHNCQUFzQjtZQUN0QixNQUFNLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUUzQixrREFBa0Q7U0FDbkQ7YUFBTSxJQUFJLElBQUksQ0FBQyxlQUFlLElBQUksSUFBSSxFQUFFO1lBQ3ZDLFlBQVksR0FBRyxJQUFJLENBQUM7WUFDcEIsb0NBQW9DO1NBQ3JDO1FBRUQsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFekQsS0FBSyxDQUFDLGdDQUFnQyxFQUFFLENBQUM7UUFFekMsNERBQTREO1FBRTVELGdFQUFnRTtRQUNoRSxTQUFTO1FBQ1QscUVBQXFFO1FBQ3JFLGlFQUFpRTtRQUNqRSxxRUFBcUU7UUFDckUsc0VBQXNFO1FBQ3RFLG1FQUFtRTtRQUNuRSxtRUFBbUU7UUFDbkUsaURBQWlEO1FBQ2pELDJCQUEyQjtRQUMzQixNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztRQUNoRCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsc0JBQXNCLEVBQWMsQ0FBQztRQUU3RCxJQUFJLFdBQXlDLENBQUM7UUFDOUMsSUFBSSxlQUF5QixDQUFDO1FBQzlCLElBQUksWUFBWSxFQUFFO1lBQ2hCLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO1lBQ3pCLFdBQVcsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1lBQ2pDLGVBQWU7Z0JBQ1gsU0FBUyxDQUFDLEtBQUssRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDOUQ7YUFBTTtZQUNMLFdBQVcsR0FBRyxJQUFJLENBQUM7WUFDbkIsTUFBTSxHQUFHLEVBQUUsQ0FBQztZQUNaLGVBQWUsR0FBRyxTQUFTLENBQUMsS0FBSyxFQUFFLENBQUM7U0FDckM7UUFFRCxNQUFNLFNBQVMsR0FBRyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN4RSxNQUFNLEdBQUcsR0FBRyxNQUFNLE9BQU8sQ0FDckIsS0FBSyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsTUFBTSxFQUM1RCxJQUFJLENBQUMsT0FBTyxFQUFFLFNBQVMsRUFBRSxXQUFXLEVBQUUsTUFBTSxFQUFFLElBQUksQ0FBQyxPQUFPLEVBQzFELGVBQWUsRUFBRSxJQUFJLENBQUMsWUFBWSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNwRCxPQUFPLEdBQUcsQ0FBQztLQUNaO1lBQVM7UUFDUixLQUFLLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQztRQUN6QixtQkFBbUI7UUFDbkIsaUJBQWlCLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQzdCLGlCQUFpQixDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM5QixpQkFBaUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDckMsaUJBQWlCLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3RDLGlCQUFpQixDQUFDLElBQWdCLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDL0MsaUJBQWlCLENBQUMsSUFBZ0IsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUMvQyxJQUFJLGFBQWEsSUFBSSxJQUFJLEVBQUU7WUFDekIsR0FBRyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUM1QjtLQUNGO0lBQ0Qsc0NBQXNDO0FBQ3hDLENBQUM7QUFFRDs7Ozs7R0FLRztBQUNILE1BQU0sVUFBVSwwQkFBMEIsQ0FBQyxPQUF3QjtJQUNqRSxNQUFNLElBQUksR0FBYSxFQUFFLENBQUM7SUFDMUIsSUFBSSxPQUFPLFlBQVksTUFBTSxFQUFFO1FBQzdCLE9BQU8sR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0tBQ3JCO0lBRUQsNEJBQTRCO0lBQzVCLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxFQUFFO1FBQ3ZDLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxQixJQUFJLE1BQU0sQ0FBQyxJQUFJLEtBQUssQ0FBQyxFQUFFO1lBQ3JCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ2xDO2FBQU0sSUFBSSxNQUFNLENBQUMsSUFBSSxLQUFLLENBQUMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUNYLDhEQUE4RDtnQkFDOUQsV0FBVyxDQUFDLENBQUM7U0FDbEI7YUFBTTtZQUNMLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDbkI7S0FDRjtJQUNELE9BQU8sSUFBSSxDQUFDO0FBQ2QsQ0FBQztBQUVEOzs7Ozs7Ozs7O0dBVUc7QUFDSCx1REFBdUQ7QUFDdkQsTUFBTSxVQUFVLGlCQUFpQixDQUM3QixPQUFzRCxFQUN0RCxVQUF5RDtJQUMzRCxJQUFJLE9BQU8sSUFBSSxJQUFJLEVBQUU7UUFDbkIsT0FBTztLQUNSO0lBQ0QsTUFBTSxZQUFZLEdBQWEsRUFBRSxDQUFDO0lBQ2xDLElBQUksVUFBVSxZQUFZLE1BQU0sRUFBRTtRQUNoQyxZQUFZLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUNsQztTQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFBRTtRQUNwQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztLQUNsRDtTQUFNLElBQUksVUFBVSxJQUFJLElBQUksRUFBRTtRQUM3QixvREFBb0Q7UUFDcEQsS0FBSyxNQUFNLElBQUksSUFBSSxVQUFVLEVBQUU7WUFDN0IsTUFBTSxTQUFTLEdBQUcsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ25DLFlBQVksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ2pDO0tBQ0Y7SUFFRCxNQUFNLGdCQUFnQixHQUFhLEVBQUUsQ0FBQztJQUN0QyxJQUFJLE9BQU8sWUFBWSxNQUFNLEVBQUU7UUFDN0IsSUFBSSxZQUFZLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtZQUMzQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDaEM7S0FDRjtTQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRTtRQUNqQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2xCLElBQUksWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7Z0JBQ3JDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUMxQjtRQUNILENBQUMsQ0FBQyxDQUFDO0tBQ0o7U0FBTSxJQUFJLE9BQU8sSUFBSSxJQUFJLEVBQUU7UUFDMUIsb0RBQW9EO1FBQ3BELEtBQUssTUFBTSxJQUFJLElBQUksT0FBTyxFQUFFO1lBQzFCLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM3QixJQUFJLFlBQVksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO2dCQUMxQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7YUFDL0I7U0FDRjtLQUNGO0lBRUQsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1FBQzNCLElBQUksQ0FBQyxDQUFDLENBQUMsVUFBVSxFQUFFO1lBQ2pCLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztTQUNiO0lBQ0gsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IDIwMTggR29vZ2xlIExMQ1xuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZVxuICogbGljZW5zZSB0aGF0IGNhbiBiZSBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIG9yIGF0XG4gKiBodHRwczovL29wZW5zb3VyY2Uub3JnL2xpY2Vuc2VzL01JVC5cbiAqID09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09XG4gKi9cblxuLyoqXG4gKiBJbnRlcmZhY2VzIGFuZCBtZXRob2RzIGZvciB0cmFpbmluZyBtb2RlbHMgdXNpbmcgdGYuVGVuc29yIG9iamVjdHMuXG4gKi9cblxuaW1wb3J0ICogYXMgdGZjIGZyb20gJ0B0ZW5zb3JmbG93L3RmanMtY29yZSc7XG5pbXBvcnQge1NjYWxhciwgVGVuc29yLCBUZW5zb3IxRCwgdGVuc29yMWQsIHV0aWx9IGZyb20gJ0B0ZW5zb3JmbG93L3RmanMtY29yZSc7XG5cbmltcG9ydCB7ZXhwYW5kRGltcywgZ2F0aGVyLCBzbGljZUFsb25nRmlyc3RBeGlzfSBmcm9tICcuLi9iYWNrZW5kL3RmanNfYmFja2VuZCc7XG5pbXBvcnQge0Jhc2VDYWxsYmFjaywgY29uZmlndXJlQ2FsbGJhY2tzLCBDdXN0b21DYWxsYmFja0FyZ3MsIEhpc3RvcnksIE1vZGVsTG9nZ2luZ1ZlcmJvc2l0eSwgc3RhbmRhcmRpemVDYWxsYmFja3MsIFlpZWxkRXZlcnlPcHRpb25zfSBmcm9tICcuLi9iYXNlX2NhbGxiYWNrcyc7XG5pbXBvcnQge05vdEltcGxlbWVudGVkRXJyb3IsIFZhbHVlRXJyb3J9IGZyb20gJy4uL2Vycm9ycyc7XG5pbXBvcnQge2Rpc3Bvc2VUZW5zb3JzSW5Mb2dzLCBVbnJlc29sdmVkTG9nc30gZnJvbSAnLi4vbG9ncyc7XG5pbXBvcnQge3JhbmdlfSBmcm9tICcuLi91dGlscy9tYXRoX3V0aWxzJztcbmltcG9ydCB7Q2xhc3NXZWlnaHQsIENsYXNzV2VpZ2h0TWFwfSBmcm9tICcuL3RyYWluaW5nX3V0aWxzJztcblxuLyoqXG4gKiBJbnRlcmZhY2UgY29uZmlndXJhdGlvbiBtb2RlbCB0cmFpbmluZyBiYXNlZCBvbiBkYXRhIGFzIGB0Zi5UZW5zb3Jgcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNb2RlbEZpdEFyZ3Mge1xuICAvKipcbiAgICogTnVtYmVyIG9mIHNhbXBsZXMgcGVyIGdyYWRpZW50IHVwZGF0ZS4gSWYgdW5zcGVjaWZpZWQsIGl0XG4gICAqIHdpbGwgZGVmYXVsdCB0byAzMi5cbiAgICovXG4gIGJhdGNoU2l6ZT86IG51bWJlcjtcblxuICAvKipcbiAgICogSW50ZWdlciBudW1iZXIgb2YgdGltZXMgdG8gaXRlcmF0ZSBvdmVyIHRoZSB0cmFpbmluZyBkYXRhIGFycmF5cy5cbiAgICovXG4gIGVwb2Nocz86IG51bWJlcjtcblxuICAvKipcbiAgICogVmVyYm9zaXR5IGxldmVsLlxuICAgKlxuICAgKiBFeHBlY3RlZCB0byBiZSAwLCAxLCBvciAyLiBEZWZhdWx0OiAxLlxuICAgKlxuICAgKiAwIC0gTm8gcHJpbnRlZCBtZXNzYWdlIGR1cmluZyBmaXQoKSBjYWxsLlxuICAgKiAxIC0gSW4gTm9kZS5qcyAodGZqcy1ub2RlKSwgcHJpbnRzIHRoZSBwcm9ncmVzcyBiYXIsIHRvZ2V0aGVyIHdpdGhcbiAgICogICAgIHJlYWwtdGltZSB1cGRhdGVzIG9mIGxvc3MgYW5kIG1ldHJpYyB2YWx1ZXMgYW5kIHRyYWluaW5nIHNwZWVkLlxuICAgKiAgICAgSW4gdGhlIGJyb3dzZXI6IG5vIGFjdGlvbi4gVGhpcyBpcyB0aGUgZGVmYXVsdC5cbiAgICogMiAtIE5vdCBpbXBsZW1lbnRlZCB5ZXQuXG4gICAqL1xuICB2ZXJib3NlPzogTW9kZWxMb2dnaW5nVmVyYm9zaXR5O1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGNhbGxiYWNrcyB0byBiZSBjYWxsZWQgZHVyaW5nIHRyYWluaW5nLlxuICAgKiBDYW4gaGF2ZSBvbmUgb3IgbW9yZSBvZiB0aGUgZm9sbG93aW5nIGNhbGxiYWNrczpcbiAgICogICAtIGBvblRyYWluQmVnaW4obG9ncylgOiBjYWxsZWQgd2hlbiB0cmFpbmluZyBzdGFydHMuXG4gICAqICAgLSBgb25UcmFpbkVuZChsb2dzKWA6IGNhbGxlZCB3aGVuIHRyYWluaW5nIGVuZHMuXG4gICAqICAgLSBgb25FcG9jaEJlZ2luKGVwb2NoLCBsb2dzKWA6IGNhbGxlZCBhdCB0aGUgc3RhcnQgb2YgZXZlcnkgZXBvY2guXG4gICAqICAgLSBgb25FcG9jaEVuZChlcG9jaCwgbG9ncylgOiBjYWxsZWQgYXQgdGhlIGVuZCBvZiBldmVyeSBlcG9jaC5cbiAgICogICAtIGBvbkJhdGNoQmVnaW4oYmF0Y2gsIGxvZ3MpYDogY2FsbGVkIGF0IHRoZSBzdGFydCBvZiBldmVyeSBiYXRjaC5cbiAgICogICAtIGBvbkJhdGNoRW5kKGJhdGNoLCBsb2dzKWA6IGNhbGxlZCBhdCB0aGUgZW5kIG9mIGV2ZXJ5IGJhdGNoLlxuICAgKiAgIC0gYG9uWWllbGQoZXBvY2gsIGJhdGNoLCBsb2dzKWA6IGNhbGxlZCBldmVyeSBgeWllbGRFdmVyeWAgbWlsbGlzZWNvbmRzXG4gICAqICAgICAgd2l0aCB0aGUgY3VycmVudCBlcG9jaCwgYmF0Y2ggYW5kIGxvZ3MuIFRoZSBsb2dzIGFyZSB0aGUgc2FtZVxuICAgKiAgICAgIGFzIGluIGBvbkJhdGNoRW5kKClgLiBOb3RlIHRoYXQgYG9uWWllbGRgIGNhbiBza2lwIGJhdGNoZXMgb3JcbiAgICogICAgICBlcG9jaHMuIFNlZSBhbHNvIGRvY3MgZm9yIGB5aWVsZEV2ZXJ5YCBiZWxvdy5cbiAgICovXG4gIGNhbGxiYWNrcz86IEJhc2VDYWxsYmFja1tdfEN1c3RvbUNhbGxiYWNrQXJnc3xDdXN0b21DYWxsYmFja0FyZ3NbXTtcblxuICAvKipcbiAgICogRmxvYXQgYmV0d2VlbiAwIGFuZCAxOiBmcmFjdGlvbiBvZiB0aGUgdHJhaW5pbmcgZGF0YVxuICAgKiB0byBiZSB1c2VkIGFzIHZhbGlkYXRpb24gZGF0YS4gVGhlIG1vZGVsIHdpbGwgc2V0IGFwYXJ0IHRoaXMgZnJhY3Rpb24gb2ZcbiAgICogdGhlIHRyYWluaW5nIGRhdGEsIHdpbGwgbm90IHRyYWluIG9uIGl0LCBhbmQgd2lsbCBldmFsdWF0ZSB0aGUgbG9zcyBhbmRcbiAgICogYW55IG1vZGVsIG1ldHJpY3Mgb24gdGhpcyBkYXRhIGF0IHRoZSBlbmQgb2YgZWFjaCBlcG9jaC5cbiAgICogVGhlIHZhbGlkYXRpb24gZGF0YSBpcyBzZWxlY3RlZCBmcm9tIHRoZSBsYXN0IHNhbXBsZXMgaW4gdGhlIGB4YCBhbmQgYHlgXG4gICAqIGRhdGEgcHJvdmlkZWQsIGJlZm9yZSBzaHVmZmxpbmcuXG4gICAqL1xuICB2YWxpZGF0aW9uU3BsaXQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIERhdGEgb24gd2hpY2ggdG8gZXZhbHVhdGUgdGhlIGxvc3MgYW5kIGFueSBtb2RlbFxuICAgKiBtZXRyaWNzIGF0IHRoZSBlbmQgb2YgZWFjaCBlcG9jaC4gVGhlIG1vZGVsIHdpbGwgbm90IGJlIHRyYWluZWQgb24gdGhpc1xuICAgKiBkYXRhLiBUaGlzIGNvdWxkIGJlIGEgdHVwbGUgW3hWYWwsIHlWYWxdIG9yIGEgdHVwbGUgW3hWYWwsIHlWYWwsXG4gICAqIHZhbFNhbXBsZVdlaWdodHNdLiBUaGUgbW9kZWwgd2lsbCBub3QgYmUgdHJhaW5lZCBvbiB0aGlzIGRhdGEuXG4gICAqIGB2YWxpZGF0aW9uRGF0YWAgd2lsbCBvdmVycmlkZSBgdmFsaWRhdGlvblNwbGl0YC5cbiAgICovXG4gIHZhbGlkYXRpb25EYXRhPzogW1xuICAgIFRlbnNvcnxUZW5zb3JbXSwgVGVuc29yfFRlbnNvcltdXG4gIF18W1RlbnNvciB8IFRlbnNvcltdLCBUZW5zb3J8VGVuc29yW10sIFRlbnNvcnxUZW5zb3JbXV07XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gc2h1ZmZsZSB0aGUgdHJhaW5pbmcgZGF0YSBiZWZvcmUgZWFjaCBlcG9jaC4gSGFzXG4gICAqIG5vIGVmZmVjdCB3aGVuIGBzdGVwc1BlckVwb2NoYCBpcyBub3QgYG51bGxgLlxuICAgKi9cbiAgc2h1ZmZsZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIG9iamVjdCBtYXBwaW5nIGNsYXNzIGluZGljZXMgKGludGVnZXJzKSB0b1xuICAgKiBhIHdlaWdodCAoZmxvYXQpIHRvIGFwcGx5IHRvIHRoZSBtb2RlbCdzIGxvc3MgZm9yIHRoZSBzYW1wbGVzIGZyb20gdGhpc1xuICAgKiBjbGFzcyBkdXJpbmcgdHJhaW5pbmcuIFRoaXMgY2FuIGJlIHVzZWZ1bCB0byB0ZWxsIHRoZSBtb2RlbCB0byBcInBheSBtb3JlXG4gICAqIGF0dGVudGlvblwiIHRvIHNhbXBsZXMgZnJvbSBhbiB1bmRlci1yZXByZXNlbnRlZCBjbGFzcy5cbiAgICpcbiAgICogSWYgdGhlIG1vZGVsIGhhcyBtdWx0aXBsZSBvdXRwdXRzLCBhIGNsYXNzIHdlaWdodCBjYW4gYmUgc3BlY2lmaWVkIGZvclxuICAgKiBlYWNoIG9mIHRoZSBvdXRwdXRzIGJ5IHNldHRpbmcgdGhpcyBmaWVsZCBhbiBhcnJheSBvZiB3ZWlnaHQgb2JqZWN0XG4gICAqIG9yIGFuIG9iamVjdCB0aGF0IG1hcHMgbW9kZWwgb3V0cHV0IG5hbWVzIChlLmcuLCBgbW9kZWwub3V0cHV0TmFtZXNbMF1gKVxuICAgKiB0byB3ZWlnaHQgb2JqZWN0cy5cbiAgICovXG4gIGNsYXNzV2VpZ2h0PzogQ2xhc3NXZWlnaHR8Q2xhc3NXZWlnaHRbXXxDbGFzc1dlaWdodE1hcDtcblxuICAvKipcbiAgICogT3B0aW9uYWwgYXJyYXkgb2YgdGhlIHNhbWUgbGVuZ3RoIGFzIHgsIGNvbnRhaW5pbmdcbiAgICogd2VpZ2h0cyB0byBhcHBseSB0byB0aGUgbW9kZWwncyBsb3NzIGZvciBlYWNoIHNhbXBsZS4gSW4gdGhlIGNhc2Ugb2ZcbiAgICogdGVtcG9yYWwgZGF0YSwgeW91IGNhbiBwYXNzIGEgMkQgYXJyYXkgd2l0aCBzaGFwZSAoc2FtcGxlcyxcbiAgICogc2VxdWVuY2VMZW5ndGgpLCB0byBhcHBseSBhIGRpZmZlcmVudCB3ZWlnaHQgdG8gZXZlcnkgdGltZXN0ZXAgb2YgZXZlcnlcbiAgICogc2FtcGxlLiBJbiB0aGlzIGNhc2UgeW91IHNob3VsZCBtYWtlIHN1cmUgdG8gc3BlY2lmeVxuICAgKiBzYW1wbGVXZWlnaHRNb2RlPVwidGVtcG9yYWxcIiBpbiBjb21waWxlKCkuXG4gICAqL1xuICBzYW1wbGVXZWlnaHQ/OiBUZW5zb3I7XG5cbiAgLyoqXG4gICAqIEVwb2NoIGF0IHdoaWNoIHRvIHN0YXJ0IHRyYWluaW5nICh1c2VmdWwgZm9yIHJlc3VtaW5nIGEgcHJldmlvdXMgdHJhaW5pbmdcbiAgICogcnVuKS4gV2hlbiB0aGlzIGlzIHVzZWQsIGBlcG9jaHNgIGlzIHRoZSBpbmRleCBvZiB0aGUgXCJmaW5hbCBlcG9jaFwiLlxuICAgKiBUaGUgbW9kZWwgaXMgbm90IHRyYWluZWQgZm9yIGEgbnVtYmVyIG9mIGl0ZXJhdGlvbnMgZ2l2ZW4gYnkgYGVwb2Noc2AsXG4gICAqIGJ1dCBtZXJlbHkgdW50aWwgdGhlIGVwb2NoIG9mIGluZGV4IGBlcG9jaHNgIGlzIHJlYWNoZWQuXG4gICAqL1xuICBpbml0aWFsRXBvY2g/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRvdGFsIG51bWJlciBvZiBzdGVwcyAoYmF0Y2hlcyBvZiBzYW1wbGVzKSBiZWZvcmVcbiAgICogZGVjbGFyaW5nIG9uZSBlcG9jaCBmaW5pc2hlZCBhbmQgc3RhcnRpbmcgdGhlIG5leHQgZXBvY2guIFdoZW4gdHJhaW5pbmdcbiAgICogd2l0aCBJbnB1dCBUZW5zb3JzIHN1Y2ggYXMgVGVuc29yRmxvdyBkYXRhIHRlbnNvcnMsIHRoZSBkZWZhdWx0IGBudWxsYCBpc1xuICAgKiBlcXVhbCB0byB0aGUgbnVtYmVyIG9mIHVuaXF1ZSBzYW1wbGVzIGluIHlvdXIgZGF0YXNldCBkaXZpZGVkIGJ5IHRoZVxuICAgKiBiYXRjaCBzaXplLCBvciAxIGlmIHRoYXQgY2Fubm90IGJlIGRldGVybWluZWQuXG4gICAqL1xuICBzdGVwc1BlckVwb2NoPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBPbmx5IHJlbGV2YW50IGlmIGBzdGVwc1BlckVwb2NoYCBpcyBzcGVjaWZpZWQuIFRvdGFsIG51bWJlciBvZiBzdGVwc1xuICAgKiAoYmF0Y2hlcyBvZiBzYW1wbGVzKSB0byB2YWxpZGF0ZSBiZWZvcmUgc3RvcHBpbmcuXG4gICAqL1xuICB2YWxpZGF0aW9uU3RlcHM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgdGhlIGZyZXF1ZW5jeSBvZiB5aWVsZGluZyB0aGUgbWFpbiB0aHJlYWQgdG8gb3RoZXIgdGFza3MuXG4gICAqXG4gICAqIEluIHRoZSBicm93c2VyIGVudmlyb25tZW50LCB5aWVsZGluZyB0aGUgbWFpbiB0aHJlYWQgY2FuIGltcHJvdmUgdGhlXG4gICAqIHJlc3BvbnNpdmVuZXNzIG9mIHRoZSBwYWdlIGR1cmluZyB0cmFpbmluZy4gSW4gdGhlIE5vZGUuanMgZW52aXJvbm1lbnQsXG4gICAqIGl0IGNhbiBlbnN1cmUgdGFza3MgcXVldWVkIGluIHRoZSBldmVudCBsb29wIGNhbiBiZSBoYW5kbGVkIGluIGEgdGltZWx5XG4gICAqIG1hbm5lci5cbiAgICpcbiAgICogVGhlIHZhbHVlIGNhbiBiZSBvbmUgb2YgdGhlIGZvbGxvd2luZzpcbiAgICogICAtIGAnYXV0bydgOiBUaGUgeWllbGRpbmcgaGFwcGVucyBhdCBhIGNlcnRhaW4gZnJhbWUgcmF0ZSAoY3VycmVudGx5IHNldFxuICAgKiAgICAgICAgICAgICAgIGF0IDEyNW1zKS4gVGhpcyBpcyB0aGUgZGVmYXVsdC5cbiAgICogICAtIGAnYmF0Y2gnYDogeWllbGQgZXZlcnkgYmF0Y2guXG4gICAqICAgLSBgJ2Vwb2NoJ2A6IHlpZWxkIGV2ZXJ5IGVwb2NoLlxuICAgKiAgIC0gYW55IGBudW1iZXJgOiB5aWVsZCBldmVyeSBgbnVtYmVyYCBtaWxsaXNlY29uZHMuXG4gICAqICAgLSBgJ25ldmVyJ2A6IG5ldmVyIHlpZWxkLiAoeWllbGRpbmcgY2FuIHN0aWxsIGhhcHBlbiB0aHJvdWdoIGBhd2FpdFxuICAgKiAgICAgIG5leHRGcmFtZSgpYCBjYWxscyBpbiBjdXN0b20gY2FsbGJhY2tzLilcbiAgICovXG4gIHlpZWxkRXZlcnk/OiBZaWVsZEV2ZXJ5T3B0aW9ucztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNoZWNrQmF0Y2hTaXplKGJhdGNoU2l6ZTogbnVtYmVyKSB7XG4gIHRmYy51dGlsLmFzc2VydChcbiAgICAgIGJhdGNoU2l6ZSA+IDAgJiYgTnVtYmVyLmlzSW50ZWdlcihiYXRjaFNpemUpLFxuICAgICAgKCkgPT4gYGJhdGNoU2l6ZSBpcyByZXF1aXJlZCB0byBiZSBhIHBvc2l0aXZlIGludGVnZXIsIGJ1dCBnb3QgJHtcbiAgICAgICAgICBiYXRjaFNpemV9YCk7XG59XG5cbi8qKlxuICogU2xpY2UgYSBUZW5zb3Igb3IgYW4gQXJyYXkgb2YgVGVuc29ycywgYnkgc3RhcnQgYW5kIHN0b3AgaW5kaWNlcy5cbiAqXG4gKiBQb3J0aW5nIE5vdGU6IFRoZSBgX3NsaWNlX2FycmF5c2AgZnVuY3Rpb24gaW4gUHlLZXJhcyBpcyBjb3ZlcmVkIGJ5IHRoaXNcbiAqICAgZnVuY3Rpb24gYW5kIGBzbGljZUFycmF5c0J5SW5kaWNlcygpYCB0b2dldGhlci5cbiAqXG4gKiBAcGFyYW0gYXJyYXlzOiB0aGUgaW5wdXQuXG4gKiBAcGFyYW0gc3RhcnQ6IHRoZSBzdGFydGluZyBpbmRleCAoaW5jbHVzaXZlKS5cbiAqIEBwYXJhbSBzdG9wOiB0aGUgc3RvcHBpbmcgaW5kZXggKGV4Y2x1c2l2ZSkuXG4gKiBAcmV0dXJucyBUaGUgcmVzdWx0IG9mIHRoZSBzbGljaW5nLiBJZiBgYXJyYXlzYCBpcyBhbiBgQXJyYXlgIG9mXG4gKiAgIGB0Zi5UZW5zb3JgcywgdGhlIHNsaWNpbmcgd2lsbCBiZSBhcHBsaWVkIHRvIGFsbCBlbGVtZW50cyBvZiB0aGUgYEFycmF5YFxuICogICBpbiB0aGUgc2FtZSB3YXkuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzbGljZUFycmF5cyhcbiAgICBhcnJheXM6IFRlbnNvcnxUZW5zb3JbXSwgc3RhcnQ6IG51bWJlciwgc3RvcDogbnVtYmVyKTogVGVuc29yfFRlbnNvcltdIHtcbiAgaWYgKGFycmF5cyA9PSBudWxsKSB7XG4gICAgcmV0dXJuIFtudWxsXTtcbiAgfSBlbHNlIGlmIChBcnJheS5pc0FycmF5KGFycmF5cykpIHtcbiAgICByZXR1cm4gYXJyYXlzLm1hcChhcnJheSA9PiBzbGljZUFsb25nRmlyc3RBeGlzKGFycmF5LCBzdGFydCwgc3RvcCAtIHN0YXJ0KSk7XG4gIH0gZWxzZSB7ICAvLyBUZW5zb3IuXG4gICAgcmV0dXJuIHNsaWNlQWxvbmdGaXJzdEF4aXMoYXJyYXlzLCBzdGFydCwgc3RvcCAtIHN0YXJ0KTtcbiAgfVxufVxuXG4vKipcbiAqIFNsaWNlIGEgVGVuc29yIG9yIGFuIEFycmF5IG9mIFRlbnNvcnMsIGJ5IHJhbmRvbS1vcmRlciBpbmRpY2VzLlxuICpcbiAqIFBvcnRpbmcgTm90ZTogVGhlIGBfc2xpY2VfYXJyYXlzYCBmdW5jdGlvbiBpbiBQeUtlcmFzIGlzIGNvdmVyZWQgYnkgdGhpc1xuICogICBmdW5jdGlvbiBhbmQgYHNsaWNlQXJyYXlzKClgIHRvZ2V0aGVyLlxuICpcbiAqIEBwYXJhbSBhcnJheXMgVGhlIGlucHV0IGB0Zi5UZW5zb3JgIG9yIGBBcnJheWAgb2YgYHRmLlRlbnNvcmBzIHRvIHNsaWNlLlxuICogICBJZiBhbiBgQXJyYXlgIG9mIGB0Zi5UZW5zb3JgcywgYWxsIGB0Zi5UZW5zb3JgcyB3aWxsIGJlIHNsaWNlZCBpbiB0aGVcbiAqICAgc2FtZSBmYXNoaW9uLlxuICogQHBhcmFtIGluZGljZXMgVGhlIGluZGljZXMgdG8gdXNlIGZvciBzbGljaW5nIGFsb25nIHRoZSBmaXJzdCAoYmF0Y2gpXG4gKiAgIGRpbWVuc2lvbi5cbiAqIEByZXR1cm5zIFJlc3VsdChzKSBvZiB0aGUgc2xpY2luZy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHNsaWNlQXJyYXlzQnlJbmRpY2VzKFxuICAgIGFycmF5czogVGVuc29yfFRlbnNvcltdLCBpbmRpY2VzOiBUZW5zb3IxRCk6IFRlbnNvcnxUZW5zb3JbXSB7XG4gIHJldHVybiB0ZmMudGlkeSgoKSA9PiB7XG4gICAgaWYgKGFycmF5cyA9PSBudWxsKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9IGVsc2UgaWYgKEFycmF5LmlzQXJyYXkoYXJyYXlzKSkge1xuICAgICAgcmV0dXJuIGFycmF5cy5tYXAoXG4gICAgICAgICAgYXJyYXkgPT4gKHNsaWNlQXJyYXlzQnlJbmRpY2VzKGFycmF5LCBpbmRpY2VzKSBhcyBUZW5zb3IpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gVE9ETyhjYWlzKTogaW5kaWNlcyBzaG91bGQgYmUgYSBwcmUtY29uc3RydWN0ZWQgVGVuc29yMUQgdG8gYXZvaWRcbiAgICAgIC8vICAgdGVuc29yMWQoKSBjYWxscy5cbiAgICAgIHJldHVybiBnYXRoZXIoXG4gICAgICAgICAgYXJyYXlzLFxuICAgICAgICAgIGluZGljZXMuZHR5cGUgPT09ICdpbnQzMicgPyBpbmRpY2VzIDogdGZjLmNhc3QoaW5kaWNlcywgJ2ludDMyJykpO1xuICAgIH1cbiAgfSk7XG59XG5cbi8qKlxuICogUmV0dXJucyBhIGxpc3Qgb2YgYmF0Y2ggaW5kaWNlcyAodHVwbGVzIG9mIGluZGljZXMpLlxuICogQHBhcmFtIHNpemU6IEludGVnZXIsIHRvdGFsIHNpemUgb2YgdGhlIGRhdGEgdG8gc2xpY2UgaW50byBiYXRjaGVzLlxuICogQHBhcmFtIGJhdGNoU2l6ZTogSW50ZWdlciwgYmF0Y2ggc2l6ZS5cbiAqIEByZXR1cm5zIEFuIEFycmF5IG9mIFtiYXRjaFN0YXJ0LCBiYXRjaEVuZF0gdHVwbGVzLiBiYXRjaFN0YXJ0IGlzXG4gKiAgIGluY2x1c2l2ZTsgYmF0Y2hFbmQgaXMgZXhjbHVzaXZlLiBJLmUuLCBlYWNoIGJhdGNoIGNvbnNpc3RzIG9mIGluZGljZXMgeFxuICogICB0aGF0IHNhdGlzZnkgYmF0Y2hTdGFydCA8PSB4IDwgYmF0Y2hFbmQuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBtYWtlQmF0Y2hlcyhcbiAgICBzaXplOiBudW1iZXIsIGJhdGNoU2l6ZTogbnVtYmVyKTogQXJyYXk8W251bWJlciwgbnVtYmVyXT4ge1xuICBjb25zdCBvdXRwdXQ6IEFycmF5PFtudW1iZXIsIG51bWJlcl0+ID0gW107XG4gIGxldCBiYXRjaFN0YXJ0ID0gMDtcbiAgbGV0IGJhdGNoRW5kOiBudW1iZXIgPSBudWxsO1xuICB3aGlsZSAoYmF0Y2hTdGFydCA8IHNpemUpIHtcbiAgICBiYXRjaEVuZCA9IGJhdGNoU3RhcnQgKyBiYXRjaFNpemU7XG4gICAgaWYgKGJhdGNoRW5kID49IHNpemUpIHtcbiAgICAgIGJhdGNoRW5kID0gc2l6ZTtcbiAgICB9XG4gICAgb3V0cHV0LnB1c2goW2JhdGNoU3RhcnQsIGJhdGNoRW5kXSk7XG4gICAgYmF0Y2hTdGFydCA9IGJhdGNoRW5kO1xuICB9XG4gIHJldHVybiBvdXRwdXQ7XG59XG5cbi8qKlxuICogQWJzdHJhY3QgZml0IGZ1bmN0aW9uIGZvciBgZihpbnMpYC5cbiAqIEBwYXJhbSBmIEEgRnVuY3Rpb24gcmV0dXJuaW5nIGEgbGlzdCBvZiB0ZW5zb3JzLiBGb3IgdHJhaW5pbmcsIHRoaXNcbiAqICAgZnVuY3Rpb24gaXMgZXhwZWN0ZWQgdG8gcGVyZm9ybSB0aGUgdXBkYXRlcyB0byB0aGUgdmFyaWFibGVzLlxuICogQHBhcmFtIGlucyBMaXN0IG9mIHRlbnNvcnMgdG8gYmUgZmVkIHRvIGBmYC5cbiAqIEBwYXJhbSBvdXRMYWJlbHMgTGlzdCBvZiBzdHJpbmdzLCBkaXNwbGF5IG5hbWVzIG9mIHRoZSBvdXRwdXRzIG9mIGBmYC5cbiAqIEBwYXJhbSBiYXRjaFNpemUgSW50ZWdlciBiYXRjaCBzaXplIG9yIGA9PSBudWxsYCBpZiB1bmtub3duLiBEZWZhdWx0IDogMzIuXG4gKiBAcGFyYW0gZXBvY2hzIE51bWJlciBvZiB0aW1lcyB0byBpdGVyYXRlIG92ZXIgdGhlIGRhdGEuIERlZmF1bHQgOiAxLlxuICogQHBhcmFtIHZlcmJvc2UgVmVyYm9zaXR5IG1vZGU6IDAsIDEsIG9yIDIuIERlZmF1bHQ6IDEuXG4gKiBAcGFyYW0gY2FsbGJhY2tzIExpc3Qgb2YgY2FsbGJhY2tzIHRvIGJlIGNhbGxlZCBkdXJpbmcgdHJhaW5pbmcuXG4gKiBAcGFyYW0gdmFsRiBGdW5jdGlvbiB0byBjYWxsIGZvciB2YWxpZGF0aW9uLlxuICogQHBhcmFtIHZhbElucyBMaXN0IG9mIHRlbnNvcnMgdG8gYmUgZmVkIHRvIGB2YWxGYC5cbiAqIEBwYXJhbSBzaHVmZmxlIFdoZXRoZXIgdG8gc2h1ZmZsZSB0aGUgZGF0YSBhdCB0aGUgYmVnaW5uaW5nIG9mIGV2ZXJ5XG4gKiBlcG9jaC4gRGVmYXVsdCA6IHRydWUuXG4gKiBAcGFyYW0gY2FsbGJhY2tNZXRyaWNzIExpc3Qgb2Ygc3RyaW5ncywgdGhlIGRpc3BsYXkgbmFtZXMgb2YgdGhlIG1ldHJpY3NcbiAqICAgcGFzc2VkIHRvIHRoZSBjYWxsYmFja3MuIFRoZXkgc2hvdWxkIGJlIHRoZSBjb25jYXRlbmF0aW9uIG9mIHRoZVxuICogICBkaXNwbGF5IG5hbWVzIG9mIHRoZSBvdXRwdXRzIG9mIGBmYCBhbmQgdGhlIGxpc3Qgb2YgZGlzcGxheSBuYW1lc1xuICogICBvZiB0aGUgb3V0cHV0cyBvZiBgdmFsRmAuXG4gKiBAcGFyYW0gaW5pdGlhbEVwb2NoIEVwb2NoIGF0IHdoaWNoIHRvIHN0YXJ0IHRyYWluaW5nICh1c2VmdWwgZm9yXG4gKiAgIHJlc3VtaW5nIGEgcHJldmlvdXMgdHJhaW5pbmcgcnVuKS4gRGVmYXVsdCA6IDAuXG4gKiBAcGFyYW0gc3RlcHNQZXJFcG9jaCBUb3RhbCBudW1iZXIgb2Ygc3RlcHMgKGJhdGNoZXMgb24gc2FtcGxlcykgYmVmb3JlXG4gKiAgIGRlY2xhcmluZyBvbmUgZXBvY2ggZmluaXNoZWQgYW5kIHN0YXJ0aW5nIHRoZSBuZXh0IGVwb2NoLiBJZ25vcmVkIHdpdGhcbiAqICAgdGhlIGRlZmF1bHQgdmFsdWUgb2YgYHVuZGVmaW5lZGAgb3IgYG51bGxgLlxuICogQHBhcmFtIHZhbGlkYXRpb25TdGVwcyBOdW1iZXIgb2Ygc3RlcHMgdG8gcnVuIHZhbGlkYXRpb24gZm9yIChvbmx5IGlmXG4gKiAgIGRvaW5nIHZhbGlkYXRpb24gZnJvbSBkYXRhIHRlbnNvcnMpLiBOb3QgYXBwbGljYWJsZSBmb3IgdGZqcy1sYXllcnMuXG4gKiBAcmV0dXJucyBBIGBIaXN0b3J5YCBvYmplY3QuXG4gKi9cbmFzeW5jIGZ1bmN0aW9uIGZpdExvb3AoXG4gICAgLy8gVHlwZSBgbW9kZWxgIGFzIGBhbnlgIGhlcmUgdG8gYXZvaWQgY2lyY3VsYXIgZGVwZW5kZW5jeSB3LyB0cmFpbmluZy50cy5cbiAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bm8tYW55XG4gICAgbW9kZWw6IGFueSwgZjogKGRhdGE6IFRlbnNvcltdKSA9PiBTY2FsYXJbXSwgaW5zOiBUZW5zb3JbXSxcbiAgICBvdXRMYWJlbHM/OiBzdHJpbmdbXSwgYmF0Y2hTaXplPzogbnVtYmVyLCBlcG9jaHM/OiBudW1iZXIsIHZlcmJvc2U/OiBudW1iZXIsXG4gICAgY2FsbGJhY2tzPzogQmFzZUNhbGxiYWNrW10sIHZhbEY/OiAoZGF0YTogVGVuc29yW10pID0+IFNjYWxhcltdLFxuICAgIHZhbElucz86IFRlbnNvcltdLCBzaHVmZmxlPzogYm9vbGVhbnxzdHJpbmcsIGNhbGxiYWNrTWV0cmljcz86IHN0cmluZ1tdLFxuICAgIGluaXRpYWxFcG9jaD86IG51bWJlciwgc3RlcHNQZXJFcG9jaD86IG51bWJlcixcbiAgICB2YWxpZGF0aW9uU3RlcHM/OiBudW1iZXIpOiBQcm9taXNlPEhpc3Rvcnk+IHtcbiAgaWYgKGJhdGNoU2l6ZSA9PSBudWxsKSB7XG4gICAgYmF0Y2hTaXplID0gMzI7XG4gIH1cbiAgaWYgKGVwb2NocyA9PSBudWxsKSB7XG4gICAgZXBvY2hzID0gMTtcbiAgfVxuICBpZiAoc2h1ZmZsZSA9PSBudWxsKSB7XG4gICAgc2h1ZmZsZSA9IHRydWU7XG4gIH1cbiAgaWYgKGluaXRpYWxFcG9jaCA9PSBudWxsKSB7XG4gICAgaW5pdGlhbEVwb2NoID0gMDtcbiAgfVxuXG4gIC8vIFRPRE8oY2Fpcyk6IENoYW5nZSBjb25zdCB0byBsZXQgYmVsb3cgd2hlbiBpbXBsZW1lbnRpbmcgdmFsaWRhdGlvbi5cbiAgbGV0IGRvVmFsaWRhdGlvbiA9IGZhbHNlO1xuICBpZiAodmFsRiAhPSBudWxsICYmIHZhbElucyAhPSBudWxsKSB7XG4gICAgZG9WYWxpZGF0aW9uID0gdHJ1ZTtcbiAgICAvLyBUT0RPKGNhaXMpOiB2ZXJib3NlIG1lc3NhZ2UuXG4gIH1cbiAgaWYgKHZhbGlkYXRpb25TdGVwcyAhPSBudWxsKSB7XG4gICAgZG9WYWxpZGF0aW9uID0gdHJ1ZTtcbiAgICBpZiAoc3RlcHNQZXJFcG9jaCA9PSBudWxsKSB7XG4gICAgICB0aHJvdyBuZXcgVmFsdWVFcnJvcihcbiAgICAgICAgICAnQ2FuIG9ubHkgdXNlIGB2YWxpZGF0aW9uU3RlcHNgIHdoZW4gZG9pbmcgc3RlcC13aXNlIHRyYWluaW5nLCAnICtcbiAgICAgICAgICAnaS5lLiwgYHN0ZXBzUGVyRXBvY2hgIG11c3QgYmUgc2V0LicpO1xuICAgIH1cbiAgfVxuXG4gIGNvbnN0IG51bVRyYWluU2FtcGxlcyA9XG4gICAgICBtb2RlbC5jaGVja051bVNhbXBsZXMoaW5zLCBiYXRjaFNpemUsIHN0ZXBzUGVyRXBvY2gsICdzdGVwc19wZXJfZXBvY2gnKTtcbiAgbGV0IGluZGV4QXJyYXk6IG51bWJlcltdO1xuICBpZiAobnVtVHJhaW5TYW1wbGVzICE9IG51bGwpIHtcbiAgICBpbmRleEFycmF5ID0gcmFuZ2UoMCwgbnVtVHJhaW5TYW1wbGVzKTtcbiAgfVxuXG4gIGlmICh2ZXJib3NlID09IG51bGwpIHtcbiAgICB2ZXJib3NlID0gMTtcbiAgfVxuXG4gIGNvbnN0IHtjYWxsYmFja0xpc3QsIGhpc3Rvcnl9ID0gY29uZmlndXJlQ2FsbGJhY2tzKFxuICAgICAgY2FsbGJhY2tzLCB2ZXJib3NlLCBlcG9jaHMsIGluaXRpYWxFcG9jaCwgbnVtVHJhaW5TYW1wbGVzLCBzdGVwc1BlckVwb2NoLFxuICAgICAgYmF0Y2hTaXplLCBkb1ZhbGlkYXRpb24sIGNhbGxiYWNrTWV0cmljcyk7XG4gIGNhbGxiYWNrTGlzdC5zZXRNb2RlbChtb2RlbCk7XG4gIG1vZGVsLmhpc3RvcnkgPSBoaXN0b3J5O1xuICBhd2FpdCBjYWxsYmFja0xpc3Qub25UcmFpbkJlZ2luKCk7XG4gIG1vZGVsLnN0b3BUcmFpbmluZ18gPSBmYWxzZTtcbiAgLy8gVE9ETyhjYWlzKTogVGFrZSBjYXJlIG9mIGNhbGxiYWNrcy52YWxpZGF0aW9uX2RhdGEgYXMgaW4gUHlLZXJhcy5cbiAgLy8gVE9ETyhjYWlzKTogUHJlLWNvbnZlcnQgZmVlZHMgZm9yIHBlcmZvcm1hbmNlIGFzIGluIFB5S2VyYXMuXG5cbiAgZm9yIChsZXQgZXBvY2ggPSBpbml0aWFsRXBvY2g7IGVwb2NoIDwgZXBvY2hzOyArK2Vwb2NoKSB7XG4gICAgYXdhaXQgY2FsbGJhY2tMaXN0Lm9uRXBvY2hCZWdpbihlcG9jaCk7XG4gICAgY29uc3QgZXBvY2hMb2dzOiBVbnJlc29sdmVkTG9ncyA9IHt9O1xuICAgIGlmIChzdGVwc1BlckVwb2NoICE9IG51bGwpIHtcbiAgICAgIHRocm93IG5ldyBOb3RJbXBsZW1lbnRlZEVycm9yKFxuICAgICAgICAgICdzdGVwc1BlckVwb2NoIG1vZGUgaXMgbm90IGltcGxlbWVudGVkIHlldC4nKTtcbiAgICB9IGVsc2Uge1xuICAgICAgaWYgKHNodWZmbGUgPT09ICdiYXRjaCcpIHtcbiAgICAgICAgdGhyb3cgbmV3IE5vdEltcGxlbWVudGVkRXJyb3IoJ2JhdGNoIHNodWZmbGluZyBpcyBub3QgaW1wbGVtbmV0ZWQgeWV0Jyk7XG4gICAgICB9IGVsc2UgaWYgKHNodWZmbGUpIHtcbiAgICAgICAgdXRpbC5zaHVmZmxlKGluZGV4QXJyYXkpO1xuICAgICAgfVxuICAgICAgLy8gQ29udmVydCB0aGUgcG90ZW50aWFsbHkgc2h1ZmZsZWQgaW5kaWNlcyB0byBUZW5zb3IxRCwgdG8gYXZvaWQgdGhlXG4gICAgICAvLyBjb3N0IG9mIHJlcGVhdGVkIGNyZWF0aW9uIG9mIEFycmF5MURzIGxhdGVyIG9uLlxuICAgICAgY29uc3QgZXBvY2hJbmRleEFycmF5MUQgPSB0ZW5zb3IxZChpbmRleEFycmF5KTtcblxuICAgICAgY29uc3QgYmF0Y2hlcyA9IG1ha2VCYXRjaGVzKG51bVRyYWluU2FtcGxlcywgYmF0Y2hTaXplKTtcbiAgICAgIGZvciAobGV0IGJhdGNoSW5kZXggPSAwOyBiYXRjaEluZGV4IDwgYmF0Y2hlcy5sZW5ndGg7ICsrYmF0Y2hJbmRleCkge1xuICAgICAgICBjb25zdCBiYXRjaExvZ3M6IFVucmVzb2x2ZWRMb2dzID0ge307XG4gICAgICAgIGF3YWl0IGNhbGxiYWNrTGlzdC5vbkJhdGNoQmVnaW4oYmF0Y2hJbmRleCwgYmF0Y2hMb2dzKTtcblxuICAgICAgICB0ZmMudGlkeSgoKSA9PiB7XG4gICAgICAgICAgY29uc3QgYmF0Y2hTdGFydCA9IGJhdGNoZXNbYmF0Y2hJbmRleF1bMF07XG4gICAgICAgICAgY29uc3QgYmF0Y2hFbmQgPSBiYXRjaGVzW2JhdGNoSW5kZXhdWzFdO1xuICAgICAgICAgIGNvbnN0IGJhdGNoSWRzID0gc2xpY2VBbG9uZ0ZpcnN0QXhpcyhcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBlcG9jaEluZGV4QXJyYXkxRCwgYmF0Y2hTdGFydCxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBiYXRjaEVuZCAtIGJhdGNoU3RhcnQpIGFzIFRlbnNvcjFEO1xuICAgICAgICAgIGJhdGNoTG9nc1snYmF0Y2gnXSA9IGJhdGNoSW5kZXg7XG4gICAgICAgICAgYmF0Y2hMb2dzWydzaXplJ10gPSBiYXRjaEVuZCAtIGJhdGNoU3RhcnQ7XG5cbiAgICAgICAgICAvLyBUT0RPKGNhaXMpOiBJbiBpbnMsIHRyYWluIGZsYWcgY2FuIGJlIGEgbnVtYmVyLCBpbnN0ZWFkIG9mIGFuXG4gICAgICAgICAgLy8gICBUZW5zb3I/IERvIHdlIG5lZWQgdG8gaGFuZGxlIHRoaXMgaW4gdGZqcy1sYXllcnM/XG4gICAgICAgICAgY29uc3QgaW5zQmF0Y2ggPSBzbGljZUFycmF5c0J5SW5kaWNlcyhpbnMsIGJhdGNoSWRzKSBhcyBUZW5zb3JbXTtcbiAgICAgICAgICBjb25zdCBvdXRzID0gZihpbnNCYXRjaCk7XG4gICAgICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCBvdXRMYWJlbHMubGVuZ3RoOyArK2kpIHtcbiAgICAgICAgICAgIGNvbnN0IGxhYmVsID0gb3V0TGFiZWxzW2ldO1xuICAgICAgICAgICAgY29uc3Qgb3V0ID0gb3V0c1tpXTtcbiAgICAgICAgICAgIGJhdGNoTG9nc1tsYWJlbF0gPSBvdXQ7XG4gICAgICAgICAgICB0ZmMua2VlcChvdXQpO1xuICAgICAgICAgICAgLy8gVE9ETyhjYWlzKTogVXNlIHNjb3BlKCkgdG8gYXZvaWQgb3duZXJzaGlwLlxuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmIChiYXRjaEluZGV4ID09PSBiYXRjaGVzLmxlbmd0aCAtIDEpIHsgIC8vIExhc3QgYmF0Y2guXG4gICAgICAgICAgICBpZiAoZG9WYWxpZGF0aW9uKSB7XG4gICAgICAgICAgICAgIGNvbnN0IHZhbE91dHMgPSBtb2RlbC50ZXN0TG9vcCh2YWxGLCB2YWxJbnMsIGJhdGNoU2l6ZSk7XG4gICAgICAgICAgICAgIC8vIFBvcnRpbmcgTm90ZXM6IEluIHRmanMtbGF5ZXJzLCB2YWxPdXRzIGlzIGFsd2F5cyBhbiBBcnJheS5cbiAgICAgICAgICAgICAgZm9yIChsZXQgaSA9IDA7IGkgPCBvdXRMYWJlbHMubGVuZ3RoOyArK2kpIHtcbiAgICAgICAgICAgICAgICBjb25zdCBsYWJlbCA9IG91dExhYmVsc1tpXTtcbiAgICAgICAgICAgICAgICBjb25zdCBvdXQgPSB2YWxPdXRzW2ldO1xuICAgICAgICAgICAgICAgIHRmYy5rZWVwKG91dCk7XG4gICAgICAgICAgICAgICAgLy8gVE9ETyhjYWlzKTogVXNlIHNjb3BlKCkgdG8gYXZvaWQgb3duZXJzaGlwLlxuICAgICAgICAgICAgICAgIGVwb2NoTG9nc1sndmFsXycgKyBsYWJlbF0gPSBvdXQ7XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuXG4gICAgICAgIGF3YWl0IGNhbGxiYWNrTGlzdC5vbkJhdGNoRW5kKGJhdGNoSW5kZXgsIGJhdGNoTG9ncyk7XG4gICAgICAgIGRpc3Bvc2VUZW5zb3JzSW5Mb2dzKGJhdGNoTG9ncyk7XG5cbiAgICAgICAgaWYgKG1vZGVsLnN0b3BUcmFpbmluZ18pIHtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgfVxuICAgICAgICAvLyBUT0RPKGNhaXMpOiByZXR1cm4gb3V0cyBhcyBsaXN0IG9mIFRlbnNvci5cbiAgICAgIH1cblxuICAgICAgZXBvY2hJbmRleEFycmF5MUQuZGlzcG9zZSgpO1xuICAgIH1cbiAgICAvLyBUT0RPKGNhaXMpOiBSdW4gdmFsaWRhdGlvbiBhdCB0aGUgZW5kIG9mIHRoZSBlcG9jaC5cbiAgICBhd2FpdCBjYWxsYmFja0xpc3Qub25FcG9jaEVuZChlcG9jaCwgZXBvY2hMb2dzKTtcbiAgICBpZiAobW9kZWwuc3RvcFRyYWluaW5nXykge1xuICAgICAgYnJlYWs7XG4gICAgfVxuICB9XG4gIGF3YWl0IGNhbGxiYWNrTGlzdC5vblRyYWluRW5kKCk7XG5cbiAgYXdhaXQgbW9kZWwuaGlzdG9yeS5zeW5jRGF0YSgpO1xuICByZXR1cm4gbW9kZWwuaGlzdG9yeTtcbn1cblxuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGZpdFRlbnNvcnMoXG4gICAgLy8gVHlwZSBgbW9kZWxgIGFzIGBhbnlgIGhlcmUgdG8gYXZvaWQgY2lyY3VsYXIgZGVwZW5kZW5jeSB3LyB0cmFpbmluZy50cy5cbiAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bm8tYW55XG4gICAgbW9kZWw6IGFueSwgeDogVGVuc29yfFRlbnNvcltdfHtbaW5wdXROYW1lOiBzdHJpbmddOiBUZW5zb3J9LFxuICAgIHk6IFRlbnNvcnxUZW5zb3JbXXx7W2lucHV0TmFtZTogc3RyaW5nXTogVGVuc29yfSxcbiAgICBhcmdzOiBNb2RlbEZpdEFyZ3MgPSB7fSk6IFByb21pc2U8SGlzdG9yeT4ge1xuICBpZiAobW9kZWwuaXNUcmFpbmluZykge1xuICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgJ0Nhbm5vdCBzdGFydCB0cmFpbmluZyBiZWNhdXNlIGFub3RoZXIgZml0KCkgY2FsbCBpcyBvbmdvaW5nLicpO1xuICB9XG4gIG1vZGVsLmlzVHJhaW5pbmcgPSB0cnVlO1xuICBsZXQgaW5wdXRzOiBUZW5zb3JbXTtcbiAgbGV0IHRhcmdldHM6IFRlbnNvcltdO1xuICBsZXQgb3JpZ2luYWxJbnB1dHM6IFRlbnNvcltdO1xuICBsZXQgb3JpZ2luYWxUYXJnZXRzOiBUZW5zb3JbXTtcbiAgbGV0IGlucHV0VmFsWDogVGVuc29yfFRlbnNvcltdO1xuICBsZXQgaW5wdXRWYWxZOiBUZW5zb3J8VGVuc29yW107XG4gIGxldCB2YWxYOiBUZW5zb3J8VGVuc29yW107XG4gIGxldCB2YWxZOiBUZW5zb3J8VGVuc29yW107XG4gIGxldCBzYW1wbGVXZWlnaHRzOiBUZW5zb3JbXTtcbiAgdHJ5IHtcbiAgICBjb25zdCBiYXRjaFNpemUgPSBhcmdzLmJhdGNoU2l6ZSA9PSBudWxsID8gMzIgOiBhcmdzLmJhdGNoU2l6ZTtcbiAgICBjaGVja0JhdGNoU2l6ZShiYXRjaFNpemUpO1xuXG4gICAgLy8gVmFsaWRhdGUgdXNlciBkYXRhLlxuICAgIC8vIFRPRE8oY2Fpcyk6IFN1cHBvcnQgc2FtcGxlV2VpZ2h0LlxuICAgIGNvbnN0IGNoZWNrQmF0Y2hBeGlzID0gZmFsc2U7XG4gICAgY29uc3Qgc3RhbmRhcmRpemVkT3V0cyA9XG4gICAgICAgIGF3YWl0IG1vZGVsLnN0YW5kYXJkaXplVXNlckRhdGEoXG4gICAgICAgICAgICB4LCB5LCBhcmdzLnNhbXBsZVdlaWdodCwgYXJncy5jbGFzc1dlaWdodCwgY2hlY2tCYXRjaEF4aXMsXG4gICAgICAgICAgICBiYXRjaFNpemUpIGFzIFtUZW5zb3JbXSwgVGVuc29yW10sIFRlbnNvcltdXTtcbiAgICBpbnB1dHMgPSBzdGFuZGFyZGl6ZWRPdXRzWzBdO1xuICAgIHRhcmdldHMgPSBzdGFuZGFyZGl6ZWRPdXRzWzFdO1xuICAgIHNhbXBsZVdlaWdodHMgPSBzdGFuZGFyZGl6ZWRPdXRzWzJdO1xuXG4gICAgLy8gUHJlcGFyZSB2YWxpZGF0aW9uIGRhdGEuXG4gICAgbGV0IGRvVmFsaWRhdGlvbiA9IGZhbHNlO1xuICAgIGxldCB2YWxJbnM6IFRlbnNvcltdO1xuICAgIGlmIChhcmdzLnZhbGlkYXRpb25EYXRhICE9IG51bGwgJiYgYXJncy52YWxpZGF0aW9uRGF0YS5sZW5ndGggPiAwKSB7XG4gICAgICBkb1ZhbGlkYXRpb24gPSB0cnVlO1xuICAgICAgaWYgKGFyZ3MudmFsaWRhdGlvbkRhdGEubGVuZ3RoID09PSAyKSB7XG4gICAgICAgIC8vIGNvbmZpZy52YWxpZGF0aW9uRGF0YSBjb25zaXN0cyBvZiB2YWxYIGFuZCB2YWxZLlxuICAgICAgICBpbnB1dFZhbFggPSBhcmdzLnZhbGlkYXRpb25EYXRhWzBdO1xuICAgICAgICBpbnB1dFZhbFkgPSBhcmdzLnZhbGlkYXRpb25EYXRhWzFdO1xuICAgICAgfSBlbHNlIGlmIChhcmdzLnZhbGlkYXRpb25EYXRhLmxlbmd0aCA9PT0gMykge1xuICAgICAgICB0aHJvdyBuZXcgTm90SW1wbGVtZW50ZWRFcnJvcihcbiAgICAgICAgICAgICd2YWxpZGF0aW9uRGF0YSBpbmNsdWRpbmcgc2FtcGxlIHdlaWdodHMgaXMgbm90IHN1cHBvcnRlZCB5ZXQuJyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBuZXcgVmFsdWVFcnJvcihcbiAgICAgICAgICAgIGBXaGVuIHBhc3NpbmcgdmFsaWRhdGlvbiBkYXRhLCBpdCBtdXN0IGNvbnRhaW4gMiAodmFsWCwgdmFsWSkgYCArXG4gICAgICAgICAgICBgb3IgMyAodmFsWCwgdmFsWSwgdmFsU2FtcGxlV2VpZ2h0KSBpdGVtczsgYCArXG4gICAgICAgICAgICBgJHthcmdzLnZhbGlkYXRpb25EYXRhfSBpcyBpbnZhbGlkLmApO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBjaGVja0JhdGNoQXhpcyA9IHRydWU7XG4gICAgICBjb25zdCB2YWxTdGFuZGFyZGl6ZWQgPVxuICAgICAgICAgIGF3YWl0IG1vZGVsLnN0YW5kYXJkaXplVXNlckRhdGEoXG4gICAgICAgICAgICAgIGlucHV0VmFsWCwgaW5wdXRWYWxZLCBudWxsLCAvKiogVW51c2VkIHNhbXBsZSB3ZWlnaHRzLiAqL1xuICAgICAgICAgICAgICBudWxsLCAgICAgICAgICAgICAgICAgICAgICAgLyoqIFVudXNlZCBjbGFzcyB3ZWlnaHRzLiAqL1xuICAgICAgICAgICAgICBjaGVja0JhdGNoQXhpcywgYmF0Y2hTaXplKSBhcyBbVGVuc29yW10sIFRlbnNvcltdLCBUZW5zb3JbXV07XG4gICAgICB2YWxYID0gdmFsU3RhbmRhcmRpemVkWzBdO1xuICAgICAgdmFsWSA9IHZhbFN0YW5kYXJkaXplZFsxXTtcbiAgICAgIHZhbElucyA9IHZhbFguY29uY2F0KHZhbFkpO1xuICAgICAgLy8gVE9ETyhjYWlzKTogQWRkIHVzZUxlYXJuaW5nUGhhc2UgZGF0YSBwcm9wZXJseS5cbiAgICB9IGVsc2UgaWYgKFxuICAgICAgICBhcmdzLnZhbGlkYXRpb25TcGxpdCAhPSBudWxsICYmIGFyZ3MudmFsaWRhdGlvblNwbGl0ID4gMCAmJlxuICAgICAgICBhcmdzLnZhbGlkYXRpb25TcGxpdCA8IDEpIHtcbiAgICAgIGRvVmFsaWRhdGlvbiA9IHRydWU7XG4gICAgICAvLyBQb3J0aW5nIE5vdGU6IEluIHRmanMtbGF5ZXJzLCBpbnB1dHNbMF0gaXMgYWx3YXlzIGEgVGVuc29yLlxuICAgICAgY29uc3Qgc3BsaXRBdCA9XG4gICAgICAgICAgTWF0aC5mbG9vcihpbnB1dHNbMF0uc2hhcGVbMF0gKiAoMSAtIGFyZ3MudmFsaWRhdGlvblNwbGl0KSk7XG4gICAgICBjb25zdCBvcmlnaW5hbEJhdGNoU2l6ZSA9IGlucHV0c1swXS5zaGFwZVswXTtcbiAgICAgIHZhbFggPSBzbGljZUFycmF5cyhpbnB1dHMsIHNwbGl0QXQsIG9yaWdpbmFsQmF0Y2hTaXplKSBhcyBUZW5zb3JbXTtcbiAgICAgIG9yaWdpbmFsSW5wdXRzID0gaW5wdXRzO1xuICAgICAgaW5wdXRzID0gc2xpY2VBcnJheXMoaW5wdXRzLCAwLCBzcGxpdEF0KSBhcyBUZW5zb3JbXTtcbiAgICAgIHZhbFkgPSBzbGljZUFycmF5cyh0YXJnZXRzLCBzcGxpdEF0LCBvcmlnaW5hbEJhdGNoU2l6ZSkgYXMgVGVuc29yW107XG4gICAgICBvcmlnaW5hbFRhcmdldHMgPSB0YXJnZXRzO1xuICAgICAgdGFyZ2V0cyA9IHNsaWNlQXJyYXlzKHRhcmdldHMsIDAsIHNwbGl0QXQpIGFzIFRlbnNvcltdO1xuICAgICAgLy8gVE9ETyhjYWlzKTogT25jZSBzYW1wbGVXZWlnaHRzIGJlY29tZXMgYXZhaWxhYmxlLCBzbGljZSBpdCB0byBnZXRcbiAgICAgIC8vICAgdmFsU2FtcGxlV2VpZ2h0cy5cbiAgICAgIHZhbElucyA9IHZhbFguY29uY2F0KHZhbFkpO1xuXG4gICAgICAvLyBUT0RPKGNhaXMpOiBBZGQgdXNlTGVhcm5pbmdQaGFzZSBkYXRhIHByb3Blcmx5LlxuICAgIH0gZWxzZSBpZiAoYXJncy52YWxpZGF0aW9uU3RlcHMgIT0gbnVsbCkge1xuICAgICAgZG9WYWxpZGF0aW9uID0gdHJ1ZTtcbiAgICAgIC8vIFRPRE8oY2Fpcyk6IEFkZCB1c2VMZWFybmluZ1BoYXNlLlxuICAgIH1cblxuICAgIGNvbnN0IGlucyA9IGlucHV0cy5jb25jYXQodGFyZ2V0cykuY29uY2F0KHNhbXBsZVdlaWdodHMpO1xuXG4gICAgbW9kZWwuY2hlY2tUcmFpbmFibGVXZWlnaHRzQ29uc2lzdGVuY3koKTtcblxuICAgIC8vIFRPRE8oY2Fpcyk6IEhhbmRsZSB1c2VfbGVhcm5pbmdfcGhhc2UgYW5kIGxlYXJuaW5nX3BoYXNlP1xuXG4gICAgLy8gUG9ydGluZyBOb3RlOiBIZXJlIHdlIHNlZSBhIGtleSBkZXZpYXRpb24gb2YgdGZqcy1sYXllcnMgZnJvbVxuICAgIC8vIEtlcmFzLlxuICAgIC8vICBEdWUgdG8gdGhlIGltcGVyYXRpdmUgbmF0dXJlIG9mIHRmanMtbGF5ZXJzJyBiYWNrZW5kICh0ZmpzLWNvcmUpLFxuICAgIC8vICB3ZSBkbyBub3QgY29uc3RydWN0IHN5bWJvbGljIGNvbXB1dGF0aW9uIGdyYXBocyB0byBlbWJvZHkgdGhlXG4gICAgLy8gIHRyYWluaW5nIHByb2Nlc3MuIEluc3RlYWQsIHdlIGRlZmluZSBhIGZ1bmN0aW9uIHRoYXQgcGVyZm9ybXMgdGhlXG4gICAgLy8gIHRyYWluaW5nIGFjdGlvbi4gSW4gUHlLZXJhcywgdGhlIGRhdGEgKGlucHV0cyBhbmQgdGFyZ2V0cykgYXJlIGZlZFxuICAgIC8vICB0aHJvdWdoIGdyYXBoIHBsYWNlaG9sZGVycy4gSW4gdGZqcy1sYXllcnMsIHRoZSBkYXRhIGFyZSBmZWQgYXNcbiAgICAvLyAgZnVuY3Rpb24gYXJndW1lbnRzLiBTaW5jZSB0aGUgZnVuY3Rpb24gYXJlIGRlZmluZWQgYmVsb3cgaW4gdGhlXG4gICAgLy8gIHNjb3BlLCB3ZSBkb24ndCBoYXZlIGVxdWl2YWxlbnRzIG9mIFB5S2VyYXMnc1xuICAgIC8vICBgX21ha2VfdHJhaW5fZnVuY2l0b25gLlxuICAgIGNvbnN0IHRyYWluRnVuY3Rpb24gPSBtb2RlbC5tYWtlVHJhaW5GdW5jdGlvbigpO1xuICAgIGNvbnN0IG91dExhYmVscyA9IG1vZGVsLmdldERlZHVwZWRNZXRyaWNzTmFtZXMoKSBhcyBzdHJpbmdbXTtcblxuICAgIGxldCB2YWxGdW5jdGlvbjogKGRhdGE6IFRlbnNvcltdKSA9PiBTY2FsYXJbXTtcbiAgICBsZXQgY2FsbGJhY2tNZXRyaWNzOiBzdHJpbmdbXTtcbiAgICBpZiAoZG9WYWxpZGF0aW9uKSB7XG4gICAgICBtb2RlbC5tYWtlVGVzdEZ1bmN0aW9uKCk7XG4gICAgICB2YWxGdW5jdGlvbiA9IG1vZGVsLnRlc3RGdW5jdGlvbjtcbiAgICAgIGNhbGxiYWNrTWV0cmljcyA9XG4gICAgICAgICAgb3V0TGFiZWxzLnNsaWNlKCkuY29uY2F0KG91dExhYmVscy5tYXAobiA9PiAndmFsXycgKyBuKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHZhbEZ1bmN0aW9uID0gbnVsbDtcbiAgICAgIHZhbElucyA9IFtdO1xuICAgICAgY2FsbGJhY2tNZXRyaWNzID0gb3V0TGFiZWxzLnNsaWNlKCk7XG4gICAgfVxuXG4gICAgY29uc3QgY2FsbGJhY2tzID0gc3RhbmRhcmRpemVDYWxsYmFja3MoYXJncy5jYWxsYmFja3MsIGFyZ3MueWllbGRFdmVyeSk7XG4gICAgY29uc3Qgb3V0ID0gYXdhaXQgZml0TG9vcChcbiAgICAgICAgbW9kZWwsIHRyYWluRnVuY3Rpb24sIGlucywgb3V0TGFiZWxzLCBiYXRjaFNpemUsIGFyZ3MuZXBvY2hzLFxuICAgICAgICBhcmdzLnZlcmJvc2UsIGNhbGxiYWNrcywgdmFsRnVuY3Rpb24sIHZhbElucywgYXJncy5zaHVmZmxlLFxuICAgICAgICBjYWxsYmFja01ldHJpY3MsIGFyZ3MuaW5pdGlhbEVwb2NoLCBudWxsLCBudWxsKTtcbiAgICByZXR1cm4gb3V0O1xuICB9IGZpbmFsbHkge1xuICAgIG1vZGVsLmlzVHJhaW5pbmcgPSBmYWxzZTtcbiAgICAvLyBNZW1vcnkgY2xlYW4gdXAuXG4gICAgZGlzcG9zZU5ld1RlbnNvcnMoaW5wdXRzLCB4KTtcbiAgICBkaXNwb3NlTmV3VGVuc29ycyh0YXJnZXRzLCB5KTtcbiAgICBkaXNwb3NlTmV3VGVuc29ycyhvcmlnaW5hbElucHV0cywgeCk7XG4gICAgZGlzcG9zZU5ld1RlbnNvcnMob3JpZ2luYWxUYXJnZXRzLCB5KTtcbiAgICBkaXNwb3NlTmV3VGVuc29ycyh2YWxYIGFzIFRlbnNvcltdLCBpbnB1dFZhbFgpO1xuICAgIGRpc3Bvc2VOZXdUZW5zb3JzKHZhbFkgYXMgVGVuc29yW10sIGlucHV0VmFsWSk7XG4gICAgaWYgKHNhbXBsZVdlaWdodHMgIT0gbnVsbCkge1xuICAgICAgdGZjLmRpc3Bvc2Uoc2FtcGxlV2VpZ2h0cyk7XG4gICAgfVxuICB9XG4gIC8vIFRPRE8oY2Fpcyk6IEFkZCB2YWx1ZSB0byBvdXRMYWJlbHMuXG59XG5cbi8qKlxuICogRW5zdXJlIHRlbnNvcnMgYWxsIGhhdmUgYSByYW5rIG9mIGF0IGxlYXN0IDIuXG4gKlxuICogSWYgYSB0ZW5zb3IgaGFzIGEgcmFuayBvZiAxLCBpdCBpcyBkaW1lbnNpb24tZXhwYW5kZWQgdG8gcmFuayAyLlxuICogSWYgYW55IHRlbnNvciBoYXMgYSByYW5rIG9mIDAgKGkuZS4sIGlzIGEgc2NhbGFyKSwgYW4gZXJyb3Igd2lsbCBiZSB0aHJvd24uXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBlbnN1cmVUZW5zb3JzUmFuazJPckhpZ2hlcih0ZW5zb3JzOiBUZW5zb3J8VGVuc29yW10pOiBUZW5zb3JbXSB7XG4gIGNvbnN0IG91dHM6IFRlbnNvcltdID0gW107XG4gIGlmICh0ZW5zb3JzIGluc3RhbmNlb2YgVGVuc29yKSB7XG4gICAgdGVuc29ycyA9IFt0ZW5zb3JzXTtcbiAgfVxuXG4gIC8vIE1ha2UgVGVuc29ycyBhdCBsZWFzdCAyRC5cbiAgZm9yIChsZXQgaSA9IDA7IGkgPCB0ZW5zb3JzLmxlbmd0aDsgKytpKSB7XG4gICAgY29uc3QgdGVuc29yID0gdGVuc29yc1tpXTtcbiAgICBpZiAodGVuc29yLnJhbmsgPT09IDEpIHtcbiAgICAgIG91dHMucHVzaChleHBhbmREaW1zKHRlbnNvciwgMSkpO1xuICAgIH0gZWxzZSBpZiAodGVuc29yLnJhbmsgPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnRXhwZWN0ZWQgdGVuc29yIHRvIGJlIGF0IGxlYXN0IDFELCBidXQgcmVjZWl2ZWQgYSAwRCB0ZW5zb3IgJyArXG4gICAgICAgICAgJyhzY2FsYXIpLicpO1xuICAgIH0gZWxzZSB7XG4gICAgICBvdXRzLnB1c2godGVuc29yKTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIG91dHM7XG59XG5cbi8qKlxuICogQ29tcGFyZSBhIHNldCBvZiB0ZW5zb3JzIHdpdGggYSByZWZlcmVuY2UgKG9sZCkgc2V0LCBkaXNjYXJkIHRoZSBvbmVzXG4gKiBpbiB0aGUgbmV3IHNldCB0aGF0IGFyZSBub3QgcHJlc2VudCBpbiB0aGUgcmVmZXJlbmNlIHNldC5cbiAqXG4gKiBUaGlzIG1ldGhvZCBpcyB1c2VkIGZvciBtZW1vcnkgY2xlbmF1cCBkdXJpbmcgY2FsbHMgc3VjaCBhc1xuICogTGF5ZXJzTW9kZWwuZml0KCkuXG4gKlxuICogQHBhcmFtIHRlbnNvcnMgTmV3IHNldCB3aGljaCBtYXkgY29udGFpbiBUZW5zb3JzIG5vdCBwcmVzZW50IGluXG4gKiAgIGByZWZUZW5zb3JzYC5cbiAqIEBwYXJhbSByZWZUZW5zb3JzIFJlZmVyZW5jZSBUZW5zb3Igc2V0LlxuICovXG4vLyBUT0RPKGNhaXMsIGthbmd5aXpoYW5nKTogRGVkdXBsaWNhdGUgd2l0aCB0ZmpzLWRhdGEuXG5leHBvcnQgZnVuY3Rpb24gZGlzcG9zZU5ld1RlbnNvcnMoXG4gICAgdGVuc29yczogVGVuc29yfFRlbnNvcltdfHtbaW5wdXROYW1lOiBzdHJpbmddOiBUZW5zb3J9LFxuICAgIHJlZlRlbnNvcnM6IFRlbnNvcnxUZW5zb3JbXXx7W2lucHV0TmFtZTogc3RyaW5nXTogVGVuc29yfSk6IHZvaWQge1xuICBpZiAodGVuc29ycyA9PSBudWxsKSB7XG4gICAgcmV0dXJuO1xuICB9XG4gIGNvbnN0IG9sZFRlbnNvcklkczogbnVtYmVyW10gPSBbXTtcbiAgaWYgKHJlZlRlbnNvcnMgaW5zdGFuY2VvZiBUZW5zb3IpIHtcbiAgICBvbGRUZW5zb3JJZHMucHVzaChyZWZUZW5zb3JzLmlkKTtcbiAgfSBlbHNlIGlmIChBcnJheS5pc0FycmF5KHJlZlRlbnNvcnMpKSB7XG4gICAgcmVmVGVuc29ycy5mb3JFYWNoKHQgPT4gb2xkVGVuc29ySWRzLnB1c2godC5pZCkpO1xuICB9IGVsc2UgaWYgKHJlZlRlbnNvcnMgIT0gbnVsbCkge1xuICAgIC8vIGBvbGRUZW5zb3JzYCBpcyBhIG1hcCBmcm9tIHN0cmluZyBuYW1lIHRvIFRlbnNvci5cbiAgICBmb3IgKGNvbnN0IG5hbWUgaW4gcmVmVGVuc29ycykge1xuICAgICAgY29uc3Qgb2xkVGVuc29yID0gcmVmVGVuc29yc1tuYW1lXTtcbiAgICAgIG9sZFRlbnNvcklkcy5wdXNoKG9sZFRlbnNvci5pZCk7XG4gICAgfVxuICB9XG5cbiAgY29uc3QgdGVuc29yc1RvRGlzcG9zZTogVGVuc29yW10gPSBbXTtcbiAgaWYgKHRlbnNvcnMgaW5zdGFuY2VvZiBUZW5zb3IpIHtcbiAgICBpZiAob2xkVGVuc29ySWRzLmluZGV4T2YodGVuc29ycy5pZCkgPT09IC0xKSB7XG4gICAgICB0ZW5zb3JzVG9EaXNwb3NlLnB1c2godGVuc29ycyk7XG4gICAgfVxuICB9IGVsc2UgaWYgKEFycmF5LmlzQXJyYXkodGVuc29ycykpIHtcbiAgICB0ZW5zb3JzLmZvckVhY2godCA9PiB7XG4gICAgICBpZiAob2xkVGVuc29ySWRzLmluZGV4T2YodC5pZCkgPT09IC0xKSB7XG4gICAgICAgIHRlbnNvcnNUb0Rpc3Bvc2UucHVzaCh0KTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfSBlbHNlIGlmICh0ZW5zb3JzICE9IG51bGwpIHtcbiAgICAvLyBgb2xkVGVuc29yc2AgaXMgYSBtYXAgZnJvbSBzdHJpbmcgbmFtZSB0byBUZW5zb3IuXG4gICAgZm9yIChjb25zdCBuYW1lIGluIHRlbnNvcnMpIHtcbiAgICAgIGNvbnN0IHRlbnNvciA9IHRlbnNvcnNbbmFtZV07XG4gICAgICBpZiAob2xkVGVuc29ySWRzLmluZGV4T2YodGVuc29yLmlkKSA9PT0gLTEpIHtcbiAgICAgICAgdGVuc29yc1RvRGlzcG9zZS5wdXNoKHRlbnNvcik7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgdGVuc29yc1RvRGlzcG9zZS5mb3JFYWNoKHQgPT4ge1xuICAgIGlmICghdC5pc0Rpc3Bvc2VkKSB7XG4gICAgICB0LmRpc3Bvc2UoKTtcbiAgICB9XG4gIH0pO1xufVxuIl19