/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
import { InputLayer } from './engine/input_layer';
import { Layer } from './engine/topology';
import { input } from './exports';
import { ELU, LeakyReLU, PReLU, ReLU, Softmax, ThresholdedReLU } from './layers/advanced_activations';
import { Conv1D, Conv2D, Conv2DTranspose, Conv3D, Cropping2D, SeparableConv2D, UpSampling2D, Conv3DTranspose } from './layers/convolutional';
import { DepthwiseConv2D } from './layers/convolutional_depthwise';
import { ConvLSTM2D, ConvLSTM2DCell } from './layers/convolutional_recurrent';
import { Activation, Dense, Dropout, Flatten, Masking, Permute, RepeatVector, Reshape, SpatialDropout1D } from './layers/core';
import { Embedding } from './layers/embeddings';
import { Add, Average, Concatenate, Dot, Maximum, Minimum, Multiply } from './layers/merge';
import { AlphaDropout, GaussianDropout, GaussianNoise } from './layers/noise';
import { BatchNormalization, LayerNormalization } from './layers/normalization';
import { ZeroPadding2D } from './layers/padding';
import { AveragePooling1D, AveragePooling2D, AveragePooling3D, GlobalAveragePooling1D, GlobalAveragePooling2D, GlobalMaxPooling1D, GlobalMaxPooling2D, MaxPooling1D, MaxPooling2D, MaxPooling3D } from './layers/pooling';
import { GRU, GRUCell, LSTM, LSTMCell, RNN, RNNCell, SimpleRNN, SimpleRNNCell, StackedRNNCells } from './layers/recurrent';
import { Bidirectional, TimeDistributed } from './layers/wrappers';
import { Rescaling } from './layers/preprocessing/image_preprocessing';
import { Resizing } from './layers/preprocessing/image_resizing';
import { CategoryEncoding } from './layers/preprocessing/category_encoding';
// TODO(cais): Add doc string to all the public static functions in this
//   class; include exectuable JavaScript code snippets where applicable
//   (b/74074458).
// Input Layer.
/**
 * An input layer is an entry point into a `tf.LayersModel`.
 *
 * `InputLayer` is generated automatically for `tf.Sequential` models by
 * specifying the `inputshape` or `batchInputShape` for the first layer.  It
 * should not be specified explicitly. However, it can be useful sometimes,
 * e.g., when constructing a sequential model from a subset of another
 * sequential model's layers. Like the code snippet below shows.
 *
 * ```js
 * // Define a model which simply adds two inputs.
 * const model1 = tf.sequential();
 * model1.add(tf.layers.dense({inputShape: [4], units: 3, activation: 'relu'}));
 * model1.add(tf.layers.dense({units: 1, activation: 'sigmoid'}));
 * model1.summary();
 * model1.predict(tf.zeros([1, 4])).print();
 *
 * // Construct another model, reusing the second layer of `model1` while
 * // not using the first layer of `model1`. Note that you cannot add the second
 * // layer of `model` directly as the first layer of the new sequential model,
 * // because doing so will lead to an error related to the fact that the layer
 * // is not an input layer. Instead, you need to create an `inputLayer` and add
 * // it to the new sequential model before adding the reused layer.
 * const model2 = tf.sequential();
 * // Use an inputShape that matches the input shape of `model1`'s second
 * // layer.
 * model2.add(tf.layers.inputLayer({inputShape: [3]}));
 * model2.add(model1.layers[1]);
 * model2.summary();
 * model2.predict(tf.zeros([1, 3])).print();
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Inputs', namespace: 'layers'}
 */
export function inputLayer(args) {
    return new InputLayer(args);
}
// Advanced Activation Layers.
/**
 * Exponential Linear Unit (ELU).
 *
 * It follows:
 * `f(x) =  alpha * (exp(x) - 1.) for x < 0`,
 * `f(x) = x for x >= 0`.
 *
 * Input shape:
 *   Arbitrary. Use the configuration `inputShape` when using this layer as the
 *   first layer in a model.
 *
 * Output shape:
 *   Same shape as the input.
 *
 * References:
 *   - [Fast and Accurate Deep Network Learning by Exponential Linear Units
 * (ELUs)](https://arxiv.org/abs/1511.07289v1)
 *
 * @doc {
 *   heading: 'Layers',
 *   subheading: 'Advanced Activation',
 *   namespace: 'layers'
 * }
 */
export function elu(args) {
    return new ELU(args);
}
/**
 * Rectified Linear Unit activation function.
 *
 * Input shape:
 *   Arbitrary. Use the config field `inputShape` (Array of integers, does
 *   not include the sample axis) when using this layer as the first layer
 *   in a model.
 *
 * Output shape:
 *   Same shape as the input.
 *
 * @doc {
 *   heading: 'Layers',
 *   subheading: 'Advanced Activation',
 *   namespace: 'layers'
 * }
 */
export function reLU(args) {
    return new ReLU(args);
}
/**
 * Leaky version of a rectified linear unit.
 *
 * It allows a small gradient when the unit is not active:
 * `f(x) = alpha * x for x < 0.`
 * `f(x) = x for x >= 0.`
 *
 * Input shape:
 *   Arbitrary. Use the configuration `inputShape` when using this layer as the
 *   first layer in a model.
 *
 * Output shape:
 *   Same shape as the input.
 *
 * @doc {
 *   heading: 'Layers',
 *   subheading: 'Advanced Activation',
 *   namespace: 'layers'
 * }
 */
export function leakyReLU(args) {
    return new LeakyReLU(args);
}
/**
 * Parameterized version of a leaky rectified linear unit.
 *
 * It follows
 * `f(x) = alpha * x for x < 0.`
 * `f(x) = x for x >= 0.`
 * wherein `alpha` is a trainable weight.
 *
 * Input shape:
 *   Arbitrary. Use the configuration `inputShape` when using this layer as the
 *   first layer in a model.
 *
 * Output shape:
 *   Same shape as the input.
 *
 * @doc {
 *   heading: 'Layers',
 *   subheading: 'Advanced Activation',
 *   namespace: 'layers'
 * }
 */
export function prelu(args) {
    return new PReLU(args);
}
/**
 * Softmax activation layer.
 *
 * Input shape:
 *   Arbitrary. Use the configuration `inputShape` when using this layer as the
 *   first layer in a model.
 *
 * Output shape:
 *   Same shape as the input.
 *
 * @doc {
 *   heading: 'Layers',
 *   subheading: 'Advanced Activation',
 *   namespace: 'layers'
 * }
 */
export function softmax(args) {
    return new Softmax(args);
}
/**
 * Thresholded Rectified Linear Unit.
 *
 * It follows:
 * `f(x) = x for x > theta`,
 * `f(x) = 0 otherwise`.
 *
 * Input shape:
 *   Arbitrary. Use the configuration `inputShape` when using this layer as the
 *   first layer in a model.
 *
 * Output shape:
 *   Same shape as the input.
 *
 * References:
 *   - [Zero-Bias Autoencoders and the Benefits of Co-Adapting
 * Features](http://arxiv.org/abs/1402.3337)
 *
 * @doc {
 *   heading: 'Layers',
 *   subheading: 'Advanced Activation',
 *   namespace: 'layers'
 * }
 */
export function thresholdedReLU(args) {
    return new ThresholdedReLU(args);
}
// Convolutional Layers.
/**
 * 1D convolution layer (e.g., temporal convolution).
 *
 * This layer creates a convolution kernel that is convolved
 * with the layer input over a single spatial (or temporal) dimension
 * to produce a tensor of outputs.
 *
 * If `use_bias` is True, a bias vector is created and added to the outputs.
 *
 * If `activation` is not `null`, it is applied to the outputs as well.
 *
 * When using this layer as the first layer in a model, provide an
 * `inputShape` argument `Array` or `null`.
 *
 * For example, `inputShape` would be:
 * - `[10, 128]` for sequences of 10 vectors of 128-dimensional vectors
 * - `[null, 128]` for variable-length sequences of 128-dimensional vectors.
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional',  namespace: 'layers'}
 */
export function conv1d(args) {
    return new Conv1D(args);
}
/**
 * 2D convolution layer (e.g. spatial convolution over images).
 *
 * This layer creates a convolution kernel that is convolved
 * with the layer input to produce a tensor of outputs.
 *
 * If `useBias` is True, a bias vector is created and added to the outputs.
 *
 * If `activation` is not `null`, it is applied to the outputs as well.
 *
 * When using this layer as the first layer in a model,
 * provide the keyword argument `inputShape`
 * (Array of integers, does not include the sample axis),
 * e.g. `inputShape=[128, 128, 3]` for 128x128 RGB pictures
 * in `dataFormat='channelsLast'`.
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function conv2d(args) {
    return new Conv2D(args);
}
/**
 * Transposed convolutional layer (sometimes called Deconvolution).
 *
 * The need for transposed convolutions generally arises
 * from the desire to use a transformation going in the opposite direction of
 * a normal convolution, i.e., from something that has the shape of the output
 * of some convolution to something that has the shape of its input while
 * maintaining a connectivity pattern that is compatible with said
 * convolution.
 *
 * When using this layer as the first layer in a model, provide the
 * configuration `inputShape` (`Array` of integers, does not include the
 * sample axis), e.g., `inputShape: [128, 128, 3]` for 128x128 RGB pictures in
 * `dataFormat: 'channelsLast'`.
 *
 * Input shape:
 *   4D tensor with shape:
 *   `[batch, channels, rows, cols]` if `dataFormat` is `'channelsFirst'`.
 *   or 4D tensor with shape
 *   `[batch, rows, cols, channels]` if `dataFormat` is `'channelsLast'`.
 *
 * Output shape:
 *   4D tensor with shape:
 *   `[batch, filters, newRows, newCols]` if `dataFormat` is
 * `'channelsFirst'`. or 4D tensor with shape:
 *   `[batch, newRows, newCols, filters]` if `dataFormat` is `'channelsLast'`.
 *
 * References:
 *   - [A guide to convolution arithmetic for deep
 * learning](https://arxiv.org/abs/1603.07285v1)
 *   - [Deconvolutional
 * Networks](http://www.matthewzeiler.com/pubs/cvpr2010/cvpr2010.pdf)
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function conv2dTranspose(args) {
    return new Conv2DTranspose(args);
}
/**
 * 3D convolution layer (e.g. spatial convolution over volumes).
 *
 * This layer creates a convolution kernel that is convolved
 * with the layer input to produce a tensor of outputs.
 *
 * If `useBias` is True, a bias vector is created and added to the outputs.
 *
 * If `activation` is not `null`, it is applied to the outputs as well.
 *
 * When using this layer as the first layer in a model,
 * provide the keyword argument `inputShape`
 * (Array of integers, does not include the sample axis),
 * e.g. `inputShape=[128, 128, 128, 1]` for 128x128x128 grayscale volumes
 * in `dataFormat='channelsLast'`.
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function conv3d(args) {
    return new Conv3D(args);
}
export function conv3dTranspose(args) {
    return new Conv3DTranspose(args);
}
/**
 * Depthwise separable 2D convolution.
 *
 * Separable convolution consists of first performing
 * a depthwise spatial convolution
 * (which acts on each input channel separately)
 * followed by a pointwise convolution which mixes together the resulting
 * output channels. The `depthMultiplier` argument controls how many
 * output channels are generated per input channel in the depthwise step.
 *
 * Intuitively, separable convolutions can be understood as
 * a way to factorize a convolution kernel into two smaller kernels,
 * or as an extreme version of an Inception block.
 *
 * Input shape:
 *   4D tensor with shape:
 *     `[batch, channels, rows, cols]` if data_format='channelsFirst'
 *   or 4D tensor with shape:
 *     `[batch, rows, cols, channels]` if data_format='channelsLast'.
 *
 * Output shape:
 *   4D tensor with shape:
 *     `[batch, filters, newRows, newCols]` if data_format='channelsFirst'
 *   or 4D tensor with shape:
 *     `[batch, newRows, newCols, filters]` if data_format='channelsLast'.
 *     `rows` and `cols` values might have changed due to padding.
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function separableConv2d(args) {
    return new SeparableConv2D(args);
}
/**
 * Cropping layer for 2D input (e.g., image).
 *
 * This layer can crop an input
 * at the top, bottom, left and right side of an image tensor.
 *
 * Input shape:
 *   4D tensor with shape:
 *   - If `dataFormat` is `"channelsLast"`:
 *     `[batch, rows, cols, channels]`
 *   - If `data_format` is `"channels_first"`:
 *     `[batch, channels, rows, cols]`.
 *
 * Output shape:
 *   4D with shape:
 *   - If `dataFormat` is `"channelsLast"`:
 *     `[batch, croppedRows, croppedCols, channels]`
 *    - If `dataFormat` is `"channelsFirst"`:
 *     `[batch, channels, croppedRows, croppedCols]`.
 *
 * Examples
 * ```js
 *
 * const model = tf.sequential();
 * model.add(tf.layers.cropping2D({cropping:[[2, 2], [2, 2]],
 *                                inputShape: [128, 128, 3]}));
 * //now output shape is [batch, 124, 124, 3]
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function cropping2D(args) {
    return new Cropping2D(args);
}
/**
 * Upsampling layer for 2D inputs.
 *
 * Repeats the rows and columns of the data
 * by size[0] and size[1] respectively.
 *
 *
 * Input shape:
 *    4D tensor with shape:
 *     - If `dataFormat` is `"channelsLast"`:
 *         `[batch, rows, cols, channels]`
 *     - If `dataFormat` is `"channelsFirst"`:
 *        `[batch, channels, rows, cols]`
 *
 * Output shape:
 *     4D tensor with shape:
 *     - If `dataFormat` is `"channelsLast"`:
 *        `[batch, upsampledRows, upsampledCols, channels]`
 *     - If `dataFormat` is `"channelsFirst"`:
 *         `[batch, channels, upsampledRows, upsampledCols]`
 *
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function upSampling2d(args) {
    return new UpSampling2D(args);
}
// Convolutional(depthwise) Layers.
/**
 * Depthwise separable 2D convolution.
 *
 * Depthwise Separable convolutions consists in performing just the first step
 * in a depthwise spatial convolution (which acts on each input channel
 * separately). The `depthMultiplier` argument controls how many output channels
 * are generated per input channel in the depthwise step.
 *
 * @doc {heading: 'Layers', subheading: 'Convolutional', namespace: 'layers'}
 */
export function depthwiseConv2d(args) {
    return new DepthwiseConv2D(args);
}
// Basic Layers.
/**
 * Applies an activation function to an output.
 *
 * This layer applies element-wise activation function.  Other layers, notably
 * `dense` can also apply activation functions.  Use this isolated activation
 * function to extract the values before and after the
 * activation. For instance:
 *
 * ```js
 * const input = tf.input({shape: [5]});
 * const denseLayer = tf.layers.dense({units: 1});
 * const activationLayer = tf.layers.activation({activation: 'relu6'});
 *
 * // Obtain the output symbolic tensors by applying the layers in order.
 * const denseOutput = denseLayer.apply(input);
 * const activationOutput = activationLayer.apply(denseOutput);
 *
 * // Create the model based on the inputs.
 * const model = tf.model({
 *     inputs: input,
 *     outputs: [denseOutput, activationOutput]
 * });
 *
 * // Collect both outputs and print separately.
 * const [denseOut, activationOut] = model.predict(tf.randomNormal([6, 5]));
 * denseOut.print();
 * activationOut.print();
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function activation(args) {
    return new Activation(args);
}
/**
 * Creates a dense (fully connected) layer.
 *
 * This layer implements the operation:
 *   `output = activation(dot(input, kernel) + bias)`
 *
 * `activation` is the element-wise activation function
 *   passed as the `activation` argument.
 *
 * `kernel` is a weights matrix created by the layer.
 *
 * `bias` is a bias vector created by the layer (only applicable if `useBias`
 * is `true`).
 *
 * **Input shape:**
 *
 *   nD `tf.Tensor` with shape: `(batchSize, ..., inputDim)`.
 *
 *   The most common situation would be
 *   a 2D input with shape `(batchSize, inputDim)`.
 *
 * **Output shape:**
 *
 *   nD tensor with shape: `(batchSize, ..., units)`.
 *
 *   For instance, for a 2D input with shape `(batchSize, inputDim)`,
 *   the output would have shape `(batchSize, units)`.
 *
 * Note: if the input to the layer has a rank greater than 2, then it is
 * flattened prior to the initial dot product with the kernel.
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function dense(args) {
    return new Dense(args);
}
/**
 * Applies
 * [dropout](http://www.cs.toronto.edu/~rsalakhu/papers/srivastava14a.pdf) to
 * the input.
 *
 * Dropout consists in randomly setting a fraction `rate` of input units to 0 at
 * each update during training time, which helps prevent overfitting.
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function dropout(args) {
    return new Dropout(args);
}
/**
 * Spatial 1D version of Dropout.
 *
 * This Layer type performs the same function as the Dropout layer, but it drops
 * entire 1D feature maps instead of individual elements. For example, if an
 * input example consists of 3 timesteps and the feature map for each timestep
 * has a size of 4, a `spatialDropout1d` layer may zero out the feature maps
 * of the 1st timesteps and 2nd timesteps completely while sparing all feature
 * elements of the 3rd timestep.
 *
 * If adjacent frames (timesteps) are strongly correlated (as is normally the
 * case in early convolution layers), regular dropout will not regularize the
 * activation and will otherwise just result in merely an effective learning
 * rate decrease. In this case, `spatialDropout1d` will help promote
 * independence among feature maps and should be used instead.
 *
 * **Arguments:**
 *   rate: A floating-point number >=0 and <=1. Fraction of the input elements
 *     to drop.
 *
 * **Input shape:**
 *   3D tensor with shape `(samples, timesteps, channels)`.
 *
 * **Output shape:**
 *   Same as the input shape.
 *
 * References:
 *   - [Efficient Object Localization Using Convolutional
 *      Networks](https://arxiv.org/abs/1411.4280)
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function spatialDropout1d(args) {
    return new SpatialDropout1D(args);
}
/**
 * Flattens the input. Does not affect the batch size.
 *
 * A `Flatten` layer flattens each batch in its inputs to 1D (making the output
 * 2D).
 *
 * For example:
 *
 * ```js
 * const input = tf.input({shape: [4, 3]});
 * const flattenLayer = tf.layers.flatten();
 * // Inspect the inferred output shape of the flatten layer, which
 * // equals `[null, 12]`. The 2nd dimension is 4 * 3, i.e., the result of the
 * // flattening. (The 1st dimension is the undermined batch size.)
 * console.log(JSON.stringify(flattenLayer.apply(input).shape));
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function flatten(args) {
    return new Flatten(args);
}
/**
 * Repeats the input n times in a new dimension.
 *
 * ```js
 *  const model = tf.sequential();
 *  model.add(tf.layers.repeatVector({n: 4, inputShape: [2]}));
 *  const x = tf.tensor2d([[10, 20]]);
 *  // Use the model to do inference on a data point the model hasn't see
 *  model.predict(x).print();
 *  // output shape is now [batch, 2, 4]
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function repeatVector(args) {
    return new RepeatVector(args);
}
/**
 * Reshapes an input to a certain shape.
 *
 * ```js
 * const input = tf.input({shape: [4, 3]});
 * const reshapeLayer = tf.layers.reshape({targetShape: [2, 6]});
 * // Inspect the inferred output shape of the Reshape layer, which
 * // equals `[null, 2, 6]`. (The 1st dimension is the undermined batch size.)
 * console.log(JSON.stringify(reshapeLayer.apply(input).shape));
 * ```
 *
 * Input shape:
 *   Arbitrary, although all dimensions in the input shape must be fixed.
 *   Use the configuration `inputShape` when using this layer as the
 *   first layer in a model.
 *
 *
 * Output shape:
 *   [batchSize, targetShape[0], targetShape[1], ...,
 *    targetShape[targetShape.length - 1]].
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function reshape(args) {
    return new Reshape(args);
}
/**
 * Permutes the dimensions of the input according to a given pattern.
 *
 * Useful for, e.g., connecting RNNs and convnets together.
 *
 * Example:
 *
 * ```js
 * const model = tf.sequential();
 * model.add(tf.layers.permute({
 *   dims: [2, 1],
 *   inputShape: [10, 64]
 * }));
 * console.log(model.outputShape);
 * // Now model's output shape is [null, 64, 10], where null is the
 * // unpermuted sample (batch) dimension.
 * ```
 *
 * Input shape:
 *   Arbitrary. Use the configuration field `inputShape` when using this
 *   layer as the first layer in a model.
 *
 * Output shape:
 *   Same rank as the input shape, but with the dimensions re-ordered (i.e.,
 *   permuted) according to the `dims` configuration of this layer.
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function permute(args) {
    return new Permute(args);
}
/**
 * Maps positive integers (indices) into dense vectors of fixed size.
 * E.g. [[4], [20]] -> [[0.25, 0.1], [0.6, -0.2]]
 *
 * **Input shape:** 2D tensor with shape: `[batchSize, sequenceLength]`.
 *
 * **Output shape:** 3D tensor with shape: `[batchSize, sequenceLength,
 * outputDim]`.
 *
 * @doc {heading: 'Layers', subheading: 'Basic', namespace: 'layers'}
 */
export function embedding(args) {
    return new Embedding(args);
}
// Merge Layers.
/**
 * Layer that performs element-wise addition on an `Array` of inputs.
 *
 * It takes as input a list of tensors, all of the same shape, and returns a
 * single tensor (also of the same shape). The inputs are specified as an
 * `Array` when the `apply` method of the `Add` layer instance is called. For
 * example:
 *
 * ```js
 * const input1 = tf.input({shape: [2, 2]});
 * const input2 = tf.input({shape: [2, 2]});
 * const addLayer = tf.layers.add();
 * const sum = addLayer.apply([input1, input2]);
 * console.log(JSON.stringify(sum.shape));
 * // You get [null, 2, 2], with the first dimension as the undetermined batch
 * // dimension.
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function add(args) {
    return new Add(args);
}
/**
 * Layer that performs element-wise averaging on an `Array` of inputs.
 *
 * It takes as input a list of tensors, all of the same shape, and returns a
 * single tensor (also of the same shape). For example:
 *
 * ```js
 * const input1 = tf.input({shape: [2, 2]});
 * const input2 = tf.input({shape: [2, 2]});
 * const averageLayer = tf.layers.average();
 * const average = averageLayer.apply([input1, input2]);
 * console.log(JSON.stringify(average.shape));
 * // You get [null, 2, 2], with the first dimension as the undetermined batch
 * // dimension.
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function average(args) {
    return new Average(args);
}
/**
 * Layer that concatenates an `Array` of inputs.
 *
 * It takes a list of tensors, all of the same shape except for the
 * concatenation axis, and returns a single tensor, the concatenation
 * of all inputs. For example:
 *
 * ```js
 * const input1 = tf.input({shape: [2, 2]});
 * const input2 = tf.input({shape: [2, 3]});
 * const concatLayer = tf.layers.concatenate();
 * const output = concatLayer.apply([input1, input2]);
 * console.log(JSON.stringify(output.shape));
 * // You get [null, 2, 5], with the first dimension as the undetermined batch
 * // dimension. The last dimension (5) is the result of concatenating the
 * // last dimensions of the inputs (2 and 3).
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function concatenate(args) {
    return new Concatenate(args);
}
/**
 * Layer that computes the element-wise maximum of an `Array` of inputs.
 *
 * It takes as input a list of tensors, all of the same shape, and returns a
 * single tensor (also of the same shape). For example:
 *
 * ```js
 * const input1 = tf.input({shape: [2, 2]});
 * const input2 = tf.input({shape: [2, 2]});
 * const maxLayer = tf.layers.maximum();
 * const max = maxLayer.apply([input1, input2]);
 * console.log(JSON.stringify(max.shape));
 * // You get [null, 2, 2], with the first dimension as the undetermined batch
 * // dimension.
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function maximum(args) {
    return new Maximum(args);
}
/**
 * Layer that computes the element-wise minimum of an `Array` of inputs.
 *
 * It takes as input a list of tensors, all of the same shape, and returns a
 * single tensor (also of the same shape). For example:
 *
 * ```js
 * const input1 = tf.input({shape: [2, 2]});
 * const input2 = tf.input({shape: [2, 2]});
 * const minLayer = tf.layers.minimum();
 * const min = minLayer.apply([input1, input2]);
 * console.log(JSON.stringify(min.shape));
 * // You get [null, 2, 2], with the first dimension as the undetermined batch
 * // dimension.
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function minimum(args) {
    return new Minimum(args);
}
/**
 * Layer that multiplies (element-wise) an `Array` of inputs.
 *
 * It takes as input an Array of tensors, all of the same
 * shape, and returns a single tensor (also of the same shape).
 * For example:
 *
 * ```js
 * const input1 = tf.input({shape: [2, 2]});
 * const input2 = tf.input({shape: [2, 2]});
 * const input3 = tf.input({shape: [2, 2]});
 * const multiplyLayer = tf.layers.multiply();
 * const product = multiplyLayer.apply([input1, input2, input3]);
 * console.log(product.shape);
 * // You get [null, 2, 2], with the first dimension as the undetermined batch
 * // dimension.
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function multiply(args) {
    return new Multiply(args);
}
/**
 * Layer that computes a dot product between samples in two tensors.
 *
 * E.g., if applied to a list of two tensors `a` and `b` both of shape
 * `[batchSize, n]`, the output will be a tensor of shape `[batchSize, 1]`,
 * where each entry at index `[i, 0]` will be the dot product between
 * `a[i, :]` and `b[i, :]`.
 *
 * Example:
 *
 * ```js
 * const dotLayer = tf.layers.dot({axes: -1});
 * const x1 = tf.tensor2d([[10, 20], [30, 40]]);
 * const x2 = tf.tensor2d([[-1, -2], [-3, -4]]);
 *
 * // Invoke the layer's apply() method in eager (imperative) mode.
 * const y = dotLayer.apply([x1, x2]);
 * y.print();
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Merge', namespace: 'layers'}
 */
export function dot(args) {
    return new Dot(args);
}
// Normalization Layers.
/**
 * Batch normalization layer (Ioffe and Szegedy, 2014).
 *
 * Normalize the activations of the previous layer at each batch,
 * i.e. applies a transformation that maintains the mean activation
 * close to 0 and the activation standard deviation close to 1.
 *
 * Input shape:
 *   Arbitrary. Use the keyword argument `inputShape` (Array of integers, does
 *   not include the sample axis) when calling the constructor of this class,
 *   if this layer is used as a first layer in a model.
 *
 * Output shape:
 *   Same shape as input.
 *
 * References:
 *   - [Batch Normalization: Accelerating Deep Network Training by Reducing
 * Internal Covariate Shift](https://arxiv.org/abs/1502.03167)
 *
 * @doc {heading: 'Layers', subheading: 'Normalization', namespace: 'layers'}
 */
export function batchNormalization(args) {
    return new BatchNormalization(args);
}
/**
 * Layer-normalization layer (Ba et al., 2016).
 *
 * Normalizes the activations of the previous layer for each given example in a
 * batch independently, instead of across a batch like in `batchNormalization`.
 * In other words, this layer applies a transformation that maintains the mean
 * activation within each example close to 0 and activation variance close to 1.
 *
 * Input shape:
 *   Arbitrary. Use the argument `inputShape` when using this layer as the first
 *   layer in a model.
 *
 * Output shape:
 *   Same as input.
 *
 * References:
 *   - [Layer Normalization](https://arxiv.org/abs/1607.06450)
 *
 * @doc {heading: 'Layers', subheading: 'Normalization', namespace: 'layers'}
 */
export function layerNormalization(args) {
    return new LayerNormalization(args);
}
// Padding Layers.
/**
 * Zero-padding layer for 2D input (e.g., image).
 *
 * This layer can add rows and columns of zeros
 * at the top, bottom, left and right side of an image tensor.
 *
 * Input shape:
 *   4D tensor with shape:
 *   - If `dataFormat` is `"channelsLast"`:
 *     `[batch, rows, cols, channels]`
 *   - If `data_format` is `"channels_first"`:
 *     `[batch, channels, rows, cols]`.
 *
 * Output shape:
 *   4D with shape:
 *   - If `dataFormat` is `"channelsLast"`:
 *     `[batch, paddedRows, paddedCols, channels]`
 *    - If `dataFormat` is `"channelsFirst"`:
 *     `[batch, channels, paddedRows, paddedCols]`.
 *
 * @doc {heading: 'Layers', subheading: 'Padding', namespace: 'layers'}
 */
export function zeroPadding2d(args) {
    return new ZeroPadding2D(args);
}
// Pooling Layers.
/**
 * Average pooling operation for spatial data.
 *
 * Input shape: `[batchSize, inLength, channels]`
 *
 * Output shape: `[batchSize, pooledLength, channels]`
 *
 * `tf.avgPool1d` is an alias.
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function averagePooling1d(args) {
    return new AveragePooling1D(args);
}
export function avgPool1d(args) {
    return averagePooling1d(args);
}
// For backwards compatibility.
// See https://github.com/tensorflow/tfjs/issues/152
export function avgPooling1d(args) {
    return averagePooling1d(args);
}
/**
 * Average pooling operation for spatial data.
 *
 * Input shape:
 *  - If `dataFormat === CHANNEL_LAST`:
 *      4D tensor with shape:
 *      `[batchSize, rows, cols, channels]`
 *  - If `dataFormat === CHANNEL_FIRST`:
 *      4D tensor with shape:
 *      `[batchSize, channels, rows, cols]`
 *
 * Output shape
 *  - If `dataFormat === CHANNEL_LAST`:
 *      4D tensor with shape:
 *      `[batchSize, pooledRows, pooledCols, channels]`
 *  - If `dataFormat === CHANNEL_FIRST`:
 *      4D tensor with shape:
 *      `[batchSize, channels, pooledRows, pooledCols]`
 *
 * `tf.avgPool2d` is an alias.
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function averagePooling2d(args) {
    return new AveragePooling2D(args);
}
export function avgPool2d(args) {
    return averagePooling2d(args);
}
// For backwards compatibility.
// See https://github.com/tensorflow/tfjs/issues/152
export function avgPooling2d(args) {
    return averagePooling2d(args);
}
/**
 * Average pooling operation for 3D data.
 *
 * Input shape
 *   - If `dataFormat === channelsLast`:
 *       5D tensor with shape:
 *       `[batchSize, depths, rows, cols, channels]`
 *   - If `dataFormat === channelsFirst`:
 *      4D tensor with shape:
 *       `[batchSize, channels, depths, rows, cols]`
 *
 * Output shape
 *   - If `dataFormat=channelsLast`:
 *       5D tensor with shape:
 *       `[batchSize, pooledDepths, pooledRows, pooledCols, channels]`
 *   - If `dataFormat=channelsFirst`:
 *       5D tensor with shape:
 *       `[batchSize, channels, pooledDepths, pooledRows, pooledCols]`
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function averagePooling3d(args) {
    return new AveragePooling3D(args);
}
export function avgPool3d(args) {
    return averagePooling3d(args);
}
// For backwards compatibility.
// See https://github.com/tensorflow/tfjs/issues/152
export function avgPooling3d(args) {
    return averagePooling3d(args);
}
/**
 * Global average pooling operation for temporal data.
 *
 * Input Shape: 3D tensor with shape: `[batchSize, steps, features]`.
 *
 * Output Shape: 2D tensor with shape: `[batchSize, features]`.
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function globalAveragePooling1d(args) {
    return new GlobalAveragePooling1D(args);
}
/**
 * Global average pooling operation for spatial data.
 *
 * Input shape:
 *   - If `dataFormat` is `CHANNEL_LAST`:
 *       4D tensor with shape: `[batchSize, rows, cols, channels]`.
 *   - If `dataFormat` is `CHANNEL_FIRST`:
 *       4D tensor with shape: `[batchSize, channels, rows, cols]`.
 *
 * Output shape:
 *   2D tensor with shape: `[batchSize, channels]`.
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function globalAveragePooling2d(args) {
    return new GlobalAveragePooling2D(args);
}
/**
 * Global max pooling operation for temporal data.
 *
 * Input Shape: 3D tensor with shape: `[batchSize, steps, features]`.
 *
 * Output Shape: 2D tensor with shape: `[batchSize, features]`.
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function globalMaxPooling1d(args) {
    return new GlobalMaxPooling1D(args);
}
/**
 * Global max pooling operation for spatial data.
 *
 * Input shape:
 *   - If `dataFormat` is `CHANNEL_LAST`:
 *       4D tensor with shape: `[batchSize, rows, cols, channels]`.
 *   - If `dataFormat` is `CHANNEL_FIRST`:
 *       4D tensor with shape: `[batchSize, channels, rows, cols]`.
 *
 * Output shape:
 *   2D tensor with shape: `[batchSize, channels]`.
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function globalMaxPooling2d(args) {
    return new GlobalMaxPooling2D(args);
}
/**
 * Max pooling operation for temporal data.
 *
 * Input shape:  `[batchSize, inLength, channels]`
 *
 * Output shape: `[batchSize, pooledLength, channels]`
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function maxPooling1d(args) {
    return new MaxPooling1D(args);
}
/**
 * Max pooling operation for spatial data.
 *
 * Input shape
 *   - If `dataFormat === CHANNEL_LAST`:
 *       4D tensor with shape:
 *       `[batchSize, rows, cols, channels]`
 *   - If `dataFormat === CHANNEL_FIRST`:
 *      4D tensor with shape:
 *       `[batchSize, channels, rows, cols]`
 *
 * Output shape
 *   - If `dataFormat=CHANNEL_LAST`:
 *       4D tensor with shape:
 *       `[batchSize, pooledRows, pooledCols, channels]`
 *   - If `dataFormat=CHANNEL_FIRST`:
 *       4D tensor with shape:
 *       `[batchSize, channels, pooledRows, pooledCols]`
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function maxPooling2d(args) {
    return new MaxPooling2D(args);
}
/**
 * Max pooling operation for 3D data.
 *
 * Input shape
 *   - If `dataFormat === channelsLast`:
 *       5D tensor with shape:
 *       `[batchSize, depths, rows, cols, channels]`
 *   - If `dataFormat === channelsFirst`:
 *      5D tensor with shape:
 *       `[batchSize, channels, depths, rows, cols]`
 *
 * Output shape
 *   - If `dataFormat=channelsLast`:
 *       5D tensor with shape:
 *       `[batchSize, pooledDepths, pooledRows, pooledCols, channels]`
 *   - If `dataFormat=channelsFirst`:
 *       5D tensor with shape:
 *       `[batchSize, channels, pooledDepths, pooledRows, pooledCols]`
 *
 * @doc {heading: 'Layers', subheading: 'Pooling', namespace: 'layers'}
 */
export function maxPooling3d(args) {
    return new MaxPooling3D(args);
}
// Recurrent Layers.
/**
 * Gated Recurrent Unit - Cho et al. 2014.
 *
 * This is an `RNN` layer consisting of one `GRUCell`. However, unlike
 * the underlying `GRUCell`, the `apply` method of `SimpleRNN` operates
 * on a sequence of inputs. The shape of the input (not including the first,
 * batch dimension) needs to be at least 2-D, with the first dimension being
 * time steps. For example:
 *
 * ```js
 * const rnn = tf.layers.gru({units: 8, returnSequences: true});
 *
 * // Create an input with 10 time steps.
 * const input = tf.input({shape: [10, 20]});
 * const output = rnn.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10, 8]: 1st dimension is unknown batch size; 2nd dimension is the
 * // same as the sequence length of `input`, due to `returnSequences`: `true`;
 * // 3rd dimension is the `GRUCell`'s number of units.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function gru(args) {
    return new GRU(args);
}
/**
 * Cell class for `GRU`.
 *
 * `GRUCell` is distinct from the `RNN` subclass `GRU` in that its
 * `apply` method takes the input data of only a single time step and returns
 * the cell's output at the time step, while `GRU` takes the input data
 * over a number of time steps. For example:
 *
 * ```js
 * const cell = tf.layers.gruCell({units: 2});
 * const input = tf.input({shape: [10]});
 * const output = cell.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10]: This is the cell's output at a single time step. The 1st
 * // dimension is the unknown batch size.
 * ```
 *
 * Instance(s) of `GRUCell` can be used to construct `RNN` layers. The
 * most typical use of this workflow is to combine a number of cells into a
 * stacked RNN cell (i.e., `StackedRNNCell` internally) and use it to create an
 * RNN. For example:
 *
 * ```js
 * const cells = [
 *   tf.layers.gruCell({units: 4}),
 *   tf.layers.gruCell({units: 8}),
 * ];
 * const rnn = tf.layers.rnn({cell: cells, returnSequences: true});
 *
 * // Create an input with 10 time steps and a length-20 vector at each step.
 * const input = tf.input({shape: [10, 20]});
 * const output = rnn.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10, 8]: 1st dimension is unknown batch size; 2nd dimension is the
 * // same as the sequence length of `input`, due to `returnSequences`: `true`;
 * // 3rd dimension is the last `gruCell`'s number of units.
 * ```
 *
 * To create an `RNN` consisting of only *one* `GRUCell`, use the
 * `tf.layers.gru`.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function gruCell(args) {
    return new GRUCell(args);
}
/**
 * Long-Short Term Memory layer - Hochreiter 1997.
 *
 * This is an `RNN` layer consisting of one `LSTMCell`. However, unlike
 * the underlying `LSTMCell`, the `apply` method of `LSTM` operates
 * on a sequence of inputs. The shape of the input (not including the first,
 * batch dimension) needs to be at least 2-D, with the first dimension being
 * time steps. For example:
 *
 * ```js
 * const lstm = tf.layers.lstm({units: 8, returnSequences: true});
 *
 * // Create an input with 10 time steps.
 * const input = tf.input({shape: [10, 20]});
 * const output = lstm.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10, 8]: 1st dimension is unknown batch size; 2nd dimension is the
 * // same as the sequence length of `input`, due to `returnSequences`: `true`;
 * // 3rd dimension is the `LSTMCell`'s number of units.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function lstm(args) {
    return new LSTM(args);
}
/**
 * Cell class for `LSTM`.
 *
 * `LSTMCell` is distinct from the `RNN` subclass `LSTM` in that its
 * `apply` method takes the input data of only a single time step and returns
 * the cell's output at the time step, while `LSTM` takes the input data
 * over a number of time steps. For example:
 *
 * ```js
 * const cell = tf.layers.lstmCell({units: 2});
 * const input = tf.input({shape: [10]});
 * const output = cell.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10]: This is the cell's output at a single time step. The 1st
 * // dimension is the unknown batch size.
 * ```
 *
 * Instance(s) of `LSTMCell` can be used to construct `RNN` layers. The
 * most typical use of this workflow is to combine a number of cells into a
 * stacked RNN cell (i.e., `StackedRNNCell` internally) and use it to create an
 * RNN. For example:
 *
 * ```js
 * const cells = [
 *   tf.layers.lstmCell({units: 4}),
 *   tf.layers.lstmCell({units: 8}),
 * ];
 * const rnn = tf.layers.rnn({cell: cells, returnSequences: true});
 *
 * // Create an input with 10 time steps and a length-20 vector at each step.
 * const input = tf.input({shape: [10, 20]});
 * const output = rnn.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10, 8]: 1st dimension is unknown batch size; 2nd dimension is the
 * // same as the sequence length of `input`, due to `returnSequences`: `true`;
 * // 3rd dimension is the last `lstmCell`'s number of units.
 * ```
 *
 * To create an `RNN` consisting of only *one* `LSTMCell`, use the
 * `tf.layers.lstm`.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function lstmCell(args) {
    return new LSTMCell(args);
}
/**
 * Fully-connected RNN where the output is to be fed back to input.
 *
 * This is an `RNN` layer consisting of one `SimpleRNNCell`. However, unlike
 * the underlying `SimpleRNNCell`, the `apply` method of `SimpleRNN` operates
 * on a sequence of inputs. The shape of the input (not including the first,
 * batch dimension) needs to be at least 2-D, with the first dimension being
 * time steps. For example:
 *
 * ```js
 * const rnn = tf.layers.simpleRNN({units: 8, returnSequences: true});
 *
 * // Create an input with 10 time steps.
 * const input = tf.input({shape: [10, 20]});
 * const output = rnn.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10, 8]: 1st dimension is unknown batch size; 2nd dimension is the
 * // same as the sequence length of `input`, due to `returnSequences`: `true`;
 * // 3rd dimension is the `SimpleRNNCell`'s number of units.
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function simpleRNN(args) {
    return new SimpleRNN(args);
}
/**
 * Cell class for `SimpleRNN`.
 *
 * `SimpleRNNCell` is distinct from the `RNN` subclass `SimpleRNN` in that its
 * `apply` method takes the input data of only a single time step and returns
 * the cell's output at the time step, while `SimpleRNN` takes the input data
 * over a number of time steps. For example:
 *
 * ```js
 * const cell = tf.layers.simpleRNNCell({units: 2});
 * const input = tf.input({shape: [10]});
 * const output = cell.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10]: This is the cell's output at a single time step. The 1st
 * // dimension is the unknown batch size.
 * ```
 *
 * Instance(s) of `SimpleRNNCell` can be used to construct `RNN` layers. The
 * most typical use of this workflow is to combine a number of cells into a
 * stacked RNN cell (i.e., `StackedRNNCell` internally) and use it to create an
 * RNN. For example:
 *
 * ```js
 * const cells = [
 *   tf.layers.simpleRNNCell({units: 4}),
 *   tf.layers.simpleRNNCell({units: 8}),
 * ];
 * const rnn = tf.layers.rnn({cell: cells, returnSequences: true});
 *
 * // Create an input with 10 time steps and a length-20 vector at each step.
 * const input = tf.input({shape: [10, 20]});
 * const output = rnn.apply(input);
 *
 * console.log(JSON.stringify(output.shape));
 * // [null, 10, 8]: 1st dimension is unknown batch size; 2nd dimension is the
 * // same as the sequence length of `input`, due to `returnSequences`: `true`;
 * // 3rd dimension is the last `SimpleRNNCell`'s number of units.
 * ```
 *
 * To create an `RNN` consisting of only *one* `SimpleRNNCell`, use the
 * `tf.layers.simpleRNN`.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function simpleRNNCell(args) {
    return new SimpleRNNCell(args);
}
/**
 * Convolutional LSTM layer - Xingjian Shi 2015.
 *
 * This is a `ConvRNN2D` layer consisting of one `ConvLSTM2DCell`. However,
 * unlike the underlying `ConvLSTM2DCell`, the `apply` method of `ConvLSTM2D`
 * operates on a sequence of inputs. The shape of the input (not including the
 * first, batch dimension) needs to be 4-D, with the first dimension being time
 * steps. For example:
 *
 * ```js
 * const filters = 3;
 * const kernelSize = 3;
 *
 * const batchSize = 4;
 * const sequenceLength = 2;
 * const size = 5;
 * const channels = 3;
 *
 * const inputShape = [batchSize, sequenceLength, size, size, channels];
 * const input = tf.ones(inputShape);
 *
 * const layer = tf.layers.convLstm2d({filters, kernelSize});
 *
 * const output = layer.apply(input);
 * ```
 */
/** @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'} */
export function convLstm2d(args) {
    return new ConvLSTM2D(args);
}
/**
 * Cell class for `ConvLSTM2D`.
 *
 * `ConvLSTM2DCell` is distinct from the `ConvRNN2D` subclass `ConvLSTM2D` in
 * that its `call` method takes the input data of only a single time step and
 * returns the cell's output at the time step, while `ConvLSTM2D` takes the
 * input data over a number of time steps. For example:
 *
 * ```js
 * const filters = 3;
 * const kernelSize = 3;
 *
 * const sequenceLength = 1;
 * const size = 5;
 * const channels = 3;
 *
 * const inputShape = [sequenceLength, size, size, channels];
 * const input = tf.ones(inputShape);
 *
 * const cell = tf.layers.convLstm2dCell({filters, kernelSize});
 *
 * cell.build(input.shape);
 *
 * const outputSize = size - kernelSize + 1;
 * const outShape = [sequenceLength, outputSize, outputSize, filters];
 *
 * const initialH = tf.zeros(outShape);
 * const initialC = tf.zeros(outShape);
 *
 * const [o, h, c] = cell.call([input, initialH, initialC], {});
 * ```
 */
/** @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'} */
export function convLstm2dCell(args) {
    return new ConvLSTM2DCell(args);
}
/**
 * Base class for recurrent layers.
 *
 * Input shape:
 *   3D tensor with shape `[batchSize, timeSteps, inputDim]`.
 *
 * Output shape:
 *   - if `returnState`, an Array of tensors (i.e., `tf.Tensor`s). The first
 *     tensor is the output. The remaining tensors are the states at the
 *     last time step, each with shape `[batchSize, units]`.
 *   - if `returnSequences`, the output will have shape
 *     `[batchSize, timeSteps, units]`.
 *   - else, the output will have shape `[batchSize, units]`.
 *
 * Masking:
 *   This layer supports masking for input data with a variable number
 *   of timesteps. To introduce masks to your data,
 *   use an embedding layer with the `mask_zero` parameter
 *   set to `True`.
 *
 * Notes on using statefulness in RNNs:
 *   You can set RNN layers to be 'stateful', which means that the states
 *   computed for the samples in one batch will be reused as initial states
 *   for the samples in the next batch. This assumes a one-to-one mapping
 *   between samples in different successive batches.
 *
 *   To enable statefulness:
 *     - specify `stateful: true` in the layer constructor.
 *     - specify a fixed batch size for your model, by passing
 *       if sequential model:
 *         `batchInputShape=[...]` to the first layer in your model.
 *       else for functional model with 1 or more Input layers:
 *         `batchShape=[...]` to all the first layers in your model.
 *       This is the expected shape of your inputs *including the batch size*.
 *       It should be a tuple of integers, e.g. `(32, 10, 100)`.
 *     - specify `shuffle=False` when calling fit().
 *
 *   To reset the states of your model, call `.resetStates()` on either
 *   a specific layer, or on your entire model.
 *
 * Note on specifying the initial state of RNNs
 *   You can specify the initial state of RNN layers symbolically by
 *   calling them with the option `initialState`. The value of
 *   `initialState` should be a tensor or list of tensors representing
 *   the initial state of the RNN layer.
 *
 *   You can specify the initial state of RNN layers numerically by
 *   calling `resetStates` with the keyword argument `states`. The value of
 *   `states` should be a numpy array or list of numpy arrays representing
 *   the initial state of the RNN layer.
 *
 * Note on passing external constants to RNNs
 *   You can pass "external" constants to the cell using the `constants`
 *   keyword argument of `RNN.call` method. This requires that the `cell.call`
 *   method accepts the same keyword argument `constants`. Such constants
 *   can be used to condition the cell transformation on additional static
 *   inputs (not changing over time), a.k.a. an attention mechanism.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function rnn(args) {
    return new RNN(args);
}
/**
 * Wrapper allowing a stack of RNN cells to behave as a single cell.
 *
 * Used to implement efficient stacked RNNs.
 *
 * @doc {heading: 'Layers', subheading: 'Recurrent', namespace: 'layers'}
 */
export function stackedRNNCells(args) {
    return new StackedRNNCells(args);
}
// Wrapper Layers.
/** @doc {heading: 'Layers', subheading: 'Wrapper', namespace: 'layers'} */
export function bidirectional(args) {
    return new Bidirectional(args);
}
/**
 * This wrapper applies a layer to every temporal slice of an input.
 *
 * The input should be at least 3D,  and the dimension of the index `1` will be
 * considered to be the temporal dimension.
 *
 * Consider a batch of 32 samples, where each sample is a sequence of 10 vectors
 * of 16 dimensions. The batch input shape of the layer is then `[32,  10,
 * 16]`, and the `inputShape`, not including the sample dimension, is
 * `[10, 16]`.
 *
 * You can then use `TimeDistributed` to apply a `Dense` layer to each of the 10
 * timesteps, independently:
 *
 * ```js
 * const model = tf.sequential();
 * model.add(tf.layers.timeDistributed({
 *   layer: tf.layers.dense({units: 8}),
 *   inputShape: [10, 16],
 * }));
 *
 * // Now model.outputShape = [null, 10, 8].
 * // The output will then have shape `[32, 10, 8]`.
 *
 * // In subsequent layers, there is no need for `inputShape`:
 * model.add(tf.layers.timeDistributed({layer: tf.layers.dense({units: 32})}));
 * console.log(JSON.stringify(model.outputs[0].shape));
 * // Now model.outputShape = [null, 10, 32].
 * ```
 *
 * The output will then have shape `[32, 10, 32]`.
 *
 * `TimeDistributed` can be used with arbitrary layers, not just `Dense`, for
 * instance a `Conv2D` layer.
 *
 * ```js
 * const model = tf.sequential();
 * model.add(tf.layers.timeDistributed({
 *   layer: tf.layers.conv2d({filters: 64, kernelSize: [3, 3]}),
 *   inputShape: [10, 299, 299, 3],
 * }));
 * console.log(JSON.stringify(model.outputs[0].shape));
 * ```
 *
 * @doc {heading: 'Layers', subheading: 'Wrapper', namespace: 'layers'}
 */
export function timeDistributed(args) {
    return new TimeDistributed(args);
}
// Aliases for pooling.
export const globalMaxPool1d = globalMaxPooling1d;
export const globalMaxPool2d = globalMaxPooling2d;
export const maxPool1d = maxPooling1d;
export const maxPool2d = maxPooling2d;
export { Layer, RNN, RNNCell, input /* alias for tf.input */ };
/**
 * Apply additive zero-centered Gaussian noise.
 *
 * As it is a regularization layer, it is only active at training time.
 *
 * This is useful to mitigate overfitting
 * (you could see it as a form of random data augmentation).
 * Gaussian Noise (GS) is a natural choice as corruption process
 * for real valued inputs.
 *
 * # Arguments
 * stddev: float, standard deviation of the noise distribution.
 *
 * # Input shape
 * Arbitrary. Use the keyword argument `input_shape`
 * (tuple of integers, does not include the samples axis)
 * when using this layer as the first layer in a model.
 *
 * # Output shape
 * Same shape as input.
 *
 * @doc {heading: 'Layers', subheading: 'Noise', namespace: 'layers'}
 */
export function gaussianNoise(args) {
    return new GaussianNoise(args);
}
/**
 * Apply multiplicative 1-centered Gaussian noise.
 *
 * As it is a regularization layer, it is only active at training time.
 *
 * Arguments:
 *   - `rate`: float, drop probability (as with `Dropout`).
 *     The multiplicative noise will have
 *     standard deviation `sqrt(rate / (1 - rate))`.
 *
 * Input shape:
 *   Arbitrary. Use the keyword argument `inputShape`
 *   (tuple of integers, does not include the samples axis)
 *   when using this layer as the first layer in a model.
 *
 * Output shape:
 *   Same shape as input.
 *
 * References:
 *   - [Dropout: A Simple Way to Prevent Neural Networks from Overfitting](
 *      http://www.cs.toronto.edu/~rsalakhu/papers/srivastava14a.pdf)
 *
 * @doc {heading: 'Layers', subheading: 'Noise', namespace: 'layers'}
 */
export function gaussianDropout(args) {
    return new GaussianDropout(args);
}
/**
 * Applies Alpha Dropout to the input.
 *
 * As it is a regularization layer, it is only active at training time.
 *
 * Alpha Dropout is a `Dropout` that keeps mean and variance of inputs
 * to their original values, in order to ensure the self-normalizing property
 * even after this dropout.
 * Alpha Dropout fits well to Scaled Exponential Linear Units
 * by randomly setting activations to the negative saturation value.
 *
 * Arguments:
 *   - `rate`: float, drop probability (as with `Dropout`).
 *     The multiplicative noise will have
 *     standard deviation `sqrt(rate / (1 - rate))`.
 *   - `noise_shape`: A 1-D `Tensor` of type `int32`, representing the
 *     shape for randomly generated keep/drop flags.
 *
 * Input shape:
 *   Arbitrary. Use the keyword argument `inputShape`
 *   (tuple of integers, does not include the samples axis)
 *   when using this layer as the first layer in a model.
 *
 * Output shape:
 *   Same shape as input.
 *
 * References:
 *   - [Self-Normalizing Neural Networks](https://arxiv.org/abs/1706.02515)
 *
 * @doc {heading: 'Layers', subheading: 'Noise', namespace: 'layers'}
 */
export function alphaDropout(args) {
    return new AlphaDropout(args);
}
/**
 * Masks a sequence by using a mask value to skip timesteps.
 *
 * If all features for a given sample timestep are equal to `mask_value`,
 * then the sample timestep will be masked (skipped) in all downstream layers
 * (as long as they support masking).
 *
 * If any downstream layer does not support masking yet receives such
 * an input mask, an exception will be raised.
 *
 * Arguments:
 *   - `maskValue`: Either None or mask value to skip.
 *
 * Input shape:
 *   Arbitrary. Use the keyword argument `inputShape`
 *   (tuple of integers, does not include the samples axis)
 *   when using this layer as the first layer in a model.
 *
 * Output shape:
 *   Same shape as input.
 *
 * @doc {heading: 'Layers', subheading: 'Mask', namespace: 'layers'}
 */
export function masking(args) {
    return new Masking(args);
}
/**
 * A preprocessing layer which rescales input values to a new range.
 *
 * This layer rescales every value of an input (often an image) by multiplying
 * by `scale` and adding `offset`.
 *
 * For instance:
 * 1. To rescale an input in the ``[0, 255]`` range
 * to be in the `[0, 1]` range, you would pass `scale=1/255`.
 * 2. To rescale an input in the ``[0, 255]`` range to be in the `[-1, 1]`
 * range, you would pass `scale=1./127.5, offset=-1`.
 * The rescaling is applied both during training and inference. Inputs can be
 * of integer or floating point dtype, and by default the layer will output
 * floats.
 *
 * Arguments:
 *   - `scale`: Float, the scale to apply to the inputs.
 *   - `offset`: Float, the offset to apply to the inputs.
 *
 * Input shape:
 *   Arbitrary.
 *
 * Output shape:
 *   Same as input.
 *
 * @doc {heading: 'Layers', subheading: 'Rescaling', namespace: 'layers'}
 */
export function rescaling(args) {
    return new Rescaling(args);
}
/**
 * A preprocessing layer which resizes images.
 * This layer resizes an image input to a target height and width. The input
 * should be a 4D (batched) or 3D (unbatched) tensor in `"channels_last"`
 * format.  Input pixel values can be of any range (e.g. `[0., 1.)` or `[0,
 * 255]`) and of interger or floating point dtype. By default, the layer will
 * output floats.
 *
 * Arguments:
 *   - `height`: number, the height for the output tensor.
 *   - `width`: number, the width for the output tensor.
 *   - `interpolation`: string, the method for image resizing interpolation.
 *   - `cropToAspectRatio`: boolean, whether to keep image aspect ratio.
 *
 * Input shape:
 *   Arbitrary.
 *
 * Output shape:
 *   height, width, num channels.
 *
 * @doc {heading: 'Layers', subheading: 'Resizing', namespace: 'layers'}
 */
export function resizing(args) {
    return new Resizing(args);
}
/**
 * A preprocessing layer which encodes integer features.
 *
 * This layer provides options for condensing data into a categorical encoding
 * when the total number of tokens are known in advance. It accepts integer
 * values as inputs, and it outputs a dense representation of those
 * inputs.
 *
 * Arguments:
 *
 * numTokens: The total number of tokens the layer should support. All
 *  inputs to the layer must integers in the range `0 <= value <
 *  numTokens`, or an error will be thrown.
 *
 * outputMode: Specification for the output of the layer.
 *  Defaults to `multiHot`. Values can be `oneHot`, `multiHot` or
 *  `count`, configuring the layer as follows:
 *
 *    oneHot: Encodes each individual element in the input into an
 *      array of `numTokens` size, containing a 1 at the element index. If
 *      the last dimension is size 1, will encode on that dimension. If the
 *      last dimension is not size 1, will append a new dimension for the
 *      encoded output.
 *
 *    multiHot: Encodes each sample in the input into a single array
 *     of `numTokens` size, containing a 1 for each vocabulary term
 *     present in the sample. Treats the last dimension as the sample
 *     dimension, if input shape is `(..., sampleLength)`, output shape
 *     will be `(..., numTokens)`.
 *
 *    count: Like `multiHot`, but the int array contains a count of
 *     the number of times the token at that index appeared in the sample.
 *
 *  For all output modes, currently only output up to rank 2 is supported.
 *   Call arguments:
 *    inputs: A 1D or 2D tensor of integer inputs.
 *    countWeights: A tensor in the same shape as `inputs` indicating the
 *    weight for each sample value when summing up in `count` mode. Not used
 *    in `multiHot` or `oneHot` modes.
 *
 *
 * @doc {heading: 'Layers', subheading: 'CategoryEncoding', namespace: 'layers'}
 */
export function categoryEncoding(args) {
    return new CategoryEncoding(args);
}
//# sourceMappingURL=data:application/json;base64,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