/**
 * @license
 * Copyright 2020 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import * as tfc from '@tensorflow/tfjs-core';
import { util } from '@tensorflow/tfjs-core';
import * as K from '../backend/tfjs_backend';
import { checkDataFormat, checkPaddingMode } from '../common';
import { InputSpec } from '../engine/topology';
import { AttributeError, NotImplementedError, ValueError } from '../errors';
import { Initializer } from '../initializers';
import { convOutputLength, normalizeArray } from '../utils/conv_utils';
import { assertPositiveInteger } from '../utils/generic_utils';
import { getExactlyOneShape } from '../utils/types_utils';
import { generateDropoutMask, LSTMCell, RNN, RNNCell } from './recurrent';
class ConvRNN2DCell extends RNNCell {
}
/**
 * Base class for convolutional-recurrent layers.
 */
class ConvRNN2D extends RNN {
    constructor(args) {
        if (args.unroll) {
            throw new NotImplementedError('Unrolling is not possible with convolutional RNNs.');
        }
        if (Array.isArray(args.cell)) {
            throw new NotImplementedError('It is not possible at the moment to stack convolutional cells.');
        }
        super(args);
        this.inputSpec = [new InputSpec({ ndim: 5 })];
    }
    call(inputs, kwargs) {
        return tfc.tidy(() => {
            if (this.cell.dropoutMask != null) {
                tfc.dispose(this.cell.dropoutMask);
                this.cell.dropoutMask = null;
            }
            if (this.cell.recurrentDropoutMask != null) {
                tfc.dispose(this.cell.recurrentDropoutMask);
                this.cell.recurrentDropoutMask = null;
            }
            if (kwargs && kwargs['constants']) {
                throw new ValueError('ConvRNN2D cell does not support constants');
            }
            const mask = kwargs == null ? null : kwargs['mask'];
            const training = kwargs == null ? null : kwargs['training'];
            const initialState = kwargs == null ? null : kwargs['initialState'];
            return super.call(inputs, { mask, training, initialState });
        });
    }
    computeOutputShape(inputShape) {
        let outShape = this.computeSingleOutputShape(inputShape);
        if (!this.returnSequences) {
            outShape = [outShape[0], ...outShape.slice(2)];
        }
        if (this.returnState) {
            outShape =
                [outShape, ...Array(2).fill([inputShape[0], ...outShape.slice(-3)])];
        }
        return outShape;
    }
    getInitialState(inputs) {
        return tfc.tidy(() => {
            const { stateSize } = this.cell;
            const inputShape = inputs.shape;
            const outputShape = this.computeSingleOutputShape(inputShape);
            const stateShape = [outputShape[0], ...outputShape.slice(2)];
            const initialState = tfc.zeros(stateShape);
            if (Array.isArray(stateSize)) {
                return Array(stateSize.length).fill(initialState);
            }
            return [initialState];
        });
    }
    resetStates(states, training = false) {
        tfc.tidy(() => {
            if (!this.stateful) {
                throw new AttributeError('Cannot call resetStates() on an RNN Layer that is not stateful.');
            }
            const inputShape = this.inputSpec[0].shape;
            const outputShape = this.computeSingleOutputShape(inputShape);
            const stateShape = [outputShape[0], ...outputShape.slice(2)];
            const batchSize = inputShape[0];
            if (batchSize == null) {
                throw new ValueError('If an RNN is stateful, it needs to know its batch size. Specify ' +
                    'the batch size of your input tensors: \n' +
                    '- If using a Sequential model, specify the batch size by ' +
                    'passing a `batchInputShape` option to your first layer.\n' +
                    '- If using the functional API, specify the batch size by ' +
                    'passing a `batchShape` option to your Input layer.');
            }
            // Initialize state if null.
            if (this.getStates() == null) {
                if (Array.isArray(this.cell.stateSize)) {
                    this.states_ = this.cell.stateSize.map(() => tfc.zeros(stateShape));
                }
                else {
                    this.states_ = [tfc.zeros(stateShape)];
                }
            }
            else if (states == null) {
                // Dispose old state tensors.
                tfc.dispose(this.states_);
                // For stateful RNNs, fully dispose kept old states.
                if (this.keptStates != null) {
                    tfc.dispose(this.keptStates);
                    this.keptStates = [];
                }
                if (Array.isArray(this.cell.stateSize)) {
                    this.states_ = this.cell.stateSize.map(() => tfc.zeros(stateShape));
                }
                else {
                    this.states_[0] = tfc.zeros(stateShape);
                }
            }
            else {
                if (!Array.isArray(states)) {
                    states = [states];
                }
                if (states.length !== this.states_.length) {
                    throw new ValueError(`Layer ${this.name} expects ${this.states_.length} state(s), ` +
                        `but it received ${states.length} state value(s). Input ` +
                        `received: ${states}`);
                }
                if (training) {
                    // Store old state tensors for complete disposal later, i.e., during
                    // the next no-arg call to this method. We do not dispose the old
                    // states immediately because that BPTT (among other things) require
                    // them.
                    this.keptStates.push(this.states_.slice());
                }
                else {
                    tfc.dispose(this.states_);
                }
                for (let index = 0; index < this.states_.length; ++index) {
                    const value = states[index];
                    const expectedShape = stateShape;
                    if (!util.arraysEqual(value.shape, expectedShape)) {
                        throw new ValueError(`State ${index} is incompatible with layer ${this.name}: ` +
                            `expected shape=${expectedShape}, received shape=${value.shape}`);
                    }
                    this.states_[index] = value;
                }
            }
            this.states_ = this.states_.map(state => tfc.keep(state.clone()));
        });
    }
    computeSingleOutputShape(inputShape) {
        const { dataFormat, filters, kernelSize, padding, strides, dilationRate } = this.cell;
        const isChannelsFirst = dataFormat === 'channelsFirst';
        const h = inputShape[isChannelsFirst ? 3 : 2];
        const w = inputShape[isChannelsFirst ? 4 : 3];
        const hOut = convOutputLength(h, kernelSize[0], padding, strides[0], dilationRate[0]);
        const wOut = convOutputLength(w, kernelSize[1], padding, strides[1], dilationRate[1]);
        const outShape = [
            ...inputShape.slice(0, 2),
            ...(isChannelsFirst ? [filters, hOut, wOut] : [hOut, wOut, filters])
        ];
        return outShape;
    }
}
/** @nocollapse */
ConvRNN2D.className = 'ConvRNN2D';
export class ConvLSTM2DCell extends LSTMCell {
    constructor(args) {
        const { filters, kernelSize, strides, padding, dataFormat, dilationRate, } = args;
        super(Object.assign(Object.assign({}, args), { units: filters }));
        this.filters = filters;
        assertPositiveInteger(this.filters, 'filters');
        this.kernelSize = normalizeArray(kernelSize, 2, 'kernelSize');
        this.kernelSize.forEach(size => assertPositiveInteger(size, 'kernelSize'));
        this.strides = normalizeArray(strides || 1, 2, 'strides');
        this.strides.forEach(stride => assertPositiveInteger(stride, 'strides'));
        this.padding = padding || 'valid';
        checkPaddingMode(this.padding);
        this.dataFormat = dataFormat || 'channelsLast';
        checkDataFormat(this.dataFormat);
        this.dilationRate = normalizeArray(dilationRate || 1, 2, 'dilationRate');
        this.dilationRate.forEach(rate => assertPositiveInteger(rate, 'dilationRate'));
    }
    build(inputShape) {
        var _a;
        inputShape = getExactlyOneShape(inputShape);
        const channelAxis = this.dataFormat === 'channelsFirst' ? 1 : inputShape.length - 1;
        if (inputShape[channelAxis] == null) {
            throw new ValueError(`The channel dimension of the input should be defined. ` +
                `Found ${inputShape[channelAxis]}`);
        }
        const inputDim = inputShape[channelAxis];
        const numOfKernels = 4;
        const kernelShape = this.kernelSize.concat([inputDim, this.filters * numOfKernels]);
        this.kernel = this.addWeight('kernel', kernelShape, null, this.kernelInitializer, this.kernelRegularizer, true, this.kernelConstraint);
        const recurrentKernelShape = this.kernelSize.concat([this.filters, this.filters * numOfKernels]);
        this.recurrentKernel = this.addWeight('recurrent_kernel', recurrentKernelShape, null, this.recurrentInitializer, this.recurrentRegularizer, true, this.recurrentConstraint);
        if (this.useBias) {
            let biasInitializer;
            if (this.unitForgetBias) {
                const init = this.biasInitializer;
                const filters = this.filters;
                biasInitializer = new (_a = class CustomInit extends Initializer {
                        apply(shape, dtype) {
                            const biasI = init.apply([filters]);
                            const biasF = tfc.ones([filters]);
                            const biasCAndO = init.apply([filters * 2]);
                            return K.concatenate([biasI, biasF, biasCAndO]);
                        }
                    },
                    /** @nocollapse */
                    _a.className = 'CustomInit',
                    _a)();
            }
            else {
                biasInitializer = this.biasInitializer;
            }
            this.bias = this.addWeight('bias', [this.filters * numOfKernels], null, biasInitializer, this.biasRegularizer, true, this.biasConstraint);
        }
        this.built = true;
    }
    call(inputs, kwargs) {
        return tfc.tidy(() => {
            if (inputs.length !== 3) {
                throw new ValueError(`ConvLSTM2DCell expects 3 input Tensors (inputs, h, c), got ` +
                    `${inputs.length}.`);
            }
            const training = kwargs['training'] || false;
            const x = inputs[0]; // Current input
            const hTMinus1 = inputs[1]; // Previous memory state.
            const cTMinus1 = inputs[2]; // Previous carry state.
            const numOfKernels = 4;
            if (0 < this.dropout && this.dropout < 1 && this.dropoutMask == null) {
                this.dropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(x),
                    rate: this.dropout,
                    training,
                    count: numOfKernels,
                    dropoutFunc: this.dropoutFunc
                });
            }
            const dropoutMask = this.dropoutMask;
            const applyDropout = (x, mask, index) => {
                if (!mask || !mask[index]) {
                    return x;
                }
                return tfc.mul(mask[index], x);
            };
            let xI = applyDropout(x, dropoutMask, 0);
            let xF = applyDropout(x, dropoutMask, 1);
            let xC = applyDropout(x, dropoutMask, 2);
            let xO = applyDropout(x, dropoutMask, 3);
            if (0 < this.recurrentDropout && this.recurrentDropout < 1 &&
                this.recurrentDropoutMask == null) {
                this.recurrentDropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(hTMinus1),
                    rate: this.recurrentDropout,
                    training,
                    count: numOfKernels,
                    dropoutFunc: this.dropoutFunc
                });
            }
            const recDropoutMask = this.recurrentDropoutMask;
            let hI = applyDropout(hTMinus1, recDropoutMask, 0);
            let hF = applyDropout(hTMinus1, recDropoutMask, 1);
            let hC = applyDropout(hTMinus1, recDropoutMask, 2);
            let hO = applyDropout(hTMinus1, recDropoutMask, 3);
            const kernelChannelAxis = 3;
            const [kernelI, kernelF, kernelC, kernelO] = tfc.split(this.kernel.read(), numOfKernels, kernelChannelAxis);
            const [biasI, biasF, biasC, biasO] = this.useBias ?
                tfc.split(this.bias.read(), numOfKernels) :
                [null, null, null, null];
            xI = this.inputConv(xI, kernelI, biasI, this.padding);
            xF = this.inputConv(xF, kernelF, biasF, this.padding);
            xC = this.inputConv(xC, kernelC, biasC, this.padding);
            xO = this.inputConv(xO, kernelO, biasO, this.padding);
            const [recKernelI, recKernelF, recKernelC, recKernelO] = tfc.split(this.recurrentKernel.read(), numOfKernels, kernelChannelAxis);
            hI = this.recurrentConv(hI, recKernelI);
            hF = this.recurrentConv(hF, recKernelF);
            hC = this.recurrentConv(hC, recKernelC);
            hO = this.recurrentConv(hO, recKernelO);
            const i = this.recurrentActivation.apply(tfc.add(xI, hI));
            const f = this.recurrentActivation.apply(tfc.add(xF, hF));
            const c = tfc.add(tfc.mul(f, cTMinus1), tfc.mul(i, this.activation.apply(tfc.add(xC, hC))));
            const h = tfc.mul(this.recurrentActivation.apply(tfc.add(xO, hO)), this.activation.apply(c));
            return [h, h, c];
        });
    }
    getConfig() {
        const _a = super.getConfig(), { 'units': _ } = _a, baseConfig = __rest(_a, ['units']);
        const config = {
            filters: this.filters,
            kernelSize: this.kernelSize,
            padding: this.padding,
            dataFormat: this.dataFormat,
            dilationRate: this.dilationRate,
            strides: this.strides,
        };
        return Object.assign(Object.assign({}, baseConfig), config);
    }
    inputConv(x, w, b, padding) {
        const out = tfc.conv2d(x, w, this.strides, (padding || 'valid'), this.dataFormat === 'channelsFirst' ? 'NCHW' : 'NHWC', this.dilationRate);
        if (b) {
            return K.biasAdd(out, b, this.dataFormat);
        }
        return out;
    }
    recurrentConv(x, w) {
        const strides = 1;
        return tfc.conv2d(x, w, strides, 'same', this.dataFormat === 'channelsFirst' ? 'NCHW' : 'NHWC');
    }
}
/** @nocollapse */
ConvLSTM2DCell.className = 'ConvLSTM2DCell';
tfc.serialization.registerClass(ConvLSTM2DCell);
export class ConvLSTM2D extends ConvRNN2D {
    constructor(args) {
        const cell = new ConvLSTM2DCell(args);
        super(Object.assign(Object.assign({}, args), { cell }));
    }
    /** @nocollapse */
    static fromConfig(cls, config) {
        return new cls(config);
    }
}
/** @nocollapse */
ConvLSTM2D.className = 'ConvLSTM2D';
tfc.serialization.registerClass(ConvLSTM2D);
//# sourceMappingURL=data:application/json;base64,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