/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/**
 * TensorFlow.js Layers: Basic Layers.
 */
import { any, cast, mul, notEqual, reshape, serialization, tidy, transpose, util } from '@tensorflow/tfjs-core';
import { getActivation, serializeActivation } from '../activations';
import * as K from '../backend/tfjs_backend';
import { getConstraint, serializeConstraint } from '../constraints';
import { InputSpec, Layer } from '../engine/topology';
import { ValueError } from '../errors';
import { getInitializer, serializeInitializer } from '../initializers';
import { getRegularizer, serializeRegularizer } from '../regularizers';
import { assertPositiveInteger, mapActivationToFusedKernel } from '../utils/generic_utils';
import { arrayProd, range } from '../utils/math_utils';
import { getExactlyOneShape, getExactlyOneTensor } from '../utils/types_utils';
export class Dropout extends Layer {
    constructor(args) {
        super(args);
        this.rate = Math.max(Math.min(args.rate, 1), 0);
        // So that the scalar doesn't get tidied up between executions.
        this.noiseShape = args.noiseShape;
        this.seed = args.seed;
        this.supportsMasking = true;
    }
    getNoiseShape(input) {
        if (this.noiseShape == null) {
            return this.noiseShape;
        }
        const inputShape = input.shape;
        const noiseShape = [];
        for (let i = 0; i < this.noiseShape.length; ++i) {
            noiseShape.push(this.noiseShape[i] == null ? inputShape[i] : this.noiseShape[i]);
        }
        return noiseShape;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            this.invokeCallHook(inputs, kwargs);
            const input = getExactlyOneTensor(inputs);
            if (0 < this.rate && this.rate < 1) {
                const training = kwargs['training'] == null ? false : kwargs['training'];
                const noiseShape = this.getNoiseShape(input);
                const output = K.inTrainPhase(() => K.dropout(input, this.rate, noiseShape, this.seed), () => input, training);
                return output;
            }
            return inputs;
        });
    }
    getConfig() {
        const config = {
            rate: this.rate,
            noiseShape: this.noiseShape,
            seed: this.seed,
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
    dispose() {
        return super.dispose();
    }
}
/** @nocollapse */
Dropout.className = 'Dropout';
serialization.registerClass(Dropout);
export class SpatialDropout1D extends Dropout {
    constructor(args) {
        super(args);
        this.inputSpec = [{ ndim: 3 }];
    }
    getNoiseShape(input) {
        const inputShape = input.shape;
        return [inputShape[0], 1, inputShape[2]];
    }
}
/** @nocollapse */
SpatialDropout1D.className = 'SpatialDropout1D';
serialization.registerClass(SpatialDropout1D);
export class Dense extends Layer {
    constructor(args) {
        super(args);
        // Default activation: Linear (none).
        this.activation = null;
        this.useBias = true;
        this.kernel = null;
        this.bias = null;
        this.DEFAULT_KERNEL_INITIALIZER = 'glorotNormal';
        this.DEFAULT_BIAS_INITIALIZER = 'zeros';
        if (args.batchInputShape == null && args.inputShape == null &&
            args.inputDim != null) {
            // This logic is copied from Layer's constructor, since we can't
            // do exactly what the Python constructor does for Dense().
            let batchSize = null;
            if (args.batchSize != null) {
                batchSize = args.batchSize;
            }
            this.batchInputShape = [batchSize, args.inputDim];
        }
        this.units = args.units;
        assertPositiveInteger(this.units, 'units');
        this.activation = getActivation(args.activation);
        if (args.useBias != null) {
            this.useBias = args.useBias;
        }
        this.kernelInitializer = getInitializer(args.kernelInitializer || this.DEFAULT_KERNEL_INITIALIZER);
        this.biasInitializer =
            getInitializer(args.biasInitializer || this.DEFAULT_BIAS_INITIALIZER);
        this.kernelConstraint = getConstraint(args.kernelConstraint);
        this.biasConstraint = getConstraint(args.biasConstraint);
        this.kernelRegularizer = getRegularizer(args.kernelRegularizer);
        this.biasRegularizer = getRegularizer(args.biasRegularizer);
        this.activityRegularizer = getRegularizer(args.activityRegularizer);
        this.supportsMasking = true;
        this.inputSpec = [{ minNDim: 2 }];
    }
    build(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        const inputLastDim = inputShape[inputShape.length - 1];
        if (this.kernel == null) {
            this.kernel = this.addWeight('kernel', [inputLastDim, this.units], null, this.kernelInitializer, this.kernelRegularizer, true, this.kernelConstraint);
            if (this.useBias) {
                this.bias = this.addWeight('bias', [this.units], null, this.biasInitializer, this.biasRegularizer, true, this.biasConstraint);
            }
        }
        this.inputSpec = [{ minNDim: 2, axes: { [-1]: inputLastDim } }];
        this.built = true;
    }
    computeOutputShape(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        const outputShape = inputShape.slice();
        outputShape[outputShape.length - 1] = this.units;
        return outputShape;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            this.invokeCallHook(inputs, kwargs);
            // Dense layer accepts only a single input.
            const input = getExactlyOneTensor(inputs);
            const fusedActivationName = mapActivationToFusedKernel(this.activation.getClassName());
            let output;
            if (fusedActivationName != null) {
                output = K.dot(input, this.kernel.read(), fusedActivationName, this.bias ? this.bias.read() : null);
            }
            else {
                output = K.dot(input, this.kernel.read());
                if (this.bias != null) {
                    output = K.biasAdd(output, this.bias.read());
                }
                if (this.activation != null) {
                    output = this.activation.apply(output);
                }
            }
            return output;
        });
    }
    getConfig() {
        const config = {
            units: this.units,
            activation: serializeActivation(this.activation),
            useBias: this.useBias,
            kernelInitializer: serializeInitializer(this.kernelInitializer),
            biasInitializer: serializeInitializer(this.biasInitializer),
            kernelRegularizer: serializeRegularizer(this.kernelRegularizer),
            biasRegularizer: serializeRegularizer(this.biasRegularizer),
            activityRegularizer: serializeRegularizer(this.activityRegularizer),
            kernelConstraint: serializeConstraint(this.kernelConstraint),
            biasConstraint: serializeConstraint(this.biasConstraint)
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
}
/** @nocollapse */
Dense.className = 'Dense';
serialization.registerClass(Dense);
export class Flatten extends Layer {
    constructor(args) {
        args = args || {};
        super(args);
        this.inputSpec = [{ minNDim: 3 }];
        this.dataFormat = args.dataFormat;
    }
    computeOutputShape(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        for (const dim of inputShape.slice(1)) {
            if (dim == null) {
                throw new ValueError(`The shape of the input to "Flatten" is not fully defined ` +
                    `(got ${inputShape.slice(1)}). Make sure to pass a complete ` +
                    `"input_shape" or "batch_input_shape" argument to the first ` +
                    `layer in your model.`);
            }
        }
        return [inputShape[0], arrayProd(inputShape, 1)];
    }
    call(inputs, kwargs) {
        return tidy(() => {
            this.invokeCallHook(inputs, kwargs);
            let input = getExactlyOneTensor(inputs);
            if (this.dataFormat === 'channelsFirst' && input.rank > 1) {
                const permutation = [0];
                for (let i = 2; i < input.rank; ++i) {
                    permutation.push(i);
                }
                permutation.push(1);
                input = transpose(input, permutation);
            }
            return K.batchFlatten(input);
        });
    }
    getConfig() {
        const config = {};
        if (this.dataFormat != null) {
            config['dataFormat'] = this.dataFormat;
        }
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
}
/** @nocollapse */
Flatten.className = 'Flatten';
serialization.registerClass(Flatten);
export class Activation extends Layer {
    constructor(args) {
        super(args);
        this.supportsMasking = true;
        this.activation = getActivation(args.activation);
    }
    call(inputs, kwargs) {
        return tidy(() => {
            this.invokeCallHook(inputs, kwargs);
            const input = getExactlyOneTensor(inputs);
            return this.activation.apply(input);
        });
    }
    getConfig() {
        const config = { activation: serializeActivation(this.activation) };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
}
/** @nocollapse */
Activation.className = 'Activation';
serialization.registerClass(Activation);
export class RepeatVector extends Layer {
    constructor(args) {
        super(args);
        this.n = args.n;
        this.inputSpec = [{ ndim: 2 }];
    }
    computeOutputShape(inputShape) {
        return [inputShape[0], this.n, inputShape[1]];
    }
    call(inputs, kwargs) {
        return tidy(() => {
            inputs = getExactlyOneTensor(inputs);
            return K.repeat(inputs, this.n);
        });
    }
    getConfig() {
        const config = {
            n: this.n,
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
}
/** @nocollapse */
RepeatVector.className = 'RepeatVector';
serialization.registerClass(RepeatVector);
export class Reshape extends Layer {
    constructor(args) {
        super(args);
        this.targetShape = args.targetShape;
        // Make sure that all unknown dimensions are represented as `null`.
        for (let i = 0; i < this.targetShape.length; ++i) {
            if (this.isUnknown(this.targetShape[i])) {
                this.targetShape[i] = null;
            }
        }
    }
    isUnknown(dim) {
        return dim < 0 || dim == null;
    }
    /**
     * Finds and replaces a missing dimension in output shape.
     *
     * This is a near direct port of the internal Numpy function
     * `_fix_unknown_dimension` in `numpy/core/src/multiarray/shape.c`.
     *
     * @param inputShape: Original shape of array begin reshape.
     * @param outputShape: Target shape of the array, with at most a single
     * `null` or negative number, which indicates an underdetermined dimension
     * that should be derived from `inputShape` and the known dimensions of
     *   `outputShape`.
     * @returns: The output shape with `null` replaced with its computed value.
     * @throws: ValueError: If `inputShape` and `outputShape` do not match.
     */
    fixUnknownDimension(inputShape, outputShape) {
        const errorMsg = 'Total size of new array must be unchanged.';
        const finalShape = outputShape.slice();
        let known = 1;
        let unknown = null;
        for (let i = 0; i < finalShape.length; ++i) {
            const dim = finalShape[i];
            if (this.isUnknown(dim)) {
                if (unknown === null) {
                    unknown = i;
                }
                else {
                    throw new ValueError('Can only specifiy one unknown dimension.');
                }
            }
            else {
                known *= dim;
            }
        }
        const originalSize = arrayProd(inputShape);
        if (unknown !== null) {
            if (known === 0 || originalSize % known !== 0) {
                throw new ValueError(errorMsg);
            }
            finalShape[unknown] = originalSize / known;
        }
        else if (originalSize !== known) {
            throw new ValueError(errorMsg);
        }
        return finalShape;
    }
    computeOutputShape(inputShape) {
        let anyUnknownDims = false;
        for (let i = 0; i < inputShape.length; ++i) {
            if (this.isUnknown(inputShape[i])) {
                anyUnknownDims = true;
                break;
            }
        }
        if (anyUnknownDims) {
            return inputShape.slice(0, 1).concat(this.targetShape);
        }
        else {
            return inputShape.slice(0, 1).concat(this.fixUnknownDimension(inputShape.slice(1), this.targetShape));
        }
    }
    call(inputs, kwargs) {
        return tidy(() => {
            this.invokeCallHook(inputs, kwargs);
            const input = getExactlyOneTensor(inputs);
            const inputShape = input.shape;
            const outputShape = inputShape.slice(0, 1).concat(this.fixUnknownDimension(inputShape.slice(1), this.targetShape));
            return reshape(input, outputShape);
        });
    }
    getConfig() {
        const config = {
            targetShape: this.targetShape,
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
}
/** @nocollapse */
Reshape.className = 'Reshape';
serialization.registerClass(Reshape);
export class Permute extends Layer {
    constructor(args) {
        super(args);
        if (args.dims == null) {
            throw new Error('Required configuration field `dims` is missing during Permute ' +
                'constructor call.');
        }
        if (!Array.isArray(args.dims)) {
            throw new Error('Permute constructor requires `dims` to be an Array, but received ' +
                `${args.dims} instead.`);
        }
        // Check the validity of the permutation indices.
        const expectedSortedIndices = range(1, args.dims.length + 1);
        if (!util.arraysEqual(args.dims.slice().sort(), expectedSortedIndices)) {
            throw new Error('Invalid permutation `dims`: ' + JSON.stringify(args.dims) +
                ' `dims` must contain consecutive integers starting from 1.');
        }
        this.dims = args.dims;
        this.dimsIncludingBatch = [0].concat(this.dims);
        this.inputSpec = [new InputSpec({ ndim: this.dims.length + 1 })];
    }
    computeOutputShape(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        const outputShape = inputShape.slice();
        this.dims.forEach((dim, i) => {
            outputShape[i + 1] = inputShape[dim];
        });
        return outputShape;
    }
    call(inputs, kwargs) {
        return transpose(getExactlyOneTensor(inputs), this.dimsIncludingBatch);
    }
    getConfig() {
        const config = {
            dims: this.dims,
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
}
/** @nocollapse */
Permute.className = 'Permute';
serialization.registerClass(Permute);
export class Masking extends Layer {
    constructor(args) {
        super(args == null ? {} : args);
        this.supportsMasking = true;
        if (args != null) {
            this.maskValue = args.maskValue == null ? 0 : args.maskValue;
        }
        else {
            this.maskValue = 0;
        }
    }
    computeOutputShape(inputShape) {
        return inputShape;
    }
    getConfig() {
        const baseConfig = super.getConfig();
        const config = { maskValue: this.maskValue };
        Object.assign(config, baseConfig);
        return config;
    }
    computeMask(inputs, mask) {
        const input = getExactlyOneTensor(inputs);
        const axis = -1;
        return any(notEqual(input, this.maskValue), axis);
    }
    call(inputs, kwargs) {
        return tidy(() => {
            this.invokeCallHook(inputs, kwargs);
            const input = getExactlyOneTensor(inputs);
            const axis = -1;
            const keepDims = true;
            const booleanMask = any(notEqual(input, this.maskValue), axis, keepDims);
            const output = mul(input, cast(booleanMask, input.dtype));
            return output;
        });
    }
}
/** @nocollapse */
Masking.className = 'Masking';
serialization.registerClass(Masking);
//# sourceMappingURL=data:application/json;base64,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