/**
 * @license
 * Copyright 2022 CodeSmith LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
import { Layer } from '../../engine/topology';
import { serialization, mul, add, tidy } from '@tensorflow/tfjs-core';
import { getExactlyOneTensor } from '../../utils/types_utils';
import * as K from '../../backend/tfjs_backend';
/**
 * Preprocessing Rescaling Layer
 *
 * This rescales images by a scaling and offset factor
 */
export class Rescaling extends Layer {
    constructor(args) {
        super(args);
        this.scale = args.scale;
        if (args.offset) {
            this.offset = args.offset;
        }
        else {
            this.offset = 0;
        }
    }
    getConfig() {
        const config = {
            'scale': this.scale,
            'offset': this.offset
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            inputs = getExactlyOneTensor(inputs);
            if (inputs.dtype !== 'float32') {
                inputs = K.cast(inputs, 'float32');
            }
            return add(mul(inputs, this.scale), this.offset);
        });
    }
}
/** @nocollapse */
Rescaling.className = 'Rescaling';
serialization.registerClass(Rescaling);
//# sourceMappingURL=data:application/json;base64,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