/**
 * @license
 * Copyright 2022 CodeSmith LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
import { image, serialization, tidy } from '@tensorflow/tfjs-core'; // mul, add
import { Layer } from '../../engine/topology';
import { ValueError } from '../../errors';
import { getExactlyOneShape } from '../../utils/types_utils'; //, getExactlyOneTensor
// tf methods unimplemented in tfjs: 'bicubic', 'area', 'lanczos3', 'lanczos5',
//                                   'gaussian', 'mitchellcubic'
const INTERPOLATION_KEYS = ['bilinear', 'nearest'];
const INTERPOLATION_METHODS = new Set(INTERPOLATION_KEYS);
/**
 * Preprocessing Resizing Layer
 *
 * This resizes images by a scaling and offset factor
 */
export class Resizing extends Layer {
    constructor(args) {
        super(args);
        this.height = args.height;
        this.width = args.width;
        if (args.interpolation) {
            if (INTERPOLATION_METHODS.has(args.interpolation)) {
                this.interpolation = args.interpolation;
            }
            else {
                throw new ValueError(`Invalid interpolation parameter: ${args.interpolation} is not implemented`);
            }
        }
        else {
            this.interpolation = 'bilinear';
        }
        this.cropToAspectRatio = Boolean(args.cropToAspectRatio);
    }
    computeOutputShape(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        const numChannels = inputShape[2];
        return [this.height, this.width, numChannels];
    }
    getConfig() {
        const config = {
            'height': this.height,
            'width': this.width,
            'interpolation': this.interpolation,
            'cropToAspectRatio': this.cropToAspectRatio
        };
        const baseConfig = super.getConfig();
        Object.assign(config, baseConfig);
        return config;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            const size = [this.height, this.width];
            if (this.interpolation === 'bilinear') {
                return image.resizeBilinear(inputs, size, !this.cropToAspectRatio);
            }
            else if (this.interpolation === 'nearest') {
                return image.resizeNearestNeighbor(inputs, size, !this.cropToAspectRatio);
            }
            else {
                throw new Error(`Interpolation is ${this.interpolation} but only ${[...INTERPOLATION_METHODS]} are supported`);
            }
        });
    }
}
/** @nocollapse */
Resizing.className = 'Resizing';
serialization.registerClass(Resizing);
//# sourceMappingURL=data:application/json;base64,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