/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/**
 * TensorFlow.js Layers: Recurrent Neural Network Layers.
 */
import * as tfc from '@tensorflow/tfjs-core';
import { serialization, tidy, util } from '@tensorflow/tfjs-core';
import { getActivation, serializeActivation } from '../activations';
import * as K from '../backend/tfjs_backend';
import { nameScope } from '../common';
import { getConstraint, serializeConstraint } from '../constraints';
import { InputSpec, SymbolicTensor } from '../engine/topology';
import { Layer } from '../engine/topology';
import { AttributeError, NotImplementedError, ValueError } from '../errors';
import { getInitializer, Initializer, Ones, serializeInitializer } from '../initializers';
import { getRegularizer, serializeRegularizer } from '../regularizers';
import { assertPositiveInteger } from '../utils/generic_utils';
import * as math_utils from '../utils/math_utils';
import { getExactlyOneShape, getExactlyOneTensor, isArrayOfShapes } from '../utils/types_utils';
import { batchGetValue, batchSetValue } from '../variables';
import { deserialize } from './serialization';
/**
 * Standardize `apply()` args to a single list of tensor inputs.
 *
 * When running a model loaded from file, the input tensors `initialState` and
 * `constants` are passed to `RNN.apply()` as part of `inputs` instead of the
 * dedicated kwargs fields. `inputs` consists of
 * `[inputs, initialState0, initialState1, ..., constant0, constant1]` in this
 * case.
 * This method makes sure that arguments are
 * separated and that `initialState` and `constants` are `Array`s of tensors
 * (or None).
 *
 * @param inputs Tensor or `Array` of  tensors.
 * @param initialState Tensor or `Array` of tensors or `null`/`undefined`.
 * @param constants Tensor or `Array` of tensors or `null`/`undefined`.
 * @returns An object consisting of
 *   inputs: A tensor.
 *   initialState: `Array` of tensors or `null`.
 *   constants: `Array` of tensors or `null`.
 * @throws ValueError, if `inputs` is an `Array` but either `initialState` or
 *   `constants` is provided.
 */
export function standardizeArgs(inputs, initialState, constants, numConstants) {
    if (Array.isArray(inputs)) {
        if (initialState != null || constants != null) {
            throw new ValueError('When inputs is an array, neither initialState or constants ' +
                'should be provided');
        }
        if (numConstants != null) {
            constants = inputs.slice(inputs.length - numConstants, inputs.length);
            inputs = inputs.slice(0, inputs.length - numConstants);
        }
        if (inputs.length > 1) {
            initialState = inputs.slice(1, inputs.length);
        }
        inputs = inputs[0];
    }
    function toListOrNull(x) {
        if (x == null || Array.isArray(x)) {
            return x;
        }
        else {
            return [x];
        }
    }
    initialState = toListOrNull(initialState);
    constants = toListOrNull(constants);
    return { inputs, initialState, constants };
}
/**
 * Iterates over the time dimension of a tensor.
 *
 * @param stepFunction RNN step function.
 *   Parameters:
 *     inputs: tensor with shape `[samples, ...]` (no time dimension),
 *       representing input for the batch of samples at a certain time step.
 *     states: an Array of tensors.
 *   Returns:
 *     outputs: tensor with shape `[samples, outputDim]` (no time dimension).
 *     newStates: list of tensors, same length and shapes as `states`. The first
 *       state in the list must be the output tensor at the previous timestep.
 * @param inputs Tensor of temporal data of shape `[samples, time, ...]` (at
 *   least 3D).
 * @param initialStates Tensor with shape `[samples, outputDim]` (no time
 *   dimension), containing the initial values of the states used in the step
 *   function.
 * @param goBackwards If `true`, do the iteration over the time dimension in
 *   reverse order and return the reversed sequence.
 * @param mask Binary tensor with shape `[sample, time, 1]`, with a zero for
 *   every element that is masked.
 * @param constants An Array of constant values passed at each step.
 * @param unroll Whether to unroll the RNN or to use a symbolic loop. *Not*
 *   applicable to this imperative deeplearn.js backend. Its value is ignored.
 * @param needPerStepOutputs Whether the per-step outputs are to be
 *   concatenated into a single tensor and returned (as the second return
 *   value). Default: `false`. This arg is included so that the relatively
 *   expensive concatenation of the stepwise outputs can be omitted unless
 *   the stepwise outputs need to be kept (e.g., for an LSTM layer of which
 *   `returnSequence` is `true`.)
 * @returns An Array: `[lastOutput, outputs, newStates]`.
 *   lastOutput: the lastest output of the RNN, of shape `[samples, ...]`.
 *   outputs: tensor with shape `[samples, time, ...]` where each entry
 *     `output[s, t]` is the output of the step function at time `t` for sample
 *     `s`. This return value is provided if and only if the
 *     `needPerStepOutputs` is set as `true`. If it is set as `false`, this
 *     return value will be `undefined`.
 *   newStates: Array of tensors, latest states returned by the step function,
 *      of shape `(samples, ...)`.
 * @throws ValueError If input dimension is less than 3.
 *
 * TODO(nielsene): This needs to be tidy-ed.
 */
export function rnn(stepFunction, inputs, initialStates, goBackwards = false, mask, constants, unroll = false, needPerStepOutputs = false) {
    return tfc.tidy(() => {
        const ndim = inputs.shape.length;
        if (ndim < 3) {
            throw new ValueError(`Input should be at least 3D, but is ${ndim}D.`);
        }
        // Transpose to time-major, i.e., from [batch, time, ...] to [time, batch,
        // ...].
        const axes = [1, 0].concat(math_utils.range(2, ndim));
        inputs = tfc.transpose(inputs, axes);
        if (constants != null) {
            throw new NotImplementedError('The rnn() functoin of the deeplearn.js backend does not support ' +
                'constants yet.');
        }
        // Porting Note: the unroll option is ignored by the imperative backend.
        if (unroll) {
            console.warn('Backend rnn(): the unroll = true option is not applicable to the ' +
                'imperative deeplearn.js backend.');
        }
        if (mask != null) {
            mask = tfc.cast(tfc.cast(mask, 'bool'), 'float32');
            if (mask.rank === ndim - 1) {
                mask = tfc.expandDims(mask, -1);
            }
            mask = tfc.transpose(mask, axes);
        }
        if (goBackwards) {
            inputs = tfc.reverse(inputs, 0);
            if (mask != null) {
                mask = tfc.reverse(mask, 0);
            }
        }
        // Porting Note: PyKeras with TensorFlow backend uses a symbolic loop
        //   (tf.while_loop). But for the imperative deeplearn.js backend, we just
        //   use the usual TypeScript control flow to iterate over the time steps in
        //   the inputs.
        // Porting Note: PyKeras patches a "_use_learning_phase" attribute to
        // outputs.
        //   This is not idiomatic in TypeScript. The info regarding whether we are
        //   in a learning (i.e., training) phase for RNN is passed in a different
        //   way.
        const perStepOutputs = [];
        let lastOutput;
        let states = initialStates;
        const timeSteps = inputs.shape[0];
        const perStepInputs = tfc.unstack(inputs);
        let perStepMasks;
        if (mask != null) {
            perStepMasks = tfc.unstack(mask);
        }
        for (let t = 0; t < timeSteps; ++t) {
            const currentInput = perStepInputs[t];
            const stepOutputs = tfc.tidy(() => stepFunction(currentInput, states));
            if (mask == null) {
                lastOutput = stepOutputs[0];
                states = stepOutputs[1];
            }
            else {
                const maskedOutputs = tfc.tidy(() => {
                    const stepMask = perStepMasks[t];
                    const negStepMask = tfc.sub(tfc.onesLike(stepMask), stepMask);
                    // TODO(cais): Would tfc.where() be better for performance?
                    const output = tfc.add(tfc.mul(stepOutputs[0], stepMask), tfc.mul(states[0], negStepMask));
                    const newStates = states.map((state, i) => {
                        return tfc.add(tfc.mul(stepOutputs[1][i], stepMask), tfc.mul(state, negStepMask));
                    });
                    return { output, newStates };
                });
                lastOutput = maskedOutputs.output;
                states = maskedOutputs.newStates;
            }
            if (needPerStepOutputs) {
                perStepOutputs.push(lastOutput);
            }
        }
        let outputs;
        if (needPerStepOutputs) {
            const axis = 1;
            outputs = tfc.stack(perStepOutputs, axis);
        }
        return [lastOutput, outputs, states];
    });
}
export class RNN extends Layer {
    constructor(args) {
        super(args);
        let cell;
        if (args.cell == null) {
            throw new ValueError('cell property is missing for the constructor of RNN.');
        }
        else if (Array.isArray(args.cell)) {
            cell = new StackedRNNCells({ cells: args.cell });
        }
        else {
            cell = args.cell;
        }
        if (cell.stateSize == null) {
            throw new ValueError('The RNN cell should have an attribute `stateSize` (tuple of ' +
                'integers, one integer per RNN state).');
        }
        this.cell = cell;
        this.returnSequences =
            args.returnSequences == null ? false : args.returnSequences;
        this.returnState = args.returnState == null ? false : args.returnState;
        this.goBackwards = args.goBackwards == null ? false : args.goBackwards;
        this._stateful = args.stateful == null ? false : args.stateful;
        this.unroll = args.unroll == null ? false : args.unroll;
        this.supportsMasking = true;
        this.inputSpec = [new InputSpec({ ndim: 3 })];
        this.stateSpec = null;
        this.states_ = null;
        // TODO(cais): Add constantsSpec and numConstants.
        this.numConstants = null;
        // TODO(cais): Look into the use of initial_state in the kwargs of the
        //   constructor.
        this.keptStates = [];
    }
    // Porting Note: This is the equivalent of `RNN.states` property getter in
    //   PyKeras.
    getStates() {
        if (this.states_ == null) {
            const numStates = Array.isArray(this.cell.stateSize) ? this.cell.stateSize.length : 1;
            return math_utils.range(0, numStates).map(x => null);
        }
        else {
            return this.states_;
        }
    }
    // Porting Note: This is the equivalent of the `RNN.states` property setter in
    //   PyKeras.
    setStates(states) {
        this.states_ = states;
    }
    computeOutputShape(inputShape) {
        if (isArrayOfShapes(inputShape)) {
            inputShape = inputShape[0];
        }
        inputShape = inputShape;
        // TODO(cais): Remove the casting once stacked RNN cells become supported.
        let stateSize = this.cell.stateSize;
        if (!Array.isArray(stateSize)) {
            stateSize = [stateSize];
        }
        const outputDim = stateSize[0];
        let outputShape;
        if (this.returnSequences) {
            outputShape = [inputShape[0], inputShape[1], outputDim];
        }
        else {
            outputShape = [inputShape[0], outputDim];
        }
        if (this.returnState) {
            const stateShape = [];
            for (const dim of stateSize) {
                stateShape.push([inputShape[0], dim]);
            }
            return [outputShape].concat(stateShape);
        }
        else {
            return outputShape;
        }
    }
    computeMask(inputs, mask) {
        return tfc.tidy(() => {
            if (Array.isArray(mask)) {
                mask = mask[0];
            }
            const outputMask = this.returnSequences ? mask : null;
            if (this.returnState) {
                const stateMask = this.states.map(s => null);
                return [outputMask].concat(stateMask);
            }
            else {
                return outputMask;
            }
        });
    }
    /**
     * Get the current state tensors of the RNN.
     *
     * If the state hasn't been set, return an array of `null`s of the correct
     * length.
     */
    get states() {
        if (this.states_ == null) {
            const numStates = Array.isArray(this.cell.stateSize) ? this.cell.stateSize.length : 1;
            const output = [];
            for (let i = 0; i < numStates; ++i) {
                output.push(null);
            }
            return output;
        }
        else {
            return this.states_;
        }
    }
    set states(s) {
        this.states_ = s;
    }
    build(inputShape) {
        // Note inputShape will be an Array of Shapes of initial states and
        // constants if these are passed in apply().
        const constantShape = null;
        if (this.numConstants != null) {
            throw new NotImplementedError('Constants support is not implemented in RNN yet.');
        }
        if (isArrayOfShapes(inputShape)) {
            inputShape = inputShape[0];
        }
        inputShape = inputShape;
        const batchSize = this.stateful ? inputShape[0] : null;
        const inputDim = inputShape.slice(2);
        this.inputSpec[0] = new InputSpec({ shape: [batchSize, null, ...inputDim] });
        // Allow cell (if RNNCell Layer) to build before we set or validate
        // stateSpec.
        const stepInputShape = [inputShape[0]].concat(inputShape.slice(2));
        if (constantShape != null) {
            throw new NotImplementedError('Constants support is not implemented in RNN yet.');
        }
        else {
            this.cell.build(stepInputShape);
        }
        // Set or validate stateSpec.
        let stateSize;
        if (Array.isArray(this.cell.stateSize)) {
            stateSize = this.cell.stateSize;
        }
        else {
            stateSize = [this.cell.stateSize];
        }
        if (this.stateSpec != null) {
            if (!util.arraysEqual(this.stateSpec.map(spec => spec.shape[spec.shape.length - 1]), stateSize)) {
                throw new ValueError(`An initialState was passed that is not compatible with ` +
                    `cell.stateSize. Received stateSpec=${this.stateSpec}; ` +
                    `However cell.stateSize is ${this.cell.stateSize}`);
            }
        }
        else {
            this.stateSpec =
                stateSize.map(dim => new InputSpec({ shape: [null, dim] }));
        }
        if (this.stateful) {
            this.resetStates();
        }
    }
    /**
     * Reset the state tensors of the RNN.
     *
     * If the `states` argument is `undefined` or `null`, will set the
     * state tensor(s) of the RNN to all-zero tensors of the appropriate
     * shape(s).
     *
     * If `states` is provided, will set the state tensors of the RNN to its
     * value.
     *
     * @param states Optional externally-provided initial states.
     * @param training Whether this call is done during training. For stateful
     *   RNNs, this affects whether the old states are kept or discarded. In
     *   particular, if `training` is `true`, the old states will be kept so
     *   that subsequent backpropgataion through time (BPTT) may work properly.
     *   Else, the old states will be discarded.
     */
    resetStates(states, training = false) {
        tidy(() => {
            if (!this.stateful) {
                throw new AttributeError('Cannot call resetStates() on an RNN Layer that is not stateful.');
            }
            const batchSize = this.inputSpec[0].shape[0];
            if (batchSize == null) {
                throw new ValueError('If an RNN is stateful, it needs to know its batch size. Specify ' +
                    'the batch size of your input tensors: \n' +
                    '- If using a Sequential model, specify the batch size by ' +
                    'passing a `batchInputShape` option to your first layer.\n' +
                    '- If using the functional API, specify the batch size by ' +
                    'passing a `batchShape` option to your Input layer.');
            }
            // Initialize state if null.
            if (this.states_ == null) {
                if (Array.isArray(this.cell.stateSize)) {
                    this.states_ =
                        this.cell.stateSize.map(dim => tfc.zeros([batchSize, dim]));
                }
                else {
                    this.states_ = [tfc.zeros([batchSize, this.cell.stateSize])];
                }
            }
            else if (states == null) {
                // Dispose old state tensors.
                tfc.dispose(this.states_);
                // For stateful RNNs, fully dispose kept old states.
                if (this.keptStates != null) {
                    tfc.dispose(this.keptStates);
                    this.keptStates = [];
                }
                if (Array.isArray(this.cell.stateSize)) {
                    this.states_ =
                        this.cell.stateSize.map(dim => tfc.zeros([batchSize, dim]));
                }
                else {
                    this.states_[0] = tfc.zeros([batchSize, this.cell.stateSize]);
                }
            }
            else {
                if (!Array.isArray(states)) {
                    states = [states];
                }
                if (states.length !== this.states_.length) {
                    throw new ValueError(`Layer ${this.name} expects ${this.states_.length} state(s), ` +
                        `but it received ${states.length} state value(s). Input ` +
                        `received: ${states}`);
                }
                if (training === true) {
                    // Store old state tensors for complete disposal later, i.e., during
                    // the next no-arg call to this method. We do not dispose the old
                    // states immediately because that BPTT (among other things) require
                    // them.
                    this.keptStates.push(this.states_.slice());
                }
                else {
                    tfc.dispose(this.states_);
                }
                for (let index = 0; index < this.states_.length; ++index) {
                    const value = states[index];
                    const dim = Array.isArray(this.cell.stateSize) ?
                        this.cell.stateSize[index] :
                        this.cell.stateSize;
                    const expectedShape = [batchSize, dim];
                    if (!util.arraysEqual(value.shape, expectedShape)) {
                        throw new ValueError(`State ${index} is incompatible with layer ${this.name}: ` +
                            `expected shape=${expectedShape}, received shape=${value.shape}`);
                    }
                    this.states_[index] = value;
                }
            }
            this.states_ = this.states_.map(state => tfc.keep(state.clone()));
        });
    }
    apply(inputs, kwargs) {
        // TODO(cais): Figure out whether initialState is in kwargs or inputs.
        let initialState = kwargs == null ? null : kwargs['initialState'];
        let constants = kwargs == null ? null : kwargs['constants'];
        if (kwargs == null) {
            kwargs = {};
        }
        const standardized = standardizeArgs(inputs, initialState, constants, this.numConstants);
        inputs = standardized.inputs;
        initialState = standardized.initialState;
        constants = standardized.constants;
        // If any of `initial_state` or `constants` are specified and are
        // `tf.SymbolicTensor`s, then add them to the inputs and temporarily modify
        // the input_spec to include them.
        let additionalInputs = [];
        let additionalSpecs = [];
        if (initialState != null) {
            kwargs['initialState'] = initialState;
            additionalInputs = additionalInputs.concat(initialState);
            this.stateSpec = [];
            for (const state of initialState) {
                this.stateSpec.push(new InputSpec({ shape: state.shape }));
            }
            // TODO(cais): Use the following instead.
            // this.stateSpec = initialState.map(state => new InputSpec({shape:
            // state.shape}));
            additionalSpecs = additionalSpecs.concat(this.stateSpec);
        }
        if (constants != null) {
            kwargs['constants'] = constants;
            additionalInputs = additionalInputs.concat(constants);
            // TODO(cais): Add this.constantsSpec.
            this.numConstants = constants.length;
        }
        const isTensor = additionalInputs[0] instanceof SymbolicTensor;
        if (isTensor) {
            // Compute full input spec, including state and constants.
            const fullInput = [inputs].concat(additionalInputs);
            const fullInputSpec = this.inputSpec.concat(additionalSpecs);
            // Perform the call with temporarily replaced inputSpec.
            const originalInputSpec = this.inputSpec;
            this.inputSpec = fullInputSpec;
            const output = super.apply(fullInput, kwargs);
            this.inputSpec = originalInputSpec;
            return output;
        }
        else {
            return super.apply(inputs, kwargs);
        }
    }
    // tslint:disable-next-line:no-any
    call(inputs, kwargs) {
        // Input shape: `[samples, time (padded with zeros), input_dim]`.
        // Note that the .build() method of subclasses **must** define
        // this.inputSpec and this.stateSpec owith complete input shapes.
        return tidy(() => {
            const mask = kwargs == null ? null : kwargs['mask'];
            const training = kwargs == null ? null : kwargs['training'];
            let initialState = kwargs == null ? null : kwargs['initialState'];
            inputs = getExactlyOneTensor(inputs);
            if (initialState == null) {
                if (this.stateful) {
                    initialState = this.states_;
                }
                else {
                    initialState = this.getInitialState(inputs);
                }
            }
            const numStates = Array.isArray(this.cell.stateSize) ? this.cell.stateSize.length : 1;
            if (initialState.length !== numStates) {
                throw new ValueError(`RNN Layer has ${numStates} state(s) but was passed ` +
                    `${initialState.length} initial state(s).`);
            }
            if (this.unroll) {
                console.warn('Ignoring unroll = true for RNN layer, due to imperative backend.');
            }
            const cellCallKwargs = { training };
            // TODO(cais): Add support for constants.
            const step = (inputs, states) => {
                // `inputs` and `states` are concatenated to form a single `Array` of
                // `tf.Tensor`s as the input to `cell.call()`.
                const outputs = this.cell.call([inputs].concat(states), cellCallKwargs);
                // Marshall the return value into output and new states.
                return [outputs[0], outputs.slice(1)];
            };
            // TODO(cais): Add support for constants.
            const rnnOutputs = rnn(step, inputs, initialState, this.goBackwards, mask, null, this.unroll, this.returnSequences);
            const lastOutput = rnnOutputs[0];
            const outputs = rnnOutputs[1];
            const states = rnnOutputs[2];
            if (this.stateful) {
                this.resetStates(states, training);
            }
            const output = this.returnSequences ? outputs : lastOutput;
            // TODO(cais): Porperty set learning phase flag.
            if (this.returnState) {
                return [output].concat(states);
            }
            else {
                return output;
            }
        });
    }
    getInitialState(inputs) {
        return tidy(() => {
            // Build an all-zero tensor of shape [samples, outputDim].
            // [Samples, timeSteps, inputDim].
            let initialState = tfc.zeros(inputs.shape);
            // [Samples].
            initialState = tfc.sum(initialState, [1, 2]);
            initialState = K.expandDims(initialState); // [Samples, 1].
            if (Array.isArray(this.cell.stateSize)) {
                return this.cell.stateSize.map(dim => dim > 1 ? K.tile(initialState, [1, dim]) : initialState);
            }
            else {
                return this.cell.stateSize > 1 ?
                    [K.tile(initialState, [1, this.cell.stateSize])] :
                    [initialState];
            }
        });
    }
    get trainableWeights() {
        if (!this.trainable) {
            return [];
        }
        // Porting Note: In TypeScript, `this` is always an instance of `Layer`.
        return this.cell.trainableWeights;
    }
    get nonTrainableWeights() {
        // Porting Note: In TypeScript, `this` is always an instance of `Layer`.
        if (!this.trainable) {
            return this.cell.weights;
        }
        return this.cell.nonTrainableWeights;
    }
    setFastWeightInitDuringBuild(value) {
        super.setFastWeightInitDuringBuild(value);
        if (this.cell != null) {
            this.cell.setFastWeightInitDuringBuild(value);
        }
    }
    getConfig() {
        const baseConfig = super.getConfig();
        const config = {
            returnSequences: this.returnSequences,
            returnState: this.returnState,
            goBackwards: this.goBackwards,
            stateful: this.stateful,
            unroll: this.unroll,
        };
        if (this.numConstants != null) {
            config['numConstants'] = this.numConstants;
        }
        const cellConfig = this.cell.getConfig();
        if (this.getClassName() === RNN.className) {
            config['cell'] = {
                'className': this.cell.getClassName(),
                'config': cellConfig,
            };
        }
        // this order is necessary, to prevent cell name from replacing layer name
        return Object.assign(Object.assign(Object.assign({}, cellConfig), baseConfig), config);
    }
    /** @nocollapse */
    static fromConfig(cls, config, customObjects = {}) {
        const cellConfig = config['cell'];
        const cell = deserialize(cellConfig, customObjects);
        return new cls(Object.assign(config, { cell }));
    }
}
/** @nocollapse */
RNN.className = 'RNN';
serialization.registerClass(RNN);
// Porting Note: This is a common parent class for RNN cells. There is no
// equivalent of this in PyKeras. Having a common parent class forgoes the
//  need for `has_attr(cell, ...)` checks or its TypeScript equivalent.
/**
 * An RNNCell layer.
 *
 * @doc {heading: 'Layers', subheading: 'Classes'}
 */
export class RNNCell extends Layer {
}
export class SimpleRNNCell extends RNNCell {
    constructor(args) {
        super(args);
        this.DEFAULT_ACTIVATION = 'tanh';
        this.DEFAULT_KERNEL_INITIALIZER = 'glorotNormal';
        this.DEFAULT_RECURRENT_INITIALIZER = 'orthogonal';
        this.DEFAULT_BIAS_INITIALIZER = 'zeros';
        this.units = args.units;
        assertPositiveInteger(this.units, `units`);
        this.activation = getActivation(args.activation == null ? this.DEFAULT_ACTIVATION : args.activation);
        this.useBias = args.useBias == null ? true : args.useBias;
        this.kernelInitializer = getInitializer(args.kernelInitializer || this.DEFAULT_KERNEL_INITIALIZER);
        this.recurrentInitializer = getInitializer(args.recurrentInitializer || this.DEFAULT_RECURRENT_INITIALIZER);
        this.biasInitializer =
            getInitializer(args.biasInitializer || this.DEFAULT_BIAS_INITIALIZER);
        this.kernelRegularizer = getRegularizer(args.kernelRegularizer);
        this.recurrentRegularizer = getRegularizer(args.recurrentRegularizer);
        this.biasRegularizer = getRegularizer(args.biasRegularizer);
        this.kernelConstraint = getConstraint(args.kernelConstraint);
        this.recurrentConstraint = getConstraint(args.recurrentConstraint);
        this.biasConstraint = getConstraint(args.biasConstraint);
        this.dropout = math_utils.min([1, math_utils.max([0, args.dropout == null ? 0 : args.dropout])]);
        this.recurrentDropout = math_utils.min([
            1,
            math_utils.max([0, args.recurrentDropout == null ? 0 : args.recurrentDropout])
        ]);
        this.dropoutFunc = args.dropoutFunc;
        this.stateSize = this.units;
        this.dropoutMask = null;
        this.recurrentDropoutMask = null;
    }
    build(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        // TODO(cais): Use regularizer.
        this.kernel = this.addWeight('kernel', [inputShape[inputShape.length - 1], this.units], null, this.kernelInitializer, this.kernelRegularizer, true, this.kernelConstraint);
        this.recurrentKernel = this.addWeight('recurrent_kernel', [this.units, this.units], null, this.recurrentInitializer, this.recurrentRegularizer, true, this.recurrentConstraint);
        if (this.useBias) {
            this.bias = this.addWeight('bias', [this.units], null, this.biasInitializer, this.biasRegularizer, true, this.biasConstraint);
        }
        else {
            this.bias = null;
        }
        this.built = true;
    }
    // Porting Note: PyKeras' equivalent of this method takes two tensor inputs:
    //   `inputs` and `states`. Here, the two tensors are combined into an
    //   `Tensor[]` Array as the first input argument.
    //   Similarly, PyKeras' equivalent of this method returns two values:
    //    `output` and `[output]`. Here the two are combined into one length-2
    //    `Tensor[]`, consisting of `output` repeated.
    call(inputs, kwargs) {
        return tidy(() => {
            inputs = inputs;
            if (inputs.length !== 2) {
                throw new ValueError(`SimpleRNNCell expects 2 input Tensors, got ${inputs.length}.`);
            }
            let prevOutput = inputs[1];
            inputs = inputs[0];
            const training = kwargs['training'] == null ? false : kwargs['training'];
            if (0 < this.dropout && this.dropout < 1 && this.dropoutMask == null) {
                this.dropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(inputs),
                    rate: this.dropout,
                    training,
                    dropoutFunc: this.dropoutFunc,
                });
            }
            if (0 < this.recurrentDropout && this.recurrentDropout < 1 &&
                this.recurrentDropoutMask == null) {
                this.recurrentDropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(prevOutput),
                    rate: this.recurrentDropout,
                    training,
                    dropoutFunc: this.dropoutFunc,
                });
            }
            let h;
            const dpMask = this.dropoutMask;
            const recDpMask = this.recurrentDropoutMask;
            if (dpMask != null) {
                h = K.dot(tfc.mul(inputs, dpMask), this.kernel.read());
            }
            else {
                h = K.dot(inputs, this.kernel.read());
            }
            if (this.bias != null) {
                h = K.biasAdd(h, this.bias.read());
            }
            if (recDpMask != null) {
                prevOutput = tfc.mul(prevOutput, recDpMask);
            }
            let output = tfc.add(h, K.dot(prevOutput, this.recurrentKernel.read()));
            if (this.activation != null) {
                output = this.activation.apply(output);
            }
            // TODO(cais): Properly set learning phase on output tensor?
            return [output, output];
        });
    }
    getConfig() {
        const baseConfig = super.getConfig();
        const config = {
            units: this.units,
            activation: serializeActivation(this.activation),
            useBias: this.useBias,
            kernelInitializer: serializeInitializer(this.kernelInitializer),
            recurrentInitializer: serializeInitializer(this.recurrentInitializer),
            biasInitializer: serializeInitializer(this.biasInitializer),
            kernelRegularizer: serializeRegularizer(this.kernelRegularizer),
            recurrentRegularizer: serializeRegularizer(this.recurrentRegularizer),
            biasRegularizer: serializeRegularizer(this.biasRegularizer),
            activityRegularizer: serializeRegularizer(this.activityRegularizer),
            kernelConstraint: serializeConstraint(this.kernelConstraint),
            recurrentConstraint: serializeConstraint(this.recurrentConstraint),
            biasConstraint: serializeConstraint(this.biasConstraint),
            dropout: this.dropout,
            recurrentDropout: this.recurrentDropout,
        };
        return Object.assign(Object.assign({}, baseConfig), config);
    }
}
/** @nocollapse */
SimpleRNNCell.className = 'SimpleRNNCell';
serialization.registerClass(SimpleRNNCell);
export class SimpleRNN extends RNN {
    constructor(args) {
        args.cell = new SimpleRNNCell(args);
        super(args);
        // TODO(cais): Add activityRegularizer.
    }
    call(inputs, kwargs) {
        return tidy(() => {
            if (this.cell.dropoutMask != null) {
                tfc.dispose(this.cell.dropoutMask);
                this.cell.dropoutMask = null;
            }
            if (this.cell.recurrentDropoutMask != null) {
                tfc.dispose(this.cell.recurrentDropoutMask);
                this.cell.recurrentDropoutMask = null;
            }
            const mask = kwargs == null ? null : kwargs['mask'];
            const training = kwargs == null ? null : kwargs['training'];
            const initialState = kwargs == null ? null : kwargs['initialState'];
            return super.call(inputs, { mask, training, initialState });
        });
    }
    /** @nocollapse */
    static fromConfig(cls, config) {
        return new cls(config);
    }
}
/** @nocollapse */
SimpleRNN.className = 'SimpleRNN';
serialization.registerClass(SimpleRNN);
export class GRUCell extends RNNCell {
    constructor(args) {
        super(args);
        this.DEFAULT_ACTIVATION = 'tanh';
        this.DEFAULT_RECURRENT_ACTIVATION = 'hardSigmoid';
        this.DEFAULT_KERNEL_INITIALIZER = 'glorotNormal';
        this.DEFAULT_RECURRENT_INITIALIZER = 'orthogonal';
        this.DEFAULT_BIAS_INITIALIZER = 'zeros';
        if (args.resetAfter) {
            throw new ValueError(`GRUCell does not support reset_after parameter set to true.`);
        }
        this.units = args.units;
        assertPositiveInteger(this.units, 'units');
        this.activation = getActivation(args.activation === undefined ? this.DEFAULT_ACTIVATION :
            args.activation);
        this.recurrentActivation = getActivation(args.recurrentActivation === undefined ?
            this.DEFAULT_RECURRENT_ACTIVATION :
            args.recurrentActivation);
        this.useBias = args.useBias == null ? true : args.useBias;
        this.kernelInitializer = getInitializer(args.kernelInitializer || this.DEFAULT_KERNEL_INITIALIZER);
        this.recurrentInitializer = getInitializer(args.recurrentInitializer || this.DEFAULT_RECURRENT_INITIALIZER);
        this.biasInitializer =
            getInitializer(args.biasInitializer || this.DEFAULT_BIAS_INITIALIZER);
        this.kernelRegularizer = getRegularizer(args.kernelRegularizer);
        this.recurrentRegularizer = getRegularizer(args.recurrentRegularizer);
        this.biasRegularizer = getRegularizer(args.biasRegularizer);
        this.kernelConstraint = getConstraint(args.kernelConstraint);
        this.recurrentConstraint = getConstraint(args.recurrentConstraint);
        this.biasConstraint = getConstraint(args.biasConstraint);
        this.dropout = math_utils.min([1, math_utils.max([0, args.dropout == null ? 0 : args.dropout])]);
        this.recurrentDropout = math_utils.min([
            1,
            math_utils.max([0, args.recurrentDropout == null ? 0 : args.recurrentDropout])
        ]);
        this.dropoutFunc = args.dropoutFunc;
        this.implementation = args.implementation;
        this.stateSize = this.units;
        this.dropoutMask = null;
        this.recurrentDropoutMask = null;
    }
    build(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        const inputDim = inputShape[inputShape.length - 1];
        this.kernel = this.addWeight('kernel', [inputDim, this.units * 3], null, this.kernelInitializer, this.kernelRegularizer, true, this.kernelConstraint);
        this.recurrentKernel = this.addWeight('recurrent_kernel', [this.units, this.units * 3], null, this.recurrentInitializer, this.recurrentRegularizer, true, this.recurrentConstraint);
        if (this.useBias) {
            this.bias = this.addWeight('bias', [this.units * 3], null, this.biasInitializer, this.biasRegularizer, true, this.biasConstraint);
        }
        else {
            this.bias = null;
        }
        // Porting Notes: Unlike the PyKeras implementation, we perform slicing
        //   of the weights and bias in the call() method, at execution time.
        this.built = true;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            inputs = inputs;
            if (inputs.length !== 2) {
                throw new ValueError(`GRUCell expects 2 input Tensors (inputs, h, c), got ` +
                    `${inputs.length}.`);
            }
            const training = kwargs['training'] == null ? false : kwargs['training'];
            let hTMinus1 = inputs[1]; // Previous memory state.
            inputs = inputs[0];
            // Note: For superior performance, TensorFlow.js always uses
            // implementation 2, regardless of the actual value of
            // config.implementation.
            if (0 < this.dropout && this.dropout < 1 && this.dropoutMask == null) {
                this.dropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(inputs),
                    rate: this.dropout,
                    training,
                    count: 3,
                    dropoutFunc: this.dropoutFunc,
                });
            }
            if (0 < this.recurrentDropout && this.recurrentDropout < 1 &&
                this.recurrentDropoutMask == null) {
                this.recurrentDropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(hTMinus1),
                    rate: this.recurrentDropout,
                    training,
                    count: 3,
                    dropoutFunc: this.dropoutFunc,
                });
            }
            const dpMask = this.dropoutMask;
            const recDpMask = this.recurrentDropoutMask;
            let z;
            let r;
            let hh;
            if (0 < this.dropout && this.dropout < 1) {
                inputs = tfc.mul(inputs, dpMask[0]);
            }
            let matrixX = K.dot(inputs, this.kernel.read());
            if (this.useBias) {
                matrixX = K.biasAdd(matrixX, this.bias.read());
            }
            if (0 < this.recurrentDropout && this.recurrentDropout < 1) {
                hTMinus1 = tfc.mul(hTMinus1, recDpMask[0]);
            }
            const recurrentKernelValue = this.recurrentKernel.read();
            const [rk1, rk2] = tfc.split(recurrentKernelValue, [2 * this.units, this.units], recurrentKernelValue.rank - 1);
            const matrixInner = K.dot(hTMinus1, rk1);
            const [xZ, xR, xH] = tfc.split(matrixX, 3, matrixX.rank - 1);
            const [recurrentZ, recurrentR] = tfc.split(matrixInner, 2, matrixInner.rank - 1);
            z = this.recurrentActivation.apply(tfc.add(xZ, recurrentZ));
            r = this.recurrentActivation.apply(tfc.add(xR, recurrentR));
            const recurrentH = K.dot(tfc.mul(r, hTMinus1), rk2);
            hh = this.activation.apply(tfc.add(xH, recurrentH));
            const h = tfc.add(tfc.mul(z, hTMinus1), tfc.mul(tfc.add(1, tfc.neg(z)), hh));
            // TODO(cais): Add use_learning_phase flag properly.
            return [h, h];
        });
    }
    getConfig() {
        const baseConfig = super.getConfig();
        const config = {
            units: this.units,
            activation: serializeActivation(this.activation),
            recurrentActivation: serializeActivation(this.recurrentActivation),
            useBias: this.useBias,
            kernelInitializer: serializeInitializer(this.kernelInitializer),
            recurrentInitializer: serializeInitializer(this.recurrentInitializer),
            biasInitializer: serializeInitializer(this.biasInitializer),
            kernelRegularizer: serializeRegularizer(this.kernelRegularizer),
            recurrentRegularizer: serializeRegularizer(this.recurrentRegularizer),
            biasRegularizer: serializeRegularizer(this.biasRegularizer),
            activityRegularizer: serializeRegularizer(this.activityRegularizer),
            kernelConstraint: serializeConstraint(this.kernelConstraint),
            recurrentConstraint: serializeConstraint(this.recurrentConstraint),
            biasConstraint: serializeConstraint(this.biasConstraint),
            dropout: this.dropout,
            recurrentDropout: this.recurrentDropout,
            implementation: this.implementation,
            resetAfter: false
        };
        return Object.assign(Object.assign({}, baseConfig), config);
    }
}
/** @nocollapse */
GRUCell.className = 'GRUCell';
serialization.registerClass(GRUCell);
export class GRU extends RNN {
    constructor(args) {
        if (args.implementation === 0) {
            console.warn('`implementation=0` has been deprecated, and now defaults to ' +
                '`implementation=1`. Please update your layer call.');
        }
        args.cell = new GRUCell(args);
        super(args);
        // TODO(cais): Add activityRegularizer.
    }
    call(inputs, kwargs) {
        return tidy(() => {
            if (this.cell.dropoutMask != null) {
                tfc.dispose(this.cell.dropoutMask);
                this.cell.dropoutMask = null;
            }
            if (this.cell.recurrentDropoutMask != null) {
                tfc.dispose(this.cell.recurrentDropoutMask);
                this.cell.recurrentDropoutMask = null;
            }
            const mask = kwargs == null ? null : kwargs['mask'];
            const training = kwargs == null ? null : kwargs['training'];
            const initialState = kwargs == null ? null : kwargs['initialState'];
            return super.call(inputs, { mask, training, initialState });
        });
    }
    /** @nocollapse */
    static fromConfig(cls, config) {
        if (config['implmentation'] === 0) {
            config['implementation'] = 1;
        }
        return new cls(config);
    }
}
/** @nocollapse */
GRU.className = 'GRU';
serialization.registerClass(GRU);
export class LSTMCell extends RNNCell {
    constructor(args) {
        super(args);
        this.DEFAULT_ACTIVATION = 'tanh';
        this.DEFAULT_RECURRENT_ACTIVATION = 'hardSigmoid';
        this.DEFAULT_KERNEL_INITIALIZER = 'glorotNormal';
        this.DEFAULT_RECURRENT_INITIALIZER = 'orthogonal';
        this.DEFAULT_BIAS_INITIALIZER = 'zeros';
        this.units = args.units;
        assertPositiveInteger(this.units, 'units');
        this.activation = getActivation(args.activation === undefined ? this.DEFAULT_ACTIVATION :
            args.activation);
        this.recurrentActivation = getActivation(args.recurrentActivation === undefined ?
            this.DEFAULT_RECURRENT_ACTIVATION :
            args.recurrentActivation);
        this.useBias = args.useBias == null ? true : args.useBias;
        this.kernelInitializer = getInitializer(args.kernelInitializer || this.DEFAULT_KERNEL_INITIALIZER);
        this.recurrentInitializer = getInitializer(args.recurrentInitializer || this.DEFAULT_RECURRENT_INITIALIZER);
        this.biasInitializer =
            getInitializer(args.biasInitializer || this.DEFAULT_BIAS_INITIALIZER);
        this.unitForgetBias = args.unitForgetBias;
        this.kernelRegularizer = getRegularizer(args.kernelRegularizer);
        this.recurrentRegularizer = getRegularizer(args.recurrentRegularizer);
        this.biasRegularizer = getRegularizer(args.biasRegularizer);
        this.kernelConstraint = getConstraint(args.kernelConstraint);
        this.recurrentConstraint = getConstraint(args.recurrentConstraint);
        this.biasConstraint = getConstraint(args.biasConstraint);
        this.dropout = math_utils.min([1, math_utils.max([0, args.dropout == null ? 0 : args.dropout])]);
        this.recurrentDropout = math_utils.min([
            1,
            math_utils.max([0, args.recurrentDropout == null ? 0 : args.recurrentDropout])
        ]);
        this.dropoutFunc = args.dropoutFunc;
        this.implementation = args.implementation;
        this.stateSize = [this.units, this.units];
        this.dropoutMask = null;
        this.recurrentDropoutMask = null;
    }
    build(inputShape) {
        var _a;
        inputShape = getExactlyOneShape(inputShape);
        const inputDim = inputShape[inputShape.length - 1];
        this.kernel = this.addWeight('kernel', [inputDim, this.units * 4], null, this.kernelInitializer, this.kernelRegularizer, true, this.kernelConstraint);
        this.recurrentKernel = this.addWeight('recurrent_kernel', [this.units, this.units * 4], null, this.recurrentInitializer, this.recurrentRegularizer, true, this.recurrentConstraint);
        let biasInitializer;
        if (this.useBias) {
            if (this.unitForgetBias) {
                const capturedBiasInit = this.biasInitializer;
                const capturedUnits = this.units;
                biasInitializer = new (_a = class CustomInit extends Initializer {
                        apply(shape, dtype) {
                            // TODO(cais): More informative variable names?
                            const bI = capturedBiasInit.apply([capturedUnits]);
                            const bF = (new Ones()).apply([capturedUnits]);
                            const bCAndH = capturedBiasInit.apply([capturedUnits * 2]);
                            return K.concatAlongFirstAxis(K.concatAlongFirstAxis(bI, bF), bCAndH);
                        }
                    },
                    /** @nocollapse */
                    _a.className = 'CustomInit',
                    _a)();
            }
            else {
                biasInitializer = this.biasInitializer;
            }
            this.bias = this.addWeight('bias', [this.units * 4], null, biasInitializer, this.biasRegularizer, true, this.biasConstraint);
        }
        else {
            this.bias = null;
        }
        // Porting Notes: Unlike the PyKeras implementation, we perform slicing
        //   of the weights and bias in the call() method, at execution time.
        this.built = true;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            const training = kwargs['training'] == null ? false : kwargs['training'];
            inputs = inputs;
            if (inputs.length !== 3) {
                throw new ValueError(`LSTMCell expects 3 input Tensors (inputs, h, c), got ` +
                    `${inputs.length}.`);
            }
            let hTMinus1 = inputs[1]; // Previous memory state.
            const cTMinus1 = inputs[2]; // Previous carry state.
            inputs = inputs[0];
            if (0 < this.dropout && this.dropout < 1 && this.dropoutMask == null) {
                this.dropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(inputs),
                    rate: this.dropout,
                    training,
                    count: 4,
                    dropoutFunc: this.dropoutFunc
                });
            }
            if (0 < this.recurrentDropout && this.recurrentDropout < 1 &&
                this.recurrentDropoutMask == null) {
                this.recurrentDropoutMask = generateDropoutMask({
                    ones: () => tfc.onesLike(hTMinus1),
                    rate: this.recurrentDropout,
                    training,
                    count: 4,
                    dropoutFunc: this.dropoutFunc
                });
            }
            const dpMask = this.dropoutMask;
            const recDpMask = this.recurrentDropoutMask;
            // Note: For superior performance, TensorFlow.js always uses
            // implementation 2 regardless of the actual value of
            // config.implementation.
            let i;
            let f;
            let c;
            let o;
            if (0 < this.dropout && this.dropout < 1) {
                inputs = tfc.mul(inputs, dpMask[0]);
            }
            let z = K.dot(inputs, this.kernel.read());
            if (0 < this.recurrentDropout && this.recurrentDropout < 1) {
                hTMinus1 = tfc.mul(hTMinus1, recDpMask[0]);
            }
            z = tfc.add(z, K.dot(hTMinus1, this.recurrentKernel.read()));
            if (this.useBias) {
                z = K.biasAdd(z, this.bias.read());
            }
            const [z0, z1, z2, z3] = tfc.split(z, 4, z.rank - 1);
            i = this.recurrentActivation.apply(z0);
            f = this.recurrentActivation.apply(z1);
            c = tfc.add(tfc.mul(f, cTMinus1), tfc.mul(i, this.activation.apply(z2)));
            o = this.recurrentActivation.apply(z3);
            const h = tfc.mul(o, this.activation.apply(c));
            // TODO(cais): Add use_learning_phase flag properly.
            return [h, h, c];
        });
    }
    getConfig() {
        const baseConfig = super.getConfig();
        const config = {
            units: this.units,
            activation: serializeActivation(this.activation),
            recurrentActivation: serializeActivation(this.recurrentActivation),
            useBias: this.useBias,
            kernelInitializer: serializeInitializer(this.kernelInitializer),
            recurrentInitializer: serializeInitializer(this.recurrentInitializer),
            biasInitializer: serializeInitializer(this.biasInitializer),
            unitForgetBias: this.unitForgetBias,
            kernelRegularizer: serializeRegularizer(this.kernelRegularizer),
            recurrentRegularizer: serializeRegularizer(this.recurrentRegularizer),
            biasRegularizer: serializeRegularizer(this.biasRegularizer),
            activityRegularizer: serializeRegularizer(this.activityRegularizer),
            kernelConstraint: serializeConstraint(this.kernelConstraint),
            recurrentConstraint: serializeConstraint(this.recurrentConstraint),
            biasConstraint: serializeConstraint(this.biasConstraint),
            dropout: this.dropout,
            recurrentDropout: this.recurrentDropout,
            implementation: this.implementation,
        };
        return Object.assign(Object.assign({}, baseConfig), config);
    }
}
/** @nocollapse */
LSTMCell.className = 'LSTMCell';
serialization.registerClass(LSTMCell);
export class LSTM extends RNN {
    constructor(args) {
        if (args.implementation === 0) {
            console.warn('`implementation=0` has been deprecated, and now defaults to ' +
                '`implementation=1`. Please update your layer call.');
        }
        args.cell = new LSTMCell(args);
        super(args);
        // TODO(cais): Add activityRegularizer.
    }
    call(inputs, kwargs) {
        return tidy(() => {
            if (this.cell.dropoutMask != null) {
                tfc.dispose(this.cell.dropoutMask);
                this.cell.dropoutMask = null;
            }
            if (this.cell.recurrentDropoutMask != null) {
                tfc.dispose(this.cell.recurrentDropoutMask);
                this.cell.recurrentDropoutMask = null;
            }
            const mask = kwargs == null ? null : kwargs['mask'];
            const training = kwargs == null ? null : kwargs['training'];
            const initialState = kwargs == null ? null : kwargs['initialState'];
            return super.call(inputs, { mask, training, initialState });
        });
    }
    /** @nocollapse */
    static fromConfig(cls, config) {
        if (config['implmentation'] === 0) {
            config['implementation'] = 1;
        }
        return new cls(config);
    }
}
/** @nocollapse */
LSTM.className = 'LSTM';
serialization.registerClass(LSTM);
export class StackedRNNCells extends RNNCell {
    constructor(args) {
        super(args);
        this.cells = args.cells;
    }
    get stateSize() {
        // States are a flat list in reverse order of the cell stack.
        // This allows perserving the requirement `stack.statesize[0] ===
        // outputDim`. E.g., states of a 2-layer LSTM would be `[h2, c2, h1, c1]`,
        // assuming one LSTM has states `[h, c]`.
        const stateSize = [];
        for (const cell of this.cells.slice().reverse()) {
            if (Array.isArray(cell.stateSize)) {
                stateSize.push(...cell.stateSize);
            }
            else {
                stateSize.push(cell.stateSize);
            }
        }
        return stateSize;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            inputs = inputs;
            let states = inputs.slice(1);
            // Recover per-cell states.
            const nestedStates = [];
            for (const cell of this.cells.slice().reverse()) {
                if (Array.isArray(cell.stateSize)) {
                    nestedStates.push(states.splice(0, cell.stateSize.length));
                }
                else {
                    nestedStates.push(states.splice(0, 1));
                }
            }
            nestedStates.reverse();
            // Call the cells in order and store the returned states.
            const newNestedStates = [];
            let callInputs;
            for (let i = 0; i < this.cells.length; ++i) {
                const cell = this.cells[i];
                states = nestedStates[i];
                // TODO(cais): Take care of constants.
                if (i === 0) {
                    callInputs = [inputs[0]].concat(states);
                }
                else {
                    callInputs = [callInputs[0]].concat(states);
                }
                callInputs = cell.call(callInputs, kwargs);
                newNestedStates.push(callInputs.slice(1));
            }
            // Format the new states as a flat list in reverse cell order.
            states = [];
            for (const cellStates of newNestedStates.slice().reverse()) {
                states.push(...cellStates);
            }
            return [callInputs[0]].concat(states);
        });
    }
    build(inputShape) {
        if (isArrayOfShapes(inputShape)) {
            // TODO(cais): Take care of input constants.
            // const constantShape = inputShape.slice(1);
            inputShape = inputShape[0];
        }
        inputShape = inputShape;
        let outputDim;
        this.cells.forEach((cell, i) => {
            nameScope(`RNNCell_${i}`, () => {
                // TODO(cais): Take care of input constants.
                cell.build(inputShape);
                if (Array.isArray(cell.stateSize)) {
                    outputDim = cell.stateSize[0];
                }
                else {
                    outputDim = cell.stateSize;
                }
                inputShape = [inputShape[0], outputDim];
            });
        });
        this.built = true;
    }
    getConfig() {
        const baseConfig = super.getConfig();
        const getCellConfig = (cell) => {
            return {
                'className': cell.getClassName(),
                'config': cell.getConfig(),
            };
        };
        const cellConfigs = this.cells.map(getCellConfig);
        const config = { 'cells': cellConfigs };
        return Object.assign(Object.assign({}, baseConfig), config);
    }
    /** @nocollapse */
    static fromConfig(cls, config, customObjects = {}) {
        const cells = [];
        for (const cellConfig of config['cells']) {
            cells.push(deserialize(cellConfig, customObjects));
        }
        return new cls({ cells });
    }
    get trainableWeights() {
        if (!this.trainable) {
            return [];
        }
        const weights = [];
        for (const cell of this.cells) {
            weights.push(...cell.trainableWeights);
        }
        return weights;
    }
    get nonTrainableWeights() {
        const weights = [];
        for (const cell of this.cells) {
            weights.push(...cell.nonTrainableWeights);
        }
        if (!this.trainable) {
            const trainableWeights = [];
            for (const cell of this.cells) {
                trainableWeights.push(...cell.trainableWeights);
            }
            return trainableWeights.concat(weights);
        }
        return weights;
    }
    /**
     * Retrieve the weights of a the model.
     *
     * @returns A flat `Array` of `tf.Tensor`s.
     */
    getWeights() {
        const weights = [];
        for (const cell of this.cells) {
            weights.push(...cell.weights);
        }
        return batchGetValue(weights);
    }
    /**
     * Set the weights of the model.
     *
     * @param weights An `Array` of `tf.Tensor`s with shapes and types matching
     *     the output of `getWeights()`.
     */
    setWeights(weights) {
        const tuples = [];
        for (const cell of this.cells) {
            const numParams = cell.weights.length;
            const inputWeights = weights.splice(numParams);
            for (let i = 0; i < cell.weights.length; ++i) {
                tuples.push([cell.weights[i], inputWeights[i]]);
            }
        }
        batchSetValue(tuples);
    }
}
/** @nocollapse */
StackedRNNCells.className = 'StackedRNNCells';
serialization.registerClass(StackedRNNCells);
export function generateDropoutMask(args) {
    const { ones, rate, training = false, count = 1, dropoutFunc } = args;
    const droppedInputs = () => dropoutFunc != null ? dropoutFunc(ones(), rate) : K.dropout(ones(), rate);
    const createMask = () => K.inTrainPhase(droppedInputs, ones, training);
    // just in case count is provided with null or undefined
    if (!count || count <= 1) {
        return tfc.keep(createMask().clone());
    }
    const masks = Array(count).fill(undefined).map(createMask);
    return masks.map(m => tfc.keep(m.clone()));
}
//# sourceMappingURL=data:application/json;base64,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