/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/* Original source keras/models.py */
import { dispose, io, serialization, util } from '@tensorflow/tfjs-core';
import { getUid } from './backend/state';
import { Input } from './engine/input_layer';
import { getSourceInputs, Node } from './engine/topology';
import { LayersModel } from './engine/training';
import { NotImplementedError, RuntimeError, ValueError } from './errors';
import { deserialize } from './layers/serialization';
import * as generic_utils from './utils/generic_utils';
import { convertPythonicToTs } from './utils/serialization_utils';
import { getExactlyOneShape } from './utils/types_utils';
/**
 * Parses a JSON model configuration file and returns a model instance.
 *
 * ```js
 * // This example shows how to serialize a model using `toJSON()` and
 * // deserialize it as another model using `tf.models.modelFromJSON()`.
 * // Note: this example serializes and deserializes only the topology
 * // of the model; the weights of the loaded model will be different
 * // from those of the the original model, due to random weight
 * // initialization.
 * // To load the topology and weights of a model, use `tf.loadLayersModel()`.
 * const model1 = tf.sequential();
 * model1.add(tf.layers.repeatVector({inputShape: [2], n: 4}));
 * // Serialize `model1` as a JSON object.
 * const model1JSON = model1.toJSON(null, false);
 * model1.summary();
 *
 * const model2 = await tf.models.modelFromJSON(model1JSON);
 * model2.summary();
 * ```
 *
 *  @param modelAndWeightsConfig JSON object or string encoding a model and
 *       weights configuration. It can also be only the topology JSON of the
 *       model, in which case the weights will not be loaded.
 *  @param custom_objects Optional dictionary mapping names
 *       (strings) to custom classes or functions to be
 *       considered during deserialization.
 * @returns A TensorFlow.js Layers `tf.LayersModel` instance (uncompiled).
 */
export async function modelFromJSON(modelAndWeightsConfig, customObjects) {
    if (!('modelTopology' in modelAndWeightsConfig)) {
        modelAndWeightsConfig = { modelTopology: modelAndWeightsConfig };
    }
    modelAndWeightsConfig = modelAndWeightsConfig;
    let modelTopology = modelAndWeightsConfig.modelTopology;
    if (modelTopology['model_config'] != null) {
        // If the model-topology JSON contains a 'model_config' field, then it is
        // a full model JSON (e.g., from `keras.Model.save()`), which contains
        // not only the model's architecture in its 'model_config' field, but
        // additional information such as the model's optimizer. We use only the
        // 'model_config' field currently.
        modelTopology = modelTopology['model_config'];
    }
    const tsConfig = convertPythonicToTs(modelTopology);
    const model = deserialize(tsConfig, customObjects);
    if (modelAndWeightsConfig.weightsManifest != null) {
        // Load the weight values keyed by the original tensor names in the model
        // file that was loaded.  These should match the keys of the weight
        // manifest.
        const weightValues = await io.loadWeights(modelAndWeightsConfig.weightsManifest, modelAndWeightsConfig.pathPrefix, model.weights.map(weight => weight.originalName));
        // Map the weights to the unique tensor names generated during model loading
        const uniqueWeightValues = {};
        for (const weight of model.weights) {
            uniqueWeightValues[weight.originalName] =
                weightValues[weight.originalName];
        }
        model.loadWeights(uniqueWeightValues);
        // Dispose temporary weight values.
        dispose(weightValues);
    }
    return model;
}
/**
 * Load a model composed of Layer objects, including its topology and optionally
 * weights. See the Tutorial named "How to import a Keras Model" for usage
 * examples.
 *
 * This method is applicable to:
 *
 * 1. Models created with the `tf.layers.*`, `tf.sequential`, and
 * `tf.model` APIs of TensorFlow.js and later saved with the
 * `tf.LayersModel.save` method.
 * 2. Models converted from Keras or TensorFlow tf.keras using the
 * [tensorflowjs_converter](https://github.com/tensorflow/tfjs/tree/master/tfjs-converter).
 *
 * This mode is *not* applicable to TensorFlow `SavedModel`s or their converted
 * forms. For those models, use `tf.loadGraphModel`.
 *
 * Example 1. Load a model from an HTTP server.
 *
 * ```js
 * const model = await tf.loadLayersModel(
 *     'https://storage.googleapis.com/tfjs-models/tfjs/iris_v1/model.json');
 * model.summary();
 * ```
 *
 * Example 2: Save `model`'s topology and weights to browser [local
 * storage](https://developer.mozilla.org/en-US/docs/Web/API/Window/localStorage);
 * then load it back.
 *
 * ```js
 * const model = tf.sequential(
 *     {layers: [tf.layers.dense({units: 1, inputShape: [3]})]});
 * console.log('Prediction from original model:');
 * model.predict(tf.ones([1, 3])).print();
 *
 * const saveResults = await model.save('localstorage://my-model-1');
 *
 * const loadedModel = await tf.loadLayersModel('localstorage://my-model-1');
 * console.log('Prediction from loaded model:');
 * loadedModel.predict(tf.ones([1, 3])).print();
 * ```
 *
 * Example 3. Saving `model`'s topology and weights to browser
 * [IndexedDB](https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API);
 * then load it back.
 *
 * ```js
 * const model = tf.sequential(
 *     {layers: [tf.layers.dense({units: 1, inputShape: [3]})]});
 * console.log('Prediction from original model:');
 * model.predict(tf.ones([1, 3])).print();
 *
 * const saveResults = await model.save('indexeddb://my-model-1');
 *
 * const loadedModel = await tf.loadLayersModel('indexeddb://my-model-1');
 * console.log('Prediction from loaded model:');
 * loadedModel.predict(tf.ones([1, 3])).print();
 * ```
 *
 * Example 4. Load a model from user-selected files from HTML
 * [file input
 * elements](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input/file).
 *
 * ```js
 * // Note: this code snippet will not work without the HTML elements in the
 * //   page
 * const jsonUpload = document.getElementById('json-upload');
 * const weightsUpload = document.getElementById('weights-upload');
 *
 * const model = await tf.loadLayersModel(
 *     tf.io.browserFiles([jsonUpload.files[0], weightsUpload.files[0]]));
 * ```
 *
 * @param pathOrIOHandler Can be either of the two formats
 *   1. A string path to the `ModelAndWeightsConfig` JSON describing
 *      the model in the canonical TensorFlow.js format. For file://
 *      (tfjs-node-only), http:// and https:// schemas, the path can be
 *      either absolute or relative. The content of the JSON file is assumed to
 *      be a JSON object with the following fields and values:
 *      - 'modelTopology': A JSON object that can be either of:
 *        1. a model architecture JSON consistent with the format of the return
 *            value of `keras.Model.to_json()`
 *        2. a full model JSON in the format of `keras.models.save_model()`.
 *      - 'weightsManifest': A TensorFlow.js weights manifest.
 *      See the Python converter function `save_model()` for more details.
 *      It is also assumed that model weights can be accessed from relative
 *      paths described by the `paths` fields in weights manifest.
 *   2. A `tf.io.IOHandler` object that loads model artifacts with its `load`
 *      method.
 * @param options Optional configuration arguments for the model loading,
 *   including:
 *   - `strict`: Require that the provided weights exactly match those required
 *     by the layers.  Default true.  Passing false means that both extra
 *     weights and missing weights will be silently ignored.
 *   - `onProgress`: A progress callback of the form:
 *     `(fraction: number) => void`. This callback can be used to monitor the
 *     model-loading process.
 * @returns A `Promise` of `tf.LayersModel`, with the topology and weights
 *     loaded.
 *
 * @doc {heading: 'Models', subheading: 'Loading'}
 */
export async function loadLayersModel(pathOrIOHandler, options) {
    if (options == null) {
        options = {};
    }
    if (typeof pathOrIOHandler === 'string') {
        const handlers = io.getLoadHandlers(pathOrIOHandler, options);
        if (handlers.length === 0) {
            // For backward compatibility: if no load handler can be found,
            // assume it is a relative http path.
            // TODO(cais): Reformat the args into a single `LoadOptions` once the core
            // is refactored.
            handlers.push(io.browserHTTPRequest(pathOrIOHandler, options));
        }
        else if (handlers.length > 1) {
            throw new ValueError(`Found more than one (${handlers.length}) load handlers for ` +
                `URL '${pathOrIOHandler}'`);
        }
        pathOrIOHandler = handlers[0];
    }
    return loadLayersModelFromIOHandler(pathOrIOHandler, undefined, options);
}
/**
 * Load a model and optionally its weights, using an IOHandler object.
 *
 * @param handler The instance of `IOHandler` to be used during the model
 *   loading.
 * @param customObjects Any optional custom objects to be used during model
 *   loading.
 * @param strict Whether the weight loading will be done in strict mode.
 *   Default: `true`.
 */
export async function loadLayersModelFromIOHandler(handler, customObjects, options) {
    if (options == null) {
        options = {};
    }
    if (handler.load == null) {
        throw new ValueError('Cannot proceed with model loading because the IOHandler provided ' +
            'does not have the `load` method implemented.');
    }
    const artifacts = await handler.load();
    let modelTopology = artifacts.modelTopology;
    if (modelTopology['model_config'] != null) {
        modelTopology = modelTopology['model_config'];
    }
    const strict = options.strict == null ? true : options.strict;
    // If weights are provided and the weight-loading mode is strict, use
    // fast weight initialization. This skips costly initializers such as
    // 'orthogonal' and saves unnecessary computation in cases where
    // the initialized weight values will immediately be overwritten by
    // loaded weight values.
    const fastWeightInit = artifacts.weightData != null && artifacts.weightSpecs != null && strict;
    const model = deserialize(convertPythonicToTs(modelTopology), customObjects, fastWeightInit);
    const trainingConfig = artifacts.trainingConfig;
    if (trainingConfig != null) {
        model.loadTrainingConfig(trainingConfig);
    }
    if (artifacts.userDefinedMetadata != null) {
        model.setUserDefinedMetadata(artifacts.userDefinedMetadata);
    }
    // If weightData is present, load the weights into the model.
    if (artifacts.weightData != null) {
        // Loading weights requires weightSpecs.
        if (artifacts.weightSpecs == null) {
            throw new ValueError('LayersModel artifacts contains weight data, but not weight specs. ' +
                'Therefore loading of weights cannot proceed.');
        }
        const { modelWeights, optimizerWeights } = decodeModelAndOptimizerWeights(artifacts.weightData, artifacts.weightSpecs);
        model.loadWeights(modelWeights, strict);
        if (model.optimizer != null && optimizerWeights.length > 0) {
            await model.optimizer.setWeights(optimizerWeights);
        }
        // Dispose temporary weight values.
        dispose(modelWeights);
        dispose(optimizerWeights.map(w => w.tensor));
    }
    return model;
}
function decodeModelAndOptimizerWeights(buffer, specs) {
    const name2Tensor = io.decodeWeights(buffer, specs);
    const modelWeights = {};
    const optimizerWeights = [];
    specs.forEach(spec => {
        if (spec.group === 'optimizer') {
            optimizerWeights.push({ name: spec.name, tensor: name2Tensor[spec.name] });
        }
        else {
            modelWeights[spec.name] = name2Tensor[spec.name];
        }
    });
    return { modelWeights, optimizerWeights };
}
/**
 * A model with a stack of layers, feeding linearly from one to the next.
 *
 * `tf.sequential` is a factory function that creates an instance of
 * `tf.Sequential`.
 *
 * ```js
 *  // Define a model for linear regression.
 *  const model = tf.sequential();
 *  model.add(tf.layers.dense({units: 1, inputShape: [1]}));
 *
 *  // Prepare the model for training: Specify the loss and the optimizer.
 *  model.compile({loss: 'meanSquaredError', optimizer: 'sgd'});
 *
 *  // Generate some synthetic data for training.
 *  const xs = tf.tensor2d([1, 2, 3, 4], [4, 1]);
 *  const ys = tf.tensor2d([1, 3, 5, 7], [4, 1]);
 *
 *  // Train the model using the data then do inference on a data point the
 *  // model hasn't seen:
 *  await model.fit(xs, ys);
 *  model.predict(tf.tensor2d([5], [1, 1])).print();
 * ```
 *
 * @doc {heading: 'Models', subheading: 'Classes'}
 */
export class Sequential extends LayersModel {
    constructor(args) {
        super({ inputs: [], outputs: [] });
        args = args || {};
        this.trainable = true;
        this.built = false;
        // Set model name.
        this.name = (args.name != null) ? args.name : getUid('sequential_');
        // Add to the model any layers passed to the constructor.
        if (args.layers != null) {
            for (const layer of args.layers) {
                this.add(layer);
            }
        }
    }
    // Helper function to Sequential.add  Throws if the new output shape will be
    // invalid.
    checkShape(layer) {
        const shape = layer.inboundNodes[0].outputTensors[0].shape;
        if (shape.some(x => x < 0)) {
            throw new ValueError('Negative dimension size caused by adding layer ' +
                `${layer.name} with input shape [` +
                `${layer.inboundNodes[0].inputTensors[0].shape}]`);
        }
    }
    /**
     * Adds a layer instance on top of the layer stack.
     *
     * ```js
     *  const model = tf.sequential();
     *  model.add(tf.layers.dense({units: 8, inputShape: [1]}));
     *  model.add(tf.layers.dense({units: 4, activation: 'relu6'}));
     *  model.add(tf.layers.dense({units: 1, activation: 'relu6'}));
     *  // Note that the untrained model is random at this point.
     *  model.predict(tf.randomNormal([10, 1])).print();
     * ```
     * @param layer Layer instance.
     *
     * @exception ValueError In case the `layer` argument does not know its
     * input shape.
     * @exception ValueError In case the `layer` argument has multiple output
     *   tensors, or is already connected somewhere else (forbidden in
     *   `Sequential` models).
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    add(layer) {
        const isLayerModelInstance = layer instanceof Sequential || layer instanceof LayersModel;
        let modelLayer;
        if (isLayerModelInstance) {
            modelLayer = layer;
            if (modelLayer.outputs.length !== 1) {
                throw new ValueError('All layers in a Sequential model ' +
                    'should have a single output tensor. ' +
                    'For multi-output layers, ' +
                    'use the functional API.');
            }
            if (modelLayer.inputs.length !== 1) {
                throw new ValueError('All layers in a Sequential model ' +
                    'should have a single input tensor. ' +
                    'For multi-input layers, ' +
                    'use the functional API.');
            }
        }
        if (this.outputs.length === 0) {
            // first layer in model: check that it is an input layer
            if (layer.inboundNodes.length === 0) {
                // create an input layer
                if (layer.batchInputShape == null) {
                    throw new ValueError('The first layer in a Sequential model must ' +
                        'get an `inputShape` or `batchInputShape` argument.');
                }
                // Instantiate the input layer.
                const x = Input({
                    batchShape: layer.batchInputShape,
                    dtype: layer.dtype,
                    name: layer.name + '_input'
                });
                // This will build the current layer and create the node connecting
                // the current layer to the input layer we just created.
                layer.apply(x);
            }
            if (isLayerModelInstance) {
                this.outputs = modelLayer.outputs;
                this.inputs = modelLayer.inputs;
            }
            else {
                if (layer.inboundNodes.length !== 1) {
                    throw new ValueError('A layer added to a Sequential model must not already be ' +
                        `connected somewhere else. LayersModel received layer ${layer.name} ` +
                        `which has ${layer.inboundNodes.length} pre-existing inbound ` +
                        'connections.');
                }
                if (layer.inboundNodes[0].outputTensors.length !== 1) {
                    throw new ValueError('All layers in a Sequential model ' +
                        'should have a single output tensor. ' +
                        'For multi-output layers, ' +
                        'use the functional API.');
                }
                this.checkShape(layer);
                this.outputs = [layer.inboundNodes[0].outputTensors[0]];
                this.inputs = getSourceInputs(this.outputs[0]);
            }
            this.inboundNodes = [];
            // We create an input node, which we will keep updated
            // as we add more layers.
            // (This call has side effects.)
            // tslint:disable-next-line:no-unused-expression
            new Node({
                outboundLayer: this,
                inboundLayers: [],
                nodeIndices: [],
                tensorIndices: [],
                inputTensors: this.inputs,
                outputTensors: this.outputs,
                // no model-level masking for now
                inputMasks: generic_utils.pyListRepeat(null, this.inputs.length),
                outputMasks: [null],
                inputShapes: this.inputs.map(x => x.shape),
                outputShapes: this.outputs[0].shape
            });
        }
        else {
            const outputTensor = layer.apply(this.outputs[0]);
            if (Array.isArray(outputTensor)) {
                throw new TypeError('All layers in a Sequential model ' +
                    'should have a single output tensor. ' +
                    'For multi-output layers, ' +
                    'use the functional API.');
            }
            this.checkShape(layer);
            this.outputs = [outputTensor];
            // update self.inbound_nodes
            this.inboundNodes[0].outputTensors = this.outputs;
            this.inboundNodes[0].outputShapes = [this.outputs[0].shape];
        }
        this.layers.push(layer);
        this.built = false;
    }
    /**
     * Removes the last layer in the model.
     *
     * @exception TypeError if there are no layers in the model.
     */
    pop() {
        if (this.layers.length === 0) {
            throw new TypeError('There are no layers in the model.');
        }
        this.layers.pop();
        if (this.layers.length === 0) {
            this.outputs = [];
            this.inboundNodes = [];
            this.outboundNodes = [];
        }
        else {
            const lastLayerIndex = this.layers.length - 1;
            this.layers[lastLayerIndex].outboundNodes = [];
            this.outputs = [this.layers[lastLayerIndex].output];
            // update self.inbound_nodes
            this.inboundNodes[0].outputTensors = this.outputs;
            this.inboundNodes[0].outputShapes = [this.outputs[0].shape];
        }
    }
    call(inputs, kwargs) {
        if (this.model == null) {
            this.build();
        }
        return this.model.call(inputs, kwargs);
    }
    build(inputShape) {
        // Call `getExactlyOneShape` without using its return value,
        // to verify that exactly one input shape is provided.
        getExactlyOneShape(inputShape);
        if (this.inputs.length === 0 || this.outputs.length === 0) {
            throw new TypeError('Sequential model cannot be built: model is empty.' +
                ' Add some layers first.');
        }
        // actually create the model
        this.model = new LayersModel({
            inputs: this.inputs,
            outputs: this.outputs[0],
            name: this.name + '_model'
        });
        this.model.trainable = this.trainable;
        // mirror model attributes
        this.supportsMasking = this.model.supportsMasking;
        // TODO(michaelterry): Add caches
        this.inputLayers = this.model.inputLayers;
        this.inputLayersNodeIndices = this.model.inputLayersNodeIndices;
        this.inputLayersTensorIndices = this.model.inputLayersTensorIndices;
        this.outputLayers = this.model.outputLayers;
        this.outputLayersNodeIndices = this.model.outputLayersNodeIndices;
        this.outputLayersTensorIndices = this.model.outputLayersTensorIndices;
        this.nodesByDepth = this.model.nodesByDepth;
        this.containerNodes = this.model.containerNodes;
        this.outputNames = this.model.outputNames;
        this.inputNames = this.model.inputNames;
        // TODO(michaelterry): Add feedInputNames, feedInputs, if needed.
        // TODO(michaelterry): Add callbackModel if needed.
        this.built = true;
    }
    countParams() {
        if (!this.built) {
            this.build();
        }
        return super.countParams();
    }
    /**
     * Print a text summary of the Sequential model's layers.
     *
     * The summary includes
     * - Name and type of all layers that comprise the model.
     * - Output shape(s) of the layers
     * - Number of weight parameters of each layer
     * - The total number of trainable and non-trainable parameters of the
     * model.
     *
     * ```js
     * const model = tf.sequential();
     * model.add(
     *     tf.layers.dense({units: 100, inputShape: [10], activation: 'relu'}));
     * model.add(tf.layers.dense({units: 1, activation: 'sigmoid'}));
     *
     * model.summary();
     * ```
     *
     * @param lineLength Custom line length, in number of characters.
     * @param positions Custom widths of each of the columns, as either
     *   fractions of `lineLength` (e.g., `[0.5, 0.75, 1]`) or absolute number
     *   of characters (e.g., `[30, 50, 65]`). Each number corresponds to
     *   right-most (i.e., ending) position of a column.
     * @param printFn Custom print function. Can be used to replace the default
     *   `console.log`. For example, you can use `x => {}` to mute the printed
     *   messages in the console.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    summary(lineLength, positions, printFn = console.log) {
        if (!this.built) {
            this.build();
        }
        super.summary(lineLength, positions, printFn);
    }
    /**
     * Sets the weights of the model.
     *
     * @param weights Should be a list of Tensors with shapes and types matching
     *   the output of `model.getWeights()`.
     */
    setWeights(weights) {
        if (this.model == null) {
            this.build();
        }
        this.model.setWeights(weights);
    }
    /**
     * Returns the loss value & metrics values for the model in test mode.
     *
     * Loss and metrics are specified during `compile()`, which needs to happen
     * before calls to `evaluate()`.
     *
     * Computation is done in batches.
     *
     * ```js
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [10]})]
     * });
     * model.compile({optimizer: 'sgd', loss: 'meanSquaredError'});
     * const result = model.evaluate(tf.ones([8, 10]), tf.ones([8, 1]), {
     *   batchSize: 4,
     * });
     * result.print();
     * ```
     *
     * @param x `tf.Tensor` of test data, or an `Array` of `tf.Tensor`s if the
     * model has multiple inputs.
     * @param y `tf.Tensor` of target data, or an `Array` of `tf.Tensor`s if the
     * model has multiple outputs.
     * @param args A `ModelEvaluateConfig`, containing optional fields.
     *
     * @return `Scalar` test loss (if the model has a single output and no
     *   metrics) or `Array` of `Scalar`s (if the model has multiple outputs
     *   and/or metrics). The attribute `model.metricsNames`
     *   will give you the display labels for the scalar outputs.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    evaluate(x, y, args = {}) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before being used.');
        }
        return this.model.evaluate(x, y, args);
    }
    // TODO(cais): Add code snippet below once real dataset objects are
    //   available.
    /**
     * Evaluate model using a dataset object.
     *
     * Note: Unlike `evaluate()`, this method is asynchronous (`async`).
     *
     * @param dataset A dataset object. Its `iterator()` method is expected
     *   to generate a dataset iterator object, the `next()` method of which
     *   is expected to produce data batches for evaluation. The return value
     *   of the `next()` call ought to contain a boolean `done` field and a
     *   `value` field. The `value` field is expected to be an array of two
     *   `tf.Tensor`s or an array of two nested `tf.Tensor` structures. The former
     *   case is for models with exactly one input and one output (e.g.
     *   a sequential model). The latter case is for models with multiple
     *   inputs and/or multiple outputs. Of the two items in the array, the
     *   first is the input feature(s) and the second is the output target(s).
     * @param args A configuration object for the dataset-based evaluation.
     * @returns Loss and metric values as an Array of `Scalar` objects.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async evaluateDataset(dataset, args) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before being used.');
        }
        return this.model.evaluateDataset(dataset, args);
    }
    /**
     * Generates output predictions for the input samples.
     *
     * Computation is done in batches.
     *
     * Note: the "step" mode of predict() is currently not supported.
     *   This is because the TensorFlow.js core backend is imperative only.
     *
     * ```js
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [10]})]
     * });
     * model.predict(tf.ones([2, 10])).print();
     * ```
     *
     * @param x The input data, as a Tensor, or an `Array` of `tf.Tensor`s if
     *   the model has multiple inputs.
     * @param conifg A `ModelPredictConfig` object containing optional fields.
     *
     * @return `tf.Tensor`(s) of predictions.
     *
     * @exception ValueError In case of mismatch between the provided input data
     *   and the model's expectations, or in case a stateful model receives a
     *   number of samples that is not a multiple of the batch size.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    predict(x, args = {}) {
        if (this.model == null) {
            this.build();
        }
        return this.model.predict(x, args);
    }
    /**
     * Returns predictions for a single batch of samples.
     *
     * @param x: Input samples, as a Tensor, or list of Tensors (if the model
     *   has multiple inputs).
     * @return Tensor(s) of predictions
     */
    predictOnBatch(x) {
        if (this.model == null) {
            this.build();
        }
        return this.model.predictOnBatch(x);
    }
    /**
     * See `LayersModel.compile`.
     *
     * @param args
     */
    compile(args) {
        this.build();
        this.model.compile(args);
        this.optimizer_ = this.model.optimizer;
        // tslint:disable-next-line:no-any
        this.isOptimizerOwned = this.model.isOptimizerOwned;
        this.loss = this.model.loss;
        this.metrics = this.model.metrics;
        // TODO(cais): Add this.lossWeights, this.sampleWeightMode,
        //   this.weightedMetrics, this.targets.
        this.metricsTensors = this.model.metricsTensors;
        this.metricsNames = this.model.metricsNames;
        // TODO(cais): Add sampleWeights.
    }
    get optimizer() {
        return this.model == null ? undefined : this.model.optimizer;
    }
    set optimizer(optimizer) {
        this.model.optimizer = optimizer;
    }
    /**
     * Trains the model for a fixed number of epochs (iterations on a dataset).
     *
     * ```js
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [10]})]
     * });
     * model.compile({optimizer: 'sgd', loss: 'meanSquaredError'});
     * const history = await model.fit(tf.ones([8, 10]), tf.ones([8, 1]), {
     *   batchSize: 4,
     *   epochs: 3
     * });
     * console.log(history.history.loss[0]);
     * ```
     *
     * @param x `tf.Tensor` of training data, or an array of `tf.Tensor`s if the
     * model has multiple inputs. If all inputs in the model are named, you can
     * also pass a dictionary mapping input names to `tf.Tensor`s.
     * @param y `tf.Tensor` of target (label) data, or an array of `tf.Tensor`s if
     * the model has multiple outputs. If all outputs in the model are named, you
     *  can also pass a dictionary mapping output names to `tf.Tensor`s.
     * @param args  A `ModelFitConfig`, containing optional fields.
     *
     * @return A `History` instance. Its `history` attribute contains all
     *   information collected during training.
     *
     * @exception ValueError In case of mismatch between the provided input data
     *   and what the model expects.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async fit(x, y, args = {}) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before ' +
                'being used.');
        }
        return this.model.fit(x, y, args);
    }
    /**
     * Trains the model using a dataset object.
     *
     * ```js
     * const xArray = [
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     * ];
     * const yArray = [1, 1, 1, 1];
     * // Create a dataset from the JavaScript array.
     * const xDataset = tf.data.array(xArray);
     * const yDataset = tf.data.array(yArray);
     * // Zip combines the `x` and `y` Datasets into a single Dataset, the
     * // iterator of which will return an object containing of two tensors,
     * // corresponding to `x` and `y`.  The call to `batch(4)` will bundle
     * // four such samples into a single object, with the same keys now pointing
     * // to tensors that hold 4 examples, organized along the batch dimension.
     * // The call to `shuffle(4)` causes each iteration through the dataset to
     * // happen in a different order.  The size of the shuffle window is 4.
     * const xyDataset = tf.data.zip({xs: xDataset, ys: yDataset})
     *     .batch(4)
     *     .shuffle(4);
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [9]})]
     * });
     * model.compile({optimizer: 'sgd', loss: 'meanSquaredError'});
     * const history = await model.fitDataset(xyDataset, {
     *   epochs: 4,
     *   callbacks: {onEpochEnd: (epoch, logs) => console.log(logs.loss)}
     * });
     * ```
     *
     * @param dataset A dataset object. Its `iterator()` method is expected to
     *   generate a dataset iterator object, the `next()` method of which is
     *   expected to produce data batches for evaluation. The return value of the
     *   `next()` call ought to contain a boolean `done` field and a `value`
     *   field.
     *
     *   The `value` field is expected to be an object of with fields
     *   `xs` and `ys`, which point to the feature tensor and the target tensor,
     *   respectively. This case is for models with exactly one input and one
     *   output (e.g. a sequential model). For example:
     *   ```js
     *   {value: {xs: xsTensor, ys: ysTensor}, done: false}
     *   ```
     *
     *   If the model has multiple inputs, the `xs` field of `value` should
     *   be an object mapping input names to their respective feature tensors.
     *   For example:
     *   ```js
     *   {
     *     value: {
     *       xs: {
     *         input_1: xsTensor1,
     *         input_2: xsTensor2
     *       },
     *       ys: ysTensor
     *     },
     *     done: false
     *   }
     *   ```
     *   If the model has multiple outputs, the `ys` field of `value` should
     *   be an object mapping output names to their respective target tensors.
     *   For example:
     *   ```js
     *   {
     *     value: {
     *       xs: xsTensor,
     *       ys: {
     *         output_1: ysTensor1,
     *         output_2: ysTensor2
     *       },
     *     },
     *     done: false
     *   }
     *   ```
     * @param args A `ModelFitDatasetArgs`, containing optional fields.
     *
     * @return A `History` instance. Its `history` attribute contains all
     *   information collected during training.
     *
     * @doc {heading: 'Models', subheading: 'Classes', ignoreCI: true}
     */
    async fitDataset(dataset, args) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before ' +
                'being used.');
        }
        return this.model.fitDataset(dataset, args);
    }
    /**
     * Runs a single gradient update on a single batch of data.
     *
     * This method differs from `fit()` and `fitDataset()` in the following
     * regards:
     *   - It operates on exactly one batch of data.
     *   - It returns only the loss and metric values, instead of
     *     returning the batch-by-batch loss and metric values.
     *   - It doesn't support fine-grained options such as verbosity and
     *     callbacks.
     *
     * @param x Input data. It could be one of the following:
     *   - A `tf.Tensor`, or an Array of `tf.Tensor`s (in case the model has
     *     multiple inputs).
     *   - An Object mapping input names to corresponding `tf.Tensor` (if the
     *     model has named inputs).
     * @param y Target data. It could be either a `tf.Tensor` or multiple
     *   `tf.Tensor`s. It should be consistent with `x`.
     * @returns Training loss or losses (in case the model has
     *   multiple outputs), along with metrics (if any), as numbers.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async trainOnBatch(x, y) {
        return this.model.trainOnBatch(x, y);
    }
    /* See parent class for JsDoc */
    /** @nocollapse */
    static fromConfig(cls, config, customObjects = {}, fastWeightInit = false) {
        let configArray;
        let extraModelConfig = {};
        if (config instanceof Array) {
            if (!(config[0].className != null) ||
                config[0]['className'] === 'Merge') {
                throw new ValueError('Legacy serialization format not supported yet.');
            }
            configArray = config;
        }
        else {
            util.assert(config['layers'] != null, () => `When the config data for a Sequential model is not an Array, ` +
                `it must be an Object that contains the 'layers' field.`);
            configArray = config['layers'];
            delete config['layers'];
            extraModelConfig = config;
        }
        const model = new cls(extraModelConfig);
        if (!(model instanceof Sequential)) {
            throw new NotImplementedError(`Sequential.fromConfig called on non-Sequential input: ${model}`);
        }
        for (const conf of configArray) {
            const customObjects = undefined;
            const layer = deserialize(conf, customObjects, fastWeightInit);
            if (fastWeightInit) {
                layer.setFastWeightInitDuringBuild(true);
            }
            model.add(layer);
        }
        return model;
    }
    /**
     * Setter used for force stopping of LayersModel.fit() (i.e., training).
     *
     * Example:
     *
     * ```js
     * const model = tf.sequential();
     * model.add(tf.layers.dense({units: 1, inputShape: [10]}));
     * model.compile({loss: 'meanSquaredError', optimizer: 'sgd'});
     * const xs = tf.ones([8, 10]);
     * const ys = tf.zeros([8, 1]);
     *
     * const history = await model.fit(xs, ys, {
     *   epochs: 10,
     *   callbacks: {
     *     onEpochEnd: async (epoch, logs) => {
     *       if (epoch === 2) {
     *         model.stopTraining = true;
     *       }
     *     }
     *   }
     * });
     *
     * // There should be only 3 values in the loss array, instead of 10 values,
     * // due to the stopping after 3 epochs.
     * console.log(history.history.loss);
     * ```
     */
    set stopTraining(stop) {
        // TODO(cais): When refactoring to remove the composition pattern happens,
        // remove this method overriding.
        if (this.model == null) {
            throw new ValueError('Cannot set the stopTraining property of a sequential model before ' +
                'it is compiled.');
        }
        this.model.stopTraining = stop;
    }
    get stopTraining() {
        if (this.model == null) {
            throw new ValueError('Cannot get the stopTraining property of a sequential model before ' +
                'it is compiled.');
        }
        return this.model.stopTraining;
    }
    // TODO(cais): Override get trainableWeights() here
    // tslint:disable-next-line:no-any
    getConfig() {
        // NOTE(cais): We override the return type of getConfig() to `any` here,
        //   because the `Sequential` class is a special case among `Container`
        //   subtypes in that its getConfig() method returns an Array (not a
        //   dict).
        const layers = [];
        for (const layer of this.layers) {
            const dict = {};
            dict['className'] = layer.getClassName();
            dict['config'] = layer.getConfig();
            layers.push(dict);
        }
        return { name: this.name, layers };
    }
}
/** @nocollapse */
Sequential.className = 'Sequential';
serialization.registerClass(Sequential);
//# sourceMappingURL=data:application/json;base64,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