/**
 * @license
 * Copyright 2022 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('@tensorflow/tfjs-core')) :
    typeof define === 'function' && define.amd ? define(['exports', '@tensorflow/tfjs-core'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.tf = global.tf || {}, global.tf));
})(this, (function (exports, tfc) { 'use strict';

    function _interopNamespace(e) {
        if (e && e.__esModule) return e;
        var n = Object.create(null);
        if (e) {
            Object.keys(e).forEach(function (k) {
                if (k !== 'default') {
                    var d = Object.getOwnPropertyDescriptor(e, k);
                    Object.defineProperty(n, k, d.get ? d : {
                        enumerable: true,
                        get: function () { return e[k]; }
                    });
                }
            });
        }
        n["default"] = e;
        return n;
    }

    function _mergeNamespaces(n, m) {
        m.forEach(function (e) {
            Object.keys(e).forEach(function (k) {
                if (k !== 'default' && !(k in n)) {
                    var d = Object.getOwnPropertyDescriptor(e, k);
                    Object.defineProperty(n, k, d.get ? d : {
                        enumerable: true,
                        get: function () { return e[k]; }
                    });
                }
            });
        });
        return n;
    }

    var tfc__namespace = /*#__PURE__*/_interopNamespace(tfc);

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Explicit error types.
     *
     * See the following link for more information about why the code includes
     * calls to setPrototypeOf:
     *
     * https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
     */
    // tslint:enable
    /**
     * Equivalent of Python's AttributeError.
     */
    class AttributeError extends Error {
        constructor(message) {
            super(message);
            // Set the prototype explicitly.
            Object.setPrototypeOf(this, AttributeError.prototype);
        }
    }
    /**
     * Equivalent of Python's RuntimeError.
     */
    class RuntimeError extends Error {
        constructor(message) {
            super(message);
            // Set the prototype explicitly.
            Object.setPrototypeOf(this, RuntimeError.prototype);
        }
    }
    /**
     * Equivalent of Python's ValueError.
     */
    class ValueError extends Error {
        constructor(message) {
            super(message);
            // Set the prototype explicitly.
            Object.setPrototypeOf(this, ValueError.prototype);
        }
    }
    /**
     * Equivalent of Python's NotImplementedError.
     */
    class NotImplementedError extends Error {
        constructor(message) {
            super(message);
            // Set the prototype explicitly.
            Object.setPrototypeOf(this, NotImplementedError.prototype);
        }
    }
    /**
     * Equivalent of Python's AssertionError.
     */
    class AssertionError extends Error {
        constructor(message) {
            super(message);
            // Set the prototype explicitly.
            Object.setPrototypeOf(this, AssertionError.prototype);
        }
    }

    /**
     * @license
     * Copyright 2022 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * LruCache: A mapping from the String to T. If the number of the entries is
     * exceeding the `maxEntries`, the LruCache will delete the least recently
     * used entry.
     */
    class LruCache {
        constructor(maxEntries) {
            this.maxEntries = maxEntries || 100;
            this.cache = new Map();
        }
        /**
         * Get the entry for the key and mark it as used recently.
         */
        get(key) {
            let entry;
            if (this.cache.has(key)) {
                entry = this.cache.get(key);
                this.cache.delete(key);
                this.cache.set(key, entry);
            }
            return entry;
        }
        /**
         * Put the entry into the cache. If the key already existed, mark the key as
         * used recently.
         */
        put(key, value) {
            if (this.cache.has(key)) {
                this.cache.delete(key);
            }
            else if (this.cache.size >= this.maxEntries) {
                const keyToDelete = this.cache.keys().next().value;
                this.cache.delete(keyToDelete);
            }
            this.cache.set(key, value);
        }
        /**
         * Get the MaxEntries of the cache.
         */
        getMaxEntries() {
            return this.maxEntries;
        }
        /**
         * Set the MaxEntries of the cache. If the maxEntries is decreased, reduce
         * entries in the cache.
         */
        setMaxEntries(maxEntries) {
            if (maxEntries < 0) {
                throw new Error(`The maxEntries of LRU caches must be at least 0, but got ${maxEntries}.`);
            }
            if (this.maxEntries > maxEntries) {
                for (let i = 0; i < this.maxEntries - maxEntries; i++) {
                    const keyToDelete = this.cache.keys().next().value;
                    this.cache.delete(keyToDelete);
                }
            }
            this.maxEntries = maxEntries;
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    // tslint:enable
    /**
     * If `value` is an Array, equivalent to Python's `value * numValues`.
     * If `value` is not an Array, equivalent to Python's `[value] * numValues`
     */
    // tslint:disable-next-line:no-any
    function pyListRepeat(value, numValues) {
        if (Array.isArray(value)) {
            // tslint:disable-next-line:no-any
            let newArray = [];
            for (let i = 0; i < numValues; i++) {
                newArray = newArray.concat(value);
            }
            return newArray;
        }
        else {
            const newArray = new Array(numValues);
            newArray.fill(value);
            return newArray;
        }
    }
    function assert$1(val, message) {
        if (!val) {
            throw new AssertionError(message);
        }
    }
    /**
     * Count the number of elements of the `array` that are equal to `reference`.
     */
    function count(array, refernce) {
        let counter = 0;
        for (const item of array) {
            if (item === refernce) {
                counter++;
            }
        }
        return counter;
    }
    /**
     * If an array is of length 1, just return the first element. Otherwise, return
     * the full array.
     * @param tensors
     */
    function singletonOrArray(xs) {
        if (xs.length === 1) {
            return xs[0];
        }
        return xs;
    }
    /**
     * Normalizes a list/tensor into a list.
     *
     * If a tensor is passed, we return
     * a list of size 1 containing the tensor.
     *
     * @param x target object to be normalized.
     */
    // tslint:disable-next-line:no-any
    function toList(x) {
        if (Array.isArray(x)) {
            return x;
        }
        return [x];
    }
    /**
     * Converts string to snake-case.
     * @param name
     */
    function toSnakeCase(name) {
        const intermediate = name.replace(/(.)([A-Z][a-z0-9]+)/g, '$1_$2');
        const insecure = intermediate.replace(/([a-z])([A-Z])/g, '$1_$2').toLowerCase();
        /*
         If the class is private the name starts with "_" which is not secure
         for creating scopes. We prefix the name with "private" in this case.
         */
        if (insecure[0] !== '_') {
            return insecure;
        }
        return 'private' + insecure;
    }
    function toCamelCase(identifier) {
        // quick return for empty string or single character strings
        if (identifier.length <= 1) {
            return identifier;
        }
        // Check for the underscore indicating snake_case
        if (identifier.indexOf('_') === -1) {
            return identifier;
        }
        return identifier.replace(/[_]+(\w|$)/g, (m, p1) => p1.toUpperCase());
    }
    // tslint:disable-next-line:no-any
    let _GLOBAL_CUSTOM_OBJECTS = {};
    function serializeKerasObject(instance) {
        if (instance === null || instance === undefined) {
            return null;
        }
        const dict = {};
        dict['className'] = instance.getClassName();
        dict['config'] = instance.getConfig();
        return dict;
    }
    /**
     * Replace ndarray-style scalar objects in serialization objects with numbers.
     *
     * Background: In some versions of tf.keras, certain scalar values in the HDF5
     * model save file can be serialized as: `{'type': 'ndarray', 'value': num}`,
     * where in `num` is a plain number. This method converts such serialization
     * to a `number`.
     *
     * @param config The keras-format serialization object to be processed
     *   (in place).
     */
    function convertNDArrayScalarsInConfig(config) {
        if (config == null || typeof config !== 'object') {
            return;
        }
        else if (Array.isArray(config)) {
            config.forEach(configItem => convertNDArrayScalarsInConfig(configItem));
        }
        else {
            const fields = Object.keys(config);
            for (const field of fields) {
                const value = config[field];
                if (value != null && typeof value === 'object') {
                    if (!Array.isArray(value) && value['type'] === 'ndarray' &&
                        typeof value['value'] === 'number') {
                        config[field] = value['value'];
                    }
                    else {
                        convertNDArrayScalarsInConfig(value);
                    }
                }
            }
        }
    }
    /**
     * Deserialize a saved Keras Object
     * @param identifier either a string ID or a saved Keras dictionary
     * @param moduleObjects a list of Python class names to object constructors
     * @param customObjects a list of Python class names to object constructors
     * @param printableModuleName debug text for the object being reconstituted
     * @param fastWeightInit Optional flag to use fast weight initialization
     *   during deserialization. This is applicable to cases in which
     *   the initialization will be immediately overwritten by loaded weight
     *   values. Default: `false`.
     * @returns a TensorFlow.js Layers object
     */
    // tslint:disable:no-any
    function deserializeKerasObject(identifier, moduleObjects = {}, customObjects = {}, printableModuleName = 'object', fastWeightInit = false) {
        // tslint:enable
        if (typeof identifier === 'string') {
            const functionName = identifier;
            let fn;
            if (functionName in customObjects) {
                fn = customObjects[functionName];
            }
            else if (functionName in _GLOBAL_CUSTOM_OBJECTS) {
                fn = _GLOBAL_CUSTOM_OBJECTS[functionName];
            }
            else {
                fn = moduleObjects[functionName];
                if (fn == null) {
                    throw new ValueError(`Unknown ${printableModuleName}: ${identifier}. ` +
                        `This may be due to one of the following reasons:\n` +
                        `1. The ${printableModuleName} is defined in Python, in which ` +
                        `case it needs to be ported to TensorFlow.js or your JavaScript ` +
                        `code.\n` +
                        `2. The custom ${printableModuleName} is defined in JavaScript, ` +
                        `but is not registered properly with ` +
                        `tf.serialization.registerClass().`);
                    // TODO(cais): Add link to tutorial page on custom layers.
                }
            }
            return fn;
        }
        else {
            // In this case we are dealing with a Keras config dictionary.
            const config = identifier;
            if (config['className'] == null || config['config'] == null) {
                throw new ValueError(`${printableModuleName}: Improper config format: ` +
                    `${JSON.stringify(config)}.\n` +
                    `'className' and 'config' must set.`);
            }
            const className = config['className'];
            let cls, fromConfig;
            if (className in customObjects) {
                [cls, fromConfig] = customObjects[className];
            }
            else if (className in _GLOBAL_CUSTOM_OBJECTS) {
                [cls, fromConfig] = _GLOBAL_CUSTOM_OBJECTS['className'];
            }
            else if (className in moduleObjects) {
                [cls, fromConfig] = moduleObjects[className];
            }
            if (cls == null) {
                throw new ValueError(`Unknown ${printableModuleName}: ${className}. ` +
                    `This may be due to one of the following reasons:\n` +
                    `1. The ${printableModuleName} is defined in Python, in which ` +
                    `case it needs to be ported to TensorFlow.js or your JavaScript ` +
                    `code.\n` +
                    `2. The custom ${printableModuleName} is defined in JavaScript, ` +
                    `but is not registered properly with ` +
                    `tf.serialization.registerClass().`);
                // TODO(cais): Add link to tutorial page on custom layers.
            }
            if (fromConfig != null) {
                // Porting notes: Instead of checking to see whether fromConfig accepts
                // customObjects, we create a customObjects dictionary and tack it on to
                // config['config'] as config['config'].customObjects. Objects can use it,
                // if they want.
                // tslint:disable-next-line:no-any
                const customObjectsCombined = {};
                for (const key of Object.keys(_GLOBAL_CUSTOM_OBJECTS)) {
                    customObjectsCombined[key] = _GLOBAL_CUSTOM_OBJECTS[key];
                }
                for (const key of Object.keys(customObjects)) {
                    customObjectsCombined[key] = customObjects[key];
                }
                // Add the customObjects to config
                const nestedConfig = config['config'];
                nestedConfig['customObjects'] = customObjectsCombined;
                const backupCustomObjects = Object.assign({}, _GLOBAL_CUSTOM_OBJECTS);
                for (const key of Object.keys(customObjects)) {
                    _GLOBAL_CUSTOM_OBJECTS[key] = customObjects[key];
                }
                convertNDArrayScalarsInConfig(config['config']);
                const returnObj = fromConfig(cls, config['config'], customObjects, fastWeightInit);
                _GLOBAL_CUSTOM_OBJECTS = Object.assign({}, backupCustomObjects);
                return returnObj;
            }
            else {
                // Then `cls` may be a function returning a class.
                // In this case by convention `config` holds
                // the kwargs of the function.
                const backupCustomObjects = Object.assign({}, _GLOBAL_CUSTOM_OBJECTS);
                for (const key of Object.keys(customObjects)) {
                    _GLOBAL_CUSTOM_OBJECTS[key] = customObjects[key];
                }
                // In python this is **config['config'], for tfjs-layers we require
                // classes that use this fall-through construction method to take
                // a config interface that mimics the expansion of named parameters.
                const returnObj = new cls(config['config']);
                _GLOBAL_CUSTOM_OBJECTS = Object.assign({}, backupCustomObjects);
                return returnObj;
            }
        }
    }
    /**
     * Compares two numbers for sorting.
     * @param a
     * @param b
     */
    function numberCompare(a, b) {
        return (a < b) ? -1 : ((a > b) ? 1 : 0);
    }
    /**
     * Comparison of two numbers for reverse sorting.
     * @param a
     * @param b
     */
    function reverseNumberCompare(a, b) {
        return -1 * numberCompare(a, b);
    }
    /**
     * Get the unique elements of an array.
     * @param xs Array.
     * @returns An Array consisting of the unique elements in `xs`.
     */
    function unique(xs) {
        if (xs == null) {
            return xs;
        }
        const out = [];
        // TODO(cais): Maybe improve performance by sorting.
        for (const x of xs) {
            if (out.indexOf(x) === -1) {
                out.push(x);
            }
        }
        return out;
    }
    /**
     * Determine if an Object is empty (i.e., does not have own properties).
     * @param obj Object
     * @returns Whether the Object is empty.
     * @throws ValueError: If object is `null` or `undefined`.
     */
    function isObjectEmpty(obj) {
        if (obj == null) {
            throw new ValueError(`Invalid value in obj: ${JSON.stringify(obj)}`);
        }
        for (const key in obj) {
            if (obj.hasOwnProperty(key)) {
                return false;
            }
        }
        return true;
    }
    /**
     * Helper function used to build type union/enum run-time checkers.
     * @param values The list of allowed values.
     * @param label A string name for the type
     * @param value The value to test.
     * @throws ValueError: If the value is not in values nor `undefined`/`null`.
     */
    function checkStringTypeUnionValue(values, label, value) {
        if (value == null) {
            return;
        }
        if (values.indexOf(value) < 0) {
            throw new ValueError(`${value} is not a valid ${label}.  Valid values are ${values} or null/undefined.`);
        }
    }
    /**
     * Helper function for verifying the types of inputs.
     *
     * Ensures that the elements of `x` are all of type `expectedType`.
     * Also verifies that the length of `x` is within bounds.
     *
     * @param x Object to test.
     * @param expectedType The string expected type of all of the elements in the
     * Array.
     * @param minLength Return false if x.length is less than this.
     * @param maxLength Return false if x.length is greater than this.
     * @returns true if and only if `x` is an `Array<expectedType>` with
     * length >= `minLength` and <= `maxLength`.
     */
    // tslint:disable:no-any
    function checkArrayTypeAndLength(x, expectedType, minLength = 0, maxLength = Infinity) {
        assert$1(minLength >= 0);
        assert$1(maxLength >= minLength);
        return (Array.isArray(x) && x.length >= minLength && x.length <= maxLength &&
            x.every(e => typeof e === expectedType));
    }
    // tslint:enable:no-any
    /**
     * Assert that a value or an array of value are positive integer.
     *
     * @param value The value being asserted on. May be a single number or an array
     *   of numbers.
     * @param name Name of the value, used to make the error message.
     */
    function assertPositiveInteger(value, name) {
        if (Array.isArray(value)) {
            tfc.util.assert(value.length > 0, () => `${name} is unexpectedly an empty array.`);
            value.forEach((v, i) => assertPositiveInteger(v, `element ${i + 1} of ${name}`));
        }
        else {
            tfc.util.assert(Number.isInteger(value) && value > 0, () => `Expected ${name} to be a positive integer, but got ` +
                `${formatAsFriendlyString(value)}.`);
        }
    }
    /**
     * Format a value into a display-friendly, human-readable fashion.
     *
     * - `null` is formatted as `'null'`
     * - Strings are formated with flanking pair of quotes.
     * - Arrays are formatted with flanking pair of square brackets.
     *
     * @param value The value to display.
     * @return Formatted string.
     */
    // tslint:disable-next-line:no-any
    function formatAsFriendlyString(value) {
        if (value === null) {
            return 'null';
        }
        else if (Array.isArray(value)) {
            return '[' + value.map(v => formatAsFriendlyString(v)).join(',') + ']';
        }
        else if (typeof value === 'string') {
            return `"${value}"`;
        }
        else {
            return `${value}`;
        }
    }
    /**
     * Returns a function `f2` (decorator) which wraps the original function
     * `f`. `f2` guarantees that `f` can be called at most once
     * every `waitMs` ms. If `f2` is called more often, it will return
     * the last returned result of `f`.
     *
     * @param f The original function `f` to wrap.
     * @param waitMs The time between two consecutive calls to `f` in ms.
     */
    function debounce(f, waitMs, nowFunc) {
        let lastTime = nowFunc != null ? nowFunc() : tfc.util.now();
        let lastResult;
        const f2 = (...args) => {
            const now = nowFunc != null ? nowFunc() : tfc.util.now();
            if (now - lastTime < waitMs) {
                return lastResult;
            }
            lastTime = now;
            lastResult = f(...args);
            return lastResult;
        };
        return f2;
    }
    /**
     * Returns the fusable activation given a layers identifier.
     *
     * @param activationName The layers identifier string.
     * @return The name of the fusable activation.
     */
    function mapActivationToFusedKernel(activationName) {
        if (activationName === 'relu') {
            return 'relu';
        }
        if (activationName === 'linear') {
            return 'linear';
        }
        if (activationName === 'elu') {
            return 'elu';
        }
        return null;
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Utilities related to persistent state in the backend.
     */
    /**
     * An ID to track `tf.SymbolicTensor`s and derived classes.
     * Required in different places in engine/topology.ts to identify unique
     * tensors.
     */
    let _nextUniqueTensorId = 0;
    function getNextUniqueTensorId() {
        return _nextUniqueTensorId++;
    }
    const _uidPrefixes = {};
    /**
     * Provides a unique UID given a string prefix.
     *
     * @param prefix
     */
    function getUid(prefix = '') {
        if (!(prefix in _uidPrefixes)) {
            _uidPrefixes[prefix] = 0;
        }
        _uidPrefixes[prefix] += 1;
        return prefix + _uidPrefixes[prefix].toString();
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    const VALID_DATA_FORMAT_VALUES = ['channelsFirst', 'channelsLast'];
    const VALID_INTERPOLATION_FORMAT_VALUES = ['nearest', 'bilinear'];
    const VALID_PADDING_MODE_VALUES = ['valid', 'same', 'causal'];
    const VALID_POOL_MODE_VALUES = ['max', 'avg'];
    const VALID_BIDIRECTIONAL_MERGE_MODES = ['sum', 'mul', 'concat', 'ave'];

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    // A map from the requested scoped name of a Tensor to the number of Tensors
    // wanting that name so far.  This allows enforcing name uniqueness by appending
    // an incrementing index, e.g. scope/name, scope/name_1, scope/name_2, etc.
    const nameMap = new Map();
    function checkDataFormat(value) {
        checkStringTypeUnionValue(VALID_DATA_FORMAT_VALUES, 'DataFormat', value);
    }
    function checkInterpolationFormat(value) {
        checkStringTypeUnionValue(VALID_INTERPOLATION_FORMAT_VALUES, 'InterpolationFormat', value);
    }
    function checkPaddingMode(value) {
        checkStringTypeUnionValue(VALID_PADDING_MODE_VALUES, 'PaddingMode', value);
    }
    function checkPoolMode(value) {
        checkStringTypeUnionValue(VALID_POOL_MODE_VALUES, 'PoolMode', value);
    }
    const _nameScopeStack = [];
    const _nameScopeDivider = '/';
    /**
     * Enter namescope, which can be nested.
     */
    function nameScope(name, fn) {
        _nameScopeStack.push(name);
        try {
            const val = fn();
            _nameScopeStack.pop();
            return val;
        }
        catch (e) {
            _nameScopeStack.pop();
            throw e;
        }
    }
    /**
     * Get the current namescope as a flat, concatenated string.
     */
    function currentNameScopePrefix() {
        if (_nameScopeStack.length === 0) {
            return '';
        }
        else {
            return _nameScopeStack.join(_nameScopeDivider) + _nameScopeDivider;
        }
    }
    /**
     * Get the name a Tensor (or Variable) would have if not uniqueified.
     * @param tensorName
     * @return Scoped name string.
     */
    function getScopedTensorName(tensorName) {
        if (!isValidTensorName(tensorName)) {
            throw new Error('Not a valid tensor name: \'' + tensorName + '\'');
        }
        return currentNameScopePrefix() + tensorName;
    }
    /**
     * Get unique names for Tensors and Variables.
     * @param scopedName The fully-qualified name of the Tensor, i.e. as produced by
     *  `getScopedTensorName()`.
     * @return A unique version of the given fully scoped name.
     *   If this is the first time that the scoped name is seen in this session,
     *   then the given `scopedName` is returned unaltered.  If the same name is
     *   seen again (producing a collision), an incrementing suffix is added to the
     *   end of the name, so it takes the form 'scope/name_1', 'scope/name_2', etc.
     */
    function getUniqueTensorName(scopedName) {
        if (!isValidTensorName(scopedName)) {
            throw new Error('Not a valid tensor name: \'' + scopedName + '\'');
        }
        if (!nameMap.has(scopedName)) {
            nameMap.set(scopedName, 0);
        }
        const index = nameMap.get(scopedName);
        nameMap.set(scopedName, nameMap.get(scopedName) + 1);
        if (index > 0) {
            const result = `${scopedName}_${index}`;
            // Mark the composed name as used in case someone wants
            // to call getUniqueTensorName("name_1").
            nameMap.set(result, 1);
            return result;
        }
        else {
            return scopedName;
        }
    }
    const tensorNameRegex = new RegExp(/^[A-Za-z0-9][-A-Za-z0-9\._\/]*$/);
    /**
     * Determine whether a string is a valid tensor name.
     * @param name
     * @returns A Boolean indicating whether `name` is a valid tensor name.
     */
    function isValidTensorName(name) {
        return !!name.match(tensorNameRegex);
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Determine if a number is an integer.
     */
    function isInteger(x) {
        return x === parseInt(x.toString(), 10);
    }
    /**
     * Calculate the product of an array of numbers.
     * @param array The array to calculate the product over.
     * @param begin Beginning index, inclusive.
     * @param end Ending index, exclusive.
     * @return The product.
     */
    function arrayProd(array, begin, end) {
        if (begin == null) {
            begin = 0;
        }
        if (end == null) {
            end = array.length;
        }
        let prod = 1;
        for (let i = begin; i < end; ++i) {
            prod *= array[i];
        }
        return prod;
    }
    /**
     * Compute minimum value.
     * @param array
     * @return minimum value.
     */
    function min$1(array) {
        // same behavior as tf.min()
        if (array.length === 0) {
            return Number.NaN;
        }
        let min = Number.POSITIVE_INFINITY;
        for (let i = 0; i < array.length; i++) {
            const value = array[i];
            if (value < min) {
                min = value;
            }
        }
        return min;
    }
    /**
     * Compute maximum value.
     * @param array
     * @return maximum value
     */
    function max$1(array) {
        // same behavior as tf.max()
        if (array.length === 0) {
            return Number.NaN;
        }
        let max = Number.NEGATIVE_INFINITY;
        for (let i = 0; i < array.length; i++) {
            const value = array[i];
            if (value > max) {
                max = value;
            }
        }
        return max;
    }
    /**
     * Generate an array of integers in [begin, end).
     * @param begin Beginning integer, inclusive.
     * @param end Ending integer, exclusive.
     * @returns Range array.
     * @throws ValueError, iff `end` < `begin`.
     */
    function range$1(begin, end) {
        if (end < begin) {
            throw new ValueError(`end (${end}) < begin (${begin}) is forbidden.`);
        }
        const out = [];
        for (let i = begin; i < end; ++i) {
            out.push(i);
        }
        return out;
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    let _epsilon;
    /**
     * Returns the value of the fuzz factor used in numeric expressions.
     */
    function epsilon() {
        if (_epsilon == null) {
            _epsilon = tfc.backend().epsilon();
        }
        return _epsilon;
    }
    /**
     * Returns the default image data format convention.
     */
    function imageDataFormat() {
        return 'channelsLast';
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Casts a tensor to a different dtype and returns it.
     * @param x Input tensor.
     * @param dtype String: 'float32'|'int32'|'bool'.
     * @returns Tensor of the specified `dtype`.
     */
    function cast$1(x, dtype) {
        return tfc__namespace.cast(x, dtype);
    }
    /**
     * Adds a 1-sized dimension at index "axis".
     * @param x Input tensor.
     * @param axis Position where to add the new axis.
     * @returns Result of the dimension expansion.
     */
    function expandDims$1(x, axis = -1) {
        const outShape = x.shape.slice();
        if (axis < 0) {
            axis = outShape.length + axis + 1;
        }
        outShape.splice(axis, 0, 1);
        return tfc__namespace.reshape(x, outShape);
    }
    /**
     * Repeats a 2D tensor.
     *
     * If `x` has shape `[samples, dim]` and `n` is 2, for example, the output
     * will have shape `[samples, 2, dim]`.
     *
     * @param x Input tensor.
     * @param n Integer, number of times to repeat.
     * @returns The result of the repeat operation.
     * @throws ValueError: If input tensor is not 2D.
     */
    function repeat(x, n) {
        return tfc.tidy(() => {
            if (x.shape.length !== 2) {
                throw new ValueError(`repeat() expects a rank-2 tensor, but received a ` +
                    `rank-${x.shape.length} tensor.`);
            }
            const y = expandDims$1(x, 1);
            return tile$1(y, [1, n, 1]);
        });
    }
    /**
     * Flatten a Tensor into 1D.
     * @param x Input tensor.
     * @return The result of the flattening `x`.
     */
    function flatten$2(x) {
        const newShape = [arrayProd(x.shape)];
        return tfc__namespace.reshape(x, newShape);
    }
    /**
     * Turn a nD tensor into a 2D tensor with same 0th dimension.
     * In other words, it flattens each data samples of a batch.
     *
     * @param x The tensor to flatten. The rank of this tensor is required to be 2
     *   or higher.
     * @return The result of the flattening.
     */
    function batchFlatten(x) {
        if (x.rank <= 1) {
            throw new ValueError(`batchFlatten requires a minimum rank of 2. Got rank: ${x.rank}.`);
        }
        const newShape = [x.shape[0], arrayProd(x.shape, 1)];
        return tfc__namespace.reshape(x, newShape);
    }
    /**
     * Do slicing along the first axis.
     * @param array input `tf.Tensor`.
     * @param start starting index, inclusive.
     * @param size size of the slice along the first axis.
     * @returns result of the slicing.
     * @throws ValueError: If `array` is of an unsupported subtype of `tf.Tensor`.
     */
    function sliceAlongFirstAxis(array, start, size) {
        return tfc.tidy(() => {
            switch (array.rank) {
                case 1:
                    return tfc__namespace.slice1d(array, start, size);
                case 2:
                    return tfc__namespace.slice2d(array, [start, 0], [size, array.shape[1]]);
                case 3:
                    return tfc__namespace.slice3d(array, [start, 0, 0], [size, array.shape[1], array.shape[2]]);
                case 4:
                    return tfc__namespace.slice4d(array, [start, 0, 0, 0], [size, array.shape[1], array.shape[2], array.shape[3]]);
                case 5:
                    return tfc__namespace.slice(array, [start, 0, 0, 0, 0], [
                        size, array.shape[1], array.shape[2], array.shape[3], array.shape[4]
                    ]);
                case 6:
                    return tfc__namespace.slice(array, [start, 0, 0, 0, 0, 0], [
                        size, array.shape[1], array.shape[2], array.shape[3], array.shape[4],
                        array.shape[5]
                    ]);
                default:
                    throw new ValueError(`sliceAlongFirstAxis() received an unsupported tensor rank: ` +
                        `${array.rank}`);
            }
        });
    }
    /**
     * Do slicing along the last axis.
     * @param array input `tf.Tensor`.
     * @param start starting index, inclusive.
     * @param size size of the slice along the last axis.
     * @returns result of the slicing.
     * @throws ValueError: If `array` is of an unsupported subtype of `tf.Tensor`.
     */
    function sliceAlongLastAxis(array, start, size) {
        return tfc.tidy(() => {
            switch (array.rank) {
                case 1:
                    return tfc__namespace.slice1d(array, start, size);
                case 2:
                    return tfc__namespace.slice2d(array, [0, start], [array.shape[0], size]);
                case 3:
                    return tfc__namespace.slice3d(array, [0, 0, start], [array.shape[0], array.shape[1], size]);
                case 4:
                    return tfc__namespace.slice4d(array, [0, 0, 0, start], [array.shape[0], array.shape[1], array.shape[2], size]);
                default:
                    throw new ValueError(`sliceAlongLastAxis() received an unsupported tensor rank: ` +
                        `${array.rank}`);
            }
        });
    }
    /**
     * Do slicing along the sepcified axis.
     * @param array input `tf.Tensor`.
     * @param start starting index, inclusive.
     * @param size of the slice along the chosen axis.
     * @param choose an axis.
     * @returns result of the slicing.
     * @throws ValueError: If `array` is of an unsupported subtype of `tf.Tensor`.
     */
    function sliceAlongAxis(array, start, size, axis) {
        return tfc.tidy(() => {
            switch (array.rank) {
                case 1:
                    return tfc__namespace.slice1d(array, start, size);
                case 2:
                    switch (axis) {
                        case 1:
                            return sliceAlongFirstAxis(array, start, size);
                        case 2:
                            return sliceAlongLastAxis(array, start, size);
                        default:
                            throw new ValueError(`The axis is not within the rank of the tensor ` +
                                `${axis}`);
                    }
                case 3:
                    switch (axis) {
                        case 1:
                            return sliceAlongFirstAxis(array, start, size);
                        case 2:
                            return tfc__namespace.slice3d(array, [0, start, 0], [array.shape[0], size, array.shape[2]]);
                        case 3:
                            return sliceAlongLastAxis(array, start, size);
                        default:
                            throw new ValueError(`The axis is not within the rank of the tensor ` +
                                `${axis}`);
                    }
                case 4:
                    switch (axis) {
                        case 1:
                            return sliceAlongFirstAxis(array, start, size);
                        case 2:
                            return tfc__namespace.slice4d(array, [0, start, 0, 0], [array.shape[0], size, array.shape[2], array.shape[3]]);
                        case 3:
                            return tfc__namespace.slice4d(array, [0, 0, start, 0], [array.shape[0], array.shape[1], size, array.shape[3]]);
                        case 4:
                            return sliceAlongLastAxis(array, start, size);
                        default:
                            throw new ValueError(`The axis is not within the rank of the tensor ` +
                                `${axis}`);
                    }
                default:
                    throw new ValueError(`sliceAlongLastAxis() received an unsupported tensor rank: ` +
                        `${array.rank}`);
            }
        });
    }
    /**
     * Concatenates a list of tensors alongside the specified axis.
     * @param tensors `Array` of tensors to concatenate.
     * @param axis Concatenation axis.
     * @returns The result of the concatenation.
     */
    function concatenate$1(tensors, axis = -1) {
        let rank;
        if (axis < 0) {
            rank = tensors[0].rank;
            if (rank !== 0) {
                axis = rank;
            }
            else {
                axis = 0;
            }
        }
        if (axis === tensors[0].rank) {
            // Porting Note: This is necessary because tfc.concat() requires axis to be
            //   in the interval [-rank, rank).
            axis = -1;
        }
        // Porting Note: Sparse concat is not supported yet.
        return tfc__namespace.concat(tensors, axis);
    }
    /**
     * Concatenate two arrays along the first dimension.
     * @param a The 1st `tf.Tensor` to concatenate.
     * @param b The 2nd `tf.Tensor` to concatenate.
     * @returns Result of the concatenation.
     * @throws ValueError: If `a` is of an unsupported subtype of `tf.Tensor`.
     */
    function concatAlongFirstAxis(a, b) {
        switch (a.rank) {
            case 1:
                return tfc__namespace.concat1d([a, b]);
            case 2:
                return tfc__namespace.concat2d([a, b], 0);
            case 3:
                return tfc__namespace.concat3d([a, b], 0);
            case 4:
                return tfc__namespace.concat4d([a, b], 0);
            default:
                throw new ValueError(`concatAlongFirstAxis() received an unsupported ` +
                    `tensor rank: ${a.rank}`);
        }
    }
    /**
     * Creates a tensor by tiling `x` by `n`.
     * @param x A tensor.
     * @param n An Array of integers or a single integer. If an Array, the length
     *   must be the same as the number of dimensions in `x`. If a single integer,
     *   it will be treated as an Array of length 1.
     */
    function tile$1(x, n) {
        if (!Array.isArray(n)) {
            n = [n];
        }
        if (x.rank !== n.length) {
            throw new ValueError(`The length of input n (${n.length}) does not match ` +
                `the number of dimensions in input x (${x.rank})`);
        }
        return tfc__namespace.tile(x, n);
    }
    /* Creation of random tensors. */
    /**
     * Get a tensor with normal distribution of values.
     *
     * @param shape Shape of the tensor.
     * @param mean mean value of the normal distribution.
     * @param stddev standard deviation of the normal distribution.
     * @param dtype
     * @param seed
     * @return The normal tensor.
     */
    function randomNormal$2(shape, mean = 0.0, stddev = 1.0, dtype, seed) {
        return tfc__namespace.randomNormal(shape, mean, stddev, dtype, seed);
    }
    /* Linear Algebra */
    /**
     * Multiply two tensors and returns the result as a tensor.
     *
     * For 2D tensors, this is equivalent to matrix multiplication (matMul).
     * For tensors of higher ranks, it follows the Theano behavior,
     * (e.g. `(2, 3) * (4, 3, 5) -> (2, 4, 5)`).  From the Theano documentation:
     *
     * For N dimensions it is a sum product over the last axis of x and the
     * second-to-last of y:
     *
     * @param a A tensor of at least rank 2.
     * @param b A tensor of at least rank 2.
     * @param activation (optional) A string identifying the activation
     *   function.
     * @return Result of the dot operation.
     */
    function dot$1(a, b, activation, bias) {
        if ((a.rank < 2) || (b.rank < 2)) {
            throw new NotImplementedError(`dot requires both inputs to be rank >= 2` +
                ` but got x shape = ${a.shape} and y shape = ${b.shape}`);
        }
        if (b.rank >= 3) {
            const xLastDim = a.shape.slice(-1)[0];
            const ySecondLastDim = b.shape.slice(-2)[0];
            if (xLastDim !== ySecondLastDim) {
                throw new NotImplementedError(`If rank y >= 3, then the second last dim` +
                    ` of y must equal the last dim of x but got x shape = ${a.shape} and ` +
                    ` y shape = ${b.shape}`);
            }
        }
        // Handle basic 2D x 2D case.
        if ((a.rank === 2) && (b.rank === 2)) {
            const transposeA = false;
            const transposeB = false;
            // tfc.fused.matMul only fuses certain activation functions. Unsupported
            // activation functions are treated as 'linear' activations, which is
            // equivalent to a no-op.
            return tfc__namespace.fused.matMul({
                a,
                b: b,
                transposeA,
                transposeB,
                bias: bias ? reshapeBias(a.rank, bias, imageDataFormat()) : null,
                activation
            });
        }
        else {
            // Reshape x into the analogous 2D Tensor.
            const aFirstDims = a.shape.slice(); // Holds all but the last dim of x.
            const aLastDim = aFirstDims.pop();
            a = tfc__namespace.reshape(a, [-1, aLastDim]);
            // Reshape y into the analogous 2D Tensor, and keep track of the
            // required dimensions to reproduce the output shape.
            const bShape = b.shape.slice();
            const bLastDim = bShape.pop();
            const ySecondLastDim = bShape.pop();
            const yOtherDims = [...bShape, bLastDim];
            // permutation should be like [r-2, 0, 1, 2, ... r-4, r-3, r-1]
            // where r is the rank of y.
            const perm = Array.from({ length: b.rank }, (_, i) => {
                if (i === 0) {
                    return b.rank - 2;
                }
                else if (i <= b.rank - 2) {
                    return i - 1;
                }
                return i;
            });
            b = tfc__namespace.reshape(tfc__namespace.transpose(b, perm), [ySecondLastDim, -1]);
            // Multiply x and y as 2D Tensors, and then reshape back to original.
            const outputShape = [...aFirstDims, ...yOtherDims];
            const transposeA = false;
            const transposeB = false;
            return tfc__namespace.reshape(tfc__namespace.fused.matMul({
                a,
                b,
                transposeA,
                transposeB,
                bias: bias ? reshapeBias(a.rank, bias, imageDataFormat()) : null,
                activation
            }), outputShape);
        }
    }
    /* Elementary math functions. */
    /**
     * Retrieves the elements of indices `indices` in the tensor `reference`.
     * @param reference A tensor.
     * @param indices An integer tensor of indices or an `Array` of integers.
     * @param axis Axis along which to perform the gather operation.
     * @returns The result of the gathering as a tensor.
     */
    function gather$1(reference, indices, axis) {
        return tfc.tidy(() => {
            if (Array.isArray(indices)) {
                indices = tfc.tensor1d(indices, 'int32');
            }
            else {
                indices = tfc__namespace.cast(indices, 'int32');
            }
            return tfc__namespace.gather(reference, indices, axis);
        });
    }
    /**
     * Element-wise square.
     * @param x Input tensor.
     * @return element-wise x^2
     */
    function square$1(x) {
        return tfc__namespace.mul(x, x);
    }
    /**
     * Reshapes bias tensor according to rank of x.
     */
    function reshapeBias(xRank, bias, dataFormat) {
        const biasShape = bias.shape;
        if (bias.rank !== 1 && bias.rank !== xRank) {
            throw new ValueError(`Unexpected bias dimensions: ${bias.rank}` +
                `; expected it to be 1 or ${xRank}`);
        }
        if (xRank === 5) {
            if (dataFormat === 'channelsFirst') {
                if (biasShape.length === 1) {
                    return tfc__namespace.reshape(bias, [1, biasShape[0], 1, 1, 1]);
                }
                else {
                    return tfc__namespace.reshape(bias, [1, biasShape[3], biasShape[0], biasShape[1], biasShape[2]]);
                }
            }
            else if (dataFormat === 'channelsLast') {
                if (biasShape.length === 1) {
                    return tfc__namespace.reshape(bias, [1, 1, 1, 1, biasShape[0]]);
                }
                else {
                    return tfc__namespace.reshape(bias, [1].concat(biasShape));
                }
            }
        }
        else if (xRank === 4) {
            if (dataFormat === 'channelsFirst') {
                if (biasShape.length === 1) {
                    return tfc__namespace.reshape(bias, [1, biasShape[0], 1, 1]);
                }
                else {
                    return tfc__namespace.reshape(bias, [1, biasShape[2], biasShape[0], biasShape[1]]);
                }
            }
            else if (dataFormat === 'channelsLast') {
                if (biasShape.length === 1) {
                    return tfc__namespace.reshape(bias, [1, 1, 1, biasShape[0]]);
                }
                else {
                    return tfc__namespace.reshape(bias, [1].concat(biasShape));
                }
            }
        }
        else if (xRank === 3) {
            if (dataFormat === 'channelsFirst') {
                if (biasShape.length === 1) {
                    return tfc__namespace.reshape(bias, [1, biasShape[0], 1]);
                }
                else {
                    return tfc__namespace.reshape(bias, [1, biasShape[1], biasShape[0]]);
                }
            }
            else if (dataFormat === 'channelsLast') {
                if (biasShape.length === 1) {
                    return tfc__namespace.reshape(bias, [1, 1, biasShape[0]]);
                }
                else {
                    return tfc__namespace.reshape(bias, [1].concat(biasShape));
                }
            }
        }
        else if (xRank < 3) {
            return bias;
        }
        throw new ValueError(`Unsupported input rank by biasAdd: ${bias.rank}`);
    }
    /* Neural-network operations. */
    /**
     * Add a bias to a tensor.
     *
     * @param x The tensor to add the bias to.
     * @param bias The bias to add to `x`. Must be 1D or the same rank as `x`.
     * @return Result of the bias adding.
     * @throws ValueError: If the rank of `bias` is incorrect.
     */
    function biasAdd(x, bias, dataFormat) {
        return tfc.tidy(() => {
            if (dataFormat == null) {
                dataFormat = imageDataFormat();
            }
            checkDataFormat(dataFormat);
            return tfc__namespace.add(x, reshapeBias(x.rank, bias, dataFormat));
        });
    }
    /**
     * Exponential linear unit (ELU).
     * @param x A tensor or variable to compute the activation function for.
     * @param alpha: A scalar, a scaling factor for the negative section.
     * @return Output of the ELU operation.
     */
    function elu$2(x, alpha = 1) {
        // TODO(cais): Add support for alpha values other than 1.
        if (alpha !== 1) {
            throw new NotImplementedError(`Support for alpha values other than 1 (${alpha}) is not implemented ` +
                `yet.`);
        }
        return tfc__namespace.elu(x);
    }
    /**
     * Softsign of a tensor.
     *
     * Defined as x / (abs(x) + 1), element-wise.
     *
     * @param x: Input.
     * @returns Output.
     */
    function softsign(x) {
        return tfc.tidy(() => tfc__namespace.div(x, tfc__namespace.add(tfc__namespace.abs(x), 1)));
    }
    /**
     * Sets entries in `x` to zero at random, while scaling the entire tensor.
     *
     * @param x input tensor.
     * @param level fraction of the entries in the tensor that will be set to 0.
     * @param noiseShape shape of randomly generated keep/drop flags, must be
     *   broadcastable to the shape of `x`. Optional.
     * @param seed random seed to ensure determinism. Optional.
     * @returns Result of the dropout operation.
     */
    function dropout$1(x, level, noiseShape, seed) {
        return tfc.tidy(() => tfc__namespace.dropout(x, level, noiseShape, seed));
    }
    /**
     * Element-wise, segment-wise linear approximation of sigmoid.
     *
     * Returns `0.` if `x < -2.5`, `1.` if `x > 2.5`.
     * In `-2.5 <= x <= 2.5`, returns `0.2 * x + 0.5`.
     *
     * @param x Input tensor.
     * @returns Output tensor.
     */
    function hardSigmoid(x) {
        return tfc.tidy(() => {
            const y = tfc__namespace.add(.5, tfc__namespace.mul(.2, x));
            return tfc__namespace.clipByValue(y, 0, 1);
        });
    }
    /**
     * Invoke `x` in the training phase, and `alt` otherwise.
     *
     * Porting Note: We do not create placeholder tensors for the `training`
     * boolean flag here, because there is no such thing in the TF.js imperative
     * backend.
     *
     * @param x The function to invoke iff `training` is `true`.
     * @param alt The function to invoke iff `training` is `false`.
     * @param training Boolean flag for whether training phase is active.
     * @returns The return value of `x()` if `training` is `true`, or the return
     *   value of `alt()` if `training` is `false`.
     */
    function inTrainPhase(x, alt, training = false) {
        return training ? x() : alt();
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    const VALID_FAN_MODE_VALUES = ['fanIn', 'fanOut', 'fanAvg'];
    const VALID_DISTRIBUTION_VALUES = ['normal', 'uniform', 'truncatedNormal'];

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    function checkFanMode(value) {
        checkStringTypeUnionValue(VALID_FAN_MODE_VALUES, 'FanMode', value);
    }
    function checkDistribution(value) {
        checkStringTypeUnionValue(VALID_DISTRIBUTION_VALUES, 'Distribution', value);
    }
    /**
     * Initializer base class.
     *
     * @doc {
     *   heading: 'Initializers', subheading: 'Classes', namespace: 'initializers'}
     */
    class Initializer extends tfc.serialization.Serializable {
        fromConfigUsesCustomObjects() {
            return false;
        }
        getConfig() {
            return {};
        }
    }
    class Zeros extends Initializer {
        apply(shape, dtype) {
            return tfc.zeros(shape, dtype);
        }
    }
    /** @nocollapse */
    Zeros.className = 'Zeros';
    tfc.serialization.registerClass(Zeros);
    class Ones extends Initializer {
        apply(shape, dtype) {
            return tfc.ones(shape, dtype);
        }
    }
    /** @nocollapse */
    Ones.className = 'Ones';
    tfc.serialization.registerClass(Ones);
    class Constant extends Initializer {
        constructor(args) {
            super();
            if (typeof args !== 'object') {
                throw new ValueError(`Expected argument of type ConstantConfig but got ${args}`);
            }
            if (args.value === undefined) {
                throw new ValueError(`config must have value set but got ${args}`);
            }
            this.value = args.value;
        }
        apply(shape, dtype) {
            return tfc.tidy(() => tfc.mul(tfc.scalar(this.value), tfc.ones(shape, dtype)));
        }
        getConfig() {
            return {
                value: this.value,
            };
        }
    }
    /** @nocollapse */
    Constant.className = 'Constant';
    tfc.serialization.registerClass(Constant);
    class RandomUniform extends Initializer {
        constructor(args) {
            super();
            this.DEFAULT_MINVAL = -0.05;
            this.DEFAULT_MAXVAL = 0.05;
            this.minval = args.minval || this.DEFAULT_MINVAL;
            this.maxval = args.maxval || this.DEFAULT_MAXVAL;
            this.seed = args.seed;
        }
        apply(shape, dtype) {
            return tfc.randomUniform(shape, this.minval, this.maxval, dtype);
        }
        getConfig() {
            return { minval: this.minval, maxval: this.maxval, seed: this.seed };
        }
    }
    /** @nocollapse */
    RandomUniform.className = 'RandomUniform';
    tfc.serialization.registerClass(RandomUniform);
    class RandomNormal extends Initializer {
        constructor(args) {
            super();
            this.DEFAULT_MEAN = 0.;
            this.DEFAULT_STDDEV = 0.05;
            this.mean = args.mean || this.DEFAULT_MEAN;
            this.stddev = args.stddev || this.DEFAULT_STDDEV;
            this.seed = args.seed;
        }
        apply(shape, dtype) {
            dtype = dtype || 'float32';
            if (dtype !== 'float32' && dtype !== 'int32') {
                throw new NotImplementedError(`randomNormal does not support dType ${dtype}.`);
            }
            return randomNormal$2(shape, this.mean, this.stddev, dtype, this.seed);
        }
        getConfig() {
            return { mean: this.mean, stddev: this.stddev, seed: this.seed };
        }
    }
    /** @nocollapse */
    RandomNormal.className = 'RandomNormal';
    tfc.serialization.registerClass(RandomNormal);
    class TruncatedNormal extends Initializer {
        constructor(args) {
            super();
            this.DEFAULT_MEAN = 0.;
            this.DEFAULT_STDDEV = 0.05;
            this.mean = args.mean || this.DEFAULT_MEAN;
            this.stddev = args.stddev || this.DEFAULT_STDDEV;
            this.seed = args.seed;
        }
        apply(shape, dtype) {
            dtype = dtype || 'float32';
            if (dtype !== 'float32' && dtype !== 'int32') {
                throw new NotImplementedError(`truncatedNormal does not support dType ${dtype}.`);
            }
            return tfc.truncatedNormal(shape, this.mean, this.stddev, dtype, this.seed);
        }
        getConfig() {
            return { mean: this.mean, stddev: this.stddev, seed: this.seed };
        }
    }
    /** @nocollapse */
    TruncatedNormal.className = 'TruncatedNormal';
    tfc.serialization.registerClass(TruncatedNormal);
    class Identity$1 extends Initializer {
        constructor(args) {
            super();
            this.gain = args.gain != null ? args.gain : 1.0;
        }
        apply(shape, dtype) {
            return tfc.tidy(() => {
                if (shape.length !== 2 || shape[0] !== shape[1]) {
                    throw new ValueError('Identity matrix initializer can only be used for' +
                        ' 2D square matrices.');
                }
                else {
                    return tfc.mul(this.gain, tfc.eye(shape[0]));
                }
            });
        }
        getConfig() {
            return { gain: this.gain };
        }
    }
    /** @nocollapse */
    Identity$1.className = 'Identity';
    tfc.serialization.registerClass(Identity$1);
    /**
     * Computes the number of input and output units for a weight shape.
     * @param shape Shape of weight.
     * @param dataFormat data format to use for convolution kernels.
     *   Note that all kernels in Keras are standardized on the
     *   CHANNEL_LAST ordering (even when inputs are set to CHANNEL_FIRST).
     * @return An length-2 array: fanIn, fanOut.
     */
    function computeFans(shape, dataFormat = 'channelsLast') {
        let fanIn;
        let fanOut;
        checkDataFormat(dataFormat);
        if (shape.length === 2) {
            fanIn = shape[0];
            fanOut = shape[1];
        }
        else if ([3, 4, 5].indexOf(shape.length) !== -1) {
            if (dataFormat === 'channelsFirst') {
                const receptiveFieldSize = arrayProd(shape, 2);
                fanIn = shape[1] * receptiveFieldSize;
                fanOut = shape[0] * receptiveFieldSize;
            }
            else if (dataFormat === 'channelsLast') {
                const receptiveFieldSize = arrayProd(shape, 0, shape.length - 2);
                fanIn = shape[shape.length - 2] * receptiveFieldSize;
                fanOut = shape[shape.length - 1] * receptiveFieldSize;
            }
        }
        else {
            const shapeProd = arrayProd(shape);
            fanIn = Math.sqrt(shapeProd);
            fanOut = Math.sqrt(shapeProd);
        }
        return [fanIn, fanOut];
    }
    class VarianceScaling extends Initializer {
        /**
         * Constructor of VarianceScaling.
         * @throws ValueError for invalid value in scale.
         */
        constructor(args) {
            super();
            if (args.scale < 0.0) {
                throw new ValueError(`scale must be a positive float. Got: ${args.scale}`);
            }
            this.scale = args.scale == null ? 1.0 : args.scale;
            this.mode = args.mode == null ? 'fanIn' : args.mode;
            checkFanMode(this.mode);
            this.distribution =
                args.distribution == null ? 'normal' : args.distribution;
            checkDistribution(this.distribution);
            this.seed = args.seed;
        }
        apply(shape, dtype) {
            const fans = computeFans(shape);
            const fanIn = fans[0];
            const fanOut = fans[1];
            let scale = this.scale;
            if (this.mode === 'fanIn') {
                scale /= Math.max(1, fanIn);
            }
            else if (this.mode === 'fanOut') {
                scale /= Math.max(1, fanOut);
            }
            else {
                scale /= Math.max(1, (fanIn + fanOut) / 2);
            }
            if (this.distribution === 'normal') {
                const stddev = Math.sqrt(scale);
                dtype = dtype || 'float32';
                if (dtype !== 'float32' && dtype !== 'int32') {
                    throw new NotImplementedError(`${this.getClassName()} does not support dType ${dtype}.`);
                }
                return tfc.truncatedNormal(shape, 0, stddev, dtype, this.seed);
            }
            else {
                const limit = Math.sqrt(3 * scale);
                return tfc.randomUniform(shape, -limit, limit, dtype);
            }
        }
        getConfig() {
            return {
                scale: this.scale,
                mode: this.mode,
                distribution: this.distribution,
                seed: this.seed
            };
        }
    }
    /** @nocollapse */
    VarianceScaling.className = 'VarianceScaling';
    tfc.serialization.registerClass(VarianceScaling);
    class GlorotUniform extends VarianceScaling {
        /**
         * Constructor of GlorotUniform
         * @param scale
         * @param mode
         * @param distribution
         * @param seed
         */
        constructor(args) {
            super({
                scale: 1.0,
                mode: 'fanAvg',
                distribution: 'uniform',
                seed: args == null ? null : args.seed
            });
        }
        getClassName() {
            // In Python Keras, GlorotUniform is not a class, but a helper method
            // that creates a VarianceScaling object. Use 'VarianceScaling' as
            // class name to be compatible with that.
            return VarianceScaling.className;
        }
    }
    /** @nocollapse */
    GlorotUniform.className = 'GlorotUniform';
    tfc.serialization.registerClass(GlorotUniform);
    class GlorotNormal extends VarianceScaling {
        /**
         * Constructor of GlorotNormal.
         * @param scale
         * @param mode
         * @param distribution
         * @param seed
         */
        constructor(args) {
            super({
                scale: 1.0,
                mode: 'fanAvg',
                distribution: 'normal',
                seed: args == null ? null : args.seed
            });
        }
        getClassName() {
            // In Python Keras, GlorotNormal is not a class, but a helper method
            // that creates a VarianceScaling object. Use 'VarianceScaling' as
            // class name to be compatible with that.
            return VarianceScaling.className;
        }
    }
    /** @nocollapse */
    GlorotNormal.className = 'GlorotNormal';
    tfc.serialization.registerClass(GlorotNormal);
    class HeNormal extends VarianceScaling {
        constructor(args) {
            super({
                scale: 2.0,
                mode: 'fanIn',
                distribution: 'normal',
                seed: args == null ? null : args.seed
            });
        }
        getClassName() {
            // In Python Keras, HeNormal is not a class, but a helper method
            // that creates a VarianceScaling object. Use 'VarianceScaling' as
            // class name to be compatible with that.
            return VarianceScaling.className;
        }
    }
    /** @nocollapse */
    HeNormal.className = 'HeNormal';
    tfc.serialization.registerClass(HeNormal);
    class HeUniform extends VarianceScaling {
        constructor(args) {
            super({
                scale: 2.0,
                mode: 'fanIn',
                distribution: 'uniform',
                seed: args == null ? null : args.seed
            });
        }
        getClassName() {
            // In Python Keras, HeUniform is not a class, but a helper method
            // that creates a VarianceScaling object. Use 'VarianceScaling' as
            // class name to be compatible with that.
            return VarianceScaling.className;
        }
    }
    /** @nocollapse */
    HeUniform.className = 'HeUniform';
    tfc.serialization.registerClass(HeUniform);
    class LeCunNormal extends VarianceScaling {
        constructor(args) {
            super({
                scale: 1.0,
                mode: 'fanIn',
                distribution: 'normal',
                seed: args == null ? null : args.seed
            });
        }
        getClassName() {
            // In Python Keras, LeCunNormal is not a class, but a helper method
            // that creates a VarianceScaling object. Use 'VarianceScaling' as
            // class name to be compatible with that.
            return VarianceScaling.className;
        }
    }
    /** @nocollapse */
    LeCunNormal.className = 'LeCunNormal';
    tfc.serialization.registerClass(LeCunNormal);
    class LeCunUniform extends VarianceScaling {
        constructor(args) {
            super({
                scale: 1.0,
                mode: 'fanIn',
                distribution: 'uniform',
                seed: args == null ? null : args.seed
            });
        }
        getClassName() {
            // In Python Keras, LeCunUniform is not a class, but a helper method
            // that creates a VarianceScaling object. Use 'VarianceScaling' as
            // class name to be compatible with that.
            return VarianceScaling.className;
        }
    }
    /** @nocollapse */
    LeCunUniform.className = 'LeCunNormal';
    tfc.serialization.registerClass(LeCunUniform);
    class Orthogonal extends Initializer {
        constructor(args) {
            super();
            this.DEFAULT_GAIN = 1;
            this.gain = args.gain == null ? this.DEFAULT_GAIN : args.gain;
            this.seed = args.seed;
            if (this.seed != null) {
                throw new NotImplementedError('Random seed is not implemented for Orthogonal Initializer yet.');
            }
        }
        apply(shape, dtype) {
            return tfc.tidy(() => {
                if (shape.length < 2) {
                    throw new NotImplementedError('Shape must be at least 2D.');
                }
                if (shape[0] * shape[1] > 2000) {
                    console.warn(`Orthogonal initializer is being called on a matrix with more ` +
                        `than 2000 (${shape[0] * shape[1]}) elements: ` +
                        `Slowness may result.`);
                }
                // TODO(cais): Add seed support.
                const normalizedShape = shape[0] > shape[1] ? [shape[1], shape[0]] : shape;
                const a = randomNormal$2(normalizedShape, 0, 1, 'float32');
                let q = tfc.linalg.gramSchmidt(a);
                if (shape[0] > shape[1]) {
                    q = tfc.transpose(q);
                }
                return tfc.mul(this.gain, q);
            });
        }
        getConfig() {
            return {
                gain: this.gain,
                seed: this.seed,
            };
        }
    }
    /** @nocollapse */
    Orthogonal.className = 'Orthogonal';
    tfc.serialization.registerClass(Orthogonal);
    // Maps the JavaScript-like identifier keys to the corresponding registry
    // symbols.
    const INITIALIZER_IDENTIFIER_REGISTRY_SYMBOL_MAP = {
        'constant': 'Constant',
        'glorotNormal': 'GlorotNormal',
        'glorotUniform': 'GlorotUniform',
        'heNormal': 'HeNormal',
        'heUniform': 'HeUniform',
        'identity': 'Identity',
        'leCunNormal': 'LeCunNormal',
        'leCunUniform': 'LeCunUniform',
        'ones': 'Ones',
        'orthogonal': 'Orthogonal',
        'randomNormal': 'RandomNormal',
        'randomUniform': 'RandomUniform',
        'truncatedNormal': 'TruncatedNormal',
        'varianceScaling': 'VarianceScaling',
        'zeros': 'Zeros'
    };
    function deserializeInitializer(config, customObjects = {}) {
        return deserializeKerasObject(config, tfc.serialization.SerializationMap.getMap().classNameMap, customObjects, 'initializer');
    }
    function serializeInitializer(initializer) {
        return serializeKerasObject(initializer);
    }
    function getInitializer(identifier) {
        if (typeof identifier === 'string') {
            const className = identifier in INITIALIZER_IDENTIFIER_REGISTRY_SYMBOL_MAP ?
                INITIALIZER_IDENTIFIER_REGISTRY_SYMBOL_MAP[identifier] :
                identifier;
            /* We have four 'helper' classes for common initializers that
            all get serialized as 'VarianceScaling' and shouldn't go through
            the deserializeInitializer pathway. */
            if (className === 'GlorotNormal') {
                return new GlorotNormal();
            }
            else if (className === 'GlorotUniform') {
                return new GlorotUniform();
            }
            else if (className === 'HeNormal') {
                return new HeNormal();
            }
            else if (className === 'HeUniform') {
                return new HeUniform();
            }
            else if (className === 'LeCunNormal') {
                return new LeCunNormal();
            }
            else if (className === 'LeCunUniform') {
                return new LeCunUniform();
            }
            else {
                const config = {};
                config['className'] = className;
                config['config'] = {};
                return deserializeInitializer(config);
            }
        }
        else if (identifier instanceof Initializer) {
            return identifier;
        }
        else {
            return deserializeInitializer(identifier);
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    // tslint:enable
    /**
     * Determine whether the input is an Array of Shapes.
     */
    function isArrayOfShapes(x) {
        return Array.isArray(x) && Array.isArray(x[0]);
    }
    /**
     * Special case of normalizing shapes to lists.
     *
     * @param x A shape or list of shapes to normalize into a list of Shapes.
     * @return A list of Shapes.
     */
    function normalizeShapeList(x) {
        if (x.length === 0) {
            return [];
        }
        if (!Array.isArray(x[0])) {
            return [x];
        }
        return x;
    }
    /**
     * Helper function to obtain exactly one Tensor.
     * @param xs: A single `tf.Tensor` or an `Array` of `tf.Tensor`s.
     * @return A single `tf.Tensor`. If `xs` is an `Array`, return the first one.
     * @throws ValueError: If `xs` is an `Array` and its length is not 1.
     */
    function getExactlyOneTensor(xs) {
        let x;
        if (Array.isArray(xs)) {
            if (xs.length !== 1) {
                throw new ValueError(`Expected Tensor length to be 1; got ${xs.length}`);
            }
            x = xs[0];
        }
        else {
            x = xs;
        }
        return x;
    }
    /**
     * Helper function to obtain exactly on instance of Shape.
     *
     * @param shapes Input single `Shape` or Array of `Shape`s.
     * @returns If input is a single `Shape`, return it unchanged. If the input is
     *   an `Array` containing exactly one instance of `Shape`, return the instance.
     *   Otherwise, throw a `ValueError`.
     * @throws ValueError: If input is an `Array` of `Shape`s, and its length is not
     *   1.
     */
    function getExactlyOneShape(shapes) {
        if (Array.isArray(shapes) && Array.isArray(shapes[0])) {
            if (shapes.length === 1) {
                shapes = shapes;
                return shapes[0];
            }
            else {
                throw new ValueError(`Expected exactly 1 Shape; got ${shapes.length}`);
            }
        }
        else {
            return shapes;
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Count the elements in an Array of LayerVariables.
     *
     * @param weights: The LayerVariables of which the constituent numbers are to
     *   be counted.
     * @returns A count of the elements in all the LayerVariables
     */
    function countParamsInWeights(weights) {
        let count = 0;
        for (const weight of weights) {
            if (weight.shape.length === 0) {
                count += 1;
            }
            else {
                count += weight.shape.reduce((a, b) => a * b);
            }
        }
        return count;
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    const DEFAULT_VARIABLE_NAME_PREFIX = 'Variable';
    /**
     * A `tf.layers.LayerVariable` is similar to a `tf.Tensor` in that it has a
     * dtype and shape, but its value is mutable.  The value is itself represented
     * as a`tf.Tensor`, and can be read with the `read()` method and updated with
     * the `write()` method.
     */
    class LayerVariable {
        /**
         * Construct Variable from a `tf.Tensor`.
         *
         * If not explicitly named, the Variable will be given a name with the
         * prefix 'Variable'. Variable names are unique. In the case of name
         * collision, suffixies '_<num>' will be added to the name.
         *
         * @param val Initial value of the Variable.
         * @param name Name of the variable. If `null` or `undefined` is provided, it
         *   will default a name with the prefix 'Variable'.
         * @param constraint Optional, projection function to be applied to the
         * variable after optimize updates
         * @throws ValueError if `name` is `null` or `undefined`.
         */
        constructor(val, dtype = 'float32', name = DEFAULT_VARIABLE_NAME_PREFIX, trainable = true, constraint = null) {
            this.dtype = dtype == null ? 'float32' : dtype;
            this.shape = val.shape;
            this.id = getNextUniqueTensorId();
            name = name == null ? DEFAULT_VARIABLE_NAME_PREFIX : name;
            this.originalName = getScopedTensorName(name);
            this.name = getUniqueTensorName(this.originalName);
            this.trainable_ = trainable;
            this.constraint = constraint;
            this.val = tfc__namespace.variable(val, this.trainable_, this.name, this.dtype);
        }
        /**
         * Get a snapshot of the Variable's value.
         *
         * The returned value is a snapshot of the Variable's value at the time of
         * the invocation. Future mutations in the value of the tensor will only
         * be reflected by future calls to this method.
         */
        read() {
            this.assertNotDisposed();
            return this.val;
        }
        /**
         * Update the value of the Variable.
         *
         * @param newVal: The new value to update to. Must be consistent with the
         *   dtype and shape of the Variable.
         * @return This Variable.
         */
        write(newVal) {
            // TODO(cais): Once  TF.js Core supports Tensor.dtype, check dtype match.
            this.assertNotDisposed();
            checkShapesMatch(this.val, newVal);
            // Skip updating if this is the exact same tensor.
            if (this.val.id !== newVal.id) {
                this.val.assign(newVal);
                if (this.constraint != null) {
                    this.val.assign(this.constraint.apply(this.val));
                }
            }
            return this;
        }
        /**
         * Dispose this LayersVariable instance from memory.
         */
        dispose() {
            this.assertNotDisposed();
            this.val.dispose();
        }
        assertNotDisposed() {
            if (this.val.isDisposed) {
                throw new Error(`LayersVariable ${this.name} is already disposed.`);
            }
        }
        get trainable() {
            return this.trainable_;
        }
        set trainable(trainable) {
            this.trainable_ = trainable;
            this.val.trainable = trainable;
        }
    }
    function checkShapesMatch(x, y) {
        if (x.shape.toString() !== y.shape.toString()) {
            throw new Error('Shape mismatch: ' + JSON.stringify(x.shape) + ' vs. ' +
                JSON.stringify(y.shape));
        }
    }
    /**
     * Get the values of an array of Variables.
     *
     * @param tensors An `Array` of `Variable`s to get the values of.
     * @return The values of the inputs, as an `Array` of`tf.Tensor`s.
     */
    function batchGetValue(xs) {
        return xs.map(x => x.read());
    }
    /**
     * Update the value of multiple Variables at once.
     *
     * @param variablesAndValues An `Array`, each element is of type
     *   [Variable, Tensor]. The first item is the
     *   `Variable` of which the value is to be updated. The second item
     *   carries the new value.
     */
    function batchSetValue(variablesAndValues) {
        variablesAndValues.forEach(variableAndValue => {
            const variable = variableAndValue[0];
            variable.write(variableAndValue[1]);
        });
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Specifies the ndim, dtype and shape of every input to a layer.
     *
     * Every layer should expose (if appropriate) an `inputSpec` attribute:
     * a list of instances of InputSpec (one per input tensor).
     *
     * A null entry in a shape is compatible with any dimension,
     * a null shape is compatible with any shape.
     */
    class InputSpec {
        constructor(args) {
            this.dtype = args.dtype;
            this.shape = args.shape;
            /*
              TODO(michaelterry): Could throw error if ndim and shape are both defined
                (then backport).
            */
            if (args.shape != null) {
                this.ndim = args.shape.length;
            }
            else {
                this.ndim = args.ndim;
            }
            this.maxNDim = args.maxNDim;
            this.minNDim = args.minNDim;
            this.axes = args.axes || {};
        }
    }
    /**
     * `tf.SymbolicTensor` is a placeholder for a Tensor without any concrete value.
     *
     * They are most often encountered when building a graph of `Layer`s for a
     * `tf.LayersModel` and the input data's shape, but not values are known.
     *
     * @doc {heading: 'Models', 'subheading': 'Classes'}
     */
    class SymbolicTensor {
        /**
         *
         * @param dtype
         * @param shape
         * @param sourceLayer The Layer that produced this symbolic tensor.
         * @param inputs The inputs passed to sourceLayer's __call__() method.
         * @param nodeIndex
         * @param tensorIndex
         * @param callArgs The keyword arguments passed to the __call__() method.
         * @param name
         * @param outputTensorIndex The index of this tensor in the list of outputs
         *   returned by apply().
         */
        constructor(dtype, shape, sourceLayer, inputs, callArgs, name, outputTensorIndex) {
            this.dtype = dtype;
            this.shape = shape;
            this.sourceLayer = sourceLayer;
            this.inputs = inputs;
            this.callArgs = callArgs;
            this.outputTensorIndex = outputTensorIndex;
            this.id = getNextUniqueTensorId();
            if (name != null) {
                this.originalName = getScopedTensorName(name);
                this.name = getUniqueTensorName(this.originalName);
            }
            this.rank = shape.length;
        }
    }
    let _nextNodeID = 0;
    /**
     * A `Node` describes the connectivity between two layers.
     *
     * Each time a layer is connected to some new input,
     * a node is added to `layer.inboundNodes`.
     *
     * Each time the output of a layer is used by another layer,
     * a node is added to `layer.outboundNodes`.
     *
     * `nodeIndices` and `tensorIndices` are basically fine-grained coordinates
     * describing the origin of the `inputTensors`, verifying the following:
     *
     * `inputTensors[i] ==
     * inboundLayers[i].inboundNodes[nodeIndices[i]].outputTensors[
     *   tensorIndices[i]]`
     *
     * A node from layer A to layer B is added to:
     *     A.outboundNodes
     *     B.inboundNodes
     */
    class Node {
        constructor(args, 
        // TODO(michaelterry): Define actual type for this.
        callArgs) {
            this.callArgs = callArgs;
            this.id = _nextNodeID++;
            /*
              Layer instance (NOT a list).
              this is the layer that takes a list of input tensors
              and turns them into a list of output tensors.
              the current node will be added to
              the inboundNodes of outboundLayer.
            */
            this.outboundLayer = args.outboundLayer;
            /*
                The following 3 properties describe where
                the input tensors come from: which layers,
                and for each layer, which node and which
                tensor output of each node.
            */
            // List of layer instances.
            this.inboundLayers = args.inboundLayers;
            // List of integers, 1:1 mapping with inboundLayers.
            this.nodeIndices = args.nodeIndices;
            // List of integers, 1:1 mapping with inboundLayers.
            this.tensorIndices = args.tensorIndices;
            /*
                Following 2 properties:
                tensor inputs and outputs of outboundLayer.
            */
            // List of tensors. 1:1 mapping with inboundLayers.
            this.inputTensors = args.inputTensors;
            // List of tensors, created by outboundLayer.call().
            this.outputTensors = args.outputTensors;
            /*
                Following 2 properties: input and output masks.
                List of tensors, 1:1 mapping with inputTensor.
            */
            this.inputMasks = args.inputMasks;
            // List of tensors, created by outboundLayer.computeMask().
            this.outputMasks = args.outputMasks;
            // Following 2 properties: input and output shapes.
            // List of shape tuples, shapes of inputTensors.
            this.inputShapes = args.inputShapes;
            // List of shape tuples, shapes of outputTensors.
            this.outputShapes = args.outputShapes;
            // Add nodes to all layers involved.
            for (const layer of args.inboundLayers) {
                if (layer != null) {
                    layer.outboundNodes.push(this);
                }
            }
            args.outboundLayer.inboundNodes.push(this);
        }
        getConfig() {
            const inboundNames = [];
            for (const layer of this.inboundLayers) {
                if (layer != null) {
                    inboundNames.push(layer.name);
                }
                else {
                    inboundNames.push(null);
                }
            }
            return {
                outboundLayer: this.outboundLayer ? this.outboundLayer.name : null,
                inboundLayers: inboundNames,
                nodeIndices: this.nodeIndices,
                tensorIndices: this.tensorIndices
            };
        }
    }
    let _nextLayerID = 0;
    /**
     * A layer is a grouping of operations and weights that can be composed to
     * create a `tf.LayersModel`.
     *
     * Layers are constructed by using the functions under the
     * [tf.layers](#Layers-Basic) namespace.
     *
     * @doc {heading: 'Layers', subheading: 'Classes', namespace: 'layers'}
     */
    class Layer extends tfc.serialization.Serializable {
        constructor(args = {}) {
            super();
            this._callHook = null;
            this._addedWeightNames = [];
            // Porting Notes: PyKeras does not have this property in this base Layer
            //   class. Instead lets Layer subclass set it dynamically and checks the
            //   value with `hasattr`. In tfjs-layers, we let this be a member of this
            //   base class.
            this._stateful = false;
            this.id = _nextLayerID++;
            this.activityRegularizer = null;
            this.inputSpec = null;
            this.supportsMasking = false;
            // These properties will be set upon call of this.build()
            this._trainableWeights = [];
            this._nonTrainableWeights = [];
            this._losses = [];
            this._updates = [];
            this._built = false;
            /*
              These lists will be filled via successive calls
              to this.addInboundNode().
             */
            this.inboundNodes = [];
            this.outboundNodes = [];
            let name = args.name;
            if (!name) {
                const prefix = this.getClassName();
                name = toSnakeCase(prefix) + '_' + getUid(prefix);
            }
            this.name = name;
            this.trainable_ = args.trainable == null ? true : args.trainable;
            if (args.inputShape != null || args.batchInputShape != null) {
                /*
                  In this case we will later create an input layer
                  to insert before the current layer
                 */
                let batchInputShape;
                if (args.batchInputShape != null) {
                    batchInputShape = args.batchInputShape;
                }
                else if (args.inputShape != null) {
                    let batchSize = null;
                    if (args.batchSize != null) {
                        batchSize = args.batchSize;
                    }
                    batchInputShape = [batchSize].concat(args.inputShape);
                }
                this.batchInputShape = batchInputShape;
                // Set dtype.
                let dtype = args.dtype;
                if (dtype == null) {
                    dtype = args.inputDType;
                }
                if (dtype == null) {
                    dtype = 'float32';
                }
                this.dtype = dtype;
            }
            if (args.weights != null) {
                this.initialWeights = args.weights;
            }
            else {
                this.initialWeights = null;
            }
            // The value of `_refCount` is initialized to null. When the layer is used
            // in a symbolic way for the first time, it will be set to 1.
            this._refCount = null;
            this.fastWeightInitDuringBuild = false;
        }
        /**
         * Converts a layer and its index to a unique (immutable type) name.
         * This function is used internally with `this.containerNodes`.
         * @param layer The layer.
         * @param nodeIndex The layer's position (e.g. via enumerate) in a list of
         *   nodes.
         *
         * @returns The unique name.
         */
        static nodeKey(layer, nodeIndex) {
            return layer.name + '_ib-' + nodeIndex.toString();
        }
        /**
         * Returns this.inboundNode at index nodeIndex.
         *
         * Porting note: This is a replacement for _get_node_attribute_at_index()
         * @param nodeIndex
         * @param attrName The name of the attribute related to request for this node.
         */
        getNodeAtIndex(nodeIndex, attrName) {
            if (this.inboundNodes.length === 0) {
                throw new RuntimeError('The layer has never been called ' +
                    `and thus has no defined ${attrName}.`);
            }
            if (this.inboundNodes.length <= nodeIndex) {
                throw new ValueError(`Asked to get ${attrName} at node ${nodeIndex}, ` +
                    `but the layer has only ${this.inboundNodes.length} inbound nodes.`);
            }
            return this.inboundNodes[nodeIndex];
        }
        /**
         * Retrieves the input tensor(s) of a layer at a given node.
         *
         * @param nodeIndex Integer, index of the node from which to retrieve the
         *   attribute. E.g. `nodeIndex=0` will correspond to the first time the layer
         *   was called.
         *
         * @return A tensor (or list of tensors if the layer has multiple inputs).
         */
        getInputAt(nodeIndex) {
            return singletonOrArray(this.getNodeAtIndex(nodeIndex, 'input').inputTensors);
        }
        /**
         * Retrieves the output tensor(s) of a layer at a given node.
         *
         * @param nodeIndex Integer, index of the node from which to retrieve the
         *   attribute. E.g. `nodeIndex=0` will correspond to the first time the layer
         *   was called.
         *
         * @return A tensor (or list of tensors if the layer has multiple outputs).
         */
        getOutputAt(nodeIndex) {
            return singletonOrArray(this.getNodeAtIndex(nodeIndex, 'output').outputTensors);
        }
        // Properties
        /**
         * Retrieves the input tensor(s) of a layer.
         *
         * Only applicable if the layer has exactly one inbound node,
         * i.e. if it is connected to one incoming layer.
         *
         * @return Input tensor or list of input tensors.
         *
         * @exception AttributeError if the layer is connected to more than one
         *   incoming layers.
         */
        get input() {
            if (this.inboundNodes.length > 1) {
                throw new AttributeError(`Layer ${this.name}` +
                    ' has multiple inbound nodes, ' +
                    'hence the notion of "layer input" ' +
                    'is ill-defined. ' +
                    'Use `getInputAt(nodeIndex)` instead.');
            }
            else if (this.inboundNodes.length === 0) {
                throw new AttributeError(`Layer ${this.name}` +
                    ' is not connected, no input to return.');
            }
            return singletonOrArray(this.getNodeAtIndex(0, 'input').inputTensors);
        }
        /**
         * Retrieves the output tensor(s) of a layer.
         *
         * Only applicable if the layer has exactly one inbound node,
         * i.e. if it is connected to one incoming layer.
         *
         * @return Output tensor or list of output tensors.
         *
         * @exception AttributeError if the layer is connected to more than one
         *   incoming layers.
         */
        get output() {
            if (this.inboundNodes.length === 0) {
                throw new AttributeError(`Layer ${this.name}` +
                    ' has no inbound nodes.');
            }
            if (this.inboundNodes.length > 1) {
                throw new AttributeError(`Layer ${this.name}` +
                    ' has multiple inbound nodes, ' +
                    'hence the notion of "layer output" ' +
                    'is ill-defined. ' +
                    'Use `getOutputAt(nodeIndex)` instead.');
            }
            return singletonOrArray(this.getNodeAtIndex(0, 'output').outputTensors);
        }
        get losses() {
            return this._losses;
        }
        /**
         * Retrieves the Layer's current loss values.
         *
         * Used for regularizers during training.
         */
        calculateLosses() {
            // Porting Node: This is an augmentation to Layer.loss in PyKeras.
            //   In PyKeras, Layer.loss returns symbolic tensors. Here a concrete
            //   Tensor (specifically Scalar) values are returned. This is due to the
            //   imperative backend.
            return this.losses.map(lossFn => lossFn());
        }
        get updates() {
            return this._updates;
        }
        get built() {
            return this._built;
        }
        set built(built) {
            this._built = built;
        }
        get trainable() {
            return this.trainable_;
        }
        set trainable(trainable) {
            this._trainableWeights.forEach(w => w.trainable = trainable);
            this.trainable_ = trainable;
        }
        get trainableWeights() {
            if (this.trainable_) {
                return this._trainableWeights.filter(w => w.trainable);
            }
            else {
                return [];
            }
        }
        set trainableWeights(weights) {
            this._trainableWeights = weights;
        }
        get nonTrainableWeights() {
            if (this.trainable) {
                return this._trainableWeights.filter(w => !w.trainable)
                    .concat(this._nonTrainableWeights);
            }
            else {
                return this._trainableWeights.concat(this._nonTrainableWeights);
            }
        }
        set nonTrainableWeights(weights) {
            this._nonTrainableWeights = weights;
        }
        /**
         * The concatenation of the lists trainableWeights and nonTrainableWeights
         * (in this order).
         */
        get weights() {
            return this.trainableWeights.concat(this.nonTrainableWeights);
        }
        get stateful() {
            return this._stateful;
        }
        /**
         * Reset the states of the layer.
         *
         * This method of the base Layer class is essentially a no-op.
         * Subclasses that are stateful (e.g., stateful RNNs) should override this
         * method.
         */
        resetStates() {
            if (!this.stateful) {
                throw new Error('Cannot call the resetStates() method of a non-stateful Layer ' +
                    'object.');
            }
        }
        /**
         * Checks compatibility between the layer and provided inputs.
         *
         * This checks that the tensor(s) `input`
         * verify the input assumptions of the layer
         * (if any). If not, exceptions are raised.
         *
         * @param inputs Input tensor or list of input tensors.
         *
         * @exception ValueError in case of mismatch between
         *   the provided inputs and the expectations of the layer.
         */
        assertInputCompatibility(inputs) {
            inputs = toList(inputs);
            if (this.inputSpec == null || this.inputSpec.length === 0) {
                return;
            }
            const inputSpec = toList(this.inputSpec);
            if (inputs.length !== inputSpec.length) {
                throw new ValueError(`Layer ${this.name} expects ${inputSpec.length} inputs, ` +
                    `but it received ${inputs.length} input tensors. ` +
                    `Input received: ${inputs}`);
            }
            for (let inputIndex = 0; inputIndex < inputs.length; inputIndex++) {
                const x = inputs[inputIndex];
                const spec = inputSpec[inputIndex];
                if (spec == null) {
                    continue;
                }
                // Check ndim.
                const ndim = x.rank;
                if (spec.ndim != null) {
                    if (ndim !== spec.ndim) {
                        throw new ValueError(`Input ${inputIndex} is incompatible with layer ${this.name}: ` +
                            `expected ndim=${spec.ndim}, found ndim=${ndim}`);
                    }
                }
                if (spec.maxNDim != null) {
                    if (ndim > spec.maxNDim) {
                        throw new ValueError(`Input ${inputIndex} is incompatible with layer ${this.name}` +
                            `: expected max_ndim=${spec.maxNDim}, found ndim=${ndim}`);
                    }
                }
                if (spec.minNDim != null) {
                    if (ndim < spec.minNDim) {
                        throw new ValueError(`Input ${inputIndex} is incompatible with layer ${this.name}` +
                            `: expected min_ndim=${spec.minNDim}, found ndim=${ndim}.`);
                    }
                }
                // Check dtype.
                if (spec.dtype != null) {
                    if (x.dtype !== spec.dtype) {
                        throw new ValueError(`Input ${inputIndex} is incompatible with layer ${this.name} ` +
                            `: expected dtype=${spec.dtype}, found dtype=${x.dtype}.`);
                    }
                }
                // Check specific shape axes.
                if (spec.axes) {
                    const xShape = x.shape;
                    for (const key in spec.axes) {
                        const axis = Number(key);
                        const value = spec.axes[key];
                        // Perform Python-style slicing in case axis < 0;
                        // TODO(cais): Use https://github.com/alvivi/typescript-underscore to
                        // ensure type safety through Underscore calls.
                        const xShapeAtAxis = axis >= 0 ? xShape[axis] : xShape[xShape.length + axis];
                        if (value != null && [value, null].indexOf(xShapeAtAxis) === -1) {
                            throw new ValueError(`Input ${inputIndex} is incompatible with layer ` +
                                `${this.name}: expected axis ${axis} of input shape to ` +
                                `have value ${value} but got shape ${xShape}.`);
                        }
                    }
                }
                // Check shape.
                if (spec.shape != null) {
                    for (let i = 0; i < spec.shape.length; ++i) {
                        const specDim = spec.shape[i];
                        const dim = x.shape[i];
                        if (specDim != null && dim != null) {
                            if (specDim !== dim) {
                                throw new ValueError(`Input ${inputIndex} is incompatible with layer ` +
                                    `${this.name}: expected shape=${spec.shape}, ` +
                                    `found shape=${x.shape}.`);
                            }
                        }
                    }
                }
            }
        }
        /**
         * This is where the layer's logic lives.
         *
         * @param inputs Input tensor, or list/tuple of input tensors.
         * @param kwargs Additional keyword arguments.
         *
         * @return A tensor or list/tuple of tensors.
         */
        call(inputs, kwargs) {
            return inputs;
        }
        invokeCallHook(inputs, kwargs) {
            if (this._callHook != null) {
                this._callHook(inputs, kwargs);
            }
        }
        /**
         * Set call hook.
         * This is currently used for testing only.
         * @param callHook
         */
        setCallHook(callHook) {
            this._callHook = callHook;
        }
        /**
         * Clear call hook.
         * This is currently used for testing only.
         */
        clearCallHook() {
            this._callHook = null;
        }
        /**
         * Builds or executes a `Layer`'s logic.
         *
         * When called with `tf.Tensor`(s), execute the `Layer`'s computation and
         * return Tensor(s). For example:
         *
         * ```js
         * const denseLayer = tf.layers.dense({
         *   units: 1,
         *   kernelInitializer: 'zeros',
         *   useBias: false
         * });
         *
         * // Invoke the layer's apply() method with a `tf.Tensor` (with concrete
         * // numeric values).
         * const input = tf.ones([2, 2]);
         * const output = denseLayer.apply(input);
         *
         * // The output's value is expected to be [[0], [0]], due to the fact that
         * // the dense layer has a kernel initialized to all-zeros and does not have
         * // a bias.
         * output.print();
         * ```
         *
         * When called with `tf.SymbolicTensor`(s), this will prepare the layer for
         * future execution.  This entails internal book-keeping on shapes of
         * expected Tensors, wiring layers together, and initializing weights.
         *
         * Calling `apply` with `tf.SymbolicTensor`s are typically used during the
         * building of non-`tf.Sequential` models. For example:
         *
         * ```js
         * const flattenLayer = tf.layers.flatten();
         * const denseLayer = tf.layers.dense({units: 1});
         *
         * // Use tf.layers.input() to obtain a SymbolicTensor as input to apply().
         * const input = tf.input({shape: [2, 2]});
         * const output1 = flattenLayer.apply(input);
         *
         * // output1.shape is [null, 4]. The first dimension is the undetermined
         * // batch size. The second dimension comes from flattening the [2, 2]
         * // shape.
         * console.log(JSON.stringify(output1.shape));
         *
         * // The output SymbolicTensor of the flatten layer can be used to call
         * // the apply() of the dense layer:
         * const output2 = denseLayer.apply(output1);
         *
         * // output2.shape is [null, 1]. The first dimension is the undetermined
         * // batch size. The second dimension matches the number of units of the
         * // dense layer.
         * console.log(JSON.stringify(output2.shape));
         *
         * // The input and output can be used to construct a model that consists
         * // of the flatten and dense layers.
         * const model = tf.model({inputs: input, outputs: output2});
         * ```
         *
         * @param inputs a `tf.Tensor` or `tf.SymbolicTensor` or an Array of them.
         * @param kwargs Additional keyword arguments to be passed to `call()`.
         *
         * @return Output of the layer's `call` method.
         *
         * @exception ValueError error in case the layer is missing shape information
         *   for its `build` call.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        // Porting Note: This is a replacement for __call__() in Python.
        apply(inputs, kwargs) {
            kwargs = kwargs || {};
            this.assertNotDisposed();
            // Ensure inputs are all the same type.
            const inputsList = toList(inputs);
            let allAreSymbolic = true;
            for (const input of inputsList) {
                if (!(input instanceof SymbolicTensor)) {
                    allAreSymbolic = false;
                    break;
                }
            }
            let noneAreSymbolic = true;
            for (const input of inputsList) {
                if (input instanceof SymbolicTensor) {
                    noneAreSymbolic = false;
                    break;
                }
            }
            if (allAreSymbolic === noneAreSymbolic) {
                throw new ValueError('Arguments to apply() must be all ' +
                    'SymbolicTensors or all Tensors');
            }
            // TODO(michaelterry): nameScope() may not be necessary.
            return nameScope(this.name, () => {
                // Handle laying building (weight creating, input spec locking).
                if (!this.built) {
                    /*
                      Throw exceptions in case the input is not compatible
                      with the inputSpec specified in the layer constructor.
                     */
                    this.assertInputCompatibility(inputs);
                    // Collect input shapes to build layer.
                    const inputShapes = [];
                    for (const xElem of toList(inputs)) {
                        inputShapes.push(xElem.shape);
                    }
                    this.build(singletonOrArray(inputShapes));
                    this.built = true;
                    // Load weights that were specified at layer instantiation.
                    if (this.initialWeights) {
                        this.setWeights(this.initialWeights);
                    }
                    if (this._refCount === null && noneAreSymbolic) {
                        // The first use of this layer is a non-symbolic call, set ref count
                        // to 1 so the Layer can be properly disposed if its dispose() method
                        // is called.
                        this._refCount = 1;
                    }
                }
                /*
                  Throw exceptions in case the input is not compatible
                  with the inputSpec set at build time.
                */
                this.assertInputCompatibility(inputs);
                // Handle mask propagation.
                // TODO(michaelterry): Mask propagation not currently implemented.
                // Actually call the layer, collecting output(s), mask(s), and shape(s).
                if (noneAreSymbolic) {
                    let output = this.call(inputs, kwargs);
                    // TODO(michaelterry): Compute the outputMask
                    // If the layer returns tensors from its inputs, unmodified,
                    // we copy them to avoid loss of tensor metadata.
                    const outputList = toList(output);
                    const outputListCopy = [];
                    // TODO(michaelterry): This copying may not be necessary given our eager
                    // backend.
                    for (let x of outputList) {
                        if (inputsList.indexOf(x) !== -1) {
                            x = x.clone();
                        }
                        outputListCopy.push(x);
                    }
                    output = singletonOrArray(outputListCopy);
                    if (this.activityRegularizer != null) {
                        throw new NotImplementedError('Layer invocation in the presence of activity ' +
                            'regularizer(s) is not supported yet.');
                    }
                    // TODO(michaelterry): Call addInboundNode()?
                    return output;
                }
                else {
                    const inputShape = collectInputShape(inputs);
                    const outputShape = this.computeOutputShape(inputShape);
                    let output;
                    const outputDType = guessOutputDType(inputs);
                    this.warnOnIncompatibleInputShape(Array.isArray(inputs) ? inputShape[0] :
                        inputShape);
                    if (outputShape != null && outputShape.length > 0 &&
                        Array.isArray(outputShape[0])) {
                        // We have multiple output shapes. Create multiple output tensors.
                        output = outputShape
                            .map((shape, index) => new SymbolicTensor(outputDType, shape, this, toList(inputs), kwargs, this.name, index));
                    }
                    else {
                        output = new SymbolicTensor(outputDType, outputShape, this, toList(inputs), kwargs, this.name);
                    }
                    /*
                      Add an inbound node to the layer, so that it keeps track
                      of the call and of all new variables created during the call.
                      This also updates the layer history of the output tensor(s).
                      If the input tensor(s) had no previous history,
                      this does nothing.
                    */
                    this.addInboundNode(inputs, output, null, null, inputShape, outputShape, kwargs);
                    this._refCount++;
                    if (this.activityRegularizer != null) {
                        throw new NotImplementedError('Layer invocation in the presence of activity ' +
                            'regularizer(s) is not supported yet.');
                    }
                    return output;
                }
            });
        }
        /**
         * Check compatibility between input shape and this layer's batchInputShape.
         *
         * Print warning if any incompatibility is found.
         *
         * @param inputShape Input shape to be checked.
         */
        warnOnIncompatibleInputShape(inputShape) {
            if (this.batchInputShape == null) {
                return;
            }
            else if (inputShape.length !== this.batchInputShape.length) {
                console.warn(`The rank of the input tensor provided (shape: ` +
                    `${JSON.stringify(inputShape)}) does not match that of the ` +
                    `batchInputShape (${JSON.stringify(this.batchInputShape)}) ` +
                    `of the layer ${this.name}`);
            }
            else {
                let dimMismatch = false;
                this.batchInputShape.forEach((dimension, i) => {
                    if (dimension != null && inputShape[i] != null &&
                        inputShape[i] !== dimension) {
                        dimMismatch = true;
                    }
                });
                if (dimMismatch) {
                    console.warn(`The shape of the input tensor ` +
                        `(${JSON.stringify(inputShape)}) does not ` +
                        `match the expectation of layer ${this.name}: ` +
                        `${JSON.stringify(this.batchInputShape)}`);
                }
            }
        }
        /**
         * Retrieves the output shape(s) of a layer.
         *
         * Only applicable if the layer has only one inbound node, or if all inbound
         * nodes have the same output shape.
         *
         * @returns Output shape or shapes.
         * @throws AttributeError: if the layer is connected to more than one incoming
         *   nodes.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        get outputShape() {
            if (this.inboundNodes == null || this.inboundNodes.length === 0) {
                throw new AttributeError(`The layer ${this.name} has never been called and thus has no ` +
                    `defined output shape.`);
            }
            const allOutputShapes = [];
            for (const node of this.inboundNodes) {
                const shapeString = JSON.stringify(node.outputShapes);
                if (allOutputShapes.indexOf(shapeString) === -1) {
                    allOutputShapes.push(shapeString);
                }
            }
            if (allOutputShapes.length === 1) {
                const outputShapes = this.inboundNodes[0].outputShapes;
                if (Array.isArray(outputShapes) && Array.isArray(outputShapes[0]) &&
                    outputShapes.length === 1) {
                    return outputShapes[0];
                }
                else {
                    return outputShapes;
                }
            }
            else {
                throw new AttributeError(`The layer ${this.name} has multiple inbound nodes with different ` +
                    `output shapes. Hence the notion of "output shape" is ill-defined ` +
                    `for the layer.`);
                // TODO(cais): Implement getOutputShapeAt().
            }
        }
        /**
         * Counts the total number of numbers (e.g., float32, int32) in the
         * weights.
         *
         * @returns An integer count.
         * @throws RuntimeError: If the layer is not built yet (in which case its
         *   weights are not defined yet.)
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        countParams() {
            if (!this.built) {
                throw new RuntimeError(`You tried to call countParams() on ${this.name}, ` +
                    `but the layer is not built yet. Build it first by calling ` +
                    `build(batchInputShape).`);
            }
            return countParamsInWeights(this.weights);
        }
        /**
         * Creates the layer weights.
         *
         * Must be implemented on all layers that have weights.
         *
         * Called when apply() is called to construct the weights.
         *
         * @param inputShape A `Shape` or array of `Shape` (unused).
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        build(inputShape) {
            this.built = true;
        }
        /**
         * Returns the current values of the weights of the layer.
         *
         * @param trainableOnly Whether to get the values of only trainable weights.
         * @returns Weight values as an `Array` of `tf.Tensor`s.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        getWeights(trainableOnly = false) {
            return batchGetValue(trainableOnly ? this.trainableWeights : this.weights);
        }
        /**
         * Sets the weights of the layer, from Tensors.
         *
         * @param weights a list of Tensors. The number of arrays and their shape
         *   must match number of the dimensions of the weights of the layer (i.e.
         *   it should match the output of `getWeights`).
         *
         * @exception ValueError If the provided weights list does not match the
         *   layer's specifications.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        setWeights(weights) {
            tfc.tidy(() => {
                const params = this.weights;
                if (params.length !== weights.length) {
                    // TODO(cais): Restore the following and use `providedWeights`, instead
                    // of `weights` in the error message, once the deeplearn.js bug is
                    // fixed: https://github.com/PAIR-code/deeplearnjs/issues/498 const
                    // providedWeights = JSON.stringify(weights).slice(0, 50);
                    throw new ValueError(`You called setWeights(weights) on layer "${this.name}" ` +
                        `with a weight list of length ${weights.length}, ` +
                        `but the layer was expecting ${params.length} weights. ` +
                        `Provided weights: ${weights}...`);
                }
                if (params.length === 0) {
                    return;
                }
                const weightValueTuples = [];
                const paramValues = batchGetValue(params);
                for (let i = 0; i < paramValues.length; ++i) {
                    const pv = paramValues[i];
                    const p = params[i];
                    const w = weights[i];
                    if (!tfc.util.arraysEqual(pv.shape, w.shape)) {
                        throw new ValueError(`Layer weight shape ${pv.shape} ` +
                            `not compatible with provided weight shape ${w.shape}`);
                    }
                    weightValueTuples.push([p, w]);
                }
                batchSetValue(weightValueTuples);
            });
        }
        /**
         * Adds a weight variable to the layer.
         *
         * @param name Name of the new weight variable.
         * @param shape The shape of the weight.
         * @param dtype The dtype of the weight.
         * @param initializer An initializer instance.
         * @param regularizer A regularizer instance.
         * @param trainable Whether the weight should be trained via backprop or not
         *   (assuming that the layer itself is also trainable).
         * @param constraint An optional trainable.
         * @return The created weight variable.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        addWeight(name, shape, dtype, initializer, regularizer, trainable, constraint, getInitializerFunc) {
            // Reject duplicate weight names.
            if (this._addedWeightNames.indexOf(name) !== -1) {
                throw new ValueError(`Duplicate weight name ${name} for layer ${this.name}`);
            }
            this._addedWeightNames.push(name);
            if (dtype == null) {
                dtype = 'float32';
            }
            if (this.fastWeightInitDuringBuild) {
                initializer = getInitializerFunc != null ? getInitializerFunc() :
                    getInitializer('zeros');
            }
            const initValue = initializer.apply(shape, dtype);
            const weight = new LayerVariable(initValue, dtype, name, trainable, constraint);
            initValue.dispose();
            // Request backend not to dispose the weights of the model on scope() exit.
            if (regularizer != null) {
                this.addLoss(() => regularizer.apply(weight.read()));
            }
            if (trainable == null) {
                trainable = true;
            }
            if (trainable) {
                this._trainableWeights.push(weight);
            }
            else {
                this._nonTrainableWeights.push(weight);
            }
            return weight;
        }
        /**
         * Set the fast-weight-initialization flag.
         *
         * In cases where the initialized weight values will be immediately
         * overwritten by loaded weight values during model loading, setting
         * the flag to `true` saves unnecessary calls to potentially expensive
         * initializers and speeds up the loading process.
         *
         * @param value Target value of the flag.
         */
        setFastWeightInitDuringBuild(value) {
            this.fastWeightInitDuringBuild = value;
        }
        /**
         * Add losses to the layer.
         *
         * The loss may potentially be conditional on some inputs tensors,
         * for instance activity losses are conditional on the layer's inputs.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        addLoss(losses) {
            if (losses == null || Array.isArray(losses) && losses.length === 0) {
                return;
            }
            // Update this.losses
            losses = toList(losses);
            if (this._losses !== undefined && this._losses !== null) {
                this.losses.push(...losses);
            }
        }
        /**
         * Computes the output shape of the layer.
         *
         * Assumes that the layer will be built to match that input shape provided.
         *
         * @param inputShape A shape (tuple of integers) or a list of shape tuples
         *   (one per output tensor of the layer). Shape tuples can include null for
         *   free dimensions, instead of an integer.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        computeOutputShape(inputShape) {
            return inputShape;
        }
        /**
         * Computes an output mask tensor.
         *
         * @param inputs Tensor or list of tensors.
         * @param mask Tensor or list of tensors.
         *
         * @return null or a tensor (or list of tensors, one per output tensor of the
         * layer).
         */
        computeMask(inputs, mask) {
            if (!this.supportsMasking) {
                if (mask != null) {
                    if (Array.isArray(mask)) {
                        mask.forEach(maskElement => {
                            if (maskElement != null) {
                                throw new TypeError(`Layer ${this.name} does not support masking, ` +
                                    'but was passed an inputMask.');
                            }
                        });
                    }
                    else {
                        throw new TypeError(`Layer ${this.name} does not support masking, ` +
                            'but was passed an inputMask.');
                    }
                }
                // masking not explicitly supported: return null as mask
                return null;
            }
            // if masking is explictly supported, by default
            // carry over the input mask
            return mask;
        }
        /**
         * Internal method to create an inbound node for the layer.
         *
         * @param inputTensors List of input tensors.
         * @param outputTensors List of output tensors.
         * @param inputMasks List of input masks (a mask can be a tensor, or null).
         * @param outputMasks List of output masks (a mask can be a tensor, or null).
         * @param inputShapes List of input shape tuples.
         * @param outputShapes List of output shape tuples.
         * @param kwargs Dictionary of keyword arguments that were passed to the
         *   `call` method of the layer at the call that created the node.
         */
        addInboundNode(inputTensors, outputTensors, inputMasks, outputMasks, inputShapes, outputShapes, kwargs = null) {
            const inputTensorList = toList(inputTensors);
            outputTensors = toList(outputTensors);
            inputMasks = toList(inputMasks);
            outputMasks = toList(outputMasks);
            inputShapes = normalizeShapeList(inputShapes);
            outputShapes = normalizeShapeList(outputShapes);
            // Collect input tensor(s) coordinates.
            const inboundLayers = [];
            const nodeIndices = [];
            const tensorIndices = [];
            for (const x of inputTensorList) {
                /*
                 * TODO(michaelterry): Keras adds this value to tensors; it's not
                 * clear whether we'll use this or not.
                 */
                inboundLayers.push(x.sourceLayer);
                nodeIndices.push(x.nodeIndex);
                tensorIndices.push(x.tensorIndex);
            }
            // Create node, add it to inbound nodes.
            // (This call has side effects.)
            // tslint:disable-next-line:no-unused-expression
            new Node({
                outboundLayer: this,
                inboundLayers,
                nodeIndices,
                tensorIndices,
                inputTensors: inputTensorList,
                outputTensors,
                inputMasks,
                outputMasks,
                inputShapes,
                outputShapes
            }, kwargs);
            // Update tensor history
            for (let i = 0; i < outputTensors.length; i++) {
                // TODO(michaelterry: _uses_learning_phase not tracked.
                outputTensors[i].sourceLayer = this;
                outputTensors[i].nodeIndex = this.inboundNodes.length - 1;
                outputTensors[i].tensorIndex = i;
            }
        }
        /**
         * Returns the config of the layer.
         *
         * A layer config is a TS dictionary (serializable)
         * containing the configuration of a layer.
         * The same layer can be reinstantiated later
         * (without its trained weights) from this configuration.
         *
         * The config of a layer does not include connectivity
         * information, nor the layer class name.  These are handled
         * by 'Container' (one layer of abstraction above).
         *
         * Porting Note: The TS dictionary follows TS naming standards for
         * keys, and uses tfjs-layers type-safe Enums.  Serialization methods
         * should use a helper function to convert to the pythonic storage
         * standard. (see serialization_utils.convertTsToPythonic)
         *
         * @returns TS dictionary of configuration.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        getConfig() {
            const config = { name: this.name, trainable: this.trainable };
            if (this.batchInputShape != null) {
                config['batchInputShape'] = this.batchInputShape;
            }
            if (this.dtype != null) {
                config['dtype'] = this.dtype;
            }
            return config;
        }
        /**
         * Dispose the weight variables that this Layer instance holds.
         *
         * @returns {number} Number of disposed variables.
         */
        disposeWeights() {
            this.weights.forEach(weight => weight.dispose());
            return this.weights.length;
        }
        assertNotDisposed() {
            if (this._refCount === 0) {
                throw new Error(`Layer '${this.name}' is already disposed.`);
            }
        }
        /**
         * Attempt to dispose layer's weights.
         *
         * This method decreases the reference count of the Layer object by 1.
         *
         * A Layer is reference-counted. Its reference count is incremented by 1
         * the first item its `apply()` method is called and when it becomes a part
         * of a new `Node` (through calling the `apply()` method on a
         * `tf.SymbolicTensor`).
         *
         * If the reference count of a Layer becomes 0, all the weights will be
         * disposed and the underlying memory (e.g., the textures allocated in WebGL)
         * will be freed.
         *
         * Note: If the reference count is greater than 0 after the decrement, the
         * weights of the Layer will *not* be disposed.
         *
         * After a Layer is disposed, it cannot be used in calls such as `apply()`,
         * `getWeights()` or `setWeights()` anymore.
         *
         * @returns A DisposeResult Object with the following fields:
         *   - refCountAfterDispose: The reference count of the Container after this
         *     `dispose()` call.
         *   - numDisposedVariables: Number of `tf.Variable`s (i.e., weights) disposed
         *     during this `dispose()` call.
         * @throws {Error} If the layer is not built yet, or if the layer has already
         *   been disposed.
         *
         * @doc {heading: 'Models', 'subheading': 'Classes'}
         */
        dispose() {
            if (!this.built) {
                throw new Error(`Cannot dispose Layer ${this.name} because it has not been ` +
                    `built yet.`);
            }
            if (this._refCount === null) {
                throw new Error(`Cannot dispose Layer ${this.name} because it has not been used ` +
                    `yet.`);
            }
            this.assertNotDisposed();
            let numDisposedVariables = 0;
            if (--this._refCount === 0) {
                numDisposedVariables = this.disposeWeights();
            }
            return { refCountAfterDispose: this._refCount, numDisposedVariables };
        }
    }
    /**
     * Collects the input shape(s) of a list of `tf.Tensor`s or
     * `tf.SymbolicTensor`s.
     *
     * TODO(michaelterry): Update PyKeras docs (backport).
     *
     * @param inputTensors List of input tensors (or single input tensor).
     *
     * @return List of shape tuples (or single tuple), one tuple per input.
     */
    function collectInputShape(inputTensors) {
        inputTensors =
            toList(inputTensors);
        const shapes = [];
        for (const x of inputTensors) {
            shapes.push(x.shape);
        }
        return singletonOrArray(shapes);
    }
    /**
     * Guesses output dtype based on inputs.
     *
     * At present, just returns 'float32' for any input.
     *
     * @param inputTensors List of input tensors (or single input tensor).
     *
     * @return The guessed DType. At present, always returns 'float32'.
     */
    function guessOutputDType(inputTensors) {
        return 'float32';
    }
    /**
     * Returns the list of input tensors necessary to compute `tensor`.
     *
     * Output will always be a list of tensors (potentially with 1 element).
     *
     * @param tensor The tensor to start from.
     * @param layer Origin layer of the tensor.
     * @param nodeIndex Origin node index of the tensor.
     *
     * @return Array of input tensors.
     */
    function getSourceInputs(tensor, layer, nodeIndex) {
        if (layer == null || (nodeIndex != null && nodeIndex > 0)) {
            layer = tensor.sourceLayer;
            nodeIndex = tensor.nodeIndex;
        }
        if (layer.inboundNodes.length === 0) {
            return [tensor];
        }
        else {
            const node = layer.inboundNodes[nodeIndex];
            if (node.inboundLayers.length === 0) {
                return node.inputTensors;
            }
            else {
                const sourceTensors = [];
                for (let i = 0; i < node.inboundLayers.length; i++) {
                    const x = node.inputTensors[i];
                    const layer = node.inboundLayers[i];
                    const nodeIndex = node.nodeIndices[i];
                    const previousSources = getSourceInputs(x, layer, nodeIndex);
                    // Avoid input redundancy.
                    for (const x of previousSources) {
                        if (sourceTensors.indexOf(x) === -1) {
                            sourceTensors.push(x);
                        }
                    }
                }
                return sourceTensors;
            }
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    class InputLayer extends Layer {
        constructor(args) {
            super({
                dtype: args.dtype,
                name: args.name != null ? args.name : getUid('input').toString()
            });
            // Normalize config.batchSize and config.sparse
            if (args.batchSize == null) {
                args.batchSize = null;
            }
            if (args.sparse == null) {
                args.sparse = false;
            }
            this.trainable = false;
            this.built = true;
            this.sparse = args.sparse;
            if (args.inputShape != null && args.batchInputShape != null) {
                throw new ValueError('Only provide the inputShape OR ' +
                    'batchInputShape argument to inputLayer, not both at the same time.');
            }
            let batchInputShape = args.batchInputShape;
            if (batchInputShape == null) {
                if (args.inputShape == null) {
                    throw new ValueError('An InputLayer should be passed either a ' +
                        '`batchInputShape` or an `inputShape`.');
                }
                else {
                    batchInputShape = [args.batchSize].concat(args.inputShape);
                }
            }
            else {
                // TODO(michaelterry): Backport to PyKeras
                if (args.batchSize != null) {
                    throw new ValueError('Cannot specify batchSize if batchInputShape is ' +
                        'specified when creating an InputLayer.');
                }
            }
            const dtype = args.dtype || 'float32';
            this.batchInputShape = batchInputShape;
            this.dtype = dtype;
            // TODO(michaelterry): Backport this to PyKeras?
            this.inputSpec = [{ shape: batchInputShape }];
            const inputTensor = new SymbolicTensor(this.dtype, this.batchInputShape, this, [], {}, this.name);
            inputTensor.nodeIndex = 0;
            inputTensor.tensorIndex = 0;
            // Create an input node to add to this.outboundNode.
            // (This call has side effects.)
            // tslint:disable-next-line:no-unused-expression
            new Node({
                outboundLayer: this,
                inboundLayers: [],
                nodeIndices: [],
                tensorIndices: [],
                inputTensors: [inputTensor],
                outputTensors: [inputTensor],
                inputMasks: [null],
                outputMasks: [null],
                inputShapes: [batchInputShape],
                outputShapes: [batchInputShape]
            });
        }
        apply(inputs, kwargs) {
            throw new ValueError('Cannot pass any input to an ' +
                `InputLayer's apply() method. InputLayer name: ${this.name}`);
        }
        dispose() {
            // dispose() for InputLayer is overridden as no-op.
            return { refCountAfterDispose: this._refCount, numDisposedVariables: 0 };
        }
        getConfig() {
            return {
                batchInputShape: this.batchInputShape,
                dtype: this.dtype,
                sparse: this.sparse,
                name: this.name
            };
        }
    }
    /** @nocollapse */
    InputLayer.className = 'InputLayer';
    tfc.serialization.registerClass(InputLayer);
    function Input(config) {
        if (config.batchShape == null && config.shape == null) {
            throw new Error('Please provide to Input either a `shape`' +
                ' or a `batchShape` argument. Note that ' +
                '`shape` does not include the batch ' +
                'dimension.');
        }
        if (config.batchShape != null && config.shape != null) {
            // TODO(michaelterry): Backport to PyKeras.
            throw new ValueError('Please provide either a `shape` or `batchShape` ' +
                'argument to Input, but not both.');
        }
        let batchShape = config.batchShape;
        if (config.shape != null && batchShape == null) {
            batchShape = [null].concat(config.shape);
        }
        let dtype = config.dtype;
        if (dtype == null) {
            dtype = 'float32';
        }
        const inputLayer = new InputLayer({
            batchInputShape: batchShape,
            name: config.name,
            dtype,
            sparse: config.sparse
        });
        const outputs = inputLayer.inboundNodes[0].outputTensors;
        return outputs[0];
    }

    /**
     * @license
     * Copyright 2018 Google LLC
     *
     * Use of this source code is governed by an MIT-style
     * license that can be found in the LICENSE file or at
     * https://opensource.org/licenses/MIT.
     * =============================================================================
     */
    /**
     * Helper function to check the dtype and shape compatibility of a feed value.
     */
    function assertFeedCompatibility(key, val) {
        // Check dtype compatibility.
        if (key.dtype == null || key.dtype === val.dtype) {
            //  a.  If types match, return val tensor as is.
            return val;
        }
        try {
            //  b. Attempt to convert to expected type.
            return tfc.cast(val, key.dtype);
        }
        catch (err) {
            //  c. If conversion fails, return helpful error.
            throw new ValueError(`The dtype of the feed (${val.dtype}) can not be cast to the dtype ` +
                `of the key '${key.name}' (${key.dtype}).`);
        }
    }
    /**
     * FeedDict: A mapping from unique SymbolicTensors to feed values for them.
     * A feed value is a concrete value represented as an `Tensor`.
     */
    class FeedDict {
        /**
         * Constructor, optionally does copy-construction.
         * @param feeds An Array of `Feed`s, or another `FeedDict`, in which case
         *   copy-construction will be performed.
         */
        constructor(feeds) {
            this.id2Value = {};
            this.id2Mask = {};
            this.name2Id = {};
            if (feeds instanceof FeedDict) {
                for (const id in feeds.id2Value) {
                    this.id2Value[id] = feeds.id2Value[id];
                    if (id in feeds.id2Mask) {
                        this.id2Mask[id] = feeds.id2Mask[id];
                    }
                }
            }
            else {
                if (feeds == null) {
                    return;
                }
                for (const feed of feeds) {
                    this.add(feed.key, feed.value);
                }
            }
        }
        /**
         * Add a key-value pair to the FeedDict.
         *
         * @param key The key of the feed.
         * @param value The value of the tensor feed.
         * @param mask The value of the mask feed (optional).
         * @returns This `FeedDict`.
         * @throws ValueError: If the key `SymbolicTensor` already exists in the
         *   `FeedDict`.
         */
        add(key, value, mask) {
            if (this.id2Value[key.id] == null) {
                this.id2Value[key.id] = assertFeedCompatibility(key, value);
                this.name2Id[key.name] = key.id;
                if (mask != null) {
                    this.id2Mask[key.id] = mask;
                }
            }
            else {
                throw new ValueError(`Duplicate key: name=${key.name}, id=${key.id}`);
            }
            return this;
        }
        /**
         * Add a Feed to the FeedDict.
         * @param feed The new `Feed` to add.
         * @returns This `FeedDict`.
         */
        addFeed(feed) {
            this.add(feed.key, feed.value);
        }
        /**
         * Probe whether a key already exists in the FeedDict.
         * @param key
         */
        hasKey(key) {
            return this.id2Value[key.id] != null;
        }
        /**
         * Get all the SymbolicTensor available in this FeedDict.
         */
        names() {
            return Object.keys(this.name2Id);
        }
        /**
         * Get the feed value for given key.
         * @param key The SymbolicTensor, or its name (as a string), of which the
         *     value is sought.
         * @returns If `key` exists, the corresponding feed value.
         * @throws ValueError: If `key` does not exist in this `FeedDict`.
         */
        getValue(key) {
            if (key instanceof SymbolicTensor) {
                if (this.id2Value[key.id] == null) {
                    throw new ValueError(`Nonexistent key: ${key.name}`);
                }
                else {
                    return this.id2Value[key.id];
                }
            }
            else {
                const id = this.name2Id[key];
                if (id == null) {
                    throw new ValueError(`Feed dict has no SymbolicTensor name: ${key}`);
                }
                return this.id2Value[id];
            }
        }
        /**
         * Get the feed mask for given key.
         * @param key The SymbolicTensor, or its name (as a string), of which the
         *     value is sought.
         * @returns If `key` exists, the corresponding feed mask.
         * @throws ValueError: If `key` does not exist in this `FeedDict`.
         */
        getMask(key) {
            if (key instanceof SymbolicTensor) {
                if (this.id2Value[key.id] == null) {
                    throw new ValueError(`Nonexistent key: ${key.name}`);
                }
                else {
                    return this.id2Mask[key.id];
                }
            }
            else {
                const id = this.name2Id[key];
                if (id == null) {
                    throw new ValueError(`Feed dict has no SymbolicTensor name: ${key}`);
                }
                return this.id2Mask[id];
            }
        }
        /** Dispose all mask Tensors held by this object. */
        disposeMasks() {
            if (this.id2Mask != null) {
                tfc.dispose(this.id2Mask);
            }
        }
    }
    // Cache for topologically sorted SymbolicTensors for given execution
    // targets (i.e., fetches).
    const cachedSorted = new LruCache();
    // Cache for recipient count maps for given execution targets (i.e., fetches).
    const cachedRecipientCounts = new LruCache();
    function updateCacheMaxEntries(maxEntries) {
        if (cachedSorted != null) {
            cachedSorted.setMaxEntries(maxEntries);
        }
        if (cachedRecipientCounts != null) {
            cachedRecipientCounts.setMaxEntries(maxEntries);
        }
    }
    /**
     * Execute a SymbolicTensor by using concrete feed values.
     *
     * A `SymbolicTensor` object is a node in a computation graph of TF.js
     * Layers. The object is backed by a source layer and input
     * `SymbolicTensor`s to the source layer. This method evaluates
     * the `call()` method of the source layer, using concrete values of the
     * inputs obtained from either
     * * `feedDict`, if the input key exists in `feedDict`, or else,
     * * a recursive call to `execute()` itself.
     *
     * @param x: The `SymbolicTensor` to execute.
     * @param feedDict: The feed values, as base condition of the recursion.
     *   execution.
     * @param kwargs: Optional keyword arguments.
     * @param probe: A probe object (of interface `ExecutionProbe`) used for
     *   testing memory footprint of `execute` calls.
     * @returns Result of the execution.
     * @throws ValueError: If any `SymbolicTensor`s from `InputLayer`s
     *   encountered during the execution lacks a feed value in `feedDict`.
     */
    function execute(fetches, feedDict, kwargs, probe) {
        const training = kwargs == null ? false : kwargs['training'];
        const arrayFetches = Array.isArray(fetches);
        const fetchArray = arrayFetches ? fetches : [fetches];
        const outputNames = fetchArray.map(t => t.name);
        const finalOutputs = [];
        const feedNames = feedDict.names();
        for (const outputName of outputNames) {
            if (feedNames.indexOf(outputName) !== -1) {
                finalOutputs.push(feedDict.getValue(outputName));
            }
            else {
                finalOutputs.push(null);
            }
        }
        if (probe != null) {
            // For optional probing of memory footprint during execution.
            probe.maxNumTensors = -Infinity;
            probe.minNumTensors = Infinity;
        }
        // Check cache.
        const fetchAndFeedKey = outputNames.join(',') + '|' + feedDict.names().sort().join(',');
        let sorted = cachedSorted.get(fetchAndFeedKey);
        let recipientCounts;
        if (sorted == null) {
            // Cache doesn't contain the desired combination of fetches. Compute
            // topological sort for the combination for the first time.
            const out = getTopologicalSortAndRecipientCounts(fetchArray, feedDict);
            sorted = out.sorted;
            recipientCounts = out.recipientCounts;
            // Store results in cache for future use.
            cachedSorted.put(fetchAndFeedKey, sorted);
            cachedRecipientCounts.put(fetchAndFeedKey, recipientCounts);
        }
        recipientCounts = {};
        if (!training) {
            Object.assign(recipientCounts, cachedRecipientCounts.get(fetchAndFeedKey));
        }
        const internalFeedDict = new FeedDict(feedDict);
        // Start iterative execution on the topologically-sorted SymbolicTensors.
        for (let i = 0; i < sorted.length; ++i) {
            if (probe != null) {
                // For optional probing of memory usage during execution.
                const numTensors = tfc.memory().numTensors;
                if (numTensors > probe.maxNumTensors) {
                    probe.maxNumTensors = numTensors;
                }
                if (numTensors < probe.minNumTensors) {
                    probe.minNumTensors = numTensors;
                }
            }
            const symbolic = sorted[i];
            const srcLayer = symbolic.sourceLayer;
            if (srcLayer instanceof InputLayer) {
                continue;
            }
            const inputValues = [];
            const inputMasks = [];
            const tensorsToDispose = [];
            let maskExists = false;
            for (const input of symbolic.inputs) {
                const value = internalFeedDict.getValue(input);
                const mask = internalFeedDict.getMask(input);
                inputValues.push(value);
                inputMasks.push(mask);
                if (mask != null) {
                    maskExists = true;
                }
                if (!training) {
                    recipientCounts[input.name]--;
                    if (recipientCounts[input.name] === 0 && !feedDict.hasKey(input) &&
                        outputNames.indexOf(input.name) === -1 && !value.isDisposed &&
                        input.sourceLayer.stateful !== true) {
                        tensorsToDispose.push(value);
                    }
                }
            }
            if (maskExists) {
                kwargs = kwargs || {};
                kwargs['mask'] = inputMasks[0];
            }
            const outputTensors = toList(srcLayer.apply(inputValues, kwargs));
            let outputMask = null;
            if (srcLayer.supportsMasking) {
                outputMask = srcLayer.computeMask(inputValues, inputMasks);
            }
            const layerOutputs = getNodeOutputs(symbolic);
            const outputSymbolicTensors = Array.isArray(layerOutputs) ? layerOutputs : [layerOutputs];
            for (let i = 0; i < outputSymbolicTensors.length; ++i) {
                if (!internalFeedDict.hasKey(outputSymbolicTensors[i])) {
                    internalFeedDict.add(outputSymbolicTensors[i], outputTensors[i], Array.isArray(outputMask) ? outputMask[0] : outputMask);
                }
                const index = outputNames.indexOf(outputSymbolicTensors[i].name);
                if (index !== -1) {
                    finalOutputs[index] = outputTensors[i];
                }
            }
            if (!training) {
                // Clean up Tensors that are no longer needed.
                tfc.dispose(tensorsToDispose);
            }
        }
        // NOTE(cais): Unlike intermediate tensors, we don't discard mask
        // tensors as we go, because these tensors are sometimes passed over a
        // series of mutliple layers, i.e., not obeying the immediate input
        // relations in the graph. If this becomes a memory-usage concern,
        // we can improve this in the future.
        internalFeedDict.disposeMasks();
        return arrayFetches ? finalOutputs : finalOutputs[0];
    }
    /**
     * Sort the `SymbolicTensor`s topologically, for an array of fetches.
     *
     * This function calls getTopologicalSortAndRecipientCountsForOneFetch and
     * merges their results.
     *
     * @param fetch The array of fetches requested. Must be a non-empty array.
     * @param feedDict The dictionary of fed values.
     * @returns sorted: Topologically-sorted array of SymbolicTensors.
     *   recipientCounts: Recipient counts for all SymbolicTensors in `sorted`.
     */
    function getTopologicalSortAndRecipientCounts(fetches, feedDict) {
        tfc.util.assert(fetches != null && fetches.length > 0, () => `Expected at least one fetch, got none`);
        let finalSorted = [];
        let finalRecipientMap = {};
        if (fetches.length === 1) {
            // Special-casing 1 fetch for efficiency.
            const out = getTopologicalSortAndRecipientCountsForOneFetch(fetches[0], feedDict);
            finalSorted = out.sorted;
            finalRecipientMap = out.recipientMap;
        }
        else {
            const visited = new Set();
            for (const fetch of fetches) {
                const { sorted, recipientMap } = getTopologicalSortAndRecipientCountsForOneFetch(fetch, feedDict);
                // Merge sorted SymbolicTensor Arrays.
                for (const symbolicTensor of sorted) {
                    if (!visited.has(symbolicTensor.name)) {
                        finalSorted.push(symbolicTensor);
                        visited.add(symbolicTensor.name);
                    }
                }
                // Merge recipient maps.
                for (const name in recipientMap) {
                    if (finalRecipientMap[name] == null) {
                        finalRecipientMap[name] = new Set();
                    }
                    recipientMap[name].forEach(recipient => finalRecipientMap[name].add(recipient));
                }
            }
        }
        return {
            sorted: finalSorted,
            recipientCounts: recipientMap2Counts(finalRecipientMap)
        };
    }
    function recipientMap2Counts(recipientMap) {
        const recipientCounts = {};
        for (const name in recipientMap) {
            recipientCounts[name] = recipientMap[name].size;
        }
        return recipientCounts;
    }
    /**
     * Sort the `SymbolicTensor`s topologically, for a single fetch.
     *
     * This helper function processes the upstream SymbolicTensors of a single
     * fetch.
     *
     * @param fetch The single fetch requested.
     * @param feedDict The dictionary of fed values.
     * @returns sorted: Topologically-sorted array of SymbolicTensors.
     *   recipientMap: Recipient names for all SymbolicTensors in `sorted`.
     */
    function getTopologicalSortAndRecipientCountsForOneFetch(fetch, feedDict) {
        const visited = new Set();
        const sorted = [];
        const recipientMap = {};
        // Put keys of the feedDict into visited first, so they don't have to be
        // walked. This is needed in case where there are feeds for intermediate
        // SymbolicTensors of the graph.
        for (const key of feedDict.names()) {
            visited.add(key);
        }
        const stack = [];
        const marks = [];
        // Initial population of stack and marks.
        stack.push(fetch);
        while (stack.length > 0) {
            const top = stack[stack.length - 1];
            if (visited.has(top.name)) {
                stack.pop();
                continue;
            }
            const topIsMarked = marks[marks.length - 1] === stack.length - 1;
            if (top.inputs.length === 0 || topIsMarked) {
                // Input SymbolicTensor or all children have been visited.
                stack.pop();
                sorted.push(top);
                visited.add(top.name);
                if (topIsMarked) {
                    marks.pop();
                }
            }
            else {
                // A non-input SymbolicTensor whose upstream SymbolicTensors haven't
                // been visited yet. Push them onto the stack.
                marks.push(stack.length - 1);
                for (const input of top.inputs) {
                    // Increment the recipient count. Note that this needs to happen
                    // regardless of whether the SymbolicTensor has been visited before.
                    if (recipientMap[input.name] == null) {
                        recipientMap[input.name] = new Set();
                    }
                    recipientMap[input.name].add(top.name);
                    if (visited.has(input.name)) {
                        continue; // Avoid repeated visits to the same SymbolicTensor.
                    }
                    stack.push(input);
                }
            }
        }
        return { sorted, recipientMap };
    }
    /**
     * Get the symbolic output tensors of the node to which a given fetch belongs.
     * @param fetch The fetched symbolic tensor.
     * @returns The Array of symbolic tensors output by the node to which `fetch`
     *   belongs.
     */
    function getNodeOutputs(fetch) {
        let layerOutputs;
        if (fetch.sourceLayer.inboundNodes.length === 1) {
            layerOutputs = fetch.sourceLayer.output;
        }
        else {
            let nodeIndex = null;
            for (let i = 0; i < fetch.sourceLayer.inboundNodes.length; ++i) {
                for (const outputTensor of fetch.sourceLayer.inboundNodes[i]
                    .outputTensors) {
                    if (outputTensor.id === fetch.id) {
                        nodeIndex = i;
                        break;
                    }
                }
            }
            layerOutputs = fetch.sourceLayer.getOutputAt(nodeIndex);
        }
        return layerOutputs;
    }

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const ENV$2 = tfc.env();
    /** The max number of entries for the caches of layers' topological sort. */
    ENV$2.registerFlag('TOPOLOGICAL_SORT_CACHE_MAX_ENTRIES', () => 100, updateCacheMaxEntries);

    const Abs = 'Abs';
    const Acos = 'Acos';
    const Acosh = 'Acosh';
    const Add$1 = 'Add';
    const AddN = 'AddN';
    const All = 'All';
    const Any = 'Any';
    const ArgMax = 'ArgMax';
    const ArgMin = 'ArgMin';
    const Asin = 'Asin';
    const Asinh = 'Asinh';
    const Atan = 'Atan';
    const Atanh = 'Atanh';
    const Atan2 = 'Atan2';
    const AvgPool = 'AvgPool';
    const AvgPoolGrad = 'AvgPoolGrad';
    const AvgPool3D = 'AvgPool3D';
    const AvgPool3DGrad = 'AvgPool3DGrad';
    const BatchMatMul = 'BatchMatMul';
    const BatchToSpaceND = 'BatchToSpaceND';
    const Bincount = 'Bincount';
    const BroadcastTo = 'BroadcastTo';
    const BroadcastArgs = 'BroadcastArgs';
    const Cast = 'Cast';
    const Ceil = 'Ceil';
    const ClipByValue = 'ClipByValue';
    const Complex = 'Complex';
    const ComplexAbs = 'ComplexAbs';
    const Concat = 'Concat';
    const Conv2D$1 = 'Conv2D';
    const Conv2DBackpropFilter = 'Conv2DBackpropFilter';
    const Conv2DBackpropInput = 'Conv2DBackpropInput';
    const Conv3D$1 = 'Conv3D';
    const Conv3DBackpropFilterV2 = 'Conv3DBackpropFilterV2';
    const Conv3DBackpropInputV2 = 'Conv3DBackpropInputV2';
    const Cos = 'Cos';
    const Cosh = 'Cosh';
    const Cumprod = 'Cumprod';
    const Cumsum = 'Cumsum';
    const CropAndResize = 'CropAndResize';
    const DenseBincount = 'DenseBincount';
    const DepthToSpace = 'DepthToSpace';
    const DepthwiseConv2dNative = 'DepthwiseConv2dNative';
    const DepthwiseConv2dNativeBackpropFilter = 'DepthwiseConv2dNativeBackpropFilter';
    const DepthwiseConv2dNativeBackpropInput = 'DepthwiseConv2dNativeBackpropInput';
    const Diag = 'Diag';
    const Dilation2D = 'Dilation2D';
    const Dilation2DBackpropInput = 'Dilation2DBackpropInput';
    const Dilation2DBackpropFilter = 'Dilation2DBackpropFilter';
    const RealDiv = 'RealDiv';
    const Einsum = 'Einsum';
    const Elu$1 = 'Elu';
    const EluGrad = 'EluGrad';
    const Erf = 'Erf';
    const Equal = 'Equal';
    const Exp = 'Exp';
    const ExpandDims = 'ExpandDims';
    const Expm1 = 'Expm1';
    const FFT = 'FFT';
    const Fill = 'Fill';
    const FlipLeftRight = 'FlipLeftRight';
    const Floor = 'Floor';
    const FloorDiv = 'FloorDiv';
    const FusedBatchNorm = 'FusedBatchNorm';
    const GatherV2 = 'GatherV2';
    const GatherNd = 'GatherNd';
    const Greater = 'Greater';
    const GreaterEqual = 'GreaterEqual';
    const Identity = 'Identity';
    const IFFT = 'IFFT';
    const Imag = 'Imag';
    const IsFinite = 'IsFinite';
    const IsInf = 'IsInf';
    const IsNan = 'IsNan';
    const LeakyRelu = 'LeakyRelu';
    const Less = 'Less';
    const LessEqual = 'LessEqual';
    const Log = 'Log';
    const Log1p = 'Log1p';
    const LogicalAnd = 'LogicalAnd';
    const LogicalNot = 'LogicalNot';
    const LogicalOr = 'LogicalOr';
    const LogSoftmax$1 = 'LogSoftmax';
    const LRN = 'LRN';
    const LRNGrad = 'LRNGrad';
    const Max = 'Max';
    const Maximum$1 = 'Maximum';
    const MaxPool = 'MaxPool';
    const MaxPoolGrad = 'MaxPoolGrad';
    const MaxPool3D = 'MaxPool3D';
    const MaxPool3DGrad = 'MaxPool3DGrad';
    const MaxPoolWithArgmax = 'MaxPoolWithArgmax';
    const Mean = 'Mean';
    const Min = 'Min';
    const Minimum$1 = 'Minimum';
    const MirrorPad = 'MirrorPad';
    const Mod = 'Mod';
    const Multinomial = 'Multinomial';
    const Multiply$1 = 'Multiply';
    const Neg = 'Neg';
    const NotEqual = 'NotEqual';
    const NonMaxSuppressionV3 = 'NonMaxSuppressionV3';
    const NonMaxSuppressionV4 = 'NonMaxSuppressionV4';
    const NonMaxSuppressionV5 = 'NonMaxSuppressionV5';
    const OnesLike = 'OnesLike';
    const OneHot = 'OneHot';
    const Pack = 'Pack';
    const PadV2 = 'PadV2';
    const Pow = 'Pow';
    const Prelu = 'Prelu';
    const Prod = 'Prod';
    const RaggedGather = 'RaggedGather';
    const RaggedRange = 'RaggedRange';
    const RaggedTensorToTensor = 'RaggedTensorToTensor';
    const Range = 'Range';
    const Real = 'Real';
    const Reciprocal = 'Reciprocal';
    const Relu$1 = 'Relu';
    const Reshape$1 = 'Reshape';
    const ResizeNearestNeighbor = 'ResizeNearestNeighbor';
    const ResizeNearestNeighborGrad = 'ResizeNearestNeighborGrad';
    const ResizeBilinear = 'ResizeBilinear';
    const ResizeBilinearGrad = 'ResizeBilinearGrad';
    const Relu6$1 = 'Relu6';
    const Reverse = 'Reverse';
    const Round = 'Round';
    const Rsqrt = 'Rsqrt';
    const ScatterNd = 'ScatterNd';
    const SearchSorted = 'SearchSorted';
    const Select = 'Select';
    const Selu$1 = 'Selu';
    const Slice = 'Slice';
    const Sin = 'Sin';
    const Sinh = 'Sinh';
    const Sign = 'Sign';
    const Sigmoid$1 = 'Sigmoid';
    const Softplus$1 = 'Softplus';
    const Sqrt = 'Sqrt';
    const Sum = 'Sum';
    const SpaceToBatchND = 'SpaceToBatchND';
    const SplitV = 'SplitV';
    const Softmax$2 = 'Softmax';
    const SparseFillEmptyRows = 'SparseFillEmptyRows';
    const SparseReshape = 'SparseReshape';
    const SparseSegmentMean = 'SparseSegmentMean';
    const SparseSegmentSum = 'SparseSegmentSum';
    const SparseToDense = 'SparseToDense';
    const SquaredDifference = 'SquaredDifference';
    const Square = 'Square';
    const StridedSlice = 'StridedSlice';
    const StringNGrams = 'StringNGrams';
    const StringSplit = 'StringSplit';
    const StringToHashBucketFast = 'StringToHashBucketFast';
    const Sub = 'Sub';
    const Tan = 'Tan';
    const Tanh$1 = 'Tanh';
    const Tile = 'Tile';
    const TopK = 'TopK';
    const Transform = 'Transform';
    const Transpose = 'Transpose';
    const Unique = 'Unique';
    const Unpack = 'Unpack';
    const UnsortedSegmentSum = 'UnsortedSegmentSum';
    const ZerosLike = 'ZerosLike';
    /**
     * TensorFlow.js-only kernels
     */
    const Step = 'Step';
    const FromPixels = 'FromPixels';
    const RotateWithOffset = 'RotateWithOffset';
    const _FusedMatMul = '_FusedMatMul';
    const FusedConv2D = 'FusedConv2D';
    const FusedDepthwiseConv2D = 'FusedDepthwiseConv2D';

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const EPSILON_FLOAT32 = 1e-7;
    const EPSILON_FLOAT16 = 1e-4;
    /**
     * The interface that defines the kernels that should be implemented when
     * adding a new backend. New backends don't need to implement every one of the
     * methods, this can be done gradually (throw an error for unimplemented
     * methods).
     */
    class KernelBackend {
        refCount(dataId) {
            return notYetImplemented('refCount');
        }
        incRef(dataId) {
            return notYetImplemented('incRef');
        }
        timerAvailable() {
            return true;
        }
        time(f) {
            return notYetImplemented('time');
        }
        read(dataId) {
            return notYetImplemented('read');
        }
        readSync(dataId) {
            return notYetImplemented('readSync');
        }
        readToGPU(dataId, options) {
            return notYetImplemented('readToGPU');
        }
        numDataIds() {
            return notYetImplemented('numDataIds');
        }
        disposeData(dataId, force) {
            return notYetImplemented('disposeData');
        }
        write(values, shape, dtype) {
            return notYetImplemented('write');
        }
        move(dataId, values, shape, dtype, refCount) {
            return notYetImplemented('move');
        }
        createTensorFromTexture(values, shape, dtype) {
            return notYetImplemented('createTensorFromTexture');
        }
        memory() {
            return notYetImplemented('memory');
        }
        /** Returns the highest precision for floats in bits (e.g. 16 or 32) */
        floatPrecision() {
            return notYetImplemented('floatPrecision');
        }
        /** Returns the smallest representable number.  */
        epsilon() {
            return this.floatPrecision() === 32 ? EPSILON_FLOAT32 : EPSILON_FLOAT16;
        }
        dispose() {
            return notYetImplemented('dispose');
        }
    }
    function notYetImplemented(kernelName) {
        throw new Error(`'${kernelName}' not yet implemented or not found in the registry. ` +
            `This kernel may not be supported by the tfjs backend you have chosen`);
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Asserts that the expression is true. Otherwise throws an error with the
     * provided message.
     *
     * ```js
     * const x = 2;
     * tf.util.assert(x === 2, 'x is not 2');
     * ```
     *
     * @param expr The expression to assert (as a boolean).
     * @param msg A function that returns the message to report when throwing an
     *     error. We use a function for performance reasons.
     *
     * @doc {heading: 'Util', namespace: 'util'}
     */
    function assert(expr, msg) {
        if (!expr) {
            throw new Error(typeof msg === 'string' ? msg : msg());
        }
    }
    function assertShapesMatch(shapeA, shapeB, errorMessagePrefix = '') {
        assert(arraysEqual(shapeA, shapeB), () => errorMessagePrefix + ` Shapes ${shapeA} and ${shapeB} must match`);
    }
    function assertNonNull(a) {
        assert(a != null, () => `The input to the tensor constructor must be a non-null value.`);
    }
    // NOTE: We explicitly type out what T extends instead of any so that
    // util.flatten on a nested array of number doesn't try to infer T as a
    // number[][], causing us to explicitly type util.flatten<number>().
    /**
     *  Flattens an arbitrarily nested array.
     *
     * ```js
     * const a = [[1, 2], [3, 4], [5, [6, [7]]]];
     * const flat = tf.util.flatten(a);
     * console.log(flat);
     * ```
     *
     *  @param arr The nested array to flatten.
     *  @param result The destination array which holds the elements.
     *  @param skipTypedArray If true, avoids flattening the typed arrays. Defaults
     *      to false.
     *
     * @doc {heading: 'Util', namespace: 'util'}
     */
    function flatten$1(arr, result = [], skipTypedArray = false) {
        if (result == null) {
            result = [];
        }
        if (Array.isArray(arr) || isTypedArray(arr) && !skipTypedArray) {
            for (let i = 0; i < arr.length; ++i) {
                flatten$1(arr[i], result, skipTypedArray);
            }
        }
        else {
            result.push(arr);
        }
        return result;
    }
    /**
     * Returns the size (number of elements) of the tensor given its shape.
     *
     * ```js
     * const shape = [3, 4, 2];
     * const size = tf.util.sizeFromShape(shape);
     * console.log(size);
     * ```
     *
     * @doc {heading: 'Util', namespace: 'util'}
     */
    function sizeFromShape(shape) {
        if (shape.length === 0) {
            // Scalar.
            return 1;
        }
        let size = shape[0];
        for (let i = 1; i < shape.length; i++) {
            size *= shape[i];
        }
        return size;
    }
    function arraysEqual(n1, n2) {
        if (n1 === n2) {
            return true;
        }
        if (n1 == null || n2 == null) {
            return false;
        }
        if (n1.length !== n2.length) {
            return false;
        }
        for (let i = 0; i < n1.length; i++) {
            if (n1[i] !== n2[i]) {
                return false;
            }
        }
        return true;
    }
    function isInt(a) {
        return a % 1 === 0;
    }
    function rightPad(a, size) {
        if (size <= a.length) {
            return a;
        }
        return a + ' '.repeat(size - a.length);
    }
    function parseAxisParam(axis, shape) {
        const rank = shape.length;
        // Normalize input
        axis = axis == null ? shape.map((s, i) => i) : [].concat(axis);
        // Check for valid range
        assert(axis.every(ax => ax >= -rank && ax < rank), () => `All values in axis param must be in range [-${rank}, ${rank}) but ` +
            `got axis ${axis}`);
        // Check for only integers
        assert(axis.every(ax => isInt(ax)), () => `All values in axis param must be integers but ` +
            `got axis ${axis}`);
        // Handle negative axis.
        return axis.map(a => a < 0 ? rank + a : a);
    }
    /** Reduces the shape by removing all dimensions of shape 1. */
    function squeezeShape(shape, axis) {
        const newShape = [];
        const keptDims = [];
        const isEmptyArray = axis != null && Array.isArray(axis) && axis.length === 0;
        const axes = (axis == null || isEmptyArray) ?
            null :
            parseAxisParam(axis, shape).sort();
        let j = 0;
        for (let i = 0; i < shape.length; ++i) {
            if (axes != null) {
                if (axes[j] === i && shape[i] !== 1) {
                    throw new Error(`Can't squeeze axis ${i} since its dim '${shape[i]}' is not 1`);
                }
                if ((axes[j] == null || axes[j] > i) && shape[i] === 1) {
                    newShape.push(shape[i]);
                    keptDims.push(i);
                }
                if (axes[j] <= i) {
                    j++;
                }
            }
            if (shape[i] !== 1) {
                newShape.push(shape[i]);
                keptDims.push(i);
            }
        }
        return { newShape, keptDims };
    }
    function getArrayFromDType(dtype, size) {
        let values = null;
        if (dtype == null || dtype === 'float32') {
            values = new Float32Array(size);
        }
        else if (dtype === 'int32') {
            values = new Int32Array(size);
        }
        else if (dtype === 'bool') {
            values = new Uint8Array(size);
        }
        else if (dtype === 'string') {
            values = new Array(size);
        }
        else {
            throw new Error(`Unknown data type ${dtype}`);
        }
        return values;
    }
    function checkConversionForErrors(vals, dtype) {
        for (let i = 0; i < vals.length; i++) {
            const num = vals[i];
            if (isNaN(num) || !isFinite(num)) {
                throw Error(`A tensor of type ${dtype} being uploaded contains ${num}.`);
            }
        }
    }
    /** Returns true if the dtype is valid. */
    function isValidDtype(dtype) {
        return dtype === 'bool' || dtype === 'complex64' || dtype === 'float32' ||
            dtype === 'int32' || dtype === 'string';
    }
    function isTypedArray(a) {
        return a instanceof Float32Array || a instanceof Int32Array ||
            a instanceof Uint8Array || a instanceof Uint8ClampedArray;
    }
    function bytesPerElement(dtype) {
        if (dtype === 'float32' || dtype === 'int32') {
            return 4;
        }
        else if (dtype === 'complex64') {
            return 8;
        }
        else if (dtype === 'bool') {
            return 1;
        }
        else {
            throw new Error(`Unknown dtype ${dtype}`);
        }
    }
    /**
     * Returns the approximate number of bytes allocated in the string array - 2
     * bytes per character. Computing the exact bytes for a native string in JS
     * is not possible since it depends on the encoding of the html page that
     * serves the website.
     */
    function bytesFromStringArray(arr) {
        if (arr == null) {
            return 0;
        }
        let bytes = 0;
        arr.forEach(x => bytes += x.length);
        return bytes;
    }
    /** Returns true if the value is a string. */
    function isString(value) {
        return typeof value === 'string' || value instanceof String;
    }
    function isBoolean(value) {
        return typeof value === 'boolean';
    }
    function isNumber(value) {
        return typeof value === 'number';
    }
    function inferDtype(values) {
        if (Array.isArray(values)) {
            return inferDtype(values[0]);
        }
        if (values instanceof Float32Array) {
            return 'float32';
        }
        else if (values instanceof Int32Array || values instanceof Uint8Array ||
            values instanceof Uint8ClampedArray) {
            return 'int32';
        }
        else if (isNumber(values)) {
            return 'float32';
        }
        else if (isString(values)) {
            return 'string';
        }
        else if (isBoolean(values)) {
            return 'bool';
        }
        return 'float32';
    }
    function isFunction(f) {
        return !!(f && f.constructor && f.call && f.apply);
    }
    function computeStrides(shape) {
        const rank = shape.length;
        if (rank < 2) {
            return [];
        }
        // Last dimension has implicit stride of 1, thus having D-1 (instead of D)
        // strides.
        const strides = new Array(rank - 1);
        strides[rank - 2] = shape[rank - 1];
        for (let i = rank - 3; i >= 0; --i) {
            strides[i] = strides[i + 1] * shape[i + 1];
        }
        return strides;
    }
    function createNestedArray(offset, shape, a, isComplex = false) {
        const ret = new Array();
        if (shape.length === 1) {
            const d = shape[0] * (isComplex ? 2 : 1);
            for (let i = 0; i < d; i++) {
                ret[i] = a[offset + i];
            }
        }
        else {
            const d = shape[0];
            const rest = shape.slice(1);
            const len = rest.reduce((acc, c) => acc * c) * (isComplex ? 2 : 1);
            for (let i = 0; i < d; i++) {
                ret[i] = createNestedArray(offset + i * len, rest, a, isComplex);
            }
        }
        return ret;
    }
    // Provide a nested array of TypedArray in given shape.
    function toNestedArray(shape, a, isComplex = false) {
        if (shape.length === 0) {
            // Scalar type should return a single number.
            return a[0];
        }
        const size = shape.reduce((acc, c) => acc * c) * (isComplex ? 2 : 1);
        if (size === 0) {
            // A tensor with shape zero should be turned into empty list.
            return [];
        }
        if (size !== a.length) {
            throw new Error(`[${shape}] does not match the input size ${a.length}${isComplex ? ' for a complex tensor' : ''}.`);
        }
        return createNestedArray(0, shape, a, isComplex);
    }
    function makeOnesTypedArray(size, dtype) {
        const array = makeZerosTypedArray(size, dtype);
        for (let i = 0; i < array.length; i++) {
            array[i] = 1;
        }
        return array;
    }
    function makeZerosTypedArray(size, dtype) {
        if (dtype == null || dtype === 'float32' || dtype === 'complex64') {
            return new Float32Array(size);
        }
        else if (dtype === 'int32') {
            return new Int32Array(size);
        }
        else if (dtype === 'bool') {
            return new Uint8Array(size);
        }
        else {
            throw new Error(`Unknown data type ${dtype}`);
        }
    }
    function assertNonNegativeIntegerDimensions(shape) {
        shape.forEach(dimSize => {
            assert(Number.isInteger(dimSize) && dimSize >= 0, () => `Tensor must have a shape comprised of positive integers but got ` +
                `shape [${shape}].`);
        });
    }
    /**
     * This method asserts whether an object is a Promise instance.
     * @param object
     */
    // tslint:disable-next-line: no-any
    function isPromise(object) {
        //  We chose to not use 'obj instanceOf Promise' for two reasons:
        //  1. It only reliably works for es6 Promise, not other Promise
        //  implementations.
        //  2. It doesn't work with framework that uses zone.js. zone.js monkey
        //  patch the async calls, so it is possible the obj (patched) is
        //  comparing to a pre-patched Promise.
        return object && object.then && typeof object.then === 'function';
    }

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // Expects flags from URL in the format ?tfjsflags=FLAG1:1,FLAG2:true.
    const TENSORFLOWJS_FLAGS_PREFIX = 'tfjsflags';
    /**
     * The environment contains evaluated flags as well as the registered platform.
     * This is always used as a global singleton and can be retrieved with
     * `tf.env()`.
     *
     * @doc {heading: 'Environment'}
     */
    class Environment {
        // tslint:disable-next-line: no-any
        constructor(global) {
            this.global = global;
            this.flags = {};
            this.flagRegistry = {};
            this.urlFlags = {};
            // Jasmine spies on this in 'environment_test.ts'
            this.getQueryParams = getQueryParams;
            this.populateURLFlags();
        }
        setPlatform(platformName, platform) {
            if (this.platform != null) {
                if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
                    console.warn(`Platform ${this.platformName} has already been set. ` +
                        `Overwriting the platform with ${platformName}.`);
                }
            }
            this.platformName = platformName;
            this.platform = platform;
        }
        registerFlag(flagName, evaluationFn, setHook) {
            this.flagRegistry[flagName] = { evaluationFn, setHook };
            // Override the flag value from the URL. This has to happen here because
            // the environment is initialized before flags get registered.
            if (this.urlFlags[flagName] != null) {
                const flagValue = this.urlFlags[flagName];
                if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
                    console.warn(`Setting feature override from URL ${flagName}: ${flagValue}.`);
                }
                this.set(flagName, flagValue);
            }
        }
        async getAsync(flagName) {
            if (flagName in this.flags) {
                return this.flags[flagName];
            }
            this.flags[flagName] = await this.evaluateFlag(flagName);
            return this.flags[flagName];
        }
        get(flagName) {
            if (flagName in this.flags) {
                return this.flags[flagName];
            }
            const flagValue = this.evaluateFlag(flagName);
            if (isPromise(flagValue)) {
                throw new Error(`Flag ${flagName} cannot be synchronously evaluated. ` +
                    `Please use getAsync() instead.`);
            }
            this.flags[flagName] = flagValue;
            return this.flags[flagName];
        }
        getNumber(flagName) {
            return this.get(flagName);
        }
        getBool(flagName) {
            return this.get(flagName);
        }
        getFlags() {
            return this.flags;
        }
        // For backwards compatibility.
        get features() {
            return this.flags;
        }
        set(flagName, value) {
            if (this.flagRegistry[flagName] == null) {
                throw new Error(`Cannot set flag ${flagName} as it has not been registered.`);
            }
            this.flags[flagName] = value;
            if (this.flagRegistry[flagName].setHook != null) {
                this.flagRegistry[flagName].setHook(value);
            }
        }
        evaluateFlag(flagName) {
            if (this.flagRegistry[flagName] == null) {
                throw new Error(`Cannot evaluate flag '${flagName}': no evaluation function found.`);
            }
            return this.flagRegistry[flagName].evaluationFn();
        }
        setFlags(flags) {
            this.flags = Object.assign({}, flags);
        }
        reset() {
            this.flags = {};
            this.urlFlags = {};
            this.populateURLFlags();
        }
        populateURLFlags() {
            if (typeof this.global === 'undefined' ||
                typeof this.global.location === 'undefined' ||
                typeof this.global.location.search === 'undefined') {
                return;
            }
            const urlParams = this.getQueryParams(this.global.location.search);
            if (TENSORFLOWJS_FLAGS_PREFIX in urlParams) {
                const keyValues = urlParams[TENSORFLOWJS_FLAGS_PREFIX].split(',');
                keyValues.forEach(keyValue => {
                    const [key, value] = keyValue.split(':');
                    this.urlFlags[key] = parseValue(key, value);
                });
            }
        }
    }
    function getQueryParams(queryString) {
        const params = {};
        queryString.replace(/[?&]([^=?&]+)(?:=([^&]*))?/g, (s, ...t) => {
            decodeParam(params, t[0], t[1]);
            return t.join('=');
        });
        return params;
    }
    function decodeParam(params, name, value) {
        params[decodeURIComponent(name)] = decodeURIComponent(value || '');
    }
    function parseValue(flagName, value) {
        value = value.toLowerCase();
        if (value === 'true' || value === 'false') {
            return value === 'true';
        }
        else if (`${+value}` === value) {
            return +value;
        }
        throw new Error(`Could not parse value flag value ${value} for flag ${flagName}.`);
    }
    /**
     * Returns the current environment (a global singleton).
     *
     * The environment object contains the evaluated feature values as well as the
     * active platform.
     *
     * @doc {heading: 'Environment'}
     */
    function env() {
        return ENV$1;
    }
    let ENV$1 = null;
    function setEnvironmentGlobal(environment) {
        ENV$1 = environment;
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // Note that the identifier globalNameSpace is scoped to this module, but will
    // always resolve to the same global object regardless of how the module is
    // resolved.
    // tslint:disable-next-line:no-any
    let globalNameSpace;
    // tslint:disable-next-line:no-any
    function getGlobalNamespace() {
        if (globalNameSpace == null) {
            // tslint:disable-next-line:no-any
            let ns;
            if (typeof (window) !== 'undefined') {
                ns = window;
            }
            else if (typeof (global) !== 'undefined') {
                ns = global;
            }
            else if (typeof (process) !== 'undefined') {
                ns = process;
            }
            else if (typeof (self) !== 'undefined') {
                ns = self;
            }
            else {
                throw new Error('Could not find a global object');
            }
            globalNameSpace = ns;
        }
        return globalNameSpace;
    }
    // tslint:disable-next-line:no-any
    function getGlobalMap() {
        const ns = getGlobalNamespace();
        if (ns._tfGlobals == null) {
            ns._tfGlobals = new Map();
        }
        return ns._tfGlobals;
    }
    /**
     * Returns a globally accessible 'singleton' object.
     *
     * @param key the name of the object
     * @param init a function to initialize to initialize this object
     *             the first time it is fetched.
     */
    function getGlobal(key, init) {
        const globalMap = getGlobalMap();
        if (globalMap.has(key)) {
            return globalMap.get(key);
        }
        else {
            const singleton = init();
            globalMap.set(key, singleton);
            return globalMap.get(key);
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function warn(...msg) {
        if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
            console.warn(...msg);
        }
    }

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const kernelRegistry = getGlobal('kernelRegistry', () => new Map());
    const gradRegistry = getGlobal('gradRegistry', () => new Map());
    /**
     * Returns the kernel function (code) associated with the provided names.
     *
     * @param kernelName The official name of the kernel.
     * @param backendName The official name of the backend.
     */
    function getKernel(kernelName, backendName) {
        const key = makeKey(kernelName, backendName);
        return kernelRegistry.get(key);
    }
    /**
     * Returns the registered gradient info associated with the provided kernel.
     * @param kernelName The official TF kernel name.
     */
    function getGradient(kernelName) {
        return gradRegistry.get(kernelName);
    }
    function getKernelsForBackend(backendName) {
        const it = kernelRegistry.entries();
        const result = [];
        while (true) {
            const { done, value } = it.next();
            if (done) {
                break;
            }
            const [key, config] = value;
            const [backend,] = key.split('_');
            if (backend === backendName) {
                result.push(config);
            }
        }
        return result;
    }
    /**
     * Registers a gradient function for a given kernel in the global registry,
     * to be used during the back-propagation of that kernel.
     *
     * @param config An object with the following properties:
     * - `kernelName` The name of the kernel that the gradient function is for.
     * - `gradFunc` The function to run during back-propagation.
     */
    function registerGradient(config) {
        const { kernelName } = config;
        if (gradRegistry.has(kernelName)) {
            // TODO (yassogba) after 3.0 assess whether we need to keep this gated
            // to debug mode.
            if (env().getBool('DEBUG')) {
                warn(`Overriding the gradient for '${kernelName}'`);
            }
        }
        gradRegistry.set(kernelName, config);
    }
    function makeKey(kernelName, backendName) {
        return `${backendName}_${kernelName}`;
    }

    var long = Long$1;

    /**
     * wasm optimizations, to do native i64 multiplication and divide
     */
    var wasm = null;

    try {
      wasm = new WebAssembly.Instance(new WebAssembly.Module(new Uint8Array([
        0, 97, 115, 109, 1, 0, 0, 0, 1, 13, 2, 96, 0, 1, 127, 96, 4, 127, 127, 127, 127, 1, 127, 3, 7, 6, 0, 1, 1, 1, 1, 1, 6, 6, 1, 127, 1, 65, 0, 11, 7, 50, 6, 3, 109, 117, 108, 0, 1, 5, 100, 105, 118, 95, 115, 0, 2, 5, 100, 105, 118, 95, 117, 0, 3, 5, 114, 101, 109, 95, 115, 0, 4, 5, 114, 101, 109, 95, 117, 0, 5, 8, 103, 101, 116, 95, 104, 105, 103, 104, 0, 0, 10, 191, 1, 6, 4, 0, 35, 0, 11, 36, 1, 1, 126, 32, 0, 173, 32, 1, 173, 66, 32, 134, 132, 32, 2, 173, 32, 3, 173, 66, 32, 134, 132, 126, 34, 4, 66, 32, 135, 167, 36, 0, 32, 4, 167, 11, 36, 1, 1, 126, 32, 0, 173, 32, 1, 173, 66, 32, 134, 132, 32, 2, 173, 32, 3, 173, 66, 32, 134, 132, 127, 34, 4, 66, 32, 135, 167, 36, 0, 32, 4, 167, 11, 36, 1, 1, 126, 32, 0, 173, 32, 1, 173, 66, 32, 134, 132, 32, 2, 173, 32, 3, 173, 66, 32, 134, 132, 128, 34, 4, 66, 32, 135, 167, 36, 0, 32, 4, 167, 11, 36, 1, 1, 126, 32, 0, 173, 32, 1, 173, 66, 32, 134, 132, 32, 2, 173, 32, 3, 173, 66, 32, 134, 132, 129, 34, 4, 66, 32, 135, 167, 36, 0, 32, 4, 167, 11, 36, 1, 1, 126, 32, 0, 173, 32, 1, 173, 66, 32, 134, 132, 32, 2, 173, 32, 3, 173, 66, 32, 134, 132, 130, 34, 4, 66, 32, 135, 167, 36, 0, 32, 4, 167, 11
      ])), {}).exports;
    } catch (e) {
      // no wasm support :(
    }

    /**
     * Constructs a 64 bit two's-complement integer, given its low and high 32 bit values as *signed* integers.
     *  See the from* functions below for more convenient ways of constructing Longs.
     * @exports Long
     * @class A Long class for representing a 64 bit two's-complement integer value.
     * @param {number} low The low (signed) 32 bits of the long
     * @param {number} high The high (signed) 32 bits of the long
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @constructor
     */
    function Long$1(low, high, unsigned) {

        /**
         * The low 32 bits as a signed value.
         * @type {number}
         */
        this.low = low | 0;

        /**
         * The high 32 bits as a signed value.
         * @type {number}
         */
        this.high = high | 0;

        /**
         * Whether unsigned or not.
         * @type {boolean}
         */
        this.unsigned = !!unsigned;
    }

    // The internal representation of a long is the two given signed, 32-bit values.
    // We use 32-bit pieces because these are the size of integers on which
    // Javascript performs bit-operations.  For operations like addition and
    // multiplication, we split each number into 16 bit pieces, which can easily be
    // multiplied within Javascript's floating-point representation without overflow
    // or change in sign.
    //
    // In the algorithms below, we frequently reduce the negative case to the
    // positive case by negating the input(s) and then post-processing the result.
    // Note that we must ALWAYS check specially whether those values are MIN_VALUE
    // (-2^63) because -MIN_VALUE == MIN_VALUE (since 2^63 cannot be represented as
    // a positive number, it overflows back into a negative).  Not handling this
    // case would often result in infinite recursion.
    //
    // Common constant values ZERO, ONE, NEG_ONE, etc. are defined below the from*
    // methods on which they depend.

    /**
     * An indicator used to reliably determine if an object is a Long or not.
     * @type {boolean}
     * @const
     * @private
     */
    Long$1.prototype.__isLong__;

    Object.defineProperty(Long$1.prototype, "__isLong__", { value: true });

    /**
     * @function
     * @param {*} obj Object
     * @returns {boolean}
     * @inner
     */
    function isLong(obj) {
        return (obj && obj["__isLong__"]) === true;
    }

    /**
     * Tests if the specified object is a Long.
     * @function
     * @param {*} obj Object
     * @returns {boolean}
     */
    Long$1.isLong = isLong;

    /**
     * A cache of the Long representations of small integer values.
     * @type {!Object}
     * @inner
     */
    var INT_CACHE = {};

    /**
     * A cache of the Long representations of small unsigned integer values.
     * @type {!Object}
     * @inner
     */
    var UINT_CACHE = {};

    /**
     * @param {number} value
     * @param {boolean=} unsigned
     * @returns {!Long}
     * @inner
     */
    function fromInt(value, unsigned) {
        var obj, cachedObj, cache;
        if (unsigned) {
            value >>>= 0;
            if (cache = (0 <= value && value < 256)) {
                cachedObj = UINT_CACHE[value];
                if (cachedObj)
                    return cachedObj;
            }
            obj = fromBits(value, (value | 0) < 0 ? -1 : 0, true);
            if (cache)
                UINT_CACHE[value] = obj;
            return obj;
        } else {
            value |= 0;
            if (cache = (-128 <= value && value < 128)) {
                cachedObj = INT_CACHE[value];
                if (cachedObj)
                    return cachedObj;
            }
            obj = fromBits(value, value < 0 ? -1 : 0, false);
            if (cache)
                INT_CACHE[value] = obj;
            return obj;
        }
    }

    /**
     * Returns a Long representing the given 32 bit integer value.
     * @function
     * @param {number} value The 32 bit integer in question
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @returns {!Long} The corresponding Long value
     */
    Long$1.fromInt = fromInt;

    /**
     * @param {number} value
     * @param {boolean=} unsigned
     * @returns {!Long}
     * @inner
     */
    function fromNumber(value, unsigned) {
        if (isNaN(value))
            return unsigned ? UZERO : ZERO;
        if (unsigned) {
            if (value < 0)
                return UZERO;
            if (value >= TWO_PWR_64_DBL)
                return MAX_UNSIGNED_VALUE;
        } else {
            if (value <= -TWO_PWR_63_DBL)
                return MIN_VALUE;
            if (value + 1 >= TWO_PWR_63_DBL)
                return MAX_VALUE;
        }
        if (value < 0)
            return fromNumber(-value, unsigned).neg();
        return fromBits((value % TWO_PWR_32_DBL) | 0, (value / TWO_PWR_32_DBL) | 0, unsigned);
    }

    /**
     * Returns a Long representing the given value, provided that it is a finite number. Otherwise, zero is returned.
     * @function
     * @param {number} value The number in question
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @returns {!Long} The corresponding Long value
     */
    Long$1.fromNumber = fromNumber;

    /**
     * @param {number} lowBits
     * @param {number} highBits
     * @param {boolean=} unsigned
     * @returns {!Long}
     * @inner
     */
    function fromBits(lowBits, highBits, unsigned) {
        return new Long$1(lowBits, highBits, unsigned);
    }

    /**
     * Returns a Long representing the 64 bit integer that comes by concatenating the given low and high bits. Each is
     *  assumed to use 32 bits.
     * @function
     * @param {number} lowBits The low 32 bits
     * @param {number} highBits The high 32 bits
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @returns {!Long} The corresponding Long value
     */
    Long$1.fromBits = fromBits;

    /**
     * @function
     * @param {number} base
     * @param {number} exponent
     * @returns {number}
     * @inner
     */
    var pow_dbl = Math.pow; // Used 4 times (4*8 to 15+4)

    /**
     * @param {string} str
     * @param {(boolean|number)=} unsigned
     * @param {number=} radix
     * @returns {!Long}
     * @inner
     */
    function fromString(str, unsigned, radix) {
        if (str.length === 0)
            throw Error('empty string');
        if (str === "NaN" || str === "Infinity" || str === "+Infinity" || str === "-Infinity")
            return ZERO;
        if (typeof unsigned === 'number') {
            // For goog.math.long compatibility
            radix = unsigned,
            unsigned = false;
        } else {
            unsigned = !! unsigned;
        }
        radix = radix || 10;
        if (radix < 2 || 36 < radix)
            throw RangeError('radix');

        var p;
        if ((p = str.indexOf('-')) > 0)
            throw Error('interior hyphen');
        else if (p === 0) {
            return fromString(str.substring(1), unsigned, radix).neg();
        }

        // Do several (8) digits each time through the loop, so as to
        // minimize the calls to the very expensive emulated div.
        var radixToPower = fromNumber(pow_dbl(radix, 8));

        var result = ZERO;
        for (var i = 0; i < str.length; i += 8) {
            var size = Math.min(8, str.length - i),
                value = parseInt(str.substring(i, i + size), radix);
            if (size < 8) {
                var power = fromNumber(pow_dbl(radix, size));
                result = result.mul(power).add(fromNumber(value));
            } else {
                result = result.mul(radixToPower);
                result = result.add(fromNumber(value));
            }
        }
        result.unsigned = unsigned;
        return result;
    }

    /**
     * Returns a Long representation of the given string, written using the specified radix.
     * @function
     * @param {string} str The textual representation of the Long
     * @param {(boolean|number)=} unsigned Whether unsigned or not, defaults to signed
     * @param {number=} radix The radix in which the text is written (2-36), defaults to 10
     * @returns {!Long} The corresponding Long value
     */
    Long$1.fromString = fromString;

    /**
     * @function
     * @param {!Long|number|string|!{low: number, high: number, unsigned: boolean}} val
     * @param {boolean=} unsigned
     * @returns {!Long}
     * @inner
     */
    function fromValue(val, unsigned) {
        if (typeof val === 'number')
            return fromNumber(val, unsigned);
        if (typeof val === 'string')
            return fromString(val, unsigned);
        // Throws for non-objects, converts non-instanceof Long:
        return fromBits(val.low, val.high, typeof unsigned === 'boolean' ? unsigned : val.unsigned);
    }

    /**
     * Converts the specified value to a Long using the appropriate from* function for its type.
     * @function
     * @param {!Long|number|string|!{low: number, high: number, unsigned: boolean}} val Value
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @returns {!Long}
     */
    Long$1.fromValue = fromValue;

    // NOTE: the compiler should inline these constant values below and then remove these variables, so there should be
    // no runtime penalty for these.

    /**
     * @type {number}
     * @const
     * @inner
     */
    var TWO_PWR_16_DBL = 1 << 16;

    /**
     * @type {number}
     * @const
     * @inner
     */
    var TWO_PWR_24_DBL = 1 << 24;

    /**
     * @type {number}
     * @const
     * @inner
     */
    var TWO_PWR_32_DBL = TWO_PWR_16_DBL * TWO_PWR_16_DBL;

    /**
     * @type {number}
     * @const
     * @inner
     */
    var TWO_PWR_64_DBL = TWO_PWR_32_DBL * TWO_PWR_32_DBL;

    /**
     * @type {number}
     * @const
     * @inner
     */
    var TWO_PWR_63_DBL = TWO_PWR_64_DBL / 2;

    /**
     * @type {!Long}
     * @const
     * @inner
     */
    var TWO_PWR_24 = fromInt(TWO_PWR_24_DBL);

    /**
     * @type {!Long}
     * @inner
     */
    var ZERO = fromInt(0);

    /**
     * Signed zero.
     * @type {!Long}
     */
    Long$1.ZERO = ZERO;

    /**
     * @type {!Long}
     * @inner
     */
    var UZERO = fromInt(0, true);

    /**
     * Unsigned zero.
     * @type {!Long}
     */
    Long$1.UZERO = UZERO;

    /**
     * @type {!Long}
     * @inner
     */
    var ONE = fromInt(1);

    /**
     * Signed one.
     * @type {!Long}
     */
    Long$1.ONE = ONE;

    /**
     * @type {!Long}
     * @inner
     */
    var UONE = fromInt(1, true);

    /**
     * Unsigned one.
     * @type {!Long}
     */
    Long$1.UONE = UONE;

    /**
     * @type {!Long}
     * @inner
     */
    var NEG_ONE = fromInt(-1);

    /**
     * Signed negative one.
     * @type {!Long}
     */
    Long$1.NEG_ONE = NEG_ONE;

    /**
     * @type {!Long}
     * @inner
     */
    var MAX_VALUE = fromBits(0xFFFFFFFF|0, 0x7FFFFFFF|0, false);

    /**
     * Maximum signed value.
     * @type {!Long}
     */
    Long$1.MAX_VALUE = MAX_VALUE;

    /**
     * @type {!Long}
     * @inner
     */
    var MAX_UNSIGNED_VALUE = fromBits(0xFFFFFFFF|0, 0xFFFFFFFF|0, true);

    /**
     * Maximum unsigned value.
     * @type {!Long}
     */
    Long$1.MAX_UNSIGNED_VALUE = MAX_UNSIGNED_VALUE;

    /**
     * @type {!Long}
     * @inner
     */
    var MIN_VALUE = fromBits(0, 0x80000000|0, false);

    /**
     * Minimum signed value.
     * @type {!Long}
     */
    Long$1.MIN_VALUE = MIN_VALUE;

    /**
     * @alias Long.prototype
     * @inner
     */
    var LongPrototype = Long$1.prototype;

    /**
     * Converts the Long to a 32 bit integer, assuming it is a 32 bit integer.
     * @returns {number}
     */
    LongPrototype.toInt = function toInt() {
        return this.unsigned ? this.low >>> 0 : this.low;
    };

    /**
     * Converts the Long to a the nearest floating-point representation of this value (double, 53 bit mantissa).
     * @returns {number}
     */
    LongPrototype.toNumber = function toNumber() {
        if (this.unsigned)
            return ((this.high >>> 0) * TWO_PWR_32_DBL) + (this.low >>> 0);
        return this.high * TWO_PWR_32_DBL + (this.low >>> 0);
    };

    /**
     * Converts the Long to a string written in the specified radix.
     * @param {number=} radix Radix (2-36), defaults to 10
     * @returns {string}
     * @override
     * @throws {RangeError} If `radix` is out of range
     */
    LongPrototype.toString = function toString(radix) {
        radix = radix || 10;
        if (radix < 2 || 36 < radix)
            throw RangeError('radix');
        if (this.isZero())
            return '0';
        if (this.isNegative()) { // Unsigned Longs are never negative
            if (this.eq(MIN_VALUE)) {
                // We need to change the Long value before it can be negated, so we remove
                // the bottom-most digit in this base and then recurse to do the rest.
                var radixLong = fromNumber(radix),
                    div = this.div(radixLong),
                    rem1 = div.mul(radixLong).sub(this);
                return div.toString(radix) + rem1.toInt().toString(radix);
            } else
                return '-' + this.neg().toString(radix);
        }

        // Do several (6) digits each time through the loop, so as to
        // minimize the calls to the very expensive emulated div.
        var radixToPower = fromNumber(pow_dbl(radix, 6), this.unsigned),
            rem = this;
        var result = '';
        while (true) {
            var remDiv = rem.div(radixToPower),
                intval = rem.sub(remDiv.mul(radixToPower)).toInt() >>> 0,
                digits = intval.toString(radix);
            rem = remDiv;
            if (rem.isZero())
                return digits + result;
            else {
                while (digits.length < 6)
                    digits = '0' + digits;
                result = '' + digits + result;
            }
        }
    };

    /**
     * Gets the high 32 bits as a signed integer.
     * @returns {number} Signed high bits
     */
    LongPrototype.getHighBits = function getHighBits() {
        return this.high;
    };

    /**
     * Gets the high 32 bits as an unsigned integer.
     * @returns {number} Unsigned high bits
     */
    LongPrototype.getHighBitsUnsigned = function getHighBitsUnsigned() {
        return this.high >>> 0;
    };

    /**
     * Gets the low 32 bits as a signed integer.
     * @returns {number} Signed low bits
     */
    LongPrototype.getLowBits = function getLowBits() {
        return this.low;
    };

    /**
     * Gets the low 32 bits as an unsigned integer.
     * @returns {number} Unsigned low bits
     */
    LongPrototype.getLowBitsUnsigned = function getLowBitsUnsigned() {
        return this.low >>> 0;
    };

    /**
     * Gets the number of bits needed to represent the absolute value of this Long.
     * @returns {number}
     */
    LongPrototype.getNumBitsAbs = function getNumBitsAbs() {
        if (this.isNegative()) // Unsigned Longs are never negative
            return this.eq(MIN_VALUE) ? 64 : this.neg().getNumBitsAbs();
        var val = this.high != 0 ? this.high : this.low;
        for (var bit = 31; bit > 0; bit--)
            if ((val & (1 << bit)) != 0)
                break;
        return this.high != 0 ? bit + 33 : bit + 1;
    };

    /**
     * Tests if this Long's value equals zero.
     * @returns {boolean}
     */
    LongPrototype.isZero = function isZero() {
        return this.high === 0 && this.low === 0;
    };

    /**
     * Tests if this Long's value equals zero. This is an alias of {@link Long#isZero}.
     * @returns {boolean}
     */
    LongPrototype.eqz = LongPrototype.isZero;

    /**
     * Tests if this Long's value is negative.
     * @returns {boolean}
     */
    LongPrototype.isNegative = function isNegative() {
        return !this.unsigned && this.high < 0;
    };

    /**
     * Tests if this Long's value is positive.
     * @returns {boolean}
     */
    LongPrototype.isPositive = function isPositive() {
        return this.unsigned || this.high >= 0;
    };

    /**
     * Tests if this Long's value is odd.
     * @returns {boolean}
     */
    LongPrototype.isOdd = function isOdd() {
        return (this.low & 1) === 1;
    };

    /**
     * Tests if this Long's value is even.
     * @returns {boolean}
     */
    LongPrototype.isEven = function isEven() {
        return (this.low & 1) === 0;
    };

    /**
     * Tests if this Long's value equals the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.equals = function equals(other) {
        if (!isLong(other))
            other = fromValue(other);
        if (this.unsigned !== other.unsigned && (this.high >>> 31) === 1 && (other.high >>> 31) === 1)
            return false;
        return this.high === other.high && this.low === other.low;
    };

    /**
     * Tests if this Long's value equals the specified's. This is an alias of {@link Long#equals}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.eq = LongPrototype.equals;

    /**
     * Tests if this Long's value differs from the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.notEquals = function notEquals(other) {
        return !this.eq(/* validates */ other);
    };

    /**
     * Tests if this Long's value differs from the specified's. This is an alias of {@link Long#notEquals}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.neq = LongPrototype.notEquals;

    /**
     * Tests if this Long's value differs from the specified's. This is an alias of {@link Long#notEquals}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.ne = LongPrototype.notEquals;

    /**
     * Tests if this Long's value is less than the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.lessThan = function lessThan(other) {
        return this.comp(/* validates */ other) < 0;
    };

    /**
     * Tests if this Long's value is less than the specified's. This is an alias of {@link Long#lessThan}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.lt = LongPrototype.lessThan;

    /**
     * Tests if this Long's value is less than or equal the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.lessThanOrEqual = function lessThanOrEqual(other) {
        return this.comp(/* validates */ other) <= 0;
    };

    /**
     * Tests if this Long's value is less than or equal the specified's. This is an alias of {@link Long#lessThanOrEqual}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.lte = LongPrototype.lessThanOrEqual;

    /**
     * Tests if this Long's value is less than or equal the specified's. This is an alias of {@link Long#lessThanOrEqual}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.le = LongPrototype.lessThanOrEqual;

    /**
     * Tests if this Long's value is greater than the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.greaterThan = function greaterThan(other) {
        return this.comp(/* validates */ other) > 0;
    };

    /**
     * Tests if this Long's value is greater than the specified's. This is an alias of {@link Long#greaterThan}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.gt = LongPrototype.greaterThan;

    /**
     * Tests if this Long's value is greater than or equal the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.greaterThanOrEqual = function greaterThanOrEqual(other) {
        return this.comp(/* validates */ other) >= 0;
    };

    /**
     * Tests if this Long's value is greater than or equal the specified's. This is an alias of {@link Long#greaterThanOrEqual}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.gte = LongPrototype.greaterThanOrEqual;

    /**
     * Tests if this Long's value is greater than or equal the specified's. This is an alias of {@link Long#greaterThanOrEqual}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {boolean}
     */
    LongPrototype.ge = LongPrototype.greaterThanOrEqual;

    /**
     * Compares this Long's value with the specified's.
     * @param {!Long|number|string} other Other value
     * @returns {number} 0 if they are the same, 1 if the this is greater and -1
     *  if the given one is greater
     */
    LongPrototype.compare = function compare(other) {
        if (!isLong(other))
            other = fromValue(other);
        if (this.eq(other))
            return 0;
        var thisNeg = this.isNegative(),
            otherNeg = other.isNegative();
        if (thisNeg && !otherNeg)
            return -1;
        if (!thisNeg && otherNeg)
            return 1;
        // At this point the sign bits are the same
        if (!this.unsigned)
            return this.sub(other).isNegative() ? -1 : 1;
        // Both are positive if at least one is unsigned
        return (other.high >>> 0) > (this.high >>> 0) || (other.high === this.high && (other.low >>> 0) > (this.low >>> 0)) ? -1 : 1;
    };

    /**
     * Compares this Long's value with the specified's. This is an alias of {@link Long#compare}.
     * @function
     * @param {!Long|number|string} other Other value
     * @returns {number} 0 if they are the same, 1 if the this is greater and -1
     *  if the given one is greater
     */
    LongPrototype.comp = LongPrototype.compare;

    /**
     * Negates this Long's value.
     * @returns {!Long} Negated Long
     */
    LongPrototype.negate = function negate() {
        if (!this.unsigned && this.eq(MIN_VALUE))
            return MIN_VALUE;
        return this.not().add(ONE);
    };

    /**
     * Negates this Long's value. This is an alias of {@link Long#negate}.
     * @function
     * @returns {!Long} Negated Long
     */
    LongPrototype.neg = LongPrototype.negate;

    /**
     * Returns the sum of this and the specified Long.
     * @param {!Long|number|string} addend Addend
     * @returns {!Long} Sum
     */
    LongPrototype.add = function add(addend) {
        if (!isLong(addend))
            addend = fromValue(addend);

        // Divide each number into 4 chunks of 16 bits, and then sum the chunks.

        var a48 = this.high >>> 16;
        var a32 = this.high & 0xFFFF;
        var a16 = this.low >>> 16;
        var a00 = this.low & 0xFFFF;

        var b48 = addend.high >>> 16;
        var b32 = addend.high & 0xFFFF;
        var b16 = addend.low >>> 16;
        var b00 = addend.low & 0xFFFF;

        var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
        c00 += a00 + b00;
        c16 += c00 >>> 16;
        c00 &= 0xFFFF;
        c16 += a16 + b16;
        c32 += c16 >>> 16;
        c16 &= 0xFFFF;
        c32 += a32 + b32;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c48 += a48 + b48;
        c48 &= 0xFFFF;
        return fromBits((c16 << 16) | c00, (c48 << 16) | c32, this.unsigned);
    };

    /**
     * Returns the difference of this and the specified Long.
     * @param {!Long|number|string} subtrahend Subtrahend
     * @returns {!Long} Difference
     */
    LongPrototype.subtract = function subtract(subtrahend) {
        if (!isLong(subtrahend))
            subtrahend = fromValue(subtrahend);
        return this.add(subtrahend.neg());
    };

    /**
     * Returns the difference of this and the specified Long. This is an alias of {@link Long#subtract}.
     * @function
     * @param {!Long|number|string} subtrahend Subtrahend
     * @returns {!Long} Difference
     */
    LongPrototype.sub = LongPrototype.subtract;

    /**
     * Returns the product of this and the specified Long.
     * @param {!Long|number|string} multiplier Multiplier
     * @returns {!Long} Product
     */
    LongPrototype.multiply = function multiply(multiplier) {
        if (this.isZero())
            return ZERO;
        if (!isLong(multiplier))
            multiplier = fromValue(multiplier);

        // use wasm support if present
        if (wasm) {
            var low = wasm.mul(this.low,
                               this.high,
                               multiplier.low,
                               multiplier.high);
            return fromBits(low, wasm.get_high(), this.unsigned);
        }

        if (multiplier.isZero())
            return ZERO;
        if (this.eq(MIN_VALUE))
            return multiplier.isOdd() ? MIN_VALUE : ZERO;
        if (multiplier.eq(MIN_VALUE))
            return this.isOdd() ? MIN_VALUE : ZERO;

        if (this.isNegative()) {
            if (multiplier.isNegative())
                return this.neg().mul(multiplier.neg());
            else
                return this.neg().mul(multiplier).neg();
        } else if (multiplier.isNegative())
            return this.mul(multiplier.neg()).neg();

        // If both longs are small, use float multiplication
        if (this.lt(TWO_PWR_24) && multiplier.lt(TWO_PWR_24))
            return fromNumber(this.toNumber() * multiplier.toNumber(), this.unsigned);

        // Divide each long into 4 chunks of 16 bits, and then add up 4x4 products.
        // We can skip products that would overflow.

        var a48 = this.high >>> 16;
        var a32 = this.high & 0xFFFF;
        var a16 = this.low >>> 16;
        var a00 = this.low & 0xFFFF;

        var b48 = multiplier.high >>> 16;
        var b32 = multiplier.high & 0xFFFF;
        var b16 = multiplier.low >>> 16;
        var b00 = multiplier.low & 0xFFFF;

        var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
        c00 += a00 * b00;
        c16 += c00 >>> 16;
        c00 &= 0xFFFF;
        c16 += a16 * b00;
        c32 += c16 >>> 16;
        c16 &= 0xFFFF;
        c16 += a00 * b16;
        c32 += c16 >>> 16;
        c16 &= 0xFFFF;
        c32 += a32 * b00;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c32 += a16 * b16;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c32 += a00 * b32;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c48 += a48 * b00 + a32 * b16 + a16 * b32 + a00 * b48;
        c48 &= 0xFFFF;
        return fromBits((c16 << 16) | c00, (c48 << 16) | c32, this.unsigned);
    };

    /**
     * Returns the product of this and the specified Long. This is an alias of {@link Long#multiply}.
     * @function
     * @param {!Long|number|string} multiplier Multiplier
     * @returns {!Long} Product
     */
    LongPrototype.mul = LongPrototype.multiply;

    /**
     * Returns this Long divided by the specified. The result is signed if this Long is signed or
     *  unsigned if this Long is unsigned.
     * @param {!Long|number|string} divisor Divisor
     * @returns {!Long} Quotient
     */
    LongPrototype.divide = function divide(divisor) {
        if (!isLong(divisor))
            divisor = fromValue(divisor);
        if (divisor.isZero())
            throw Error('division by zero');

        // use wasm support if present
        if (wasm) {
            // guard against signed division overflow: the largest
            // negative number / -1 would be 1 larger than the largest
            // positive number, due to two's complement.
            if (!this.unsigned &&
                this.high === -0x80000000 &&
                divisor.low === -1 && divisor.high === -1) {
                // be consistent with non-wasm code path
                return this;
            }
            var low = (this.unsigned ? wasm.div_u : wasm.div_s)(
                this.low,
                this.high,
                divisor.low,
                divisor.high
            );
            return fromBits(low, wasm.get_high(), this.unsigned);
        }

        if (this.isZero())
            return this.unsigned ? UZERO : ZERO;
        var approx, rem, res;
        if (!this.unsigned) {
            // This section is only relevant for signed longs and is derived from the
            // closure library as a whole.
            if (this.eq(MIN_VALUE)) {
                if (divisor.eq(ONE) || divisor.eq(NEG_ONE))
                    return MIN_VALUE;  // recall that -MIN_VALUE == MIN_VALUE
                else if (divisor.eq(MIN_VALUE))
                    return ONE;
                else {
                    // At this point, we have |other| >= 2, so |this/other| < |MIN_VALUE|.
                    var halfThis = this.shr(1);
                    approx = halfThis.div(divisor).shl(1);
                    if (approx.eq(ZERO)) {
                        return divisor.isNegative() ? ONE : NEG_ONE;
                    } else {
                        rem = this.sub(divisor.mul(approx));
                        res = approx.add(rem.div(divisor));
                        return res;
                    }
                }
            } else if (divisor.eq(MIN_VALUE))
                return this.unsigned ? UZERO : ZERO;
            if (this.isNegative()) {
                if (divisor.isNegative())
                    return this.neg().div(divisor.neg());
                return this.neg().div(divisor).neg();
            } else if (divisor.isNegative())
                return this.div(divisor.neg()).neg();
            res = ZERO;
        } else {
            // The algorithm below has not been made for unsigned longs. It's therefore
            // required to take special care of the MSB prior to running it.
            if (!divisor.unsigned)
                divisor = divisor.toUnsigned();
            if (divisor.gt(this))
                return UZERO;
            if (divisor.gt(this.shru(1))) // 15 >>> 1 = 7 ; with divisor = 8 ; true
                return UONE;
            res = UZERO;
        }

        // Repeat the following until the remainder is less than other:  find a
        // floating-point that approximates remainder / other *from below*, add this
        // into the result, and subtract it from the remainder.  It is critical that
        // the approximate value is less than or equal to the real value so that the
        // remainder never becomes negative.
        rem = this;
        while (rem.gte(divisor)) {
            // Approximate the result of division. This may be a little greater or
            // smaller than the actual value.
            approx = Math.max(1, Math.floor(rem.toNumber() / divisor.toNumber()));

            // We will tweak the approximate result by changing it in the 48-th digit or
            // the smallest non-fractional digit, whichever is larger.
            var log2 = Math.ceil(Math.log(approx) / Math.LN2),
                delta = (log2 <= 48) ? 1 : pow_dbl(2, log2 - 48),

            // Decrease the approximation until it is smaller than the remainder.  Note
            // that if it is too large, the product overflows and is negative.
                approxRes = fromNumber(approx),
                approxRem = approxRes.mul(divisor);
            while (approxRem.isNegative() || approxRem.gt(rem)) {
                approx -= delta;
                approxRes = fromNumber(approx, this.unsigned);
                approxRem = approxRes.mul(divisor);
            }

            // We know the answer can't be zero... and actually, zero would cause
            // infinite recursion since we would make no progress.
            if (approxRes.isZero())
                approxRes = ONE;

            res = res.add(approxRes);
            rem = rem.sub(approxRem);
        }
        return res;
    };

    /**
     * Returns this Long divided by the specified. This is an alias of {@link Long#divide}.
     * @function
     * @param {!Long|number|string} divisor Divisor
     * @returns {!Long} Quotient
     */
    LongPrototype.div = LongPrototype.divide;

    /**
     * Returns this Long modulo the specified.
     * @param {!Long|number|string} divisor Divisor
     * @returns {!Long} Remainder
     */
    LongPrototype.modulo = function modulo(divisor) {
        if (!isLong(divisor))
            divisor = fromValue(divisor);

        // use wasm support if present
        if (wasm) {
            var low = (this.unsigned ? wasm.rem_u : wasm.rem_s)(
                this.low,
                this.high,
                divisor.low,
                divisor.high
            );
            return fromBits(low, wasm.get_high(), this.unsigned);
        }

        return this.sub(this.div(divisor).mul(divisor));
    };

    /**
     * Returns this Long modulo the specified. This is an alias of {@link Long#modulo}.
     * @function
     * @param {!Long|number|string} divisor Divisor
     * @returns {!Long} Remainder
     */
    LongPrototype.mod = LongPrototype.modulo;

    /**
     * Returns this Long modulo the specified. This is an alias of {@link Long#modulo}.
     * @function
     * @param {!Long|number|string} divisor Divisor
     * @returns {!Long} Remainder
     */
    LongPrototype.rem = LongPrototype.modulo;

    /**
     * Returns the bitwise NOT of this Long.
     * @returns {!Long}
     */
    LongPrototype.not = function not() {
        return fromBits(~this.low, ~this.high, this.unsigned);
    };

    /**
     * Returns the bitwise AND of this Long and the specified.
     * @param {!Long|number|string} other Other Long
     * @returns {!Long}
     */
    LongPrototype.and = function and(other) {
        if (!isLong(other))
            other = fromValue(other);
        return fromBits(this.low & other.low, this.high & other.high, this.unsigned);
    };

    /**
     * Returns the bitwise OR of this Long and the specified.
     * @param {!Long|number|string} other Other Long
     * @returns {!Long}
     */
    LongPrototype.or = function or(other) {
        if (!isLong(other))
            other = fromValue(other);
        return fromBits(this.low | other.low, this.high | other.high, this.unsigned);
    };

    /**
     * Returns the bitwise XOR of this Long and the given one.
     * @param {!Long|number|string} other Other Long
     * @returns {!Long}
     */
    LongPrototype.xor = function xor(other) {
        if (!isLong(other))
            other = fromValue(other);
        return fromBits(this.low ^ other.low, this.high ^ other.high, this.unsigned);
    };

    /**
     * Returns this Long with bits shifted to the left by the given amount.
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shiftLeft = function shiftLeft(numBits) {
        if (isLong(numBits))
            numBits = numBits.toInt();
        if ((numBits &= 63) === 0)
            return this;
        else if (numBits < 32)
            return fromBits(this.low << numBits, (this.high << numBits) | (this.low >>> (32 - numBits)), this.unsigned);
        else
            return fromBits(0, this.low << (numBits - 32), this.unsigned);
    };

    /**
     * Returns this Long with bits shifted to the left by the given amount. This is an alias of {@link Long#shiftLeft}.
     * @function
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shl = LongPrototype.shiftLeft;

    /**
     * Returns this Long with bits arithmetically shifted to the right by the given amount.
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shiftRight = function shiftRight(numBits) {
        if (isLong(numBits))
            numBits = numBits.toInt();
        if ((numBits &= 63) === 0)
            return this;
        else if (numBits < 32)
            return fromBits((this.low >>> numBits) | (this.high << (32 - numBits)), this.high >> numBits, this.unsigned);
        else
            return fromBits(this.high >> (numBits - 32), this.high >= 0 ? 0 : -1, this.unsigned);
    };

    /**
     * Returns this Long with bits arithmetically shifted to the right by the given amount. This is an alias of {@link Long#shiftRight}.
     * @function
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shr = LongPrototype.shiftRight;

    /**
     * Returns this Long with bits logically shifted to the right by the given amount.
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shiftRightUnsigned = function shiftRightUnsigned(numBits) {
        if (isLong(numBits))
            numBits = numBits.toInt();
        numBits &= 63;
        if (numBits === 0)
            return this;
        else {
            var high = this.high;
            if (numBits < 32) {
                var low = this.low;
                return fromBits((low >>> numBits) | (high << (32 - numBits)), high >>> numBits, this.unsigned);
            } else if (numBits === 32)
                return fromBits(high, 0, this.unsigned);
            else
                return fromBits(high >>> (numBits - 32), 0, this.unsigned);
        }
    };

    /**
     * Returns this Long with bits logically shifted to the right by the given amount. This is an alias of {@link Long#shiftRightUnsigned}.
     * @function
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shru = LongPrototype.shiftRightUnsigned;

    /**
     * Returns this Long with bits logically shifted to the right by the given amount. This is an alias of {@link Long#shiftRightUnsigned}.
     * @function
     * @param {number|!Long} numBits Number of bits
     * @returns {!Long} Shifted Long
     */
    LongPrototype.shr_u = LongPrototype.shiftRightUnsigned;

    /**
     * Converts this Long to signed.
     * @returns {!Long} Signed long
     */
    LongPrototype.toSigned = function toSigned() {
        if (!this.unsigned)
            return this;
        return fromBits(this.low, this.high, false);
    };

    /**
     * Converts this Long to unsigned.
     * @returns {!Long} Unsigned long
     */
    LongPrototype.toUnsigned = function toUnsigned() {
        if (this.unsigned)
            return this;
        return fromBits(this.low, this.high, true);
    };

    /**
     * Converts this Long to its byte representation.
     * @param {boolean=} le Whether little or big endian, defaults to big endian
     * @returns {!Array.<number>} Byte representation
     */
    LongPrototype.toBytes = function toBytes(le) {
        return le ? this.toBytesLE() : this.toBytesBE();
    };

    /**
     * Converts this Long to its little endian byte representation.
     * @returns {!Array.<number>} Little endian byte representation
     */
    LongPrototype.toBytesLE = function toBytesLE() {
        var hi = this.high,
            lo = this.low;
        return [
            lo        & 0xff,
            lo >>>  8 & 0xff,
            lo >>> 16 & 0xff,
            lo >>> 24       ,
            hi        & 0xff,
            hi >>>  8 & 0xff,
            hi >>> 16 & 0xff,
            hi >>> 24
        ];
    };

    /**
     * Converts this Long to its big endian byte representation.
     * @returns {!Array.<number>} Big endian byte representation
     */
    LongPrototype.toBytesBE = function toBytesBE() {
        var hi = this.high,
            lo = this.low;
        return [
            hi >>> 24       ,
            hi >>> 16 & 0xff,
            hi >>>  8 & 0xff,
            hi        & 0xff,
            lo >>> 24       ,
            lo >>> 16 & 0xff,
            lo >>>  8 & 0xff,
            lo        & 0xff
        ];
    };

    /**
     * Creates a Long from its byte representation.
     * @param {!Array.<number>} bytes Byte representation
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @param {boolean=} le Whether little or big endian, defaults to big endian
     * @returns {Long} The corresponding Long value
     */
    Long$1.fromBytes = function fromBytes(bytes, unsigned, le) {
        return le ? Long$1.fromBytesLE(bytes, unsigned) : Long$1.fromBytesBE(bytes, unsigned);
    };

    /**
     * Creates a Long from its little endian byte representation.
     * @param {!Array.<number>} bytes Little endian byte representation
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @returns {Long} The corresponding Long value
     */
    Long$1.fromBytesLE = function fromBytesLE(bytes, unsigned) {
        return new Long$1(
            bytes[0]       |
            bytes[1] <<  8 |
            bytes[2] << 16 |
            bytes[3] << 24,
            bytes[4]       |
            bytes[5] <<  8 |
            bytes[6] << 16 |
            bytes[7] << 24,
            unsigned
        );
    };

    /**
     * Creates a Long from its big endian byte representation.
     * @param {!Array.<number>} bytes Big endian byte representation
     * @param {boolean=} unsigned Whether unsigned or not, defaults to signed
     * @returns {Long} The corresponding Long value
     */
    Long$1.fromBytesBE = function fromBytesBE(bytes, unsigned) {
        return new Long$1(
            bytes[4] << 24 |
            bytes[5] << 16 |
            bytes[6] <<  8 |
            bytes[7],
            bytes[0] << 24 |
            bytes[1] << 16 |
            bytes[2] <<  8 |
            bytes[3],
            unsigned
        );
    };

    var LongExports = /*#__PURE__*/_mergeNamespaces({
        __proto__: null,
        'default': long
    }, [long]);

    /**
     * @license
     * Copyright 2021 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // tslint:disable-next-line
    const Long = 
    // tslint:disable-next-line
    long || LongExports;
    function hexToLong(hex) {
        return Long.fromString(hex, true, 16);
    }
    // Some primes between 2^63 and 2^64 for various uses.
    // Hex 0xc3a5c85c97cb3127
    hexToLong('c3a5c85c97cb3127');
    // Hex 0xb492b66fbe98f273
    hexToLong('b492b66fbe98f273');
    // Hex 0x9ae16a3b2f90404f
    hexToLong('9ae16a3b2f90404f');

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function noConversionNeeded(a, dtype) {
        return (a instanceof Float32Array && dtype === 'float32') ||
            (a instanceof Int32Array && dtype === 'int32') ||
            (a instanceof Uint8Array && dtype === 'bool');
    }
    function toTypedArray(a, dtype) {
        if (dtype === 'string') {
            throw new Error('Cannot convert a string[] to a TypedArray');
        }
        if (Array.isArray(a)) {
            a = flatten$1(a);
        }
        if (env().getBool('DEBUG')) {
            checkConversionForErrors(a, dtype);
        }
        if (noConversionNeeded(a, dtype)) {
            return a;
        }
        if (dtype == null || dtype === 'float32' || dtype === 'complex64') {
            return new Float32Array(a);
        }
        else if (dtype === 'int32') {
            return new Int32Array(a);
        }
        else if (dtype === 'bool') {
            const bool = new Uint8Array(a.length);
            for (let i = 0; i < bool.length; ++i) {
                if (Math.round(a[i]) !== 0) {
                    bool[i] = 1;
                }
            }
            return bool;
        }
        else {
            throw new Error(`Unknown data type ${dtype}`);
        }
    }
    /**
     * Returns the current high-resolution time in milliseconds relative to an
     * arbitrary time in the past. It works across different platforms (node.js,
     * browsers).
     *
     * ```js
     * console.log(tf.util.now());
     * ```
     *
     * @doc {heading: 'Util', namespace: 'util'}
     */
    function now() {
        return env().platform.now();
    }
    /**
     * Encodes the provided string into bytes using the provided encoding scheme.
     *
     * @param s The string to encode.
     * @param encoding The encoding scheme. Defaults to utf-8.
     *
     * @doc {heading: 'Util'}
     */
    function encodeString(s, encoding = 'utf-8') {
        encoding = encoding || 'utf-8';
        return env().platform.encode(s, encoding);
    }
    /**
     * Decodes the provided bytes into a string using the provided encoding scheme.
     * @param bytes The bytes to decode.
     *
     * @param encoding The encoding scheme. Defaults to utf-8.
     *
     * @doc {heading: 'Util'}
     */
    function decodeString(bytes, encoding = 'utf-8') {
        encoding = encoding || 'utf-8';
        return env().platform.decode(bytes, encoding);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    class Profiler {
        constructor(backendTimer, logger) {
            this.backendTimer = backendTimer;
            this.logger = logger;
            if (logger == null) {
                this.logger = new Logger();
            }
        }
        profileKernel(kernelName, inputs, f) {
            let outputs;
            const holdResultWrapperFn = () => {
                outputs = f();
            };
            let timer;
            const start = now();
            if (this.backendTimer.timerAvailable()) {
                timer = this.backendTimer.time(holdResultWrapperFn);
            }
            else {
                holdResultWrapperFn();
                for (const output of outputs) {
                    output.dataSync();
                }
                timer = Promise.resolve({ kernelMs: now() - start });
            }
            if (env().getBool('CHECK_COMPUTATION_FOR_ERRORS')) {
                for (let i = 0; i < outputs.length; i++) {
                    const output = outputs[i];
                    // Dangling promise here because we don't want to propagate up
                    // asynchronicity.
                    output.data().then(tensorVals => {
                        checkComputationForErrors(tensorVals, output.dtype, kernelName);
                    });
                }
            }
            const kernelProfile = {
                kernelName,
                outputs,
                inputs,
                timeMs: timer.then(timing => timing.kernelMs),
                extraInfo: timer.then(timing => timing.getExtraProfileInfo != null ?
                    timing.getExtraProfileInfo() :
                    '')
            };
            return kernelProfile;
        }
        logKernelProfile(kernelProfile) {
            const { kernelName, outputs, timeMs, inputs, extraInfo } = kernelProfile;
            outputs.forEach(result => {
                Promise.all([result.data(), timeMs, extraInfo]).then(valueContainer => {
                    this.logger.logKernelProfile(kernelName, result, valueContainer[0], valueContainer[1], inputs, valueContainer[2]);
                });
            });
        }
    }
    function checkComputationForErrors(vals, dtype, kernelName) {
        if (dtype !== 'float32') {
            // Only floating point computations will generate NaN values
            return false;
        }
        for (let i = 0; i < vals.length; i++) {
            const num = vals[i];
            if (isNaN(num) || !isFinite(num)) {
                // Throwing custom exception so behavior is testable.
                console.warn(`Found ${num} in the result of '${kernelName}'`);
                return true;
            }
        }
        return false;
    }
    class Logger {
        logKernelProfile(name, result, vals, timeMs, inputs, extraInfo) {
            const time = typeof timeMs === 'number' ? rightPad(`${timeMs}ms`, 9) :
                timeMs['error'];
            const paddedName = rightPad(name, 25);
            const rank = result.rank;
            const size = result.size;
            const shape = rightPad(result.shape.toString(), 14);
            let inputShapesDescription = '';
            for (const name in inputs) {
                const input = inputs[name];
                if (input != null) {
                    // The input might be a non-tensor (e.g HTMLImageElement), in which case
                    // we claim the output shape as input shape.
                    const inputShape = input.shape || result.shape;
                    const inputRank = inputShape.length;
                    inputShapesDescription +=
                        `${name}: ${inputRank}D ${inputRank > 0 ? inputShape : ''} `;
                }
            }
            console.log(`%c${paddedName}\t%c${time}\t%c${rank}D ${shape}\t%c${size}\t%c${inputShapesDescription}\t%c${extraInfo}`, 'font-weight:bold', 'color:red', 'color:blue', 'color: orange', 'color: green', 'color: steelblue');
        }
    }

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes a list of TapeNodes that connect x to y, filtering everything else
     * out and preserving the order of the original tape elements.
     *
     * @param tape The tape elements to filter.
     * @param xs The input Tensors.
     * @param y The output Tensor.
     */
    function getFilteredNodesXToY(tape, xs, y) {
        // Forward pass to compute all the nodes and Tensors that are transitively a
        // function of x.
        const tensorsFromX = {};
        const nodesFromX = {};
        for (let i = 0; i < xs.length; i++) {
            tensorsFromX[xs[i].id] = true;
        }
        for (let i = 0; i < tape.length; i++) {
            const node = tape[i];
            const nodeInputs = node.inputs;
            for (const inputName in nodeInputs) {
                const input = nodeInputs[inputName];
                let anyInputFromX = false;
                for (let j = 0; j < xs.length; j++) {
                    if (tensorsFromX[input.id]) {
                        node.outputs.forEach(output => tensorsFromX[output.id] = true);
                        anyInputFromX = true;
                        nodesFromX[node.id] = true;
                        break;
                    }
                }
                if (anyInputFromX) {
                    break;
                }
            }
        }
        // Backward pass to find all of the nodes and Tensors that lead to y.
        const tensorsLeadToY = {};
        tensorsLeadToY[y.id] = true;
        const nodesToY = {};
        for (let i = tape.length - 1; i >= 0; i--) {
            const node = tape[i];
            const nodeInputs = node.inputs;
            // If any of the outputs lead to y, mark all of the inputs as leading to y.
            for (let j = 0; j < node.outputs.length; j++) {
                if (tensorsLeadToY[node.outputs[j].id]) {
                    for (const inputName in nodeInputs) {
                        tensorsLeadToY[nodeInputs[inputName].id] = true;
                        nodesToY[node.id] = true;
                    }
                    break;
                }
            }
        }
        // Return the paths that come from x and lead to y.
        const filteredTape = [];
        for (let i = 0; i < tape.length; i++) {
            const node = tape[i];
            if (nodesFromX[node.id] && nodesToY[node.id]) {
                // Prune the inputs from the node that aren't a function of x.
                const prunedInputs = {};
                for (const inputName in node.inputs) {
                    const nodeInput = node.inputs[inputName];
                    if (tensorsFromX[nodeInput.id]) {
                        prunedInputs[inputName] = nodeInput;
                    }
                }
                // Copy the node and overwrite inputsAndArgs to the pruned version.
                const prunedNode = Object.assign({}, node);
                prunedNode.inputs = prunedInputs;
                prunedNode.outputs = node.outputs;
                filteredTape.push(prunedNode);
            }
        }
        return filteredTape;
    }
    /**
     * Backpropagate gradients through the filtered TapeNodes.
     *
     * @param tensorAccumulatedGradientMap A map of Tensor to its gradient. This map
     * is mutated by this method.
     * @param filteredTape The filtered TapeNodes to backprop through.
     */
    function backpropagateGradients(tensorAccumulatedGradientMap, filteredTape, tidy, add) {
        // Walk the tape backward and keep a map of Tensor to its gradient.
        for (let i = filteredTape.length - 1; i >= 0; i--) {
            const node = filteredTape[i];
            const dys = [];
            node.outputs.forEach(o => {
                const gradTensor = tensorAccumulatedGradientMap[o.id];
                if (gradTensor != null) {
                    dys.push(gradTensor);
                }
                else {
                    // This particular output is not in the back-propagation subgraph, so it
                    // does not affect the final output, thus we put null for its dy.
                    dys.push(null);
                }
            });
            if (node.gradient == null) {
                throw new Error(`Cannot compute gradient: gradient function not found ` +
                    `for ${node.kernelName}.`);
            }
            // Backprop dy through this node and accumulate gradients over the inputs.
            const inputGradients = node.gradient(dys);
            for (const inputName in node.inputs) {
                if (!(inputName in inputGradients)) {
                    throw new Error(`Cannot backprop through input ${inputName}. ` +
                        `Available gradients found: ${Object.keys(inputGradients)}.`);
                }
                // Call the gradient function.
                const dx = tidy(() => inputGradients[inputName]());
                if (dx.dtype !== 'float32') {
                    throw new Error(`Error in gradient for op ${node.kernelName}. The gradient of input ` +
                        `${inputName} must have 'float32' dtype, but has '${dx.dtype}'`);
                }
                const x = node.inputs[inputName];
                if (!arraysEqual(dx.shape, x.shape)) {
                    throw new Error(`Error in gradient for op ${node.kernelName}. The gradient of input ` +
                        `'${inputName}' has shape '${dx.shape}', which does not match ` +
                        `the shape of the input '${x.shape}'`);
                }
                if (tensorAccumulatedGradientMap[x.id] == null) {
                    tensorAccumulatedGradientMap[x.id] = dx;
                }
                else {
                    const curGradient = tensorAccumulatedGradientMap[x.id];
                    tensorAccumulatedGradientMap[x.id] = add(curGradient, dx);
                    curGradient.dispose();
                }
            }
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // Maximum number of values before we decide to show ellipsis.
    const FORMAT_LIMIT_NUM_VALS = 20;
    // Number of first and last values to show when displaying a, b,...,y, z.
    const FORMAT_NUM_FIRST_LAST_VALS = 3;
    // Number of significant digits to show.
    const FORMAT_NUM_SIG_DIGITS = 7;
    function tensorToString(vals, shape, dtype, verbose) {
        const strides = computeStrides(shape);
        const padPerCol = computeMaxSizePerColumn(vals, shape, dtype, strides);
        const rank = shape.length;
        const valsLines = subTensorToString(vals, shape, dtype, strides, padPerCol);
        const lines = ['Tensor'];
        if (verbose) {
            lines.push(`  dtype: ${dtype}`);
            lines.push(`  rank: ${rank}`);
            lines.push(`  shape: [${shape}]`);
            lines.push(`  values:`);
        }
        lines.push(valsLines.map(l => '    ' + l).join('\n'));
        return lines.join('\n');
    }
    function computeMaxSizePerColumn(vals, shape, dtype, strides) {
        const n = sizeFromShape(shape);
        const numCols = strides[strides.length - 1];
        const padPerCol = new Array(numCols).fill(0);
        const rank = shape.length;
        const valuesOrTuples = dtype === 'complex64' ? createComplexTuples(vals) : vals;
        if (rank > 1) {
            for (let row = 0; row < n / numCols; row++) {
                const offset = row * numCols;
                for (let j = 0; j < numCols; j++) {
                    padPerCol[j] = Math.max(padPerCol[j], valToString(valuesOrTuples[offset + j], 0, dtype).length);
                }
            }
        }
        return padPerCol;
    }
    function valToString(val, pad, dtype) {
        let valStr;
        if (Array.isArray(val)) {
            valStr = `${parseFloat(val[0].toFixed(FORMAT_NUM_SIG_DIGITS))} + ` +
                `${parseFloat(val[1].toFixed(FORMAT_NUM_SIG_DIGITS))}j`;
        }
        else if (isString(val)) {
            valStr = `'${val}'`;
        }
        else if (dtype === 'bool') {
            valStr = boolNumToString(val);
        }
        else {
            valStr = parseFloat(val.toFixed(FORMAT_NUM_SIG_DIGITS)).toString();
        }
        return rightPad(valStr, pad);
    }
    function boolNumToString(v) {
        return v === 0 ? 'false' : 'true';
    }
    function subTensorToString(vals, shape, dtype, strides, padPerCol, isLast = true) {
        const storagePerElement = dtype === 'complex64' ? 2 : 1;
        const size = shape[0];
        const rank = shape.length;
        if (rank === 0) {
            if (dtype === 'complex64') {
                const complexTuple = createComplexTuples(vals);
                return [valToString(complexTuple[0], 0, dtype)];
            }
            if (dtype === 'bool') {
                return [boolNumToString(vals[0])];
            }
            return [vals[0].toString()];
        }
        if (rank === 1) {
            if (size > FORMAT_LIMIT_NUM_VALS) {
                const firstValsSize = FORMAT_NUM_FIRST_LAST_VALS * storagePerElement;
                let firstVals = Array.from(vals.slice(0, firstValsSize));
                let lastVals = Array.from(vals.slice((size - FORMAT_NUM_FIRST_LAST_VALS) * storagePerElement, size * storagePerElement));
                if (dtype === 'complex64') {
                    firstVals = createComplexTuples(firstVals);
                    lastVals = createComplexTuples(lastVals);
                }
                return [
                    '[' +
                        firstVals.map((x, i) => valToString(x, padPerCol[i], dtype))
                            .join(', ') +
                        ', ..., ' +
                        lastVals
                            .map((x, i) => valToString(x, padPerCol[size - FORMAT_NUM_FIRST_LAST_VALS + i], dtype))
                            .join(', ') +
                        ']'
                ];
            }
            const displayVals = dtype === 'complex64' ? createComplexTuples(vals) :
                Array.from(vals);
            return [
                '[' +
                    displayVals.map((x, i) => valToString(x, padPerCol[i], dtype))
                        .join(', ') +
                    ']'
            ];
        }
        // The array is rank 2 or more.
        const subshape = shape.slice(1);
        const substrides = strides.slice(1);
        const stride = strides[0] * storagePerElement;
        const lines = [];
        if (size > FORMAT_LIMIT_NUM_VALS) {
            for (let i = 0; i < FORMAT_NUM_FIRST_LAST_VALS; i++) {
                const start = i * stride;
                const end = start + stride;
                lines.push(...subTensorToString(vals.slice(start, end), subshape, dtype, substrides, padPerCol, false /* isLast */));
            }
            lines.push('...');
            for (let i = size - FORMAT_NUM_FIRST_LAST_VALS; i < size; i++) {
                const start = i * stride;
                const end = start + stride;
                lines.push(...subTensorToString(vals.slice(start, end), subshape, dtype, substrides, padPerCol, i === size - 1 /* isLast */));
            }
        }
        else {
            for (let i = 0; i < size; i++) {
                const start = i * stride;
                const end = start + stride;
                lines.push(...subTensorToString(vals.slice(start, end), subshape, dtype, substrides, padPerCol, i === size - 1 /* isLast */));
            }
        }
        const sep = rank === 2 ? ',' : '';
        lines[0] = '[' + lines[0] + sep;
        for (let i = 1; i < lines.length - 1; i++) {
            lines[i] = ' ' + lines[i] + sep;
        }
        let newLineSep = ',\n';
        for (let i = 2; i < rank; i++) {
            newLineSep += '\n';
        }
        lines[lines.length - 1] =
            ' ' + lines[lines.length - 1] + ']' + (isLast ? '' : newLineSep);
        return lines;
    }
    function createComplexTuples(vals) {
        const complexTuples = [];
        for (let i = 0; i < vals.length; i += 2) {
            complexTuples.push([vals[i], vals[i + 1]]);
        }
        return complexTuples;
    }

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * A mutable object, similar to `tf.Tensor`, that allows users to set values
     * at locations before converting to an immutable `tf.Tensor`.
     *
     * See `tf.buffer` for creating a tensor buffer.
     *
     * @doc {heading: 'Tensors', subheading: 'Classes'}
     */
    class TensorBuffer {
        constructor(shape, dtype, values) {
            this.dtype = dtype;
            this.shape = shape.slice();
            this.size = sizeFromShape(shape);
            if (values != null) {
                const n = values.length;
                assert(n === this.size, () => `Length of values '${n}' does not match the size ` +
                    `inferred by the shape '${this.size}'.`);
            }
            if (dtype === 'complex64') {
                throw new Error(`complex64 dtype TensorBuffers are not supported. Please create ` +
                    `a TensorBuffer for the real and imaginary parts separately and ` +
                    `call tf.complex(real, imag).`);
            }
            this.values = values || getArrayFromDType(dtype, this.size);
            this.strides = computeStrides(shape);
        }
        /**
         * Sets a value in the buffer at a given location.
         *
         * @param value The value to set.
         * @param locs  The location indices.
         *
         * @doc {heading: 'Tensors', subheading: 'Creation'}
         */
        set(value, ...locs) {
            if (locs.length === 0) {
                locs = [0];
            }
            assert(locs.length === this.rank, () => `The number of provided coordinates (${locs.length}) must ` +
                `match the rank (${this.rank})`);
            const index = this.locToIndex(locs);
            this.values[index] = value;
        }
        /**
         * Returns the value in the buffer at the provided location.
         *
         * @param locs The location indices.
         *
         * @doc {heading: 'Tensors', subheading: 'Creation'}
         */
        get(...locs) {
            if (locs.length === 0) {
                locs = [0];
            }
            let i = 0;
            for (const loc of locs) {
                if (loc < 0 || loc >= this.shape[i]) {
                    const msg = `Requested out of range element at ${locs}. ` +
                        `  Buffer shape=${this.shape}`;
                    throw new Error(msg);
                }
                i++;
            }
            let index = locs[locs.length - 1];
            for (let i = 0; i < locs.length - 1; ++i) {
                index += this.strides[i] * locs[i];
            }
            return this.values[index];
        }
        locToIndex(locs) {
            if (this.rank === 0) {
                return 0;
            }
            else if (this.rank === 1) {
                return locs[0];
            }
            let index = locs[locs.length - 1];
            for (let i = 0; i < locs.length - 1; ++i) {
                index += this.strides[i] * locs[i];
            }
            return index;
        }
        indexToLoc(index) {
            if (this.rank === 0) {
                return [];
            }
            else if (this.rank === 1) {
                return [index];
            }
            const locs = new Array(this.shape.length);
            for (let i = 0; i < locs.length - 1; ++i) {
                locs[i] = Math.floor(index / this.strides[i]);
                index -= locs[i] * this.strides[i];
            }
            locs[locs.length - 1] = index;
            return locs;
        }
        get rank() {
            return this.shape.length;
        }
        /**
         * Creates an immutable `tf.Tensor` object from the buffer.
         *
         * @doc {heading: 'Tensors', subheading: 'Creation'}
         */
        toTensor() {
            return trackerFn().makeTensor(this.values, this.shape, this.dtype);
        }
    }
    // For tracking tensor creation and disposal.
    let trackerFn = null;
    // Used by chaining methods to call into ops.
    let opHandler = null;
    /**
     * An external consumer can register itself as the tensor tracker. This way
     * the Tensor class can notify the tracker for every tensor created and
     * disposed.
     */
    function setTensorTracker(fn) {
        trackerFn = fn;
    }
    /**
     * A `tf.Tensor` object represents an immutable, multidimensional array of
     * numbers that has a shape and a data type.
     *
     * For performance reasons, functions that create tensors do not necessarily
     * perform a copy of the data passed to them (e.g. if the data is passed as a
     * `Float32Array`), and changes to the data will change the tensor. This is not
     * a feature and is not supported. To avoid this behavior, use the tensor before
     * changing the input data or create a copy with `copy = tf.add(yourTensor, 0)`.
     *
     * See `tf.tensor` for details on how to create a `tf.Tensor`.
     *
     * @doc {heading: 'Tensors', subheading: 'Classes'}
     */
    class Tensor {
        constructor(shape, dtype, dataId, id) {
            /** Whether this tensor has been globally kept. */
            this.kept = false;
            this.isDisposedInternal = false;
            this.shape = shape.slice();
            this.dtype = dtype || 'float32';
            this.size = sizeFromShape(shape);
            this.strides = computeStrides(shape);
            this.dataId = dataId;
            this.id = id;
            this.rankType = (this.rank < 5 ? this.rank.toString() : 'higher');
        }
        get rank() {
            return this.shape.length;
        }
        /**
         * Returns a promise of `tf.TensorBuffer` that holds the underlying data.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        async buffer() {
            const vals = await this.data();
            return opHandler.buffer(this.shape, this.dtype, vals);
        }
        /**
         * Returns a `tf.TensorBuffer` that holds the underlying data.
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        bufferSync() {
            return opHandler.buffer(this.shape, this.dtype, this.dataSync());
        }
        /**
         * Returns the tensor data as a nested array. The transfer of data is done
         * asynchronously.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        async array() {
            const vals = await this.data();
            return toNestedArray(this.shape, vals, this.dtype === 'complex64');
        }
        /**
         * Returns the tensor data as a nested array. The transfer of data is done
         * synchronously.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        arraySync() {
            return toNestedArray(this.shape, this.dataSync(), this.dtype === 'complex64');
        }
        /**
         * Asynchronously downloads the values from the `tf.Tensor`. Returns a
         * promise of `TypedArray` that resolves when the computation has finished.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        async data() {
            this.throwIfDisposed();
            const data = trackerFn().read(this.dataId);
            if (this.dtype === 'string') {
                const bytes = await data;
                try {
                    return bytes.map(b => decodeString(b));
                }
                catch (_a) {
                    throw new Error('Failed to decode the string bytes into utf-8. ' +
                        'To get the original bytes, call tensor.bytes().');
                }
            }
            return data;
        }
        /**
         * Copy the tensor's data to a new GPU resource. Comparing to the `dataSync()`
         * and `data()`, this method prevents data from being downloaded to CPU.
         *
         * For WebGL backend, the data will be stored on a densely packed texture.
         * This means that the texture will use the RGBA channels to store value.
         *
         * For WebGPU backend, the data will be stored on a buffer. There is no
         * parameter, so can not use a user-defined size to create the buffer.
         *
         * @param options:
         *     For WebGL,
         *         - customTexShape: Optional. If set, will use the user defined
         *     texture shape to create the texture.
         *
         * @returns For WebGL backend, a GPUData contains the new texture and
         *     its information.
         *     {
         *        tensorRef: The tensor that is associated with this texture,
         *        texture: WebGLTexture,
         *        texShape: [number, number] // [height, width]
         *     }
         *
         *     For WebGPU backend, a GPUData contains the new buffer and
         *     its information.
         *     {
         *        tensorRef: The tensor that is associated with this buffer,
         *        buffer: GPUBuffer,
         *        bufSize: number
         *     }
         *
         *     Remember to dispose the GPUData after it is used by
         *     `res.tensorRef.dispose()`.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        dataToGPU(options) {
            this.throwIfDisposed();
            return trackerFn().readToGPU(this.dataId, options);
        }
        /**
         * Synchronously downloads the values from the `tf.Tensor`. This blocks the
         * UI thread until the values are ready, which can cause performance issues.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        dataSync() {
            this.throwIfDisposed();
            const data = trackerFn().readSync(this.dataId);
            if (this.dtype === 'string') {
                try {
                    return data.map(b => decodeString(b));
                }
                catch (_a) {
                    throw new Error('Failed to decode the string bytes into utf-8. ' +
                        'To get the original bytes, call tensor.bytes().');
                }
            }
            return data;
        }
        /** Returns the underlying bytes of the tensor's data. */
        async bytes() {
            this.throwIfDisposed();
            const data = await trackerFn().read(this.dataId);
            if (this.dtype === 'string') {
                return data;
            }
            else {
                return new Uint8Array(data.buffer);
            }
        }
        /**
         * Disposes `tf.Tensor` from memory.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        dispose() {
            if (this.isDisposed) {
                return;
            }
            trackerFn().disposeTensor(this);
            this.isDisposedInternal = true;
        }
        get isDisposed() {
            return this.isDisposedInternal;
        }
        throwIfDisposed() {
            if (this.isDisposed) {
                throw new Error(`Tensor is disposed.`);
            }
        }
        /**
         * Prints the `tf.Tensor`. See `tf.print` for details.
         *
         * @param verbose Whether to print verbose information about the tensor,
         *    including dtype and size.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        print(verbose = false) {
            return opHandler.print(this, verbose);
        }
        /**
         * Returns a copy of the tensor. See `tf.clone` for details.
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        clone() {
            this.throwIfDisposed();
            return opHandler.clone(this);
        }
        /**
         * Returns a human-readable description of the tensor. Useful for logging.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        toString(verbose = false) {
            const vals = this.dataSync();
            return tensorToString(vals, this.shape, this.dtype, verbose);
        }
        cast(dtype) {
            this.throwIfDisposed();
            return opHandler.cast(this, dtype);
        }
        variable(trainable = true, name, dtype) {
            this.throwIfDisposed();
            return trackerFn().makeVariable(this, trainable, name, dtype);
        }
    }
    Object.defineProperty(Tensor, Symbol.hasInstance, {
        value: (instance) => {
            // Implementation note: we should use properties of the object that will be
            // defined before the constructor body has finished executing (methods).
            // This is because when this code is transpiled by babel, babel will call
            // classCallCheck before the constructor body is run.
            // See https://github.com/tensorflow/tfjs/issues/3384 for backstory.
            return !!instance && instance.data != null && instance.dataSync != null &&
                instance.throwIfDisposed != null;
        }
    });
    function getGlobalTensorClass() {
        // Use getGlobal so that we can augment the Tensor class across package
        // boundaries becase the node resolution alg may result in different modules
        // being returned for this file depending on the path they are loaded from.
        return getGlobal('Tensor', () => {
            return Tensor;
        });
    }
    // Global side effect. Cache global reference to Tensor class
    getGlobalTensorClass();
    /**
     * A mutable `tf.Tensor`, useful for persisting state, e.g. for training.
     *
     * @doc {heading: 'Tensors', subheading: 'Classes'}
     */
    class Variable extends Tensor {
        constructor(initialValue, trainable, name, tensorId) {
            super(initialValue.shape, initialValue.dtype, initialValue.dataId, tensorId);
            this.trainable = trainable;
            this.name = name;
        }
        /**
         * Assign a new `tf.Tensor` to this variable. The new `tf.Tensor` must have
         * the same shape and dtype as the old `tf.Tensor`.
         *
         * @param newValue New tensor to be assigned to this variable.
         *
         * @doc {heading: 'Tensors', subheading: 'Classes'}
         */
        assign(newValue) {
            if (newValue.dtype !== this.dtype) {
                throw new Error(`dtype of the new value (${newValue.dtype}) and ` +
                    `previous value (${this.dtype}) must match`);
            }
            if (!arraysEqual(newValue.shape, this.shape)) {
                throw new Error(`shape of the new value (${newValue.shape}) and ` +
                    `previous value (${this.shape}) must match`);
            }
            trackerFn().disposeTensor(this);
            this.dataId = newValue.dataId;
            trackerFn().incRef(this, null /* backend */);
        }
        dispose() {
            trackerFn().disposeVariable(this);
            this.isDisposedInternal = true;
        }
    }
    Object.defineProperty(Variable, Symbol.hasInstance, {
        value: (instance) => {
            return instance instanceof Tensor && instance.assign != null &&
                instance.assign instanceof Function;
        }
    });

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    var Rank;
    (function (Rank) {
        Rank["R0"] = "R0";
        Rank["R1"] = "R1";
        Rank["R2"] = "R2";
        Rank["R3"] = "R3";
        Rank["R4"] = "R4";
        Rank["R5"] = "R5";
        Rank["R6"] = "R6";
    })(Rank || (Rank = {}));
    // Looks for upcasting types. Used, for example, in operations with mixed dtype
    // inputs.
    var UpcastInt32AndMap;
    (function (UpcastInt32AndMap) {
        UpcastInt32AndMap["float32"] = "float32";
        UpcastInt32AndMap["int32"] = "int32";
        UpcastInt32AndMap["bool"] = "int32";
        UpcastInt32AndMap["complex64"] = "complex64";
    })(UpcastInt32AndMap || (UpcastInt32AndMap = {}));
    var UpcastBoolAndMap;
    (function (UpcastBoolAndMap) {
        UpcastBoolAndMap["float32"] = "float32";
        UpcastBoolAndMap["int32"] = "int32";
        UpcastBoolAndMap["bool"] = "bool";
        UpcastBoolAndMap["complex64"] = "complex64";
    })(UpcastBoolAndMap || (UpcastBoolAndMap = {}));
    var UpcastFloat32AndMap;
    (function (UpcastFloat32AndMap) {
        UpcastFloat32AndMap["float32"] = "float32";
        UpcastFloat32AndMap["int32"] = "float32";
        UpcastFloat32AndMap["bool"] = "float32";
        UpcastFloat32AndMap["complex64"] = "complex64";
    })(UpcastFloat32AndMap || (UpcastFloat32AndMap = {}));
    var UpcastComplex64AndMap;
    (function (UpcastComplex64AndMap) {
        UpcastComplex64AndMap["float32"] = "complex64";
        UpcastComplex64AndMap["int32"] = "complex64";
        UpcastComplex64AndMap["bool"] = "complex64";
        UpcastComplex64AndMap["complex64"] = "complex64";
    })(UpcastComplex64AndMap || (UpcastComplex64AndMap = {}));
    const upcastTypeMap = {
        'float32': UpcastFloat32AndMap,
        'int32': UpcastInt32AndMap,
        'bool': UpcastBoolAndMap,
        'complex64': UpcastComplex64AndMap
    };
    function upcastType(typeA, typeB) {
        if (typeA === 'string' || typeB === 'string') {
            if (typeA === 'string' && typeB === 'string') {
                return 'string';
            }
            throw new Error(`Can not upcast ${typeA} with ${typeB}`);
        }
        return upcastTypeMap[typeA][typeB];
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function makeTypesMatch(a, b) {
        if (a.dtype === b.dtype) {
            return [a, b];
        }
        const dtype = upcastType(a.dtype, b.dtype);
        return [a.cast(dtype), b.cast(dtype)];
    }
    function assertTypesMatch(a, b) {
        assert(a.dtype === b.dtype, () => `The dtypes of the first(${a.dtype}) and` +
            ` second(${b.dtype}) input must match`);
    }
    /**
     * Extracts any `Tensor`s found within the provided object.
     *
     * @param container an object that may be a `Tensor` or may directly contain
     *   `Tensor`s, such as a `Tensor[]` or `{key: Tensor, ...}`. In general it
     *   is safe to pass any object here, except that `Promise`s are not
     *   supported.
     * @returns An array of `Tensors` found within the passed object. If the
     *   argument is simply a `Tensor', a list containing that `Tensor` is
     *   returned. If the object is not a `Tensor` or does not
     *   contain `Tensors`, an empty list is returned.
     */
    function getTensorsInContainer(result) {
        const list = [];
        const seen = new Set();
        walkTensorContainer(result, list, seen);
        return list;
    }
    function walkTensorContainer(container, list, seen) {
        if (container == null) {
            return;
        }
        if (container instanceof Tensor) {
            list.push(container);
            return;
        }
        if (!isIterable(container)) {
            return;
        }
        // Iteration over keys works also for arrays.
        const iterable = container;
        for (const k in iterable) {
            const val = iterable[k];
            if (!seen.has(val)) {
                seen.add(val);
                walkTensorContainer(val, list, seen);
            }
        }
    }
    // tslint:disable-next-line:no-any
    function isIterable(obj) {
        return Array.isArray(obj) || typeof obj === 'object';
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function isRegisteredKernelInvocation(kernelInvocation) {
        return kernelInvocation.kernelName != null;
    }
    class EngineState {
        constructor() {
            // Public since optimizers will use it.
            this.registeredVariables = {};
            this.nextTapeNodeId = 0;
            this.numBytes = 0;
            this.numTensors = 0;
            this.numStringTensors = 0;
            this.numDataBuffers = 0;
            // Number of nested tf.grad() statements when computing higher-order
            // gradients. E.g. `1` for first-order gradients and `2` for second-order
            // gradients. Used to track if the tape should be removed after a backprop.
            this.gradientDepth = 0;
            // Number of nested kernel calls. When kernel depth is greater than 1, we turn
            // off the tape.
            this.kernelDepth = 0;
            this.scopeStack = [];
            /**
             * Keeps track of the number of data moves during a kernel execution. We
             * maintain a stack since kernels can call other kernels, recursively.
             */
            this.numDataMovesStack = [];
            this.nextScopeId = 0;
            this.tensorInfo = new WeakMap();
            this.profiling = false;
            this.activeProfile = {
                newBytes: 0,
                newTensors: 0,
                peakBytes: 0,
                kernels: [],
                result: null,
                get kernelNames() {
                    return Array.from(new Set(this.kernels.map(k => k.name)));
                }
            };
        }
        dispose() {
            for (const variableName in this.registeredVariables) {
                this.registeredVariables[variableName].dispose();
            }
        }
    }
    class Engine {
        constructor(ENV) {
            this.ENV = ENV;
            this.registry = {};
            this.registryFactory = {};
            this.pendingBackendInitId = 0;
            this.state = new EngineState();
        }
        async ready() {
            if (this.pendingBackendInit != null) {
                return this.pendingBackendInit.then(() => { });
            }
            if (this.backendInstance != null) {
                return;
            }
            const sortedBackends = this.getSortedBackends();
            for (let i = 0; i < sortedBackends.length; i++) {
                const backendName = sortedBackends[i];
                const success = await this.initializeBackend(backendName).success;
                if (success) {
                    await this.setBackend(backendName);
                    return;
                }
            }
            throw new Error(`Could not initialize any backends, all backend initializations ` +
                `failed.`);
        }
        get backend() {
            if (this.pendingBackendInit != null) {
                throw new Error(`Backend '${this.backendName}' has not yet been initialized. Make ` +
                    `sure to await tf.ready() or await tf.setBackend() before calling ` +
                    `other methods`);
            }
            if (this.backendInstance == null) {
                const { name, asyncInit } = this.initializeBackendsAndReturnBest();
                if (asyncInit) {
                    throw new Error(`The highest priority backend '${name}' has not yet been ` +
                        `initialized. Make sure to await tf.ready() or ` +
                        `await tf.setBackend() before calling other methods`);
                }
                this.setBackend(name);
            }
            return this.backendInstance;
        }
        backendNames() {
            return Object.keys(this.registryFactory);
        }
        findBackend(backendName) {
            if (!(backendName in this.registry)) {
                // If the backend hasn't been initialized but we have a registry entry for
                // it, initialize it and return it.
                if (backendName in this.registryFactory) {
                    const { asyncInit } = this.initializeBackend(backendName);
                    if (asyncInit) {
                        // Backend is not ready yet.
                        return null;
                    }
                }
                else {
                    return null;
                }
            }
            return this.registry[backendName];
        }
        findBackendFactory(backendName) {
            if (!(backendName in this.registryFactory)) {
                return null;
            }
            return this.registryFactory[backendName].factory;
        }
        registerBackend(backendName, factory, priority = 1) {
            if (backendName in this.registryFactory) {
                warn(`${backendName} backend was already registered. ` +
                    `Reusing existing backend factory.`);
                return false;
            }
            this.registryFactory[backendName] = { factory, priority };
            return true;
        }
        async setBackend(backendName) {
            if (this.registryFactory[backendName] == null) {
                throw new Error(`Backend name '${backendName}' not found in registry`);
            }
            this.backendName = backendName;
            if (this.registry[backendName] == null) {
                this.backendInstance = null;
                const { success, asyncInit } = this.initializeBackend(backendName);
                const result = asyncInit ? await success : success;
                if (!result) {
                    return false;
                }
            }
            this.backendInstance = this.registry[backendName];
            this.setupRegisteredKernels();
            // Reset the profiler.
            this.profiler = new Profiler(this.backendInstance);
            return true;
        }
        setupRegisteredKernels() {
            const kernels = getKernelsForBackend(this.backendName);
            kernels.forEach(kernel => {
                if (kernel.setupFunc != null) {
                    kernel.setupFunc(this.backendInstance);
                }
            });
        }
        disposeRegisteredKernels(backendName) {
            const kernels = getKernelsForBackend(backendName);
            kernels.forEach(kernel => {
                if (kernel.disposeFunc != null) {
                    kernel.disposeFunc(this.registry[backendName]);
                }
            });
        }
        /**
         * Initializes a backend by looking up the backend name in the factory
         * registry and calling the factory method. Returns a boolean representing
         * whether the initialization of the backend suceeded. Throws an error if
         * there is no backend in the factory registry.
         */
        initializeBackend(backendName) {
            const registryFactoryEntry = this.registryFactory[backendName];
            if (registryFactoryEntry == null) {
                throw new Error(`Cannot initialize backend ${backendName}, no registration found.`);
            }
            try {
                const backend = registryFactoryEntry.factory();
                /* Test if the factory returns a promise.
                Done in a more liberal way than
                previous 'Promise.resolve(backend)===backend'
                as we needed to account for custom Promise
                implementations (e.g. Angular) */
                if (backend && !(backend instanceof KernelBackend) &&
                    typeof backend.then === 'function') {
                    const promiseId = ++this.pendingBackendInitId;
                    const success = backend
                        .then(backendInstance => {
                        // Outdated promise. Another backend was set in the meantime.
                        if (promiseId < this.pendingBackendInitId) {
                            return false;
                        }
                        this.registry[backendName] = backendInstance;
                        this.pendingBackendInit = null;
                        return true;
                    })
                        .catch(err => {
                        // Outdated promise. Another backend was set in the meantime.
                        if (promiseId < this.pendingBackendInitId) {
                            return false;
                        }
                        this.pendingBackendInit = null;
                        warn(`Initialization of backend ${backendName} failed`);
                        warn(err.stack || err.message);
                        return false;
                    });
                    this.pendingBackendInit = success;
                    return { success, asyncInit: true };
                }
                else {
                    this.registry[backendName] = backend;
                    return { success: true, asyncInit: false };
                }
            }
            catch (err) {
                warn(`Initialization of backend ${backendName} failed`);
                warn(err.stack || err.message);
                return { success: false, asyncInit: false };
            }
        }
        removeBackend(backendName) {
            if (!(backendName in this.registryFactory)) {
                throw new Error(`${backendName} backend not found in registry`);
            }
            if (this.backendName === backendName && this.pendingBackendInit != null) {
                // There is a pending promise of the backend we want to remove. Make it
                // obsolete.
                this.pendingBackendInitId++;
            }
            if (backendName in this.registry) {
                this.disposeRegisteredKernels(backendName);
                this.registry[backendName].dispose();
                delete this.registry[backendName];
            }
            delete this.registryFactory[backendName];
            // Unset the backend if it is active.
            if (this.backendName === backendName) {
                this.pendingBackendInit = null;
                this.backendName = null;
                this.backendInstance = null;
            }
        }
        getSortedBackends() {
            if (Object.keys(this.registryFactory).length === 0) {
                throw new Error('No backend found in registry.');
            }
            return Object.keys(this.registryFactory).sort((a, b) => {
                // Highest priority comes first.
                return this.registryFactory[b].priority -
                    this.registryFactory[a].priority;
            });
        }
        initializeBackendsAndReturnBest() {
            const sortedBackends = this.getSortedBackends();
            for (let i = 0; i < sortedBackends.length; i++) {
                const backendName = sortedBackends[i];
                const { success, asyncInit } = this.initializeBackend(backendName);
                if (asyncInit || success) {
                    return { name: backendName, asyncInit };
                }
            }
            throw new Error(`Could not initialize any backends, all backend initializations ` +
                `failed.`);
        }
        moveData(backend, dataId) {
            const info = this.state.tensorInfo.get(dataId);
            const srcBackend = info.backend;
            const values = this.readSync(dataId);
            const refCount = srcBackend.refCount(dataId);
            // Delete the tensor from the old backend and move it to the new
            // backend.
            srcBackend.disposeData(dataId, true);
            info.backend = backend;
            backend.move(dataId, values, info.shape, info.dtype, refCount);
            if (this.shouldCheckForMemLeaks()) {
                // Track the number of moves during a kernel execution to correctly
                // detect memory leaks.
                this.state.numDataMovesStack[this.state.numDataMovesStack.length - 1]++;
            }
        }
        tidy(nameOrFn, fn) {
            let name = null;
            if (fn == null) {
                // Called with only 1 argument.
                if (typeof nameOrFn !== 'function') {
                    throw new Error('Please provide a function to tidy()');
                }
                fn = nameOrFn;
            }
            else {
                // Called with 2 arguments.
                if (typeof nameOrFn !== 'string' && !(nameOrFn instanceof String)) {
                    throw new Error('When calling with two arguments, the first argument ' +
                        'to tidy() must be a string');
                }
                if (typeof fn !== 'function') {
                    throw new Error('When calling with two arguments, the 2nd argument ' +
                        'to tidy() must be a function');
                }
                name = nameOrFn;
                // TODO(nsthorat,smilkov): Do operation logging and performance
                // profiling.
            }
            let result;
            return this.scopedRun(() => this.startScope(name), () => this.endScope(result), () => {
                result = fn();
                if (result instanceof Promise) {
                    console.error('Cannot return a Promise inside of tidy.');
                }
                return result;
            });
        }
        scopedRun(start, end, f) {
            start();
            try {
                const res = f();
                end();
                return res;
            }
            catch (ex) {
                end();
                throw ex;
            }
        }
        nextTensorId() {
            return Engine.nextTensorId++;
        }
        nextVariableId() {
            return Engine.nextVariableId++;
        }
        /**
         * This method is called instead of the public-facing tensor.clone() when
         * saving a tensor for backwards pass. It makes sure to add the clone
         * operation to the tape regardless of being called inside a kernel
         * execution.
         */
        clone(x) {
            const y = ENGINE.runKernel(Identity, { x });
            const inputs = { x };
            const grad = (dy) => ({
                x: () => {
                    const dtype = 'float32';
                    const gradInputs = { x: dy };
                    const attrs = { dtype };
                    return ENGINE.runKernel(Cast, gradInputs, 
                    // tslint:disable-next-line: no-unnecessary-type-assertion
                    attrs);
                }
            });
            const saved = [];
            this.addTapeNode(this.state.activeScope.name, inputs, [y], grad, saved, {});
            return y;
        }
        /**
         * Execute a kernel with the given name and return the output tensor.
         *
         * @param kernelName The name of the kernel to execute.
         * @param inputs A map of input names to tensors.
         * @param attrs A map of attribute names to their values. An attribute is a
         *     primitive (non-tensor) input to the kernel.
         * @param inputsToSave A list of tensors, inputs to save for the backprop
         *     computation.
         * @param outputsToSave A list of booleans, specifying which output to save
         *     for the backprop computation. These are booleans since the output
         * tensors are not visible to the user.
         */
        runKernel(kernelName, inputs, attrs) {
            if (this.backendName == null) {
                // backend has not been initialized yet (backend initialization is lazy
                // can be deferred until an op/ kernel is run).
                // The below getter has side effects that will try to initialize the
                // backend and set properties like this.backendName
                // tslint:disable-next-line: no-unused-expression
                this.backend;
            }
            const hasKernel = getKernel(kernelName, this.backendName) != null;
            if (!hasKernel) {
                throw new Error(`Kernel '${kernelName}' not registered for backend '${this.backendName}'`);
            }
            return this.runKernelFunc({ kernelName, inputs, attrs });
        }
        shouldCheckForMemLeaks() {
            return this.ENV.getBool('IS_TEST');
        }
        checkKernelForMemLeak(kernelName, numDataIdsBefore, outInfos) {
            const numDataIdsAfter = this.backend.numDataIds();
            // Count the number of data ids associated with the result of the kernel.
            let numOutputDataIds = 0;
            outInfos.forEach(info => {
                // Complex numbers allocate 3 data ids, one for 'real', one for
                // 'imaginary', and one for the container that holds the former two.
                numOutputDataIds += (info.dtype === 'complex64' ? 3 : 1);
            });
            // Account for the number of moves during kernel execution. A "data move"
            // can happen in the middle of a kernel execution, placing a new (key,value)
            // pair in the data storage. Since data moves have net zero effect (we
            // always remove the data from the old backend), we have to cancel them out
            // when detecting memory leaks.
            const numMoves = this.state.numDataMovesStack[this.state.numDataMovesStack.length - 1];
            const dataIdsLeaked = numDataIdsAfter - numDataIdsBefore - numOutputDataIds - numMoves;
            if (dataIdsLeaked > 0) {
                throw new Error(`Backend '${this.backendName}' has an internal memory leak ` +
                    `(${dataIdsLeaked} data ids) after running '${kernelName}'`);
            }
        }
        /**
         * Internal helper method to execute a kernel Func
         *
         * Use `runKernel` to execute kernels from outside of engine.
         */
        runKernelFunc(kernelParams) {
            let outputs;
            let saved = [];
            const isTapeOn = this.isTapeOn();
            const startingBytecount = this.state.numBytes;
            const startingNumTensors = this.state.numTensors;
            if (this.shouldCheckForMemLeaks()) {
                this.state.numDataMovesStack.push(0);
            }
            let kernelFunc;
            if (this.backendName == null) {
                // backend has not been initialized yet (backend initialization is lazy
                // can be deferred until an op/ kernel is run).
                // The below getter has side effects that will try to initialize the
                // backend and set properties like this.backendName
                // tslint:disable-next-line: no-unused-expression
                this.backend;
            }
            let out;
            const kernelOrScopeName = isRegisteredKernelInvocation(kernelParams) ?
                kernelParams.kernelName :
                this.state.activeScope != null ? this.state.activeScope.name : '';
            // Create the kernelFunc from either a registered kernel OR passed in
            // forward/backward functions (used by custom grad). In this context a
            // kernelFunc wraps a kernel implementation with some bookkeeping.
            if (isRegisteredKernelInvocation(kernelParams)) {
                const { kernelName, inputs, attrs } = kernelParams;
                if (this.backendName == null) {
                    // backend has not been initialized yet (backend initialization is lazy
                    // can be deferred until an op/ kernel is run).
                    // The below getter has side effects that will try to initialize the
                    // backend and set properties like this.backendName
                    // tslint:disable-next-line: no-unused-expression
                    this.backend;
                }
                const kernel = getKernel(kernelName, this.backendName);
                assert(kernel != null, () => `Cannot find registered kernel '${kernelName}' for backend '${this.backendName}'`);
                kernelFunc = () => {
                    const numDataIdsBefore = this.backend.numDataIds();
                    out = kernel.kernelFunc({ inputs, attrs, backend: this.backend });
                    const outInfos = Array.isArray(out) ? out : [out];
                    if (this.shouldCheckForMemLeaks()) {
                        this.checkKernelForMemLeak(kernelName, numDataIdsBefore, outInfos);
                    }
                    const outTensors = outInfos.map((outInfo) => {
                        // todo (yassogba) remove this option (Tensor) when node backend
                        // methods have been modularized and they all return tensorInfo.
                        // TensorInfos do not have a rank attribute.
                        if (outInfo.rank != null) {
                            return outInfo;
                        }
                        return this.makeTensorFromTensorInfo(outInfo);
                    });
                    // Save any required inputs and outputs.
                    // Do not save unless we are recording to the tape. Otherwise it would
                    // cause a mem leak since there would be no backprop for these tensors
                    // (which would otherwise dispose them).
                    if (isTapeOn) {
                        const tensorsToSave = this.getTensorsForGradient(kernelName, inputs, outTensors);
                        saved = this.saveTensorsForBackwardMode(tensorsToSave);
                    }
                    return outTensors;
                };
            }
            else {
                const { forwardFunc } = kernelParams;
                // Running a customGrad op.
                const saveFunc = (tensors) => {
                    // Do not save unless we are recording to the tape. Otherwise it would
                    // cause a mem leak since we would never run backprop, which disposes
                    // the kept tensors.
                    if (!isTapeOn) {
                        return;
                    }
                    saved = tensors.map(tensor => this.keep(this.clone(tensor)));
                };
                kernelFunc = () => {
                    const numDataIdsBefore = this.backend.numDataIds();
                    out = this.tidy(() => forwardFunc(this.backend, saveFunc));
                    const outs = (Array.isArray(out) ? out : [out]);
                    if (this.shouldCheckForMemLeaks()) {
                        // Scope name is used to print a more helpful error message if needed.
                        this.checkKernelForMemLeak(kernelOrScopeName, numDataIdsBefore, outs);
                    }
                    return outs;
                };
            }
            //
            // Run the kernelFunc. Optionally profiling it.
            //
            const { inputs, attrs } = kernelParams;
            const backwardsFunc = isRegisteredKernelInvocation(kernelParams) ?
                null :
                kernelParams.backwardsFunc;
            let kernelProfile;
            this.scopedRun(
            // Stop recording to a tape when running a kernel.
            () => this.state.kernelDepth++, () => this.state.kernelDepth--, () => {
                if (!this.ENV.getBool('DEBUG') && !this.state.profiling) {
                    outputs = kernelFunc();
                }
                else {
                    kernelProfile = this.profiler.profileKernel(kernelOrScopeName, inputs, () => kernelFunc());
                    if (this.ENV.getBool('DEBUG')) {
                        this.profiler.logKernelProfile(kernelProfile);
                    }
                    outputs = kernelProfile.outputs;
                }
            });
            if (isTapeOn) {
                this.addTapeNode(kernelOrScopeName, inputs, outputs, backwardsFunc, saved, attrs);
            }
            if (this.state.profiling) {
                this.state.activeProfile.kernels.push({
                    name: kernelOrScopeName,
                    bytesAdded: this.state.numBytes - startingBytecount,
                    totalBytesSnapshot: this.state.numBytes,
                    tensorsAdded: this.state.numTensors - startingNumTensors,
                    totalTensorsSnapshot: this.state.numTensors,
                    inputShapes: Object.keys(inputs).map(key => inputs[key] != null ? inputs[key].shape : null),
                    outputShapes: outputs.map(item => item.shape),
                    kernelTimeMs: kernelProfile.timeMs,
                    extraInfo: kernelProfile.extraInfo
                });
            }
            return (Array.isArray(out) ? outputs : outputs[0]);
        }
        /**
         * Saves tensors used in forward mode for use in backward mode.
         *
         * @param tensors the list of tensors to save.
         */
        saveTensorsForBackwardMode(tensors) {
            const saved = tensors.map(tensor => this.keep(this.clone(tensor)));
            return saved;
        }
        /**
         * Returns a list of tensors to save for a given gradient calculation.
         *
         * @param kernelName name of kernel to look up gradient for.
         * @param inputs a map of input tensors.
         * @param outputs an array of output tensors from forward mode of kernel.
         */
        getTensorsForGradient(kernelName, inputs, outputs) {
            const gradConfig = getGradient(kernelName);
            if (gradConfig != null) {
                const inputsToSave = gradConfig.inputsToSave || [];
                const outputsToSave = gradConfig.outputsToSave || [];
                // If saveAllInputs is true, all inputs will be saved. Otherwise, inputs
                // specified in inputsToSave will be saved.
                let inputTensorsToSave;
                if (gradConfig.saveAllInputs) {
                    assert(Array.isArray(inputs), () => 'saveAllInputs is true, expected inputs to be an array.');
                    inputTensorsToSave = Object.keys(inputs).map((key) => inputs[key]);
                }
                else {
                    inputTensorsToSave = inputsToSave.map((inputName) => inputs[inputName]);
                }
                const outputTensorsToSave = outputs.filter((_, i) => outputsToSave[i]);
                return inputTensorsToSave.concat(outputTensorsToSave);
            }
            // We return an empty list rather than throw an error because the kernel we
            // are looking up may not actually be relevant to backproping through the
            // overall function
            //
            // See 'does not error if irrelevant (pruned) ops are missing grads' test
            // in gradients_test.ts for an example.
            return [];
        }
        /**
         * Internal method used by public APIs for tensor creation. Makes a new
         * tensor with the provided shape, dtype and values. It always
         * creates a new data id and writes the values to the underlying backend.
         */
        makeTensor(values, shape, dtype, backend) {
            if (values == null) {
                throw new Error('Values passed to engine.makeTensor() are null');
            }
            dtype = dtype || 'float32';
            backend = backend || this.backend;
            let backendVals = values;
            if (dtype === 'string' && isString(values[0])) {
                backendVals = values.map(d => encodeString(d));
            }
            const dataId = backend.write(backendVals, shape, dtype);
            const t = new Tensor(shape, dtype, dataId, this.nextTensorId());
            this.trackTensor(t, backend);
            // Count bytes for string tensors.
            if (dtype === 'string') {
                const info = this.state.tensorInfo.get(dataId);
                const newBytes = bytesFromStringArray(backendVals);
                this.state.numBytes += newBytes - info.bytes;
                info.bytes = newBytes;
            }
            return t;
        }
        /**
         * Internal method used by backends. Makes a new tensor
         * that is a wrapper around an existing data id. It doesn't create
         * a new data id, only increments the ref count used in memory tracking.
         * @deprecated
         */
        makeTensorFromDataId(dataId, shape, dtype, backend) {
            dtype = dtype || 'float32';
            const tensorInfo = { dataId, shape, dtype };
            return this.makeTensorFromTensorInfo(tensorInfo, backend);
        }
        /**
         * Internal method used by backends. Makes a new tensor that is a wrapper
         * around an existing data id in TensorInfo. It doesn't create a new data id,
         * only increments the ref count used in memory tracking.
         */
        makeTensorFromTensorInfo(tensorInfo, backend) {
            const { dataId, shape, dtype } = tensorInfo;
            const t = new Tensor(shape, dtype, dataId, this.nextTensorId());
            this.trackTensor(t, backend);
            return t;
        }
        makeVariable(initialValue, trainable = true, name, dtype) {
            name = name || this.nextVariableId().toString();
            if (dtype != null && dtype !== initialValue.dtype) {
                initialValue = initialValue.cast(dtype);
            }
            const v = new Variable(initialValue, trainable, name, this.nextTensorId());
            if (this.state.registeredVariables[v.name] != null) {
                throw new Error(`Variable with name ${v.name} was already registered`);
            }
            this.state.registeredVariables[v.name] = v;
            this.incRef(v, this.backend);
            return v;
        }
        trackTensor(a, backend) {
            this.state.numTensors++;
            if (a.dtype === 'string') {
                this.state.numStringTensors++;
            }
            // Bytes for complex numbers are counted by their components. Bytes for
            // string tensors are counted when writing values.
            let bytes = 0;
            if (a.dtype !== 'complex64' && a.dtype !== 'string') {
                bytes = a.size * bytesPerElement(a.dtype);
            }
            this.state.numBytes += bytes;
            if (!this.state.tensorInfo.has(a.dataId)) {
                this.state.numDataBuffers++;
                this.state.tensorInfo.set(a.dataId, {
                    backend: backend || this.backend,
                    dtype: a.dtype,
                    shape: a.shape,
                    bytes
                });
            }
            if (!(a instanceof Variable)) {
                this.track(a);
            }
        }
        // Track the tensor by dataId and increase the refCount for the dataId in the
        // backend.
        // TODO(pyu10055): This is currently used by makeVariable method, to increase
        // refCount on the backend for the dataId. It can potentially be replaced with
        // Identity op indead of calling backend directly.
        incRef(a, backend) {
            this.trackTensor(a, backend);
            this.backend.incRef(a.dataId);
        }
        removeDataId(dataId, backend) {
            if (this.state.tensorInfo.has(dataId) &&
                this.state.tensorInfo.get(dataId).backend === backend) {
                this.state.tensorInfo.delete(dataId);
                this.state.numDataBuffers--;
            }
        }
        disposeTensor(a) {
            if (!this.state.tensorInfo.has(a.dataId)) {
                return;
            }
            const info = this.state.tensorInfo.get(a.dataId);
            this.state.numTensors--;
            if (a.dtype === 'string') {
                this.state.numStringTensors--;
                this.state.numBytes -= info.bytes;
            }
            // Don't count bytes for complex numbers as they are counted by their
            // components.
            if (a.dtype !== 'complex64' && a.dtype !== 'string') {
                const bytes = a.size * bytesPerElement(a.dtype);
                this.state.numBytes -= bytes;
            }
            // Remove the reference to dataId if backend dispose the data successfully
            if (info.backend.disposeData(a.dataId)) {
                this.removeDataId(a.dataId, info.backend);
            }
            // TODO(nsthorat): Construct an error and save the stack trace for
            // debugging when in debug mode. Creating a stack trace is too expensive
            // to do unconditionally.
        }
        disposeVariables() {
            for (const varName in this.state.registeredVariables) {
                const v = this.state.registeredVariables[varName];
                this.disposeVariable(v);
            }
        }
        disposeVariable(v) {
            this.disposeTensor(v);
            if (this.state.registeredVariables[v.name] != null) {
                delete this.state.registeredVariables[v.name];
            }
        }
        memory() {
            const info = this.backend.memory();
            info.numTensors = this.state.numTensors;
            info.numDataBuffers = this.state.numDataBuffers;
            info.numBytes = this.state.numBytes;
            if (this.state.numStringTensors > 0) {
                info.unreliable = true;
                if (info.reasons == null) {
                    info.reasons = [];
                }
                info.reasons.push('Memory usage by string tensors is approximate ' +
                    '(2 bytes per character)');
            }
            return info;
        }
        async profile(query) {
            this.state.profiling = true;
            const startBytes = this.state.numBytes;
            const startNumTensors = this.state.numTensors;
            this.state.activeProfile.kernels = [];
            this.state.activeProfile.result = await query();
            this.state.profiling = false;
            this.state.activeProfile.peakBytes = Math.max(...this.state.activeProfile.kernels.map(d => d.totalBytesSnapshot));
            this.state.activeProfile.newBytes = this.state.numBytes - startBytes;
            this.state.activeProfile.newTensors =
                this.state.numTensors - startNumTensors;
            for (const kernel of this.state.activeProfile.kernels) {
                kernel.kernelTimeMs = await kernel.kernelTimeMs;
                kernel.extraInfo = await kernel.extraInfo;
            }
            return this.state.activeProfile;
        }
        isTapeOn() {
            return this.state.gradientDepth > 0 && this.state.kernelDepth === 0;
        }
        addTapeNode(kernelName, inputs, outputs, gradientsFunc, saved, attrs) {
            const tapeNode = { id: this.state.nextTapeNodeId++, kernelName, inputs, outputs, saved };
            const gradConfig = getGradient(kernelName);
            if (gradConfig != null) {
                gradientsFunc = gradConfig.gradFunc;
            }
            if (gradientsFunc != null) {
                tapeNode.gradient = (dys) => {
                    // TODO(smilkov): To optimize back-prop, pass dys that are not used in
                    // the backprop graph to the user as null instead of zeros
                    dys = dys.map((dy, i) => {
                        if (dy == null) {
                            const output = outputs[i];
                            const vals = makeZerosTypedArray(output.size, output.dtype);
                            return this.makeTensor(vals, output.shape, output.dtype);
                        }
                        return dy;
                    });
                    // Grad functions of ops with single outputs expect a dy, while ops
                    // with multiple outputs expect dys (array of dy).
                    return gradientsFunc(dys.length > 1 ? dys : dys[0], saved, attrs);
                };
            }
            this.state.activeTape.push(tapeNode);
        }
        keep(result) {
            result.kept = true;
            return result;
        }
        startTape() {
            if (this.state.gradientDepth === 0) {
                this.state.activeTape = [];
            }
            this.state.gradientDepth++;
        }
        endTape() {
            this.state.gradientDepth--;
        }
        /**
         * Start a scope. Use this with endScope() to achieve the same functionality
         * as scope() without the need for a function closure.
         */
        startScope(name) {
            const scopeInfo = {
                track: [],
                name: 'unnamed scope',
                id: this.state.nextScopeId++
            };
            if (name) {
                scopeInfo.name = name;
            }
            this.state.scopeStack.push(scopeInfo);
            this.state.activeScope = scopeInfo;
        }
        /**
         * End a scope. Use this with startScope() to achieve the same functionality
         * as scope() without the need for a function closure.
         */
        endScope(result) {
            const tensorsToTrackInParent = getTensorsInContainer(result);
            const tensorsToTrackInParentSet = new Set(tensorsToTrackInParent.map(t => t.id));
            // Dispose the arrays tracked in this scope.
            for (let i = 0; i < this.state.activeScope.track.length; i++) {
                const tensor = this.state.activeScope.track[i];
                if (!tensor.kept && !tensorsToTrackInParentSet.has(tensor.id)) {
                    tensor.dispose();
                }
            }
            const oldScope = this.state.scopeStack.pop();
            this.state.activeScope = this.state.scopeStack.length === 0 ?
                null :
                this.state.scopeStack[this.state.scopeStack.length - 1];
            // Track the current result in the parent scope.
            tensorsToTrackInParent.forEach(tensor => {
                // Only track the tensor if was allocated in the inner scope and is not
                // globally kept.
                if (!tensor.kept && tensor.scopeId === oldScope.id) {
                    this.track(tensor);
                }
            });
        }
        /**
         * Returns gradients of `f` with respect to each of the `xs`. The gradients
         * returned are of the same length as `xs`, but some might be null if `f`
         * was not a function of that `x`. It also takes optional dy to multiply the
         * gradient, which defaults to `1`.
         */
        gradients(f, xs, dy, allowNoGradients = false) {
            assert(xs.length > 0, () => 'gradients() received an empty list of xs.');
            if (dy != null && dy.dtype !== 'float32') {
                throw new Error(`dy must have 'float32' dtype, but has '${dy.dtype}'`);
            }
            const y = this.scopedRun(() => this.startTape(), () => this.endTape(), () => this.tidy('forward', f));
            assert(y instanceof Tensor, () => 'The result y returned by f() must be a tensor.');
            // Filter out the nodes that don't connect x => y.
            const filteredTape = getFilteredNodesXToY(this.state.activeTape, xs, y);
            if (!allowNoGradients && filteredTape.length === 0 && xs.length > 0) {
                throw new Error('Cannot compute gradient of y=f(x) with respect to x. Make sure ' +
                    'that the f you passed encloses all operations that lead from x ' +
                    'to y.');
            }
            return this.tidy('backward', () => {
                const accumulatedGradientMap = {};
                accumulatedGradientMap[y.id] = (dy == null) ? ones$2(y.shape) : dy;
                // Backprop gradients through the filtered nodes.
                backpropagateGradients(accumulatedGradientMap, filteredTape, 
                // Pass the tidy function to avoid circular dep with `tape.ts`.
                f => this.tidy(f), 
                // Pass an add function to avoide a circular dep with `tape.ts`.
                add$2);
                const grads = xs.map(x => accumulatedGradientMap[x.id]);
                if (this.state.gradientDepth === 0) {
                    // This means that we are not computing higher-order gradients
                    // and can clean up the tape.
                    this.state.activeTape.forEach(node => {
                        for (const tensor of node.saved) {
                            tensor.dispose();
                        }
                    });
                    this.state.activeTape = null;
                }
                return { value: y, grads };
            });
        }
        customGrad(f) {
            assert(isFunction(f), () => 'The f passed in customGrad(f) must be a function.');
            return (...inputs) => {
                assert(inputs.every(t => t instanceof Tensor), () => 'The args passed in customGrad(f)(x1, x2,...) must all be ' +
                    'tensors');
                let res;
                const inputMap = {};
                inputs.forEach((input, i) => {
                    inputMap[i] = input;
                });
                const forwardFunc = (_, save) => {
                    res = f(...[...inputs, save]);
                    assert(res.value instanceof Tensor, () => 'The function f passed in customGrad(f) must return an ' +
                        'object where `obj.value` is a tensor');
                    assert(isFunction(res.gradFunc), () => 'The function f passed in customGrad(f) must return an ' +
                        'object where `obj.gradFunc` is a function.');
                    return res.value;
                };
                const backwardsFunc = (dy, saved) => {
                    const gradRes = res.gradFunc(dy, saved);
                    const grads = Array.isArray(gradRes) ? gradRes : [gradRes];
                    assert(grads.length === inputs.length, () => 'The function f passed in customGrad(f) must return an ' +
                        'object where `obj.gradFunc` is a function that returns ' +
                        'the same number of tensors as inputs passed to f(...).');
                    assert(grads.every(t => t instanceof Tensor), () => 'The function f passed in customGrad(f) must return an ' +
                        'object where `obj.gradFunc` is a function that returns ' +
                        'a list of only tensors.');
                    const gradMap = {};
                    grads.forEach((grad, i) => {
                        gradMap[i] = () => grad;
                    });
                    return gradMap;
                };
                return this.runKernelFunc({
                    forwardFunc,
                    backwardsFunc,
                    inputs: inputMap,
                });
            };
        }
        readSync(dataId) {
            // Route the read to the correct backend.
            const info = this.state.tensorInfo.get(dataId);
            return info.backend.readSync(dataId);
        }
        read(dataId) {
            // Route the read to the correct backend.
            const info = this.state.tensorInfo.get(dataId);
            return info.backend.read(dataId);
        }
        readToGPU(dataId, options) {
            // Route the read to the correct backend.
            const info = this.state.tensorInfo.get(dataId);
            return info.backend.readToGPU(dataId, options);
        }
        async time(query) {
            const start = now();
            const timingInfo = await this.backend.time(query);
            timingInfo.wallMs = now() - start;
            return timingInfo;
        }
        /**
         * Tracks a Tensor in the current scope to be automatically cleaned up
         * when the current scope ends, and returns the value.
         *
         * @param result The Tensor to track in the current scope.
         */
        track(result) {
            if (this.state.activeScope != null) {
                result.scopeId = this.state.activeScope.id;
                this.state.activeScope.track.push(result);
            }
            return result;
        }
        get registeredVariables() {
            return this.state.registeredVariables;
        }
        /**
         * Resets the engine state. Removes all backends but does not remove
         * registered backend factories.
         */
        reset() {
            // Make any pending promise obsolete.
            this.pendingBackendInitId++;
            this.state.dispose();
            this.ENV.reset();
            this.state = new EngineState();
            for (const backendName in this.registry) {
                this.disposeRegisteredKernels(backendName);
                this.registry[backendName].dispose();
                delete this.registry[backendName];
            }
            this.backendName = null;
            this.backendInstance = null;
            this.pendingBackendInit = null;
        }
    }
    Engine.nextTensorId = 0;
    Engine.nextVariableId = 0;
    function ones$2(shape) {
        const values = makeOnesTypedArray(sizeFromShape(shape), 'float32');
        return ENGINE.makeTensor(values, shape, 'float32');
    }
    function getOrMakeEngine() {
        const ns = getGlobalNamespace();
        if (ns._tfengine == null) {
            const environment = new Environment(ns);
            ns._tfengine = new Engine(environment);
        }
        setEnvironmentGlobal(ns._tfengine.ENV);
        // Tell the current tensor interface that the global engine is responsible
        // for tracking.
        setTensorTracker(() => ns._tfengine);
        return ns._tfengine;
    }
    const ENGINE = getOrMakeEngine();
    /**
     * A implementation of the add op for use within engine and tape.
     *
     * This allows us to avoid a circular dependency between add.ts and engine.
     * It is exported to be available in tape tests.
     */
    function add$2(a, b) {
        // We duplicate Add here to avoid a circular dependency with add.ts.
        const inputs = { a, b };
        return ENGINE.runKernel(Add$1, inputs);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function inferShape(val, dtype) {
        let firstElem = val;
        if (isTypedArray(val)) {
            return dtype === 'string' ? [] : [val.length];
        }
        if (typeof val === 'object' && 'texture' in val) {
            const usedChannels = val.channels || 'RGBA';
            return [val.height, val.width * usedChannels.length];
        }
        if (!Array.isArray(val)) {
            return []; // Scalar.
        }
        const shape = [];
        while (Array.isArray(firstElem) ||
            isTypedArray(firstElem) && dtype !== 'string') {
            shape.push(firstElem.length);
            firstElem = firstElem[0];
        }
        if (Array.isArray(val) &&
            env().getBool('TENSORLIKE_CHECK_SHAPE_CONSISTENCY')) {
            deepAssertShapeConsistency(val, shape, []);
        }
        return shape;
    }
    function deepAssertShapeConsistency(val, shape, indices) {
        indices = indices || [];
        if (!(Array.isArray(val)) && !isTypedArray(val)) {
            assert(shape.length === 0, () => `Element arr[${indices.join('][')}] is a primitive, ` +
                `but should be an array/TypedArray of ${shape[0]} elements`);
            return;
        }
        assert(shape.length > 0, () => `Element arr[${indices.join('][')}] should be a primitive, ` +
            `but is an array of ${val.length} elements`);
        assert(val.length === shape[0], () => `Element arr[${indices.join('][')}] should have ${shape[0]} ` +
            `elements, but has ${val.length} elements`);
        const subShape = shape.slice(1);
        for (let i = 0; i < val.length; ++i) {
            deepAssertShapeConsistency(val[i], subShape, indices.concat(i));
        }
    }
    function assertDtype(expectedDtype, actualDType, argName, functionName) {
        if (expectedDtype === 'string_or_numeric') {
            return;
        }
        if (expectedDtype == null) {
            throw new Error(`Expected dtype cannot be null.`);
        }
        if (expectedDtype !== 'numeric' && expectedDtype !== actualDType ||
            expectedDtype === 'numeric' && actualDType === 'string') {
            throw new Error(`Argument '${argName}' passed to '${functionName}' must ` +
                `be ${expectedDtype} tensor, but got ${actualDType} tensor`);
        }
    }
    function convertToTensor(x, argName, functionName, parseAsDtype = 'numeric') {
        if (x instanceof Tensor) {
            assertDtype(parseAsDtype, x.dtype, argName, functionName);
            return x;
        }
        let inferredDtype = inferDtype(x);
        // If the user expects a bool/int/float, use that info to update the
        // inferredDtype when it is not a string.
        if (inferredDtype !== 'string' &&
            ['bool', 'int32', 'float32'].indexOf(parseAsDtype) >= 0) {
            inferredDtype = parseAsDtype;
        }
        assertDtype(parseAsDtype, inferredDtype, argName, functionName);
        if ((x == null) ||
            (!isTypedArray(x) && !Array.isArray(x) && typeof x !== 'number' &&
                typeof x !== 'boolean' && typeof x !== 'string')) {
            const type = x == null ? 'null' : x.constructor.name;
            throw new Error(`Argument '${argName}' passed to '${functionName}' must be a ` +
                `Tensor or TensorLike, but got '${type}'`);
        }
        const inferredShape = inferShape(x, inferredDtype);
        if (!isTypedArray(x) && !Array.isArray(x)) {
            x = [x];
        }
        const skipTypedArray = true;
        const values = inferredDtype !== 'string' ?
            toTypedArray(x, inferredDtype) :
            flatten$1(x, [], skipTypedArray);
        return ENGINE.makeTensor(values, inferredShape, inferredDtype);
    }
    function convertToTensorArray(arg, argName, functionName, parseAsDtype = 'numeric') {
        if (!Array.isArray(arg)) {
            throw new Error(`Argument ${argName} passed to ${functionName} must be a ` +
                '`Tensor[]` or `TensorLike[]`');
        }
        const tensors = arg;
        return tensors.map((t, i) => convertToTensor(t, `${argName}[${i}]`, functionName, parseAsDtype));
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const OP_SCOPE_SUFFIX = '__op';
    /**
     * Used for wrapping functions that perform math operations on
     * Tensors. The function will be wrapped in a named scope that cleans all
     * memory usage after the function is done.
     */
    function op(f) {
        const keys = Object.keys(f);
        if (keys.length !== 1) {
            throw new Error(`Please provide an object with a single key ` +
                `(operation name) mapping to a function. Got an object with ` +
                `${keys.length} keys.`);
        }
        let opName = keys[0];
        const fn = f[opName];
        // Strip the underscore from the end of the function name.
        if (opName.endsWith('_')) {
            opName = opName.substring(0, opName.length - 1);
        }
        // add an __op suffix to distinguish ops from kernels in tf.profile
        opName = opName + OP_SCOPE_SUFFIX;
        // tslint:disable-next-line:no-any
        const f2 = (...args) => {
            ENGINE.startScope(opName);
            try {
                const result = fn(...args);
                if (isPromise(result)) {
                    console.error('Cannot return a Promise inside of tidy.');
                }
                ENGINE.endScope(result);
                return result;
            }
            catch (ex) {
                ENGINE.endScope(null);
                throw ex;
            }
        };
        Object.defineProperty(f2, 'name', { value: opName, configurable: true });
        // tslint:disable-next-line:no-any
        return f2;
    }

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Casts a `tf.Tensor` to a new dtype.
     *
     * ```js
     * const x = tf.tensor1d([1.5, 2.5, 3]);
     * tf.cast(x, 'int32').print();
     * ```
     * @param x The input tensor to be casted.
     * @param dtype The dtype to cast the input tensor to.
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function cast_(x, dtype) {
        const $x = convertToTensor(x, 'x', 'cast');
        // Sanity checks.
        if (!isValidDtype(dtype)) {
            throw new Error(`Failed to cast to unknown dtype ${dtype}`);
        }
        if (dtype === 'string' && $x.dtype !== 'string' ||
            dtype !== 'string' && $x.dtype === 'string') {
            throw new Error('Only strings can be casted to strings');
        }
        const inputs = { x: $x };
        const attrs = { dtype };
        return ENGINE.runKernel(Cast, inputs, attrs);
    }
    const cast = op({ cast_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Multiplies two `tf.Tensor`s element-wise, A * B. Supports broadcasting.
     *
     * We also expose `tf.mulStrict` which has the same signature as this op and
     * asserts that `a` and `b` are the same shape (does not broadcast).
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3, 4]);
     * const b = tf.tensor1d([2, 3, 4, 5]);
     *
     * a.mul(b).print();  // or tf.mul(a, b)
     * ```
     *
     * ```js
     * // Broadcast mul a with b.
     * const a = tf.tensor1d([1, 2, 3, 4]);
     * const b = tf.scalar(5);
     *
     * a.mul(b).print();  // or tf.mul(a, b)
     * ```
     * @param a The first tensor to multiply.
     * @param b The second tensor to multiply. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function mul_(a, b) {
        let $a = convertToTensor(a, 'a', 'mul');
        let $b = convertToTensor(b, 'b', 'mul');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Multiply$1, inputs);
    }
    const mul = op({ mul_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes step of the input `tf.Tensor` element-wise: `x > 0 ? 1 : alpha * x`
     *
     * ```js
     * const x = tf.tensor1d([0, 2, -1, -3]);
     *
     * x.step(.5).print();  // or tf.step(x, .5)
     * ```
     * @param x The input tensor.
     * @param alpha The gradient when input is negative.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function step_(x, alpha = 0.0) {
        const $x = convertToTensor(x, 'x', 'step');
        const inputs = { x: $x };
        const attrs = { alpha };
        return ENGINE.runKernel(Step, inputs, attrs);
    }
    const step = op({ step_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const absGradConfig = {
        kernelName: Abs,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => mul(dy, step(cast(x, 'float32'), -1)) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Divides two `tf.Tensor`s element-wise, A / B. Supports broadcasting.
     * The result is rounded with floor function.
     *
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 9, 16]);
     * const b = tf.tensor1d([1, 2, 3, 4]);
     *
     * a.floorDiv(b).print();  // or tf.div(a, b)
     * ```
     *
     * ```js
     * // Broadcast div a with b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(2);
     *
     * a.floorDiv(b).print();  // or tf.floorDiv(a, b)
     * ```
     *
     * @param a The first tensor as the numerator.
     * @param b The second tensor as the denominator. Must have the same dtype as
     * `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function floorDiv_(a, b) {
        let $a = convertToTensor(a, 'a', 'floorDiv');
        let $b = convertToTensor(b, 'b', 'floorDiv');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(FloorDiv, inputs);
    }
    const floorDiv = op({ floorDiv_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Divides two `tf.Tensor`s element-wise, A / B. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 9, 16]);
     * const b = tf.tensor1d([1, 2, 3, 4]);
     *
     * a.div(b).print();  // or tf.div(a, b)
     * ```
     *
     * ```js
     * // Broadcast div a with b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(2);
     *
     * a.div(b).print();  // or tf.div(a, b)
     * ```
     *
     * @param a The first tensor as the numerator.
     * @param b The second tensor as the denominator. Must have the same dtype as
     * `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function div_(a, b) {
        let $a = convertToTensor(a, 'a', 'div');
        let $b = convertToTensor(b, 'b', 'div');
        [$a, $b] = makeTypesMatch($a, $b);
        if ($a.dtype === 'int32' && $b.dtype === 'int32') {
            return floorDiv($a, $b);
        }
        const inputs = { a: $a, b: $b };
        const attrs = {};
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(RealDiv, inputs, attrs);
    }
    const div = op({ div_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes `-1 * x` element-wise.
     *
     * ```js
     * const x = tf.tensor2d([1, 2, -2, 0], [2, 2]);
     *
     * x.neg().print();  // or tf.neg(x)
     * ```
     *
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function neg_(x) {
        const $x = convertToTensor(x, 'x', 'neg');
        const inputs = { x: $x };
        return ENGINE.runKernel(Neg, inputs);
    }
    const neg = op({ neg_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /** This is shared code across all tensor creation methods. */
    function makeTensor(values, shape, inferredShape, dtype) {
        if (dtype == null) {
            dtype = inferDtype(values);
        }
        if (dtype === 'complex64') {
            throw new Error(`Cannot construct a complex64 tensor directly. ` +
                `Please use tf.complex(real, imag).`);
        }
        if (typeof values === 'object' && 'texture' in values) {
            if (dtype !== 'float32' && dtype !== 'int32') {
                throw new Error(`Creating tensor from texture only supports ` +
                    `'float32'|'int32' dtype, while the dtype is ${dtype}.`);
            }
            values.channels = values.channels || 'RGBA';
            return ENGINE.backend.createTensorFromTexture(values, shape || inferredShape, dtype);
        }
        if (!isTypedArray(values) && !Array.isArray(values) &&
            typeof values !== 'number' && typeof values !== 'boolean' &&
            typeof values !== 'string') {
            throw new Error('values passed to tensor(values) must be a number/boolean/string or ' +
                'an array of numbers/booleans/strings, or a TypedArray');
        }
        // Verify that the shape matches the inferred shape.
        if (shape != null) {
            assertNonNegativeIntegerDimensions(shape);
            const providedSize = sizeFromShape(shape);
            const inferredSize = sizeFromShape(inferredShape);
            assert(providedSize === inferredSize, () => `Based on the provided shape, [${shape}], the tensor should have ` +
                `${providedSize} values but has ${inferredSize}`);
            for (let i = 0; i < inferredShape.length; ++i) {
                const inferred = inferredShape[i];
                const flatDimsDontMatch = i === inferredShape.length - 1 ?
                    inferred !== sizeFromShape(shape.slice(i)) :
                    true;
                assert(inferredShape[i] === shape[i] || !flatDimsDontMatch, () => `Error creating a new Tensor. Inferred shape ` +
                    `(${inferredShape}) does not match the provided ` +
                    `shape (${shape}). `);
            }
        }
        if (!isTypedArray(values) && !Array.isArray(values)) {
            values = [values];
        }
        shape = shape || inferredShape;
        values = dtype !== 'string' ?
            toTypedArray(values, dtype) :
            flatten$1(values, [], true);
        return ENGINE.makeTensor(values, shape, dtype);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates rank-0 `tf.Tensor` (scalar) with the provided value and dtype.
     *
     * The same functionality can be achieved with `tf.tensor`, but in general
     * we recommend using `tf.scalar` as it makes the code more readable.
     *
     * ```js
     * tf.scalar(3.14).print();
     * ```
     *
     * @param value The value of the scalar.
     * @param dtype The data type.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function scalar(value, dtype) {
        if (((isTypedArray(value) && dtype !== 'string') || Array.isArray(value)) &&
            dtype !== 'complex64') {
            throw new Error('Error creating a new Scalar: value must be a primitive ' +
                '(number|boolean|string)');
        }
        if (dtype === 'string' && isTypedArray(value) &&
            !(value instanceof Uint8Array)) {
            throw new Error('When making a scalar from encoded string, ' +
                'the value must be `Uint8Array`.');
        }
        const shape = [];
        const inferredShape = [];
        return makeTensor(value, shape, inferredShape, dtype);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes square root of the input `tf.Tensor` element-wise: `y = sqrt(x)`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 4, -1]);
     *
     * x.sqrt().print();  // or tf.sqrt(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function sqrt_(x) {
        const $x = convertToTensor(x, 'x', 'sqrt', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Sqrt, inputs);
    }
    const sqrt = op({ sqrt_ });

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes square of `x` element-wise: `x ^ 2`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, Math.sqrt(2), -1]);
     *
     * x.square().print();  // or tf.square(x)
     * ```
     * @param x The input Tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function square_(x) {
        const $x = convertToTensor(x, 'x', 'square');
        const attrs = {};
        return ENGINE.runKernel('Square', { x: $x }, attrs);
    }
    const square = op({ square_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Subtracts two `tf.Tensor`s element-wise, A - B. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([10, 20, 30, 40]);
     * const b = tf.tensor1d([1, 2, 3, 4]);
     *
     * a.sub(b).print();  // or tf.sub(a, b)
     * ```
     *
     * ```js
     * // Broadcast subtract a with b.
     * const a = tf.tensor1d([10, 20, 30, 40]);
     * const b = tf.scalar(5);
     *
     * a.sub(b).print();  // or tf.sub(a, b)
     * ```
     * @param a The first `tf.Tensor` to subtract from.
     * @param b The second `tf.Tensor` to be subtracted. Must have the same dtype as
     * `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function sub_(a, b) {
        let $a = convertToTensor(a, 'a', 'sub');
        let $b = convertToTensor(b, 'b', 'sub');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Sub, inputs);
    }
    const sub = op({ sub_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const acosGradConfig = {
        kernelName: Acos,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return {
                x: () => {
                    const a = square(cast(x, 'float32'));
                    const b = sqrt(sub(scalar(1), a));
                    return neg(div(dy, b));
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const acoshGradConfig = {
        kernelName: Acosh,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return {
                x: () => {
                    const a = sqrt(sub(square(cast(x, 'float32')), 1));
                    return div(dy, a);
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the axes in the output space that should be reduced to produce
     * the input space.
     */
    function getReductionAxes(inShape, outShape) {
        const result = [];
        for (let i = 0; i < outShape.length; i++) {
            const inDim = inShape[inShape.length - i - 1];
            const outAxis = outShape.length - i - 1;
            const outDim = outShape[outAxis];
            if (inDim == null || (inDim === 1 && outDim > 1)) {
                result.unshift(outAxis);
            }
        }
        return result;
    }
    function assertAndGetBroadcastShape(shapeA, shapeB) {
        const result = [];
        const l = Math.max(shapeA.length, shapeB.length);
        for (let i = 0; i < l; i++) {
            let a = shapeA[shapeA.length - i - 1];
            if (a == null) {
                a = 1;
            }
            let b = shapeB[shapeB.length - i - 1];
            if (b == null) {
                b = 1;
            }
            if (a === 1) {
                result.unshift(b);
            }
            else if (b === 1) {
                result.unshift(a);
            }
            else if (a !== b) {
                const errMsg = `Operands could not be broadcast together with shapes ` +
                    `${shapeA} and ${shapeB}.`;
                throw Error(errMsg);
            }
            else {
                result.unshift(a);
            }
        }
        return result;
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reshapes a `tf.Tensor` to a given shape.
     *
     * Given an input tensor, returns a new tensor with the same values as the
     * input tensor with shape `shape`.
     *
     * If one component of shape is the special value -1, the size of that
     * dimension is computed so that the total size remains constant. In
     * particular, a shape of [-1] flattens into 1-D. At most one component of
     * shape can be -1.
     *
     * If shape is 1-D or higher, then the operation returns a tensor with shape
     * shape filled with the values of tensor. In this case, the number of
     * elements implied by shape must be the same as the number of elements in
     * tensor.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     * x.reshape([2, 2]).print();
     * ```
     *
     * @param x The input tensor to be reshaped.
     * @param shape An array of integers defining the output tensor shape.
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function reshape_(x, shape) {
        const $x = convertToTensor(x, 'x', 'reshape', 'string_or_numeric');
        const inputs = { x: $x };
        const attrs = { shape };
        return ENGINE.runKernel(Reshape$1, inputs, attrs);
    }
    const reshape$1 = op({ reshape_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the sum of elements across dimensions of a `tf.Tensor`.
     *
     * Reduces the input along the dimensions given in `axes`. Unless `keepDims`
     * is true, the rank of the `tf.Tensor` is reduced by 1 for each entry in
     * `axes`. If `keepDims` is true, the reduced dimensions are retained with
     * length 1. If axes has no entries, all dimensions are reduced, and a
     * `tf.Tensor` with a single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.sum().print();  // or tf.sum(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.sum(axis).print();  // or tf.sum(x, axis)
     * ```
     *
     * @param x The input tensor to compute the sum over. If the dtype is `bool`
     *   it will be converted to `int32` and the output dtype will be `int32`.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function sum_(x, axis = null, keepDims = false) {
        let $x = convertToTensor(x, 'x', 'sum');
        if ($x.dtype === 'bool') {
            $x = cast($x, 'int32');
        }
        const inputs = { x: $x };
        const attrs = { axis, keepDims };
        return ENGINE.runKernel(Sum, inputs, attrs);
    }
    const sum = op({ sum_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const addGradConfig = {
        kernelName: Add$1,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const outShape = assertAndGetBroadcastShape(a.shape, b.shape);
            const derA = () => {
                let res = dy;
                const reduceAxes = getReductionAxes(a.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = sum(res, reduceAxes);
                }
                return reshape$1(res, a.shape);
            };
            const derB = () => {
                let res = dy;
                const reduceAxes = getReductionAxes(b.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = sum(res, reduceAxes);
                }
                return reshape$1(res, b.shape);
            };
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const addNGradConfig = {
        kernelName: AddN,
        saveAllInputs: true,
        gradFunc: (dy, saved) => {
            const ders = {};
            saved.forEach((_, i) => {
                ders[i] = () => dy.clone();
            });
            return ders;
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with all elements set to 0 with the same shape as the
     * given tensor.
     *
     * ```js
     * const x = tf.tensor([1, 2]);
     * tf.zerosLike(x).print();
     * ```
     *
     * @param x The tensor of required shape.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function zerosLike_(x) {
        const $x = convertToTensor(x, 'x', 'zerosLike');
        const inputs = { x: $x };
        return ENGINE.runKernel(ZerosLike, inputs);
    }
    const zerosLike = op({ zerosLike_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const argMaxGradConfig = {
        kernelName: ArgMax,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => zerosLike(x) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const argMinGradConfig = {
        kernelName: ArgMin,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => zerosLike(x) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const asinGradConfig = {
        kernelName: Asin,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => div(dy, sqrt(sub(scalar(1), square(cast(x, 'float32'))))) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Adds two `tf.Tensor`s element-wise, A + B. Supports broadcasting.
     *
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3, 4]);
     * const b = tf.tensor1d([10, 20, 30, 40]);
     *
     * a.add(b).print();  // or tf.add(a, b)
     * ```
     *
     * ```js
     * // Broadcast add a with b.
     * const a = tf.scalar(5);
     * const b = tf.tensor1d([10, 20, 30, 40]);
     *
     * a.add(b).print();  // or tf.add(a, b)
     * ```
     * @param a The first `tf.Tensor` to add.
     * @param b The second `tf.Tensor` to add. Must have the same type as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function add_(a, b) {
        let $a = convertToTensor(a, 'a', 'add');
        let $b = convertToTensor(b, 'b', 'add');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Add$1, inputs);
    }
    const add$1 = op({ add_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const asinhGradConfig = {
        kernelName: Asinh,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return {
                x: () => {
                    const a = sqrt(add$1(scalar(1), square(cast(x, 'float32'))));
                    return div(dy, a);
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const atan2GradConfig = {
        kernelName: Atan2,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const outShape = assertAndGetBroadcastShape(a.shape, b.shape);
            const derA = () => {
                const d = add$1(square(a), square(b));
                let res = mul(dy, div(b, d));
                const reduceAxes = getReductionAxes(a.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = sum(res, reduceAxes);
                }
                return reshape$1(res, a.shape);
            };
            const derB = () => {
                const d = add$1(square(a), square(b));
                let res = neg(mul(dy, div(a, d)));
                const reduceAxes = getReductionAxes(b.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = sum(res, reduceAxes);
                }
                return reshape$1(res, b.shape);
            };
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const atanGradConfig = {
        kernelName: Atan,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => div(dy, add$1(square(cast(x, 'float32')), 1)) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const atanhGradConfig = {
        kernelName: Atanh,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => div(dy, sub(scalar(1), square(cast(x, 'float32')))) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function computePool2DInfo(inShape, filterSize, strides, dilations, pad, roundingMode, dataFormat = 'channelsLast') {
        const [filterHeight, filterWidth] = parseTupleParam(filterSize);
        let filterShape;
        if (dataFormat === 'channelsLast') {
            filterShape = [filterHeight, filterWidth, inShape[3], inShape[3]];
        }
        else if (dataFormat === 'channelsFirst') {
            filterShape = [filterHeight, filterWidth, inShape[1], inShape[1]];
        }
        else {
            throw new Error(`Unknown dataFormat ${dataFormat}`);
        }
        return computeConv2DInfo(inShape, filterShape, strides, dilations, pad, roundingMode, false, dataFormat);
    }
    /**
     * Computes the information for a forward pass of a convolution/pooling
     * operation.
     */
    function computeConv2DInfo(inShape, filterShape, strides, dilations, pad, roundingMode, depthwise = false, dataFormat = 'channelsLast') {
        let [batchSize, inHeight, inWidth, inChannels] = [-1, -1, -1, -1];
        if (dataFormat === 'channelsLast') {
            [batchSize, inHeight, inWidth, inChannels] = inShape;
        }
        else if (dataFormat === 'channelsFirst') {
            [batchSize, inChannels, inHeight, inWidth] = inShape;
        }
        else {
            throw new Error(`Unknown dataFormat ${dataFormat}`);
        }
        const [filterHeight, filterWidth, , filterChannels] = filterShape;
        const [strideHeight, strideWidth] = parseTupleParam(strides);
        const [dilationHeight, dilationWidth] = parseTupleParam(dilations);
        const effectiveFilterHeight = getEffectiveFilterSize(filterHeight, dilationHeight);
        const effectiveFilterWidth = getEffectiveFilterSize(filterWidth, dilationWidth);
        const { padInfo, outHeight, outWidth } = getPadAndOutInfo(pad, inHeight, inWidth, strideHeight, strideWidth, effectiveFilterHeight, effectiveFilterWidth, roundingMode, dataFormat);
        const outChannels = depthwise ? filterChannels * inChannels : filterChannels;
        let outShape;
        if (dataFormat === 'channelsFirst') {
            outShape = [batchSize, outChannels, outHeight, outWidth];
        }
        else if (dataFormat === 'channelsLast') {
            outShape = [batchSize, outHeight, outWidth, outChannels];
        }
        return {
            batchSize,
            dataFormat,
            inHeight,
            inWidth,
            inChannels,
            outHeight,
            outWidth,
            outChannels,
            padInfo,
            strideHeight,
            strideWidth,
            filterHeight,
            filterWidth,
            effectiveFilterHeight,
            effectiveFilterWidth,
            dilationHeight,
            dilationWidth,
            inShape,
            outShape,
            filterShape
        };
    }
    function computeOutputShape2D(inShape, fieldSize, stride, zeroPad, roundingMode) {
        if (zeroPad == null) {
            zeroPad = computeDefaultPad(inShape, fieldSize, stride);
        }
        const inputRows = inShape[0];
        const inputCols = inShape[1];
        const outputRows = round$1((inputRows - fieldSize + 2 * zeroPad) / stride + 1, roundingMode);
        const outputCols = round$1((inputCols - fieldSize + 2 * zeroPad) / stride + 1, roundingMode);
        return [outputRows, outputCols];
    }
    function computeDefaultPad(inputShape, fieldSize, stride, dilation = 1) {
        const effectiveFieldSize = getEffectiveFilterSize(fieldSize, dilation);
        return Math.floor((inputShape[0] * (stride - 1) - stride + effectiveFieldSize) / 2);
    }
    function parseTupleParam(param) {
        if (typeof param === 'number') {
            return [param, param, param];
        }
        if (param.length === 2) {
            return [param[0], param[1], 1];
        }
        return param;
    }
    /* See https://www.tensorflow.org/api_docs/python/tf/nn/atrous_conv2d
     * Atrous convolution is equivalent to standard convolution with upsampled
     * filters with effective_filter_height =
     * filter_height + (filter_height - 1) * (dilation - 1)
     * and effective_filter_width =
     * filter_width + (filter_width - 1) * (dilation - 1),
     * produced by inserting dilation - 1 zeros along consecutive elements across
     * the filters' spatial dimensions.
     * When there is a dilation, this converts a filter dimension to the
     * effective filter dimension, so it can be used in a standard convolution.
     */
    function getEffectiveFilterSize(filterSize, dilation) {
        if (dilation <= 1) {
            return filterSize;
        }
        return filterSize + (filterSize - 1) * (dilation - 1);
    }
    function getPadAndOutInfo(pad, inHeight, inWidth, strideHeight, strideWidth, filterHeight, filterWidth, roundingMode, dataFormat) {
        let padInfo;
        let outHeight;
        let outWidth;
        if (typeof pad === 'number') {
            const padType = (pad === 0) ? 'VALID' : 'NUMBER';
            padInfo = { top: pad, bottom: pad, left: pad, right: pad, type: padType };
            const outShape = computeOutputShape2D([inHeight, inWidth], filterHeight, strideHeight, pad, roundingMode);
            outHeight = outShape[0];
            outWidth = outShape[1];
        }
        else if (pad === 'same') {
            outHeight = Math.ceil(inHeight / strideHeight);
            outWidth = Math.ceil(inWidth / strideWidth);
            const padAlongHeight = Math.max(0, (outHeight - 1) * strideHeight + filterHeight - inHeight);
            const padAlongWidth = Math.max(0, (outWidth - 1) * strideWidth + filterWidth - inWidth);
            const top = Math.floor(padAlongHeight / 2);
            const bottom = padAlongHeight - top;
            const left = Math.floor(padAlongWidth / 2);
            const right = padAlongWidth - left;
            padInfo = { top, bottom, left, right, type: 'SAME' };
        }
        else if (pad === 'valid') {
            padInfo = { top: 0, bottom: 0, left: 0, right: 0, type: 'VALID' };
            outHeight = Math.ceil((inHeight - filterHeight + 1) / strideHeight);
            outWidth = Math.ceil((inWidth - filterWidth + 1) / strideWidth);
        }
        else if (typeof pad === 'object') {
            const top = dataFormat === 'channelsLast' ? pad[1][0] : pad[2][0];
            const bottom = dataFormat === 'channelsLast' ? pad[1][1] : pad[2][1];
            const left = dataFormat === 'channelsLast' ? pad[2][0] : pad[3][0];
            const right = dataFormat === 'channelsLast' ? pad[2][1] : pad[3][1];
            const padType = (top === 0 && bottom === 0 && left === 0 && right === 0) ?
                'VALID' :
                'EXPLICIT';
            padInfo = { top, bottom, left, right, type: padType };
            outHeight = round$1((inHeight - filterHeight + top + bottom) / strideHeight + 1, roundingMode);
            outWidth = round$1((inWidth - filterWidth + left + right) / strideWidth + 1, roundingMode);
        }
        else {
            throw Error(`Unknown padding parameter: ${pad}`);
        }
        return { padInfo, outHeight, outWidth };
    }
    /**
     * Rounds a value depending on the rounding mode
     * @param value
     * @param roundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function round$1(value, roundingMode) {
        if (!roundingMode) {
            return Math.trunc(value);
        }
        switch (roundingMode) {
            case 'round':
                // used for Caffe Conv
                return Math.round(value);
            case 'ceil':
                // used for Caffe Pool
                return Math.ceil(value);
            case 'floor':
                return Math.floor(value);
            default:
                throw new Error(`Unknown roundingMode ${roundingMode}`);
        }
    }
    function tupleValuesAreOne(param) {
        const [dimA, dimB, dimC] = parseTupleParam(param);
        return dimA === 1 && dimB === 1 && dimC === 1;
    }
    function eitherStridesOrDilationsAreOne(strides, dilations) {
        return tupleValuesAreOne(strides) || tupleValuesAreOne(dilations);
    }
    /**
     * Check validity of pad when using dimRoundingMode.
     * @param opDesc A string of op description
     * @param pad The type of padding algorithm.
     *   - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *   - `valid` output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     * @throws unknown padding parameter
     */
    function checkPadOnDimRoundingMode(opDesc, pad, dimRoundingMode) {
        if (dimRoundingMode != null) {
            if (typeof pad === 'string') {
                throw Error(`Error in ${opDesc}: pad must be an integer when using ` +
                    `dimRoundingMode ${dimRoundingMode} but got pad ${pad}.`);
            }
            else if (typeof pad === 'number') {
                assert(isInt(pad), () => `Error in ${opDesc}: pad must be an integer when using ` +
                    `dimRoundingMode ${dimRoundingMode} but got pad ${pad}.`);
            }
            else if (typeof pad === 'object') {
                pad.forEach(p => {
                    p.forEach(v => {
                        assert(isInt(v), () => `Error in ${opDesc}: pad must be an integer when using ` +
                            `dimRoundingMode ${dimRoundingMode} but got pad ${v}.`);
                    });
                });
            }
            else {
                throw Error(`Error in ${opDesc}: Unknown padding parameter: ${pad}`);
            }
        }
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the backprop of a 3d avg pool.
     *
     * @param dy The dy error, of rank 5 of shape
     *     [batchSize, depth, height, width, channels].
     * assumed.
     * @param input The original input image, of rank 5 or rank4 of shape
     *     [batchSize, depth, height, width, channels].
     * @param filterSize The filter size:
     *     `[filterDepth, filterHeight, filterWidth]`.
     *     `filterSize` is a single number,
     *     then `filterDepth == filterHeight == filterWidth`.
     * @param strides The strides of the pooling:
     *     `[strideDepth, strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param pad A string from: 'same', 'valid'. The type of padding algorithm
     *     used in the forward prop of the op.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function avgPool3dGrad_(dy, input, filterSize, strides, pad, dimRoundingMode) {
        const $dy = convertToTensor(dy, 'dy', 'avgPool3dGrad');
        const $input = convertToTensor(input, 'input', 'avgPool3dGrad');
        let dy5D = $dy;
        let input5D = $input;
        let reshapedTo5D = false;
        if ($input.rank === 4) {
            reshapedTo5D = true;
            dy5D = reshape$1($dy, [1, $dy.shape[0], $dy.shape[1], $dy.shape[2], $dy.shape[3]]);
            input5D = reshape$1($input, [
                1, $input.shape[0], $input.shape[1], $input.shape[2], $input.shape[3]
            ]);
        }
        assert(dy5D.rank === 5, () => `Error in avgPool3dGrad: dy must be rank 5 but got rank ` +
            `${dy5D.rank}.`);
        assert(input5D.rank === 5, () => `Error in avgPool3dGrad: input must be rank 5 but got rank ` +
            `${input5D.rank}.`);
        checkPadOnDimRoundingMode('avgPool3dGrad', pad, dimRoundingMode);
        const inputs = { dy: dy5D, input: input5D };
        const attrs = { filterSize, strides, pad, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(AvgPool3DGrad, inputs, attrs);
        if (reshapedTo5D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
        }
        return res;
    }
    const avgPool3dGrad = op({ avgPool3dGrad_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const avgPool3DGradConfig = {
        kernelName: AvgPool3D,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            const [x] = saved;
            const { filterSize, strides, pad, dimRoundingMode } = attrs;
            return {
                x: () => avgPool3dGrad(dy, x, filterSize, strides, pad, dimRoundingMode)
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the backprop of an 2D avg pool.
     *
     * @param dy The dy error, of rank 4 or rank 3 of shape
     *     [batchSize, height, width, channels]. If rank 3, batch of 1 is
     * assumed.
     * @param input The input image, of rank 4 or rank 3 of shape
     *     [batchSize, height, width, channels]. If rank 3, batch of 1 is
     * assumed.
     * @param filterSize The filter size: `[filterHeight, filterWidth]`. If
     *     `filterSize` is a single number, then `filterHeight == filterWidth`.
     * @param strides The strides of the pooling: `[strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param pad The type of padding algorithm used in the forward prop of the op.
     *     'same', 'valid', for more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *         https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     */
    function avgPoolGrad_(dy, input, filterSize, strides, pad) {
        const $dy = convertToTensor(dy, 'dy', 'avgPoolGrad');
        const $input = convertToTensor(input, 'input', 'avgPoolGrad');
        assert($input.rank === $dy.rank, () => `Rank of input (${$input.rank}) does not match rank of dy (${$dy.rank})`);
        let input4D = $input;
        let dy4D = $dy;
        let reshapedTo4D = false;
        if ($input.rank === 3) {
            reshapedTo4D = true;
            input4D =
                reshape$1($input, [1, $input.shape[0], $input.shape[1], $input.shape[2]]);
            dy4D = reshape$1($dy, [1, $dy.shape[0], $dy.shape[1], $dy.shape[2]]);
        }
        assert(dy4D.rank === 4, () => `Error in avgPoolGrad: dy must be rank 4 but got rank ` +
            `${dy4D.rank}.`);
        assert(input4D.rank === 4, () => `Error in avgPoolGrad: input must be rank 4 but got rank ` +
            `${input4D.rank}.`);
        const inputs = { dy: dy4D, input: input4D };
        const attrs = { filterSize, strides, pad };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(AvgPoolGrad, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const avgPoolGrad = op({ avgPoolGrad_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const avgPoolGradConfig = {
        kernelName: AvgPool,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            const [x] = saved;
            const { filterSize, strides, pad } = attrs;
            return { x: () => avgPoolGrad(dy, x, filterSize, strides, pad) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the dot product of two matrices, A * B. These must be matrices.
     *
     * ```js
     * const a = tf.tensor2d([1, 2], [1, 2]);
     * const b = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * a.matMul(b).print();  // or tf.matMul(a, b)
     * ```
     * @param a First matrix in dot product operation.
     * @param b Second matrix in dot product operation.
     * @param transposeA If true, `a` is transposed before multiplication.
     * @param transposeB If true, `b` is transposed before multiplication.
     *
     * @doc {heading: 'Operations', subheading: 'Matrices'}
     */
    function matMul_(a, b, transposeA = false, transposeB = false) {
        let $a = convertToTensor(a, 'a', 'matMul');
        let $b = convertToTensor(b, 'b', 'matMul');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        const attrs = { transposeA, transposeB };
        return ENGINE.runKernel(BatchMatMul, inputs, attrs);
    }
    const matMul = op({ matMul_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const batchMatMulGradConfig = {
        kernelName: BatchMatMul,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved, attrs) => {
            const [a, b] = saved;
            const { transposeA, transposeB } = attrs;
            if (!transposeA && !transposeB) {
                return {
                    a: () => matMul(dy, b, false, true),
                    b: () => matMul(a, dy, true, false)
                };
            }
            else if (!transposeA && transposeB) {
                return {
                    a: () => matMul(dy, b, false, false),
                    b: () => matMul(dy, a, true, false)
                };
            }
            else if (transposeA && !transposeB) {
                return {
                    a: () => matMul(b, dy, false, true),
                    b: () => matMul(a, dy, false, false)
                };
            }
            else {
                return {
                    a: () => matMul(b, dy, true, true),
                    b: () => matMul(dy, a, true, true)
                };
            }
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * This operation divides "spatial" dimensions `[1, ..., M]` of the input into
     * a grid of blocks of shape `blockShape`, and interleaves these blocks with
     * the "batch" dimension (0) such that in the output, the spatial
     * dimensions `[1, ..., M]` correspond to the position within the grid,
     * and the batch dimension combines both the position within a spatial block
     * and the original batch position. Prior to division into blocks,
     * the spatial dimensions of the input are optionally zero padded
     * according to `paddings`. See below for a precise description.
     *
     * ```js
     * const x = tf.tensor4d([1, 2, 3, 4], [1, 2, 2, 1]);
     * const blockShape = [2, 2];
     * const paddings = [[0, 0], [0, 0]];
     *
     * x.spaceToBatchND(blockShape, paddings).print();
     * ```
     *
     * @param x A `tf.Tensor`. N-D with `x.shape` = `[batch] + spatialShape +
     * remainingShape`, where spatialShape has `M` dimensions.
     * @param blockShape A 1-D array. Must have shape `[M]`, all values must
     * be >= 1.
     * @param paddings A 2-D array. Must have shape `[M, 2]`, all values must be >=
     *     0. `paddings[i] = [padStart, padEnd]` specifies the amount to zero-pad
     * from input dimension `i + 1`, which corresponds to spatial dimension `i`. It
     * is required that
     * `(inputShape[i + 1] + padStart + padEnd) % blockShape[i] === 0`
     *
     * This operation is equivalent to the following steps:
     *
     * 1. Zero-pad the start and end of dimensions `[1, ..., M]` of the input
     * according to `paddings` to produce `padded` of shape paddedShape.
     *
     * 2. Reshape `padded` to `reshapedPadded` of shape:
     * `[batch] + [paddedShape[1] / blockShape[0], blockShape[0], ...,
     * paddedShape[M] / blockShape[M-1], blockShape[M-1]] + remainingShape`
     *
     * 3. Permute dimensions of `reshapedPadded` to produce `permutedReshapedPadded`
     * of shape: `blockShape + [batch] + [paddedShape[1] / blockShape[0], ...,
     * paddedShape[M] / blockShape[M-1]] + remainingShape`
     *
     * 4. Reshape `permutedReshapedPadded` to flatten `blockShape` into the
     * batch dimension, producing an output tensor of shape:
     * `[batch * prod(blockShape)] + [paddedShape[1] / blockShape[0], ...,
     * paddedShape[M] / blockShape[M-1]] + remainingShape`
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function spaceToBatchND_(x, blockShape, paddings) {
        const $x = convertToTensor(x, 'x', 'spaceToBatchND');
        assert($x.rank >= 1 + blockShape.length, () => `input rank ${$x.rank} should be > than [blockShape] ${blockShape.length}`);
        assert(paddings.length === blockShape.length, () => `paddings.shape[0] ${paddings.length} must be equal to [blockShape] ${blockShape.length}`);
        assert($x.shape.reduce((a, b, i) => {
            if (i > 0 && i <= blockShape.length) {
                return a &&
                    ((b + paddings[i - 1][0] + paddings[i - 1][1]) %
                        blockShape[i - 1] ===
                        0);
            }
            return a;
        }, true), () => `input spatial dimensions ${$x.shape.slice(1)} with paddings ${paddings.toString()} must be divisible by blockShapes ${blockShape.toString()}`);
        const inputs = { x: $x };
        const attrs = { blockShape, paddings };
        return ENGINE.runKernel(SpaceToBatchND, inputs, attrs);
    }
    const spaceToBatchND = op({ spaceToBatchND_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const batchToSpaceNDGradConfig = {
        kernelName: BatchToSpaceND,
        gradFunc: (dy, saved, attrs) => {
            const { blockShape, crops } = attrs;
            return { x: () => spaceToBatchND(dy, blockShape, crops) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const broadcastToGradConfig = {
        kernelName: BroadcastTo,
        gradFunc: (dy, saved, attrs) => {
            const broadCastToAttrs = attrs;
            const inputShape = broadCastToAttrs.inputShape;
            const outputShape = broadCastToAttrs.shape;
            const reps = Array.from(outputShape);
            for (let i = inputShape.length - 1; i >= 0; i--) {
                if (inputShape[i] === outputShape[i]) {
                    reps[i] = 1;
                }
                else if (inputShape[i] !== 1) {
                    throw new Error(`broadcastTo(): [${inputShape}] cannot be broadcast to [${outputShape}].`);
                }
            }
            const axes = [];
            for (let i = 0; i < reps.length; i++) {
                if (reps[i] > 1) {
                    axes.push(i);
                }
            }
            return { x: () => sum(dy, axes, true /* keepDims */) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const castGradConfig = {
        kernelName: Cast,
        gradFunc: (dy) => {
            return { x: () => dy.clone() };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const ceilGradConfig = {
        kernelName: Ceil,
        gradFunc: (dy) => {
            // TODO(manrajgrover): Return null for gradients when backprop supports it.
            return { x: () => zerosLike(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of (a >= b) element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([2, 2, 2]);
     *
     * a.greaterEqual(b).print();
     * ```
     *
     * @param a The first input tensor.
     * @param b The second input tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function greaterEqual_(a, b) {
        let $a = convertToTensor(a, 'a', 'greaterEqual', 'string_or_numeric');
        let $b = convertToTensor(b, 'b', 'greaterEqual', 'string_or_numeric');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(GreaterEqual, inputs);
    }
    const greaterEqual = op({ greaterEqual_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of (a <= b) element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([2, 2, 2]);
     *
     * a.lessEqual(b).print();
     * ```
     *
     * @param a The first input tensor.
     * @param b The second input tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function lessEqual_(a, b) {
        let $a = convertToTensor(a, 'a', 'lessEqual', 'string_or_numeric');
        let $b = convertToTensor(b, 'b', 'lessEqual', 'string_or_numeric');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(LessEqual, inputs);
    }
    const lessEqual = op({ lessEqual_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of `a AND b` element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([false, false, true, true], 'bool');
     * const b = tf.tensor1d([false, true, false, true], 'bool');
     *
     * a.logicalAnd(b).print();
     * ```
     *
     * @param a The first input tensor. Must be of dtype bool.
     * @param b The second input tensor. Must be of dtype bool.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function logicalAnd_(a, b) {
        const $a = convertToTensor(a, 'a', 'logicalAnd', 'bool');
        const $b = convertToTensor(b, 'b', 'logicalAnd', 'bool');
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(LogicalAnd, inputs);
    }
    const logicalAnd = op({ logicalAnd_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a new tensor with the same values and shape as the specified
     * tensor.
     *
     * ```js
     * const x = tf.tensor([1, 2]);
     *
     * x.clone().print();
     * ```
     *
     * @param x The tensor to clone.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function clone_(x) {
        const $x = convertToTensor(x, 'x', 'clone', 'string_or_numeric');
        const inputs = { x: $x };
        // Note this op is called tf.identity in python. Hence the kernel name used
        // here.
        return ENGINE.runKernel(Identity, inputs);
    }
    const clone = op({ clone_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Broadcast an array to a compatible shape NumPy-style.
     *
     * The tensor's shape is compared to the broadcast shape from end to beginning.
     * Ones are prepended to the tensor's shape until it has the same length as
     * the broadcast shape. If input.shape[i]==shape[i], the (i+1)-th axis is
     * already broadcast-compatible. If input.shape[i]==1 and shape[i]==N, then
     * the input tensor is tiled N times along that axis (using tf.tile).
     *
     * @param input The tensor that is to be broadcasted.
     * @param shape The input is to be broadcast to this shape.
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function broadcastTo_(x, shape) {
        let input = convertToTensor(x, 'broadcastTo', 'x');
        const xShape = input.shape;
        if (shape.some(d => !(d > 0) || d % 1 !== 0)) {
            throw new Error(`broadcastTo(): Invalid broadcast shape [${shape}].`);
        }
        if (shape.length < input.rank) {
            throw new Error(`broadcastTo(): shape.length=${shape.length} < input.rank=${input.rank}.`);
        }
        if (shape.length > input.rank) {
            const newShape = input.shape.slice();
            while (newShape.length < shape.length) {
                newShape.unshift(1);
            }
            input = reshape$1(input, newShape);
        }
        const inputShape = input.shape;
        const reps = Array.from(shape);
        for (let i = shape.length - 1; i >= 0; i--) {
            if (inputShape[i] === shape[i]) {
                reps[i] = 1;
            }
            else if (input.shape[i] !== 1) {
                throw new Error(`broadcastTo(): [${xShape}] cannot be broadcast to [${shape}].`);
            }
        }
        const axes = reps.map((n, i) => n > 1 ? i : -1).filter(i => i >= 0);
        if (axes.length === 0) {
            return clone(input);
        }
        // TODO call broadcastTo kernel directly once backends implement broadcstTo
        const inputs = { x: input };
        const attrs = { reps };
        return ENGINE.runKernel(Tile, inputs, attrs);
    }
    const broadcastTo = op({ broadcastTo_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the elements, either `a` or `b` depending on the `condition`.
     *
     * If the condition is true, select from `a`, otherwise select from `b`.
     *
     * ```js
     * const cond = tf.tensor1d([false, false, true], 'bool');
     * const a = tf.tensor1d([1 , 2, 3]);
     * const b = tf.tensor1d([-1, -2, -3]);
     *
     * a.where(cond, b).print();
     * ```
     *
     * @param condition The input condition. Must be of dtype bool.
     * @param a If `condition` is rank 1, `a` may have a higher rank but
     *     its first dimension must match the size of `condition`.
     * @param b A tensor with the same dtype as `a` and with shape that is
     *     compatible with `a`.
     * @return A tensor with same dtype as `a` and `b`, and shape that is
     *     broadcastable from `a` and `b`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function where_(condition, a, b) {
        const $a = convertToTensor(a, 'a', 'where');
        const $b = convertToTensor(b, 'b', 'where');
        const $condition = convertToTensor(condition, 'condition', 'where', 'bool');
        // TODO: move this logic to forward function when the broadcastTo op is
        // implemented in WASM.
        // Find the broadcastable shape for $condition, $a, and $b.
        const broadcastShape = assertAndGetBroadcastShape(assertAndGetBroadcastShape($condition.shape, $a.shape), $b.shape);
        const $broadcastedCondition = broadcastTo($condition, broadcastShape);
        const $broadcastedA = broadcastTo($a, broadcastShape);
        const $broadcastedB = broadcastTo($b, broadcastShape);
        const inputs = {
            condition: $broadcastedCondition,
            t: $broadcastedA,
            e: $broadcastedB
        };
        return ENGINE.runKernel(Select, inputs);
    }
    const where = op({ where_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const clipByValueGradConfig = {
        kernelName: ClipByValue,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            const [x] = saved;
            const { clipValueMin, clipValueMax } = attrs;
            return {
                x: () => where(logicalAnd(greaterEqual(x, clipValueMin), lessEqual(x, clipValueMax)), dy, zerosLike(dy)),
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const complexAbsGradConfig = {
        kernelName: ComplexAbs,
        inputsToSave: ['x'],
        gradFunc: absGradConfig.gradFunc,
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Splits a `tf.Tensor` into sub tensors.
     *
     * If `numOrSizeSplits` is a number, splits `x` along dimension `axis`
     * into `numOrSizeSplits` smaller tensors.
     * Requires that `numOrSizeSplits` evenly divides `x.shape[axis]`.
     *
     * If `numOrSizeSplits` is a number array, splits `x` into
     * `numOrSizeSplits.length` pieces. The shape of the `i`-th piece has the
     * same size as `x` except along dimension `axis` where the size is
     * `numOrSizeSplits[i]`.
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4, 5, 6, 7, 8], [2, 4]);
     * const [a, b] = tf.split(x, 2, 1);
     * a.print();
     * b.print();
     *
     * const [c, d, e] = tf.split(x, [1, 2, 1], 1);
     * c.print();
     * d.print();
     * e.print();
     * ```
     *
     * @param x The input tensor to split.
     * @param numOrSizeSplits Either an integer indicating the number of
     * splits along the axis or an array of integers containing the sizes of
     * each output tensor along the axis. If a number then it must evenly divide
     * `x.shape[axis]`; otherwise the sum of sizes must match `x.shape[axis]`.
     * Can contain one -1 indicating that dimension is to be inferred.
     * @param axis The dimension along which to split. Defaults to 0 (the first
     * dim).
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function split_(x, numOrSizeSplits, axis = 0) {
        const $x = convertToTensor(x, 'x', 'split');
        const inputs = { x: $x };
        const attr = { numOrSizeSplits, axis };
        return ENGINE.runKernel(SplitV, inputs, attr);
    }
    const split = op({ split_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const concatGradConfig = {
        kernelName: Concat,
        saveAllInputs: true,
        gradFunc: (dy, saved, attrs) => {
            const shapes = saved.map(t => t.shape);
            const { axis } = attrs;
            const $axis = parseAxisParam(axis, saved[0].shape)[0];
            const sizeSplits = shapes.map(s => s[$axis]);
            const derTensors = split(dy, sizeSplits, $axis);
            return derTensors.map(t => () => t);
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the derivative of the filter of a 2D convolution.
     *
     * @param x The input tensor, of rank 4 or rank 3 of shape
     *     [batch, height, width, inChannels]. If rank 3, batch of 1 is assumed.
     * @param dy The dy image, of rank 4 or rank 3, of shape
     *     [batch, height, width, outDepth]. If rank 3, batch of 1 is assumed.
     * @param filterShape The shape of the filter, length 4,
     *     [filterHeight, filterWidth, inDepth, outDepth].
     * @param strides The strides of the convolution: [strideHeight,
     * strideWidth].
     * @param pad A string from: 'same', 'valid'. The type of padding algorithm
     *     used in the forward prop of the op.
     * @param dataFormat: An optional string from: "NHWC", "NCHW". Defaults to
     *     "NHWC". Specify the data format of the input and output data. With the
     *     default format "NHWC", the data is stored in the order of: [batch,
     *     height, width, channels].
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function conv2DBackpropFilter_(x, dy, filterShape, strides, pad, dataFormat = 'NHWC', dimRoundingMode) {
        let x4D = x;
        if (x.rank === 3) {
            x4D = reshape$1(x, [1, x.shape[0], x.shape[1], x.shape[2]]);
        }
        let dy4D = dy;
        if (dy4D.rank === 3) {
            dy4D = reshape$1(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2]]);
        }
        assert(x4D.rank === 4, () => `Error in conv2dDerFilter: input must be rank 4, but got shape ` +
            `${x4D.shape}.`);
        assert(dy4D.rank === 4, () => `Error in conv2dDerFilter: dy must be rank 4, but got shape ` +
            `${dy4D.shape}.`);
        assert(filterShape.length === 4, () => `Error in conv2dDerFilter: filterShape must be length 4, but got ` +
            `${filterShape}.`);
        const inDepth = dataFormat === 'NHWC' ? x4D.shape[3] : x4D.shape[1];
        const outDepth = dataFormat === 'NHWC' ? dy4D.shape[3] : dy4D.shape[1];
        assert(inDepth === filterShape[2], () => `Error in conv2dDerFilter: depth of input ${inDepth}) must ` +
            `match input depth in filter (${filterShape[2]}.`);
        assert(outDepth === filterShape[3], () => `Error in conv2dDerFilter: depth of dy (${outDepth}) must ` +
            `match output depth for filter (${filterShape[3]}).`);
        checkPadOnDimRoundingMode('conv2dDerFilter', pad, dimRoundingMode);
        const inputs = { x: x4D, dy: dy4D };
        const attrs = { strides, pad, dataFormat, dimRoundingMode, filterShape };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(Conv2DBackpropFilter, inputs, attrs);
    }
    const conv2DBackpropFilter = op({ conv2DBackpropFilter_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the derivative of the input of a 2D convolution.
     *
     * @param xShape The shape of the input: [batch, height, width, inDepth].
     * If length of 3, batch of 1 is assumed.
     * @param dy The derivative of the output, of rank 4 or rank 3 of shape
     *   `[batch, outHeight, outWidth, outDepth]`. If rank 3, batch of 1 is
     * assumed.
     * @param filter The filter, rank 4, of shape
     *     `[filterHeight, filterWidth, inDepth, outDepth]`.
     * @param strides The strides of the convolution: `[strideHeight,
     * strideWidth]`.
     * @param pad The type of padding algorithm used:
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     * @param dataFormat: An optional string from: "NHWC", "NCHW". Defaults to
     *     "NHWC". Specify the data format of the input and output data. With the
     *     default format "NHWC", the data is stored in the order of: [batch,
     *     height, width, channels].
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function conv2DBackpropInput_(xShape, dy, filter, strides, pad, dataFormat = 'NHWC', dimRoundingMode) {
        assert(xShape.length === dy.rank, () => `Length of inShape ` +
            `(${xShape.length}) and rank of dy (${dy.rank}) must match`);
        let xShape4D = xShape;
        let dy4D = dy;
        let reshapedTo4D = false;
        if (dy.rank === 3) {
            reshapedTo4D = true;
            dy4D = reshape$1(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2]]);
            xShape4D = [1, xShape[0], xShape[1], xShape[2]];
        }
        assert(xShape4D.length === 4, () => `Error in conv2dDerInput: inShape must be length 4, but got length ` +
            `${xShape4D.length}.`);
        assert(dy4D.rank === 4, () => `Error in conv2dDerInput: dy must be rank 4, but got ` +
            `rank ${dy4D.rank}`);
        assert(filter.rank === 4, () => `Error in conv2dDerInput: filter must be rank 4, but got ` +
            `rank ${filter.rank}`);
        const inDepth = dataFormat === 'NHWC' ? xShape4D[3] : xShape4D[1];
        const outDepth = dataFormat === 'NHWC' ? dy4D.shape[3] : dy4D.shape[1];
        assert(inDepth === filter.shape[2], () => `Error in conv2dDerInput: depth of input (${inDepth}) must ` +
            `match input depth for filter ${filter.shape[2]}.`);
        assert(outDepth === filter.shape[3], () => `Error in conv2dDerInput: depth of output (${outDepth}) must ` +
            `match output depth for filter ${filter.shape[3]}.`);
        checkPadOnDimRoundingMode('conv2dDerInput', pad, dimRoundingMode);
        const inputs = { dy: dy4D, filter };
        const attrs = { strides, pad, dataFormat, dimRoundingMode, inputShape: xShape4D };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(Conv2DBackpropInput, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const conv2DBackpropInput = op({ conv2DBackpropInput_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const conv2DGradConfig = {
        kernelName: Conv2D$1,
        inputsToSave: ['x', 'filter'],
        gradFunc: (dy, saved, attrs) => {
            const [x4D, $filter] = saved;
            const { dilations, strides, pad, dataFormat } = attrs;
            assert(tupleValuesAreOne(dilations), () => 'Error in gradient of conv2D: dilation rates greater than 1 ' +
                `are not yet supported in gradients. Got dilations '${dilations}'`);
            return {
                x: () => conv2DBackpropInput(x4D.shape, dy, $filter, strides, pad, dataFormat),
                filter: () => conv2DBackpropFilter(x4D, dy, $filter.shape, strides, pad, dataFormat)
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes a 2D convolution over the input x.
     *
     * @param x The input tensor, of rank 4 or rank 3, of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is
     * assumed.
     * @param filter The filter, rank 4, of shape
     *     `[filterHeight, filterWidth, inDepth, outDepth]`.
     * @param strides The strides of the convolution: `[strideHeight,
     * strideWidth]`.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dataFormat: An optional string from: "NHWC", "NCHW". Defaults to
     *     "NHWC". Specify the data format of the input and output data. With the
     *     default format "NHWC", the data is stored in the order of: [batch,
     *     height, width, channels].
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     in atrous convolution. Defaults to `[1, 1]`. If `dilations` is a single
     *     number, then `dilationHeight == dilationWidth`. If it is greater than
     *     1, then all values of `strides` must be 1.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function conv2d_(x, filter, strides, pad, dataFormat = 'NHWC', dilations = [1, 1], dimRoundingMode) {
        const $x = convertToTensor(x, 'x', 'conv2d', 'float32');
        const $filter = convertToTensor(filter, 'filter', 'conv2d', 'float32');
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        assert(x4D.rank === 4, () => `Error in conv2d: input must be rank 4, but got rank ${x4D.rank}.`);
        assert($filter.rank === 4, () => `Error in conv2d: filter must be rank 4, but got rank ` +
            `${$filter.rank}.`);
        checkPadOnDimRoundingMode('conv2d', pad, dimRoundingMode);
        const inDepth = dataFormat === 'NHWC' ? x4D.shape[3] : x4D.shape[1];
        assert(inDepth === $filter.shape[2], () => `Error in conv2d: depth of input (${inDepth}) must match ` +
            `input depth for filter ${$filter.shape[2]}.`);
        assert(eitherStridesOrDilationsAreOne(strides, dilations), () => 'Error in conv2D: Either strides or dilations must be 1. ' +
            `Got strides ${strides} and dilations '${dilations}'`);
        const inputs = { x: x4D, filter: $filter };
        const attrs = { strides, pad, dataFormat, dilations, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(Conv2D$1, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const conv2d$1 = op({ conv2d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const conv2DBackpropInputGradConfig = {
        kernelName: Conv2DBackpropInput,
        inputsToSave: ['dy', 'filter'],
        gradFunc: (ddx, saved, attrs) => {
            const [dy, filter] = saved;
            const { strides, pad, dataFormat, dimRoundingMode } = attrs;
            return {
                dy: () => conv2d$1(ddx, filter, strides, pad, dataFormat, 1 /* dilations */, dimRoundingMode),
                filter: () => conv2DBackpropFilter(ddx, dy, filter.shape, strides, pad, dataFormat, dimRoundingMode)
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the derivative of the filter of a 3D convolution.
     *
     * @param x The input tensor, of rank 5 or rank 4 of shape
     *     [batch, depth, height, width, inChannels]. If rank 4, batch of 1 is
     *     assumed.
     * @param dy The dy image, of rank 5 or rank 4, of shape
     *     [batch, depth, height, width, outDepth]. If rank 4, batch of 1 is
     *     assumed.
     * @param filterShape The shape of the filter, length 5,
     *     [filterDepth, filterHeight, filterWidth, inDepth, outDepth].
     * @param strides The strides of the convolution: [strideDepth, strideHeight,
     * strideWidth].
     * @param pad A string from: 'same', 'valid'. The type of padding algorithm
     *     used in the forward prop of the op.
     */
    function conv3DBackpropFilter_(x, dy, filterShape, strides, pad) {
        let x5D = x;
        if (x.rank === 4) {
            x5D = reshape$1(x, [1, x.shape[0], x.shape[1], x.shape[2], x.shape[3]]);
        }
        let dy5D = dy;
        if (dy5D.rank === 4) {
            dy5D = reshape$1(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2], dy.shape[3]]);
        }
        assert(x5D.rank === 5, () => `Error in conv3dDerFilter: input must be rank 5, but got shape ` +
            `${x5D.shape}.`);
        assert(dy5D.rank === 5, () => `Error in conv3dDerFilter: dy must be rank 5, but got shape ` +
            `${dy5D.shape}.`);
        assert(filterShape.length === 5, () => `Error in conv3dDerFilter: filterShape must be length 5, but got ` +
            `${filterShape}.`);
        assert(x5D.shape[4] === filterShape[3], () => `Error in conv3dDerFilter: depth of input ${x5D.shape[4]}) must ` +
            `match input depth in filter (${filterShape[3]}.`);
        assert(dy5D.shape[4] === filterShape[4], () => `Error in conv3dDerFilter: depth of dy (${dy5D.shape[4]}) must ` +
            `match output depth for filter (${filterShape[4]}).`);
        const inputs = { x: x5D, dy: dy5D };
        const attrs = { strides, pad, filterShape };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(Conv3DBackpropFilterV2, inputs, attrs);
    }
    const conv3DBackpropFilter = op({ conv3DBackpropFilter_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the derivative of the input of a 3D convolution.
     *
     * @param xShape The shape of the input: [batch, depth, height, width,
     * in_channels]. If length of 4, batch of 1 is assumed.
     * @param dy The derivative of the output, of rank 5 or rank 4 of shape
     *   `[batch, outDepth, outHeight, outWidth, in_channels]`.
     * If rank 4, batch of 1 is assumed.
     * @param filter The filter, rank 5, of shape
     *     `[filterDepth, filterHeight, filterWidth, inDepth, outDepth]`.
     * @param strides The strides of the convolution: `[strideDepth, strideHeight,
     * strideWidth]`.
     * @param pad The type of padding algorithm used:
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     */
    function conv3DBackpropInput_(xShape, dy, filter, strides, pad) {
        assert(xShape.length === dy.rank, () => `Length of inShape ` +
            `(${xShape.length}) and rank of dy (${dy.rank}) must match`);
        let xShape5D = xShape;
        let dy5D = dy;
        let reshapedTo5D = false;
        if (dy.rank === 4) {
            reshapedTo5D = true;
            dy5D = reshape$1(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2], dy.shape[3]]);
            xShape5D = [1, xShape[0], xShape[1], xShape[2], xShape[3]];
        }
        const inDepth = xShape5D[4];
        const outDepth = dy5D.shape[4];
        assert(xShape5D.length === 5, () => `Error in conv3dDerInput: inShape must be length 5, but got length ` +
            `${xShape5D.length}.`);
        assert(dy5D.rank === 5, () => `Error in conv3dDerInput: dy must be rank 5, but got ` +
            `rank ${dy5D.rank}`);
        assert(filter.rank === 5, () => `Error in conv3dDerInput: filter must be rank 5, but got ` +
            `rank ${filter.rank}`);
        assert(inDepth === filter.shape[3], () => `Error in conv3dDerInput: depth of input (${inDepth}) must ` +
            `match input depth for filter ${filter.shape[3]}.`);
        assert(outDepth === filter.shape[4], () => `Error in conv3dDerInput: depth of output (${outDepth}) must ` +
            `match output depth for filter ${filter.shape[4]}.`);
        const inputs = { dy: dy5D, filter };
        const attrs = { pad, strides, inputShape: xShape5D };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(Conv3DBackpropInputV2, inputs, attrs);
        if (reshapedTo5D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
        }
        return res;
    }
    const conv3DBackpropInput = op({ conv3DBackpropInput_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const conv3DGradConfig = {
        kernelName: Conv3D$1,
        inputsToSave: ['x', 'filter'],
        gradFunc: (dy, saved, attrs) => {
            const { dilations, strides, pad } = attrs;
            assert(tupleValuesAreOne(dilations), () => 'Error in gradient of conv3D: dilation rates greater than 1 are ' +
                `not yet supported in gradients. Got dilations '${dilations}'`);
            const [x5D, $filter] = saved;
            return {
                x: () => conv3DBackpropInput(x5D.shape, dy, $filter, strides, pad),
                filter: () => conv3DBackpropFilter(x5D, dy, $filter.shape, strides, pad)
            };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes sin of the input Tensor element-wise: `sin(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, Math.PI / 2, Math.PI * 3 / 4]);
     *
     * x.sin().print();  // or tf.sin(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function sin_(x) {
        const $x = convertToTensor(x, 'x', 'sin', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Sin, inputs);
    }
    const sin = op({ sin_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const cosGradConfig = {
        kernelName: Cos,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => mul(neg(sin(cast(x, 'float32'))), dy) };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes hyperbolic sin of the input `tf.Tensor` element-wise: `sinh(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.sinh().print();  // or tf.sinh(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function sinh_(x) {
        const $x = convertToTensor(x, 'x', 'sinh');
        const inputs = { x: $x };
        return ENGINE.runKernel(Sinh, inputs);
    }
    const sinh = op({ sinh_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const coshGradConfig = {
        kernelName: Cosh,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => mul(sinh(cast(x, 'float32')), dy) };
        }
    };

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns true if the axis specifies the inner most dimensions of the
     * array.
     */
    function axesAreInnerMostDims(axes, rank) {
        for (let i = 0; i < axes.length; ++i) {
            if (axes[axes.length - i - 1] !== rank - 1 - i) {
                return false;
            }
        }
        return true;
    }
    function combineLocations(outputLoc, reduceLoc, axes) {
        const rank = outputLoc.length + reduceLoc.length;
        const loc = [];
        let outIdx = 0;
        let reduceIdx = 0;
        for (let dim = 0; dim < rank; dim++) {
            if (axes.indexOf(dim) === -1) {
                loc.push(outputLoc[outIdx++]);
            }
            else {
                loc.push(reduceLoc[reduceIdx++]);
            }
        }
        return loc;
    }
    function computeOutAndReduceShapes(aShape, axes) {
        const outShape = [];
        const rank = aShape.length;
        for (let dim = 0; dim < rank; dim++) {
            if (axes.indexOf(dim) === -1) {
                outShape.push(aShape[dim]);
            }
        }
        const reduceShape = axes.map(dim => aShape[dim]);
        return [outShape, reduceShape];
    }
    function expandShapeToKeepDim(shape, axes) {
        const reduceSubShape = axes.map(x => 1);
        return combineLocations(shape, reduceSubShape, axes);
    }
    /**
     * Returns the axes permutation to be used with `tf.transpose`, if such
     * permutation is necessary. Otherwise it returns null. This method is used by
     * operations that operate only on inner-most axes.
     */
    function getAxesPermutation(axes, rank) {
        if (axesAreInnerMostDims(axes, rank)) {
            return null;
        }
        const result = [];
        for (let i = 0; i < rank; ++i) {
            if (axes.indexOf(i) === -1) {
                result.push(i);
            }
        }
        axes.forEach(axis => result.push(axis));
        return result;
    }
    /** Returns the axes permutation that undoes the original permutation. */
    function getUndoAxesPermutation(axes) {
        return axes.map((axis, i) => [i, axis])
            .sort((a, b) => a[1] - b[1])
            .map(x => x[0]);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the cumulative sum of a `tf.Tensor` along `axis`.
     *
     * ```js
     * const x = tf.tensor([1, 2, 3, 4]);
     * x.cumsum().print();
     * ```
     * ```js
     * const x = tf.tensor([[1, 2], [3, 4]]);
     * x.cumsum().print();
     * ```
     *
     * @param x The input tensor to be summed.
     * @param axis The axis along which to sum. Optional. Defaults to 0.
     * @param exclusive Whether to perform exclusive cumulative sum. Optional.
     *     Defaults to false. If set to true then the sum of each tensor entry
     *     does not include its own value, but only the values previous to it
     *     along the specified axis.
     * @param reverse Whether to sum in the opposite direction. Optional.
     *     Defaults to false.
     *
     * @doc {heading: 'Operations', subheading: 'Scan'}
     */
    function cumsum_(x, axis = 0, exclusive = false, reverse = false) {
        const $x = convertToTensor(x, 'x', 'cumsum');
        const inputs = { x: $x };
        const attrs = { axis, exclusive, reverse };
        return ENGINE.runKernel(Cumsum, inputs, attrs);
    }
    const cumsum = op({ cumsum_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Executes the provided function `fn` and after it is executed, cleans up all
     * intermediate tensors allocated by `fn` except those returned by `fn`.
     * `fn` must not return a Promise (async functions not allowed). The returned
     * result can be a complex object.
     *
     * Using this method helps avoid memory leaks. In general, wrap calls to
     * operations in `tf.tidy` for automatic memory cleanup.
     *
     * NOTE: Variables do *not* get cleaned up when inside a tidy(). If you want to
     * dispose variables, please use `tf.disposeVariables` or call dispose()
     * directly on variables.
     *
     * ```js
     * // y = 2 ^ 2 + 1
     * const y = tf.tidy(() => {
     *   // a, b, and one will be cleaned up when the tidy ends.
     *   const one = tf.scalar(1);
     *   const a = tf.scalar(2);
     *   const b = a.square();
     *
     *   console.log('numTensors (in tidy): ' + tf.memory().numTensors);
     *
     *   // The value returned inside the tidy function will return
     *   // through the tidy, in this case to the variable y.
     *   return b.add(one);
     * });
     *
     * console.log('numTensors (outside tidy): ' + tf.memory().numTensors);
     * y.print();
     * ```
     *
     * @param nameOrFn The name of the closure, or the function to execute.
     *     If a name is provided, the 2nd argument should be the function.
     *     If debug mode is on, the timing and the memory usage of the function
     *     will be tracked and displayed on the console using the provided name.
     * @param fn The function to execute.
     *
     * @doc {heading: 'Performance', subheading: 'Memory'}
     */
    function tidy(nameOrFn, fn) {
        return ENGINE.tidy(nameOrFn, fn);
    }
    /**
     * Disposes any `tf.Tensor`s found within the provided object.
     *
     * @param container an object that may be a `tf.Tensor` or may directly
     *     contain `tf.Tensor`s, such as a `Tensor[]` or `{key: Tensor, ...}`. If
     *     the object is not a `tf.Tensor` or does not contain `Tensors`, nothing
     *     happens. In general it is safe to pass any object here, except that
     *     `Promise`s are not supported.
     *
     * @doc {heading: 'Performance', subheading: 'Memory'}
     */
    function dispose(container) {
        const tensors = getTensorsInContainer(container);
        tensors.forEach(tensor => tensor.dispose());
    }
    /**
     * Keeps a `tf.Tensor` generated inside a `tf.tidy` from being disposed
     * automatically.
     *
     * ```js
     * let b;
     * const y = tf.tidy(() => {
     *   const one = tf.scalar(1);
     *   const a = tf.scalar(2);
     *
     *   // b will not be cleaned up by the tidy. a and one will be cleaned up
     *   // when the tidy ends.
     *   b = tf.keep(a.square());
     *
     *   console.log('numTensors (in tidy): ' + tf.memory().numTensors);
     *
     *   // The value returned inside the tidy function will return
     *   // through the tidy, in this case to the variable y.
     *   return b.add(one);
     * });
     *
     * console.log('numTensors (outside tidy): ' + tf.memory().numTensors);
     * console.log('y:');
     * y.print();
     * console.log('b:');
     * b.print();
     * ```
     *
     * @param result The tensor to keep from being disposed.
     *
     * @doc {heading: 'Performance', subheading: 'Memory'}
     */
    function keep(result) {
        return ENGINE.keep(result);
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Converts two real numbers to a complex number.
     *
     * Given a tensor `real` representing the real part of a complex number, and a
     * tensor `imag` representing the imaginary part of a complex number, this
     * operation returns complex numbers elementwise of the form [r0, i0, r1, i1],
     * where r represents the real part and i represents the imag part.
     *
     * The input tensors real and imag must have the same shape.
     *
     * ```js
     * const real = tf.tensor1d([2.25, 3.25]);
     * const imag = tf.tensor1d([4.75, 5.75]);
     * const complex = tf.complex(real, imag);
     *
     * complex.print();
     * ```
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function complex_(real, imag) {
        const $real = convertToTensor(real, 'real', 'complex');
        const $imag = convertToTensor(imag, 'imag', 'complex');
        assertShapesMatch($real.shape, $imag.shape, `real and imag shapes, ${$real.shape} and ${$imag.shape}, ` +
            `must match in call to tf.complex().`);
        const inputs = { real: $real, imag: $imag };
        return ENGINE.runKernel(Complex, inputs);
    }
    const complex = op({ complex_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the imaginary part of a complex (or real) tensor.
     *
     * Given a tensor input, this operation returns a tensor of type float that is
     * the imaginary part of each element in input considered as a complex number.
     * If input is real, a tensor of all zeros is returned.
     *
     * ```js
     * const x = tf.complex([-2.25, 3.25], [4.75, 5.75]);
     * tf.imag(x).print();
     * ```
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function imag_(input) {
        const $input = convertToTensor(input, 'input', 'imag');
        const inputs = { input: $input };
        return ENGINE.runKernel(Imag, inputs);
    }
    const imag = op({ imag_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the real part of a complex (or real) tensor.
     *
     * Given a tensor input, this operation returns a tensor of type float that is
     * the real part of each element in input considered as a complex number.
     *
     * If the input is real, it simply makes a clone.
     *
     * ```js
     * const x = tf.complex([-2.25, 3.25], [4.75, 5.75]);
     * tf.real(x).print();
     * ```
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function real_(input) {
        const $input = convertToTensor(input, 'input', 'real');
        const inputs = { input: $input };
        return ENGINE.runKernel(Real, inputs);
    }
    const real = op({ real_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Transposes the `tf.Tensor`. Permutes the dimensions according to `perm`.
     *
     * The returned `tf.Tensor`'s dimension `i` will correspond to the input
     * dimension `perm[i]`. If `perm` is not given, it is set to `[n-1...0]`,
     * where `n` is the rank of the input `tf.Tensor`. Hence by default, this
     * operation performs a regular matrix transpose on 2-D input `tf.Tensor`s.
     *
     * ```js
     * const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
     *
     * a.transpose().print();  // or tf.transpose(a)
     * ```
     *
     * @param x The tensor to transpose.
     * @param perm The permutation of the dimensions of a.
     * @param conjugate Will conjugate complex input if true.
     *
     * @doc {heading: 'Operations', subheading: 'Matrices'}
     */
    function transpose_(x, perm, conjugate) {
        const $x = convertToTensor(x, 'x', 'transpose');
        if (perm == null) {
            perm = $x.shape.map((s, i) => i).reverse();
        }
        assert($x.rank === perm.length, () => `Error in transpose: rank of input ${$x.rank} ` +
            `must match length of perm ${perm}.`);
        perm.forEach(axis => {
            assert(axis >= 0 && axis < $x.rank, () => `All entries in 'perm' must be between 0 and ${$x.rank - 1}` +
                ` but got ${perm}`);
        });
        if ($x.rank <= 1) {
            return $x.clone();
        }
        const inputs = { x: $x };
        const attrs = { perm };
        if ($x.dtype === 'complex64') {
            return tidy(() => {
                let $real = real($x);
                let $imag = imag($x);
                $real = ENGINE.runKernel(Transpose, { x: $real }, attrs);
                $imag = ENGINE.runKernel(Transpose, { x: $imag }, attrs);
                if (conjugate) {
                    $imag = neg($imag);
                }
                return complex($real, $imag);
            });
        }
        return ENGINE.runKernel(Transpose, inputs, attrs);
    }
    const transpose = op({ transpose_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const cumsumGradConfig = {
        kernelName: Cumsum,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            const [x] = saved;
            const { axis, exclusive, reverse } = attrs;
            return {
                x: () => {
                    const permutation = getAxesPermutation([axis], x.rank);
                    let out = cumsum(dy, axis, exclusive, !reverse);
                    if (permutation != null) {
                        out = transpose(out, permutation);
                    }
                    return out;
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function depthwiseConv2dNativeBackpropFilter_(x, dy, filterShape, strides, pad, dilations = [1, 1], dimRoundingMode) {
        let x4D = x;
        if (x.rank === 3) {
            x4D = reshape$1(x, [1, x.shape[0], x.shape[1], x.shape[2]]);
        }
        let dy4D = dy;
        if (dy4D.rank === 3) {
            dy4D = reshape$1(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2]]);
        }
        const inputs = { x: x4D, dy: dy4D };
        const attrs = { strides, pad, dimRoundingMode, dilations, filterShape };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(DepthwiseConv2dNativeBackpropFilter, inputs, attrs);
    }
    const depthwiseConv2dNativeBackpropFilter = op({ depthwiseConv2dNativeBackpropFilter_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function depthwiseConv2dNativeBackpropInput_(xShape, dy, filter, strides, pad, dilations = [1, 1], dimRoundingMode) {
        let dy4D = dy;
        let reshapedTo4D = false;
        if (dy.rank === 3) {
            reshapedTo4D = true;
            dy4D = reshape$1(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2]]);
        }
        const inputs = { dy: dy4D, filter };
        const attrs = { strides, pad, dimRoundingMode, dilations, inputShape: xShape };
        const res = 
        // tslint:disable-next-line: no-unnecessary-type-assertion
        ENGINE.runKernel(DepthwiseConv2dNativeBackpropInput, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const depthwiseConv2dNativeBackpropInput = op({ depthwiseConv2dNativeBackpropInput_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const depthwiseConv2dNativeGradConfig = {
        kernelName: DepthwiseConv2dNative,
        inputsToSave: ['x', 'filter'],
        gradFunc: (dy, saved, attrs) => {
            const { dilations, strides, pad, dimRoundingMode } = attrs;
            const $dilations = dilations == null ? [1, 1] : dilations;
            assert(tupleValuesAreOne($dilations), () => 'Error in gradient of depthwiseConv2dNative: dilation rates ' +
                `greater than 1 are not yet supported. Got dilations ` +
                `'${$dilations}'`);
            const [x, filter] = saved;
            assert(x.rank === 4, () => `Error in gradient of depthwiseConv2dNative: input must be ` +
                `rank 4, but got rank ${x.rank}.`);
            assert(filter.rank === 4, () => `Error in gradient of depthwiseConv2dNative: filter must be ` +
                `rank 4, but got rank ${filter.rank}.`);
            assert(x.shape[3] === filter.shape[2], () => `Error in gradient of depthwiseConv2d: number of input ` +
                `channels (${x.shape[3]}) must match the inChannels dimension ` +
                `in filter ${filter.shape[2]}.`);
            assert(eitherStridesOrDilationsAreOne(strides, $dilations), () => 'Error in gradient of depthwiseConv2d: Either strides or ' +
                `dilations must be  1. Got strides ${strides} and dilations ` +
                `'${$dilations}'.`);
            checkPadOnDimRoundingMode('depthwiseConv2d', pad, dimRoundingMode);
            return {
                x: () => depthwiseConv2dNativeBackpropInput(x.shape, dy, filter, strides, pad, $dilations, dimRoundingMode),
                filter: () => depthwiseConv2dNativeBackpropFilter(x, dy, filter.shape, strides, pad, $dilations, dimRoundingMode),
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const dilation2dGradConfig = {
        kernelName: Dilation2D,
        inputsToSave: ['x', 'filter'],
        gradFunc: (dy, saved, attrs) => {
            const [x, filter] = saved;
            const inputInputs = { x, filter, dy };
            const filterInputs = { x, filter, dy };
            return {
                x: () => ENGINE.runKernel(Dilation2DBackpropInput, inputInputs, attrs),
                filter: () => ENGINE.runKernel(Dilation2DBackpropFilter, filterInputs, attrs)
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const eluGradConfig = {
        kernelName: Elu$1,
        outputsToSave: [true],
        gradFunc: (dy, saved) => {
            const [y] = saved;
            const inputs = { dy, y };
            return { x: () => ENGINE.runKernel(EluGrad, inputs) };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes exponential of the input `tf.Tensor` element-wise. `e ^ x`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, -3]);
     *
     * x.exp().print();  // or tf.exp(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function exp_(x) {
        const $x = convertToTensor(x, 'x', 'exp');
        const inputs = { x: $x };
        return ENGINE.runKernel(Exp, inputs);
    }
    const exp = op({ exp_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const erfGradConfig = {
        kernelName: Erf,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            const a = mul(exp(neg(square(x))), 2 / Math.sqrt(Math.PI));
            return { x: () => mul(dy, a) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const expGradConfig = {
        kernelName: Exp,
        outputsToSave: [true],
        gradFunc: (dy, saved) => {
            const [y] = saved;
            return { x: () => mul(dy, y) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const expandDimsGradConfig = {
        kernelName: ExpandDims,
        inputsToSave: ['input'],
        gradFunc: (dy, saved) => {
            const [input] = saved;
            return { input: () => reshape$1(dy, input.shape) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const expm1GradConfig = {
        kernelName: Expm1,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => mul(dy, exp(x)) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const floorGradConfig = {
        kernelName: Floor,
        gradFunc: (dy) => {
            return { x: () => zerosLike(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const floorDivGradConfig = {
        kernelName: FloorDiv,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const outShape = assertAndGetBroadcastShape(a.shape, b.shape);
            const derA = () => {
                const res = div(dy, cast(b, 'float32'));
                const reduceAxes = getReductionAxes(a.shape, outShape);
                if (reduceAxes.length > 0) {
                    return reshape$1(sum(res, reduceAxes), a.shape);
                }
                return res;
            };
            const derB = () => {
                let res = mul(dy, cast(a, 'float32'));
                const reduceAxes = getReductionAxes(b.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = reshape$1(sum(res, reduceAxes), b.shape);
                }
                const tmp = square(b);
                return neg(div(res, cast(tmp, 'float32')));
            };
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes reciprocal of square root of the input `tf.Tensor` element-wise:
     * `y = 1 / sqrt(x)`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 4, -1]);
     *
     * x.rsqrt().print();  // or tf.rsqrt(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function rsqrt_(x) {
        const $x = convertToTensor(x, 'x', 'rsqrt', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Rsqrt, inputs);
    }
    const rsqrt = op({ rsqrt_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Construct a tensor by repeating it the number of times given by reps.
     *
     * This operation creates a new tensor by replicating `input` `reps`
     * times. The output tensor's `i`th dimension has `input.shape[i] *
     * reps[i]` elements, and the values of `input` are replicated
     * `reps[i]` times along the `i`th dimension. For example, tiling
     * `[a, b, c, d]` by `[2]` produces `[a, b, c, d, a, b, c, d]`.
     *
     * ```js
     * const a = tf.tensor1d([1, 2]);
     *
     * a.tile([2]).print();    // or a.tile([2])
     * ```
     *
     * ```js
     * const a = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * a.tile([1, 2]).print();  // or a.tile([1, 2])
     * ```
     * @param x The tensor to tile.
     * @param reps Determines the number of replications per dimension.
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function tile_(x, reps) {
        const $x = convertToTensor(x, 'x', 'tile', 'string_or_numeric');
        assert($x.rank === reps.length, () => `Error in transpose: rank of input ${$x.rank} ` +
            `must match length of reps ${reps}.`);
        const inputs = { x: $x };
        const attrs = { reps };
        return ENGINE.runKernel(Tile, inputs, attrs);
    }
    const tile = op({ tile_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const fusedBatchNormGradConfig = {
        kernelName: FusedBatchNorm,
        inputsToSave: ['x', 'mean', 'variance', 'scale'],
        gradFunc: (dy, saved, attrs) => {
            const { varianceEpsilon } = attrs;
            const [x, mean, variance, scale] = saved;
            const scaleValue = scale == null ? scalar(1) : scale;
            const reductionAxes = getReductionAxes(mean.shape, x.shape);
            const tileShape = [];
            if (mean.rank === 1) {
                for (let i = 0; i < x.shape.length - 1; ++i) {
                    tileShape.push(x.shape[i]);
                }
                tileShape.push(1);
            }
            const xMinusMean = sub(x, mean);
            const dyTimesScaleValue = mul(dy, scaleValue);
            const oneOverSqrtVariance = rsqrt(add$1(variance, scalar(varianceEpsilon)));
            const minusHalfRCube = mul(mul(mul(oneOverSqrtVariance, oneOverSqrtVariance), oneOverSqrtVariance), scalar(-0.5));
            const derX = () => {
                if (mean.rank === 1) {
                    return reshape$1(mul(mul(dy, tile(reshape$1(oneOverSqrtVariance, [1, 1, 1, mean.shape[0]]), tileShape)), scaleValue), x.shape);
                }
                else {
                    return reshape$1(mul(mul(dy, oneOverSqrtVariance), scaleValue), x.shape);
                }
            };
            const derMean = () => {
                let meanDer = mul(mul(oneOverSqrtVariance, scalar(-1)), dyTimesScaleValue);
                if (mean.rank === 1) {
                    meanDer = sum(meanDer, reductionAxes);
                }
                return reshape$1(meanDer, mean.shape);
            };
            const derVariance = () => {
                let varianceDer = mul(mul(minusHalfRCube, xMinusMean), dyTimesScaleValue);
                if (mean.rank === 1) {
                    varianceDer = sum(varianceDer, reductionAxes);
                }
                return reshape$1(varianceDer, mean.shape);
            };
            const derScale = () => {
                const xMinusMean2TimesRsqrt = mul(xMinusMean, oneOverSqrtVariance);
                let scaleDer = mul(dy, xMinusMean2TimesRsqrt);
                if (mean.rank === 1) {
                    scaleDer = sum(scaleDer, reductionAxes);
                }
                return reshape$1(scaleDer, mean.shape);
            };
            const derOffset = () => {
                let offsetDer = dy;
                if (mean.rank === 1) {
                    offsetDer = sum(offsetDer, reductionAxes);
                }
                return reshape$1(offsetDer, mean.shape);
            };
            return {
                x: derX,
                mean: derMean,
                variance: derVariance,
                scale: derScale,
                offset: derOffset
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the sum along segments of a `tf.Tensor`.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     * const segmentIds = tf.tensor1d([1, 2, 0, 1], 'int32');
     * const numSegments = 3;
     *
     * x.unsortedSegmentSum(segmentIds, numSegments).print()
     * //or tf.unsortedSegmentSum(x, segmentIds, numSegments)
     * ```
     * @param x The `tf.Tensor` that will be summed along its segments.
     * @param segmentIds A `tf.Tensor1D` whose rank is equal to the rank of `x`'s
     * dimension along the `axis`.  Maps each element of `x` to a segment.
     * @param numSegments The number of distinct `segmentIds`.
     *
     * @doc {heading: 'Operations', subheading: 'Segment'}
     */
    function unsortedSegmentSum_(x, segmentIds, numSegments) {
        const $x = convertToTensor(x, 'x', 'unsortedSegmentSum');
        const $segmentIds = convertToTensor(segmentIds, 'segmentIds', 'unsortedSegmentSum', 'int32');
        assert(isInt(numSegments), () => 'numSegments must be of dtype int');
        const inputs = { x: $x, segmentIds: $segmentIds };
        const attrs = { numSegments };
        return ENGINE.runKernel(UnsortedSegmentSum, inputs, attrs);
    }
    const unsortedSegmentSum = op({ unsortedSegmentSum_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const gatherGradConfig = {
        kernelName: GatherV2,
        inputsToSave: ['x', 'indices'],
        gradFunc: (dy, saved, attrs) => {
            const [x, indices] = saved;
            const { axis } = attrs;
            const parsedAxis = parseAxisParam(axis, x.shape)[0];
            const derX = () => {
                const paramsShape = x.shape;
                const indicesSize = indices.size;
                const outerShape = paramsShape.slice(0, parsedAxis);
                const outerDims = outerShape.length;
                const innerShape = paramsShape.slice(axis, paramsShape.length).slice(1);
                const innerDims = innerShape.length;
                const outerAxesIndices = arrayRange(0, outerDims);
                const innerAxesIndices = arrayRange(outerDims + 1, outerDims + 1 + innerDims);
                const valuesShape = arrayConcat([outerShape, [indicesSize], innerShape]);
                const values = reshape$1(dy, valuesShape);
                const reshapedIndices = reshape$1(indices, [indicesSize]);
                const transposeDims = arrayConcat([[outerDims], outerAxesIndices, innerAxesIndices]);
                const valuesTranspose = transpose(values, transposeDims);
                let paramsGrad = unsortedSegmentSum(valuesTranspose, reshapedIndices, x.shape[parsedAxis]);
                const invertTransposeDims = getUndoAxesPermutation(transposeDims);
                paramsGrad = transpose(paramsGrad, invertTransposeDims);
                return paramsGrad;
            };
            return { x: derX, indices: () => indices };
        }
    };
    function arrayRange(start, stop) {
        const result = [];
        for (let i = start; i < stop; ++i) {
            result.push(i);
        }
        return result;
    }
    function arrayConcat(arrays) {
        const result = [];
        for (let i = 0; i < arrays.length; ++i) {
            for (let j = 0; j < arrays[i].length; ++j) {
                result.push(arrays[i][j]);
            }
        }
        return result;
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const greaterEqualGradConfig = {
        kernelName: GreaterEqual,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            return { a: () => zerosLike(a), b: () => zerosLike(b) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const identityGradConfig = {
        kernelName: Identity,
        gradFunc: (dy) => {
            return { x: () => cast(dy, 'float32') };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const isFiniteGradConfig = {
        kernelName: IsFinite,
        gradFunc: (dy) => {
            // TODO(nsthorat): Let gradients be null for cases where we want to stop
            // backpropgation.
            return { x: () => zerosLike(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const isInfGradConfig = {
        kernelName: IsInf,
        gradFunc: (dy) => {
            // TODO(nsthorat): Let gradients be null for cases where we want to stop
            // backpropgation.
            return { x: () => zerosLike(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const isNanGradConfig = {
        kernelName: IsNan,
        gradFunc: (dy) => {
            // TODO(nsthorat): Let gradients be null for cases where we want to stop
            // backpropgation.
            return { x: () => zerosLike(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of (a > b) element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([2, 2, 2]);
     *
     * a.greater(b).print();
     * ```
     *
     * @param a The first input tensor.
     * @param b The second input tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function greater_(a, b) {
        let $a = convertToTensor(a, 'a', 'greater', 'string_or_numeric');
        let $b = convertToTensor(b, 'b', 'greater', 'string_or_numeric');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Greater, inputs);
    }
    const greater$1 = op({ greater_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const leakyReluGradConfig = {
        kernelName: LeakyRelu,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            const [x] = saved;
            const { alpha } = attrs;
            const mask = greater$1(x, 0);
            // Returns `gradients * (features > 0) + alpha * gradients * (features <=
            // 0)`.
            return { x: () => where(mask, dy, mul(dy, alpha)) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const log1pGradConfig = {
        kernelName: Log1p,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => div(dy, add$1(x, 1)) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const logGradConfig = {
        kernelName: Log,
        inputsToSave: ['x'],
        gradFunc: (dy, saved) => {
            const [x] = saved;
            return { x: () => div(dy, cast(x, 'float32')) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const logSoftmaxGradConfig = {
        kernelName: LogSoftmax$1,
        inputsToSave: [],
        outputsToSave: [true],
        gradFunc: (dy, saved, attrs) => {
            const [value] = saved;
            const { axis } = attrs;
            return {
                logits: () => {
                    const keepDims = true;
                    const softmax = exp(value);
                    return sub(dy, mul(sum(dy, axis, keepDims), softmax));
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function localResponseNormalizationBackprop_(x, y, dy, depthRadius = 5, bias = 1, alpha = 1, beta = 0.5) {
        const inputs = { x, y, dy };
        const attrs = { depthRadius, bias, alpha, beta };
        return ENGINE.runKernel(LRNGrad, inputs, attrs);
    }
    const localResponseNormalizationBackprop = op({ localResponseNormalizationBackprop_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const lrnGradConfig = {
        kernelName: LRN,
        inputsToSave: ['x'],
        outputsToSave: [true],
        gradFunc: (dy, saved, attrs) => {
            const [x, y] = saved;
            const { depthRadius, bias, alpha, beta } = attrs;
            return {
                x: () => localResponseNormalizationBackprop(x, y, dy, depthRadius, bias, alpha, beta)
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of (a == b) element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([2, 2, 2]);
     *
     * a.equal(b).print();
     * ```
     *
     * @param a The first input tensor.
     * @param b The second input tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function equal_(a, b) {
        let $a = convertToTensor(a, 'a', 'equal', 'string_or_numeric');
        let $b = convertToTensor(b, 'b', 'equal', 'string_or_numeric');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Equal, inputs);
    }
    const equal = op({ equal_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Gradient helper function for the min and max operations.
     */
    function gradForMinAndMax(dy, y, xOrig, origAxes) {
        if (y.rank < xOrig.rank) {
            y = reshape$1(y, expandShapeToKeepDim(y.shape, origAxes));
        }
        if (dy.rank < xOrig.rank) {
            dy = reshape$1(dy, expandShapeToKeepDim(dy.shape, origAxes));
        }
        return {
            x: () => {
                const dx = mul(dy, cast(equal(xOrig, y), dy.dtype));
                return dx;
            }
        };
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const maxGradConfig = {
        kernelName: Max,
        inputsToSave: ['x'],
        outputsToSave: [true],
        gradFunc: (dy, saved, attrs) => {
            const maxAttrs = attrs;
            const { reductionIndices } = maxAttrs;
            const x = saved[0];
            const y = saved[1];
            const origAxes = parseAxisParam(reductionIndices, x.shape);
            const maxGrad = gradForMinAndMax(dy, y, x, origAxes);
            return {
                x: () => {
                    return maxGrad['x']();
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of (a < b) element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([2, 2, 2]);
     *
     * a.less(b).print();
     * ```
     * @param a The first input tensor.
     * @param b The second input tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function less_(a, b) {
        let $a = convertToTensor(a, 'a', 'less', 'string_or_numeric');
        let $b = convertToTensor(b, 'b', 'less', 'string_or_numeric');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Less, inputs);
    }
    const less$1 = op({ less_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const maximumGradConfig = {
        kernelName: Maximum$1,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const derA = () => mul(dy, cast(greaterEqual(a, b), 'float32'));
            const derB = () => mul(dy, cast(less$1(a, b), 'float32'));
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the backprop of a 3d max pool.
     *
     * @param dy The dy error, of rank 5 of shape
     *     [batchSize, depth, height, width, channels].
     * assumed.
     * @param input The original input image, of rank 5 or rank 4 of shape
     *     [batchSize, depth, height, width, channels].
     * @param output The original output image, of rank 5 of shape
     *     [batchSize, outDepth, outHeight, outWidth, channels].
     * @param filterSize The filter size:
     *     `[filterDepth, filterHeight, filterWidth]`.
     *     `filterSize` is a single number,
     *     then `filterDepth == filterHeight == filterWidth`.
     * @param strides The strides of the pooling:
     *     `[strideDepth, strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param pad A string from: 'same', 'valid'. The type of padding algorithm
     *     used in the forward prop of the op.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function maxPool3dGrad_(dy, input, output, filterSize, strides, pad, dimRoundingMode) {
        const $dy = convertToTensor(dy, 'dy', 'maxPool3dGrad');
        const $input = convertToTensor(input, 'input', 'maxPool3dGrad');
        const $output = convertToTensor(output, 'output', 'maxPool3dGrad');
        let dy5D = $dy;
        let input5D = $input;
        let output5D = $output;
        let reshapedTo5D = false;
        if ($input.rank === 4) {
            reshapedTo5D = true;
            dy5D = reshape$1($dy, [1, $dy.shape[0], $dy.shape[1], $dy.shape[2], $dy.shape[3]]);
            input5D = reshape$1($input, [
                1, $input.shape[0], $input.shape[1], $input.shape[2], $input.shape[3]
            ]);
            output5D = reshape$1($output, [
                1, $output.shape[0], $output.shape[1], $output.shape[2], $output.shape[3]
            ]);
        }
        assert(dy5D.rank === 5, () => `Error in maxPool3dGrad: dy must be rank 5 but got rank ` +
            `${dy5D.rank}.`);
        assert(input5D.rank === 5, () => `Error in maxPool3dGrad: input must be rank 5 but got rank ` +
            `${input5D.rank}.`);
        assert(output5D.rank === 5, () => `Error in maxPool3dGrad: output must be rank 5 but got rank ` +
            `${output5D.rank}.`);
        checkPadOnDimRoundingMode('maxPool3dGrad', pad, dimRoundingMode);
        const inputs = { dy: dy5D, input: input5D, output: output5D };
        const attrs = { filterSize, strides, pad, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(MaxPool3DGrad, inputs, attrs);
        if (reshapedTo5D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
        }
        return res;
    }
    const maxPool3dGrad = op({ maxPool3dGrad_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const maxPool3DGradConfig = {
        kernelName: MaxPool3D,
        inputsToSave: ['x'],
        outputsToSave: [true],
        gradFunc: (dy, saved, attrs) => {
            const [x, y] = saved;
            const { filterSize, strides, pad, dimRoundingMode } = attrs;
            return {
                x: () => maxPool3dGrad(dy, x, y, filterSize, strides, pad, dimRoundingMode)
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the backprop of a 2D max pool.
     *
     * @param dy The dy error, of rank 4 or rank 3 of shape
     *     [batchSize, height, width, channels]. If rank 3, batch of 1 is
     * assumed.
     * @param input The original input image, of rank 4, of shape
     *     [batchSize, height, width, channels].
     * @param output The original output image, of rank 4, of shape
     *     [batchSize, outHeight, outWidth, channels].
     * @param filterSize The filter size: `[filterHeight, filterWidth]`. If
     *     `filterSize` is a single number, then `filterHeight == filterWidth`.
     * @param strides The strides of the pooling: `[strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param pad The type of padding algorithm used in the forward prop of the op.
     *     'same', 'valid', for more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function maxPoolGrad_(dy, input, output, filterSize, strides, pad, dimRoundingMode) {
        const $dy = convertToTensor(dy, 'dy', 'maxPoolGrad');
        const $input = convertToTensor(input, 'input', 'maxPoolGrad');
        const $output = convertToTensor(output, 'output', 'maxPoolGrad');
        assert($input.rank === $dy.rank, () => `Rank of input (${$input.rank}) does not match rank of dy ` +
            `(${$dy.rank})`);
        assert($dy.rank === 4, () => `Error in maxPoolGrad: dy must be rank 4 but got rank ` +
            `${$dy.rank}.`);
        assert($input.rank === 4, () => `Error in maxPoolGrad: input must be rank 4 but got rank ` +
            `${$input.rank}.`);
        checkPadOnDimRoundingMode('maxPoolGrad', pad, dimRoundingMode);
        const inputs = { dy: $dy, input: $input, output: $output };
        const attrs = { filterSize, strides, pad, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(MaxPoolGrad, inputs, attrs);
    }
    const maxPoolGrad = op({ maxPoolGrad_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const maxPoolGradConfig = {
        kernelName: MaxPool,
        inputsToSave: ['x'],
        outputsToSave: [true],
        gradFunc: (dy, saved, attrs) => {
            const [x, y] = saved;
            const { filterSize, strides, pad } = attrs;
            return {
                x: () => maxPoolGrad(dy, x, y, filterSize, strides, pad)
            };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with all elements set to 0.
     *
     * ```js
     * tf.zeros([2, 2]).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param dtype The type of an element in the resulting tensor. Can
     *     be 'float32', 'int32' or 'bool'. Defaults to 'float'.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function zeros$1(shape, dtype = 'float32') {
        if (dtype === 'complex64') {
            const real = zeros$1(shape, 'float32');
            const imag = zeros$1(shape, 'float32');
            return complex(real, imag);
        }
        const values = makeZerosTypedArray(sizeFromShape(shape), dtype);
        return ENGINE.makeTensor(values, shape, dtype);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with all elements set to 1.
     *
     * ```js
     * tf.ones([2, 2]).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param dtype The type of an element in the resulting tensor. Defaults to
     *     'float'.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function ones$1(shape, dtype = 'float32') {
        if (dtype === 'complex64') {
            const real = ones$1(shape, 'float32');
            const imag = zeros$1(shape, 'float32');
            return complex(real, imag);
        }
        const values = makeOnesTypedArray(sizeFromShape(shape), dtype);
        return ENGINE.makeTensor(values, shape, dtype);
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const meanGradConfig = {
        kernelName: Mean,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            const [x] = saved;
            const { axis } = attrs;
            const axes = parseAxisParam(axis, x.shape);
            const shapes = computeOutAndReduceShapes(x.shape, axes);
            const reduceShape = shapes[1];
            const reduceSize = sizeFromShape(reduceShape);
            const derX = () => {
                const expandedDyShape = x.shape.slice();
                axes.forEach(axis => {
                    expandedDyShape[axis] = 1;
                });
                const expandedDy = reshape$1(dy, expandedDyShape);
                const res = div(mul(expandedDy, ones$1(x.shape, 'float32')), reduceSize);
                return res;
            };
            return { x: derX };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const minGradConfig = {
        kernelName: Min,
        inputsToSave: ['x'],
        outputsToSave: [true],
        gradFunc: (dy, saved, attrs) => {
            const minAttrs = attrs;
            const { axis } = minAttrs;
            const [x, y] = saved;
            const origAxes = parseAxisParam(axis, x.shape);
            const minGrad = gradForMinAndMax(dy, y, x, origAxes);
            return {
                x: () => {
                    return minGrad['x']();
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const minimumGradConfig = {
        kernelName: Minimum$1,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const derA = () => mul(dy, cast(lessEqual(a, b), 'float32'));
            const derB = () => mul(dy, cast(greater$1(a, b), 'float32'));
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Extracts a slice from a `tf.Tensor` starting at coordinates `begin`
     * and is of size `size`.
     *
     * Also available are stricter rank-specific methods with the same signature
     * as this method that assert that `x` is of the given rank:
     *   - `tf.slice1d`
     *   - `tf.slice2d`
     *   - `tf.slice3d`
     *   - `tf.slice4d`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     *
     * x.slice([1], [2]).print();
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * x.slice([1, 0], [1, 2]).print();
     * ```
     * @param x The input `tf.Tensor` to slice from.
     * @param begin The coordinates to start the slice from. The length can be
     *     less than the rank of x - the rest of the axes will have implicit 0 as
     *     start. Can also be a single number, in which case it specifies the
     *     first axis.
     * @param size The size of the slice. The length can be less than the rank of
     *     x - the rest of the axes will have implicit -1. A value of -1 requests
     *     the rest of the dimensions in the axis. Can also be a single number,
     *     in which case it specifies the size of the first axis.
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function slice_(x, begin, size) {
        const $x = convertToTensor(x, 'x', 'slice', 'string_or_numeric');
        if ($x.rank === 0) {
            throw new Error('Slicing scalar is not possible');
        }
        const inputs = { x: $x };
        const attrs = { begin, size };
        return ENGINE.runKernel(Slice, inputs, attrs);
    }
    const slice = op({ slice_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const mirrorPadGradConfig = {
        kernelName: MirrorPad,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            // Pad introduces values around the original tensor, so the gradient
            // slices the original shape out of the gradient.
            const x = saved[0];
            const { paddings } = attrs;
            const begin = paddings.map(p => p[0]);
            return { x: () => slice(dy, begin, x.shape) };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes floor of input `tf.Tensor` element-wise: `floor(x)`.
     *
     * ```js
     * const x = tf.tensor1d([.6, 1.1, -3.3]);
     *
     * x.floor().print();  // or tf.floor(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function floor_(x) {
        const $x = convertToTensor(x, 'x', 'floor', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Floor, inputs);
    }
    const floor = op({ floor_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const modGradConfig = {
        kernelName: Mod,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const outShape = assertAndGetBroadcastShape(a.shape, b.shape);
            const derA = () => {
                const reduceAxes = getReductionAxes(a.shape, outShape);
                if (reduceAxes.length > 0) {
                    return reshape$1(sum(dy, reduceAxes), a.shape);
                }
                return dy;
            };
            const derB = () => {
                const res = mul(dy, neg(floor(div(a, b))));
                const reduceAxes = getReductionAxes(b.shape, outShape);
                if (reduceAxes.length > 0) {
                    return reshape$1(sum(res, reduceAxes), b.shape);
                }
                return res;
            };
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const multiplyGradConfig = {
        kernelName: Multiply$1,
        inputsToSave: ['a', 'b'],
        gradFunc: (dy, saved) => {
            const [a, b] = saved;
            const outShape = assertAndGetBroadcastShape(a.shape, b.shape);
            const derA = () => {
                const res = mul(dy, cast(b, 'float32'));
                const reduceAxes = getReductionAxes(a.shape, outShape);
                if (reduceAxes.length > 0) {
                    return reshape$1(sum(res, reduceAxes), a.shape);
                }
                return res;
            };
            const derB = () => {
                const res = mul(dy, cast(a, 'float32'));
                const reduceAxes = getReductionAxes(b.shape, outShape);
                if (reduceAxes.length > 0) {
                    return reshape$1(sum(res, reduceAxes), b.shape);
                }
                return res;
            };
            return { a: derA, b: derB };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const negGradConfig = {
        kernelName: Neg,
        gradFunc: (dy) => {
            return { x: () => neg(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const oneHotGradConfig = {
        kernelName: OneHot,
        inputsToSave: ['indices'],
        gradFunc: (dy, saved) => {
            const indices = saved[0];
            return { indices: () => zeros$1(indices.shape, 'float32') };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const onesLikeGradConfig = {
        kernelName: OnesLike,
        gradFunc: (dy) => {
            return { x: () => zerosLike(dy) };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Unstacks a `tf.Tensor` of rank-`R` into a list of rank-`(R-1)` `tf.Tensor`s.
     *
     * ```js
     * const a = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * tf.unstack(a).forEach(tensor => tensor.print());
     * ```
     *
     * @param x A tensor object.
     * @param axis The axis to unstack along. Defaults to 0 (the first dim).
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function unstack_(x, axis = 0) {
        const $x = convertToTensor(x, 'x', 'unstack', 'string_or_numeric');
        assert(axis >= -$x.shape.length && axis < $x.shape.length, () => `Axis = ${axis} is not in [-${$x.shape.length}, ${$x.shape.length})`);
        const inputs = { value: $x };
        const attrs = { axis };
        return ENGINE.runKernel(Unpack, inputs, attrs);
    }
    const unstack = op({ unstack_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const packGradConfig = {
        kernelName: Pack,
        saveAllInputs: true,
        gradFunc: (dy, saved, attrs) => {
            const { axis } = attrs;
            const derTensors = unstack(dy, axis);
            return derTensors.map(t => () => t);
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const padV2GradConfig = {
        kernelName: PadV2,
        inputsToSave: ['x'],
        gradFunc: (dy, saved, attrs) => {
            // Pad introduces values around the original tensor, so the gradient
            // slices the original shape out of the gradient.
            const x = saved[0];
            const { paddings } = attrs;
            const begin = paddings.map(p => p[0]);
            return { x: () => slice(dy, begin, x.shape) };
        }
    };

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes natural logarithm of the input `tf.Tensor` element-wise: `ln(x)`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, Math.E]);
     *
     * x.log().print();  // or tf.log(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function log_(x) {
        const $x = convertToTensor(x, 'x', 'log', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Log, inputs);
    }
    const log = op({ log_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the power of one `tf.Tensor` to another. Supports broadcasting.
     *
     * Given a `tf.Tensor` x and a `tf.Tensor` y, this operation computes x^y for
     * corresponding elements in x and y. The result's dtype will be the upcasted
     * type of the `base` and `exp` dtypes.
     *
     * ```js
     * const a = tf.tensor([[2, 3], [4, 5]])
     * const b = tf.tensor([[1, 2], [3, 0]]).toInt();
     *
     * a.pow(b).print();  // or tf.pow(a, b)
     * ```
     *
     * ```js
     * const a = tf.tensor([[1, 2], [3, 4]])
     * const b = tf.tensor(2).toInt();
     *
     * a.pow(b).print();  // or tf.pow(a, b)
     * ```
     * We also expose `powStrict` which has the same signature as this op and
     * asserts that `base` and `exp` are the same shape (does not broadcast).
     *
     * @param base The base `tf.Tensor` to pow element-wise.
     * @param exp The exponent `tf.Tensor` to pow element-wise.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function pow_(base, exp) {
        let $base = convertToTensor(base, 'base', 'pow');
        let $exp = convertToTensor(exp, 'exp', 'pow');
        [$base, $exp] = makeTypesMatch($base, $exp);
        const inputs = { a: $base, b: $exp };
        return ENGINE.runKernel(Pow, inputs);
    }
    const pow = op({ pow_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const powGradConfig = {
        kernelName: Pow,
        inputsToSave: ['a', 'b'],
        outputsToSave: [true],
        gradFunc: (dy, saved) => {
            const [a, b, y] = saved;
            const base = a;
            const exp = b;
            const outShape = assertAndGetBroadcastShape(base.shape, exp.shape);
            const derBase = () => {
                const expFloat = cast(exp, 'float32');
                let res = mul(dy, mul(expFloat, pow(base, sub(expFloat, scalar(1)))));
                const reduceAxes = getReductionAxes(base.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = sum(res, reduceAxes);
                }
                return reshape$1(res, base.shape);
            };
            const derExp = () => {
                const condition = greater$1(base, 0);
                const logBase = where(condition, log(base), zerosLike(base));
                let res = mul(dy, mul(y, logBase));
                const reduceAxes = getReductionAxes(exp.shape, outShape);
                if (reduceAxes.length > 0) {
                    res = sum(res, reduceAxes);
                }
                return reshape$1(res, exp.shape);
            };
            return { a: derBase, b: derExp };
        }
    };

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const preluGradConfig = {
        kernelName: Prelu,
        inputsToSave: ['x', 'alpha'],
        gradFunc: (dy, saved) => {
            const [x, alpha] = saved;
            const mask = greater$1(x, 0);
            return {
                x: () => where(mask, dy, mul(dy, alpha)),
                alpha: () => {
                    let res = where(mask, zerosLike(dy), mul(dy, x));
                    const reduceAxes = getReductionAxes(alpha.shape, dy.shape);
                    if (reduceAxes.length > 0) {
                        res = sum(res, reduceAxes);
                    }
                    return reshape$1(res, alpha.shape);
                }
            };
        }
    };

    /**
     * @license
     * Copyright 2017 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function isBrowser() {
        return (typeof window !== 'undefined' && window.document != null) ||
            //@ts-ignore
            (typeof WorkerGlobalScope !== 'undefined');
    }

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const ENV = env();
    /**
     * This file contains environment-related flag registrations.
     */
    /** Whether to enable debug mode. */
    ENV.registerFlag('DEBUG', () => false, debugValue => {
        if (debugValue) {
            console.warn('Debugging mode is ON. The output of every math call will ' +
                'be downloaded to CPU and checked for NaNs. ' +
                'This significantly impacts performance.');
        }
    });
    /** Whether we are in a browser (as versus, say, node.js) environment. */
    ENV.registerFlag('IS_BROWSER', () => isBrowser());
    /** Whether we are in a browser (as versus, say, node.js) environment. */
    ENV.registerFlag('IS_NODE', () => (typeof process !== 'undefined') &&
        (typeof process.versions !== 'undefined') &&
        (typeof process.versions.node !== 'undefined'));
    /** Whether this browser is Chrome. */
    ENV.registerFlag('IS_CHROME', () => typeof navigator !== 'undefined' && navigator != null &&
        navigator.userAgent != null && /Chrome/.test(navigator.userAgent) &&
        /Google Inc/.test(navigator.vendor));
    /**
     * True when the environment is "production" where we disable safety checks
     * to gain performance.
     */
    ENV.registerFlag('PROD', () => false);
    /**
     * Whether to do sanity checks when inferring a shape from user-provided
     * values, used when creating a new tensor.
     */
    ENV.registerFlag('TENSORLIKE_CHECK_SHAPE_CONSISTENCY', () => ENV.getBool('DEBUG'));
    /** Whether deprecation warnings are enabled. */
    ENV.registerFlag('DEPRECATION_WARNINGS_ENABLED', () => true);
    /** True if running unit tests. */
    ENV.registerFlag('IS_TEST', () => false);
    /** Whether to check computation result for errors. */
    ENV.registerFlag('CHECK_COMPUTATION_FOR_ERRORS', () => true);
    /** Whether the backend needs to wrap input to imageBitmap. */
    ENV.registerFlag('WRAP_TO_IMAGEBITMAP', () => false);
    /** Experimental flag, whether enter compile only phase. */
    ENV.registerFlag('ENGINE_COMPILE_ONLY', () => false);
    /** Whether to enable canvas2d willReadFrequently for GPU backends */
    ENV.registerFlag('CANVAS2D_WILL_READ_FREQUENTLY_FOR_GPU', () => false);
    /** Whether to use setTimeoutCustom */
    ENV.registerFlag('USE_SETTIMEOUTCUSTOM', () => false);

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with the provided values, shape and dtype.
     *
     * ```js
     * // Pass an array of values to create a vector.
     * tf.tensor([1, 2, 3, 4]).print();
     * ```
     *
     * ```js
     * // Pass a nested array of values to make a matrix or a higher
     * // dimensional tensor.
     * tf.tensor([[1, 2], [3, 4]]).print();
     * ```
     *
     * ```js
     * // Pass a flat array and specify a shape yourself.
     * tf.tensor([1, 2, 3, 4], [2, 2]).print();
     * ```
     *
     * ```js
     * // Pass a `WebGLData` object and specify a shape yourself.
     *
     * // This makes it possible for TF.js applications to avoid GPU / CPU sync.
     * // For example, if your application includes a preprocessing step on the GPU,
     * // you could upload the GPU output directly to TF.js, rather than first
     * // downloading the values.
     *
     * // Example for WebGL2:
     * const customCanvas = document.createElement('canvas');
     * const customBackend = new tf.MathBackendWebGL(customCanvas);
     * tf.registerBackend('custom-webgl', () => customBackend);
     * await tf.setBackend('custom-webgl');
     * const gl = customBackend.gpgpu.gl;
     * const texture = gl.createTexture();
     * const tex2d = gl.TEXTURE_2D;
     * const width = 2;
     * const height = 2;
     *
     * gl.bindTexture(tex2d, texture);
     * gl.texParameteri(tex2d, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
     * gl.texParameteri(tex2d, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
     * gl.texParameteri(tex2d, gl.TEXTURE_MIN_FILTER, gl.NEAREST);
     * gl.texParameteri(tex2d, gl.TEXTURE_MAG_FILTER, gl.NEAREST);
     * gl.texImage2D(
     *   tex2d, 0, gl.RGBA32F, // internalFormat
     *   width, height, 0,
     *   gl.RGBA, // textureFormat
     *   gl.FLOAT, // textureType
     *   new Float32Array([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15])
     * );
     *
     * // Currently, the `texture` has 4 pixels:
     * // Pixel0 is {R:0, G:1, B:2, A:3}
     * // Pixel1 is {R:4, G:5, B:6, A:7}
     * // Pixel2 is {R:8, G:9, B:10, A:11}
     * // Pixel3 is {R:12, G:13, B:14, A:15}
     *
     * const logicalShape = [height * width * 2];
     * const a = tf.tensor({texture, height, width, channels: 'BR'}, logicalShape);
     * // Tensor value will be [2, 0, 6, 4, 10, 8, 14, 12], since [2, 0] is the
     * // values of 'B' and 'R' channels of Pixel0, [6, 4] is the values of 'B' and
     * 'R'
     * // channels of Pixel1...
     *
     * // For postprocessing on the GPU, it's possible to retrieve the texture
     * // backing any tensor by calling the tensor's `dataToGPU` method like
     * // so:
     *
     * const tex = a.dataToGPU();
     * ```
     * @param values The values of the tensor. Can be nested array of numbers,
     *     or a flat array, or a `TypedArray`, or a `WebGLData` object. If the
     * values are strings, they will be encoded as utf-8 and kept as `Uint8Array[]`.
     * If the values is a `WebGLData` object, the dtype could only be 'float32' or
     * 'int32' and the object has to have: 1. texture, a `WebGLTexture`, the texture
     * must share the same `WebGLRenderingContext` with TFJS's WebGL backend (you
     * could create a custom WebGL backend from your texture's canvas) and the
     * internal texture format for the input texture must be floating point or
     * normalized integer; 2. height, the height of the texture; 3. width, the width
     * of the texture; 4. channels, a non-empty subset of 'RGBA', indicating the
     * values of which channels will be passed to the tensor, such as 'R' or 'BR'
     * (The order of the channels affect the order of tensor values. ). (If the
     * values passed from texture is less than the tensor size, zeros will be padded
     * at the rear.)
     * @param shape The shape of the tensor. Optional. If not provided,
     *   it is inferred from `values`.
     * @param dtype The data type.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function tensor(values, shape, dtype) {
        const inferredShape = inferShape(values, dtype);
        return makeTensor(values, shape, inferredShape, dtype);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // Use Buffer on Node.js instead of Blob/atob/btoa
    const useNodeBuffer = typeof Buffer !== 'undefined' &&
        (typeof Blob === 'undefined' || typeof atob === 'undefined' ||
            typeof btoa === 'undefined');
    /**
     * Calculate the byte length of a JavaScript string.
     *
     * Note that a JavaScript string can contain wide characters, therefore the
     * length of the string is not necessarily equal to the byte length.
     *
     * @param str Input string.
     * @returns Byte length.
     */
    function stringByteLength(str) {
        if (useNodeBuffer) {
            return Buffer.byteLength(str);
        }
        return new Blob([str]).size;
    }
    /**
     * Encode an ArrayBuffer as a base64 encoded string.
     *
     * @param buffer `ArrayBuffer` to be converted.
     * @returns A string that base64-encodes `buffer`.
     */
    function arrayBufferToBase64String(buffer) {
        if (useNodeBuffer) {
            return Buffer.from(buffer).toString('base64');
        }
        const buf = new Uint8Array(buffer);
        let s = '';
        for (let i = 0, l = buf.length; i < l; i++) {
            s += String.fromCharCode(buf[i]);
        }
        return btoa(s);
    }
    /**
     * Decode a base64 string as an ArrayBuffer.
     *
     * @param str Base64 string.
     * @returns Decoded `ArrayBuffer`.
     */
    function base64StringToArrayBuffer(str) {
        if (useNodeBuffer) {
            const buf = Buffer.from(str, 'base64');
            return buf.buffer.slice(buf.byteOffset, buf.byteOffset + buf.byteLength);
        }
        const s = atob(str);
        const buffer = new Uint8Array(s.length);
        for (let i = 0; i < s.length; ++i) {
            buffer.set([s.charCodeAt(i)], i);
        }
        return buffer.buffer;
    }
    /**
     * Concatenate a number of ArrayBuffers into one.
     *
     * @param buffers A number of array buffers to concatenate.
     * @returns Result of concatenating `buffers` in order.
     */
    function concatenateArrayBuffers(buffers) {
        if (buffers.length === 1) {
            return buffers[0];
        }
        let totalByteLength = 0;
        buffers.forEach((buffer) => {
            totalByteLength += buffer.byteLength;
        });
        const temp = new Uint8Array(totalByteLength);
        let offset = 0;
        buffers.forEach((buffer) => {
            temp.set(new Uint8Array(buffer), offset);
            offset += buffer.byteLength;
        });
        return temp.buffer;
    }
    /**
     * Create `ModelJSON` from `ModelArtifacts`.
     *
     * @param artifacts Model artifacts, describing the model and its weights.
     * @param manifest Weight manifest, describing where the weights of the
     *     `ModelArtifacts` are stored, and some metadata about them.
     * @returns Object representing the `model.json` file describing the model
     *     artifacts and weights
     */
    function getModelJSONForModelArtifacts(artifacts, manifest) {
        const result = {
            modelTopology: artifacts.modelTopology,
            format: artifacts.format,
            generatedBy: artifacts.generatedBy,
            convertedBy: artifacts.convertedBy,
            weightsManifest: manifest
        };
        if (artifacts.signature != null) {
            result.signature = artifacts.signature;
        }
        if (artifacts.userDefinedMetadata != null) {
            result.userDefinedMetadata = artifacts.userDefinedMetadata;
        }
        if (artifacts.modelInitializer != null) {
            result.modelInitializer = artifacts.modelInitializer;
        }
        if (artifacts.initializerSignature != null) {
            result.initializerSignature = artifacts.initializerSignature;
        }
        if (artifacts.trainingConfig != null) {
            result.trainingConfig = artifacts.trainingConfig;
        }
        return result;
    }
    /**
     * Create `ModelArtifacts` from a JSON file and weights.
     *
     * @param modelJSON Object containing the parsed JSON of `model.json`
     * @param weightSpecs The list of WeightsManifestEntry for the model. Must be
     *     passed if the modelJSON has a weightsManifest.
     * @param weightData An ArrayBuffer of weight data for the model corresponding
     *     to the weights in weightSpecs. Must be passed if the modelJSON has a
     *     weightsManifest.
     * @returns A Promise of the `ModelArtifacts`, as described by the JSON file.
     */
    function getModelArtifactsForJSONSync(modelJSON, weightSpecs, weightData) {
        const modelArtifacts = {
            modelTopology: modelJSON.modelTopology,
            format: modelJSON.format,
            generatedBy: modelJSON.generatedBy,
            convertedBy: modelJSON.convertedBy
        };
        if (modelJSON.trainingConfig != null) {
            modelArtifacts.trainingConfig = modelJSON.trainingConfig;
        }
        if (modelJSON.weightsManifest != null) {
            if (!weightSpecs) {
                throw new Error('modelJSON has weightsManifest but weightSpecs is null');
            }
            if (!weightData) {
                throw new Error('modelJSON has weightsManifest but weightData is null');
            }
            modelArtifacts.weightSpecs = weightSpecs;
            modelArtifacts.weightData = weightData;
        }
        if (modelJSON.signature != null) {
            modelArtifacts.signature = modelJSON.signature;
        }
        if (modelJSON.userDefinedMetadata != null) {
            modelArtifacts.userDefinedMetadata = modelJSON.userDefinedMetadata;
        }
        if (modelJSON.modelInitializer != null) {
            modelArtifacts.modelInitializer = modelJSON.modelInitializer;
        }
        if (modelJSON.initializerSignature != null) {
            modelArtifacts.initializerSignature = modelJSON.initializerSignature;
        }
        return modelArtifacts;
    }
    /**
     * Create `ModelArtifacts` from a JSON file.
     *
     * @param modelJSON Object containing the parsed JSON of `model.json`
     * @param loadWeights Function that takes the JSON file's weights manifest,
     *     reads weights from the listed path(s), and returns a Promise of the
     *     weight manifest entries along with the weights data.
     * @returns A Promise of the `ModelArtifacts`, as described by the JSON file.
     */
    async function getModelArtifactsForJSON(modelJSON, loadWeights) {
        let weightSpecs;
        let weightData;
        if (modelJSON.weightsManifest != null) {
            [weightSpecs, weightData] = await loadWeights(modelJSON.weightsManifest);
        }
        return getModelArtifactsForJSONSync(modelJSON, weightSpecs, weightData);
    }
    /**
     * Populate ModelArtifactsInfo fields for a model with JSON topology.
     * @param modelArtifacts
     * @returns A ModelArtifactsInfo object.
     */
    function getModelArtifactsInfoForJSON(modelArtifacts) {
        if (modelArtifacts.modelTopology instanceof ArrayBuffer) {
            throw new Error('Expected JSON model topology, received ArrayBuffer.');
        }
        return {
            dateSaved: new Date(),
            modelTopologyType: 'JSON',
            modelTopologyBytes: modelArtifacts.modelTopology == null ?
                0 :
                stringByteLength(JSON.stringify(modelArtifacts.modelTopology)),
            weightSpecsBytes: modelArtifacts.weightSpecs == null ?
                0 :
                stringByteLength(JSON.stringify(modelArtifacts.weightSpecs)),
            weightDataBytes: modelArtifacts.weightData == null ?
                0 :
                modelArtifacts.weightData.byteLength,
        };
    }
    /**
     * Concatenate the weights stored in a WeightsManifestConfig into a list of
     * WeightsManifestEntry
     *
     * @param weightsManifest The WeightsManifestConfig to extract weights from.
     * @returns A list of WeightsManifestEntry of the weights in the weightsManifest
     */
    function getWeightSpecs(weightsManifest) {
        const weightSpecs = [];
        for (const entry of weightsManifest) {
            weightSpecs.push(...entry.weights);
        }
        return weightSpecs;
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    class IORouterRegistry {
        constructor() {
            this.saveRouters = [];
            this.loadRouters = [];
        }
        static getInstance() {
            if (IORouterRegistry.instance == null) {
                IORouterRegistry.instance = new IORouterRegistry();
            }
            return IORouterRegistry.instance;
        }
        /**
         * Register a save-handler router.
         *
         * @param saveRouter A function that maps a URL-like string onto an instance
         * of `IOHandler` with the `save` method defined or `null`.
         */
        static registerSaveRouter(saveRouter) {
            IORouterRegistry.getInstance().saveRouters.push(saveRouter);
        }
        /**
         * Register a load-handler router.
         *
         * @param loadRouter A function that maps a URL-like string onto an instance
         * of `IOHandler` with the `load` method defined or `null`.
         */
        static registerLoadRouter(loadRouter) {
            IORouterRegistry.getInstance().loadRouters.push(loadRouter);
        }
        /**
         * Look up IOHandler for saving, given a URL-like string.
         *
         * @param url
         * @returns If only one match is found, an instance of IOHandler with the
         * `save` method defined. If no match is found, `null`.
         * @throws Error, if more than one match is found.
         */
        static getSaveHandlers(url) {
            return IORouterRegistry.getHandlers(url, 'save');
        }
        /**
         * Look up IOHandler for loading, given a URL-like string.
         *
         * @param url
         * @param loadOptions Optional, custom load options.
         * @returns All valid handlers for `url`, given the currently registered
         *   handler routers.
         */
        static getLoadHandlers(url, loadOptions) {
            return IORouterRegistry.getHandlers(url, 'load', loadOptions);
        }
        static getHandlers(url, handlerType, loadOptions) {
            const validHandlers = [];
            const routers = handlerType === 'load' ?
                IORouterRegistry.getInstance().loadRouters :
                IORouterRegistry.getInstance().saveRouters;
            routers.forEach(router => {
                const handler = router(url, loadOptions);
                if (handler !== null) {
                    validHandlers.push(handler);
                }
            });
            return validHandlers;
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const DATABASE_NAME = 'tensorflowjs';
    const DATABASE_VERSION = 1;
    // Model data and ModelArtifactsInfo (metadata) are stored in two separate
    // stores for efficient access of the list of stored models and their metadata.
    // 1. The object store for model data: topology, weights and weight manifests.
    const MODEL_STORE_NAME = 'models_store';
    // 2. The object store for ModelArtifactsInfo, including meta-information such
    //    as the type of topology (JSON vs binary), byte size of the topology, byte
    //    size of the weights, etc.
    const INFO_STORE_NAME = 'model_info_store';
    function getIndexedDBFactory() {
        if (!env().getBool('IS_BROWSER')) {
            // TODO(cais): Add more info about what IOHandler subtypes are available.
            //   Maybe point to a doc page on the web and/or automatically determine
            //   the available IOHandlers and print them in the error message.
            throw new Error('Failed to obtain IndexedDB factory because the current environment' +
                'is not a web browser.');
        }
        // tslint:disable-next-line:no-any
        const theWindow = typeof window === 'undefined' ? self : window;
        const factory = theWindow.indexedDB || theWindow.mozIndexedDB ||
            theWindow.webkitIndexedDB || theWindow.msIndexedDB ||
            theWindow.shimIndexedDB;
        if (factory == null) {
            throw new Error('The current browser does not appear to support IndexedDB.');
        }
        return factory;
    }
    function setUpDatabase(openRequest) {
        const db = openRequest.result;
        db.createObjectStore(MODEL_STORE_NAME, { keyPath: 'modelPath' });
        db.createObjectStore(INFO_STORE_NAME, { keyPath: 'modelPath' });
    }
    /**
     * IOHandler subclass: Browser IndexedDB.
     *
     * See the doc string of `browserIndexedDB` for more details.
     */
    class BrowserIndexedDB {
        constructor(modelPath) {
            this.indexedDB = getIndexedDBFactory();
            if (modelPath == null || !modelPath) {
                throw new Error('For IndexedDB, modelPath must not be null, undefined or empty.');
            }
            this.modelPath = modelPath;
        }
        async save(modelArtifacts) {
            // TODO(cais): Support saving GraphDef models.
            if (modelArtifacts.modelTopology instanceof ArrayBuffer) {
                throw new Error('BrowserLocalStorage.save() does not support saving model topology ' +
                    'in binary formats yet.');
            }
            return this.databaseAction(this.modelPath, modelArtifacts);
        }
        async load() {
            return this.databaseAction(this.modelPath);
        }
        /**
         * Perform database action to put model artifacts into or read model artifacts
         * from IndexedDB object store.
         *
         * Whether the action is put or get depends on whether `modelArtifacts` is
         * specified. If it is specified, the action will be put; otherwise the action
         * will be get.
         *
         * @param modelPath A unique string path for the model.
         * @param modelArtifacts If specified, it will be the model artifacts to be
         *   stored in IndexedDB.
         * @returns A `Promise` of `SaveResult`, if the action is put, or a `Promise`
         *   of `ModelArtifacts`, if the action is get.
         */
        databaseAction(modelPath, modelArtifacts) {
            return new Promise((resolve, reject) => {
                const openRequest = this.indexedDB.open(DATABASE_NAME, DATABASE_VERSION);
                openRequest.onupgradeneeded = () => setUpDatabase(openRequest);
                openRequest.onsuccess = () => {
                    const db = openRequest.result;
                    if (modelArtifacts == null) {
                        // Read model out from object store.
                        const modelTx = db.transaction(MODEL_STORE_NAME, 'readonly');
                        const modelStore = modelTx.objectStore(MODEL_STORE_NAME);
                        const getRequest = modelStore.get(this.modelPath);
                        getRequest.onsuccess = () => {
                            if (getRequest.result == null) {
                                db.close();
                                return reject(new Error(`Cannot find model with path '${this.modelPath}' ` +
                                    `in IndexedDB.`));
                            }
                            else {
                                resolve(getRequest.result.modelArtifacts);
                            }
                        };
                        getRequest.onerror = error => {
                            db.close();
                            return reject(getRequest.error);
                        };
                        modelTx.oncomplete = () => db.close();
                    }
                    else {
                        // Put model into object store.
                        const modelArtifactsInfo = getModelArtifactsInfoForJSON(modelArtifacts);
                        // First, put ModelArtifactsInfo into info store.
                        const infoTx = db.transaction(INFO_STORE_NAME, 'readwrite');
                        let infoStore = infoTx.objectStore(INFO_STORE_NAME);
                        const putInfoRequest = infoStore.put({ modelPath: this.modelPath, modelArtifactsInfo });
                        let modelTx;
                        putInfoRequest.onsuccess = () => {
                            // Second, put model data into model store.
                            modelTx = db.transaction(MODEL_STORE_NAME, 'readwrite');
                            const modelStore = modelTx.objectStore(MODEL_STORE_NAME);
                            const putModelRequest = modelStore.put({
                                modelPath: this.modelPath,
                                modelArtifacts,
                                modelArtifactsInfo
                            });
                            putModelRequest.onsuccess = () => resolve({ modelArtifactsInfo });
                            putModelRequest.onerror = error => {
                                // If the put-model request fails, roll back the info entry as
                                // well.
                                infoStore = infoTx.objectStore(INFO_STORE_NAME);
                                const deleteInfoRequest = infoStore.delete(this.modelPath);
                                deleteInfoRequest.onsuccess = () => {
                                    db.close();
                                    return reject(putModelRequest.error);
                                };
                                deleteInfoRequest.onerror = error => {
                                    db.close();
                                    return reject(putModelRequest.error);
                                };
                            };
                        };
                        putInfoRequest.onerror = error => {
                            db.close();
                            return reject(putInfoRequest.error);
                        };
                        infoTx.oncomplete = () => {
                            if (modelTx == null) {
                                db.close();
                            }
                            else {
                                modelTx.oncomplete = () => db.close();
                            }
                        };
                    }
                };
                openRequest.onerror = error => reject(openRequest.error);
            });
        }
    }
    BrowserIndexedDB.URL_SCHEME = 'indexeddb://';
    const indexedDBRouter = (url) => {
        if (!env().getBool('IS_BROWSER')) {
            return null;
        }
        else {
            if (!Array.isArray(url) && url.startsWith(BrowserIndexedDB.URL_SCHEME)) {
                return browserIndexedDB(url.slice(BrowserIndexedDB.URL_SCHEME.length));
            }
            else {
                return null;
            }
        }
    };
    IORouterRegistry.registerSaveRouter(indexedDBRouter);
    IORouterRegistry.registerLoadRouter(indexedDBRouter);
    /**
     * Creates a browser IndexedDB IOHandler for saving and loading models.
     *
     * ```js
     * const model = tf.sequential();
     * model.add(
     *     tf.layers.dense({units: 1, inputShape: [100], activation: 'sigmoid'}));
     *
     * const saveResult = await model.save('indexeddb://MyModel'));
     * console.log(saveResult);
     * ```
     *
     * @param modelPath A unique identifier for the model to be saved. Must be a
     *   non-empty string.
     * @returns An instance of `BrowserIndexedDB` (sublcass of `IOHandler`),
     *   which can be used with, e.g., `tf.Model.save`.
     */
    function browserIndexedDB(modelPath) {
        return new BrowserIndexedDB(modelPath);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const PATH_SEPARATOR = '/';
    const PATH_PREFIX = 'tensorflowjs_models';
    const INFO_SUFFIX = 'info';
    const MODEL_TOPOLOGY_SUFFIX = 'model_topology';
    const WEIGHT_SPECS_SUFFIX = 'weight_specs';
    const WEIGHT_DATA_SUFFIX = 'weight_data';
    const MODEL_METADATA_SUFFIX = 'model_metadata';
    function getModelKeys(path) {
        return {
            info: [PATH_PREFIX, path, INFO_SUFFIX].join(PATH_SEPARATOR),
            topology: [PATH_PREFIX, path, MODEL_TOPOLOGY_SUFFIX].join(PATH_SEPARATOR),
            weightSpecs: [PATH_PREFIX, path, WEIGHT_SPECS_SUFFIX].join(PATH_SEPARATOR),
            weightData: [PATH_PREFIX, path, WEIGHT_DATA_SUFFIX].join(PATH_SEPARATOR),
            modelMetadata: [PATH_PREFIX, path, MODEL_METADATA_SUFFIX].join(PATH_SEPARATOR)
        };
    }
    function removeItems(keys) {
        for (const key of Object.values(keys)) {
            window.localStorage.removeItem(key);
        }
    }
    /**
     * IOHandler subclass: Browser Local Storage.
     *
     * See the doc string to `browserLocalStorage` for more details.
     */
    class BrowserLocalStorage {
        constructor(modelPath) {
            if (!env().getBool('IS_BROWSER') || typeof window === 'undefined' ||
                typeof window.localStorage === 'undefined') {
                // TODO(cais): Add more info about what IOHandler subtypes are
                // available.
                //   Maybe point to a doc page on the web and/or automatically determine
                //   the available IOHandlers and print them in the error message.
                throw new Error('The current environment does not support local storage.');
            }
            this.LS = window.localStorage;
            if (modelPath == null || !modelPath) {
                throw new Error('For local storage, modelPath must not be null, undefined or empty.');
            }
            this.modelPath = modelPath;
            this.keys = getModelKeys(this.modelPath);
        }
        /**
         * Save model artifacts to browser local storage.
         *
         * See the documentation to `browserLocalStorage` for details on the saved
         * artifacts.
         *
         * @param modelArtifacts The model artifacts to be stored.
         * @returns An instance of SaveResult.
         */
        async save(modelArtifacts) {
            if (modelArtifacts.modelTopology instanceof ArrayBuffer) {
                throw new Error('BrowserLocalStorage.save() does not support saving model topology ' +
                    'in binary formats yet.');
            }
            else {
                const topology = JSON.stringify(modelArtifacts.modelTopology);
                const weightSpecs = JSON.stringify(modelArtifacts.weightSpecs);
                const modelArtifactsInfo = getModelArtifactsInfoForJSON(modelArtifacts);
                try {
                    this.LS.setItem(this.keys.info, JSON.stringify(modelArtifactsInfo));
                    this.LS.setItem(this.keys.topology, topology);
                    this.LS.setItem(this.keys.weightSpecs, weightSpecs);
                    this.LS.setItem(this.keys.weightData, arrayBufferToBase64String(modelArtifacts.weightData));
                    // Note that JSON.stringify doesn't write out keys that have undefined
                    // values, so for some keys, we set undefined instead of a null-ish
                    // value.
                    const metadata = {
                        format: modelArtifacts.format,
                        generatedBy: modelArtifacts.generatedBy,
                        convertedBy: modelArtifacts.convertedBy,
                        signature: modelArtifacts.signature != null ?
                            modelArtifacts.signature :
                            undefined,
                        userDefinedMetadata: modelArtifacts.userDefinedMetadata != null ?
                            modelArtifacts.userDefinedMetadata :
                            undefined,
                        modelInitializer: modelArtifacts.modelInitializer != null ?
                            modelArtifacts.modelInitializer :
                            undefined,
                        initializerSignature: modelArtifacts.initializerSignature != null ?
                            modelArtifacts.initializerSignature :
                            undefined,
                        trainingConfig: modelArtifacts.trainingConfig != null ?
                            modelArtifacts.trainingConfig :
                            undefined
                    };
                    this.LS.setItem(this.keys.modelMetadata, JSON.stringify(metadata));
                    return { modelArtifactsInfo };
                }
                catch (err) {
                    // If saving failed, clean up all items saved so far.
                    removeItems(this.keys);
                    throw new Error(`Failed to save model '${this.modelPath}' to local storage: ` +
                        `size quota being exceeded is a possible cause of this failure: ` +
                        `modelTopologyBytes=${modelArtifactsInfo.modelTopologyBytes}, ` +
                        `weightSpecsBytes=${modelArtifactsInfo.weightSpecsBytes}, ` +
                        `weightDataBytes=${modelArtifactsInfo.weightDataBytes}.`);
                }
            }
        }
        /**
         * Load a model from local storage.
         *
         * See the documentation to `browserLocalStorage` for details on the saved
         * artifacts.
         *
         * @returns The loaded model (if loading succeeds).
         */
        async load() {
            const info = JSON.parse(this.LS.getItem(this.keys.info));
            if (info == null) {
                throw new Error(`In local storage, there is no model with name '${this.modelPath}'`);
            }
            if (info.modelTopologyType !== 'JSON') {
                throw new Error('BrowserLocalStorage does not support loading non-JSON model ' +
                    'topology yet.');
            }
            const out = {};
            // Load topology.
            const topology = JSON.parse(this.LS.getItem(this.keys.topology));
            if (topology == null) {
                throw new Error(`In local storage, the topology of model '${this.modelPath}' ` +
                    `is missing.`);
            }
            out.modelTopology = topology;
            // Load weight specs.
            const weightSpecs = JSON.parse(this.LS.getItem(this.keys.weightSpecs));
            if (weightSpecs == null) {
                throw new Error(`In local storage, the weight specs of model '${this.modelPath}' ` +
                    `are missing.`);
            }
            out.weightSpecs = weightSpecs;
            // Load meta-data fields.
            const metadataString = this.LS.getItem(this.keys.modelMetadata);
            if (metadataString != null) {
                const metadata = JSON.parse(metadataString);
                out.format = metadata.format;
                out.generatedBy = metadata.generatedBy;
                out.convertedBy = metadata.convertedBy;
                if (metadata.signature != null) {
                    out.signature = metadata.signature;
                }
                if (metadata.userDefinedMetadata != null) {
                    out.userDefinedMetadata = metadata.userDefinedMetadata;
                }
                if (metadata.modelInitializer != null) {
                    out.modelInitializer = metadata.modelInitializer;
                }
                if (metadata.initializerSignature != null) {
                    out.initializerSignature = metadata.initializerSignature;
                }
                if (metadata.trainingConfig != null) {
                    out.trainingConfig = metadata.trainingConfig;
                }
            }
            // Load weight data.
            const weightDataBase64 = this.LS.getItem(this.keys.weightData);
            if (weightDataBase64 == null) {
                throw new Error(`In local storage, the binary weight values of model ` +
                    `'${this.modelPath}' are missing.`);
            }
            out.weightData = base64StringToArrayBuffer(weightDataBase64);
            return out;
        }
    }
    BrowserLocalStorage.URL_SCHEME = 'localstorage://';
    const localStorageRouter = (url) => {
        if (!env().getBool('IS_BROWSER')) {
            return null;
        }
        else {
            if (!Array.isArray(url) && url.startsWith(BrowserLocalStorage.URL_SCHEME)) {
                return browserLocalStorage(url.slice(BrowserLocalStorage.URL_SCHEME.length));
            }
            else {
                return null;
            }
        }
    };
    IORouterRegistry.registerSaveRouter(localStorageRouter);
    IORouterRegistry.registerLoadRouter(localStorageRouter);
    /**
     * Factory function for local storage IOHandler.
     *
     * This `IOHandler` supports both `save` and `load`.
     *
     * For each model's saved artifacts, four items are saved to local storage.
     *   - `${PATH_SEPARATOR}/${modelPath}/info`: Contains meta-info about the
     *     model, such as date saved, type of the topology, size in bytes, etc.
     *   - `${PATH_SEPARATOR}/${modelPath}/topology`: Model topology. For Keras-
     *     style models, this is a stringized JSON.
     *   - `${PATH_SEPARATOR}/${modelPath}/weight_specs`: Weight specs of the
     *     model, can be used to decode the saved binary weight values (see
     *     item below).
     *   - `${PATH_SEPARATOR}/${modelPath}/weight_data`: Concatenated binary
     *     weight values, stored as a base64-encoded string.
     *
     * Saving may throw an `Error` if the total size of the artifacts exceed the
     * browser-specific quota.
     *
     * @param modelPath A unique identifier for the model to be saved. Must be a
     *   non-empty string.
     * @returns An instance of `IOHandler`, which can be used with, e.g.,
     *   `tf.Model.save`.
     */
    function browserLocalStorage(modelPath) {
        return new BrowserLocalStorage(modelPath);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const DEFAULT_FILE_NAME_PREFIX = 'model';
    const DEFAULT_JSON_EXTENSION_NAME = '.json';
    const DEFAULT_WEIGHT_DATA_EXTENSION_NAME = '.weights.bin';
    function defer(f) {
        return new Promise(resolve => setTimeout(resolve)).then(f);
    }
    class BrowserDownloads {
        constructor(fileNamePrefix) {
            if (!env().getBool('IS_BROWSER')) {
                // TODO(cais): Provide info on what IOHandlers are available under the
                //   current environment.
                throw new Error('browserDownloads() cannot proceed because the current environment ' +
                    'is not a browser.');
            }
            if (fileNamePrefix.startsWith(BrowserDownloads.URL_SCHEME)) {
                fileNamePrefix = fileNamePrefix.slice(BrowserDownloads.URL_SCHEME.length);
            }
            if (fileNamePrefix == null || fileNamePrefix.length === 0) {
                fileNamePrefix = DEFAULT_FILE_NAME_PREFIX;
            }
            this.modelJsonFileName = fileNamePrefix + DEFAULT_JSON_EXTENSION_NAME;
            this.weightDataFileName =
                fileNamePrefix + DEFAULT_WEIGHT_DATA_EXTENSION_NAME;
        }
        async save(modelArtifacts) {
            if (typeof (document) === 'undefined') {
                throw new Error('Browser downloads are not supported in ' +
                    'this environment since `document` is not present');
            }
            const weightsURL = window.URL.createObjectURL(new Blob([modelArtifacts.weightData], { type: 'application/octet-stream' }));
            if (modelArtifacts.modelTopology instanceof ArrayBuffer) {
                throw new Error('BrowserDownloads.save() does not support saving model topology ' +
                    'in binary formats yet.');
            }
            else {
                const weightsManifest = [{
                        paths: ['./' + this.weightDataFileName],
                        weights: modelArtifacts.weightSpecs
                    }];
                const modelJSON = getModelJSONForModelArtifacts(modelArtifacts, weightsManifest);
                const modelJsonURL = window.URL.createObjectURL(new Blob([JSON.stringify(modelJSON)], { type: 'application/json' }));
                // If anchor elements are not provided, create them without attaching them
                // to parents, so that the downloaded file names can be controlled.
                const jsonAnchor = this.modelJsonAnchor == null ?
                    document.createElement('a') :
                    this.modelJsonAnchor;
                jsonAnchor.download = this.modelJsonFileName;
                jsonAnchor.href = modelJsonURL;
                // Trigger downloads by evoking a click event on the download anchors.
                // When multiple downloads are started synchronously, Firefox will only
                // save the last one.
                await defer(() => jsonAnchor.dispatchEvent(new MouseEvent('click')));
                if (modelArtifacts.weightData != null) {
                    const weightDataAnchor = this.weightDataAnchor == null ?
                        document.createElement('a') :
                        this.weightDataAnchor;
                    weightDataAnchor.download = this.weightDataFileName;
                    weightDataAnchor.href = weightsURL;
                    await defer(() => weightDataAnchor.dispatchEvent(new MouseEvent('click')));
                }
                return { modelArtifactsInfo: getModelArtifactsInfoForJSON(modelArtifacts) };
            }
        }
    }
    BrowserDownloads.URL_SCHEME = 'downloads://';
    const browserDownloadsRouter = (url) => {
        if (!env().getBool('IS_BROWSER')) {
            return null;
        }
        else {
            if (!Array.isArray(url) && url.startsWith(BrowserDownloads.URL_SCHEME)) {
                return browserDownloads(url.slice(BrowserDownloads.URL_SCHEME.length));
            }
            else {
                return null;
            }
        }
    };
    IORouterRegistry.registerSaveRouter(browserDownloadsRouter);
    /**
     * Creates an IOHandler that triggers file downloads from the browser.
     *
     * The returned `IOHandler` instance can be used as model exporting methods such
     * as `tf.Model.save` and supports only saving.
     *
     * ```js
     * const model = tf.sequential();
     * model.add(tf.layers.dense(
     *     {units: 1, inputShape: [10], activation: 'sigmoid'}));
     * const saveResult = await model.save('downloads://mymodel');
     * // This will trigger downloading of two files:
     * //   'mymodel.json' and 'mymodel.weights.bin'.
     * console.log(saveResult);
     * ```
     *
     * @param fileNamePrefix Prefix name of the files to be downloaded. For use with
     *   `tf.Model`, `fileNamePrefix` should follow either of the following two
     *   formats:
     *   1. `null` or `undefined`, in which case the default file
     *      names will be used:
     *      - 'model.json' for the JSON file containing the model topology and
     *        weights manifest.
     *      - 'model.weights.bin' for the binary file containing the binary weight
     *        values.
     *   2. A single string or an Array of a single string, as the file name prefix.
     *      For example, if `'foo'` is provided, the downloaded JSON
     *      file and binary weights file will be named 'foo.json' and
     *      'foo.weights.bin', respectively.
     * @param config Additional configuration for triggering downloads.
     * @returns An instance of `BrowserDownloads` `IOHandler`.
     *
     * @doc {
     *   heading: 'Models',
     *   subheading: 'Loading',
     *   namespace: 'io',
     *   ignoreCI: true
     * }
     */
    function browserDownloads(fileNamePrefix = 'model') {
        return new BrowserDownloads(fileNamePrefix);
    }

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Monitor Promise.all progress, fire onProgress callback function.
     *
     * @param promises Promise list going to be monitored
     * @param onProgress Callback function. Fired when a promise resolved.
     * @param startFraction Optional fraction start. Default to 0.
     * @param endFraction Optional fraction end. Default to 1.
     */
    function monitorPromisesProgress(promises, onProgress, startFraction, endFraction) {
        checkPromises(promises);
        startFraction = startFraction == null ? 0 : startFraction;
        endFraction = endFraction == null ? 1 : endFraction;
        checkFraction(startFraction, endFraction);
        let resolvedPromise = 0;
        const registerMonitor = (promise) => {
            promise.then(value => {
                const fraction = startFraction +
                    ++resolvedPromise / promises.length * (endFraction - startFraction);
                // pass fraction as parameter to callback function.
                onProgress(fraction);
                return value;
            });
            return promise;
        };
        function checkPromises(promises) {
            assert(promises != null && Array.isArray(promises) && promises.length > 0, () => 'promises must be a none empty array');
        }
        function checkFraction(startFraction, endFraction) {
            assert(startFraction >= 0 && startFraction <= 1, () => `Progress fraction must be in range [0, 1], but ` +
                `got startFraction ${startFraction}`);
            assert(endFraction >= 0 && endFraction <= 1, () => `Progress fraction must be in range [0, 1], but ` +
                `got endFraction ${endFraction}`);
            assert(endFraction >= startFraction, () => `startFraction must be no more than endFraction, but ` +
                `got startFraction ${startFraction} and endFraction ` +
                `${endFraction}`);
        }
        return Promise.all(promises.map(registerMonitor));
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reads binary weights data from a number of URLs.
     *
     * @param fetchURLs URLs to send the HTTP requests at, using `fetch` calls.
     * @param requestOptions RequestInit (options) for the HTTP requests.
     * @param fetchFunc Optional overriding value for the `window.fetch` function.
     * @param onProgress Optional, progress callback function, fired periodically
     *   before the load is completed.
     * @returns A `Promise` of an Array of `ArrayBuffer`. The Array has the same
     *   length as `fetchURLs`.
     */
    async function loadWeightsAsArrayBuffer(fetchURLs, loadOptions) {
        if (loadOptions == null) {
            loadOptions = {};
        }
        const fetchFunc = loadOptions.fetchFunc == null ? env().platform.fetch :
            loadOptions.fetchFunc;
        // Create the requests for all of the weights in parallel.
        const requests = fetchURLs.map(fetchURL => fetchFunc(fetchURL, loadOptions.requestInit, { isBinary: true }));
        const fetchStartFraction = 0;
        const fetchEndFraction = 0.5;
        const responses = loadOptions.onProgress == null ?
            await Promise.all(requests) :
            await monitorPromisesProgress(requests, loadOptions.onProgress, fetchStartFraction, fetchEndFraction);
        const bufferPromises = responses.map(response => response.arrayBuffer());
        const bufferStartFraction = 0.5;
        const bufferEndFraction = 1;
        const buffers = loadOptions.onProgress == null ?
            await Promise.all(bufferPromises) :
            await monitorPromisesProgress(bufferPromises, loadOptions.onProgress, bufferStartFraction, bufferEndFraction);
        return buffers;
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const OCTET_STREAM_MIME_TYPE = 'application/octet-stream';
    const JSON_TYPE = 'application/json';
    class HTTPRequest {
        constructor(path, loadOptions) {
            this.DEFAULT_METHOD = 'POST';
            if (loadOptions == null) {
                loadOptions = {};
            }
            this.weightPathPrefix = loadOptions.weightPathPrefix;
            this.onProgress = loadOptions.onProgress;
            this.weightUrlConverter = loadOptions.weightUrlConverter;
            if (loadOptions.fetchFunc != null) {
                assert(typeof loadOptions.fetchFunc === 'function', () => 'Must pass a function that matches the signature of ' +
                    '`fetch` (see ' +
                    'https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API)');
                this.fetch = loadOptions.fetchFunc;
            }
            else {
                this.fetch = env().platform.fetch;
            }
            assert(path != null && path.length > 0, () => 'URL path for http must not be null, undefined or ' +
                'empty.');
            if (Array.isArray(path)) {
                assert(path.length === 2, () => 'URL paths for http must have a length of 2, ' +
                    `(actual length is ${path.length}).`);
            }
            this.path = path;
            if (loadOptions.requestInit != null &&
                loadOptions.requestInit.body != null) {
                throw new Error('requestInit is expected to have no pre-existing body, but has one.');
            }
            this.requestInit = loadOptions.requestInit || {};
        }
        async save(modelArtifacts) {
            if (modelArtifacts.modelTopology instanceof ArrayBuffer) {
                throw new Error('BrowserHTTPRequest.save() does not support saving model topology ' +
                    'in binary formats yet.');
            }
            const init = Object.assign({ method: this.DEFAULT_METHOD }, this.requestInit);
            init.body = new FormData();
            const weightsManifest = [{
                    paths: ['./model.weights.bin'],
                    weights: modelArtifacts.weightSpecs,
                }];
            const modelTopologyAndWeightManifest = getModelJSONForModelArtifacts(modelArtifacts, weightsManifest);
            init.body.append('model.json', new Blob([JSON.stringify(modelTopologyAndWeightManifest)], { type: JSON_TYPE }), 'model.json');
            if (modelArtifacts.weightData != null) {
                init.body.append('model.weights.bin', new Blob([modelArtifacts.weightData], { type: OCTET_STREAM_MIME_TYPE }), 'model.weights.bin');
            }
            const response = await this.fetch(this.path, init);
            if (response.ok) {
                return {
                    modelArtifactsInfo: getModelArtifactsInfoForJSON(modelArtifacts),
                    responses: [response],
                };
            }
            else {
                throw new Error(`BrowserHTTPRequest.save() failed due to HTTP response status ` +
                    `${response.status}.`);
            }
        }
        /**
         * Load model artifacts via HTTP request(s).
         *
         * See the documentation to `tf.io.http` for details on the saved
         * artifacts.
         *
         * @returns The loaded model artifacts (if loading succeeds).
         */
        async load() {
            const modelConfigRequest = await this.fetch(this.path, this.requestInit);
            if (!modelConfigRequest.ok) {
                throw new Error(`Request to ${this.path} failed with status code ` +
                    `${modelConfigRequest.status}. Please verify this URL points to ` +
                    `the model JSON of the model to load.`);
            }
            let modelJSON;
            try {
                modelJSON = await modelConfigRequest.json();
            }
            catch (e) {
                let message = `Failed to parse model JSON of response from ${this.path}.`;
                // TODO(nsthorat): Remove this after some time when we're comfortable that
                // .pb files are mostly gone.
                if (this.path.endsWith('.pb')) {
                    message += ' Your path contains a .pb file extension. ' +
                        'Support for .pb models have been removed in TensorFlow.js 1.0 ' +
                        'in favor of .json models. You can re-convert your Python ' +
                        'TensorFlow model using the TensorFlow.js 1.0 conversion scripts ' +
                        'or you can convert your.pb models with the \'pb2json\'' +
                        'NPM script in the tensorflow/tfjs-converter repository.';
                }
                else {
                    message += ' Please make sure the server is serving valid ' +
                        'JSON for this request.';
                }
                throw new Error(message);
            }
            // We do not allow both modelTopology and weightsManifest to be missing.
            const modelTopology = modelJSON.modelTopology;
            const weightsManifest = modelJSON.weightsManifest;
            if (modelTopology == null && weightsManifest == null) {
                throw new Error(`The JSON from HTTP path ${this.path} contains neither model ` +
                    `topology or manifest for weights.`);
            }
            return getModelArtifactsForJSON(modelJSON, (weightsManifest) => this.loadWeights(weightsManifest));
        }
        async loadWeights(weightsManifest) {
            const weightPath = Array.isArray(this.path) ? this.path[1] : this.path;
            const [prefix, suffix] = parseUrl(weightPath);
            const pathPrefix = this.weightPathPrefix || prefix;
            const weightSpecs = getWeightSpecs(weightsManifest);
            const fetchURLs = [];
            const urlPromises = [];
            for (const weightsGroup of weightsManifest) {
                for (const path of weightsGroup.paths) {
                    if (this.weightUrlConverter != null) {
                        urlPromises.push(this.weightUrlConverter(path));
                    }
                    else {
                        fetchURLs.push(pathPrefix + path + suffix);
                    }
                }
            }
            if (this.weightUrlConverter) {
                fetchURLs.push(...await Promise.all(urlPromises));
            }
            const buffers = await loadWeightsAsArrayBuffer(fetchURLs, {
                requestInit: this.requestInit,
                fetchFunc: this.fetch,
                onProgress: this.onProgress
            });
            return [weightSpecs, concatenateArrayBuffers(buffers)];
        }
    }
    HTTPRequest.URL_SCHEME_REGEX = /^https?:\/\//;
    /**
     * Extract the prefix and suffix of the url, where the prefix is the path before
     * the last file, and suffix is the search params after the last file.
     * ```
     * const url = 'http://tfhub.dev/model/1/tensorflowjs_model.pb?tfjs-format=file'
     * [prefix, suffix] = parseUrl(url)
     * // prefix = 'http://tfhub.dev/model/1/'
     * // suffix = '?tfjs-format=file'
     * ```
     * @param url the model url to be parsed.
     */
    function parseUrl(url) {
        const lastSlash = url.lastIndexOf('/');
        const lastSearchParam = url.lastIndexOf('?');
        const prefix = url.substring(0, lastSlash);
        const suffix = lastSearchParam > lastSlash ? url.substring(lastSearchParam) : '';
        return [prefix + '/', suffix];
    }
    function isHTTPScheme(url) {
        return url.match(HTTPRequest.URL_SCHEME_REGEX) != null;
    }
    const httpRouter = (url, loadOptions) => {
        if (typeof fetch === 'undefined' &&
            (loadOptions == null || loadOptions.fetchFunc == null)) {
            // `http` uses `fetch` or `node-fetch`, if one wants to use it in
            // an environment that is not the browser or node they have to setup a
            // global fetch polyfill.
            return null;
        }
        else {
            let isHTTP = true;
            if (Array.isArray(url)) {
                isHTTP = url.every(urlItem => isHTTPScheme(urlItem));
            }
            else {
                isHTTP = isHTTPScheme(url);
            }
            if (isHTTP) {
                return http(url, loadOptions);
            }
        }
        return null;
    };
    IORouterRegistry.registerSaveRouter(httpRouter);
    IORouterRegistry.registerLoadRouter(httpRouter);
    /**
     * Creates an IOHandler subtype that sends model artifacts to HTTP server.
     *
     * An HTTP request of the `multipart/form-data` mime type will be sent to the
     * `path` URL. The form data includes artifacts that represent the topology
     * and/or weights of the model. In the case of Keras-style `tf.Model`, two
     * blobs (files) exist in form-data:
     *   - A JSON file consisting of `modelTopology` and `weightsManifest`.
     *   - A binary weights file consisting of the concatenated weight values.
     * These files are in the same format as the one generated by
     * [tfjs_converter](https://js.tensorflow.org/tutorials/import-keras.html).
     *
     * The following code snippet exemplifies the client-side code that uses this
     * function:
     *
     * ```js
     * const model = tf.sequential();
     * model.add(
     *     tf.layers.dense({units: 1, inputShape: [100], activation: 'sigmoid'}));
     *
     * const saveResult = await model.save(tf.io.http(
     *     'http://model-server:5000/upload', {requestInit: {method: 'PUT'}}));
     * console.log(saveResult);
     * ```
     *
     * If the default `POST` method is to be used, without any custom parameters
     * such as headers, you can simply pass an HTTP or HTTPS URL to `model.save`:
     *
     * ```js
     * const saveResult = await model.save('http://model-server:5000/upload');
     * ```
     *
     * The following GitHub Gist
     * https://gist.github.com/dsmilkov/1b6046fd6132d7408d5257b0976f7864
     * implements a server based on [flask](https://github.com/pallets/flask) that
     * can receive the request. Upon receiving the model artifacts via the requst,
     * this particular server reconstitutes instances of [Keras
     * Models](https://keras.io/models/model/) in memory.
     *
     *
     * @param path A URL path to the model.
     *   Can be an absolute HTTP path (e.g.,
     *   'http://localhost:8000/model-upload)') or a relative path (e.g.,
     *   './model-upload').
     * @param requestInit Request configurations to be used when sending
     *    HTTP request to server using `fetch`. It can contain fields such as
     *    `method`, `credentials`, `headers`, `mode`, etc. See
     *    https://developer.mozilla.org/en-US/docs/Web/API/Request/Request
     *    for more information. `requestInit` must not have a body, because the
     * body will be set by TensorFlow.js. File blobs representing the model
     * topology (filename: 'model.json') and the weights of the model (filename:
     * 'model.weights.bin') will be appended to the body. If `requestInit` has a
     * `body`, an Error will be thrown.
     * @param loadOptions Optional configuration for the loading. It includes the
     *   following fields:
     *   - weightPathPrefix Optional, this specifies the path prefix for weight
     *     files, by default this is calculated from the path param.
     *   - fetchFunc Optional, custom `fetch` function. E.g., in Node.js,
     *     the `fetch` from node-fetch can be used here.
     *   - onProgress Optional, progress callback function, fired periodically
     *     before the load is completed.
     * @returns An instance of `IOHandler`.
     *
     * @doc {
     *   heading: 'Models',
     *   subheading: 'Loading',
     *   namespace: 'io',
     *   ignoreCI: true
     * }
     */
    function http(path, loadOptions) {
        return new HTTPRequest(path, loadOptions);
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a one-hot `tf.Tensor`. The locations represented by `indices` take
     * value `onValue` (defaults to 1), while all other locations take value
     * `offValue` (defaults to 0). If `indices` is rank `R`, the output has rank
     * `R+1` with the last axis of size `depth`.
     * `indices` used to encode prediction class must start from 0. For example,
     *  if you have 3 classes of data, class 1 should be encoded as 0, class 2
     *  should be 1, and class 3 should be 2.
     *
     * ```js
     * tf.oneHot(tf.tensor1d([0, 1], 'int32'), 3).print();
     * ```
     *
     * @param indices `tf.Tensor` of indices with dtype `int32`. Indices must
     * start from 0.
     * @param depth The depth of the one hot dimension.
     * @param onValue A number used to fill in the output when the index matches
     * the location.
     * @param offValue A number used to fill in the output when the index does
     *     not match the location.
     * @param dtype The dtype of the output tensor, default to 'int32'.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function oneHot_(indices, depth, onValue = 1, offValue = 0, dtype = 'int32') {
        if (depth < 2) {
            throw new Error(`Error in oneHot: depth must be >=2, but it is ${depth}`);
        }
        const $indices = convertToTensor(indices, 'indices', 'oneHot', 'int32');
        const inputs = { indices: $indices };
        const attrs = { dtype, depth, onValue, offValue };
        return ENGINE.runKernel(OneHot, inputs, attrs);
    }
    const oneHot = op({ oneHot_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the confusion matrix from true labels and predicted labels.
     *
     * ```js
     * const labels = tf.tensor1d([0, 1, 2, 1, 0], 'int32');
     * const predictions = tf.tensor1d([0, 2, 2, 1, 0], 'int32');
     * const numClasses = 3;
     * const out = tf.math.confusionMatrix(labels, predictions, numClasses);
     * out.print();
     * // Expected output matrix:
     * // [[2, 0, 0],
     * //  [0, 1, 1],
     * //  [0, 0, 1]]
     * ```
     *
     * @param labels The target labels, assumed to be 0-based integers
     *   for the classes. The shape is `[numExamples]`, where
     *   `numExamples` is the number of examples included.
     * @param predictions The predicted classes, assumed to be
     *   0-based integers for the classes. Must have the same shape as `labels`.
     * @param numClasses Number of all classes, as an integer.
     *   Its value must be larger than the largest element in `labels` and
     *   `predictions`.
     * @returns The confusion matrix as a int32-type 2D tensor. The value at
     *   row `r` and column `c` is the number of times examples of actual class
     *   `r` were predicted as class `c`.
     *
     * @doc {heading: 'Operations', subheading: 'Evaluation'}
     */
    function confusionMatrix_(labels, predictions, numClasses) {
        const $labels = convertToTensor(labels, 'labels', 'confusionMatrix');
        const $predictions = convertToTensor(predictions, 'predictions', 'confusionMatrix');
        assert(numClasses == null || numClasses > 0 && Number.isInteger(numClasses), () => `If provided, numClasses must be a positive integer, ` +
            `but got ${numClasses}`);
        assert($labels.rank === 1, () => `Expected the rank of labels to be 1, but got ${$labels.rank}`);
        assert($predictions.rank === 1, () => `Expected the rank of predictions to be 1, ` +
            `but got ${$predictions.rank}`);
        assert($labels.shape[0] === $predictions.shape[0], () => `Mismatch in the number of examples: ` +
            `${$labels.shape[0]} vs. ${$predictions.shape[0]}. ` +
            `Labels and predictions should have the same number of elements.`);
        assert(numClasses > 0 && Number.isInteger(numClasses), () => `numClasses is required to be a positive integer, but got ` +
            `${numClasses}`);
        // TODO(cais): In the future, if oneHot supports tensors inputs for
        //   `numClasses`, `confusionMatrix` can make `numClasses` optional.
        const oneHotLabels = oneHot(cast($labels, 'int32'), numClasses);
        const oneHotPredictions = oneHot(cast($predictions, 'int32'), numClasses);
        const oneHotLabelsT = transpose(oneHotLabels);
        const product = matMul(oneHotLabelsT, oneHotPredictions);
        return cast(product, 'int32');
    }
    op({ confusionMatrix_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates rank-3 `tf.Tensor` with the provided values, shape and dtype.
     *
     * The same functionality can be achieved with `tf.tensor`, but in general
     * we recommend using `tf.tensor3d` as it makes the code more readable.
     *
     *  ```js
     * // Pass a nested array.
     * tf.tensor3d([[[1], [2]], [[3], [4]]]).print();
     * ```
     * ```js
     * // Pass a flat array and specify a shape.
     * tf.tensor3d([1, 2, 3, 4], [2, 2, 1]).print();
     * ```
     *
     * @param values The values of the tensor. Can be nested array of numbers,
     *     or a flat array, or a `TypedArray`.
     * @param shape The shape of the tensor. If not provided,  it is inferred from
     *     `values`.
     * @param dtype The data type.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function tensor3d(values, shape, dtype) {
        assertNonNull(values);
        if (shape != null && shape.length !== 3) {
            throw new Error('tensor3d() requires shape to have three numbers');
        }
        const inferredShape = inferShape(values, dtype);
        if (inferredShape.length !== 3 && inferredShape.length !== 1) {
            throw new Error('tensor3d() requires values to be number[][][] or flat/TypedArray');
        }
        if (inferredShape.length === 1 && shape == null) {
            throw new Error('tensor3d() requires shape to be provided when `values` ' +
                'are a flat array');
        }
        return makeTensor(values, shape, inferredShape, dtype);
    }

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    let fromPixels2DContext;
    /**
     * Creates a `tf.Tensor` from an image.
     *
     * ```js
     * const image = new ImageData(1, 1);
     * image.data[0] = 100;
     * image.data[1] = 150;
     * image.data[2] = 200;
     * image.data[3] = 255;
     *
     * tf.browser.fromPixels(image).print();
     * ```
     *
     * @param pixels The input image to construct the tensor from. The
     * supported image types are all 4-channel. You can also pass in an image
     * object with following attributes:
     * `{data: Uint8Array; width: number; height: number}`
     * @param numChannels The number of channels of the output tensor. A
     * numChannels value less than 4 allows you to ignore channels. Defaults to
     * 3 (ignores alpha channel of input image).
     *
     * @returns A Tensor3D with the shape `[height, width, numChannels]`.
     *
     * Note: fromPixels can be lossy in some cases, same image may result in
     * slightly different tensor values, if rendered by different rendering
     * engines. This means that results from different browsers, or even same
     * browser with CPU and GPU rendering engines can be different. See discussion
     * in details:
     * https://github.com/tensorflow/tfjs/issues/5482
     *
     * @doc {heading: 'Browser', namespace: 'browser', ignoreCI: true}
     */
    function fromPixels_(pixels, numChannels = 3) {
        // Sanity checks.
        if (numChannels > 4) {
            throw new Error('Cannot construct Tensor with more than 4 channels from pixels.');
        }
        if (pixels == null) {
            throw new Error('pixels passed to tf.browser.fromPixels() can not be null');
        }
        let isPixelData = false;
        let isImageData = false;
        let isVideo = false;
        let isImage = false;
        let isCanvasLike = false;
        let isImageBitmap = false;
        if (pixels.data instanceof Uint8Array) {
            isPixelData = true;
        }
        else if (typeof (ImageData) !== 'undefined' && pixels instanceof ImageData) {
            isImageData = true;
        }
        else if (typeof (HTMLVideoElement) !== 'undefined' &&
            pixels instanceof HTMLVideoElement) {
            isVideo = true;
        }
        else if (typeof (HTMLImageElement) !== 'undefined' &&
            pixels instanceof HTMLImageElement) {
            isImage = true;
            // tslint:disable-next-line: no-any
        }
        else if (pixels.getContext != null) {
            isCanvasLike = true;
        }
        else if (typeof (ImageBitmap) !== 'undefined' && pixels instanceof ImageBitmap) {
            isImageBitmap = true;
        }
        else {
            throw new Error('pixels passed to tf.browser.fromPixels() must be either an ' +
                `HTMLVideoElement, HTMLImageElement, HTMLCanvasElement, ImageData ` +
                `in browser, or OffscreenCanvas, ImageData in webworker` +
                ` or {data: Uint32Array, width: number, height: number}, ` +
                `but was ${pixels.constructor.name}`);
        }
        // If the current backend has 'FromPixels' registered, it has a more
        // efficient way of handling pixel uploads, so we call that.
        const kernel = getKernel(FromPixels, ENGINE.backendName);
        if (kernel != null) {
            const inputs = { pixels };
            const attrs = { numChannels };
            return ENGINE.runKernel(FromPixels, inputs, attrs);
        }
        const [width, height] = isVideo ?
            [
                pixels.videoWidth,
                pixels.videoHeight
            ] :
            [pixels.width, pixels.height];
        let vals;
        if (isCanvasLike) {
            vals =
                // tslint:disable-next-line:no-any
                pixels.getContext('2d').getImageData(0, 0, width, height).data;
        }
        else if (isImageData || isPixelData) {
            vals = pixels.data;
        }
        else if (isImage || isVideo || isImageBitmap) {
            if (fromPixels2DContext == null) {
                if (typeof document === 'undefined') {
                    if (typeof OffscreenCanvas !== 'undefined' &&
                        typeof OffscreenCanvasRenderingContext2D !== 'undefined') {
                        // @ts-ignore
                        fromPixels2DContext = new OffscreenCanvas(1, 1).getContext('2d');
                    }
                    else {
                        throw new Error('Cannot parse input in current context. ' +
                            'Reason: OffscreenCanvas Context2D rendering is not supported.');
                    }
                }
                else {
                    fromPixels2DContext =
                        document.createElement('canvas').getContext('2d', { willReadFrequently: true });
                }
            }
            fromPixels2DContext.canvas.width = width;
            fromPixels2DContext.canvas.height = height;
            fromPixels2DContext.drawImage(pixels, 0, 0, width, height);
            vals = fromPixels2DContext.getImageData(0, 0, width, height).data;
        }
        let values;
        if (numChannels === 4) {
            values = new Int32Array(vals);
        }
        else {
            const numPixels = width * height;
            values = new Int32Array(numPixels * numChannels);
            for (let i = 0; i < numPixels; i++) {
                for (let channel = 0; channel < numChannels; ++channel) {
                    values[i * numChannels + channel] = vals[i * 4 + channel];
                }
            }
        }
        const outShape = [height, width, numChannels];
        return tensor3d(values, outShape, 'int32');
    }
    op({ fromPixels_ });

    /**
     * Check whether updates.shape = indices.shape[:batchDim] +
     * shape[sliceDim:]
     *
     * @param x The input tensor.
     */
    function validateUpdateShape(shape, indices, updates) {
        const sliceDim = (indices.rank > 1) ? indices.shape[indices.rank - 1] : 1;
        const batchDim = (indices.rank > 1) ? indices.rank - 1 : 1;
        const shapeError = 'Must have updates.shape = indices.shape[:batchDim] + ' +
            `shape[sliceDim:], got updates.shape: ${updates.shape}` +
            `, indices.shape: ${indices.shape}, shape: ${shape}` +
            `, sliceDim: ${sliceDim}, and batchDim: ${batchDim}.`;
        if (updates.rank < batchDim) {
            throw new Error(shapeError + ` update.rank < ${batchDim}. `);
        }
        if (shape.length < sliceDim + (updates.rank - batchDim)) {
            throw new Error(shapeError +
                ` Output shape length < ${sliceDim + (updates.rank - batchDim)}`);
        }
        if (updates.rank !== batchDim + shape.length - sliceDim) {
            throw new Error(shapeError + ` update.rank != ${batchDim + shape.length - sliceDim}`);
        }
        for (let d = 0; d < batchDim; ++d) {
            if (updates.shape[d] !== indices.shape[d]) {
                throw new Error(shapeError +
                    ` updates.shape[${d}] (${updates.shape[d]}) != indices.shape[${d}] (${indices.shape[d]}).`);
            }
        }
        for (let d = 0; d < updates.rank - batchDim; ++d) {
            if (updates.shape[d + batchDim] !== shape[d + sliceDim]) {
                throw new Error(shapeError +
                    ` updates.shape[${d + batchDim}] (${updates.shape[d + batchDim]}) != shape[${d + batchDim}] (${shape[d + batchDim]})`);
            }
        }
    }
    /**
     * Validate scatter nd inputs.
     *
     * @param update The tensor contains the update values.
     * @param indices The tensor contains the indices for the update values.
     * @param shape The shape of the output tensor.
     */
    function validateInput$1(updates, indices, shape) {
        if (indices.rank < 1) {
            throw new Error('tf.scatterND() expects the indices to be rank 1 or higher,' +
                ` but the rank was ${indices.rank}.`);
        }
        if (updates.rank < 1) {
            throw new Error('tf.scatterND() expects the updates to be rank 1 or higher,' +
                ` but the rank was ${updates.rank}.`);
        }
        if (indices.dtype !== 'int32') {
            throw new Error(`The dtype of 'indices' should be int32, but got dtype: ${indices.dtype}`);
        }
        if (shape.length < 1) {
            throw new Error(`Output rank must be greater or equal to 1, but got shape: ${shape}`);
        }
        if (shape.length === 0) {
            if (indices.size === 0) {
                throw new Error(`Indices specified for empty output. indices shape: ${indices.shape}`);
            }
            if (updates.size === 0) {
                throw new Error(`Updates specified for empty output. updates shape: ${updates.shape}`);
            }
        }
        validateUpdateShape(shape, indices, updates);
    }

    /**
     * @license
     * Copyright 2021 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function parseSliceParams(x, begin, size) {
        // The following logic allows for more ergonomic calls.
        let begin_;
        const xRank = x.shape.length;
        if (typeof begin === 'number') {
            begin_ = [begin, ...new Array(xRank - 1).fill(0)];
        }
        else if (begin.length < xRank) {
            begin_ = begin.concat(new Array(xRank - begin.length).fill(0));
        }
        else {
            begin_ = begin.slice();
        }
        begin_.forEach(d => {
            assert(d !== -1, () => 'slice() does not support negative begin indexing.');
        });
        let size_;
        if (size == null) {
            size_ = new Array(xRank).fill(-1);
        }
        else if (typeof size === 'number') {
            size_ = [size, ...new Array(xRank - 1).fill(-1)];
        }
        else if (size.length < xRank) {
            size_ = size.concat(new Array(xRank - size.length).fill(-1));
        }
        else {
            size_ = size;
        }
        size_ = size_.map((d, i) => {
            if (d >= 0) {
                return d;
            }
            else {
                assert(d === -1, () => `Negative size values should be exactly -1 but got ` +
                    `${d} for the slice() size at index ${i}.`);
                return x.shape[i] - begin_[i];
            }
        });
        return [begin_, size_];
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Serializable defines the serialization contract.
     *
     * TFJS requires serializable classes to return their className when asked
     * to avoid issues with minification.
     */
    class Serializable {
        /**
         * Return the class name for this class to use in serialization contexts.
         *
         * Generally speaking this will be the same thing that constructor.name
         * would have returned.  However, the class name needs to be robust
         * against minification for serialization/deserialization to work properly.
         *
         * There's also places such as initializers.VarianceScaling, where
         * implementation details between different languages led to different
         * class hierarchies and a non-leaf node is used for serialization purposes.
         */
        getClassName() {
            return this.constructor
                .className;
        }
        /**
         * Creates an instance of T from a ConfigDict.
         *
         * This works for most descendants of serializable.  A few need to
         * provide special handling.
         * @param cls A Constructor for the class to instantiate.
         * @param config The Configuration for the object.
         */
        /** @nocollapse */
        static fromConfig(cls, config) {
            return new cls(config);
        }
    }
    /**
     * Maps string keys to class constructors.
     *
     * Used during (de)serialization from the cross-language JSON format, which
     * requires the class name in the serialization format matches the class
     * names as used in Python, should it exist.
     */
    class SerializationMap {
        constructor() {
            this.classNameMap = {};
        }
        /**
         * Returns the singleton instance of the map.
         */
        static getMap() {
            if (SerializationMap.instance == null) {
                SerializationMap.instance = new SerializationMap();
            }
            return SerializationMap.instance;
        }
        /**
         * Registers the class as serializable.
         */
        static register(cls) {
            SerializationMap.getMap().classNameMap[cls.className] =
                [cls, cls.fromConfig];
        }
    }
    /**
     * Register a class with the serialization map of TensorFlow.js.
     *
     * This is often used for registering custom Layers, so they can be
     * serialized and deserialized.
     *
     * Example:
     *
     * ```js
     * class MyCustomLayer extends tf.layers.Layer {
     *   static className = 'MyCustomLayer';
     *
     *   constructor(config) {
     *     super(config);
     *   }
     * }
     * tf.serialization.registerClass(MyCustomLayer);
     * ```
     *
     * @param cls The class to be registered. It must have a public static member
     *   called `className` defined and the value must be a non-empty string.
     *
     * @doc {heading: 'Models', subheading: 'Serialization', ignoreCI: true}
     */
    function registerClass(cls) {
        assert(cls.className != null, () => `Class being registered does not have the static className ` +
            `property defined.`);
        assert(typeof cls.className === 'string', () => `className is required to be a string, but got type ` +
            typeof cls.className);
        assert(cls.className.length > 0, () => `Class being registered has an empty-string as its className, ` +
            `which is disallowed.`);
        SerializationMap.register(cls);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes absolute value element-wise: `abs(x)`
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 4]);
     *
     * x.abs().print();  // or tf.abs(x)
     * ```
     * @param x The input `tf.Tensor`.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function abs_(x) {
        const $x = convertToTensor(x, 'x', 'abs');
        if ($x.dtype === 'complex64') {
            const inputs = { x: $x };
            return ENGINE.runKernel(ComplexAbs, inputs);
        }
        else {
            const inputs = { x: $x };
            return ENGINE.runKernel(Abs, inputs);
        }
    }
    const abs = op({ abs_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes acos of the input `tf.Tensor` element-wise: `acos(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.acos().print();  // or tf.acos(x)
     * ```
     * @param x The input tensor.
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function acos_(x) {
        const $x = convertToTensor(x, 'x', 'acos');
        const inputs = { x: $x };
        return ENGINE.runKernel(Acos, inputs);
    }
    op({ acos_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the inverse hyperbolic cos of the input `tf.Tensor` element-wise:
     * `acosh(x)`
     *
     * ```js
     * const x = tf.tensor1d([10, 1, 3, 5.7]);
     *
     * x.acosh().print();  // or tf.acosh(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function acosh_(x) {
        const $x = convertToTensor(x, 'x', 'acosh');
        const inputs = { x: $x };
        return ENGINE.runKernel(Acosh, inputs);
    }
    op({ acosh_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Adds a list of `tf.Tensor`s element-wise, each with the same shape and dtype.
     *
     * ```js
     * const a = tf.tensor1d([1, 2]);
     * const b = tf.tensor1d([3, 4]);
     * const c = tf.tensor1d([5, 6]);
     *
     * tf.addN([a, b, c]).print();
     * ```
     * @param tensors A list of tensors with the same shape and dtype.
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function addN_(tensors) {
        assert(Array.isArray(tensors), () => 'The argument passed to tf.addN() must be a list of tensors');
        assert(tensors.length >= 1, () => `Must pass at least one tensor to tf.addN(), but got ` +
            `${tensors.length}`);
        const $tensors = tensors.map((t, i) => convertToTensor(t, `tensors${i}`, 'addN'));
        const firstTensor = $tensors[0];
        $tensors.forEach(t => {
            if (t.dtype !== firstTensor.dtype) {
                throw new Error('All tensors passed to tf.addN() must have the same dtype');
            }
        });
        $tensors.forEach(t => {
            if (!arraysEqual(t.shape, firstTensor.shape)) {
                throw new Error('All tensors passed to tf.addN() must have the same shape');
            }
        });
        const inputs = $tensors;
        return ENGINE.runKernel(AddN, inputs);
    }
    op({ addN_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the logical and of elements across dimensions of a `tf.Tensor`.
     *
     * Reduces the input along the dimensions given in `axes`. Unless `keepDims`
     * is true, the rank of the `tf.Tensor` is reduced by 1 for each entry in
     * `axes`. If `keepDims` is true, the reduced dimensions are retained with
     * length 1. If `axes` has no entries, all dimensions are reduced, and a
     * `tf.Tensor` with a single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 1, 1], 'bool');
     *
     * x.all().print();  // or tf.all(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 1, 0, 0], [2, 2], 'bool');
     *
     * const axis = 1;
     * x.all(axis).print();  // or tf.all(x, axis)
     * ```
     *
     * @param x The input tensor. Must be of dtype bool.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function all_(x, axis = null, keepDims = false) {
        const $x = convertToTensor(x, 'x', 'all', 'bool');
        const inputs = { x: $x };
        const attrs = { axis, keepDims };
        return ENGINE.runKernel(All, inputs, attrs);
    }
    op({ all_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the logical or of elements across dimensions of a `tf.Tensor`.
     *
     * Reduces the input along the dimensions given in `axes`. Unless `keepDims`
     * is true, the rank of the `tf.Tensor` is reduced by 1 for each entry in
     * `axes`. If `keepDims` is true, the reduced dimensions are retained with
     * length 1. If `axes` has no entries, all dimensions are reduced, and a
     * `tf.Tensor` with a single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 1, 1], 'bool');
     *
     * x.any().print();  // or tf.any(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 1, 0, 0], [2, 2], 'bool');
     *
     * const axis = 1;
     * x.any(axis).print();  // or tf.any(x, axis)
     * ```
     *
     * @param x The input tensor. Must be of dtype bool.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function any_(x, axis = null, keepDims = false) {
        const $x = convertToTensor(x, 'x', 'any', 'bool');
        const inputs = { x: $x };
        const attrs = { axis, keepDims };
        return ENGINE.runKernel(Any, inputs, attrs);
    }
    // tslint:disable-next-line:variable-name
    op({ any_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the indices of the maximum values along an `axis`.
     *
     * The result has the same shape as `input` with the dimension along `axis`
     * removed.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.argMax().print();  // or tf.argMax(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 4, 3], [2, 2]);
     *
     * const axis = 1;
     * x.argMax(axis).print();  // or tf.argMax(x, axis)
     * ```
     *
     * @param x The input tensor.
     * @param axis The dimension to reduce. Defaults to 0 (outer-most dimension).
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function argMax_(x, axis = 0) {
        const $x = convertToTensor(x, 'x', 'argMax');
        const inputs = { x: $x };
        const attrs = { axis };
        return ENGINE.runKernel(ArgMax, inputs, attrs);
    }
    op({ argMax_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the indices of the minimum values along an `axis`.
     *
     * The result has the same shape as `input` with the dimension along `axis`
     * removed.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.argMin().print();  // or tf.argMin(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 4, 3], [2, 2]);
     *
     * const axis = 1;
     * x.argMin(axis).print();  // or tf.argMin(x, axis)
     * ```
     *
     * @param x The input tensor.
     * @param axis The dimension to reduce. Defaults to 0 (outer-most dimension).
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function argMin_(x, axis = 0) {
        const $x = convertToTensor(x, 'x', 'argMin');
        const inputs = { x: $x };
        const attrs = { axis };
        return ENGINE.runKernel(ArgMin, inputs, attrs);
    }
    op({ argMin_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes asin of the input `tf.Tensor` element-wise: `asin(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.asin().print();  // or tf.asin(x)
     * ```
     * @param x The input tensor.
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function asin_(x) {
        const $x = convertToTensor(x, 'x', 'asin');
        const inputs = { x: $x };
        return ENGINE.runKernel(Asin, inputs);
    }
    op({ asin_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes inverse hyperbolic sin of the input `tf.Tensor` element-wise:
     * `asinh(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.asinh().print();  // or tf.asinh(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function asinh_(x) {
        const $x = convertToTensor(x, 'x', 'asinh');
        const inputs = { x: $x };
        return ENGINE.runKernel(Asinh, inputs);
    }
    op({ asinh_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes atan of the input `tf.Tensor` element-wise: `atan(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.atan().print();  // or tf.atan(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function atan_(x) {
        const $x = convertToTensor(x, 'x', 'atan');
        const inputs = { x: $x };
        return ENGINE.runKernel(Atan, inputs);
    }
    op({ atan_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes arctangent of `tf.Tensor`s a / b element-wise: `atan2(a, b)`.
     * Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1.0, 1.0, -1.0, .7]);
     * const b = tf.tensor1d([2.0, 13.0, 3.5, .21]);
     *
     * tf.atan2(a, b).print()
     * ```
     *
     * @param a The first tensor.
     * @param b The second tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function atan2_(a, b) {
        let $a = convertToTensor(a, 'a', 'atan2');
        let $b = convertToTensor(b, 'b', 'atan2');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Atan2, inputs);
    }
    op({ atan2_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes inverse hyperbolic tan of the input `tf.Tensor` element-wise:
     * `atanh(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, .1, -.1, .7]);
     *
     * x.atanh().print();  // or tf.atanh(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function atanh_(x) {
        const $x = convertToTensor(x, 'x', 'atanh');
        const inputs = { x: $x };
        return ENGINE.runKernel(Atanh, inputs);
    }
    op({ atanh_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the 2D average pooling of an image.
     *
     * @param x The input tensor, of rank 4 or rank 3 of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is assumed.
     * @param filterSize The filter size: `[filterHeight, filterWidth]`. If
     *     `filterSize` is a single number, then `filterHeight == filterWidth`.
     * @param strides The strides of the pooling: `[strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param pad The type of padding algorithm:
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *         https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function avgPool_(x, filterSize, strides, pad, dimRoundingMode) {
        const $x = convertToTensor(x, 'x', 'avgPool', 'float32');
        const dilations = 1;
        assert(eitherStridesOrDilationsAreOne(strides, dilations), () => 'Error in avgPool: Either strides or dilations must be 1. ' +
            `Got strides ${strides} and dilations '${dilations}'`);
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        assert(x4D.rank === 4, () => `Error in avgPool: x must be rank 4 but got rank ${x4D.rank}.`);
        checkPadOnDimRoundingMode('avgPool', pad, dimRoundingMode);
        const inputs = { x: x4D };
        const attrs = { filterSize, strides, pad, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        let res = ENGINE.runKernel(AvgPool, inputs, attrs);
        res = cast(res, $x.dtype);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const avgPool = op({ avgPool_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the 3D average pooling.
     *
     * ```js
     * const x = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
     * const result = tf.avgPool3d(x, 2, 1, 'valid');
     * result.print();
     * ```
     *
     * @param x The input tensor, of rank 5 or rank 4 of shape
     *     `[batch, depth, height, width, inChannels]`.
     * @param filterSize The filter size:
     *     `[filterDepth, filterHeight, filterWidth]`.
     *     If `filterSize` is a single number,
     *     then `filterDepth == filterHeight == filterWidth`.
     * @param strides The strides of the pooling:
     *     `[strideDepth, strideHeight, strideWidth]`.
     *     If `strides` is a single number,
     *     then `strideDepth == strideHeight == strideWidth`.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1*1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     * @param dataFormat An optional string from: "NDHWC", "NCDHW". Defaults to
     *     "NDHWC". Specify the data format of the input and output data. With the
     *     default format "NDHWC", the data is stored in the order of: [batch,
     *     depth, height, width, channels]. Only "NDHWC" is currently supported.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function avgPool3d_(x, filterSize, strides, pad, dimRoundingMode, dataFormat = 'NDHWC') {
        const $x = convertToTensor(x, 'x', 'avgPool3d', 'float32');
        let x5D = $x;
        let reshapedTo5D = false;
        if ($x.rank === 4) {
            reshapedTo5D = true;
            x5D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2], $x.shape[3]]);
        }
        assert(x5D.rank === 5, () => `Error in avgPool3d: x must be rank 5 but got rank ${x5D.rank}.`);
        assert(dataFormat === 'NDHWC', () => `Error in avgPool3d: Only NDHWC is currently supported, ` +
            `but got dataFormat of ${dataFormat}`);
        checkPadOnDimRoundingMode('avgPool3d', pad, dimRoundingMode);
        const inputs = { x: x5D };
        const attrs = { filterSize, strides, pad, dimRoundingMode, dataFormat };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        let res = ENGINE.runKernel(AvgPool3D, inputs, attrs);
        res = cast(res, x5D.dtype);
        if (reshapedTo5D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
        }
        return res;
    }
    op({ avgPool3d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Concatenates a list of `tf.Tensor`s along a given axis.
     *
     * The tensors ranks and types must match, and their sizes must match in all
     * dimensions except `axis`.
     *
     * Also available are stricter rank-specific methods that assert that
     * `tensors` are of the given rank:
     *   - `tf.concat1d`
     *   - `tf.concat2d`
     *   - `tf.concat3d`
     *   - `tf.concat4d`
     *
     * Except `tf.concat1d` (which does not have axis param), all methods have
     * same signature as this method.
     *
     * ```js
     * const a = tf.tensor1d([1, 2]);
     * const b = tf.tensor1d([3, 4]);
     * a.concat(b).print();  // or a.concat(b)
     * ```
     *
     * ```js
     * const a = tf.tensor1d([1, 2]);
     * const b = tf.tensor1d([3, 4]);
     * const c = tf.tensor1d([5, 6]);
     * tf.concat([a, b, c]).print();
     * ```
     *
     * ```js
     * const a = tf.tensor2d([[1, 2], [10, 20]]);
     * const b = tf.tensor2d([[3, 4], [30, 40]]);
     * const axis = 1;
     * tf.concat([a, b], axis).print();
     * ```
     * @param tensors A list of tensors to concatenate.
     * @param axis The axis to concatenate along. Defaults to 0 (the first dim).
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function concat_(tensors, axis = 0) {
        assert(tensors.length >= 1, () => 'Pass at least one tensor to concat');
        const $tensors = convertToTensorArray(tensors, 'tensors', 'concat', 'string_or_numeric');
        if ($tensors[0].dtype === 'complex64') {
            $tensors.forEach(tensor => {
                if (tensor.dtype !== 'complex64') {
                    throw new Error(`Cannot concatenate complex64 tensors with a tensor
          with dtype ${tensor.dtype}. `);
                }
            });
        }
        if ($tensors.length === 1) {
            return clone($tensors[0]);
        }
        const inputs = $tensors;
        const attr = { axis };
        return ENGINE.runKernel(Concat, inputs, attr);
    }
    const concat = op({ concat_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes sigmoid element-wise, `1 / (1 + exp(-x))`
     *
     * ```js
     * const x = tf.tensor1d([0, -1, 2, -3]);
     *
     * x.sigmoid().print();  // or tf.sigmoid(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function sigmoid_(x) {
        const $x = convertToTensor(x, 'x', 'sigmoid', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Sigmoid$1, inputs);
    }
    const sigmoid = op({ sigmoid_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes hyperbolic tangent of the input `tf.Tensor` element-wise: `tanh(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, 70]);
     *
     * x.tanh().print();  // or tf.tanh(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function tanh_(x) {
        const $x = convertToTensor(x, 'x', 'tanh', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Tanh$1, inputs);
    }
    const tanh = op({ tanh_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the next state and output of a BasicLSTMCell.
     *
     * Returns `[newC, newH]`.
     *
     * Derived from tf.contrib.rnn.BasicLSTMCell.
     *
     * @param forgetBias Forget bias for the cell.
     * @param lstmKernel The weights for the cell.
     * @param lstmBias The bias for the cell.
     * @param data The input to the cell.
     * @param c Previous cell state.
     * @param h Previous cell output.
     *
     * @doc {heading: 'Operations', subheading: 'RNN'}
     */
    function basicLSTMCell_(forgetBias, lstmKernel, lstmBias, data, c, h) {
        const $forgetBias = convertToTensor(forgetBias, 'forgetBias', 'basicLSTMCell');
        const $lstmKernel = convertToTensor(lstmKernel, 'lstmKernel', 'basicLSTMCell');
        const $lstmBias = convertToTensor(lstmBias, 'lstmBias', 'basicLSTMCell');
        const $data = convertToTensor(data, 'data', 'basicLSTMCell');
        const $c = convertToTensor(c, 'c', 'basicLSTMCell');
        const $h = convertToTensor(h, 'h', 'basicLSTMCell');
        const combined = concat([$data, $h], 1);
        const weighted = matMul(combined, $lstmKernel);
        const res = add$1(weighted, $lstmBias);
        // i = input_gate, j = new_input, f = forget_gate, o = output_gate
        const batchSize = res.shape[0];
        const sliceCols = res.shape[1] / 4;
        const sliceSize = [batchSize, sliceCols];
        const i = slice(res, [0, 0], sliceSize);
        const j = slice(res, [0, sliceCols], sliceSize);
        const f = slice(res, [0, sliceCols * 2], sliceSize);
        const o = slice(res, [0, sliceCols * 3], sliceSize);
        const newC = add$1(mul(sigmoid(i), tanh(j)), mul($c, sigmoid(add$1($forgetBias, f))));
        const newH = mul(tanh(newC), sigmoid(o));
        return [newC, newH];
    }
    op({ basicLSTMCell_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * This operation reshapes the "batch" dimension 0 into `M + 1` dimensions of
     * shape `blockShape + [batch]`, interleaves these blocks back into the grid
     * defined by the spatial dimensions `[1, ..., M]`, to obtain a result with
     * the same rank as the input. The spatial dimensions of this intermediate
     * result are then optionally cropped according to `crops` to produce the
     * output. This is the reverse of `tf.spaceToBatchND`. See below for a precise
     * description.
     *
     * ```js
     * const x = tf.tensor4d([1, 2, 3, 4], [4, 1, 1, 1]);
     * const blockShape = [2, 2];
     * const crops = [[0, 0], [0, 0]];
     *
     * x.batchToSpaceND(blockShape, crops).print();
     * ```
     *
     * @param x A `tf.Tensor`. N-D with `x.shape` = `[batch] + spatialShape +
     * remainingShape`, where spatialShape has `M` dimensions.
     * @param blockShape A 1-D array. Must have shape `[M]`, all values must
     * be >= 1.
     * @param crops A 2-D array.  Must have shape `[M, 2]`, all values must be >= 0.
     * `crops[i] = [cropStart, cropEnd]` specifies the amount to crop from input
     * dimension `i + 1`, which corresponds to spatial dimension `i`. It is required
     * that `cropStart[i] + cropEnd[i] <= blockShape[i] * inputShape[i + 1]`
     *
     * This operation is equivalent to the following steps:
     *
     * 1. Reshape `x` to `reshaped` of shape: `[blockShape[0], ...,
     * blockShape[M-1], batch / prod(blockShape), x.shape[1], ...,
     * x.shape[N-1]]`
     *
     * 2. Permute dimensions of `reshaped` to produce `permuted` of shape `[batch /
     * prod(blockShape),x.shape[1], blockShape[0], ..., x.shape[M],
     * blockShape[M-1],x.shape[M+1], ..., x.shape[N-1]]`
     *
     * 3. Reshape `permuted` to produce `reshapedPermuted` of shape `[batch /
     * prod(blockShape),x.shape[1] * blockShape[0], ..., x.shape[M] *
     * blockShape[M-1],x.shape[M+1], ..., x.shape[N-1]]`
     *
     * 4. Crop the start and end of dimensions `[1, ..., M]` of `reshapedPermuted`
     * according to `crops` to produce the output of shape: `[batch /
     * prod(blockShape),x.shape[1] * blockShape[0] - crops[0,0] - crops[0,1],
     * ..., x.shape[M] * blockShape[M-1] - crops[M-1,0] -
     * crops[M-1,1],x.shape[M+1], ..., x.shape[N-1]]`
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function batchToSpaceND_(x, blockShape, crops) {
        const $x = convertToTensor(x, 'x', 'batchToSpaceND');
        const prod = blockShape.reduce((a, b) => a * b);
        assert($x.rank >= 1 + blockShape.length, () => `input rank is ${$x.rank} but should be > than blockShape.length ${blockShape.length}`);
        assert(crops.length === blockShape.length, () => `crops.length is ${crops.length} but should be equal to blockShape.length  ${blockShape.length}`);
        assert($x.shape[0] % prod === 0, () => `input tensor batch is ${$x.shape[0]} but is not divisible by the product of ` +
            `the elements of blockShape ${blockShape.join(' * ')} === ${prod}`);
        const inputs = { x: $x };
        const attrs = { blockShape, crops };
        return ENGINE.runKernel(BatchToSpaceND, inputs, attrs);
    }
    const batchToSpaceND = op({ batchToSpaceND_ });

    function xAs4D(x) {
        let x4D;
        if (x.rank === 0 || x.rank === 1) {
            x4D = reshape$1(x, [1, 1, 1, x.size]);
        }
        else if (x.rank === 2) {
            x4D = reshape$1(x, [1, 1, x.shape[0], x.shape[1]]);
        }
        else if (x.rank === 3) {
            x4D = reshape$1(x, [1, x.shape[0], x.shape[1], x.shape[2]]);
        }
        else {
            x4D = x;
        }
        return x4D;
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Batch normalization.
     *
     * As described in
     * [http://arxiv.org/abs/1502.03167](http://arxiv.org/abs/1502.03167).
     *
     * Mean, variance, scale, and offset can be of two shapes:
     *   - The same shape as the input.
     *   - In the common case, the depth dimension is the last dimension of x, so
     *     the values would be a `tf.Tensor1D` of shape [depth].
     *
     * Also available are stricter rank-specific methods with the same signature
     * as this method that assert that parameters passed are of given rank
     *   - `tf.batchNorm2d`
     *   - `tf.batchNorm3d`
     *   - `tf.batchNorm4d`
     *
     * @param x The input Tensor.
     * @param mean A mean Tensor.
     * @param variance A variance Tensor.
     * @param offset An offset Tensor.
     * @param scale A scale Tensor.
     * @param varianceEpsilon A small float number to avoid dividing by 0.
     *
     * @doc {heading: 'Operations', subheading: 'Normalization'}
     */
    function batchNorm_(x, mean, variance, offset, scale, varianceEpsilon) {
        if (varianceEpsilon == null) {
            varianceEpsilon = 0.001;
        }
        const $x = convertToTensor(x, 'x', 'batchNorm');
        const $mean = convertToTensor(mean, 'mean', 'batchNorm');
        const $variance = convertToTensor(variance, 'variance', 'batchNorm');
        let $scale;
        if (scale != null) {
            $scale = convertToTensor(scale, 'scale', 'batchNorm');
        }
        let $offset;
        if (offset != null) {
            $offset = convertToTensor(offset, 'offset', 'batchNorm');
        }
        assert($mean.rank === $variance.rank, () => 'Batch normalization gradient requires mean and variance to have ' +
            'equal ranks.');
        assert($offset == null || $mean.rank === $offset.rank, () => 'Batch normalization gradient requires mean and offset to have ' +
            'equal ranks.');
        assert($scale == null || $mean.rank === $scale.rank, () => 'Batch normalization gradient requires mean and scale to have ' +
            'equal ranks.');
        const x4D = xAs4D($x);
        const inputs = {
            x: x4D,
            scale: $scale,
            offset: $offset,
            mean: $mean,
            variance: $variance
        };
        const attrs = { varianceEpsilon };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(FusedBatchNorm, inputs, attrs);
        return reshape$1(res, $x.shape);
    }
    const batchNorm = op({ batchNorm_ });

    /**
     * Batch normalization, strictly for 2D. For the more relaxed version, see
     * `tf.batchNorm`.
     *
     * @param x The input Tensor.
     * @param mean A mean Tensor.
     * @param variance A variance Tensor.
     * @param offset An offset Tensor.
     * @param scale A scale Tensor.
     * @param varianceEpsilon A small float number to avoid dividing by 0.
     */
    function batchNorm2d_(x, mean, variance, offset, scale, varianceEpsilon) {
        const $x = convertToTensor(x, 'x', 'batchNorm');
        const $mean = convertToTensor(mean, 'mean', 'batchNorm');
        const $variance = convertToTensor(variance, 'variance', 'batchNorm');
        let $scale;
        if (scale != null) {
            $scale = convertToTensor(scale, 'scale', 'batchNorm');
        }
        let $offset;
        if (offset != null) {
            $offset = convertToTensor(offset, 'offset', 'batchNorm');
        }
        assert($x.rank === 2, () => `Error in batchNorm2D: x must be rank 2 but got rank ` +
            `${$x.rank}.`);
        assert($mean.rank === 2 || $mean.rank === 1, () => `Error in batchNorm2D: mean must be rank 2 or rank 1 but ` +
            `got rank ${$mean.rank}.`);
        assert($variance.rank === 2 || $variance.rank === 1, () => `Error in batchNorm2D: variance must be rank 2 or rank 1 ` +
            `but got rank ${$variance.rank}.`);
        if ($scale != null) {
            assert($scale.rank === 2 || $scale.rank === 1, () => `Error in batchNorm2D: scale must be rank 2 or rank 1 ` +
                `but got rank ${$scale.rank}.`);
        }
        if ($offset != null) {
            assert($offset.rank === 2 || $offset.rank === 1, () => `Error in batchNorm2D: offset must be rank 2 or rank 1 ` +
                `but got rank ${$offset.rank}.`);
        }
        return batchNorm($x, $mean, $variance, $offset, $scale, varianceEpsilon);
    }
    op({ batchNorm2d_ });

    /**
     * Batch normalization, strictly for 3D. For the more relaxed version, see
     * `tf.batchNorm`.
     *
     * @param x The input Tensor.
     * @param mean A mean Tensor.
     * @param variance A variance Tensor.
     * @param offset An offset Tensor.
     * @param scale A scale Tensor.
     * @param varianceEpsilon A small float number to avoid dividing by 0.
     */
    function batchNorm3d_(x, mean, variance, offset, scale, varianceEpsilon) {
        const $x = convertToTensor(x, 'x', 'batchNorm');
        const $mean = convertToTensor(mean, 'mean', 'batchNorm');
        const $variance = convertToTensor(variance, 'variance', 'batchNorm');
        let $scale;
        if (scale != null) {
            $scale = convertToTensor(scale, 'scale', 'batchNorm');
        }
        let $offset;
        if (offset != null) {
            $offset = convertToTensor(offset, 'offset', 'batchNorm');
        }
        assert($x.rank === 3, () => `Error in batchNorm3D: x must be rank 3 but got rank ` +
            `${$x.rank}.`);
        assert($mean.rank === 3 || $mean.rank === 1, () => `Error in batchNorm3D: mean must be rank 3 or rank 1 but ` +
            `got rank ${$mean.rank}.`);
        assert($variance.rank === 3 || $variance.rank === 1, () => `Error in batchNorm3D: variance must be rank 3 or rank 1 ` +
            `but got rank ${$variance.rank}.`);
        if ($scale != null) {
            assert($scale.rank === 3 || $scale.rank === 1, () => `Error in batchNorm3D: scale must be rank 3 or rank 1 ` +
                `but got rank ${$scale.rank}.`);
        }
        if ($offset != null) {
            assert($offset.rank === 3 || $offset.rank === 1, () => `Error in batchNorm3D: offset must be rank 3 or rank 1 ` +
                `but got rank ${$offset.rank}.`);
        }
        return batchNorm($x, $mean, $variance, $offset, $scale, varianceEpsilon);
    }
    op({ batchNorm3d_ });

    /**
     * Batch normalization, strictly for 4D. For the more relaxed version, see
     * `tf.batchNorm`.
     *
     * @param x The input Tensor.
     * @param mean A mean Tensor.
     * @param variance A variance Tensor.
     * @param offset An offset Tensor.
     * @param scale A scale Tensor.
     * @param varianceEpsilon A small float number to avoid dividing by 0.
     */
    function batchNorm4d_(x, mean, variance, offset, scale, varianceEpsilon) {
        const $x = convertToTensor(x, 'x', 'batchNorm');
        const $mean = convertToTensor(mean, 'mean', 'batchNorm');
        const $variance = convertToTensor(variance, 'variance', 'batchNorm');
        let $scale;
        if (scale != null) {
            $scale = convertToTensor(scale, 'scale', 'batchNorm');
        }
        let $offset;
        if (offset != null) {
            $offset = convertToTensor(offset, 'offset', 'batchNorm');
        }
        assert($x.rank === 4, () => `Error in batchNorm4D: x must be rank 4 but got rank ` +
            `${$x.rank}.`);
        assert($mean.rank === 4 || $mean.rank === 1, () => `Error in batchNorm4D: mean must be rank 4 or rank 1 but ` +
            `got rank ${$mean.rank}.`);
        assert($variance.rank === 4 || $variance.rank === 1, () => `Error in batchNorm4D: variance must be rank 4 or rank 1 ` +
            `but got rank ${$variance.rank}.`);
        if ($scale != null) {
            assert($scale.rank === 4 || $scale.rank === 1, () => `Error in batchNorm4D: scale must be rank 4 or rank 1 ` +
                `but got rank ${$scale.rank}.`);
        }
        if ($offset != null) {
            assert($offset.rank === 4 || $offset.rank === 1, () => `Error in batchNorm4D: offset must be rank 4 or rank 1 ` +
                `but got rank ${$offset.rank}.`);
        }
        return batchNorm($x, $mean, $variance, $offset, $scale, varianceEpsilon);
    }
    op({ batchNorm4d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Outputs a vector with length `size` and the same dtype as `weights`.
     *
     * If `weights` are empty, then index `i` stores the number of times the value
     * `i` is counted in `x`. If `weights` are non-empty, then index `i` stores the
     * sum of the value in `weights` at each index where the corresponding value in
     * `x` is `i`.
     *
     * Values in `x` outside of the range [0, size) are ignored.
     *
     * @param x The input int tensor, rank 1.
     * @param weights The weights tensor, must have the same shape as x, or a
     *     length-0 Tensor, in which case it acts as all weights equal to 1.
     * @param size Non-negative integer.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function bincount_(x, weights, size) {
        const $x = convertToTensor(x, 'x', 'bincount');
        const $weights = convertToTensor(weights, 'weights', 'bincount');
        assert($x.dtype === 'int32', () => `Error in bincount: input ` +
            `dtype must be int32, but got ${$x.dtype}`);
        assert(size >= 0, () => `size must be non-negative, but got ${size}.`);
        assert($weights.size === $x.size || $weights.size === 0, () => `Error in bincount: weights must have the same size as input or` +
            `0-length, but got input shape: ${$x.shape}, weights shape: ` +
            `${$weights.shape}.`);
        const inputs = { x: $x, weights: $weights };
        const attrs = { size };
        return ENGINE.runKernel(Bincount, inputs, attrs);
    }
    const bincount = op({ bincount_ });

    /**
     * @license
     * Copyright 2021 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Return the shape of s0 op s1 with broadcast.
     *
     * compute r0, the broadcasted shape as a tensor.
     * s0, s1 and r0 are all integer vectors.
     *
     * This function returns the shape of the result of an operation between
     * two tensors of size s0 and s1 performed with broadcast.
     *
     * @param s0 A tensor representing a shape
     * @param s1 A tensor representing a shape
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function broadcastArgs_(s0, s1) {
        const shape1Input = convertToTensor(s0, 's0', 'broadcastArgs', 'int32');
        const shape2Input = convertToTensor(s1, 's1', 'broadcastArgs', 'int32');
        if (shape1Input.rank !== 1) {
            throw new Error('broadcastArgs(): first input must be a vector (rank=1). ' +
                `Has rank ${shape1Input.rank}`);
        }
        if (shape2Input.rank !== 1) {
            throw new Error('broadcastArgs(): second input must be a vector (rank=1). ' +
                `Has rank ${shape2Input.rank}`);
        }
        const inputs = { s0: shape1Input, s1: shape2Input };
        return ENGINE.runKernel(BroadcastArgs, inputs);
    }
    op({ broadcastArgs_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates an empty `tf.TensorBuffer` with the specified `shape` and `dtype`.
     *
     * The values are stored in CPU as `TypedArray`. Fill the buffer using
     * `buffer.set()`, or by modifying directly `buffer.values`.
     *
     * When done, call `buffer.toTensor()` to get an immutable `tf.Tensor` with
     * those values.
     *
     * ```js
     * // Create a buffer and set values at particular indices.
     * const buffer = tf.buffer([2, 2]);
     * buffer.set(3, 0, 0);
     * buffer.set(5, 1, 0);
     *
     * // Convert the buffer back to a tensor.
     * buffer.toTensor().print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param dtype The dtype of the buffer. Defaults to 'float32'.
     * @param values The values of the buffer as `TypedArray`. Defaults to
     * zeros.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function buffer(shape, dtype = 'float32', values) {
        dtype = dtype || 'float32';
        assertNonNegativeIntegerDimensions(shape);
        return new TensorBuffer(shape, dtype, values);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes ceiling of input `tf.Tensor` element-wise: `ceil(x)`
     *
     * ```js
     * const x = tf.tensor1d([.6, 1.1, -3.3]);
     *
     * x.ceil().print();  // or tf.ceil(x)
     * ```
     * @param x The input Tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function ceil_(x) {
        const $x = convertToTensor(x, 'x', 'ceil', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Ceil, inputs);
    }
    op({ ceil_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` filled with a scalar value.
     *
     * ```js
     * tf.fill([2, 2], 4).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param value The scalar value to fill the tensor with.
     * @param dtype The type of an element in the resulting tensor. Defaults to
     * 'float'.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function fill(shape, value, dtype) {
        const attrs = { shape, value, dtype };
        return ENGINE.runKernel(Fill, {}, attrs);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Clips values element-wise. `max(min(x, clipValueMax), clipValueMin)`
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 4]);
     *
     * x.clipByValue(-2, 3).print();  // or tf.clipByValue(x, -2, 3)
     * ```
     * @param x The input tensor.
     * @param clipValueMin Lower bound of range to be clipped to.
     * @param clipValueMax Upper bound of range to be clipped to.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function clipByValue_(x, clipValueMin, clipValueMax) {
        const $x = convertToTensor(x, 'x', 'clipByValue');
        assert((clipValueMin <= clipValueMax), () => `Error in clip: min (${clipValueMin}) must be ` +
            `less than or equal to max (${clipValueMax}).`);
        if (clipValueMin === clipValueMax) {
            return fill($x.shape, clipValueMin, $x.dtype);
        }
        const inputs = { x: $x };
        const attrs = { clipValueMin, clipValueMax };
        return ENGINE.runKernel(ClipByValue, inputs, attrs);
    }
    op({ clipByValue_ });

    /**
     * Concatenates a list of`tf.Tensor1D`s along an axis. See `concat` for details.
     *
     * For example, if:
     * A: shape(3) = |r1, g1, b1|
     * B: shape(2) = |r2, g2|
     * C = tf.concat1d([A, B]) == |r1, g1, b1, r2, g2|
     *
     * @param tensors A list of`tf.Tensor`s to concatenate.
     * @return The concatenated array.
     */
    function concat1d_(tensors) {
        return concat(tensors, 0 /* axis */);
    }
    op({ concat1d_ });

    /**
     * Concatenates a list of`tf.Tensor2D`s along an axis. See `concat` for details.
     *
     * For example, if:
     * A: shape(2, 3) = | r1, g1, b1 |
     *                  | r2, g2, b2 |
     *
     * B: shape(2, 3) = | r3, g3, b3 |
     *                  | r4, g4, b4 |
     *
     * C = tf.concat2d([A, B], axis)
     *
     * if axis = 0:
     * C: shape(4, 3) = | r1, g1, b1 |
     *                  | r2, g2, b2 |
     *                  | r3, g3, b3 |
     *                  | r4, g4, b4 |
     *
     * if axis = 1:
     * C = shape(2, 6) = | r1, g1, b1, r3, g3, b3 |
     *                   | r2, g2, b2, r4, g4, b4 |
     *
     *
     * @param tensors A list of `tf.Tensor`s to concatenate.
     * @param axis The axis to concatenate along.
     * @return The concatenated array.
     */
    function concat2d_(tensors, axis) {
        return concat(tensors, axis);
    }
    op({ concat2d_ });

    /**
     * Concatenates a list of `tf.Tensor3D`s along an axis.
     * See `concat` for details.
     *
     * For example, if:
     * A: shape(2, 1, 3) = | r1, g1, b1 |
     *                     | r2, g2, b2 |
     *
     * B: shape(2, 1, 3) = | r3, g3, b3 |
     *                     | r4, g4, b4 |
     *
     * C = tf.concat3d([A, B], axis)
     *
     * if axis = 0:
     * C: shape(4, 1, 3) = | r1, g1, b1 |
     *                     | r2, g2, b2 |
     *                     | r3, g3, b3 |
     *                     | r4, g4, b4 |
     *
     * if axis = 1:
     * C: shape(2, 2, 3) = | r1, g1, b1, r3, g3, b3 |
     *                     | r2, g2, b2, r4, g4, b4 |
     *
     * if axis = 2:
     * C = shape(2, 1, 6) = | r1, g1, b1, r3, g3, b3 |
     *                      | r2, g2, b2, r4, g4, b4 |
     *
     * @param tensors A list of`tf.Tensor`s to concatenate.
     * @param axis The axis to concate along.
     * @return The concatenated array.
     */
    function concat3d_(tensors, axis) {
        return concat(tensors, axis);
    }
    op({ concat3d_ });

    /**
     * Concatenates a list of `tf.Tensor4D`s along an axis.
     * See `concat` for details.
     *
     * @param tensors A list of `tf.Tensor`s to concatenate.
     * @param axis The axis to concate along.
     * @return The concatenated array.
     */
    function concat4d_(tensors, axis) {
        return concat(tensors, axis);
    }
    op({ concat4d_ });

    /**
     * Computes a 1D convolution over the input x.
     *
     * @param x The input tensor, of rank 3 or rank 2, of shape
     *     `[batch, width, inChannels]`. If rank 2, batch of 1 is assumed.
     * @param filter The filter, rank 3, of shape
     *     `[filterWidth, inDepth, outDepth]`.
     * @param stride The number of entries by which the filter is moved right at
     *     each step.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dataFormat An optional string from "NWC", "NCW". Defaults to "NWC",
     *     the data is stored in the order of [batch, in_width, in_channels]. Only
     *     "NWC" is currently supported.
     * @param dilation The dilation rate in which we sample input values in
     *     atrous convolution. Defaults to `1`. If it is greater than 1, then
     *     stride must be `1`.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function conv1d_(x, filter, stride, pad, dataFormat = 'NWC', dilation = 1, dimRoundingMode) {
        const $x = convertToTensor(x, 'x', 'conv1d');
        const $filter = convertToTensor(filter, 'filter', 'conv1d');
        let x3D = $x;
        let reshapedTo3D = false;
        if ($x.rank === 2) {
            reshapedTo3D = true;
            x3D = reshape$1($x, [1, $x.shape[0], $x.shape[1]]);
        }
        assert(x3D.rank === 3, () => `Error in conv1d: input must be rank 3, but got rank ${x3D.rank}.`);
        assert($filter.rank === 3, () => `Error in conv1d: filter must be rank 3, but got rank ` +
            `${$filter.rank}.`);
        checkPadOnDimRoundingMode('conv1d', pad, dimRoundingMode);
        assert(x3D.shape[2] === $filter.shape[1], () => `Error in conv1d: depth of input (${x3D.shape[2]}) must match ` +
            `input depth for filter ${$filter.shape[1]}.`);
        assert(eitherStridesOrDilationsAreOne(stride, dilation), () => 'Error in conv1D: Either stride or dilation must be 1. ' +
            `Got stride ${stride} and dilation '${dilation}'`);
        assert(dataFormat === 'NWC', () => `Error in conv1d: got dataFormat of ${dataFormat} but only NWC is currently supported.`);
        const filter4D = reshape$1($filter, [1, $filter.shape[0], $filter.shape[1], $filter.shape[2]]);
        const input4D = reshape$1(x3D, [x3D.shape[0], 1, x3D.shape[1], x3D.shape[2]]);
        const strides = [1, stride];
        const dilations = [1, dilation];
        const conv2dDataFormat = 'NHWC';
        const res = conv2d$1(input4D, filter4D, strides, pad, conv2dDataFormat, dilations, dimRoundingMode);
        if (reshapedTo3D) {
            return reshape$1(res, [res.shape[2], res.shape[3]]);
        }
        return reshape$1(res, [res.shape[0], res.shape[2], res.shape[3]]);
    }
    op({ conv1d_ });

    /**
     * Computes the transposed 2D convolution of an image, also known as a
     * deconvolution.
     *
     * @param x The input image, of rank 4 or rank 3, of shape
     *   `[batch, height, width, inDepth]`. If rank 3, batch of 1 is assumed.
     * @param filter The filter, rank 4, of shape
     *     `[filterHeight, filterWidth, outDepth, inDepth]`.
     *     `inDepth` must match `inDepth` in `x`.
     * @param outputShape Output shape, of rank 4 or rank 3:
     *     `[batch, height, width, outDepth]`. If rank 3, batch of 1 is assumed.
     * @param strides The strides of the original convolution:
     *     `[strideHeight, strideWidth]`.
     * @param pad  The type of padding algorithm used in the non-transpose version
     *    of the op.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function conv2dTranspose_(x, filter, outputShape, strides, pad, dimRoundingMode) {
        const $x = convertToTensor(x, 'x', 'conv2dTranspose');
        const $filter = convertToTensor(filter, 'filter', 'conv2dTranspose');
        return conv2DBackpropInput(outputShape, $x, $filter, strides, pad, 'NHWC', dimRoundingMode);
    }
    op({ conv2dTranspose_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes a 3D convolution over the input x.
     *
     * @param x The input tensor, of rank 5 or rank 4, of shape
     *     `[batch, depth, height, width, channels]`. If rank 4,
     * batch of 1 is assumed.
     * @param filter The filter, rank 5, of shape
     *     `[filterDepth, filterHeight, filterWidth, inChannels, outChannels]`.
     *      inChannels must match between input and filter.
     * @param strides The strides of the convolution: `[strideDepth, strideHeight,
     * strideWidth]`.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dataFormat: An optional string from: "NDHWC", "NCDHW". Defaults to
     *     "NDHWC". Specify the data format of the input and output data. With the
     *     default format "NDHWC", the data is stored in the order of: [batch,
     *     depth, height, width, channels]. Only "NDHWC" is currently supported.
     * @param dilations The dilation rates: `[dilationDepth, dilationHeight,
     *     dilationWidth]` in which we sample input values across the height
     *     and width dimensions in atrous convolution. Defaults to `[1, 1, 1]`.
     *     If `dilations` is a single number, then
     *     `dilationDepth == dilationHeight == dilationWidth`. If it is greater
     *     than 1, then all values of `strides` must be 1.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function conv3d_(x, filter, strides, pad, dataFormat = 'NDHWC', dilations = [1, 1, 1]) {
        const $x = convertToTensor(x, 'x', 'conv3d');
        const $filter = convertToTensor(filter, 'filter', 'conv3d');
        let x5D = $x;
        let reshapedTo5D = false;
        if ($x.rank === 4) {
            reshapedTo5D = true;
            x5D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2], $x.shape[3]]);
        }
        assert(x5D.rank === 5, () => `Error in conv3d: input must be rank 5, but got rank ${x5D.rank}.`);
        assert($filter.rank === 5, () => `Error in conv3d: filter must be rank 5, but got rank ` +
            `${$filter.rank}.`);
        assert(x5D.shape[4] === $filter.shape[3], () => `Error in conv3d: depth of input (${x5D.shape[4]}) must match ` +
            `input depth for filter ${$filter.shape[3]}.`);
        assert(eitherStridesOrDilationsAreOne(strides, dilations), () => 'Error in conv3D: Either strides or dilations must be 1. ' +
            `Got strides ${strides} and dilations '${dilations}'`);
        assert(dataFormat === 'NDHWC', () => `Error in conv3d: got dataFormat of ${dataFormat} but only NDHWC is currently supported.`);
        const inputs = { x: x5D, filter: $filter };
        const attrs = { strides, pad, dataFormat, dilations };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(Conv3D$1, inputs, attrs);
        if (reshapedTo5D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
        }
        return res;
    }
    op({ conv3d_ });

    /**
     * Computes the transposed 3D convolution of a volume, also known as a
     * deconvolution.
     *
     * @param x The input image, of rank 5 or rank 4, of shape
     *   `[batch, depth, height, width, inDepth]`. If rank 4, batch of 1 is assumed.
     * @param filter The filter, rank 4, of shape
     *     `[depth, filterHeight, filterWidth, outDepth, inDepth]`.
     *     `inDepth` must match `inDepth` in `x`.
     * @param outputShape Output shape, of rank 5 or rank 4:
     *     `[batch, depth, height, width, outDepth]`. If rank 3, batch of 1 is
     *    assumed.
     * @param strides The strides of the original convolution:
     *     `[strideDepth, strideHeight, strideWidth]`.
     * @param pad  The type of padding algorithm used in the non-transpose version
     *    of the op.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function conv3dTranspose_(x, filter, outputShape, strides, pad) {
        const $x = convertToTensor(x, 'x', 'conv3dTranspose');
        const $filter = convertToTensor(filter, 'filter', 'conv3dTranspose');
        return conv3DBackpropInput(outputShape, $x, $filter, strides, pad);
    }
    op({ conv3dTranspose_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes cos of the input `tf.Tensor` element-wise: `cos(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, Math.PI / 2, Math.PI * 3 / 4]);
     *
     * x.cos().print();  // or tf.cos(x)
     * ```
     * @param x The input tensor. Must be float32 type.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function cos_(x) {
        const $x = convertToTensor(x, 'x', 'cos', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Cos, inputs);
    }
    const cos = op({ cos_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes hyperbolic cos of the input `tf.Tensor` element-wise: `cosh(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.cosh().print();  // or tf.cosh(x)
     * ```
     * @param x The input tensor. Must be float32 type.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function cosh_(x) {
        const $x = convertToTensor(x, 'x', 'cosh', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Cosh, inputs);
    }
    const cosh = op({ cosh_ });

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the 'License');
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an 'AS IS' BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the cumulative product of a `tf.Tensor` along `axis`.
     *
     * ```js
     * const x = tf.tensor([1, 2, 3, 4]);
     * x.cumprod().print();
     * ```
     * ```js
     * const x = tf.tensor([[1, 2], [3, 4]]);
     * x.cumprod().print();
     * ```
     *
     * @param x The input tensor to cumulatively multiply.
     * @param axis The axis along which to multiply. Optional. Defaults to 0.
     * @param exclusive Whether to perform exclusive cumulative product. Optional.
     *     Defaults to false. If set to true then the product of each tensor entry
     *     does not include its own value, but only the values previous to it
     *     along the specified axis.
     * @param reverse Whether to multiply in the opposite direction. Optional.
     *     Defaults to false.
     *
     * @doc {heading: 'Operations', subheading: 'Scan'}
     */
    function cumprod_(x, axis = 0, exclusive = false, reverse = false) {
        const $x = convertToTensor(x, 'x', 'cumprod');
        const inputs = { x: $x };
        const attrs = { axis, exclusive, reverse };
        return ENGINE.runKernel(Cumprod, inputs, attrs);
    }
    const cumprod = op({ cumprod_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Outputs a vector with length `size` and the same dtype as `weights`.
     *
     * If `weights` are empty, then index `i` stores the number of times the value
     * `i` is counted in `x`. If `weights` are non-empty, then index `i` stores the
     * sum of the value in `weights` at each index where the corresponding value in
     * `x` is `i`.
     *
     * Values in `x` outside of the range [0, size) are ignored.
     *
     * @param x The input int tensor, rank 1 or rank 2.
     * @param weights The weights tensor, must have the same shape as x, or a
     *     length-0 Tensor, in which case it acts as all weights equal to 1.
     * @param size Non-negative integer.
     * @param binaryOutput Optional. Whether the kernel should count the appearance
     *     or number of occurrences. Defaults to False.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function denseBincount_(x, weights, size, binaryOutput = false) {
        const $x = convertToTensor(x, 'x', 'denseBincount');
        const $weights = convertToTensor(weights, 'weights', 'denseBincount');
        assert($x.dtype === 'int32', () => `Error in denseBincount: input ` +
            `dtype must be int32, but got ${$x.dtype}`);
        assert($x.rank <= 2, () => `Error in denseBincount: input must be at most rank 2, but got ` +
            `rank ${$x.rank}.`);
        assert(size >= 0, () => `size must be non-negative, but got ${size}.`);
        assert($weights.size === $x.size || $weights.size === 0, () => `Error in denseBincount: weights must have the same shape as x or ` +
            `0-length, but got x shape: ${$x.shape}, weights shape: ` +
            `${$weights.shape}.`);
        const inputs = { x: $x, weights: $weights };
        const attrs = { size, binaryOutput };
        return ENGINE.runKernel(DenseBincount, inputs, attrs);
    }
    op({ denseBincount_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Rearranges data from depth into blocks of spatial data. More specifically,
     * this op outputs a copy of the input tensor where values from the `depth`
     * dimension are moved in spatial blocks to the `height` and `width` dimensions.
     * The attr `blockSize` indicates the input block size and how the data is
     * moved.
     *
     *  - Chunks of data of size `blockSize * blockSize` from depth are rearranged
     * into non-overlapping blocks of size `blockSize x blockSize`
     *
     *  - The width the output tensor is `inputWidth * blockSize`, whereas the
     * height is `inputHeight * blockSize`
     *
     *  - The Y, X coordinates within each block of the output image are determined
     * by the high order component of the input channel index
     *
     *  - The depth of the input tensor must be divisible by `blockSize *
     * blockSize`
     *
     * The `dataFormat` attr specifies the layout of the input and output tensors
     * with the following options: "NHWC": [ `batch, height, width, channels` ]
     * "NCHW": [ `batch, channels, height, width` ]
     *
     * ```js
     * const x = tf.tensor4d([1, 2, 3, 4], [1, 1, 1, 4]);
     * const blockSize = 2;
     * const dataFormat = "NHWC";
     *
     * tf.depthToSpace(x, blockSize, dataFormat).print();
     * ```
     *
     * @param x The input tensor of rank 4
     * @param blockSIze  An `int` that is `>= 2`. The size of the spatial block
     * @param dataFormat An optional string from: "NHWC", "NCHW". Defaults to "NHWC"
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function depthToSpace_(x, blockSize, dataFormat = 'NHWC') {
        const $x = convertToTensor(x, 'x', 'depthToSpace', 'float32');
        const inputHeight = (dataFormat === 'NHWC') ? $x.shape[1] : $x.shape[2];
        const inputWidth = (dataFormat === 'NHWC') ? $x.shape[2] : $x.shape[3];
        const inputDepth = (dataFormat === 'NHWC') ? $x.shape[3] : $x.shape[1];
        assert(blockSize > 1, () => `blockSize should be > 1 for depthToSpace, but was: ${blockSize}`);
        assert(inputHeight * blockSize >= 0, () => `Negative dimension size caused by overflow when multiplying
    ${inputHeight} and ${blockSize}  for depthToSpace with input shape
    ${$x.shape}`);
        assert(inputWidth * blockSize >= 0, () => `Negative dimension size caused by overflow when multiplying
    ${inputWidth} and ${blockSize} for depthToSpace with input shape
        ${$x.shape}`);
        assert((inputDepth % (blockSize * blockSize) === 0), () => `Dimension size must be evenly divisible by ${blockSize * blockSize} but is ${inputDepth} for depthToSpace with input shape ${$x.shape}`);
        const inputs = { x: $x };
        const attrs = { blockSize, dataFormat };
        return ENGINE.runKernel(DepthToSpace, inputs, attrs);
    }
    op({ depthToSpace_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Depthwise 2D convolution.
     *
     * Given a 4D `input` array and a `filter` array of shape
     * `[filterHeight, filterWidth, inChannels, channelMultiplier]` containing
     * `inChannels` convolutional filters of depth 1, this op applies a
     * different filter to each input channel (expanding from 1 channel to
     * `channelMultiplier` channels for each), then concatenates the results
     * together. The output has `inChannels * channelMultiplier` channels.
     *
     * See
     * [https://www.tensorflow.org/api_docs/python/tf/nn/depthwise_conv2d](
     *     https://www.tensorflow.org/api_docs/python/tf/nn/depthwise_conv2d)
     * for more details.
     *
     * @param x The input tensor, of rank 4 or rank 3, of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is
     * assumed.
     * @param filter The filter tensor, rank 4, of shape
     *     `[filterHeight, filterWidth, inChannels, channelMultiplier]`.
     * @param strides The strides of the convolution: `[strideHeight,
     * strideWidth]`. If strides is a single number, then `strideHeight ==
     * strideWidth`.
     * @param pad The type of padding algorithm.
     *   - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *   - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     in atrous convolution. Defaults to `[1, 1]`. If `rate` is a single
     *     number, then `dilationHeight == dilationWidth`. If it is greater than
     *     1, then all values of `strides` must be 1.
     * @param dataFormat: An optional string from: "NHWC", "NCHW". Defaults to
     *     "NHWC". Specify the data format of the input and output data. With the
     *     default format "NHWC", the data is stored in the order of: [batch,
     *     height, width, channels]. Only "NHWC" is currently supported.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function depthwiseConv2d_(x, filter, strides, pad, dataFormat = 'NHWC', dilations = [1, 1], dimRoundingMode) {
        const $x = convertToTensor(x, 'x', 'depthwiseConv2d', 'float32');
        const $filter = convertToTensor(filter, 'filter', 'depthwiseConv2d', 'float32');
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        assert(x4D.rank === 4, () => `Error in depthwiseConv2d: input must be rank 4, but got ` +
            `rank ${x4D.rank}.`);
        assert($filter.rank === 4, () => `Error in depthwiseConv2d: filter must be rank 4, but got rank ` +
            `${$filter.rank}.`);
        const inChannels = dataFormat === 'NHWC' ? x4D.shape[3] : x4D.shape[1];
        assert(inChannels === $filter.shape[2], () => `Error in depthwiseConv2d: number of input channels ` +
            `(${inChannels}) must match the inChannels dimension in ` +
            `filter ${$filter.shape[2]}.`);
        checkPadOnDimRoundingMode('depthwiseConv2d', pad, dimRoundingMode);
        const inputs = { x: x4D, filter: $filter };
        const attrs = { strides, pad, dataFormat, dilations, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(DepthwiseConv2dNative, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const depthwiseConv2d$2 = op({ depthwiseConv2d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns a diagonal tensor with given diagonal values.
     *
     * Given a diagonal, this operation returns a tensor with the diagonal and
     * everything else padded with zeros.
     *
     * Assume the input has dimensions `[D1,..., Dk]`, then the output is a tensor
     * of rank 2k with dimensions `[D1,..., Dk, D1,..., Dk]`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     *
     * tf.diag(x).print()
     * ```
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4, 5, 6, 6, 8], [4, 2])
     *
     * tf.diag(x).print()
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function diag_(x) {
        const $x = convertToTensor(x, 'x', 'diag');
        const inputs = { x: $x };
        return ENGINE.runKernel(Diag, inputs);
    }
    op({ diag_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the grayscale dilation over the input `x`.
     *
     * @param x The input tensor, rank 3 or rank 4 of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is assumed.
     * @param filter The filter tensor, rank 3, of shape
     *     `[filterHeight, filterWidth, depth]`.
     * @param strides The strides of the sliding window for each dimension of the
     *     input tensor: `[strideHeight, strideWidth]`.
     *     If `strides` is a single number,
     *     then `strideHeight == strideWidth`.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1*1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dataFormat Specify the data format of the input and output data.
     *      Defaults to 'NHWC'. Only 'NHWC' is currently supported. With the
     *      default format "NHWC", the data is stored in the order of: [batch,
     *      height, width, channels].
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     for atrous morphological dilation. Defaults to `[1, 1]`. If `dilations`
     *     is a single number, then `dilationHeight == dilationWidth`. If it is
     *     greater than 1, then all values of `strides` must be 1.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function dilation2d_(x, filter, strides, pad, dilations = [1, 1], dataFormat = 'NHWC') {
        const $x = convertToTensor(x, 'x', 'dilation2d');
        const $filter = convertToTensor(filter, 'filter', 'dilation2d');
        assert($x.rank === 3 || $x.rank === 4, () => `Error in dilation2d: input must be rank 3 or 4, but got rank ` +
            `${$x.rank}.`);
        assert($filter.rank === 3, () => `Error in dilation2d: filter must be rank 3, but got rank ` +
            `${$filter.rank}.`);
        assert(dataFormat === 'NHWC', () => `Error in dilation2d: Only NHWC is currently supported, ` +
            `but got dataFormat of ${dataFormat}`);
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
            reshapedTo4D = true;
        }
        const inputs = { x: x4D, filter: $filter };
        const attrs = { strides, pad, dilations };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(Dilation2D, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    op({ dilation2d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Divides two `tf.Tensor`s element-wise, A / B. Supports broadcasting. Return 0
     * if denominator is 0.
     *
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 9, 16]);
     * const b = tf.tensor1d([1, 2, 3, 4]);
     * const c = tf.tensor1d([0, 0, 0, 0]);
     *
     * a.divNoNan(b).print();  // or tf.divNoNan(a, b)
     * a.divNoNan(c).print();  // or tf.divNoNan(a, c)
     * ```
     *
     * ```js
     * // Broadcast div a with b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(2);
     * const c = tf.scalar(0);
     *
     * a.divNoNan(b).print();  // or tf.divNoNan(a, b)
     * a.divNoNan(c).print();  // or tf.divNoNan(a, c)
     * ```
     *
     * @param a The first tensor as the numerator.
     * @param b The second tensor as the denominator. Must have the same dtype as
     * `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function divNoNan_(a, b) {
        // TODO: Make this into its own kernel.
        let $a = convertToTensor(a, 'a', 'div');
        let $b = convertToTensor(b, 'b', 'div');
        [$a, $b] = makeTypesMatch($a, $b);
        const divResult = div($a, $b);
        const zeros = zerosLike(divResult);
        const bEqualsZero = equal($b, zeros);
        return where(bEqualsZero, zeros, divResult);
    }
    op({ divNoNan_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the dot product of two matrices and/or vectors, `t1` and `t2`.
     *
     * ```js
     * const a = tf.tensor1d([1, 2]);
     * const b = tf.tensor2d([[1, 2], [3, 4]]);
     * const c = tf.tensor2d([[1, 2, 3], [4, 5, 6]]);
     *
     * a.dot(b).print();  // or tf.dot(a, b)
     * b.dot(a).print();
     * b.dot(c).print();
     * ```
     * @param t1 The first tensor in the dot operation.
     * @param t2 The second tensor in the dot operation.
     *
     * @doc {heading: 'Operations', subheading: 'Matrices'}
     */
    function dot_(t1, t2) {
        const $t1 = convertToTensor(t1, 't1', 'dot');
        const $t2 = convertToTensor(t2, 't2', 'dot');
        assert(($t1.rank === 1 || $t1.rank === 2) && ($t2.rank === 1 || $t2.rank === 2), () => `Error in dot: inputs must all be rank 1 or 2, but got ranks ` +
            `${$t1.rank} and ${$t2.rank}.`);
        const t1Inner = ($t1.rank === 1 ? $t1.size : $t1.shape[1]);
        const t2Inner = ($t2.rank === 1 ? $t2.size : $t2.shape[0]);
        assert(t1Inner === t2Inner, () => `Error in dot: inner dimensions of inputs must match, but got ` +
            `${t1Inner} and ${t2Inner}.`);
        if ($t1.rank === 1 && $t2.rank === 1) {
            const t12D = reshape$1($t1, [1, -1]);
            const t22D = reshape$1($t2, [-1, 1]);
            const t1t2 = matMul(t12D, t22D);
            return reshape$1(t1t2, []);
        }
        else if ($t1.rank === 1 && $t2.rank === 2) {
            const t12D = reshape$1($t1, [1, -1]);
            const t22D = reshape$1($t2, [$t2.shape[0], $t2.shape[1]]);
            const t1t2 = matMul(t12D, t22D);
            return reshape$1(t1t2, [t1t2.size]);
        }
        else if ($t1.rank === 2 && $t2.rank === 1) {
            const t22D = reshape$1($t2, [-1, 1]);
            const t1t2 = matMul($t1, t22D);
            return reshape$1(t1t2, [t1t2.size]);
        }
        else {
            const t22D = reshape$1($t2, [$t2.shape[0], $t2.shape[1]]);
            const t1t2 = matMul($t1, t22D);
            return t1t2;
        }
    }
    op({ dot_ });

    /**
     * @license
     * Copyright 2021 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Tensor contraction over specified indices and outer product.
     *
     * `einsum` allows defining Tensors by defining their element-wise computation.
     * This computation is based on
     * [Einstein summation](https://en.wikipedia.org/wiki/Einstein_notation).
     *
     * Some special cases include:
     *
     * Matrix multiplication:
     * ```js
     * const x = tf.tensor2d([[1, 2, 3], [4, 5, 6]]);
     * const y = tf.tensor2d([[0, 1], [2, 3], [4, 5]]);
     * x.print();
     * y.print();
     * tf.einsum('ij,jk->ik', x, y).print();
     * ```
     *
     * Dot product:
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     * const y = tf.tensor1d([0, 1, 2]);
     * x.print();
     * y.print();
     * tf.einsum('i,i->', x, y).print();
     * ```
     *
     * Batch dot product:
     * ```js
     * const x = tf.tensor2d([[1, 2, 3], [4, 5, 6]]);
     * const y = tf.tensor2d([[0, 1, 2], [3, 4, 5]]);
     * x.print();
     * y.print();
     * tf.einsum('bi,bi->b', x, y).print();
     * ```
     *
     * Outer prouduct:
     * ```js
     * const x = tf.tensor1d([1, 3, 5]);
     * const y = tf.tensor1d([2, 4, 6]);
     * x.print();
     * y.print();
     * tf.einsum('i,j->ij', x, y).print();
     * ```
     *
     * Matrix transpose:
     * ```js
     * const x = tf.tensor2d([[1, 2], [3, 4]]);
     * x.print();
     * tf.einsum('ij->ji', x).print();
     * ```
     *
     * Batch matrix transpose:
     * ```js
     * const x = tf.tensor3d([[[1, 2], [3, 4]], [[-1, -2], [-3, -4]]]);
     * x.print();
     * tf.einsum('bij->bji', x).print();
     * ```
     *
     * Limitations:
     *
     * This implementation of einsum has the following limitations:
     *
     * - Does not support >2 input tensors.
     * - Does not support duplicate axes for any given input tensor. E.g., equation
     *   'ii->' is not supported.
     * - The `...` notation is not supported.
     *
     * @param equation a string describing the contraction, in the same format as
     * [numpy.einsum](https://numpy.org/doc/stable/reference/generated/numpy.einsum.html).
     * @param tensors the input(s) to contract (each one a Tensor), whose shapes
     *     should be consistent with equation.
     * @returns The output tensor.
     *
     * @doc {heading: 'Tensors', subheading: 'Matrices'}
     */
    function einsum_(equation, ...tensors) {
        const $tensors = tensors.map((t, i) => convertToTensor(t, `tensors${i}`, 'einsum'));
        const attrs = { equation };
        return ENGINE.runKernel(Einsum, $tensors, attrs);
    }
    op({ einsum_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes exponential linear element-wise: `x > 0 ? x : (e ^ x) - 1`.
     *
     * ```js
     * const x = tf.tensor1d([-1, 1, -3, 2]);
     *
     * x.elu().print();  // or tf.elu(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function elu_(x) {
        const $x = convertToTensor(x, 'x', 'elu', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Elu$1, inputs);
    }
    const elu$1 = op({ elu_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes Gauss error function of the input `tf.Tensor` element-wise:
     * `erf(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, .1, -.1, .7]);
     *
     * x.erf().print(); // or tf.erf(x);
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function erf_(x) {
        let $x = convertToTensor(x, 'x', 'erf');
        assert($x.dtype === 'int32' || $x.dtype === 'float32', () => 'Input dtype must be `int32` or `float32`.');
        if ($x.dtype === 'int32') {
            $x = cast($x, 'float32');
        }
        const inputs = { x: $x };
        return ENGINE.runKernel(Erf, inputs);
    }
    op({ erf_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the maximum of elements across dimensions of a `tf.Tensor`.
     *
     * Reduces the input along the dimensions given in `axes`. Unless `keepDims`
     * is true, the rank of the `tf.Tensor` is reduced by 1 for each entry in
     * `axes`. If `keepDims` is true, the reduced dimensions are retained with
     * length 1. If `axes` has no entries, all dimensions are reduced, and a
     * `tf.Tensor` with a single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.max().print();  // or tf.max(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.max(axis).print();  // or tf.max(x, axis)
     * ```
     *
     * @param x The input tensor.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function max_(x, axis = null, keepDims = false) {
        const $x = convertToTensor(x, 'x', 'max');
        const inputs = { x: $x };
        const attrs = { reductionIndices: axis, keepDims };
        return ENGINE.runKernel(Max, inputs, attrs);
    }
    const max = op({ max_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the minimum value from the input.
     *
     * Reduces the input along the dimensions given in `axes`. Unless `keepDims`
     * is true, the rank of the array is reduced by 1 for each entry in `axes`.
     * If `keepDims` is true, the reduced dimensions are retained with length 1.
     * If `axes` has no entries, all dimensions are reduced, and an array with a
     * single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.min().print();  // or tf.min(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.min(axis).print();  // or tf.min(x, axis)
     * ```
     *
     * @param x The input Tensor.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function min_(x, axis = null, keepDims = false) {
        const $x = convertToTensor(x, 'x', 'min');
        const inputs = { x: $x };
        const attrs = { axis, keepDims };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(Min, inputs, attrs);
    }
    const min = op({ min_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the norm of scalar, vectors, and matrices.
     * This function can compute several different vector norms (the 1-norm, the
     * Euclidean or 2-norm, the inf-norm, and in general the p-norm for p > 0)
     * and matrix norms (Frobenius, 1-norm, and inf-norm).
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     *
     * x.norm().print();  // or tf.norm(x)
     * ```
     *
     * @param x The input array.
     * @param ord Optional. Order of the norm. Supported norm types are
     * following:
     *
     *  | ord        | norm for matrices         | norm for vectors
     *  |------------|---------------------------|---------------------
     *  |'euclidean' |Frobenius norm             |2-norm
     *  |'fro'       |Frobenius norm	           |
     *  |Infinity    |max(sum(abs(x), axis=1))   |max(abs(x))
     *  |-Infinity   |min(sum(abs(x), axis=1))   |min(abs(x))
     *  |1           |max(sum(abs(x), axis=0))   |sum(abs(x))
     *  |2           |                           |sum(abs(x)^2)^(1/2)
     *
     * @param axis Optional. If axis is null (the default), the input is
     * considered a vector and a single vector norm is computed over the entire
     * set of values in the Tensor, i.e. norm(x, ord) is equivalent
     * to norm(x.reshape([-1]), ord). If axis is an integer, the input
     * is considered a batch of vectors, and axis determines the axis in x
     * over which to compute vector norms. If axis is a 2-tuple of integer it is
     * considered a batch of matrices and axis determines the axes in NDArray
     * over which to compute a matrix norm.
     * @param keepDims Optional. If true, the norm has the same dimensionality
     * as the input.
     *
     * @doc {heading: 'Operations', subheading: 'Matrices'}
     */
    function norm_(x, ord = 'euclidean', axis = null, keepDims = false) {
        x = convertToTensor(x, 'x', 'norm');
        const norm = normImpl(x, ord, axis);
        let keepDimsShape = norm.shape;
        if (keepDims) {
            const axes = parseAxisParam(axis, x.shape);
            keepDimsShape = expandShapeToKeepDim(norm.shape, axes);
        }
        return reshape$1(norm, keepDimsShape);
    }
    function normImpl(x, p, axis = null) {
        if (x.rank === 0) {
            return abs(x);
        }
        // consider vector when no axis is specified
        if (x.rank !== 1 && axis === null) {
            return normImpl(reshape$1(x, [-1]), p, axis);
        }
        // vector
        if (x.rank === 1 || typeof axis === 'number' ||
            Array.isArray(axis) && axis.length === 1) {
            if (p === 1) {
                return sum(abs(x), axis);
            }
            if (p === Infinity) {
                return max(abs(x), axis);
            }
            if (p === -Infinity) {
                return min(abs(x), axis);
            }
            if (p === 'euclidean' || p === 2) {
                // norm(x, 2) = sum(abs(xi) ^ 2) ^ 1/2
                return sqrt(sum(pow(abs(x), scalar(2, 'int32')), axis));
            }
            throw new Error(`Error in norm: invalid ord value: ${p}`);
        }
        // matrix (assumption axis[0] < axis[1])
        if (Array.isArray(axis) && axis.length === 2) {
            if (p === 1) {
                return max(sum(abs(x), axis[0]), axis[1] - 1);
            }
            if (p === Infinity) {
                return max(sum(abs(x), axis[1]), axis[0]);
            }
            if (p === -Infinity) {
                return min(sum(abs(x), axis[1]), axis[0]);
            }
            if (p === 'fro' || p === 'euclidean') {
                // norm(x) = sqrt(sum(pow(x, 2)))
                return sqrt(sum(square(x), axis));
            }
            throw new Error(`Error in norm: invalid ord value: ${p}`);
        }
        throw new Error(`Error in norm: invalid axis: ${axis}`);
    }
    const norm = op({ norm_ });

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the Euclidean norm of scalar, vectors, and matrices.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     *
     * x.euclideanNorm().print();  // or tf.euclideanNorm(x)
     * ```
     *
     * @param x The input array.
     * @param axis Optional. If axis is null (the default), the input is
     * considered a vector and a single vector norm is computed over the entire
     * set of values in the Tensor, i.e. euclideanNorm(x) is equivalent
     * to euclideanNorm(x.reshape([-1])). If axis is an integer, the input
     * is considered a batch of vectors, and axis determines the axis in x
     * over which to compute vector norms. If axis is a 2-tuple of integer it is
     * considered a batch of matrices and axis determines the axes in NDArray
     * over which to compute a matrix norm.
     * @param keepDims Optional. If true, the norm has the same dimensionality
     * as the input.
     *
     * @doc {heading: 'Operations', subheading: 'Matrices'}
     */
    function euclideanNorm_(x, axis = null, keepDims = false) {
        return norm(x, 'euclidean', axis, keepDims);
    }
    op({ euclideanNorm_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns a `tf.Tensor` that has expanded rank, by inserting a dimension
     * into the tensor's shape.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     * const axis = 1;
     * x.expandDims(axis).print();
     * ```
     *
     * @param x The input tensor whose dimensions are to be expanded.
     * @param axis The dimension index at which to insert shape of `1`. Defaults
     *     to 0 (the first dimension).
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function expandDims_(x, axis = 0) {
        const $x = convertToTensor(x, 'x', 'expandDims', 'string_or_numeric');
        assert(axis <= $x.rank, () => 'Axis must be <= rank of the tensor');
        const inputs = { input: $x };
        const attrs = { dim: axis };
        return ENGINE.runKernel(ExpandDims, inputs, attrs);
    }
    const expandDims = op({ expandDims_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes exponential of the input `tf.Tensor` minus one element-wise.
     * `e ^ x - 1`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, -3]);
     *
     * x.expm1().print();  // or tf.expm1(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function expm1_(x) {
        const $x = convertToTensor(x, 'x', 'expm1');
        const inputs = { x: $x };
        return ENGINE.runKernel(Expm1, inputs);
    }
    op({ expm1_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Create an identity matrix.
     *
     * @param numRows Number of rows.
     * @param numColumns Number of columns. Defaults to `numRows`.
     * @param batchShape If provided, will add the batch shape to the beginning
     *   of the shape of the returned `tf.Tensor` by repeating the identity
     *   matrix.
     * @param dtype Data type.
     * @returns Identity matrix of the specified size and data type, possibly
     *   with batch repetition if `batchShape` is specified.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function eye_(numRows, numColumns, batchShape, dtype = 'float32') {
        if (numColumns == null) {
            numColumns = numRows;
        }
        const buff = buffer([numRows, numColumns], dtype);
        const n = numRows <= numColumns ? numRows : numColumns;
        for (let i = 0; i < n; ++i) {
            buff.set(1, i, i);
        }
        const out = reshape$1(buff.toTensor(), [numRows, numColumns]);
        if (batchShape == null) {
            return out;
        }
        else {
            if (batchShape.length === 1) {
                return tile(expandDims(out, 0), [batchShape[0], 1, 1]);
            }
            else if (batchShape.length === 2) {
                // tslint:disable-next-line:no-unnecessary-type-assertion
                return tile(expandDims(expandDims(out, 0), 0), [batchShape[0], batchShape[1], 1, 1]);
            }
            else if (batchShape.length === 3) {
                // tslint:disable-next-line:no-unnecessary-type-assertion
                return tile(expandDims(expandDims(expandDims(out, 0), 0), 0), [
                    batchShape[0], batchShape[1], batchShape[2], 1, 1
                ]);
            }
            else {
                throw new Error(`eye() currently supports only 1D and 2D ` +
                    // tslint:disable-next-line:no-any
                    `batchShapes, but received ${batchShape.length}D.`);
            }
        }
    }
    const eye = op({ eye_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Gather slices from tensor `x`'s axis `axis` according to `indices`.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     * const indices = tf.tensor1d([1, 3, 3], 'int32');
     *
     * x.gather(indices).print();
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     * const indices = tf.tensor1d([1, 1, 0], 'int32');
     *
     * x.gather(indices).print();
     * ```
     * @param x The input tensor whose slices are to be gathered.
     * @param indices The indices of the values to extract.
     * @param axis The axis over which to select values. Defaults to 0.
     * @param batchDims Optional. The number of batch dimensions. It must be less
     *     than or equal to rank(indices). Defaults to 0.
     *     The output tensor will have shape of
     *     `x.shape[:axis] + indices.shape[batchDims:] + x.shape[axis + 1:]`
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function gather_(x, indices, axis = 0, batchDims = 0) {
        const $x = convertToTensor(x, 'x', 'gather');
        const $indices = convertToTensor(indices, 'indices', 'gather', 'int32');
        const inputs = { x: $x, indices: $indices };
        const attrs = { axis, batchDims };
        return ENGINE.runKernel(GatherV2, inputs, attrs);
    }
    const gather = op({ gather_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns which elements of x are finite.
     *
     * ```js
     * const x = tf.tensor1d([NaN, Infinity, -Infinity, 0, 1]);
     *
     * x.isFinite().print();  // or tf.isNaN(x)
     * ```
     * @param x The input Tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function isFinite_(x) {
        const $x = convertToTensor(x, 'x', 'isFinite');
        const inputs = { x: $x };
        return ENGINE.runKernel(IsFinite, inputs);
    }
    op({ isFinite_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns which elements of x are Infinity or -Infinity.
     *
     * ```js
     * const x = tf.tensor1d([NaN, Infinity, -Infinity, 0, 1]);
     *
     * x.isInf().print();  // or tf.isNaN(x)
     * ```
     * @param x The input Tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function isInf_(x) {
        const $x = convertToTensor(x, 'x', 'isInf');
        const inputs = { x: $x };
        return ENGINE.runKernel(IsInf, inputs);
    }
    op({ isInf_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns which elements of x are NaN.
     *
     * ```js
     * const x = tf.tensor1d([NaN, Infinity, -Infinity, 0, 1]);
     *
     * x.isNaN().print();  // or tf.isNaN(x)
     * ```
     * @param x The input Tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function isNaN_(x) {
        const $x = convertToTensor(x, 'x', 'isNaN');
        const inputs = { x: $x };
        return ENGINE.runKernel(IsNan, inputs);
    }
    op({ isNaN_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes leaky rectified linear element-wise.
     *
     * See
     * [http://web.stanford.edu/~awni/papers/relu_hybrid_icml2013_final.pdf](
     *     http://web.stanford.edu/~awni/papers/relu_hybrid_icml2013_final.pdf)
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 4]);
     *
     * x.leakyRelu(0.1).print();  // or tf.leakyRelu(x, 0.1)
     * ```
     * @param x The input tensor.
     * @param alpha The scaling factor for negative values, defaults to 0.2.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function leakyRelu_(x, alpha = 0.2) {
        const $x = convertToTensor(x, 'x', 'leakyRelu');
        const inputs = { x: $x };
        const attrs = { alpha };
        return ENGINE.runKernel(LeakyRelu, inputs, attrs);
    }
    const leakyRelu = op({ leakyRelu_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Normalizes the activation of a local neighborhood across or within
     * channels.
     *
     * @param x The input tensor. The 4-D input tensor is treated as a 3-D array
     *     of 1D vectors (along the last dimension), and each vector is
     *     normalized independently.
     * @param depthRadius The number of adjacent channels in the 1D normalization
     *     window.
     * @param bias A constant bias term for the basis.
     * @param alpha A scale factor, usually positive.
     * @param beta An exponent.
     *
     * @doc {heading: 'Operations', subheading: 'Normalization'}
     */
    function localResponseNormalization_(x, depthRadius = 5, bias = 1, alpha = 1, beta = 0.5) {
        const $x = convertToTensor(x, 'x', 'localResponseNormalization');
        assert($x.rank === 4 || $x.rank === 3, () => `Error in localResponseNormalization: x must be rank 3 or 4 but got
               rank ${$x.rank}.`);
        assert(isInt(depthRadius), () => `Error in localResponseNormalization: depthRadius must be an ` +
            `integer but got depthRadius ${depthRadius}.`);
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        const inputs = { x: x4D };
        const attrs = { depthRadius, bias, alpha, beta };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(LRN, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        else {
            return res;
        }
    }
    op({ localResponseNormalization_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes natural logarithm of the input `tf.Tensor` plus one
     * element-wise: `ln(1 + x)`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, Math.E - 1]);
     *
     * x.log1p().print();  // or tf.log1p(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function log1p_(x) {
        const $x = convertToTensor(x, 'x', 'log1p');
        const inputs = { x: $x };
        return ENGINE.runKernel(Log1p, inputs);
    }
    const log1p = op({ log1p_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes and returns the gradient of f(x) with respect to the list of
     * trainable variables provided by `varList`. If no list is provided, it
     * defaults to all trainable variables.
     *
     * ```js
     * const a = tf.variable(tf.tensor1d([3, 4]));
     * const b = tf.variable(tf.tensor1d([5, 6]));
     * const x = tf.tensor1d([1, 2]);
     *
     * // f(a, b) = a * x ^ 2 + b * x
     * const f = () => a.mul(x.square()).add(b.mul(x)).sum();
     * // df/da = x ^ 2, df/db = x
     * const {value, grads} = tf.variableGrads(f);
     *
     * Object.keys(grads).forEach(varName => grads[varName].print());
     * ```
     *
     * @param f The function to execute. f() should return a scalar.
     * @param varList The list of variables to compute the gradients with respect
     *     to. Defaults to all trainable variables.
     * @returns An object with the following keys and values:
     *   - `value`: The value of the function `f`.
     *   - `grads`: A map from the names of the variables to the gradients.
     *     If the `varList` argument is provided explicitly and contains a subset of
     *     non-trainable variables, this map in the return value will contain keys
     *     that map the names of the non-trainable variables to `null`.
     *
     * @doc {heading: 'Training', subheading: 'Gradients'}
     */
    function variableGrads(f, varList) {
        assert(isFunction(f), () => 'The f passed in variableGrads(f) must be a function');
        assert(varList == null ||
            Array.isArray(varList) && varList.every(v => v instanceof Variable), () => 'The varList passed in variableGrads(f, varList) must be an array ' +
            'of variables');
        const specifiedVarList = varList != null;
        if (!specifiedVarList) {
            // Get all of the trainable variables.
            varList = [];
            for (const varName in ENGINE.registeredVariables) {
                varList.push(ENGINE.registeredVariables[varName]);
            }
        }
        const specifiedNonTrainable = specifiedVarList ? varList.filter(variable => !variable.trainable) : null;
        // Prune non-trainable variables.
        const originalVarCount = varList.length;
        varList = varList.filter(variable => variable.trainable);
        assert(varList.length > 0, () => `variableGrads() expects at least one of the input variables to ` +
            `be trainable, but none of the ${originalVarCount} variables is ` +
            `trainable.`);
        const allowNoGradients = true;
        const { value, grads } = ENGINE.gradients(f, varList, null, allowNoGradients);
        assert(grads.some(g => g != null), () => 'Cannot find a connection between any variable and the result of ' +
            'the loss function y=f(x). Please make sure the operations that ' +
            'use variables are inside the function f passed to minimize().');
        assert(value.rank === 0, () => `The f passed in variableGrads(f) must return a scalar, but it ` +
            `returned a rank-${value.rank} tensor`);
        const namedGrads = {};
        varList.forEach((v, i) => {
            if (grads[i] != null) {
                namedGrads[v.name] = grads[i];
            }
        });
        if (specifiedNonTrainable != null) {
            // If varList is explicitly provided and contains non-trainable values,
            // add them to the returned gradients with `null` values.
            specifiedNonTrainable.forEach(v => namedGrads[v.name] = null);
        }
        return { value, grads: namedGrads };
    }
    /**
     * Overrides the gradient computation of a function `f`.
     *
     * Takes a function
     * `f(...inputs, save) => {value: Tensor, gradFunc: (dy, saved) => Tensor[]}`
     * and returns another function `g(...inputs)` which takes the same inputs as
     * `f`. When called, `g` returns `f().value`. In backward mode, custom gradients
     * with respect to each input of `f` are computed using `f().gradFunc`.
     *
     * The `save` function passed to `f` should be used for saving tensors needed
     * in the gradient. And the `saved` passed to the `gradFunc` is a
     * `NamedTensorMap`, which contains those saved tensors.
     *
     * ```js
     * const customOp = tf.customGrad((x, save) => {
     *   // Save x to make sure it's available later for the gradient.
     *   save([x]);
     *   // Override gradient of our custom x ^ 2 op to be dy * abs(x);
     *   return {
     *     value: x.square(),
     *     // Note `saved.x` which points to the `x` we saved earlier.
     *     gradFunc: (dy, saved) => [dy.mul(saved[0].abs())]
     *   };
     * });
     *
     * const x = tf.tensor1d([-1, -2, 3]);
     * const dx = tf.grad(x => customOp(x));
     *
     * console.log(`f(x):`);
     * customOp(x).print();
     * console.log(`f'(x):`);
     * dx(x).print();
     * ```
     *
     * @param f The function to evaluate in forward mode, which should return
     *     `{value: Tensor, gradFunc: (dy, saved) => Tensor[]}`, where `gradFunc`
     *     returns the custom gradients of `f` with respect to its inputs.
     *
     * @doc {heading: 'Training', subheading: 'Gradients'}
     */
    function customGrad(f) {
        return ENGINE.customGrad(f);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes softplus of the input `tf.Tensor` element-wise: `log(exp(x) + 1)`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.softplus().print();  // or tf.softplus(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function softplus_(x) {
        const $x = convertToTensor(x, 'x', 'softplus');
        const inputs = { x: $x };
        return ENGINE.runKernel(Softplus$1, inputs);
    }
    const softplus = op({ softplus_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes log sigmoid of the input `tf.Tensor` element-wise:
     * `logSigmoid(x)`. For numerical stability, we use `-tf.softplus(-x)`.
     *
     * ```js
     * const x = tf.tensor1d([0, 1, -1, .7]);
     *
     * x.logSigmoid().print();  // or tf.logSigmoid(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function logSigmoid_(x) {
        const $x = convertToTensor(x, 'x', 'logSigmoid');
        // Use a custom gradient to maintain previous implementation.
        // There is no LogSigmoid kernel in TF so we can't use engine.runKernel
        // directly
        const customOp = customGrad((x) => {
            // TODO(yassogba) we can remove the chained softplus call here only
            // after backends have modualrized softplus at which point we can call
            // engine runKernel(..., Sotfplus, ...) directly.
            const value = neg(softplus(neg(x)));
            const gradFunc = (dy) => {
                const derX = mul(dy, sigmoid(neg(x)));
                return derX;
            };
            return { value, gradFunc };
        });
        return customOp($x);
    }
    op({ logSigmoid_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the log softmax.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     *
     * a.logSoftmax().print();  // or tf.logSoftmax(a)
     * ```
     *
     * ```js
     * const a = tf.tensor2d([2, 4, 6, 1, 2, 3], [2, 3]);
     *
     * a.logSoftmax().print();  // or tf.logSoftmax(a)
     * ```
     *
     * @param logits The logits array.
     * @param axis The dimension softmax would be performed on. Defaults to `-1`
     *     which indicates the last dimension.
     *
     * @doc {heading: 'Operations', subheading: 'Normalization'}
     */
    function logSoftmax_(logits, axis = -1) {
        const $logits = convertToTensor(logits, 'logits', 'logSoftmax');
        if (axis === -1) {
            axis = $logits.rank - 1;
        }
        if (axis !== $logits.rank - 1) {
            throw Error('Log Softmax along a non-last dimension is not yet supported. ' +
                `Logits was rank ${$logits.rank} and axis was ${axis}`);
        }
        // const forward: ForwardFunc<Tensor> = (backend, save) => {
        //   const keepDims = true;
        //   const xMax = max(logits, axis, true);
        //   const shifted = sub(logits, xMax);
        //   const value =
        //       sub(cast(shifted, 'float32'), log(sum(exp(shifted), axis,
        //       keepDims)));
        //   save([value]);
        //   return value;
        // };
        // Use a custom gradient for numerical stability.
        const customOp = customGrad((logits, save) => {
            const keepDims = true;
            const xMax = max(logits, axis, true);
            const shifted = sub(logits, xMax);
            const value = sub(cast(shifted, 'float32'), log(sum(exp(shifted), axis, keepDims)));
            save([value]);
            const gradFunc = (dy, saved) => {
                const [value] = saved;
                const keepDims = true;
                const softmax = exp(value);
                return sub(dy, mul(sum(dy, axis, keepDims), softmax));
            };
            return { value, gradFunc };
        });
        return customOp($logits);
        // TODO Use Engine.runKernel when CPU/WebGL/WASM backends implement this.
        // const inputs: LogSoftmaxInputs = {logits: $logits};
        // const attrs: LogSoftmaxAttrs = {axis};
        // return ENGINE.runKernel(
        //            LogSoftmax, inputs as {} as NamedTensorMap,
        //            attrs as {} as NamedAttrMap);
    }
    op({ logSoftmax_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the log(sum(exp(elements across the reduction dimensions))).
     *
     * Reduces the input along the dimensions given in `axis`. Unless `keepDims`
     * is true, the rank of the array is reduced by 1 for each entry in `axis`.
     * If `keepDims` is true, the reduced dimensions are retained with length 1.
     * If `axis` has no entries, all dimensions are reduced, and an array with a
     * single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.logSumExp().print();  // or tf.logSumExp(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.logSumExp(axis).print();  // or tf.logSumExp(a, axis)
     * ```
     * @param x The input tensor.
     * @param axis The dimension(s) to reduce. If null (the default),
     *     reduces all dimensions.
     * @param keepDims If true, retains reduced dimensions with length
     *     of 1. Defaults to false.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function logSumExp_(x, axis = null, keepDims = false) {
        const $x = convertToTensor(x, 'x', 'logSumExp');
        const axes = parseAxisParam(axis, $x.shape);
        const xMax = max($x, axes, true /* keepDims */);
        const a = sub($x, xMax);
        const b = exp(a);
        const c = sum(b, axes);
        const d = log(c);
        const res = add$1(reshape$1(xMax, d.shape), d);
        if (keepDims) {
            const newShape = expandShapeToKeepDim(res.shape, axes);
            return reshape$1(res, newShape);
        }
        return res;
    }
    const logSumExp = op({ logSumExp_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of `NOT x` element-wise.
     *
     * ```js
     * const a = tf.tensor1d([false, true], 'bool');
     *
     * a.logicalNot().print();
     * ```
     *
     * @param x The input tensor. Must be of dtype 'bool'.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function logicalNot_(x) {
        const $x = convertToTensor(x, 'x', 'logicalNot', 'bool');
        const inputs = { x: $x };
        return ENGINE.runKernel(LogicalNot, inputs);
    }
    const logicalNot = op({ logicalNot_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of `a OR b` element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([false, false, true, true], 'bool');
     * const b = tf.tensor1d([false, true, false, true], 'bool');
     *
     * a.logicalOr(b).print();
     * ```
     * @param a The first input tensor. Must be of dtype bool.
     * @param b The second input tensor. Must be of dtype bool.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function logicalOr_(a, b) {
        const $a = convertToTensor(a, 'a', 'logicalOr', 'bool');
        const $b = convertToTensor(b, 'b', 'logicalOr', 'bool');
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(LogicalOr, inputs);
    }
    const logicalOr = op({ logicalOr_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of `a XOR b` element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([false, false, true, true], 'bool');
     * const b = tf.tensor1d([false, true, false, true], 'bool');
     *
     * a.logicalXor(b).print();
     * ```
     *
     * @param a The first input tensor. Must be of dtype bool.
     * @param b The second input tensor. Must be of dtype bool.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function logicalXor_(a, b) {
        const $a = convertToTensor(a, 'a', 'logicalXor', 'bool');
        const $b = convertToTensor(b, 'b', 'logicalXor', 'bool');
        assertAndGetBroadcastShape($a.shape, $b.shape);
        // x ^ y = (x | y) & ~(x & y)
        return logicalAnd(logicalOr(a, b), logicalNot(logicalAnd(a, b)));
    }
    op({ logicalXor_ });

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    const INT32_MAX = 2147483648;
    /**
     * Searches for where a value would go in a sorted sequence.
     *
     * This is not a method for checking containment (like javascript in).
     *
     * The typical use case for this operation is "binning", "bucketing", or
     * "discretizing". The values are assigned to bucket-indices based on the edges
     * listed in 'sortedSequence'. This operation returns the bucket-index for each
     * value.
     *
     * The side argument controls which index is returned if a value lands exactly
     * on an edge.
     *
     * The axis is not settable for this operation. It always operates on the
     * innermost dimension (axis=-1). The operation will accept any number of outer
     * dimensions.
     *
     * Note: This operation assumes that 'sortedSequence' is sorted along the
     * innermost axis, maybe using 'sort(..., axis=-1)'. If the sequence is not
     * sorted no error is raised and the content of the returned tensor is not well
     * defined.
     *
     * ```js
     * const edges = tf.tensor1d([-1, 3.3, 9.1, 10.0]);
     * let values = tf.tensor1d([0.0, 4.1, 12.0]);
     * const result1 = tf.searchSorted(edges, values, 'left');
     * result1.print(); // [1, 2, 4]
     *
     * const seq = tf.tensor1d([0, 3, 9, 10, 10]);
     * values = tf.tensor1d([0, 4, 10]);
     * const result2 = tf.searchSorted(seq, values, 'left');
     * result2.print(); // [0, 2, 3]
     * const result3 = tf.searchSorted(seq, values, 'right');
     * result3.print(); // [1, 2, 5]
     *
     * const sortedSequence = tf.tensor2d([[0., 3., 8., 9., 10.],
     *                                     [1., 2., 3., 4., 5.]]);
     * values = tf.tensor2d([[9.8, 2.1, 4.3],
     *                       [0.1, 6.6, 4.5, ]]);
     * const result4 = tf.searchSorted(sortedSequence, values, 'left');
     * result4.print(); // [[4, 1, 2], [0, 5, 4]]
     * ```
     * @param sortedSequence: N-D. Sorted sequence.
     * @param values: N-D. Search values.
     * @param side: 'left'|'right'. Defaults to 'left'. 'left' corresponds to lower
     *     bound and 'right' to upper bound.
     * @return An N-D int32 tensor the size of values containing the result of
     *     applying either lower bound or upper bound (depending on side) to each
     *     value. The result is not a global index to the entire Tensor, but the
     *     index in the last dimension.
     * @doc {heading: 'Operations', subheading: 'Evaluation'}
     */
    function searchSorted_(sortedSequence, values, side = 'left') {
        const $sortedSequence = convertToTensor(sortedSequence, 'sortedSequence', 'searchSorted');
        const $values = convertToTensor(values, 'values', 'searchSorted');
        const sequenceSize = $sortedSequence.shape[$sortedSequence.shape.length - 1];
        const valuesSize = $values.shape[$values.shape.length - 1];
        const $sortedSequence2D = reshape$1($sortedSequence, [-1, sequenceSize]);
        const $values2D = reshape$1($values, [-1, valuesSize]);
        if ($sortedSequence2D.rank < 2) {
            throw new Error(`Sorted input argument must be at least 2-dimensional`);
        }
        if ($sortedSequence2D.shape[0] !== $values2D.shape[0]) {
            throw new Error(`Leading dimension of 'sortedSequence' and 'values' must match.`);
        }
        if (sizeFromShape($values2D.shape) >= INT32_MAX) {
            throw new Error(`values tensor size must less than ${INT32_MAX}`);
        }
        if ($sortedSequence2D.shape[1] >= INT32_MAX) {
            throw new Error(`trailing dim_size must less than ${INT32_MAX} for int32 output type, was ${$sortedSequence2D.shape[1]}`);
        }
        const inputs = {
            sortedSequence: $sortedSequence2D,
            values: $values2D,
        };
        const attrs = { side };
        return ENGINE.runKernel(SearchSorted, inputs, attrs);
    }
    op({ searchSorted_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the 2D max pooling of an image.
     *
     * @param x The input tensor, of rank 4 or rank 3 of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is assumed.
     * @param filterSize The filter size: `[filterHeight, filterWidth]`. If
     *     `filterSize` is a single number, then `filterHeight == filterWidth`.
     * @param strides The strides of the pooling: `[strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     in dilated pooling. Defaults to `[1, 1]`. If `dilations` is a single
     *     number, then `dilationHeight == dilationWidth`. If it is greater than
     *     1, then all values of `strides` must be 1.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     */
    function maxPool_(x, filterSize, strides, pad, dimRoundingMode) {
        const $x = convertToTensor(x, 'x', 'maxPool');
        const dilations = 1;
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        assert(x4D.rank === 4, () => `Error in maxPool: input must be rank 4 but got rank ${x4D.rank}.`);
        assert(eitherStridesOrDilationsAreOne(strides, dilations), () => 'Error in maxPool: Either strides or dilations must be 1. ' +
            `Got strides ${strides} and dilations '${dilations}'`);
        checkPadOnDimRoundingMode('maxPool', pad, dimRoundingMode);
        const inputs = { x: x4D };
        const attrs = { filterSize, strides, pad, dimRoundingMode };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(MaxPool, inputs, attrs);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    const maxPool = op({ maxPool_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the 3D max pooling.
     *
     * ```js
     * const x = tf.tensor5d([1, 2, 3, 4, 5, 6, 7, 8], [1, 2, 2, 2, 1]);
     * const result = tf.maxPool3d(x, 2, 1, 'valid');
     * result.print();
     * ```
     *
     * @param x The input tensor, of rank 5 or rank 4 of shape
     *     `[batch, depth, height, width, inChannels]`.
     * @param filterSize The filter size:
     *     `[filterDepth, filterHeight, filterWidth]`.
     *     If `filterSize` is a single number,
     *     then `filterDepth == filterHeight == filterWidth`.
     * @param strides The strides of the pooling:
     *     `[strideDepth, strideHeight, strideWidth]`.
     *     If `strides` is a single number,
     *     then `strideDepth == strideHeight == strideWidth`.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1*1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     * @param dataFormat An optional string from: "NDHWC", "NCDHW". Defaults to
     *     "NDHWC". Specify the data format of the input and output data. With the
     *     default format "NDHWC", the data is stored in the order of: [batch,
     *     depth, height, width, channels]. Only "NDHWC" is currently supported.
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function maxPool3d_(x, filterSize = [1, 1, 1], strides, pad, dimRoundingMode, dataFormat = 'NDHWC') {
        const $x = convertToTensor(x, 'x', 'maxPool3d');
        let x5D = $x;
        let reshapedTo5D = false;
        if ($x.rank === 4) {
            reshapedTo5D = true;
            x5D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2], $x.shape[3]]);
        }
        assert(x5D.rank === 5, () => `Error in maxPool3d: x must be rank 5 but got rank ${x5D.rank}.`);
        assert(dataFormat === 'NDHWC', () => `Error in maxPool3d: Only NDHWC is currently supported, ` +
            `but got dataFormat of ${dataFormat}`);
        checkPadOnDimRoundingMode('maxPool3d', pad, dimRoundingMode);
        const inputs = { x: x5D };
        const attrs = { filterSize, strides, pad, dimRoundingMode, dataFormat };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(MaxPool3D, inputs, attrs);
        if (reshapedTo5D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
        }
        return res;
    }
    op({ maxPool3d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the 2D max pooling of an image with Argmax index.
     * The indices in argmax are flattened, so that a maximum value at position `[b,
     * y, x, c]` becomes flattened index: `(y * width + x) * channels + c` if
     * include_batch_in_index is False; `((b * height + y) * width + x) * channels
     * +c` if include_batch_in_index is True.
     *
     * The indices returned are always in `[0, height) x [0, width)` before
     * flattening.
     *
     * @param x The input tensor, of rank 4 or rank 3 of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is assumed.
     * @param filterSize The filter size: `[filterHeight, filterWidth]`. If
     *     `filterSize` is a single number, then `filterHeight == filterWidth`.
     * @param strides The strides of the pooling: `[strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param dataFormat An optional string from: "NDHWC", "NCDHW". Defaults to
     *     "NDHWC". Specify the data format of the input and output data. With the
     *     default format "NDHWC", the data is stored in the order of: [batch,
     *     depth, height, width, channels]. Only "NDHWC" is currently supported.
     * @param pad The type of padding algorithm.
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param includeBatchIndex Defaults to False. Whether to include batch
     *    dimension in flattened index of argmax.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function maxPoolWithArgmax_(x, filterSize, strides, pad, includeBatchInIndex = false) {
        const $x = convertToTensor(x, 'x', 'maxPoolWithArgmax');
        const inputs = { x: $x };
        const attrs = { filterSize, strides, pad, includeBatchInIndex };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const result = ENGINE.runKernel(MaxPoolWithArgmax, inputs, attrs);
        return { result: result[0], indexes: result[1] };
    }
    op({ maxPoolWithArgmax_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the max of a and b (`a > b ? a : b`) element-wise.
     * Supports broadcasting.
     *
     * We also expose `tf.maximumStrict` which has the same signature as this op and
     * asserts that `a` and `b` are the same shape (does not broadcast).
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 3, 16]);
     * const b = tf.tensor1d([1, 2, 9, 4]);
     *
     * a.maximum(b).print();  // or tf.maximum(a, b)
     * ```
     *
     * ```js
     * // Broadcast maximum a with b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(5);
     *
     * a.maximum(b).print();  // or tf.maximum(a, b)
     * ```
     *
     * @param a The first tensor.
     * @param b The second tensor. Must have the same type as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function maximum_(a, b) {
        let $a = convertToTensor(a, 'a', 'maximum');
        let $b = convertToTensor(b, 'b', 'maximum');
        [$a, $b] = makeTypesMatch($a, $b);
        if ($a.dtype === 'bool') {
            $a = cast($a, 'int32');
            $b = cast($b, 'int32');
        }
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Maximum$1, inputs);
    }
    const maximum$1 = op({ maximum_ });

    /**
     * @license
     * Copyright 2020 Google Inc. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the mean of elements across dimensions of a `tf.Tensor`.
     *
     * Reduces `x` along the dimensions given in `axis`. Unless `keepDims` is
     * true, the rank of the `tf.Tensor` is reduced by 1 for each entry in `axis`.
     * If `keepDims` is true, the reduced dimensions are retained with length 1.
     * If `axis` has no entries, all dimensions are reduced, and a `tf.Tensor` with
     * a single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.mean().print();  // or tf.mean(a)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.mean(axis).print();  // or tf.mean(x, axis)
     * ```
     *
     * @param x The input tensor.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function mean_(x, axis = null, keepDims = false) {
        const $x = convertToTensor(x, 'x', 'mean');
        const inputs = { x: $x };
        const attrs = { axis, keepDims };
        return ENGINE.runKernel(Mean, inputs, attrs);
    }
    const mean = op({ mean_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the min of a and b (`a < b ? a : b`) element-wise.
     * Supports broadcasting.
     *
     * We also expose `minimumStrict` which has the same signature as this op and
     * asserts that `a` and `b` are the same shape (does not broadcast).
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 3, 16]);
     * const b = tf.tensor1d([1, 2, 9, 4]);
     *
     * a.minimum(b).print();  // or tf.minimum(a, b)
     * ```
     *
     * ```js
     * // Broadcast minimum a with b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(5);
     *
     * a.minimum(b).print();  // or tf.minimum(a, b)
     * ```
     *
     * @param a The first tensor.
     * @param b The second tensor. Must have the same type as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function minimum_(a, b) {
        let $a = convertToTensor(a, 'a', 'minimum');
        let $b = convertToTensor(b, 'b', 'minimum');
        [$a, $b] = makeTypesMatch($a, $b);
        if ($a.dtype === 'bool') {
            $a = cast($a, 'int32');
            $b = cast($b, 'int32');
        }
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Minimum$1, inputs);
    }
    const minimum$1 = op({ minimum_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Pads a `tf.Tensor` using mirror padding.
     *
     * This operation implements the `REFLECT` and `SYMMETRIC` modes of pad.
     *
     * ```js
     * const x = tf.range(0, 9).reshape([1, 1, 3, 3]);
     * x.mirrorPad([[0, 0], [0, 0], [2, 2], [2, 2]], 'reflect').print();
     * ```
     * @param x The tensor to pad.
     * @param paddings An array of length `R` (the rank of the tensor), where
     * each element is a length-2 tuple of ints `[padBefore, padAfter]`,
     * specifying how much to pad along each dimension of the tensor.
     * In "reflect" mode, the padded regions do not include the borders,
     * while in "symmetric" mode the padded regions do include the borders.
     * For example, if the input is `[1, 2, 3]` and paddings is `[0, 2]`,
     * then the output is `[1, 2, 3, 2, 1]` in "reflect" mode, and
     * `[1, 2, 3, 3, 2]` in "symmetric" mode.
     * If `mode` is "reflect" then both `paddings[D, 0]` and `paddings[D, 1]`
     * must be no greater than `x.shape[D] - 1`. If mode is "symmetric"
     * then both `paddings[D, 0]` and `paddings[D, 1]` must be no greater than
     * `x.shape[D]`
     * @param mode String to specify padding mode. Can be `'reflect' | 'symmetric'`
     */
    /** @doc {heading: 'Tensors', subheading: 'Transformations'} */
    function mirrorPad_(x, paddings, mode) {
        assert(mode === 'reflect' || mode === 'symmetric', () => `Invalid mode. Mode must be either reflect or symmetric. ` +
            `Got ${mode}.`);
        const $x = convertToTensor(x, 'x', 'mirrorPad');
        if ($x.rank === 0) {
            throw new Error('mirrorPad(scalar) is not defined. ' +
                'Pass non-scalar to mirrorPad');
        }
        assert(paddings.length === $x.rank, () => `Padding doesn't match input. Must be ${$x.rank}. ` +
            `Got ${paddings.length}.`);
        const shapeOffset = mode === 'reflect' ? 1 : 0;
        for (let i = 0; i < $x.rank; i++) {
            assert(paddings[i].length === 2, () => `Invalid number of paddings. Must be length of 2 each.`);
            assert(paddings[i][0] >= 0 && paddings[i][0] <= $x.shape[i] - shapeOffset &&
                paddings[i][1] >= 0 && paddings[i][1] <= $x.shape[i] - shapeOffset, () => `Padding in dimension ${i} cannot be greater than or equal ` +
                `to ${$x.shape[i] - shapeOffset} or less than 0 for input of ` +
                `shape ${$x.shape}`);
        }
        const attrs = { paddings, mode };
        const inputs = { x: $x };
        return ENGINE.runKernel(MirrorPad, inputs, attrs);
    }
    op({ mirrorPad_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the mod of a and b element-wise.
     * `floor(x / y) * y + mod(x, y) = x`
     * Supports broadcasting.
     *
     * We also expose `tf.modStrict` which has the same signature as this op and
     * asserts that `a` and `b` are the same shape (does not broadcast).
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 3, 16]);
     * const b = tf.tensor1d([1, 2, 9, 4]);
     *
     * a.mod(b).print();  // or tf.mod(a, b)
     * ```
     *
     * ```js
     * // Broadcast a mod b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(5);
     *
     * a.mod(b).print();  // or tf.mod(a, b)
     * ```
     *
     * @param a The first tensor.
     * @param b The second tensor. Must have the same type as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function mod_(a, b) {
        let $a = convertToTensor(a, 'a', 'mod');
        let $b = convertToTensor(b, 'b', 'mod');
        [$a, $b] = makeTypesMatch($a, $b);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(Mod, inputs);
    }
    op({ mod_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Calculates the mean and variance of `x`. The mean and variance are
     * calculated by aggregating the contents of `x` across `axes`. If `x` is
     * 1-D and `axes = [0]` this is just the mean and variance of a vector.
     *
     * @param x The input tensor.
     * @param axis The dimension(s) along with to compute mean and
     *     variance. By default it reduces all dimensions.
     * @param keepDims If true, the moments have the same dimensionality as the
     *     input.
     * @return An object with two keys: `mean` and `variance`.
     *
     * @doc {heading: 'Operations', subheading: 'Normalization'}
     */
    function moments_(x, axis = null, keepDims = false) {
        x = convertToTensor(x, 'x', 'moments');
        const axes = parseAxisParam(axis, x.shape);
        const xMean = mean(x, axes, keepDims);
        let keepDimsShape = xMean.shape;
        if (!keepDims) {
            keepDimsShape = expandShapeToKeepDim(xMean.shape, axes);
        }
        const devSquared = square(sub(cast(x, 'float32'), reshape$1(xMean, keepDimsShape)));
        const variance = mean(devSquared, axes, keepDims);
        return { mean: xMean, variance };
    }
    op({ moments_ });

    /**
     * Computes the next states and outputs of a stack of LSTMCells.
     *
     * Each cell output is used as input to the next cell.
     *
     * Returns `[cellState, cellOutput]`.
     *
     * Derived from tf.contrib.rn.MultiRNNCell.
     *
     * @param lstmCells Array of LSTMCell functions.
     * @param data The input to the cell.
     * @param c Array of previous cell states.
     * @param h Array of previous cell outputs.
     *
     * @doc {heading: 'Operations', subheading: 'RNN'}
     */
    function multiRNNCell_(lstmCells, data, c, h) {
        const $data = convertToTensor(data, 'data', 'multiRNNCell');
        const $c = convertToTensorArray(c, 'c', 'multiRNNCell');
        const $h = convertToTensorArray(h, 'h', 'multiRNNCell');
        let input = $data;
        const newStates = [];
        for (let i = 0; i < lstmCells.length; i++) {
            const output = lstmCells[i](input, $c[i], $h[i]);
            newStates.push(output[0]);
            newStates.push(output[1]);
            input = output[1];
        }
        const newC = [];
        const newH = [];
        for (let i = 0; i < newStates.length; i += 2) {
            newC.push(newStates[i]);
            newH.push(newStates[i + 1]);
        }
        return [newC, newH];
    }
    op({ multiRNNCell_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values drawn from a multinomial distribution.
     *
     * ```js
     * const probs = tf.tensor([.75, .25]);
     * tf.multinomial(probs, 3).print();
     * ```
     *
     * @param logits 1D array with unnormalized log-probabilities, or
     *     2D array of shape `[batchSize, numOutcomes]`. See the `normalized`
     *     parameter.
     * @param numSamples Number of samples to draw for each row slice.
     * @param seed The seed number.
     * @param normalized Whether the provided `logits` are normalized true
     *     probabilities (sum to 1). Defaults to false.
     * @return 1D array of shape `[numSamples]`, or 2D array of shape
     *     `[batchSize, numSamples]`, depending on the rank of the input.
     *
     * @doc {heading: 'Tensors', subheading: 'Random'}
     */
    function multinomial_(logits, numSamples, seed, normalized = false) {
        const $logits = convertToTensor(logits, 'logits', 'multinomial');
        const numOutcomes = $logits.size;
        const origRank = $logits.rank;
        if (numOutcomes < 2) {
            throw new Error(`Error in multinomial: you need at least 2 outcomes, but got ` +
                `${numOutcomes}.`);
        }
        if (origRank > 2) {
            throw new Error(`Rank of probabilities must be 1 or 2, but is ${origRank}`);
        }
        // TODO(lina128): Investigate correct seed behavior. The code seems not allow
        // setting see to 0.
        seed = seed || Math.random();
        // The kernel only accepts (and returns) rank 2 tensors.
        const logits2D = origRank === 1 ? reshape$1($logits, [1, -1]) : $logits;
        const inputs = { logits: logits2D };
        const attrs = { numSamples, seed, normalized };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        const res = ENGINE.runKernel(Multinomial, inputs, attrs);
        // tslint:disable-next-line:no-unnecessary-type-assertion
        return origRank === 1 ? reshape$1(res, [res.size]) : res;
    }
    op({ multinomial_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns the truth value of (a != b) element-wise. Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([0, 2, 3]);
     *
     * a.notEqual(b).print();
     * ```
     * @param a The first input tensor.
     * @param b The second input tensor. Must have the same dtype as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Logical'}
     */
    function notEqual_(a, b) {
        let $a = convertToTensor(a, 'a', 'notEqual', 'string_or_numeric');
        let $b = convertToTensor(b, 'b', 'notEqual', 'string_or_numeric');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        return ENGINE.runKernel(NotEqual, inputs);
    }
    const notEqual = op({ notEqual_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with all elements set to 1 with the same shape as the
     * given tensor.
     *
     * ```js
     * const x = tf.tensor([1, 2]);
     * tf.onesLike(x).print();
     * ```
     * @param x A tensor.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function onesLike_(x) {
        const $x = convertToTensor(x, 'x', 'onesLike');
        const inputs = { x: $x };
        return ENGINE.runKernel(OnesLike, inputs);
    }
    op({ onesLike_ });

    /**
     * Computes the outer product of two vectors, `v1` and `v2`.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     * const b = tf.tensor1d([3, 4, 5]);
     *
     * tf.outerProduct(a, b).print();
     * ```
     * @param v1 The first vector in the outer product operation.
     * @param v2 The second vector in the outer product operation.
     *
     * @doc {heading: 'Operations', subheading: 'Matrices'}
     */
    function outerProduct_(v1, v2) {
        const $v1 = convertToTensor(v1, 'v1', 'outerProduct');
        const $v2 = convertToTensor(v2, 'v2', 'outerProduct');
        assert($v1.rank === 1 && $v2.rank === 1, () => `Error in outerProduct: inputs must be rank 1, but got ranks ` +
            `${$v1.rank} and ${$v2.rank}.`);
        const v12D = reshape$1($v1, [-1, 1]);
        const v22D = reshape$1($v2, [1, -1]);
        return matMul(v12D, v22D);
    }
    op({ outerProduct_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Pads a `tf.Tensor` with a given value and paddings.
     *
     * This operation implements `CONSTANT` mode. For `REFLECT` and `SYMMETRIC`,
     * refer to `tf.mirrorPad`.
     *
     * Also available are stricter rank-specific methods with the same signature
     * as this method that assert that `paddings` is of given length.
     *   - `tf.pad1d`
     *   - `tf.pad2d`
     *   - `tf.pad3d`
     *   - `tf.pad4d`
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     * x.pad([[1, 2]]).print();
     * ```
     * @param x The tensor to pad.
     * @param paddings An array of length `R` (the rank of the tensor), where
     * each element is a length-2 tuple of ints `[padBefore, padAfter]`,
     * specifying how much to pad along each dimension of the tensor.
     * @param constantValue The pad value to use. Defaults to 0.
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function pad_(x, paddings, constantValue = 0) {
        const $x = convertToTensor(x, 'x', 'pad');
        if ($x.rank === 0) {
            throw new Error('pad(scalar) is not defined. Pass non-scalar to pad');
        }
        const attrs = { paddings, constantValue };
        const inputs = { x: $x };
        return ENGINE.runKernel(PadV2, inputs, attrs);
    }
    const pad = op({ pad_ });

    /**
     * Pads a `tf.Tensor1D` with a given value and paddings. See `pad` for details.
     */
    function pad1d_(x, paddings, constantValue = 0) {
        assert(paddings.length === 2, () => 'Invalid number of paddings. Must be length of 2.');
        return pad(x, [paddings], constantValue);
    }
    op({ pad1d_ });

    /**
     * Pads a `tf.Tensor2D` with a given value and paddings. See `pad` for details.
     */
    function pad2d_(x, paddings, constantValue = 0) {
        assert(paddings.length === 2 && paddings[0].length === 2 &&
            paddings[1].length === 2, () => 'Invalid number of paddings. Must be length of 2 each.');
        return pad(x, paddings, constantValue);
    }
    op({ pad2d_ });

    /**
     * Pads a `tf.Tensor3D` with a given value and paddings. See `pad` for details.
     */
    function pad3d_(x, paddings, constantValue = 0) {
        assert(paddings.length === 3 && paddings[0].length === 2 &&
            paddings[1].length === 2 && paddings[2].length === 2, () => 'Invalid number of paddings. Must be length of 2 each.');
        return pad(x, paddings, constantValue);
    }
    op({ pad3d_ });

    /**
     * Pads a `tf.Tensor4D` with a given value and paddings. See `pad` for details.
     */
    function pad4d_(x, paddings, constantValue = 0) {
        assert(paddings.length === 4 && paddings[0].length === 2 &&
            paddings[1].length === 2 && paddings[2].length === 2 &&
            paddings[3].length === 2, () => 'Invalid number of paddings. Must be length of 2 each.');
        return pad(x, paddings, constantValue);
    }
    op({ pad4d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Performs an N-D pooling operation
     *
     * @param input The input tensor, of rank 4 or rank 3 of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is assumed.
     * @param windowShape The filter size: `[filterHeight, filterWidth]`. If
     *     `filterSize` is a single number, then `filterHeight == filterWidth`.
     * @param poolingType The type of pooling, either 'max' or 'avg'.
     * @param pad The type of padding algorithm:
     *    - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *    - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *    - For more info, see this guide:
     *     [https://www.tensorflow.org/api_guides/python/nn#Convolution](
     *         https://www.tensorflow.org/api_guides/python/nn#Convolution)
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     in dilated pooling. Defaults to `[1, 1]`. If `dilationRate` is a single
     *     number, then `dilationHeight == dilationWidth`. If it is greater than
     *     1, then all values of `strides` must be 1.
     * @param strides The strides of the pooling: `[strideHeight, strideWidth]`. If
     *     `strides` is a single number, then `strideHeight == strideWidth`.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function pool_(input, windowShape, poolingType, pad, dilations, strides, dimRoundingMode) {
        if (dilations == null) {
            dilations = [1, 1];
        }
        if (strides == null) {
            strides = 1;
        }
        if (pad === 0) {
            pad = 'valid';
        }
        const $x = convertToTensor(input, 'x', 'maxPool');
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        assert(eitherStridesOrDilationsAreOne(strides, dilations), () => 'Error in pool: Either strides or dilations must be 1. ' +
            `Got strides ${strides} and dilations '${dilations}'`);
        const convInfo = computePool2DInfo(x4D.shape, windowShape, strides, dilations, pad);
        const dilation = [convInfo.dilationHeight, convInfo.dilationWidth];
        // The following implementation does batchToSpace(pool(spaceToBatch(x)))
        // whenever dilation > 1 since the TF kernels do not support dilation > 1.
        // tslint:disable-next-line:max-line-length
        // https://github.com/tensorflow/tensorflow/blob/50f6bb67dc98c9b74630b6047aae7a4f8a40fd02/tensorflow/python/ops/nn_ops.py#L1037
        let basePadding;
        if (pad === 'same') {
            basePadding = withSpaceToBatchBasePaddings([convInfo.filterHeight, convInfo.filterWidth], dilation);
        }
        else {
            basePadding = [[0, 0], [0, 0]];
        }
        const isDilationOne = dilation[0] === 1 && dilation[1] === 1;
        const [adjustedPadding, adjustedCrops] = requiredSpaceToBatchPaddings([convInfo.inHeight, convInfo.inWidth], dilation, basePadding);
        const convertedPad = isDilationOne ? pad : 'valid';
        const convertedX = isDilationOne ? x4D : spaceToBatchND(x4D, dilation, adjustedPadding);
        const forwardOp = poolingType === 'avg' ?
            () => avgPool(convertedX, windowShape, strides, convertedPad, dimRoundingMode) :
            () => maxPool(convertedX, windowShape, strides, convertedPad, dimRoundingMode);
        const y = forwardOp();
        const res = isDilationOne ? y : batchToSpaceND(y, dilation, adjustedCrops);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    // Helper function to compute crops and paddings for pool with dilation > 1.
    // tslint:disable-next-line:max-line-length
    // https://github.com/tensorflow/tensorflow/blob/50f6bb67dc98c9b74630b6047aae7a4f8a40fd02/tensorflow/python/ops/array_ops.py#L2184
    function requiredSpaceToBatchPaddings(inputShape, blockShape, basePadding) {
        const padStart = basePadding.map(b => b[0]);
        const origPadEnd = basePadding.map(b => b[1]);
        const fullInputShape = inputShape.concat(padStart, origPadEnd);
        const padEndExtra = blockShape.map((b, i) => (b - fullInputShape[i] % b) % b);
        const padEnd = origPadEnd.map((s, i) => s + padEndExtra[i]);
        const paddings = blockShape.map((_, i) => [padStart[i], padEnd[i]]);
        const crops = blockShape.map((_, i) => [0, padEndExtra[i]]);
        return [paddings, crops];
    }
    // Helper function to compute base paddings for pool with dilation > 1.
    // tslint:disable-next-line:max-line-length
    // https://github.com/tensorflow/tensorflow/blob/50f6bb67dc98c9b74630b6047aae7a4f8a40fd02/tensorflow/python/ops/nn_ops.py#L524
    function withSpaceToBatchBasePaddings(filterShape, dilation) {
        // Spatial dimensions of the filters and the upsampled filters in which we
        // introduce (rate - 1) zeros between consecutive filter values.
        const dilatedFilterShape = filterShape.map((s, i) => {
            return s + (s - 1) * (dilation[i] - 1);
        });
        const padExtraShape = dilatedFilterShape.map(s => s - 1);
        // When padding is odd, we pad more at end, following the same
        // convention as conv2d.
        const padExtraStart = padExtraShape.map(s => Math.floor(s / 2));
        const padExtraEnd = padExtraShape.map((s, i) => s - padExtraStart[i]);
        return padExtraShape.map((_, i) => {
            return [padExtraStart[i], padExtraEnd[i]];
        });
    }
    op({ pool_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes leaky rectified linear element-wise with parametric alphas.
     *
     * `x < 0 ? alpha * x : f(x) = x`
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 4]);
     * const alpha = tf.scalar(0.1);
     *
     * x.prelu(alpha).print();  // or tf.prelu(x, alpha)
     * ```
     * @param x The input tensor.
     * @param alpha Scaling factor for negative values.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function prelu_(x, alpha) {
        const $x = convertToTensor(x, 'x', 'prelu');
        const $alpha = convertToTensor(alpha, 'alpha', 'prelu');
        const inputs = { x: $x, alpha: $alpha };
        return ENGINE.runKernel(Prelu, inputs);
    }
    const prelu$1 = op({ prelu_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the product of elements across dimensions of a `tf.Tensor`.
     *
     * Reduces the input along the dimensions given in `axes`. Unless `keepDims`
     * is true, the rank of the `tf.Tensor` is reduced by 1 for each entry in
     * `axes`. If `keepDims` is true, the reduced dimensions are retained with
     * length 1. If `axes` has no entries, all dimensions are reduced, and a
     * `tf.Tensor` with a single element is returned.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3]);
     *
     * x.prod().print();  // or tf.prod(x)
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.prod(axis).print();  // or tf.prod(x, axis)
     * ```
     *
     * @param x The input tensor to compute the product over. If the dtype is `bool`
     *   it will be converted to `int32` and the output dtype will be `int32`.
     * @param axis The dimension(s) to reduce. By default it reduces
     *     all dimensions.
     * @param keepDims If true, retains reduced dimensions with size 1.
     *
     * @doc {heading: 'Operations', subheading: 'Reduction'}
     */
    function prod_(x, axis = null, keepDims = false) {
        let $x = convertToTensor(x, 'x', 'prod');
        if ($x.dtype === 'bool') {
            // bool is not an allowed type for the underlying kernel.
            $x = cast($x, 'int32');
        }
        const inputs = { x: $x };
        const attrs = { axis, keepDims };
        return ENGINE.runKernel(Prod, inputs, attrs);
    }
    op({ prod_ });

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function raggedGather_(paramsNestedSplits, paramsDenseValues, indices, outputRaggedRank) {
        const $paramsNestedSplits = paramsNestedSplits.map((t, i) => convertToTensor(t, `tensors${i}`, 'raggedGather', 'int32'));
        const $paramsDenseValues = convertToTensor(paramsDenseValues, 'paramsDenseValues', 'raggedGather');
        const $indices = convertToTensor(indices, 'indices', 'raggedGather', 'int32');
        const inputs = {
            paramsNestedSplits: $paramsNestedSplits,
            paramsDenseValues: $paramsDenseValues,
            indices: $indices,
        };
        const attrs = { outputRaggedRank };
        const result = ENGINE.runKernel(RaggedGather, inputs, attrs);
        return {
            outputNestedSplits: result.slice(0, result.length - 1),
            outputDenseValues: result[result.length - 1],
        };
    }
    op({ raggedGather_ });

    /**
     * @license
     * Copyright 2022 Google LLC.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns a RaggedTensor result composed from rtDenseValues and rtNestedSplits,
     * such that result[i] = [starts[i], starts[i] + deltas[i], ..., limits[i]]).
     *
     * @param starts: A Tensor. Must be one of the following types:
     *     'float32', 'int32'. The starts of each range.
     * @param limits: A Tensor. Must have the same type as starts. The limits of
     *     each range.
     * @param deltas: A Tensor. Must have the same type as starts. The deltas of
     *     each range.
     * @return A map with the following properties:
     *     - rtNestedSplits: A Tensor of type 'int32'.
     *     - rtDenseValues: A Tensor. Has the same type as starts.
     */
    function raggedRange_(starts, limits, deltas) {
        const $starts = convertToTensor(starts, 'starts', 'raggedRange');
        const $limits = convertToTensor(limits, 'limits', 'raggedRange', $starts.dtype);
        const $deltas = convertToTensor(deltas, 'deltas', 'raggedRange', $starts.dtype);
        const inputs = {
            starts: $starts,
            limits: $limits,
            deltas: $deltas,
        };
        const result = ENGINE.runKernel(RaggedRange, inputs);
        return {
            rtNestedSplits: result[0],
            rtDenseValues: result[1],
        };
    }
    op({ raggedRange_ });

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Create a dense tensor from a ragged tensor, possibly altering its shape.
     *
     * The raggedTensorToTensor op creates a dense tensor from am array of row
     * partition tensors, a value vector, and default values. If the shape is
     * unspecified, the minimal shape required to contain all the elements in the
     * ragged tensor (the natural shape) will be used. If some dimensions are left
     * unspecified, then the size of the natural shape is used in that dimension.
     *
     * The defaultValue will be broadcast to the output shape. After that, the
     * values from the ragged tensor overwrite the default values. Note that the
     * defaultValue must have less dimensions than the value.
     *
     * The row partition tensors are in the order of the dimensions. At present, the
     * types can be: "ROW_SPLITS": the row_splits tensor from the ragged tensor.
     *   "VALUE_ROWIDS": the value_rowids tensor from the ragged tensor.
     *   "FIRST_DIM_SIZE": if value_rowids is used for the first dimension, then it
     * is preceded by "FIRST_DIM_SIZE".
     * ```
     * @param shape: A Tensor. Must be one of the following types: 'int32'. The
     *     desired shape of the output tensor. If left unspecified (empty), the
     *     minimal shape required to contain all the elements in the ragged tensor
     *     (the natural shape) will be used. If some dimensions are left
     *     unspecified, then the size of the natural shape is used in that
     *     dimension.
     *
     *     Note that dense dimensions cannot be modified by the shape argument.
     *     Trying to change the size of a dense dimension will cause the op to fail.
     *     Examples: natural shape: [4, 5, 6] shape: -1 output shape: [4, 5, 6]
     *
     *     natural shape: [4, 5, 6] shape: [3, -1, 2] output shape: [3, 5, 2]
     *
     *     natural shape: [4, 5, 6] shape: [3, 7, 2] output shape: [3, 7, 2]
     * @param values: A Tensor. A 1D tensor representing the values of the ragged
     *     tensor.
     * @param defaultValue: A Tensor. Must have the same type as values. The
     *     defaultValue when the shape is larger than the ragged tensor. The
     *     defaultValue is broadcast until it is the shape of the output tensor,
     *     and then overwritten by values in the ragged tensor. The default value
     *     must be compatible with this broadcast operation, and must have fewer
     *     dimensions than the value tensor.
     * @param rowPartitionTensors: A list of at least 1 Tensor objects with the same
     *     type in: 'int32'.
     * @param rowPartitionTypes: A list of strings. The types of the row partition
     *     tensors. At present, these can be:
     *     "ROW_SPLITS": the row_splits tensor from the ragged tensor.
     *     "VALUE_ROWIDS": the value_rowids tensor from the ragged tensor.
     *     "FIRST_DIM_SIZE": if value_rowids is used for the first dimension, then
     *         it is preceeded by "FIRST_DIM_SIZE". The tensors are in the order of
     *         the dimensions.
     * @return A Tensor. Has the same type as values.
     * @doc {heading: 'Operations', subheading: 'Ragged'}
     */
    function raggedTensorToTensor_(shape, values, defaultValue, rowPartitionTensors, rowPartitionTypes) {
        const $shape = convertToTensor(shape, 'shape', 'raggedTensorToTensor', 'int32');
        const $values = convertToTensor(values, 'values', 'raggedTensorToTensor');
        const $defaultValue = convertToTensor(defaultValue, 'defaultValue', 'raggedTensorToTensor', $values.dtype);
        const $rowPartitionTensors = rowPartitionTensors.map((t, i) => convertToTensor(t, `tensors${i}`, 'raggedTensorToTensor', 'int32'));
        const inputs = {
            shape: $shape,
            values: $values,
            defaultValue: $defaultValue,
            rowPartitionTensors: $rowPartitionTensors
        };
        const attrs = { rowPartitionTypes };
        return ENGINE.runKernel(RaggedTensorToTensor, inputs, attrs);
    }
    op({ raggedTensorToTensor_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values sampled from a random number generator
     * function defined by the user.
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param randFunction A random number generator function which is called
     * for each element in the output tensor.
     * @param dtype The data type of the output tensor. Defaults to 'float32'.
     *
     * @doc {heading: 'Tensors', subheading: 'Random'}
     */
    function rand_(shape, randFunction, dtype) {
        const size = sizeFromShape(shape);
        let values = null;
        if (dtype == null || dtype === 'float32') {
            values = new Float32Array(size);
        }
        else if (dtype === 'int32') {
            values = new Int32Array(size);
        }
        else if (dtype === 'bool') {
            values = new Uint8Array(size);
        }
        else {
            throw new Error(`Unknown data type ${dtype}`);
        }
        for (let i = 0; i < size; i++) {
            values[i] = randFunction();
        }
        return ENGINE.makeTensor(values, shape, dtype);
    }
    op({ rand_ });

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function getAugmentedNamespace(n) {
    	if (n.__esModule) return n;
    	var a = Object.defineProperty({}, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    function createCommonjsModule(fn) {
      var module = { exports: {} };
    	return fn(module, module.exports), module.exports;
    }

    var alea = createCommonjsModule(function (module) {
    // A port of an algorithm by Johannes Baagøe <baagoe@baagoe.com>, 2010
    // http://baagoe.com/en/RandomMusings/javascript/
    // https://github.com/nquinlan/better-random-numbers-for-javascript-mirror
    // Original work is under MIT license -

    // Copyright (C) 2010 by Johannes Baagøe <baagoe@baagoe.org>
    //
    // Permission is hereby granted, free of charge, to any person obtaining a copy
    // of this software and associated documentation files (the "Software"), to deal
    // in the Software without restriction, including without limitation the rights
    // to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    // copies of the Software, and to permit persons to whom the Software is
    // furnished to do so, subject to the following conditions:
    //
    // The above copyright notice and this permission notice shall be included in
    // all copies or substantial portions of the Software.
    //
    // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    // IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    // FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    // AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    // LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    // OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    // THE SOFTWARE.



    (function(global, module, define) {

    function Alea(seed) {
      var me = this, mash = Mash();

      me.next = function() {
        var t = 2091639 * me.s0 + me.c * 2.3283064365386963e-10; // 2^-32
        me.s0 = me.s1;
        me.s1 = me.s2;
        return me.s2 = t - (me.c = t | 0);
      };

      // Apply the seeding algorithm from Baagoe.
      me.c = 1;
      me.s0 = mash(' ');
      me.s1 = mash(' ');
      me.s2 = mash(' ');
      me.s0 -= mash(seed);
      if (me.s0 < 0) { me.s0 += 1; }
      me.s1 -= mash(seed);
      if (me.s1 < 0) { me.s1 += 1; }
      me.s2 -= mash(seed);
      if (me.s2 < 0) { me.s2 += 1; }
      mash = null;
    }

    function copy(f, t) {
      t.c = f.c;
      t.s0 = f.s0;
      t.s1 = f.s1;
      t.s2 = f.s2;
      return t;
    }

    function impl(seed, opts) {
      var xg = new Alea(seed),
          state = opts && opts.state,
          prng = xg.next;
      prng.int32 = function() { return (xg.next() * 0x100000000) | 0; };
      prng.double = function() {
        return prng() + (prng() * 0x200000 | 0) * 1.1102230246251565e-16; // 2^-53
      };
      prng.quick = prng;
      if (state) {
        if (typeof(state) == 'object') copy(state, xg);
        prng.state = function() { return copy(xg, {}); };
      }
      return prng;
    }

    function Mash() {
      var n = 0xefc8249d;

      var mash = function(data) {
        data = String(data);
        for (var i = 0; i < data.length; i++) {
          n += data.charCodeAt(i);
          var h = 0.02519603282416938 * n;
          n = h >>> 0;
          h -= n;
          h *= n;
          n = h >>> 0;
          h -= n;
          n += h * 0x100000000; // 2^32
        }
        return (n >>> 0) * 2.3283064365386963e-10; // 2^-32
      };

      return mash;
    }


    if (module && module.exports) {
      module.exports = impl;
    } else if (define && define.amd) {
      define(function() { return impl; });
    } else {
      this.alea = impl;
    }

    })(
      commonjsGlobal,
      module,    // present in node.js
      (typeof undefined) == 'function'    // present with an AMD loader
    );
    });

    var xor128 = createCommonjsModule(function (module) {
    // A Javascript implementaion of the "xor128" prng algorithm by
    // George Marsaglia.  See http://www.jstatsoft.org/v08/i14/paper

    (function(global, module, define) {

    function XorGen(seed) {
      var me = this, strseed = '';

      me.x = 0;
      me.y = 0;
      me.z = 0;
      me.w = 0;

      // Set up generator function.
      me.next = function() {
        var t = me.x ^ (me.x << 11);
        me.x = me.y;
        me.y = me.z;
        me.z = me.w;
        return me.w ^= (me.w >>> 19) ^ t ^ (t >>> 8);
      };

      if (seed === (seed | 0)) {
        // Integer seed.
        me.x = seed;
      } else {
        // String seed.
        strseed += seed;
      }

      // Mix in string seed, then discard an initial batch of 64 values.
      for (var k = 0; k < strseed.length + 64; k++) {
        me.x ^= strseed.charCodeAt(k) | 0;
        me.next();
      }
    }

    function copy(f, t) {
      t.x = f.x;
      t.y = f.y;
      t.z = f.z;
      t.w = f.w;
      return t;
    }

    function impl(seed, opts) {
      var xg = new XorGen(seed),
          state = opts && opts.state,
          prng = function() { return (xg.next() >>> 0) / 0x100000000; };
      prng.double = function() {
        do {
          var top = xg.next() >>> 11,
              bot = (xg.next() >>> 0) / 0x100000000,
              result = (top + bot) / (1 << 21);
        } while (result === 0);
        return result;
      };
      prng.int32 = xg.next;
      prng.quick = prng;
      if (state) {
        if (typeof(state) == 'object') copy(state, xg);
        prng.state = function() { return copy(xg, {}); };
      }
      return prng;
    }

    if (module && module.exports) {
      module.exports = impl;
    } else if (define && define.amd) {
      define(function() { return impl; });
    } else {
      this.xor128 = impl;
    }

    })(
      commonjsGlobal,
      module,    // present in node.js
      (typeof undefined) == 'function'    // present with an AMD loader
    );
    });

    var xorwow = createCommonjsModule(function (module) {
    // A Javascript implementaion of the "xorwow" prng algorithm by
    // George Marsaglia.  See http://www.jstatsoft.org/v08/i14/paper

    (function(global, module, define) {

    function XorGen(seed) {
      var me = this, strseed = '';

      // Set up generator function.
      me.next = function() {
        var t = (me.x ^ (me.x >>> 2));
        me.x = me.y; me.y = me.z; me.z = me.w; me.w = me.v;
        return (me.d = (me.d + 362437 | 0)) +
           (me.v = (me.v ^ (me.v << 4)) ^ (t ^ (t << 1))) | 0;
      };

      me.x = 0;
      me.y = 0;
      me.z = 0;
      me.w = 0;
      me.v = 0;

      if (seed === (seed | 0)) {
        // Integer seed.
        me.x = seed;
      } else {
        // String seed.
        strseed += seed;
      }

      // Mix in string seed, then discard an initial batch of 64 values.
      for (var k = 0; k < strseed.length + 64; k++) {
        me.x ^= strseed.charCodeAt(k) | 0;
        if (k == strseed.length) {
          me.d = me.x << 10 ^ me.x >>> 4;
        }
        me.next();
      }
    }

    function copy(f, t) {
      t.x = f.x;
      t.y = f.y;
      t.z = f.z;
      t.w = f.w;
      t.v = f.v;
      t.d = f.d;
      return t;
    }

    function impl(seed, opts) {
      var xg = new XorGen(seed),
          state = opts && opts.state,
          prng = function() { return (xg.next() >>> 0) / 0x100000000; };
      prng.double = function() {
        do {
          var top = xg.next() >>> 11,
              bot = (xg.next() >>> 0) / 0x100000000,
              result = (top + bot) / (1 << 21);
        } while (result === 0);
        return result;
      };
      prng.int32 = xg.next;
      prng.quick = prng;
      if (state) {
        if (typeof(state) == 'object') copy(state, xg);
        prng.state = function() { return copy(xg, {}); };
      }
      return prng;
    }

    if (module && module.exports) {
      module.exports = impl;
    } else if (define && define.amd) {
      define(function() { return impl; });
    } else {
      this.xorwow = impl;
    }

    })(
      commonjsGlobal,
      module,    // present in node.js
      (typeof undefined) == 'function'    // present with an AMD loader
    );
    });

    var xorshift7 = createCommonjsModule(function (module) {
    // A Javascript implementaion of the "xorshift7" algorithm by
    // François Panneton and Pierre L'ecuyer:
    // "On the Xorgshift Random Number Generators"
    // http://saluc.engr.uconn.edu/refs/crypto/rng/panneton05onthexorshift.pdf

    (function(global, module, define) {

    function XorGen(seed) {
      var me = this;

      // Set up generator function.
      me.next = function() {
        // Update xor generator.
        var X = me.x, i = me.i, t, v;
        t = X[i]; t ^= (t >>> 7); v = t ^ (t << 24);
        t = X[(i + 1) & 7]; v ^= t ^ (t >>> 10);
        t = X[(i + 3) & 7]; v ^= t ^ (t >>> 3);
        t = X[(i + 4) & 7]; v ^= t ^ (t << 7);
        t = X[(i + 7) & 7]; t = t ^ (t << 13); v ^= t ^ (t << 9);
        X[i] = v;
        me.i = (i + 1) & 7;
        return v;
      };

      function init(me, seed) {
        var j, X = [];

        if (seed === (seed | 0)) {
          // Seed state array using a 32-bit integer.
          X[0] = seed;
        } else {
          // Seed state using a string.
          seed = '' + seed;
          for (j = 0; j < seed.length; ++j) {
            X[j & 7] = (X[j & 7] << 15) ^
                (seed.charCodeAt(j) + X[(j + 1) & 7] << 13);
          }
        }
        // Enforce an array length of 8, not all zeroes.
        while (X.length < 8) X.push(0);
        for (j = 0; j < 8 && X[j] === 0; ++j);
        if (j == 8) X[7] = -1;

        me.x = X;
        me.i = 0;

        // Discard an initial 256 values.
        for (j = 256; j > 0; --j) {
          me.next();
        }
      }

      init(me, seed);
    }

    function copy(f, t) {
      t.x = f.x.slice();
      t.i = f.i;
      return t;
    }

    function impl(seed, opts) {
      if (seed == null) seed = +(new Date);
      var xg = new XorGen(seed),
          state = opts && opts.state,
          prng = function() { return (xg.next() >>> 0) / 0x100000000; };
      prng.double = function() {
        do {
          var top = xg.next() >>> 11,
              bot = (xg.next() >>> 0) / 0x100000000,
              result = (top + bot) / (1 << 21);
        } while (result === 0);
        return result;
      };
      prng.int32 = xg.next;
      prng.quick = prng;
      if (state) {
        if (state.x) copy(state, xg);
        prng.state = function() { return copy(xg, {}); };
      }
      return prng;
    }

    if (module && module.exports) {
      module.exports = impl;
    } else if (define && define.amd) {
      define(function() { return impl; });
    } else {
      this.xorshift7 = impl;
    }

    })(
      commonjsGlobal,
      module,    // present in node.js
      (typeof undefined) == 'function'    // present with an AMD loader
    );
    });

    var xor4096 = createCommonjsModule(function (module) {
    // A Javascript implementaion of Richard Brent's Xorgens xor4096 algorithm.
    //
    // This fast non-cryptographic random number generator is designed for
    // use in Monte-Carlo algorithms. It combines a long-period xorshift
    // generator with a Weyl generator, and it passes all common batteries
    // of stasticial tests for randomness while consuming only a few nanoseconds
    // for each prng generated.  For background on the generator, see Brent's
    // paper: "Some long-period random number generators using shifts and xors."
    // http://arxiv.org/pdf/1004.3115v1.pdf
    //
    // Usage:
    //
    // var xor4096 = require('xor4096');
    // random = xor4096(1);                        // Seed with int32 or string.
    // assert.equal(random(), 0.1520436450538547); // (0, 1) range, 53 bits.
    // assert.equal(random.int32(), 1806534897);   // signed int32, 32 bits.
    //
    // For nonzero numeric keys, this impelementation provides a sequence
    // identical to that by Brent's xorgens 3 implementaion in C.  This
    // implementation also provides for initalizing the generator with
    // string seeds, or for saving and restoring the state of the generator.
    //
    // On Chrome, this prng benchmarks about 2.1 times slower than
    // Javascript's built-in Math.random().

    (function(global, module, define) {

    function XorGen(seed) {
      var me = this;

      // Set up generator function.
      me.next = function() {
        var w = me.w,
            X = me.X, i = me.i, t, v;
        // Update Weyl generator.
        me.w = w = (w + 0x61c88647) | 0;
        // Update xor generator.
        v = X[(i + 34) & 127];
        t = X[i = ((i + 1) & 127)];
        v ^= v << 13;
        t ^= t << 17;
        v ^= v >>> 15;
        t ^= t >>> 12;
        // Update Xor generator array state.
        v = X[i] = v ^ t;
        me.i = i;
        // Result is the combination.
        return (v + (w ^ (w >>> 16))) | 0;
      };

      function init(me, seed) {
        var t, v, i, j, w, X = [], limit = 128;
        if (seed === (seed | 0)) {
          // Numeric seeds initialize v, which is used to generates X.
          v = seed;
          seed = null;
        } else {
          // String seeds are mixed into v and X one character at a time.
          seed = seed + '\0';
          v = 0;
          limit = Math.max(limit, seed.length);
        }
        // Initialize circular array and weyl value.
        for (i = 0, j = -32; j < limit; ++j) {
          // Put the unicode characters into the array, and shuffle them.
          if (seed) v ^= seed.charCodeAt((j + 32) % seed.length);
          // After 32 shuffles, take v as the starting w value.
          if (j === 0) w = v;
          v ^= v << 10;
          v ^= v >>> 15;
          v ^= v << 4;
          v ^= v >>> 13;
          if (j >= 0) {
            w = (w + 0x61c88647) | 0;     // Weyl.
            t = (X[j & 127] ^= (v + w));  // Combine xor and weyl to init array.
            i = (0 == t) ? i + 1 : 0;     // Count zeroes.
          }
        }
        // We have detected all zeroes; make the key nonzero.
        if (i >= 128) {
          X[(seed && seed.length || 0) & 127] = -1;
        }
        // Run the generator 512 times to further mix the state before using it.
        // Factoring this as a function slows the main generator, so it is just
        // unrolled here.  The weyl generator is not advanced while warming up.
        i = 127;
        for (j = 4 * 128; j > 0; --j) {
          v = X[(i + 34) & 127];
          t = X[i = ((i + 1) & 127)];
          v ^= v << 13;
          t ^= t << 17;
          v ^= v >>> 15;
          t ^= t >>> 12;
          X[i] = v ^ t;
        }
        // Storing state as object members is faster than using closure variables.
        me.w = w;
        me.X = X;
        me.i = i;
      }

      init(me, seed);
    }

    function copy(f, t) {
      t.i = f.i;
      t.w = f.w;
      t.X = f.X.slice();
      return t;
    }
    function impl(seed, opts) {
      if (seed == null) seed = +(new Date);
      var xg = new XorGen(seed),
          state = opts && opts.state,
          prng = function() { return (xg.next() >>> 0) / 0x100000000; };
      prng.double = function() {
        do {
          var top = xg.next() >>> 11,
              bot = (xg.next() >>> 0) / 0x100000000,
              result = (top + bot) / (1 << 21);
        } while (result === 0);
        return result;
      };
      prng.int32 = xg.next;
      prng.quick = prng;
      if (state) {
        if (state.X) copy(state, xg);
        prng.state = function() { return copy(xg, {}); };
      }
      return prng;
    }

    if (module && module.exports) {
      module.exports = impl;
    } else if (define && define.amd) {
      define(function() { return impl; });
    } else {
      this.xor4096 = impl;
    }

    })(
      commonjsGlobal,                                     // window object or global
      module,    // present in node.js
      (typeof undefined) == 'function'    // present with an AMD loader
    );
    });

    var tychei = createCommonjsModule(function (module) {
    // A Javascript implementaion of the "Tyche-i" prng algorithm by
    // Samuel Neves and Filipe Araujo.
    // See https://eden.dei.uc.pt/~sneves/pubs/2011-snfa2.pdf

    (function(global, module, define) {

    function XorGen(seed) {
      var me = this, strseed = '';

      // Set up generator function.
      me.next = function() {
        var b = me.b, c = me.c, d = me.d, a = me.a;
        b = (b << 25) ^ (b >>> 7) ^ c;
        c = (c - d) | 0;
        d = (d << 24) ^ (d >>> 8) ^ a;
        a = (a - b) | 0;
        me.b = b = (b << 20) ^ (b >>> 12) ^ c;
        me.c = c = (c - d) | 0;
        me.d = (d << 16) ^ (c >>> 16) ^ a;
        return me.a = (a - b) | 0;
      };

      /* The following is non-inverted tyche, which has better internal
       * bit diffusion, but which is about 25% slower than tyche-i in JS.
      me.next = function() {
        var a = me.a, b = me.b, c = me.c, d = me.d;
        a = (me.a + me.b | 0) >>> 0;
        d = me.d ^ a; d = d << 16 ^ d >>> 16;
        c = me.c + d | 0;
        b = me.b ^ c; b = b << 12 ^ d >>> 20;
        me.a = a = a + b | 0;
        d = d ^ a; me.d = d = d << 8 ^ d >>> 24;
        me.c = c = c + d | 0;
        b = b ^ c;
        return me.b = (b << 7 ^ b >>> 25);
      }
      */

      me.a = 0;
      me.b = 0;
      me.c = 2654435769 | 0;
      me.d = 1367130551;

      if (seed === Math.floor(seed)) {
        // Integer seed.
        me.a = (seed / 0x100000000) | 0;
        me.b = seed | 0;
      } else {
        // String seed.
        strseed += seed;
      }

      // Mix in string seed, then discard an initial batch of 64 values.
      for (var k = 0; k < strseed.length + 20; k++) {
        me.b ^= strseed.charCodeAt(k) | 0;
        me.next();
      }
    }

    function copy(f, t) {
      t.a = f.a;
      t.b = f.b;
      t.c = f.c;
      t.d = f.d;
      return t;
    }
    function impl(seed, opts) {
      var xg = new XorGen(seed),
          state = opts && opts.state,
          prng = function() { return (xg.next() >>> 0) / 0x100000000; };
      prng.double = function() {
        do {
          var top = xg.next() >>> 11,
              bot = (xg.next() >>> 0) / 0x100000000,
              result = (top + bot) / (1 << 21);
        } while (result === 0);
        return result;
      };
      prng.int32 = xg.next;
      prng.quick = prng;
      if (state) {
        if (typeof(state) == 'object') copy(state, xg);
        prng.state = function() { return copy(xg, {}); };
      }
      return prng;
    }

    if (module && module.exports) {
      module.exports = impl;
    } else if (define && define.amd) {
      define(function() { return impl; });
    } else {
      this.tychei = impl;
    }

    })(
      commonjsGlobal,
      module,    // present in node.js
      (typeof undefined) == 'function'    // present with an AMD loader
    );
    });

    var _nodeResolve_empty = {};

    var _nodeResolve_empty$1 = {
        __proto__: null,
        'default': _nodeResolve_empty
    };

    var require$$0 = /*@__PURE__*/getAugmentedNamespace(_nodeResolve_empty$1);

    /*
    Copyright 2019 David Bau.

    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
    IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
    CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
    TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
    SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

    */

    var seedrandom$1 = createCommonjsModule(function (module) {
    (function (global, pool, math) {
    //
    // The following constants are related to IEEE 754 limits.
    //

    var width = 256,        // each RC4 output is 0 <= x < 256
        chunks = 6,         // at least six RC4 outputs for each double
        digits = 52,        // there are 52 significant digits in a double
        rngname = 'random', // rngname: name for Math.random and Math.seedrandom
        startdenom = math.pow(width, chunks),
        significance = math.pow(2, digits),
        overflow = significance * 2,
        mask = width - 1,
        nodecrypto;         // node.js crypto module, initialized at the bottom.

    //
    // seedrandom()
    // This is the seedrandom function described above.
    //
    function seedrandom(seed, options, callback) {
      var key = [];
      options = (options == true) ? { entropy: true } : (options || {});

      // Flatten the seed string or build one from local entropy if needed.
      var shortseed = mixkey(flatten(
        options.entropy ? [seed, tostring(pool)] :
        (seed == null) ? autoseed() : seed, 3), key);

      // Use the seed to initialize an ARC4 generator.
      var arc4 = new ARC4(key);

      // This function returns a random double in [0, 1) that contains
      // randomness in every bit of the mantissa of the IEEE 754 value.
      var prng = function() {
        var n = arc4.g(chunks),             // Start with a numerator n < 2 ^ 48
            d = startdenom,                 //   and denominator d = 2 ^ 48.
            x = 0;                          //   and no 'extra last byte'.
        while (n < significance) {          // Fill up all significant digits by
          n = (n + x) * width;              //   shifting numerator and
          d *= width;                       //   denominator and generating a
          x = arc4.g(1);                    //   new least-significant-byte.
        }
        while (n >= overflow) {             // To avoid rounding up, before adding
          n /= 2;                           //   last byte, shift everything
          d /= 2;                           //   right using integer math until
          x >>>= 1;                         //   we have exactly the desired bits.
        }
        return (n + x) / d;                 // Form the number within [0, 1).
      };

      prng.int32 = function() { return arc4.g(4) | 0; };
      prng.quick = function() { return arc4.g(4) / 0x100000000; };
      prng.double = prng;

      // Mix the randomness into accumulated entropy.
      mixkey(tostring(arc4.S), pool);

      // Calling convention: what to return as a function of prng, seed, is_math.
      return (options.pass || callback ||
          function(prng, seed, is_math_call, state) {
            if (state) {
              // Load the arc4 state from the given state if it has an S array.
              if (state.S) { copy(state, arc4); }
              // Only provide the .state method if requested via options.state.
              prng.state = function() { return copy(arc4, {}); };
            }

            // If called as a method of Math (Math.seedrandom()), mutate
            // Math.random because that is how seedrandom.js has worked since v1.0.
            if (is_math_call) { math[rngname] = prng; return seed; }

            // Otherwise, it is a newer calling convention, so return the
            // prng directly.
            else return prng;
          })(
      prng,
      shortseed,
      'global' in options ? options.global : (this == math),
      options.state);
    }

    //
    // ARC4
    //
    // An ARC4 implementation.  The constructor takes a key in the form of
    // an array of at most (width) integers that should be 0 <= x < (width).
    //
    // The g(count) method returns a pseudorandom integer that concatenates
    // the next (count) outputs from ARC4.  Its return value is a number x
    // that is in the range 0 <= x < (width ^ count).
    //
    function ARC4(key) {
      var t, keylen = key.length,
          me = this, i = 0, j = me.i = me.j = 0, s = me.S = [];

      // The empty key [] is treated as [0].
      if (!keylen) { key = [keylen++]; }

      // Set up S using the standard key scheduling algorithm.
      while (i < width) {
        s[i] = i++;
      }
      for (i = 0; i < width; i++) {
        s[i] = s[j = mask & (j + key[i % keylen] + (t = s[i]))];
        s[j] = t;
      }

      // The "g" method returns the next (count) outputs as one number.
      (me.g = function(count) {
        // Using instance members instead of closure state nearly doubles speed.
        var t, r = 0,
            i = me.i, j = me.j, s = me.S;
        while (count--) {
          t = s[i = mask & (i + 1)];
          r = r * width + s[mask & ((s[i] = s[j = mask & (j + t)]) + (s[j] = t))];
        }
        me.i = i; me.j = j;
        return r;
        // For robust unpredictability, the function call below automatically
        // discards an initial batch of values.  This is called RC4-drop[256].
        // See http://google.com/search?q=rsa+fluhrer+response&btnI
      })(width);
    }

    //
    // copy()
    // Copies internal state of ARC4 to or from a plain object.
    //
    function copy(f, t) {
      t.i = f.i;
      t.j = f.j;
      t.S = f.S.slice();
      return t;
    }
    //
    // flatten()
    // Converts an object tree to nested arrays of strings.
    //
    function flatten(obj, depth) {
      var result = [], typ = (typeof obj), prop;
      if (depth && typ == 'object') {
        for (prop in obj) {
          try { result.push(flatten(obj[prop], depth - 1)); } catch (e) {}
        }
      }
      return (result.length ? result : typ == 'string' ? obj : obj + '\0');
    }

    //
    // mixkey()
    // Mixes a string seed into a key that is an array of integers, and
    // returns a shortened string seed that is equivalent to the result key.
    //
    function mixkey(seed, key) {
      var stringseed = seed + '', smear, j = 0;
      while (j < stringseed.length) {
        key[mask & j] =
          mask & ((smear ^= key[mask & j] * 19) + stringseed.charCodeAt(j++));
      }
      return tostring(key);
    }

    //
    // autoseed()
    // Returns an object for autoseeding, using window.crypto and Node crypto
    // module if available.
    //
    function autoseed() {
      try {
        var out;
        if (nodecrypto && (out = nodecrypto.randomBytes)) {
          // The use of 'out' to remember randomBytes makes tight minified code.
          out = out(width);
        } else {
          out = new Uint8Array(width);
          (global.crypto || global.msCrypto).getRandomValues(out);
        }
        return tostring(out);
      } catch (e) {
        var browser = global.navigator,
            plugins = browser && browser.plugins;
        return [+new Date, global, plugins, global.screen, tostring(pool)];
      }
    }

    //
    // tostring()
    // Converts an array of charcodes to a string
    //
    function tostring(a) {
      return String.fromCharCode.apply(0, a);
    }

    //
    // When seedrandom.js is loaded, we immediately mix a few bits
    // from the built-in RNG into the entropy pool.  Because we do
    // not want to interfere with deterministic PRNG state later,
    // seedrandom will not call math.random on its own again after
    // initialization.
    //
    mixkey(math.random(), pool);

    //
    // Nodejs and AMD support: export the implementation as a module using
    // either convention.
    //
    if (module.exports) {
      module.exports = seedrandom;
      // When in node.js, try using crypto package for autoseeding.
      try {
        nodecrypto = require$$0;
      } catch (ex) {}
    } else {
      // When included as a plain script, set up Math.seedrandom global.
      math['seed' + rngname] = seedrandom;
    }


    // End anonymous scope, and pass initial values.
    })(
      // global: `self` in browsers (including strict mode and web workers),
      // otherwise `this` in Node and other environments
      (typeof self !== 'undefined') ? self : commonjsGlobal,
      [],     // pool: entropy pool starts empty
      Math    // math: package containing random, pow, and seedrandom
    );
    });

    // A library of seedable RNGs implemented in Javascript.
    //
    // Usage:
    //
    // var seedrandom = require('seedrandom');
    // var random = seedrandom(1); // or any seed.
    // var x = random();       // 0 <= x < 1.  Every bit is random.
    // var x = random.quick(); // 0 <= x < 1.  32 bits of randomness.

    // alea, a 53-bit multiply-with-carry generator by Johannes Baagøe.
    // Period: ~2^116
    // Reported to pass all BigCrush tests.


    // xor128, a pure xor-shift generator by George Marsaglia.
    // Period: 2^128-1.
    // Reported to fail: MatrixRank and LinearComp.


    // xorwow, George Marsaglia's 160-bit xor-shift combined plus weyl.
    // Period: 2^192-2^32
    // Reported to fail: CollisionOver, SimpPoker, and LinearComp.


    // xorshift7, by François Panneton and Pierre L'ecuyer, takes
    // a different approach: it adds robustness by allowing more shifts
    // than Marsaglia's original three.  It is a 7-shift generator
    // with 256 bits, that passes BigCrush with no systmatic failures.
    // Period 2^256-1.
    // No systematic BigCrush failures reported.


    // xor4096, by Richard Brent, is a 4096-bit xor-shift with a
    // very long period that also adds a Weyl generator. It also passes
    // BigCrush with no systematic failures.  Its long period may
    // be useful if you have many generators and need to avoid
    // collisions.
    // Period: 2^4128-2^32.
    // No systematic BigCrush failures reported.


    // Tyche-i, by Samuel Neves and Filipe Araujo, is a bit-shifting random
    // number generator derived from ChaCha, a modern stream cipher.
    // https://eden.dei.uc.pt/~sneves/pubs/2011-snfa2.pdf
    // Period: ~2^127
    // No systematic BigCrush failures reported.


    // The original ARC4-based prng included in this library.
    // Period: ~2^1600


    seedrandom$1.alea = alea;
    seedrandom$1.xor128 = xor128;
    seedrandom$1.xorwow = xorwow;
    seedrandom$1.xorshift7 = xorshift7;
    seedrandom$1.xor4096 = xor4096;
    seedrandom$1.tychei = tychei;

    var seedrandom = seedrandom$1;

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // https://en.wikipedia.org/wiki/Marsaglia_polar_method
    class MPRandGauss {
        constructor(mean, stdDeviation, dtype, truncated, seed) {
            this.mean = mean;
            this.stdDev = stdDeviation;
            this.dtype = dtype;
            this.nextVal = NaN;
            this.truncated = truncated;
            if (this.truncated) {
                this.upper = this.mean + this.stdDev * 2;
                this.lower = this.mean - this.stdDev * 2;
            }
            const seedValue = seed ? seed : Math.random();
            this.random = seedrandom.alea(seedValue.toString());
        }
        /** Returns next sample from a Gaussian distribution. */
        nextValue() {
            if (!isNaN(this.nextVal)) {
                const value = this.nextVal;
                this.nextVal = NaN;
                return value;
            }
            let resultX, resultY;
            let isValid = false;
            while (!isValid) {
                let v1, v2, s;
                do {
                    v1 = 2 * this.random() - 1;
                    v2 = 2 * this.random() - 1;
                    s = v1 * v1 + v2 * v2;
                } while (s >= 1 || s === 0);
                const mul = Math.sqrt(-2.0 * Math.log(s) / s);
                resultX = this.mean + this.stdDev * v1 * mul;
                resultY = this.mean + this.stdDev * v2 * mul;
                if (!this.truncated || this.isValidTruncated(resultX)) {
                    isValid = true;
                }
            }
            if (!this.truncated || this.isValidTruncated(resultY)) {
                this.nextVal = this.convertValue(resultY);
            }
            return this.convertValue(resultX);
        }
        /** Handles proper rounding for non-floating-point numbers. */
        convertValue(value) {
            if (this.dtype == null || this.dtype === 'float32') {
                return value;
            }
            return Math.round(value);
        }
        /** Returns true if less than 2-standard-deviations from the mean. */
        isValidTruncated(value) {
            return value <= this.upper && value >= this.lower;
        }
    }
    // Marsaglia, George, and Wai Wan Tsang. 2000. "A Simple Method for Generating
    // Gamma Variables."
    class RandGamma {
        constructor(alpha, beta, dtype, seed) {
            this.alpha = alpha;
            this.beta = 1 / beta; // convert rate to scale parameter
            this.dtype = dtype;
            const seedValue = seed ? seed : Math.random();
            this.randu = seedrandom.alea(seedValue.toString());
            this.randn = new MPRandGauss(0, 1, dtype, false, this.randu());
            if (alpha < 1) {
                this.d = alpha + (2 / 3);
            }
            else {
                this.d = alpha - (1 / 3);
            }
            this.c = 1 / Math.sqrt(9 * this.d);
        }
        /** Returns next sample from a gamma distribution. */
        nextValue() {
            let x2, v0, v1, x, u, v;
            while (true) {
                do {
                    x = this.randn.nextValue();
                    v = 1 + (this.c * x);
                } while (v <= 0);
                v *= v * v;
                x2 = x * x;
                v0 = 1 - (0.331 * x2 * x2);
                v1 = (0.5 * x2) + (this.d * (1 - v + Math.log(v)));
                u = this.randu();
                if (u < v0 || Math.log(u) < v1) {
                    break;
                }
            }
            v = (1 / this.beta) * this.d * v;
            if (this.alpha < 1) {
                v *= Math.pow(this.randu(), 1 / this.alpha);
            }
            return this.convertValue(v);
        }
        /** Handles proper rounding for non-floating-point numbers. */
        convertValue(value) {
            if (this.dtype === 'float32') {
                return value;
            }
            return Math.round(value);
        }
    }
    class UniformRandom {
        constructor(min = 0, max = 1, dtype, seed) {
            /** Handles proper rounding for non floating point numbers. */
            this.canReturnFloat = () => (this.dtype == null || this.dtype === 'float32');
            this.min = min;
            this.range = max - min;
            this.dtype = dtype;
            if (seed == null) {
                seed = Math.random();
            }
            if (typeof seed === 'number') {
                seed = seed.toString();
            }
            if (!this.canReturnFloat() && this.range <= 1) {
                throw new Error(`The difference between ${min} - ${max} <= 1 and dtype is not float`);
            }
            this.random = seedrandom.alea(seed);
        }
        convertValue(value) {
            if (this.canReturnFloat()) {
                return value;
            }
            return Math.round(value);
        }
        nextValue() {
            return this.convertValue(this.min + this.range * this.random());
        }
    }

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values sampled from a gamma distribution.
     *
     * ```js
     * tf.randomGamma([2, 2], 1).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param alpha The shape parameter of the gamma distribution.
     * @param beta The inverse scale parameter of the gamma distribution. Defaults
     *     to 1.
     * @param dtype The data type of the output. Defaults to float32.
     * @param seed The seed for the random number generator.
     *
     * @doc {heading: 'Tensors', subheading: 'Random'}
     */
    function randomGamma_(shape, alpha, beta = 1, dtype = 'float32', seed) {
        if (beta == null) {
            beta = 1;
        }
        if (dtype == null) {
            dtype = 'float32';
        }
        if (dtype !== 'float32' && dtype !== 'int32') {
            throw new Error(`Unsupported data type ${dtype}`);
        }
        const rgamma = new RandGamma(alpha, beta, dtype, seed);
        const res = buffer(shape, dtype);
        for (let i = 0; i < res.values.length; i++) {
            res.values[i] = rgamma.nextValue();
        }
        return res.toTensor();
    }
    op({ randomGamma_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values sampled from a normal distribution.
     *
     * ```js
     * tf.randomNormal([2, 2]).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param mean The mean of the normal distribution.
     * @param stdDev The standard deviation of the normal distribution.
     * @param dtype The data type of the output.
     * @param seed The seed for the random number generator.
     *
     * @doc {heading: 'Tensors', subheading: 'Random'}
     */
    function randomNormal_(shape, mean = 0, stdDev = 1, dtype, seed) {
        if (dtype != null && dtype === 'bool') {
            throw new Error(`Unsupported data type ${dtype}`);
        }
        const randGauss = new MPRandGauss(mean, stdDev, dtype, false /* truncated */, seed);
        const res = buffer(shape, dtype);
        for (let i = 0; i < res.values.length; i++) {
            res.values[i] = randGauss.nextValue();
        }
        return res.toTensor();
    }
    const randomNormal$1 = op({ randomNormal_ });

    /**
     * @license
     * Copyright 2022 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values sampled from a normal distribution.
     *
     * The generated values will have mean 0 and standard deviation 1.
     *
     * ```js
     * tf.randomStandardNormal([2, 2]).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param dtype The data type of the output.
     * @param seed The seed for the random number generator.
     *
     * @doc {heading: 'Tensors', subheading: 'Random'}
     */
    function randomStandardNormal_(shape, dtype, seed) {
        if (dtype != null && dtype === 'bool') {
            throw new Error(`Unsupported data type ${dtype}`);
        }
        return randomNormal$1(shape, 0, 1, dtype, seed);
    }
    op({ randomStandardNormal_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values sampled from a uniform distribution.
     *
     * The generated values follow a uniform distribution in the range [minval,
     * maxval). The lower bound minval is included in the range, while the upper
     * bound maxval is excluded.
     *
     * ```js
     * tf.randomUniform([2, 2]).print();
     * ```
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param minval The lower bound on the range of random values to generate.
     *   Defaults to 0.
     * @param maxval The upper bound on the range of random values to generate.
     *   Defaults to 1.
     * @param dtype The data type of the output tensor. Defaults to 'float32'.
     *
     * @doc {heading: 'Tensors', subheading: 'Random'}
     */
    function randomUniform_(shape, minval = 0, maxval = 1, dtype = 'float32', seed) {
        const res = buffer(shape, dtype);
        const random = new UniformRandom(minval, maxval, null, seed);
        for (let i = 0; i < res.values.length; i++) {
            res.values[i] = random.nextValue();
        }
        return res.toTensor();
    }
    const randomUniform$1 = op({ randomUniform_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a new `tf.Tensor1D` filled with the numbers in the range provided.
     *
     * The tensor is a half-open interval meaning it includes start, but
     * excludes stop. Decrementing ranges and negative step values are also
     * supported.
     *
     *
     * ```js
     * tf.range(0, 9, 2).print();
     * ```
     *
     * @param start An integer start value
     * @param stop An integer stop value
     * @param step An integer increment (will default to 1 or -1)
     * @param dtype The data type of the output tensor. Defaults to 'float32'.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function range(start, stop, step = 1, dtype = 'float32') {
        if (step === 0) {
            throw new Error('Cannot have a step of zero');
        }
        const attrs = { start, stop, step, dtype };
        return ENGINE.runKernel(Range, {} /* inputs */, attrs);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes reciprocal of x element-wise: `1 / x`
     *
     * ```js
     * const x = tf.tensor1d([0, 1, 2]);
     *
     * x.reciprocal().print();  // or tf.reciprocal(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function reciprocal_(x) {
        const $x = convertToTensor(x, 'x', 'reciprocal');
        const inputs = { x: $x };
        return ENGINE.runKernel(Reciprocal, inputs);
    }
    op({ reciprocal_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes rectified linear element-wise: `max(x, 0)`.
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 4]);
     *
     * x.relu().print();  // or tf.relu(x)
     * ```
     * @param x The input tensor. If the dtype is `bool`, the output dtype will be
     *     `int32`.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function relu_(x) {
        const $x = convertToTensor(x, 'x', 'relu');
        const inputs = { x: $x };
        return ENGINE.runKernel(Relu$1, inputs);
    }
    const relu = op({ relu_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes rectified linear 6 element-wise: `min(max(x, 0), 6)`.
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 8]);
     *
     * x.relu6().print();  // or tf.relu6(x)
     * ```
     * @param x The input tensor. If the dtype is `bool`, the output dtype will be
     *     `int32`.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function relu6_(x) {
        const $x = convertToTensor(x, 'x', 'relu6');
        const inputs = { x: $x };
        return ENGINE.runKernel(Relu6$1, inputs);
    }
    const relu6 = op({ relu6_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reverses a `tf.Tensor` along a specified axis.
     *
     * Also available are stricter rank-specific methods that assert that `x` is
     * of the given rank:
     *   - `tf.reverse1d`
     *   - `tf.reverse2d`
     *   - `tf.reverse3d`
     *   - `tf.reverse4d`
     *
     * Except `tf.reverse1d` (which does not have axis param), all methods have
     * same signature as this method.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 3, 4]);
     *
     * x.reverse().print();
     * ```
     *
     * ```js
     * const x = tf.tensor2d([1, 2, 3, 4], [2, 2]);
     *
     * const axis = 1;
     * x.reverse(axis).print();
     * ```
     * @param x The input tensor to be reversed.
     * @param axis The set of dimensions to reverse. Must be in the
     *     range [-rank(x), rank(x)). Defaults to all axes.
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function reverse_(x, axis) {
        const $x = convertToTensor(x, 'x', 'reverse');
        const inputs = { x: $x };
        const attrs = { dims: axis };
        return ENGINE.runKernel(Reverse, inputs, attrs);
    }
    const reverse = op({ reverse_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reverses a `tf.Tensor1D`.
     *
     * @param x The input tensor.
     */
    function reverse1d_(x) {
        const $x = convertToTensor(x, 'x', 'reverse');
        assert($x.rank === 1, () => `Error in reverse1D: x must be rank 1 but got rank ${$x.rank}.`);
        return reverse($x, 0);
    }
    op({ reverse1d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reverses a `tf.Tensor2D` along a specified axis.
     *
     * @param x The input tensor.
     * @param axis The set of dimensions to reverse. Must be in the
     *     range [-rank(x), rank(x)). Defaults to all axes.
     */
    function reverse2d_(x, axis) {
        const $x = convertToTensor(x, 'x', 'reverse');
        assert($x.rank === 2, () => `Error in reverse2D: x must be rank 2 but got rank ${$x.rank}.`);
        return reverse($x, axis);
    }
    op({ reverse2d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reverses a `tf.Tensor3D` along a specified axis.
     *
     * @param x The input tensor.
     * @param axis The set of dimensions to reverse. Must be in the
     *     range [-rank(x), rank(x)). Defaults to all axes.
     */
    function reverse3d_(x, axis) {
        const $x = convertToTensor(x, 'x', 'reverse');
        assert($x.rank === 3, () => `Error in reverse3D: x must be rank 3 but got rank ${$x.rank}.`);
        return reverse($x, axis);
    }
    op({ reverse3d_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Reverses a `tf.Tensor4D` along a specified axis.
     *
     * @param x The input tensor.
     * @param axis The set of dimensions to reverse. Must be in the
     *     range [-rank(x), rank(x)). Defaults to all axes.
     */
    function reverse4d_(x, axis) {
        const $x = convertToTensor(x, 'x', 'reverse');
        assert($x.rank === 4, () => `Error in reverse4D: x must be rank 4 but got rank ${$x.rank}.`);
        return reverse($x, axis);
    }
    op({ reverse4d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes round of input `tf.Tensor` element-wise: `round(x)`.
     * It implements banker's rounding.
     *
     * ```js
     * const x = tf.tensor1d([.6, 1.1, -3.3]);
     *
     * x.round().print();  // or tf.round(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function round_(x) {
        const $x = convertToTensor(x, 'x', 'round');
        const inputs = { x: $x };
        return ENGINE.runKernel(Round, inputs);
    }
    const round = op({ round_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes scaled exponential linear element-wise.
     *
     * `x < 0 ? scale * alpha * (exp(x) - 1) : scale * x`
     *
     * ```js
     * const x = tf.tensor1d([-1, 2, -3, 4]);
     *
     * x.selu().print();  // or tf.selu(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function selu_(x) {
        const $x = convertToTensor(x, 'x', 'selu');
        const inputs = { x: $x };
        return ENGINE.runKernel(Selu$1, inputs);
    }
    op({ selu_ });

    /**
     * 2-D convolution with separable filters.
     *
     * Performs a depthwise convolution that acts separately on channels followed
     * by a pointwise convolution that mixes channels. Note that this is
     * separability between dimensions [1, 2] and 3, not spatial separability
     * between dimensions 1 and 2.
     *
     * See
     * [https://www.tensorflow.org/api_docs/python/tf/nn/separable_conv2d](
     *     https://www.tensorflow.org/api_docs/python/tf/nn/separable_conv2d)
     * for more details.
     *
     * @param x The input tensor, of rank 4 or rank 3, of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is
     * assumed.
     * @param depthwiseFilter The depthwise filter tensor, rank 4, of shape
     *     `[filterHeight, filterWidth, inChannels, channelMultiplier]`. This is
     *     the filter used in the first step.
     * @param pointwiseFilter The pointwise filter tensor, rank 4, of shape
     *     `[1, 1, inChannels * channelMultiplier, outChannels]`. This is
     *     the filter used in the second step.
     * @param strides The strides of the convolution: `[strideHeight,
     * strideWidth]`. If strides is a single number, then `strideHeight ==
     * strideWidth`.
     * @param pad The type of padding algorithm.
     *   - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *   - `valid`: output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     in atrous convolution. Defaults to `[1, 1]`. If `rate` is a single
     *     number, then `dilationHeight == dilationWidth`. If it is greater than
     *     1, then all values of `strides` must be 1.
     * @param dataFormat: An optional string from: "NHWC", "NCHW". Defaults to
     *     "NHWC". Specify the data format of the input and output data. With the
     *     default format "NHWC", the data is stored in the order of: [batch,
     *     height, width, channels]. Only "NHWC" is currently supported.
     *
     * @doc {heading: 'Operations', subheading: 'Convolution'}
     */
    function separableConv2d_(x, depthwiseFilter, pointwiseFilter, strides, pad, dilation = [1, 1], dataFormat = 'NHWC') {
        const $x = convertToTensor(x, 'x', 'separableConv2d');
        const $depthwiseFilter = convertToTensor(depthwiseFilter, 'depthwiseFilter', 'separableConv2d');
        const $pointwiseFilter = convertToTensor(pointwiseFilter, 'pointwiseFilter', 'separableConv2d');
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        if (dataFormat === 'NCHW') {
            throw new Error('separableConv2d currently does not support dataFormat NCHW; only ' +
                'NHWC is supported');
        }
        assert(x4D.rank === 4, () => `Error in separableConv2d: input must be rank 4, but got ` +
            `rank ${x4D.rank}.`);
        assert($depthwiseFilter.rank === 4, () => `Error in separableConv2d: depthwise filter must be rank 4, but ` +
            `got rank ${$depthwiseFilter.rank}.`);
        assert($pointwiseFilter.rank === 4, () => `Error in separableConv2d: pointwise filter must be rank 4, but ` +
            `got rank ${$depthwiseFilter.rank}.`);
        assert($pointwiseFilter.shape[0] === 1, () => `Error in separableConv2d: the first dimension of pointwise filter ` +
            ` must be 1, but got ${$pointwiseFilter.shape[0]}.`);
        assert($pointwiseFilter.shape[1] === 1, () => `Error in separableConv2d: the second dimension of pointwise ` +
            `filter must be 1, but got ${$pointwiseFilter.shape[1]}.`);
        const inChannels = $depthwiseFilter.shape[2];
        const channelMultiplier = $depthwiseFilter.shape[3];
        assert($pointwiseFilter.shape[2] === inChannels * channelMultiplier, () => `Error in separableConv2d: the third dimension of pointwise filter ` +
            `must be ${inChannels * channelMultiplier}, ` +
            `but got ${$pointwiseFilter.shape[2]}.`);
        const depthwise = depthwiseConv2d$2(x4D, $depthwiseFilter, strides, pad, dataFormat, dilation);
        const pointwiseStride = 1;
        const res = conv2d$1(depthwise, $pointwiseFilter, pointwiseStride, 'valid', dataFormat);
        if (reshapedTo4D) {
            return reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
        }
        return res;
    }
    op({ separableConv2d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns an element-wise indication of the sign of a number.
     *
     * ```js
     * const x = tf.tensor1d([.6, 1.1, -3.3, NaN, 0]);
     *
     * x.sign().print();  // or tf.sign(x)
     * ```
     * @param x The input Tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function sign_(x) {
        const $x = convertToTensor(x, 'x', 'sign');
        const inputs = { x: $x };
        return ENGINE.runKernel(Sign, inputs);
    }
    op({ sign_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Extracts a 1D slice from 1D array starting at coordinates `begin` and is
     * of length `size`. See `slice` for details.
     */
    function slice1d_(x, begin, size) {
        const $x = convertToTensor(x, 'x', 'slice1d');
        assert($x.rank === 1, () => `slice1d expects a rank-1 tensor, but got a rank-${$x.rank} tensor`);
        return slice($x, [begin], [size]);
    }
    op({ slice1d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Extracts a 2D slice from a 2D array starting at coordinates `begin` and
     * is of size `size`. See `slice` for details.
     */
    function slice2d_(x, begin, size) {
        const $x = convertToTensor(x, 'x', 'slice2d');
        assert($x.rank === 2, () => `slice2d expects a rank-2 tensor, but got a rank-${$x.rank} tensor`);
        return slice($x, begin, size);
    }
    op({ slice2d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Extracts a 3D slice from a 3D array starting at coordinates `begin` and
     * is of size `size`. See `slice` for details.
     */
    function slice3d_(x, begin, size) {
        const $x = convertToTensor(x, 'x', 'slice3d');
        assert($x.rank === 3, () => `slice3d expects a rank-3 tensor, but got a rank-${$x.rank} tensor`);
        return slice($x, begin, size);
    }
    op({ slice3d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Extracts a 4D slice from a 4D array starting at coordinates `begin` and
     * is of size `size`. See `slice` for details.
     */
    function slice4d_(x, begin, size) {
        const $x = convertToTensor(x, 'x', 'slice4d');
        assert($x.rank === 4, () => `slice4d expects a rank-4 tensor, but got a rank-${$x.rank} tensor`);
        return slice($x, begin, size);
    }
    op({ slice4d_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes the softmax normalized vector given the logits.
     *
     * ```js
     * const a = tf.tensor1d([1, 2, 3]);
     *
     * a.softmax().print();  // or tf.softmax(a)
     * ```
     *
     * ```js
     * const a = tf.tensor2d([2, 4, 6, 1, 2, 3], [2, 3]);
     *
     * a.softmax().print();  // or tf.softmax(a)
     * ```
     *
     * @param logits The logits array.
     * @param dim The dimension softmax would be performed on. Defaults to `-1`
     *     which indicates the last dimension.
     *
     * @doc {heading: 'Operations', subheading: 'Normalization'}
     */
    function softmax_(logits, dim = -1) {
        const $logits = convertToTensor(logits, 'logits', 'softmax', 'float32');
        if (dim === -1) {
            dim = $logits.rank - 1;
        }
        if (dim !== $logits.rank - 1) {
            throw Error('Softmax along a non-last dimension is not yet supported. ' +
                `Logits was rank ${$logits.rank} and dim was ${dim}`);
        }
        const inputs = { logits: $logits };
        const attrs = { dim };
        return ENGINE.runKernel(Softmax$2, inputs, attrs);
    }
    op({ softmax_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Fast Fourier transform.
     *
     * Computes the 1-dimensional discrete Fourier transform over the inner-most
     * dimension of input.
     *
     * ```js
     * const real = tf.tensor1d([1, 2, 3]);
     * const imag = tf.tensor1d([1, 2, 3]);
     * const x = tf.complex(real, imag);
     *
     * x.fft().print();  // tf.spectral.fft(x).print();
     * ```
     * @param input The complex input to compute an fft over.
     *
     * @doc {heading: 'Operations', subheading: 'Spectral', namespace: 'spectral'}
     */
    function fft_(input) {
        assert(input.dtype === 'complex64', () => `The dtype for tf.spectral.fft() must be complex64 ` +
            `but got ${input.dtype}.`);
        const inputs = { input };
        return ENGINE.runKernel(FFT, inputs);
    }
    const fft = op({ fft_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Inverse fast Fourier transform.
     *
     * Computes the inverse 1-dimensional discrete Fourier transform over the
     * inner-most dimension of input.
     *
     * ```js
     * const real = tf.tensor1d([1, 2, 3]);
     * const imag = tf.tensor1d([1, 2, 3]);
     * const x = tf.complex(real, imag);
     *
     * x.ifft().print();  // tf.spectral.ifft(x).print();
     * ```
     * @param input The complex input to compute an ifft over.
     *
     * @doc {heading: 'Operations', subheading: 'Spectral', namespace: 'spectral'}
     */
    function ifft_(input) {
        assert(input.dtype === 'complex64', () => `The dtype for tf.spectral.ifft() must be complex64 ` +
            `but got ${input.dtype}.`);
        const inputs = { input };
        return ENGINE.runKernel(IFFT, inputs);
    }
    const ifft = op({ ifft_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Inversed real value input fast Fourier transform.
     *
     * Computes the 1-dimensional inversed discrete Fourier transform over the
     * inner-most dimension of the real input.
     *
     * ```js
     * const real = tf.tensor1d([1, 2, 3]);
     * const imag = tf.tensor1d([0, 0, 0]);
     * const x = tf.complex(real, imag);
     *
     * x.irfft().print();
     * ```
     * @param input The real value input to compute an irfft over.
     *
     * @doc {heading: 'Operations', subheading: 'Spectral', namespace: 'spectral'}
     */
    function irfft_(input) {
        const innerDimensionSize = input.shape[input.shape.length - 1];
        const batch = input.size / innerDimensionSize;
        let ret;
        if (innerDimensionSize <= 2) {
            const complexInput = reshape$1(input, [batch, innerDimensionSize]);
            ret = ifft(complexInput);
        }
        else {
            // The length of unique components of the DFT of a real-valued signal
            // is 2 * (input_len - 1)
            const outputShape = [batch, 2 * (innerDimensionSize - 1)];
            const realInput = reshape$1(real(input), [batch, innerDimensionSize]);
            const imagInput = reshape$1(imag(input), [batch, innerDimensionSize]);
            const realConjugate = reverse(slice(realInput, [0, 1], [batch, innerDimensionSize - 2]), 1);
            const imagConjugate = mul(reverse(slice(imagInput, [0, 1], [batch, innerDimensionSize - 2]), 1), scalar(-1));
            const r = concat([realInput, realConjugate], 1);
            const i = concat([imagInput, imagConjugate], 1);
            const complexInput = reshape$1(complex(r, i), [outputShape[0], outputShape[1]]);
            ret = ifft(complexInput);
        }
        ret = real(ret);
        // reshape the result if the input is 3D tensor.
        if (input.rank === 3 && input.shape[0] !== 0) {
            const temp = ret;
            const batch = input.shape[0];
            ret = reshape$1(ret, [batch, ret.shape[0] / batch, ret.shape[1]]);
            temp.dispose();
        }
        return ret;
    }
    op({ irfft_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Real value input fast Fourier transform.
     *
     * Computes the 1-dimensional discrete Fourier transform over the
     * inner-most dimension of the real input.
     *
     * ```js
     * const real = tf.tensor1d([1, 2, 3]);
     *
     * real.rfft().print();
     * ```
     * @param input The real value input to compute an rfft over.
     *
     * @doc {heading: 'Operations', subheading: 'Spectral', namespace: 'spectral'}
     */
    function rfft_(input, fftLength) {
        assert(input.dtype === 'float32', () => `The dtype for rfft() must be real value but got ${input.dtype}`);
        let innerDimensionSize = input.shape[input.shape.length - 1];
        const batch = input.size / innerDimensionSize;
        let adjustedInput;
        if (fftLength != null && fftLength < innerDimensionSize) {
            // Need to crop
            const begin = input.shape.map(v => 0);
            const size = input.shape.map(v => v);
            size[input.shape.length - 1] = fftLength;
            adjustedInput = slice(input, begin, size);
            innerDimensionSize = fftLength;
        }
        else if (fftLength != null && fftLength > innerDimensionSize) {
            // Need to pad with zeros
            const zerosShape = input.shape.map(v => v);
            zerosShape[input.shape.length - 1] = fftLength - innerDimensionSize;
            adjustedInput = concat([input, zeros$1(zerosShape)], input.shape.length - 1);
            innerDimensionSize = fftLength;
        }
        else {
            adjustedInput = input;
        }
        // Complement the input with zero imaginary numbers.
        const zerosInput = zerosLike(adjustedInput);
        const complexInput = reshape$1(complex(adjustedInput, zerosInput), [batch, innerDimensionSize]);
        const ret = fft(complexInput);
        // Exclude complex conjugations. These conjugations are put symmetrically.
        const half = Math.floor(innerDimensionSize / 2) + 1;
        const realValues = real(ret);
        const imagValues = imag(ret);
        const realComplexConjugate = split(realValues, [half, innerDimensionSize - half], realValues.shape.length - 1);
        const imagComplexConjugate = split(imagValues, [half, innerDimensionSize - half], imagValues.shape.length - 1);
        const outputShape = adjustedInput.shape.slice();
        outputShape[adjustedInput.shape.length - 1] = half;
        return reshape$1(complex(realComplexConjugate[0], imagComplexConjugate[0]), outputShape);
    }
    const rfft = op({ rfft_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Returns (a - b) * (a - b) element-wise.
     * Supports broadcasting.
     *
     * ```js
     * const a = tf.tensor1d([1, 4, 3, 16]);
     * const b = tf.tensor1d([1, 2, 9, 4]);
     *
     * a.squaredDifference(b).print();  // or tf.squaredDifference(a, b)
     * ```
     *
     * ```js
     * // Broadcast squared difference  a with b.
     * const a = tf.tensor1d([2, 4, 6, 8]);
     * const b = tf.scalar(5);
     *
     * a.squaredDifference(b).print();  // or tf.squaredDifference(a, b)
     * ```
     *
     * @param a The first tensor.
     * @param b The second tensor. Must have the same type as `a`.
     *
     * @doc {heading: 'Operations', subheading: 'Arithmetic'}
     */
    function squaredDifference_(a, b) {
        let $a = convertToTensor(a, 'a', 'squaredDifference');
        let $b = convertToTensor(b, 'b', 'squaredDifference');
        [$a, $b] = makeTypesMatch($a, $b);
        assertAndGetBroadcastShape($a.shape, $b.shape);
        const inputs = { a: $a, b: $b };
        const attrs = {};
        return ENGINE.runKernel(SquaredDifference, inputs, attrs);
    }
    const squaredDifference = op({ squaredDifference_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Removes dimensions of size 1 from the shape of a `tf.Tensor`.
     *
     * ```js
     * const x = tf.tensor([1, 2, 3, 4], [1, 1, 4]);
     * x.squeeze().print();
     * ```
     *
     * @param x The input tensor to be squeezed.
     * @param axis An optional list of numbers. If specified, only
     *     squeezes the dimensions listed. The dimension index starts at 0. It
     * is an error to squeeze a dimension that is not 1.
     *
     * @doc {heading: 'Tensors', subheading: 'Transformations'}
     */
    function squeeze_(x, axis) {
        const $x = convertToTensor(x, 'x', 'squeeze', 'string_or_numeric');
        return reshape$1($x, squeezeShape($x.shape, axis).newShape);
    }
    const squeeze = op({ squeeze_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Stacks a list of rank-`R` `tf.Tensor`s into one rank-`(R+1)` `tf.Tensor`.
     *
     * ```js
     * const a = tf.tensor1d([1, 2]);
     * const b = tf.tensor1d([3, 4]);
     * const c = tf.tensor1d([5, 6]);
     * tf.stack([a, b, c]).print();
     * ```
     *
     * @param tensors A list of tensor objects with the same shape and dtype.
     * @param axis The axis to stack along. Defaults to 0 (the first dim).
     *
     * @doc {heading: 'Tensors', subheading: 'Slicing and Joining'}
     */
    function stack_(tensors, axis = 0) {
        const $tensors = convertToTensorArray(tensors, 'tensors', 'stack', 'string_or_numeric');
        assert($tensors.length >= 1, () => 'Pass at least one tensor to tf.stack');
        if ($tensors.length > 0) {
            assert(axis <= $tensors[0].rank, () => 'Axis must be <= rank of the tensor');
        }
        const inputs = $tensors;
        const attrs = { axis };
        return ENGINE.runKernel(Pack, inputs, attrs);
    }
    const stack = op({ stack_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Extracts a strided slice of a tensor.
     *
     * Roughly speaking, this op extracts a slice of size (end-begin)/stride from
     * the given input tensor (x). Starting at the location specified by begin the
     * slice continues by adding stride to the index until all dimensions are not
     * less than end. Note that a stride can be negative, which causes a reverse
     * slice.
     *
     * ```js
     * const t = tf.tensor3d([1, 1, 1 ,2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5, 6, 6, 6],
     *    [3, 2, 3]);
     * t.stridedSlice([1, 0, 0], [2, 1, 3], [1, 1, 1]).print()  // [[[3, 3, 3]]]
     * t.stridedSlice([1, 0, 0], [2, 2, 3], [1, 1, 1]).print()  // [[[3, 3, 3],
     *                                                     // [4, 4, 4]]]
     * t.stridedSlice([1, -1, 0], [2, -3, 3], [1, -1, 1]).print() // [[[4, 4, 4],
     *                                                     // [3, 3, 3]]]
     * ```
     *
     * @param x The tensor to stride slice.
     * @param begin The coordinates to start the slice from.
     * @param end: The coordinates to end the slice at.
     * @param strides: The size of the slice.
     * @param beginMask: If the ith bit of beginMask is set, begin[i] is ignored
     *      and the fullest possible range in that dimension is used instead.
     * @param endMask: If the ith bit of endMask is set, end[i] is ignored
     *      and the fullest possible range in that dimension is used instead.
     * @param shrinkAxisMask: a bitmask where bit i implies that
     * the ith specification should shrink the dimensionality. begin and end must
     * imply a slice of size 1 in the dimension.
     *
     * @doc {heading: 'Operations', subheading: 'Slicing and Joining'}
     */
    function stridedSlice_(x, begin, end, strides, beginMask = 0, endMask = 0, ellipsisMask = 0, newAxisMask = 0, shrinkAxisMask = 0) {
        const $x = convertToTensor(x, 'x', 'stridedSlice', 'string_or_numeric');
        const inputs = { x: $x };
        const attrs = {
            begin,
            end,
            strides,
            beginMask,
            endMask,
            ellipsisMask,
            newAxisMask,
            shrinkAxisMask
        };
        return ENGINE.runKernel(StridedSlice, inputs, attrs);
    }
    op({ stridedSlice_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes tan of the input `tf.Tensor` element-wise, `tan(x)`
     *
     * ```js
     * const x = tf.tensor1d([0, Math.PI / 2, Math.PI * 3 / 4]);
     *
     * x.tan().print();  // or tf.tan(x)
     * ```
     * @param x The input tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Basic math'}
     */
    function tan_(x) {
        const $x = convertToTensor(x, 'x', 'tan', 'float32');
        const inputs = { x: $x };
        return ENGINE.runKernel(Tan, inputs);
    }
    op({ tan_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates rank-1 `tf.Tensor` with the provided values, shape and dtype.
     *
     * The same functionality can be achieved with `tf.tensor`, but in general
     * we recommend using `tf.tensor1d` as it makes the code more readable.
     *
     * ```js
     * tf.tensor1d([1, 2, 3]).print();
     * ```
     *
     * @param values The values of the tensor. Can be array of numbers,
     *     or a `TypedArray`.
     * @param dtype The data type.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function tensor1d(values, dtype) {
        assertNonNull(values);
        const inferredShape = inferShape(values, dtype);
        if (inferredShape.length !== 1) {
            throw new Error('tensor1d() requires values to be a flat/TypedArray');
        }
        const shape = null;
        return makeTensor(values, shape, inferredShape, dtype);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates rank-2 `tf.Tensor` with the provided values, shape and dtype.
     *
     * The same functionality can be achieved with `tf.tensor`, but in general
     * we recommend using `tf.tensor2d` as it makes the code more readable.
     *
     *  ```js
     * // Pass a nested array.
     * tf.tensor2d([[1, 2], [3, 4]]).print();
     * ```
     * ```js
     * // Pass a flat array and specify a shape.
     * tf.tensor2d([1, 2, 3, 4], [2, 2]).print();
     * ```
     *
     * @param values The values of the tensor. Can be nested array of numbers,
     *     or a flat array, or a `TypedArray`.
     * @param shape The shape of the tensor. If not provided, it is inferred from
     *     `values`.
     * @param dtype The data type.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function tensor2d(values, shape, dtype) {
        assertNonNull(values);
        if (shape != null && shape.length !== 2) {
            throw new Error('tensor2d() requires shape to have two numbers');
        }
        const inferredShape = inferShape(values, dtype);
        if (inferredShape.length !== 2 && inferredShape.length !== 1) {
            throw new Error('tensor2d() requires values to be number[][] or flat/TypedArray');
        }
        if (inferredShape.length === 1 && shape == null) {
            throw new Error('tensor2d() requires shape to be provided when `values` ' +
                'are a flat/TypedArray');
        }
        return makeTensor(values, shape, inferredShape, dtype);
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Finds the values and indices of the `k` largest entries along the last
     * dimension.
     *
     * If the input is a vector (rank=1), finds the k largest entries in the vector
     * and outputs their values and indices as vectors. Thus values[j] is the j-th
     * largest entry in input, and its index is indices[j].
     * For higher rank inputs, computes the top k entries along the last dimension.
     *
     * If two elements are equal, the lower-index element appears first.
     *
     * ```js
     * const a = tf.tensor2d([[1, 5], [4, 3]]);
     * const {values, indices} = tf.topk(a);
     * values.print();
     * indices.print();
     * ```
     * @param x 1-D or higher `tf.Tensor` with last dimension being at least `k`.
     * @param k Number of top elements to look for along the last dimension.
     * @param sorted If true, the resulting `k` elements will be sorted by the
     *     values in descending order.
     *
     * @doc {heading: 'Operations', subheading: 'Evaluation'}
     */
    function topk_(x, k = 1, sorted = true) {
        const $x = convertToTensor(x, 'x', 'topk');
        if ($x.rank === 0) {
            throw new Error('topk() expects the input to be of rank 1 or higher');
        }
        const lastDim = $x.shape[$x.shape.length - 1];
        if (k < 0) {
            throw new Error(`'k' passed to topk() must be >= 0 but got ${k}`);
        }
        if (k > lastDim) {
            throw new Error(`'k' passed to topk() must be <= the last dimension (${lastDim}) ` +
                `but got ${k}`);
        }
        const inputs = { x: $x };
        const attrs = { k, sorted };
        const [values, indices] = ENGINE.runKernel(TopK, inputs, attrs);
        return { values, indices };
    }
    op({ topk_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a `tf.Tensor` with values sampled from a truncated normal
     * distribution.
     *
     * ```js
     * tf.truncatedNormal([2, 2]).print();
     * ```
     *
     * The generated values follow a normal distribution with specified mean and
     * standard deviation, except that values whose magnitude is more than 2
     * standard deviations from the mean are dropped and re-picked.
     *
     * @param shape An array of integers defining the output tensor shape.
     * @param mean The mean of the normal distribution.
     * @param stdDev The standard deviation of the normal distribution.
     * @param dtype The data type of the output tensor.
     * @param seed The seed for the random number generator.
     *
     * @doc {heading: 'Tensors', subheading: 'Creation'}
     */
    function truncatedNormal_(shape, mean = 0, stdDev = 1, dtype, seed) {
        if (dtype != null && dtype === 'bool') {
            throw new Error(`Unsupported data type $ { dtype }`);
        }
        const randGauss = new MPRandGauss(mean, stdDev, dtype, true /* truncated */, seed);
        const res = buffer(shape, dtype);
        for (let i = 0; i < res.values.length; i++) {
            res.values[i] = randGauss.nextValue();
        }
        return res.toTensor();
    }
    op({ truncatedNormal_ });

    /**
     * @license
     * Copyright 2020 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Finds unique elements along an axis of a tensor.
     *
     * It returns a tensor `values` containing all of the unique elements along the
     * `axis` of the given tensor `x` in the same order that they occur along the
     * `axis` in `x`; `x` does not need to be sorted. It also returns a tensor
     * `indices` the same size as the number of the elements in `x` along the `axis`
     * dimension. It contains the index in the unique output `values`.
     *
     * ```js
     * // A 1-D tensor
     * const a = tf.tensor1d([1, 1, 2, 4, 4, 4, 7, 8, 8]);
     * const {values, indices} = tf.unique(a);
     * values.print();   // [1, 2, 4, 7, 8,]
     * indices.print();  // [0, 0, 1, 2, 2, 2, 3, 4, 4]
     * ```
     *
     * ```js
     * // A 2-D tensor with axis=0
     * //
     * // 'a' is: [[1, 0, 0],
     * //          [1, 0, 0],
     * //          [2, 0, 0]]
     * const a = tf.tensor2d([[1, 0, 0], [1, 0, 0], [2, 0, 0]]);
     * const {values, indices} = tf.unique(a, 0)
     * values.print();   // [[1, 0, 0],
     *                   //  [2, 0, 0]]
     * indices.print();  // [0, 0, 1]
     * ```
     *
     * ```js
     * // A 2-D tensor with axis=1
     * //
     * // 'a' is: [[1, 0, 0],
     * //          [1, 0, 0],
     * //          [2, 0, 0]]
     * const a = tf.tensor2d([[1, 0, 0], [1, 0, 0], [2, 0, 0]]);
     * const {values, indices} = tf.unique(a, 1)
     * values.print();   // [[1, 0],
     *                   //  [1, 0],
     *                   //  [2, 0]]
     * indices.print();  // [0, 1, 1]
     * ```
     * @param x A tensor (int32, string, bool).
     * @param axis The axis of the tensor to find the unique elements.
     * @returns [uniqueElements, indices] (see above for details)
     *
     * @doc {heading: 'Operations', subheading: 'Evaluation'}
     */
    function unique_(x, axis = 0) {
        const $x = convertToTensor(x, 'x', 'unique', 'string_or_numeric');
        assert($x.rank > 0, () => 'The input tensor must be at least 1D');
        const inputs = { x: $x };
        const attrs = { axis };
        const [values, indices] = ENGINE.runKernel(Unique, inputs, attrs);
        return { values, indices };
    }
    op({ unique_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Compute the moving average of a variable.
     *
     * Without zeroDebias, the moving average operation is defined by:
     *   `v += delta`
     * where
     *   `delta = (1 - decay) * (x - v)`
     *
     * With zeroDebias (default), the `delta` term is scaled to debias the
     * effect of the (assumed) zero-initialization of `v`.
     *   `delta /= (1 - decay ^ step)`
     *
     * For more details on the zero-debiasing algorithm, see:
     *   https://arxiv.org/abs/1412.6980
     *
     * Note that this function is completely stateless and does not keep track of
     * step count. The step count needs to be maintained by the caller and passed
     * in as `step`.
     *
     * @param v The current moving average value.
     * @param x New input value, must have the same shape and dtype as `v`.
     * @param decay The decay factor. Typical values are 0.95 and 0.99.
     * @param step Step count.
     * @param zeroDebias: Whether zeroDebias is to be performed (default: `true`).
     * @returns The new moving average value.
     *
     * @doc {heading: 'Operations', subheading: 'Moving Average'}
     */
    function movingAverage_(v, x, decay, step, zeroDebias = true) {
        const $v = convertToTensor(v, 'v', 'movingAverage');
        const $x = convertToTensor(x, 'x', 'movingAverage');
        const $decay = convertToTensor(decay, 'decay', 'movingAverage');
        assertTypesMatch($v, $x);
        assert(arraysEqual($v.shape, $x.shape), () => 'Shape mismatch in v and x');
        const one = scalar(1);
        const oneMinusDecay = sub(one, $decay);
        let update = mul(sub($x, $v), oneMinusDecay);
        if (zeroDebias) {
            assert(step != null, () => 'When using zeroDebias: true, step is required.');
            const $step = convertToTensor(step, 'step', 'movingAverage');
            update = div(update, sub(one, pow($decay, $step)));
        }
        return add$1($v, update);
    }
    op({ movingAverage_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Creates a new tensor by applying sparse updates to individual
     * values or slices within a zero tensor of the given shape tensor according to
     * indices. This operator is the inverse of the `tf.gatherND` operator which
     * extracts values or slices from a given tensor.
     *
     * ```js
     * const indices = tf.tensor2d([4, 3, 1, 7], [4, 1], 'int32');
     * const updates = tf.tensor1d([9, 10, 11, 12]);
     * const shape = [8];
     * tf.scatterND(indices, updates, shape).print() //[0, 11, 0, 10, 9, 0, 0, 12]
     * ```
     *
     * @param indices The tensor contains the indices into the output tensor.
     * @param updates The tensor contains the value for the indices.
     * @param shape: The shape of the output tensor.
     *
     * @doc {heading: 'Operations', subheading: 'Slicing and Joining'}
     */
    function scatterND_(indices, updates, shape) {
        const $indices = convertToTensor(indices, 'indices', 'scatterND', 'int32');
        const $updates = convertToTensor(updates, 'updates', 'scatterND');
        validateInput$1($updates, $indices, shape);
        const inputs = { indices: $indices, updates: $updates };
        const attrs = { shape };
        // tslint:disable-next-line: no-unnecessary-type-assertion
        return ENGINE.runKernel(ScatterNd, inputs, attrs);
    }
    op({ scatterND_ });

    /**
     * Validate sparseToDense inputs.
     *
     * @param sparseIndices A 0-D, 1-D, or 2-D Tensor of type int32.
     * sparseIndices[i] contains the complete index where sparseValues[i] will be
     * placed.
     * @param sparseValues A 0-D or 1-D Tensor. Values
     * corresponding to each row of sparseIndices, or a scalar value to be used for
     * all sparse indices.
     * @param outputShape number[]. Shape of the dense output tensor.
     * @param validateIndices boolean. indice validation is not supported, error
     * will be thrown if it is set.
     */
    function validateInput(sparseIndices, sparseValues, outputShape, defaultValues) {
        if (sparseIndices.dtype !== 'int32') {
            throw new Error('tf.sparseToDense() expects the indices to be int32 type,' +
                ` but the dtype was ${sparseIndices.dtype}.`);
        }
        if (sparseIndices.rank > 2) {
            throw new Error('sparseIndices should be a scalar, vector, or matrix,' +
                ` but got shape ${sparseIndices.shape}.`);
        }
        const numElems = sparseIndices.rank > 0 ? sparseIndices.shape[0] : 1;
        const numDims = sparseIndices.rank > 1 ? sparseIndices.shape[1] : 1;
        if (outputShape.length !== numDims) {
            throw new Error('outputShape has incorrect number of elements:,' +
                ` ${outputShape.length}, should be: ${numDims}.`);
        }
        const numValues = sparseValues.size;
        if (!(sparseValues.rank === 0 ||
            sparseValues.rank === 1 && numValues === numElems)) {
            throw new Error('sparseValues has incorrect shape ' +
                `${sparseValues.shape}, should be [] or [${numElems}]`);
        }
        if (sparseValues.dtype !== defaultValues.dtype) {
            throw new Error('sparseValues.dtype must match defaultValues.dtype');
        }
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Converts a sparse representation into a dense tensor.
     *
     * Builds an array dense with shape outputShape such that:
     *
     * // If sparseIndices is scalar
     * dense[i] = (i == sparseIndices ? sparseValues : defaultValue)
     *
     * // If sparseIndices is a vector, then for each i
     * dense[sparseIndices[i]] = sparseValues[i]
     *
     * // If sparseIndices is an n by d matrix, then for each i in [0, n)
     * dense[sparseIndices[i][0], ..., sparseIndices[i][d-1]] = sparseValues[i]
     * All other values in dense are set to defaultValue. If sparseValues is a
     * scalar, all sparse indices are set to this single value.
     *
     * If indices are repeated the final value is summed over all values for those
     * indices.
     *
     * ```js
     * const indices = tf.tensor1d([4, 5, 6, 1, 2, 3], 'int32');
     * const values = tf.tensor1d([10, 11, 12, 13, 14, 15], 'float32');
     * const shape = [8];
     * tf.sparseToDense(indices, values, shape).print();
     * ```
     *
     * @param sparseIndices A 0-D, 1-D, or 2-D Tensor of type int32.
     * sparseIndices[i] contains the complete index where sparseValues[i] will be
     * placed.
     * @param sparseValues A 0-D or 1-D Tensor. Values
     * corresponding to each row of sparseIndices, or a scalar value to be used for
     * all sparse indices.
     * @param outputShape Shape of the dense output tensor. The type is inferred.
     * @param defaultValue Scalar. Value to set for indices not specified in
     * sparseIndices. Defaults to zero.
     *
     * @doc {heading: 'Operations', subheading: 'Normalization'}
     */
    function sparseToDense_(sparseIndices, sparseValues, outputShape, defaultValue = 0) {
        const $sparseIndices = convertToTensor(sparseIndices, 'sparseIndices', 'sparseToDense', 'int32');
        const $sparseValues = convertToTensor(sparseValues, 'sparseValues', 'sparseToDense', 'string_or_numeric');
        const $defaultValue = convertToTensor(defaultValue, 'defaultValue', 'sparseToDense', $sparseValues.dtype);
        validateInput($sparseIndices, $sparseValues, outputShape, $defaultValue);
        const inputs = {
            sparseIndices: $sparseIndices,
            sparseValues: $sparseValues,
            defaultValue: $defaultValue
        };
        const attrs = { outputShape };
        return ENGINE.runKernel(SparseToDense, inputs, attrs);
    }
    op({ sparseToDense_ });

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Gather slices from input tensor into a Tensor with shape specified by
     * `indices`.
     *
     * `indices` is a K-dimensional integer tensor, best thought of as a
     * (K-1)-dimensional tensor of indices into input, where each element defines a
     * slice of input:
     * output[\\(i_0, ..., i_{K-2}\\)] = input[indices[\\(i_0, ..., i_{K-2}\\)]]
     *
     * Whereas in `tf.gather`, `indices` defines slices into the first dimension of
     * input, in `tf.gatherND`, `indices` defines slices into the first N dimensions
     * of input, where N = indices.shape[-1].
     *
     * The last dimension of indices can be at most the rank of input:
     * indices.shape[-1] <= input.rank
     *
     * The last dimension of `indices` corresponds to elements
     * (if indices.shape[-1] == input.rank) or slices
     * (if indices.shape[-1] < input.rank) along dimension indices.shape[-1] of
     * input.
     * The output tensor has shape
     * indices.shape[:-1] + input.shape[indices.shape[-1]:]
     *
     * Note that on CPU, if an out of bound index is found, an error is returned. On
     * GPU, if an out of bound index is found, a 0 is stored in the corresponding
     * output value.
     *
     * ```js
     * const indices = tf.tensor2d([0, 1, 1, 0], [2,2], 'int32');
     * const input = tf.tensor2d([9, 10, 11, 12], [2, 2]);
     * tf.gatherND(input, indices).print() // [10, 11]
     * ```
     *
     * @param x The tensor from which to gather values.
     * @param indices Index tensor, must be of type int32.
     *
     * @doc {heading: 'Operations', subheading: 'Slicing and Joining'}
     */
    function gatherND_(x, indices) {
        const $indices = convertToTensor(indices, 'indices', 'gatherND', 'int32');
        const $x = convertToTensor(x, 'x', 'gatherND', 'string_or_numeric');
        const inputs = { params: $x, indices: $indices };
        return ENGINE.runKernel(GatherNd, inputs);
    }
    op({ gatherND_ });

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Normalize noise shape based on provided tensor and noise shape.
     *
     * @param x Tensor.
     * @param noiseShape The shape for the randomly generated keep/drop flags, as
     *   an array of numbers. Optional.
     * @returns Normalized noise shape.
     */
    function getNoiseShape(x, noiseShape) {
        if (noiseShape == null) {
            return x.shape.slice();
        }
        if (arraysEqual(x.shape, noiseShape)) {
            return noiseShape;
        }
        if (x.shape.length === noiseShape.length) {
            const newDimension = [];
            for (let i = 0; i < x.shape.length; i++) {
                if (noiseShape[i] == null && x.shape[i] != null) {
                    newDimension.push(x.shape[i]);
                }
                else {
                    newDimension.push(noiseShape[i]);
                }
            }
            return newDimension;
        }
        return noiseShape;
    }

    /**
     * @license
     * Copyright 2018 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes dropout.
     *
     * ```js
     * const x = tf.tensor1d([1, 2, 2, 1]);
     * const rate = 0.75;
     * const output = tf.dropout(x, rate);
     * output.print();
     * ```
     *
     * @param x A floating point Tensor or TensorLike.
     * @param rate A float in the range [0, 1). The probability that each element
     *   of x is discarded.
     * @param noiseShape An array of numbers of type int32, representing the
     * shape for randomly generated keep/drop flags. If the noiseShape has null
     * value, it will be automatically replaced with the x's relative dimension
     * size. Optional.
     * @param seed Used to create random seeds. Optional.
     * @returns A Tensor of the same shape of x.
     *
     * @doc {heading: 'Operations', subheading: 'Dropout'}
     */
    function dropout_(x, rate, noiseShape, seed) {
        const $x = convertToTensor(x, 'x', 'dropout');
        assert($x.dtype === 'float32', () => `x has to be a floating point tensor since it's going to be ` +
            `scaled, but got a ${$x.dtype} tensor instead.`);
        assert(rate >= 0 && rate < 1, () => `rate must be a float in the range [0, 1), but got ${rate}.`);
        if (rate === 0) {
            return x instanceof Tensor ? $x.clone() : $x;
        }
        const $noiseShape = getNoiseShape($x, noiseShape);
        const keepProb = 1 - rate;
        const multiplier = div(floor(add$1(randomUniform$1($noiseShape, 0, 1, 'float32', seed), keepProb)), keepProb);
        return mul($x, multiplier);
    }
    op({ dropout_ });

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    function enclosingPowerOfTwo(value) {
        // Return 2**N for integer N such that 2**N >= value.
        return Math.floor(Math.pow(2, Math.ceil(Math.log(value) / Math.log(2.0))));
    }
    function cosineWindow(windowLength, a, b) {
        const even = 1 - windowLength % 2;
        const newValues = new Float32Array(windowLength);
        for (let i = 0; i < windowLength; ++i) {
            const cosArg = (2.0 * Math.PI * i) / (windowLength + even - 1);
            newValues[i] = a - b * Math.cos(cosArg);
        }
        return tensor1d(newValues, 'float32');
    }

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    // Returns gradient for fused activation.
    function getFusedDyActivation(dy, y, activation) {
        if (activation == null || activation === 'linear') {
            return dy;
        }
        if (activation === 'relu') {
            return mul(dy, step(y));
        }
        throw new Error(`Cannot compute gradient for fused activation ${activation}.`);
    }
    // Returns gradient for fused bias.
    function getFusedBiasGradient(bias, dyActivation) {
        let res = dyActivation;
        const reduceAxes = getReductionAxes(bias.shape, dyActivation.shape);
        if (reduceAxes.length > 0) {
            res = sum(res, reduceAxes);
        }
        return reshape$1(res, bias.shape);
    }
    function applyActivation(x, activation, preluActivationWeights, leakyreluAlpha) {
        if (activation === 'linear') {
            return x;
        }
        else if (activation === 'relu') {
            return relu(x);
        }
        else if (activation === 'elu') {
            return elu$1(x);
        }
        else if (activation === 'relu6') {
            return relu6(x);
        }
        else if (activation === 'prelu') {
            return prelu$1(x, preluActivationWeights);
        }
        else if (activation === 'leakyrelu') {
            return leakyRelu(x, leakyreluAlpha);
        }
        else if (activation === 'sigmoid') {
            return sigmoid(x);
        }
        throw new Error(`Unknown fused activation ${activation}.`);
    }
    // Whether we should call fused ops.
    const shouldFuse = (gradientDepth, activation) => {
        const gradientMode = gradientDepth > 0;
        return !gradientMode || activation === 'linear';
    };

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes a 2D convolution over the input x, optionally fused with adding a
     * bias and applying an activation.
     *
     * ```js
     * const inputDepth = 2;
     * const inShape = [2, 2, 2, inputDepth];
     * const outputDepth = 2;
     * const fSize = 1;
     * const pad = 0;
     * const strides = 1;
     *
     * const x = tf.tensor4d( [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
     * 16], inShape);
     * const w = tf.tensor4d([-1, 1, -2, 0.5], [fSize, fSize, inputDepth,
     * outputDepth]);
     *
     * tf.fused.conv2d({ x, filter: w, strides, pad, dataFormat: 'NHWC',
     * dilations: [1, 1], bias: tf.scalar(5), activation: 'relu' }).print();
     * ```
     *
     * @param obj An object with the following properties:
     * @param x The input tensor, of rank 4 or rank 3, of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is
     * assumed.
     * @param filter The filter, rank 4, of shape
     *     `[filterHeight, filterWidth, inDepth, outDepth]`.
     * @param strides The strides of the convolution: `[strideHeight,
     * strideWidth]`.
     * @param pad The type of padding algorithm.
     *   - `same` and stride 1: output will be of same size as input,
     *       regardless of filter size.
     *   - `valid` output will be smaller than input if filter is larger
     *       than 1x1.
     *   - For more info, see this guide:
     *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
     *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
     * @param dataFormat An optional string from: "NHWC", "NCHW". Defaults to
     *     "NHWC". Specify the data format of the input and output data. With the
     *     default format "NHWC", the data is stored in the order of: [batch,
     *     height, width, channels]. Only "NHWC" is currently supported.
     * @param dilations The dilation rates: `[dilationHeight, dilationWidth]`
     *     in which we sample input values across the height and width dimensions
     *     in atrous convolution. Defaults to `[1, 1]`. If `dilations` is a single
     *     number, then `dilationHeight == dilationWidth`. If it is greater than
     *     1, then all values of `strides` must be 1.
     * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
     *     provided, it will default to truncate.
     * @param bias Tensor to be added to the result.
     * @param activation Name of activation kernel (defaults to `linear`) to be
     *     applied
     *      after biasAdd.
     * @param preluActivationWeights Tensor of prelu weights to be applied as part
     *     of a `prelu` activation, typically the same shape as `x`.
     * @param leakyreluAlpha Optional. Alpha to be applied as part of a `leakyrelu`
     *     activation.
     */
    function fusedConv2d_({ x, filter, strides, pad, dataFormat = 'NHWC', dilations = [1, 1], dimRoundingMode, bias, activation = 'linear', preluActivationWeights, leakyreluAlpha }) {
        activation = activation || 'linear';
        if (shouldFuse(ENGINE.state.gradientDepth, activation) === false) {
            // TODO: Transpose bias and preluActivationWeights properly for NCHW
            // format before computation.
            assert(dataFormat === 'NHWC', () => `Error in fused conv2d: got dataFormat of ${dataFormat} but ` +
                `only NHWC is currently supported for the case of gradient depth ` +
                `is 0 and the activation is not linear.`);
            let result = conv2d$1(x, filter, strides, pad, dataFormat, dilations, dimRoundingMode);
            if (bias != null) {
                result = add$1(result, bias);
            }
            return applyActivation(result, activation, preluActivationWeights, leakyreluAlpha);
        }
        const $x = convertToTensor(x, 'x', 'conv2d', 'float32');
        const $filter = convertToTensor(filter, 'filter', 'conv2d', 'float32');
        let x4D = $x;
        let reshapedTo4D = false;
        if ($x.rank === 3) {
            reshapedTo4D = true;
            x4D = reshape$1($x, [1, $x.shape[0], $x.shape[1], $x.shape[2]]);
        }
        assert(x4D.rank === 4, () => `Error in fused conv2d: input must be rank 4, but got rank ` +
            `${x4D.rank}.`);
        assert($filter.rank === 4, () => `Error in fused conv2d: filter must be rank 4, but got rank ` +
            `${$filter.rank}.`);
        checkPadOnDimRoundingMode('fused conv2d', pad, dimRoundingMode);
        const inputChannels = dataFormat === 'NHWC' ? x4D.shape[3] : x4D.shape[1];
        assert($filter.shape[2] === inputChannels, () => `Error in conv2d: depth of input (${inputChannels}) must match ` +
            `input depth for filter ${$filter.shape[2]}.`);
        assert(eitherStridesOrDilationsAreOne(strides, dilations), () => 'Error in conv2D: Either strides or dilations must be 1. ' +
            `Got strides ${strides} and dilations '${dilations}'`);
        const convInfo = computeConv2DInfo(x4D.shape, $filter.shape, strides, dilations, pad, dimRoundingMode);
        let $bias;
        if (bias != null) {
            $bias = convertToTensor(bias, 'bias', 'fused conv2d');
            [$bias] = makeTypesMatch($bias, $x);
            // According to TensorFlow, the bias is supposed be a 1-D tensor or a
            // scalar.
            //
            // 3-D or 4-D bias is not disabled for NHWC format, because they are
            // currently being used in some cases. For examplem in our code base,
            // https://github.com/tensorflow/tfjs/blob/b53bd47e880367ae57493f0ea628abaf08db2d5d/tfjs-core/src/ops/fused/fused_conv2d_test.ts#L1972.
            if (dataFormat === 'NHWC') {
                assertAndGetBroadcastShape(convInfo.outShape, $bias.shape);
            }
            else {
                assert($bias.shape.length <= 1, () => `Error in fused conv2d: only supports scalar or 1-D Tensor ` +
                    `bias for NCHW format but got the bias of ` +
                    `rank-${$bias.shape.length}.`);
                assert($bias.shape.length === 0 || $bias.shape[0] === convInfo.outChannels ||
                    $bias.shape[0] === 1, () => `Error in fused conv2d: bias shape (${$bias.shape}) is not ` +
                    `compatible with the number of output channels ` +
                    `(${convInfo.outChannels})`);
            }
        }
        let $preluActivationWeights;
        if (preluActivationWeights != null) {
            // PReLU's activation weights could be a scalar, a 1-D tensor or a 3-D
            // tensor.
            const alphaShape = preluActivationWeights.shape;
            assert(alphaShape.length <= 1 || alphaShape.length === 3, () => `Error in fused conv2d: only supports scalar, 1-D Tensor or ` +
                `3-D Tensor PReLU activation weights but got a tensor of ` +
                `rank-${alphaShape.length}.`);
            if (alphaShape.length === 1) {
                // Whether the data format is NCHW or NHWC, the 1-D PReLU activation
                // weights tensor should be aligned with the output channels of conv2d
                // result.
                assert(alphaShape[0] === 1 || alphaShape[0] === convInfo.outChannels, () => `Error in fused conv2d: PReLU activation weights ` +
                    `(${alphaShape}) is not compatible with the number of output ` +
                    `channels (${convInfo.outChannels}).`);
            }
            else if (alphaShape.length === 3) {
                // Whether the data format is NCHW or NHWC, the PReLU activation weights
                // tensor should has the compatible shape with the result of conv2d.
                try {
                    assertAndGetBroadcastShape(alphaShape, convInfo.outShape);
                }
                catch (e) {
                    const errMsg = `Error in fused conv2d: PReLU activation weights (${alphaShape}) ` +
                        `is not compatible with the output shape of the conv2d ` +
                        `(${convInfo.outShape}).`;
                    throw Error(errMsg);
                }
            }
            $preluActivationWeights = convertToTensor(preluActivationWeights, 'prelu weights', 'fused conv2d');
        }
        const grad = (dy, saved) => {
            assert(dataFormat === 'NHWC', () => `Error in gradient of fused conv2D: got dataFormat of ${dataFormat} but only NHWC is currently supported.`);
            const [$filter, x4D, y, $bias] = saved;
            const dyActivation = getFusedDyActivation(dy, y, activation);
            assert(tupleValuesAreOne(dilations), () => 'Error in gradient of fused conv2D: ' +
                `dilation rates greater than 1 ` +
                `are not yet supported in gradients. Got dilations '${dilations}'`);
            const xDer = conv2DBackpropInput(x4D.shape, dyActivation, $filter, strides, pad);
            const filterDer = conv2DBackpropFilter(x4D, dyActivation, $filter.shape, strides, pad);
            const der = [xDer, filterDer];
            if ($bias != null) {
                const biasDer = getFusedBiasGradient($bias, dyActivation);
                der.push(biasDer);
            }
            return der;
        };
        const inputs = {
            x: x4D,
            filter: $filter,
            bias: $bias,
            preluActivationWeights: $preluActivationWeights
        };
        const attrs = {
            strides,
            pad,
            dataFormat,
            dilations,
            dimRoundingMode,
            activation,
            leakyreluAlpha
        };
        // Depending on the the params passed in we will have different number of
        // inputs and thus a a different number of elements in the gradient.
        if (bias == null) {
            const customOp = customGrad((x4D, filter, save) => {
                let res = 
                // tslint:disable-next-line: no-unnecessary-type-assertion
                ENGINE.runKernel(FusedConv2D, inputs, attrs);
                save([filter, x4D, res]);
                if (reshapedTo4D) {
                    // tslint:disable-next-line: no-unnecessary-type-assertion
                    res = reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
                }
                return { value: res, gradFunc: grad };
            });
            return customOp(x4D, $filter);
        }
        else {
            const customOpWithBias = customGrad((x4D, filter, bias, save) => {
                let res = ENGINE.runKernel(FusedConv2D, inputs, attrs);
                save([filter, x4D, res, bias]);
                if (reshapedTo4D) {
                    // tslint:disable-next-line: no-unnecessary-type-assertion
                    res = reshape$1(res, [res.shape[1], res.shape[2], res.shape[3]]);
                }
                return { value: res, gradFunc: grad };
            });
            return customOpWithBias(x4D, $filter, $bias);
        }
    }
    op({ fusedConv2d_ });

    /**
     * @license
     * Copyright 2019 Google LLC. All Rights Reserved.
     * Licensed under the Apache License, Version 2.0 (the "License");
     * you may not use this file except in compliance with the License.
     * You may obtain a copy of the License at
     *
     * http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     * =============================================================================
     */
    /**
     * Computes depthwise 2D convolution, optionally fused with adding a
     * bias and applying an activation.
     *
     * Given a 4D `input` array and a `filter` array of shape
     * `[filterHeight, filterWidth, inChannels, channelMultiplier]` containing
     * `inChannels` convolutional filters of depth 1, this op applies a
     * different filter to each input channel (expanding from 1 channel to
     * `channelMultiplier` channels for each), then concatenates the results
     * together. The output has `inChannels * channelMultiplier` channels.
     *
     * See
     * [https://www.tensorflow.org/api_docs/python/tf/nn/depthwise_conv2d](
     *     https://www.tensorflow.org/api_docs/python/tf/nn/depthwise_conv2d)
     * for more details.
     *
     * @param obj An object with the following properties:
     * @param x The input tensor, of rank 4 or rank 3, of shape
     *     `[batch, height, width, inChannels]`. If rank 3, batch of 1 is
     * assumed.
     * @param filter The filter tensor, rank 4, of shape
     *     `[filterHeight, filterWidth, inChannels, channelMultiplier]`.
     * @param strides The strides of the convolution: `[strideHeigh