'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var arrayUtils = require('@ultraq/array-utils');
var stringUtils = require('@ultraq/string-utils');
var domUtils = require('@ultraq/dom-utils');
var dumbQuerySelector = require('dumb-query-selector');

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function ownKeys(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);
    if (enumerableOnly) symbols = symbols.filter(function (sym) {
      return Object.getOwnPropertyDescriptor(object, sym).enumerable;
    });
    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread2(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys(Object(source), true).forEach(function (key) {
        _defineProperty(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys(Object(source)).forEach(function (key) {
        Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
      });
    }
  }

  return target;
}

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * A special kind of expression that requires the referenced rule consume all
 * available input.
 * 
 * @param {String} ruleName
 * @return {Matchable}
 */
const AllInput = ruleName => (input, parser) => {
  let matchResult = parser.parseWithExpression(input, ruleName);
  return matchResult !== null && input.exhausted() ? matchResult : null;
};

/*
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Default processor which returns the result as is.
 * 
 * @template T
 * @param {T} result
 * @return {T}
 */
const defaultMatchProcessor = result => result;
/**
 * A rule describes a string in the language.
 * 
 * @author Emanuel Rabina
 */


class Rule {
  /**
   * @param {String} name
   * @param {Object} expression
   * @param {Function} [matchProcessor=defaultMatchProcessor]
   */
  constructor(name, expression, matchProcessor = defaultMatchProcessor) {
    this.name = name;
    this.expression = expression;
    this.matchProcessor = matchProcessor;
  }
  /**
   * Given an input string and a parser, return whether or not the input is
   * accepted by this rule.
   * 
   * @param {InputBuffer} input
   * @param {Parser} parser
   * @return {Object} If the input is accepted, this will be the non-null result
   *   of matching against the rule.
   */


  accept(input, parser) {
    let {
      expression,
      name
    } = this;
    return parser.trackExpression(input, expression, name, () => {
      let matchResult = parser.parseWithExpression(input, expression);
      return matchResult !== null ? this.matchProcessor(matchResult) : null;
    });
  }

}

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A custom rule where the default match processor returns a function to execute
 * against the current context to discover the matched value.
 */

class ThymeleafRule extends Rule {
  /**
   * @param {String} name
   * @param {Object} expression
   * @param {Function} [matchProcessor]
   */
  constructor(name, expression, matchProcessor) {
    const contextSensitiveMatchProcessor = result => (...args) => {
      // console.log(`Processing rule: ${name}`);
      return typeof result === 'function' ? result.apply(null, args) : result;
    };

    super(name, expression, matchProcessor || contextSensitiveMatchProcessor);
  }

}

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * A collection of Rules that describes a language.
 * 
 * @author Emanuel Rabina
 */
class Grammar {
  /**
   * @param {String} name
   * @param {Rule} startingRule
   * @param {...Rule} additionalRules
   */
  constructor(name, startingRule, ...additionalRules) {
    this.name = name;
    this.rules = [].concat(startingRule, additionalRules);
  }
  /**
   * Given an input string and a parser, return whether or not the input is
   * accepted by this grammar.
   * 
   * @param {InputBuffer} input
   * @param {Parser} parser
   * @return {Object} If the input is accepted, this will be the non-null result
   *   of matching against the rules of this grammar.
   */


  accept(input, parser) {
    return this.startingRule.accept(input, parser);
  }
  /**
   * Return the rule that has the given name.
   * 
   * @param {String} name
   * @return {Rule}
   */


  findRuleByName(name) {
    let rule = this.rules.find(rule => rule.name === name);

    if (!rule) {
      throw new Error(`Failed to find a rule named "${name}" in the grammar`);
    }

    return rule;
  }
  /**
   * Returns the grammar's starting rule.
   * 
   * @return {Rule}
   */


  get startingRule() {
    return this.rules[0];
  }

}

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Returns an expression function where the underlying expression doesn't need
 * to be matched.  Thus, optional expressions "always" match.
 * 
 * @param {Matchable} expression
 * @return {Matchable}
 */
const Optional = expression => (input, parser) => {
  return input.markAndClearOrReset(() => {
    let result = parser.parseWithExpression(input, expression);
    return result !== null ? result : '';
  });
};
/**
 * Returns an expression function where the expression must be matched against
 * at least once to be considered a match.
 * 
 * @param {Matchable} expression
 * @return {Matchable}
 */

const OneOrMore = expression => (input, parser) => {
  return input.markAndClearOrReset(() => {
    let results = [];

    while (true) {
      let result = input.markAndClearOrReset(() => {
        return parser.parseWithExpression(input, expression);
      });

      if (result) {
        results.push(result);
      } else {
        break;
      }
    }

    return results.length > 0 ? results : null;
  });
};
/**
 * Returns an expression function where only one of the underlying expressions
 * must be matched in order to consider the expression a match.
 * 
 * @param {...Matchable} expressions
 * @return {Matchable}
 */

const OrderedChoice = (...expressions) => (input, parser) => {
  return input.markAndClearOrReset(() => {
    for (let expression of expressions) {
      let result = input.markAndClearOrReset(() => {
        return parser.parseWithExpression(input, expression);
      });

      if (result !== null) {
        return result;
      }
    }

    return null;
  });
};
/**
 * Returns an expression whose underlying expressions must be matched in order
 * to consider the entire expression a match.
 * 
 * @param {...Matchable} expressions
 * @return {Matchable}
 */

const Sequence = (...expressions) => (input, parser) => {
  return input.markAndClearOrReset(() => {
    let results = [];

    for (let expression of expressions) {
      let result = input.markAndClearOrReset(() => {
        return parser.parseWithExpression(input, expression);
      });

      if (result === null) {
        return null;
      }

      results.push(result);
    }

    return results;
  });
};
/**
 * Returns an expression function where the expression can be matched a repeated
 * number of times or none at all to be considered a match.
 * 
 * @param {Matchable} expression
 * @return {Matchable}
 */

const ZeroOrMore = expression => (input, parser) => {
  return input.markAndClearOrReset(() => {
    let results = [];

    while (true) {
      let result = input.markAndClearOrReset(() => {
        return parser.parseWithExpression(input, expression);
      });

      if (result) {
        results.push(result);
      } else {
        break;
      }
    }

    return results;
  });
};

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Wrapper around the string being parsed, with a position that keeps track of
 * what part of the imput is currently being read/tested.
 * 
 * @author Emanuel Rabina
 */
class InputBuffer {
  /**
   * @private
   * @type {Number}
   */

  /**
   * @private
   * @type {Array<Number>}
   */

  /**
   * @param {String} input
   */
  constructor(input) {
    _defineProperty(this, "position", 0);

    _defineProperty(this, "positionStack", []);

    this.input = input;
  }
  /**
   * Clear the previously {@link #mark}ed position.
   */


  clear() {
    let lastPosition = this.positionStack.pop();

    if (lastPosition === undefined) {
      throw new Error('Called clear() but no matching mark()');
    }
  }
  /**
   * Returns whether or not the current position is at the end of the input,
   * meaning we've exhausted the entire input string.
   * 
   * @return {Boolean}
   */


  exhausted() {
    return this.position === this.input.length;
  }
  /**
   * Sets a mark at the current position so that it can be returned to by a
   * matching {@link #reset} call.
   */


  mark() {
    this.positionStack.push(this.position);
  }
  /**
   * Convenience method for surrounding a function with a call to {@link #mark},
   * then {@link #clear} if the result of the function is non-null, or
   * {@link #reset} if `null`.
   * 
   * @template T
   * @param {Function<T>} func
   * @return {T}
   */


  markAndClearOrReset(func) {
    this.mark();
    let result = func();

    if (result !== null) {
      this.clear();
      return result;
    }

    this.reset();
    return null;
  }
  /**
   * Reads as many characters from the current position as satisfies the given
   * pattern, returning the read characters and advancing the mark by as many
   * characters.
   * 
   * @param {RegExp} pattern
   * @return {Array} The array of matched strings, or `null` if the pattern
   *   didn't match.
   */


  read(pattern) {
    let remaining = this.input.substring(this.position);
    let leadingWhitespace = remaining.match(/^\s+/);

    if (leadingWhitespace) {
      leadingWhitespace = leadingWhitespace[0];
      remaining = remaining.substring(leadingWhitespace.length);
    }

    let result = new RegExp(pattern.source).exec(remaining);

    if (result) {
      let [value] = result;

      if (remaining.startsWith(value)) {
        this.position += value.length + (leadingWhitespace ? leadingWhitespace.length : 0);
        return result;
      }
    }

    return null;
  }
  /**
   * Revert to the last @{link #mark}ed position.
   */


  reset() {
    let newPosition = this.positionStack.pop();

    if (newPosition === undefined) {
      throw new Error('Called reset() but no matching mark()');
    }

    this.position = newPosition;
  }

}

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A special kind of expression that understands matched portions of regular
 * expressions to run processing over, which may lead to additional parsing
 * expressions.
 * 
 * This expression should be used sparingly as the regexes within need to take
 * care of whitespace between tokens themselves, which can be annoying.
 * 
 * @param {RegExp} expression
 * @param {Array<Matchable>} matchers
 * @return {Matchable}
 */

const RegularExpression = (expression, matchers) => (input, parser) => {
  return input.markAndClearOrReset(() => {
    let result = input.read(expression);

    if (result) {
      let parseResults = [result[0]];

      for (let i = 1; i < result.length; i++) {
        let match = result[i];

        if (match !== undefined) {
          let parseResult = parser.parseWithExpression(new InputBuffer(match), matchers[i - 1]);

          if (parseResult === null) {
            return null;
          }

          parseResults.push(parseResult);
        }
      }

      return parseResults;
    }

    return null;
  });
};

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const METADATA_FRAGMENT = 'fragment';
const METADATA_ITERATION = 'iteration';
const METADATA_MESSAGE = 'message';
/**
 * Grammar for the Thymeleaf expression language.  Describes the language and
 * how to parse it.
 * 
 * @author Emanuel Rabina
 */

var ThymeleafExpressionLanguage = new Grammar('Thymeleaf Expression Language', // Ordered as at https://www.thymeleaf.org/doc/tutorials/3.0/usingthymeleaf.html#standard-expression-syntax
new ThymeleafRule('ThymeleafExpression', OrderedChoice(AllInput('VariableExpression'), AllInput('MessageExpression'), AllInput('LinkExpression'), AllInput('FragmentExpression'), AllInput('Iteration'), AllInput('StringConcatenation'), AllInput('ScopedVariables'), AllInput('Literal'), AllInput('LogicalExpression'), AllInput('IfThenCondition'), AllInput('IfThenElseCondition'), AllInput('TokenLiteral'), AllInput('Nothing'))), // Simple expressions
// ==================

/**
 * Variable expressions, `${variable}`.  Represents a value to be retrieved
 * from the current context.  Also is an entry into the underlying expression
 * language, so this part often extends to do what OGNL (and thus SpEL) can
 * do.
 */
new ThymeleafRule('VariableExpression', Sequence(/\${/, 'Chain', /\}/), ([, chain]) => context => {
  let result = chain(context);
  return result !== null && result !== undefined ? result : '';
}), new ThymeleafRule('Chain', Sequence(Optional('Negation'), 'ChainLink', ZeroOrMore(Sequence(/\./, 'ChainLink'))), ([negation, ...chain]) => context => {
  let result = arrayUtils.flatten(chain).filter(link => link !== '.').reduce((linkContext, nextLink) => {
    if (linkContext === null || linkContext === undefined) {
      return linkContext;
    }

    return nextLink(linkContext, context);
  }, context);
  return typeof negation === 'function' ? !result : result;
}), new ThymeleafRule('ChainLink', OrderedChoice('MethodCall', 'PropertyName', 'Literal')), new ThymeleafRule('Negation', /!/),
/**
 * Message expressions, `#{messageKey(parameters)}`.  Used for referencing
 * text from a resource file, often for localization purposes.
 */
new ThymeleafRule('MessageExpression', Sequence(/#{/, 'MessageKey', Optional(Sequence(/\(/, Sequence('Expression', ZeroOrMore(Sequence(/,/, 'Expression'))), /\)/)), /}/), ([, messageKey, [, messageParameters]]) => context => {
  return {
    type: METADATA_MESSAGE,
    key: messageKey(context),
    parameters: messageParameters ? arrayUtils.flatten(messageParameters).filter(messageParameter => typeof messageParameter === 'function').map(messageParameter => messageParameter(context)) : null
  };
}), new ThymeleafRule('MessageKey', /[\w.-]+/),
/**
 * Link expressions, `@{url(parameters)}`.  Used for generating URLs out of
 * context parameters.
 * 
 * TODO: Change this to use the other expression types as it causes a circular
 *       dependency in the build.
 */
new ThymeleafRule('LinkExpression', RegularExpression(/^@\{(.+?)(\(.+\))?\}$/, ['Url', 'UrlParameters']), ([, urlFunc, parameters]) => context => {
  let url = urlFunc(context);

  if (parameters) {
    // TODO: Push this parsing of the parameters list back into the grammar
    let expressionProcessor = new ExpressionProcessor();
    let paramsList = parameters(context).slice(1, -1).split(',').map(param => {
      let [lhs, rhs] = param.split('=');
      return [lhs, expressionProcessor.process(rhs, context)];
    }); // Fill out any placeholders in the URL from the parameters

    while (true) {
      // eslint-disable-line
      let urlTemplate = /(.*?)\{(.+?)\}(.*)/.exec(url);

      if (urlTemplate) {
        let [, head, placeholder, tail] = urlTemplate;
        let paramEntry = arrayUtils.remove(paramsList, ([lhs]) => lhs === placeholder);

        if (paramEntry) {
          url = `${head}${paramEntry[1]}${tail}`;
        }
      } else {
        break;
      }
    } // Remaining parameters become search query parameters


    if (paramsList.length) {
      url += `?${paramsList.map(([key, value]) => `${key}=${value}`).join('&')}`;
    }
  }

  return url;
}), new ThymeleafRule('Url', /.+/), new ThymeleafRule('UrlParameters', /\((.+)\)/),
/**
 * Fragment expressions, `~{template :: fragment(parameters)}`.  A locator for
 * a piece of HTML in the same or another template.
 */
new ThymeleafRule('FragmentExpression', Sequence(/~{/, 'TemplateName', /::/, 'FragmentName', Optional('FragmentParametersSection'), /}/), ([, templateName,, fragmentName, parameters]) => context => {
  // TODO: Should executing a fragment expression locate and return the
  //       fragment?  If so, then it'll make expression execution
  //       asynchronous!
  return {
    type: METADATA_FRAGMENT,
    templateName: templateName(context),
    fragmentName: fragmentName(context),
    parameters: parameters ? parameters(context) : null
  };
}), new ThymeleafRule('TemplateName', OrderedChoice(/[\w-._/]+/, 'VariableExpression')), new ThymeleafRule('FragmentName', /[\w-._]+/), new ThymeleafRule('FragmentParametersSection', Sequence(/\(/, Optional('FragmentParameters'), /\)/), ([, parameters]) => context => {
  return parameters(context);
}), new ThymeleafRule('FragmentParameters', Sequence('Expression', ZeroOrMore(Sequence(/,/, 'Expression'))), expressionsAndSeparators => context => {
  return expressionsAndSeparators ? arrayUtils.flatten(expressionsAndSeparators).filter(item => item !== ',').map(expressions => expressions(context)) : [];
}), // Complex expressions
// ===================

/**
 * Iteration, `localVar : ${collection}`.  The name of the variable for each
 * loop, followed by the collection being iterated over.
 */
new ThymeleafRule('Iteration', Sequence('Identifier', Optional(Sequence(/,/, 'Identifier')), /:/, 'VariableExpression'), ([localValueName, [, iterationStatusVariable],, collectionExpressionAction]) => context => ({
  type: METADATA_ITERATION,
  localValueName: localValueName(context),
  iterable: collectionExpressionAction(context),
  iterationStatusVariable: iterationStatusVariable ? iterationStatusVariable(context) : null
})),
/**
 * String concatenation, `'...' + '...'` or even `${...} + ${...}`, the
 * joining of 2 expressions by way of the `+` operator.
 */
new ThymeleafRule('StringConcatenation', Sequence('Concatenatable', OneOrMore(Sequence(/\+/, 'Concatenatable'))), values => context => {
  return arrayUtils.flatten(values).filter(item => item !== '+').reduce((result, value) => {
    return result + (typeof value === 'function' ? value(context) : value);
  }, '');
}), new ThymeleafRule('Concatenatable', OrderedChoice('StringLiteral', 'VariableExpression')),
/**
 * Scoped variable aliases, `key=${expression},...`, describes one or more
 * names for scoped variables with the expressions that can be their values.
 */
new ThymeleafRule('ScopedVariables', Sequence('ScopedVariable', ZeroOrMore(Sequence(/,/, 'ScopedVariable'))), aliases => context => {
  return arrayUtils.flatten(aliases).map(alias => alias(context));
}), new ThymeleafRule('ScopedVariable', Sequence('Identifier', /=/, 'Expression'), ([name,, expression]) => context => ({
  name: name(context),
  value: expression(context)
})), // Literals
// ========
new ThymeleafRule('Literal', OrderedChoice('StringLiteral', 'NumberLiteral', 'BooleanLiteral', 'NullLiteral')),
/**
 * String literal, characters surrounded by `'` (single quotes).
 * 
 * This is trying to emulate negative lookbehind so that escaped quotes don't
 * get counted as string terminators, but JavaScript only got that feature in
 * ES2018, so if I used it it'd leave too many JS engines without support.
 */
new ThymeleafRule('StringLiteral', /'.*?(?!\\').'/, result => () => result.slice(1, -1).replace(/\\/g, '')),
/**
 * A number.
 */
new ThymeleafRule('NumberLiteral', /\d+(\.\d+)?/, result => () => parseFloat(result)),
/**
 * One of `true` or `false`.
 */
new ThymeleafRule('BooleanLiteral', /(true|false)/, result => () => result === 'true'),
/**
 * The word `null` to represent the null value.
 */
// TODO: The parser uses null to mean 'failed parse', so this might not work?
new ThymeleafRule('NullLiteral', /null/, () => () => null),
/**
 * A token literal, which is pretty much anything else that can't be categorized
 * by the other literal types.  This is often used as a fallback in the
 * expression language so that, for any unknown input, we're still returning
 * something.
 */
new ThymeleafRule('TokenLiteral', /[^: ${}]+/, result => () => result), // Text operations
// ===============
// Arithmetic operations
// =====================
// Boolean operations
// ==================
// Comparisons and equality
// ========================

/**
 * A logical expression is any expression that resolves to a `true`/`false`
 * value.
 */
new ThymeleafRule('LogicalExpression', Sequence('Expression', 'Comparator', 'Expression'), ([leftOperand, comparator, rightOperand]) => context => {
  let lhs = leftOperand(context);
  let rhs = rightOperand(context);

  switch (comparator(context)) {
    case '==':
      return lhs == rhs;
    // eslint-disable-line

    case '===':
      return lhs === rhs;

    case '!=':
      return lhs != rhs;
    // eslint-disable-line

    case '!==':
      return lhs !== rhs;
  }

  return false;
}), new ThymeleafRule('Comparator', OrderedChoice(/[=!]==?/)), // Conditional operators
// =====================

/**
 * If-then condition, `if ? then`.  This is the truthy branch only of the
 * classic ternary operator.  The falsey branch is a no-op.
 */
new ThymeleafRule('IfThenCondition', Sequence('Condition', /\?/, 'Expression'), ([condition,, truthyExpression]) => context => {
  return condition(context) ? truthyExpression(context) : undefined;
}),
/**
 * If-then-else condition, `if ? then : else`, the classic ternary operator.
 */
new ThymeleafRule('IfThenElseCondition', Sequence('Condition', /\?/, 'Expression', /:/, 'Expression'), ([condition,, truthyExpression,, falseyExpression]) => context => {
  return condition(context) ? truthyExpression(context) : falseyExpression(context);
}),
/**
 * A condition is some expression or value that resolves to a true/false
 * value.
 */
new ThymeleafRule('Condition', OrderedChoice('LogicalExpression', 'Expression')), // Special tokens
// ==============

/**
 * An expression that matches the empty string.
 */
new ThymeleafRule('Nothing', /^$/), // Common language basics
// ======================
new ThymeleafRule('Identifier', /[#a-zA-Z_][\w]*/), new ThymeleafRule('PropertyName', 'Identifier', propertyName => context => {
  let property = propertyName(context);
  return Object.prototype.hasOwnProperty.call(context, property) ? context[property] : '';
}), new ThymeleafRule('MethodCall', Sequence('MethodName', /\(/, Optional('MethodParameters'), /\)/), ([name,, parameters]) => (context, parameterContext) => {
  let methodName = name(context);
  let method = context[methodName];

  if (!method) {
    console.warn(`No method '${methodName}' present on the current context.  Expression: ${context.expression}`);
    return '';
  }

  return method.apply(context, parameters(parameterContext || context));
}), new ThymeleafRule('MethodName', 'Identifier'), new ThymeleafRule('MethodParameters', Sequence('Chain', ZeroOrMore(Sequence(/,/, 'Chain'))), parametersAndSeparators => context => {
  return parametersAndSeparators ? arrayUtils.flatten(parametersAndSeparators).filter(item => item !== ',').map(parameter => parameter(context)) : [];
}),
/**
 * Any valid unit of code that resolves to some value.
 */
new ThymeleafRule('Expression', OrderedChoice('VariableExpression', 'StringConcatenation', 'Literal')));

/**
 * Any one of the objects that can be matched:
 *  - an expression function
 *  - a string that references another rule
 *  - a regular expression
 * 
 * @typedef {String|RegExp|Function} Matchable
 */

/**
 * A recursive descent parser for any parsing expression grammar defined by the
 * constructs in this module.
 * 
 * TODO: Own module?
 * 
 * @author Emanuel Rabina
 */

class Parser {
  /**
   * @param {Grammar} grammar
   */
  constructor(grammar) {
    _defineProperty(this, "expressionStack", []);

    this.grammar = grammar;
  }
  /**
   * Parse a string, attempting to build the parse tree defined by the rules in
   * the configured grammar.  Parsing is considered successful when there are no
   * more non-terminating symbols in the grammar and all of the input has been
   * read.
   * 
   * @param {String} input
   * @return {Object} The parse tree if the input could be parsed, `null`
   *   otherwise.
   */


  parse(input) {
    let inputBuffer = new InputBuffer(input);
    let matchResult = this.grammar.accept(inputBuffer, this);

    if (matchResult === null || !inputBuffer.exhausted()) {
      let errorMessage = `Failed to parse "${input}"`; // Don't bring down the thread if we can't parse

      if (process.env.NODE_ENV === 'production') {
        console.error(errorMessage);
        return null;
      } else {
        throw new Error(errorMessage);
      }
    }

    return matchResult;
  }
  /**
   * Parse the input against the given expression.  An expression can either be
   * a reference to another rule in the current grammar, or a regular expression
   * that consumes input.
   * 
   * @param {InputBuffer} input
   * @param {Matchable} expression
   * @return {Object}
   */


  parseWithExpression(input, expression) {
    // Name of another rule in the grammar
    if (typeof expression === 'string') {
      let rule = this.grammar.findRuleByName(expression);
      return rule ? rule.accept(input, this) : null;
    } // A regular expression that must be matched
    else if (expression instanceof RegExp) {
        let result = input.read(expression);

        if (result) {
          return result[0];
        }
      } // An expression function to be executed
      else if (typeof expression === 'function') {
          return expression(input, this);
        }

    return null;
  }
  /**
   * Surrounds a function that evaluates an expression with tracking it against
   * the current stack.  Useful for knowing how the current expression was
   * arrived at through the grammar's rules.
   * 
   * @template T
   * @param {InputBuffer} input
   * @param {Matchable} expression
   * @param {String} name
   * @param {Function<T>} func
   * @return {T}
   */


  trackExpression(input, expression, name, func) {
    let stackSize = this.expressionStack.push({
      name,
      expression: typeof expression !== 'function' ? expression.toString() : null,
      input: input.input.substring(input.position)
    });
    let result = func();

    if (result !== null) {
      return result;
    }

    this.expressionStack.splice(stackSize - 1);
    return null;
  }

}

/**
 * Parses and executes Thymeleaf expressions.
 * 
 * TODO: Create a shared instance of this for a processing context so that it
 *       doesn't need to be recreated over and over.
 * 
 * @author Emanuel Rabina
 */

class ExpressionProcessor {
  /**
   * Constructor, create a new processor that can parse/execute a string in the
   * given grammar.
   * 
   * @param {Grammar} [grammar=ThymeleafExpressionLanguage]
   */
  constructor(grammar = ThymeleafExpressionLanguage) {
    this.grammar = grammar;
  }
  /**
   * Parse and execute the given input as a Thymeleaf expression.
   * 
   * @param {String} input
   * @param {Object} [context={}]
   * @return {*}
   */


  process(input, context = {}) {
    // TODO: Probably don't need to create a new parser every time?
    let parser = new Parser(this.grammar);
    let expression = parser.parse(input);
    return expression ? expression(_objectSpread2(_objectSpread2({}, context), {}, {
      expression: input
    })) : null;
  }

}

/* 
 * Copyright 2017, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Common class for attribute processors.
 * 
 * @author Emanuel Rabina
 */
class AttributeProcessor {
  /**
   * Constructor, sets this processor's prefix and name.
   * 
   * @param {String} prefix
   * @param {String} name
   */
  constructor(prefix, name) {
    this.prefix = prefix;
    this.name = name;
  }

}

/**
 * JS equivalent of Thymeleaf's `th:attr` attribute processor, modifies or sets
 * a target attribute to whatever its associated expression evaluates to.
 * 
 * @author Emanuel Rabina
 */

class AttrAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `attr` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, AttrAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:attr` or `data-th-attr` attribute
   * on it, picking out the target attributes and setting them to whatever their
   * expressions evaluate to.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    // TODO: This regex, is this some kind of value list that needs to be
    //       turned into an expression?
    if (/(.+=.+,)*.+=.+/.test(attributeValue)) {
      attributeValue.split(',').forEach(attribute => {
        let attributeParts = attribute.split('=');
        element.setAttribute(attributeParts[0], stringUtils.escapeHtml(new ExpressionProcessor().process(attributeParts[1], context)));
      });
    }
    /* istanbul ignore next */
    else if (process.env.NODE_ENV !== 'test') {
        console.warn(`Value to ${attribute}, ${attributeValue}, doesn't seem to contain an attribute assignment expression.  Ignoring.`);
      }

    element.removeAttribute(attribute);
  }

}

_defineProperty(AttrAttributeProcessor, "NAME", 'attr');

/* 
 * Copyright 2019, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Parent class for element processors.
 * 
 * @author Emanuel Rabina
 */
class ElementProcessor {
  /**
   * Constructor, sets this processor's prefix and name.
   * 
   * @param {String} prefix
   * @param {String} name
   */
  constructor(prefix, name) {
    this.prefix = prefix;
    this.name = name;
  }

}

/**
 * Equivalent of Thymeleaf's "synthetic tag", `th:block`, which removes itself,
 * leaving the body of the tag behind.
 * 
 * @author Emanuel Rabina
 */

class BlockElementProcessor extends ElementProcessor {
  /**
   * Constructor, set this processor to use the `block` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, BlockElementProcessor.NAME);
  }
  /**
   * Processes an element named `th:block`, removing itself to leave its
   * body/contents behind.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {Object} context
   * @return {Boolean} `true` to indicate that the elements need reprocessing.
   */


  process(element, context) {
    let parent = element.parentElement;

    while (element.firstChild) {
      let child = element.firstChild;
      parent.insertBefore(child, element);

      if (child instanceof Element && element.__thymeleafLocalVariables) {
        child.__thymeleafLocalVariables = _objectSpread2(_objectSpread2({}, child.__thymeleafLocalVariables || {}), element.__thymeleafLocalVariables);
      }
    }

    parent.removeChild(element);
    return true;
  }

}

_defineProperty(BlockElementProcessor, "NAME", 'block');

/**
 * Thymeleaf's `th:checked` attribute processor, sets or removes the `checked`
 * attribute from an element based on the result of the expression within it.
 * 
 * TODO: This is one of HTML5s "boolean attributes", attributes whose values are
 *       true simply by being present in the element, regardless of the value
 *       inside it.  To act as false, the attribute has to be removed.  Find a
 *       way to generate these from some list of boolean attributes so that I
 *       don't need to write a class for each one!
 * 
 * @author Emanuel Rabina
 */

class CheckedAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `checked` name and supplied
   * prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, CheckedAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:checked` or `data-th-checked`
   * attribute on it, either setting or removing a `checked` attribute to the
   * current element based on the result of the inner expression.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    let result = new ExpressionProcessor().process(attributeValue, context);

    if (result) {
      element.setAttribute('checked', '');
    } else {
      element.removeAttribute('checked');
    }

    element.removeAttribute(attribute);
  }

}

_defineProperty(CheckedAttributeProcessor, "NAME", 'checked');

/**
 * The `th:classappend` is a special attribute that applies the expression to
 * any existing classes already on an element.
 * 
 * @author Emanuel Rabina
 */

class ClassAppendAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `attr` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, ClassAppendAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:classappend` or `data-th-classappend`
   * attribute on it, adding the resulting classes to any existing classes on
   * the current element.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    let classes = new ExpressionProcessor().process(attributeValue, context);

    if (classes) {
      element.className += ` ${classes}`;
    }

    element.removeAttribute(attribute);
  }

}

_defineProperty(ClassAppendAttributeProcessor, "NAME", 'classappend');

/**
 * JS equivalent of Thymeleaf's `th:each` attribute processor, iterates over an
 * [iterable object](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Iteration_protocols),
 * executing a piece of template for every iteration.
 * 
 * @author Emanuel Rabina
 */

class EachAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `each` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, EachAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:each`/`data-th-each` attribute,
   * repeating the markup for every object in the iterable value.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   * @return {Boolean} Whether or not the parent element needs to do a second
   *   pass as its children have been modified by this processor.
   */


  process(element, attribute, attributeValue, context) {
    element.removeAttribute(attribute);
    let iterationInfo = new ExpressionProcessor().process(attributeValue, context);

    if (iterationInfo) {
      let {
        localValueName,
        iterable
      } = iterationInfo;
      let templateNode = element.cloneNode(true);

      for (let value of iterable) {
        let localClone = templateNode.cloneNode(true);
        localClone.__thymeleafLocalVariables = {
          [localValueName]: value
        };
        element.parentElement.appendChild(localClone);
      }
    }

    element.parentElement.removeChild(element);
    return true;
  }

}

_defineProperty(EachAttributeProcessor, "NAME", 'each');

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Configurable attribute processor that sets or empties an attribute value on
 * an element if the result of its expression is truthy or falsey respectively.
 * 
 * @author Emanuel Rabina
 */

class EmptyableAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set the name of the attribute this processor will operate on.
   * 
   * @param {String} prefix
   * @param {String} name
   */
  constructor(prefix, name) {
    super(prefix, name);
  }
  /**
   * Processes an element that contains the configured attribute to be worked
   * on, setting it if the expression resolves to a truthy value, or removing it
   * if it resolves to a falsey value.
   * 
   * @param {Element} element 
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    let value = new ExpressionProcessor().process(attributeValue, context);
    element.setAttribute(this.name, value ? value.toString() : '');
    element.removeAttribute(attribute);
  }

}
const EMPTYABLE_ATTRIBUTE_NAMES = ['datetime', 'href', 'src', 'style', 'value'];

/**
 * JS equivalent of Thymeleaf's `th:fragment` attribute processor, marks an
 * element as a template fragment that can be imported by other processors like
 * `th:insert`.
 * 
 * @author Emanuel Rabina
 */

class FragmentAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `fragment` name and supplied
   * prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, FragmentAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:fragment` or `data-th-fragment`
   * attribute on it.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    element.removeAttribute(attribute);
  }

}

_defineProperty(FragmentAttributeProcessor, "NAME", 'fragment');

/**
 * JS equivalent of Thymeleaf's `th:if` attribute processor, includes or
 * excludes the current element and its children from rendering, depending on
 * the evaluation of the expression in the attribute value.
 * 
 * @author Emanuel Rabina
 */

class IfAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `if` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, IfAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:if` or `data-th-if` attribute
   * on it, evaluating the expression for truthy/falsey, rendering/excluding the
   * element and its children based on the result.
   * 
   * @param {Element} element 
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   * @return {Boolean} `true` if the element was removed.
   */


  process(element, attribute, attributeValue, context) {
    let expressionResult = new ExpressionProcessor().process(attributeValue, context);

    if (!expressionResult) {
      domUtils.clearChildren(element);
      element.parentNode.removeChild(element);
      return true;
    }

    element.removeAttribute(attribute);
  }

}

_defineProperty(IfAttributeProcessor, "NAME", 'if');

/* 
 * Copyright 2019, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Grammar for Thymeleaf fragment signatures.  This is separate from the
 * expression language as these are not expressions, but rather marker syntaxes.
 * 
 * @author Emanuel Rabina
 */

var FragmentSignatureGrammar = new Grammar('Thymeleaf fragment signature', new ThymeleafRule('ThymeleafFragmentSignatureRule', AllInput('FragmentSignature')),
/**
 * The target end of a fragment expression, contains the fragment name and
 * optional parameter names.
 */
new ThymeleafRule('FragmentSignature', Sequence('FragmentName', Optional('FragmentParameters')), ([fragmentName, parameterNames]) => context => {
  return {
    fragmentName: fragmentName(context),
    parameterNames: parameterNames ? parameterNames(context) : null
  };
}), new ThymeleafRule('FragmentName', /[\w-._]+/), new ThymeleafRule('FragmentParameters', Sequence(/\(/, ZeroOrMore('FragmentParameterNames'), /\)/), ([, [parameterNames]]) => context => {
  return parameterNames(context);
}), new ThymeleafRule('FragmentParameterNames', Sequence('Identifier', ZeroOrMore(Sequence(/,/, 'Identifier'))), namesAndSeparators => context => {
  return namesAndSeparators ? arrayUtils.flatten(namesAndSeparators).filter(item => item !== ',').map(name => name(context)) : [];
}), // Common language basics
// ======================
new ThymeleafRule('Identifier', /[#a-zA-Z_][\w]*/));

/* 
 * Copyright 2017, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Returns the value of a Thymeleaf attribute processor.
 * 
 * @param {Element} element
 * @param {String} prefix
 * @param {String} processorName
 * @return {String} The value of the Thymeleaf attribute processor, or `null`
 *   if the attribute processor wasn't present.
 */

function getThymeleafAttributeValue(element, prefix, processorName) {
  return element.getAttribute(`${prefix}:${processorName}`) || element.getAttribute(`data-${prefix}-${processorName}`);
}
/**
 * Use either JSDOM or the browser's native DOM parsing to deserialize the HTML
 * string into a document fragment.
 * 
 * @param {String} htmlString
 * @return {DocumentFragment}
 */

function deserialize(htmlString) {
  /* istanbul ignore if */
  {
    const {
      JSDOM
    } = require('jsdom');

    return new JSDOM(htmlString).window.document;
  }
}
/**
 * Use either JSDOM or the browser's native DOM serialization to serialize a
 * document fragment into an HTML string.
 * 
 * @param {DocumentFragment} documentFragment
 * @return {String}
 */

function serialize(documentFragment) {
  /* istanbul ignore if */
  {
    let result = '';
    let {
      firstChild,
      firstElementChild
    } = documentFragment;

    if (firstChild.nodeType === Node.DOCUMENT_TYPE_NODE) {
      result += `<!DOCTYPE ${firstChild.name}>`;
    }

    return result + firstElementChild.outerHTML;
  }
}

/* 
 * Copyright 2019, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Extract HTML from the target identified by the given fragment information.
 * 
 * @param {Object} fragmentInfo
 * @param {Object} context
 * @return {Promise<Element>}
 */

async function extractFragment(fragmentInfo, context) {
  let {
    templateResolver
  } = context;

  if (templateResolver) {
    let {
      templateName,
      fragmentName
    } = fragmentInfo;
    let template = deserialize(await templateResolver(templateName));
    let standardDialect = context.dialects.find(dialect => dialect.name === StandardDialect.NAME);
    let dialectPrefix = standardDialect.prefix;
    let fragmentProcessorName = FragmentAttributeProcessor.NAME;
    return dumbQuerySelector.$(`[${dialectPrefix}\\:${fragmentProcessorName}^="${fragmentName}"]`, template) || dumbQuerySelector.$(`[data-${dialectPrefix}-${fragmentProcessorName}^="${fragmentName}"]`, template);
  }

  console.log('No template resolver configured');
  return null;
}

/**
 * JS equivalent of Thymeleaf's `th:insert` attribute processor, inserts the
 * referenced template fragment as a child of the current element.
 * 
 * @author Emanuel Rabina
 */

class InsertAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `insert` name and supplied
   * prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, InsertAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:insert`/`data-th-insert` attribute,
   * replacing the current element's children with the DOM in the referenced
   * fragment.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   * @return {Promise<Boolean>} Whether or not the parent element needs to do a
   *   second pass as its children have been modified by this processor.
   */


  async process(element, attribute, attributeValue, context) {
    element.removeAttribute(attribute);
    domUtils.clearChildren(element);
    let fragmentInfo = new ExpressionProcessor().process(attributeValue, context);

    if (fragmentInfo) {
      let fragment = await extractFragment(fragmentInfo, context);

      if (fragment) {
        let standardDialect = context.dialects.find(dialect => dialect.name === StandardDialect.NAME);
        let dialectPrefix = standardDialect.prefix;
        let fragmentProcessorName = FragmentAttributeProcessor.NAME;
        let fragmentSignature = getThymeleafAttributeValue(fragment, dialectPrefix, fragmentProcessorName);
        let {
          parameterNames
        } = new ExpressionProcessor(FragmentSignatureGrammar).process(fragmentSignature, context);

        if (parameterNames) {
          let {
            parameters
          } = fragmentInfo;
          let localContext = {};
          parameterNames.forEach((parameterName, index) => {
            localContext[parameterName] = parameters[parameterName] || parameters[index] || null;
          });
          fragment.__thymeleafLocalVariables = localContext;
        }

        element.appendChild(fragment);
        return true;
      }
    }

    return false;
  }

}

_defineProperty(InsertAttributeProcessor, "NAME", 'insert');

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Configurable attribute processor that sets or removes an attribute on an
 * element if the result of its expression is truthy or falsey respectively.
 * 
 * @author Emanuel Rabina
 */

class RemovableAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set the name of the attribute this processor will operate on.
   * 
   * @param {String} prefix
   * @param {String} name
   */
  constructor(prefix, name) {
    super(prefix, name);
  }
  /**
   * Processes an element that contains the configured attribute to be worked
   * on, setting it if the expression resolves to a truthy value, or removing it
   * if it resolves to a falsey value.
   * 
   * @param {Element} element 
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    let value = new ExpressionProcessor().process(attributeValue, context);

    if (value) {
      element.setAttribute(this.name, value.toString());
    } else {
      element.removeAttribute(this.name);
    }

    element.removeAttribute(attribute);
  }

}
const REMOVABLE_ATTRIBUTE_NAMES = ['alt', 'class'];

/**
 * `th:remove`, used to remove the current element or select parts of it (and
 * its children).
 * 
 * @author Emanuel Rabina
 */

class RemoveAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `remove` name and supplied
   * prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, RemoveAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:remove`/`data-th-remove`
   * attribute, removing the current element or parts of it based on the
   * attribute value.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   * @return {Boolean} Whether reprocessing behaviour needs to be applied, only
   *   when the current tag has been removed.
   */


  process(element, attribute, attributeValue, context) {
    element.removeAttribute(attribute);

    switch (attributeValue) {
      case 'all':
        element.parentElement.removeChild(element);
        return true;

      case 'all-but-first':
        while (element.lastElementChild !== element.firstElementChild) {
          element.removeChild(element.lastElementChild);
        }

        return false;
    }
  }

}

_defineProperty(RemoveAttributeProcessor, "NAME", 'remove');

/**
 * JS equivalent of Thymeleaf's `th:relace` attribute processor, replaces the
 * current element with the fragment referenced by the processor.
 * 
 * @author Emanuel Rabina
 */

class ReplaceAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `replace` name and supplied
   * prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, ReplaceAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:replace`/`data-th-replace`
   * attribute, replacing the current element with the DOM in the referenced
   * fragment.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   * @return {Promise<Boolean>} Whether or not the parent element needs to do a
   *   second pass as its children have been modified by this processor.
   */


  async process(element, attribute, attributeValue, context) {
    element.removeAttribute(attribute);
    domUtils.clearChildren(element);
    let fragmentInfo = new ExpressionProcessor().process(attributeValue, context);

    if (fragmentInfo) {
      let fragment = await extractFragment(fragmentInfo, context);

      if (fragment) {
        let standardDialect = context.dialects.find(dialect => dialect.name === StandardDialect.NAME);
        let dialectPrefix = standardDialect.prefix;
        let fragmentProcessorName = FragmentAttributeProcessor.NAME;
        let fragmentSignature = getThymeleafAttributeValue(fragment, dialectPrefix, fragmentProcessorName);
        let {
          parameterNames
        } = new ExpressionProcessor(FragmentSignatureGrammar).process(fragmentSignature, context);

        if (parameterNames) {
          let {
            parameters
          } = fragmentInfo;
          let localContext = {};
          parameterNames.forEach((parameterName, index) => {
            localContext[parameterName] = parameters[parameterName] || parameters[index] || null;
          });
          fragment.__thymeleafLocalVariables = localContext;
        } // TODO: Can simplify this with insertAdjacent*(), but need to upgrade
        //       JSDOM first.


        element.parentElement.insertBefore(fragment, element);
        element.remove();
        return true;
      }
    }

    element.remove();
    return false;
  }

}

_defineProperty(ReplaceAttributeProcessor, "NAME", 'replace');

/* 
 * Copyright 2019, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Build a message string from a processed message expression.
 * 
 * @param {Object} messageInfo
 * @param {Object} context
 * @return {Promise<String>}
 */
async function buildMessage(messageInfo, context) {
  let {
    messageResolver
  } = context;

  if (messageResolver) {
    let {
      key,
      parameters
    } = messageInfo;
    return (await messageResolver(key, parameters)) || '';
  }

  console.log('No message resolver configured');
  return null;
}

/**
 * JS equivalent of Thymeleaf's `th:text` attribute processor, applies the
 * expression in the attribute value to the text content of the element being
 * processed, escaping any unsafe input.
 * 
 * @author Emanuel Rabina
 */

class TextAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `text` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, TextAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:text` or `data-th-text` attribute
   * on it, taking the text expression in the value and applying it to the text
   * content of the element.
   * 
   * @param {Element} element 
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  async process(element, attribute, attributeValue, context) {
    // TODO: Move message constructon to the expression language?  Need to make
    //       all the executions async!
    let messageResult = new ExpressionProcessor().process(attributeValue, context);
    element.textContent = typeof messageResult === 'object' ? await buildMessage(messageResult, context) : messageResult;
    element.removeAttribute(attribute);
  }

}

_defineProperty(TextAttributeProcessor, "NAME", 'text');

/**
 * JS equivalent of Thymeleaf's `th:unless` attribute processor, excludes or
 * includes the current element and its children from rendering, depending on
 * the evaluation of the expression in the attribute value.
 * 
 * @author Robbie Bardijn
 */

class UnlessAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `unless` name and supplied prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, UnlessAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:unless` or `data-th-unless` attribute
   * on it, evaluating the expression for falsey/truthy, excluding/rendering the
   * element and its children based on the result.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    let expressionResult = new ExpressionProcessor().process(attributeValue, context);

    if (expressionResult) {
      domUtils.clearChildren(element);
      element.parentNode.removeChild(element);
    }

    element.removeAttribute(attribute);
  }

}

_defineProperty(UnlessAttributeProcessor, "NAME", 'unless');

/**
 * JS equivalent of Thymeleaf's `th:utext` attribute processor, applies the
 * expression in the attribute value to the text content of the element being
 * processed.
 * 
 * @author Emanuel Rabina
 */

class UTextAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `utext` name and supplied
   * prefix.
   * 
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, UTextAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:utext` or `data-th-utext`
   * attribute on it, taking the text expression in the value and applying it to
   * the text content of the element.
   *
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   */


  process(element, attribute, attributeValue, context) {
    element.innerHTML = new ExpressionProcessor().process(attributeValue, context);
    element.removeAttribute(attribute);
  }

}

_defineProperty(UTextAttributeProcessor, "NAME", 'utext');

/**
 * `th:with`, used for creating scoped variables, useful for aliasing things.
 * 
 * @author Emanuel Rabina
 */

class WithAttributeProcessor extends AttributeProcessor {
  /**
   * Constructor, set this processor to use the `with` name and supplied
   * prefix.
   *
   * @param {String} prefix
   */
  constructor(prefix) {
    super(prefix, WithAttributeProcessor.NAME);
  }
  /**
   * Processes an element that contains a `th:with`/`data-th-with` attribute,
   * setting a variable scoped to the current element with the given name.
   * 
   * @param {Element} element
   *   Element being processed.
   * @param {String} attribute
   *   The attribute that was encountered to invoke this processor.
   * @param {String} attributeValue
   *   The value given by the attribute.
   * @param {Object} context
   * @return {Boolean} `true` as adding new local variables needs to re-run
   *   processing.
   */


  process(element, attribute, attributeValue, context) {
    element.removeAttribute(attribute);
    let localVariables = {};
    let aliases = new ExpressionProcessor().process(attributeValue, context);
    aliases.forEach(({
      name,
      value
    }) => {
      localVariables[name] = value;
    });
    element.__thymeleafLocalVariables = localVariables;
    return true;
  }

}

_defineProperty(WithAttributeProcessor, "NAME", 'with');

/* 
 * Copyright 2017, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Common class for dialects.
 * 
 * @author Emanuel Rabina
 */
class Dialect {
  /**
   * Constructor, sets this dialect's name and optional prefix.
   * 
   * @param {String} name
   * @param {String} [prefix]
   */
  constructor(name, prefix) {
    this.name = name;
    this.prefix = prefix;
  }
  /**
   * Return an object whose keys are the expression object names, the values the
   * expression object available properties and methods.
   * 
   * @return {Object}
   */


  get expressionObjects() {
    return null;
  }
  /**
   * Return an array of processors.
   * 
   * @return {Array}
   */


  get processors() {
    return null;
  }

}

/**
 * The out-of-the-box dialect for Thymeleaf, the "Standard Dialect".
 * 
 * @author Emanuel Rabina
 */

class StandardDialect extends Dialect {
  /**
   * Create an instance of this dialect with the name "Standard" and
   * given prefix, defaulting to "th" if not supplied.
   * 
   * @param {String} [prefix='thjs']
   */
  constructor(prefix = StandardDialect.DEFAULT_PREFIX) {
    super(StandardDialect.NAME, prefix);
  }
  /**
   * Returns the supported standard processors.
   * 
   * @return {Array} A list of the processors included in this dialect.
   */


  get processors() {
    // TODO: This is a very basic way of imposing the order of attribute
    //       processors.  It's currently ordered in the same way as OG
    //       Thymeleaf.  Figure out a 'proper' way to do the ordering.
    // Order taken from https://www.thymeleaf.org/doc/tutorials/3.0/usingthymeleaf.html#attribute-precedence
    let {
      prefix
    } = this;
    return [// Fragment inclusion
    new InsertAttributeProcessor(prefix), new ReplaceAttributeProcessor(prefix), // Fragment iteration
    new EachAttributeProcessor(prefix), // Conditional evaluation
    new IfAttributeProcessor(prefix), new UnlessAttributeProcessor(prefix), // Local variable definition
    new WithAttributeProcessor(prefix), // General attribute modification
    new AttrAttributeProcessor(prefix), new ClassAppendAttributeProcessor(prefix), ...EMPTYABLE_ATTRIBUTE_NAMES.map(attributeName => {
      return new EmptyableAttributeProcessor(prefix, attributeName);
    }), ...REMOVABLE_ATTRIBUTE_NAMES.map(attributeName => {
      return new RemovableAttributeProcessor(prefix, attributeName);
    }), // Specific attribute modification
    new CheckedAttributeProcessor(prefix), // Text
    new TextAttributeProcessor(prefix), new UTextAttributeProcessor(prefix), // Fragment specification
    new FragmentAttributeProcessor(prefix), // Fragment removal
    new RemoveAttributeProcessor(prefix), // Element processors
    new BlockElementProcessor(prefix)];
  }

}

_defineProperty(StandardDialect, "NAME", 'Standard');

_defineProperty(StandardDialect, "DEFAULT_PREFIX", 'thjs');

/**
 * Configuration object for the template engine.
 * 
 * @typedef {Object} Configuration
 * @property {Array<Dialect>} dialects
 *   A list of dialects to include with this instance of the template engine.
 * @property {Object} [isomorphic]
 *   An object which configures the isomorphic capabilities of the template
 *   engine.
 * @property {Function} messageResolver
 *   A function for building a message string from some external source, given a
 *   message key and optional parameters for that particular message.
 * @property {Function} templateResolver
 *   A function for returning the text of templates named by fragment
 *   expressions in templates.  Is given only 1 argument, the template name from
 *   a fragment expression, and should return a Promise of the template text.
 */

/**
 * Default configuration for the template engine, configures the standard
 * dialect with no options (uses `thjs` as the prefix).
 * 
 * @type {Configuration}
 */

const DEFAULT_CONFIGURATION = {
  dialects: [new StandardDialect()]
};
/**
 * Standard configuration, configures the standard dialect with the `th` prefix
 * and enables isomorphic mode which enables the ability to use much of the same
 * processors across original Thymeleaf and ThymeleafJS.
 * 
 * @type {Configuration}
 */

const STANDARD_CONFIGURATION = _objectSpread2(_objectSpread2({}, DEFAULT_CONFIGURATION), {}, {
  dialects: [new StandardDialect('th')],
  isomorphic: {
    prefix: 'thjs'
  }
});

/* 
 * Copyright 2017, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Class for determining if an element contains a processor on it.
 */

class Matcher {
  /**
   * Create a matcher to work with the current context and isomorphic processing
   * settings.
   * 
   * @param {Object} context
   * @param {Object} isomorphic
   */
  constructor(context, isomorphic) {
    this.context = context;
    this.isomorphic = isomorphic;
  }
  /**
   * Return the matching attribute or element that a processor can work over.
   * 
   * @param {Element} element
   * @param {AttributeProcessor} processor
   * @return {String}
   *   The attribute or element that matched processing by this processor, or
   *   `null` if no match was found.
   */


  matches(element, processor) {
    let {
      name
    } = processor; // TODO: Some way to do this generically and not have to type check?
    // Attribute processor matching, can be of the name prefix:name or data-prefix-name

    if (processor instanceof AttributeProcessor) {
      let prefixes = [].concat(this.isomorphic ? this.isomorphic.prefix : [], processor.prefix);

      for (let prefix of prefixes) {
        let attribute;
        attribute = `${prefix}:${name}`;

        if (element.hasAttribute(attribute)) {
          return attribute;
        }

        attribute = `data-${prefix}-${name}`;

        if (element.hasAttribute(attribute)) {
          return attribute;
        }
      }
    } // Element processor, can only be of the name prefix:name
    else if (processor instanceof ElementProcessor) {
        let elementName = `${processor.prefix}:${name}`;

        if (element.tagName === elementName.toUpperCase()) {
          return elementName;
        }
      }

    return null;
  }

}

/* 
 * Copyright 2018, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Convert a standard node callback-style function into a function that returns
 * a Promise.
 * 
 * @param {Function} func
 * @return {Function}
 */
function promisify(func) {
  return function () {
    return new Promise((resolve, reject) => {
      func(...arguments, (error, result) => {
        if (error) {
          reject(new Error(result));
        } else {
          resolve(result);
        }
      });
    });
  };
}

/**
 * A highly-configurable class responsible for processing the Thymeleaf
 * directives found in HTML documents and fragments.
 * 
 * @author Emanuel Rabina
 */

class TemplateEngine {
  /**
   * Constructor, set up a new template engine instance.
   * 
   * @param {Object} [config=DEFAULT_CONFIGURATION]
   */
  constructor({
    dialects,
    isomorphic,
    messageResolver,
    templateResolver
  } = DEFAULT_CONFIGURATION) {
    this.dialects = dialects;
    this.isomorphic = isomorphic;
    this.messageResolver = messageResolver;
    this.templateResolver = templateResolver;
    let standardDialect = dialects.find(dialect => dialect instanceof StandardDialect);
    this.xmlNamespaceAttribute = `xmlns:${standardDialect ? standardDialect.prefix : StandardDialect.DEFAULT_PREFIX}`; // Combine all processors into a unified list

    this.processors = dialects.reduce((acc, {
      processors
    }) => processors ? [...acc, ...processors] : acc, []); // Combine all expression objects into a unified object

    this.expressionObjects = dialects.reduce((acc, {
      expressionObjects
    }) => expressionObjects ? _objectSpread2(_objectSpread2({}, acc), expressionObjects) : acc, {});
  }
  /**
   * Process the Thymeleaf template data, returning the processed template.
   *
   * @param {String} template
   * @param {Object} [context={}]
   * @return {Promise<String>}
   *   A promise resolved with the processed template, or rejected with an error
   *   message.
   */


  process(template, context = {}) {
    let document = deserialize(template);
    let rootElement = document.firstElementChild;
    return this.processNode(rootElement, _objectSpread2(_objectSpread2(_objectSpread2({}, context), this.expressionObjects), {}, {
      dialects: this.dialects,
      messageResolver: this.messageResolver,
      templateResolver: this.templateResolver
    })).then(() => {
      // TODO: Special case, remove the xmlns:thjs namespace from the
      //       document.  This should be handled like in main Thymeleaf where
      //       it's just another processor that runs on the document.
      if (rootElement.hasAttribute(this.xmlNamespaceAttribute)) {
        rootElement.removeAttribute(this.xmlNamespaceAttribute);
      }

      return serialize(document);
    });
  }
  /**
   * Process the Thymeleaf template at the given path, returning a promise of the
   * processed template.
   * 
   * @param {String} filePath
   * @param {Object} [context={}]
   * @return {Promise<String>}
   *   A promise resolved with the processed template, or rejected with an error
   *   message.
   */


  processFile(filePath, context = {}) {
    /* global "node" */
    return  promisify(require('fs').readFile)(filePath).then(data => {
      return this.process(data, context);
    });
  }
  /**
   * Process a DOM element.
   * 
   * @private
   * @param {Element} element
   * @param {Object} [context={}]
   * @return {Promise<Boolean>} Whether or not the parent node needs
   *   reprocessing.
   */


  async processNode(element, context = {}) {
    let localVariables = element.__thymeleafLocalVariables || {};

    let localContext = _objectSpread2(_objectSpread2({}, context), localVariables);

    let matcher = new Matcher(localContext, this.isomorphic); // Run the current element through the gamut of registered processors.  If
    // one of them sends a reprocessing signal, return from this method to let
    // the caller re-run everything.

    for (let i = 0; i < this.processors.length; i++) {
      let processor = this.processors[i];
      let processorResult = false; // TODO: Some way to do this generically and not have to type check?

      let attributeOrElementName = matcher.matches(element, processor);

      if (attributeOrElementName) {
        if (processor instanceof AttributeProcessor) {
          processorResult = await processor.process(element, attributeOrElementName, element.getAttribute(attributeOrElementName), localContext);
        } else if (processor instanceof ElementProcessor) {
          processorResult = await processor.process(element, localContext);
        }
      }

      if (processorResult) {
        return true;
      }
    } // Process this element's children, handling the reprocessing signal to
    // re-run the 'current' child element (could have been shifted due to being
    // removed etc).


    for (let i = 0; i < element.children.length;) {
      let child = element.children[i];
      let reprocess = await this.processNode(child, localContext);

      if (!reprocess) {
        i++;
      }
    }
  }

}

exports.AttributeProcessor = AttributeProcessor;
exports.Dialect = Dialect;
exports.ExpressionProcessor = ExpressionProcessor;
exports.STANDARD_CONFIGURATION = STANDARD_CONFIGURATION;
exports.StandardDialect = StandardDialect;
exports.TemplateEngine = TemplateEngine;
//# sourceMappingURL=thymeleaf.node.cjs.js.map
