(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.TinyMDE = {}));
})(this, (function (exports) { 'use strict';

  const svg = {
    blockquote: `<svg height="18" width="18"><rect width="5" height="5" x="3" y="4" ry="1"/><rect ry="1" y="4" x="10" height="5" width="5"/><path d="M8 6.999v3c0 1-1 3-1 3s-.331 1-1.331 1h-1c-1 0-.669-1-.669-1s1-2 1-3v-3zm7 0v3c0 1-1 3-1 3s-.331 1-1.331 1h-1c-1 0-.669-1-.669-1s1-2 1-3v-3z"/></svg>`,
    bold: `<svg height="18" width="18"><path d="M4 2a1 1 0 00-1 1v12a1 1 0 001 1h6c4 0 5-2 5-4 0-1.322-.434-2.636-1.885-3.381C13.772 7.885 14 6.945 14 6c0-2-1-4-5-4zm1 2h4c1.668 0 2.32.393 2.6.672.28.279.4.662.4 1.328s-.12 1.057-.4 1.338c-.275.274-1.014.646-2.6.662H5zm5 6c1.666.005 2.318.388 2.596.666.277.278.404.667.404 1.334s-.127 1.06-.404 1.338c-.278.278-.93.66-2.596.662l-4.992.004L5 10z"/></svg>`,
    clear_formatting: `<svg height="18" width="18"><path d="M11.03 1a1 1 0 00-.74.3l-9 9a1 1 0 000 1.4l4 4A1 1 0 006 16h2a1 1 0 00.7-.3l8-8a1 1 0 000-1.4l-5-5a1 1 0 00-.67-.3zM8.7 5.7l3.58 3.6L7.6 14H6.4l-3-3 5.3-5.3z"/><rect ry=".8" rx=".8" y="14" x="16" height="2" width="2"/><rect width="2" height="2" x="13" y="14" rx=".8" ry=".8"/><rect ry=".8" rx=".8" y="14" x="10" height="2" width="2"/></svg>`,
    code: `<svg height="18" width="18"><path d="M13.5 2.994a.5.5 0 00-.5.5.5.5 0 000 .034V4.53a5.993 5.993 0 00-7.451-.445A6 6 0 1012.45 13.9a5.99 5.99 0 001.346-1.334.5.5 0 00.096-.101.5.5 0 00-.12-.698.5.5 0 00-.697.12l-.004-.005a5 5 0 01-1.197 1.2 5 5 0 111.215-6.965.5.5 0 00.697.12.5.5 0 00.211-.44V4.745H14V3.493a.5.5 0 00-.5-.5z"/></svg>`,
    h1: `<svg height="18" width="18"><path d="M3 2s0-1 1-1h1c1 0 1 1 1 1v6h6V2s0-1 1-1h1c1 0 1 1 1 1v14s0 1-1 1h-1c-1 0-1-1-1-1v-6H6v6s0 1-1 1H4c-1 0-1-1-1-1z"/></svg>`,
    h2: `<svg height="18" width="18"><path d="M4 2s0-1 1-1 1 1 1 1v6c1-1 2-1 4-1 3 0 4 2 4 4v5s0 1-1 1-1-1-1-1v-5c0-1.094-1-2-2-2-2 0-3 0-4 2v5s0 1-1 1-1-1-1-1z"/></svg>`,
    hr: `<svg height="18" width="18"><rect ry="1" y="8" height="2" width="18" style="font-variation-settings:normal;marker:none"/></svg>`,
    image: `<svg height="18" width="18"><path d="M1 2v14h16V2H1zm2 2h12v7l-3-3-4 4-2-2-3 3V4z"/><circle r="1.5" cy="6.5" cx="5.5"/></svg>`,
    italic: `<svg height="18" width="18"><path d="M9 2a1 1 0 000 2L7 14a1 1 0 100 2h2a1 1 0 000-2l2-10a1 1 0 100-2z"/></svg>`,
    link: `<svg height="18" width="18"><path d="M9.07 5.18a3.9 3.9 0 00-1.52.43C6.31 6.22 5.3 7.29 4.3 8.29c-1 1-2.07 2.02-2.68 3.26-.31.62-.5 1.33-.41 2.07.09.75.48 1.47 1.1 2.09.61.61 1.33 1 2.08 1.1.74.09 1.45-.1 2.07-.42 1.24-.61 2.26-1.68 3.26-2.68.46-.47.94-.94 1.39-1.44l-.43.26c-.68.34-1.49.56-2.36.46-.2-.03-.4-.08-.6-.14-.79.76-1.55 1.45-2.16 1.76-.38.19-.67.24-.92.21-.26-.03-.54-.14-.92-.53-.39-.38-.5-.66-.53-.91-.03-.26.02-.55.21-.93.39-.76 1.32-1.74 2.32-2.74 1-1 1.98-1.93 2.74-2.32.38-.19.67-.24.93-.21.25.03.53.14.91.53.39.38.5.66.53.92v.06l1.12-1.06.44-.47c-.18-.3-.4-.6-.67-.87-.62-.61-1.34-1-2.09-1.1a3.08 3.08 0 00-.55-.01z"/><path d="M13.07.86a3.9 3.9 0 00-1.52.43c-1.24.62-2.26 1.69-3.26 2.69-.46.47-.94.94-1.39 1.43.15-.08.28-.18.43-.25a4.4 4.4 0 012.36-.46c.2.02.4.07.6.14.79-.77 1.55-1.46 2.16-1.76.38-.19.67-.25.93-.21.25.03.53.14.91.52.39.38.5.66.53.92.03.26-.02.55-.21.93-.39.76-1.32 1.74-2.32 2.74-1 1-1.98 1.93-2.74 2.31-.38.2-.67.25-.93.22-.25-.04-.53-.15-.91-.53-.39-.38-.5-.66-.53-.92V9c-.36.33-.73.67-1.12 1.06l-.43.46c.17.3.4.6.66.87.62.62 1.34 1 2.08 1.1.75.1 1.46-.1 2.08-.41 1.24-.62 2.26-1.69 3.26-2.69s2.07-2.02 2.68-3.26c.31-.62.5-1.32.41-2.07a3.63 3.63 0 00-1.1-2.08c-.61-.62-1.33-1-2.07-1.1a3.08 3.08 0 00-.56-.02z"/></svg>`,
    ol: `<svg height="18" width="18"><path d="M1.5 7a.5.5 0 100 1h1a.5.5 0 01.5.5c0 .164-.08.31-.14.355l-1.655 1.25A.492.492 0 001 10.5a.5.5 0 00.5.5h2a.5.5 0 000-1H3l.398-.299A1.5 1.5 0 002.5 7z"/><path d="M1.5 12c-.667 0-.667 1 0 1h1.25c.333 0 .333.5 0 .5H2.5c-.667 0-.667 1 0 1h.25c.333 0 .333.5 0 .5H1.5c-.667 0-.667 1 0 1h1c.398 0 .78-.131 1.06-.365.282-.235.44-.554.44-.885a1.121 1.121 0 00-.303-.75c.191-.204.3-.47.303-.75 0-.332-.158-.651-.44-.885-.3-.241-.675-.37-1.06-.365z"/><rect y="13" x="6" height="2" width="12" ry="1"/><rect ry="1" width="12" height="2" x="6" y="8"/><rect y="3" x="6" height="2" width="12" ry="1"/><path d="M1.5 2a.5.5 0 100 1H2v2h-.5a.5.5 0 100 1h2a.5.5 0 100-1H3V2.5c0-.277-.223-.5-.5-.5z"/></svg>`,
    strikethrough: `<svg width="18" height="18"><path d="M10 2C6 2 4 4 4 6c0 .338.08.672.193 1h2.34C6.113 6.629 6 6.295 6 6c0-.334.117-.725.691-1.154C7.265 4.416 8.331 4 10 4h3a1 1 0 000-2zm1.477 9c.413.368.523.706.523 1 0 .334-.127.712-.701 1.143-.575.43-1.632.85-3.299.857l-1.006.007V14H5a1 1 0 000 2h3c4 0 6-2 6-4 0-.338-.081-.672-.195-1z"/><rect ry="1" y="8" x="1" height="2" width="16"/></svg>`,
    ul: `<svg height="18" width="18"><circle cx="2" cy="9" r="2"/><circle cy="4" cx="2" r="2"/><rect y="3" x="6" height="2" width="12" ry="1"/><circle cx="2" cy="14" r="2"/><rect ry="1" width="12" height="2" x="6" y="8"/><rect y="13" x="6" height="2" width="12" ry="1"/></svg>`
  };

  const isMacLike = /(Mac|iPhone|iPod|iPad)/i.test(typeof navigator !== "undefined" ? navigator.platform : "");
  const DefaultCommands = {
    'bold': {
      name: 'bold',
      action: 'bold',
      innerHTML: svg.bold,
      title: 'Bold',
      hotkey: 'Mod-B'
    },
    'italic': {
      name: 'italic',
      action: 'italic',
      innerHTML: svg.italic,
      title: 'Italic',
      hotkey: 'Mod-I'
    },
    'strikethrough': {
      name: 'strikethrough',
      action: 'strikethrough',
      innerHTML: svg.strikethrough,
      title: 'Strikethrough',
      hotkey: 'Mod2-Shift-5'
    },
    'code': {
      name: 'code',
      action: 'code',
      innerHTML: svg.code,
      title: 'Format as code'
    },
    'h1': {
      name: 'h1',
      action: 'h1',
      innerHTML: svg.h1,
      title: 'Level 1 heading',
      hotkey: 'Mod-Shift-1'
    },
    'h2': {
      name: 'h2',
      action: 'h2',
      innerHTML: svg.h2,
      title: 'Level 2 heading',
      hotkey: 'Mod-Shift-2'
    },
    'ul': {
      name: 'ul',
      action: 'ul',
      innerHTML: svg.ul,
      title: 'Bulleted list'
    },
    'ol': {
      name: 'ol',
      action: 'ol',
      innerHTML: svg.ol,
      title: 'Numbered list'
    },
    'blockquote': {
      name: 'blockquote',
      action: 'blockquote',
      innerHTML: svg.blockquote,
      title: 'Quote',
      hotkey: 'Mod2-Shift-Q'
    },
    'insertLink': {
      name: 'insertLink',
      action: editor => {
        if (editor.isInlineFormattingAllowed()) editor.wrapSelection('[', ']()');
      },
      enabled: (editor, focus, anchor) => editor.isInlineFormattingAllowed(focus, anchor) ? false : null,
      innerHTML: svg.link,
      title: 'Insert link',
      hotkey: 'Mod-K'
    },
    'insertImage': {
      name: 'insertImage',
      action: editor => {
        if (editor.isInlineFormattingAllowed()) editor.wrapSelection('![', ']()');
      },
      enabled: (editor, focus, anchor) => editor.isInlineFormattingAllowed(focus, anchor) ? false : null,
      innerHTML: svg.image,
      title: 'Insert image',
      hotkey: 'Mod2-Shift-I'
    },
    'hr': {
      name: 'hr',
      action: editor => editor.paste('\n***\n'),
      enabled: () => false,
      innerHTML: svg.hr,
      title: 'Insert horizontal line',
      hotkey: 'Mod2-Shift-L'
    }
  };
  class CommandBar {
    constructor(props) {
      this.e = null;
      this.editor = null;
      this.commands = [];
      this.buttons = {};
      this.state = {};
      this.hotkeys = [];
      let element = props.element;
      if (element && !element.tagName) {
        element = document.getElementById(props.element);
      }
      if (!element) {
        element = document.body;
      }
      this.createCommandBarElement(element, props.commands || ['bold', 'italic', 'strikethrough', '|', 'code', '|', 'h1', 'h2', '|', 'ul', 'ol', '|', 'blockquote', 'hr', '|', 'insertLink', 'insertImage']);
      document.addEventListener('keydown', e => this.handleKeydown(e));
      if (props.editor) this.setEditor(props.editor);
    }
    createCommandBarElement(parentElement, commands) {
      this.e = document.createElement('div');
      this.e.className = 'TMCommandBar';
      for (let command of commands) {
        if (command == '|') {
          let el = document.createElement('div');
          el.className = 'TMCommandDivider';
          this.e.appendChild(el);
        } else {
          let commandName;
          if (typeof command == "string") {
            // Reference to default command

            if (DefaultCommands[command]) {
              commandName = command;
              this.commands[commandName] = DefaultCommands[commandName];
            } else {
              continue;
            }
          } else if (typeof command == "object" && command.name) {
            commandName = command.name;
            this.commands[commandName] = {};
            if (DefaultCommands[commandName]) Object.assign(this.commands[commandName], DefaultCommands[commandName]);
            Object.assign(this.commands[commandName], command);
          } else {
            continue;
          }
          let title = this.commands[commandName].title || commandName;
          if (this.commands[commandName].hotkey) {
            const keys = this.commands[commandName].hotkey.split('-');
            // construct modifiers
            let modifiers = [];
            let modifierexplanation = [];
            for (let i = 0; i < keys.length - 1; i++) {
              switch (keys[i]) {
                case 'Ctrl':
                  modifiers.push('ctrlKey');
                  modifierexplanation.push('Ctrl');
                  break;
                case 'Cmd':
                  modifiers.push('metaKey');
                  modifierexplanation.push('⌘');
                  break;
                case 'Alt':
                  modifiers.push('altKey');
                  modifierexplanation.push('Alt');
                  break;
                case 'Option':
                  modifiers.push('altKey');
                  modifierexplanation.push('⌥');
                  break;
                case 'Win':
                  modifiers.push('metaKey');
                  modifierexplanation.push('⊞ Win');
                  break;
                case 'Shift':
                  modifiers.push('shiftKey');
                  modifierexplanation.push('⇧');
                  break;
                case 'Mod':
                  // Mod is a convenience mechanism: Ctrl on Windows, Cmd on Mac
                  if (isMacLike) {
                    modifiers.push('metaKey');
                    modifierexplanation.push('⌘');
                  } else {
                    modifiers.push('ctrlKey');
                    modifierexplanation.push('Ctrl');
                  }
                  break;
                case 'Mod2':
                  modifiers.push('altKey');
                  if (isMacLike) modifierexplanation.push('⌥');else modifierexplanation.push('Alt');
                  break;
                // Mod2 is a convenience mechanism: Alt on Windows, Option on Mac
              }
            }

            modifierexplanation.push(keys[keys.length - 1]);
            let hotkey = {
              modifiers: modifiers,
              command: commandName
            };
            // TODO Right now this is working only for letters and numbers
            if (keys[keys.length - 1].match(/^[0-9]$/)) {
              hotkey.code = `Digit${keys[keys.length - 1]}`;
            } else {
              hotkey.key = keys[keys.length - 1].toLowerCase();
            }
            this.hotkeys.push(hotkey);
            title = title.concat(` (${modifierexplanation.join('+')})`);
          }
          this.buttons[commandName] = document.createElement('div');
          this.buttons[commandName].className = 'TMCommandButton TMCommandButton_Disabled';
          this.buttons[commandName].title = title;
          this.buttons[commandName].innerHTML = this.commands[commandName].innerHTML;
          this.buttons[commandName].addEventListener('mousedown', e => this.handleClick(commandName, e));
          this.e.appendChild(this.buttons[commandName]);
        }
      }
      parentElement.appendChild(this.e);
    }
    handleClick(commandName, event) {
      if (!this.editor) return;
      event.preventDefault();
      if (typeof this.commands[commandName].action == "string") {
        if (this.state[commandName] === false) this.editor.setCommandState(commandName, true);else this.editor.setCommandState(commandName, false);
      } else if (typeof this.commands[commandName].action == "function") {
        this.commands[commandName].action(this.editor);
      }
    }
    setEditor(editor) {
      this.editor = editor;
      editor.addEventListener('selection', e => this.handleSelection(e));
    }
    handleSelection(event) {
      if (event.commandState) {
        for (let command in this.commands) {
          if (event.commandState[command] === undefined) {
            if (this.commands[command].enabled) this.state[command] = this.commands[command].enabled(this.editor, event.focus, event.anchor);else this.state[command] = event.focus ? false : null;
          } else {
            this.state[command] = event.commandState[command];
          }
          if (this.state[command] === true) {
            this.buttons[command].className = 'TMCommandButton TMCommandButton_Active';
          } else if (this.state[command] === false) {
            this.buttons[command].className = 'TMCommandButton TMCommandButton_Inactive';
          } else {
            this.buttons[command].className = 'TMCommandButton TMCommandButton_Disabled';
          }
        }
      }
    }
    handleKeydown(event) {
      outer: for (let hotkey of this.hotkeys) {
        if (hotkey.key && event.key.toLowerCase() == hotkey.key || hotkey.code && event.code == hotkey.code) {
          // Key matches hotkey. Look for any required modifier that wasn't pressed
          for (let modifier of hotkey.modifiers) {
            if (!event[modifier]) continue outer;
          }
          // Everything matches.
          this.handleClick(hotkey.command, event);
          return;
        }
      }
    }
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  var check = function (it) {
    return it && it.Math === Math && it;
  };

  // https://github.com/zloirock/core-js/issues/86#issuecomment-115759028
  var global$b =
  // eslint-disable-next-line es/no-global-this -- safe
  check(typeof globalThis == 'object' && globalThis) || check(typeof window == 'object' && window) ||
  // eslint-disable-next-line no-restricted-globals -- safe
  check(typeof self == 'object' && self) || check(typeof commonjsGlobal == 'object' && commonjsGlobal) ||
  // eslint-disable-next-line no-new-func -- fallback
  function () {
    return this;
  }() || commonjsGlobal || Function('return this')();

  var fails$7 = function (exec) {
    try {
      return !!exec();
    } catch (error) {
      return true;
    }
  };

  var fails$6 = fails$7;

  // Detect IE8's incomplete defineProperty implementation
  var descriptors = !fails$6(function () {
    // eslint-disable-next-line es/no-object-defineproperty -- required for testing
    return Object.defineProperty({}, 1, {
      get: function () {
        return 7;
      }
    })[1] !== 7;
  });

  var makeBuiltInExports = {};
  var makeBuiltIn$2 = {
    get exports(){ return makeBuiltInExports; },
    set exports(v){ makeBuiltInExports = v; },
  };

  var fails$5 = fails$7;
  var functionBindNative = !fails$5(function () {
    // eslint-disable-next-line es/no-function-prototype-bind -- safe
    var test = function () {/* empty */}.bind();
    // eslint-disable-next-line no-prototype-builtins -- safe
    return typeof test != 'function' || test.hasOwnProperty('prototype');
  });

  var NATIVE_BIND$1 = functionBindNative;
  var FunctionPrototype$1 = Function.prototype;
  var call$3 = FunctionPrototype$1.call;
  var uncurryThisWithBind = NATIVE_BIND$1 && FunctionPrototype$1.bind.bind(call$3, call$3);
  var functionUncurryThis = NATIVE_BIND$1 ? uncurryThisWithBind : function (fn) {
    return function () {
      return call$3.apply(fn, arguments);
    };
  };

  var documentAll$2 = typeof document == 'object' && document.all;

  // https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot
  // eslint-disable-next-line unicorn/no-typeof-undefined -- required for testing
  var IS_HTMLDDA = typeof documentAll$2 == 'undefined' && documentAll$2 !== undefined;
  var documentAll_1 = {
    all: documentAll$2,
    IS_HTMLDDA: IS_HTMLDDA
  };

  var $documentAll$1 = documentAll_1;
  var documentAll$1 = $documentAll$1.all;

  // `IsCallable` abstract operation
  // https://tc39.es/ecma262/#sec-iscallable
  var isCallable$8 = $documentAll$1.IS_HTMLDDA ? function (argument) {
    return typeof argument == 'function' || argument === documentAll$1;
  } : function (argument) {
    return typeof argument == 'function';
  };

  // we can't use just `it == null` since of `document.all` special case
  // https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot-aec
  var isNullOrUndefined$2 = function (it) {
    return it === null || it === undefined;
  };

  var isNullOrUndefined$1 = isNullOrUndefined$2;
  var $TypeError$5 = TypeError;

  // `RequireObjectCoercible` abstract operation
  // https://tc39.es/ecma262/#sec-requireobjectcoercible
  var requireObjectCoercible$1 = function (it) {
    if (isNullOrUndefined$1(it)) throw new $TypeError$5("Can't call method on " + it);
    return it;
  };

  var requireObjectCoercible = requireObjectCoercible$1;
  var $Object$1 = Object;

  // `ToObject` abstract operation
  // https://tc39.es/ecma262/#sec-toobject
  var toObject$1 = function (argument) {
    return $Object$1(requireObjectCoercible(argument));
  };

  var uncurryThis$4 = functionUncurryThis;
  var toObject = toObject$1;
  var hasOwnProperty = uncurryThis$4({}.hasOwnProperty);

  // `HasOwnProperty` abstract operation
  // https://tc39.es/ecma262/#sec-hasownproperty
  // eslint-disable-next-line es/no-object-hasown -- safe
  var hasOwnProperty_1 = Object.hasOwn || function hasOwn(it, key) {
    return hasOwnProperty(toObject(it), key);
  };

  var DESCRIPTORS$6 = descriptors;
  var hasOwn$3 = hasOwnProperty_1;
  var FunctionPrototype = Function.prototype;
  // eslint-disable-next-line es/no-object-getownpropertydescriptor -- safe
  var getDescriptor = DESCRIPTORS$6 && Object.getOwnPropertyDescriptor;
  var EXISTS$1 = hasOwn$3(FunctionPrototype, 'name');
  // additional protection from minified / mangled / dropped function names
  var PROPER = EXISTS$1 && function something() {/* empty */}.name === 'something';
  var CONFIGURABLE$1 = EXISTS$1 && (!DESCRIPTORS$6 || DESCRIPTORS$6 && getDescriptor(FunctionPrototype, 'name').configurable);
  var functionName = {
    EXISTS: EXISTS$1,
    PROPER: PROPER,
    CONFIGURABLE: CONFIGURABLE$1
  };

  var global$a = global$b;

  // eslint-disable-next-line es/no-object-defineproperty -- safe
  var defineProperty$2 = Object.defineProperty;
  var defineGlobalProperty$1 = function (key, value) {
    try {
      defineProperty$2(global$a, key, {
        value: value,
        configurable: true,
        writable: true
      });
    } catch (error) {
      global$a[key] = value;
    }
    return value;
  };

  var global$9 = global$b;
  var defineGlobalProperty = defineGlobalProperty$1;
  var SHARED = '__core-js_shared__';
  var store$3 = global$9[SHARED] || defineGlobalProperty(SHARED, {});
  var sharedStore = store$3;

  var uncurryThis$3 = functionUncurryThis;
  var isCallable$7 = isCallable$8;
  var store$2 = sharedStore;
  var functionToString = uncurryThis$3(Function.toString);

  // this helper broken in `core-js@3.4.1-3.4.4`, so we can't use `shared` helper
  if (!isCallable$7(store$2.inspectSource)) {
    store$2.inspectSource = function (it) {
      return functionToString(it);
    };
  }
  var inspectSource$1 = store$2.inspectSource;

  var global$8 = global$b;
  var isCallable$6 = isCallable$8;
  var WeakMap$1 = global$8.WeakMap;
  var weakMapBasicDetection = isCallable$6(WeakMap$1) && /native code/.test(String(WeakMap$1));

  var isCallable$5 = isCallable$8;
  var $documentAll = documentAll_1;
  var documentAll = $documentAll.all;
  var isObject$5 = $documentAll.IS_HTMLDDA ? function (it) {
    return typeof it == 'object' ? it !== null : isCallable$5(it) || it === documentAll;
  } : function (it) {
    return typeof it == 'object' ? it !== null : isCallable$5(it);
  };

  var objectDefineProperty = {};

  var global$7 = global$b;
  var isObject$4 = isObject$5;
  var document$1 = global$7.document;
  // typeof document.createElement is 'object' in old IE
  var EXISTS = isObject$4(document$1) && isObject$4(document$1.createElement);
  var documentCreateElement = function (it) {
    return EXISTS ? document$1.createElement(it) : {};
  };

  var DESCRIPTORS$5 = descriptors;
  var fails$4 = fails$7;
  var createElement = documentCreateElement;

  // Thanks to IE8 for its funny defineProperty
  var ie8DomDefine = !DESCRIPTORS$5 && !fails$4(function () {
    // eslint-disable-next-line es/no-object-defineproperty -- required for testing
    return Object.defineProperty(createElement('div'), 'a', {
      get: function () {
        return 7;
      }
    }).a !== 7;
  });

  var DESCRIPTORS$4 = descriptors;
  var fails$3 = fails$7;

  // V8 ~ Chrome 36-
  // https://bugs.chromium.org/p/v8/issues/detail?id=3334
  var v8PrototypeDefineBug = DESCRIPTORS$4 && fails$3(function () {
    // eslint-disable-next-line es/no-object-defineproperty -- required for testing
    return Object.defineProperty(function () {/* empty */}, 'prototype', {
      value: 42,
      writable: false
    }).prototype !== 42;
  });

  var isObject$3 = isObject$5;
  var $String$3 = String;
  var $TypeError$4 = TypeError;

  // `Assert: Type(argument) is Object`
  var anObject$2 = function (argument) {
    if (isObject$3(argument)) return argument;
    throw new $TypeError$4($String$3(argument) + ' is not an object');
  };

  var NATIVE_BIND = functionBindNative;
  var call$2 = Function.prototype.call;
  var functionCall = NATIVE_BIND ? call$2.bind(call$2) : function () {
    return call$2.apply(call$2, arguments);
  };

  var global$6 = global$b;
  var isCallable$4 = isCallable$8;
  var aFunction = function (argument) {
    return isCallable$4(argument) ? argument : undefined;
  };
  var getBuiltIn$1 = function (namespace, method) {
    return arguments.length < 2 ? aFunction(global$6[namespace]) : global$6[namespace] && global$6[namespace][method];
  };

  var uncurryThis$2 = functionUncurryThis;
  var objectIsPrototypeOf = uncurryThis$2({}.isPrototypeOf);

  var engineUserAgent = typeof navigator != 'undefined' && String(navigator.userAgent) || '';

  var global$5 = global$b;
  var userAgent = engineUserAgent;
  var process = global$5.process;
  var Deno = global$5.Deno;
  var versions = process && process.versions || Deno && Deno.version;
  var v8 = versions && versions.v8;
  var match, version;
  if (v8) {
    match = v8.split('.');
    // in old Chrome, versions of V8 isn't V8 = Chrome / 10
    // but their correct versions are not interesting for us
    version = match[0] > 0 && match[0] < 4 ? 1 : +(match[0] + match[1]);
  }

  // BrowserFS NodeJS `process` polyfill incorrectly set `.v8` to `0.0`
  // so check `userAgent` even if `.v8` exists, but 0
  if (!version && userAgent) {
    match = userAgent.match(/Edge\/(\d+)/);
    if (!match || match[1] >= 74) {
      match = userAgent.match(/Chrome\/(\d+)/);
      if (match) version = +match[1];
    }
  }
  var engineV8Version = version;

  /* eslint-disable es/no-symbol -- required for testing */
  var V8_VERSION = engineV8Version;
  var fails$2 = fails$7;
  var global$4 = global$b;
  var $String$2 = global$4.String;

  // eslint-disable-next-line es/no-object-getownpropertysymbols -- required for testing
  var symbolConstructorDetection = !!Object.getOwnPropertySymbols && !fails$2(function () {
    var symbol = Symbol('symbol detection');
    // Chrome 38 Symbol has incorrect toString conversion
    // `get-own-property-symbols` polyfill symbols converted to object are not Symbol instances
    // nb: Do not call `String` directly to avoid this being optimized out to `symbol+''` which will,
    // of course, fail.
    return !$String$2(symbol) || !(Object(symbol) instanceof Symbol) ||
    // Chrome 38-40 symbols are not inherited from DOM collections prototypes to instances
    !Symbol.sham && V8_VERSION && V8_VERSION < 41;
  });

  /* eslint-disable es/no-symbol -- required for testing */
  var NATIVE_SYMBOL$1 = symbolConstructorDetection;
  var useSymbolAsUid = NATIVE_SYMBOL$1 && !Symbol.sham && typeof Symbol.iterator == 'symbol';

  var getBuiltIn = getBuiltIn$1;
  var isCallable$3 = isCallable$8;
  var isPrototypeOf = objectIsPrototypeOf;
  var USE_SYMBOL_AS_UID$1 = useSymbolAsUid;
  var $Object = Object;
  var isSymbol$2 = USE_SYMBOL_AS_UID$1 ? function (it) {
    return typeof it == 'symbol';
  } : function (it) {
    var $Symbol = getBuiltIn('Symbol');
    return isCallable$3($Symbol) && isPrototypeOf($Symbol.prototype, $Object(it));
  };

  var $String$1 = String;
  var tryToString$1 = function (argument) {
    try {
      return $String$1(argument);
    } catch (error) {
      return 'Object';
    }
  };

  var isCallable$2 = isCallable$8;
  var tryToString = tryToString$1;
  var $TypeError$3 = TypeError;

  // `Assert: IsCallable(argument) is true`
  var aCallable$1 = function (argument) {
    if (isCallable$2(argument)) return argument;
    throw new $TypeError$3(tryToString(argument) + ' is not a function');
  };

  var aCallable = aCallable$1;
  var isNullOrUndefined = isNullOrUndefined$2;

  // `GetMethod` abstract operation
  // https://tc39.es/ecma262/#sec-getmethod
  var getMethod$1 = function (V, P) {
    var func = V[P];
    return isNullOrUndefined(func) ? undefined : aCallable(func);
  };

  var call$1 = functionCall;
  var isCallable$1 = isCallable$8;
  var isObject$2 = isObject$5;
  var $TypeError$2 = TypeError;

  // `OrdinaryToPrimitive` abstract operation
  // https://tc39.es/ecma262/#sec-ordinarytoprimitive
  var ordinaryToPrimitive$1 = function (input, pref) {
    var fn, val;
    if (pref === 'string' && isCallable$1(fn = input.toString) && !isObject$2(val = call$1(fn, input))) return val;
    if (isCallable$1(fn = input.valueOf) && !isObject$2(val = call$1(fn, input))) return val;
    if (pref !== 'string' && isCallable$1(fn = input.toString) && !isObject$2(val = call$1(fn, input))) return val;
    throw new $TypeError$2("Can't convert object to primitive value");
  };

  var sharedExports = {};
  var shared$3 = {
    get exports(){ return sharedExports; },
    set exports(v){ sharedExports = v; },
  };

  var store$1 = sharedStore;
  (shared$3.exports = function (key, value) {
    return store$1[key] || (store$1[key] = value !== undefined ? value : {});
  })('versions', []).push({
    version: '3.33.0',
    mode: 'global',
    copyright: '© 2014-2023 Denis Pushkarev (zloirock.ru)',
    license: 'https://github.com/zloirock/core-js/blob/v3.33.0/LICENSE',
    source: 'https://github.com/zloirock/core-js'
  });

  var uncurryThis$1 = functionUncurryThis;
  var id = 0;
  var postfix = Math.random();
  var toString = uncurryThis$1(1.0.toString);
  var uid$2 = function (key) {
    return 'Symbol(' + (key === undefined ? '' : key) + ')_' + toString(++id + postfix, 36);
  };

  var global$3 = global$b;
  var shared$2 = sharedExports;
  var hasOwn$2 = hasOwnProperty_1;
  var uid$1 = uid$2;
  var NATIVE_SYMBOL = symbolConstructorDetection;
  var USE_SYMBOL_AS_UID = useSymbolAsUid;
  var Symbol$1 = global$3.Symbol;
  var WellKnownSymbolsStore = shared$2('wks');
  var createWellKnownSymbol = USE_SYMBOL_AS_UID ? Symbol$1['for'] || Symbol$1 : Symbol$1 && Symbol$1.withoutSetter || uid$1;
  var wellKnownSymbol$1 = function (name) {
    if (!hasOwn$2(WellKnownSymbolsStore, name)) {
      WellKnownSymbolsStore[name] = NATIVE_SYMBOL && hasOwn$2(Symbol$1, name) ? Symbol$1[name] : createWellKnownSymbol('Symbol.' + name);
    }
    return WellKnownSymbolsStore[name];
  };

  var call = functionCall;
  var isObject$1 = isObject$5;
  var isSymbol$1 = isSymbol$2;
  var getMethod = getMethod$1;
  var ordinaryToPrimitive = ordinaryToPrimitive$1;
  var wellKnownSymbol = wellKnownSymbol$1;
  var $TypeError$1 = TypeError;
  var TO_PRIMITIVE = wellKnownSymbol('toPrimitive');

  // `ToPrimitive` abstract operation
  // https://tc39.es/ecma262/#sec-toprimitive
  var toPrimitive$1 = function (input, pref) {
    if (!isObject$1(input) || isSymbol$1(input)) return input;
    var exoticToPrim = getMethod(input, TO_PRIMITIVE);
    var result;
    if (exoticToPrim) {
      if (pref === undefined) pref = 'default';
      result = call(exoticToPrim, input, pref);
      if (!isObject$1(result) || isSymbol$1(result)) return result;
      throw new $TypeError$1("Can't convert object to primitive value");
    }
    if (pref === undefined) pref = 'number';
    return ordinaryToPrimitive(input, pref);
  };

  var toPrimitive = toPrimitive$1;
  var isSymbol = isSymbol$2;

  // `ToPropertyKey` abstract operation
  // https://tc39.es/ecma262/#sec-topropertykey
  var toPropertyKey$1 = function (argument) {
    var key = toPrimitive(argument, 'string');
    return isSymbol(key) ? key : key + '';
  };

  var DESCRIPTORS$3 = descriptors;
  var IE8_DOM_DEFINE = ie8DomDefine;
  var V8_PROTOTYPE_DEFINE_BUG = v8PrototypeDefineBug;
  var anObject$1 = anObject$2;
  var toPropertyKey = toPropertyKey$1;
  var $TypeError = TypeError;
  // eslint-disable-next-line es/no-object-defineproperty -- safe
  var $defineProperty = Object.defineProperty;
  // eslint-disable-next-line es/no-object-getownpropertydescriptor -- safe
  var $getOwnPropertyDescriptor = Object.getOwnPropertyDescriptor;
  var ENUMERABLE = 'enumerable';
  var CONFIGURABLE = 'configurable';
  var WRITABLE = 'writable';

  // `Object.defineProperty` method
  // https://tc39.es/ecma262/#sec-object.defineproperty
  objectDefineProperty.f = DESCRIPTORS$3 ? V8_PROTOTYPE_DEFINE_BUG ? function defineProperty(O, P, Attributes) {
    anObject$1(O);
    P = toPropertyKey(P);
    anObject$1(Attributes);
    if (typeof O === 'function' && P === 'prototype' && 'value' in Attributes && WRITABLE in Attributes && !Attributes[WRITABLE]) {
      var current = $getOwnPropertyDescriptor(O, P);
      if (current && current[WRITABLE]) {
        O[P] = Attributes.value;
        Attributes = {
          configurable: CONFIGURABLE in Attributes ? Attributes[CONFIGURABLE] : current[CONFIGURABLE],
          enumerable: ENUMERABLE in Attributes ? Attributes[ENUMERABLE] : current[ENUMERABLE],
          writable: false
        };
      }
    }
    return $defineProperty(O, P, Attributes);
  } : $defineProperty : function defineProperty(O, P, Attributes) {
    anObject$1(O);
    P = toPropertyKey(P);
    anObject$1(Attributes);
    if (IE8_DOM_DEFINE) try {
      return $defineProperty(O, P, Attributes);
    } catch (error) {/* empty */}
    if ('get' in Attributes || 'set' in Attributes) throw new $TypeError('Accessors not supported');
    if ('value' in Attributes) O[P] = Attributes.value;
    return O;
  };

  var createPropertyDescriptor$1 = function (bitmap, value) {
    return {
      enumerable: !(bitmap & 1),
      configurable: !(bitmap & 2),
      writable: !(bitmap & 4),
      value: value
    };
  };

  var DESCRIPTORS$2 = descriptors;
  var definePropertyModule = objectDefineProperty;
  var createPropertyDescriptor = createPropertyDescriptor$1;
  var createNonEnumerableProperty$1 = DESCRIPTORS$2 ? function (object, key, value) {
    return definePropertyModule.f(object, key, createPropertyDescriptor(1, value));
  } : function (object, key, value) {
    object[key] = value;
    return object;
  };

  var shared$1 = sharedExports;
  var uid = uid$2;
  var keys = shared$1('keys');
  var sharedKey$1 = function (key) {
    return keys[key] || (keys[key] = uid(key));
  };

  var NATIVE_WEAK_MAP = weakMapBasicDetection;
  var global$2 = global$b;
  var isObject = isObject$5;
  var createNonEnumerableProperty = createNonEnumerableProperty$1;
  var hasOwn$1 = hasOwnProperty_1;
  var shared = sharedStore;
  var sharedKey = sharedKey$1;
  var OBJECT_ALREADY_INITIALIZED = 'Object already initialized';
  var TypeError$1 = global$2.TypeError;
  var WeakMap = global$2.WeakMap;
  var set, get, has;
  var enforce = function (it) {
    return has(it) ? get(it) : set(it, {});
  };
  var getterFor = function (TYPE) {
    return function (it) {
      var state;
      if (!isObject(it) || (state = get(it)).type !== TYPE) {
        throw new TypeError$1('Incompatible receiver, ' + TYPE + ' required');
      }
      return state;
    };
  };
  if (NATIVE_WEAK_MAP || shared.state) {
    var store = shared.state || (shared.state = new WeakMap());
    /* eslint-disable no-self-assign -- prototype methods protection */
    store.get = store.get;
    store.has = store.has;
    store.set = store.set;
    /* eslint-enable no-self-assign -- prototype methods protection */
    set = function (it, metadata) {
      if (store.has(it)) throw new TypeError$1(OBJECT_ALREADY_INITIALIZED);
      metadata.facade = it;
      store.set(it, metadata);
      return metadata;
    };
    get = function (it) {
      return store.get(it) || {};
    };
    has = function (it) {
      return store.has(it);
    };
  } else {
    var STATE = sharedKey('state');
    set = function (it, metadata) {
      if (hasOwn$1(it, STATE)) throw new TypeError$1(OBJECT_ALREADY_INITIALIZED);
      metadata.facade = it;
      createNonEnumerableProperty(it, STATE, metadata);
      return metadata;
    };
    get = function (it) {
      return hasOwn$1(it, STATE) ? it[STATE] : {};
    };
    has = function (it) {
      return hasOwn$1(it, STATE);
    };
  }
  var internalState = {
    set: set,
    get: get,
    has: has,
    enforce: enforce,
    getterFor: getterFor
  };

  var uncurryThis = functionUncurryThis;
  var fails$1 = fails$7;
  var isCallable = isCallable$8;
  var hasOwn = hasOwnProperty_1;
  var DESCRIPTORS$1 = descriptors;
  var CONFIGURABLE_FUNCTION_NAME = functionName.CONFIGURABLE;
  var inspectSource = inspectSource$1;
  var InternalStateModule = internalState;
  var enforceInternalState = InternalStateModule.enforce;
  var getInternalState = InternalStateModule.get;
  var $String = String;
  // eslint-disable-next-line es/no-object-defineproperty -- safe
  var defineProperty$1 = Object.defineProperty;
  var stringSlice = uncurryThis(''.slice);
  var replace = uncurryThis(''.replace);
  var join = uncurryThis([].join);
  var CONFIGURABLE_LENGTH = DESCRIPTORS$1 && !fails$1(function () {
    return defineProperty$1(function () {/* empty */}, 'length', {
      value: 8
    }).length !== 8;
  });
  var TEMPLATE = String(String).split('String');
  var makeBuiltIn$1 = makeBuiltIn$2.exports = function (value, name, options) {
    if (stringSlice($String(name), 0, 7) === 'Symbol(') {
      name = '[' + replace($String(name), /^Symbol\(([^)]*)\)/, '$1') + ']';
    }
    if (options && options.getter) name = 'get ' + name;
    if (options && options.setter) name = 'set ' + name;
    if (!hasOwn(value, 'name') || CONFIGURABLE_FUNCTION_NAME && value.name !== name) {
      if (DESCRIPTORS$1) defineProperty$1(value, 'name', {
        value: name,
        configurable: true
      });else value.name = name;
    }
    if (CONFIGURABLE_LENGTH && options && hasOwn(options, 'arity') && value.length !== options.arity) {
      defineProperty$1(value, 'length', {
        value: options.arity
      });
    }
    try {
      if (options && hasOwn(options, 'constructor') && options.constructor) {
        if (DESCRIPTORS$1) defineProperty$1(value, 'prototype', {
          writable: false
        });
        // in V8 ~ Chrome 53, prototypes of some methods, like `Array.prototype.values`, are non-writable
      } else if (value.prototype) value.prototype = undefined;
    } catch (error) {/* empty */}
    var state = enforceInternalState(value);
    if (!hasOwn(state, 'source')) {
      state.source = join(TEMPLATE, typeof name == 'string' ? name : '');
    }
    return value;
  };

  // add fake Function#toString for correct work wrapped methods / constructors with methods like LoDash isNative
  // eslint-disable-next-line no-extend-native -- required
  Function.prototype.toString = makeBuiltIn$1(function toString() {
    return isCallable(this) && getInternalState(this).source || inspectSource(this);
  }, 'toString');

  var makeBuiltIn = makeBuiltInExports;
  var defineProperty = objectDefineProperty;
  var defineBuiltInAccessor$1 = function (target, name, descriptor) {
    if (descriptor.get) makeBuiltIn(descriptor.get, name, {
      getter: true
    });
    if (descriptor.set) makeBuiltIn(descriptor.set, name, {
      setter: true
    });
    return defineProperty.f(target, name, descriptor);
  };

  var anObject = anObject$2;

  // `RegExp.prototype.flags` getter implementation
  // https://tc39.es/ecma262/#sec-get-regexp.prototype.flags
  var regexpFlags = function () {
    var that = anObject(this);
    var result = '';
    if (that.hasIndices) result += 'd';
    if (that.global) result += 'g';
    if (that.ignoreCase) result += 'i';
    if (that.multiline) result += 'm';
    if (that.dotAll) result += 's';
    if (that.unicode) result += 'u';
    if (that.unicodeSets) result += 'v';
    if (that.sticky) result += 'y';
    return result;
  };

  var global$1 = global$b;
  var DESCRIPTORS = descriptors;
  var defineBuiltInAccessor = defineBuiltInAccessor$1;
  var regExpFlags = regexpFlags;
  var fails = fails$7;

  // babel-minify and Closure Compiler transpiles RegExp('.', 'd') -> /./d and it causes SyntaxError
  var RegExp$1 = global$1.RegExp;
  var RegExpPrototype = RegExp$1.prototype;
  var FORCED = DESCRIPTORS && fails(function () {
    var INDICES_SUPPORT = true;
    try {
      RegExp$1('.', 'd');
    } catch (error) {
      INDICES_SUPPORT = false;
    }
    var O = {};
    // modern V8 bug
    var calls = '';
    var expected = INDICES_SUPPORT ? 'dgimsy' : 'gimsy';
    var addGetter = function (key, chr) {
      // eslint-disable-next-line es/no-object-defineproperty -- safe
      Object.defineProperty(O, key, {
        get: function () {
          calls += chr;
          return true;
        }
      });
    };
    var pairs = {
      dotAll: 's',
      global: 'g',
      ignoreCase: 'i',
      multiline: 'm',
      sticky: 'y'
    };
    if (INDICES_SUPPORT) pairs.hasIndices = 'd';
    for (var key in pairs) addGetter(key, pairs[key]);

    // eslint-disable-next-line es/no-object-getownpropertydescriptor -- safe
    var result = Object.getOwnPropertyDescriptor(RegExpPrototype, 'flags').get.call(O);
    return result !== expected || calls !== expected;
  });

  // `RegExp.prototype.flags` getter
  // https://tc39.es/ecma262/#sec-get-regexp.prototype.flags
  if (FORCED) defineBuiltInAccessor(RegExpPrototype, 'flags', {
    configurable: true,
    get: regExpFlags
  });

  // const replacements = {
  //   ASCIIPunctuation: '!"#$%&\'()*+,\\-./:;<=>?@\\[\\]^_`{|}~',
  //   TriggerChars: '`_\*\[\]\(\)',
  //   Scheme: `[A-Za-z][A-Za-z0-9\+\.\-]{1,31}`,
  //   Email: `[a-zA-Z0-9.!#$%&'*+/=?^_\`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*`, // From CommonMark spec

  // }
  const replacements = {
    ASCIIPunctuation: /[!"#$%&'()*+,\-./:;<=>?@[\]^_`{|}~\\]/,
    NotTriggerChar: /[^`_*[\]()<>!~]/,
    Scheme: /[A-Za-z][A-Za-z0-9+.-]{1,31}/,
    Email: /[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*/,
    // From CommonMark spec
    HTMLOpenTag: /<HTMLTagName(?:HTMLAttribute)*\s*\/?>/,
    HTMLCloseTag: /<\/HTMLTagName\s*>/,
    HTMLTagName: /[A-Za-z][A-Za-z0-9-]*/,
    HTMLComment: /<!--(?:[^>-]|(?:[^>-](?:[^-]|-[^-])*[^-]))-->/,
    HTMLPI: /<\?(?:|.|(?:[^?]|\?[^>])*)\?>/,
    HTMLDeclaration: /<![A-Z]+\s[^>]*>/,
    HTMLCDATA: /<!\[CDATA\[.*?\]\]>/,
    HTMLAttribute: /\s+[A-Za-z_:][A-Za-z0-9_.:-]*(?:HTMLAttValue)?/,
    HTMLAttValue: /\s*=\s*(?:(?:'[^']*')|(?:"[^"]*")|(?:[^\s"'=<>`]+))/,
    KnownTag: /address|article|aside|base|basefont|blockquote|body|caption|center|col|colgroup|dd|details|dialog|dir|div|dl|dt|fieldset|figcaption|figure|footer|form|frame|frameset|h1|h2|h3|h4|h5|h6|head|header|hr|html|iframe|legend|li|link|main|menu|menuitem|nav|noframes|ol|optgroup|option|p|param|section|source|summary|table|tbody|td|tfoot|th|thead|title|tr|track|ul/
  };

  // From CommonMark.js. 
  const punctuationLeading = new RegExp(/^(?:[!"#$%&'()*+,\-./:;<=>?@[\]\\^_`{|}~\xA1\xA7\xAB\xB6\xB7\xBB\xBF\u037E\u0387\u055A-\u055F\u0589\u058A\u05BE\u05C0\u05C3\u05C6\u05F3\u05F4\u0609\u060A\u060C\u060D\u061B\u061E\u061F\u066A-\u066D\u06D4\u0700-\u070D\u07F7-\u07F9\u0830-\u083E\u085E\u0964\u0965\u0970\u0AF0\u0DF4\u0E4F\u0E5A\u0E5B\u0F04-\u0F12\u0F14\u0F3A-\u0F3D\u0F85\u0FD0-\u0FD4\u0FD9\u0FDA\u104A-\u104F\u10FB\u1360-\u1368\u1400\u166D\u166E\u169B\u169C\u16EB-\u16ED\u1735\u1736\u17D4-\u17D6\u17D8-\u17DA\u1800-\u180A\u1944\u1945\u1A1E\u1A1F\u1AA0-\u1AA6\u1AA8-\u1AAD\u1B5A-\u1B60\u1BFC-\u1BFF\u1C3B-\u1C3F\u1C7E\u1C7F\u1CC0-\u1CC7\u1CD3\u2010-\u2027\u2030-\u2043\u2045-\u2051\u2053-\u205E\u207D\u207E\u208D\u208E\u2308-\u230B\u2329\u232A\u2768-\u2775\u27C5\u27C6\u27E6-\u27EF\u2983-\u2998\u29D8-\u29DB\u29FC\u29FD\u2CF9-\u2CFC\u2CFE\u2CFF\u2D70\u2E00-\u2E2E\u2E30-\u2E42\u3001-\u3003\u3008-\u3011\u3014-\u301F\u3030\u303D\u30A0\u30FB\uA4FE\uA4FF\uA60D-\uA60F\uA673\uA67E\uA6F2-\uA6F7\uA874-\uA877\uA8CE\uA8CF\uA8F8-\uA8FA\uA8FC\uA92E\uA92F\uA95F\uA9C1-\uA9CD\uA9DE\uA9DF\uAA5C-\uAA5F\uAADE\uAADF\uAAF0\uAAF1\uABEB\uFD3E\uFD3F\uFE10-\uFE19\uFE30-\uFE52\uFE54-\uFE61\uFE63\uFE68\uFE6A\uFE6B\uFF01-\uFF03\uFF05-\uFF0A\uFF0C-\uFF0F\uFF1A\uFF1B\uFF1F\uFF20\uFF3B-\uFF3D\uFF3F\uFF5B\uFF5D\uFF5F-\uFF65]|\uD800[\uDD00-\uDD02\uDF9F\uDFD0]|\uD801\uDD6F|\uD802[\uDC57\uDD1F\uDD3F\uDE50-\uDE58\uDE7F\uDEF0-\uDEF6\uDF39-\uDF3F\uDF99-\uDF9C]|\uD804[\uDC47-\uDC4D\uDCBB\uDCBC\uDCBE-\uDCC1\uDD40-\uDD43\uDD74\uDD75\uDDC5-\uDDC9\uDDCD\uDDDB\uDDDD-\uDDDF\uDE38-\uDE3D\uDEA9]|\uD805[\uDCC6\uDDC1-\uDDD7\uDE41-\uDE43\uDF3C-\uDF3E]|\uD809[\uDC70-\uDC74]|\uD81A[\uDE6E\uDE6F\uDEF5\uDF37-\uDF3B\uDF44]|\uD82F\uDC9F|\uD836[\uDE87-\uDE8B])/);
  const punctuationTrailing = new RegExp(/(?:[!"#$%&'()*+,\-./:;<=>?@[\]\\^_`{|}~\xA1\xA7\xAB\xB6\xB7\xBB\xBF\u037E\u0387\u055A-\u055F\u0589\u058A\u05BE\u05C0\u05C3\u05C6\u05F3\u05F4\u0609\u060A\u060C\u060D\u061B\u061E\u061F\u066A-\u066D\u06D4\u0700-\u070D\u07F7-\u07F9\u0830-\u083E\u085E\u0964\u0965\u0970\u0AF0\u0DF4\u0E4F\u0E5A\u0E5B\u0F04-\u0F12\u0F14\u0F3A-\u0F3D\u0F85\u0FD0-\u0FD4\u0FD9\u0FDA\u104A-\u104F\u10FB\u1360-\u1368\u1400\u166D\u166E\u169B\u169C\u16EB-\u16ED\u1735\u1736\u17D4-\u17D6\u17D8-\u17DA\u1800-\u180A\u1944\u1945\u1A1E\u1A1F\u1AA0-\u1AA6\u1AA8-\u1AAD\u1B5A-\u1B60\u1BFC-\u1BFF\u1C3B-\u1C3F\u1C7E\u1C7F\u1CC0-\u1CC7\u1CD3\u2010-\u2027\u2030-\u2043\u2045-\u2051\u2053-\u205E\u207D\u207E\u208D\u208E\u2308-\u230B\u2329\u232A\u2768-\u2775\u27C5\u27C6\u27E6-\u27EF\u2983-\u2998\u29D8-\u29DB\u29FC\u29FD\u2CF9-\u2CFC\u2CFE\u2CFF\u2D70\u2E00-\u2E2E\u2E30-\u2E42\u3001-\u3003\u3008-\u3011\u3014-\u301F\u3030\u303D\u30A0\u30FB\uA4FE\uA4FF\uA60D-\uA60F\uA673\uA67E\uA6F2-\uA6F7\uA874-\uA877\uA8CE\uA8CF\uA8F8-\uA8FA\uA8FC\uA92E\uA92F\uA95F\uA9C1-\uA9CD\uA9DE\uA9DF\uAA5C-\uAA5F\uAADE\uAADF\uAAF0\uAAF1\uABEB\uFD3E\uFD3F\uFE10-\uFE19\uFE30-\uFE52\uFE54-\uFE61\uFE63\uFE68\uFE6A\uFE6B\uFF01-\uFF03\uFF05-\uFF0A\uFF0C-\uFF0F\uFF1A\uFF1B\uFF1F\uFF20\uFF3B-\uFF3D\uFF3F\uFF5B\uFF5D\uFF5F-\uFF65]|\uD800[\uDD00-\uDD02\uDF9F\uDFD0]|\uD801\uDD6F|\uD802[\uDC57\uDD1F\uDD3F\uDE50-\uDE58\uDE7F\uDEF0-\uDEF6\uDF39-\uDF3F\uDF99-\uDF9C]|\uD804[\uDC47-\uDC4D\uDCBB\uDCBC\uDCBE-\uDCC1\uDD40-\uDD43\uDD74\uDD75\uDDC5-\uDDC9\uDDCD\uDDDB\uDDDD-\uDDDF\uDE38-\uDE3D\uDEA9]|\uD805[\uDCC6\uDDC1-\uDDD7\uDE41-\uDE43\uDF3C-\uDF3E]|\uD809[\uDC70-\uDC74]|\uD81A[\uDE6E\uDE6F\uDEF5\uDF37-\uDF3B\uDF44]|\uD82F\uDC9F|\uD836[\uDE87-\uDE8B])$/);

  // export const inlineTriggerChars = new RegExp(`[${replacements.TriggerChars}]`);

  /**
   * This is CommonMark's block grammar, but we're ignoring nested blocks here.  
   */
  const lineGrammar = {
    TMH1: {
      regexp: /^( {0,3}#\s)(.*?)((?:\s+#+\s*)?)$/,
      replacement: '<span class="TMMark TMMark_TMH1">$1</span>$$2<span class="TMMark TMMark_TMH1">$3</span>'
    },
    TMH2: {
      regexp: /^( {0,3}##\s)(.*?)((?:\s+#+\s*)?)$/,
      replacement: '<span class="TMMark TMMark_TMH2">$1</span>$$2<span class="TMMark TMMark_TMH2">$3</span>'
    },
    TMH3: {
      regexp: /^( {0,3}###\s)(.*?)((?:\s+#+\s*)?)$/,
      replacement: '<span class="TMMark TMMark_TMH3">$1</span>$$2<span class="TMMark TMMark_TMH3">$3</span>'
    },
    TMH4: {
      regexp: /^( {0,3}####\s)(.*?)((?:\s+#+\s*)?)$/,
      replacement: '<span class="TMMark TMMark_TMH4">$1</span>$$2<span class="TMMark TMMark_TMH4">$3</span>'
    },
    TMH5: {
      regexp: /^( {0,3}#####\s)(.*?)((?:\s+#+\s*)?)$/,
      replacement: '<span class="TMMark TMMark_TMH5">$1</span>$$2<span class="TMMark TMMark_TMH5">$3</span>'
    },
    TMH6: {
      regexp: /^( {0,3}######\s)(.*?)((?:\s+#+\s*)?)$/,
      replacement: '<span class="TMMark TMMark_TMH6">$1</span>$$2<span class="TMMark TMMark_TMH6">$3</span>'
    },
    TMBlockquote: {
      regexp: /^( {0,3}>[ ]?)(.*)$/,
      replacement: '<span class="TMMark TMMark_TMBlockquote">$1</span>$$2'
    },
    TMCodeFenceBacktickOpen: {
      regexp: /^( {0,3}(?<seq>````*)\s*)([^`]*?)(\s*)$/,
      replacement: '<span class="TMMark TMMark_TMCodeFenceBacktick">$1</span><span class="TMInfoString">$3</span>$4'
    },
    TMCodeFenceTildeOpen: {
      regexp: /^( {0,3}(?<seq>~~~~*)\s*)(.*?)(\s*)$/,
      replacement: '<span class="TMMark TMMark_TMCodeFenceTilde">$1</span><span class="TMInfoString">$3</span>$4'
    },
    TMCodeFenceBacktickClose: {
      regexp: /^( {0,3}(?<seq>````*))(\s*)$/,
      replacement: '<span class="TMMark TMMark_TMCodeFenceBacktick">$1</span>$3'
    },
    TMCodeFenceTildeClose: {
      regexp: /^( {0,3}(?<seq>~~~~*))(\s*)$/,
      replacement: '<span class="TMMark TMMark_TMCodeFenceTilde">$1</span>$3'
    },
    TMBlankLine: {
      regexp: /^([ \t]*)$/,
      replacement: '$0'
    },
    TMSetextH1Marker: {
      regexp: /^ {0,3}=+\s*$/,
      replacement: '<span class="TMMark TMMark_TMSetextH1Marker">$0</span>'
    },
    TMSetextH2Marker: {
      regexp: /^ {0,3}-+\s*$/,
      replacement: '<span class="TMMark TMMark_TMSetextH1Marker">$0</span>'
    },
    TMHR: {
      regexp: /^( {0,3}(\*[ \t]*\*[ \t]*\*[ \t*]*)|(-[ \t]*-[ \t]*-[ \t-]*)|(_[ \t]*_[ \t]*_[ \t_]*))$/,
      replacement: '<span class="TMMark TMMark_TMHR">$0</span>'
    },
    TMUL: {
      regexp: /^( {0,3}[+*-] {1,4})(.*)$/,
      replacement: '<span class="TMMark TMMark_TMUL">$1</span>$$2'
    },
    TMOL: {
      regexp: /^( {0,3}\d{1,9}[.)] {1,4})(.*)$/,
      replacement: '<span class="TMMark TMMark_TMOL">$1</span>$$2'
    },
    // TODO: This is currently preventing sublists (and any content within list items, really) from working
    TMIndentedCode: {
      regexp: /^( {4}|\t)(.*)$/,
      replacement: '<span class="TMMark TMMark_TMIndentedCode">$1</span>$2'
    },
    TMLinkReferenceDefinition: {
      // TODO: Link destination can't include unbalanced parantheses, but we just ignore that here 
      regexp: /^( {0,3}\[\s*)([^\s\]](?:[^\]]|\\\])*?)(\s*\]:\s*)((?:[^\s<>]+)|(?:<(?:[^<>\\]|\\.)*>))?(\s*)((?:\((?:[^()\\]|\\.)*\))|(?:"(?:[^"\\]|\\.)*")|(?:'(?:[^'\\]|\\.)*'))?(\s*)$/,
      replacement: '<span class="TMMark TMMark_TMLinkReferenceDefinition">$1</span><span class="TMLinkLabel TMLinkLabel_Definition">$2</span><span class="TMMark TMMark_TMLinkReferenceDefinition">$3</span><span class="TMLinkDestination">$4</span>$5<span class="TMLinkTitle">$6</span>$7',
      labelPlaceholder: 2 // this defines which placeholder in the above regex is the link "label"
    }
  };

  /**
   * HTML blocks have multiple different classes of opener and closer. This array defines all the cases
   */
  var htmlBlockGrammar = [{
    start: /^ {0,3}<(?:script|pre|style)(?:\s|>|$)/i,
    end: /(?:<\/script>|<\/pre>|<\/style>)/i,
    paraInterrupt: true
  }, {
    start: /^ {0,3}<!--/,
    end: /-->/,
    paraInterrupt: true
  }, {
    start: /^ {0,3}<\?/,
    end: /\?>/,
    paraInterrupt: true
  }, {
    start: /^ {0,3}<![A-Z]/,
    end: />/,
    paraInterrupt: true
  }, {
    start: /^ {0,3}<!\[CDATA\[/,
    end: /\]\]>/,
    paraInterrupt: true
  }, {
    start: /^ {0,3}(?:<|<\/)(?:KnownTag)(?:\s|>|\/>|$)/i,
    end: false,
    paraInterrupt: true
  }, {
    start: /^ {0,3}(?:HTMLOpenTag|HTMLCloseTag)\s*$/,
    end: false,
    paraInterrupt: false
  }];

  /**
   * Structure of the object:
   * Top level entries are rules, each consisting of a regular expressions (in string format) as well as a replacement.
   * In the regular expressions, replacements from the object 'replacements' will be processed before compiling into the property regexp.
   */
  var inlineGrammar = {
    escape: {
      regexp: /^\\(ASCIIPunctuation)/,
      replacement: '<span class="TMMark TMMark_TMEscape">\\</span>$1'
    },
    code: {
      regexp: /^(`+)((?:[^`])|(?:[^`].*?[^`]))(\1)/,
      replacement: '<span class="TMMark TMMark_TMCode">$1</span><code class="TMCode">$2</code><span class="TMMark TMMark_TMCode">$3</span>'
    },
    autolink: {
      regexp: /^<((?:Scheme:[^\s<>]*)|(?:Email))>/,
      replacement: '<span class="TMMark TMMark_TMAutolink">&lt;</span><span class="TMAutolink">$1</span><span class="TMMark TMMark_TMAutolink">&gt;</span>'
    },
    html: {
      regexp: /^((?:HTMLOpenTag)|(?:HTMLCloseTag)|(?:HTMLComment)|(?:HTMLPI)|(?:HTMLDeclaration)|(?:HTMLCDATA))/,
      replacement: '<span class="TMHTML">$1</span>'
    },
    linkOpen: {
      regexp: /^\[/,
      replacement: ''
    },
    imageOpen: {
      regexp: /^!\[/,
      replacement: ''
    },
    linkLabel: {
      regexp: /^(\[\s*)([^\]]*?)(\s*\])/,
      replacement: '',
      labelPlaceholder: 2
    },
    default: {
      regexp: /^(.|(?:NotTriggerChar+))/,
      replacement: '$1'
    }
  };

  // Process replacements in regexps
  const replacementRegexp = new RegExp(Object.keys(replacements).join('|'));

  // Inline
  const inlineRules = [...Object.keys(inlineGrammar)];
  for (let rule of inlineRules) {
    let re = inlineGrammar[rule].regexp.source;
    // Replace while there is something to replace. This means it also works over multiple levels (replacements containing replacements)
    while (re.match(replacementRegexp)) {
      re = re.replace(replacementRegexp, string => {
        return replacements[string].source;
      });
    }
    inlineGrammar[rule].regexp = new RegExp(re, inlineGrammar[rule].regexp.flags);
  }

  // HTML Block (only opening rule is processed currently)
  for (let rule of htmlBlockGrammar) {
    let re = rule.start.source;
    // Replace while there is something to replace. This means it also works over multiple levels (replacements containing replacements)
    while (re.match(replacementRegexp)) {
      re = re.replace(replacementRegexp, string => {
        return replacements[string].source;
      });
    }
    rule.start = new RegExp(re, rule.start.flags);
  }

  /**
   * Escapes HTML special characters (<, >, and &) in the string.
   * @param {string} string The raw string to be escaped
   * @returns {string} The string, ready to be used in HTML
   */
  function htmlescape(string) {
    return (string ? string : '').replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
  }
  /**
   * Contains the commands that can be sent to the editor. Contains objects with a name representing the name of the command.
   * Each of the objects contains the following keys:
   * 
   *   - type: Can be either inline (for inline formatting) or line (for block / line formatting).
   *   - className: Used to determine whether the command is active at a given position. 
   *     For line formatting, this looks at the class of the line element. For inline elements, tries to find an enclosing element with that class.
   *   - set / unset: Contain instructions how to set and unset the command. For line type commands, both consist of a pattern and replacement that 
   *     will be applied to each line (using String.replace). For inline type commands, the set object contains a pre and post string which will
   *     be inserted before and after the selection. The unset object contains a prePattern and a postPattern. Both should be regular expressions and 
   *     they will be applied to the portion of the line before and after the selection (using String.replace, with an empty replacement string).
   */
  const commands = {
    // Replacements for unset for inline commands are '' by default
    bold: {
      type: 'inline',
      className: 'TMStrong',
      set: {
        pre: '**',
        post: '**'
      },
      unset: {
        prePattern: /(?:\*\*|__)$/,
        postPattern: /^(?:\*\*|__)/
      }
    },
    italic: {
      type: 'inline',
      className: 'TMEm',
      set: {
        pre: '*',
        post: '*'
      },
      unset: {
        prePattern: /(?:\*|_)$/,
        postPattern: /^(?:\*|_)/
      }
    },
    code: {
      type: 'inline',
      className: 'TMCode',
      set: {
        pre: '`',
        post: '`'
      },
      unset: {
        prePattern: /`+$/,
        postPattern: /^`+/
      } // FIXME this doesn't ensure balanced backticks right now
    },

    strikethrough: {
      type: 'inline',
      className: 'TMStrikethrough',
      set: {
        pre: '~~',
        post: '~~'
      },
      unset: {
        prePattern: /~~$/,
        postPattern: /^~~/
      }
    },
    h1: {
      type: 'line',
      className: 'TMH1',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '# $2'
      },
      unset: {
        pattern: /^( {0,3}#\s+)(.*?)((?:\s+#+\s*)?)$/,
        replacement: '$2'
      }
    },
    h2: {
      type: 'line',
      className: 'TMH2',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '## $2'
      },
      unset: {
        pattern: /^( {0,3}##\s+)(.*?)((?:\s+#+\s*)?)$/,
        replacement: '$2'
      }
    },
    h3: {
      type: 'line',
      className: 'TMH3',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '### $2'
      },
      unset: {
        pattern: /^( {0,3}###\s+)(.*?)((?:\s+#+\s*)?)$/,
        replacement: '$2'
      }
    },
    h4: {
      type: 'line',
      className: 'TMH4',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '#### $2'
      },
      unset: {
        pattern: /^( {0,3}####\s+)(.*?)((?:\s+#+\s*)?)$/,
        replacement: '$2'
      }
    },
    h5: {
      type: 'line',
      className: 'TMH5',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '##### $2'
      },
      unset: {
        pattern: /^( {0,3}#####\s+)(.*?)((?:\s+#+\s*)?)$/,
        replacement: '$2'
      }
    },
    h6: {
      type: 'line',
      className: 'TMH6',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '###### $2'
      },
      unset: {
        pattern: /^( {0,3}######\s+)(.*?)((?:\s+#+\s*)?)$/,
        replacement: '$2'
      }
    },
    ul: {
      type: 'line',
      className: 'TMUL',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '- $2'
      },
      unset: {
        pattern: /^( {0,3}[+*-] {1,4})(.*)$/,
        replacement: '$2'
      }
    },
    ol: {
      type: 'line',
      className: 'TMOL',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '$#. $2'
      },
      unset: {
        pattern: /^( {0,3}\d{1,9}[.)] {1,4})(.*)$/,
        replacement: '$2'
      }
    },
    blockquote: {
      type: 'line',
      className: 'TMBlockquote',
      set: {
        pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\-*+])\s+)?)(.*)$/,
        replacement: '> $2'
      },
      unset: {
        pattern: /^( {0,3}>[ ]?)(.*)$/,
        replacement: '$2'
      }
    }
  };

  class Editor {
    constructor() {
      let props = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      this.e = null;
      this.textarea = null;
      this.lines = [];
      this.lineElements = [];
      this.lineTypes = [];
      this.lineCaptures = [];
      this.lineReplacements = [];
      this.linkLabels = [];
      this.lineDirty = [];
      this.lastCommandState = null;
      this.listeners = {
        change: [],
        selection: [],
        drop: []
      };
      let element = props.element;
      this.textarea = props.textarea;
      if (this.textarea) {
        if (!this.textarea.tagName) {
          this.textarea = document.getElementById(this.textarea);
        }
        if (!element) element = this.textarea;
      }
      if (element && !element.tagName) {
        element = document.getElementById(props.element);
      }
      if (!element) {
        element = document.getElementsByTagName("body")[0];
      }
      if (element.tagName == "TEXTAREA") {
        this.textarea = element;
        element = this.textarea.parentNode;
      }
      if (this.textarea) {
        this.textarea.style.display = "none";
      }
      this.createEditorElement(element, props);
      this.setContent(typeof props.content === "string" ? props.content : this.textarea ? this.textarea.value : "# Hello TinyMDE!\nEdit **here**");
    }

    /**
     * Creates the editor element inside the target element of the DOM tree
     * @param element The target element of the DOM tree
     * @param props options, passed from constructor's props
     */
    createEditorElement(element, props) {
      if (props && props.editor !== undefined) {
        if (props.editor.tagName) this.e = props.editor;else this.e = document.getElementById(props.editor);
      } else this.e = document.createElement("div");
      this.e.classList.add("TinyMDE");
      this.e.contentEditable = true;
      // The following is important for formatting purposes, but also since otherwise the browser replaces subsequent spaces with  &nbsp; &nbsp;
      // That breaks a lot of stuff, so we do this here and not in CSS—therefore, you don't have to remember to put this in the CSS file
      this.e.style.whiteSpace = "pre-wrap";
      // Avoid formatting (B / I / U) popping up on iOS
      this.e.style.webkitUserModify = "read-write-plaintext-only";
      if (props.editor === undefined) {
        if (this.textarea && this.textarea.parentNode == element && this.textarea.nextSibling) {
          element.insertBefore(this.e, this.textarea.nextSibling);
        } else {
          element.appendChild(this.e);
        }
      }
      this.e.addEventListener("input", e => this.handleInputEvent(e));
      this.e.addEventListener("compositionend", e => this.handleInputEvent(e));
      document.addEventListener("selectionchange", e => this.handleSelectionChangeEvent(e));
      this.e.addEventListener("paste", e => this.handlePaste(e));
      this.e.addEventListener("drop", e => this.handleDrop(e));
      this.lineElements = this.e.childNodes; // this will automatically update
    }

    /**
     * Sets the editor content.
     * @param {string} content The new Markdown content
     */
    setContent(content) {
      // Delete any existing content
      while (this.e.firstChild) {
        this.e.removeChild(this.e.firstChild);
      }
      this.lines = content.split(/(?:\r\n|\r|\n)/);
      this.lineDirty = [];
      for (let lineNum = 0; lineNum < this.lines.length; lineNum++) {
        let le = document.createElement("div");
        this.e.appendChild(le);
        this.lineDirty.push(true);
      }
      this.lineTypes = new Array(this.lines.length);
      this.updateFormatting();
      this.fireChange();
    }

    /**
     * Gets the editor content as a Markdown string.
     * @returns {string} The editor content as a markdown string
     */
    getContent() {
      return this.lines.join("\n");
    }

    /**
     * This is the main method to update the formatting (from this.lines to HTML output)
     */
    updateFormatting() {
      // First, parse line types. This will update this.lineTypes, this.lineReplacements, and this.lineCaptures
      // We don't apply the formatting yet
      this.updateLineTypes();
      // Collect any valid link labels from link reference definitions—we need that for formatting to determine what's a valid link
      this.updateLinkLabels();
      // Now, apply the formatting
      this.applyLineTypes();
    }

    /**
     * Updates this.linkLabels: For every link reference definition (line type TMLinkReferenceDefinition), we collect the label
     */
    updateLinkLabels() {
      this.linkLabels = [];
      for (let l = 0; l < this.lines.length; l++) {
        if (this.lineTypes[l] == "TMLinkReferenceDefinition") {
          this.linkLabels.push(this.lineCaptures[l][lineGrammar.TMLinkReferenceDefinition.labelPlaceholder]);
        }
      }
    }

    /**
     * Helper function to replace placeholders from a RegExp capture. The replacement string can contain regular dollar placeholders (e.g., $1),
     * which are interpreted like in String.replace(), but also double dollar placeholders ($$1). In the case of double dollar placeholders,
     * Markdown inline grammar is applied on the content of the captured subgroup, i.e., $$1 processes inline Markdown grammar in the content of the
     * first captured subgroup, and replaces `$$1` with the result.
     *
     * @param {string} replacement The replacement string, including placeholders.
     * @param  capture The result of a RegExp.exec() call
     * @returns The replacement string, with placeholders replaced from the capture result.
     */
    replace(replacement, capture) {
      return replacement.replace(/(\${1,2})([0-9])/g, (str, p1, p2) => {
        if (p1 == "$") return htmlescape(capture[p2]);else return `<span class="TMInlineFormatted">${this.processInlineStyles(capture[p2])}</span>`;
      });
    }

    /**
     * Applies the line types (from this.lineTypes as well as the capture result in this.lineReplacements and this.lineCaptures)
     * and processes inline formatting for all lines.
     */
    applyLineTypes() {
      for (let lineNum = 0; lineNum < this.lines.length; lineNum++) {
        if (this.lineDirty[lineNum]) {
          let contentHTML = this.replace(this.lineReplacements[lineNum], this.lineCaptures[lineNum]);
          // this.lineHTML[lineNum] = (contentHTML == '' ? '<br />' : contentHTML); // Prevent empty elements which can't be selected etc.
          this.lineElements[lineNum].className = this.lineTypes[lineNum];
          this.lineElements[lineNum].removeAttribute("style");
          this.lineElements[lineNum].innerHTML = contentHTML == "" ? "<br />" : contentHTML; // Prevent empty elements which can't be selected etc.
        }

        this.lineElements[lineNum].dataset.lineNum = lineNum;
      }
    }

    /**
     * Determines line types for all lines based on the line / block grammar. Captures the results of the respective line
     * grammar regular expressions.
     * Updates this.lineTypes, this.lineCaptures, and this.lineReplacements, as well as this.lineDirty.
     */
    updateLineTypes() {
      let codeBlockType = false;
      let codeBlockSeqLength = 0;
      let htmlBlock = false;
      for (let lineNum = 0; lineNum < this.lines.length; lineNum++) {
        let lineType = "TMPara";
        let lineCapture = [this.lines[lineNum]];
        let lineReplacement = "$$0"; // Default replacement for paragraph: Inline format the entire line

        // Check ongoing code blocks
        // if (lineNum > 0 && (this.lineTypes[lineNum - 1] == 'TMCodeFenceBacktickOpen' || this.lineTypes[lineNum - 1] == 'TMFencedCodeBacktick')) {
        if (codeBlockType == "TMCodeFenceBacktickOpen") {
          // We're in a backtick-fenced code block, check if the current line closes it
          let capture = lineGrammar.TMCodeFenceBacktickClose.regexp.exec(this.lines[lineNum]);
          if (capture && capture.groups["seq"].length >= codeBlockSeqLength) {
            lineType = "TMCodeFenceBacktickClose";
            lineReplacement = lineGrammar.TMCodeFenceBacktickClose.replacement;
            lineCapture = capture;
            codeBlockType = false;
          } else {
            lineType = "TMFencedCodeBacktick";
            lineReplacement = '<span class="TMFencedCode">$0<br /></span>';
            lineCapture = [this.lines[lineNum]];
          }
        }
        // if (lineNum > 0 && (this.lineTypes[lineNum - 1] == 'TMCodeFenceTildeOpen' || this.lineTypes[lineNum - 1] == 'TMFencedCodeTilde')) {
        else if (codeBlockType == "TMCodeFenceTildeOpen") {
          // We're in a tilde-fenced code block
          let capture = lineGrammar.TMCodeFenceTildeClose.regexp.exec(this.lines[lineNum]);
          if (capture && capture.groups["seq"].length >= codeBlockSeqLength) {
            lineType = "TMCodeFenceTildeClose";
            lineReplacement = lineGrammar.TMCodeFenceTildeClose.replacement;
            lineCapture = capture;
            codeBlockType = false;
          } else {
            lineType = "TMFencedCodeTilde";
            lineReplacement = '<span class="TMFencedCode">$0<br /></span>';
            lineCapture = [this.lines[lineNum]];
          }
        }

        // Check HTML block types
        if (lineType == "TMPara" && htmlBlock === false) {
          for (let htmlBlockType of htmlBlockGrammar) {
            if (this.lines[lineNum].match(htmlBlockType.start)) {
              // Matching start condition. Check if this tag can start here (not all start conditions allow breaking a paragraph).
              if (htmlBlockType.paraInterrupt || lineNum == 0 || !(this.lineTypes[lineNum - 1] == "TMPara" || this.lineTypes[lineNum - 1] == "TMUL" || this.lineTypes[lineNum - 1] == "TMOL" || this.lineTypes[lineNum - 1] == "TMBlockquote")) {
                htmlBlock = htmlBlockType;
                break;
              }
            }
          }
        }
        if (htmlBlock !== false) {
          lineType = "TMHTMLBlock";
          lineReplacement = '<span class="TMHTMLContent">$0<br /></span>'; // No formatting in TMHTMLBlock
          lineCapture = [this.lines[lineNum]]; // This should already be set but better safe than sorry

          // Check if HTML block should be closed
          if (htmlBlock.end) {
            // Specific end condition
            if (this.lines[lineNum].match(htmlBlock.end)) {
              htmlBlock = false;
            }
          } else {
            // No specific end condition, ends with blank line
            if (lineNum == this.lines.length - 1 || this.lines[lineNum + 1].match(lineGrammar.TMBlankLine.regexp)) {
              htmlBlock = false;
            }
          }
        }

        // Check all regexps if we haven't applied one of the code block types
        if (lineType == "TMPara") {
          for (let type in lineGrammar) {
            if (lineGrammar[type].regexp) {
              let capture = lineGrammar[type].regexp.exec(this.lines[lineNum]);
              if (capture) {
                lineType = type;
                lineReplacement = lineGrammar[type].replacement;
                lineCapture = capture;
                break;
              }
            }
          }
        }

        // If we've opened a code block, remember that
        if (lineType == "TMCodeFenceBacktickOpen" || lineType == "TMCodeFenceTildeOpen") {
          codeBlockType = lineType;
          codeBlockSeqLength = lineCapture.groups["seq"].length;
        }

        // Link reference definition and indented code can't interrupt a paragraph
        if ((lineType == "TMIndentedCode" || lineType == "TMLinkReferenceDefinition") && lineNum > 0 && (this.lineTypes[lineNum - 1] == "TMPara" || this.lineTypes[lineNum - 1] == "TMUL" || this.lineTypes[lineNum - 1] == "TMOL" || this.lineTypes[lineNum - 1] == "TMBlockquote")) {
          // Fall back to TMPara
          lineType = "TMPara";
          lineCapture = [this.lines[lineNum]];
          lineReplacement = "$$0";
        }

        // Setext H2 markers that can also be interpreted as an empty list item should be regarded as such (as per CommonMark spec)
        if (lineType == "TMSetextH2Marker") {
          let capture = lineGrammar.TMUL.regexp.exec(this.lines[lineNum]);
          if (capture) {
            lineType = "TMUL";
            lineReplacement = lineGrammar.TMUL.replacement;
            lineCapture = capture;
          }
        }

        // Setext headings are only valid if preceded by a paragraph (and if so, they change the type of the previous paragraph)
        if (lineType == "TMSetextH1Marker" || lineType == "TMSetextH2Marker") {
          if (lineNum == 0 || this.lineTypes[lineNum - 1] != "TMPara") {
            // Setext marker is invalid. However, a H2 marker might still be a valid HR, so let's check that
            let capture = lineGrammar.TMHR.regexp.exec(this.lines[lineNum]);
            if (capture) {
              // Valid HR
              lineType = "TMHR";
              lineCapture = capture;
              lineReplacement = lineGrammar.TMHR.replacement;
            } else {
              // Not valid HR, format as TMPara
              lineType = "TMPara";
              lineCapture = [this.lines[lineNum]];
              lineReplacement = "$$0";
            }
          } else {
            // Valid setext marker. Change types of preceding para lines
            let headingLine = lineNum - 1;
            const headingLineType = lineType == "TMSetextH1Marker" ? "TMSetextH1" : "TMSetextH2";
            do {
              if (this.lineTypes[headingLineType] != headingLineType) {
                this.lineTypes[headingLine] = headingLineType;
                this.lineDirty[headingLineType] = true;
              }
              this.lineReplacements[headingLine] = "$$0";
              this.lineCaptures[headingLine] = [this.lines[headingLine]];
              headingLine--;
            } while (headingLine >= 0 && this.lineTypes[headingLine] == "TMPara");
          }
        }
        // Lastly, save the line style to be applied later
        if (this.lineTypes[lineNum] != lineType) {
          this.lineTypes[lineNum] = lineType;
          this.lineDirty[lineNum] = true;
        }
        this.lineReplacements[lineNum] = lineReplacement;
        this.lineCaptures[lineNum] = lineCapture;
      }
    }

    /**
     * Updates all line contents from the HTML, then re-applies formatting.
     */
    updateLineContentsAndFormatting() {
      this.clearDirtyFlag();
      this.updateLineContents();
      this.updateFormatting();
    }

    /**
     * Attempts to parse a link or image at the current position. This assumes that the opening [ or ![ has already been matched.
     * Returns false if this is not a valid link, image. See below for more information
     * @param {string} originalString The original string, starting at the opening marker ([ or ![)
     * @param {boolean} isImage Whether or not this is an image (opener == ![)
     * @returns false if not a valid link / image.
     * Otherwise returns an object with two properties: output is the string to be included in the processed output,
     * charCount is the number of input characters (from originalString) consumed.
     */
    parseLinkOrImage(originalString, isImage) {
      // Skip the opening bracket
      let textOffset = isImage ? 2 : 1;
      let opener = originalString.substr(0, textOffset);
      let type = isImage ? "TMImage" : "TMLink";
      let currentOffset = textOffset;
      let bracketLevel = 1;
      let linkText = false;
      let linkRef = false;
      let linkLabel = [];
      let linkDetails = []; // If matched, this will be an array: [whitespace + link destination delimiter, link destination, link destination delimiter, whitespace, link title delimiter, link title, link title delimiter + whitespace]. All can be empty strings.

      textOuter: while (currentOffset < originalString.length && linkText === false /* empty string is okay */) {
        let string = originalString.substr(currentOffset);

        // Capture any escapes and code blocks at current position, they bind more strongly than links
        // We don't have to actually process them here, that'll be done later in case the link / image is valid, but we need to skip over them.
        for (let rule of ["escape", "code", "autolink", "html"]) {
          let cap = inlineGrammar[rule].regexp.exec(string);
          if (cap) {
            currentOffset += cap[0].length;
            continue textOuter;
          }
        }

        // Check for image. It's okay for an image to be included in a link or image
        if (string.match(inlineGrammar.imageOpen.regexp)) {
          // Opening image. It's okay if this is a matching pair of brackets
          bracketLevel++;
          currentOffset += 2;
          continue textOuter;
        }

        // Check for link (not an image because that would have been captured and skipped over above)
        if (string.match(inlineGrammar.linkOpen.regexp)) {
          // Opening bracket. Two things to do:
          // 1) it's okay if this part of a pair of brackets.
          // 2) If we are currently trying to parse a link, this nested bracket musn't start a valid link (no nested links allowed)
          bracketLevel++;
          // if (bracketLevel >= 2) return false; // Nested unescaped brackets, this doesn't qualify as a link / image
          if (!isImage) {
            if (this.parseLinkOrImage(string, false)) {
              // Valid link inside this possible link, which makes this link invalid (inner links beat outer ones)
              return false;
            }
          }
          currentOffset += 1;
          continue textOuter;
        }

        // Check for closing bracket
        if (string.match(/^\]/)) {
          bracketLevel--;
          if (bracketLevel == 0) {
            // Found matching bracket and haven't found anything disqualifying this as link / image.
            linkText = originalString.substr(textOffset, currentOffset - textOffset);
            currentOffset++;
            continue textOuter;
          }
        }

        // Nothing matches, proceed to next char
        currentOffset++;
      }

      // Did we find a link text (i.e., find a matching closing bracket?)
      if (linkText === false) return false; // Nope

      // So far, so good. We've got a valid link text. Let's see what type of link this is
      let nextChar = currentOffset < originalString.length ? originalString.substr(currentOffset, 1) : "";

      // REFERENCE LINKS
      if (nextChar == "[") {
        let string = originalString.substr(currentOffset);
        let cap = inlineGrammar.linkLabel.regexp.exec(string);
        if (cap) {
          // Valid link label
          currentOffset += cap[0].length;
          linkLabel.push(cap[1], cap[2], cap[3]);
          if (cap[inlineGrammar.linkLabel.labelPlaceholder]) {
            // Full reference link
            linkRef = cap[inlineGrammar.linkLabel.labelPlaceholder];
          } else {
            // Collapsed reference link
            linkRef = linkText.trim();
          }
        } else {
          // Not a valid link label
          return false;
        }
      } else if (nextChar != "(") {
        // Shortcut ref link
        linkRef = linkText.trim();

        // INLINE LINKS
      } else {
        // nextChar == '('

        // Potential inline link
        currentOffset++;
        let parenthesisLevel = 1;
        inlineOuter: while (currentOffset < originalString.length && parenthesisLevel > 0) {
          let string = originalString.substr(currentOffset);

          // Process whitespace
          let cap = /^\s+/.exec(string);
          if (cap) {
            switch (linkDetails.length) {
              case 0:
                linkDetails.push(cap[0]);
                break;
              // Opening whitespace
              case 1:
                linkDetails.push(cap[0]);
                break;
              // Open destination, but not a destination yet; desination opened with <
              case 2:
                // Open destination with content in it. Whitespace only allowed if opened by angle bracket, otherwise this closes the destination
                if (linkDetails[0].match(/</)) {
                  linkDetails[1] = linkDetails[1].concat(cap[0]);
                } else {
                  if (parenthesisLevel != 1) return false; // Unbalanced parenthesis
                  linkDetails.push(""); // Empty end delimiter for destination
                  linkDetails.push(cap[0]); // Whitespace in between destination and title
                }

                break;
              case 3:
                linkDetails.push(cap[0]);
                break;
              // Whitespace between destination and title
              case 4:
                return false;
              // This should never happen (no opener for title yet, but more whitespace to capture)
              case 5:
                linkDetails.push("");
              // Whitespace at beginning of title, push empty title and continue
              case 6:
                linkDetails[5] = linkDetails[5].concat(cap[0]);
                break;
              // Whitespace in title
              case 7:
                linkDetails[6] = linkDetails[6].concat(cap[0]);
                break;
              // Whitespace after closing delimiter
              default:
                return false;
              // We should never get here
            }

            currentOffset += cap[0].length;
            continue inlineOuter;
          }

          // Process backslash escapes
          cap = inlineGrammar.escape.regexp.exec(string);
          if (cap) {
            switch (linkDetails.length) {
              case 0:
                linkDetails.push("");
              // this opens the link destination, add empty opening delimiter and proceed to next case
              case 1:
                linkDetails.push(cap[0]);
                break;
              // This opens the link destination, append it
              case 2:
                linkDetails[1] = linkDetails[1].concat(cap[0]);
                break;
              // Part of the link destination
              case 3:
                return false;
              // Lacking opening delimiter for link title
              case 4:
                return false;
              // Lcaking opening delimiter for link title
              case 5:
                linkDetails.push("");
              // This opens the link title
              case 6:
                linkDetails[5] = linkDetails[5].concat(cap[0]);
                break;
              // Part of the link title
              default:
                return false;
              // After link title was closed, without closing parenthesis
            }

            currentOffset += cap[0].length;
            continue inlineOuter;
          }

          // Process opening angle bracket as deilimiter of destination
          if (linkDetails.length < 2 && string.match(/^</)) {
            if (linkDetails.length == 0) linkDetails.push("");
            linkDetails[0] = linkDetails[0].concat("<");
            currentOffset++;
            continue inlineOuter;
          }

          // Process closing angle bracket as delimiter of destination
          if ((linkDetails.length == 1 || linkDetails.length == 2) && string.match(/^>/)) {
            if (linkDetails.length == 1) linkDetails.push(""); // Empty link destination
            linkDetails.push(">");
            currentOffset++;
            continue inlineOuter;
          }

          // Process  non-parenthesis delimiter for title.
          cap = /^["']/.exec(string);
          // For this to be a valid opener, we have to either have no destination, only whitespace so far,
          // or a destination with trailing whitespace.
          if (cap && (linkDetails.length == 0 || linkDetails.length == 1 || linkDetails.length == 4)) {
            while (linkDetails.length < 4) linkDetails.push("");
            linkDetails.push(cap[0]);
            currentOffset++;
            continue inlineOuter;
          }

          // For this to be a valid closer, we have to have an opener and some or no title, and this has to match the opener
          if (cap && (linkDetails.length == 5 || linkDetails.length == 6) && linkDetails[4] == cap[0]) {
            if (linkDetails.length == 5) linkDetails.push(""); // Empty link title
            linkDetails.push(cap[0]);
            currentOffset++;
            continue inlineOuter;
          }
          // Other cases (linkDetails.length == 2, 3, 7) will be handled with the "default" case below.

          // Process opening parenthesis
          if (string.match(/^\(/)) {
            switch (linkDetails.length) {
              case 0:
                linkDetails.push("");
              // this opens the link destination, add empty opening delimiter and proceed to next case
              case 1:
                linkDetails.push("");
              // This opens the link destination
              case 2:
                // Part of the link destination
                linkDetails[1] = linkDetails[1].concat("(");
                if (!linkDetails[0].match(/<$/)) parenthesisLevel++;
                break;
              case 3:
                linkDetails.push("");
              //  opening delimiter for link title
              case 4:
                linkDetails.push("(");
                break;
              // opening delimiter for link title
              case 5:
                linkDetails.push("");
              // opens the link title, add empty title content and proceed to next case
              case 6:
                // Part of the link title. Un-escaped parenthesis only allowed in " or ' delimited title
                if (linkDetails[4] == "(") return false;
                linkDetails[5] = linkDetails[5].concat("(");
                break;
              default:
                return false;
              // After link title was closed, without closing parenthesis
            }

            currentOffset++;
            continue inlineOuter;
          }

          // Process closing parenthesis
          if (string.match(/^\)/)) {
            if (linkDetails.length <= 2) {
              // We are inside the link destination. Parentheses have to be matched if not in angle brackets
              while (linkDetails.length < 2) linkDetails.push("");
              if (!linkDetails[0].match(/<$/)) parenthesisLevel--;
              if (parenthesisLevel > 0) {
                linkDetails[1] = linkDetails[1].concat(")");
              }
            } else if (linkDetails.length == 5 || linkDetails.length == 6) {
              // We are inside the link title.
              if (linkDetails[4] == "(") {
                // This closes the link title
                if (linkDetails.length == 5) linkDetails.push("");
                linkDetails.push(")");
              } else {
                // Just regular ol' content
                if (linkDetails.length == 5) linkDetails.push(")");else linkDetails[5] = linkDetails[5].concat(")");
              }
            } else {
              parenthesisLevel--; // This should decrease it from 1 to 0...
            }

            if (parenthesisLevel == 0) {
              // No invalid condition, let's make sure the linkDetails array is complete
              while (linkDetails.length < 7) linkDetails.push("");
            }
            currentOffset++;
            continue inlineOuter;
          }

          // Any old character
          cap = /^./.exec(string);
          if (cap) {
            switch (linkDetails.length) {
              case 0:
                linkDetails.push("");
              // this opens the link destination, add empty opening delimiter and proceed to next case
              case 1:
                linkDetails.push(cap[0]);
                break;
              // This opens the link destination, append it
              case 2:
                linkDetails[1] = linkDetails[1].concat(cap[0]);
                break;
              // Part of the link destination
              case 3:
                return false;
              // Lacking opening delimiter for link title
              case 4:
                return false;
              // Lcaking opening delimiter for link title
              case 5:
                linkDetails.push("");
              // This opens the link title
              case 6:
                linkDetails[5] = linkDetails[5].concat(cap[0]);
                break;
              // Part of the link title
              default:
                return false;
              // After link title was closed, without closing parenthesis
            }

            currentOffset += cap[0].length;
            continue inlineOuter;
          }
          throw "Infinite loop"; // we should never get here since the last test matches any character
        }

        if (parenthesisLevel > 0) return false; // Parenthes(es) not closed
      }

      if (linkRef !== false) {
        // Ref link; check that linkRef is valid
        let valid = false;
        for (let label of this.linkLabels) {
          if (label == linkRef) {
            valid = true;
            break;
          }
        }
        let label = valid ? "TMLinkLabel TMLinkLabel_Valid" : "TMLinkLabel TMLinkLabel_Invalid";
        let output = `<span class="TMMark TMMark_${type}">${opener}</span><span class="${type} ${linkLabel.length < 3 || !linkLabel[1] ? label : ""}">${this.processInlineStyles(linkText)}</span><span class="TMMark TMMark_${type}">]</span>`;
        if (linkLabel.length >= 3) {
          output = output.concat(`<span class="TMMark TMMark_${type}">${linkLabel[0]}</span>`, `<span class="${label}">${linkLabel[1]}</span>`, `<span class="TMMark TMMark_${type}">${linkLabel[2]}</span>`);
        }
        return {
          output: output,
          charCount: currentOffset
        };
      } else if (linkDetails) {
        // Inline link

        // This should never happen, but better safe than sorry.
        while (linkDetails.length < 7) {
          linkDetails.push("");
        }
        return {
          output: `<span class="TMMark TMMark_${type}">${opener}</span><span class="${type}">${this.processInlineStyles(linkText)}</span><span class="TMMark TMMark_${type}">](${linkDetails[0]}</span><span class="${type}Destination">${linkDetails[1]}</span><span class="TMMark TMMark_${type}">${linkDetails[2]}${linkDetails[3]}${linkDetails[4]}</span><span class="${type}Title">${linkDetails[5]}</span><span class="TMMark TMMark_${type}">${linkDetails[6]})</span>`,
          charCount: currentOffset
        };
      }
      return false;
    }

    /**
     * Formats a markdown string as HTML, using Markdown inline formatting.
     * @param {string} originalString The input (markdown inline formatted) string
     * @returns {string} The HTML formatted output
     */
    processInlineStyles(originalString) {
      let processed = "";
      let stack = []; // Stack is an array of objects of the format: {delimiter, delimString, count, output}
      let offset = 0;
      let string = originalString;
      outer: while (string) {
        // Process simple rules (non-delimiter)
        for (let rule of ["escape", "code", "autolink", "html"]) {
          let cap = inlineGrammar[rule].regexp.exec(string);
          if (cap) {
            string = string.substr(cap[0].length);
            offset += cap[0].length;
            processed += inlineGrammar[rule].replacement
            // .replace(/\$\$([1-9])/g, (str, p1) => processInlineStyles(cap[p1])) // todo recursive calling
            .replace(/\$([1-9])/g, (str, p1) => htmlescape(cap[p1]));
            continue outer;
          }
        }

        // Check for links / images
        let potentialLink = string.match(inlineGrammar.linkOpen.regexp);
        let potentialImage = string.match(inlineGrammar.imageOpen.regexp);
        if (potentialImage || potentialLink) {
          let result = this.parseLinkOrImage(string, potentialImage);
          if (result) {
            processed = `${processed}${result.output}`;
            string = string.substr(result.charCount);
            offset += result.charCount;
            continue outer;
          }
        }

        // Check for em / strong delimiters
        let cap = /(^\*+)|(^_+)/.exec(string);
        if (cap) {
          let delimCount = cap[0].length;
          const delimString = cap[0];
          const currentDelimiter = cap[0][0]; // This should be * or _

          string = string.substr(cap[0].length);

          // We have a delimiter run. Let's check if it can open or close an emphasis.

          const preceding = offset > 0 ? originalString.substr(0, offset) : " "; // beginning and end of line count as whitespace
          const following = offset + cap[0].length < originalString.length ? string : " ";
          const punctuationFollows = following.match(punctuationLeading);
          const punctuationPrecedes = preceding.match(punctuationTrailing);
          const whitespaceFollows = following.match(/^\s/);
          const whitespacePrecedes = preceding.match(/\s$/);

          // These are the rules for right-flanking and left-flanking delimiter runs as per CommonMark spec
          let canOpen = !whitespaceFollows && (!punctuationFollows || !!whitespacePrecedes || !!punctuationPrecedes);
          let canClose = !whitespacePrecedes && (!punctuationPrecedes || !!whitespaceFollows || !!punctuationFollows);

          // Underscores have more detailed rules than just being part of left- or right-flanking run:
          if (currentDelimiter == "_" && canOpen && canClose) {
            canOpen = punctuationPrecedes;
            canClose = punctuationFollows;
          }

          // If the delimiter can close, check the stack if there's something it can close
          if (canClose) {
            let stackPointer = stack.length - 1;
            // See if we can find a matching opening delimiter, move down through the stack
            while (delimCount && stackPointer >= 0) {
              if (stack[stackPointer].delimiter == currentDelimiter) {
                // We found a matching delimiter, let's construct the formatted string

                // Firstly, if we skipped any stack levels, pop them immediately (non-matching delimiters)
                while (stackPointer < stack.length - 1) {
                  const entry = stack.pop();
                  processed = `${entry.output}${entry.delimString.substr(0, entry.count)}${processed}`;
                }

                // Then, format the string
                if (delimCount >= 2 && stack[stackPointer].count >= 2) {
                  // Strong
                  processed = `<span class="TMMark">${currentDelimiter}${currentDelimiter}</span><strong class="TMStrong">${processed}</strong><span class="TMMark">${currentDelimiter}${currentDelimiter}</span>`;
                  delimCount -= 2;
                  stack[stackPointer].count -= 2;
                } else {
                  // Em
                  processed = `<span class="TMMark">${currentDelimiter}</span><em class="TMEm">${processed}</em><span class="TMMark">${currentDelimiter}</span>`;
                  delimCount -= 1;
                  stack[stackPointer].count -= 1;
                }

                // If that stack level is empty now, pop it
                if (stack[stackPointer].count == 0) {
                  let entry = stack.pop();
                  processed = `${entry.output}${processed}`;
                  stackPointer--;
                }
              } else {
                // This stack level's delimiter type doesn't match the current delimiter type
                // Go down one level in the stack
                stackPointer--;
              }
            }
          }
          // If there are still delimiters left, and the delimiter run can open, push it on the stack
          if (delimCount && canOpen) {
            stack.push({
              delimiter: currentDelimiter,
              delimString: delimString,
              count: delimCount,
              output: processed
            });
            processed = ""; // Current formatted output has been pushed on the stack and will be prepended when the stack gets popped
            delimCount = 0;
          }

          // Any delimiters that are left (closing unmatched) are appended to the output.
          if (delimCount) {
            processed = `${processed}${delimString.substr(0, delimCount)}`;
          }
          offset += cap[0].length;
          continue outer;
        }

        // Check for strikethrough delimiter
        cap = /^~~/.exec(string);
        if (cap) {
          let consumed = false;
          let stackPointer = stack.length - 1;
          // See if we can find a matching opening delimiter, move down through the stack
          while (!consumed && stackPointer >= 0) {
            if (stack[stackPointer].delimiter == "~") {
              // We found a matching delimiter, let's construct the formatted string

              // Firstly, if we skipped any stack levels, pop them immediately (non-matching delimiters)
              while (stackPointer < stack.length - 1) {
                const entry = stack.pop();
                processed = `${entry.output}${entry.delimString.substr(0, entry.count)}${processed}`;
              }

              // Then, format the string
              processed = `<span class="TMMark">~~</span><del class="TMStrikethrough">${processed}</del><span class="TMMark">~~</span>`;
              let entry = stack.pop();
              processed = `${entry.output}${processed}`;
              consumed = true;
            } else {
              // This stack level's delimiter type doesn't match the current delimiter type
              // Go down one level in the stack
              stackPointer--;
            }
          }

          // If there are still delimiters left, and the delimiter run can open, push it on the stack
          if (!consumed) {
            stack.push({
              delimiter: "~",
              delimString: "~~",
              count: 2,
              output: processed
            });
            processed = ""; // Current formatted output has been pushed on the stack and will be prepended when the stack gets popped
          }

          offset += cap[0].length;
          string = string.substr(cap[0].length);
          continue outer;
        }

        // Process 'default' rule
        cap = inlineGrammar.default.regexp.exec(string);
        if (cap) {
          string = string.substr(cap[0].length);
          offset += cap[0].length;
          processed += inlineGrammar.default.replacement.replace(/\$([1-9])/g, (str, p1) => htmlescape(cap[p1]));
          continue outer;
        }
        throw "Infinite loop!";
      }

      // Empty the stack, any opening delimiters are unused
      while (stack.length) {
        const entry = stack.pop();
        processed = `${entry.output}${entry.delimString.substr(0, entry.count)}${processed}`;
      }
      return processed;
    }

    /**
     * Clears the line dirty flag (resets it to an array of false)
     */
    clearDirtyFlag() {
      this.lineDirty = new Array(this.lines.length);
      for (let i = 0; i < this.lineDirty.length; i++) {
        this.lineDirty[i] = false;
      }
    }

    /**
     * Updates the class properties (lines, lineElements) from the DOM.
     * @returns true if contents changed
     */
    updateLineContents() {
      // this.lineDirty = [];
      // Check if we have changed anything about the number of lines (inserted or deleted a paragraph)
      // < 0 means line(s) removed; > 0 means line(s) added
      let lineDelta = this.e.childElementCount - this.lines.length;
      if (lineDelta) {
        // yup. Let's try how much we can salvage (find out which lines from beginning and end were unchanged)
        // Find lines from the beginning that haven't changed...
        let firstChangedLine = 0;
        while (firstChangedLine <= this.lines.length && firstChangedLine <= this.lineElements.length && this.lineElements[firstChangedLine] &&
        // Check that the line element hasn't been deleted
        this.lines[firstChangedLine] == this.lineElements[firstChangedLine].textContent && this.lineTypes[firstChangedLine] == this.lineElements[firstChangedLine].className) {
          firstChangedLine++;
        }

        // End also from the end
        let lastChangedLine = -1;
        while (-lastChangedLine < this.lines.length && -lastChangedLine < this.lineElements.length && this.lines[this.lines.length + lastChangedLine] == this.lineElements[this.lineElements.length + lastChangedLine].textContent && this.lineTypes[this.lines.length + lastChangedLine] == this.lineElements[this.lineElements.length + lastChangedLine].className) {
          lastChangedLine--;
        }
        let linesToDelete = this.lines.length + lastChangedLine + 1 - firstChangedLine;
        if (linesToDelete < -lineDelta) linesToDelete = -lineDelta;
        if (linesToDelete < 0) linesToDelete = 0;
        let linesToAdd = [];
        for (let l = 0; l < linesToDelete + lineDelta; l++) {
          linesToAdd.push(this.lineElements[firstChangedLine + l].textContent);
        }
        this.spliceLines(firstChangedLine, linesToDelete, linesToAdd, false);
      } else {
        // No lines added or removed
        for (let line = 0; line < this.lineElements.length; line++) {
          let e = this.lineElements[line];
          let ct = e.textContent;
          if (this.lines[line] !== ct || this.lineTypes[line] !== e.className) {
            // Line changed, update it
            this.lines[line] = ct;
            this.lineTypes[line] = e.className;
            this.lineDirty[line] = true;
          }
        }
      }
    }

    /**
     * Processes a new paragraph.
     * @param sel The current selection
     */
    processNewParagraph(sel) {
      if (!sel) return;

      // Update lines from content
      this.updateLineContents();
      let continuableType = false;
      // Let's see if we need to continue a list

      let checkLine = sel.col > 0 ? sel.row : sel.row - 1;
      switch (this.lineTypes[checkLine]) {
        case "TMUL":
          continuableType = "TMUL";
          break;
        case "TMOL":
          continuableType = "TMOL";
          break;
        case "TMIndentedCode":
          continuableType = "TMIndentedCode";
          break;
      }
      let lines = this.lines[sel.row].replace(/\n\n$/, "\n").split(/(?:\r\n|\n|\r)/);
      if (lines.length > 1) {
        // New line
        this.spliceLines(sel.row, 1, lines, true);
        sel.row++;
        sel.col = 0;
      }
      if (continuableType) {
        // Check if the previous line was non-empty
        let capture = lineGrammar[continuableType].regexp.exec(this.lines[sel.row - 1]);
        if (capture) {
          // Convention: capture[1] is the line type marker, capture[2] is the content
          if (capture[2]) {
            // Previous line has content, continue the continuable type

            // Hack for OL: increment number
            if (continuableType == "TMOL") {
              capture[1] = capture[1].replace(/\d{1,9}/, result => {
                return parseInt(result[0]) + 1;
              });
            }
            this.lines[sel.row] = `${capture[1]}${this.lines[sel.row]}`;
            this.lineDirty[sel.row] = true;
            sel.col = capture[1].length;
          } else {
            // Previous line has no content, remove the continuable type from the previous row
            this.lines[sel.row - 1] = "";
            this.lineDirty[sel.row - 1] = true;
          }
        }
      }
      this.updateFormatting();
    }

    // /**
    //  * Processes a "delete" input action.
    //  * @param {object} focus The selection
    //  * @param {boolean} forward If true, performs a forward delete, otherwise performs a backward delete
    //  */
    // processDelete(focus, forward) {
    //   if (!focus) return;
    //   let anchor = this.getSelection(true);
    //   // Do we have a non-empty selection?
    //   if (focus.col != anchor.col || focus.row != anchor.row) {
    //     // non-empty. direction doesn't matter.
    //     this.paste('', anchor, focus);
    //   } else {
    //     if (forward) {
    //       if (focus.col < this.lines[focus.row].length) this.paste('', {row: focus.row, col: focus.col + 1}, focus);
    //       else if (focus.col < this.lines.length) this.paste('', {row: focus.row + 1, col: 0}, focus);
    //       // Otherwise, we're at the very end and can't delete forward
    //     } else {
    //       if (focus.col > 0) this.paste('', {row: focus.row, col: focus.col - 1}, focus);
    //       else if (focus.row > 0) this.paste('', {row: focus.row - 1, col: this.lines[focus.row - 1].length - 1}, focus);
    //       // Otherwise, we're at the very beginning and can't delete backwards
    //     }
    //   }

    // }

    /**
     * Gets the current position of the selection counted by row and column of the editor Markdown content (as opposed to the position in the DOM).
     *
     * @param {boolean} getAnchor if set to true, gets the selection anchor (start point of the selection), otherwise gets the focus (end point).
     * @return {object} An object representing the selection, with properties col and row.
     */
    getSelection() {
      let getAnchor = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;
      const selection = window.getSelection();
      let startNode = getAnchor ? selection.anchorNode : selection.focusNode;
      if (!startNode) return null;
      let offset = getAnchor ? selection.anchorOffset : selection.focusOffset;
      if (startNode == this.e) {
        if (offset < this.lines.length) return {
          row: offset,
          col: 0
        };
        return {
          row: offset - 1,
          col: this.lines[offset - 1].length
        };
      }
      let col = this.computeColumn(startNode, offset);
      if (col === null) return null; // We are outside of the editor

      // Find the row node
      let node = startNode;
      while (node.parentElement != this.e) {
        node = node.parentElement;
      }
      let row = 0;
      // Check if we can read a line number from the data-line-num attribute.
      // The last condition is a security measure since inserting a new paragraph copies the previous rows' line number
      if (node.dataset && node.dataset.lineNum && (!node.previousSibling || node.previousSibling.dataset.lineNum != node.dataset.lineNum)) {
        row = parseInt(node.dataset.lineNum);
      } else {
        while (node.previousSibling) {
          row++;
          node = node.previousSibling;
        }
      }
      return {
        row: row,
        col: col,
        node: startNode
      };
    }

    /**
     * Computes a column within an editor line from a node and offset within that node.
     * @param {Node} startNode The node
     * @param {int} offset THe selection
     * @returns {int} the column, or null if the node is not inside the editor
     */
    computeColumn(startNode, offset) {
      let node = startNode;
      let col;
      // First, make sure we're actually in the editor.
      while (node && node.parentNode != this.e) {
        node = node.parentNode;
      }
      if (node == null) return null;

      // There are two ways that offset can be defined:
      // - Either, the node is a text node, in which case it is the offset within the text
      // - Or, the node is an element with child notes, in which case the offset refers to the
      //   child node after which the selection is located
      if (startNode.nodeType === Node.TEXT_NODE || offset === 0) {
        // In the case that the node is non-text node but the offset is 0,
        // The selection is at the beginning of that element so we
        // can simply use the same approach as if it were at the beginning
        // of a text node.
        col = offset;
        node = startNode;
      } else if (offset > 0) {
        node = startNode.childNodes[offset - 1];
        col = node.textContent.length;
      }
      while (node.parentNode != this.e) {
        if (node.previousSibling) {
          node = node.previousSibling;
          col += node.textContent.length;
        } else {
          node = node.parentNode;
        }
      }
      return col;
    }

    /**
     * Computes DOM node and offset within that node from a position expressed as row and column.
     * @param {int} row Row (line number)
     * @param {int} col Column
     * @returns An object with two properties: node and offset. offset may be null;
     */
    computeNodeAndOffset(row, col) {
      let bindRight = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
      if (row >= this.lineElements.length) {
        // Selection past the end of text, set selection to end of text
        row = this.lineElements.length - 1;
        col = this.lines[row].length;
      }
      if (col > this.lines[row].length) {
        col = this.lines[row].length;
      }
      const parentNode = this.lineElements[row];
      let node = parentNode.firstChild;
      let childrenComplete = false;
      // default return value
      let rv = {
        node: parentNode.firstChild ? parentNode.firstChild : parentNode,
        offset: 0
      };
      while (node != parentNode) {
        if (!childrenComplete && node.nodeType === Node.TEXT_NODE) {
          if (node.nodeValue.length >= col) {
            if (bindRight && node.nodeValue.length == col) {
              // Selection is at the end of this text node, but we are binding right (prefer an offset of 0 in the next text node)
              // Remember return value in case we don't find another text node
              rv = {
                node: node,
                offset: col
              };
              col = 0;
            } else {
              return {
                node: node,
                offset: col
              };
            }
          } else {
            col -= node.nodeValue.length;
          }
        }
        if (!childrenComplete && node.firstChild) {
          node = node.firstChild;
        } else if (node.nextSibling) {
          childrenComplete = false;
          node = node.nextSibling;
        } else {
          childrenComplete = true;
          node = node.parentNode;
        }
      }

      // Either, the position was invalid and we just return the default return value
      // Or we are binding right and the selection is at the end of the line
      return rv;
    }

    /**
     * Sets the selection based on rows and columns within the editor Markdown content.
     * @param {object} focus Object representing the selection, needs to have properties row and col.
     * @param anchor Anchor of the selection. If not given, assumes the current anchor.
     */
    setSelection(focus) {
      let anchor = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
      if (!focus) return;
      let range = document.createRange();
      let {
        node: focusNode,
        offset: focusOffset
      } = this.computeNodeAndOffset(focus.row, focus.col, anchor && anchor.row == focus.row && anchor.col > focus.col); // Bind selection right if anchor is in the same row and behind the focus
      let anchorNode = null,
        anchorOffset = null;
      if (anchor && (anchor.row != focus.row || anchor.col != focus.col)) {
        let {
          node,
          offset
        } = this.computeNodeAndOffset(anchor.row, anchor.col, focus.row == anchor.row && focus.col > anchor.col);
        anchorNode = node;
        anchorOffset = offset;
      }
      if (anchorNode) range.setStart(anchorNode, anchorOffset);else range.setStart(focusNode, focusOffset);
      range.setEnd(focusNode, focusOffset);
      let windowSelection = window.getSelection();
      windowSelection.removeAllRanges();
      windowSelection.addRange(range);
    }

    /**
     * Event handler for input events
     */
    handleInputEvent(event) {
      // For composition input, we are only updating the text after we have received
      // a compositionend event, so we return upon insertCompositionText.
      // Otherwise, the DOM changes break the text input.
      if (event.inputType == "insertCompositionText") return;
      let focus = this.getSelection();
      if ((event.inputType == "insertParagraph" || event.inputType == "insertLineBreak") && focus) {
        this.clearDirtyFlag();
        this.processNewParagraph(focus);
      } else {
        if (!this.e.firstChild) {
          this.e.innerHTML = '<div class="TMBlankLine"><br></div>';
        } else {
          this.fixNodeHierarchy();
        }
        this.updateLineContentsAndFormatting();
      }
      if (focus) {
        this.setSelection(focus);
      }
      this.fireChange();
    }

    /**
     * Fixes the node hierarchy – makes sure that each line is in a div, and there are no nested divs
     */
    fixNodeHierarchy() {
      const originalChildren = Array.from(this.e.childNodes);
      const replaceChild = function (child) {
        const parent = child.parentElement;
        const nextSibling = child.nextSibling;
        parent.removeChild(child);
        for (var _len = arguments.length, newChildren = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          newChildren[_key - 1] = arguments[_key];
        }
        newChildren.forEach(newChild => nextSibling ? parent.insertBefore(newChild, nextSibling) : parent.appendChild(newChild));
      };
      originalChildren.forEach(child => {
        if (child.nodeType !== Node.ELEMENT_NODE || child.tagName !== "DIV") {
          // Found a child node that's either not an element or not a div. Wrap it in a div.
          const divWrapper = document.createElement("div");
          replaceChild(child, divWrapper);
          divWrapper.appendChild(child);
        } else if (child.childNodes.length == 0) {
          // Empty div child node, include at least a <br />
          child.appendChild(document.createElement("br"));
        } else {
          const grandChildren = Array.from(child.childNodes);
          if (grandChildren.some(grandChild => grandChild.nodeType === Node.ELEMENT_NODE && grandChild.tagName === "DIV")) {
            return replaceChild(child, grandChildren);
          }
        }
      });
    }

    /**
     * Event handler for the "drop" event
     */
    handleDrop(event) {
      event.preventDefault();
      this.fireDrop(event.dataTransfer);
    }

    /**
     * Event handler for "selectionchange" events.
     */
    handleSelectionChangeEvent() {
      this.fireSelection();
    }

    /**
     * Convenience function to "splice" new lines into the arrays this.lines, this.lineDirty, this.lineTypes, and the DOM elements
     * underneath the editor element.
     * This method is essentially Array.splice, only that the third parameter takes an un-spread array (and the forth parameter)
     * determines whether the DOM should also be adjusted.
     *
     * @param {int} startLine Position at which to start changing the array of lines
     * @param {int} linesToDelete Number of lines to delete
     * @param {array} linesToInsert Array of strings representing the lines to be inserted
     * @param {boolean} adjustLineElements If true, then <div> elements are also inserted in the DOM at the respective position
     */
    spliceLines(startLine) {
      let linesToDelete = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
      let linesToInsert = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : [];
      let adjustLineElements = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : true;
      if (adjustLineElements) {
        for (let i = 0; i < linesToDelete; i++) {
          this.e.removeChild(this.e.childNodes[startLine]);
        }
      }
      let insertedBlank = [];
      let insertedDirty = [];
      for (let i = 0; i < linesToInsert.length; i++) {
        insertedBlank.push("");
        insertedDirty.push(true);
        if (adjustLineElements) {
          if (this.e.childNodes[startLine]) this.e.insertBefore(document.createElement("div"), this.e.childNodes[startLine]);else this.e.appendChild(document.createElement("div"));
        }
      }
      this.lines.splice(startLine, linesToDelete, ...linesToInsert);
      this.lineTypes.splice(startLine, linesToDelete, ...insertedBlank);
      this.lineDirty.splice(startLine, linesToDelete, ...insertedDirty);
    }

    /**
     * Event handler for the "paste" event
     */
    handlePaste(event) {
      event.preventDefault();

      // get text representation of clipboard
      let text = (event.originalEvent || event).clipboardData.getData("text/plain");

      // insert text manually
      this.paste(text);
    }

    /**
     * Pastes the text at the current selection (or at the end, if no current selection)
     * @param {string} text
     */
    paste(text) {
      let anchor = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
      let focus = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;
      if (!anchor) anchor = this.getSelection(true);
      if (!focus) focus = this.getSelection(false);
      let beginning, end;
      if (!focus) {
        focus = {
          row: this.lines.length - 1,
          col: this.lines[this.lines.length - 1].length
        }; // Insert at end
      }

      if (!anchor) {
        anchor = focus;
      }
      if (anchor.row < focus.row || anchor.row == focus.row && anchor.col <= focus.col) {
        beginning = anchor;
        end = focus;
      } else {
        beginning = focus;
        end = anchor;
      }
      let insertedLines = text.split(/(?:\r\n|\r|\n)/);
      let lineBefore = this.lines[beginning.row].substr(0, beginning.col);
      let lineEnd = this.lines[end.row].substr(end.col);
      insertedLines[0] = lineBefore.concat(insertedLines[0]);
      let endColPos = insertedLines[insertedLines.length - 1].length;
      insertedLines[insertedLines.length - 1] = insertedLines[insertedLines.length - 1].concat(lineEnd);
      this.spliceLines(beginning.row, 1 + end.row - beginning.row, insertedLines);
      focus.row = beginning.row + insertedLines.length - 1;
      focus.col = endColPos;
      this.updateFormatting();
      this.setSelection(focus);
      this.fireChange();
    }

    /**
     * Computes the (lowest in the DOM tree) common ancestor of two DOM nodes.
     * @param {Node} node1 the first node
     * @param {Node} node2 the second node
     * @returns {Node} The commen ancestor node, or null if there is no common ancestor
     */
    computeCommonAncestor(node1, node2) {
      if (!node1 || !node2) return null;
      if (node1 == node2) return node1;
      const ancestry = node => {
        let ancestry = [];
        while (node) {
          ancestry.unshift(node);
          node = node.parentNode;
        }
        return ancestry;
      };
      const ancestry1 = ancestry(node1);
      const ancestry2 = ancestry(node2);
      if (ancestry1[0] != ancestry2[0]) return null;
      let i;
      for (i = 0; ancestry1[i] == ancestry2[i]; i++);
      return ancestry1[i - 1];
    }

    /**
     * Finds the (lowest in the DOM tree) enclosing DOM node with a given class.
     * @param {object} focus The focus selection object
     * @param {object} anchor The anchor selection object
     * @param {string} className The class name to find
     * @returns {Node} The enclosing DOM node with the respective class (inside the editor), if there is one; null otherwise.
     */
    computeEnclosingMarkupNode(focus, anchor, className) {
      let node = null;
      if (!focus) return null;
      if (!anchor) {
        node = focus.node;
      } else {
        if (focus.row != anchor.row) return null;
        node = this.computeCommonAncestor(focus.node, anchor.node);
      }
      if (!node) return null;
      while (node != this.e) {
        if (node.className && node.className.includes(className)) return node;
        node = node.parentNode;
      }
      // Ascended all the way to the editor element
      return null;
    }

    /**
     * Returns the state (true / false) of all commands.
     * @param focus Focus of the selection. If not given, assumes the current focus.
     * @param anchor Anchor of the selection. If not given, assumes the current anchor.
     */
    getCommandState() {
      let focus = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;
      let anchor = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
      let commandState = {};
      if (!focus) focus = this.getSelection(false);
      if (!anchor) anchor = this.getSelection(true);
      if (!focus) {
        for (let cmd in commands) {
          commandState[cmd] = null;
        }
        return commandState;
      }
      if (!anchor) anchor = focus;
      let start, end;
      if (anchor.row < focus.row || anchor.row == focus.row && anchor.col < focus.col) {
        start = anchor;
        end = focus;
      } else {
        start = focus;
        end = anchor;
      }
      if (end.row > start.row && end.col == 0) {
        end.row--;
        end.col = this.lines[end.row].length; // Selection to beginning of next line is said to end at the beginning of the last line
      }

      for (let cmd in commands) {
        if (commands[cmd].type == "inline") {
          if (!focus || focus.row != anchor.row || !this.isInlineFormattingAllowed(focus, anchor)) {
            commandState[cmd] = null;
          } else {
            // The command state is true if there is a respective enclosing markup node (e.g., the selection is enclosed in a <b>..</b>) ...
            commandState[cmd] = !!this.computeEnclosingMarkupNode(focus, anchor, commands[cmd].className) ||
            // ... or if it's an empty string preceded by and followed by formatting markers, e.g. **|** where | is the cursor
            focus.col == anchor.col && !!this.lines[focus.row].substr(0, focus.col).match(commands[cmd].unset.prePattern) && !!this.lines[focus.row].substr(focus.col).match(commands[cmd].unset.postPattern);
          }
        }
        if (commands[cmd].type == "line") {
          if (!focus) {
            commandState[cmd] = null;
          } else {
            let state = this.lineTypes[start.row] == commands[cmd].className;
            for (let line = start.row; line <= end.row; line++) {
              if (this.lineTypes[line] == commands[cmd].className != state) {
                state = null;
                break;
              }
            }
            commandState[cmd] = state;
          }
        }
      }
      return commandState;
    }

    /**
     * Sets a command state
     * @param {string} command
     * @param {boolean} state
     */
    setCommandState(command, state) {
      if (commands[command].type == "inline") {
        let anchor = this.getSelection(true);
        let focus = this.getSelection(false);
        if (!anchor) anchor = focus;
        if (!anchor) return;
        if (anchor.row != focus.row) return;
        if (!this.isInlineFormattingAllowed(focus, anchor)) return;
        let markupNode = this.computeEnclosingMarkupNode(focus, anchor, commands[command].className);
        this.clearDirtyFlag();

        // First case: There's an enclosing markup node, remove the markers around that markup node
        if (markupNode) {
          this.lineDirty[focus.row] = true;
          const startCol = this.computeColumn(markupNode, 0);
          const len = markupNode.textContent.length;
          const left = this.lines[focus.row].substr(0, startCol).replace(commands[command].unset.prePattern, "");
          const mid = this.lines[focus.row].substr(startCol, len);
          const right = this.lines[focus.row].substr(startCol + len).replace(commands[command].unset.postPattern, "");
          this.lines[focus.row] = left.concat(mid, right);
          anchor.col = left.length;
          focus.col = anchor.col + len;
          this.updateFormatting();
          this.setSelection(focus, anchor);
          this.fireChange();

          // Second case: Empty selection with surrounding formatting markers, remove those
        } else if (focus.col == anchor.col && !!this.lines[focus.row].substr(0, focus.col).match(commands[command].unset.prePattern) && !!this.lines[focus.row].substr(focus.col).match(commands[command].unset.postPattern)) {
          this.lineDirty[focus.row] = true;
          const left = this.lines[focus.row].substr(0, focus.col).replace(commands[command].unset.prePattern, "");
          const right = this.lines[focus.row].substr(focus.col).replace(commands[command].unset.postPattern, "");
          this.lines[focus.row] = left.concat(right);
          focus.col = anchor.col = left.length;
          this.updateFormatting();
          this.setSelection(focus, anchor);
          this.fireChange();

          // Not currently formatted, insert formatting markers
        } else {
          // Trim any spaces from the selection
          let {
            startCol,
            endCol
          } = focus.col < anchor.col ? {
            startCol: focus.col,
            endCol: anchor.col
          } : {
            startCol: anchor.col,
            endCol: focus.col
          };
          let match = this.lines[focus.row].substr(startCol, endCol - startCol).match(/^(?<leading>\s*).*\S(?<trailing>\s*)$/);
          if (match) {
            startCol += match.groups.leading.length;
            endCol -= match.groups.trailing.length;
          }
          focus.col = startCol;
          anchor.col = endCol;

          // Just insert markup before and after and hope for the best.
          this.wrapSelection(commands[command].set.pre, commands[command].set.post, focus, anchor);
          this.fireChange();
          // TODO clean this up so that markup remains properly nested
        }
      } else if (commands[command].type == "line") {
        let anchor = this.getSelection(true);
        let focus = this.getSelection(false);
        if (!anchor) anchor = focus;
        if (!focus) return;
        this.clearDirtyFlag();
        let start = anchor.row > focus.row ? focus : anchor;
        let end = anchor.row > focus.row ? anchor : focus;
        if (end.row > start.row && end.col == 0) {
          end.row--;
        }
        for (let line = start.row; line <= end.row; line++) {
          if (state && this.lineTypes[line] != commands[command].className) {
            this.lines[line] = this.lines[line].replace(commands[command].set.pattern, commands[command].set.replacement.replace("$#", line - start.row + 1));
            this.lineDirty[line] = true;
          }
          if (!state && this.lineTypes[line] == commands[command].className) {
            this.lines[line] = this.lines[line].replace(commands[command].unset.pattern, commands[command].unset.replacement);
            this.lineDirty[line] = true;
          }
        }
        this.updateFormatting();
        this.setSelection({
          row: end.row,
          col: this.lines[end.row].length
        }, {
          row: start.row,
          col: 0
        });
        this.fireChange();
      }
    }

    /**
     * Returns whether or not inline formatting is allowed at the current focus
     * @param {object} focus The current focus
     */
    isInlineFormattingAllowed() {
      // TODO Remove parameters from all calls
      const sel = window.getSelection();
      if (!sel || !sel.focusNode || !sel.anchorNode) return false;

      // Check if we can find a common ancestor with the class `TMInlineFormatted`

      // Special case: Empty selection right before `TMInlineFormatted`
      if (sel.isCollapsed && sel.focusNode.nodeType == 3 && sel.focusOffset == sel.focusNode.nodeValue.length) {
        let node;
        for (node = sel.focusNode; node && node.nextSibling == null; node = node.parentNode);
        if (node && node.nextSibling.className && node.nextSibling.className.includes("TMInlineFormatted")) return true;
      }

      // Look for a common ancestor
      let ancestor = this.computeCommonAncestor(sel.focusNode, sel.anchorNode);
      if (!ancestor) return false;

      // Check if there's an ancestor of class 'TMInlineFormatted' or 'TMBlankLine'
      while (ancestor && ancestor != this.e) {
        if (ancestor.className && typeof ancestor.className.includes == "function" && (ancestor.className.includes("TMInlineFormatted") || ancestor.className.includes("TMBlankLine"))) return true;
        ancestor = ancestor.parentNode;
      }
      return false;
    }

    /**
     * Wraps the current selection in the strings pre and post. If the selection is not on one line, returns.
     * @param {string} pre The string to insert before the selection.
     * @param {string} post The string to insert after the selection.
     * @param {object} focus The current selection focus. If null, selection will be computed.
     * @param {object} anchor The current selection focus. If null, selection will be computed.
     */
    wrapSelection(pre, post) {
      let focus = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;
      let anchor = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : null;
      if (!focus) focus = this.getSelection(false);
      if (!anchor) anchor = this.getSelection(true);
      if (!focus || !anchor || focus.row != anchor.row) return;
      this.lineDirty[focus.row] = true;
      const startCol = focus.col < anchor.col ? focus.col : anchor.col;
      const endCol = focus.col < anchor.col ? anchor.col : focus.col;
      const left = this.lines[focus.row].substr(0, startCol).concat(pre);
      const mid = endCol == startCol ? "" : this.lines[focus.row].substr(startCol, endCol - startCol);
      const right = post.concat(this.lines[focus.row].substr(endCol));
      this.lines[focus.row] = left.concat(mid, right);
      anchor.col = left.length;
      focus.col = anchor.col + mid.length;
      this.updateFormatting();
      this.setSelection(focus, anchor);
    }

    /**
     * Toggles the command state for a command (true <-> false)
     * @param {string} command The editor command
     */
    toggleCommandState(command) {
      if (!this.lastCommandState) this.lastCommandState = this.getCommandState();
      this.setCommandState(command, !this.lastCommandState[command]);
    }

    /**
     * Fires a change event. Updates the linked textarea and notifies any event listeners.
     */
    fireChange() {
      if (!this.textarea && !this.listeners.change.length) return;
      const content = this.getContent();
      if (this.textarea) this.textarea.value = content;
      for (let listener of this.listeners.change) {
        listener({
          content: content,
          linesDirty: this.linesDirty
        });
      }
    }

    /**
     * Fires a "selection changed" event.
     */
    fireSelection() {
      if (this.listeners.selection && this.listeners.selection.length) {
        let focus = this.getSelection(false);
        let anchor = this.getSelection(true);
        let commandState = this.getCommandState(focus, anchor);
        if (this.lastCommandState) {
          Object.assign(this.lastCommandState, commandState);
        } else {
          this.lastCommandState = Object.assign({}, commandState);
        }
        for (let listener of this.listeners.selection) {
          listener({
            focus: focus,
            anchor: anchor,
            commandState: this.lastCommandState
          });
        }
      }
    }

    /**
     * Fires a drop event.
     */
    fireDrop(dataTransfer) {
      for (let listener of this.listeners.drop) {
        listener({
          dataTransfer
        });
      }
    }

    /**
     * Adds an event listener.
     * @param {string} type The type of event to listen to. Can be 'change', 'selection' or 'drop'.
     * @param {*} listener Function of the type (event) => {} to be called when the event occurs.
     */
    addEventListener(type, listener) {
      if (type.match(/^(?:change|input)$/i)) {
        this.listeners.change.push(listener);
      }
      if (type.match(/^(?:selection|selectionchange)$/i)) {
        this.listeners.selection.push(listener);
      }
      if (type.match(/^(?:drop)$/i)) {
        this.listeners.drop.push(listener);
      }
    }
  }

  exports.CommandBar = CommandBar;
  exports.Editor = Editor;

}));
//# sourceMappingURL=index.js.map

//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VzIjpbInNyYy9zdmcvc3ZnLmpzIiwic3JjL1RpbnlNREVDb21tYW5kQmFyLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2dsb2JhbC5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9mYWlscy5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9kZXNjcmlwdG9ycy5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9mdW5jdGlvbi1iaW5kLW5hdGl2ZS5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9mdW5jdGlvbi11bmN1cnJ5LXRoaXMuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvZG9jdW1lbnQtYWxsLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2lzLWNhbGxhYmxlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2lzLW51bGwtb3ItdW5kZWZpbmVkLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL3JlcXVpcmUtb2JqZWN0LWNvZXJjaWJsZS5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy90by1vYmplY3QuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvaGFzLW93bi1wcm9wZXJ0eS5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9mdW5jdGlvbi1uYW1lLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2RlZmluZS1nbG9iYWwtcHJvcGVydHkuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvc2hhcmVkLXN0b3JlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2luc3BlY3Qtc291cmNlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL3dlYWstbWFwLWJhc2ljLWRldGVjdGlvbi5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9pcy1vYmplY3QuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvZG9jdW1lbnQtY3JlYXRlLWVsZW1lbnQuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvaWU4LWRvbS1kZWZpbmUuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvdjgtcHJvdG90eXBlLWRlZmluZS1idWcuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvYW4tb2JqZWN0LmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2Z1bmN0aW9uLWNhbGwuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvZ2V0LWJ1aWx0LWluLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL29iamVjdC1pcy1wcm90b3R5cGUtb2YuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvZW5naW5lLXVzZXItYWdlbnQuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvZW5naW5lLXY4LXZlcnNpb24uanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvc3ltYm9sLWNvbnN0cnVjdG9yLWRldGVjdGlvbi5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy91c2Utc3ltYm9sLWFzLXVpZC5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9pcy1zeW1ib2wuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvdHJ5LXRvLXN0cmluZy5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9hLWNhbGxhYmxlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2dldC1tZXRob2QuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvb3JkaW5hcnktdG8tcHJpbWl0aXZlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL3NoYXJlZC5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy91aWQuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvd2VsbC1rbm93bi1zeW1ib2wuanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvdG8tcHJpbWl0aXZlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL3RvLXByb3BlcnR5LWtleS5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9vYmplY3QtZGVmaW5lLXByb3BlcnR5LmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2NyZWF0ZS1wcm9wZXJ0eS1kZXNjcmlwdG9yLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2NyZWF0ZS1ub24tZW51bWVyYWJsZS1wcm9wZXJ0eS5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL2ludGVybmFscy9zaGFyZWQta2V5LmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL2ludGVybmFsLXN0YXRlLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL21ha2UtYnVpbHQtaW4uanMiLCJub2RlX21vZHVsZXMvY29yZS1qcy9pbnRlcm5hbHMvZGVmaW5lLWJ1aWx0LWluLWFjY2Vzc29yLmpzIiwibm9kZV9tb2R1bGVzL2NvcmUtanMvaW50ZXJuYWxzL3JlZ2V4cC1mbGFncy5qcyIsIm5vZGVfbW9kdWxlcy9jb3JlLWpzL21vZHVsZXMvZXMucmVnZXhwLmZsYWdzLmpzIiwic3JjL2dyYW1tYXIuanMiLCJzcmMvVGlueU1ERS5qcyJdLCJzb3VyY2VzQ29udGVudCI6WyJjb25zdCBzdmcgPSB7XG4gIGJsb2NrcXVvdGU6IGA8c3ZnIGhlaWdodD1cIjE4XCIgd2lkdGg9XCIxOFwiPjxyZWN0IHdpZHRoPVwiNVwiIGhlaWdodD1cIjVcIiB4PVwiM1wiIHk9XCI0XCIgcnk9XCIxXCIvPjxyZWN0IHJ5PVwiMVwiIHk9XCI0XCIgeD1cIjEwXCIgaGVpZ2h0PVwiNVwiIHdpZHRoPVwiNVwiLz48cGF0aCBkPVwiTTggNi45OTl2M2MwIDEtMSAzLTEgM3MtLjMzMSAxLTEuMzMxIDFoLTFjLTEgMC0uNjY5LTEtLjY2OS0xczEtMiAxLTN2LTN6bTcgMHYzYzAgMS0xIDMtMSAzcy0uMzMxIDEtMS4zMzEgMWgtMWMtMSAwLS42NjktMS0uNjY5LTFzMS0yIDEtM3YtM3pcIi8+PC9zdmc+YCxcbiAgYm9sZDogYDxzdmcgaGVpZ2h0PVwiMThcIiB3aWR0aD1cIjE4XCI+PHBhdGggZD1cIk00IDJhMSAxIDAgMDAtMSAxdjEyYTEgMSAwIDAwMSAxaDZjNCAwIDUtMiA1LTQgMC0xLjMyMi0uNDM0LTIuNjM2LTEuODg1LTMuMzgxQzEzLjc3MiA3Ljg4NSAxNCA2Ljk0NSAxNCA2YzAtMi0xLTQtNS00em0xIDJoNGMxLjY2OCAwIDIuMzIuMzkzIDIuNi42NzIuMjguMjc5LjQuNjYyLjQgMS4zMjhzLS4xMiAxLjA1Ny0uNCAxLjMzOGMtLjI3NS4yNzQtMS4wMTQuNjQ2LTIuNi42NjJINXptNSA2YzEuNjY2LjAwNSAyLjMxOC4zODggMi41OTYuNjY2LjI3Ny4yNzguNDA0LjY2Ny40MDQgMS4zMzRzLS4xMjcgMS4wNi0uNDA0IDEuMzM4Yy0uMjc4LjI3OC0uOTMuNjYtMi41OTYuNjYybC00Ljk5Mi4wMDRMNSAxMHpcIi8+PC9zdmc+YCxcbiAgY2xlYXJfZm9ybWF0dGluZzogYDxzdmcgaGVpZ2h0PVwiMThcIiB3aWR0aD1cIjE4XCI+PHBhdGggZD1cIk0xMS4wMyAxYTEgMSAwIDAwLS43NC4zbC05IDlhMSAxIDAgMDAwIDEuNGw0IDRBMSAxIDAgMDA2IDE2aDJhMSAxIDAgMDAuNy0uM2w4LThhMSAxIDAgMDAwLTEuNGwtNS01YTEgMSAwIDAwLS42Ny0uM3pNOC43IDUuN2wzLjU4IDMuNkw3LjYgMTRINi40bC0zLTMgNS4zLTUuM3pcIi8+PHJlY3Qgcnk9XCIuOFwiIHJ4PVwiLjhcIiB5PVwiMTRcIiB4PVwiMTZcIiBoZWlnaHQ9XCIyXCIgd2lkdGg9XCIyXCIvPjxyZWN0IHdpZHRoPVwiMlwiIGhlaWdodD1cIjJcIiB4PVwiMTNcIiB5PVwiMTRcIiByeD1cIi44XCIgcnk9XCIuOFwiLz48cmVjdCByeT1cIi44XCIgcng9XCIuOFwiIHk9XCIxNFwiIHg9XCIxMFwiIGhlaWdodD1cIjJcIiB3aWR0aD1cIjJcIi8+PC9zdmc+YCxcbiAgY29kZTogYDxzdmcgaGVpZ2h0PVwiMThcIiB3aWR0aD1cIjE4XCI+PHBhdGggZD1cIk0xMy41IDIuOTk0YS41LjUgMCAwMC0uNS41LjUuNSAwIDAwMCAuMDM0VjQuNTNhNS45OTMgNS45OTMgMCAwMC03LjQ1MS0uNDQ1QTYgNiAwIDEwMTIuNDUgMTMuOWE1Ljk5IDUuOTkgMCAwMDEuMzQ2LTEuMzM0LjUuNSAwIDAwLjA5Ni0uMTAxLjUuNSAwIDAwLS4xMi0uNjk4LjUuNSAwIDAwLS42OTcuMTJsLS4wMDQtLjAwNWE1IDUgMCAwMS0xLjE5NyAxLjIgNSA1IDAgMTExLjIxNS02Ljk2NS41LjUgMCAwMC42OTcuMTIuNS41IDAgMDAuMjExLS40NFY0Ljc0NUgxNFYzLjQ5M2EuNS41IDAgMDAtLjUtLjV6XCIvPjwvc3ZnPmAsXG4gIGgxOiBgPHN2ZyBoZWlnaHQ9XCIxOFwiIHdpZHRoPVwiMThcIj48cGF0aCBkPVwiTTMgMnMwLTEgMS0xaDFjMSAwIDEgMSAxIDF2Nmg2VjJzMC0xIDEtMWgxYzEgMCAxIDEgMSAxdjE0czAgMS0xIDFoLTFjLTEgMC0xLTEtMS0xdi02SDZ2NnMwIDEtMSAxSDRjLTEgMC0xLTEtMS0xelwiLz48L3N2Zz5gLFxuICBoMjogYDxzdmcgaGVpZ2h0PVwiMThcIiB3aWR0aD1cIjE4XCI+PHBhdGggZD1cIk00IDJzMC0xIDEtMSAxIDEgMSAxdjZjMS0xIDItMSA0LTEgMyAwIDQgMiA0IDR2NXMwIDEtMSAxLTEtMS0xLTF2LTVjMC0xLjA5NC0xLTItMi0yLTIgMC0zIDAtNCAydjVzMCAxLTEgMS0xLTEtMS0xelwiLz48L3N2Zz5gLFxuICBocjogYDxzdmcgaGVpZ2h0PVwiMThcIiB3aWR0aD1cIjE4XCI+PHJlY3Qgcnk9XCIxXCIgeT1cIjhcIiBoZWlnaHQ9XCIyXCIgd2lkdGg9XCIxOFwiIHN0eWxlPVwiZm9udC12YXJpYXRpb24tc2V0dGluZ3M6bm9ybWFsO21hcmtlcjpub25lXCIvPjwvc3ZnPmAsXG4gIGltYWdlOiBgPHN2ZyBoZWlnaHQ9XCIxOFwiIHdpZHRoPVwiMThcIj48cGF0aCBkPVwiTTEgMnYxNGgxNlYySDF6bTIgMmgxMnY3bC0zLTMtNCA0LTItMi0zIDNWNHpcIi8+PGNpcmNsZSByPVwiMS41XCIgY3k9XCI2LjVcIiBjeD1cIjUuNVwiLz48L3N2Zz5gLFxuICBpdGFsaWM6IGA8c3ZnIGhlaWdodD1cIjE4XCIgd2lkdGg9XCIxOFwiPjxwYXRoIGQ9XCJNOSAyYTEgMSAwIDAwMCAyTDcgMTRhMSAxIDAgMTAwIDJoMmExIDEgMCAwMDAtMmwyLTEwYTEgMSAwIDEwMC0yelwiLz48L3N2Zz5gLFxuICBsaW5rOiBgPHN2ZyBoZWlnaHQ9XCIxOFwiIHdpZHRoPVwiMThcIj48cGF0aCBkPVwiTTkuMDcgNS4xOGEzLjkgMy45IDAgMDAtMS41Mi40M0M2LjMxIDYuMjIgNS4zIDcuMjkgNC4zIDguMjljLTEgMS0yLjA3IDIuMDItMi42OCAzLjI2LS4zMS42Mi0uNSAxLjMzLS40MSAyLjA3LjA5Ljc1LjQ4IDEuNDcgMS4xIDIuMDkuNjEuNjEgMS4zMyAxIDIuMDggMS4xLjc0LjA5IDEuNDUtLjEgMi4wNy0uNDIgMS4yNC0uNjEgMi4yNi0xLjY4IDMuMjYtMi42OC40Ni0uNDcuOTQtLjk0IDEuMzktMS40NGwtLjQzLjI2Yy0uNjguMzQtMS40OS41Ni0yLjM2LjQ2LS4yLS4wMy0uNC0uMDgtLjYtLjE0LS43OS43Ni0xLjU1IDEuNDUtMi4xNiAxLjc2LS4zOC4xOS0uNjcuMjQtLjkyLjIxLS4yNi0uMDMtLjU0LS4xNC0uOTItLjUzLS4zOS0uMzgtLjUtLjY2LS41My0uOTEtLjAzLS4yNi4wMi0uNTUuMjEtLjkzLjM5LS43NiAxLjMyLTEuNzQgMi4zMi0yLjc0IDEtMSAxLjk4LTEuOTMgMi43NC0yLjMyLjM4LS4xOS42Ny0uMjQuOTMtLjIxLjI1LjAzLjUzLjE0LjkxLjUzLjM5LjM4LjUuNjYuNTMuOTJ2LjA2bDEuMTItMS4wNi40NC0uNDdjLS4xOC0uMy0uNC0uNi0uNjctLjg3LS42Mi0uNjEtMS4zNC0xLTIuMDktMS4xYTMuMDggMy4wOCAwIDAwLS41NS0uMDF6XCIvPjxwYXRoIGQ9XCJNMTMuMDcuODZhMy45IDMuOSAwIDAwLTEuNTIuNDNjLTEuMjQuNjItMi4yNiAxLjY5LTMuMjYgMi42OS0uNDYuNDctLjk0Ljk0LTEuMzkgMS40My4xNS0uMDguMjgtLjE4LjQzLS4yNWE0LjQgNC40IDAgMDEyLjM2LS40NmMuMi4wMi40LjA3LjYuMTQuNzktLjc3IDEuNTUtMS40NiAyLjE2LTEuNzYuMzgtLjE5LjY3LS4yNS45My0uMjEuMjUuMDMuNTMuMTQuOTEuNTIuMzkuMzguNS42Ni41My45Mi4wMy4yNi0uMDIuNTUtLjIxLjkzLS4zOS43Ni0xLjMyIDEuNzQtMi4zMiAyLjc0LTEgMS0xLjk4IDEuOTMtMi43NCAyLjMxLS4zOC4yLS42Ny4yNS0uOTMuMjItLjI1LS4wNC0uNTMtLjE1LS45MS0uNTMtLjM5LS4zOC0uNS0uNjYtLjUzLS45MlY5Yy0uMzYuMzMtLjczLjY3LTEuMTIgMS4wNmwtLjQzLjQ2Yy4xNy4zLjQuNi42Ni44Ny42Mi42MiAxLjM0IDEgMi4wOCAxLjEuNzUuMSAxLjQ2LS4xIDIuMDgtLjQxIDEuMjQtLjYyIDIuMjYtMS42OSAzLjI2LTIuNjlzMi4wNy0yLjAyIDIuNjgtMy4yNmMuMzEtLjYyLjUtMS4zMi40MS0yLjA3YTMuNjMgMy42MyAwIDAwLTEuMS0yLjA4Yy0uNjEtLjYyLTEuMzMtMS0yLjA3LTEuMWEzLjA4IDMuMDggMCAwMC0uNTYtLjAyelwiLz48L3N2Zz5gLFxuICBvbDogYDxzdmcgaGVpZ2h0PVwiMThcIiB3aWR0aD1cIjE4XCI+PHBhdGggZD1cIk0xLjUgN2EuNS41IDAgMTAwIDFoMWEuNS41IDAgMDEuNS41YzAgLjE2NC0uMDguMzEtLjE0LjM1NWwtMS42NTUgMS4yNUEuNDkyLjQ5MiAwIDAwMSAxMC41YS41LjUgMCAwMC41LjVoMmEuNS41IDAgMDAwLTFIM2wuMzk4LS4yOTlBMS41IDEuNSAwIDAwMi41IDd6XCIvPjxwYXRoIGQ9XCJNMS41IDEyYy0uNjY3IDAtLjY2NyAxIDAgMWgxLjI1Yy4zMzMgMCAuMzMzLjUgMCAuNUgyLjVjLS42NjcgMC0uNjY3IDEgMCAxaC4yNWMuMzMzIDAgLjMzMy41IDAgLjVIMS41Yy0uNjY3IDAtLjY2NyAxIDAgMWgxYy4zOTggMCAuNzgtLjEzMSAxLjA2LS4zNjUuMjgyLS4yMzUuNDQtLjU1NC40NC0uODg1YTEuMTIxIDEuMTIxIDAgMDAtLjMwMy0uNzVjLjE5MS0uMjA0LjMtLjQ3LjMwMy0uNzUgMC0uMzMyLS4xNTgtLjY1MS0uNDQtLjg4NS0uMy0uMjQxLS42NzUtLjM3LTEuMDYtLjM2NXpcIi8+PHJlY3QgeT1cIjEzXCIgeD1cIjZcIiBoZWlnaHQ9XCIyXCIgd2lkdGg9XCIxMlwiIHJ5PVwiMVwiLz48cmVjdCByeT1cIjFcIiB3aWR0aD1cIjEyXCIgaGVpZ2h0PVwiMlwiIHg9XCI2XCIgeT1cIjhcIi8+PHJlY3QgeT1cIjNcIiB4PVwiNlwiIGhlaWdodD1cIjJcIiB3aWR0aD1cIjEyXCIgcnk9XCIxXCIvPjxwYXRoIGQ9XCJNMS41IDJhLjUuNSAwIDEwMCAxSDJ2MmgtLjVhLjUuNSAwIDEwMCAxaDJhLjUuNSAwIDEwMC0xSDNWMi41YzAtLjI3Ny0uMjIzLS41LS41LS41elwiLz48L3N2Zz5gLFxuICBzdHJpa2V0aHJvdWdoOiBgPHN2ZyB3aWR0aD1cIjE4XCIgaGVpZ2h0PVwiMThcIj48cGF0aCBkPVwiTTEwIDJDNiAyIDQgNCA0IDZjMCAuMzM4LjA4LjY3Mi4xOTMgMWgyLjM0QzYuMTEzIDYuNjI5IDYgNi4yOTUgNiA2YzAtLjMzNC4xMTctLjcyNS42OTEtMS4xNTRDNy4yNjUgNC40MTYgOC4zMzEgNCAxMCA0aDNhMSAxIDAgMDAwLTJ6bTEuNDc3IDljLjQxMy4zNjguNTIzLjcwNi41MjMgMSAwIC4zMzQtLjEyNy43MTItLjcwMSAxLjE0My0uNTc1LjQzLTEuNjMyLjg1LTMuMjk5Ljg1N2wtMS4wMDYuMDA3VjE0SDVhMSAxIDAgMDAwIDJoM2M0IDAgNi0yIDYtNCAwLS4zMzgtLjA4MS0uNjcyLS4xOTUtMXpcIi8+PHJlY3Qgcnk9XCIxXCIgeT1cIjhcIiB4PVwiMVwiIGhlaWdodD1cIjJcIiB3aWR0aD1cIjE2XCIvPjwvc3ZnPmAsXG4gIHVsOiBgPHN2ZyBoZWlnaHQ9XCIxOFwiIHdpZHRoPVwiMThcIj48Y2lyY2xlIGN4PVwiMlwiIGN5PVwiOVwiIHI9XCIyXCIvPjxjaXJjbGUgY3k9XCI0XCIgY3g9XCIyXCIgcj1cIjJcIi8+PHJlY3QgeT1cIjNcIiB4PVwiNlwiIGhlaWdodD1cIjJcIiB3aWR0aD1cIjEyXCIgcnk9XCIxXCIvPjxjaXJjbGUgY3g9XCIyXCIgY3k9XCIxNFwiIHI9XCIyXCIvPjxyZWN0IHJ5PVwiMVwiIHdpZHRoPVwiMTJcIiBoZWlnaHQ9XCIyXCIgeD1cIjZcIiB5PVwiOFwiLz48cmVjdCB5PVwiMTNcIiB4PVwiNlwiIGhlaWdodD1cIjJcIiB3aWR0aD1cIjEyXCIgcnk9XCIxXCIvPjwvc3ZnPmBcbn07XG5cbmV4cG9ydCBkZWZhdWx0IHN2ZzsiLCJpbXBvcnQgc3ZnIGZyb20gJy4vc3ZnL3N2Zyc7XG5cbmNvbnN0IGlzTWFjTGlrZSA9IC8oTWFjfGlQaG9uZXxpUG9kfGlQYWQpL2kudGVzdCh0eXBlb2YgbmF2aWdhdG9yICE9PSBcInVuZGVmaW5lZFwiID8gbmF2aWdhdG9yLnBsYXRmb3JtIDogXCJcIik7XG5cbmNvbnN0IERlZmF1bHRDb21tYW5kcyA9IHtcbiAgJ2JvbGQnOiB7XG4gICAgbmFtZTogJ2JvbGQnLFxuICAgIGFjdGlvbjogJ2JvbGQnLFxuICAgIGlubmVySFRNTDogc3ZnLmJvbGQsXG4gICAgdGl0bGU6ICdCb2xkJyxcbiAgICBob3RrZXk6ICdNb2QtQicsXG4gIH0sXG4gICdpdGFsaWMnOiB7XG4gICAgbmFtZTogJ2l0YWxpYycsXG4gICAgYWN0aW9uOiAnaXRhbGljJyxcbiAgICBpbm5lckhUTUw6IHN2Zy5pdGFsaWMsXG4gICAgdGl0bGU6ICdJdGFsaWMnLFxuICAgIGhvdGtleTogJ01vZC1JJyxcbiAgfSxcbiAgJ3N0cmlrZXRocm91Z2gnOiB7XG4gICAgbmFtZTogJ3N0cmlrZXRocm91Z2gnLFxuICAgIGFjdGlvbjogJ3N0cmlrZXRocm91Z2gnLFxuICAgIGlubmVySFRNTDogc3ZnLnN0cmlrZXRocm91Z2gsXG4gICAgdGl0bGU6ICdTdHJpa2V0aHJvdWdoJyxcbiAgICBob3RrZXk6ICdNb2QyLVNoaWZ0LTUnLFxuICB9LFxuICAnY29kZSc6IHtcbiAgICBuYW1lOiAnY29kZScsXG4gICAgYWN0aW9uOiAnY29kZScsXG4gICAgaW5uZXJIVE1MOiBzdmcuY29kZSxcbiAgICB0aXRsZTogJ0Zvcm1hdCBhcyBjb2RlJyxcbiAgfSxcbiAgJ2gxJzoge1xuICAgIG5hbWU6ICdoMScsXG4gICAgYWN0aW9uOiAnaDEnLFxuICAgIGlubmVySFRNTDogc3ZnLmgxLFxuICAgIHRpdGxlOiAnTGV2ZWwgMSBoZWFkaW5nJyxcbiAgICBob3RrZXk6ICdNb2QtU2hpZnQtMScsXG4gIH0sXG4gICdoMic6IHtcbiAgICBuYW1lOiAnaDInLFxuICAgIGFjdGlvbjogJ2gyJyxcbiAgICBpbm5lckhUTUw6IHN2Zy5oMixcbiAgICB0aXRsZTogJ0xldmVsIDIgaGVhZGluZycsXG4gICAgaG90a2V5OiAnTW9kLVNoaWZ0LTInLFxuICB9LFxuICAndWwnOiB7XG4gICAgbmFtZTogJ3VsJyxcbiAgICBhY3Rpb246ICd1bCcsXG4gICAgaW5uZXJIVE1MOiBzdmcudWwsXG4gICAgdGl0bGU6ICdCdWxsZXRlZCBsaXN0JyxcbiAgfSxcbiAgJ29sJzoge1xuICAgIG5hbWU6ICdvbCcsXG4gICAgYWN0aW9uOiAnb2wnLFxuICAgIGlubmVySFRNTDogc3ZnLm9sLFxuICAgIHRpdGxlOiAnTnVtYmVyZWQgbGlzdCcsXG4gIH0sXG4gICdibG9ja3F1b3RlJzoge1xuICAgIG5hbWU6ICdibG9ja3F1b3RlJyxcbiAgICBhY3Rpb246ICdibG9ja3F1b3RlJyxcbiAgICBpbm5lckhUTUw6IHN2Zy5ibG9ja3F1b3RlLFxuICAgIHRpdGxlOiAnUXVvdGUnLFxuICAgIGhvdGtleTogJ01vZDItU2hpZnQtUScsXG4gIH0sXG4gICdpbnNlcnRMaW5rJzoge1xuICAgIG5hbWU6ICdpbnNlcnRMaW5rJyxcbiAgICBhY3Rpb246IChlZGl0b3IpID0+IHtpZiAoZWRpdG9yLmlzSW5saW5lRm9ybWF0dGluZ0FsbG93ZWQoKSkgZWRpdG9yLndyYXBTZWxlY3Rpb24oJ1snLCAnXSgpJyl9LFxuICAgIGVuYWJsZWQ6IChlZGl0b3IsIGZvY3VzLCBhbmNob3IpID0+IGVkaXRvci5pc0lubGluZUZvcm1hdHRpbmdBbGxvd2VkKGZvY3VzLCBhbmNob3IpID8gZmFsc2UgOiBudWxsLFxuICAgIGlubmVySFRNTDogc3ZnLmxpbmssXG4gICAgdGl0bGU6ICdJbnNlcnQgbGluaycsXG4gICAgaG90a2V5OiAnTW9kLUsnLFxuICB9LFxuICAnaW5zZXJ0SW1hZ2UnOiB7XG4gICAgbmFtZTogJ2luc2VydEltYWdlJyxcbiAgICBhY3Rpb246IChlZGl0b3IpID0+IHtpZiAoZWRpdG9yLmlzSW5saW5lRm9ybWF0dGluZ0FsbG93ZWQoKSkgZWRpdG9yLndyYXBTZWxlY3Rpb24oJyFbJywgJ10oKScpfSxcbiAgICBlbmFibGVkOiAoZWRpdG9yLCBmb2N1cywgYW5jaG9yKSA9PiBlZGl0b3IuaXNJbmxpbmVGb3JtYXR0aW5nQWxsb3dlZChmb2N1cywgYW5jaG9yKSA/IGZhbHNlIDogbnVsbCxcbiAgICBpbm5lckhUTUw6IHN2Zy5pbWFnZSxcbiAgICB0aXRsZTogJ0luc2VydCBpbWFnZScsXG4gICAgaG90a2V5OiAnTW9kMi1TaGlmdC1JJyxcbiAgfSxcbiAgJ2hyJzoge1xuICAgIG5hbWU6ICdocicsXG4gICAgYWN0aW9uOiAoZWRpdG9yKSA9PiBlZGl0b3IucGFzdGUoJ1xcbioqKlxcbicpLFxuICAgIGVuYWJsZWQ6ICgpID0+IGZhbHNlLFxuICAgIGlubmVySFRNTDogc3ZnLmhyLFxuICAgIHRpdGxlOiAnSW5zZXJ0IGhvcml6b250YWwgbGluZScsXG4gICAgaG90a2V5OiAnTW9kMi1TaGlmdC1MJ1xuICB9XG59XG5cblxuY2xhc3MgQ29tbWFuZEJhciB7XG4gIGNvbnN0cnVjdG9yKHByb3BzKSB7XG4gICAgdGhpcy5lID0gbnVsbDtcbiAgICB0aGlzLmVkaXRvciA9IG51bGw7XG4gICAgdGhpcy5jb21tYW5kcyA9IFtdO1xuICAgIHRoaXMuYnV0dG9ucyA9IHt9O1xuICAgIHRoaXMuc3RhdGUgPSB7fTtcbiAgICB0aGlzLmhvdGtleXMgPSBbXTtcblxuICAgIGxldCBlbGVtZW50ID0gcHJvcHMuZWxlbWVudDtcbiAgICBpZiAoZWxlbWVudCAmJiAhZWxlbWVudC50YWdOYW1lKSB7XG4gICAgICBlbGVtZW50ID0gZG9jdW1lbnQuZ2V0RWxlbWVudEJ5SWQocHJvcHMuZWxlbWVudCk7XG4gICAgfVxuICAgIGlmICghZWxlbWVudCkge1xuICAgICAgZWxlbWVudCA9IGRvY3VtZW50LmJvZHk7IFxuICAgIH1cbiAgICB0aGlzLmNyZWF0ZUNvbW1hbmRCYXJFbGVtZW50KGVsZW1lbnQsIHByb3BzLmNvbW1hbmRzIHx8IFsnYm9sZCcsICdpdGFsaWMnLCAnc3RyaWtldGhyb3VnaCcsICd8JywgJ2NvZGUnLCAnfCcsICdoMScsICdoMicsICd8JywgJ3VsJywgJ29sJywgJ3wnLCAnYmxvY2txdW90ZScsICdocicsICd8JywgJ2luc2VydExpbmsnLCAnaW5zZXJ0SW1hZ2UnXSk7XG4gICAgZG9jdW1lbnQuYWRkRXZlbnRMaXN0ZW5lcigna2V5ZG93bicsIChlKSA9PiB0aGlzLmhhbmRsZUtleWRvd24oZSkpO1xuICAgIGlmIChwcm9wcy5lZGl0b3IpIHRoaXMuc2V0RWRpdG9yKHByb3BzLmVkaXRvcik7XG4gIH1cblxuICBjcmVhdGVDb21tYW5kQmFyRWxlbWVudChwYXJlbnRFbGVtZW50LCBjb21tYW5kcykge1xuICAgIHRoaXMuZSA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoJ2RpdicpO1xuICAgIHRoaXMuZS5jbGFzc05hbWUgPSAnVE1Db21tYW5kQmFyJztcblxuICAgIGZvciAobGV0IGNvbW1hbmQgb2YgY29tbWFuZHMpIHtcbiAgICAgIGlmIChjb21tYW5kID09ICd8Jykge1xuICAgICAgICBsZXQgZWwgPSBkb2N1bWVudC5jcmVhdGVFbGVtZW50KCdkaXYnKTtcbiAgICAgICAgZWwuY2xhc3NOYW1lID0gJ1RNQ29tbWFuZERpdmlkZXInO1xuICAgICAgICB0aGlzLmUuYXBwZW5kQ2hpbGQoZWwpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgbGV0IGNvbW1hbmROYW1lO1xuICAgICAgICBpZiAodHlwZW9mIGNvbW1hbmQgPT0gXCJzdHJpbmdcIikge1xuICAgICAgICAgIC8vIFJlZmVyZW5jZSB0byBkZWZhdWx0IGNvbW1hbmRcblxuICAgICAgICAgIGlmIChEZWZhdWx0Q29tbWFuZHNbY29tbWFuZF0pIHtcbiAgICAgICAgICAgIGNvbW1hbmROYW1lID0gY29tbWFuZDtcbiAgICAgICAgICAgIHRoaXMuY29tbWFuZHNbY29tbWFuZE5hbWVdID0gRGVmYXVsdENvbW1hbmRzW2NvbW1hbmROYW1lXTtcblxuICAgICAgICAgICAgXG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICAgIH1cbiAgICAgICAgICBcbiAgICAgICAgfSBlbHNlIGlmICh0eXBlb2YgY29tbWFuZCA9PSBcIm9iamVjdFwiICYmIGNvbW1hbmQubmFtZSkge1xuICAgICAgICAgIGNvbW1hbmROYW1lID0gY29tbWFuZC5uYW1lO1xuICAgICAgICAgIHRoaXMuY29tbWFuZHNbY29tbWFuZE5hbWVdID0ge307IFxuICAgICAgICAgIGlmIChEZWZhdWx0Q29tbWFuZHNbY29tbWFuZE5hbWVdKSBPYmplY3QuYXNzaWduKHRoaXMuY29tbWFuZHNbY29tbWFuZE5hbWVdLCBEZWZhdWx0Q29tbWFuZHNbY29tbWFuZE5hbWVdKTtcbiAgICAgICAgICBPYmplY3QuYXNzaWduKHRoaXMuY29tbWFuZHNbY29tbWFuZE5hbWVdLCBjb21tYW5kKTtcbiAgICAgICAgXG5cbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGxldCB0aXRsZSA9IHRoaXMuY29tbWFuZHNbY29tbWFuZE5hbWVdLnRpdGxlIHx8IGNvbW1hbmROYW1lO1xuXG4gICAgICAgIGlmICh0aGlzLmNvbW1hbmRzW2NvbW1hbmROYW1lXS5ob3RrZXkpIHtcbiAgICAgICAgICBjb25zdCBrZXlzID0gdGhpcy5jb21tYW5kc1tjb21tYW5kTmFtZV0uaG90a2V5LnNwbGl0KCctJyk7XG4gICAgICAgICAgLy8gY29uc3RydWN0IG1vZGlmaWVyc1xuICAgICAgICAgIGxldCBtb2RpZmllcnMgPSBbXTtcbiAgICAgICAgICBsZXQgbW9kaWZpZXJleHBsYW5hdGlvbiA9IFtdO1xuICAgICAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwga2V5cy5sZW5ndGggLSAxOyBpKyspIHtcbiAgICAgICAgICAgIHN3aXRjaCAoa2V5c1tpXSkge1xuICAgICAgICAgICAgICBjYXNlICdDdHJsJzogbW9kaWZpZXJzLnB1c2goJ2N0cmxLZXknKTsgbW9kaWZpZXJleHBsYW5hdGlvbi5wdXNoKCdDdHJsJyk7IGJyZWFrO1xuICAgICAgICAgICAgICBjYXNlICdDbWQnOiBtb2RpZmllcnMucHVzaCgnbWV0YUtleScpOyBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goJ+KMmCcpOyBicmVhaztcbiAgICAgICAgICAgICAgY2FzZSAnQWx0JzogbW9kaWZpZXJzLnB1c2goJ2FsdEtleScpOyBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goJ0FsdCcpOyBicmVhaztcbiAgICAgICAgICAgICAgY2FzZSAnT3B0aW9uJzogbW9kaWZpZXJzLnB1c2goJ2FsdEtleScpOyBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goJ+KMpScpOyBicmVhaztcbiAgICAgICAgICAgICAgY2FzZSAnV2luJzogbW9kaWZpZXJzLnB1c2goJ21ldGFLZXknKTsgbW9kaWZpZXJleHBsYW5hdGlvbi5wdXNoKCfiip4gV2luJyk7IGJyZWFrO1xuXG4gICAgICAgICAgICAgIGNhc2UgJ1NoaWZ0JzogIG1vZGlmaWVycy5wdXNoKCdzaGlmdEtleScpOyBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goJ+KHpycpOyBicmVhaztcblxuICAgICAgICAgICAgICBjYXNlICdNb2QnOiAvLyBNb2QgaXMgYSBjb252ZW5pZW5jZSBtZWNoYW5pc206IEN0cmwgb24gV2luZG93cywgQ21kIG9uIE1hY1xuICAgICAgICAgICAgICAgIGlmIChpc01hY0xpa2UpIHttb2RpZmllcnMucHVzaCgnbWV0YUtleScpOyBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goJ+KMmCcpO30gXG4gICAgICAgICAgICAgICAgZWxzZSB7bW9kaWZpZXJzLnB1c2goJ2N0cmxLZXknKTsgbW9kaWZpZXJleHBsYW5hdGlvbi5wdXNoKCdDdHJsJyk7fSBcbiAgICAgICAgICAgICAgICBicmVhazsgXG4gICAgICAgICAgICAgIGNhc2UgJ01vZDInOiBcbiAgICAgICAgICAgICAgICBtb2RpZmllcnMucHVzaCgnYWx0S2V5Jyk7IFxuICAgICAgICAgICAgICAgIGlmIChpc01hY0xpa2UpIG1vZGlmaWVyZXhwbGFuYXRpb24ucHVzaCgn4oylJyk7XG4gICAgICAgICAgICAgICAgZWxzZSBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goJ0FsdCcpO1xuICAgICAgICAgICAgICAgIGJyZWFrOyAvLyBNb2QyIGlzIGEgY29udmVuaWVuY2UgbWVjaGFuaXNtOiBBbHQgb24gV2luZG93cywgT3B0aW9uIG9uIE1hY1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgICBtb2RpZmllcmV4cGxhbmF0aW9uLnB1c2goa2V5c1trZXlzLmxlbmd0aCAtIDFdKTtcbiAgICAgICAgICBsZXQgaG90a2V5ID0ge1xuICAgICAgICAgICAgXG4gICAgICAgICAgICBtb2RpZmllcnM6IG1vZGlmaWVycyxcbiAgICAgICAgICAgIGNvbW1hbmQ6IGNvbW1hbmROYW1lLFxuICAgICAgICAgIH07XG4gICAgICAgICAgLy8gVE9ETyBSaWdodCBub3cgdGhpcyBpcyB3b3JraW5nIG9ubHkgZm9yIGxldHRlcnMgYW5kIG51bWJlcnNcbiAgICAgICAgICBpZiAoa2V5c1trZXlzLmxlbmd0aCAtIDFdLm1hdGNoKC9eWzAtOV0kLykpIHtcbiAgICAgICAgICAgIGhvdGtleS5jb2RlID0gYERpZ2l0JHtrZXlzW2tleXMubGVuZ3RoIC0gMV19YDtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgaG90a2V5LmtleSA9IGtleXNba2V5cy5sZW5ndGggLSAxXS50b0xvd2VyQ2FzZSgpO1xuICAgICAgICAgIH1cbiAgICAgICAgICB0aGlzLmhvdGtleXMucHVzaChob3RrZXkpO1xuICAgICAgICAgIHRpdGxlID0gdGl0bGUuY29uY2F0KGAgKCR7bW9kaWZpZXJleHBsYW5hdGlvbi5qb2luKCcrJyl9KWApO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5idXR0b25zW2NvbW1hbmROYW1lXSA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoJ2RpdicpO1xuICAgICAgICB0aGlzLmJ1dHRvbnNbY29tbWFuZE5hbWVdLmNsYXNzTmFtZSA9ICdUTUNvbW1hbmRCdXR0b24gVE1Db21tYW5kQnV0dG9uX0Rpc2FibGVkJztcbiAgICAgICAgdGhpcy5idXR0b25zW2NvbW1hbmROYW1lXS50aXRsZSA9IHRpdGxlO1xuICAgICAgICB0aGlzLmJ1dHRvbnNbY29tbWFuZE5hbWVdLmlubmVySFRNTCA9IHRoaXMuY29tbWFuZHNbY29tbWFuZE5hbWVdLmlubmVySFRNTDtcblxuICAgICAgICB0aGlzLmJ1dHRvbnNbY29tbWFuZE5hbWVdLmFkZEV2ZW50TGlzdGVuZXIoJ21vdXNlZG93bicsIChlKSA9PiB0aGlzLmhhbmRsZUNsaWNrKGNvbW1hbmROYW1lLCBlKSk7XG4gICAgICAgIHRoaXMuZS5hcHBlbmRDaGlsZCh0aGlzLmJ1dHRvbnNbY29tbWFuZE5hbWVdKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcGFyZW50RWxlbWVudC5hcHBlbmRDaGlsZCh0aGlzLmUpO1xuICB9XG5cbiAgaGFuZGxlQ2xpY2soY29tbWFuZE5hbWUsIGV2ZW50KSB7XG4gICAgaWYgKCF0aGlzLmVkaXRvcikgcmV0dXJuO1xuICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG4gICAgaWYgKHR5cGVvZiB0aGlzLmNvbW1hbmRzW2NvbW1hbmROYW1lXS5hY3Rpb24gPT0gXCJzdHJpbmdcIikge1xuICAgICAgaWYgKHRoaXMuc3RhdGVbY29tbWFuZE5hbWVdID09PSBmYWxzZSkgdGhpcy5lZGl0b3Iuc2V0Q29tbWFuZFN0YXRlKGNvbW1hbmROYW1lLCB0cnVlKTtcbiAgICAgIGVsc2UgdGhpcy5lZGl0b3Iuc2V0Q29tbWFuZFN0YXRlKGNvbW1hbmROYW1lLCBmYWxzZSk7ICBcbiAgICB9IGVsc2UgaWYgKHR5cGVvZiB0aGlzLmNvbW1hbmRzW2NvbW1hbmROYW1lXS5hY3Rpb24gPT0gXCJmdW5jdGlvblwiKSB7XG4gICAgICB0aGlzLmNvbW1hbmRzW2NvbW1hbmROYW1lXS5hY3Rpb24odGhpcy5lZGl0b3IpO1xuICAgIH1cbiAgfVxuXG4gIHNldEVkaXRvcihlZGl0b3IpIHtcbiAgICB0aGlzLmVkaXRvciA9IGVkaXRvcjtcbiAgICBlZGl0b3IuYWRkRXZlbnRMaXN0ZW5lcignc2VsZWN0aW9uJywgKGUpID0+IHRoaXMuaGFuZGxlU2VsZWN0aW9uKGUpKTtcbiAgfVxuXG4gIGhhbmRsZVNlbGVjdGlvbihldmVudCkge1xuICAgIGlmIChldmVudC5jb21tYW5kU3RhdGUpIHtcbiAgICAgIGZvciAobGV0IGNvbW1hbmQgaW4gdGhpcy5jb21tYW5kcykge1xuICAgICAgICBpZiAoZXZlbnQuY29tbWFuZFN0YXRlW2NvbW1hbmRdID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICBpZiAodGhpcy5jb21tYW5kc1tjb21tYW5kXS5lbmFibGVkKSB0aGlzLnN0YXRlW2NvbW1hbmRdID0gdGhpcy5jb21tYW5kc1tjb21tYW5kXS5lbmFibGVkKHRoaXMuZWRpdG9yLCBldmVudC5mb2N1cywgZXZlbnQuYW5jaG9yKTtcbiAgICAgICAgICBlbHNlIHRoaXMuc3RhdGVbY29tbWFuZF0gPSBldmVudC5mb2N1cyA/IGZhbHNlIDogbnVsbDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICB0aGlzLnN0YXRlW2NvbW1hbmRdID0gZXZlbnQuY29tbWFuZFN0YXRlW2NvbW1hbmRdO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHRoaXMuc3RhdGVbY29tbWFuZF0gPT09IHRydWUpIHtcbiAgICAgICAgICB0aGlzLmJ1dHRvbnNbY29tbWFuZF0uY2xhc3NOYW1lID0gJ1RNQ29tbWFuZEJ1dHRvbiBUTUNvbW1hbmRCdXR0b25fQWN0aXZlJztcbiAgICAgICAgfSBlbHNlIGlmICh0aGlzLnN0YXRlW2NvbW1hbmRdID09PSBmYWxzZSkge1xuICAgICAgICAgIHRoaXMuYnV0dG9uc1tjb21tYW5kXS5jbGFzc05hbWUgPSAnVE1Db21tYW5kQnV0dG9uIFRNQ29tbWFuZEJ1dHRvbl9JbmFjdGl2ZSc7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhpcy5idXR0b25zW2NvbW1hbmRdLmNsYXNzTmFtZSA9ICAnVE1Db21tYW5kQnV0dG9uIFRNQ29tbWFuZEJ1dHRvbl9EaXNhYmxlZCc7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBoYW5kbGVLZXlkb3duKGV2ZW50KSB7XG4gICAgb3V0ZXI6IGZvciAobGV0IGhvdGtleSBvZiB0aGlzLmhvdGtleXMpIHtcbiAgICAgIGlmICgoaG90a2V5LmtleSAmJiBldmVudC5rZXkudG9Mb3dlckNhc2UoKSA9PSBob3RrZXkua2V5KSB8fCAoaG90a2V5LmNvZGUgJiYgZXZlbnQuY29kZSA9PSBob3RrZXkuY29kZSkpIHtcbiAgICAgICAgLy8gS2V5IG1hdGNoZXMgaG90a2V5LiBMb29rIGZvciBhbnkgcmVxdWlyZWQgbW9kaWZpZXIgdGhhdCB3YXNuJ3QgcHJlc3NlZFxuICAgICAgICBmb3IgKGxldCBtb2RpZmllciBvZiBob3RrZXkubW9kaWZpZXJzKSB7XG4gICAgICAgICAgaWYgKCFldmVudFttb2RpZmllcl0pIGNvbnRpbnVlIG91dGVyO1xuICAgICAgICB9XG4gICAgICAgIC8vIEV2ZXJ5dGhpbmcgbWF0Y2hlcy5cbiAgICAgICAgdGhpcy5oYW5kbGVDbGljayhob3RrZXkuY29tbWFuZCwgZXZlbnQpO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgfVxuICB9XG59XG5cbmV4cG9ydCBkZWZhdWx0IENvbW1hbmRCYXI7IiwiJ3VzZSBzdHJpY3QnO1xudmFyIGNoZWNrID0gZnVuY3Rpb24gKGl0KSB7XG4gIHJldHVybiBpdCAmJiBpdC5NYXRoID09PSBNYXRoICYmIGl0O1xufTtcblxuLy8gaHR0cHM6Ly9naXRodWIuY29tL3psb2lyb2NrL2NvcmUtanMvaXNzdWVzLzg2I2lzc3VlY29tbWVudC0xMTU3NTkwMjhcbm1vZHVsZS5leHBvcnRzID1cbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLWdsb2JhbC10aGlzIC0tIHNhZmVcbiAgY2hlY2sodHlwZW9mIGdsb2JhbFRoaXMgPT0gJ29iamVjdCcgJiYgZ2xvYmFsVGhpcykgfHxcbiAgY2hlY2sodHlwZW9mIHdpbmRvdyA9PSAnb2JqZWN0JyAmJiB3aW5kb3cpIHx8XG4gIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1yZXN0cmljdGVkLWdsb2JhbHMgLS0gc2FmZVxuICBjaGVjayh0eXBlb2Ygc2VsZiA9PSAnb2JqZWN0JyAmJiBzZWxmKSB8fFxuICBjaGVjayh0eXBlb2YgZ2xvYmFsID09ICdvYmplY3QnICYmIGdsb2JhbCkgfHxcbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLW5ldy1mdW5jIC0tIGZhbGxiYWNrXG4gIChmdW5jdGlvbiAoKSB7IHJldHVybiB0aGlzOyB9KSgpIHx8IHRoaXMgfHwgRnVuY3Rpb24oJ3JldHVybiB0aGlzJykoKTtcbiIsIid1c2Ugc3RyaWN0Jztcbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKGV4ZWMpIHtcbiAgdHJ5IHtcbiAgICByZXR1cm4gISFleGVjKCk7XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgZmFpbHMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZmFpbHMnKTtcblxuLy8gRGV0ZWN0IElFOCdzIGluY29tcGxldGUgZGVmaW5lUHJvcGVydHkgaW1wbGVtZW50YXRpb25cbm1vZHVsZS5leHBvcnRzID0gIWZhaWxzKGZ1bmN0aW9uICgpIHtcbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLW9iamVjdC1kZWZpbmVwcm9wZXJ0eSAtLSByZXF1aXJlZCBmb3IgdGVzdGluZ1xuICByZXR1cm4gT2JqZWN0LmRlZmluZVByb3BlcnR5KHt9LCAxLCB7IGdldDogZnVuY3Rpb24gKCkgeyByZXR1cm4gNzsgfSB9KVsxXSAhPT0gNztcbn0pO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGZhaWxzID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2ZhaWxzJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gIWZhaWxzKGZ1bmN0aW9uICgpIHtcbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLWZ1bmN0aW9uLXByb3RvdHlwZS1iaW5kIC0tIHNhZmVcbiAgdmFyIHRlc3QgPSAoZnVuY3Rpb24gKCkgeyAvKiBlbXB0eSAqLyB9KS5iaW5kKCk7XG4gIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1wcm90b3R5cGUtYnVpbHRpbnMgLS0gc2FmZVxuICByZXR1cm4gdHlwZW9mIHRlc3QgIT0gJ2Z1bmN0aW9uJyB8fCB0ZXN0Lmhhc093blByb3BlcnR5KCdwcm90b3R5cGUnKTtcbn0pO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIE5BVElWRV9CSU5EID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2Z1bmN0aW9uLWJpbmQtbmF0aXZlJyk7XG5cbnZhciBGdW5jdGlvblByb3RvdHlwZSA9IEZ1bmN0aW9uLnByb3RvdHlwZTtcbnZhciBjYWxsID0gRnVuY3Rpb25Qcm90b3R5cGUuY2FsbDtcbnZhciB1bmN1cnJ5VGhpc1dpdGhCaW5kID0gTkFUSVZFX0JJTkQgJiYgRnVuY3Rpb25Qcm90b3R5cGUuYmluZC5iaW5kKGNhbGwsIGNhbGwpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IE5BVElWRV9CSU5EID8gdW5jdXJyeVRoaXNXaXRoQmluZCA6IGZ1bmN0aW9uIChmbikge1xuICByZXR1cm4gZnVuY3Rpb24gKCkge1xuICAgIHJldHVybiBjYWxsLmFwcGx5KGZuLCBhcmd1bWVudHMpO1xuICB9O1xufTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBkb2N1bWVudEFsbCA9IHR5cGVvZiBkb2N1bWVudCA9PSAnb2JqZWN0JyAmJiBkb2N1bWVudC5hbGw7XG5cbi8vIGh0dHBzOi8vdGMzOS5lcy9lY21hMjYyLyNzZWMtSXNIVE1MRERBLWludGVybmFsLXNsb3Rcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSB1bmljb3JuL25vLXR5cGVvZi11bmRlZmluZWQgLS0gcmVxdWlyZWQgZm9yIHRlc3RpbmdcbnZhciBJU19IVE1MRERBID0gdHlwZW9mIGRvY3VtZW50QWxsID09ICd1bmRlZmluZWQnICYmIGRvY3VtZW50QWxsICE9PSB1bmRlZmluZWQ7XG5cbm1vZHVsZS5leHBvcnRzID0ge1xuICBhbGw6IGRvY3VtZW50QWxsLFxuICBJU19IVE1MRERBOiBJU19IVE1MRERBXG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyICRkb2N1bWVudEFsbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9kb2N1bWVudC1hbGwnKTtcblxudmFyIGRvY3VtZW50QWxsID0gJGRvY3VtZW50QWxsLmFsbDtcblxuLy8gYElzQ2FsbGFibGVgIGFic3RyYWN0IG9wZXJhdGlvblxuLy8gaHR0cHM6Ly90YzM5LmVzL2VjbWEyNjIvI3NlYy1pc2NhbGxhYmxlXG5tb2R1bGUuZXhwb3J0cyA9ICRkb2N1bWVudEFsbC5JU19IVE1MRERBID8gZnVuY3Rpb24gKGFyZ3VtZW50KSB7XG4gIHJldHVybiB0eXBlb2YgYXJndW1lbnQgPT0gJ2Z1bmN0aW9uJyB8fCBhcmd1bWVudCA9PT0gZG9jdW1lbnRBbGw7XG59IDogZnVuY3Rpb24gKGFyZ3VtZW50KSB7XG4gIHJldHVybiB0eXBlb2YgYXJndW1lbnQgPT0gJ2Z1bmN0aW9uJztcbn07XG4iLCIndXNlIHN0cmljdCc7XG4vLyB3ZSBjYW4ndCB1c2UganVzdCBgaXQgPT0gbnVsbGAgc2luY2Ugb2YgYGRvY3VtZW50LmFsbGAgc3BlY2lhbCBjYXNlXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLUlzSFRNTEREQS1pbnRlcm5hbC1zbG90LWFlY1xubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiAoaXQpIHtcbiAgcmV0dXJuIGl0ID09PSBudWxsIHx8IGl0ID09PSB1bmRlZmluZWQ7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGlzTnVsbE9yVW5kZWZpbmVkID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLW51bGwtb3ItdW5kZWZpbmVkJyk7XG5cbnZhciAkVHlwZUVycm9yID0gVHlwZUVycm9yO1xuXG4vLyBgUmVxdWlyZU9iamVjdENvZXJjaWJsZWAgYWJzdHJhY3Qgb3BlcmF0aW9uXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLXJlcXVpcmVvYmplY3Rjb2VyY2libGVcbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKGl0KSB7XG4gIGlmIChpc051bGxPclVuZGVmaW5lZChpdCkpIHRocm93IG5ldyAkVHlwZUVycm9yKFwiQ2FuJ3QgY2FsbCBtZXRob2Qgb24gXCIgKyBpdCk7XG4gIHJldHVybiBpdDtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgcmVxdWlyZU9iamVjdENvZXJjaWJsZSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9yZXF1aXJlLW9iamVjdC1jb2VyY2libGUnKTtcblxudmFyICRPYmplY3QgPSBPYmplY3Q7XG5cbi8vIGBUb09iamVjdGAgYWJzdHJhY3Qgb3BlcmF0aW9uXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLXRvb2JqZWN0XG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChhcmd1bWVudCkge1xuICByZXR1cm4gJE9iamVjdChyZXF1aXJlT2JqZWN0Q29lcmNpYmxlKGFyZ3VtZW50KSk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIHVuY3VycnlUaGlzID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2Z1bmN0aW9uLXVuY3VycnktdGhpcycpO1xudmFyIHRvT2JqZWN0ID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL3RvLW9iamVjdCcpO1xuXG52YXIgaGFzT3duUHJvcGVydHkgPSB1bmN1cnJ5VGhpcyh7fS5oYXNPd25Qcm9wZXJ0eSk7XG5cbi8vIGBIYXNPd25Qcm9wZXJ0eWAgYWJzdHJhY3Qgb3BlcmF0aW9uXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLWhhc293bnByb3BlcnR5XG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgZXMvbm8tb2JqZWN0LWhhc293biAtLSBzYWZlXG5tb2R1bGUuZXhwb3J0cyA9IE9iamVjdC5oYXNPd24gfHwgZnVuY3Rpb24gaGFzT3duKGl0LCBrZXkpIHtcbiAgcmV0dXJuIGhhc093blByb3BlcnR5KHRvT2JqZWN0KGl0KSwga2V5KTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgREVTQ1JJUFRPUlMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVzY3JpcHRvcnMnKTtcbnZhciBoYXNPd24gPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaGFzLW93bi1wcm9wZXJ0eScpO1xuXG52YXIgRnVuY3Rpb25Qcm90b3R5cGUgPSBGdW5jdGlvbi5wcm90b3R5cGU7XG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgZXMvbm8tb2JqZWN0LWdldG93bnByb3BlcnR5ZGVzY3JpcHRvciAtLSBzYWZlXG52YXIgZ2V0RGVzY3JpcHRvciA9IERFU0NSSVBUT1JTICYmIE9iamVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3I7XG5cbnZhciBFWElTVFMgPSBoYXNPd24oRnVuY3Rpb25Qcm90b3R5cGUsICduYW1lJyk7XG4vLyBhZGRpdGlvbmFsIHByb3RlY3Rpb24gZnJvbSBtaW5pZmllZCAvIG1hbmdsZWQgLyBkcm9wcGVkIGZ1bmN0aW9uIG5hbWVzXG52YXIgUFJPUEVSID0gRVhJU1RTICYmIChmdW5jdGlvbiBzb21ldGhpbmcoKSB7IC8qIGVtcHR5ICovIH0pLm5hbWUgPT09ICdzb21ldGhpbmcnO1xudmFyIENPTkZJR1VSQUJMRSA9IEVYSVNUUyAmJiAoIURFU0NSSVBUT1JTIHx8IChERVNDUklQVE9SUyAmJiBnZXREZXNjcmlwdG9yKEZ1bmN0aW9uUHJvdG90eXBlLCAnbmFtZScpLmNvbmZpZ3VyYWJsZSkpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IHtcbiAgRVhJU1RTOiBFWElTVFMsXG4gIFBST1BFUjogUFJPUEVSLFxuICBDT05GSUdVUkFCTEU6IENPTkZJR1VSQUJMRVxufTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBnbG9iYWwgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZ2xvYmFsJyk7XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBlcy9uby1vYmplY3QtZGVmaW5lcHJvcGVydHkgLS0gc2FmZVxudmFyIGRlZmluZVByb3BlcnR5ID0gT2JqZWN0LmRlZmluZVByb3BlcnR5O1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChrZXksIHZhbHVlKSB7XG4gIHRyeSB7XG4gICAgZGVmaW5lUHJvcGVydHkoZ2xvYmFsLCBrZXksIHsgdmFsdWU6IHZhbHVlLCBjb25maWd1cmFibGU6IHRydWUsIHdyaXRhYmxlOiB0cnVlIH0pO1xuICB9IGNhdGNoIChlcnJvcikge1xuICAgIGdsb2JhbFtrZXldID0gdmFsdWU7XG4gIH0gcmV0dXJuIHZhbHVlO1xufTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBnbG9iYWwgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZ2xvYmFsJyk7XG52YXIgZGVmaW5lR2xvYmFsUHJvcGVydHkgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVmaW5lLWdsb2JhbC1wcm9wZXJ0eScpO1xuXG52YXIgU0hBUkVEID0gJ19fY29yZS1qc19zaGFyZWRfXyc7XG52YXIgc3RvcmUgPSBnbG9iYWxbU0hBUkVEXSB8fCBkZWZpbmVHbG9iYWxQcm9wZXJ0eShTSEFSRUQsIHt9KTtcblxubW9kdWxlLmV4cG9ydHMgPSBzdG9yZTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciB1bmN1cnJ5VGhpcyA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9mdW5jdGlvbi11bmN1cnJ5LXRoaXMnKTtcbnZhciBpc0NhbGxhYmxlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLWNhbGxhYmxlJyk7XG52YXIgc3RvcmUgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvc2hhcmVkLXN0b3JlJyk7XG5cbnZhciBmdW5jdGlvblRvU3RyaW5nID0gdW5jdXJyeVRoaXMoRnVuY3Rpb24udG9TdHJpbmcpO1xuXG4vLyB0aGlzIGhlbHBlciBicm9rZW4gaW4gYGNvcmUtanNAMy40LjEtMy40LjRgLCBzbyB3ZSBjYW4ndCB1c2UgYHNoYXJlZGAgaGVscGVyXG5pZiAoIWlzQ2FsbGFibGUoc3RvcmUuaW5zcGVjdFNvdXJjZSkpIHtcbiAgc3RvcmUuaW5zcGVjdFNvdXJjZSA9IGZ1bmN0aW9uIChpdCkge1xuICAgIHJldHVybiBmdW5jdGlvblRvU3RyaW5nKGl0KTtcbiAgfTtcbn1cblxubW9kdWxlLmV4cG9ydHMgPSBzdG9yZS5pbnNwZWN0U291cmNlO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGdsb2JhbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9nbG9iYWwnKTtcbnZhciBpc0NhbGxhYmxlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLWNhbGxhYmxlJyk7XG5cbnZhciBXZWFrTWFwID0gZ2xvYmFsLldlYWtNYXA7XG5cbm1vZHVsZS5leHBvcnRzID0gaXNDYWxsYWJsZShXZWFrTWFwKSAmJiAvbmF0aXZlIGNvZGUvLnRlc3QoU3RyaW5nKFdlYWtNYXApKTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBpc0NhbGxhYmxlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLWNhbGxhYmxlJyk7XG52YXIgJGRvY3VtZW50QWxsID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2RvY3VtZW50LWFsbCcpO1xuXG52YXIgZG9jdW1lbnRBbGwgPSAkZG9jdW1lbnRBbGwuYWxsO1xuXG5tb2R1bGUuZXhwb3J0cyA9ICRkb2N1bWVudEFsbC5JU19IVE1MRERBID8gZnVuY3Rpb24gKGl0KSB7XG4gIHJldHVybiB0eXBlb2YgaXQgPT0gJ29iamVjdCcgPyBpdCAhPT0gbnVsbCA6IGlzQ2FsbGFibGUoaXQpIHx8IGl0ID09PSBkb2N1bWVudEFsbDtcbn0gOiBmdW5jdGlvbiAoaXQpIHtcbiAgcmV0dXJuIHR5cGVvZiBpdCA9PSAnb2JqZWN0JyA/IGl0ICE9PSBudWxsIDogaXNDYWxsYWJsZShpdCk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGdsb2JhbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9nbG9iYWwnKTtcbnZhciBpc09iamVjdCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9pcy1vYmplY3QnKTtcblxudmFyIGRvY3VtZW50ID0gZ2xvYmFsLmRvY3VtZW50O1xuLy8gdHlwZW9mIGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQgaXMgJ29iamVjdCcgaW4gb2xkIElFXG52YXIgRVhJU1RTID0gaXNPYmplY3QoZG9jdW1lbnQpICYmIGlzT2JqZWN0KGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQpO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChpdCkge1xuICByZXR1cm4gRVhJU1RTID8gZG9jdW1lbnQuY3JlYXRlRWxlbWVudChpdCkgOiB7fTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgREVTQ1JJUFRPUlMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVzY3JpcHRvcnMnKTtcbnZhciBmYWlscyA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9mYWlscycpO1xudmFyIGNyZWF0ZUVsZW1lbnQgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZG9jdW1lbnQtY3JlYXRlLWVsZW1lbnQnKTtcblxuLy8gVGhhbmtzIHRvIElFOCBmb3IgaXRzIGZ1bm55IGRlZmluZVByb3BlcnR5XG5tb2R1bGUuZXhwb3J0cyA9ICFERVNDUklQVE9SUyAmJiAhZmFpbHMoZnVuY3Rpb24gKCkge1xuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgZXMvbm8tb2JqZWN0LWRlZmluZXByb3BlcnR5IC0tIHJlcXVpcmVkIGZvciB0ZXN0aW5nXG4gIHJldHVybiBPYmplY3QuZGVmaW5lUHJvcGVydHkoY3JlYXRlRWxlbWVudCgnZGl2JyksICdhJywge1xuICAgIGdldDogZnVuY3Rpb24gKCkgeyByZXR1cm4gNzsgfVxuICB9KS5hICE9PSA3O1xufSk7XG4iLCIndXNlIHN0cmljdCc7XG52YXIgREVTQ1JJUFRPUlMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVzY3JpcHRvcnMnKTtcbnZhciBmYWlscyA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9mYWlscycpO1xuXG4vLyBWOCB+IENocm9tZSAzNi1cbi8vIGh0dHBzOi8vYnVncy5jaHJvbWl1bS5vcmcvcC92OC9pc3N1ZXMvZGV0YWlsP2lkPTMzMzRcbm1vZHVsZS5leHBvcnRzID0gREVTQ1JJUFRPUlMgJiYgZmFpbHMoZnVuY3Rpb24gKCkge1xuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgZXMvbm8tb2JqZWN0LWRlZmluZXByb3BlcnR5IC0tIHJlcXVpcmVkIGZvciB0ZXN0aW5nXG4gIHJldHVybiBPYmplY3QuZGVmaW5lUHJvcGVydHkoZnVuY3Rpb24gKCkgeyAvKiBlbXB0eSAqLyB9LCAncHJvdG90eXBlJywge1xuICAgIHZhbHVlOiA0MixcbiAgICB3cml0YWJsZTogZmFsc2VcbiAgfSkucHJvdG90eXBlICE9PSA0Mjtcbn0pO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGlzT2JqZWN0ID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLW9iamVjdCcpO1xuXG52YXIgJFN0cmluZyA9IFN0cmluZztcbnZhciAkVHlwZUVycm9yID0gVHlwZUVycm9yO1xuXG4vLyBgQXNzZXJ0OiBUeXBlKGFyZ3VtZW50KSBpcyBPYmplY3RgXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChhcmd1bWVudCkge1xuICBpZiAoaXNPYmplY3QoYXJndW1lbnQpKSByZXR1cm4gYXJndW1lbnQ7XG4gIHRocm93IG5ldyAkVHlwZUVycm9yKCRTdHJpbmcoYXJndW1lbnQpICsgJyBpcyBub3QgYW4gb2JqZWN0Jyk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIE5BVElWRV9CSU5EID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2Z1bmN0aW9uLWJpbmQtbmF0aXZlJyk7XG5cbnZhciBjYWxsID0gRnVuY3Rpb24ucHJvdG90eXBlLmNhbGw7XG5cbm1vZHVsZS5leHBvcnRzID0gTkFUSVZFX0JJTkQgPyBjYWxsLmJpbmQoY2FsbCkgOiBmdW5jdGlvbiAoKSB7XG4gIHJldHVybiBjYWxsLmFwcGx5KGNhbGwsIGFyZ3VtZW50cyk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGdsb2JhbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9nbG9iYWwnKTtcbnZhciBpc0NhbGxhYmxlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLWNhbGxhYmxlJyk7XG5cbnZhciBhRnVuY3Rpb24gPSBmdW5jdGlvbiAoYXJndW1lbnQpIHtcbiAgcmV0dXJuIGlzQ2FsbGFibGUoYXJndW1lbnQpID8gYXJndW1lbnQgOiB1bmRlZmluZWQ7XG59O1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChuYW1lc3BhY2UsIG1ldGhvZCkge1xuICByZXR1cm4gYXJndW1lbnRzLmxlbmd0aCA8IDIgPyBhRnVuY3Rpb24oZ2xvYmFsW25hbWVzcGFjZV0pIDogZ2xvYmFsW25hbWVzcGFjZV0gJiYgZ2xvYmFsW25hbWVzcGFjZV1bbWV0aG9kXTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgdW5jdXJyeVRoaXMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZnVuY3Rpb24tdW5jdXJyeS10aGlzJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gdW5jdXJyeVRoaXMoe30uaXNQcm90b3R5cGVPZik7XG4iLCIndXNlIHN0cmljdCc7XG5tb2R1bGUuZXhwb3J0cyA9IHR5cGVvZiBuYXZpZ2F0b3IgIT0gJ3VuZGVmaW5lZCcgJiYgU3RyaW5nKG5hdmlnYXRvci51c2VyQWdlbnQpIHx8ICcnO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGdsb2JhbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9nbG9iYWwnKTtcbnZhciB1c2VyQWdlbnQgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZW5naW5lLXVzZXItYWdlbnQnKTtcblxudmFyIHByb2Nlc3MgPSBnbG9iYWwucHJvY2VzcztcbnZhciBEZW5vID0gZ2xvYmFsLkRlbm87XG52YXIgdmVyc2lvbnMgPSBwcm9jZXNzICYmIHByb2Nlc3MudmVyc2lvbnMgfHwgRGVubyAmJiBEZW5vLnZlcnNpb247XG52YXIgdjggPSB2ZXJzaW9ucyAmJiB2ZXJzaW9ucy52ODtcbnZhciBtYXRjaCwgdmVyc2lvbjtcblxuaWYgKHY4KSB7XG4gIG1hdGNoID0gdjguc3BsaXQoJy4nKTtcbiAgLy8gaW4gb2xkIENocm9tZSwgdmVyc2lvbnMgb2YgVjggaXNuJ3QgVjggPSBDaHJvbWUgLyAxMFxuICAvLyBidXQgdGhlaXIgY29ycmVjdCB2ZXJzaW9ucyBhcmUgbm90IGludGVyZXN0aW5nIGZvciB1c1xuICB2ZXJzaW9uID0gbWF0Y2hbMF0gPiAwICYmIG1hdGNoWzBdIDwgNCA/IDEgOiArKG1hdGNoWzBdICsgbWF0Y2hbMV0pO1xufVxuXG4vLyBCcm93c2VyRlMgTm9kZUpTIGBwcm9jZXNzYCBwb2x5ZmlsbCBpbmNvcnJlY3RseSBzZXQgYC52OGAgdG8gYDAuMGBcbi8vIHNvIGNoZWNrIGB1c2VyQWdlbnRgIGV2ZW4gaWYgYC52OGAgZXhpc3RzLCBidXQgMFxuaWYgKCF2ZXJzaW9uICYmIHVzZXJBZ2VudCkge1xuICBtYXRjaCA9IHVzZXJBZ2VudC5tYXRjaCgvRWRnZVxcLyhcXGQrKS8pO1xuICBpZiAoIW1hdGNoIHx8IG1hdGNoWzFdID49IDc0KSB7XG4gICAgbWF0Y2ggPSB1c2VyQWdlbnQubWF0Y2goL0Nocm9tZVxcLyhcXGQrKS8pO1xuICAgIGlmIChtYXRjaCkgdmVyc2lvbiA9ICttYXRjaFsxXTtcbiAgfVxufVxuXG5tb2R1bGUuZXhwb3J0cyA9IHZlcnNpb247XG4iLCIndXNlIHN0cmljdCc7XG4vKiBlc2xpbnQtZGlzYWJsZSBlcy9uby1zeW1ib2wgLS0gcmVxdWlyZWQgZm9yIHRlc3RpbmcgKi9cbnZhciBWOF9WRVJTSU9OID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2VuZ2luZS12OC12ZXJzaW9uJyk7XG52YXIgZmFpbHMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZmFpbHMnKTtcbnZhciBnbG9iYWwgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZ2xvYmFsJyk7XG5cbnZhciAkU3RyaW5nID0gZ2xvYmFsLlN0cmluZztcblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLW9iamVjdC1nZXRvd25wcm9wZXJ0eXN5bWJvbHMgLS0gcmVxdWlyZWQgZm9yIHRlc3Rpbmdcbm1vZHVsZS5leHBvcnRzID0gISFPYmplY3QuZ2V0T3duUHJvcGVydHlTeW1ib2xzICYmICFmYWlscyhmdW5jdGlvbiAoKSB7XG4gIHZhciBzeW1ib2wgPSBTeW1ib2woJ3N5bWJvbCBkZXRlY3Rpb24nKTtcbiAgLy8gQ2hyb21lIDM4IFN5bWJvbCBoYXMgaW5jb3JyZWN0IHRvU3RyaW5nIGNvbnZlcnNpb25cbiAgLy8gYGdldC1vd24tcHJvcGVydHktc3ltYm9sc2AgcG9seWZpbGwgc3ltYm9scyBjb252ZXJ0ZWQgdG8gb2JqZWN0IGFyZSBub3QgU3ltYm9sIGluc3RhbmNlc1xuICAvLyBuYjogRG8gbm90IGNhbGwgYFN0cmluZ2AgZGlyZWN0bHkgdG8gYXZvaWQgdGhpcyBiZWluZyBvcHRpbWl6ZWQgb3V0IHRvIGBzeW1ib2wrJydgIHdoaWNoIHdpbGwsXG4gIC8vIG9mIGNvdXJzZSwgZmFpbC5cbiAgcmV0dXJuICEkU3RyaW5nKHN5bWJvbCkgfHwgIShPYmplY3Qoc3ltYm9sKSBpbnN0YW5jZW9mIFN5bWJvbCkgfHxcbiAgICAvLyBDaHJvbWUgMzgtNDAgc3ltYm9scyBhcmUgbm90IGluaGVyaXRlZCBmcm9tIERPTSBjb2xsZWN0aW9ucyBwcm90b3R5cGVzIHRvIGluc3RhbmNlc1xuICAgICFTeW1ib2wuc2hhbSAmJiBWOF9WRVJTSU9OICYmIFY4X1ZFUlNJT04gPCA0MTtcbn0pO1xuIiwiJ3VzZSBzdHJpY3QnO1xuLyogZXNsaW50LWRpc2FibGUgZXMvbm8tc3ltYm9sIC0tIHJlcXVpcmVkIGZvciB0ZXN0aW5nICovXG52YXIgTkFUSVZFX1NZTUJPTCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9zeW1ib2wtY29uc3RydWN0b3ItZGV0ZWN0aW9uJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gTkFUSVZFX1NZTUJPTFxuICAmJiAhU3ltYm9sLnNoYW1cbiAgJiYgdHlwZW9mIFN5bWJvbC5pdGVyYXRvciA9PSAnc3ltYm9sJztcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBnZXRCdWlsdEluID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2dldC1idWlsdC1pbicpO1xudmFyIGlzQ2FsbGFibGUgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaXMtY2FsbGFibGUnKTtcbnZhciBpc1Byb3RvdHlwZU9mID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL29iamVjdC1pcy1wcm90b3R5cGUtb2YnKTtcbnZhciBVU0VfU1lNQk9MX0FTX1VJRCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy91c2Utc3ltYm9sLWFzLXVpZCcpO1xuXG52YXIgJE9iamVjdCA9IE9iamVjdDtcblxubW9kdWxlLmV4cG9ydHMgPSBVU0VfU1lNQk9MX0FTX1VJRCA/IGZ1bmN0aW9uIChpdCkge1xuICByZXR1cm4gdHlwZW9mIGl0ID09ICdzeW1ib2wnO1xufSA6IGZ1bmN0aW9uIChpdCkge1xuICB2YXIgJFN5bWJvbCA9IGdldEJ1aWx0SW4oJ1N5bWJvbCcpO1xuICByZXR1cm4gaXNDYWxsYWJsZSgkU3ltYm9sKSAmJiBpc1Byb3RvdHlwZU9mKCRTeW1ib2wucHJvdG90eXBlLCAkT2JqZWN0KGl0KSk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyICRTdHJpbmcgPSBTdHJpbmc7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKGFyZ3VtZW50KSB7XG4gIHRyeSB7XG4gICAgcmV0dXJuICRTdHJpbmcoYXJndW1lbnQpO1xuICB9IGNhdGNoIChlcnJvcikge1xuICAgIHJldHVybiAnT2JqZWN0JztcbiAgfVxufTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBpc0NhbGxhYmxlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLWNhbGxhYmxlJyk7XG52YXIgdHJ5VG9TdHJpbmcgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvdHJ5LXRvLXN0cmluZycpO1xuXG52YXIgJFR5cGVFcnJvciA9IFR5cGVFcnJvcjtcblxuLy8gYEFzc2VydDogSXNDYWxsYWJsZShhcmd1bWVudCkgaXMgdHJ1ZWBcbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKGFyZ3VtZW50KSB7XG4gIGlmIChpc0NhbGxhYmxlKGFyZ3VtZW50KSkgcmV0dXJuIGFyZ3VtZW50O1xuICB0aHJvdyBuZXcgJFR5cGVFcnJvcih0cnlUb1N0cmluZyhhcmd1bWVudCkgKyAnIGlzIG5vdCBhIGZ1bmN0aW9uJyk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGFDYWxsYWJsZSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9hLWNhbGxhYmxlJyk7XG52YXIgaXNOdWxsT3JVbmRlZmluZWQgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaXMtbnVsbC1vci11bmRlZmluZWQnKTtcblxuLy8gYEdldE1ldGhvZGAgYWJzdHJhY3Qgb3BlcmF0aW9uXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLWdldG1ldGhvZFxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiAoViwgUCkge1xuICB2YXIgZnVuYyA9IFZbUF07XG4gIHJldHVybiBpc051bGxPclVuZGVmaW5lZChmdW5jKSA/IHVuZGVmaW5lZCA6IGFDYWxsYWJsZShmdW5jKTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgY2FsbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9mdW5jdGlvbi1jYWxsJyk7XG52YXIgaXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9pcy1jYWxsYWJsZScpO1xudmFyIGlzT2JqZWN0ID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2lzLW9iamVjdCcpO1xuXG52YXIgJFR5cGVFcnJvciA9IFR5cGVFcnJvcjtcblxuLy8gYE9yZGluYXJ5VG9QcmltaXRpdmVgIGFic3RyYWN0IG9wZXJhdGlvblxuLy8gaHR0cHM6Ly90YzM5LmVzL2VjbWEyNjIvI3NlYy1vcmRpbmFyeXRvcHJpbWl0aXZlXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChpbnB1dCwgcHJlZikge1xuICB2YXIgZm4sIHZhbDtcbiAgaWYgKHByZWYgPT09ICdzdHJpbmcnICYmIGlzQ2FsbGFibGUoZm4gPSBpbnB1dC50b1N0cmluZykgJiYgIWlzT2JqZWN0KHZhbCA9IGNhbGwoZm4sIGlucHV0KSkpIHJldHVybiB2YWw7XG4gIGlmIChpc0NhbGxhYmxlKGZuID0gaW5wdXQudmFsdWVPZikgJiYgIWlzT2JqZWN0KHZhbCA9IGNhbGwoZm4sIGlucHV0KSkpIHJldHVybiB2YWw7XG4gIGlmIChwcmVmICE9PSAnc3RyaW5nJyAmJiBpc0NhbGxhYmxlKGZuID0gaW5wdXQudG9TdHJpbmcpICYmICFpc09iamVjdCh2YWwgPSBjYWxsKGZuLCBpbnB1dCkpKSByZXR1cm4gdmFsO1xuICB0aHJvdyBuZXcgJFR5cGVFcnJvcihcIkNhbid0IGNvbnZlcnQgb2JqZWN0IHRvIHByaW1pdGl2ZSB2YWx1ZVwiKTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgSVNfUFVSRSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9pcy1wdXJlJyk7XG52YXIgc3RvcmUgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvc2hhcmVkLXN0b3JlJyk7XG5cbihtb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChrZXksIHZhbHVlKSB7XG4gIHJldHVybiBzdG9yZVtrZXldIHx8IChzdG9yZVtrZXldID0gdmFsdWUgIT09IHVuZGVmaW5lZCA/IHZhbHVlIDoge30pO1xufSkoJ3ZlcnNpb25zJywgW10pLnB1c2goe1xuICB2ZXJzaW9uOiAnMy4zMy4wJyxcbiAgbW9kZTogSVNfUFVSRSA/ICdwdXJlJyA6ICdnbG9iYWwnLFxuICBjb3B5cmlnaHQ6ICfCqSAyMDE0LTIwMjMgRGVuaXMgUHVzaGthcmV2ICh6bG9pcm9jay5ydSknLFxuICBsaWNlbnNlOiAnaHR0cHM6Ly9naXRodWIuY29tL3psb2lyb2NrL2NvcmUtanMvYmxvYi92My4zMy4wL0xJQ0VOU0UnLFxuICBzb3VyY2U6ICdodHRwczovL2dpdGh1Yi5jb20vemxvaXJvY2svY29yZS1qcydcbn0pO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIHVuY3VycnlUaGlzID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2Z1bmN0aW9uLXVuY3VycnktdGhpcycpO1xuXG52YXIgaWQgPSAwO1xudmFyIHBvc3RmaXggPSBNYXRoLnJhbmRvbSgpO1xudmFyIHRvU3RyaW5nID0gdW5jdXJyeVRoaXMoMS4wLnRvU3RyaW5nKTtcblxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiAoa2V5KSB7XG4gIHJldHVybiAnU3ltYm9sKCcgKyAoa2V5ID09PSB1bmRlZmluZWQgPyAnJyA6IGtleSkgKyAnKV8nICsgdG9TdHJpbmcoKytpZCArIHBvc3RmaXgsIDM2KTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgZ2xvYmFsID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2dsb2JhbCcpO1xudmFyIHNoYXJlZCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9zaGFyZWQnKTtcbnZhciBoYXNPd24gPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaGFzLW93bi1wcm9wZXJ0eScpO1xudmFyIHVpZCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy91aWQnKTtcbnZhciBOQVRJVkVfU1lNQk9MID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL3N5bWJvbC1jb25zdHJ1Y3Rvci1kZXRlY3Rpb24nKTtcbnZhciBVU0VfU1lNQk9MX0FTX1VJRCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy91c2Utc3ltYm9sLWFzLXVpZCcpO1xuXG52YXIgU3ltYm9sID0gZ2xvYmFsLlN5bWJvbDtcbnZhciBXZWxsS25vd25TeW1ib2xzU3RvcmUgPSBzaGFyZWQoJ3drcycpO1xudmFyIGNyZWF0ZVdlbGxLbm93blN5bWJvbCA9IFVTRV9TWU1CT0xfQVNfVUlEID8gU3ltYm9sWydmb3InXSB8fCBTeW1ib2wgOiBTeW1ib2wgJiYgU3ltYm9sLndpdGhvdXRTZXR0ZXIgfHwgdWlkO1xuXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChuYW1lKSB7XG4gIGlmICghaGFzT3duKFdlbGxLbm93blN5bWJvbHNTdG9yZSwgbmFtZSkpIHtcbiAgICBXZWxsS25vd25TeW1ib2xzU3RvcmVbbmFtZV0gPSBOQVRJVkVfU1lNQk9MICYmIGhhc093bihTeW1ib2wsIG5hbWUpXG4gICAgICA/IFN5bWJvbFtuYW1lXVxuICAgICAgOiBjcmVhdGVXZWxsS25vd25TeW1ib2woJ1N5bWJvbC4nICsgbmFtZSk7XG4gIH0gcmV0dXJuIFdlbGxLbm93blN5bWJvbHNTdG9yZVtuYW1lXTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgY2FsbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9mdW5jdGlvbi1jYWxsJyk7XG52YXIgaXNPYmplY3QgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaXMtb2JqZWN0Jyk7XG52YXIgaXNTeW1ib2wgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaXMtc3ltYm9sJyk7XG52YXIgZ2V0TWV0aG9kID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2dldC1tZXRob2QnKTtcbnZhciBvcmRpbmFyeVRvUHJpbWl0aXZlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL29yZGluYXJ5LXRvLXByaW1pdGl2ZScpO1xudmFyIHdlbGxLbm93blN5bWJvbCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy93ZWxsLWtub3duLXN5bWJvbCcpO1xuXG52YXIgJFR5cGVFcnJvciA9IFR5cGVFcnJvcjtcbnZhciBUT19QUklNSVRJVkUgPSB3ZWxsS25vd25TeW1ib2woJ3RvUHJpbWl0aXZlJyk7XG5cbi8vIGBUb1ByaW1pdGl2ZWAgYWJzdHJhY3Qgb3BlcmF0aW9uXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLXRvcHJpbWl0aXZlXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uIChpbnB1dCwgcHJlZikge1xuICBpZiAoIWlzT2JqZWN0KGlucHV0KSB8fCBpc1N5bWJvbChpbnB1dCkpIHJldHVybiBpbnB1dDtcbiAgdmFyIGV4b3RpY1RvUHJpbSA9IGdldE1ldGhvZChpbnB1dCwgVE9fUFJJTUlUSVZFKTtcbiAgdmFyIHJlc3VsdDtcbiAgaWYgKGV4b3RpY1RvUHJpbSkge1xuICAgIGlmIChwcmVmID09PSB1bmRlZmluZWQpIHByZWYgPSAnZGVmYXVsdCc7XG4gICAgcmVzdWx0ID0gY2FsbChleG90aWNUb1ByaW0sIGlucHV0LCBwcmVmKTtcbiAgICBpZiAoIWlzT2JqZWN0KHJlc3VsdCkgfHwgaXNTeW1ib2wocmVzdWx0KSkgcmV0dXJuIHJlc3VsdDtcbiAgICB0aHJvdyBuZXcgJFR5cGVFcnJvcihcIkNhbid0IGNvbnZlcnQgb2JqZWN0IHRvIHByaW1pdGl2ZSB2YWx1ZVwiKTtcbiAgfVxuICBpZiAocHJlZiA9PT0gdW5kZWZpbmVkKSBwcmVmID0gJ251bWJlcic7XG4gIHJldHVybiBvcmRpbmFyeVRvUHJpbWl0aXZlKGlucHV0LCBwcmVmKTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgdG9QcmltaXRpdmUgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvdG8tcHJpbWl0aXZlJyk7XG52YXIgaXNTeW1ib2wgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaXMtc3ltYm9sJyk7XG5cbi8vIGBUb1Byb3BlcnR5S2V5YCBhYnN0cmFjdCBvcGVyYXRpb25cbi8vIGh0dHBzOi8vdGMzOS5lcy9lY21hMjYyLyNzZWMtdG9wcm9wZXJ0eWtleVxubW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiAoYXJndW1lbnQpIHtcbiAgdmFyIGtleSA9IHRvUHJpbWl0aXZlKGFyZ3VtZW50LCAnc3RyaW5nJyk7XG4gIHJldHVybiBpc1N5bWJvbChrZXkpID8ga2V5IDoga2V5ICsgJyc7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIERFU0NSSVBUT1JTID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2Rlc2NyaXB0b3JzJyk7XG52YXIgSUU4X0RPTV9ERUZJTkUgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaWU4LWRvbS1kZWZpbmUnKTtcbnZhciBWOF9QUk9UT1RZUEVfREVGSU5FX0JVRyA9IHJlcXVpcmUoJy4uL2ludGVybmFscy92OC1wcm90b3R5cGUtZGVmaW5lLWJ1ZycpO1xudmFyIGFuT2JqZWN0ID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2FuLW9iamVjdCcpO1xudmFyIHRvUHJvcGVydHlLZXkgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvdG8tcHJvcGVydHkta2V5Jyk7XG5cbnZhciAkVHlwZUVycm9yID0gVHlwZUVycm9yO1xuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLW9iamVjdC1kZWZpbmVwcm9wZXJ0eSAtLSBzYWZlXG52YXIgJGRlZmluZVByb3BlcnR5ID0gT2JqZWN0LmRlZmluZVByb3BlcnR5O1xuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLW9iamVjdC1nZXRvd25wcm9wZXJ0eWRlc2NyaXB0b3IgLS0gc2FmZVxudmFyICRnZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IgPSBPYmplY3QuZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yO1xudmFyIEVOVU1FUkFCTEUgPSAnZW51bWVyYWJsZSc7XG52YXIgQ09ORklHVVJBQkxFID0gJ2NvbmZpZ3VyYWJsZSc7XG52YXIgV1JJVEFCTEUgPSAnd3JpdGFibGUnO1xuXG4vLyBgT2JqZWN0LmRlZmluZVByb3BlcnR5YCBtZXRob2Rcbi8vIGh0dHBzOi8vdGMzOS5lcy9lY21hMjYyLyNzZWMtb2JqZWN0LmRlZmluZXByb3BlcnR5XG5leHBvcnRzLmYgPSBERVNDUklQVE9SUyA/IFY4X1BST1RPVFlQRV9ERUZJTkVfQlVHID8gZnVuY3Rpb24gZGVmaW5lUHJvcGVydHkoTywgUCwgQXR0cmlidXRlcykge1xuICBhbk9iamVjdChPKTtcbiAgUCA9IHRvUHJvcGVydHlLZXkoUCk7XG4gIGFuT2JqZWN0KEF0dHJpYnV0ZXMpO1xuICBpZiAodHlwZW9mIE8gPT09ICdmdW5jdGlvbicgJiYgUCA9PT0gJ3Byb3RvdHlwZScgJiYgJ3ZhbHVlJyBpbiBBdHRyaWJ1dGVzICYmIFdSSVRBQkxFIGluIEF0dHJpYnV0ZXMgJiYgIUF0dHJpYnV0ZXNbV1JJVEFCTEVdKSB7XG4gICAgdmFyIGN1cnJlbnQgPSAkZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yKE8sIFApO1xuICAgIGlmIChjdXJyZW50ICYmIGN1cnJlbnRbV1JJVEFCTEVdKSB7XG4gICAgICBPW1BdID0gQXR0cmlidXRlcy52YWx1ZTtcbiAgICAgIEF0dHJpYnV0ZXMgPSB7XG4gICAgICAgIGNvbmZpZ3VyYWJsZTogQ09ORklHVVJBQkxFIGluIEF0dHJpYnV0ZXMgPyBBdHRyaWJ1dGVzW0NPTkZJR1VSQUJMRV0gOiBjdXJyZW50W0NPTkZJR1VSQUJMRV0sXG4gICAgICAgIGVudW1lcmFibGU6IEVOVU1FUkFCTEUgaW4gQXR0cmlidXRlcyA/IEF0dHJpYnV0ZXNbRU5VTUVSQUJMRV0gOiBjdXJyZW50W0VOVU1FUkFCTEVdLFxuICAgICAgICB3cml0YWJsZTogZmFsc2VcbiAgICAgIH07XG4gICAgfVxuICB9IHJldHVybiAkZGVmaW5lUHJvcGVydHkoTywgUCwgQXR0cmlidXRlcyk7XG59IDogJGRlZmluZVByb3BlcnR5IDogZnVuY3Rpb24gZGVmaW5lUHJvcGVydHkoTywgUCwgQXR0cmlidXRlcykge1xuICBhbk9iamVjdChPKTtcbiAgUCA9IHRvUHJvcGVydHlLZXkoUCk7XG4gIGFuT2JqZWN0KEF0dHJpYnV0ZXMpO1xuICBpZiAoSUU4X0RPTV9ERUZJTkUpIHRyeSB7XG4gICAgcmV0dXJuICRkZWZpbmVQcm9wZXJ0eShPLCBQLCBBdHRyaWJ1dGVzKTtcbiAgfSBjYXRjaCAoZXJyb3IpIHsgLyogZW1wdHkgKi8gfVxuICBpZiAoJ2dldCcgaW4gQXR0cmlidXRlcyB8fCAnc2V0JyBpbiBBdHRyaWJ1dGVzKSB0aHJvdyBuZXcgJFR5cGVFcnJvcignQWNjZXNzb3JzIG5vdCBzdXBwb3J0ZWQnKTtcbiAgaWYgKCd2YWx1ZScgaW4gQXR0cmlidXRlcykgT1tQXSA9IEF0dHJpYnV0ZXMudmFsdWU7XG4gIHJldHVybiBPO1xufTtcbiIsIid1c2Ugc3RyaWN0Jztcbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKGJpdG1hcCwgdmFsdWUpIHtcbiAgcmV0dXJuIHtcbiAgICBlbnVtZXJhYmxlOiAhKGJpdG1hcCAmIDEpLFxuICAgIGNvbmZpZ3VyYWJsZTogIShiaXRtYXAgJiAyKSxcbiAgICB3cml0YWJsZTogIShiaXRtYXAgJiA0KSxcbiAgICB2YWx1ZTogdmFsdWVcbiAgfTtcbn07XG4iLCIndXNlIHN0cmljdCc7XG52YXIgREVTQ1JJUFRPUlMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVzY3JpcHRvcnMnKTtcbnZhciBkZWZpbmVQcm9wZXJ0eU1vZHVsZSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9vYmplY3QtZGVmaW5lLXByb3BlcnR5Jyk7XG52YXIgY3JlYXRlUHJvcGVydHlEZXNjcmlwdG9yID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2NyZWF0ZS1wcm9wZXJ0eS1kZXNjcmlwdG9yJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gREVTQ1JJUFRPUlMgPyBmdW5jdGlvbiAob2JqZWN0LCBrZXksIHZhbHVlKSB7XG4gIHJldHVybiBkZWZpbmVQcm9wZXJ0eU1vZHVsZS5mKG9iamVjdCwga2V5LCBjcmVhdGVQcm9wZXJ0eURlc2NyaXB0b3IoMSwgdmFsdWUpKTtcbn0gOiBmdW5jdGlvbiAob2JqZWN0LCBrZXksIHZhbHVlKSB7XG4gIG9iamVjdFtrZXldID0gdmFsdWU7XG4gIHJldHVybiBvYmplY3Q7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIHNoYXJlZCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9zaGFyZWQnKTtcbnZhciB1aWQgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvdWlkJyk7XG5cbnZhciBrZXlzID0gc2hhcmVkKCdrZXlzJyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKGtleSkge1xuICByZXR1cm4ga2V5c1trZXldIHx8IChrZXlzW2tleV0gPSB1aWQoa2V5KSk7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIE5BVElWRV9XRUFLX01BUCA9IHJlcXVpcmUoJy4uL2ludGVybmFscy93ZWFrLW1hcC1iYXNpYy1kZXRlY3Rpb24nKTtcbnZhciBnbG9iYWwgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZ2xvYmFsJyk7XG52YXIgaXNPYmplY3QgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaXMtb2JqZWN0Jyk7XG52YXIgY3JlYXRlTm9uRW51bWVyYWJsZVByb3BlcnR5ID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2NyZWF0ZS1ub24tZW51bWVyYWJsZS1wcm9wZXJ0eScpO1xudmFyIGhhc093biA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9oYXMtb3duLXByb3BlcnR5Jyk7XG52YXIgc2hhcmVkID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL3NoYXJlZC1zdG9yZScpO1xudmFyIHNoYXJlZEtleSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9zaGFyZWQta2V5Jyk7XG52YXIgaGlkZGVuS2V5cyA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9oaWRkZW4ta2V5cycpO1xuXG52YXIgT0JKRUNUX0FMUkVBRFlfSU5JVElBTElaRUQgPSAnT2JqZWN0IGFscmVhZHkgaW5pdGlhbGl6ZWQnO1xudmFyIFR5cGVFcnJvciA9IGdsb2JhbC5UeXBlRXJyb3I7XG52YXIgV2Vha01hcCA9IGdsb2JhbC5XZWFrTWFwO1xudmFyIHNldCwgZ2V0LCBoYXM7XG5cbnZhciBlbmZvcmNlID0gZnVuY3Rpb24gKGl0KSB7XG4gIHJldHVybiBoYXMoaXQpID8gZ2V0KGl0KSA6IHNldChpdCwge30pO1xufTtcblxudmFyIGdldHRlckZvciA9IGZ1bmN0aW9uIChUWVBFKSB7XG4gIHJldHVybiBmdW5jdGlvbiAoaXQpIHtcbiAgICB2YXIgc3RhdGU7XG4gICAgaWYgKCFpc09iamVjdChpdCkgfHwgKHN0YXRlID0gZ2V0KGl0KSkudHlwZSAhPT0gVFlQRSkge1xuICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignSW5jb21wYXRpYmxlIHJlY2VpdmVyLCAnICsgVFlQRSArICcgcmVxdWlyZWQnKTtcbiAgICB9IHJldHVybiBzdGF0ZTtcbiAgfTtcbn07XG5cbmlmIChOQVRJVkVfV0VBS19NQVAgfHwgc2hhcmVkLnN0YXRlKSB7XG4gIHZhciBzdG9yZSA9IHNoYXJlZC5zdGF0ZSB8fCAoc2hhcmVkLnN0YXRlID0gbmV3IFdlYWtNYXAoKSk7XG4gIC8qIGVzbGludC1kaXNhYmxlIG5vLXNlbGYtYXNzaWduIC0tIHByb3RvdHlwZSBtZXRob2RzIHByb3RlY3Rpb24gKi9cbiAgc3RvcmUuZ2V0ID0gc3RvcmUuZ2V0O1xuICBzdG9yZS5oYXMgPSBzdG9yZS5oYXM7XG4gIHN0b3JlLnNldCA9IHN0b3JlLnNldDtcbiAgLyogZXNsaW50LWVuYWJsZSBuby1zZWxmLWFzc2lnbiAtLSBwcm90b3R5cGUgbWV0aG9kcyBwcm90ZWN0aW9uICovXG4gIHNldCA9IGZ1bmN0aW9uIChpdCwgbWV0YWRhdGEpIHtcbiAgICBpZiAoc3RvcmUuaGFzKGl0KSkgdGhyb3cgbmV3IFR5cGVFcnJvcihPQkpFQ1RfQUxSRUFEWV9JTklUSUFMSVpFRCk7XG4gICAgbWV0YWRhdGEuZmFjYWRlID0gaXQ7XG4gICAgc3RvcmUuc2V0KGl0LCBtZXRhZGF0YSk7XG4gICAgcmV0dXJuIG1ldGFkYXRhO1xuICB9O1xuICBnZXQgPSBmdW5jdGlvbiAoaXQpIHtcbiAgICByZXR1cm4gc3RvcmUuZ2V0KGl0KSB8fCB7fTtcbiAgfTtcbiAgaGFzID0gZnVuY3Rpb24gKGl0KSB7XG4gICAgcmV0dXJuIHN0b3JlLmhhcyhpdCk7XG4gIH07XG59IGVsc2Uge1xuICB2YXIgU1RBVEUgPSBzaGFyZWRLZXkoJ3N0YXRlJyk7XG4gIGhpZGRlbktleXNbU1RBVEVdID0gdHJ1ZTtcbiAgc2V0ID0gZnVuY3Rpb24gKGl0LCBtZXRhZGF0YSkge1xuICAgIGlmIChoYXNPd24oaXQsIFNUQVRFKSkgdGhyb3cgbmV3IFR5cGVFcnJvcihPQkpFQ1RfQUxSRUFEWV9JTklUSUFMSVpFRCk7XG4gICAgbWV0YWRhdGEuZmFjYWRlID0gaXQ7XG4gICAgY3JlYXRlTm9uRW51bWVyYWJsZVByb3BlcnR5KGl0LCBTVEFURSwgbWV0YWRhdGEpO1xuICAgIHJldHVybiBtZXRhZGF0YTtcbiAgfTtcbiAgZ2V0ID0gZnVuY3Rpb24gKGl0KSB7XG4gICAgcmV0dXJuIGhhc093bihpdCwgU1RBVEUpID8gaXRbU1RBVEVdIDoge307XG4gIH07XG4gIGhhcyA9IGZ1bmN0aW9uIChpdCkge1xuICAgIHJldHVybiBoYXNPd24oaXQsIFNUQVRFKTtcbiAgfTtcbn1cblxubW9kdWxlLmV4cG9ydHMgPSB7XG4gIHNldDogc2V0LFxuICBnZXQ6IGdldCxcbiAgaGFzOiBoYXMsXG4gIGVuZm9yY2U6IGVuZm9yY2UsXG4gIGdldHRlckZvcjogZ2V0dGVyRm9yXG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIHVuY3VycnlUaGlzID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2Z1bmN0aW9uLXVuY3VycnktdGhpcycpO1xudmFyIGZhaWxzID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2ZhaWxzJyk7XG52YXIgaXNDYWxsYWJsZSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9pcy1jYWxsYWJsZScpO1xudmFyIGhhc093biA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9oYXMtb3duLXByb3BlcnR5Jyk7XG52YXIgREVTQ1JJUFRPUlMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVzY3JpcHRvcnMnKTtcbnZhciBDT05GSUdVUkFCTEVfRlVOQ1RJT05fTkFNRSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9mdW5jdGlvbi1uYW1lJykuQ09ORklHVVJBQkxFO1xudmFyIGluc3BlY3RTb3VyY2UgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvaW5zcGVjdC1zb3VyY2UnKTtcbnZhciBJbnRlcm5hbFN0YXRlTW9kdWxlID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2ludGVybmFsLXN0YXRlJyk7XG5cbnZhciBlbmZvcmNlSW50ZXJuYWxTdGF0ZSA9IEludGVybmFsU3RhdGVNb2R1bGUuZW5mb3JjZTtcbnZhciBnZXRJbnRlcm5hbFN0YXRlID0gSW50ZXJuYWxTdGF0ZU1vZHVsZS5nZXQ7XG52YXIgJFN0cmluZyA9IFN0cmluZztcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBlcy9uby1vYmplY3QtZGVmaW5lcHJvcGVydHkgLS0gc2FmZVxudmFyIGRlZmluZVByb3BlcnR5ID0gT2JqZWN0LmRlZmluZVByb3BlcnR5O1xudmFyIHN0cmluZ1NsaWNlID0gdW5jdXJyeVRoaXMoJycuc2xpY2UpO1xudmFyIHJlcGxhY2UgPSB1bmN1cnJ5VGhpcygnJy5yZXBsYWNlKTtcbnZhciBqb2luID0gdW5jdXJyeVRoaXMoW10uam9pbik7XG5cbnZhciBDT05GSUdVUkFCTEVfTEVOR1RIID0gREVTQ1JJUFRPUlMgJiYgIWZhaWxzKGZ1bmN0aW9uICgpIHtcbiAgcmV0dXJuIGRlZmluZVByb3BlcnR5KGZ1bmN0aW9uICgpIHsgLyogZW1wdHkgKi8gfSwgJ2xlbmd0aCcsIHsgdmFsdWU6IDggfSkubGVuZ3RoICE9PSA4O1xufSk7XG5cbnZhciBURU1QTEFURSA9IFN0cmluZyhTdHJpbmcpLnNwbGl0KCdTdHJpbmcnKTtcblxudmFyIG1ha2VCdWlsdEluID0gbW9kdWxlLmV4cG9ydHMgPSBmdW5jdGlvbiAodmFsdWUsIG5hbWUsIG9wdGlvbnMpIHtcbiAgaWYgKHN0cmluZ1NsaWNlKCRTdHJpbmcobmFtZSksIDAsIDcpID09PSAnU3ltYm9sKCcpIHtcbiAgICBuYW1lID0gJ1snICsgcmVwbGFjZSgkU3RyaW5nKG5hbWUpLCAvXlN5bWJvbFxcKChbXildKilcXCkvLCAnJDEnKSArICddJztcbiAgfVxuICBpZiAob3B0aW9ucyAmJiBvcHRpb25zLmdldHRlcikgbmFtZSA9ICdnZXQgJyArIG5hbWU7XG4gIGlmIChvcHRpb25zICYmIG9wdGlvbnMuc2V0dGVyKSBuYW1lID0gJ3NldCAnICsgbmFtZTtcbiAgaWYgKCFoYXNPd24odmFsdWUsICduYW1lJykgfHwgKENPTkZJR1VSQUJMRV9GVU5DVElPTl9OQU1FICYmIHZhbHVlLm5hbWUgIT09IG5hbWUpKSB7XG4gICAgaWYgKERFU0NSSVBUT1JTKSBkZWZpbmVQcm9wZXJ0eSh2YWx1ZSwgJ25hbWUnLCB7IHZhbHVlOiBuYW1lLCBjb25maWd1cmFibGU6IHRydWUgfSk7XG4gICAgZWxzZSB2YWx1ZS5uYW1lID0gbmFtZTtcbiAgfVxuICBpZiAoQ09ORklHVVJBQkxFX0xFTkdUSCAmJiBvcHRpb25zICYmIGhhc093bihvcHRpb25zLCAnYXJpdHknKSAmJiB2YWx1ZS5sZW5ndGggIT09IG9wdGlvbnMuYXJpdHkpIHtcbiAgICBkZWZpbmVQcm9wZXJ0eSh2YWx1ZSwgJ2xlbmd0aCcsIHsgdmFsdWU6IG9wdGlvbnMuYXJpdHkgfSk7XG4gIH1cbiAgdHJ5IHtcbiAgICBpZiAob3B0aW9ucyAmJiBoYXNPd24ob3B0aW9ucywgJ2NvbnN0cnVjdG9yJykgJiYgb3B0aW9ucy5jb25zdHJ1Y3Rvcikge1xuICAgICAgaWYgKERFU0NSSVBUT1JTKSBkZWZpbmVQcm9wZXJ0eSh2YWx1ZSwgJ3Byb3RvdHlwZScsIHsgd3JpdGFibGU6IGZhbHNlIH0pO1xuICAgIC8vIGluIFY4IH4gQ2hyb21lIDUzLCBwcm90b3R5cGVzIG9mIHNvbWUgbWV0aG9kcywgbGlrZSBgQXJyYXkucHJvdG90eXBlLnZhbHVlc2AsIGFyZSBub24td3JpdGFibGVcbiAgICB9IGVsc2UgaWYgKHZhbHVlLnByb3RvdHlwZSkgdmFsdWUucHJvdG90eXBlID0gdW5kZWZpbmVkO1xuICB9IGNhdGNoIChlcnJvcikgeyAvKiBlbXB0eSAqLyB9XG4gIHZhciBzdGF0ZSA9IGVuZm9yY2VJbnRlcm5hbFN0YXRlKHZhbHVlKTtcbiAgaWYgKCFoYXNPd24oc3RhdGUsICdzb3VyY2UnKSkge1xuICAgIHN0YXRlLnNvdXJjZSA9IGpvaW4oVEVNUExBVEUsIHR5cGVvZiBuYW1lID09ICdzdHJpbmcnID8gbmFtZSA6ICcnKTtcbiAgfSByZXR1cm4gdmFsdWU7XG59O1xuXG4vLyBhZGQgZmFrZSBGdW5jdGlvbiN0b1N0cmluZyBmb3IgY29ycmVjdCB3b3JrIHdyYXBwZWQgbWV0aG9kcyAvIGNvbnN0cnVjdG9ycyB3aXRoIG1ldGhvZHMgbGlrZSBMb0Rhc2ggaXNOYXRpdmVcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1leHRlbmQtbmF0aXZlIC0tIHJlcXVpcmVkXG5GdW5jdGlvbi5wcm90b3R5cGUudG9TdHJpbmcgPSBtYWtlQnVpbHRJbihmdW5jdGlvbiB0b1N0cmluZygpIHtcbiAgcmV0dXJuIGlzQ2FsbGFibGUodGhpcykgJiYgZ2V0SW50ZXJuYWxTdGF0ZSh0aGlzKS5zb3VyY2UgfHwgaW5zcGVjdFNvdXJjZSh0aGlzKTtcbn0sICd0b1N0cmluZycpO1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIG1ha2VCdWlsdEluID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL21ha2UtYnVpbHQtaW4nKTtcbnZhciBkZWZpbmVQcm9wZXJ0eSA9IHJlcXVpcmUoJy4uL2ludGVybmFscy9vYmplY3QtZGVmaW5lLXByb3BlcnR5Jyk7XG5cbm1vZHVsZS5leHBvcnRzID0gZnVuY3Rpb24gKHRhcmdldCwgbmFtZSwgZGVzY3JpcHRvcikge1xuICBpZiAoZGVzY3JpcHRvci5nZXQpIG1ha2VCdWlsdEluKGRlc2NyaXB0b3IuZ2V0LCBuYW1lLCB7IGdldHRlcjogdHJ1ZSB9KTtcbiAgaWYgKGRlc2NyaXB0b3Iuc2V0KSBtYWtlQnVpbHRJbihkZXNjcmlwdG9yLnNldCwgbmFtZSwgeyBzZXR0ZXI6IHRydWUgfSk7XG4gIHJldHVybiBkZWZpbmVQcm9wZXJ0eS5mKHRhcmdldCwgbmFtZSwgZGVzY3JpcHRvcik7XG59O1xuIiwiJ3VzZSBzdHJpY3QnO1xudmFyIGFuT2JqZWN0ID0gcmVxdWlyZSgnLi4vaW50ZXJuYWxzL2FuLW9iamVjdCcpO1xuXG4vLyBgUmVnRXhwLnByb3RvdHlwZS5mbGFnc2AgZ2V0dGVyIGltcGxlbWVudGF0aW9uXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLWdldC1yZWdleHAucHJvdG90eXBlLmZsYWdzXG5tb2R1bGUuZXhwb3J0cyA9IGZ1bmN0aW9uICgpIHtcbiAgdmFyIHRoYXQgPSBhbk9iamVjdCh0aGlzKTtcbiAgdmFyIHJlc3VsdCA9ICcnO1xuICBpZiAodGhhdC5oYXNJbmRpY2VzKSByZXN1bHQgKz0gJ2QnO1xuICBpZiAodGhhdC5nbG9iYWwpIHJlc3VsdCArPSAnZyc7XG4gIGlmICh0aGF0Lmlnbm9yZUNhc2UpIHJlc3VsdCArPSAnaSc7XG4gIGlmICh0aGF0Lm11bHRpbGluZSkgcmVzdWx0ICs9ICdtJztcbiAgaWYgKHRoYXQuZG90QWxsKSByZXN1bHQgKz0gJ3MnO1xuICBpZiAodGhhdC51bmljb2RlKSByZXN1bHQgKz0gJ3UnO1xuICBpZiAodGhhdC51bmljb2RlU2V0cykgcmVzdWx0ICs9ICd2JztcbiAgaWYgKHRoYXQuc3RpY2t5KSByZXN1bHQgKz0gJ3knO1xuICByZXR1cm4gcmVzdWx0O1xufTtcbiIsIid1c2Ugc3RyaWN0JztcbnZhciBnbG9iYWwgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZ2xvYmFsJyk7XG52YXIgREVTQ1JJUFRPUlMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVzY3JpcHRvcnMnKTtcbnZhciBkZWZpbmVCdWlsdEluQWNjZXNzb3IgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZGVmaW5lLWJ1aWx0LWluLWFjY2Vzc29yJyk7XG52YXIgcmVnRXhwRmxhZ3MgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvcmVnZXhwLWZsYWdzJyk7XG52YXIgZmFpbHMgPSByZXF1aXJlKCcuLi9pbnRlcm5hbHMvZmFpbHMnKTtcblxuLy8gYmFiZWwtbWluaWZ5IGFuZCBDbG9zdXJlIENvbXBpbGVyIHRyYW5zcGlsZXMgUmVnRXhwKCcuJywgJ2QnKSAtPiAvLi9kIGFuZCBpdCBjYXVzZXMgU3ludGF4RXJyb3JcbnZhciBSZWdFeHAgPSBnbG9iYWwuUmVnRXhwO1xudmFyIFJlZ0V4cFByb3RvdHlwZSA9IFJlZ0V4cC5wcm90b3R5cGU7XG5cbnZhciBGT1JDRUQgPSBERVNDUklQVE9SUyAmJiBmYWlscyhmdW5jdGlvbiAoKSB7XG4gIHZhciBJTkRJQ0VTX1NVUFBPUlQgPSB0cnVlO1xuICB0cnkge1xuICAgIFJlZ0V4cCgnLicsICdkJyk7XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgSU5ESUNFU19TVVBQT1JUID0gZmFsc2U7XG4gIH1cblxuICB2YXIgTyA9IHt9O1xuICAvLyBtb2Rlcm4gVjggYnVnXG4gIHZhciBjYWxscyA9ICcnO1xuICB2YXIgZXhwZWN0ZWQgPSBJTkRJQ0VTX1NVUFBPUlQgPyAnZGdpbXN5JyA6ICdnaW1zeSc7XG5cbiAgdmFyIGFkZEdldHRlciA9IGZ1bmN0aW9uIChrZXksIGNocikge1xuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBlcy9uby1vYmplY3QtZGVmaW5lcHJvcGVydHkgLS0gc2FmZVxuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eShPLCBrZXksIHsgZ2V0OiBmdW5jdGlvbiAoKSB7XG4gICAgICBjYWxscyArPSBjaHI7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9IH0pO1xuICB9O1xuXG4gIHZhciBwYWlycyA9IHtcbiAgICBkb3RBbGw6ICdzJyxcbiAgICBnbG9iYWw6ICdnJyxcbiAgICBpZ25vcmVDYXNlOiAnaScsXG4gICAgbXVsdGlsaW5lOiAnbScsXG4gICAgc3RpY2t5OiAneSdcbiAgfTtcblxuICBpZiAoSU5ESUNFU19TVVBQT1JUKSBwYWlycy5oYXNJbmRpY2VzID0gJ2QnO1xuXG4gIGZvciAodmFyIGtleSBpbiBwYWlycykgYWRkR2V0dGVyKGtleSwgcGFpcnNba2V5XSk7XG5cbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGVzL25vLW9iamVjdC1nZXRvd25wcm9wZXJ0eWRlc2NyaXB0b3IgLS0gc2FmZVxuICB2YXIgcmVzdWx0ID0gT2JqZWN0LmdldE93blByb3BlcnR5RGVzY3JpcHRvcihSZWdFeHBQcm90b3R5cGUsICdmbGFncycpLmdldC5jYWxsKE8pO1xuXG4gIHJldHVybiByZXN1bHQgIT09IGV4cGVjdGVkIHx8IGNhbGxzICE9PSBleHBlY3RlZDtcbn0pO1xuXG4vLyBgUmVnRXhwLnByb3RvdHlwZS5mbGFnc2AgZ2V0dGVyXG4vLyBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jc2VjLWdldC1yZWdleHAucHJvdG90eXBlLmZsYWdzXG5pZiAoRk9SQ0VEKSBkZWZpbmVCdWlsdEluQWNjZXNzb3IoUmVnRXhwUHJvdG90eXBlLCAnZmxhZ3MnLCB7XG4gIGNvbmZpZ3VyYWJsZTogdHJ1ZSxcbiAgZ2V0OiByZWdFeHBGbGFnc1xufSk7XG4iLCIvLyBjb25zdCByZXBsYWNlbWVudHMgPSB7XG4vLyAgIEFTQ0lJUHVuY3R1YXRpb246ICchXCIjJCUmXFwnKCkqKyxcXFxcLS4vOjs8PT4/QFxcXFxbXFxcXF1eX2B7fH1+Jyxcbi8vICAgVHJpZ2dlckNoYXJzOiAnYF9cXCpcXFtcXF1cXChcXCknLFxuLy8gICBTY2hlbWU6IGBbQS1aYS16XVtBLVphLXowLTlcXCtcXC5cXC1dezEsMzF9YCxcbi8vICAgRW1haWw6IGBbYS16QS1aMC05LiEjJCUmJyorLz0/Xl9cXGB7fH1+LV0rQFthLXpBLVowLTldKD86W2EtekEtWjAtOS1dezAsNjF9W2EtekEtWjAtOV0pPyg/OlxcXFwuW2EtekEtWjAtOV0oPzpbYS16QS1aMC05LV17MCw2MX1bYS16QS1aMC05XSk/KSpgLCAvLyBGcm9tIENvbW1vbk1hcmsgc3BlY1xuXG4vLyB9XG5jb25zdCByZXBsYWNlbWVudHMgPSB7XG4gIEFTQ0lJUHVuY3R1YXRpb246IC9bIVwiIyQlJicoKSorLFxcLS4vOjs8PT4/QFtcXF1eX2B7fH1+XFxcXF0vLCAgXG4gIE5vdFRyaWdnZXJDaGFyOiAvW15gXypbXFxdKCk8PiF+XS8sXG4gIFNjaGVtZTogL1tBLVphLXpdW0EtWmEtejAtOSsuLV17MSwzMX0vLFxuICBFbWFpbDogL1thLXpBLVowLTkuISMkJSYnKisvPT9eX2B7fH1+LV0rQFthLXpBLVowLTldKD86W2EtekEtWjAtOS1dezAsNjF9W2EtekEtWjAtOV0pPyg/OlxcLlthLXpBLVowLTldKD86W2EtekEtWjAtOS1dezAsNjF9W2EtekEtWjAtOV0pPykqLywgLy8gRnJvbSBDb21tb25NYXJrIHNwZWNcbiAgSFRNTE9wZW5UYWc6IC88SFRNTFRhZ05hbWUoPzpIVE1MQXR0cmlidXRlKSpcXHMqXFwvPz4vLFxuICBIVE1MQ2xvc2VUYWc6IC88XFwvSFRNTFRhZ05hbWVcXHMqPi8sXG4gIEhUTUxUYWdOYW1lOiAvW0EtWmEtel1bQS1aYS16MC05LV0qLywgXG4gIEhUTUxDb21tZW50OiAvPCEtLSg/OltePi1dfCg/OltePi1dKD86W14tXXwtW14tXSkqW14tXSkpLS0+LyxcbiAgSFRNTFBJOiAvPFxcPyg/OnwufCg/OlteP118XFw/W14+XSkqKVxcPz4vLFxuICBIVE1MRGVjbGFyYXRpb246IC88IVtBLVpdK1xcc1tePl0qPi8sXG4gIEhUTUxDREFUQTogLzwhXFxbQ0RBVEFcXFsuKj9cXF1cXF0+LyxcbiAgSFRNTEF0dHJpYnV0ZTogL1xccytbQS1aYS16XzpdW0EtWmEtejAtOV8uOi1dKig/OkhUTUxBdHRWYWx1ZSk/LyxcbiAgSFRNTEF0dFZhbHVlOiAvXFxzKj1cXHMqKD86KD86J1teJ10qJyl8KD86XCJbXlwiXSpcIil8KD86W15cXHNcIic9PD5gXSspKS8sXG4gIEtub3duVGFnOiAvYWRkcmVzc3xhcnRpY2xlfGFzaWRlfGJhc2V8YmFzZWZvbnR8YmxvY2txdW90ZXxib2R5fGNhcHRpb258Y2VudGVyfGNvbHxjb2xncm91cHxkZHxkZXRhaWxzfGRpYWxvZ3xkaXJ8ZGl2fGRsfGR0fGZpZWxkc2V0fGZpZ2NhcHRpb258ZmlndXJlfGZvb3Rlcnxmb3JtfGZyYW1lfGZyYW1lc2V0fGgxfGgyfGgzfGg0fGg1fGg2fGhlYWR8aGVhZGVyfGhyfGh0bWx8aWZyYW1lfGxlZ2VuZHxsaXxsaW5rfG1haW58bWVudXxtZW51aXRlbXxuYXZ8bm9mcmFtZXN8b2x8b3B0Z3JvdXB8b3B0aW9ufHB8cGFyYW18c2VjdGlvbnxzb3VyY2V8c3VtbWFyeXx0YWJsZXx0Ym9keXx0ZHx0Zm9vdHx0aHx0aGVhZHx0aXRsZXx0cnx0cmFja3x1bC9cbn1cblxuLy8gRnJvbSBDb21tb25NYXJrLmpzLiBcbmNvbnN0IHB1bmN0dWF0aW9uTGVhZGluZyA9IG5ldyBSZWdFeHAoL14oPzpbIVwiIyQlJicoKSorLFxcLS4vOjs8PT4/QFtcXF1cXFxcXl9ge3x9flxceEExXFx4QTdcXHhBQlxceEI2XFx4QjdcXHhCQlxceEJGXFx1MDM3RVxcdTAzODdcXHUwNTVBLVxcdTA1NUZcXHUwNTg5XFx1MDU4QVxcdTA1QkVcXHUwNUMwXFx1MDVDM1xcdTA1QzZcXHUwNUYzXFx1MDVGNFxcdTA2MDlcXHUwNjBBXFx1MDYwQ1xcdTA2MERcXHUwNjFCXFx1MDYxRVxcdTA2MUZcXHUwNjZBLVxcdTA2NkRcXHUwNkQ0XFx1MDcwMC1cXHUwNzBEXFx1MDdGNy1cXHUwN0Y5XFx1MDgzMC1cXHUwODNFXFx1MDg1RVxcdTA5NjRcXHUwOTY1XFx1MDk3MFxcdTBBRjBcXHUwREY0XFx1MEU0RlxcdTBFNUFcXHUwRTVCXFx1MEYwNC1cXHUwRjEyXFx1MEYxNFxcdTBGM0EtXFx1MEYzRFxcdTBGODVcXHUwRkQwLVxcdTBGRDRcXHUwRkQ5XFx1MEZEQVxcdTEwNEEtXFx1MTA0RlxcdTEwRkJcXHUxMzYwLVxcdTEzNjhcXHUxNDAwXFx1MTY2RFxcdTE2NkVcXHUxNjlCXFx1MTY5Q1xcdTE2RUItXFx1MTZFRFxcdTE3MzVcXHUxNzM2XFx1MTdENC1cXHUxN0Q2XFx1MTdEOC1cXHUxN0RBXFx1MTgwMC1cXHUxODBBXFx1MTk0NFxcdTE5NDVcXHUxQTFFXFx1MUExRlxcdTFBQTAtXFx1MUFBNlxcdTFBQTgtXFx1MUFBRFxcdTFCNUEtXFx1MUI2MFxcdTFCRkMtXFx1MUJGRlxcdTFDM0ItXFx1MUMzRlxcdTFDN0VcXHUxQzdGXFx1MUNDMC1cXHUxQ0M3XFx1MUNEM1xcdTIwMTAtXFx1MjAyN1xcdTIwMzAtXFx1MjA0M1xcdTIwNDUtXFx1MjA1MVxcdTIwNTMtXFx1MjA1RVxcdTIwN0RcXHUyMDdFXFx1MjA4RFxcdTIwOEVcXHUyMzA4LVxcdTIzMEJcXHUyMzI5XFx1MjMyQVxcdTI3NjgtXFx1Mjc3NVxcdTI3QzVcXHUyN0M2XFx1MjdFNi1cXHUyN0VGXFx1Mjk4My1cXHUyOTk4XFx1MjlEOC1cXHUyOURCXFx1MjlGQ1xcdTI5RkRcXHUyQ0Y5LVxcdTJDRkNcXHUyQ0ZFXFx1MkNGRlxcdTJENzBcXHUyRTAwLVxcdTJFMkVcXHUyRTMwLVxcdTJFNDJcXHUzMDAxLVxcdTMwMDNcXHUzMDA4LVxcdTMwMTFcXHUzMDE0LVxcdTMwMUZcXHUzMDMwXFx1MzAzRFxcdTMwQTBcXHUzMEZCXFx1QTRGRVxcdUE0RkZcXHVBNjBELVxcdUE2MEZcXHVBNjczXFx1QTY3RVxcdUE2RjItXFx1QTZGN1xcdUE4NzQtXFx1QTg3N1xcdUE4Q0VcXHVBOENGXFx1QThGOC1cXHVBOEZBXFx1QThGQ1xcdUE5MkVcXHVBOTJGXFx1QTk1RlxcdUE5QzEtXFx1QTlDRFxcdUE5REVcXHVBOURGXFx1QUE1Qy1cXHVBQTVGXFx1QUFERVxcdUFBREZcXHVBQUYwXFx1QUFGMVxcdUFCRUJcXHVGRDNFXFx1RkQzRlxcdUZFMTAtXFx1RkUxOVxcdUZFMzAtXFx1RkU1MlxcdUZFNTQtXFx1RkU2MVxcdUZFNjNcXHVGRTY4XFx1RkU2QVxcdUZFNkJcXHVGRjAxLVxcdUZGMDNcXHVGRjA1LVxcdUZGMEFcXHVGRjBDLVxcdUZGMEZcXHVGRjFBXFx1RkYxQlxcdUZGMUZcXHVGRjIwXFx1RkYzQi1cXHVGRjNEXFx1RkYzRlxcdUZGNUJcXHVGRjVEXFx1RkY1Ri1cXHVGRjY1XXxcXHVEODAwW1xcdUREMDAtXFx1REQwMlxcdURGOUZcXHVERkQwXXxcXHVEODAxXFx1REQ2RnxcXHVEODAyW1xcdURDNTdcXHVERDFGXFx1REQzRlxcdURFNTAtXFx1REU1OFxcdURFN0ZcXHVERUYwLVxcdURFRjZcXHVERjM5LVxcdURGM0ZcXHVERjk5LVxcdURGOUNdfFxcdUQ4MDRbXFx1REM0Ny1cXHVEQzREXFx1RENCQlxcdURDQkNcXHVEQ0JFLVxcdURDQzFcXHVERDQwLVxcdURENDNcXHVERDc0XFx1REQ3NVxcdUREQzUtXFx1RERDOVxcdUREQ0RcXHVERERCXFx1RERERC1cXHVERERGXFx1REUzOC1cXHVERTNEXFx1REVBOV18XFx1RDgwNVtcXHVEQ0M2XFx1RERDMS1cXHVEREQ3XFx1REU0MS1cXHVERTQzXFx1REYzQy1cXHVERjNFXXxcXHVEODA5W1xcdURDNzAtXFx1REM3NF18XFx1RDgxQVtcXHVERTZFXFx1REU2RlxcdURFRjVcXHVERjM3LVxcdURGM0JcXHVERjQ0XXxcXHVEODJGXFx1REM5RnxcXHVEODM2W1xcdURFODctXFx1REU4Ql0pLyk7XG5cbmNvbnN0IHB1bmN0dWF0aW9uVHJhaWxpbmcgPSBuZXcgUmVnRXhwKC8oPzpbIVwiIyQlJicoKSorLFxcLS4vOjs8PT4/QFtcXF1cXFxcXl9ge3x9flxceEExXFx4QTdcXHhBQlxceEI2XFx4QjdcXHhCQlxceEJGXFx1MDM3RVxcdTAzODdcXHUwNTVBLVxcdTA1NUZcXHUwNTg5XFx1MDU4QVxcdTA1QkVcXHUwNUMwXFx1MDVDM1xcdTA1QzZcXHUwNUYzXFx1MDVGNFxcdTA2MDlcXHUwNjBBXFx1MDYwQ1xcdTA2MERcXHUwNjFCXFx1MDYxRVxcdTA2MUZcXHUwNjZBLVxcdTA2NkRcXHUwNkQ0XFx1MDcwMC1cXHUwNzBEXFx1MDdGNy1cXHUwN0Y5XFx1MDgzMC1cXHUwODNFXFx1MDg1RVxcdTA5NjRcXHUwOTY1XFx1MDk3MFxcdTBBRjBcXHUwREY0XFx1MEU0RlxcdTBFNUFcXHUwRTVCXFx1MEYwNC1cXHUwRjEyXFx1MEYxNFxcdTBGM0EtXFx1MEYzRFxcdTBGODVcXHUwRkQwLVxcdTBGRDRcXHUwRkQ5XFx1MEZEQVxcdTEwNEEtXFx1MTA0RlxcdTEwRkJcXHUxMzYwLVxcdTEzNjhcXHUxNDAwXFx1MTY2RFxcdTE2NkVcXHUxNjlCXFx1MTY5Q1xcdTE2RUItXFx1MTZFRFxcdTE3MzVcXHUxNzM2XFx1MTdENC1cXHUxN0Q2XFx1MTdEOC1cXHUxN0RBXFx1MTgwMC1cXHUxODBBXFx1MTk0NFxcdTE5NDVcXHUxQTFFXFx1MUExRlxcdTFBQTAtXFx1MUFBNlxcdTFBQTgtXFx1MUFBRFxcdTFCNUEtXFx1MUI2MFxcdTFCRkMtXFx1MUJGRlxcdTFDM0ItXFx1MUMzRlxcdTFDN0VcXHUxQzdGXFx1MUNDMC1cXHUxQ0M3XFx1MUNEM1xcdTIwMTAtXFx1MjAyN1xcdTIwMzAtXFx1MjA0M1xcdTIwNDUtXFx1MjA1MVxcdTIwNTMtXFx1MjA1RVxcdTIwN0RcXHUyMDdFXFx1MjA4RFxcdTIwOEVcXHUyMzA4LVxcdTIzMEJcXHUyMzI5XFx1MjMyQVxcdTI3NjgtXFx1Mjc3NVxcdTI3QzVcXHUyN0M2XFx1MjdFNi1cXHUyN0VGXFx1Mjk4My1cXHUyOTk4XFx1MjlEOC1cXHUyOURCXFx1MjlGQ1xcdTI5RkRcXHUyQ0Y5LVxcdTJDRkNcXHUyQ0ZFXFx1MkNGRlxcdTJENzBcXHUyRTAwLVxcdTJFMkVcXHUyRTMwLVxcdTJFNDJcXHUzMDAxLVxcdTMwMDNcXHUzMDA4LVxcdTMwMTFcXHUzMDE0LVxcdTMwMUZcXHUzMDMwXFx1MzAzRFxcdTMwQTBcXHUzMEZCXFx1QTRGRVxcdUE0RkZcXHVBNjBELVxcdUE2MEZcXHVBNjczXFx1QTY3RVxcdUE2RjItXFx1QTZGN1xcdUE4NzQtXFx1QTg3N1xcdUE4Q0VcXHVBOENGXFx1QThGOC1cXHVBOEZBXFx1QThGQ1xcdUE5MkVcXHVBOTJGXFx1QTk1RlxcdUE5QzEtXFx1QTlDRFxcdUE5REVcXHVBOURGXFx1QUE1Qy1cXHVBQTVGXFx1QUFERVxcdUFBREZcXHVBQUYwXFx1QUFGMVxcdUFCRUJcXHVGRDNFXFx1RkQzRlxcdUZFMTAtXFx1RkUxOVxcdUZFMzAtXFx1RkU1MlxcdUZFNTQtXFx1RkU2MVxcdUZFNjNcXHVGRTY4XFx1RkU2QVxcdUZFNkJcXHVGRjAxLVxcdUZGMDNcXHVGRjA1LVxcdUZGMEFcXHVGRjBDLVxcdUZGMEZcXHVGRjFBXFx1RkYxQlxcdUZGMUZcXHVGRjIwXFx1RkYzQi1cXHVGRjNEXFx1RkYzRlxcdUZGNUJcXHVGRjVEXFx1RkY1Ri1cXHVGRjY1XXxcXHVEODAwW1xcdUREMDAtXFx1REQwMlxcdURGOUZcXHVERkQwXXxcXHVEODAxXFx1REQ2RnxcXHVEODAyW1xcdURDNTdcXHVERDFGXFx1REQzRlxcdURFNTAtXFx1REU1OFxcdURFN0ZcXHVERUYwLVxcdURFRjZcXHVERjM5LVxcdURGM0ZcXHVERjk5LVxcdURGOUNdfFxcdUQ4MDRbXFx1REM0Ny1cXHVEQzREXFx1RENCQlxcdURDQkNcXHVEQ0JFLVxcdURDQzFcXHVERDQwLVxcdURENDNcXHVERDc0XFx1REQ3NVxcdUREQzUtXFx1RERDOVxcdUREQ0RcXHVERERCXFx1RERERC1cXHVERERGXFx1REUzOC1cXHVERTNEXFx1REVBOV18XFx1RDgwNVtcXHVEQ0M2XFx1RERDMS1cXHVEREQ3XFx1REU0MS1cXHVERTQzXFx1REYzQy1cXHVERjNFXXxcXHVEODA5W1xcdURDNzAtXFx1REM3NF18XFx1RDgxQVtcXHVERTZFXFx1REU2RlxcdURFRjVcXHVERjM3LVxcdURGM0JcXHVERjQ0XXxcXHVEODJGXFx1REM5RnxcXHVEODM2W1xcdURFODctXFx1REU4Ql0pJC8pO1xuXG4vLyBleHBvcnQgY29uc3QgaW5saW5lVHJpZ2dlckNoYXJzID0gbmV3IFJlZ0V4cChgWyR7cmVwbGFjZW1lbnRzLlRyaWdnZXJDaGFyc31dYCk7XG5cbi8qKlxuICogVGhpcyBpcyBDb21tb25NYXJrJ3MgYmxvY2sgZ3JhbW1hciwgYnV0IHdlJ3JlIGlnbm9yaW5nIG5lc3RlZCBibG9ja3MgaGVyZS4gIFxuICovIFxuY29uc3QgbGluZUdyYW1tYXIgPSB7IFxuICBUTUgxOiB7IFxuICAgIHJlZ2V4cDogL14oIHswLDN9I1xccykoLio/KSgoPzpcXHMrIytcXHMqKT8pJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNSDFcIj4kMTwvc3Bhbj4kJDI8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1IMVwiPiQzPC9zcGFuPidcbiAgfSxcbiAgVE1IMjogeyBcbiAgICByZWdleHA6IC9eKCB7MCwzfSMjXFxzKSguKj8pKCg/OlxccysjK1xccyopPykkLywgXG4gICAgcmVwbGFjZW1lbnQ6ICc8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1IMlwiPiQxPC9zcGFuPiQkMjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUgyXCI+JDM8L3NwYW4+J1xuICB9LFxuICBUTUgzOiB7IFxuICAgIHJlZ2V4cDogL14oIHswLDN9IyMjXFxzKSguKj8pKCg/OlxccysjK1xccyopPykkLywgXG4gICAgcmVwbGFjZW1lbnQ6ICc8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1IM1wiPiQxPC9zcGFuPiQkMjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUgzXCI+JDM8L3NwYW4+J1xuICB9LFxuICBUTUg0OiB7IFxuICAgIHJlZ2V4cDogL14oIHswLDN9IyMjI1xccykoLio/KSgoPzpcXHMrIytcXHMqKT8pJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNSDRcIj4kMTwvc3Bhbj4kJDI8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1INFwiPiQzPC9zcGFuPidcbiAgfSxcbiAgVE1INTogeyBcbiAgICByZWdleHA6IC9eKCB7MCwzfSMjIyMjXFxzKSguKj8pKCg/OlxccysjK1xccyopPykkLywgXG4gICAgcmVwbGFjZW1lbnQ6ICc8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1INVwiPiQxPC9zcGFuPiQkMjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUg1XCI+JDM8L3NwYW4+J1xuICB9LFxuICBUTUg2OiB7IFxuICAgIHJlZ2V4cDogL14oIHswLDN9IyMjIyMjXFxzKSguKj8pKCg/OlxccysjK1xccyopPykkLywgXG4gICAgcmVwbGFjZW1lbnQ6ICc8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1INlwiPiQxPC9zcGFuPiQkMjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUg2XCI+JDM8L3NwYW4+J1xuICB9LFxuICBUTUJsb2NrcXVvdGU6IHsgXG4gICAgcmVnZXhwOiAvXiggezAsM30+WyBdPykoLiopJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNQmxvY2txdW90ZVwiPiQxPC9zcGFuPiQkMidcbiAgfSxcbiAgVE1Db2RlRmVuY2VCYWNrdGlja09wZW46IHsgXG4gICAgcmVnZXhwOiAvXiggezAsM30oPzxzZXE+YGBgYCopXFxzKikoW15gXSo/KShcXHMqKSQvLCBcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUNvZGVGZW5jZUJhY2t0aWNrXCI+JDE8L3NwYW4+PHNwYW4gY2xhc3M9XCJUTUluZm9TdHJpbmdcIj4kMzwvc3Bhbj4kNCdcbiAgfSxcbiAgVE1Db2RlRmVuY2VUaWxkZU9wZW46IHsgXG4gICAgcmVnZXhwOiAvXiggezAsM30oPzxzZXE+fn5+fiopXFxzKikoLio/KShcXHMqKSQvLCBcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUNvZGVGZW5jZVRpbGRlXCI+JDE8L3NwYW4+PHNwYW4gY2xhc3M9XCJUTUluZm9TdHJpbmdcIj4kMzwvc3Bhbj4kNCdcbiAgfSxcbiAgVE1Db2RlRmVuY2VCYWNrdGlja0Nsb3NlOiB7IFxuICAgIHJlZ2V4cDogL14oIHswLDN9KD88c2VxPmBgYGAqKSkoXFxzKikkLywgXG4gICAgcmVwbGFjZW1lbnQ6ICc8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1Db2RlRmVuY2VCYWNrdGlja1wiPiQxPC9zcGFuPiQzJ1xuICB9LFxuICBUTUNvZGVGZW5jZVRpbGRlQ2xvc2U6IHsgXG4gICAgcmVnZXhwOiAvXiggezAsM30oPzxzZXE+fn5+fiopKShcXHMqKSQvLCBcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUNvZGVGZW5jZVRpbGRlXCI+JDE8L3NwYW4+JDMnXG4gIH0sXG4gIFRNQmxhbmtMaW5lOiB7IFxuICAgIHJlZ2V4cDogL14oWyBcXHRdKikkLywgXG4gICAgcmVwbGFjZW1lbnQ6ICckMCdcbiAgfSxcbiAgVE1TZXRleHRIMU1hcmtlcjogeyBcbiAgICByZWdleHA6IC9eIHswLDN9PStcXHMqJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNU2V0ZXh0SDFNYXJrZXJcIj4kMDwvc3Bhbj4nXG4gIH0sXG4gIFRNU2V0ZXh0SDJNYXJrZXI6IHsgXG4gICAgcmVnZXhwOiAvXiB7MCwzfS0rXFxzKiQvLCBcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTVNldGV4dEgxTWFya2VyXCI+JDA8L3NwYW4+J1xuICB9LFxuICBUTUhSOiB7IFxuICAgIHJlZ2V4cDogL14oIHswLDN9KFxcKlsgXFx0XSpcXCpbIFxcdF0qXFwqWyBcXHQqXSopfCgtWyBcXHRdKi1bIFxcdF0qLVsgXFx0LV0qKXwoX1sgXFx0XSpfWyBcXHRdKl9bIFxcdF9dKikpJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNSFJcIj4kMDwvc3Bhbj4nXG4gIH0sXG4gIFRNVUw6IHsgXG4gICAgcmVnZXhwOiAvXiggezAsM31bKyotXSB7MSw0fSkoLiopJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNVUxcIj4kMTwvc3Bhbj4kJDInXG4gIH0sXG4gIFRNT0w6IHsgXG4gICAgcmVnZXhwOiAvXiggezAsM31cXGR7MSw5fVsuKV0gezEsNH0pKC4qKSQvLCBcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTU9MXCI+JDE8L3NwYW4+JCQyJ1xuICB9LFxuICAvLyBUT0RPOiBUaGlzIGlzIGN1cnJlbnRseSBwcmV2ZW50aW5nIHN1Ymxpc3RzIChhbmQgYW55IGNvbnRlbnQgd2l0aGluIGxpc3QgaXRlbXMsIHJlYWxseSkgZnJvbSB3b3JraW5nXG4gIFRNSW5kZW50ZWRDb2RlOiB7IFxuICAgIHJlZ2V4cDogL14oIHs0fXxcXHQpKC4qKSQvLCBcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUluZGVudGVkQ29kZVwiPiQxPC9zcGFuPiQyJ1xuICB9LFxuICBUTUxpbmtSZWZlcmVuY2VEZWZpbml0aW9uOiB7XG4gICAgLy8gVE9ETzogTGluayBkZXN0aW5hdGlvbiBjYW4ndCBpbmNsdWRlIHVuYmFsYW5jZWQgcGFyYW50aGVzZXMsIGJ1dCB3ZSBqdXN0IGlnbm9yZSB0aGF0IGhlcmUgXG4gICAgcmVnZXhwOiAvXiggezAsM31cXFtcXHMqKShbXlxcc1xcXV0oPzpbXlxcXV18XFxcXFxcXSkqPykoXFxzKlxcXTpcXHMqKSgoPzpbXlxcczw+XSspfCg/OjwoPzpbXjw+XFxcXF18XFxcXC4pKj4pKT8oXFxzKikoKD86XFwoKD86W14oKVxcXFxdfFxcXFwuKSpcXCkpfCg/OlwiKD86W15cIlxcXFxdfFxcXFwuKSpcIil8KD86Jyg/OlteJ1xcXFxdfFxcXFwuKSonKSk/KFxccyopJC8sIFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNTGlua1JlZmVyZW5jZURlZmluaXRpb25cIj4kMTwvc3Bhbj48c3BhbiBjbGFzcz1cIlRNTGlua0xhYmVsIFRNTGlua0xhYmVsX0RlZmluaXRpb25cIj4kMjwvc3Bhbj48c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1MaW5rUmVmZXJlbmNlRGVmaW5pdGlvblwiPiQzPC9zcGFuPjxzcGFuIGNsYXNzPVwiVE1MaW5rRGVzdGluYXRpb25cIj4kNDwvc3Bhbj4kNTxzcGFuIGNsYXNzPVwiVE1MaW5rVGl0bGVcIj4kNjwvc3Bhbj4kNycsXG4gICAgbGFiZWxQbGFjZWhvbGRlcjogMiwgLy8gdGhpcyBkZWZpbmVzIHdoaWNoIHBsYWNlaG9sZGVyIGluIHRoZSBhYm92ZSByZWdleCBpcyB0aGUgbGluayBcImxhYmVsXCJcbiAgfSxcbn07XG5cbi8qKlxuICogSFRNTCBibG9ja3MgaGF2ZSBtdWx0aXBsZSBkaWZmZXJlbnQgY2xhc3NlcyBvZiBvcGVuZXIgYW5kIGNsb3Nlci4gVGhpcyBhcnJheSBkZWZpbmVzIGFsbCB0aGUgY2FzZXNcbiAqL1xudmFyIGh0bWxCbG9ja0dyYW1tYXIgPSBbXG4gIHsgc3RhcnQ6IC9eIHswLDN9PCg/OnNjcmlwdHxwcmV8c3R5bGUpKD86XFxzfD58JCkvaSwgZW5kOiAvKD86PFxcL3NjcmlwdD58PFxcL3ByZT58PFxcL3N0eWxlPikvaSwgcGFyYUludGVycnVwdDogdHJ1ZSB9LFxuICB7IHN0YXJ0OiAvXiB7MCwzfTwhLS0vLCBlbmQ6IC8tLT4vLCBwYXJhSW50ZXJydXB0OiB0cnVlIH0sXG4gIHsgc3RhcnQ6IC9eIHswLDN9PFxcPy8sIGVuZDogL1xcPz4vLCBwYXJhSW50ZXJydXB0OiB0cnVlIH0sXG4gIHsgc3RhcnQ6IC9eIHswLDN9PCFbQS1aXS8sIGVuZDogLz4vLCBwYXJhSW50ZXJydXB0IDogdHJ1ZX0sXG4gIHsgc3RhcnQ6IC9eIHswLDN9PCFcXFtDREFUQVxcWy8sIGVuZDogL1xcXVxcXT4vLCBwYXJhSW50ZXJydXB0IDogdHJ1ZX0sXG4gIHsgc3RhcnQ6IC9eIHswLDN9KD86PHw8XFwvKSg/Oktub3duVGFnKSg/Olxcc3w+fFxcLz58JCkvaSwgZW5kOiBmYWxzZSwgcGFyYUludGVycnVwdDogdHJ1ZX0sXG4gIHsgc3RhcnQ6IC9eIHswLDN9KD86SFRNTE9wZW5UYWd8SFRNTENsb3NlVGFnKVxccyokLywgZW5kOiBmYWxzZSwgcGFyYUludGVycnVwdDogZmFsc2V9LFxuXTtcblxuLyoqXG4gKiBTdHJ1Y3R1cmUgb2YgdGhlIG9iamVjdDpcbiAqIFRvcCBsZXZlbCBlbnRyaWVzIGFyZSBydWxlcywgZWFjaCBjb25zaXN0aW5nIG9mIGEgcmVndWxhciBleHByZXNzaW9ucyAoaW4gc3RyaW5nIGZvcm1hdCkgYXMgd2VsbCBhcyBhIHJlcGxhY2VtZW50LlxuICogSW4gdGhlIHJlZ3VsYXIgZXhwcmVzc2lvbnMsIHJlcGxhY2VtZW50cyBmcm9tIHRoZSBvYmplY3QgJ3JlcGxhY2VtZW50cycgd2lsbCBiZSBwcm9jZXNzZWQgYmVmb3JlIGNvbXBpbGluZyBpbnRvIHRoZSBwcm9wZXJ0eSByZWdleHAuXG4gKi9cbnZhciBpbmxpbmVHcmFtbWFyID0ge1xuICBlc2NhcGUgOiB7XG4gICAgcmVnZXhwOiAvXlxcXFwoQVNDSUlQdW5jdHVhdGlvbikvLFxuICAgIHJlcGxhY2VtZW50IDogJzxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUVzY2FwZVwiPlxcXFw8L3NwYW4+JDEnXG4gIH0sXG4gIGNvZGUgOiB7XG4gICAgcmVnZXhwOiAvXihgKykoKD86W15gXSl8KD86W15gXS4qP1teYF0pKShcXDEpLyxcbiAgICByZXBsYWNlbWVudCA6ICc8c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1Db2RlXCI+JDE8L3NwYW4+PGNvZGUgY2xhc3M9XCJUTUNvZGVcIj4kMjwvY29kZT48c3BhbiBjbGFzcz1cIlRNTWFyayBUTU1hcmtfVE1Db2RlXCI+JDM8L3NwYW4+JyBcbiAgfSxcbiAgYXV0b2xpbmsgOiB7XG4gICAgcmVnZXhwOiAvXjwoKD86U2NoZW1lOlteXFxzPD5dKil8KD86RW1haWwpKT4vLFxuICAgIHJlcGxhY2VtZW50OiAnPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrX1RNQXV0b2xpbmtcIj4mbHQ7PC9zcGFuPjxzcGFuIGNsYXNzPVwiVE1BdXRvbGlua1wiPiQxPC9zcGFuPjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya19UTUF1dG9saW5rXCI+Jmd0Ozwvc3Bhbj4nXG4gIH0sXG4gIGh0bWwgOiB7XG4gICAgcmVnZXhwOiAvXigoPzpIVE1MT3BlblRhZyl8KD86SFRNTENsb3NlVGFnKXwoPzpIVE1MQ29tbWVudCl8KD86SFRNTFBJKXwoPzpIVE1MRGVjbGFyYXRpb24pfCg/OkhUTUxDREFUQSkpLyxcbiAgICByZXBsYWNlbWVudDogJzxzcGFuIGNsYXNzPVwiVE1IVE1MXCI+JDE8L3NwYW4+JyxcbiAgfSxcbiAgbGlua09wZW4gOiB7XG4gICAgcmVnZXhwOiAvXlxcWy8sXG4gICAgcmVwbGFjZW1lbnQ6ICcnXG4gIH0sXG4gIGltYWdlT3BlbiA6IHtcbiAgICByZWdleHA6IC9eIVxcWy8sXG4gICAgcmVwbGFjZW1lbnQgOiAnJ1xuICB9LFxuICBsaW5rTGFiZWwgOiB7XG4gICAgcmVnZXhwOiAvXihcXFtcXHMqKShbXlxcXV0qPykoXFxzKlxcXSkvLFxuICAgIHJlcGxhY2VtZW50OiAnJyxcbiAgICBsYWJlbFBsYWNlaG9sZGVyOiAyXG4gIH0sXG4gIGRlZmF1bHQgOiB7XG4gICAgcmVnZXhwOiAvXigufCg/Ok5vdFRyaWdnZXJDaGFyKykpLyxcbiAgICByZXBsYWNlbWVudDogJyQxJ1xuICB9XG59O1xuXG4vLyBQcm9jZXNzIHJlcGxhY2VtZW50cyBpbiByZWdleHBzXG5jb25zdCByZXBsYWNlbWVudFJlZ2V4cCA9IG5ldyBSZWdFeHAoT2JqZWN0LmtleXMocmVwbGFjZW1lbnRzKS5qb2luKCd8JykpO1xuXG4vLyBJbmxpbmVcbmNvbnN0IGlubGluZVJ1bGVzID1bLi4uT2JqZWN0LmtleXMoaW5saW5lR3JhbW1hcildO1xuZm9yIChsZXQgcnVsZSBvZiBpbmxpbmVSdWxlcykge1xuICBsZXQgcmUgPSBpbmxpbmVHcmFtbWFyW3J1bGVdLnJlZ2V4cC5zb3VyY2U7XG4gIC8vIFJlcGxhY2Ugd2hpbGUgdGhlcmUgaXMgc29tZXRoaW5nIHRvIHJlcGxhY2UuIFRoaXMgbWVhbnMgaXQgYWxzbyB3b3JrcyBvdmVyIG11bHRpcGxlIGxldmVscyAocmVwbGFjZW1lbnRzIGNvbnRhaW5pbmcgcmVwbGFjZW1lbnRzKVxuICB3aGlsZSAocmUubWF0Y2gocmVwbGFjZW1lbnRSZWdleHApKSB7XG4gICAgcmUgPSByZS5yZXBsYWNlKHJlcGxhY2VtZW50UmVnZXhwLCAoc3RyaW5nKSA9PiB7IHJldHVybiByZXBsYWNlbWVudHNbc3RyaW5nXS5zb3VyY2U7IH0pO1xuICB9XG4gIGlubGluZUdyYW1tYXJbcnVsZV0ucmVnZXhwID0gbmV3IFJlZ0V4cChyZSwgaW5saW5lR3JhbW1hcltydWxlXS5yZWdleHAuZmxhZ3MpO1xufVxuXG4vLyBIVE1MIEJsb2NrIChvbmx5IG9wZW5pbmcgcnVsZSBpcyBwcm9jZXNzZWQgY3VycmVudGx5KVxuZm9yIChsZXQgcnVsZSBvZiBodG1sQmxvY2tHcmFtbWFyKSB7XG4gIGxldCByZSA9IHJ1bGUuc3RhcnQuc291cmNlO1xuICAvLyBSZXBsYWNlIHdoaWxlIHRoZXJlIGlzIHNvbWV0aGluZyB0byByZXBsYWNlLiBUaGlzIG1lYW5zIGl0IGFsc28gd29ya3Mgb3ZlciBtdWx0aXBsZSBsZXZlbHMgKHJlcGxhY2VtZW50cyBjb250YWluaW5nIHJlcGxhY2VtZW50cylcbiAgd2hpbGUgKHJlLm1hdGNoKHJlcGxhY2VtZW50UmVnZXhwKSkge1xuICAgIHJlID0gcmUucmVwbGFjZShyZXBsYWNlbWVudFJlZ2V4cCwgKHN0cmluZykgPT4geyByZXR1cm4gcmVwbGFjZW1lbnRzW3N0cmluZ10uc291cmNlOyB9KTtcbiAgfVxuICBydWxlLnN0YXJ0ID0gbmV3IFJlZ0V4cChyZSwgcnVsZS5zdGFydC5mbGFncyk7XG59XG5cbi8qKlxuICogRXNjYXBlcyBIVE1MIHNwZWNpYWwgY2hhcmFjdGVycyAoPCwgPiwgYW5kICYpIGluIHRoZSBzdHJpbmcuXG4gKiBAcGFyYW0ge3N0cmluZ30gc3RyaW5nIFRoZSByYXcgc3RyaW5nIHRvIGJlIGVzY2FwZWRcbiAqIEByZXR1cm5zIHtzdHJpbmd9IFRoZSBzdHJpbmcsIHJlYWR5IHRvIGJlIHVzZWQgaW4gSFRNTFxuICovXG5mdW5jdGlvbiBodG1sZXNjYXBlKHN0cmluZykge1xuICByZXR1cm4gKHN0cmluZyA/IHN0cmluZyA6ICcnKVxuICAgIC5yZXBsYWNlKC8mL2csICcmYW1wOycpXG4gICAgLnJlcGxhY2UoLzwvZywgJyZsdDsnKVxuICAgIC5yZXBsYWNlKC8+L2csICcmZ3Q7Jyk7XG59XG4vKipcbiAqIENvbnRhaW5zIHRoZSBjb21tYW5kcyB0aGF0IGNhbiBiZSBzZW50IHRvIHRoZSBlZGl0b3IuIENvbnRhaW5zIG9iamVjdHMgd2l0aCBhIG5hbWUgcmVwcmVzZW50aW5nIHRoZSBuYW1lIG9mIHRoZSBjb21tYW5kLlxuICogRWFjaCBvZiB0aGUgb2JqZWN0cyBjb250YWlucyB0aGUgZm9sbG93aW5nIGtleXM6XG4gKiBcbiAqICAgLSB0eXBlOiBDYW4gYmUgZWl0aGVyIGlubGluZSAoZm9yIGlubGluZSBmb3JtYXR0aW5nKSBvciBsaW5lIChmb3IgYmxvY2sgLyBsaW5lIGZvcm1hdHRpbmcpLlxuICogICAtIGNsYXNzTmFtZTogVXNlZCB0byBkZXRlcm1pbmUgd2hldGhlciB0aGUgY29tbWFuZCBpcyBhY3RpdmUgYXQgYSBnaXZlbiBwb3NpdGlvbi4gXG4gKiAgICAgRm9yIGxpbmUgZm9ybWF0dGluZywgdGhpcyBsb29rcyBhdCB0aGUgY2xhc3Mgb2YgdGhlIGxpbmUgZWxlbWVudC4gRm9yIGlubGluZSBlbGVtZW50cywgdHJpZXMgdG8gZmluZCBhbiBlbmNsb3NpbmcgZWxlbWVudCB3aXRoIHRoYXQgY2xhc3MuXG4gKiAgIC0gc2V0IC8gdW5zZXQ6IENvbnRhaW4gaW5zdHJ1Y3Rpb25zIGhvdyB0byBzZXQgYW5kIHVuc2V0IHRoZSBjb21tYW5kLiBGb3IgbGluZSB0eXBlIGNvbW1hbmRzLCBib3RoIGNvbnNpc3Qgb2YgYSBwYXR0ZXJuIGFuZCByZXBsYWNlbWVudCB0aGF0IFxuICogICAgIHdpbGwgYmUgYXBwbGllZCB0byBlYWNoIGxpbmUgKHVzaW5nIFN0cmluZy5yZXBsYWNlKS4gRm9yIGlubGluZSB0eXBlIGNvbW1hbmRzLCB0aGUgc2V0IG9iamVjdCBjb250YWlucyBhIHByZSBhbmQgcG9zdCBzdHJpbmcgd2hpY2ggd2lsbFxuICogICAgIGJlIGluc2VydGVkIGJlZm9yZSBhbmQgYWZ0ZXIgdGhlIHNlbGVjdGlvbi4gVGhlIHVuc2V0IG9iamVjdCBjb250YWlucyBhIHByZVBhdHRlcm4gYW5kIGEgcG9zdFBhdHRlcm4uIEJvdGggc2hvdWxkIGJlIHJlZ3VsYXIgZXhwcmVzc2lvbnMgYW5kIFxuICogICAgIHRoZXkgd2lsbCBiZSBhcHBsaWVkIHRvIHRoZSBwb3J0aW9uIG9mIHRoZSBsaW5lIGJlZm9yZSBhbmQgYWZ0ZXIgdGhlIHNlbGVjdGlvbiAodXNpbmcgU3RyaW5nLnJlcGxhY2UsIHdpdGggYW4gZW1wdHkgcmVwbGFjZW1lbnQgc3RyaW5nKS5cbiAqL1xuY29uc3QgY29tbWFuZHMgPSB7XG4gIC8vIFJlcGxhY2VtZW50cyBmb3IgdW5zZXQgZm9yIGlubGluZSBjb21tYW5kcyBhcmUgJycgYnkgZGVmYXVsdFxuICBib2xkOiB7XG4gICAgdHlwZTogJ2lubGluZScsIFxuICAgIGNsYXNzTmFtZTogJ1RNU3Ryb25nJywgXG4gICAgc2V0OiB7cHJlOiAnKionLCBwb3N0OiAnKionfSwgXG4gICAgdW5zZXQ6IHtwcmVQYXR0ZXJuOiAvKD86XFwqXFwqfF9fKSQvLCBwb3N0UGF0dGVybjogL14oPzpcXCpcXCp8X18pL31cbiAgfSwgXG4gIGl0YWxpYzoge1xuICAgIHR5cGU6ICdpbmxpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTUVtJywgXG4gICAgc2V0OiB7cHJlOiAnKicsIHBvc3Q6ICcqJ30sIFxuICAgIHVuc2V0OiB7cHJlUGF0dGVybjogLyg/OlxcKnxfKSQvLCBwb3N0UGF0dGVybjogL14oPzpcXCp8XykvfVxuICB9LFxuICBjb2RlOiB7XG4gICAgdHlwZTogJ2lubGluZScsIFxuICAgIGNsYXNzTmFtZTogJ1RNQ29kZScsIFxuICAgIHNldDoge3ByZTogJ2AnLCBwb3N0OiAnYCd9LCBcbiAgICB1bnNldDoge3ByZVBhdHRlcm46IC9gKyQvLCBwb3N0UGF0dGVybjogL15gKy99IC8vIEZJWE1FIHRoaXMgZG9lc24ndCBlbnN1cmUgYmFsYW5jZWQgYmFja3RpY2tzIHJpZ2h0IG5vd1xuICB9LCBcbiAgc3RyaWtldGhyb3VnaDoge1xuICAgIHR5cGU6ICdpbmxpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTVN0cmlrZXRocm91Z2gnLCBcbiAgICBzZXQ6IHtwcmU6ICd+ficsIHBvc3Q6ICd+fid9LCBcbiAgICB1bnNldDoge3ByZVBhdHRlcm46L35+JC8sIHBvc3RQYXR0ZXJuOiAvXn5+LyB9XG4gIH0sXG4gIGgxOiB7XG4gICAgdHlwZTogJ2xpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTUgxJywgXG4gICAgc2V0OiB7cGF0dGVybjogL14oIHswLDN9KD86KD86Iyt8WzAtOV17MSw5fVspLl18Wz5cXC0qK10pXFxzKyk/KSguKikkLywgcmVwbGFjZW1lbnQ6ICcjICQyJ30sIFxuICAgIHVuc2V0OiB7cGF0dGVybjogL14oIHswLDN9I1xccyspKC4qPykoKD86XFxzKyMrXFxzKik/KSQvLCByZXBsYWNlbWVudDogJyQyJ31cbiAgfSxcbiAgaDI6IHtcbiAgICB0eXBlOiAnbGluZScsIFxuICAgIGNsYXNzTmFtZTogJ1RNSDInLCBcbiAgICBzZXQ6IHtwYXR0ZXJuOiAvXiggezAsM30oPzooPzojK3xbMC05XXsxLDl9WykuXXxbPlxcLSorXSlcXHMrKT8pKC4qKSQvLCByZXBsYWNlbWVudDogJyMjICQyJ30sIFxuICAgIHVuc2V0OiB7cGF0dGVybjogL14oIHswLDN9IyNcXHMrKSguKj8pKCg/OlxccysjK1xccyopPykkLywgcmVwbGFjZW1lbnQ6ICckMid9XG4gIH0sXG4gIGgzOiB7XG4gICAgdHlwZTogJ2xpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTUgzJywgXG4gICAgc2V0OiB7cGF0dGVybjogL14oIHswLDN9KD86KD86Iyt8WzAtOV17MSw5fVspLl18Wz5cXC0qK10pXFxzKyk/KSguKikkLywgcmVwbGFjZW1lbnQ6ICcjIyMgJDInfSwgXG4gICAgdW5zZXQ6IHtwYXR0ZXJuOiAvXiggezAsM30jIyNcXHMrKSguKj8pKCg/OlxccysjK1xccyopPykkLywgcmVwbGFjZW1lbnQ6ICckMid9XG4gIH0sXG4gIGg0OiB7XG4gICAgdHlwZTogJ2xpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTUg0JywgXG4gICAgc2V0OiB7cGF0dGVybjogL14oIHswLDN9KD86KD86Iyt8WzAtOV17MSw5fVspLl18Wz5cXC0qK10pXFxzKyk/KSguKikkLywgcmVwbGFjZW1lbnQ6ICcjIyMjICQyJ30sIFxuICAgIHVuc2V0OiB7cGF0dGVybjogL14oIHswLDN9IyMjI1xccyspKC4qPykoKD86XFxzKyMrXFxzKik/KSQvLCByZXBsYWNlbWVudDogJyQyJ31cbiAgfSxcbiAgaDU6IHtcbiAgICB0eXBlOiAnbGluZScsIFxuICAgIGNsYXNzTmFtZTogJ1RNSDUnLCBcbiAgICBzZXQ6IHtwYXR0ZXJuOiAvXiggezAsM30oPzooPzojK3xbMC05XXsxLDl9WykuXXxbPlxcLSorXSlcXHMrKT8pKC4qKSQvLCByZXBsYWNlbWVudDogJyMjIyMjICQyJ30sIFxuICAgIHVuc2V0OiB7cGF0dGVybjogL14oIHswLDN9IyMjIyNcXHMrKSguKj8pKCg/OlxccysjK1xccyopPykkLywgcmVwbGFjZW1lbnQ6ICckMid9XG4gIH0sXG4gIGg2OiB7XG4gICAgdHlwZTogJ2xpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTUg2JywgXG4gICAgc2V0OiB7cGF0dGVybjogL14oIHswLDN9KD86KD86Iyt8WzAtOV17MSw5fVspLl18Wz5cXC0qK10pXFxzKyk/KSguKikkLywgcmVwbGFjZW1lbnQ6ICcjIyMjIyMgJDInfSwgXG4gICAgdW5zZXQ6IHtwYXR0ZXJuOiAvXiggezAsM30jIyMjIyNcXHMrKSguKj8pKCg/OlxccysjK1xccyopPykkLywgcmVwbGFjZW1lbnQ6ICckMid9XG4gIH0sXG4gIHVsOiB7XG4gICAgdHlwZTogJ2xpbmUnLCBcbiAgICBjbGFzc05hbWU6ICdUTVVMJywgXG4gICAgc2V0OiB7cGF0dGVybjogL14oIHswLDN9KD86KD86Iyt8WzAtOV17MSw5fVspLl18Wz5cXC0qK10pXFxzKyk/KSguKikkLywgcmVwbGFjZW1lbnQ6ICctICQyJ30sIFxuICAgIHVuc2V0OiB7cGF0dGVybjogL14oIHswLDN9WysqLV0gezEsNH0pKC4qKSQvLCByZXBsYWNlbWVudDogJyQyJ31cbiAgfSxcbiAgb2w6IHtcbiAgICB0eXBlOiAnbGluZScsIFxuICAgIGNsYXNzTmFtZTogJ1RNT0wnLCBcbiAgICBzZXQ6IHtwYXR0ZXJuOiAvXiggezAsM30oPzooPzojK3xbMC05XXsxLDl9WykuXXxbPlxcLSorXSlcXHMrKT8pKC4qKSQvLCByZXBsYWNlbWVudDogJyQjLiAkMid9LCBcbiAgICB1bnNldDoge3BhdHRlcm46IC9eKCB7MCwzfVxcZHsxLDl9Wy4pXSB7MSw0fSkoLiopJC8sIHJlcGxhY2VtZW50OiAnJDInfVxuICB9LCBcbiAgYmxvY2txdW90ZToge1xuICAgIHR5cGU6ICdsaW5lJywgXG4gICAgY2xhc3NOYW1lOiAnVE1CbG9ja3F1b3RlJywgXG4gICAgc2V0OiB7cGF0dGVybjogL14oIHswLDN9KD86KD86Iyt8WzAtOV17MSw5fVspLl18Wz5cXC0qK10pXFxzKyk/KSguKikkLywgcmVwbGFjZW1lbnQ6ICc+ICQyJ30sIFxuICAgIHVuc2V0OiB7cGF0dGVybjogL14oIHswLDN9PlsgXT8pKC4qKSQvLCByZXBsYWNlbWVudDogJyQyJ31cbiAgfSxcbn07XG5cbmV4cG9ydCB7IGxpbmVHcmFtbWFyLCBpbmxpbmVHcmFtbWFyLCBwdW5jdHVhdGlvbkxlYWRpbmcsIHB1bmN0dWF0aW9uVHJhaWxpbmcsIGh0bWxlc2NhcGUsIGh0bWxCbG9ja0dyYW1tYXIsIGNvbW1hbmRzIH07IiwiaW1wb3J0IHtcbiAgaW5saW5lR3JhbW1hcixcbiAgbGluZUdyYW1tYXIsXG4gIHB1bmN0dWF0aW9uTGVhZGluZyxcbiAgcHVuY3R1YXRpb25UcmFpbGluZyxcbiAgaHRtbGVzY2FwZSxcbiAgaHRtbEJsb2NrR3JhbW1hcixcbiAgY29tbWFuZHMsXG59IGZyb20gXCIuL2dyYW1tYXJcIjtcblxuY2xhc3MgRWRpdG9yIHtcbiAgY29uc3RydWN0b3IocHJvcHMgPSB7fSkge1xuICAgIHRoaXMuZSA9IG51bGw7XG4gICAgdGhpcy50ZXh0YXJlYSA9IG51bGw7XG4gICAgdGhpcy5saW5lcyA9IFtdO1xuICAgIHRoaXMubGluZUVsZW1lbnRzID0gW107XG4gICAgdGhpcy5saW5lVHlwZXMgPSBbXTtcbiAgICB0aGlzLmxpbmVDYXB0dXJlcyA9IFtdO1xuICAgIHRoaXMubGluZVJlcGxhY2VtZW50cyA9IFtdO1xuICAgIHRoaXMubGlua0xhYmVscyA9IFtdO1xuICAgIHRoaXMubGluZURpcnR5ID0gW107XG4gICAgdGhpcy5sYXN0Q29tbWFuZFN0YXRlID0gbnVsbDtcblxuICAgIHRoaXMubGlzdGVuZXJzID0ge1xuICAgICAgY2hhbmdlOiBbXSxcbiAgICAgIHNlbGVjdGlvbjogW10sXG4gICAgICBkcm9wOiBbXSxcbiAgICB9O1xuXG4gICAgbGV0IGVsZW1lbnQgPSBwcm9wcy5lbGVtZW50O1xuICAgIHRoaXMudGV4dGFyZWEgPSBwcm9wcy50ZXh0YXJlYTtcblxuICAgIGlmICh0aGlzLnRleHRhcmVhKSB7XG4gICAgICBpZiAoIXRoaXMudGV4dGFyZWEudGFnTmFtZSkge1xuICAgICAgICB0aGlzLnRleHRhcmVhID0gZG9jdW1lbnQuZ2V0RWxlbWVudEJ5SWQodGhpcy50ZXh0YXJlYSk7XG4gICAgICB9XG4gICAgICBpZiAoIWVsZW1lbnQpIGVsZW1lbnQgPSB0aGlzLnRleHRhcmVhO1xuICAgIH1cblxuICAgIGlmIChlbGVtZW50ICYmICFlbGVtZW50LnRhZ05hbWUpIHtcbiAgICAgIGVsZW1lbnQgPSBkb2N1bWVudC5nZXRFbGVtZW50QnlJZChwcm9wcy5lbGVtZW50KTtcbiAgICB9XG4gICAgaWYgKCFlbGVtZW50KSB7XG4gICAgICBlbGVtZW50ID0gZG9jdW1lbnQuZ2V0RWxlbWVudHNCeVRhZ05hbWUoXCJib2R5XCIpWzBdO1xuICAgIH1cbiAgICBpZiAoZWxlbWVudC50YWdOYW1lID09IFwiVEVYVEFSRUFcIikge1xuICAgICAgdGhpcy50ZXh0YXJlYSA9IGVsZW1lbnQ7XG4gICAgICBlbGVtZW50ID0gdGhpcy50ZXh0YXJlYS5wYXJlbnROb2RlO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnRleHRhcmVhKSB7XG4gICAgICB0aGlzLnRleHRhcmVhLnN0eWxlLmRpc3BsYXkgPSBcIm5vbmVcIjtcbiAgICB9XG5cbiAgICB0aGlzLmNyZWF0ZUVkaXRvckVsZW1lbnQoZWxlbWVudCwgcHJvcHMpO1xuICAgIHRoaXMuc2V0Q29udGVudChcbiAgICAgIHR5cGVvZiBwcm9wcy5jb250ZW50ID09PSBcInN0cmluZ1wiXG4gICAgICAgID8gcHJvcHMuY29udGVudFxuICAgICAgICA6IHRoaXMudGV4dGFyZWFcbiAgICAgICAgPyB0aGlzLnRleHRhcmVhLnZhbHVlXG4gICAgICAgIDogXCIjIEhlbGxvIFRpbnlNREUhXFxuRWRpdCAqKmhlcmUqKlwiXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIHRoZSBlZGl0b3IgZWxlbWVudCBpbnNpZGUgdGhlIHRhcmdldCBlbGVtZW50IG9mIHRoZSBET00gdHJlZVxuICAgKiBAcGFyYW0gZWxlbWVudCBUaGUgdGFyZ2V0IGVsZW1lbnQgb2YgdGhlIERPTSB0cmVlXG4gICAqIEBwYXJhbSBwcm9wcyBvcHRpb25zLCBwYXNzZWQgZnJvbSBjb25zdHJ1Y3RvcidzIHByb3BzXG4gICAqL1xuICBjcmVhdGVFZGl0b3JFbGVtZW50KGVsZW1lbnQsIHByb3BzKSB7XG4gICAgaWYgKHByb3BzICYmIHByb3BzLmVkaXRvciAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBpZiAocHJvcHMuZWRpdG9yLnRhZ05hbWUpIHRoaXMuZSA9IHByb3BzLmVkaXRvcjtcbiAgICAgIGVsc2UgdGhpcy5lID0gZG9jdW1lbnQuZ2V0RWxlbWVudEJ5SWQocHJvcHMuZWRpdG9yKTtcbiAgICB9IGVsc2UgdGhpcy5lID0gZG9jdW1lbnQuY3JlYXRlRWxlbWVudChcImRpdlwiKTtcblxuICAgIHRoaXMuZS5jbGFzc0xpc3QuYWRkKFwiVGlueU1ERVwiKTtcbiAgICB0aGlzLmUuY29udGVudEVkaXRhYmxlID0gdHJ1ZTtcbiAgICAvLyBUaGUgZm9sbG93aW5nIGlzIGltcG9ydGFudCBmb3IgZm9ybWF0dGluZyBwdXJwb3NlcywgYnV0IGFsc28gc2luY2Ugb3RoZXJ3aXNlIHRoZSBicm93c2VyIHJlcGxhY2VzIHN1YnNlcXVlbnQgc3BhY2VzIHdpdGggICZuYnNwOyAmbmJzcDtcbiAgICAvLyBUaGF0IGJyZWFrcyBhIGxvdCBvZiBzdHVmZiwgc28gd2UgZG8gdGhpcyBoZXJlIGFuZCBub3QgaW4gQ1NT4oCUdGhlcmVmb3JlLCB5b3UgZG9uJ3QgaGF2ZSB0byByZW1lbWJlciB0byBwdXQgdGhpcyBpbiB0aGUgQ1NTIGZpbGVcbiAgICB0aGlzLmUuc3R5bGUud2hpdGVTcGFjZSA9IFwicHJlLXdyYXBcIjtcbiAgICAvLyBBdm9pZCBmb3JtYXR0aW5nIChCIC8gSSAvIFUpIHBvcHBpbmcgdXAgb24gaU9TXG4gICAgdGhpcy5lLnN0eWxlLndlYmtpdFVzZXJNb2RpZnkgPSBcInJlYWQtd3JpdGUtcGxhaW50ZXh0LW9ubHlcIjtcblxuICAgIGlmIChwcm9wcy5lZGl0b3IgPT09IHVuZGVmaW5lZCkge1xuICAgICAgaWYgKFxuICAgICAgICB0aGlzLnRleHRhcmVhICYmXG4gICAgICAgIHRoaXMudGV4dGFyZWEucGFyZW50Tm9kZSA9PSBlbGVtZW50ICYmXG4gICAgICAgIHRoaXMudGV4dGFyZWEubmV4dFNpYmxpbmdcbiAgICAgICkge1xuICAgICAgICBlbGVtZW50Lmluc2VydEJlZm9yZSh0aGlzLmUsIHRoaXMudGV4dGFyZWEubmV4dFNpYmxpbmcpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgZWxlbWVudC5hcHBlbmRDaGlsZCh0aGlzLmUpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMuZS5hZGRFdmVudExpc3RlbmVyKFwiaW5wdXRcIiwgKGUpID0+IHRoaXMuaGFuZGxlSW5wdXRFdmVudChlKSk7XG4gICAgdGhpcy5lLmFkZEV2ZW50TGlzdGVuZXIoXCJjb21wb3NpdGlvbmVuZFwiLCAoZSkgPT4gdGhpcy5oYW5kbGVJbnB1dEV2ZW50KGUpKTtcbiAgICBkb2N1bWVudC5hZGRFdmVudExpc3RlbmVyKFwic2VsZWN0aW9uY2hhbmdlXCIsIChlKSA9PlxuICAgICAgdGhpcy5oYW5kbGVTZWxlY3Rpb25DaGFuZ2VFdmVudChlKVxuICAgICk7XG4gICAgdGhpcy5lLmFkZEV2ZW50TGlzdGVuZXIoXCJwYXN0ZVwiLCAoZSkgPT4gdGhpcy5oYW5kbGVQYXN0ZShlKSk7XG4gICAgdGhpcy5lLmFkZEV2ZW50TGlzdGVuZXIoXCJkcm9wXCIsIChlKSA9PiB0aGlzLmhhbmRsZURyb3AoZSkpO1xuICAgIHRoaXMubGluZUVsZW1lbnRzID0gdGhpcy5lLmNoaWxkTm9kZXM7IC8vIHRoaXMgd2lsbCBhdXRvbWF0aWNhbGx5IHVwZGF0ZVxuICB9XG5cbiAgLyoqXG4gICAqIFNldHMgdGhlIGVkaXRvciBjb250ZW50LlxuICAgKiBAcGFyYW0ge3N0cmluZ30gY29udGVudCBUaGUgbmV3IE1hcmtkb3duIGNvbnRlbnRcbiAgICovXG4gIHNldENvbnRlbnQoY29udGVudCkge1xuICAgIC8vIERlbGV0ZSBhbnkgZXhpc3RpbmcgY29udGVudFxuICAgIHdoaWxlICh0aGlzLmUuZmlyc3RDaGlsZCkge1xuICAgICAgdGhpcy5lLnJlbW92ZUNoaWxkKHRoaXMuZS5maXJzdENoaWxkKTtcbiAgICB9XG4gICAgdGhpcy5saW5lcyA9IGNvbnRlbnQuc3BsaXQoLyg/OlxcclxcbnxcXHJ8XFxuKS8pO1xuICAgIHRoaXMubGluZURpcnR5ID0gW107XG4gICAgZm9yIChsZXQgbGluZU51bSA9IDA7IGxpbmVOdW0gPCB0aGlzLmxpbmVzLmxlbmd0aDsgbGluZU51bSsrKSB7XG4gICAgICBsZXQgbGUgPSBkb2N1bWVudC5jcmVhdGVFbGVtZW50KFwiZGl2XCIpO1xuICAgICAgdGhpcy5lLmFwcGVuZENoaWxkKGxlKTtcbiAgICAgIHRoaXMubGluZURpcnR5LnB1c2godHJ1ZSk7XG4gICAgfVxuICAgIHRoaXMubGluZVR5cGVzID0gbmV3IEFycmF5KHRoaXMubGluZXMubGVuZ3RoKTtcbiAgICB0aGlzLnVwZGF0ZUZvcm1hdHRpbmcoKTtcbiAgICB0aGlzLmZpcmVDaGFuZ2UoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIHRoZSBlZGl0b3IgY29udGVudCBhcyBhIE1hcmtkb3duIHN0cmluZy5cbiAgICogQHJldHVybnMge3N0cmluZ30gVGhlIGVkaXRvciBjb250ZW50IGFzIGEgbWFya2Rvd24gc3RyaW5nXG4gICAqL1xuICBnZXRDb250ZW50KCkge1xuICAgIHJldHVybiB0aGlzLmxpbmVzLmpvaW4oXCJcXG5cIik7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBpcyB0aGUgbWFpbiBtZXRob2QgdG8gdXBkYXRlIHRoZSBmb3JtYXR0aW5nIChmcm9tIHRoaXMubGluZXMgdG8gSFRNTCBvdXRwdXQpXG4gICAqL1xuICB1cGRhdGVGb3JtYXR0aW5nKCkge1xuICAgIC8vIEZpcnN0LCBwYXJzZSBsaW5lIHR5cGVzLiBUaGlzIHdpbGwgdXBkYXRlIHRoaXMubGluZVR5cGVzLCB0aGlzLmxpbmVSZXBsYWNlbWVudHMsIGFuZCB0aGlzLmxpbmVDYXB0dXJlc1xuICAgIC8vIFdlIGRvbid0IGFwcGx5IHRoZSBmb3JtYXR0aW5nIHlldFxuICAgIHRoaXMudXBkYXRlTGluZVR5cGVzKCk7XG4gICAgLy8gQ29sbGVjdCBhbnkgdmFsaWQgbGluayBsYWJlbHMgZnJvbSBsaW5rIHJlZmVyZW5jZSBkZWZpbml0aW9uc+KAlHdlIG5lZWQgdGhhdCBmb3IgZm9ybWF0dGluZyB0byBkZXRlcm1pbmUgd2hhdCdzIGEgdmFsaWQgbGlua1xuICAgIHRoaXMudXBkYXRlTGlua0xhYmVscygpO1xuICAgIC8vIE5vdywgYXBwbHkgdGhlIGZvcm1hdHRpbmdcbiAgICB0aGlzLmFwcGx5TGluZVR5cGVzKCk7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGlzLmxpbmtMYWJlbHM6IEZvciBldmVyeSBsaW5rIHJlZmVyZW5jZSBkZWZpbml0aW9uIChsaW5lIHR5cGUgVE1MaW5rUmVmZXJlbmNlRGVmaW5pdGlvbiksIHdlIGNvbGxlY3QgdGhlIGxhYmVsXG4gICAqL1xuICB1cGRhdGVMaW5rTGFiZWxzKCkge1xuICAgIHRoaXMubGlua0xhYmVscyA9IFtdO1xuICAgIGZvciAobGV0IGwgPSAwOyBsIDwgdGhpcy5saW5lcy5sZW5ndGg7IGwrKykge1xuICAgICAgaWYgKHRoaXMubGluZVR5cGVzW2xdID09IFwiVE1MaW5rUmVmZXJlbmNlRGVmaW5pdGlvblwiKSB7XG4gICAgICAgIHRoaXMubGlua0xhYmVscy5wdXNoKFxuICAgICAgICAgIHRoaXMubGluZUNhcHR1cmVzW2xdW1xuICAgICAgICAgICAgbGluZUdyYW1tYXIuVE1MaW5rUmVmZXJlbmNlRGVmaW5pdGlvbi5sYWJlbFBsYWNlaG9sZGVyXG4gICAgICAgICAgXVxuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBIZWxwZXIgZnVuY3Rpb24gdG8gcmVwbGFjZSBwbGFjZWhvbGRlcnMgZnJvbSBhIFJlZ0V4cCBjYXB0dXJlLiBUaGUgcmVwbGFjZW1lbnQgc3RyaW5nIGNhbiBjb250YWluIHJlZ3VsYXIgZG9sbGFyIHBsYWNlaG9sZGVycyAoZS5nLiwgJDEpLFxuICAgKiB3aGljaCBhcmUgaW50ZXJwcmV0ZWQgbGlrZSBpbiBTdHJpbmcucmVwbGFjZSgpLCBidXQgYWxzbyBkb3VibGUgZG9sbGFyIHBsYWNlaG9sZGVycyAoJCQxKS4gSW4gdGhlIGNhc2Ugb2YgZG91YmxlIGRvbGxhciBwbGFjZWhvbGRlcnMsXG4gICAqIE1hcmtkb3duIGlubGluZSBncmFtbWFyIGlzIGFwcGxpZWQgb24gdGhlIGNvbnRlbnQgb2YgdGhlIGNhcHR1cmVkIHN1Ymdyb3VwLCBpLmUuLCAkJDEgcHJvY2Vzc2VzIGlubGluZSBNYXJrZG93biBncmFtbWFyIGluIHRoZSBjb250ZW50IG9mIHRoZVxuICAgKiBmaXJzdCBjYXB0dXJlZCBzdWJncm91cCwgYW5kIHJlcGxhY2VzIGAkJDFgIHdpdGggdGhlIHJlc3VsdC5cbiAgICpcbiAgICogQHBhcmFtIHtzdHJpbmd9IHJlcGxhY2VtZW50IFRoZSByZXBsYWNlbWVudCBzdHJpbmcsIGluY2x1ZGluZyBwbGFjZWhvbGRlcnMuXG4gICAqIEBwYXJhbSAgY2FwdHVyZSBUaGUgcmVzdWx0IG9mIGEgUmVnRXhwLmV4ZWMoKSBjYWxsXG4gICAqIEByZXR1cm5zIFRoZSByZXBsYWNlbWVudCBzdHJpbmcsIHdpdGggcGxhY2Vob2xkZXJzIHJlcGxhY2VkIGZyb20gdGhlIGNhcHR1cmUgcmVzdWx0LlxuICAgKi9cbiAgcmVwbGFjZShyZXBsYWNlbWVudCwgY2FwdHVyZSkge1xuICAgIHJldHVybiByZXBsYWNlbWVudC5yZXBsYWNlKC8oXFwkezEsMn0pKFswLTldKS9nLCAoc3RyLCBwMSwgcDIpID0+IHtcbiAgICAgIGlmIChwMSA9PSBcIiRcIikgcmV0dXJuIGh0bWxlc2NhcGUoY2FwdHVyZVtwMl0pO1xuICAgICAgZWxzZVxuICAgICAgICByZXR1cm4gYDxzcGFuIGNsYXNzPVwiVE1JbmxpbmVGb3JtYXR0ZWRcIj4ke3RoaXMucHJvY2Vzc0lubGluZVN0eWxlcyhcbiAgICAgICAgICBjYXB0dXJlW3AyXVxuICAgICAgICApfTwvc3Bhbj5gO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFwcGxpZXMgdGhlIGxpbmUgdHlwZXMgKGZyb20gdGhpcy5saW5lVHlwZXMgYXMgd2VsbCBhcyB0aGUgY2FwdHVyZSByZXN1bHQgaW4gdGhpcy5saW5lUmVwbGFjZW1lbnRzIGFuZCB0aGlzLmxpbmVDYXB0dXJlcylcbiAgICogYW5kIHByb2Nlc3NlcyBpbmxpbmUgZm9ybWF0dGluZyBmb3IgYWxsIGxpbmVzLlxuICAgKi9cbiAgYXBwbHlMaW5lVHlwZXMoKSB7XG4gICAgZm9yIChsZXQgbGluZU51bSA9IDA7IGxpbmVOdW0gPCB0aGlzLmxpbmVzLmxlbmd0aDsgbGluZU51bSsrKSB7XG4gICAgICBpZiAodGhpcy5saW5lRGlydHlbbGluZU51bV0pIHtcbiAgICAgICAgbGV0IGNvbnRlbnRIVE1MID0gdGhpcy5yZXBsYWNlKFxuICAgICAgICAgIHRoaXMubGluZVJlcGxhY2VtZW50c1tsaW5lTnVtXSxcbiAgICAgICAgICB0aGlzLmxpbmVDYXB0dXJlc1tsaW5lTnVtXVxuICAgICAgICApO1xuICAgICAgICAvLyB0aGlzLmxpbmVIVE1MW2xpbmVOdW1dID0gKGNvbnRlbnRIVE1MID09ICcnID8gJzxiciAvPicgOiBjb250ZW50SFRNTCk7IC8vIFByZXZlbnQgZW1wdHkgZWxlbWVudHMgd2hpY2ggY2FuJ3QgYmUgc2VsZWN0ZWQgZXRjLlxuICAgICAgICB0aGlzLmxpbmVFbGVtZW50c1tsaW5lTnVtXS5jbGFzc05hbWUgPSB0aGlzLmxpbmVUeXBlc1tsaW5lTnVtXTtcbiAgICAgICAgdGhpcy5saW5lRWxlbWVudHNbbGluZU51bV0ucmVtb3ZlQXR0cmlidXRlKFwic3R5bGVcIik7XG4gICAgICAgIHRoaXMubGluZUVsZW1lbnRzW2xpbmVOdW1dLmlubmVySFRNTCA9XG4gICAgICAgICAgY29udGVudEhUTUwgPT0gXCJcIiA/IFwiPGJyIC8+XCIgOiBjb250ZW50SFRNTDsgLy8gUHJldmVudCBlbXB0eSBlbGVtZW50cyB3aGljaCBjYW4ndCBiZSBzZWxlY3RlZCBldGMuXG4gICAgICB9XG4gICAgICB0aGlzLmxpbmVFbGVtZW50c1tsaW5lTnVtXS5kYXRhc2V0LmxpbmVOdW0gPSBsaW5lTnVtO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIGxpbmUgdHlwZXMgZm9yIGFsbCBsaW5lcyBiYXNlZCBvbiB0aGUgbGluZSAvIGJsb2NrIGdyYW1tYXIuIENhcHR1cmVzIHRoZSByZXN1bHRzIG9mIHRoZSByZXNwZWN0aXZlIGxpbmVcbiAgICogZ3JhbW1hciByZWd1bGFyIGV4cHJlc3Npb25zLlxuICAgKiBVcGRhdGVzIHRoaXMubGluZVR5cGVzLCB0aGlzLmxpbmVDYXB0dXJlcywgYW5kIHRoaXMubGluZVJlcGxhY2VtZW50cywgYXMgd2VsbCBhcyB0aGlzLmxpbmVEaXJ0eS5cbiAgICovXG4gIHVwZGF0ZUxpbmVUeXBlcygpIHtcbiAgICBsZXQgY29kZUJsb2NrVHlwZSA9IGZhbHNlO1xuICAgIGxldCBjb2RlQmxvY2tTZXFMZW5ndGggPSAwO1xuICAgIGxldCBodG1sQmxvY2sgPSBmYWxzZTtcblxuICAgIGZvciAobGV0IGxpbmVOdW0gPSAwOyBsaW5lTnVtIDwgdGhpcy5saW5lcy5sZW5ndGg7IGxpbmVOdW0rKykge1xuICAgICAgbGV0IGxpbmVUeXBlID0gXCJUTVBhcmFcIjtcbiAgICAgIGxldCBsaW5lQ2FwdHVyZSA9IFt0aGlzLmxpbmVzW2xpbmVOdW1dXTtcbiAgICAgIGxldCBsaW5lUmVwbGFjZW1lbnQgPSBcIiQkMFwiOyAvLyBEZWZhdWx0IHJlcGxhY2VtZW50IGZvciBwYXJhZ3JhcGg6IElubGluZSBmb3JtYXQgdGhlIGVudGlyZSBsaW5lXG5cbiAgICAgIC8vIENoZWNrIG9uZ29pbmcgY29kZSBibG9ja3NcbiAgICAgIC8vIGlmIChsaW5lTnVtID4gMCAmJiAodGhpcy5saW5lVHlwZXNbbGluZU51bSAtIDFdID09ICdUTUNvZGVGZW5jZUJhY2t0aWNrT3BlbicgfHwgdGhpcy5saW5lVHlwZXNbbGluZU51bSAtIDFdID09ICdUTUZlbmNlZENvZGVCYWNrdGljaycpKSB7XG4gICAgICBpZiAoY29kZUJsb2NrVHlwZSA9PSBcIlRNQ29kZUZlbmNlQmFja3RpY2tPcGVuXCIpIHtcbiAgICAgICAgLy8gV2UncmUgaW4gYSBiYWNrdGljay1mZW5jZWQgY29kZSBibG9jaywgY2hlY2sgaWYgdGhlIGN1cnJlbnQgbGluZSBjbG9zZXMgaXRcbiAgICAgICAgbGV0IGNhcHR1cmUgPSBsaW5lR3JhbW1hci5UTUNvZGVGZW5jZUJhY2t0aWNrQ2xvc2UucmVnZXhwLmV4ZWMoXG4gICAgICAgICAgdGhpcy5saW5lc1tsaW5lTnVtXVxuICAgICAgICApO1xuICAgICAgICBpZiAoY2FwdHVyZSAmJiBjYXB0dXJlLmdyb3Vwc1tcInNlcVwiXS5sZW5ndGggPj0gY29kZUJsb2NrU2VxTGVuZ3RoKSB7XG4gICAgICAgICAgbGluZVR5cGUgPSBcIlRNQ29kZUZlbmNlQmFja3RpY2tDbG9zZVwiO1xuICAgICAgICAgIGxpbmVSZXBsYWNlbWVudCA9IGxpbmVHcmFtbWFyLlRNQ29kZUZlbmNlQmFja3RpY2tDbG9zZS5yZXBsYWNlbWVudDtcbiAgICAgICAgICBsaW5lQ2FwdHVyZSA9IGNhcHR1cmU7XG4gICAgICAgICAgY29kZUJsb2NrVHlwZSA9IGZhbHNlO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIGxpbmVUeXBlID0gXCJUTUZlbmNlZENvZGVCYWNrdGlja1wiO1xuICAgICAgICAgIGxpbmVSZXBsYWNlbWVudCA9ICc8c3BhbiBjbGFzcz1cIlRNRmVuY2VkQ29kZVwiPiQwPGJyIC8+PC9zcGFuPic7XG4gICAgICAgICAgbGluZUNhcHR1cmUgPSBbdGhpcy5saW5lc1tsaW5lTnVtXV07XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIC8vIGlmIChsaW5lTnVtID4gMCAmJiAodGhpcy5saW5lVHlwZXNbbGluZU51bSAtIDFdID09ICdUTUNvZGVGZW5jZVRpbGRlT3BlbicgfHwgdGhpcy5saW5lVHlwZXNbbGluZU51bSAtIDFdID09ICdUTUZlbmNlZENvZGVUaWxkZScpKSB7XG4gICAgICBlbHNlIGlmIChjb2RlQmxvY2tUeXBlID09IFwiVE1Db2RlRmVuY2VUaWxkZU9wZW5cIikge1xuICAgICAgICAvLyBXZSdyZSBpbiBhIHRpbGRlLWZlbmNlZCBjb2RlIGJsb2NrXG4gICAgICAgIGxldCBjYXB0dXJlID0gbGluZUdyYW1tYXIuVE1Db2RlRmVuY2VUaWxkZUNsb3NlLnJlZ2V4cC5leGVjKFxuICAgICAgICAgIHRoaXMubGluZXNbbGluZU51bV1cbiAgICAgICAgKTtcbiAgICAgICAgaWYgKGNhcHR1cmUgJiYgY2FwdHVyZS5ncm91cHNbXCJzZXFcIl0ubGVuZ3RoID49IGNvZGVCbG9ja1NlcUxlbmd0aCkge1xuICAgICAgICAgIGxpbmVUeXBlID0gXCJUTUNvZGVGZW5jZVRpbGRlQ2xvc2VcIjtcbiAgICAgICAgICBsaW5lUmVwbGFjZW1lbnQgPSBsaW5lR3JhbW1hci5UTUNvZGVGZW5jZVRpbGRlQ2xvc2UucmVwbGFjZW1lbnQ7XG4gICAgICAgICAgbGluZUNhcHR1cmUgPSBjYXB0dXJlO1xuICAgICAgICAgIGNvZGVCbG9ja1R5cGUgPSBmYWxzZTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBsaW5lVHlwZSA9IFwiVE1GZW5jZWRDb2RlVGlsZGVcIjtcbiAgICAgICAgICBsaW5lUmVwbGFjZW1lbnQgPSAnPHNwYW4gY2xhc3M9XCJUTUZlbmNlZENvZGVcIj4kMDxiciAvPjwvc3Bhbj4nO1xuICAgICAgICAgIGxpbmVDYXB0dXJlID0gW3RoaXMubGluZXNbbGluZU51bV1dO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIENoZWNrIEhUTUwgYmxvY2sgdHlwZXNcbiAgICAgIGlmIChsaW5lVHlwZSA9PSBcIlRNUGFyYVwiICYmIGh0bWxCbG9jayA9PT0gZmFsc2UpIHtcbiAgICAgICAgZm9yIChsZXQgaHRtbEJsb2NrVHlwZSBvZiBodG1sQmxvY2tHcmFtbWFyKSB7XG4gICAgICAgICAgaWYgKHRoaXMubGluZXNbbGluZU51bV0ubWF0Y2goaHRtbEJsb2NrVHlwZS5zdGFydCkpIHtcbiAgICAgICAgICAgIC8vIE1hdGNoaW5nIHN0YXJ0IGNvbmRpdGlvbi4gQ2hlY2sgaWYgdGhpcyB0YWcgY2FuIHN0YXJ0IGhlcmUgKG5vdCBhbGwgc3RhcnQgY29uZGl0aW9ucyBhbGxvdyBicmVha2luZyBhIHBhcmFncmFwaCkuXG4gICAgICAgICAgICBpZiAoXG4gICAgICAgICAgICAgIGh0bWxCbG9ja1R5cGUucGFyYUludGVycnVwdCB8fFxuICAgICAgICAgICAgICBsaW5lTnVtID09IDAgfHxcbiAgICAgICAgICAgICAgIShcbiAgICAgICAgICAgICAgICB0aGlzLmxpbmVUeXBlc1tsaW5lTnVtIC0gMV0gPT0gXCJUTVBhcmFcIiB8fFxuICAgICAgICAgICAgICAgIHRoaXMubGluZVR5cGVzW2xpbmVOdW0gLSAxXSA9PSBcIlRNVUxcIiB8fFxuICAgICAgICAgICAgICAgIHRoaXMubGluZVR5cGVzW2xpbmVOdW0gLSAxXSA9PSBcIlRNT0xcIiB8fFxuICAgICAgICAgICAgICAgIHRoaXMubGluZVR5cGVzW2xpbmVOdW0gLSAxXSA9PSBcIlRNQmxvY2txdW90ZVwiXG4gICAgICAgICAgICAgIClcbiAgICAgICAgICAgICkge1xuICAgICAgICAgICAgICBodG1sQmxvY2sgPSBodG1sQmxvY2tUeXBlO1xuICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgaWYgKGh0bWxCbG9jayAhPT0gZmFsc2UpIHtcbiAgICAgICAgbGluZVR5cGUgPSBcIlRNSFRNTEJsb2NrXCI7XG4gICAgICAgIGxpbmVSZXBsYWNlbWVudCA9ICc8c3BhbiBjbGFzcz1cIlRNSFRNTENvbnRlbnRcIj4kMDxiciAvPjwvc3Bhbj4nOyAvLyBObyBmb3JtYXR0aW5nIGluIFRNSFRNTEJsb2NrXG4gICAgICAgIGxpbmVDYXB0dXJlID0gW3RoaXMubGluZXNbbGluZU51bV1dOyAvLyBUaGlzIHNob3VsZCBhbHJlYWR5IGJlIHNldCBidXQgYmV0dGVyIHNhZmUgdGhhbiBzb3JyeVxuXG4gICAgICAgIC8vIENoZWNrIGlmIEhUTUwgYmxvY2sgc2hvdWxkIGJlIGNsb3NlZFxuICAgICAgICBpZiAoaHRtbEJsb2NrLmVuZCkge1xuICAgICAgICAgIC8vIFNwZWNpZmljIGVuZCBjb25kaXRpb25cbiAgICAgICAgICBpZiAodGhpcy5saW5lc1tsaW5lTnVtXS5tYXRjaChodG1sQmxvY2suZW5kKSkge1xuICAgICAgICAgICAgaHRtbEJsb2NrID0gZmFsc2U7XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIE5vIHNwZWNpZmljIGVuZCBjb25kaXRpb24sIGVuZHMgd2l0aCBibGFuayBsaW5lXG4gICAgICAgICAgaWYgKFxuICAgICAgICAgICAgbGluZU51bSA9PSB0aGlzLmxpbmVzLmxlbmd0aCAtIDEgfHxcbiAgICAgICAgICAgIHRoaXMubGluZXNbbGluZU51bSArIDFdLm1hdGNoKGxpbmVHcmFtbWFyLlRNQmxhbmtMaW5lLnJlZ2V4cClcbiAgICAgICAgICApIHtcbiAgICAgICAgICAgIGh0bWxCbG9jayA9IGZhbHNlO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICAvLyBDaGVjayBhbGwgcmVnZXhwcyBpZiB3ZSBoYXZlbid0IGFwcGxpZWQgb25lIG9mIHRoZSBjb2RlIGJsb2NrIHR5cGVzXG4gICAgICBpZiAobGluZVR5cGUgPT0gXCJUTVBhcmFcIikge1xuICAgICAgICBmb3IgKGxldCB0eXBlIGluIGxpbmVHcmFtbWFyKSB7XG4gICAgICAgICAgaWYgKGxpbmVHcmFtbWFyW3R5cGVdLnJlZ2V4cCkge1xuICAgICAgICAgICAgbGV0IGNhcHR1cmUgPSBsaW5lR3JhbW1hclt0eXBlXS5yZWdleHAuZXhlYyh0aGlzLmxpbmVzW2xpbmVOdW1dKTtcbiAgICAgICAgICAgIGlmIChjYXB0dXJlKSB7XG4gICAgICAgICAgICAgIGxpbmVUeXBlID0gdHlwZTtcbiAgICAgICAgICAgICAgbGluZVJlcGxhY2VtZW50ID0gbGluZUdyYW1tYXJbdHlwZV0ucmVwbGFjZW1lbnQ7XG4gICAgICAgICAgICAgIGxpbmVDYXB0dXJlID0gY2FwdHVyZTtcbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIElmIHdlJ3ZlIG9wZW5lZCBhIGNvZGUgYmxvY2ssIHJlbWVtYmVyIHRoYXRcbiAgICAgIGlmIChcbiAgICAgICAgbGluZVR5cGUgPT0gXCJUTUNvZGVGZW5jZUJhY2t0aWNrT3BlblwiIHx8XG4gICAgICAgIGxpbmVUeXBlID09IFwiVE1Db2RlRmVuY2VUaWxkZU9wZW5cIlxuICAgICAgKSB7XG4gICAgICAgIGNvZGVCbG9ja1R5cGUgPSBsaW5lVHlwZTtcbiAgICAgICAgY29kZUJsb2NrU2VxTGVuZ3RoID0gbGluZUNhcHR1cmUuZ3JvdXBzW1wic2VxXCJdLmxlbmd0aDtcbiAgICAgIH1cblxuICAgICAgLy8gTGluayByZWZlcmVuY2UgZGVmaW5pdGlvbiBhbmQgaW5kZW50ZWQgY29kZSBjYW4ndCBpbnRlcnJ1cHQgYSBwYXJhZ3JhcGhcbiAgICAgIGlmIChcbiAgICAgICAgKGxpbmVUeXBlID09IFwiVE1JbmRlbnRlZENvZGVcIiB8fFxuICAgICAgICAgIGxpbmVUeXBlID09IFwiVE1MaW5rUmVmZXJlbmNlRGVmaW5pdGlvblwiKSAmJlxuICAgICAgICBsaW5lTnVtID4gMCAmJlxuICAgICAgICAodGhpcy5saW5lVHlwZXNbbGluZU51bSAtIDFdID09IFwiVE1QYXJhXCIgfHxcbiAgICAgICAgICB0aGlzLmxpbmVUeXBlc1tsaW5lTnVtIC0gMV0gPT0gXCJUTVVMXCIgfHxcbiAgICAgICAgICB0aGlzLmxpbmVUeXBlc1tsaW5lTnVtIC0gMV0gPT0gXCJUTU9MXCIgfHxcbiAgICAgICAgICB0aGlzLmxpbmVUeXBlc1tsaW5lTnVtIC0gMV0gPT0gXCJUTUJsb2NrcXVvdGVcIilcbiAgICAgICkge1xuICAgICAgICAvLyBGYWxsIGJhY2sgdG8gVE1QYXJhXG4gICAgICAgIGxpbmVUeXBlID0gXCJUTVBhcmFcIjtcbiAgICAgICAgbGluZUNhcHR1cmUgPSBbdGhpcy5saW5lc1tsaW5lTnVtXV07XG4gICAgICAgIGxpbmVSZXBsYWNlbWVudCA9IFwiJCQwXCI7XG4gICAgICB9XG5cbiAgICAgIC8vIFNldGV4dCBIMiBtYXJrZXJzIHRoYXQgY2FuIGFsc28gYmUgaW50ZXJwcmV0ZWQgYXMgYW4gZW1wdHkgbGlzdCBpdGVtIHNob3VsZCBiZSByZWdhcmRlZCBhcyBzdWNoIChhcyBwZXIgQ29tbW9uTWFyayBzcGVjKVxuICAgICAgaWYgKGxpbmVUeXBlID09IFwiVE1TZXRleHRIMk1hcmtlclwiKSB7XG4gICAgICAgIGxldCBjYXB0dXJlID0gbGluZUdyYW1tYXIuVE1VTC5yZWdleHAuZXhlYyh0aGlzLmxpbmVzW2xpbmVOdW1dKTtcbiAgICAgICAgaWYgKGNhcHR1cmUpIHtcbiAgICAgICAgICBsaW5lVHlwZSA9IFwiVE1VTFwiO1xuICAgICAgICAgIGxpbmVSZXBsYWNlbWVudCA9IGxpbmVHcmFtbWFyLlRNVUwucmVwbGFjZW1lbnQ7XG4gICAgICAgICAgbGluZUNhcHR1cmUgPSBjYXB0dXJlO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIFNldGV4dCBoZWFkaW5ncyBhcmUgb25seSB2YWxpZCBpZiBwcmVjZWRlZCBieSBhIHBhcmFncmFwaCAoYW5kIGlmIHNvLCB0aGV5IGNoYW5nZSB0aGUgdHlwZSBvZiB0aGUgcHJldmlvdXMgcGFyYWdyYXBoKVxuICAgICAgaWYgKGxpbmVUeXBlID09IFwiVE1TZXRleHRIMU1hcmtlclwiIHx8IGxpbmVUeXBlID09IFwiVE1TZXRleHRIMk1hcmtlclwiKSB7XG4gICAgICAgIGlmIChsaW5lTnVtID09IDAgfHwgdGhpcy5saW5lVHlwZXNbbGluZU51bSAtIDFdICE9IFwiVE1QYXJhXCIpIHtcbiAgICAgICAgICAvLyBTZXRleHQgbWFya2VyIGlzIGludmFsaWQuIEhvd2V2ZXIsIGEgSDIgbWFya2VyIG1pZ2h0IHN0aWxsIGJlIGEgdmFsaWQgSFIsIHNvIGxldCdzIGNoZWNrIHRoYXRcbiAgICAgICAgICBsZXQgY2FwdHVyZSA9IGxpbmVHcmFtbWFyLlRNSFIucmVnZXhwLmV4ZWModGhpcy5saW5lc1tsaW5lTnVtXSk7XG4gICAgICAgICAgaWYgKGNhcHR1cmUpIHtcbiAgICAgICAgICAgIC8vIFZhbGlkIEhSXG4gICAgICAgICAgICBsaW5lVHlwZSA9IFwiVE1IUlwiO1xuICAgICAgICAgICAgbGluZUNhcHR1cmUgPSBjYXB0dXJlO1xuICAgICAgICAgICAgbGluZVJlcGxhY2VtZW50ID0gbGluZUdyYW1tYXIuVE1IUi5yZXBsYWNlbWVudDtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgLy8gTm90IHZhbGlkIEhSLCBmb3JtYXQgYXMgVE1QYXJhXG4gICAgICAgICAgICBsaW5lVHlwZSA9IFwiVE1QYXJhXCI7XG4gICAgICAgICAgICBsaW5lQ2FwdHVyZSA9IFt0aGlzLmxpbmVzW2xpbmVOdW1dXTtcbiAgICAgICAgICAgIGxpbmVSZXBsYWNlbWVudCA9IFwiJCQwXCI7XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIFZhbGlkIHNldGV4dCBtYXJrZXIuIENoYW5nZSB0eXBlcyBvZiBwcmVjZWRpbmcgcGFyYSBsaW5lc1xuICAgICAgICAgIGxldCBoZWFkaW5nTGluZSA9IGxpbmVOdW0gLSAxO1xuICAgICAgICAgIGNvbnN0IGhlYWRpbmdMaW5lVHlwZSA9XG4gICAgICAgICAgICBsaW5lVHlwZSA9PSBcIlRNU2V0ZXh0SDFNYXJrZXJcIiA/IFwiVE1TZXRleHRIMVwiIDogXCJUTVNldGV4dEgyXCI7XG4gICAgICAgICAgZG8ge1xuICAgICAgICAgICAgaWYgKHRoaXMubGluZVR5cGVzW2hlYWRpbmdMaW5lVHlwZV0gIT0gaGVhZGluZ0xpbmVUeXBlKSB7XG4gICAgICAgICAgICAgIHRoaXMubGluZVR5cGVzW2hlYWRpbmdMaW5lXSA9IGhlYWRpbmdMaW5lVHlwZTtcbiAgICAgICAgICAgICAgdGhpcy5saW5lRGlydHlbaGVhZGluZ0xpbmVUeXBlXSA9IHRydWU7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB0aGlzLmxpbmVSZXBsYWNlbWVudHNbaGVhZGluZ0xpbmVdID0gXCIkJDBcIjtcbiAgICAgICAgICAgIHRoaXMubGluZUNhcHR1cmVzW2hlYWRpbmdMaW5lXSA9IFt0aGlzLmxpbmVzW2hlYWRpbmdMaW5lXV07XG5cbiAgICAgICAgICAgIGhlYWRpbmdMaW5lLS07XG4gICAgICAgICAgfSB3aGlsZSAoaGVhZGluZ0xpbmUgPj0gMCAmJiB0aGlzLmxpbmVUeXBlc1toZWFkaW5nTGluZV0gPT0gXCJUTVBhcmFcIik7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIC8vIExhc3RseSwgc2F2ZSB0aGUgbGluZSBzdHlsZSB0byBiZSBhcHBsaWVkIGxhdGVyXG4gICAgICBpZiAodGhpcy5saW5lVHlwZXNbbGluZU51bV0gIT0gbGluZVR5cGUpIHtcbiAgICAgICAgdGhpcy5saW5lVHlwZXNbbGluZU51bV0gPSBsaW5lVHlwZTtcbiAgICAgICAgdGhpcy5saW5lRGlydHlbbGluZU51bV0gPSB0cnVlO1xuICAgICAgfVxuICAgICAgdGhpcy5saW5lUmVwbGFjZW1lbnRzW2xpbmVOdW1dID0gbGluZVJlcGxhY2VtZW50O1xuICAgICAgdGhpcy5saW5lQ2FwdHVyZXNbbGluZU51bV0gPSBsaW5lQ2FwdHVyZTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbGwgbGluZSBjb250ZW50cyBmcm9tIHRoZSBIVE1MLCB0aGVuIHJlLWFwcGxpZXMgZm9ybWF0dGluZy5cbiAgICovXG4gIHVwZGF0ZUxpbmVDb250ZW50c0FuZEZvcm1hdHRpbmcoKSB7XG4gICAgdGhpcy5jbGVhckRpcnR5RmxhZygpO1xuICAgIHRoaXMudXBkYXRlTGluZUNvbnRlbnRzKCk7XG4gICAgdGhpcy51cGRhdGVGb3JtYXR0aW5nKCk7XG4gIH1cblxuICAvKipcbiAgICogQXR0ZW1wdHMgdG8gcGFyc2UgYSBsaW5rIG9yIGltYWdlIGF0IHRoZSBjdXJyZW50IHBvc2l0aW9uLiBUaGlzIGFzc3VtZXMgdGhhdCB0aGUgb3BlbmluZyBbIG9yICFbIGhhcyBhbHJlYWR5IGJlZW4gbWF0Y2hlZC5cbiAgICogUmV0dXJucyBmYWxzZSBpZiB0aGlzIGlzIG5vdCBhIHZhbGlkIGxpbmssIGltYWdlLiBTZWUgYmVsb3cgZm9yIG1vcmUgaW5mb3JtYXRpb25cbiAgICogQHBhcmFtIHtzdHJpbmd9IG9yaWdpbmFsU3RyaW5nIFRoZSBvcmlnaW5hbCBzdHJpbmcsIHN0YXJ0aW5nIGF0IHRoZSBvcGVuaW5nIG1hcmtlciAoWyBvciAhWylcbiAgICogQHBhcmFtIHtib29sZWFufSBpc0ltYWdlIFdoZXRoZXIgb3Igbm90IHRoaXMgaXMgYW4gaW1hZ2UgKG9wZW5lciA9PSAhWylcbiAgICogQHJldHVybnMgZmFsc2UgaWYgbm90IGEgdmFsaWQgbGluayAvIGltYWdlLlxuICAgKiBPdGhlcndpc2UgcmV0dXJucyBhbiBvYmplY3Qgd2l0aCB0d28gcHJvcGVydGllczogb3V0cHV0IGlzIHRoZSBzdHJpbmcgdG8gYmUgaW5jbHVkZWQgaW4gdGhlIHByb2Nlc3NlZCBvdXRwdXQsXG4gICAqIGNoYXJDb3VudCBpcyB0aGUgbnVtYmVyIG9mIGlucHV0IGNoYXJhY3RlcnMgKGZyb20gb3JpZ2luYWxTdHJpbmcpIGNvbnN1bWVkLlxuICAgKi9cbiAgcGFyc2VMaW5rT3JJbWFnZShvcmlnaW5hbFN0cmluZywgaXNJbWFnZSkge1xuICAgIC8vIFNraXAgdGhlIG9wZW5pbmcgYnJhY2tldFxuICAgIGxldCB0ZXh0T2Zmc2V0ID0gaXNJbWFnZSA/IDIgOiAxO1xuICAgIGxldCBvcGVuZXIgPSBvcmlnaW5hbFN0cmluZy5zdWJzdHIoMCwgdGV4dE9mZnNldCk7XG4gICAgbGV0IHR5cGUgPSBpc0ltYWdlID8gXCJUTUltYWdlXCIgOiBcIlRNTGlua1wiO1xuICAgIGxldCBjdXJyZW50T2Zmc2V0ID0gdGV4dE9mZnNldDtcblxuICAgIGxldCBicmFja2V0TGV2ZWwgPSAxO1xuICAgIGxldCBsaW5rVGV4dCA9IGZhbHNlO1xuICAgIGxldCBsaW5rUmVmID0gZmFsc2U7XG4gICAgbGV0IGxpbmtMYWJlbCA9IFtdO1xuICAgIGxldCBsaW5rRGV0YWlscyA9IFtdOyAvLyBJZiBtYXRjaGVkLCB0aGlzIHdpbGwgYmUgYW4gYXJyYXk6IFt3aGl0ZXNwYWNlICsgbGluayBkZXN0aW5hdGlvbiBkZWxpbWl0ZXIsIGxpbmsgZGVzdGluYXRpb24sIGxpbmsgZGVzdGluYXRpb24gZGVsaW1pdGVyLCB3aGl0ZXNwYWNlLCBsaW5rIHRpdGxlIGRlbGltaXRlciwgbGluayB0aXRsZSwgbGluayB0aXRsZSBkZWxpbWl0ZXIgKyB3aGl0ZXNwYWNlXS4gQWxsIGNhbiBiZSBlbXB0eSBzdHJpbmdzLlxuXG4gICAgdGV4dE91dGVyOiB3aGlsZSAoXG4gICAgICBjdXJyZW50T2Zmc2V0IDwgb3JpZ2luYWxTdHJpbmcubGVuZ3RoICYmXG4gICAgICBsaW5rVGV4dCA9PT0gZmFsc2UgLyogZW1wdHkgc3RyaW5nIGlzIG9rYXkgKi9cbiAgICApIHtcbiAgICAgIGxldCBzdHJpbmcgPSBvcmlnaW5hbFN0cmluZy5zdWJzdHIoY3VycmVudE9mZnNldCk7XG5cbiAgICAgIC8vIENhcHR1cmUgYW55IGVzY2FwZXMgYW5kIGNvZGUgYmxvY2tzIGF0IGN1cnJlbnQgcG9zaXRpb24sIHRoZXkgYmluZCBtb3JlIHN0cm9uZ2x5IHRoYW4gbGlua3NcbiAgICAgIC8vIFdlIGRvbid0IGhhdmUgdG8gYWN0dWFsbHkgcHJvY2VzcyB0aGVtIGhlcmUsIHRoYXQnbGwgYmUgZG9uZSBsYXRlciBpbiBjYXNlIHRoZSBsaW5rIC8gaW1hZ2UgaXMgdmFsaWQsIGJ1dCB3ZSBuZWVkIHRvIHNraXAgb3ZlciB0aGVtLlxuICAgICAgZm9yIChsZXQgcnVsZSBvZiBbXCJlc2NhcGVcIiwgXCJjb2RlXCIsIFwiYXV0b2xpbmtcIiwgXCJodG1sXCJdKSB7XG4gICAgICAgIGxldCBjYXAgPSBpbmxpbmVHcmFtbWFyW3J1bGVdLnJlZ2V4cC5leGVjKHN0cmluZyk7XG4gICAgICAgIGlmIChjYXApIHtcbiAgICAgICAgICBjdXJyZW50T2Zmc2V0ICs9IGNhcFswXS5sZW5ndGg7XG4gICAgICAgICAgY29udGludWUgdGV4dE91dGVyO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIENoZWNrIGZvciBpbWFnZS4gSXQncyBva2F5IGZvciBhbiBpbWFnZSB0byBiZSBpbmNsdWRlZCBpbiBhIGxpbmsgb3IgaW1hZ2VcbiAgICAgIGlmIChzdHJpbmcubWF0Y2goaW5saW5lR3JhbW1hci5pbWFnZU9wZW4ucmVnZXhwKSkge1xuICAgICAgICAvLyBPcGVuaW5nIGltYWdlLiBJdCdzIG9rYXkgaWYgdGhpcyBpcyBhIG1hdGNoaW5nIHBhaXIgb2YgYnJhY2tldHNcbiAgICAgICAgYnJhY2tldExldmVsKys7XG4gICAgICAgIGN1cnJlbnRPZmZzZXQgKz0gMjtcbiAgICAgICAgY29udGludWUgdGV4dE91dGVyO1xuICAgICAgfVxuXG4gICAgICAvLyBDaGVjayBmb3IgbGluayAobm90IGFuIGltYWdlIGJlY2F1c2UgdGhhdCB3b3VsZCBoYXZlIGJlZW4gY2FwdHVyZWQgYW5kIHNraXBwZWQgb3ZlciBhYm92ZSlcbiAgICAgIGlmIChzdHJpbmcubWF0Y2goaW5saW5lR3JhbW1hci5saW5rT3Blbi5yZWdleHApKSB7XG4gICAgICAgIC8vIE9wZW5pbmcgYnJhY2tldC4gVHdvIHRoaW5ncyB0byBkbzpcbiAgICAgICAgLy8gMSkgaXQncyBva2F5IGlmIHRoaXMgcGFydCBvZiBhIHBhaXIgb2YgYnJhY2tldHMuXG4gICAgICAgIC8vIDIpIElmIHdlIGFyZSBjdXJyZW50bHkgdHJ5aW5nIHRvIHBhcnNlIGEgbGluaywgdGhpcyBuZXN0ZWQgYnJhY2tldCBtdXNuJ3Qgc3RhcnQgYSB2YWxpZCBsaW5rIChubyBuZXN0ZWQgbGlua3MgYWxsb3dlZClcbiAgICAgICAgYnJhY2tldExldmVsKys7XG4gICAgICAgIC8vIGlmIChicmFja2V0TGV2ZWwgPj0gMikgcmV0dXJuIGZhbHNlOyAvLyBOZXN0ZWQgdW5lc2NhcGVkIGJyYWNrZXRzLCB0aGlzIGRvZXNuJ3QgcXVhbGlmeSBhcyBhIGxpbmsgLyBpbWFnZVxuICAgICAgICBpZiAoIWlzSW1hZ2UpIHtcbiAgICAgICAgICBpZiAodGhpcy5wYXJzZUxpbmtPckltYWdlKHN0cmluZywgZmFsc2UpKSB7XG4gICAgICAgICAgICAvLyBWYWxpZCBsaW5rIGluc2lkZSB0aGlzIHBvc3NpYmxlIGxpbmssIHdoaWNoIG1ha2VzIHRoaXMgbGluayBpbnZhbGlkIChpbm5lciBsaW5rcyBiZWF0IG91dGVyIG9uZXMpXG4gICAgICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGN1cnJlbnRPZmZzZXQgKz0gMTtcbiAgICAgICAgY29udGludWUgdGV4dE91dGVyO1xuICAgICAgfVxuXG4gICAgICAvLyBDaGVjayBmb3IgY2xvc2luZyBicmFja2V0XG4gICAgICBpZiAoc3RyaW5nLm1hdGNoKC9eXFxdLykpIHtcbiAgICAgICAgYnJhY2tldExldmVsLS07XG4gICAgICAgIGlmIChicmFja2V0TGV2ZWwgPT0gMCkge1xuICAgICAgICAgIC8vIEZvdW5kIG1hdGNoaW5nIGJyYWNrZXQgYW5kIGhhdmVuJ3QgZm91bmQgYW55dGhpbmcgZGlzcXVhbGlmeWluZyB0aGlzIGFzIGxpbmsgLyBpbWFnZS5cbiAgICAgICAgICBsaW5rVGV4dCA9IG9yaWdpbmFsU3RyaW5nLnN1YnN0cihcbiAgICAgICAgICAgIHRleHRPZmZzZXQsXG4gICAgICAgICAgICBjdXJyZW50T2Zmc2V0IC0gdGV4dE9mZnNldFxuICAgICAgICAgICk7XG4gICAgICAgICAgY3VycmVudE9mZnNldCsrO1xuICAgICAgICAgIGNvbnRpbnVlIHRleHRPdXRlcjtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICAvLyBOb3RoaW5nIG1hdGNoZXMsIHByb2NlZWQgdG8gbmV4dCBjaGFyXG4gICAgICBjdXJyZW50T2Zmc2V0Kys7XG4gICAgfVxuXG4gICAgLy8gRGlkIHdlIGZpbmQgYSBsaW5rIHRleHQgKGkuZS4sIGZpbmQgYSBtYXRjaGluZyBjbG9zaW5nIGJyYWNrZXQ/KVxuICAgIGlmIChsaW5rVGV4dCA9PT0gZmFsc2UpIHJldHVybiBmYWxzZTsgLy8gTm9wZVxuXG4gICAgLy8gU28gZmFyLCBzbyBnb29kLiBXZSd2ZSBnb3QgYSB2YWxpZCBsaW5rIHRleHQuIExldCdzIHNlZSB3aGF0IHR5cGUgb2YgbGluayB0aGlzIGlzXG4gICAgbGV0IG5leHRDaGFyID1cbiAgICAgIGN1cnJlbnRPZmZzZXQgPCBvcmlnaW5hbFN0cmluZy5sZW5ndGhcbiAgICAgICAgPyBvcmlnaW5hbFN0cmluZy5zdWJzdHIoY3VycmVudE9mZnNldCwgMSlcbiAgICAgICAgOiBcIlwiO1xuXG4gICAgLy8gUkVGRVJFTkNFIExJTktTXG4gICAgaWYgKG5leHRDaGFyID09IFwiW1wiKSB7XG4gICAgICBsZXQgc3RyaW5nID0gb3JpZ2luYWxTdHJpbmcuc3Vic3RyKGN1cnJlbnRPZmZzZXQpO1xuICAgICAgbGV0IGNhcCA9IGlubGluZUdyYW1tYXIubGlua0xhYmVsLnJlZ2V4cC5leGVjKHN0cmluZyk7XG4gICAgICBpZiAoY2FwKSB7XG4gICAgICAgIC8vIFZhbGlkIGxpbmsgbGFiZWxcbiAgICAgICAgY3VycmVudE9mZnNldCArPSBjYXBbMF0ubGVuZ3RoO1xuICAgICAgICBsaW5rTGFiZWwucHVzaChjYXBbMV0sIGNhcFsyXSwgY2FwWzNdKTtcbiAgICAgICAgaWYgKGNhcFtpbmxpbmVHcmFtbWFyLmxpbmtMYWJlbC5sYWJlbFBsYWNlaG9sZGVyXSkge1xuICAgICAgICAgIC8vIEZ1bGwgcmVmZXJlbmNlIGxpbmtcbiAgICAgICAgICBsaW5rUmVmID0gY2FwW2lubGluZUdyYW1tYXIubGlua0xhYmVsLmxhYmVsUGxhY2Vob2xkZXJdO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIENvbGxhcHNlZCByZWZlcmVuY2UgbGlua1xuICAgICAgICAgIGxpbmtSZWYgPSBsaW5rVGV4dC50cmltKCk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIE5vdCBhIHZhbGlkIGxpbmsgbGFiZWxcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuICAgIH0gZWxzZSBpZiAobmV4dENoYXIgIT0gXCIoXCIpIHtcbiAgICAgIC8vIFNob3J0Y3V0IHJlZiBsaW5rXG4gICAgICBsaW5rUmVmID0gbGlua1RleHQudHJpbSgpO1xuXG4gICAgICAvLyBJTkxJTkUgTElOS1NcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gbmV4dENoYXIgPT0gJygnXG5cbiAgICAgIC8vIFBvdGVudGlhbCBpbmxpbmUgbGlua1xuICAgICAgY3VycmVudE9mZnNldCsrO1xuXG4gICAgICBsZXQgcGFyZW50aGVzaXNMZXZlbCA9IDE7XG4gICAgICBpbmxpbmVPdXRlcjogd2hpbGUgKFxuICAgICAgICBjdXJyZW50T2Zmc2V0IDwgb3JpZ2luYWxTdHJpbmcubGVuZ3RoICYmXG4gICAgICAgIHBhcmVudGhlc2lzTGV2ZWwgPiAwXG4gICAgICApIHtcbiAgICAgICAgbGV0IHN0cmluZyA9IG9yaWdpbmFsU3RyaW5nLnN1YnN0cihjdXJyZW50T2Zmc2V0KTtcblxuICAgICAgICAvLyBQcm9jZXNzIHdoaXRlc3BhY2VcbiAgICAgICAgbGV0IGNhcCA9IC9eXFxzKy8uZXhlYyhzdHJpbmcpO1xuICAgICAgICBpZiAoY2FwKSB7XG4gICAgICAgICAgc3dpdGNoIChsaW5rRGV0YWlscy5sZW5ndGgpIHtcbiAgICAgICAgICAgIGNhc2UgMDpcbiAgICAgICAgICAgICAgbGlua0RldGFpbHMucHVzaChjYXBbMF0pO1xuICAgICAgICAgICAgICBicmVhazsgLy8gT3BlbmluZyB3aGl0ZXNwYWNlXG4gICAgICAgICAgICBjYXNlIDE6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goY2FwWzBdKTtcbiAgICAgICAgICAgICAgYnJlYWs7IC8vIE9wZW4gZGVzdGluYXRpb24sIGJ1dCBub3QgYSBkZXN0aW5hdGlvbiB5ZXQ7IGRlc2luYXRpb24gb3BlbmVkIHdpdGggPFxuICAgICAgICAgICAgY2FzZSAyOiAvLyBPcGVuIGRlc3RpbmF0aW9uIHdpdGggY29udGVudCBpbiBpdC4gV2hpdGVzcGFjZSBvbmx5IGFsbG93ZWQgaWYgb3BlbmVkIGJ5IGFuZ2xlIGJyYWNrZXQsIG90aGVyd2lzZSB0aGlzIGNsb3NlcyB0aGUgZGVzdGluYXRpb25cbiAgICAgICAgICAgICAgaWYgKGxpbmtEZXRhaWxzWzBdLm1hdGNoKC88LykpIHtcbiAgICAgICAgICAgICAgICBsaW5rRGV0YWlsc1sxXSA9IGxpbmtEZXRhaWxzWzFdLmNvbmNhdChjYXBbMF0pO1xuICAgICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgIGlmIChwYXJlbnRoZXNpc0xldmVsICE9IDEpIHJldHVybiBmYWxzZTsgLy8gVW5iYWxhbmNlZCBwYXJlbnRoZXNpc1xuICAgICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7IC8vIEVtcHR5IGVuZCBkZWxpbWl0ZXIgZm9yIGRlc3RpbmF0aW9uXG4gICAgICAgICAgICAgICAgbGlua0RldGFpbHMucHVzaChjYXBbMF0pOyAvLyBXaGl0ZXNwYWNlIGluIGJldHdlZW4gZGVzdGluYXRpb24gYW5kIHRpdGxlXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICBjYXNlIDM6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goY2FwWzBdKTtcbiAgICAgICAgICAgICAgYnJlYWs7IC8vIFdoaXRlc3BhY2UgYmV0d2VlbiBkZXN0aW5hdGlvbiBhbmQgdGl0bGVcbiAgICAgICAgICAgIGNhc2UgNDpcbiAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlOyAvLyBUaGlzIHNob3VsZCBuZXZlciBoYXBwZW4gKG5vIG9wZW5lciBmb3IgdGl0bGUgeWV0LCBidXQgbW9yZSB3aGl0ZXNwYWNlIHRvIGNhcHR1cmUpXG4gICAgICAgICAgICBjYXNlIDU6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7IC8vIFdoaXRlc3BhY2UgYXQgYmVnaW5uaW5nIG9mIHRpdGxlLCBwdXNoIGVtcHR5IHRpdGxlIGFuZCBjb250aW51ZVxuICAgICAgICAgICAgY2FzZSA2OlxuICAgICAgICAgICAgICBsaW5rRGV0YWlsc1s1XSA9IGxpbmtEZXRhaWxzWzVdLmNvbmNhdChjYXBbMF0pO1xuICAgICAgICAgICAgICBicmVhazsgLy8gV2hpdGVzcGFjZSBpbiB0aXRsZVxuICAgICAgICAgICAgY2FzZSA3OlxuICAgICAgICAgICAgICBsaW5rRGV0YWlsc1s2XSA9IGxpbmtEZXRhaWxzWzZdLmNvbmNhdChjYXBbMF0pO1xuICAgICAgICAgICAgICBicmVhazsgLy8gV2hpdGVzcGFjZSBhZnRlciBjbG9zaW5nIGRlbGltaXRlclxuICAgICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlOyAvLyBXZSBzaG91bGQgbmV2ZXIgZ2V0IGhlcmVcbiAgICAgICAgICB9XG4gICAgICAgICAgY3VycmVudE9mZnNldCArPSBjYXBbMF0ubGVuZ3RoO1xuICAgICAgICAgIGNvbnRpbnVlIGlubGluZU91dGVyO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gUHJvY2VzcyBiYWNrc2xhc2ggZXNjYXBlc1xuICAgICAgICBjYXAgPSBpbmxpbmVHcmFtbWFyLmVzY2FwZS5yZWdleHAuZXhlYyhzdHJpbmcpO1xuICAgICAgICBpZiAoY2FwKSB7XG4gICAgICAgICAgc3dpdGNoIChsaW5rRGV0YWlscy5sZW5ndGgpIHtcbiAgICAgICAgICAgIGNhc2UgMDpcbiAgICAgICAgICAgICAgbGlua0RldGFpbHMucHVzaChcIlwiKTsgLy8gdGhpcyBvcGVucyB0aGUgbGluayBkZXN0aW5hdGlvbiwgYWRkIGVtcHR5IG9wZW5pbmcgZGVsaW1pdGVyIGFuZCBwcm9jZWVkIHRvIG5leHQgY2FzZVxuICAgICAgICAgICAgY2FzZSAxOlxuICAgICAgICAgICAgICBsaW5rRGV0YWlscy5wdXNoKGNhcFswXSk7XG4gICAgICAgICAgICAgIGJyZWFrOyAvLyBUaGlzIG9wZW5zIHRoZSBsaW5rIGRlc3RpbmF0aW9uLCBhcHBlbmQgaXRcbiAgICAgICAgICAgIGNhc2UgMjpcbiAgICAgICAgICAgICAgbGlua0RldGFpbHNbMV0gPSBsaW5rRGV0YWlsc1sxXS5jb25jYXQoY2FwWzBdKTtcbiAgICAgICAgICAgICAgYnJlYWs7IC8vIFBhcnQgb2YgdGhlIGxpbmsgZGVzdGluYXRpb25cbiAgICAgICAgICAgIGNhc2UgMzpcbiAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlOyAvLyBMYWNraW5nIG9wZW5pbmcgZGVsaW1pdGVyIGZvciBsaW5rIHRpdGxlXG4gICAgICAgICAgICBjYXNlIDQ6XG4gICAgICAgICAgICAgIHJldHVybiBmYWxzZTsgLy8gTGNha2luZyBvcGVuaW5nIGRlbGltaXRlciBmb3IgbGluayB0aXRsZVxuICAgICAgICAgICAgY2FzZSA1OlxuICAgICAgICAgICAgICBsaW5rRGV0YWlscy5wdXNoKFwiXCIpOyAvLyBUaGlzIG9wZW5zIHRoZSBsaW5rIHRpdGxlXG4gICAgICAgICAgICBjYXNlIDY6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzWzVdID0gbGlua0RldGFpbHNbNV0uY29uY2F0KGNhcFswXSk7XG4gICAgICAgICAgICAgIGJyZWFrOyAvLyBQYXJ0IG9mIHRoZSBsaW5rIHRpdGxlXG4gICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICByZXR1cm4gZmFsc2U7IC8vIEFmdGVyIGxpbmsgdGl0bGUgd2FzIGNsb3NlZCwgd2l0aG91dCBjbG9zaW5nIHBhcmVudGhlc2lzXG4gICAgICAgICAgfVxuICAgICAgICAgIGN1cnJlbnRPZmZzZXQgKz0gY2FwWzBdLmxlbmd0aDtcbiAgICAgICAgICBjb250aW51ZSBpbmxpbmVPdXRlcjtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFByb2Nlc3Mgb3BlbmluZyBhbmdsZSBicmFja2V0IGFzIGRlaWxpbWl0ZXIgb2YgZGVzdGluYXRpb25cbiAgICAgICAgaWYgKGxpbmtEZXRhaWxzLmxlbmd0aCA8IDIgJiYgc3RyaW5nLm1hdGNoKC9ePC8pKSB7XG4gICAgICAgICAgaWYgKGxpbmtEZXRhaWxzLmxlbmd0aCA9PSAwKSBsaW5rRGV0YWlscy5wdXNoKFwiXCIpO1xuICAgICAgICAgIGxpbmtEZXRhaWxzWzBdID0gbGlua0RldGFpbHNbMF0uY29uY2F0KFwiPFwiKTtcbiAgICAgICAgICBjdXJyZW50T2Zmc2V0Kys7XG4gICAgICAgICAgY29udGludWUgaW5saW5lT3V0ZXI7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBQcm9jZXNzIGNsb3NpbmcgYW5nbGUgYnJhY2tldCBhcyBkZWxpbWl0ZXIgb2YgZGVzdGluYXRpb25cbiAgICAgICAgaWYgKFxuICAgICAgICAgIChsaW5rRGV0YWlscy5sZW5ndGggPT0gMSB8fCBsaW5rRGV0YWlscy5sZW5ndGggPT0gMikgJiZcbiAgICAgICAgICBzdHJpbmcubWF0Y2goL14+LylcbiAgICAgICAgKSB7XG4gICAgICAgICAgaWYgKGxpbmtEZXRhaWxzLmxlbmd0aCA9PSAxKSBsaW5rRGV0YWlscy5wdXNoKFwiXCIpOyAvLyBFbXB0eSBsaW5rIGRlc3RpbmF0aW9uXG4gICAgICAgICAgbGlua0RldGFpbHMucHVzaChcIj5cIik7XG4gICAgICAgICAgY3VycmVudE9mZnNldCsrO1xuICAgICAgICAgIGNvbnRpbnVlIGlubGluZU91dGVyO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gUHJvY2VzcyAgbm9uLXBhcmVudGhlc2lzIGRlbGltaXRlciBmb3IgdGl0bGUuXG4gICAgICAgIGNhcCA9IC9eW1wiJ10vLmV4ZWMoc3RyaW5nKTtcbiAgICAgICAgLy8gRm9yIHRoaXMgdG8gYmUgYSB2YWxpZCBvcGVuZXIsIHdlIGhhdmUgdG8gZWl0aGVyIGhhdmUgbm8gZGVzdGluYXRpb24sIG9ubHkgd2hpdGVzcGFjZSBzbyBmYXIsXG4gICAgICAgIC8vIG9yIGEgZGVzdGluYXRpb24gd2l0aCB0cmFpbGluZyB3aGl0ZXNwYWNlLlxuICAgICAgICBpZiAoXG4gICAgICAgICAgY2FwICYmXG4gICAgICAgICAgKGxpbmtEZXRhaWxzLmxlbmd0aCA9PSAwIHx8XG4gICAgICAgICAgICBsaW5rRGV0YWlscy5sZW5ndGggPT0gMSB8fFxuICAgICAgICAgICAgbGlua0RldGFpbHMubGVuZ3RoID09IDQpXG4gICAgICAgICkge1xuICAgICAgICAgIHdoaWxlIChsaW5rRGV0YWlscy5sZW5ndGggPCA0KSBsaW5rRGV0YWlscy5wdXNoKFwiXCIpO1xuICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goY2FwWzBdKTtcbiAgICAgICAgICBjdXJyZW50T2Zmc2V0Kys7XG4gICAgICAgICAgY29udGludWUgaW5saW5lT3V0ZXI7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBGb3IgdGhpcyB0byBiZSBhIHZhbGlkIGNsb3Nlciwgd2UgaGF2ZSB0byBoYXZlIGFuIG9wZW5lciBhbmQgc29tZSBvciBubyB0aXRsZSwgYW5kIHRoaXMgaGFzIHRvIG1hdGNoIHRoZSBvcGVuZXJcbiAgICAgICAgaWYgKFxuICAgICAgICAgIGNhcCAmJlxuICAgICAgICAgIChsaW5rRGV0YWlscy5sZW5ndGggPT0gNSB8fCBsaW5rRGV0YWlscy5sZW5ndGggPT0gNikgJiZcbiAgICAgICAgICBsaW5rRGV0YWlsc1s0XSA9PSBjYXBbMF1cbiAgICAgICAgKSB7XG4gICAgICAgICAgaWYgKGxpbmtEZXRhaWxzLmxlbmd0aCA9PSA1KSBsaW5rRGV0YWlscy5wdXNoKFwiXCIpOyAvLyBFbXB0eSBsaW5rIHRpdGxlXG4gICAgICAgICAgbGlua0RldGFpbHMucHVzaChjYXBbMF0pO1xuICAgICAgICAgIGN1cnJlbnRPZmZzZXQrKztcbiAgICAgICAgICBjb250aW51ZSBpbmxpbmVPdXRlcjtcbiAgICAgICAgfVxuICAgICAgICAvLyBPdGhlciBjYXNlcyAobGlua0RldGFpbHMubGVuZ3RoID09IDIsIDMsIDcpIHdpbGwgYmUgaGFuZGxlZCB3aXRoIHRoZSBcImRlZmF1bHRcIiBjYXNlIGJlbG93LlxuXG4gICAgICAgIC8vIFByb2Nlc3Mgb3BlbmluZyBwYXJlbnRoZXNpc1xuICAgICAgICBpZiAoc3RyaW5nLm1hdGNoKC9eXFwoLykpIHtcbiAgICAgICAgICBzd2l0Y2ggKGxpbmtEZXRhaWxzLmxlbmd0aCkge1xuICAgICAgICAgICAgY2FzZSAwOlxuICAgICAgICAgICAgICBsaW5rRGV0YWlscy5wdXNoKFwiXCIpOyAvLyB0aGlzIG9wZW5zIHRoZSBsaW5rIGRlc3RpbmF0aW9uLCBhZGQgZW1wdHkgb3BlbmluZyBkZWxpbWl0ZXIgYW5kIHByb2NlZWQgdG8gbmV4dCBjYXNlXG4gICAgICAgICAgICBjYXNlIDE6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7IC8vIFRoaXMgb3BlbnMgdGhlIGxpbmsgZGVzdGluYXRpb25cbiAgICAgICAgICAgIGNhc2UgMjogLy8gUGFydCBvZiB0aGUgbGluayBkZXN0aW5hdGlvblxuICAgICAgICAgICAgICBsaW5rRGV0YWlsc1sxXSA9IGxpbmtEZXRhaWxzWzFdLmNvbmNhdChcIihcIik7XG4gICAgICAgICAgICAgIGlmICghbGlua0RldGFpbHNbMF0ubWF0Y2goLzwkLykpIHBhcmVudGhlc2lzTGV2ZWwrKztcbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICBjYXNlIDM6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7IC8vICBvcGVuaW5nIGRlbGltaXRlciBmb3IgbGluayB0aXRsZVxuICAgICAgICAgICAgY2FzZSA0OlxuICAgICAgICAgICAgICBsaW5rRGV0YWlscy5wdXNoKFwiKFwiKTtcbiAgICAgICAgICAgICAgYnJlYWs7IC8vIG9wZW5pbmcgZGVsaW1pdGVyIGZvciBsaW5rIHRpdGxlXG4gICAgICAgICAgICBjYXNlIDU6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7IC8vIG9wZW5zIHRoZSBsaW5rIHRpdGxlLCBhZGQgZW1wdHkgdGl0bGUgY29udGVudCBhbmQgcHJvY2VlZCB0byBuZXh0IGNhc2VcbiAgICAgICAgICAgIGNhc2UgNjogLy8gUGFydCBvZiB0aGUgbGluayB0aXRsZS4gVW4tZXNjYXBlZCBwYXJlbnRoZXNpcyBvbmx5IGFsbG93ZWQgaW4gXCIgb3IgJyBkZWxpbWl0ZWQgdGl0bGVcbiAgICAgICAgICAgICAgaWYgKGxpbmtEZXRhaWxzWzRdID09IFwiKFwiKSByZXR1cm4gZmFsc2U7XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzWzVdID0gbGlua0RldGFpbHNbNV0uY29uY2F0KFwiKFwiKTtcbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICByZXR1cm4gZmFsc2U7IC8vIEFmdGVyIGxpbmsgdGl0bGUgd2FzIGNsb3NlZCwgd2l0aG91dCBjbG9zaW5nIHBhcmVudGhlc2lzXG4gICAgICAgICAgfVxuICAgICAgICAgIGN1cnJlbnRPZmZzZXQrKztcbiAgICAgICAgICBjb250aW51ZSBpbmxpbmVPdXRlcjtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFByb2Nlc3MgY2xvc2luZyBwYXJlbnRoZXNpc1xuICAgICAgICBpZiAoc3RyaW5nLm1hdGNoKC9eXFwpLykpIHtcbiAgICAgICAgICBpZiAobGlua0RldGFpbHMubGVuZ3RoIDw9IDIpIHtcbiAgICAgICAgICAgIC8vIFdlIGFyZSBpbnNpZGUgdGhlIGxpbmsgZGVzdGluYXRpb24uIFBhcmVudGhlc2VzIGhhdmUgdG8gYmUgbWF0Y2hlZCBpZiBub3QgaW4gYW5nbGUgYnJhY2tldHNcbiAgICAgICAgICAgIHdoaWxlIChsaW5rRGV0YWlscy5sZW5ndGggPCAyKSBsaW5rRGV0YWlscy5wdXNoKFwiXCIpO1xuXG4gICAgICAgICAgICBpZiAoIWxpbmtEZXRhaWxzWzBdLm1hdGNoKC88JC8pKSBwYXJlbnRoZXNpc0xldmVsLS07XG5cbiAgICAgICAgICAgIGlmIChwYXJlbnRoZXNpc0xldmVsID4gMCkge1xuICAgICAgICAgICAgICBsaW5rRGV0YWlsc1sxXSA9IGxpbmtEZXRhaWxzWzFdLmNvbmNhdChcIilcIik7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIGlmIChsaW5rRGV0YWlscy5sZW5ndGggPT0gNSB8fCBsaW5rRGV0YWlscy5sZW5ndGggPT0gNikge1xuICAgICAgICAgICAgLy8gV2UgYXJlIGluc2lkZSB0aGUgbGluayB0aXRsZS5cbiAgICAgICAgICAgIGlmIChsaW5rRGV0YWlsc1s0XSA9PSBcIihcIikge1xuICAgICAgICAgICAgICAvLyBUaGlzIGNsb3NlcyB0aGUgbGluayB0aXRsZVxuICAgICAgICAgICAgICBpZiAobGlua0RldGFpbHMubGVuZ3RoID09IDUpIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCIpXCIpO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgLy8gSnVzdCByZWd1bGFyIG9sJyBjb250ZW50XG4gICAgICAgICAgICAgIGlmIChsaW5rRGV0YWlscy5sZW5ndGggPT0gNSkgbGlua0RldGFpbHMucHVzaChcIilcIik7XG4gICAgICAgICAgICAgIGVsc2UgbGlua0RldGFpbHNbNV0gPSBsaW5rRGV0YWlsc1s1XS5jb25jYXQoXCIpXCIpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBwYXJlbnRoZXNpc0xldmVsLS07IC8vIFRoaXMgc2hvdWxkIGRlY3JlYXNlIGl0IGZyb20gMSB0byAwLi4uXG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKHBhcmVudGhlc2lzTGV2ZWwgPT0gMCkge1xuICAgICAgICAgICAgLy8gTm8gaW52YWxpZCBjb25kaXRpb24sIGxldCdzIG1ha2Ugc3VyZSB0aGUgbGlua0RldGFpbHMgYXJyYXkgaXMgY29tcGxldGVcbiAgICAgICAgICAgIHdoaWxlIChsaW5rRGV0YWlscy5sZW5ndGggPCA3KSBsaW5rRGV0YWlscy5wdXNoKFwiXCIpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGN1cnJlbnRPZmZzZXQrKztcbiAgICAgICAgICBjb250aW51ZSBpbmxpbmVPdXRlcjtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEFueSBvbGQgY2hhcmFjdGVyXG4gICAgICAgIGNhcCA9IC9eLi8uZXhlYyhzdHJpbmcpO1xuICAgICAgICBpZiAoY2FwKSB7XG4gICAgICAgICAgc3dpdGNoIChsaW5rRGV0YWlscy5sZW5ndGgpIHtcbiAgICAgICAgICAgIGNhc2UgMDpcbiAgICAgICAgICAgICAgbGlua0RldGFpbHMucHVzaChcIlwiKTsgLy8gdGhpcyBvcGVucyB0aGUgbGluayBkZXN0aW5hdGlvbiwgYWRkIGVtcHR5IG9wZW5pbmcgZGVsaW1pdGVyIGFuZCBwcm9jZWVkIHRvIG5leHQgY2FzZVxuICAgICAgICAgICAgY2FzZSAxOlxuICAgICAgICAgICAgICBsaW5rRGV0YWlscy5wdXNoKGNhcFswXSk7XG4gICAgICAgICAgICAgIGJyZWFrOyAvLyBUaGlzIG9wZW5zIHRoZSBsaW5rIGRlc3RpbmF0aW9uLCBhcHBlbmQgaXRcbiAgICAgICAgICAgIGNhc2UgMjpcbiAgICAgICAgICAgICAgbGlua0RldGFpbHNbMV0gPSBsaW5rRGV0YWlsc1sxXS5jb25jYXQoY2FwWzBdKTtcbiAgICAgICAgICAgICAgYnJlYWs7IC8vIFBhcnQgb2YgdGhlIGxpbmsgZGVzdGluYXRpb25cbiAgICAgICAgICAgIGNhc2UgMzpcbiAgICAgICAgICAgICAgcmV0dXJuIGZhbHNlOyAvLyBMYWNraW5nIG9wZW5pbmcgZGVsaW1pdGVyIGZvciBsaW5rIHRpdGxlXG4gICAgICAgICAgICBjYXNlIDQ6XG4gICAgICAgICAgICAgIHJldHVybiBmYWxzZTsgLy8gTGNha2luZyBvcGVuaW5nIGRlbGltaXRlciBmb3IgbGluayB0aXRsZVxuICAgICAgICAgICAgY2FzZSA1OlxuICAgICAgICAgICAgICBsaW5rRGV0YWlscy5wdXNoKFwiXCIpOyAvLyBUaGlzIG9wZW5zIHRoZSBsaW5rIHRpdGxlXG4gICAgICAgICAgICBjYXNlIDY6XG4gICAgICAgICAgICAgIGxpbmtEZXRhaWxzWzVdID0gbGlua0RldGFpbHNbNV0uY29uY2F0KGNhcFswXSk7XG4gICAgICAgICAgICAgIGJyZWFrOyAvLyBQYXJ0IG9mIHRoZSBsaW5rIHRpdGxlXG4gICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICByZXR1cm4gZmFsc2U7IC8vIEFmdGVyIGxpbmsgdGl0bGUgd2FzIGNsb3NlZCwgd2l0aG91dCBjbG9zaW5nIHBhcmVudGhlc2lzXG4gICAgICAgICAgfVxuICAgICAgICAgIGN1cnJlbnRPZmZzZXQgKz0gY2FwWzBdLmxlbmd0aDtcbiAgICAgICAgICBjb250aW51ZSBpbmxpbmVPdXRlcjtcbiAgICAgICAgfVxuICAgICAgICB0aHJvdyBcIkluZmluaXRlIGxvb3BcIjsgLy8gd2Ugc2hvdWxkIG5ldmVyIGdldCBoZXJlIHNpbmNlIHRoZSBsYXN0IHRlc3QgbWF0Y2hlcyBhbnkgY2hhcmFjdGVyXG4gICAgICB9XG4gICAgICBpZiAocGFyZW50aGVzaXNMZXZlbCA+IDApIHJldHVybiBmYWxzZTsgLy8gUGFyZW50aGVzKGVzKSBub3QgY2xvc2VkXG4gICAgfVxuXG4gICAgaWYgKGxpbmtSZWYgIT09IGZhbHNlKSB7XG4gICAgICAvLyBSZWYgbGluazsgY2hlY2sgdGhhdCBsaW5rUmVmIGlzIHZhbGlkXG4gICAgICBsZXQgdmFsaWQgPSBmYWxzZTtcbiAgICAgIGZvciAobGV0IGxhYmVsIG9mIHRoaXMubGlua0xhYmVscykge1xuICAgICAgICBpZiAobGFiZWwgPT0gbGlua1JlZikge1xuICAgICAgICAgIHZhbGlkID0gdHJ1ZTtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgbGV0IGxhYmVsID0gdmFsaWRcbiAgICAgICAgPyBcIlRNTGlua0xhYmVsIFRNTGlua0xhYmVsX1ZhbGlkXCJcbiAgICAgICAgOiBcIlRNTGlua0xhYmVsIFRNTGlua0xhYmVsX0ludmFsaWRcIjtcbiAgICAgIGxldCBvdXRwdXQgPSBgPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrXyR7dHlwZX1cIj4ke29wZW5lcn08L3NwYW4+PHNwYW4gY2xhc3M9XCIke3R5cGV9ICR7XG4gICAgICAgIGxpbmtMYWJlbC5sZW5ndGggPCAzIHx8ICFsaW5rTGFiZWxbMV0gPyBsYWJlbCA6IFwiXCJcbiAgICAgIH1cIj4ke3RoaXMucHJvY2Vzc0lubGluZVN0eWxlcyhcbiAgICAgICAgbGlua1RleHRcbiAgICAgICl9PC9zcGFuPjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya18ke3R5cGV9XCI+XTwvc3Bhbj5gO1xuXG4gICAgICBpZiAobGlua0xhYmVsLmxlbmd0aCA+PSAzKSB7XG4gICAgICAgIG91dHB1dCA9IG91dHB1dC5jb25jYXQoXG4gICAgICAgICAgYDxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya18ke3R5cGV9XCI+JHtsaW5rTGFiZWxbMF19PC9zcGFuPmAsXG4gICAgICAgICAgYDxzcGFuIGNsYXNzPVwiJHtsYWJlbH1cIj4ke2xpbmtMYWJlbFsxXX08L3NwYW4+YCxcbiAgICAgICAgICBgPHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrXyR7dHlwZX1cIj4ke2xpbmtMYWJlbFsyXX08L3NwYW4+YFxuICAgICAgICApO1xuICAgICAgfVxuICAgICAgcmV0dXJuIHtcbiAgICAgICAgb3V0cHV0OiBvdXRwdXQsXG4gICAgICAgIGNoYXJDb3VudDogY3VycmVudE9mZnNldCxcbiAgICAgIH07XG4gICAgfSBlbHNlIGlmIChsaW5rRGV0YWlscykge1xuICAgICAgLy8gSW5saW5lIGxpbmtcblxuICAgICAgLy8gVGhpcyBzaG91bGQgbmV2ZXIgaGFwcGVuLCBidXQgYmV0dGVyIHNhZmUgdGhhbiBzb3JyeS5cbiAgICAgIHdoaWxlIChsaW5rRGV0YWlscy5sZW5ndGggPCA3KSB7XG4gICAgICAgIGxpbmtEZXRhaWxzLnB1c2goXCJcIik7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiB7XG4gICAgICAgIG91dHB1dDogYDxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya18ke3R5cGV9XCI+JHtvcGVuZXJ9PC9zcGFuPjxzcGFuIGNsYXNzPVwiJHt0eXBlfVwiPiR7dGhpcy5wcm9jZXNzSW5saW5lU3R5bGVzKFxuICAgICAgICAgIGxpbmtUZXh0XG4gICAgICAgICl9PC9zcGFuPjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya18ke3R5cGV9XCI+XSgke1xuICAgICAgICAgIGxpbmtEZXRhaWxzWzBdXG4gICAgICAgIH08L3NwYW4+PHNwYW4gY2xhc3M9XCIke3R5cGV9RGVzdGluYXRpb25cIj4ke1xuICAgICAgICAgIGxpbmtEZXRhaWxzWzFdXG4gICAgICAgIH08L3NwYW4+PHNwYW4gY2xhc3M9XCJUTU1hcmsgVE1NYXJrXyR7dHlwZX1cIj4ke2xpbmtEZXRhaWxzWzJdfSR7XG4gICAgICAgICAgbGlua0RldGFpbHNbM11cbiAgICAgICAgfSR7bGlua0RldGFpbHNbNF19PC9zcGFuPjxzcGFuIGNsYXNzPVwiJHt0eXBlfVRpdGxlXCI+JHtcbiAgICAgICAgICBsaW5rRGV0YWlsc1s1XVxuICAgICAgICB9PC9zcGFuPjxzcGFuIGNsYXNzPVwiVE1NYXJrIFRNTWFya18ke3R5cGV9XCI+JHtsaW5rRGV0YWlsc1s2XX0pPC9zcGFuPmAsXG4gICAgICAgIGNoYXJDb3VudDogY3VycmVudE9mZnNldCxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgLyoqXG4gICAqIEZvcm1hdHMgYSBtYXJrZG93biBzdHJpbmcgYXMgSFRNTCwgdXNpbmcgTWFya2Rvd24gaW5saW5lIGZvcm1hdHRpbmcuXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBvcmlnaW5hbFN0cmluZyBUaGUgaW5wdXQgKG1hcmtkb3duIGlubGluZSBmb3JtYXR0ZWQpIHN0cmluZ1xuICAgKiBAcmV0dXJucyB7c3RyaW5nfSBUaGUgSFRNTCBmb3JtYXR0ZWQgb3V0cHV0XG4gICAqL1xuICBwcm9jZXNzSW5saW5lU3R5bGVzKG9yaWdpbmFsU3RyaW5nKSB7XG4gICAgbGV0IHByb2Nlc3NlZCA9IFwiXCI7XG4gICAgbGV0IHN0YWNrID0gW107IC8vIFN0YWNrIGlzIGFuIGFycmF5IG9mIG9iamVjdHMgb2YgdGhlIGZvcm1hdDoge2RlbGltaXRlciwgZGVsaW1TdHJpbmcsIGNvdW50LCBvdXRwdXR9XG4gICAgbGV0IG9mZnNldCA9IDA7XG4gICAgbGV0IHN0cmluZyA9IG9yaWdpbmFsU3RyaW5nO1xuXG4gICAgb3V0ZXI6IHdoaWxlIChzdHJpbmcpIHtcbiAgICAgIC8vIFByb2Nlc3Mgc2ltcGxlIHJ1bGVzIChub24tZGVsaW1pdGVyKVxuICAgICAgZm9yIChsZXQgcnVsZSBvZiBbXCJlc2NhcGVcIiwgXCJjb2RlXCIsIFwiYXV0b2xpbmtcIiwgXCJodG1sXCJdKSB7XG4gICAgICAgIGxldCBjYXAgPSBpbmxpbmVHcmFtbWFyW3J1bGVdLnJlZ2V4cC5leGVjKHN0cmluZyk7XG4gICAgICAgIGlmIChjYXApIHtcbiAgICAgICAgICBzdHJpbmcgPSBzdHJpbmcuc3Vic3RyKGNhcFswXS5sZW5ndGgpO1xuICAgICAgICAgIG9mZnNldCArPSBjYXBbMF0ubGVuZ3RoO1xuICAgICAgICAgIHByb2Nlc3NlZCArPSBpbmxpbmVHcmFtbWFyW3J1bGVdLnJlcGxhY2VtZW50XG4gICAgICAgICAgICAvLyAucmVwbGFjZSgvXFwkXFwkKFsxLTldKS9nLCAoc3RyLCBwMSkgPT4gcHJvY2Vzc0lubGluZVN0eWxlcyhjYXBbcDFdKSkgLy8gdG9kbyByZWN1cnNpdmUgY2FsbGluZ1xuICAgICAgICAgICAgLnJlcGxhY2UoL1xcJChbMS05XSkvZywgKHN0ciwgcDEpID0+IGh0bWxlc2NhcGUoY2FwW3AxXSkpO1xuICAgICAgICAgIGNvbnRpbnVlIG91dGVyO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIENoZWNrIGZvciBsaW5rcyAvIGltYWdlc1xuICAgICAgbGV0IHBvdGVudGlhbExpbmsgPSBzdHJpbmcubWF0Y2goaW5saW5lR3JhbW1hci5saW5rT3Blbi5yZWdleHApO1xuICAgICAgbGV0IHBvdGVudGlhbEltYWdlID0gc3RyaW5nLm1hdGNoKGlubGluZUdyYW1tYXIuaW1hZ2VPcGVuLnJlZ2V4cCk7XG4gICAgICBpZiAocG90ZW50aWFsSW1hZ2UgfHwgcG90ZW50aWFsTGluaykge1xuICAgICAgICBsZXQgcmVzdWx0ID0gdGhpcy5wYXJzZUxpbmtPckltYWdlKHN0cmluZywgcG90ZW50aWFsSW1hZ2UpO1xuICAgICAgICBpZiAocmVzdWx0KSB7XG4gICAgICAgICAgcHJvY2Vzc2VkID0gYCR7cHJvY2Vzc2VkfSR7cmVzdWx0Lm91dHB1dH1gO1xuICAgICAgICAgIHN0cmluZyA9IHN0cmluZy5zdWJzdHIocmVzdWx0LmNoYXJDb3VudCk7XG4gICAgICAgICAgb2Zmc2V0ICs9IHJlc3VsdC5jaGFyQ291bnQ7XG4gICAgICAgICAgY29udGludWUgb3V0ZXI7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgLy8gQ2hlY2sgZm9yIGVtIC8gc3Ryb25nIGRlbGltaXRlcnNcbiAgICAgIGxldCBjYXAgPSAvKF5cXCorKXwoXl8rKS8uZXhlYyhzdHJpbmcpO1xuICAgICAgaWYgKGNhcCkge1xuICAgICAgICBsZXQgZGVsaW1Db3VudCA9IGNhcFswXS5sZW5ndGg7XG4gICAgICAgIGNvbnN0IGRlbGltU3RyaW5nID0gY2FwWzBdO1xuICAgICAgICBjb25zdCBjdXJyZW50RGVsaW1pdGVyID0gY2FwWzBdWzBdOyAvLyBUaGlzIHNob3VsZCBiZSAqIG9yIF9cblxuICAgICAgICBzdHJpbmcgPSBzdHJpbmcuc3Vic3RyKGNhcFswXS5sZW5ndGgpO1xuXG4gICAgICAgIC8vIFdlIGhhdmUgYSBkZWxpbWl0ZXIgcnVuLiBMZXQncyBjaGVjayBpZiBpdCBjYW4gb3BlbiBvciBjbG9zZSBhbiBlbXBoYXNpcy5cblxuICAgICAgICBjb25zdCBwcmVjZWRpbmcgPSBvZmZzZXQgPiAwID8gb3JpZ2luYWxTdHJpbmcuc3Vic3RyKDAsIG9mZnNldCkgOiBcIiBcIjsgLy8gYmVnaW5uaW5nIGFuZCBlbmQgb2YgbGluZSBjb3VudCBhcyB3aGl0ZXNwYWNlXG4gICAgICAgIGNvbnN0IGZvbGxvd2luZyA9XG4gICAgICAgICAgb2Zmc2V0ICsgY2FwWzBdLmxlbmd0aCA8IG9yaWdpbmFsU3RyaW5nLmxlbmd0aCA/IHN0cmluZyA6IFwiIFwiO1xuXG4gICAgICAgIGNvbnN0IHB1bmN0dWF0aW9uRm9sbG93cyA9IGZvbGxvd2luZy5tYXRjaChwdW5jdHVhdGlvbkxlYWRpbmcpO1xuICAgICAgICBjb25zdCBwdW5jdHVhdGlvblByZWNlZGVzID0gcHJlY2VkaW5nLm1hdGNoKHB1bmN0dWF0aW9uVHJhaWxpbmcpO1xuICAgICAgICBjb25zdCB3aGl0ZXNwYWNlRm9sbG93cyA9IGZvbGxvd2luZy5tYXRjaCgvXlxccy8pO1xuICAgICAgICBjb25zdCB3aGl0ZXNwYWNlUHJlY2VkZXMgPSBwcmVjZWRpbmcubWF0Y2goL1xccyQvKTtcblxuICAgICAgICAvLyBUaGVzZSBhcmUgdGhlIHJ1bGVzIGZvciByaWdodC1mbGFua2luZyBhbmQgbGVmdC1mbGFua2luZyBkZWxpbWl0ZXIgcnVucyBhcyBwZXIgQ29tbW9uTWFyayBzcGVjXG4gICAgICAgIGxldCBjYW5PcGVuID1cbiAgICAgICAgICAhd2hpdGVzcGFjZUZvbGxvd3MgJiZcbiAgICAgICAgICAoIXB1bmN0dWF0aW9uRm9sbG93cyB8fFxuICAgICAgICAgICAgISF3aGl0ZXNwYWNlUHJlY2VkZXMgfHxcbiAgICAgICAgICAgICEhcHVuY3R1YXRpb25QcmVjZWRlcyk7XG4gICAgICAgIGxldCBjYW5DbG9zZSA9XG4gICAgICAgICAgIXdoaXRlc3BhY2VQcmVjZWRlcyAmJlxuICAgICAgICAgICghcHVuY3R1YXRpb25QcmVjZWRlcyB8fCAhIXdoaXRlc3BhY2VGb2xsb3dzIHx8ICEhcHVuY3R1YXRpb25Gb2xsb3dzKTtcblxuICAgICAgICAvLyBVbmRlcnNjb3JlcyBoYXZlIG1vcmUgZGV0YWlsZWQgcnVsZXMgdGhhbiBqdXN0IGJlaW5nIHBhcnQgb2YgbGVmdC0gb3IgcmlnaHQtZmxhbmtpbmcgcnVuOlxuICAgICAgICBpZiAoY3VycmVudERlbGltaXRlciA9PSBcIl9cIiAmJiBjYW5PcGVuICYmIGNhbkNsb3NlKSB7XG4gICAgICAgICAgY2FuT3BlbiA9IHB1bmN0dWF0aW9uUHJlY2VkZXM7XG4gICAgICAgICAgY2FuQ2xvc2UgPSBwdW5jdHVhdGlvbkZvbGxvd3M7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBJZiB0aGUgZGVsaW1pdGVyIGNhbiBjbG9zZSwgY2hlY2sgdGhlIHN0YWNrIGlmIHRoZXJlJ3Mgc29tZXRoaW5nIGl0IGNhbiBjbG9zZVxuICAgICAgICBpZiAoY2FuQ2xvc2UpIHtcbiAgICAgICAgICBsZXQgc3RhY2tQb2ludGVyID0gc3RhY2subGVuZ3RoIC0gMTtcbiAgICAgICAgICAvLyBTZWUgaWYgd2UgY2FuIGZpbmQgYSBtYXRjaGluZyBvcGVuaW5nIGRlbGltaXRlciwgbW92ZSBkb3duIHRocm91Z2ggdGhlIHN0YWNrXG4gICAgICAgICAgd2hpbGUgKGRlbGltQ291bnQgJiYgc3RhY2tQb2ludGVyID49IDApIHtcbiAgICAgICAgICAgIGlmIChzdGFja1tzdGFja1BvaW50ZXJdLmRlbGltaXRlciA9PSBjdXJyZW50RGVsaW1pdGVyKSB7XG4gICAgICAgICAgICAgIC8vIFdlIGZvdW5kIGEgbWF0Y2hpbmcgZGVsaW1pdGVyLCBsZXQncyBjb25zdHJ1Y3QgdGhlIGZvcm1hdHRlZCBzdHJpbmdcblxuICAgICAgICAgICAgICAvLyBGaXJzdGx5LCBpZiB3ZSBza2lwcGVkIGFueSBzdGFjayBsZXZlbHMsIHBvcCB0aGVtIGltbWVkaWF0ZWx5IChub24tbWF0Y2hpbmcgZGVsaW1pdGVycylcbiAgICAgICAgICAgICAgd2hpbGUgKHN0YWNrUG9pbnRlciA8IHN0YWNrLmxlbmd0aCAtIDEpIHtcbiAgICAgICAgICAgICAgICBjb25zdCBlbnRyeSA9IHN0YWNrLnBvcCgpO1xuICAgICAgICAgICAgICAgIHByb2Nlc3NlZCA9IGAke2VudHJ5Lm91dHB1dH0ke2VudHJ5LmRlbGltU3RyaW5nLnN1YnN0cihcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICBlbnRyeS5jb3VudFxuICAgICAgICAgICAgICAgICl9JHtwcm9jZXNzZWR9YDtcbiAgICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAgIC8vIFRoZW4sIGZvcm1hdCB0aGUgc3RyaW5nXG4gICAgICAgICAgICAgIGlmIChkZWxpbUNvdW50ID49IDIgJiYgc3RhY2tbc3RhY2tQb2ludGVyXS5jb3VudCA+PSAyKSB7XG4gICAgICAgICAgICAgICAgLy8gU3Ryb25nXG4gICAgICAgICAgICAgICAgcHJvY2Vzc2VkID0gYDxzcGFuIGNsYXNzPVwiVE1NYXJrXCI+JHtjdXJyZW50RGVsaW1pdGVyfSR7Y3VycmVudERlbGltaXRlcn08L3NwYW4+PHN0cm9uZyBjbGFzcz1cIlRNU3Ryb25nXCI+JHtwcm9jZXNzZWR9PC9zdHJvbmc+PHNwYW4gY2xhc3M9XCJUTU1hcmtcIj4ke2N1cnJlbnREZWxpbWl0ZXJ9JHtjdXJyZW50RGVsaW1pdGVyfTwvc3Bhbj5gO1xuICAgICAgICAgICAgICAgIGRlbGltQ291bnQgLT0gMjtcbiAgICAgICAgICAgICAgICBzdGFja1tzdGFja1BvaW50ZXJdLmNvdW50IC09IDI7XG4gICAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgICAgLy8gRW1cbiAgICAgICAgICAgICAgICBwcm9jZXNzZWQgPSBgPHNwYW4gY2xhc3M9XCJUTU1hcmtcIj4ke2N1cnJlbnREZWxpbWl0ZXJ9PC9zcGFuPjxlbSBjbGFzcz1cIlRNRW1cIj4ke3Byb2Nlc3NlZH08L2VtPjxzcGFuIGNsYXNzPVwiVE1NYXJrXCI+JHtjdXJyZW50RGVsaW1pdGVyfTwvc3Bhbj5gO1xuICAgICAgICAgICAgICAgIGRlbGltQ291bnQgLT0gMTtcbiAgICAgICAgICAgICAgICBzdGFja1tzdGFja1BvaW50ZXJdLmNvdW50IC09IDE7XG4gICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICAvLyBJZiB0aGF0IHN0YWNrIGxldmVsIGlzIGVtcHR5IG5vdywgcG9wIGl0XG4gICAgICAgICAgICAgIGlmIChzdGFja1tzdGFja1BvaW50ZXJdLmNvdW50ID09IDApIHtcbiAgICAgICAgICAgICAgICBsZXQgZW50cnkgPSBzdGFjay5wb3AoKTtcbiAgICAgICAgICAgICAgICBwcm9jZXNzZWQgPSBgJHtlbnRyeS5vdXRwdXR9JHtwcm9jZXNzZWR9YDtcbiAgICAgICAgICAgICAgICBzdGFja1BvaW50ZXItLTtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgLy8gVGhpcyBzdGFjayBsZXZlbCdzIGRlbGltaXRlciB0eXBlIGRvZXNuJ3QgbWF0Y2ggdGhlIGN1cnJlbnQgZGVsaW1pdGVyIHR5cGVcbiAgICAgICAgICAgICAgLy8gR28gZG93biBvbmUgbGV2ZWwgaW4gdGhlIHN0YWNrXG4gICAgICAgICAgICAgIHN0YWNrUG9pbnRlci0tO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICAvLyBJZiB0aGVyZSBhcmUgc3RpbGwgZGVsaW1pdGVycyBsZWZ0LCBhbmQgdGhlIGRlbGltaXRlciBydW4gY2FuIG9wZW4sIHB1c2ggaXQgb24gdGhlIHN0YWNrXG4gICAgICAgIGlmIChkZWxpbUNvdW50ICYmIGNhbk9wZW4pIHtcbiAgICAgICAgICBzdGFjay5wdXNoKHtcbiAgICAgICAgICAgIGRlbGltaXRlcjogY3VycmVudERlbGltaXRlcixcbiAgICAgICAgICAgIGRlbGltU3RyaW5nOiBkZWxpbVN0cmluZyxcbiAgICAgICAgICAgIGNvdW50OiBkZWxpbUNvdW50LFxuICAgICAgICAgICAgb3V0cHV0OiBwcm9jZXNzZWQsXG4gICAgICAgICAgfSk7XG4gICAgICAgICAgcHJvY2Vzc2VkID0gXCJcIjsgLy8gQ3VycmVudCBmb3JtYXR0ZWQgb3V0cHV0IGhhcyBiZWVuIHB1c2hlZCBvbiB0aGUgc3RhY2sgYW5kIHdpbGwgYmUgcHJlcGVuZGVkIHdoZW4gdGhlIHN0YWNrIGdldHMgcG9wcGVkXG4gICAgICAgICAgZGVsaW1Db3VudCA9IDA7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBBbnkgZGVsaW1pdGVycyB0aGF0IGFyZSBsZWZ0IChjbG9zaW5nIHVubWF0Y2hlZCkgYXJlIGFwcGVuZGVkIHRvIHRoZSBvdXRwdXQuXG4gICAgICAgIGlmIChkZWxpbUNvdW50KSB7XG4gICAgICAgICAgcHJvY2Vzc2VkID0gYCR7cHJvY2Vzc2VkfSR7ZGVsaW1TdHJpbmcuc3Vic3RyKDAsIGRlbGltQ291bnQpfWA7XG4gICAgICAgIH1cblxuICAgICAgICBvZmZzZXQgKz0gY2FwWzBdLmxlbmd0aDtcbiAgICAgICAgY29udGludWUgb3V0ZXI7XG4gICAgICB9XG5cbiAgICAgIC8vIENoZWNrIGZvciBzdHJpa2V0aHJvdWdoIGRlbGltaXRlclxuICAgICAgY2FwID0gL15+fi8uZXhlYyhzdHJpbmcpO1xuICAgICAgaWYgKGNhcCkge1xuICAgICAgICBsZXQgY29uc3VtZWQgPSBmYWxzZTtcbiAgICAgICAgbGV0IHN0YWNrUG9pbnRlciA9IHN0YWNrLmxlbmd0aCAtIDE7XG4gICAgICAgIC8vIFNlZSBpZiB3ZSBjYW4gZmluZCBhIG1hdGNoaW5nIG9wZW5pbmcgZGVsaW1pdGVyLCBtb3ZlIGRvd24gdGhyb3VnaCB0aGUgc3RhY2tcbiAgICAgICAgd2hpbGUgKCFjb25zdW1lZCAmJiBzdGFja1BvaW50ZXIgPj0gMCkge1xuICAgICAgICAgIGlmIChzdGFja1tzdGFja1BvaW50ZXJdLmRlbGltaXRlciA9PSBcIn5cIikge1xuICAgICAgICAgICAgLy8gV2UgZm91bmQgYSBtYXRjaGluZyBkZWxpbWl0ZXIsIGxldCdzIGNvbnN0cnVjdCB0aGUgZm9ybWF0dGVkIHN0cmluZ1xuXG4gICAgICAgICAgICAvLyBGaXJzdGx5LCBpZiB3ZSBza2lwcGVkIGFueSBzdGFjayBsZXZlbHMsIHBvcCB0aGVtIGltbWVkaWF0ZWx5IChub24tbWF0Y2hpbmcgZGVsaW1pdGVycylcbiAgICAgICAgICAgIHdoaWxlIChzdGFja1BvaW50ZXIgPCBzdGFjay5sZW5ndGggLSAxKSB7XG4gICAgICAgICAgICAgIGNvbnN0IGVudHJ5ID0gc3RhY2sucG9wKCk7XG4gICAgICAgICAgICAgIHByb2Nlc3NlZCA9IGAke2VudHJ5Lm91dHB1dH0ke2VudHJ5LmRlbGltU3RyaW5nLnN1YnN0cihcbiAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgIGVudHJ5LmNvdW50XG4gICAgICAgICAgICAgICl9JHtwcm9jZXNzZWR9YDtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgLy8gVGhlbiwgZm9ybWF0IHRoZSBzdHJpbmdcbiAgICAgICAgICAgIHByb2Nlc3NlZCA9IGA8c3BhbiBjbGFzcz1cIlRNTWFya1wiPn5+PC9zcGFuPjxkZWwgY2xhc3M9XCJUTVN0cmlrZXRocm91Z2hcIj4ke3Byb2Nlc3NlZH08L2RlbD48c3BhbiBjbGFzcz1cIlRNTWFya1wiPn5+PC9zcGFuPmA7XG4gICAgICAgICAgICBsZXQgZW50cnkgPSBzdGFjay5wb3AoKTtcbiAgICAgICAgICAgIHByb2Nlc3NlZCA9IGAke2VudHJ5Lm91dHB1dH0ke3Byb2Nlc3NlZH1gO1xuICAgICAgICAgICAgY29uc3VtZWQgPSB0cnVlO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAvLyBUaGlzIHN0YWNrIGxldmVsJ3MgZGVsaW1pdGVyIHR5cGUgZG9lc24ndCBtYXRjaCB0aGUgY3VycmVudCBkZWxpbWl0ZXIgdHlwZVxuICAgICAgICAgICAgLy8gR28gZG93biBvbmUgbGV2ZWwgaW4gdGhlIHN0YWNrXG4gICAgICAgICAgICBzdGFja1BvaW50ZXItLTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICAvLyBJZiB0aGVyZSBhcmUgc3RpbGwgZGVsaW1pdGVycyBsZWZ0LCBhbmQgdGhlIGRlbGltaXRlciBydW4gY2FuIG9wZW4sIHB1c2ggaXQgb24gdGhlIHN0YWNrXG4gICAgICAgIGlmICghY29uc3VtZWQpIHtcbiAgICAgICAgICBzdGFjay5wdXNoKHtcbiAgICAgICAgICAgIGRlbGltaXRlcjogXCJ+XCIsXG4gICAgICAgICAgICBkZWxpbVN0cmluZzogXCJ+flwiLFxuICAgICAgICAgICAgY291bnQ6IDIsXG4gICAgICAgICAgICBvdXRwdXQ6IHByb2Nlc3NlZCxcbiAgICAgICAgICB9KTtcbiAgICAgICAgICBwcm9jZXNzZWQgPSBcIlwiOyAvLyBDdXJyZW50IGZvcm1hdHRlZCBvdXRwdXQgaGFzIGJlZW4gcHVzaGVkIG9uIHRoZSBzdGFjayBhbmQgd2lsbCBiZSBwcmVwZW5kZWQgd2hlbiB0aGUgc3RhY2sgZ2V0cyBwb3BwZWRcbiAgICAgICAgfVxuXG4gICAgICAgIG9mZnNldCArPSBjYXBbMF0ubGVuZ3RoO1xuICAgICAgICBzdHJpbmcgPSBzdHJpbmcuc3Vic3RyKGNhcFswXS5sZW5ndGgpO1xuICAgICAgICBjb250aW51ZSBvdXRlcjtcbiAgICAgIH1cblxuICAgICAgLy8gUHJvY2VzcyAnZGVmYXVsdCcgcnVsZVxuICAgICAgY2FwID0gaW5saW5lR3JhbW1hci5kZWZhdWx0LnJlZ2V4cC5leGVjKHN0cmluZyk7XG4gICAgICBpZiAoY2FwKSB7XG4gICAgICAgIHN0cmluZyA9IHN0cmluZy5zdWJzdHIoY2FwWzBdLmxlbmd0aCk7XG4gICAgICAgIG9mZnNldCArPSBjYXBbMF0ubGVuZ3RoO1xuICAgICAgICBwcm9jZXNzZWQgKz0gaW5saW5lR3JhbW1hci5kZWZhdWx0LnJlcGxhY2VtZW50LnJlcGxhY2UoXG4gICAgICAgICAgL1xcJChbMS05XSkvZyxcbiAgICAgICAgICAoc3RyLCBwMSkgPT4gaHRtbGVzY2FwZShjYXBbcDFdKVxuICAgICAgICApO1xuICAgICAgICBjb250aW51ZSBvdXRlcjtcbiAgICAgIH1cbiAgICAgIHRocm93IFwiSW5maW5pdGUgbG9vcCFcIjtcbiAgICB9XG5cbiAgICAvLyBFbXB0eSB0aGUgc3RhY2ssIGFueSBvcGVuaW5nIGRlbGltaXRlcnMgYXJlIHVudXNlZFxuICAgIHdoaWxlIChzdGFjay5sZW5ndGgpIHtcbiAgICAgIGNvbnN0IGVudHJ5ID0gc3RhY2sucG9wKCk7XG4gICAgICBwcm9jZXNzZWQgPSBgJHtlbnRyeS5vdXRwdXR9JHtlbnRyeS5kZWxpbVN0cmluZy5zdWJzdHIoXG4gICAgICAgIDAsXG4gICAgICAgIGVudHJ5LmNvdW50XG4gICAgICApfSR7cHJvY2Vzc2VkfWA7XG4gICAgfVxuXG4gICAgcmV0dXJuIHByb2Nlc3NlZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBDbGVhcnMgdGhlIGxpbmUgZGlydHkgZmxhZyAocmVzZXRzIGl0IHRvIGFuIGFycmF5IG9mIGZhbHNlKVxuICAgKi9cbiAgY2xlYXJEaXJ0eUZsYWcoKSB7XG4gICAgdGhpcy5saW5lRGlydHkgPSBuZXcgQXJyYXkodGhpcy5saW5lcy5sZW5ndGgpO1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdGhpcy5saW5lRGlydHkubGVuZ3RoOyBpKyspIHtcbiAgICAgIHRoaXMubGluZURpcnR5W2ldID0gZmFsc2U7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIGNsYXNzIHByb3BlcnRpZXMgKGxpbmVzLCBsaW5lRWxlbWVudHMpIGZyb20gdGhlIERPTS5cbiAgICogQHJldHVybnMgdHJ1ZSBpZiBjb250ZW50cyBjaGFuZ2VkXG4gICAqL1xuICB1cGRhdGVMaW5lQ29udGVudHMoKSB7XG4gICAgLy8gdGhpcy5saW5lRGlydHkgPSBbXTtcbiAgICAvLyBDaGVjayBpZiB3ZSBoYXZlIGNoYW5nZWQgYW55dGhpbmcgYWJvdXQgdGhlIG51bWJlciBvZiBsaW5lcyAoaW5zZXJ0ZWQgb3IgZGVsZXRlZCBhIHBhcmFncmFwaClcbiAgICAvLyA8IDAgbWVhbnMgbGluZShzKSByZW1vdmVkOyA+IDAgbWVhbnMgbGluZShzKSBhZGRlZFxuICAgIGxldCBsaW5lRGVsdGEgPSB0aGlzLmUuY2hpbGRFbGVtZW50Q291bnQgLSB0aGlzLmxpbmVzLmxlbmd0aDtcbiAgICBpZiAobGluZURlbHRhKSB7XG4gICAgICAvLyB5dXAuIExldCdzIHRyeSBob3cgbXVjaCB3ZSBjYW4gc2FsdmFnZSAoZmluZCBvdXQgd2hpY2ggbGluZXMgZnJvbSBiZWdpbm5pbmcgYW5kIGVuZCB3ZXJlIHVuY2hhbmdlZClcbiAgICAgIC8vIEZpbmQgbGluZXMgZnJvbSB0aGUgYmVnaW5uaW5nIHRoYXQgaGF2ZW4ndCBjaGFuZ2VkLi4uXG4gICAgICBsZXQgZmlyc3RDaGFuZ2VkTGluZSA9IDA7XG4gICAgICB3aGlsZSAoXG4gICAgICAgIGZpcnN0Q2hhbmdlZExpbmUgPD0gdGhpcy5saW5lcy5sZW5ndGggJiZcbiAgICAgICAgZmlyc3RDaGFuZ2VkTGluZSA8PSB0aGlzLmxpbmVFbGVtZW50cy5sZW5ndGggJiZcbiAgICAgICAgdGhpcy5saW5lRWxlbWVudHNbZmlyc3RDaGFuZ2VkTGluZV0gJiYgLy8gQ2hlY2sgdGhhdCB0aGUgbGluZSBlbGVtZW50IGhhc24ndCBiZWVuIGRlbGV0ZWRcbiAgICAgICAgdGhpcy5saW5lc1tmaXJzdENoYW5nZWRMaW5lXSA9PVxuICAgICAgICAgIHRoaXMubGluZUVsZW1lbnRzW2ZpcnN0Q2hhbmdlZExpbmVdLnRleHRDb250ZW50ICYmXG4gICAgICAgIHRoaXMubGluZVR5cGVzW2ZpcnN0Q2hhbmdlZExpbmVdID09XG4gICAgICAgICAgdGhpcy5saW5lRWxlbWVudHNbZmlyc3RDaGFuZ2VkTGluZV0uY2xhc3NOYW1lXG4gICAgICApIHtcbiAgICAgICAgZmlyc3RDaGFuZ2VkTGluZSsrO1xuICAgICAgfVxuXG4gICAgICAvLyBFbmQgYWxzbyBmcm9tIHRoZSBlbmRcbiAgICAgIGxldCBsYXN0Q2hhbmdlZExpbmUgPSAtMTtcbiAgICAgIHdoaWxlIChcbiAgICAgICAgLWxhc3RDaGFuZ2VkTGluZSA8IHRoaXMubGluZXMubGVuZ3RoICYmXG4gICAgICAgIC1sYXN0Q2hhbmdlZExpbmUgPCB0aGlzLmxpbmVFbGVtZW50cy5sZW5ndGggJiZcbiAgICAgICAgdGhpcy5saW5lc1t0aGlzLmxpbmVzLmxlbmd0aCArIGxhc3RDaGFuZ2VkTGluZV0gPT1cbiAgICAgICAgICB0aGlzLmxpbmVFbGVtZW50c1t0aGlzLmxpbmVFbGVtZW50cy5sZW5ndGggKyBsYXN0Q2hhbmdlZExpbmVdXG4gICAgICAgICAgICAudGV4dENvbnRlbnQgJiZcbiAgICAgICAgdGhpcy5saW5lVHlwZXNbdGhpcy5saW5lcy5sZW5ndGggKyBsYXN0Q2hhbmdlZExpbmVdID09XG4gICAgICAgICAgdGhpcy5saW5lRWxlbWVudHNbdGhpcy5saW5lRWxlbWVudHMubGVuZ3RoICsgbGFzdENoYW5nZWRMaW5lXVxuICAgICAgICAgICAgLmNsYXNzTmFtZVxuICAgICAgKSB7XG4gICAgICAgIGxhc3RDaGFuZ2VkTGluZS0tO1xuICAgICAgfVxuXG4gICAgICBsZXQgbGluZXNUb0RlbGV0ZSA9XG4gICAgICAgIHRoaXMubGluZXMubGVuZ3RoICsgbGFzdENoYW5nZWRMaW5lICsgMSAtIGZpcnN0Q2hhbmdlZExpbmU7XG4gICAgICBpZiAobGluZXNUb0RlbGV0ZSA8IC1saW5lRGVsdGEpIGxpbmVzVG9EZWxldGUgPSAtbGluZURlbHRhO1xuICAgICAgaWYgKGxpbmVzVG9EZWxldGUgPCAwKSBsaW5lc1RvRGVsZXRlID0gMDtcblxuICAgICAgbGV0IGxpbmVzVG9BZGQgPSBbXTtcbiAgICAgIGZvciAobGV0IGwgPSAwOyBsIDwgbGluZXNUb0RlbGV0ZSArIGxpbmVEZWx0YTsgbCsrKSB7XG4gICAgICAgIGxpbmVzVG9BZGQucHVzaCh0aGlzLmxpbmVFbGVtZW50c1tmaXJzdENoYW5nZWRMaW5lICsgbF0udGV4dENvbnRlbnQpO1xuICAgICAgfVxuICAgICAgdGhpcy5zcGxpY2VMaW5lcyhmaXJzdENoYW5nZWRMaW5lLCBsaW5lc1RvRGVsZXRlLCBsaW5lc1RvQWRkLCBmYWxzZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIE5vIGxpbmVzIGFkZGVkIG9yIHJlbW92ZWRcbiAgICAgIGZvciAobGV0IGxpbmUgPSAwOyBsaW5lIDwgdGhpcy5saW5lRWxlbWVudHMubGVuZ3RoOyBsaW5lKyspIHtcbiAgICAgICAgbGV0IGUgPSB0aGlzLmxpbmVFbGVtZW50c1tsaW5lXTtcbiAgICAgICAgbGV0IGN0ID0gZS50ZXh0Q29udGVudDtcbiAgICAgICAgaWYgKHRoaXMubGluZXNbbGluZV0gIT09IGN0IHx8IHRoaXMubGluZVR5cGVzW2xpbmVdICE9PSBlLmNsYXNzTmFtZSkge1xuICAgICAgICAgIC8vIExpbmUgY2hhbmdlZCwgdXBkYXRlIGl0XG4gICAgICAgICAgdGhpcy5saW5lc1tsaW5lXSA9IGN0O1xuICAgICAgICAgIHRoaXMubGluZVR5cGVzW2xpbmVdID0gZS5jbGFzc05hbWU7XG4gICAgICAgICAgdGhpcy5saW5lRGlydHlbbGluZV0gPSB0cnVlO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFByb2Nlc3NlcyBhIG5ldyBwYXJhZ3JhcGguXG4gICAqIEBwYXJhbSBzZWwgVGhlIGN1cnJlbnQgc2VsZWN0aW9uXG4gICAqL1xuICBwcm9jZXNzTmV3UGFyYWdyYXBoKHNlbCkge1xuICAgIGlmICghc2VsKSByZXR1cm47XG5cbiAgICAvLyBVcGRhdGUgbGluZXMgZnJvbSBjb250ZW50XG4gICAgdGhpcy51cGRhdGVMaW5lQ29udGVudHMoKTtcblxuICAgIGxldCBjb250aW51YWJsZVR5cGUgPSBmYWxzZTtcbiAgICAvLyBMZXQncyBzZWUgaWYgd2UgbmVlZCB0byBjb250aW51ZSBhIGxpc3RcblxuICAgIGxldCBjaGVja0xpbmUgPSBzZWwuY29sID4gMCA/IHNlbC5yb3cgOiBzZWwucm93IC0gMTtcbiAgICBzd2l0Y2ggKHRoaXMubGluZVR5cGVzW2NoZWNrTGluZV0pIHtcbiAgICAgIGNhc2UgXCJUTVVMXCI6XG4gICAgICAgIGNvbnRpbnVhYmxlVHlwZSA9IFwiVE1VTFwiO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgXCJUTU9MXCI6XG4gICAgICAgIGNvbnRpbnVhYmxlVHlwZSA9IFwiVE1PTFwiO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgXCJUTUluZGVudGVkQ29kZVwiOlxuICAgICAgICBjb250aW51YWJsZVR5cGUgPSBcIlRNSW5kZW50ZWRDb2RlXCI7XG4gICAgICAgIGJyZWFrO1xuICAgIH1cblxuICAgIGxldCBsaW5lcyA9IHRoaXMubGluZXNbc2VsLnJvd11cbiAgICAgIC5yZXBsYWNlKC9cXG5cXG4kLywgXCJcXG5cIilcbiAgICAgIC5zcGxpdCgvKD86XFxyXFxufFxcbnxcXHIpLyk7XG4gICAgaWYgKGxpbmVzLmxlbmd0aCA+IDEpIHtcbiAgICAgIC8vIE5ldyBsaW5lXG4gICAgICB0aGlzLnNwbGljZUxpbmVzKHNlbC5yb3csIDEsIGxpbmVzLCB0cnVlKTtcbiAgICAgIHNlbC5yb3crKztcbiAgICAgIHNlbC5jb2wgPSAwO1xuICAgIH1cblxuICAgIGlmIChjb250aW51YWJsZVR5cGUpIHtcbiAgICAgIC8vIENoZWNrIGlmIHRoZSBwcmV2aW91cyBsaW5lIHdhcyBub24tZW1wdHlcbiAgICAgIGxldCBjYXB0dXJlID0gbGluZUdyYW1tYXJbY29udGludWFibGVUeXBlXS5yZWdleHAuZXhlYyhcbiAgICAgICAgdGhpcy5saW5lc1tzZWwucm93IC0gMV1cbiAgICAgICk7XG4gICAgICBpZiAoY2FwdHVyZSkge1xuICAgICAgICAvLyBDb252ZW50aW9uOiBjYXB0dXJlWzFdIGlzIHRoZSBsaW5lIHR5cGUgbWFya2VyLCBjYXB0dXJlWzJdIGlzIHRoZSBjb250ZW50XG4gICAgICAgIGlmIChjYXB0dXJlWzJdKSB7XG4gICAgICAgICAgLy8gUHJldmlvdXMgbGluZSBoYXMgY29udGVudCwgY29udGludWUgdGhlIGNvbnRpbnVhYmxlIHR5cGVcblxuICAgICAgICAgIC8vIEhhY2sgZm9yIE9MOiBpbmNyZW1lbnQgbnVtYmVyXG4gICAgICAgICAgaWYgKGNvbnRpbnVhYmxlVHlwZSA9PSBcIlRNT0xcIikge1xuICAgICAgICAgICAgY2FwdHVyZVsxXSA9IGNhcHR1cmVbMV0ucmVwbGFjZSgvXFxkezEsOX0vLCAocmVzdWx0KSA9PiB7XG4gICAgICAgICAgICAgIHJldHVybiBwYXJzZUludChyZXN1bHRbMF0pICsgMTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH1cbiAgICAgICAgICB0aGlzLmxpbmVzW3NlbC5yb3ddID0gYCR7Y2FwdHVyZVsxXX0ke3RoaXMubGluZXNbc2VsLnJvd119YDtcbiAgICAgICAgICB0aGlzLmxpbmVEaXJ0eVtzZWwucm93XSA9IHRydWU7XG4gICAgICAgICAgc2VsLmNvbCA9IGNhcHR1cmVbMV0ubGVuZ3RoO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIFByZXZpb3VzIGxpbmUgaGFzIG5vIGNvbnRlbnQsIHJlbW92ZSB0aGUgY29udGludWFibGUgdHlwZSBmcm9tIHRoZSBwcmV2aW91cyByb3dcbiAgICAgICAgICB0aGlzLmxpbmVzW3NlbC5yb3cgLSAxXSA9IFwiXCI7XG4gICAgICAgICAgdGhpcy5saW5lRGlydHlbc2VsLnJvdyAtIDFdID0gdHJ1ZTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICB0aGlzLnVwZGF0ZUZvcm1hdHRpbmcoKTtcbiAgfVxuXG4gIC8vIC8qKlxuICAvLyAgKiBQcm9jZXNzZXMgYSBcImRlbGV0ZVwiIGlucHV0IGFjdGlvbi5cbiAgLy8gICogQHBhcmFtIHtvYmplY3R9IGZvY3VzIFRoZSBzZWxlY3Rpb25cbiAgLy8gICogQHBhcmFtIHtib29sZWFufSBmb3J3YXJkIElmIHRydWUsIHBlcmZvcm1zIGEgZm9yd2FyZCBkZWxldGUsIG90aGVyd2lzZSBwZXJmb3JtcyBhIGJhY2t3YXJkIGRlbGV0ZVxuICAvLyAgKi9cbiAgLy8gcHJvY2Vzc0RlbGV0ZShmb2N1cywgZm9yd2FyZCkge1xuICAvLyAgIGlmICghZm9jdXMpIHJldHVybjtcbiAgLy8gICBsZXQgYW5jaG9yID0gdGhpcy5nZXRTZWxlY3Rpb24odHJ1ZSk7XG4gIC8vICAgLy8gRG8gd2UgaGF2ZSBhIG5vbi1lbXB0eSBzZWxlY3Rpb24/XG4gIC8vICAgaWYgKGZvY3VzLmNvbCAhPSBhbmNob3IuY29sIHx8IGZvY3VzLnJvdyAhPSBhbmNob3Iucm93KSB7XG4gIC8vICAgICAvLyBub24tZW1wdHkuIGRpcmVjdGlvbiBkb2Vzbid0IG1hdHRlci5cbiAgLy8gICAgIHRoaXMucGFzdGUoJycsIGFuY2hvciwgZm9jdXMpO1xuICAvLyAgIH0gZWxzZSB7XG4gIC8vICAgICBpZiAoZm9yd2FyZCkge1xuICAvLyAgICAgICBpZiAoZm9jdXMuY29sIDwgdGhpcy5saW5lc1tmb2N1cy5yb3ddLmxlbmd0aCkgdGhpcy5wYXN0ZSgnJywge3JvdzogZm9jdXMucm93LCBjb2w6IGZvY3VzLmNvbCArIDF9LCBmb2N1cyk7XG4gIC8vICAgICAgIGVsc2UgaWYgKGZvY3VzLmNvbCA8IHRoaXMubGluZXMubGVuZ3RoKSB0aGlzLnBhc3RlKCcnLCB7cm93OiBmb2N1cy5yb3cgKyAxLCBjb2w6IDB9LCBmb2N1cyk7XG4gIC8vICAgICAgIC8vIE90aGVyd2lzZSwgd2UncmUgYXQgdGhlIHZlcnkgZW5kIGFuZCBjYW4ndCBkZWxldGUgZm9yd2FyZFxuICAvLyAgICAgfSBlbHNlIHtcbiAgLy8gICAgICAgaWYgKGZvY3VzLmNvbCA+IDApIHRoaXMucGFzdGUoJycsIHtyb3c6IGZvY3VzLnJvdywgY29sOiBmb2N1cy5jb2wgLSAxfSwgZm9jdXMpO1xuICAvLyAgICAgICBlbHNlIGlmIChmb2N1cy5yb3cgPiAwKSB0aGlzLnBhc3RlKCcnLCB7cm93OiBmb2N1cy5yb3cgLSAxLCBjb2w6IHRoaXMubGluZXNbZm9jdXMucm93IC0gMV0ubGVuZ3RoIC0gMX0sIGZvY3VzKTtcbiAgLy8gICAgICAgLy8gT3RoZXJ3aXNlLCB3ZSdyZSBhdCB0aGUgdmVyeSBiZWdpbm5pbmcgYW5kIGNhbid0IGRlbGV0ZSBiYWNrd2FyZHNcbiAgLy8gICAgIH1cbiAgLy8gICB9XG5cbiAgLy8gfVxuXG4gIC8qKlxuICAgKiBHZXRzIHRoZSBjdXJyZW50IHBvc2l0aW9uIG9mIHRoZSBzZWxlY3Rpb24gY291bnRlZCBieSByb3cgYW5kIGNvbHVtbiBvZiB0aGUgZWRpdG9yIE1hcmtkb3duIGNvbnRlbnQgKGFzIG9wcG9zZWQgdG8gdGhlIHBvc2l0aW9uIGluIHRoZSBET00pLlxuICAgKlxuICAgKiBAcGFyYW0ge2Jvb2xlYW59IGdldEFuY2hvciBpZiBzZXQgdG8gdHJ1ZSwgZ2V0cyB0aGUgc2VsZWN0aW9uIGFuY2hvciAoc3RhcnQgcG9pbnQgb2YgdGhlIHNlbGVjdGlvbiksIG90aGVyd2lzZSBnZXRzIHRoZSBmb2N1cyAoZW5kIHBvaW50KS5cbiAgICogQHJldHVybiB7b2JqZWN0fSBBbiBvYmplY3QgcmVwcmVzZW50aW5nIHRoZSBzZWxlY3Rpb24sIHdpdGggcHJvcGVydGllcyBjb2wgYW5kIHJvdy5cbiAgICovXG4gIGdldFNlbGVjdGlvbihnZXRBbmNob3IgPSBmYWxzZSkge1xuICAgIGNvbnN0IHNlbGVjdGlvbiA9IHdpbmRvdy5nZXRTZWxlY3Rpb24oKTtcbiAgICBsZXQgc3RhcnROb2RlID0gZ2V0QW5jaG9yID8gc2VsZWN0aW9uLmFuY2hvck5vZGUgOiBzZWxlY3Rpb24uZm9jdXNOb2RlO1xuICAgIGlmICghc3RhcnROb2RlKSByZXR1cm4gbnVsbDtcbiAgICBsZXQgb2Zmc2V0ID0gZ2V0QW5jaG9yID8gc2VsZWN0aW9uLmFuY2hvck9mZnNldCA6IHNlbGVjdGlvbi5mb2N1c09mZnNldDtcbiAgICBpZiAoc3RhcnROb2RlID09IHRoaXMuZSkge1xuICAgICAgaWYgKG9mZnNldCA8IHRoaXMubGluZXMubGVuZ3RoKVxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIHJvdzogb2Zmc2V0LFxuICAgICAgICAgIGNvbDogMCxcbiAgICAgICAgfTtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIHJvdzogb2Zmc2V0IC0gMSxcbiAgICAgICAgY29sOiB0aGlzLmxpbmVzW29mZnNldCAtIDFdLmxlbmd0aCxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgbGV0IGNvbCA9IHRoaXMuY29tcHV0ZUNvbHVtbihzdGFydE5vZGUsIG9mZnNldCk7XG4gICAgaWYgKGNvbCA9PT0gbnVsbCkgcmV0dXJuIG51bGw7IC8vIFdlIGFyZSBvdXRzaWRlIG9mIHRoZSBlZGl0b3JcblxuICAgIC8vIEZpbmQgdGhlIHJvdyBub2RlXG4gICAgbGV0IG5vZGUgPSBzdGFydE5vZGU7XG4gICAgd2hpbGUgKG5vZGUucGFyZW50RWxlbWVudCAhPSB0aGlzLmUpIHtcbiAgICAgIG5vZGUgPSBub2RlLnBhcmVudEVsZW1lbnQ7XG4gICAgfVxuXG4gICAgbGV0IHJvdyA9IDA7XG4gICAgLy8gQ2hlY2sgaWYgd2UgY2FuIHJlYWQgYSBsaW5lIG51bWJlciBmcm9tIHRoZSBkYXRhLWxpbmUtbnVtIGF0dHJpYnV0ZS5cbiAgICAvLyBUaGUgbGFzdCBjb25kaXRpb24gaXMgYSBzZWN1cml0eSBtZWFzdXJlIHNpbmNlIGluc2VydGluZyBhIG5ldyBwYXJhZ3JhcGggY29waWVzIHRoZSBwcmV2aW91cyByb3dzJyBsaW5lIG51bWJlclxuICAgIGlmIChcbiAgICAgIG5vZGUuZGF0YXNldCAmJlxuICAgICAgbm9kZS5kYXRhc2V0LmxpbmVOdW0gJiZcbiAgICAgICghbm9kZS5wcmV2aW91c1NpYmxpbmcgfHxcbiAgICAgICAgbm9kZS5wcmV2aW91c1NpYmxpbmcuZGF0YXNldC5saW5lTnVtICE9IG5vZGUuZGF0YXNldC5saW5lTnVtKVxuICAgICkge1xuICAgICAgcm93ID0gcGFyc2VJbnQobm9kZS5kYXRhc2V0LmxpbmVOdW0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB3aGlsZSAobm9kZS5wcmV2aW91c1NpYmxpbmcpIHtcbiAgICAgICAgcm93Kys7XG4gICAgICAgIG5vZGUgPSBub2RlLnByZXZpb3VzU2libGluZztcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHsgcm93OiByb3csIGNvbDogY29sLCBub2RlOiBzdGFydE5vZGUgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wdXRlcyBhIGNvbHVtbiB3aXRoaW4gYW4gZWRpdG9yIGxpbmUgZnJvbSBhIG5vZGUgYW5kIG9mZnNldCB3aXRoaW4gdGhhdCBub2RlLlxuICAgKiBAcGFyYW0ge05vZGV9IHN0YXJ0Tm9kZSBUaGUgbm9kZVxuICAgKiBAcGFyYW0ge2ludH0gb2Zmc2V0IFRIZSBzZWxlY3Rpb25cbiAgICogQHJldHVybnMge2ludH0gdGhlIGNvbHVtbiwgb3IgbnVsbCBpZiB0aGUgbm9kZSBpcyBub3QgaW5zaWRlIHRoZSBlZGl0b3JcbiAgICovXG4gIGNvbXB1dGVDb2x1bW4oc3RhcnROb2RlLCBvZmZzZXQpIHtcbiAgICBsZXQgbm9kZSA9IHN0YXJ0Tm9kZTtcbiAgICBsZXQgY29sO1xuICAgIC8vIEZpcnN0LCBtYWtlIHN1cmUgd2UncmUgYWN0dWFsbHkgaW4gdGhlIGVkaXRvci5cbiAgICB3aGlsZSAobm9kZSAmJiBub2RlLnBhcmVudE5vZGUgIT0gdGhpcy5lKSB7XG4gICAgICBub2RlID0gbm9kZS5wYXJlbnROb2RlO1xuICAgIH1cbiAgICBpZiAobm9kZSA9PSBudWxsKSByZXR1cm4gbnVsbDtcblxuICAgIC8vIFRoZXJlIGFyZSB0d28gd2F5cyB0aGF0IG9mZnNldCBjYW4gYmUgZGVmaW5lZDpcbiAgICAvLyAtIEVpdGhlciwgdGhlIG5vZGUgaXMgYSB0ZXh0IG5vZGUsIGluIHdoaWNoIGNhc2UgaXQgaXMgdGhlIG9mZnNldCB3aXRoaW4gdGhlIHRleHRcbiAgICAvLyAtIE9yLCB0aGUgbm9kZSBpcyBhbiBlbGVtZW50IHdpdGggY2hpbGQgbm90ZXMsIGluIHdoaWNoIGNhc2UgdGhlIG9mZnNldCByZWZlcnMgdG8gdGhlXG4gICAgLy8gICBjaGlsZCBub2RlIGFmdGVyIHdoaWNoIHRoZSBzZWxlY3Rpb24gaXMgbG9jYXRlZFxuICAgIGlmIChzdGFydE5vZGUubm9kZVR5cGUgPT09IE5vZGUuVEVYVF9OT0RFIHx8IG9mZnNldCA9PT0gMCkge1xuICAgICAgLy8gSW4gdGhlIGNhc2UgdGhhdCB0aGUgbm9kZSBpcyBub24tdGV4dCBub2RlIGJ1dCB0aGUgb2Zmc2V0IGlzIDAsXG4gICAgICAvLyBUaGUgc2VsZWN0aW9uIGlzIGF0IHRoZSBiZWdpbm5pbmcgb2YgdGhhdCBlbGVtZW50IHNvIHdlXG4gICAgICAvLyBjYW4gc2ltcGx5IHVzZSB0aGUgc2FtZSBhcHByb2FjaCBhcyBpZiBpdCB3ZXJlIGF0IHRoZSBiZWdpbm5pbmdcbiAgICAgIC8vIG9mIGEgdGV4dCBub2RlLlxuICAgICAgY29sID0gb2Zmc2V0O1xuICAgICAgbm9kZSA9IHN0YXJ0Tm9kZTtcbiAgICB9IGVsc2UgaWYgKG9mZnNldCA+IDApIHtcbiAgICAgIG5vZGUgPSBzdGFydE5vZGUuY2hpbGROb2Rlc1tvZmZzZXQgLSAxXTtcbiAgICAgIGNvbCA9IG5vZGUudGV4dENvbnRlbnQubGVuZ3RoO1xuICAgIH1cbiAgICB3aGlsZSAobm9kZS5wYXJlbnROb2RlICE9IHRoaXMuZSkge1xuICAgICAgaWYgKG5vZGUucHJldmlvdXNTaWJsaW5nKSB7XG4gICAgICAgIG5vZGUgPSBub2RlLnByZXZpb3VzU2libGluZztcbiAgICAgICAgY29sICs9IG5vZGUudGV4dENvbnRlbnQubGVuZ3RoO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgbm9kZSA9IG5vZGUucGFyZW50Tm9kZTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIGNvbDtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wdXRlcyBET00gbm9kZSBhbmQgb2Zmc2V0IHdpdGhpbiB0aGF0IG5vZGUgZnJvbSBhIHBvc2l0aW9uIGV4cHJlc3NlZCBhcyByb3cgYW5kIGNvbHVtbi5cbiAgICogQHBhcmFtIHtpbnR9IHJvdyBSb3cgKGxpbmUgbnVtYmVyKVxuICAgKiBAcGFyYW0ge2ludH0gY29sIENvbHVtblxuICAgKiBAcmV0dXJucyBBbiBvYmplY3Qgd2l0aCB0d28gcHJvcGVydGllczogbm9kZSBhbmQgb2Zmc2V0LiBvZmZzZXQgbWF5IGJlIG51bGw7XG4gICAqL1xuICBjb21wdXRlTm9kZUFuZE9mZnNldChyb3csIGNvbCwgYmluZFJpZ2h0ID0gZmFsc2UpIHtcbiAgICBpZiAocm93ID49IHRoaXMubGluZUVsZW1lbnRzLmxlbmd0aCkge1xuICAgICAgLy8gU2VsZWN0aW9uIHBhc3QgdGhlIGVuZCBvZiB0ZXh0LCBzZXQgc2VsZWN0aW9uIHRvIGVuZCBvZiB0ZXh0XG4gICAgICByb3cgPSB0aGlzLmxpbmVFbGVtZW50cy5sZW5ndGggLSAxO1xuICAgICAgY29sID0gdGhpcy5saW5lc1tyb3ddLmxlbmd0aDtcbiAgICB9XG4gICAgaWYgKGNvbCA+IHRoaXMubGluZXNbcm93XS5sZW5ndGgpIHtcbiAgICAgIGNvbCA9IHRoaXMubGluZXNbcm93XS5sZW5ndGg7XG4gICAgfVxuICAgIGNvbnN0IHBhcmVudE5vZGUgPSB0aGlzLmxpbmVFbGVtZW50c1tyb3ddO1xuICAgIGxldCBub2RlID0gcGFyZW50Tm9kZS5maXJzdENoaWxkO1xuXG4gICAgbGV0IGNoaWxkcmVuQ29tcGxldGUgPSBmYWxzZTtcbiAgICAvLyBkZWZhdWx0IHJldHVybiB2YWx1ZVxuICAgIGxldCBydiA9IHtcbiAgICAgIG5vZGU6IHBhcmVudE5vZGUuZmlyc3RDaGlsZCA/IHBhcmVudE5vZGUuZmlyc3RDaGlsZCA6IHBhcmVudE5vZGUsXG4gICAgICBvZmZzZXQ6IDAsXG4gICAgfTtcblxuICAgIHdoaWxlIChub2RlICE9IHBhcmVudE5vZGUpIHtcbiAgICAgIGlmICghY2hpbGRyZW5Db21wbGV0ZSAmJiBub2RlLm5vZGVUeXBlID09PSBOb2RlLlRFWFRfTk9ERSkge1xuICAgICAgICBpZiAobm9kZS5ub2RlVmFsdWUubGVuZ3RoID49IGNvbCkge1xuICAgICAgICAgIGlmIChiaW5kUmlnaHQgJiYgbm9kZS5ub2RlVmFsdWUubGVuZ3RoID09IGNvbCkge1xuICAgICAgICAgICAgLy8gU2VsZWN0aW9uIGlzIGF0IHRoZSBlbmQgb2YgdGhpcyB0ZXh0IG5vZGUsIGJ1dCB3ZSBhcmUgYmluZGluZyByaWdodCAocHJlZmVyIGFuIG9mZnNldCBvZiAwIGluIHRoZSBuZXh0IHRleHQgbm9kZSlcbiAgICAgICAgICAgIC8vIFJlbWVtYmVyIHJldHVybiB2YWx1ZSBpbiBjYXNlIHdlIGRvbid0IGZpbmQgYW5vdGhlciB0ZXh0IG5vZGVcbiAgICAgICAgICAgIHJ2ID0geyBub2RlOiBub2RlLCBvZmZzZXQ6IGNvbCB9O1xuICAgICAgICAgICAgY29sID0gMDtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgcmV0dXJuIHsgbm9kZTogbm9kZSwgb2Zmc2V0OiBjb2wgfTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29sIC09IG5vZGUubm9kZVZhbHVlLmxlbmd0aDtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgaWYgKCFjaGlsZHJlbkNvbXBsZXRlICYmIG5vZGUuZmlyc3RDaGlsZCkge1xuICAgICAgICBub2RlID0gbm9kZS5maXJzdENoaWxkO1xuICAgICAgfSBlbHNlIGlmIChub2RlLm5leHRTaWJsaW5nKSB7XG4gICAgICAgIGNoaWxkcmVuQ29tcGxldGUgPSBmYWxzZTtcbiAgICAgICAgbm9kZSA9IG5vZGUubmV4dFNpYmxpbmc7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBjaGlsZHJlbkNvbXBsZXRlID0gdHJ1ZTtcbiAgICAgICAgbm9kZSA9IG5vZGUucGFyZW50Tm9kZTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBFaXRoZXIsIHRoZSBwb3NpdGlvbiB3YXMgaW52YWxpZCBhbmQgd2UganVzdCByZXR1cm4gdGhlIGRlZmF1bHQgcmV0dXJuIHZhbHVlXG4gICAgLy8gT3Igd2UgYXJlIGJpbmRpbmcgcmlnaHQgYW5kIHRoZSBzZWxlY3Rpb24gaXMgYXQgdGhlIGVuZCBvZiB0aGUgbGluZVxuICAgIHJldHVybiBydjtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXRzIHRoZSBzZWxlY3Rpb24gYmFzZWQgb24gcm93cyBhbmQgY29sdW1ucyB3aXRoaW4gdGhlIGVkaXRvciBNYXJrZG93biBjb250ZW50LlxuICAgKiBAcGFyYW0ge29iamVjdH0gZm9jdXMgT2JqZWN0IHJlcHJlc2VudGluZyB0aGUgc2VsZWN0aW9uLCBuZWVkcyB0byBoYXZlIHByb3BlcnRpZXMgcm93IGFuZCBjb2wuXG4gICAqIEBwYXJhbSBhbmNob3IgQW5jaG9yIG9mIHRoZSBzZWxlY3Rpb24uIElmIG5vdCBnaXZlbiwgYXNzdW1lcyB0aGUgY3VycmVudCBhbmNob3IuXG4gICAqL1xuICBzZXRTZWxlY3Rpb24oZm9jdXMsIGFuY2hvciA9IG51bGwpIHtcbiAgICBpZiAoIWZvY3VzKSByZXR1cm47XG5cbiAgICBsZXQgcmFuZ2UgPSBkb2N1bWVudC5jcmVhdGVSYW5nZSgpO1xuXG4gICAgbGV0IHsgbm9kZTogZm9jdXNOb2RlLCBvZmZzZXQ6IGZvY3VzT2Zmc2V0IH0gPSB0aGlzLmNvbXB1dGVOb2RlQW5kT2Zmc2V0KFxuICAgICAgZm9jdXMucm93LFxuICAgICAgZm9jdXMuY29sLFxuICAgICAgYW5jaG9yICYmIGFuY2hvci5yb3cgPT0gZm9jdXMucm93ICYmIGFuY2hvci5jb2wgPiBmb2N1cy5jb2xcbiAgICApOyAvLyBCaW5kIHNlbGVjdGlvbiByaWdodCBpZiBhbmNob3IgaXMgaW4gdGhlIHNhbWUgcm93IGFuZCBiZWhpbmQgdGhlIGZvY3VzXG4gICAgbGV0IGFuY2hvck5vZGUgPSBudWxsLFxuICAgICAgYW5jaG9yT2Zmc2V0ID0gbnVsbDtcbiAgICBpZiAoYW5jaG9yICYmIChhbmNob3Iucm93ICE9IGZvY3VzLnJvdyB8fCBhbmNob3IuY29sICE9IGZvY3VzLmNvbCkpIHtcbiAgICAgIGxldCB7IG5vZGUsIG9mZnNldCB9ID0gdGhpcy5jb21wdXRlTm9kZUFuZE9mZnNldChcbiAgICAgICAgYW5jaG9yLnJvdyxcbiAgICAgICAgYW5jaG9yLmNvbCxcbiAgICAgICAgZm9jdXMucm93ID09IGFuY2hvci5yb3cgJiYgZm9jdXMuY29sID4gYW5jaG9yLmNvbFxuICAgICAgKTtcbiAgICAgIGFuY2hvck5vZGUgPSBub2RlO1xuICAgICAgYW5jaG9yT2Zmc2V0ID0gb2Zmc2V0O1xuICAgIH1cblxuICAgIGlmIChhbmNob3JOb2RlKSByYW5nZS5zZXRTdGFydChhbmNob3JOb2RlLCBhbmNob3JPZmZzZXQpO1xuICAgIGVsc2UgcmFuZ2Uuc2V0U3RhcnQoZm9jdXNOb2RlLCBmb2N1c09mZnNldCk7XG4gICAgcmFuZ2Uuc2V0RW5kKGZvY3VzTm9kZSwgZm9jdXNPZmZzZXQpO1xuXG4gICAgbGV0IHdpbmRvd1NlbGVjdGlvbiA9IHdpbmRvdy5nZXRTZWxlY3Rpb24oKTtcbiAgICB3aW5kb3dTZWxlY3Rpb24ucmVtb3ZlQWxsUmFuZ2VzKCk7XG4gICAgd2luZG93U2VsZWN0aW9uLmFkZFJhbmdlKHJhbmdlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFdmVudCBoYW5kbGVyIGZvciBpbnB1dCBldmVudHNcbiAgICovXG4gIGhhbmRsZUlucHV0RXZlbnQoZXZlbnQpIHtcbiAgICAvLyBGb3IgY29tcG9zaXRpb24gaW5wdXQsIHdlIGFyZSBvbmx5IHVwZGF0aW5nIHRoZSB0ZXh0IGFmdGVyIHdlIGhhdmUgcmVjZWl2ZWRcbiAgICAvLyBhIGNvbXBvc2l0aW9uZW5kIGV2ZW50LCBzbyB3ZSByZXR1cm4gdXBvbiBpbnNlcnRDb21wb3NpdGlvblRleHQuXG4gICAgLy8gT3RoZXJ3aXNlLCB0aGUgRE9NIGNoYW5nZXMgYnJlYWsgdGhlIHRleHQgaW5wdXQuXG4gICAgaWYgKGV2ZW50LmlucHV0VHlwZSA9PSBcImluc2VydENvbXBvc2l0aW9uVGV4dFwiKSByZXR1cm47XG5cbiAgICBsZXQgZm9jdXMgPSB0aGlzLmdldFNlbGVjdGlvbigpO1xuXG4gICAgaWYgKFxuICAgICAgKGV2ZW50LmlucHV0VHlwZSA9PSBcImluc2VydFBhcmFncmFwaFwiIHx8XG4gICAgICAgIGV2ZW50LmlucHV0VHlwZSA9PSBcImluc2VydExpbmVCcmVha1wiKSAmJlxuICAgICAgZm9jdXNcbiAgICApIHtcbiAgICAgIHRoaXMuY2xlYXJEaXJ0eUZsYWcoKTtcbiAgICAgIHRoaXMucHJvY2Vzc05ld1BhcmFncmFwaChmb2N1cyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmICghdGhpcy5lLmZpcnN0Q2hpbGQpIHtcbiAgICAgICAgdGhpcy5lLmlubmVySFRNTCA9ICc8ZGl2IGNsYXNzPVwiVE1CbGFua0xpbmVcIj48YnI+PC9kaXY+JztcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRoaXMuZml4Tm9kZUhpZXJhcmNoeSgpO1xuICAgICAgfVxuICAgICAgdGhpcy51cGRhdGVMaW5lQ29udGVudHNBbmRGb3JtYXR0aW5nKCk7XG4gICAgfVxuICAgIGlmIChmb2N1cykge1xuICAgICAgdGhpcy5zZXRTZWxlY3Rpb24oZm9jdXMpO1xuICAgIH1cblxuICAgIHRoaXMuZmlyZUNoYW5nZSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpeGVzIHRoZSBub2RlIGhpZXJhcmNoeSDigJMgbWFrZXMgc3VyZSB0aGF0IGVhY2ggbGluZSBpcyBpbiBhIGRpdiwgYW5kIHRoZXJlIGFyZSBubyBuZXN0ZWQgZGl2c1xuICAgKi9cbiAgZml4Tm9kZUhpZXJhcmNoeSgpIHtcbiAgICBjb25zdCBvcmlnaW5hbENoaWxkcmVuID0gQXJyYXkuZnJvbSh0aGlzLmUuY2hpbGROb2Rlcyk7XG5cbiAgICBjb25zdCByZXBsYWNlQ2hpbGQgPSAoY2hpbGQsIC4uLm5ld0NoaWxkcmVuKSA9PiB7XG4gICAgICBjb25zdCBwYXJlbnQgPSBjaGlsZC5wYXJlbnRFbGVtZW50O1xuICAgICAgY29uc3QgbmV4dFNpYmxpbmcgPSBjaGlsZC5uZXh0U2libGluZztcbiAgICAgIHBhcmVudC5yZW1vdmVDaGlsZChjaGlsZCk7XG4gICAgICBuZXdDaGlsZHJlbi5mb3JFYWNoKChuZXdDaGlsZCkgPT5cbiAgICAgICAgbmV4dFNpYmxpbmdcbiAgICAgICAgICA/IHBhcmVudC5pbnNlcnRCZWZvcmUobmV3Q2hpbGQsIG5leHRTaWJsaW5nKVxuICAgICAgICAgIDogcGFyZW50LmFwcGVuZENoaWxkKG5ld0NoaWxkKVxuICAgICAgKTtcbiAgICB9O1xuXG4gICAgb3JpZ2luYWxDaGlsZHJlbi5mb3JFYWNoKChjaGlsZCkgPT4ge1xuICAgICAgaWYgKGNoaWxkLm5vZGVUeXBlICE9PSBOb2RlLkVMRU1FTlRfTk9ERSB8fCBjaGlsZC50YWdOYW1lICE9PSBcIkRJVlwiKSB7XG4gICAgICAgIC8vIEZvdW5kIGEgY2hpbGQgbm9kZSB0aGF0J3MgZWl0aGVyIG5vdCBhbiBlbGVtZW50IG9yIG5vdCBhIGRpdi4gV3JhcCBpdCBpbiBhIGRpdi5cbiAgICAgICAgY29uc3QgZGl2V3JhcHBlciA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoXCJkaXZcIik7XG4gICAgICAgIHJlcGxhY2VDaGlsZChjaGlsZCwgZGl2V3JhcHBlcik7XG4gICAgICAgIGRpdldyYXBwZXIuYXBwZW5kQ2hpbGQoY2hpbGQpO1xuICAgICAgfSBlbHNlIGlmIChjaGlsZC5jaGlsZE5vZGVzLmxlbmd0aCA9PSAwKSB7XG4gICAgICAgIC8vIEVtcHR5IGRpdiBjaGlsZCBub2RlLCBpbmNsdWRlIGF0IGxlYXN0IGEgPGJyIC8+XG4gICAgICAgIGNoaWxkLmFwcGVuZENoaWxkKGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoXCJiclwiKSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBjb25zdCBncmFuZENoaWxkcmVuID0gQXJyYXkuZnJvbShjaGlsZC5jaGlsZE5vZGVzKTtcbiAgICAgICAgaWYgKFxuICAgICAgICAgIGdyYW5kQ2hpbGRyZW4uc29tZShcbiAgICAgICAgICAgIChncmFuZENoaWxkKSA9PlxuICAgICAgICAgICAgICBncmFuZENoaWxkLm5vZGVUeXBlID09PSBOb2RlLkVMRU1FTlRfTk9ERSAmJlxuICAgICAgICAgICAgICBncmFuZENoaWxkLnRhZ05hbWUgPT09IFwiRElWXCJcbiAgICAgICAgICApXG4gICAgICAgICkge1xuICAgICAgICAgIHJldHVybiByZXBsYWNlQ2hpbGQoY2hpbGQsIGdyYW5kQ2hpbGRyZW4pO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogRXZlbnQgaGFuZGxlciBmb3IgdGhlIFwiZHJvcFwiIGV2ZW50XG4gICAqL1xuICBoYW5kbGVEcm9wKGV2ZW50KSB7XG4gICAgZXZlbnQucHJldmVudERlZmF1bHQoKTtcblxuICAgIHRoaXMuZmlyZURyb3AoZXZlbnQuZGF0YVRyYW5zZmVyKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFdmVudCBoYW5kbGVyIGZvciBcInNlbGVjdGlvbmNoYW5nZVwiIGV2ZW50cy5cbiAgICovXG4gIGhhbmRsZVNlbGVjdGlvbkNoYW5nZUV2ZW50KCkge1xuICAgIHRoaXMuZmlyZVNlbGVjdGlvbigpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnZlbmllbmNlIGZ1bmN0aW9uIHRvIFwic3BsaWNlXCIgbmV3IGxpbmVzIGludG8gdGhlIGFycmF5cyB0aGlzLmxpbmVzLCB0aGlzLmxpbmVEaXJ0eSwgdGhpcy5saW5lVHlwZXMsIGFuZCB0aGUgRE9NIGVsZW1lbnRzXG4gICAqIHVuZGVybmVhdGggdGhlIGVkaXRvciBlbGVtZW50LlxuICAgKiBUaGlzIG1ldGhvZCBpcyBlc3NlbnRpYWxseSBBcnJheS5zcGxpY2UsIG9ubHkgdGhhdCB0aGUgdGhpcmQgcGFyYW1ldGVyIHRha2VzIGFuIHVuLXNwcmVhZCBhcnJheSAoYW5kIHRoZSBmb3J0aCBwYXJhbWV0ZXIpXG4gICAqIGRldGVybWluZXMgd2hldGhlciB0aGUgRE9NIHNob3VsZCBhbHNvIGJlIGFkanVzdGVkLlxuICAgKlxuICAgKiBAcGFyYW0ge2ludH0gc3RhcnRMaW5lIFBvc2l0aW9uIGF0IHdoaWNoIHRvIHN0YXJ0IGNoYW5naW5nIHRoZSBhcnJheSBvZiBsaW5lc1xuICAgKiBAcGFyYW0ge2ludH0gbGluZXNUb0RlbGV0ZSBOdW1iZXIgb2YgbGluZXMgdG8gZGVsZXRlXG4gICAqIEBwYXJhbSB7YXJyYXl9IGxpbmVzVG9JbnNlcnQgQXJyYXkgb2Ygc3RyaW5ncyByZXByZXNlbnRpbmcgdGhlIGxpbmVzIHRvIGJlIGluc2VydGVkXG4gICAqIEBwYXJhbSB7Ym9vbGVhbn0gYWRqdXN0TGluZUVsZW1lbnRzIElmIHRydWUsIHRoZW4gPGRpdj4gZWxlbWVudHMgYXJlIGFsc28gaW5zZXJ0ZWQgaW4gdGhlIERPTSBhdCB0aGUgcmVzcGVjdGl2ZSBwb3NpdGlvblxuICAgKi9cbiAgc3BsaWNlTGluZXMoXG4gICAgc3RhcnRMaW5lLFxuICAgIGxpbmVzVG9EZWxldGUgPSAwLFxuICAgIGxpbmVzVG9JbnNlcnQgPSBbXSxcbiAgICBhZGp1c3RMaW5lRWxlbWVudHMgPSB0cnVlXG4gICkge1xuICAgIGlmIChhZGp1c3RMaW5lRWxlbWVudHMpIHtcbiAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgbGluZXNUb0RlbGV0ZTsgaSsrKSB7XG4gICAgICAgIHRoaXMuZS5yZW1vdmVDaGlsZCh0aGlzLmUuY2hpbGROb2Rlc1tzdGFydExpbmVdKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBsZXQgaW5zZXJ0ZWRCbGFuayA9IFtdO1xuICAgIGxldCBpbnNlcnRlZERpcnR5ID0gW107XG5cbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IGxpbmVzVG9JbnNlcnQubGVuZ3RoOyBpKyspIHtcbiAgICAgIGluc2VydGVkQmxhbmsucHVzaChcIlwiKTtcbiAgICAgIGluc2VydGVkRGlydHkucHVzaCh0cnVlKTtcbiAgICAgIGlmIChhZGp1c3RMaW5lRWxlbWVudHMpIHtcbiAgICAgICAgaWYgKHRoaXMuZS5jaGlsZE5vZGVzW3N0YXJ0TGluZV0pXG4gICAgICAgICAgdGhpcy5lLmluc2VydEJlZm9yZShcbiAgICAgICAgICAgIGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoXCJkaXZcIiksXG4gICAgICAgICAgICB0aGlzLmUuY2hpbGROb2Rlc1tzdGFydExpbmVdXG4gICAgICAgICAgKTtcbiAgICAgICAgZWxzZSB0aGlzLmUuYXBwZW5kQ2hpbGQoZG9jdW1lbnQuY3JlYXRlRWxlbWVudChcImRpdlwiKSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdGhpcy5saW5lcy5zcGxpY2Uoc3RhcnRMaW5lLCBsaW5lc1RvRGVsZXRlLCAuLi5saW5lc1RvSW5zZXJ0KTtcbiAgICB0aGlzLmxpbmVUeXBlcy5zcGxpY2Uoc3RhcnRMaW5lLCBsaW5lc1RvRGVsZXRlLCAuLi5pbnNlcnRlZEJsYW5rKTtcbiAgICB0aGlzLmxpbmVEaXJ0eS5zcGxpY2Uoc3RhcnRMaW5lLCBsaW5lc1RvRGVsZXRlLCAuLi5pbnNlcnRlZERpcnR5KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFdmVudCBoYW5kbGVyIGZvciB0aGUgXCJwYXN0ZVwiIGV2ZW50XG4gICAqL1xuICBoYW5kbGVQYXN0ZShldmVudCkge1xuICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG5cbiAgICAvLyBnZXQgdGV4dCByZXByZXNlbnRhdGlvbiBvZiBjbGlwYm9hcmRcbiAgICBsZXQgdGV4dCA9IChldmVudC5vcmlnaW5hbEV2ZW50IHx8IGV2ZW50KS5jbGlwYm9hcmREYXRhLmdldERhdGEoXG4gICAgICBcInRleHQvcGxhaW5cIlxuICAgICk7XG5cbiAgICAvLyBpbnNlcnQgdGV4dCBtYW51YWxseVxuICAgIHRoaXMucGFzdGUodGV4dCk7XG4gIH1cblxuICAvKipcbiAgICogUGFzdGVzIHRoZSB0ZXh0IGF0IHRoZSBjdXJyZW50IHNlbGVjdGlvbiAob3IgYXQgdGhlIGVuZCwgaWYgbm8gY3VycmVudCBzZWxlY3Rpb24pXG4gICAqIEBwYXJhbSB7c3RyaW5nfSB0ZXh0XG4gICAqL1xuICBwYXN0ZSh0ZXh0LCBhbmNob3IgPSBudWxsLCBmb2N1cyA9IG51bGwpIHtcbiAgICBpZiAoIWFuY2hvcikgYW5jaG9yID0gdGhpcy5nZXRTZWxlY3Rpb24odHJ1ZSk7XG4gICAgaWYgKCFmb2N1cykgZm9jdXMgPSB0aGlzLmdldFNlbGVjdGlvbihmYWxzZSk7XG4gICAgbGV0IGJlZ2lubmluZywgZW5kO1xuICAgIGlmICghZm9jdXMpIHtcbiAgICAgIGZvY3VzID0ge1xuICAgICAgICByb3c6IHRoaXMubGluZXMubGVuZ3RoIC0gMSxcbiAgICAgICAgY29sOiB0aGlzLmxpbmVzW3RoaXMubGluZXMubGVuZ3RoIC0gMV0ubGVuZ3RoLFxuICAgICAgfTsgLy8gSW5zZXJ0IGF0IGVuZFxuICAgIH1cbiAgICBpZiAoIWFuY2hvcikge1xuICAgICAgYW5jaG9yID0gZm9jdXM7XG4gICAgfVxuICAgIGlmIChcbiAgICAgIGFuY2hvci5yb3cgPCBmb2N1cy5yb3cgfHxcbiAgICAgIChhbmNob3Iucm93ID09IGZvY3VzLnJvdyAmJiBhbmNob3IuY29sIDw9IGZvY3VzLmNvbClcbiAgICApIHtcbiAgICAgIGJlZ2lubmluZyA9IGFuY2hvcjtcbiAgICAgIGVuZCA9IGZvY3VzO1xuICAgIH0gZWxzZSB7XG4gICAgICBiZWdpbm5pbmcgPSBmb2N1cztcbiAgICAgIGVuZCA9IGFuY2hvcjtcbiAgICB9XG4gICAgbGV0IGluc2VydGVkTGluZXMgPSB0ZXh0LnNwbGl0KC8oPzpcXHJcXG58XFxyfFxcbikvKTtcbiAgICBsZXQgbGluZUJlZm9yZSA9IHRoaXMubGluZXNbYmVnaW5uaW5nLnJvd10uc3Vic3RyKDAsIGJlZ2lubmluZy5jb2wpO1xuICAgIGxldCBsaW5lRW5kID0gdGhpcy5saW5lc1tlbmQucm93XS5zdWJzdHIoZW5kLmNvbCk7XG4gICAgaW5zZXJ0ZWRMaW5lc1swXSA9IGxpbmVCZWZvcmUuY29uY2F0KGluc2VydGVkTGluZXNbMF0pO1xuICAgIGxldCBlbmRDb2xQb3MgPSBpbnNlcnRlZExpbmVzW2luc2VydGVkTGluZXMubGVuZ3RoIC0gMV0ubGVuZ3RoO1xuICAgIGluc2VydGVkTGluZXNbaW5zZXJ0ZWRMaW5lcy5sZW5ndGggLSAxXSA9XG4gICAgICBpbnNlcnRlZExpbmVzW2luc2VydGVkTGluZXMubGVuZ3RoIC0gMV0uY29uY2F0KGxpbmVFbmQpO1xuICAgIHRoaXMuc3BsaWNlTGluZXMoYmVnaW5uaW5nLnJvdywgMSArIGVuZC5yb3cgLSBiZWdpbm5pbmcucm93LCBpbnNlcnRlZExpbmVzKTtcbiAgICBmb2N1cy5yb3cgPSBiZWdpbm5pbmcucm93ICsgaW5zZXJ0ZWRMaW5lcy5sZW5ndGggLSAxO1xuICAgIGZvY3VzLmNvbCA9IGVuZENvbFBvcztcbiAgICB0aGlzLnVwZGF0ZUZvcm1hdHRpbmcoKTtcbiAgICB0aGlzLnNldFNlbGVjdGlvbihmb2N1cyk7XG4gICAgdGhpcy5maXJlQ2hhbmdlKCk7XG4gIH1cblxuICAvKipcbiAgICogQ29tcHV0ZXMgdGhlIChsb3dlc3QgaW4gdGhlIERPTSB0cmVlKSBjb21tb24gYW5jZXN0b3Igb2YgdHdvIERPTSBub2Rlcy5cbiAgICogQHBhcmFtIHtOb2RlfSBub2RlMSB0aGUgZmlyc3Qgbm9kZVxuICAgKiBAcGFyYW0ge05vZGV9IG5vZGUyIHRoZSBzZWNvbmQgbm9kZVxuICAgKiBAcmV0dXJucyB7Tm9kZX0gVGhlIGNvbW1lbiBhbmNlc3RvciBub2RlLCBvciBudWxsIGlmIHRoZXJlIGlzIG5vIGNvbW1vbiBhbmNlc3RvclxuICAgKi9cbiAgY29tcHV0ZUNvbW1vbkFuY2VzdG9yKG5vZGUxLCBub2RlMikge1xuICAgIGlmICghbm9kZTEgfHwgIW5vZGUyKSByZXR1cm4gbnVsbDtcbiAgICBpZiAobm9kZTEgPT0gbm9kZTIpIHJldHVybiBub2RlMTtcbiAgICBjb25zdCBhbmNlc3RyeSA9IChub2RlKSA9PiB7XG4gICAgICBsZXQgYW5jZXN0cnkgPSBbXTtcbiAgICAgIHdoaWxlIChub2RlKSB7XG4gICAgICAgIGFuY2VzdHJ5LnVuc2hpZnQobm9kZSk7XG4gICAgICAgIG5vZGUgPSBub2RlLnBhcmVudE5vZGU7XG4gICAgICB9XG4gICAgICByZXR1cm4gYW5jZXN0cnk7XG4gICAgfTtcblxuICAgIGNvbnN0IGFuY2VzdHJ5MSA9IGFuY2VzdHJ5KG5vZGUxKTtcbiAgICBjb25zdCBhbmNlc3RyeTIgPSBhbmNlc3RyeShub2RlMik7XG5cbiAgICBpZiAoYW5jZXN0cnkxWzBdICE9IGFuY2VzdHJ5MlswXSkgcmV0dXJuIG51bGw7XG4gICAgbGV0IGk7XG4gICAgZm9yIChpID0gMDsgYW5jZXN0cnkxW2ldID09IGFuY2VzdHJ5MltpXTsgaSsrKTtcbiAgICByZXR1cm4gYW5jZXN0cnkxW2kgLSAxXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaW5kcyB0aGUgKGxvd2VzdCBpbiB0aGUgRE9NIHRyZWUpIGVuY2xvc2luZyBET00gbm9kZSB3aXRoIGEgZ2l2ZW4gY2xhc3MuXG4gICAqIEBwYXJhbSB7b2JqZWN0fSBmb2N1cyBUaGUgZm9jdXMgc2VsZWN0aW9uIG9iamVjdFxuICAgKiBAcGFyYW0ge29iamVjdH0gYW5jaG9yIFRoZSBhbmNob3Igc2VsZWN0aW9uIG9iamVjdFxuICAgKiBAcGFyYW0ge3N0cmluZ30gY2xhc3NOYW1lIFRoZSBjbGFzcyBuYW1lIHRvIGZpbmRcbiAgICogQHJldHVybnMge05vZGV9IFRoZSBlbmNsb3NpbmcgRE9NIG5vZGUgd2l0aCB0aGUgcmVzcGVjdGl2ZSBjbGFzcyAoaW5zaWRlIHRoZSBlZGl0b3IpLCBpZiB0aGVyZSBpcyBvbmU7IG51bGwgb3RoZXJ3aXNlLlxuICAgKi9cbiAgY29tcHV0ZUVuY2xvc2luZ01hcmt1cE5vZGUoZm9jdXMsIGFuY2hvciwgY2xhc3NOYW1lKSB7XG4gICAgbGV0IG5vZGUgPSBudWxsO1xuICAgIGlmICghZm9jdXMpIHJldHVybiBudWxsO1xuICAgIGlmICghYW5jaG9yKSB7XG4gICAgICBub2RlID0gZm9jdXMubm9kZTtcbiAgICB9IGVsc2Uge1xuICAgICAgaWYgKGZvY3VzLnJvdyAhPSBhbmNob3Iucm93KSByZXR1cm4gbnVsbDtcbiAgICAgIG5vZGUgPSB0aGlzLmNvbXB1dGVDb21tb25BbmNlc3Rvcihmb2N1cy5ub2RlLCBhbmNob3Iubm9kZSk7XG4gICAgfVxuICAgIGlmICghbm9kZSkgcmV0dXJuIG51bGw7XG4gICAgd2hpbGUgKG5vZGUgIT0gdGhpcy5lKSB7XG4gICAgICBpZiAobm9kZS5jbGFzc05hbWUgJiYgbm9kZS5jbGFzc05hbWUuaW5jbHVkZXMoY2xhc3NOYW1lKSkgcmV0dXJuIG5vZGU7XG4gICAgICBub2RlID0gbm9kZS5wYXJlbnROb2RlO1xuICAgIH1cbiAgICAvLyBBc2NlbmRlZCBhbGwgdGhlIHdheSB0byB0aGUgZWRpdG9yIGVsZW1lbnRcbiAgICByZXR1cm4gbnVsbDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBzdGF0ZSAodHJ1ZSAvIGZhbHNlKSBvZiBhbGwgY29tbWFuZHMuXG4gICAqIEBwYXJhbSBmb2N1cyBGb2N1cyBvZiB0aGUgc2VsZWN0aW9uLiBJZiBub3QgZ2l2ZW4sIGFzc3VtZXMgdGhlIGN1cnJlbnQgZm9jdXMuXG4gICAqIEBwYXJhbSBhbmNob3IgQW5jaG9yIG9mIHRoZSBzZWxlY3Rpb24uIElmIG5vdCBnaXZlbiwgYXNzdW1lcyB0aGUgY3VycmVudCBhbmNob3IuXG4gICAqL1xuICBnZXRDb21tYW5kU3RhdGUoZm9jdXMgPSBudWxsLCBhbmNob3IgPSBudWxsKSB7XG4gICAgbGV0IGNvbW1hbmRTdGF0ZSA9IHt9O1xuICAgIGlmICghZm9jdXMpIGZvY3VzID0gdGhpcy5nZXRTZWxlY3Rpb24oZmFsc2UpO1xuICAgIGlmICghYW5jaG9yKSBhbmNob3IgPSB0aGlzLmdldFNlbGVjdGlvbih0cnVlKTtcbiAgICBpZiAoIWZvY3VzKSB7XG4gICAgICBmb3IgKGxldCBjbWQgaW4gY29tbWFuZHMpIHtcbiAgICAgICAgY29tbWFuZFN0YXRlW2NtZF0gPSBudWxsO1xuICAgICAgfVxuICAgICAgcmV0dXJuIGNvbW1hbmRTdGF0ZTtcbiAgICB9XG4gICAgaWYgKCFhbmNob3IpIGFuY2hvciA9IGZvY3VzO1xuXG4gICAgbGV0IHN0YXJ0LCBlbmQ7XG4gICAgaWYgKFxuICAgICAgYW5jaG9yLnJvdyA8IGZvY3VzLnJvdyB8fFxuICAgICAgKGFuY2hvci5yb3cgPT0gZm9jdXMucm93ICYmIGFuY2hvci5jb2wgPCBmb2N1cy5jb2wpXG4gICAgKSB7XG4gICAgICBzdGFydCA9IGFuY2hvcjtcbiAgICAgIGVuZCA9IGZvY3VzO1xuICAgIH0gZWxzZSB7XG4gICAgICBzdGFydCA9IGZvY3VzO1xuICAgICAgZW5kID0gYW5jaG9yO1xuICAgIH1cbiAgICBpZiAoZW5kLnJvdyA+IHN0YXJ0LnJvdyAmJiBlbmQuY29sID09IDApIHtcbiAgICAgIGVuZC5yb3ctLTtcbiAgICAgIGVuZC5jb2wgPSB0aGlzLmxpbmVzW2VuZC5yb3ddLmxlbmd0aDsgLy8gU2VsZWN0aW9uIHRvIGJlZ2lubmluZyBvZiBuZXh0IGxpbmUgaXMgc2FpZCB0byBlbmQgYXQgdGhlIGJlZ2lubmluZyBvZiB0aGUgbGFzdCBsaW5lXG4gICAgfVxuXG4gICAgZm9yIChsZXQgY21kIGluIGNvbW1hbmRzKSB7XG4gICAgICBpZiAoY29tbWFuZHNbY21kXS50eXBlID09IFwiaW5saW5lXCIpIHtcbiAgICAgICAgaWYgKFxuICAgICAgICAgICFmb2N1cyB8fFxuICAgICAgICAgIGZvY3VzLnJvdyAhPSBhbmNob3Iucm93IHx8XG4gICAgICAgICAgIXRoaXMuaXNJbmxpbmVGb3JtYXR0aW5nQWxsb3dlZChmb2N1cywgYW5jaG9yKVxuICAgICAgICApIHtcbiAgICAgICAgICBjb21tYW5kU3RhdGVbY21kXSA9IG51bGw7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgLy8gVGhlIGNvbW1hbmQgc3RhdGUgaXMgdHJ1ZSBpZiB0aGVyZSBpcyBhIHJlc3BlY3RpdmUgZW5jbG9zaW5nIG1hcmt1cCBub2RlIChlLmcuLCB0aGUgc2VsZWN0aW9uIGlzIGVuY2xvc2VkIGluIGEgPGI+Li48L2I+KSAuLi5cbiAgICAgICAgICBjb21tYW5kU3RhdGVbY21kXSA9XG4gICAgICAgICAgICAhIXRoaXMuY29tcHV0ZUVuY2xvc2luZ01hcmt1cE5vZGUoXG4gICAgICAgICAgICAgIGZvY3VzLFxuICAgICAgICAgICAgICBhbmNob3IsXG4gICAgICAgICAgICAgIGNvbW1hbmRzW2NtZF0uY2xhc3NOYW1lXG4gICAgICAgICAgICApIHx8XG4gICAgICAgICAgICAvLyAuLi4gb3IgaWYgaXQncyBhbiBlbXB0eSBzdHJpbmcgcHJlY2VkZWQgYnkgYW5kIGZvbGxvd2VkIGJ5IGZvcm1hdHRpbmcgbWFya2VycywgZS5nLiAqKnwqKiB3aGVyZSB8IGlzIHRoZSBjdXJzb3JcbiAgICAgICAgICAgIChmb2N1cy5jb2wgPT0gYW5jaG9yLmNvbCAmJlxuICAgICAgICAgICAgICAhIXRoaXMubGluZXNbZm9jdXMucm93XVxuICAgICAgICAgICAgICAgIC5zdWJzdHIoMCwgZm9jdXMuY29sKVxuICAgICAgICAgICAgICAgIC5tYXRjaChjb21tYW5kc1tjbWRdLnVuc2V0LnByZVBhdHRlcm4pICYmXG4gICAgICAgICAgICAgICEhdGhpcy5saW5lc1tmb2N1cy5yb3ddXG4gICAgICAgICAgICAgICAgLnN1YnN0cihmb2N1cy5jb2wpXG4gICAgICAgICAgICAgICAgLm1hdGNoKGNvbW1hbmRzW2NtZF0udW5zZXQucG9zdFBhdHRlcm4pKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgaWYgKGNvbW1hbmRzW2NtZF0udHlwZSA9PSBcImxpbmVcIikge1xuICAgICAgICBpZiAoIWZvY3VzKSB7XG4gICAgICAgICAgY29tbWFuZFN0YXRlW2NtZF0gPSBudWxsO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIGxldCBzdGF0ZSA9IHRoaXMubGluZVR5cGVzW3N0YXJ0LnJvd10gPT0gY29tbWFuZHNbY21kXS5jbGFzc05hbWU7XG5cbiAgICAgICAgICBmb3IgKGxldCBsaW5lID0gc3RhcnQucm93OyBsaW5lIDw9IGVuZC5yb3c7IGxpbmUrKykge1xuICAgICAgICAgICAgaWYgKCh0aGlzLmxpbmVUeXBlc1tsaW5lXSA9PSBjb21tYW5kc1tjbWRdLmNsYXNzTmFtZSkgIT0gc3RhdGUpIHtcbiAgICAgICAgICAgICAgc3RhdGUgPSBudWxsO1xuICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgICAgY29tbWFuZFN0YXRlW2NtZF0gPSBzdGF0ZTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gY29tbWFuZFN0YXRlO1xuICB9XG5cbiAgLyoqXG4gICAqIFNldHMgYSBjb21tYW5kIHN0YXRlXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBjb21tYW5kXG4gICAqIEBwYXJhbSB7Ym9vbGVhbn0gc3RhdGVcbiAgICovXG4gIHNldENvbW1hbmRTdGF0ZShjb21tYW5kLCBzdGF0ZSkge1xuICAgIGlmIChjb21tYW5kc1tjb21tYW5kXS50eXBlID09IFwiaW5saW5lXCIpIHtcbiAgICAgIGxldCBhbmNob3IgPSB0aGlzLmdldFNlbGVjdGlvbih0cnVlKTtcbiAgICAgIGxldCBmb2N1cyA9IHRoaXMuZ2V0U2VsZWN0aW9uKGZhbHNlKTtcbiAgICAgIGlmICghYW5jaG9yKSBhbmNob3IgPSBmb2N1cztcbiAgICAgIGlmICghYW5jaG9yKSByZXR1cm47XG4gICAgICBpZiAoYW5jaG9yLnJvdyAhPSBmb2N1cy5yb3cpIHJldHVybjtcbiAgICAgIGlmICghdGhpcy5pc0lubGluZUZvcm1hdHRpbmdBbGxvd2VkKGZvY3VzLCBhbmNob3IpKSByZXR1cm47XG4gICAgICBsZXQgbWFya3VwTm9kZSA9IHRoaXMuY29tcHV0ZUVuY2xvc2luZ01hcmt1cE5vZGUoXG4gICAgICAgIGZvY3VzLFxuICAgICAgICBhbmNob3IsXG4gICAgICAgIGNvbW1hbmRzW2NvbW1hbmRdLmNsYXNzTmFtZVxuICAgICAgKTtcbiAgICAgIHRoaXMuY2xlYXJEaXJ0eUZsYWcoKTtcblxuICAgICAgLy8gRmlyc3QgY2FzZTogVGhlcmUncyBhbiBlbmNsb3NpbmcgbWFya3VwIG5vZGUsIHJlbW92ZSB0aGUgbWFya2VycyBhcm91bmQgdGhhdCBtYXJrdXAgbm9kZVxuICAgICAgaWYgKG1hcmt1cE5vZGUpIHtcbiAgICAgICAgdGhpcy5saW5lRGlydHlbZm9jdXMucm93XSA9IHRydWU7XG4gICAgICAgIGNvbnN0IHN0YXJ0Q29sID0gdGhpcy5jb21wdXRlQ29sdW1uKG1hcmt1cE5vZGUsIDApO1xuICAgICAgICBjb25zdCBsZW4gPSBtYXJrdXBOb2RlLnRleHRDb250ZW50Lmxlbmd0aDtcbiAgICAgICAgY29uc3QgbGVmdCA9IHRoaXMubGluZXNbZm9jdXMucm93XVxuICAgICAgICAgIC5zdWJzdHIoMCwgc3RhcnRDb2wpXG4gICAgICAgICAgLnJlcGxhY2UoY29tbWFuZHNbY29tbWFuZF0udW5zZXQucHJlUGF0dGVybiwgXCJcIik7XG4gICAgICAgIGNvbnN0IG1pZCA9IHRoaXMubGluZXNbZm9jdXMucm93XS5zdWJzdHIoc3RhcnRDb2wsIGxlbik7XG4gICAgICAgIGNvbnN0IHJpZ2h0ID0gdGhpcy5saW5lc1tmb2N1cy5yb3ddXG4gICAgICAgICAgLnN1YnN0cihzdGFydENvbCArIGxlbilcbiAgICAgICAgICAucmVwbGFjZShjb21tYW5kc1tjb21tYW5kXS51bnNldC5wb3N0UGF0dGVybiwgXCJcIik7XG4gICAgICAgIHRoaXMubGluZXNbZm9jdXMucm93XSA9IGxlZnQuY29uY2F0KG1pZCwgcmlnaHQpO1xuICAgICAgICBhbmNob3IuY29sID0gbGVmdC5sZW5ndGg7XG4gICAgICAgIGZvY3VzLmNvbCA9IGFuY2hvci5jb2wgKyBsZW47XG4gICAgICAgIHRoaXMudXBkYXRlRm9ybWF0dGluZygpO1xuICAgICAgICB0aGlzLnNldFNlbGVjdGlvbihmb2N1cywgYW5jaG9yKTtcbiAgICAgICAgdGhpcy5maXJlQ2hhbmdlKCk7XG5cbiAgICAgICAgLy8gU2Vjb25kIGNhc2U6IEVtcHR5IHNlbGVjdGlvbiB3aXRoIHN1cnJvdW5kaW5nIGZvcm1hdHRpbmcgbWFya2VycywgcmVtb3ZlIHRob3NlXG4gICAgICB9IGVsc2UgaWYgKFxuICAgICAgICBmb2N1cy5jb2wgPT0gYW5jaG9yLmNvbCAmJlxuICAgICAgICAhIXRoaXMubGluZXNbZm9jdXMucm93XVxuICAgICAgICAgIC5zdWJzdHIoMCwgZm9jdXMuY29sKVxuICAgICAgICAgIC5tYXRjaChjb21tYW5kc1tjb21tYW5kXS51bnNldC5wcmVQYXR0ZXJuKSAmJlxuICAgICAgICAhIXRoaXMubGluZXNbZm9jdXMucm93XVxuICAgICAgICAgIC5zdWJzdHIoZm9jdXMuY29sKVxuICAgICAgICAgIC5tYXRjaChjb21tYW5kc1tjb21tYW5kXS51bnNldC5wb3N0UGF0dGVybilcbiAgICAgICkge1xuICAgICAgICB0aGlzLmxpbmVEaXJ0eVtmb2N1cy5yb3ddID0gdHJ1ZTtcbiAgICAgICAgY29uc3QgbGVmdCA9IHRoaXMubGluZXNbZm9jdXMucm93XVxuICAgICAgICAgIC5zdWJzdHIoMCwgZm9jdXMuY29sKVxuICAgICAgICAgIC5yZXBsYWNlKGNvbW1hbmRzW2NvbW1hbmRdLnVuc2V0LnByZVBhdHRlcm4sIFwiXCIpO1xuICAgICAgICBjb25zdCByaWdodCA9IHRoaXMubGluZXNbZm9jdXMucm93XVxuICAgICAgICAgIC5zdWJzdHIoZm9jdXMuY29sKVxuICAgICAgICAgIC5yZXBsYWNlKGNvbW1hbmRzW2NvbW1hbmRdLnVuc2V0LnBvc3RQYXR0ZXJuLCBcIlwiKTtcbiAgICAgICAgdGhpcy5saW5lc1tmb2N1cy5yb3ddID0gbGVmdC5jb25jYXQocmlnaHQpO1xuICAgICAgICBmb2N1cy5jb2wgPSBhbmNob3IuY29sID0gbGVmdC5sZW5ndGg7XG4gICAgICAgIHRoaXMudXBkYXRlRm9ybWF0dGluZygpO1xuICAgICAgICB0aGlzLnNldFNlbGVjdGlvbihmb2N1cywgYW5jaG9yKTtcbiAgICAgICAgdGhpcy5maXJlQ2hhbmdlKCk7XG5cbiAgICAgICAgLy8gTm90IGN1cnJlbnRseSBmb3JtYXR0ZWQsIGluc2VydCBmb3JtYXR0aW5nIG1hcmtlcnNcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIFRyaW0gYW55IHNwYWNlcyBmcm9tIHRoZSBzZWxlY3Rpb25cbiAgICAgICAgbGV0IHsgc3RhcnRDb2wsIGVuZENvbCB9ID1cbiAgICAgICAgICBmb2N1cy5jb2wgPCBhbmNob3IuY29sXG4gICAgICAgICAgICA/IHsgc3RhcnRDb2w6IGZvY3VzLmNvbCwgZW5kQ29sOiBhbmNob3IuY29sIH1cbiAgICAgICAgICAgIDogeyBzdGFydENvbDogYW5jaG9yLmNvbCwgZW5kQ29sOiBmb2N1cy5jb2wgfTtcblxuICAgICAgICBsZXQgbWF0Y2ggPSB0aGlzLmxpbmVzW2ZvY3VzLnJvd11cbiAgICAgICAgICAuc3Vic3RyKHN0YXJ0Q29sLCBlbmRDb2wgLSBzdGFydENvbClcbiAgICAgICAgICAubWF0Y2goL14oPzxsZWFkaW5nPlxccyopLipcXFMoPzx0cmFpbGluZz5cXHMqKSQvKTtcbiAgICAgICAgaWYgKG1hdGNoKSB7XG4gICAgICAgICAgc3RhcnRDb2wgKz0gbWF0Y2guZ3JvdXBzLmxlYWRpbmcubGVuZ3RoO1xuICAgICAgICAgIGVuZENvbCAtPSBtYXRjaC5ncm91cHMudHJhaWxpbmcubGVuZ3RoO1xuICAgICAgICB9XG5cbiAgICAgICAgZm9jdXMuY29sID0gc3RhcnRDb2w7XG4gICAgICAgIGFuY2hvci5jb2wgPSBlbmRDb2w7XG5cbiAgICAgICAgLy8gSnVzdCBpbnNlcnQgbWFya3VwIGJlZm9yZSBhbmQgYWZ0ZXIgYW5kIGhvcGUgZm9yIHRoZSBiZXN0LlxuICAgICAgICB0aGlzLndyYXBTZWxlY3Rpb24oXG4gICAgICAgICAgY29tbWFuZHNbY29tbWFuZF0uc2V0LnByZSxcbiAgICAgICAgICBjb21tYW5kc1tjb21tYW5kXS5zZXQucG9zdCxcbiAgICAgICAgICBmb2N1cyxcbiAgICAgICAgICBhbmNob3JcbiAgICAgICAgKTtcbiAgICAgICAgdGhpcy5maXJlQ2hhbmdlKCk7XG4gICAgICAgIC8vIFRPRE8gY2xlYW4gdGhpcyB1cCBzbyB0aGF0IG1hcmt1cCByZW1haW5zIHByb3Blcmx5IG5lc3RlZFxuICAgICAgfVxuICAgIH0gZWxzZSBpZiAoY29tbWFuZHNbY29tbWFuZF0udHlwZSA9PSBcImxpbmVcIikge1xuICAgICAgbGV0IGFuY2hvciA9IHRoaXMuZ2V0U2VsZWN0aW9uKHRydWUpO1xuICAgICAgbGV0IGZvY3VzID0gdGhpcy5nZXRTZWxlY3Rpb24oZmFsc2UpO1xuICAgICAgaWYgKCFhbmNob3IpIGFuY2hvciA9IGZvY3VzO1xuICAgICAgaWYgKCFmb2N1cykgcmV0dXJuO1xuICAgICAgdGhpcy5jbGVhckRpcnR5RmxhZygpO1xuICAgICAgbGV0IHN0YXJ0ID0gYW5jaG9yLnJvdyA+IGZvY3VzLnJvdyA/IGZvY3VzIDogYW5jaG9yO1xuICAgICAgbGV0IGVuZCA9IGFuY2hvci5yb3cgPiBmb2N1cy5yb3cgPyBhbmNob3IgOiBmb2N1cztcbiAgICAgIGlmIChlbmQucm93ID4gc3RhcnQucm93ICYmIGVuZC5jb2wgPT0gMCkge1xuICAgICAgICBlbmQucm93LS07XG4gICAgICB9XG5cbiAgICAgIGZvciAobGV0IGxpbmUgPSBzdGFydC5yb3c7IGxpbmUgPD0gZW5kLnJvdzsgbGluZSsrKSB7XG4gICAgICAgIGlmIChzdGF0ZSAmJiB0aGlzLmxpbmVUeXBlc1tsaW5lXSAhPSBjb21tYW5kc1tjb21tYW5kXS5jbGFzc05hbWUpIHtcbiAgICAgICAgICB0aGlzLmxpbmVzW2xpbmVdID0gdGhpcy5saW5lc1tsaW5lXS5yZXBsYWNlKFxuICAgICAgICAgICAgY29tbWFuZHNbY29tbWFuZF0uc2V0LnBhdHRlcm4sXG4gICAgICAgICAgICBjb21tYW5kc1tjb21tYW5kXS5zZXQucmVwbGFjZW1lbnQucmVwbGFjZShcbiAgICAgICAgICAgICAgXCIkI1wiLFxuICAgICAgICAgICAgICBsaW5lIC0gc3RhcnQucm93ICsgMVxuICAgICAgICAgICAgKVxuICAgICAgICAgICk7XG4gICAgICAgICAgdGhpcy5saW5lRGlydHlbbGluZV0gPSB0cnVlO1xuICAgICAgICB9XG4gICAgICAgIGlmICghc3RhdGUgJiYgdGhpcy5saW5lVHlwZXNbbGluZV0gPT0gY29tbWFuZHNbY29tbWFuZF0uY2xhc3NOYW1lKSB7XG4gICAgICAgICAgdGhpcy5saW5lc1tsaW5lXSA9IHRoaXMubGluZXNbbGluZV0ucmVwbGFjZShcbiAgICAgICAgICAgIGNvbW1hbmRzW2NvbW1hbmRdLnVuc2V0LnBhdHRlcm4sXG4gICAgICAgICAgICBjb21tYW5kc1tjb21tYW5kXS51bnNldC5yZXBsYWNlbWVudFxuICAgICAgICAgICk7XG4gICAgICAgICAgdGhpcy5saW5lRGlydHlbbGluZV0gPSB0cnVlO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICB0aGlzLnVwZGF0ZUZvcm1hdHRpbmcoKTtcbiAgICAgIHRoaXMuc2V0U2VsZWN0aW9uKFxuICAgICAgICB7IHJvdzogZW5kLnJvdywgY29sOiB0aGlzLmxpbmVzW2VuZC5yb3ddLmxlbmd0aCB9LFxuICAgICAgICB7IHJvdzogc3RhcnQucm93LCBjb2w6IDAgfVxuICAgICAgKTtcbiAgICAgIHRoaXMuZmlyZUNoYW5nZSgpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHdoZXRoZXIgb3Igbm90IGlubGluZSBmb3JtYXR0aW5nIGlzIGFsbG93ZWQgYXQgdGhlIGN1cnJlbnQgZm9jdXNcbiAgICogQHBhcmFtIHtvYmplY3R9IGZvY3VzIFRoZSBjdXJyZW50IGZvY3VzXG4gICAqL1xuICBpc0lubGluZUZvcm1hdHRpbmdBbGxvd2VkKCkge1xuICAgIC8vIFRPRE8gUmVtb3ZlIHBhcmFtZXRlcnMgZnJvbSBhbGwgY2FsbHNcbiAgICBjb25zdCBzZWwgPSB3aW5kb3cuZ2V0U2VsZWN0aW9uKCk7XG4gICAgaWYgKCFzZWwgfHwgIXNlbC5mb2N1c05vZGUgfHwgIXNlbC5hbmNob3JOb2RlKSByZXR1cm4gZmFsc2U7XG5cbiAgICAvLyBDaGVjayBpZiB3ZSBjYW4gZmluZCBhIGNvbW1vbiBhbmNlc3RvciB3aXRoIHRoZSBjbGFzcyBgVE1JbmxpbmVGb3JtYXR0ZWRgXG5cbiAgICAvLyBTcGVjaWFsIGNhc2U6IEVtcHR5IHNlbGVjdGlvbiByaWdodCBiZWZvcmUgYFRNSW5saW5lRm9ybWF0dGVkYFxuICAgIGlmIChcbiAgICAgIHNlbC5pc0NvbGxhcHNlZCAmJlxuICAgICAgc2VsLmZvY3VzTm9kZS5ub2RlVHlwZSA9PSAzICYmXG4gICAgICBzZWwuZm9jdXNPZmZzZXQgPT0gc2VsLmZvY3VzTm9kZS5ub2RlVmFsdWUubGVuZ3RoXG4gICAgKSB7XG4gICAgICBsZXQgbm9kZTtcbiAgICAgIGZvciAoXG4gICAgICAgIG5vZGUgPSBzZWwuZm9jdXNOb2RlO1xuICAgICAgICBub2RlICYmIG5vZGUubmV4dFNpYmxpbmcgPT0gbnVsbDtcbiAgICAgICAgbm9kZSA9IG5vZGUucGFyZW50Tm9kZVxuICAgICAgKTtcbiAgICAgIGlmIChcbiAgICAgICAgbm9kZSAmJlxuICAgICAgICBub2RlLm5leHRTaWJsaW5nLmNsYXNzTmFtZSAmJlxuICAgICAgICBub2RlLm5leHRTaWJsaW5nLmNsYXNzTmFtZS5pbmNsdWRlcyhcIlRNSW5saW5lRm9ybWF0dGVkXCIpXG4gICAgICApXG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIC8vIExvb2sgZm9yIGEgY29tbW9uIGFuY2VzdG9yXG4gICAgbGV0IGFuY2VzdG9yID0gdGhpcy5jb21wdXRlQ29tbW9uQW5jZXN0b3Ioc2VsLmZvY3VzTm9kZSwgc2VsLmFuY2hvck5vZGUpO1xuICAgIGlmICghYW5jZXN0b3IpIHJldHVybiBmYWxzZTtcblxuICAgIC8vIENoZWNrIGlmIHRoZXJlJ3MgYW4gYW5jZXN0b3Igb2YgY2xhc3MgJ1RNSW5saW5lRm9ybWF0dGVkJyBvciAnVE1CbGFua0xpbmUnXG4gICAgd2hpbGUgKGFuY2VzdG9yICYmIGFuY2VzdG9yICE9IHRoaXMuZSkge1xuICAgICAgaWYgKFxuICAgICAgICBhbmNlc3Rvci5jbGFzc05hbWUgJiZcbiAgICAgICAgdHlwZW9mIGFuY2VzdG9yLmNsYXNzTmFtZS5pbmNsdWRlcyA9PSBcImZ1bmN0aW9uXCIgJiZcbiAgICAgICAgKGFuY2VzdG9yLmNsYXNzTmFtZS5pbmNsdWRlcyhcIlRNSW5saW5lRm9ybWF0dGVkXCIpIHx8XG4gICAgICAgICAgYW5jZXN0b3IuY2xhc3NOYW1lLmluY2x1ZGVzKFwiVE1CbGFua0xpbmVcIikpXG4gICAgICApXG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgYW5jZXN0b3IgPSBhbmNlc3Rvci5wYXJlbnROb2RlO1xuICAgIH1cblxuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBXcmFwcyB0aGUgY3VycmVudCBzZWxlY3Rpb24gaW4gdGhlIHN0cmluZ3MgcHJlIGFuZCBwb3N0LiBJZiB0aGUgc2VsZWN0aW9uIGlzIG5vdCBvbiBvbmUgbGluZSwgcmV0dXJucy5cbiAgICogQHBhcmFtIHtzdHJpbmd9IHByZSBUaGUgc3RyaW5nIHRvIGluc2VydCBiZWZvcmUgdGhlIHNlbGVjdGlvbi5cbiAgICogQHBhcmFtIHtzdHJpbmd9IHBvc3QgVGhlIHN0cmluZyB0byBpbnNlcnQgYWZ0ZXIgdGhlIHNlbGVjdGlvbi5cbiAgICogQHBhcmFtIHtvYmplY3R9IGZvY3VzIFRoZSBjdXJyZW50IHNlbGVjdGlvbiBmb2N1cy4gSWYgbnVsbCwgc2VsZWN0aW9uIHdpbGwgYmUgY29tcHV0ZWQuXG4gICAqIEBwYXJhbSB7b2JqZWN0fSBhbmNob3IgVGhlIGN1cnJlbnQgc2VsZWN0aW9uIGZvY3VzLiBJZiBudWxsLCBzZWxlY3Rpb24gd2lsbCBiZSBjb21wdXRlZC5cbiAgICovXG4gIHdyYXBTZWxlY3Rpb24ocHJlLCBwb3N0LCBmb2N1cyA9IG51bGwsIGFuY2hvciA9IG51bGwpIHtcbiAgICBpZiAoIWZvY3VzKSBmb2N1cyA9IHRoaXMuZ2V0U2VsZWN0aW9uKGZhbHNlKTtcbiAgICBpZiAoIWFuY2hvcikgYW5jaG9yID0gdGhpcy5nZXRTZWxlY3Rpb24odHJ1ZSk7XG4gICAgaWYgKCFmb2N1cyB8fCAhYW5jaG9yIHx8IGZvY3VzLnJvdyAhPSBhbmNob3Iucm93KSByZXR1cm47XG4gICAgdGhpcy5saW5lRGlydHlbZm9jdXMucm93XSA9IHRydWU7XG5cbiAgICBjb25zdCBzdGFydENvbCA9IGZvY3VzLmNvbCA8IGFuY2hvci5jb2wgPyBmb2N1cy5jb2wgOiBhbmNob3IuY29sO1xuICAgIGNvbnN0IGVuZENvbCA9IGZvY3VzLmNvbCA8IGFuY2hvci5jb2wgPyBhbmNob3IuY29sIDogZm9jdXMuY29sO1xuICAgIGNvbnN0IGxlZnQgPSB0aGlzLmxpbmVzW2ZvY3VzLnJvd10uc3Vic3RyKDAsIHN0YXJ0Q29sKS5jb25jYXQocHJlKTtcbiAgICBjb25zdCBtaWQgPVxuICAgICAgZW5kQ29sID09IHN0YXJ0Q29sXG4gICAgICAgID8gXCJcIlxuICAgICAgICA6IHRoaXMubGluZXNbZm9jdXMucm93XS5zdWJzdHIoc3RhcnRDb2wsIGVuZENvbCAtIHN0YXJ0Q29sKTtcbiAgICBjb25zdCByaWdodCA9IHBvc3QuY29uY2F0KHRoaXMubGluZXNbZm9jdXMucm93XS5zdWJzdHIoZW5kQ29sKSk7XG4gICAgdGhpcy5saW5lc1tmb2N1cy5yb3ddID0gbGVmdC5jb25jYXQobWlkLCByaWdodCk7XG4gICAgYW5jaG9yLmNvbCA9IGxlZnQubGVuZ3RoO1xuICAgIGZvY3VzLmNvbCA9IGFuY2hvci5jb2wgKyBtaWQubGVuZ3RoO1xuXG4gICAgdGhpcy51cGRhdGVGb3JtYXR0aW5nKCk7XG4gICAgdGhpcy5zZXRTZWxlY3Rpb24oZm9jdXMsIGFuY2hvcik7XG4gIH1cblxuICAvKipcbiAgICogVG9nZ2xlcyB0aGUgY29tbWFuZCBzdGF0ZSBmb3IgYSBjb21tYW5kICh0cnVlIDwtPiBmYWxzZSlcbiAgICogQHBhcmFtIHtzdHJpbmd9IGNvbW1hbmQgVGhlIGVkaXRvciBjb21tYW5kXG4gICAqL1xuICB0b2dnbGVDb21tYW5kU3RhdGUoY29tbWFuZCkge1xuICAgIGlmICghdGhpcy5sYXN0Q29tbWFuZFN0YXRlKSB0aGlzLmxhc3RDb21tYW5kU3RhdGUgPSB0aGlzLmdldENvbW1hbmRTdGF0ZSgpO1xuICAgIHRoaXMuc2V0Q29tbWFuZFN0YXRlKGNvbW1hbmQsICF0aGlzLmxhc3RDb21tYW5kU3RhdGVbY29tbWFuZF0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpcmVzIGEgY2hhbmdlIGV2ZW50LiBVcGRhdGVzIHRoZSBsaW5rZWQgdGV4dGFyZWEgYW5kIG5vdGlmaWVzIGFueSBldmVudCBsaXN0ZW5lcnMuXG4gICAqL1xuICBmaXJlQ2hhbmdlKCkge1xuICAgIGlmICghdGhpcy50ZXh0YXJlYSAmJiAhdGhpcy5saXN0ZW5lcnMuY2hhbmdlLmxlbmd0aCkgcmV0dXJuO1xuICAgIGNvbnN0IGNvbnRlbnQgPSB0aGlzLmdldENvbnRlbnQoKTtcbiAgICBpZiAodGhpcy50ZXh0YXJlYSkgdGhpcy50ZXh0YXJlYS52YWx1ZSA9IGNvbnRlbnQ7XG4gICAgZm9yIChsZXQgbGlzdGVuZXIgb2YgdGhpcy5saXN0ZW5lcnMuY2hhbmdlKSB7XG4gICAgICBsaXN0ZW5lcih7XG4gICAgICAgIGNvbnRlbnQ6IGNvbnRlbnQsXG4gICAgICAgIGxpbmVzRGlydHk6IHRoaXMubGluZXNEaXJ0eSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBGaXJlcyBhIFwic2VsZWN0aW9uIGNoYW5nZWRcIiBldmVudC5cbiAgICovXG4gIGZpcmVTZWxlY3Rpb24oKSB7XG4gICAgaWYgKHRoaXMubGlzdGVuZXJzLnNlbGVjdGlvbiAmJiB0aGlzLmxpc3RlbmVycy5zZWxlY3Rpb24ubGVuZ3RoKSB7XG4gICAgICBsZXQgZm9jdXMgPSB0aGlzLmdldFNlbGVjdGlvbihmYWxzZSk7XG4gICAgICBsZXQgYW5jaG9yID0gdGhpcy5nZXRTZWxlY3Rpb24odHJ1ZSk7XG4gICAgICBsZXQgY29tbWFuZFN0YXRlID0gdGhpcy5nZXRDb21tYW5kU3RhdGUoZm9jdXMsIGFuY2hvcik7XG4gICAgICBpZiAodGhpcy5sYXN0Q29tbWFuZFN0YXRlKSB7XG4gICAgICAgIE9iamVjdC5hc3NpZ24odGhpcy5sYXN0Q29tbWFuZFN0YXRlLCBjb21tYW5kU3RhdGUpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy5sYXN0Q29tbWFuZFN0YXRlID0gT2JqZWN0LmFzc2lnbih7fSwgY29tbWFuZFN0YXRlKTtcbiAgICAgIH1cbiAgICAgIGZvciAobGV0IGxpc3RlbmVyIG9mIHRoaXMubGlzdGVuZXJzLnNlbGVjdGlvbikge1xuICAgICAgICBsaXN0ZW5lcih7XG4gICAgICAgICAgZm9jdXM6IGZvY3VzLFxuICAgICAgICAgIGFuY2hvcjogYW5jaG9yLFxuICAgICAgICAgIGNvbW1hbmRTdGF0ZTogdGhpcy5sYXN0Q29tbWFuZFN0YXRlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRmlyZXMgYSBkcm9wIGV2ZW50LlxuICAgKi9cbiAgZmlyZURyb3AoZGF0YVRyYW5zZmVyKSB7XG4gICAgZm9yIChsZXQgbGlzdGVuZXIgb2YgdGhpcy5saXN0ZW5lcnMuZHJvcCkge1xuICAgICAgbGlzdGVuZXIoe1xuICAgICAgICBkYXRhVHJhbnNmZXIsXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBldmVudCBsaXN0ZW5lci5cbiAgICogQHBhcmFtIHtzdHJpbmd9IHR5cGUgVGhlIHR5cGUgb2YgZXZlbnQgdG8gbGlzdGVuIHRvLiBDYW4gYmUgJ2NoYW5nZScsICdzZWxlY3Rpb24nIG9yICdkcm9wJy5cbiAgICogQHBhcmFtIHsqfSBsaXN0ZW5lciBGdW5jdGlvbiBvZiB0aGUgdHlwZSAoZXZlbnQpID0+IHt9IHRvIGJlIGNhbGxlZCB3aGVuIHRoZSBldmVudCBvY2N1cnMuXG4gICAqL1xuICBhZGRFdmVudExpc3RlbmVyKHR5cGUsIGxpc3RlbmVyKSB7XG4gICAgaWYgKHR5cGUubWF0Y2goL14oPzpjaGFuZ2V8aW5wdXQpJC9pKSkge1xuICAgICAgdGhpcy5saXN0ZW5lcnMuY2hhbmdlLnB1c2gobGlzdGVuZXIpO1xuICAgIH1cbiAgICBpZiAodHlwZS5tYXRjaCgvXig/OnNlbGVjdGlvbnxzZWxlY3Rpb25jaGFuZ2UpJC9pKSkge1xuICAgICAgdGhpcy5saXN0ZW5lcnMuc2VsZWN0aW9uLnB1c2gobGlzdGVuZXIpO1xuICAgIH1cbiAgICBpZiAodHlwZS5tYXRjaCgvXig/OmRyb3ApJC9pKSkge1xuICAgICAgdGhpcy5saXN0ZW5lcnMuZHJvcC5wdXNoKGxpc3RlbmVyKTtcbiAgICB9XG4gIH1cbn1cblxuZXhwb3J0IGRlZmF1bHQgRWRpdG9yO1xuIl0sIm5hbWVzIjpbInN2ZyIsImJsb2NrcXVvdGUiLCJib2xkIiwiY2xlYXJfZm9ybWF0dGluZyIsImNvZGUiLCJoMSIsImgyIiwiaHIiLCJpbWFnZSIsIml0YWxpYyIsImxpbmsiLCJvbCIsInN0cmlrZXRocm91Z2giLCJ1bCIsImlzTWFjTGlrZSIsInRlc3QiLCJuYXZpZ2F0b3IiLCJwbGF0Zm9ybSIsIkRlZmF1bHRDb21tYW5kcyIsIm5hbWUiLCJhY3Rpb24iLCJpbm5lckhUTUwiLCJ0aXRsZSIsImhvdGtleSIsImVkaXRvciIsImlzSW5saW5lRm9ybWF0dGluZ0FsbG93ZWQiLCJ3cmFwU2VsZWN0aW9uIiwiZW5hYmxlZCIsImZvY3VzIiwiYW5jaG9yIiwicGFzdGUiLCJDb21tYW5kQmFyIiwiY29uc3RydWN0b3IiLCJwcm9wcyIsImUiLCJjb21tYW5kcyIsImJ1dHRvbnMiLCJzdGF0ZSIsImhvdGtleXMiLCJlbGVtZW50IiwidGFnTmFtZSIsImRvY3VtZW50IiwiZ2V0RWxlbWVudEJ5SWQiLCJib2R5IiwiY3JlYXRlQ29tbWFuZEJhckVsZW1lbnQiLCJhZGRFdmVudExpc3RlbmVyIiwiaGFuZGxlS2V5ZG93biIsInNldEVkaXRvciIsInBhcmVudEVsZW1lbnQiLCJjcmVhdGVFbGVtZW50IiwiY2xhc3NOYW1lIiwiY29tbWFuZCIsImVsIiwiYXBwZW5kQ2hpbGQiLCJjb21tYW5kTmFtZSIsIk9iamVjdCIsImFzc2lnbiIsImtleXMiLCJzcGxpdCIsIm1vZGlmaWVycyIsIm1vZGlmaWVyZXhwbGFuYXRpb24iLCJpIiwibGVuZ3RoIiwicHVzaCIsIm1hdGNoIiwia2V5IiwidG9Mb3dlckNhc2UiLCJjb25jYXQiLCJqb2luIiwiaGFuZGxlQ2xpY2siLCJldmVudCIsInByZXZlbnREZWZhdWx0Iiwic2V0Q29tbWFuZFN0YXRlIiwiaGFuZGxlU2VsZWN0aW9uIiwiY29tbWFuZFN0YXRlIiwidW5kZWZpbmVkIiwib3V0ZXIiLCJtb2RpZmllciIsImNoZWNrIiwiaXQiLCJNYXRoIiwibW9kdWxlIiwiZ2xvYmFsVGhpcyIsIndpbmRvdyIsInNlbGYiLCJnbG9iYWwiLCJ0aGlzIiwiRnVuY3Rpb24iLCJleGVjIiwiZXJyb3IiLCJmYWlscyIsInJlcXVpcmUiLCJkZWZpbmVQcm9wZXJ0eSIsImdldCIsImJpbmQiLCJoYXNPd25Qcm9wZXJ0eSIsIk5BVElWRV9CSU5EIiwiRnVuY3Rpb25Qcm90b3R5cGUiLCJwcm90b3R5cGUiLCJjYWxsIiwidW5jdXJyeVRoaXNXaXRoQmluZCIsImZuIiwiYXBwbHkiLCJhcmd1bWVudHMiLCJkb2N1bWVudEFsbCIsImFsbCIsIklTX0hUTUxEREEiLCIkZG9jdW1lbnRBbGwiLCJhcmd1bWVudCIsImlzTnVsbE9yVW5kZWZpbmVkIiwiJFR5cGVFcnJvciIsIlR5cGVFcnJvciIsInJlcXVpcmVPYmplY3RDb2VyY2libGUiLCIkT2JqZWN0IiwidW5jdXJyeVRoaXMiLCJ0b09iamVjdCIsImhhc093biIsIkRFU0NSSVBUT1JTIiwiZ2V0RGVzY3JpcHRvciIsImdldE93blByb3BlcnR5RGVzY3JpcHRvciIsIkVYSVNUUyIsIlBST1BFUiIsInNvbWV0aGluZyIsIkNPTkZJR1VSQUJMRSIsImNvbmZpZ3VyYWJsZSIsInZhbHVlIiwid3JpdGFibGUiLCJkZWZpbmVHbG9iYWxQcm9wZXJ0eSIsIlNIQVJFRCIsInN0b3JlIiwiaXNDYWxsYWJsZSIsImZ1bmN0aW9uVG9TdHJpbmciLCJ0b1N0cmluZyIsImluc3BlY3RTb3VyY2UiLCJXZWFrTWFwIiwiU3RyaW5nIiwiaXNPYmplY3QiLCJhIiwiJFN0cmluZyIsImFGdW5jdGlvbiIsIm5hbWVzcGFjZSIsIm1ldGhvZCIsImlzUHJvdG90eXBlT2YiLCJ1c2VyQWdlbnQiLCJwcm9jZXNzIiwiRGVubyIsInZlcnNpb25zIiwidmVyc2lvbiIsInY4IiwiVjhfVkVSU0lPTiIsImdldE93blByb3BlcnR5U3ltYm9scyIsInN5bWJvbCIsIlN5bWJvbCIsInNoYW0iLCJOQVRJVkVfU1lNQk9MIiwiaXRlcmF0b3IiLCJnZXRCdWlsdEluIiwiVVNFX1NZTUJPTF9BU19VSUQiLCIkU3ltYm9sIiwidHJ5VG9TdHJpbmciLCJhQ2FsbGFibGUiLCJWIiwiUCIsImZ1bmMiLCJpbnB1dCIsInByZWYiLCJ2YWwiLCJ2YWx1ZU9mIiwibW9kZSIsImNvcHlyaWdodCIsImxpY2Vuc2UiLCJzb3VyY2UiLCJpZCIsInBvc3RmaXgiLCJyYW5kb20iLCJzaGFyZWQiLCJ1aWQiLCJXZWxsS25vd25TeW1ib2xzU3RvcmUiLCJjcmVhdGVXZWxsS25vd25TeW1ib2wiLCJ3aXRob3V0U2V0dGVyIiwiaXNTeW1ib2wiLCJnZXRNZXRob2QiLCJvcmRpbmFyeVRvUHJpbWl0aXZlIiwid2VsbEtub3duU3ltYm9sIiwiVE9fUFJJTUlUSVZFIiwiZXhvdGljVG9QcmltIiwicmVzdWx0IiwidG9QcmltaXRpdmUiLCJJRThfRE9NX0RFRklORSIsIlY4X1BST1RPVFlQRV9ERUZJTkVfQlVHIiwiYW5PYmplY3QiLCJ0b1Byb3BlcnR5S2V5IiwiJGRlZmluZVByb3BlcnR5IiwiJGdldE93blByb3BlcnR5RGVzY3JpcHRvciIsIkVOVU1FUkFCTEUiLCJXUklUQUJMRSIsImV4cG9ydHMiLCJPIiwiQXR0cmlidXRlcyIsImN1cnJlbnQiLCJlbnVtZXJhYmxlIiwiYml0bWFwIiwiZGVmaW5lUHJvcGVydHlNb2R1bGUiLCJjcmVhdGVQcm9wZXJ0eURlc2NyaXB0b3IiLCJvYmplY3QiLCJmIiwiTkFUSVZFX1dFQUtfTUFQIiwiY3JlYXRlTm9uRW51bWVyYWJsZVByb3BlcnR5Iiwic2hhcmVkS2V5IiwiT0JKRUNUX0FMUkVBRFlfSU5JVElBTElaRUQiLCJzZXQiLCJoYXMiLCJlbmZvcmNlIiwiZ2V0dGVyRm9yIiwiVFlQRSIsInR5cGUiLCJtZXRhZGF0YSIsImZhY2FkZSIsIlNUQVRFIiwiQ09ORklHVVJBQkxFX0ZVTkNUSU9OX05BTUUiLCJJbnRlcm5hbFN0YXRlTW9kdWxlIiwiZW5mb3JjZUludGVybmFsU3RhdGUiLCJnZXRJbnRlcm5hbFN0YXRlIiwic3RyaW5nU2xpY2UiLCJzbGljZSIsInJlcGxhY2UiLCJDT05GSUdVUkFCTEVfTEVOR1RIIiwiVEVNUExBVEUiLCJtYWtlQnVpbHRJbiIsIm9wdGlvbnMiLCJnZXR0ZXIiLCJzZXR0ZXIiLCJhcml0eSIsInRhcmdldCIsImRlc2NyaXB0b3IiLCJ0aGF0IiwiaGFzSW5kaWNlcyIsImlnbm9yZUNhc2UiLCJtdWx0aWxpbmUiLCJkb3RBbGwiLCJ1bmljb2RlIiwidW5pY29kZVNldHMiLCJzdGlja3kiLCJkZWZpbmVCdWlsdEluQWNjZXNzb3IiLCJyZWdFeHBGbGFncyIsIlJlZ0V4cCIsIlJlZ0V4cFByb3RvdHlwZSIsIkZPUkNFRCIsIklORElDRVNfU1VQUE9SVCIsImNhbGxzIiwiZXhwZWN0ZWQiLCJhZGRHZXR0ZXIiLCJjaHIiLCJwYWlycyIsInJlcGxhY2VtZW50cyIsIkFTQ0lJUHVuY3R1YXRpb24iLCJOb3RUcmlnZ2VyQ2hhciIsIlNjaGVtZSIsIkVtYWlsIiwiSFRNTE9wZW5UYWciLCJIVE1MQ2xvc2VUYWciLCJIVE1MVGFnTmFtZSIsIkhUTUxDb21tZW50IiwiSFRNTFBJIiwiSFRNTERlY2xhcmF0aW9uIiwiSFRNTENEQVRBIiwiSFRNTEF0dHJpYnV0ZSIsIkhUTUxBdHRWYWx1ZSIsIktub3duVGFnIiwicHVuY3R1YXRpb25MZWFkaW5nIiwicHVuY3R1YXRpb25UcmFpbGluZyIsImxpbmVHcmFtbWFyIiwiVE1IMSIsInJlZ2V4cCIsInJlcGxhY2VtZW50IiwiVE1IMiIsIlRNSDMiLCJUTUg0IiwiVE1INSIsIlRNSDYiLCJUTUJsb2NrcXVvdGUiLCJUTUNvZGVGZW5jZUJhY2t0aWNrT3BlbiIsIlRNQ29kZUZlbmNlVGlsZGVPcGVuIiwiVE1Db2RlRmVuY2VCYWNrdGlja0Nsb3NlIiwiVE1Db2RlRmVuY2VUaWxkZUNsb3NlIiwiVE1CbGFua0xpbmUiLCJUTVNldGV4dEgxTWFya2VyIiwiVE1TZXRleHRIMk1hcmtlciIsIlRNSFIiLCJUTVVMIiwiVE1PTCIsIlRNSW5kZW50ZWRDb2RlIiwiVE1MaW5rUmVmZXJlbmNlRGVmaW5pdGlvbiIsImxhYmVsUGxhY2Vob2xkZXIiLCJodG1sQmxvY2tHcmFtbWFyIiwic3RhcnQiLCJlbmQiLCJwYXJhSW50ZXJydXB0IiwiaW5saW5lR3JhbW1hciIsImVzY2FwZSIsImF1dG9saW5rIiwiaHRtbCIsImxpbmtPcGVuIiwiaW1hZ2VPcGVuIiwibGlua0xhYmVsIiwiZGVmYXVsdCIsInJlcGxhY2VtZW50UmVnZXhwIiwiaW5saW5lUnVsZXMiLCJydWxlIiwicmUiLCJzdHJpbmciLCJmbGFncyIsImh0bWxlc2NhcGUiLCJwcmUiLCJwb3N0IiwidW5zZXQiLCJwcmVQYXR0ZXJuIiwicG9zdFBhdHRlcm4iLCJwYXR0ZXJuIiwiaDMiLCJoNCIsImg1IiwiaDYiLCJFZGl0b3IiLCJ0ZXh0YXJlYSIsImxpbmVzIiwibGluZUVsZW1lbnRzIiwibGluZVR5cGVzIiwibGluZUNhcHR1cmVzIiwibGluZVJlcGxhY2VtZW50cyIsImxpbmtMYWJlbHMiLCJsaW5lRGlydHkiLCJsYXN0Q29tbWFuZFN0YXRlIiwibGlzdGVuZXJzIiwiY2hhbmdlIiwic2VsZWN0aW9uIiwiZHJvcCIsImdldEVsZW1lbnRzQnlUYWdOYW1lIiwicGFyZW50Tm9kZSIsInN0eWxlIiwiZGlzcGxheSIsImNyZWF0ZUVkaXRvckVsZW1lbnQiLCJzZXRDb250ZW50IiwiY29udGVudCIsImNsYXNzTGlzdCIsImFkZCIsImNvbnRlbnRFZGl0YWJsZSIsIndoaXRlU3BhY2UiLCJ3ZWJraXRVc2VyTW9kaWZ5IiwibmV4dFNpYmxpbmciLCJpbnNlcnRCZWZvcmUiLCJoYW5kbGVJbnB1dEV2ZW50IiwiaGFuZGxlU2VsZWN0aW9uQ2hhbmdlRXZlbnQiLCJoYW5kbGVQYXN0ZSIsImhhbmRsZURyb3AiLCJjaGlsZE5vZGVzIiwiZmlyc3RDaGlsZCIsInJlbW92ZUNoaWxkIiwibGluZU51bSIsImxlIiwiQXJyYXkiLCJ1cGRhdGVGb3JtYXR0aW5nIiwiZmlyZUNoYW5nZSIsImdldENvbnRlbnQiLCJ1cGRhdGVMaW5lVHlwZXMiLCJ1cGRhdGVMaW5rTGFiZWxzIiwiYXBwbHlMaW5lVHlwZXMiLCJsIiwiY2FwdHVyZSIsInN0ciIsInAxIiwicDIiLCJwcm9jZXNzSW5saW5lU3R5bGVzIiwiY29udGVudEhUTUwiLCJyZW1vdmVBdHRyaWJ1dGUiLCJkYXRhc2V0IiwiY29kZUJsb2NrVHlwZSIsImNvZGVCbG9ja1NlcUxlbmd0aCIsImh0bWxCbG9jayIsImxpbmVUeXBlIiwibGluZUNhcHR1cmUiLCJsaW5lUmVwbGFjZW1lbnQiLCJncm91cHMiLCJodG1sQmxvY2tUeXBlIiwiaGVhZGluZ0xpbmUiLCJoZWFkaW5nTGluZVR5cGUiLCJ1cGRhdGVMaW5lQ29udGVudHNBbmRGb3JtYXR0aW5nIiwiY2xlYXJEaXJ0eUZsYWciLCJ1cGRhdGVMaW5lQ29udGVudHMiLCJwYXJzZUxpbmtPckltYWdlIiwib3JpZ2luYWxTdHJpbmciLCJpc0ltYWdlIiwidGV4dE9mZnNldCIsIm9wZW5lciIsInN1YnN0ciIsImN1cnJlbnRPZmZzZXQiLCJicmFja2V0TGV2ZWwiLCJsaW5rVGV4dCIsImxpbmtSZWYiLCJsaW5rRGV0YWlscyIsInRleHRPdXRlciIsImNhcCIsIm5leHRDaGFyIiwidHJpbSIsInBhcmVudGhlc2lzTGV2ZWwiLCJpbmxpbmVPdXRlciIsInZhbGlkIiwibGFiZWwiLCJvdXRwdXQiLCJjaGFyQ291bnQiLCJwcm9jZXNzZWQiLCJzdGFjayIsIm9mZnNldCIsInBvdGVudGlhbExpbmsiLCJwb3RlbnRpYWxJbWFnZSIsImRlbGltQ291bnQiLCJkZWxpbVN0cmluZyIsImN1cnJlbnREZWxpbWl0ZXIiLCJwcmVjZWRpbmciLCJmb2xsb3dpbmciLCJwdW5jdHVhdGlvbkZvbGxvd3MiLCJwdW5jdHVhdGlvblByZWNlZGVzIiwid2hpdGVzcGFjZUZvbGxvd3MiLCJ3aGl0ZXNwYWNlUHJlY2VkZXMiLCJjYW5PcGVuIiwiY2FuQ2xvc2UiLCJzdGFja1BvaW50ZXIiLCJkZWxpbWl0ZXIiLCJlbnRyeSIsInBvcCIsImNvdW50IiwiY29uc3VtZWQiLCJsaW5lRGVsdGEiLCJjaGlsZEVsZW1lbnRDb3VudCIsImZpcnN0Q2hhbmdlZExpbmUiLCJ0ZXh0Q29udGVudCIsImxhc3RDaGFuZ2VkTGluZSIsImxpbmVzVG9EZWxldGUiLCJsaW5lc1RvQWRkIiwic3BsaWNlTGluZXMiLCJsaW5lIiwiY3QiLCJwcm9jZXNzTmV3UGFyYWdyYXBoIiwic2VsIiwiY29udGludWFibGVUeXBlIiwiY2hlY2tMaW5lIiwiY29sIiwicm93IiwicGFyc2VJbnQiLCJnZXRTZWxlY3Rpb24iLCJnZXRBbmNob3IiLCJzdGFydE5vZGUiLCJhbmNob3JOb2RlIiwiZm9jdXNOb2RlIiwiYW5jaG9yT2Zmc2V0IiwiZm9jdXNPZmZzZXQiLCJjb21wdXRlQ29sdW1uIiwibm9kZSIsInByZXZpb3VzU2libGluZyIsIm5vZGVUeXBlIiwiTm9kZSIsIlRFWFRfTk9ERSIsImNvbXB1dGVOb2RlQW5kT2Zmc2V0IiwiYmluZFJpZ2h0IiwiY2hpbGRyZW5Db21wbGV0ZSIsInJ2Iiwibm9kZVZhbHVlIiwic2V0U2VsZWN0aW9uIiwicmFuZ2UiLCJjcmVhdGVSYW5nZSIsInNldFN0YXJ0Iiwic2V0RW5kIiwid2luZG93U2VsZWN0aW9uIiwicmVtb3ZlQWxsUmFuZ2VzIiwiYWRkUmFuZ2UiLCJpbnB1dFR5cGUiLCJmaXhOb2RlSGllcmFyY2h5Iiwib3JpZ2luYWxDaGlsZHJlbiIsImZyb20iLCJyZXBsYWNlQ2hpbGQiLCJjaGlsZCIsInBhcmVudCIsIl9sZW4iLCJuZXdDaGlsZHJlbiIsIl9rZXkiLCJmb3JFYWNoIiwibmV3Q2hpbGQiLCJFTEVNRU5UX05PREUiLCJkaXZXcmFwcGVyIiwiZ3JhbmRDaGlsZHJlbiIsInNvbWUiLCJncmFuZENoaWxkIiwiZmlyZURyb3AiLCJkYXRhVHJhbnNmZXIiLCJmaXJlU2VsZWN0aW9uIiwic3RhcnRMaW5lIiwibGluZXNUb0luc2VydCIsImFkanVzdExpbmVFbGVtZW50cyIsImluc2VydGVkQmxhbmsiLCJpbnNlcnRlZERpcnR5Iiwic3BsaWNlIiwidGV4dCIsIm9yaWdpbmFsRXZlbnQiLCJjbGlwYm9hcmREYXRhIiwiZ2V0RGF0YSIsImJlZ2lubmluZyIsImluc2VydGVkTGluZXMiLCJsaW5lQmVmb3JlIiwibGluZUVuZCIsImVuZENvbFBvcyIsImNvbXB1dGVDb21tb25BbmNlc3RvciIsIm5vZGUxIiwibm9kZTIiLCJhbmNlc3RyeSIsInVuc2hpZnQiLCJhbmNlc3RyeTEiLCJhbmNlc3RyeTIiLCJjb21wdXRlRW5jbG9zaW5nTWFya3VwTm9kZSIsImluY2x1ZGVzIiwiZ2V0Q29tbWFuZFN0YXRlIiwiY21kIiwibWFya3VwTm9kZSIsInN0YXJ0Q29sIiwibGVuIiwibGVmdCIsIm1pZCIsInJpZ2h0IiwiZW5kQ29sIiwibGVhZGluZyIsInRyYWlsaW5nIiwiaXNDb2xsYXBzZWQiLCJhbmNlc3RvciIsInRvZ2dsZUNvbW1hbmRTdGF0ZSIsImxpc3RlbmVyIiwibGluZXNEaXJ0eSJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0VBQUEsTUFBTUEsR0FBRyxHQUFHO0VBQ1ZDLEVBQUFBLFVBQVUsRUFBRyxDQUEwUix5UkFBQSxDQUFBO0VBQ3ZTQyxFQUFBQSxJQUFJLEVBQUcsQ0FBeVksd1lBQUEsQ0FBQTtFQUNoWkMsRUFBQUEsZ0JBQWdCLEVBQUcsQ0FBMFgseVhBQUEsQ0FBQTtFQUM3WUMsRUFBQUEsSUFBSSxFQUFHLENBQThVLDZVQUFBLENBQUE7RUFDclZDLEVBQUFBLEVBQUUsRUFBRyxDQUErSiw4SkFBQSxDQUFBO0VBQ3BLQyxFQUFBQSxFQUFFLEVBQUcsQ0FBaUssZ0tBQUEsQ0FBQTtFQUN0S0MsRUFBQUEsRUFBRSxFQUFHLENBQWdJLCtIQUFBLENBQUE7RUFDcklDLEVBQUFBLEtBQUssRUFBRyxDQUE4SCw2SEFBQSxDQUFBO0VBQ3RJQyxFQUFBQSxNQUFNLEVBQUcsQ0FBZ0gsK0dBQUEsQ0FBQTtFQUN6SEMsRUFBQUEsSUFBSSxFQUFHLENBQTB1Qyx5dUNBQUEsQ0FBQTtFQUNqdkNDLEVBQUFBLEVBQUUsRUFBRyxDQUFvdEIsbXRCQUFBLENBQUE7RUFDenRCQyxFQUFBQSxhQUFhLEVBQUcsQ0FBMlgsMFhBQUEsQ0FBQTtFQUMzWUMsRUFBQUEsRUFBRSxFQUFHLENBQUEsMlFBQUEsQ0FBQTtFQUNQLENBQUM7O0VDWkQsTUFBTUMsU0FBUyxHQUFHLHlCQUF5QixDQUFDQyxJQUFJLENBQUMsT0FBT0MsU0FBUyxLQUFLLFdBQVcsR0FBR0EsU0FBUyxDQUFDQyxRQUFRLEdBQUcsRUFBRSxDQUFDLENBQUE7RUFFNUcsTUFBTUMsZUFBZSxHQUFHO0VBQ3RCLEVBQUEsTUFBTSxFQUFFO0VBQ05DLElBQUFBLElBQUksRUFBRSxNQUFNO0VBQ1pDLElBQUFBLE1BQU0sRUFBRSxNQUFNO01BQ2RDLFNBQVMsRUFBRXJCLEdBQUcsQ0FBQ0UsSUFBSTtFQUNuQm9CLElBQUFBLEtBQUssRUFBRSxNQUFNO0VBQ2JDLElBQUFBLE1BQU0sRUFBRSxPQUFBO0tBQ1Q7RUFDRCxFQUFBLFFBQVEsRUFBRTtFQUNSSixJQUFBQSxJQUFJLEVBQUUsUUFBUTtFQUNkQyxJQUFBQSxNQUFNLEVBQUUsUUFBUTtNQUNoQkMsU0FBUyxFQUFFckIsR0FBRyxDQUFDUyxNQUFNO0VBQ3JCYSxJQUFBQSxLQUFLLEVBQUUsUUFBUTtFQUNmQyxJQUFBQSxNQUFNLEVBQUUsT0FBQTtLQUNUO0VBQ0QsRUFBQSxlQUFlLEVBQUU7RUFDZkosSUFBQUEsSUFBSSxFQUFFLGVBQWU7RUFDckJDLElBQUFBLE1BQU0sRUFBRSxlQUFlO01BQ3ZCQyxTQUFTLEVBQUVyQixHQUFHLENBQUNZLGFBQWE7RUFDNUJVLElBQUFBLEtBQUssRUFBRSxlQUFlO0VBQ3RCQyxJQUFBQSxNQUFNLEVBQUUsY0FBQTtLQUNUO0VBQ0QsRUFBQSxNQUFNLEVBQUU7RUFDTkosSUFBQUEsSUFBSSxFQUFFLE1BQU07RUFDWkMsSUFBQUEsTUFBTSxFQUFFLE1BQU07TUFDZEMsU0FBUyxFQUFFckIsR0FBRyxDQUFDSSxJQUFJO0VBQ25Ca0IsSUFBQUEsS0FBSyxFQUFFLGdCQUFBO0tBQ1I7RUFDRCxFQUFBLElBQUksRUFBRTtFQUNKSCxJQUFBQSxJQUFJLEVBQUUsSUFBSTtFQUNWQyxJQUFBQSxNQUFNLEVBQUUsSUFBSTtNQUNaQyxTQUFTLEVBQUVyQixHQUFHLENBQUNLLEVBQUU7RUFDakJpQixJQUFBQSxLQUFLLEVBQUUsaUJBQWlCO0VBQ3hCQyxJQUFBQSxNQUFNLEVBQUUsYUFBQTtLQUNUO0VBQ0QsRUFBQSxJQUFJLEVBQUU7RUFDSkosSUFBQUEsSUFBSSxFQUFFLElBQUk7RUFDVkMsSUFBQUEsTUFBTSxFQUFFLElBQUk7TUFDWkMsU0FBUyxFQUFFckIsR0FBRyxDQUFDTSxFQUFFO0VBQ2pCZ0IsSUFBQUEsS0FBSyxFQUFFLGlCQUFpQjtFQUN4QkMsSUFBQUEsTUFBTSxFQUFFLGFBQUE7S0FDVDtFQUNELEVBQUEsSUFBSSxFQUFFO0VBQ0pKLElBQUFBLElBQUksRUFBRSxJQUFJO0VBQ1ZDLElBQUFBLE1BQU0sRUFBRSxJQUFJO01BQ1pDLFNBQVMsRUFBRXJCLEdBQUcsQ0FBQ2EsRUFBRTtFQUNqQlMsSUFBQUEsS0FBSyxFQUFFLGVBQUE7S0FDUjtFQUNELEVBQUEsSUFBSSxFQUFFO0VBQ0pILElBQUFBLElBQUksRUFBRSxJQUFJO0VBQ1ZDLElBQUFBLE1BQU0sRUFBRSxJQUFJO01BQ1pDLFNBQVMsRUFBRXJCLEdBQUcsQ0FBQ1csRUFBRTtFQUNqQlcsSUFBQUEsS0FBSyxFQUFFLGVBQUE7S0FDUjtFQUNELEVBQUEsWUFBWSxFQUFFO0VBQ1pILElBQUFBLElBQUksRUFBRSxZQUFZO0VBQ2xCQyxJQUFBQSxNQUFNLEVBQUUsWUFBWTtNQUNwQkMsU0FBUyxFQUFFckIsR0FBRyxDQUFDQyxVQUFVO0VBQ3pCcUIsSUFBQUEsS0FBSyxFQUFFLE9BQU87RUFDZEMsSUFBQUEsTUFBTSxFQUFFLGNBQUE7S0FDVDtFQUNELEVBQUEsWUFBWSxFQUFFO0VBQ1pKLElBQUFBLElBQUksRUFBRSxZQUFZO01BQ2xCQyxNQUFNLEVBQUdJLE1BQU0sSUFBSztFQUFDLE1BQUEsSUFBSUEsTUFBTSxDQUFDQyx5QkFBeUIsRUFBRSxFQUFFRCxNQUFNLENBQUNFLGFBQWEsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUE7T0FBQztFQUM5RkMsSUFBQUEsT0FBTyxFQUFFQSxDQUFDSCxNQUFNLEVBQUVJLEtBQUssRUFBRUMsTUFBTSxLQUFLTCxNQUFNLENBQUNDLHlCQUF5QixDQUFDRyxLQUFLLEVBQUVDLE1BQU0sQ0FBQyxHQUFHLEtBQUssR0FBRyxJQUFJO01BQ2xHUixTQUFTLEVBQUVyQixHQUFHLENBQUNVLElBQUk7RUFDbkJZLElBQUFBLEtBQUssRUFBRSxhQUFhO0VBQ3BCQyxJQUFBQSxNQUFNLEVBQUUsT0FBQTtLQUNUO0VBQ0QsRUFBQSxhQUFhLEVBQUU7RUFDYkosSUFBQUEsSUFBSSxFQUFFLGFBQWE7TUFDbkJDLE1BQU0sRUFBR0ksTUFBTSxJQUFLO0VBQUMsTUFBQSxJQUFJQSxNQUFNLENBQUNDLHlCQUF5QixFQUFFLEVBQUVELE1BQU0sQ0FBQ0UsYUFBYSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQTtPQUFDO0VBQy9GQyxJQUFBQSxPQUFPLEVBQUVBLENBQUNILE1BQU0sRUFBRUksS0FBSyxFQUFFQyxNQUFNLEtBQUtMLE1BQU0sQ0FBQ0MseUJBQXlCLENBQUNHLEtBQUssRUFBRUMsTUFBTSxDQUFDLEdBQUcsS0FBSyxHQUFHLElBQUk7TUFDbEdSLFNBQVMsRUFBRXJCLEdBQUcsQ0FBQ1EsS0FBSztFQUNwQmMsSUFBQUEsS0FBSyxFQUFFLGNBQWM7RUFDckJDLElBQUFBLE1BQU0sRUFBRSxjQUFBO0tBQ1Q7RUFDRCxFQUFBLElBQUksRUFBRTtFQUNKSixJQUFBQSxJQUFJLEVBQUUsSUFBSTtNQUNWQyxNQUFNLEVBQUdJLE1BQU0sSUFBS0EsTUFBTSxDQUFDTSxLQUFLLENBQUMsU0FBUyxDQUFDO01BQzNDSCxPQUFPLEVBQUVBLE1BQU0sS0FBSztNQUNwQk4sU0FBUyxFQUFFckIsR0FBRyxDQUFDTyxFQUFFO0VBQ2pCZSxJQUFBQSxLQUFLLEVBQUUsd0JBQXdCO0VBQy9CQyxJQUFBQSxNQUFNLEVBQUUsY0FBQTtFQUNWLEdBQUE7RUFDRixDQUFDLENBQUE7RUFHRCxNQUFNUSxVQUFVLENBQUM7SUFDZkMsV0FBV0EsQ0FBQ0MsS0FBSyxFQUFFO01BQ2pCLElBQUksQ0FBQ0MsQ0FBQyxHQUFHLElBQUksQ0FBQTtNQUNiLElBQUksQ0FBQ1YsTUFBTSxHQUFHLElBQUksQ0FBQTtNQUNsQixJQUFJLENBQUNXLFFBQVEsR0FBRyxFQUFFLENBQUE7RUFDbEIsSUFBQSxJQUFJLENBQUNDLE9BQU8sR0FBRyxFQUFFLENBQUE7RUFDakIsSUFBQSxJQUFJLENBQUNDLEtBQUssR0FBRyxFQUFFLENBQUE7TUFDZixJQUFJLENBQUNDLE9BQU8sR0FBRyxFQUFFLENBQUE7RUFFakIsSUFBQSxJQUFJQyxPQUFPLEdBQUdOLEtBQUssQ0FBQ00sT0FBTyxDQUFBO0VBQzNCLElBQUEsSUFBSUEsT0FBTyxJQUFJLENBQUNBLE9BQU8sQ0FBQ0MsT0FBTyxFQUFFO1FBQy9CRCxPQUFPLEdBQUdFLFFBQVEsQ0FBQ0MsY0FBYyxDQUFDVCxLQUFLLENBQUNNLE9BQU8sQ0FBQyxDQUFBO0VBQ2xELEtBQUE7TUFDQSxJQUFJLENBQUNBLE9BQU8sRUFBRTtRQUNaQSxPQUFPLEdBQUdFLFFBQVEsQ0FBQ0UsSUFBSSxDQUFBO0VBQ3pCLEtBQUE7TUFDQSxJQUFJLENBQUNDLHVCQUF1QixDQUFDTCxPQUFPLEVBQUVOLEtBQUssQ0FBQ0UsUUFBUSxJQUFJLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRSxlQUFlLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsWUFBWSxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsWUFBWSxFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUE7RUFDdE1NLElBQUFBLFFBQVEsQ0FBQ0ksZ0JBQWdCLENBQUMsU0FBUyxFQUFHWCxDQUFDLElBQUssSUFBSSxDQUFDWSxhQUFhLENBQUNaLENBQUMsQ0FBQyxDQUFDLENBQUE7TUFDbEUsSUFBSUQsS0FBSyxDQUFDVCxNQUFNLEVBQUUsSUFBSSxDQUFDdUIsU0FBUyxDQUFDZCxLQUFLLENBQUNULE1BQU0sQ0FBQyxDQUFBO0VBQ2hELEdBQUE7RUFFQW9CLEVBQUFBLHVCQUF1QkEsQ0FBQ0ksYUFBYSxFQUFFYixRQUFRLEVBQUU7TUFDL0MsSUFBSSxDQUFDRCxDQUFDLEdBQUdPLFFBQVEsQ0FBQ1EsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFBO0VBQ3RDLElBQUEsSUFBSSxDQUFDZixDQUFDLENBQUNnQixTQUFTLEdBQUcsY0FBYyxDQUFBO0VBRWpDLElBQUEsS0FBSyxJQUFJQyxPQUFPLElBQUloQixRQUFRLEVBQUU7UUFDNUIsSUFBSWdCLE9BQU8sSUFBSSxHQUFHLEVBQUU7RUFDbEIsUUFBQSxJQUFJQyxFQUFFLEdBQUdYLFFBQVEsQ0FBQ1EsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFBO1VBQ3RDRyxFQUFFLENBQUNGLFNBQVMsR0FBRyxrQkFBa0IsQ0FBQTtFQUNqQyxRQUFBLElBQUksQ0FBQ2hCLENBQUMsQ0FBQ21CLFdBQVcsQ0FBQ0QsRUFBRSxDQUFDLENBQUE7RUFDeEIsT0FBQyxNQUFNO0VBQ0wsUUFBQSxJQUFJRSxXQUFXLENBQUE7RUFDZixRQUFBLElBQUksT0FBT0gsT0FBTyxJQUFJLFFBQVEsRUFBRTtFQUM5Qjs7RUFFQSxVQUFBLElBQUlqQyxlQUFlLENBQUNpQyxPQUFPLENBQUMsRUFBRTtFQUM1QkcsWUFBQUEsV0FBVyxHQUFHSCxPQUFPLENBQUE7Y0FDckIsSUFBSSxDQUFDaEIsUUFBUSxDQUFDbUIsV0FBVyxDQUFDLEdBQUdwQyxlQUFlLENBQUNvQyxXQUFXLENBQUMsQ0FBQTtFQUczRCxXQUFDLE1BQU07RUFDTCxZQUFBLFNBQUE7RUFDRixXQUFBO1dBRUQsTUFBTSxJQUFJLE9BQU9ILE9BQU8sSUFBSSxRQUFRLElBQUlBLE9BQU8sQ0FBQ2hDLElBQUksRUFBRTtZQUNyRG1DLFdBQVcsR0FBR0gsT0FBTyxDQUFDaEMsSUFBSSxDQUFBO0VBQzFCLFVBQUEsSUFBSSxDQUFDZ0IsUUFBUSxDQUFDbUIsV0FBVyxDQUFDLEdBQUcsRUFBRSxDQUFBO1lBQy9CLElBQUlwQyxlQUFlLENBQUNvQyxXQUFXLENBQUMsRUFBRUMsTUFBTSxDQUFDQyxNQUFNLENBQUMsSUFBSSxDQUFDckIsUUFBUSxDQUFDbUIsV0FBVyxDQUFDLEVBQUVwQyxlQUFlLENBQUNvQyxXQUFXLENBQUMsQ0FBQyxDQUFBO1lBQ3pHQyxNQUFNLENBQUNDLE1BQU0sQ0FBQyxJQUFJLENBQUNyQixRQUFRLENBQUNtQixXQUFXLENBQUMsRUFBRUgsT0FBTyxDQUFDLENBQUE7RUFHcEQsU0FBQyxNQUFNO0VBQ0wsVUFBQSxTQUFBO0VBQ0YsU0FBQTtVQUVBLElBQUk3QixLQUFLLEdBQUcsSUFBSSxDQUFDYSxRQUFRLENBQUNtQixXQUFXLENBQUMsQ0FBQ2hDLEtBQUssSUFBSWdDLFdBQVcsQ0FBQTtVQUUzRCxJQUFJLElBQUksQ0FBQ25CLFFBQVEsQ0FBQ21CLFdBQVcsQ0FBQyxDQUFDL0IsTUFBTSxFQUFFO0VBQ3JDLFVBQUEsTUFBTWtDLElBQUksR0FBRyxJQUFJLENBQUN0QixRQUFRLENBQUNtQixXQUFXLENBQUMsQ0FBQy9CLE1BQU0sQ0FBQ21DLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQTtFQUN6RDtZQUNBLElBQUlDLFNBQVMsR0FBRyxFQUFFLENBQUE7WUFDbEIsSUFBSUMsbUJBQW1CLEdBQUcsRUFBRSxDQUFBO0VBQzVCLFVBQUEsS0FBSyxJQUFJQyxDQUFDLEdBQUcsQ0FBQyxFQUFFQSxDQUFDLEdBQUdKLElBQUksQ0FBQ0ssTUFBTSxHQUFHLENBQUMsRUFBRUQsQ0FBQyxFQUFFLEVBQUU7Y0FDeEMsUUFBUUosSUFBSSxDQUFDSSxDQUFDLENBQUM7RUFDYixjQUFBLEtBQUssTUFBTTtFQUFFRixnQkFBQUEsU0FBUyxDQUFDSSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUE7RUFBRUgsZ0JBQUFBLG1CQUFtQixDQUFDRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUE7RUFBRSxnQkFBQSxNQUFBO0VBQzFFLGNBQUEsS0FBSyxLQUFLO0VBQUVKLGdCQUFBQSxTQUFTLENBQUNJLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQTtFQUFFSCxnQkFBQUEsbUJBQW1CLENBQUNHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQTtFQUFFLGdCQUFBLE1BQUE7RUFDdEUsY0FBQSxLQUFLLEtBQUs7RUFBRUosZ0JBQUFBLFNBQVMsQ0FBQ0ksSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFBO0VBQUVILGdCQUFBQSxtQkFBbUIsQ0FBQ0csSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFBO0VBQUUsZ0JBQUEsTUFBQTtFQUN2RSxjQUFBLEtBQUssUUFBUTtFQUFFSixnQkFBQUEsU0FBUyxDQUFDSSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUE7RUFBRUgsZ0JBQUFBLG1CQUFtQixDQUFDRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUE7RUFBRSxnQkFBQSxNQUFBO0VBQ3hFLGNBQUEsS0FBSyxLQUFLO0VBQUVKLGdCQUFBQSxTQUFTLENBQUNJLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQTtFQUFFSCxnQkFBQUEsbUJBQW1CLENBQUNHLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQTtFQUFFLGdCQUFBLE1BQUE7RUFFMUUsY0FBQSxLQUFLLE9BQU87RUFBR0osZ0JBQUFBLFNBQVMsQ0FBQ0ksSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFBO0VBQUVILGdCQUFBQSxtQkFBbUIsQ0FBQ0csSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFBO0VBQUUsZ0JBQUEsTUFBQTtFQUUxRSxjQUFBLEtBQUssS0FBSztFQUFFO0VBQ1YsZ0JBQUEsSUFBSWpELFNBQVMsRUFBRTtFQUFDNkMsa0JBQUFBLFNBQVMsQ0FBQ0ksSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFBO0VBQUVILGtCQUFBQSxtQkFBbUIsQ0FBQ0csSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFBO0VBQUMsaUJBQUMsTUFDckU7RUFBQ0osa0JBQUFBLFNBQVMsQ0FBQ0ksSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFBO0VBQUVILGtCQUFBQSxtQkFBbUIsQ0FBQ0csSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFBO0VBQUMsaUJBQUE7RUFDbEUsZ0JBQUEsTUFBQTtFQUNGLGNBQUEsS0FBSyxNQUFNO0VBQ1RKLGdCQUFBQSxTQUFTLENBQUNJLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQTtFQUN4QixnQkFBQSxJQUFJakQsU0FBUyxFQUFFOEMsbUJBQW1CLENBQUNHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUN4Q0gsbUJBQW1CLENBQUNHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQTtFQUNwQyxnQkFBQSxNQUFBO0VBQU87RUFDWCxhQUFBO0VBQ0YsV0FBQTs7WUFDQUgsbUJBQW1CLENBQUNHLElBQUksQ0FBQ04sSUFBSSxDQUFDQSxJQUFJLENBQUNLLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFBO0VBQy9DLFVBQUEsSUFBSXZDLE1BQU0sR0FBRztFQUVYb0MsWUFBQUEsU0FBUyxFQUFFQSxTQUFTO0VBQ3BCUixZQUFBQSxPQUFPLEVBQUVHLFdBQUFBO2FBQ1YsQ0FBQTtFQUNEO0VBQ0EsVUFBQSxJQUFJRyxJQUFJLENBQUNBLElBQUksQ0FBQ0ssTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDRSxLQUFLLENBQUMsU0FBUyxDQUFDLEVBQUU7RUFDMUN6QyxZQUFBQSxNQUFNLENBQUNuQixJQUFJLEdBQUksQ0FBQSxLQUFBLEVBQU9xRCxJQUFJLENBQUNBLElBQUksQ0FBQ0ssTUFBTSxHQUFHLENBQUMsQ0FBRSxDQUFDLENBQUEsQ0FBQTtFQUMvQyxXQUFDLE1BQU07RUFDTHZDLFlBQUFBLE1BQU0sQ0FBQzBDLEdBQUcsR0FBR1IsSUFBSSxDQUFDQSxJQUFJLENBQUNLLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQ0ksV0FBVyxFQUFFLENBQUE7RUFDbEQsV0FBQTtFQUNBLFVBQUEsSUFBSSxDQUFDNUIsT0FBTyxDQUFDeUIsSUFBSSxDQUFDeEMsTUFBTSxDQUFDLENBQUE7RUFDekJELFVBQUFBLEtBQUssR0FBR0EsS0FBSyxDQUFDNkMsTUFBTSxDQUFFLENBQUEsRUFBQSxFQUFJUCxtQkFBbUIsQ0FBQ1EsSUFBSSxDQUFDLEdBQUcsQ0FBRSxHQUFFLENBQUMsQ0FBQTtFQUM3RCxTQUFBO1VBRUEsSUFBSSxDQUFDaEMsT0FBTyxDQUFDa0IsV0FBVyxDQUFDLEdBQUdiLFFBQVEsQ0FBQ1EsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFBO1VBQ3pELElBQUksQ0FBQ2IsT0FBTyxDQUFDa0IsV0FBVyxDQUFDLENBQUNKLFNBQVMsR0FBRywwQ0FBMEMsQ0FBQTtVQUNoRixJQUFJLENBQUNkLE9BQU8sQ0FBQ2tCLFdBQVcsQ0FBQyxDQUFDaEMsS0FBSyxHQUFHQSxLQUFLLENBQUE7RUFDdkMsUUFBQSxJQUFJLENBQUNjLE9BQU8sQ0FBQ2tCLFdBQVcsQ0FBQyxDQUFDakMsU0FBUyxHQUFHLElBQUksQ0FBQ2MsUUFBUSxDQUFDbUIsV0FBVyxDQUFDLENBQUNqQyxTQUFTLENBQUE7VUFFMUUsSUFBSSxDQUFDZSxPQUFPLENBQUNrQixXQUFXLENBQUMsQ0FBQ1QsZ0JBQWdCLENBQUMsV0FBVyxFQUFHWCxDQUFDLElBQUssSUFBSSxDQUFDbUMsV0FBVyxDQUFDZixXQUFXLEVBQUVwQixDQUFDLENBQUMsQ0FBQyxDQUFBO1VBQ2hHLElBQUksQ0FBQ0EsQ0FBQyxDQUFDbUIsV0FBVyxDQUFDLElBQUksQ0FBQ2pCLE9BQU8sQ0FBQ2tCLFdBQVcsQ0FBQyxDQUFDLENBQUE7RUFDL0MsT0FBQTtFQUNGLEtBQUE7RUFDQU4sSUFBQUEsYUFBYSxDQUFDSyxXQUFXLENBQUMsSUFBSSxDQUFDbkIsQ0FBQyxDQUFDLENBQUE7RUFDbkMsR0FBQTtFQUVBbUMsRUFBQUEsV0FBV0EsQ0FBQ2YsV0FBVyxFQUFFZ0IsS0FBSyxFQUFFO0VBQzlCLElBQUEsSUFBSSxDQUFDLElBQUksQ0FBQzlDLE1BQU0sRUFBRSxPQUFBO01BQ2xCOEMsS0FBSyxDQUFDQyxjQUFjLEVBQUUsQ0FBQTtNQUN0QixJQUFJLE9BQU8sSUFBSSxDQUFDcEMsUUFBUSxDQUFDbUIsV0FBVyxDQUFDLENBQUNsQyxNQUFNLElBQUksUUFBUSxFQUFFO0VBQ3hELE1BQUEsSUFBSSxJQUFJLENBQUNpQixLQUFLLENBQUNpQixXQUFXLENBQUMsS0FBSyxLQUFLLEVBQUUsSUFBSSxDQUFDOUIsTUFBTSxDQUFDZ0QsZUFBZSxDQUFDbEIsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDLEtBQ2pGLElBQUksQ0FBQzlCLE1BQU0sQ0FBQ2dELGVBQWUsQ0FBQ2xCLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQTtFQUN0RCxLQUFDLE1BQU0sSUFBSSxPQUFPLElBQUksQ0FBQ25CLFFBQVEsQ0FBQ21CLFdBQVcsQ0FBQyxDQUFDbEMsTUFBTSxJQUFJLFVBQVUsRUFBRTtRQUNqRSxJQUFJLENBQUNlLFFBQVEsQ0FBQ21CLFdBQVcsQ0FBQyxDQUFDbEMsTUFBTSxDQUFDLElBQUksQ0FBQ0ksTUFBTSxDQUFDLENBQUE7RUFDaEQsS0FBQTtFQUNGLEdBQUE7SUFFQXVCLFNBQVNBLENBQUN2QixNQUFNLEVBQUU7TUFDaEIsSUFBSSxDQUFDQSxNQUFNLEdBQUdBLE1BQU0sQ0FBQTtFQUNwQkEsSUFBQUEsTUFBTSxDQUFDcUIsZ0JBQWdCLENBQUMsV0FBVyxFQUFHWCxDQUFDLElBQUssSUFBSSxDQUFDdUMsZUFBZSxDQUFDdkMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUN0RSxHQUFBO0lBRUF1QyxlQUFlQSxDQUFDSCxLQUFLLEVBQUU7TUFDckIsSUFBSUEsS0FBSyxDQUFDSSxZQUFZLEVBQUU7RUFDdEIsTUFBQSxLQUFLLElBQUl2QixPQUFPLElBQUksSUFBSSxDQUFDaEIsUUFBUSxFQUFFO1VBQ2pDLElBQUltQyxLQUFLLENBQUNJLFlBQVksQ0FBQ3ZCLE9BQU8sQ0FBQyxLQUFLd0IsU0FBUyxFQUFFO1lBQzdDLElBQUksSUFBSSxDQUFDeEMsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN4QixPQUFPLEVBQUUsSUFBSSxDQUFDVSxLQUFLLENBQUNjLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQ2hCLFFBQVEsQ0FBQ2dCLE9BQU8sQ0FBQyxDQUFDeEIsT0FBTyxDQUFDLElBQUksQ0FBQ0gsTUFBTSxFQUFFOEMsS0FBSyxDQUFDMUMsS0FBSyxFQUFFMEMsS0FBSyxDQUFDekMsTUFBTSxDQUFDLENBQUMsS0FDNUgsSUFBSSxDQUFDUSxLQUFLLENBQUNjLE9BQU8sQ0FBQyxHQUFHbUIsS0FBSyxDQUFDMUMsS0FBSyxHQUFHLEtBQUssR0FBRyxJQUFJLENBQUE7RUFDdkQsU0FBQyxNQUFNO1lBQ0wsSUFBSSxDQUFDUyxLQUFLLENBQUNjLE9BQU8sQ0FBQyxHQUFHbUIsS0FBSyxDQUFDSSxZQUFZLENBQUN2QixPQUFPLENBQUMsQ0FBQTtFQUNuRCxTQUFBO1VBRUEsSUFBSSxJQUFJLENBQUNkLEtBQUssQ0FBQ2MsT0FBTyxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQ2hDLElBQUksQ0FBQ2YsT0FBTyxDQUFDZSxPQUFPLENBQUMsQ0FBQ0QsU0FBUyxHQUFHLHdDQUF3QyxDQUFBO1dBQzNFLE1BQU0sSUFBSSxJQUFJLENBQUNiLEtBQUssQ0FBQ2MsT0FBTyxDQUFDLEtBQUssS0FBSyxFQUFFO1lBQ3hDLElBQUksQ0FBQ2YsT0FBTyxDQUFDZSxPQUFPLENBQUMsQ0FBQ0QsU0FBUyxHQUFHLDBDQUEwQyxDQUFBO0VBQzlFLFNBQUMsTUFBTTtZQUNMLElBQUksQ0FBQ2QsT0FBTyxDQUFDZSxPQUFPLENBQUMsQ0FBQ0QsU0FBUyxHQUFJLDBDQUEwQyxDQUFBO0VBQy9FLFNBQUE7RUFDRixPQUFBO0VBQ0YsS0FBQTtFQUNGLEdBQUE7SUFFQUosYUFBYUEsQ0FBQ3dCLEtBQUssRUFBRTtNQUNuQk0sS0FBSyxFQUFFLEtBQUssSUFBSXJELE1BQU0sSUFBSSxJQUFJLENBQUNlLE9BQU8sRUFBRTtRQUN0QyxJQUFLZixNQUFNLENBQUMwQyxHQUFHLElBQUlLLEtBQUssQ0FBQ0wsR0FBRyxDQUFDQyxXQUFXLEVBQUUsSUFBSTNDLE1BQU0sQ0FBQzBDLEdBQUcsSUFBTTFDLE1BQU0sQ0FBQ25CLElBQUksSUFBSWtFLEtBQUssQ0FBQ2xFLElBQUksSUFBSW1CLE1BQU0sQ0FBQ25CLElBQUssRUFBRTtFQUN2RztFQUNBLFFBQUEsS0FBSyxJQUFJeUUsUUFBUSxJQUFJdEQsTUFBTSxDQUFDb0MsU0FBUyxFQUFFO0VBQ3JDLFVBQUEsSUFBSSxDQUFDVyxLQUFLLENBQUNPLFFBQVEsQ0FBQyxFQUFFLFNBQVNELEtBQUssQ0FBQTtFQUN0QyxTQUFBO0VBQ0E7VUFDQSxJQUFJLENBQUNQLFdBQVcsQ0FBQzlDLE1BQU0sQ0FBQzRCLE9BQU8sRUFBRW1CLEtBQUssQ0FBQyxDQUFBO0VBQ3ZDLFFBQUEsT0FBQTtFQUNGLE9BQUE7RUFDRixLQUFBO0VBQ0YsR0FBQTtFQUNGOzs7O0VDNVBBLElBQUlRLEtBQUssR0FBRyxVQUFVQyxFQUFFLEVBQUU7SUFDeEIsT0FBT0EsRUFBRSxJQUFJQSxFQUFFLENBQUNDLElBQUksS0FBS0EsSUFBSSxJQUFJRCxFQUFFLENBQUE7RUFDckMsQ0FBQyxDQUFBOztFQUVEO01BQ0FFLFFBQWM7RUFDWjtFQUNBSCxLQUFLLENBQUMsT0FBT0ksVUFBVSxJQUFJLFFBQVEsSUFBSUEsVUFBVSxDQUFDLElBQ2xESixLQUFLLENBQUMsT0FBT0ssTUFBTSxJQUFJLFFBQVEsSUFBSUEsTUFBTSxDQUFDO0VBQzFDO0VBQ0FMLEtBQUssQ0FBQyxPQUFPTSxJQUFJLElBQUksUUFBUSxJQUFJQSxJQUFJLENBQUMsSUFDdENOLEtBQUssQ0FBQyxPQUFPTyxjQUFNLElBQUksUUFBUSxJQUFJQSxjQUFNLENBQUM7RUFDMUM7RUFDQyxZQUFZO0VBQUUsRUFBQSxPQUFPLElBQUksQ0FBQTtFQUFFLENBQUMsRUFBRyxJQUFJQyxjQUFJLElBQUlDLFFBQVEsQ0FBQyxhQUFhLENBQUMsRUFBRTs7TUNidkVOLE9BQWMsR0FBRyxVQUFVTyxJQUFJLEVBQUU7SUFDL0IsSUFBSTtFQUNGLElBQUEsT0FBTyxDQUFDLENBQUNBLElBQUksRUFBRSxDQUFBO0tBQ2hCLENBQUMsT0FBT0MsS0FBSyxFQUFFO0VBQ2QsSUFBQSxPQUFPLElBQUksQ0FBQTtFQUNiLEdBQUE7RUFDRixDQUFDOztFQ05ELElBQUlDLE9BQUssR0FBR0MsT0FBNkIsQ0FBQTs7RUFFekM7RUFDQVYsSUFBQUEsV0FBYyxHQUFHLENBQUNTLE9BQUssQ0FBQyxZQUFZO0VBQ2xDO0lBQ0EsT0FBT25DLE1BQU0sQ0FBQ3FDLGNBQWMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxFQUFFO01BQUVDLEdBQUcsRUFBRSxZQUFZO0VBQUUsTUFBQSxPQUFPLENBQUMsQ0FBQTtFQUFFLEtBQUE7RUFBRSxHQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUE7RUFDbEYsQ0FBQyxDQUFDOzs7Ozs7OztFQ05GLElBQUlILE9BQUssR0FBR0MsT0FBNkIsQ0FBQTtFQUV6Q1YsSUFBQUEsa0JBQWMsR0FBRyxDQUFDUyxPQUFLLENBQUMsWUFBWTtFQUNsQztJQUNBLElBQUkzRSxJQUFJLEdBQUksWUFBWSxhQUFlLENBQUUrRSxJQUFJLEVBQUUsQ0FBQTtFQUMvQztJQUNBLE9BQU8sT0FBTy9FLElBQUksSUFBSSxVQUFVLElBQUlBLElBQUksQ0FBQ2dGLGNBQWMsQ0FBQyxXQUFXLENBQUMsQ0FBQTtFQUN0RSxDQUFDLENBQUM7O0VDUEYsSUFBSUMsYUFBVyxHQUFHTCxrQkFBNEMsQ0FBQTtFQUU5RCxJQUFJTSxtQkFBaUIsR0FBR1YsUUFBUSxDQUFDVyxTQUFTLENBQUE7RUFDMUMsSUFBSUMsTUFBSSxHQUFHRixtQkFBaUIsQ0FBQ0UsSUFBSSxDQUFBO0VBQ2pDLElBQUlDLG1CQUFtQixHQUFHSixhQUFXLElBQUlDLG1CQUFpQixDQUFDSCxJQUFJLENBQUNBLElBQUksQ0FBQ0ssTUFBSSxFQUFFQSxNQUFJLENBQUMsQ0FBQTtFQUVoRmxCLElBQUFBLG1CQUFjLEdBQUdlLGFBQVcsR0FBR0ksbUJBQW1CLEdBQUcsVUFBVUMsRUFBRSxFQUFFO0VBQ2pFLEVBQUEsT0FBTyxZQUFZO0VBQ2pCLElBQUEsT0FBT0YsTUFBSSxDQUFDRyxLQUFLLENBQUNELEVBQUUsRUFBRUUsU0FBUyxDQUFDLENBQUE7S0FDakMsQ0FBQTtFQUNILENBQUM7O0VDVkQsSUFBSUMsYUFBVyxHQUFHLE9BQU8vRCxRQUFRLElBQUksUUFBUSxJQUFJQSxRQUFRLENBQUNnRSxHQUFHLENBQUE7O0VBRTdEO0VBQ0E7RUFDQSxJQUFJQyxVQUFVLEdBQUcsT0FBT0YsYUFBVyxJQUFJLFdBQVcsSUFBSUEsYUFBVyxLQUFLN0IsU0FBUyxDQUFBO0VBRS9FTSxJQUFBQSxhQUFjLEdBQUc7RUFDZndCLEVBQUFBLEdBQUcsRUFBRUQsYUFBVztFQUNoQkUsRUFBQUEsVUFBVSxFQUFFQSxVQUFBQTtFQUNkLENBQUM7O0VDVEQsSUFBSUMsY0FBWSxHQUFHaEIsYUFBb0MsQ0FBQTtFQUV2RCxJQUFJYSxhQUFXLEdBQUdHLGNBQVksQ0FBQ0YsR0FBRyxDQUFBOztFQUVsQztFQUNBO0VBQ0F4QixJQUFBQSxZQUFjLEdBQUcwQixjQUFZLENBQUNELFVBQVUsR0FBRyxVQUFVRSxRQUFRLEVBQUU7RUFDN0QsRUFBQSxPQUFPLE9BQU9BLFFBQVEsSUFBSSxVQUFVLElBQUlBLFFBQVEsS0FBS0osYUFBVyxDQUFBO0VBQ2xFLENBQUMsR0FBRyxVQUFVSSxRQUFRLEVBQUU7SUFDdEIsT0FBTyxPQUFPQSxRQUFRLElBQUksVUFBVSxDQUFBO0VBQ3RDLENBQUM7O0VDVkQ7RUFDQTtNQUNBM0IsbUJBQWMsR0FBRyxVQUFVRixFQUFFLEVBQUU7RUFDN0IsRUFBQSxPQUFPQSxFQUFFLEtBQUssSUFBSSxJQUFJQSxFQUFFLEtBQUtKLFNBQVMsQ0FBQTtFQUN4QyxDQUFDOztFQ0pELElBQUlrQyxtQkFBaUIsR0FBR2xCLG1CQUE0QyxDQUFBO0VBRXBFLElBQUltQixZQUFVLEdBQUdDLFNBQVMsQ0FBQTs7RUFFMUI7RUFDQTtNQUNBOUIsd0JBQWMsR0FBRyxVQUFVRixFQUFFLEVBQUU7RUFDN0IsRUFBQSxJQUFJOEIsbUJBQWlCLENBQUM5QixFQUFFLENBQUMsRUFBRSxNQUFNLElBQUkrQixZQUFVLENBQUMsdUJBQXVCLEdBQUcvQixFQUFFLENBQUMsQ0FBQTtFQUM3RSxFQUFBLE9BQU9BLEVBQUUsQ0FBQTtFQUNYLENBQUM7O0VDVEQsSUFBSWlDLHNCQUFzQixHQUFHckIsd0JBQWdELENBQUE7RUFFN0UsSUFBSXNCLFNBQU8sR0FBRzFELE1BQU0sQ0FBQTs7RUFFcEI7RUFDQTtNQUNBMEIsVUFBYyxHQUFHLFVBQVUyQixRQUFRLEVBQUU7RUFDbkMsRUFBQSxPQUFPSyxTQUFPLENBQUNELHNCQUFzQixDQUFDSixRQUFRLENBQUMsQ0FBQyxDQUFBO0VBQ2xELENBQUM7O0VDUkQsSUFBSU0sYUFBVyxHQUFHdkIsbUJBQTZDLENBQUE7RUFDL0QsSUFBSXdCLFFBQVEsR0FBR3hCLFVBQWlDLENBQUE7RUFFaEQsSUFBSUksY0FBYyxHQUFHbUIsYUFBVyxDQUFDLEVBQUUsQ0FBQ25CLGNBQWMsQ0FBQyxDQUFBOztFQUVuRDtFQUNBO0VBQ0E7TUFDQWQsZ0JBQWMsR0FBRzFCLE1BQU0sQ0FBQzZELE1BQU0sSUFBSSxTQUFTQSxNQUFNQSxDQUFDckMsRUFBRSxFQUFFZCxHQUFHLEVBQUU7SUFDekQsT0FBTzhCLGNBQWMsQ0FBQ29CLFFBQVEsQ0FBQ3BDLEVBQUUsQ0FBQyxFQUFFZCxHQUFHLENBQUMsQ0FBQTtFQUMxQyxDQUFDOztFQ1ZELElBQUlvRCxhQUFXLEdBQUcxQixXQUFtQyxDQUFBO0VBQ3JELElBQUl5QixRQUFNLEdBQUd6QixnQkFBd0MsQ0FBQTtFQUVyRCxJQUFJTSxpQkFBaUIsR0FBR1YsUUFBUSxDQUFDVyxTQUFTLENBQUE7RUFDMUM7RUFDQSxJQUFJb0IsYUFBYSxHQUFHRCxhQUFXLElBQUk5RCxNQUFNLENBQUNnRSx3QkFBd0IsQ0FBQTtFQUVsRSxJQUFJQyxRQUFNLEdBQUdKLFFBQU0sQ0FBQ25CLGlCQUFpQixFQUFFLE1BQU0sQ0FBQyxDQUFBO0VBQzlDO0VBQ0EsSUFBSXdCLE1BQU0sR0FBR0QsUUFBTSxJQUFLLFNBQVNFLFNBQVNBLEdBQUcsYUFBZSxDQUFFdkcsSUFBSSxLQUFLLFdBQVcsQ0FBQTtFQUNsRixJQUFJd0csY0FBWSxHQUFHSCxRQUFNLEtBQUssQ0FBQ0gsYUFBVyxJQUFLQSxhQUFXLElBQUlDLGFBQWEsQ0FBQ3JCLGlCQUFpQixFQUFFLE1BQU0sQ0FBQyxDQUFDMkIsWUFBYSxDQUFDLENBQUE7RUFFckgzQyxJQUFBQSxZQUFjLEdBQUc7RUFDZnVDLEVBQUFBLE1BQU0sRUFBRUEsUUFBTTtFQUNkQyxFQUFBQSxNQUFNLEVBQUVBLE1BQU07RUFDZEUsRUFBQUEsWUFBWSxFQUFFQSxjQUFBQTtFQUNoQixDQUFDOztFQ2hCRCxJQUFJdEMsUUFBTSxHQUFHTSxRQUE4QixDQUFBOztFQUUzQztFQUNBLElBQUlDLGdCQUFjLEdBQUdyQyxNQUFNLENBQUNxQyxjQUFjLENBQUE7RUFFMUNYLElBQUFBLHNCQUFjLEdBQUcsVUFBVWhCLEdBQUcsRUFBRTRELEtBQUssRUFBRTtJQUNyQyxJQUFJO0VBQ0ZqQyxJQUFBQSxnQkFBYyxDQUFDUCxRQUFNLEVBQUVwQixHQUFHLEVBQUU7RUFBRTRELE1BQUFBLEtBQUssRUFBRUEsS0FBSztFQUFFRCxNQUFBQSxZQUFZLEVBQUUsSUFBSTtFQUFFRSxNQUFBQSxRQUFRLEVBQUUsSUFBQTtFQUFLLEtBQUMsQ0FBQyxDQUFBO0tBQ2xGLENBQUMsT0FBT3JDLEtBQUssRUFBRTtFQUNkSixJQUFBQSxRQUFNLENBQUNwQixHQUFHLENBQUMsR0FBRzRELEtBQUssQ0FBQTtFQUNyQixHQUFBO0VBQUUsRUFBQSxPQUFPQSxLQUFLLENBQUE7RUFDaEIsQ0FBQzs7RUNYRCxJQUFJeEMsUUFBTSxHQUFHTSxRQUE4QixDQUFBO0VBQzNDLElBQUlvQyxvQkFBb0IsR0FBR3BDLHNCQUE4QyxDQUFBO0VBRXpFLElBQUlxQyxNQUFNLEdBQUcsb0JBQW9CLENBQUE7RUFDakMsSUFBSUMsT0FBSyxHQUFHNUMsUUFBTSxDQUFDMkMsTUFBTSxDQUFDLElBQUlELG9CQUFvQixDQUFDQyxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUE7RUFFOUQvQyxJQUFBQSxXQUFjLEdBQUdnRCxPQUFLOztFQ050QixJQUFJZixhQUFXLEdBQUd2QixtQkFBNkMsQ0FBQTtFQUMvRCxJQUFJdUMsWUFBVSxHQUFHdkMsWUFBbUMsQ0FBQTtFQUNwRCxJQUFJc0MsT0FBSyxHQUFHdEMsV0FBb0MsQ0FBQTtFQUVoRCxJQUFJd0MsZ0JBQWdCLEdBQUdqQixhQUFXLENBQUMzQixRQUFRLENBQUM2QyxRQUFRLENBQUMsQ0FBQTs7RUFFckQ7RUFDQSxJQUFJLENBQUNGLFlBQVUsQ0FBQ0QsT0FBSyxDQUFDSSxhQUFhLENBQUMsRUFBRTtFQUNwQ0osRUFBQUEsT0FBSyxDQUFDSSxhQUFhLEdBQUcsVUFBVXRELEVBQUUsRUFBRTtNQUNsQyxPQUFPb0QsZ0JBQWdCLENBQUNwRCxFQUFFLENBQUMsQ0FBQTtLQUM1QixDQUFBO0VBQ0gsQ0FBQTtNQUVBRSxlQUFjLEdBQUdnRCxPQUFLLENBQUNJLGFBQWE7O0VDYnBDLElBQUloRCxRQUFNLEdBQUdNLFFBQThCLENBQUE7RUFDM0MsSUFBSXVDLFlBQVUsR0FBR3ZDLFlBQW1DLENBQUE7RUFFcEQsSUFBSTJDLFNBQU8sR0FBR2pELFFBQU0sQ0FBQ2lELE9BQU8sQ0FBQTtFQUU1QnJELElBQUFBLHFCQUFjLEdBQUdpRCxZQUFVLENBQUNJLFNBQU8sQ0FBQyxJQUFJLGFBQWEsQ0FBQ3ZILElBQUksQ0FBQ3dILE1BQU0sQ0FBQ0QsU0FBTyxDQUFDLENBQUM7O0VDTDNFLElBQUlKLFlBQVUsR0FBR3ZDLFlBQW1DLENBQUE7RUFDcEQsSUFBSWdCLFlBQVksR0FBR2hCLGFBQW9DLENBQUE7RUFFdkQsSUFBSWEsV0FBVyxHQUFHRyxZQUFZLENBQUNGLEdBQUcsQ0FBQTtFQUVsQ3hCLElBQUFBLFVBQWMsR0FBRzBCLFlBQVksQ0FBQ0QsVUFBVSxHQUFHLFVBQVUzQixFQUFFLEVBQUU7RUFDdkQsRUFBQSxPQUFPLE9BQU9BLEVBQUUsSUFBSSxRQUFRLEdBQUdBLEVBQUUsS0FBSyxJQUFJLEdBQUdtRCxZQUFVLENBQUNuRCxFQUFFLENBQUMsSUFBSUEsRUFBRSxLQUFLeUIsV0FBVyxDQUFBO0VBQ25GLENBQUMsR0FBRyxVQUFVekIsRUFBRSxFQUFFO0VBQ2hCLEVBQUEsT0FBTyxPQUFPQSxFQUFFLElBQUksUUFBUSxHQUFHQSxFQUFFLEtBQUssSUFBSSxHQUFHbUQsWUFBVSxDQUFDbkQsRUFBRSxDQUFDLENBQUE7RUFDN0QsQ0FBQzs7OztFQ1RELElBQUlNLFFBQU0sR0FBR00sUUFBOEIsQ0FBQTtFQUMzQyxJQUFJNkMsVUFBUSxHQUFHN0MsVUFBaUMsQ0FBQTtFQUVoRCxJQUFJbEQsVUFBUSxHQUFHNEMsUUFBTSxDQUFDNUMsUUFBUSxDQUFBO0VBQzlCO0VBQ0EsSUFBSStFLE1BQU0sR0FBR2dCLFVBQVEsQ0FBQy9GLFVBQVEsQ0FBQyxJQUFJK0YsVUFBUSxDQUFDL0YsVUFBUSxDQUFDUSxhQUFhLENBQUMsQ0FBQTtNQUVuRWdDLHFCQUFjLEdBQUcsVUFBVUYsRUFBRSxFQUFFO0lBQzdCLE9BQU95QyxNQUFNLEdBQUcvRSxVQUFRLENBQUNRLGFBQWEsQ0FBQzhCLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQTtFQUNqRCxDQUFDOztFQ1RELElBQUlzQyxhQUFXLEdBQUcxQixXQUFtQyxDQUFBO0VBQ3JELElBQUlELE9BQUssR0FBR0MsT0FBNkIsQ0FBQTtFQUN6QyxJQUFJMUMsYUFBYSxHQUFHMEMscUJBQStDLENBQUE7O0VBRW5FO0VBQ0FWLElBQUFBLFlBQWMsR0FBRyxDQUFDb0MsYUFBVyxJQUFJLENBQUMzQixPQUFLLENBQUMsWUFBWTtFQUNsRDtJQUNBLE9BQU9uQyxNQUFNLENBQUNxQyxjQUFjLENBQUMzQyxhQUFhLENBQUMsS0FBSyxDQUFDLEVBQUUsR0FBRyxFQUFFO01BQ3RENEMsR0FBRyxFQUFFLFlBQVk7RUFBRSxNQUFBLE9BQU8sQ0FBQyxDQUFBO0VBQUUsS0FBQTtFQUMvQixHQUFDLENBQUMsQ0FBQzRDLENBQUMsS0FBSyxDQUFDLENBQUE7RUFDWixDQUFDLENBQUM7O0VDVkYsSUFBSXBCLGFBQVcsR0FBRzFCLFdBQW1DLENBQUE7RUFDckQsSUFBSUQsT0FBSyxHQUFHQyxPQUE2QixDQUFBOztFQUV6QztFQUNBO0VBQ0FWLElBQUFBLG9CQUFjLEdBQUdvQyxhQUFXLElBQUkzQixPQUFLLENBQUMsWUFBWTtFQUNoRDtJQUNBLE9BQU9uQyxNQUFNLENBQUNxQyxjQUFjLENBQUMsWUFBWSxhQUFlLEVBQUUsV0FBVyxFQUFFO0VBQ3JFaUMsSUFBQUEsS0FBSyxFQUFFLEVBQUU7RUFDVEMsSUFBQUEsUUFBUSxFQUFFLEtBQUE7RUFDWixHQUFDLENBQUMsQ0FBQzVCLFNBQVMsS0FBSyxFQUFFLENBQUE7RUFDckIsQ0FBQyxDQUFDOztFQ1hGLElBQUlzQyxVQUFRLEdBQUc3QyxVQUFpQyxDQUFBO0VBRWhELElBQUkrQyxTQUFPLEdBQUdILE1BQU0sQ0FBQTtFQUNwQixJQUFJekIsWUFBVSxHQUFHQyxTQUFTLENBQUE7O0VBRTFCO01BQ0E5QixVQUFjLEdBQUcsVUFBVTJCLFFBQVEsRUFBRTtFQUNuQyxFQUFBLElBQUk0QixVQUFRLENBQUM1QixRQUFRLENBQUMsRUFBRSxPQUFPQSxRQUFRLENBQUE7SUFDdkMsTUFBTSxJQUFJRSxZQUFVLENBQUM0QixTQUFPLENBQUM5QixRQUFRLENBQUMsR0FBRyxtQkFBbUIsQ0FBQyxDQUFBO0VBQy9ELENBQUM7O0VDVEQsSUFBSVosV0FBVyxHQUFHTCxrQkFBNEMsQ0FBQTtFQUU5RCxJQUFJUSxNQUFJLEdBQUdaLFFBQVEsQ0FBQ1csU0FBUyxDQUFDQyxJQUFJLENBQUE7TUFFbENsQixZQUFjLEdBQUdlLFdBQVcsR0FBR0csTUFBSSxDQUFDTCxJQUFJLENBQUNLLE1BQUksQ0FBQyxHQUFHLFlBQVk7RUFDM0QsRUFBQSxPQUFPQSxNQUFJLENBQUNHLEtBQUssQ0FBQ0gsTUFBSSxFQUFFSSxTQUFTLENBQUMsQ0FBQTtFQUNwQyxDQUFDOztFQ05ELElBQUlsQixRQUFNLEdBQUdNLFFBQThCLENBQUE7RUFDM0MsSUFBSXVDLFlBQVUsR0FBR3ZDLFlBQW1DLENBQUE7RUFFcEQsSUFBSWdELFNBQVMsR0FBRyxVQUFVL0IsUUFBUSxFQUFFO0VBQ2xDLEVBQUEsT0FBT3NCLFlBQVUsQ0FBQ3RCLFFBQVEsQ0FBQyxHQUFHQSxRQUFRLEdBQUdqQyxTQUFTLENBQUE7RUFDcEQsQ0FBQyxDQUFBO0VBRURNLElBQUFBLFlBQWMsR0FBRyxVQUFVMkQsU0FBUyxFQUFFQyxNQUFNLEVBQUU7SUFDNUMsT0FBT3RDLFNBQVMsQ0FBQ3pDLE1BQU0sR0FBRyxDQUFDLEdBQUc2RSxTQUFTLENBQUN0RCxRQUFNLENBQUN1RCxTQUFTLENBQUMsQ0FBQyxHQUFHdkQsUUFBTSxDQUFDdUQsU0FBUyxDQUFDLElBQUl2RCxRQUFNLENBQUN1RCxTQUFTLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUE7RUFDN0csQ0FBQzs7RUNURCxJQUFJM0IsYUFBVyxHQUFHdkIsbUJBQTZDLENBQUE7RUFFL0RWLElBQUFBLG1CQUFjLEdBQUdpQyxhQUFXLENBQUMsRUFBRSxDQUFDNEIsYUFBYSxDQUFDOztFQ0Y5QzdELElBQUFBLGVBQWMsR0FBRyxPQUFPakUsU0FBUyxJQUFJLFdBQVcsSUFBSXVILE1BQU0sQ0FBQ3ZILFNBQVMsQ0FBQytILFNBQVMsQ0FBQyxJQUFJLEVBQUU7O0VDQXJGLElBQUkxRCxRQUFNLEdBQUdNLFFBQThCLENBQUE7RUFDM0MsSUFBSW9ELFNBQVMsR0FBR3BELGVBQXlDLENBQUE7RUFFekQsSUFBSXFELE9BQU8sR0FBRzNELFFBQU0sQ0FBQzJELE9BQU8sQ0FBQTtFQUM1QixJQUFJQyxJQUFJLEdBQUc1RCxRQUFNLENBQUM0RCxJQUFJLENBQUE7RUFDdEIsSUFBSUMsUUFBUSxHQUFHRixPQUFPLElBQUlBLE9BQU8sQ0FBQ0UsUUFBUSxJQUFJRCxJQUFJLElBQUlBLElBQUksQ0FBQ0UsT0FBTyxDQUFBO0VBQ2xFLElBQUlDLEVBQUUsR0FBR0YsUUFBUSxJQUFJQSxRQUFRLENBQUNFLEVBQUUsQ0FBQTtFQUNoQyxJQUFJcEYsS0FBSyxFQUFFbUYsT0FBTyxDQUFBO0VBRWxCLElBQUlDLEVBQUUsRUFBRTtFQUNOcEYsRUFBQUEsS0FBSyxHQUFHb0YsRUFBRSxDQUFDMUYsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFBO0VBQ3JCO0VBQ0E7SUFDQXlGLE9BQU8sR0FBR25GLEtBQUssQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUlBLEtBQUssQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUVBLEtBQUssQ0FBQyxDQUFDLENBQUMsR0FBR0EsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDckUsQ0FBQTs7RUFFQTtFQUNBO0VBQ0EsSUFBSSxDQUFDbUYsT0FBTyxJQUFJSixTQUFTLEVBQUU7RUFDekIvRSxFQUFBQSxLQUFLLEdBQUcrRSxTQUFTLENBQUMvRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUE7SUFDdEMsSUFBSSxDQUFDQSxLQUFLLElBQUlBLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7RUFDNUJBLElBQUFBLEtBQUssR0FBRytFLFNBQVMsQ0FBQy9FLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQTtNQUN4QyxJQUFJQSxLQUFLLEVBQUVtRixPQUFPLEdBQUcsQ0FBQ25GLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUNoQyxHQUFBO0VBQ0YsQ0FBQTtFQUVBaUIsSUFBQUEsZUFBYyxHQUFHa0UsT0FBTzs7RUMxQnhCO0VBQ0EsSUFBSUUsVUFBVSxHQUFHMUQsZUFBeUMsQ0FBQTtFQUMxRCxJQUFJRCxPQUFLLEdBQUdDLE9BQTZCLENBQUE7RUFDekMsSUFBSU4sUUFBTSxHQUFHTSxRQUE4QixDQUFBO0VBRTNDLElBQUkrQyxTQUFPLEdBQUdyRCxRQUFNLENBQUNrRCxNQUFNLENBQUE7O0VBRTNCO01BQ0F0RCwwQkFBYyxHQUFHLENBQUMsQ0FBQzFCLE1BQU0sQ0FBQytGLHFCQUFxQixJQUFJLENBQUM1RCxPQUFLLENBQUMsWUFBWTtFQUNwRSxFQUFBLElBQUk2RCxNQUFNLEdBQUdDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFBO0VBQ3ZDO0VBQ0E7RUFDQTtFQUNBO0VBQ0EsRUFBQSxPQUFPLENBQUNkLFNBQU8sQ0FBQ2EsTUFBTSxDQUFDLElBQUksRUFBRWhHLE1BQU0sQ0FBQ2dHLE1BQU0sQ0FBQyxZQUFZQyxNQUFNLENBQUM7RUFDNUQ7SUFDQSxDQUFDQSxNQUFNLENBQUNDLElBQUksSUFBSUosVUFBVSxJQUFJQSxVQUFVLEdBQUcsRUFBRSxDQUFBO0VBQ2pELENBQUMsQ0FBQzs7RUNqQkY7RUFDQSxJQUFJSyxlQUFhLEdBQUcvRCwwQkFBb0QsQ0FBQTtFQUV4RVYsSUFBQUEsY0FBYyxHQUFHeUUsZUFBYSxJQUN6QixDQUFDRixNQUFNLENBQUNDLElBQUksSUFDWixPQUFPRCxNQUFNLENBQUNHLFFBQVEsSUFBSSxRQUFROztFQ0x2QyxJQUFJQyxVQUFVLEdBQUdqRSxZQUFvQyxDQUFBO0VBQ3JELElBQUl1QyxZQUFVLEdBQUd2QyxZQUFtQyxDQUFBO0VBQ3BELElBQUltRCxhQUFhLEdBQUduRCxtQkFBOEMsQ0FBQTtFQUNsRSxJQUFJa0UsbUJBQWlCLEdBQUdsRSxjQUF5QyxDQUFBO0VBRWpFLElBQUlzQixPQUFPLEdBQUcxRCxNQUFNLENBQUE7RUFFcEIwQixJQUFBQSxVQUFjLEdBQUc0RSxtQkFBaUIsR0FBRyxVQUFVOUUsRUFBRSxFQUFFO0lBQ2pELE9BQU8sT0FBT0EsRUFBRSxJQUFJLFFBQVEsQ0FBQTtFQUM5QixDQUFDLEdBQUcsVUFBVUEsRUFBRSxFQUFFO0VBQ2hCLEVBQUEsSUFBSStFLE9BQU8sR0FBR0YsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFBO0VBQ2xDLEVBQUEsT0FBTzFCLFlBQVUsQ0FBQzRCLE9BQU8sQ0FBQyxJQUFJaEIsYUFBYSxDQUFDZ0IsT0FBTyxDQUFDNUQsU0FBUyxFQUFFZSxPQUFPLENBQUNsQyxFQUFFLENBQUMsQ0FBQyxDQUFBO0VBQzdFLENBQUM7O0VDWkQsSUFBSTJELFNBQU8sR0FBR0gsTUFBTSxDQUFBO01BRXBCdEQsYUFBYyxHQUFHLFVBQVUyQixRQUFRLEVBQUU7SUFDbkMsSUFBSTtNQUNGLE9BQU84QixTQUFPLENBQUM5QixRQUFRLENBQUMsQ0FBQTtLQUN6QixDQUFDLE9BQU9uQixLQUFLLEVBQUU7RUFDZCxJQUFBLE9BQU8sUUFBUSxDQUFBO0VBQ2pCLEdBQUE7RUFDRixDQUFDOztFQ1JELElBQUl5QyxZQUFVLEdBQUd2QyxZQUFtQyxDQUFBO0VBQ3BELElBQUlvRSxXQUFXLEdBQUdwRSxhQUFxQyxDQUFBO0VBRXZELElBQUltQixZQUFVLEdBQUdDLFNBQVMsQ0FBQTs7RUFFMUI7TUFDQTlCLFdBQWMsR0FBRyxVQUFVMkIsUUFBUSxFQUFFO0VBQ25DLEVBQUEsSUFBSXNCLFlBQVUsQ0FBQ3RCLFFBQVEsQ0FBQyxFQUFFLE9BQU9BLFFBQVEsQ0FBQTtJQUN6QyxNQUFNLElBQUlFLFlBQVUsQ0FBQ2lELFdBQVcsQ0FBQ25ELFFBQVEsQ0FBQyxHQUFHLG9CQUFvQixDQUFDLENBQUE7RUFDcEUsQ0FBQzs7RUNURCxJQUFJb0QsU0FBUyxHQUFHckUsV0FBa0MsQ0FBQTtFQUNsRCxJQUFJa0IsaUJBQWlCLEdBQUdsQixtQkFBNEMsQ0FBQTs7RUFFcEU7RUFDQTtFQUNBVixJQUFBQSxXQUFjLEdBQUcsVUFBVWdGLENBQUMsRUFBRUMsQ0FBQyxFQUFFO0VBQy9CLEVBQUEsSUFBSUMsSUFBSSxHQUFHRixDQUFDLENBQUNDLENBQUMsQ0FBQyxDQUFBO0lBQ2YsT0FBT3JELGlCQUFpQixDQUFDc0QsSUFBSSxDQUFDLEdBQUd4RixTQUFTLEdBQUdxRixTQUFTLENBQUNHLElBQUksQ0FBQyxDQUFBO0VBQzlELENBQUM7O0VDUkQsSUFBSWhFLE1BQUksR0FBR1IsWUFBcUMsQ0FBQTtFQUNoRCxJQUFJdUMsWUFBVSxHQUFHdkMsWUFBbUMsQ0FBQTtFQUNwRCxJQUFJNkMsVUFBUSxHQUFHN0MsVUFBaUMsQ0FBQTtFQUVoRCxJQUFJbUIsWUFBVSxHQUFHQyxTQUFTLENBQUE7O0VBRTFCO0VBQ0E7RUFDQTlCLElBQUFBLHFCQUFjLEdBQUcsVUFBVW1GLEtBQUssRUFBRUMsSUFBSSxFQUFFO0lBQ3RDLElBQUloRSxFQUFFLEVBQUVpRSxHQUFHLENBQUE7SUFDWCxJQUFJRCxJQUFJLEtBQUssUUFBUSxJQUFJbkMsWUFBVSxDQUFDN0IsRUFBRSxHQUFHK0QsS0FBSyxDQUFDaEMsUUFBUSxDQUFDLElBQUksQ0FBQ0ksVUFBUSxDQUFDOEIsR0FBRyxHQUFHbkUsTUFBSSxDQUFDRSxFQUFFLEVBQUUrRCxLQUFLLENBQUMsQ0FBQyxFQUFFLE9BQU9FLEdBQUcsQ0FBQTtJQUN4RyxJQUFJcEMsWUFBVSxDQUFDN0IsRUFBRSxHQUFHK0QsS0FBSyxDQUFDRyxPQUFPLENBQUMsSUFBSSxDQUFDL0IsVUFBUSxDQUFDOEIsR0FBRyxHQUFHbkUsTUFBSSxDQUFDRSxFQUFFLEVBQUUrRCxLQUFLLENBQUMsQ0FBQyxFQUFFLE9BQU9FLEdBQUcsQ0FBQTtJQUNsRixJQUFJRCxJQUFJLEtBQUssUUFBUSxJQUFJbkMsWUFBVSxDQUFDN0IsRUFBRSxHQUFHK0QsS0FBSyxDQUFDaEMsUUFBUSxDQUFDLElBQUksQ0FBQ0ksVUFBUSxDQUFDOEIsR0FBRyxHQUFHbkUsTUFBSSxDQUFDRSxFQUFFLEVBQUUrRCxLQUFLLENBQUMsQ0FBQyxFQUFFLE9BQU9FLEdBQUcsQ0FBQTtFQUN4RyxFQUFBLE1BQU0sSUFBSXhELFlBQVUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFBO0VBQ2pFLENBQUM7Ozs7Ozs7O0VDYkQsSUFBSW1CLE9BQUssR0FBR3RDLFdBQW9DLENBQUE7RUFFaEQsQ0FBQ1YsZ0JBQWMsR0FBRyxVQUFVaEIsR0FBRyxFQUFFNEQsS0FBSyxFQUFFO0VBQ3RDLEVBQUEsT0FBT0ksT0FBSyxDQUFDaEUsR0FBRyxDQUFDLEtBQUtnRSxPQUFLLENBQUNoRSxHQUFHLENBQUMsR0FBRzRELEtBQUssS0FBS2xELFNBQVMsR0FBR2tELEtBQUssR0FBRyxFQUFFLENBQUMsQ0FBQTtFQUN0RSxDQUFDLEVBQUUsVUFBVSxFQUFFLEVBQUUsQ0FBQyxDQUFDOUQsSUFBSSxDQUFDO0VBQ3RCb0YsRUFBQUEsT0FBTyxFQUFFLFFBQVE7RUFDakJxQixFQUFBQSxJQUFJLEVBQXFCLFFBQVE7RUFDakNDLEVBQUFBLFNBQVMsRUFBRSwyQ0FBMkM7RUFDdERDLEVBQUFBLE9BQU8sRUFBRSwwREFBMEQ7RUFDbkVDLEVBQUFBLE1BQU0sRUFBRSxxQ0FBQTtFQUNWLENBQUMsQ0FBQzs7RUNYRixJQUFJekQsYUFBVyxHQUFHdkIsbUJBQTZDLENBQUE7RUFFL0QsSUFBSWlGLEVBQUUsR0FBRyxDQUFDLENBQUE7RUFDVixJQUFJQyxPQUFPLEdBQUc3RixJQUFJLENBQUM4RixNQUFNLEVBQUUsQ0FBQTtFQUMzQixJQUFJMUMsUUFBUSxHQUFHbEIsYUFBVyxDQUFDLEdBQUcsQ0FBQ2tCLFFBQVEsQ0FBQyxDQUFBO01BRXhDbkQsS0FBYyxHQUFHLFVBQVVoQixHQUFHLEVBQUU7SUFDOUIsT0FBTyxTQUFTLElBQUlBLEdBQUcsS0FBS1UsU0FBUyxHQUFHLEVBQUUsR0FBR1YsR0FBRyxDQUFDLEdBQUcsSUFBSSxHQUFHbUUsUUFBUSxDQUFDLEVBQUV3QyxFQUFFLEdBQUdDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQTtFQUN6RixDQUFDOztFQ1JELElBQUl4RixRQUFNLEdBQUdNLFFBQThCLENBQUE7RUFDM0MsSUFBSW9GLFFBQU0sR0FBR3BGLGFBQThCLENBQUE7RUFDM0MsSUFBSXlCLFFBQU0sR0FBR3pCLGdCQUF3QyxDQUFBO0VBQ3JELElBQUlxRixLQUFHLEdBQUdyRixLQUEyQixDQUFBO0VBQ3JDLElBQUkrRCxhQUFhLEdBQUcvRCwwQkFBb0QsQ0FBQTtFQUN4RSxJQUFJa0UsaUJBQWlCLEdBQUdsRSxjQUF5QyxDQUFBO0VBRWpFLElBQUk2RCxRQUFNLEdBQUduRSxRQUFNLENBQUNtRSxNQUFNLENBQUE7RUFDMUIsSUFBSXlCLHFCQUFxQixHQUFHRixRQUFNLENBQUMsS0FBSyxDQUFDLENBQUE7RUFDekMsSUFBSUcscUJBQXFCLEdBQUdyQixpQkFBaUIsR0FBR0wsUUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJQSxRQUFNLEdBQUdBLFFBQU0sSUFBSUEsUUFBTSxDQUFDMkIsYUFBYSxJQUFJSCxLQUFHLENBQUE7TUFFL0cvRixpQkFBYyxHQUFHLFVBQVU5RCxJQUFJLEVBQUU7RUFDL0IsRUFBQSxJQUFJLENBQUNpRyxRQUFNLENBQUM2RCxxQkFBcUIsRUFBRTlKLElBQUksQ0FBQyxFQUFFO01BQ3hDOEoscUJBQXFCLENBQUM5SixJQUFJLENBQUMsR0FBR3VJLGFBQWEsSUFBSXRDLFFBQU0sQ0FBQ29DLFFBQU0sRUFBRXJJLElBQUksQ0FBQyxHQUMvRHFJLFFBQU0sQ0FBQ3JJLElBQUksQ0FBQyxHQUNaK0oscUJBQXFCLENBQUMsU0FBUyxHQUFHL0osSUFBSSxDQUFDLENBQUE7RUFDN0MsR0FBQTtJQUFFLE9BQU84SixxQkFBcUIsQ0FBQzlKLElBQUksQ0FBQyxDQUFBO0VBQ3RDLENBQUM7O0VDakJELElBQUlnRixJQUFJLEdBQUdSLFlBQXFDLENBQUE7RUFDaEQsSUFBSTZDLFVBQVEsR0FBRzdDLFVBQWlDLENBQUE7RUFDaEQsSUFBSXlGLFVBQVEsR0FBR3pGLFVBQWlDLENBQUE7RUFDaEQsSUFBSTBGLFNBQVMsR0FBRzFGLFdBQWtDLENBQUE7RUFDbEQsSUFBSTJGLG1CQUFtQixHQUFHM0YscUJBQTZDLENBQUE7RUFDdkUsSUFBSTRGLGVBQWUsR0FBRzVGLGlCQUF5QyxDQUFBO0VBRS9ELElBQUltQixZQUFVLEdBQUdDLFNBQVMsQ0FBQTtFQUMxQixJQUFJeUUsWUFBWSxHQUFHRCxlQUFlLENBQUMsYUFBYSxDQUFDLENBQUE7O0VBRWpEO0VBQ0E7RUFDQXRHLElBQUFBLGFBQWMsR0FBRyxVQUFVbUYsS0FBSyxFQUFFQyxJQUFJLEVBQUU7RUFDdEMsRUFBQSxJQUFJLENBQUM3QixVQUFRLENBQUM0QixLQUFLLENBQUMsSUFBSWdCLFVBQVEsQ0FBQ2hCLEtBQUssQ0FBQyxFQUFFLE9BQU9BLEtBQUssQ0FBQTtFQUNyRCxFQUFBLElBQUlxQixZQUFZLEdBQUdKLFNBQVMsQ0FBQ2pCLEtBQUssRUFBRW9CLFlBQVksQ0FBQyxDQUFBO0VBQ2pELEVBQUEsSUFBSUUsTUFBTSxDQUFBO0VBQ1YsRUFBQSxJQUFJRCxZQUFZLEVBQUU7RUFDaEIsSUFBQSxJQUFJcEIsSUFBSSxLQUFLMUYsU0FBUyxFQUFFMEYsSUFBSSxHQUFHLFNBQVMsQ0FBQTtNQUN4Q3FCLE1BQU0sR0FBR3ZGLElBQUksQ0FBQ3NGLFlBQVksRUFBRXJCLEtBQUssRUFBRUMsSUFBSSxDQUFDLENBQUE7RUFDeEMsSUFBQSxJQUFJLENBQUM3QixVQUFRLENBQUNrRCxNQUFNLENBQUMsSUFBSU4sVUFBUSxDQUFDTSxNQUFNLENBQUMsRUFBRSxPQUFPQSxNQUFNLENBQUE7RUFDeEQsSUFBQSxNQUFNLElBQUk1RSxZQUFVLENBQUMseUNBQXlDLENBQUMsQ0FBQTtFQUNqRSxHQUFBO0VBQ0EsRUFBQSxJQUFJdUQsSUFBSSxLQUFLMUYsU0FBUyxFQUFFMEYsSUFBSSxHQUFHLFFBQVEsQ0FBQTtFQUN2QyxFQUFBLE9BQU9pQixtQkFBbUIsQ0FBQ2xCLEtBQUssRUFBRUMsSUFBSSxDQUFDLENBQUE7RUFDekMsQ0FBQzs7RUN4QkQsSUFBSXNCLFdBQVcsR0FBR2hHLGFBQW9DLENBQUE7RUFDdEQsSUFBSXlGLFFBQVEsR0FBR3pGLFVBQWlDLENBQUE7O0VBRWhEO0VBQ0E7TUFDQVYsZUFBYyxHQUFHLFVBQVUyQixRQUFRLEVBQUU7RUFDbkMsRUFBQSxJQUFJM0MsR0FBRyxHQUFHMEgsV0FBVyxDQUFDL0UsUUFBUSxFQUFFLFFBQVEsQ0FBQyxDQUFBO0lBQ3pDLE9BQU93RSxRQUFRLENBQUNuSCxHQUFHLENBQUMsR0FBR0EsR0FBRyxHQUFHQSxHQUFHLEdBQUcsRUFBRSxDQUFBO0VBQ3ZDLENBQUM7O0VDUkQsSUFBSW9ELGFBQVcsR0FBRzFCLFdBQW1DLENBQUE7RUFDckQsSUFBSWlHLGNBQWMsR0FBR2pHLFlBQXNDLENBQUE7RUFDM0QsSUFBSWtHLHVCQUF1QixHQUFHbEcsb0JBQStDLENBQUE7RUFDN0UsSUFBSW1HLFVBQVEsR0FBR25HLFVBQWlDLENBQUE7RUFDaEQsSUFBSW9HLGFBQWEsR0FBR3BHLGVBQXVDLENBQUE7RUFFM0QsSUFBSW1CLFVBQVUsR0FBR0MsU0FBUyxDQUFBO0VBQzFCO0VBQ0EsSUFBSWlGLGVBQWUsR0FBR3pJLE1BQU0sQ0FBQ3FDLGNBQWMsQ0FBQTtFQUMzQztFQUNBLElBQUlxRyx5QkFBeUIsR0FBRzFJLE1BQU0sQ0FBQ2dFLHdCQUF3QixDQUFBO0VBQy9ELElBQUkyRSxVQUFVLEdBQUcsWUFBWSxDQUFBO0VBQzdCLElBQUl2RSxZQUFZLEdBQUcsY0FBYyxDQUFBO0VBQ2pDLElBQUl3RSxRQUFRLEdBQUcsVUFBVSxDQUFBOztFQUV6QjtFQUNBO0VBQ0FDLG9CQUFBQSxDQUFBQSxDQUFTLEdBQUcvRSxhQUFXLEdBQUd3RSx1QkFBdUIsR0FBRyxTQUFTakcsY0FBY0EsQ0FBQ3lHLENBQUMsRUFBRW5DLENBQUMsRUFBRW9DLFVBQVUsRUFBRTtJQUM1RlIsVUFBUSxDQUFDTyxDQUFDLENBQUMsQ0FBQTtFQUNYbkMsRUFBQUEsQ0FBQyxHQUFHNkIsYUFBYSxDQUFDN0IsQ0FBQyxDQUFDLENBQUE7SUFDcEI0QixVQUFRLENBQUNRLFVBQVUsQ0FBQyxDQUFBO0lBQ3BCLElBQUksT0FBT0QsQ0FBQyxLQUFLLFVBQVUsSUFBSW5DLENBQUMsS0FBSyxXQUFXLElBQUksT0FBTyxJQUFJb0MsVUFBVSxJQUFJSCxRQUFRLElBQUlHLFVBQVUsSUFBSSxDQUFDQSxVQUFVLENBQUNILFFBQVEsQ0FBQyxFQUFFO0VBQzVILElBQUEsSUFBSUksT0FBTyxHQUFHTix5QkFBeUIsQ0FBQ0ksQ0FBQyxFQUFFbkMsQ0FBQyxDQUFDLENBQUE7RUFDN0MsSUFBQSxJQUFJcUMsT0FBTyxJQUFJQSxPQUFPLENBQUNKLFFBQVEsQ0FBQyxFQUFFO0VBQ2hDRSxNQUFBQSxDQUFDLENBQUNuQyxDQUFDLENBQUMsR0FBR29DLFVBQVUsQ0FBQ3pFLEtBQUssQ0FBQTtFQUN2QnlFLE1BQUFBLFVBQVUsR0FBRztFQUNYMUUsUUFBQUEsWUFBWSxFQUFFRCxZQUFZLElBQUkyRSxVQUFVLEdBQUdBLFVBQVUsQ0FBQzNFLFlBQVksQ0FBQyxHQUFHNEUsT0FBTyxDQUFDNUUsWUFBWSxDQUFDO0VBQzNGNkUsUUFBQUEsVUFBVSxFQUFFTixVQUFVLElBQUlJLFVBQVUsR0FBR0EsVUFBVSxDQUFDSixVQUFVLENBQUMsR0FBR0ssT0FBTyxDQUFDTCxVQUFVLENBQUM7RUFDbkZwRSxRQUFBQSxRQUFRLEVBQUUsS0FBQTtTQUNYLENBQUE7RUFDSCxLQUFBO0VBQ0YsR0FBQTtFQUFFLEVBQUEsT0FBT2tFLGVBQWUsQ0FBQ0ssQ0FBQyxFQUFFbkMsQ0FBQyxFQUFFb0MsVUFBVSxDQUFDLENBQUE7RUFDNUMsQ0FBQyxHQUFHTixlQUFlLEdBQUcsU0FBU3BHLGNBQWNBLENBQUN5RyxDQUFDLEVBQUVuQyxDQUFDLEVBQUVvQyxVQUFVLEVBQUU7SUFDOURSLFVBQVEsQ0FBQ08sQ0FBQyxDQUFDLENBQUE7RUFDWG5DLEVBQUFBLENBQUMsR0FBRzZCLGFBQWEsQ0FBQzdCLENBQUMsQ0FBQyxDQUFBO0lBQ3BCNEIsVUFBUSxDQUFDUSxVQUFVLENBQUMsQ0FBQTtJQUNwQixJQUFJVixjQUFjLEVBQUUsSUFBSTtFQUN0QixJQUFBLE9BQU9JLGVBQWUsQ0FBQ0ssQ0FBQyxFQUFFbkMsQ0FBQyxFQUFFb0MsVUFBVSxDQUFDLENBQUE7RUFDMUMsR0FBQyxDQUFDLE9BQU83RyxLQUFLLEVBQUUsYUFBRTtFQUNsQixFQUFBLElBQUksS0FBSyxJQUFJNkcsVUFBVSxJQUFJLEtBQUssSUFBSUEsVUFBVSxFQUFFLE1BQU0sSUFBSXhGLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFBO0lBQy9GLElBQUksT0FBTyxJQUFJd0YsVUFBVSxFQUFFRCxDQUFDLENBQUNuQyxDQUFDLENBQUMsR0FBR29DLFVBQVUsQ0FBQ3pFLEtBQUssQ0FBQTtFQUNsRCxFQUFBLE9BQU93RSxDQUFDLENBQUE7RUFDVjs7RUMxQ0FwSCxJQUFBQSwwQkFBYyxHQUFHLFVBQVV3SCxNQUFNLEVBQUU1RSxLQUFLLEVBQUU7SUFDeEMsT0FBTztFQUNMMkUsSUFBQUEsVUFBVSxFQUFFLEVBQUVDLE1BQU0sR0FBRyxDQUFDLENBQUM7RUFDekI3RSxJQUFBQSxZQUFZLEVBQUUsRUFBRTZFLE1BQU0sR0FBRyxDQUFDLENBQUM7RUFDM0IzRSxJQUFBQSxRQUFRLEVBQUUsRUFBRTJFLE1BQU0sR0FBRyxDQUFDLENBQUM7RUFDdkI1RSxJQUFBQSxLQUFLLEVBQUVBLEtBQUFBO0tBQ1IsQ0FBQTtFQUNILENBQUM7O0VDUEQsSUFBSVIsYUFBVyxHQUFHMUIsV0FBbUMsQ0FBQTtFQUNyRCxJQUFJK0csb0JBQW9CLEdBQUcvRyxvQkFBOEMsQ0FBQTtFQUN6RSxJQUFJZ0gsd0JBQXdCLEdBQUdoSCwwQkFBa0QsQ0FBQTtNQUVqRlYsNkJBQWMsR0FBR29DLGFBQVcsR0FBRyxVQUFVdUYsTUFBTSxFQUFFM0ksR0FBRyxFQUFFNEQsS0FBSyxFQUFFO0VBQzNELEVBQUEsT0FBTzZFLG9CQUFvQixDQUFDRyxDQUFDLENBQUNELE1BQU0sRUFBRTNJLEdBQUcsRUFBRTBJLHdCQUF3QixDQUFDLENBQUMsRUFBRTlFLEtBQUssQ0FBQyxDQUFDLENBQUE7RUFDaEYsQ0FBQyxHQUFHLFVBQVUrRSxNQUFNLEVBQUUzSSxHQUFHLEVBQUU0RCxLQUFLLEVBQUU7RUFDaEMrRSxFQUFBQSxNQUFNLENBQUMzSSxHQUFHLENBQUMsR0FBRzRELEtBQUssQ0FBQTtFQUNuQixFQUFBLE9BQU8rRSxNQUFNLENBQUE7RUFDZixDQUFDOztFQ1RELElBQUk3QixRQUFNLEdBQUdwRixhQUE4QixDQUFBO0VBQzNDLElBQUlxRixHQUFHLEdBQUdyRixLQUEyQixDQUFBO0VBRXJDLElBQUlsQyxJQUFJLEdBQUdzSCxRQUFNLENBQUMsTUFBTSxDQUFDLENBQUE7TUFFekI5RixXQUFjLEdBQUcsVUFBVWhCLEdBQUcsRUFBRTtFQUM5QixFQUFBLE9BQU9SLElBQUksQ0FBQ1EsR0FBRyxDQUFDLEtBQUtSLElBQUksQ0FBQ1EsR0FBRyxDQUFDLEdBQUcrRyxHQUFHLENBQUMvRyxHQUFHLENBQUMsQ0FBQyxDQUFBO0VBQzVDLENBQUM7O0VDUEQsSUFBSTZJLGVBQWUsR0FBR25ILHFCQUFnRCxDQUFBO0VBQ3RFLElBQUlOLFFBQU0sR0FBR00sUUFBOEIsQ0FBQTtFQUMzQyxJQUFJNkMsUUFBUSxHQUFHN0MsVUFBaUMsQ0FBQTtFQUNoRCxJQUFJb0gsMkJBQTJCLEdBQUdwSCw2QkFBc0QsQ0FBQTtFQUN4RixJQUFJeUIsUUFBTSxHQUFHekIsZ0JBQXdDLENBQUE7RUFDckQsSUFBSW9GLE1BQU0sR0FBR3BGLFdBQW9DLENBQUE7RUFDakQsSUFBSXFILFNBQVMsR0FBR3JILFdBQWtDLENBQUE7RUFHbEQsSUFBSXNILDBCQUEwQixHQUFHLDRCQUE0QixDQUFBO0VBQzdELElBQUlsRyxXQUFTLEdBQUcxQixRQUFNLENBQUMwQixTQUFTLENBQUE7RUFDaEMsSUFBSXVCLE9BQU8sR0FBR2pELFFBQU0sQ0FBQ2lELE9BQU8sQ0FBQTtFQUM1QixJQUFJNEUsR0FBRyxFQUFFckgsR0FBRyxFQUFFc0gsR0FBRyxDQUFBO0VBRWpCLElBQUlDLE9BQU8sR0FBRyxVQUFVckksRUFBRSxFQUFFO0VBQzFCLEVBQUEsT0FBT29JLEdBQUcsQ0FBQ3BJLEVBQUUsQ0FBQyxHQUFHYyxHQUFHLENBQUNkLEVBQUUsQ0FBQyxHQUFHbUksR0FBRyxDQUFDbkksRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFBO0VBQ3hDLENBQUMsQ0FBQTtFQUVELElBQUlzSSxTQUFTLEdBQUcsVUFBVUMsSUFBSSxFQUFFO0lBQzlCLE9BQU8sVUFBVXZJLEVBQUUsRUFBRTtFQUNuQixJQUFBLElBQUkxQyxLQUFLLENBQUE7RUFDVCxJQUFBLElBQUksQ0FBQ21HLFFBQVEsQ0FBQ3pELEVBQUUsQ0FBQyxJQUFJLENBQUMxQyxLQUFLLEdBQUd3RCxHQUFHLENBQUNkLEVBQUUsQ0FBQyxFQUFFd0ksSUFBSSxLQUFLRCxJQUFJLEVBQUU7UUFDcEQsTUFBTSxJQUFJdkcsV0FBUyxDQUFDLHlCQUF5QixHQUFHdUcsSUFBSSxHQUFHLFdBQVcsQ0FBQyxDQUFBO0VBQ3JFLEtBQUE7RUFBRSxJQUFBLE9BQU9qTCxLQUFLLENBQUE7S0FDZixDQUFBO0VBQ0gsQ0FBQyxDQUFBO0VBRUQsSUFBSXlLLGVBQWUsSUFBSS9CLE1BQU0sQ0FBQzFJLEtBQUssRUFBRTtFQUNuQyxFQUFBLElBQUk0RixLQUFLLEdBQUc4QyxNQUFNLENBQUMxSSxLQUFLLEtBQUswSSxNQUFNLENBQUMxSSxLQUFLLEdBQUcsSUFBSWlHLE9BQU8sRUFBRSxDQUFDLENBQUE7RUFDMUQ7RUFDQUwsRUFBQUEsS0FBSyxDQUFDcEMsR0FBRyxHQUFHb0MsS0FBSyxDQUFDcEMsR0FBRyxDQUFBO0VBQ3JCb0MsRUFBQUEsS0FBSyxDQUFDa0YsR0FBRyxHQUFHbEYsS0FBSyxDQUFDa0YsR0FBRyxDQUFBO0VBQ3JCbEYsRUFBQUEsS0FBSyxDQUFDaUYsR0FBRyxHQUFHakYsS0FBSyxDQUFDaUYsR0FBRyxDQUFBO0VBQ3JCO0VBQ0FBLEVBQUFBLEdBQUcsR0FBRyxVQUFVbkksRUFBRSxFQUFFeUksUUFBUSxFQUFFO0VBQzVCLElBQUEsSUFBSXZGLEtBQUssQ0FBQ2tGLEdBQUcsQ0FBQ3BJLEVBQUUsQ0FBQyxFQUFFLE1BQU0sSUFBSWdDLFdBQVMsQ0FBQ2tHLDBCQUEwQixDQUFDLENBQUE7TUFDbEVPLFFBQVEsQ0FBQ0MsTUFBTSxHQUFHMUksRUFBRSxDQUFBO0VBQ3BCa0QsSUFBQUEsS0FBSyxDQUFDaUYsR0FBRyxDQUFDbkksRUFBRSxFQUFFeUksUUFBUSxDQUFDLENBQUE7RUFDdkIsSUFBQSxPQUFPQSxRQUFRLENBQUE7S0FDaEIsQ0FBQTtFQUNEM0gsRUFBQUEsR0FBRyxHQUFHLFVBQVVkLEVBQUUsRUFBRTtNQUNsQixPQUFPa0QsS0FBSyxDQUFDcEMsR0FBRyxDQUFDZCxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUE7S0FDM0IsQ0FBQTtFQUNEb0ksRUFBQUEsR0FBRyxHQUFHLFVBQVVwSSxFQUFFLEVBQUU7RUFDbEIsSUFBQSxPQUFPa0QsS0FBSyxDQUFDa0YsR0FBRyxDQUFDcEksRUFBRSxDQUFDLENBQUE7S0FDckIsQ0FBQTtFQUNILENBQUMsTUFBTTtFQUNMLEVBQUEsSUFBSTJJLEtBQUssR0FBR1YsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFBO0VBRTlCRSxFQUFBQSxHQUFHLEdBQUcsVUFBVW5JLEVBQUUsRUFBRXlJLFFBQVEsRUFBRTtFQUM1QixJQUFBLElBQUlwRyxRQUFNLENBQUNyQyxFQUFFLEVBQUUySSxLQUFLLENBQUMsRUFBRSxNQUFNLElBQUkzRyxXQUFTLENBQUNrRywwQkFBMEIsQ0FBQyxDQUFBO01BQ3RFTyxRQUFRLENBQUNDLE1BQU0sR0FBRzFJLEVBQUUsQ0FBQTtFQUNwQmdJLElBQUFBLDJCQUEyQixDQUFDaEksRUFBRSxFQUFFMkksS0FBSyxFQUFFRixRQUFRLENBQUMsQ0FBQTtFQUNoRCxJQUFBLE9BQU9BLFFBQVEsQ0FBQTtLQUNoQixDQUFBO0VBQ0QzSCxFQUFBQSxHQUFHLEdBQUcsVUFBVWQsRUFBRSxFQUFFO0VBQ2xCLElBQUEsT0FBT3FDLFFBQU0sQ0FBQ3JDLEVBQUUsRUFBRTJJLEtBQUssQ0FBQyxHQUFHM0ksRUFBRSxDQUFDMkksS0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFBO0tBQzFDLENBQUE7RUFDRFAsRUFBQUEsR0FBRyxHQUFHLFVBQVVwSSxFQUFFLEVBQUU7RUFDbEIsSUFBQSxPQUFPcUMsUUFBTSxDQUFDckMsRUFBRSxFQUFFMkksS0FBSyxDQUFDLENBQUE7S0FDekIsQ0FBQTtFQUNILENBQUE7RUFFQXpJLElBQUFBLGFBQWMsR0FBRztFQUNmaUksRUFBQUEsR0FBRyxFQUFFQSxHQUFHO0VBQ1JySCxFQUFBQSxHQUFHLEVBQUVBLEdBQUc7RUFDUnNILEVBQUFBLEdBQUcsRUFBRUEsR0FBRztFQUNSQyxFQUFBQSxPQUFPLEVBQUVBLE9BQU87RUFDaEJDLEVBQUFBLFNBQVMsRUFBRUEsU0FBQUE7RUFDYixDQUFDOztFQ3JFRCxJQUFJbkcsV0FBVyxHQUFHdkIsbUJBQTZDLENBQUE7RUFDL0QsSUFBSUQsT0FBSyxHQUFHQyxPQUE2QixDQUFBO0VBQ3pDLElBQUl1QyxVQUFVLEdBQUd2QyxZQUFtQyxDQUFBO0VBQ3BELElBQUl5QixNQUFNLEdBQUd6QixnQkFBd0MsQ0FBQTtFQUNyRCxJQUFJMEIsYUFBVyxHQUFHMUIsV0FBbUMsQ0FBQTtFQUNyRCxJQUFJZ0ksMEJBQTBCLEdBQUdoSSxZQUFxQyxDQUFDZ0MsWUFBWSxDQUFBO0VBQ25GLElBQUlVLGFBQWEsR0FBRzFDLGVBQXNDLENBQUE7RUFDMUQsSUFBSWlJLG1CQUFtQixHQUFHakksYUFBc0MsQ0FBQTtFQUVoRSxJQUFJa0ksb0JBQW9CLEdBQUdELG1CQUFtQixDQUFDUixPQUFPLENBQUE7RUFDdEQsSUFBSVUsZ0JBQWdCLEdBQUdGLG1CQUFtQixDQUFDL0gsR0FBRyxDQUFBO0VBQzlDLElBQUk2QyxPQUFPLEdBQUdILE1BQU0sQ0FBQTtFQUNwQjtFQUNBLElBQUkzQyxnQkFBYyxHQUFHckMsTUFBTSxDQUFDcUMsY0FBYyxDQUFBO0VBQzFDLElBQUltSSxXQUFXLEdBQUc3RyxXQUFXLENBQUMsRUFBRSxDQUFDOEcsS0FBSyxDQUFDLENBQUE7RUFDdkMsSUFBSUMsT0FBTyxHQUFHL0csV0FBVyxDQUFDLEVBQUUsQ0FBQytHLE9BQU8sQ0FBQyxDQUFBO0VBQ3JDLElBQUk3SixJQUFJLEdBQUc4QyxXQUFXLENBQUMsRUFBRSxDQUFDOUMsSUFBSSxDQUFDLENBQUE7RUFFL0IsSUFBSThKLG1CQUFtQixHQUFHN0csYUFBVyxJQUFJLENBQUMzQixPQUFLLENBQUMsWUFBWTtFQUMxRCxFQUFBLE9BQU9FLGdCQUFjLENBQUMsWUFBWSxhQUFlLEVBQUUsUUFBUSxFQUFFO0VBQUVpQyxJQUFBQSxLQUFLLEVBQUUsQ0FBQTtFQUFFLEdBQUMsQ0FBQyxDQUFDL0QsTUFBTSxLQUFLLENBQUMsQ0FBQTtFQUN6RixDQUFDLENBQUMsQ0FBQTtFQUVGLElBQUlxSyxRQUFRLEdBQUc1RixNQUFNLENBQUNBLE1BQU0sQ0FBQyxDQUFDN0UsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFBO0VBRTdDLElBQUkwSyxhQUFXLEdBQUduSixhQUFBQSxDQUFBQSxPQUFjLEdBQUcsVUFBVTRDLEtBQUssRUFBRTFHLElBQUksRUFBRWtOLE9BQU8sRUFBRTtFQUNqRSxFQUFBLElBQUlOLFdBQVcsQ0FBQ3JGLE9BQU8sQ0FBQ3ZILElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxTQUFTLEVBQUU7RUFDbERBLElBQUFBLElBQUksR0FBRyxHQUFHLEdBQUc4TSxPQUFPLENBQUN2RixPQUFPLENBQUN2SCxJQUFJLENBQUMsRUFBRSxvQkFBb0IsRUFBRSxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUE7RUFDdkUsR0FBQTtJQUNBLElBQUlrTixPQUFPLElBQUlBLE9BQU8sQ0FBQ0MsTUFBTSxFQUFFbk4sSUFBSSxHQUFHLE1BQU0sR0FBR0EsSUFBSSxDQUFBO0lBQ25ELElBQUlrTixPQUFPLElBQUlBLE9BQU8sQ0FBQ0UsTUFBTSxFQUFFcE4sSUFBSSxHQUFHLE1BQU0sR0FBR0EsSUFBSSxDQUFBO0VBQ25ELEVBQUEsSUFBSSxDQUFDaUcsTUFBTSxDQUFDUyxLQUFLLEVBQUUsTUFBTSxDQUFDLElBQUs4RiwwQkFBMEIsSUFBSTlGLEtBQUssQ0FBQzFHLElBQUksS0FBS0EsSUFBSyxFQUFFO0VBQ2pGLElBQUEsSUFBSWtHLGFBQVcsRUFBRXpCLGdCQUFjLENBQUNpQyxLQUFLLEVBQUUsTUFBTSxFQUFFO0VBQUVBLE1BQUFBLEtBQUssRUFBRTFHLElBQUk7RUFBRXlHLE1BQUFBLFlBQVksRUFBRSxJQUFBO0VBQUssS0FBQyxDQUFDLENBQUMsS0FDL0VDLEtBQUssQ0FBQzFHLElBQUksR0FBR0EsSUFBSSxDQUFBO0VBQ3hCLEdBQUE7RUFDQSxFQUFBLElBQUkrTSxtQkFBbUIsSUFBSUcsT0FBTyxJQUFJakgsTUFBTSxDQUFDaUgsT0FBTyxFQUFFLE9BQU8sQ0FBQyxJQUFJeEcsS0FBSyxDQUFDL0QsTUFBTSxLQUFLdUssT0FBTyxDQUFDRyxLQUFLLEVBQUU7RUFDaEc1SSxJQUFBQSxnQkFBYyxDQUFDaUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtRQUFFQSxLQUFLLEVBQUV3RyxPQUFPLENBQUNHLEtBQUFBO0VBQU0sS0FBQyxDQUFDLENBQUE7RUFDM0QsR0FBQTtJQUNBLElBQUk7RUFDRixJQUFBLElBQUlILE9BQU8sSUFBSWpILE1BQU0sQ0FBQ2lILE9BQU8sRUFBRSxhQUFhLENBQUMsSUFBSUEsT0FBTyxDQUFDck0sV0FBVyxFQUFFO0VBQ3BFLE1BQUEsSUFBSXFGLGFBQVcsRUFBRXpCLGdCQUFjLENBQUNpQyxLQUFLLEVBQUUsV0FBVyxFQUFFO0VBQUVDLFFBQUFBLFFBQVEsRUFBRSxLQUFBO0VBQU0sT0FBQyxDQUFDLENBQUE7RUFDMUU7T0FDQyxNQUFNLElBQUlELEtBQUssQ0FBQzNCLFNBQVMsRUFBRTJCLEtBQUssQ0FBQzNCLFNBQVMsR0FBR3ZCLFNBQVMsQ0FBQTtFQUN6RCxHQUFDLENBQUMsT0FBT2MsS0FBSyxFQUFFLGFBQUU7RUFDbEIsRUFBQSxJQUFJcEQsS0FBSyxHQUFHd0wsb0JBQW9CLENBQUNoRyxLQUFLLENBQUMsQ0FBQTtFQUN2QyxFQUFBLElBQUksQ0FBQ1QsTUFBTSxDQUFDL0UsS0FBSyxFQUFFLFFBQVEsQ0FBQyxFQUFFO0VBQzVCQSxJQUFBQSxLQUFLLENBQUNzSSxNQUFNLEdBQUd2RyxJQUFJLENBQUMrSixRQUFRLEVBQUUsT0FBT2hOLElBQUksSUFBSSxRQUFRLEdBQUdBLElBQUksR0FBRyxFQUFFLENBQUMsQ0FBQTtFQUNwRSxHQUFBO0VBQUUsRUFBQSxPQUFPMEcsS0FBSyxDQUFBO0VBQ2hCLENBQUMsQ0FBQTs7RUFFRDtFQUNBO0VBQ0F0QyxRQUFRLENBQUNXLFNBQVMsQ0FBQ2tDLFFBQVEsR0FBR2dHLGFBQVcsQ0FBQyxTQUFTaEcsUUFBUUEsR0FBRztFQUM1RCxFQUFBLE9BQU9GLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSTRGLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDbkQsTUFBTSxJQUFJdEMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFBO0VBQ2pGLENBQUMsRUFBRSxVQUFVLENBQUM7O0VDckRkLElBQUkrRixXQUFXLEdBQUd6SSxrQkFBcUMsQ0FBQTtFQUN2RCxJQUFJQyxjQUFjLEdBQUdELG9CQUE4QyxDQUFBO0VBRW5FVixJQUFBQSx1QkFBYyxHQUFHLFVBQVV3SixNQUFNLEVBQUV0TixJQUFJLEVBQUV1TixVQUFVLEVBQUU7SUFDbkQsSUFBSUEsVUFBVSxDQUFDN0ksR0FBRyxFQUFFdUksV0FBVyxDQUFDTSxVQUFVLENBQUM3SSxHQUFHLEVBQUUxRSxJQUFJLEVBQUU7RUFBRW1OLElBQUFBLE1BQU0sRUFBRSxJQUFBO0VBQUssR0FBQyxDQUFDLENBQUE7SUFDdkUsSUFBSUksVUFBVSxDQUFDeEIsR0FBRyxFQUFFa0IsV0FBVyxDQUFDTSxVQUFVLENBQUN4QixHQUFHLEVBQUUvTCxJQUFJLEVBQUU7RUFBRW9OLElBQUFBLE1BQU0sRUFBRSxJQUFBO0VBQUssR0FBQyxDQUFDLENBQUE7SUFDdkUsT0FBTzNJLGNBQWMsQ0FBQ2lILENBQUMsQ0FBQzRCLE1BQU0sRUFBRXROLElBQUksRUFBRXVOLFVBQVUsQ0FBQyxDQUFBO0VBQ25ELENBQUM7O0VDUEQsSUFBSTVDLFFBQVEsR0FBR25HLFVBQWlDLENBQUE7O0VBRWhEO0VBQ0E7RUFDQVYsSUFBQUEsV0FBYyxHQUFHLFlBQVk7RUFDM0IsRUFBQSxJQUFJMEosSUFBSSxHQUFHN0MsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFBO0lBQ3pCLElBQUlKLE1BQU0sR0FBRyxFQUFFLENBQUE7RUFDZixFQUFBLElBQUlpRCxJQUFJLENBQUNDLFVBQVUsRUFBRWxELE1BQU0sSUFBSSxHQUFHLENBQUE7RUFDbEMsRUFBQSxJQUFJaUQsSUFBSSxDQUFDdEosTUFBTSxFQUFFcUcsTUFBTSxJQUFJLEdBQUcsQ0FBQTtFQUM5QixFQUFBLElBQUlpRCxJQUFJLENBQUNFLFVBQVUsRUFBRW5ELE1BQU0sSUFBSSxHQUFHLENBQUE7RUFDbEMsRUFBQSxJQUFJaUQsSUFBSSxDQUFDRyxTQUFTLEVBQUVwRCxNQUFNLElBQUksR0FBRyxDQUFBO0VBQ2pDLEVBQUEsSUFBSWlELElBQUksQ0FBQ0ksTUFBTSxFQUFFckQsTUFBTSxJQUFJLEdBQUcsQ0FBQTtFQUM5QixFQUFBLElBQUlpRCxJQUFJLENBQUNLLE9BQU8sRUFBRXRELE1BQU0sSUFBSSxHQUFHLENBQUE7RUFDL0IsRUFBQSxJQUFJaUQsSUFBSSxDQUFDTSxXQUFXLEVBQUV2RCxNQUFNLElBQUksR0FBRyxDQUFBO0VBQ25DLEVBQUEsSUFBSWlELElBQUksQ0FBQ08sTUFBTSxFQUFFeEQsTUFBTSxJQUFJLEdBQUcsQ0FBQTtFQUM5QixFQUFBLE9BQU9BLE1BQU0sQ0FBQTtFQUNmLENBQUM7O0VDaEJELElBQUlyRyxRQUFNLEdBQUdNLFFBQThCLENBQUE7RUFDM0MsSUFBSTBCLFdBQVcsR0FBRzFCLFdBQW1DLENBQUE7RUFDckQsSUFBSXdKLHFCQUFxQixHQUFHeEosdUJBQWdELENBQUE7RUFDNUUsSUFBSXlKLFdBQVcsR0FBR3pKLFdBQW9DLENBQUE7RUFDdEQsSUFBSUQsS0FBSyxHQUFHQyxPQUE2QixDQUFBOztFQUV6QztFQUNBLElBQUkwSixRQUFNLEdBQUdoSyxRQUFNLENBQUNnSyxNQUFNLENBQUE7RUFDMUIsSUFBSUMsZUFBZSxHQUFHRCxRQUFNLENBQUNuSixTQUFTLENBQUE7RUFFdEMsSUFBSXFKLE1BQU0sR0FBR2xJLFdBQVcsSUFBSTNCLEtBQUssQ0FBQyxZQUFZO0lBQzVDLElBQUk4SixlQUFlLEdBQUcsSUFBSSxDQUFBO0lBQzFCLElBQUk7RUFDRkgsSUFBQUEsUUFBTSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQTtLQUNqQixDQUFDLE9BQU81SixLQUFLLEVBQUU7RUFDZCtKLElBQUFBLGVBQWUsR0FBRyxLQUFLLENBQUE7RUFDekIsR0FBQTtJQUVBLElBQUluRCxDQUFDLEdBQUcsRUFBRSxDQUFBO0VBQ1Y7SUFDQSxJQUFJb0QsS0FBSyxHQUFHLEVBQUUsQ0FBQTtFQUNkLEVBQUEsSUFBSUMsUUFBUSxHQUFHRixlQUFlLEdBQUcsUUFBUSxHQUFHLE9BQU8sQ0FBQTtFQUVuRCxFQUFBLElBQUlHLFNBQVMsR0FBRyxVQUFVMUwsR0FBRyxFQUFFMkwsR0FBRyxFQUFFO0VBQ2xDO0VBQ0FyTSxJQUFBQSxNQUFNLENBQUNxQyxjQUFjLENBQUN5RyxDQUFDLEVBQUVwSSxHQUFHLEVBQUU7UUFBRTRCLEdBQUcsRUFBRSxZQUFZO0VBQy9DNEosUUFBQUEsS0FBSyxJQUFJRyxHQUFHLENBQUE7RUFDWixRQUFBLE9BQU8sSUFBSSxDQUFBO0VBQ2IsT0FBQTtFQUFFLEtBQUMsQ0FBQyxDQUFBO0tBQ0wsQ0FBQTtFQUVELEVBQUEsSUFBSUMsS0FBSyxHQUFHO0VBQ1ZkLElBQUFBLE1BQU0sRUFBRSxHQUFHO0VBQ1gxSixJQUFBQSxNQUFNLEVBQUUsR0FBRztFQUNYd0osSUFBQUEsVUFBVSxFQUFFLEdBQUc7RUFDZkMsSUFBQUEsU0FBUyxFQUFFLEdBQUc7RUFDZEksSUFBQUEsTUFBTSxFQUFFLEdBQUE7S0FDVCxDQUFBO0VBRUQsRUFBQSxJQUFJTSxlQUFlLEVBQUVLLEtBQUssQ0FBQ2pCLFVBQVUsR0FBRyxHQUFHLENBQUE7RUFFM0MsRUFBQSxLQUFLLElBQUkzSyxHQUFHLElBQUk0TCxLQUFLLEVBQUVGLFNBQVMsQ0FBQzFMLEdBQUcsRUFBRTRMLEtBQUssQ0FBQzVMLEdBQUcsQ0FBQyxDQUFDLENBQUE7O0VBRWpEO0VBQ0EsRUFBQSxJQUFJeUgsTUFBTSxHQUFHbkksTUFBTSxDQUFDZ0Usd0JBQXdCLENBQUMrSCxlQUFlLEVBQUUsT0FBTyxDQUFDLENBQUN6SixHQUFHLENBQUNNLElBQUksQ0FBQ2tHLENBQUMsQ0FBQyxDQUFBO0VBRWxGLEVBQUEsT0FBT1gsTUFBTSxLQUFLZ0UsUUFBUSxJQUFJRCxLQUFLLEtBQUtDLFFBQVEsQ0FBQTtFQUNsRCxDQUFDLENBQUMsQ0FBQTs7RUFFRjtFQUNBO0VBQ0EsSUFBSUgsTUFBTSxFQUFFSixxQkFBcUIsQ0FBQ0csZUFBZSxFQUFFLE9BQU8sRUFBRTtFQUMxRDFILEVBQUFBLFlBQVksRUFBRSxJQUFJO0VBQ2xCL0IsRUFBQUEsR0FBRyxFQUFFdUosV0FBQUE7RUFDUCxDQUFDLENBQUM7O0VDdkRGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7O0VBRUE7RUFDQSxNQUFNVSxZQUFZLEdBQUc7RUFDbkJDLEVBQUFBLGdCQUFnQixFQUFFLHVDQUF1QztFQUN6REMsRUFBQUEsY0FBYyxFQUFFLGlCQUFpQjtFQUNqQ0MsRUFBQUEsTUFBTSxFQUFFLDhCQUE4QjtFQUN0Q0MsRUFBQUEsS0FBSyxFQUFFLG9JQUFvSTtFQUFFO0VBQzdJQyxFQUFBQSxXQUFXLEVBQUUsdUNBQXVDO0VBQ3BEQyxFQUFBQSxZQUFZLEVBQUUsb0JBQW9CO0VBQ2xDQyxFQUFBQSxXQUFXLEVBQUUsdUJBQXVCO0VBQ3BDQyxFQUFBQSxXQUFXLEVBQUUsK0NBQStDO0VBQzVEQyxFQUFBQSxNQUFNLEVBQUUsK0JBQStCO0VBQ3ZDQyxFQUFBQSxlQUFlLEVBQUUsa0JBQWtCO0VBQ25DQyxFQUFBQSxTQUFTLEVBQUUscUJBQXFCO0VBQ2hDQyxFQUFBQSxhQUFhLEVBQUUsZ0RBQWdEO0VBQy9EQyxFQUFBQSxZQUFZLEVBQUUscURBQXFEO0VBQ25FQyxFQUFBQSxRQUFRLEVBQUUscVdBQUE7RUFDWixDQUFDLENBQUE7O0VBRUQ7RUFDQSxNQUFNQyxrQkFBa0IsR0FBRyxJQUFJeEIsTUFBTSxDQUFDLHFwREFBcXBELENBQUMsQ0FBQTtFQUU1ckQsTUFBTXlCLG1CQUFtQixHQUFHLElBQUl6QixNQUFNLENBQUMscXBEQUFxcEQsQ0FBQyxDQUFBOztFQUU3ckQ7O0VBRUE7RUFDQTtFQUNBO0VBQ0EsTUFBTTBCLFdBQVcsR0FBRztFQUNsQkMsRUFBQUEsSUFBSSxFQUFFO0VBQ0pDLElBQUFBLE1BQU0sRUFBRSxtQ0FBbUM7RUFDM0NDLElBQUFBLFdBQVcsRUFBRSx5RkFBQTtLQUNkO0VBQ0RDLEVBQUFBLElBQUksRUFBRTtFQUNKRixJQUFBQSxNQUFNLEVBQUUsb0NBQW9DO0VBQzVDQyxJQUFBQSxXQUFXLEVBQUUseUZBQUE7S0FDZDtFQUNERSxFQUFBQSxJQUFJLEVBQUU7RUFDSkgsSUFBQUEsTUFBTSxFQUFFLHFDQUFxQztFQUM3Q0MsSUFBQUEsV0FBVyxFQUFFLHlGQUFBO0tBQ2Q7RUFDREcsRUFBQUEsSUFBSSxFQUFFO0VBQ0pKLElBQUFBLE1BQU0sRUFBRSxzQ0FBc0M7RUFDOUNDLElBQUFBLFdBQVcsRUFBRSx5RkFBQTtLQUNkO0VBQ0RJLEVBQUFBLElBQUksRUFBRTtFQUNKTCxJQUFBQSxNQUFNLEVBQUUsdUNBQXVDO0VBQy9DQyxJQUFBQSxXQUFXLEVBQUUseUZBQUE7S0FDZDtFQUNESyxFQUFBQSxJQUFJLEVBQUU7RUFDSk4sSUFBQUEsTUFBTSxFQUFFLHdDQUF3QztFQUNoREMsSUFBQUEsV0FBVyxFQUFFLHlGQUFBO0tBQ2Q7RUFDRE0sRUFBQUEsWUFBWSxFQUFFO0VBQ1pQLElBQUFBLE1BQU0sRUFBRSxxQkFBcUI7RUFDN0JDLElBQUFBLFdBQVcsRUFBRSx1REFBQTtLQUNkO0VBQ0RPLEVBQUFBLHVCQUF1QixFQUFFO0VBQ3ZCUixJQUFBQSxNQUFNLEVBQUUseUNBQXlDO0VBQ2pEQyxJQUFBQSxXQUFXLEVBQUUsaUdBQUE7S0FDZDtFQUNEUSxFQUFBQSxvQkFBb0IsRUFBRTtFQUNwQlQsSUFBQUEsTUFBTSxFQUFFLHNDQUFzQztFQUM5Q0MsSUFBQUEsV0FBVyxFQUFFLDhGQUFBO0tBQ2Q7RUFDRFMsRUFBQUEsd0JBQXdCLEVBQUU7RUFDeEJWLElBQUFBLE1BQU0sRUFBRSw4QkFBOEI7RUFDdENDLElBQUFBLFdBQVcsRUFBRSw2REFBQTtLQUNkO0VBQ0RVLEVBQUFBLHFCQUFxQixFQUFFO0VBQ3JCWCxJQUFBQSxNQUFNLEVBQUUsOEJBQThCO0VBQ3RDQyxJQUFBQSxXQUFXLEVBQUUsMERBQUE7S0FDZDtFQUNEVyxFQUFBQSxXQUFXLEVBQUU7RUFDWFosSUFBQUEsTUFBTSxFQUFFLFlBQVk7RUFDcEJDLElBQUFBLFdBQVcsRUFBRSxJQUFBO0tBQ2Q7RUFDRFksRUFBQUEsZ0JBQWdCLEVBQUU7RUFDaEJiLElBQUFBLE1BQU0sRUFBRSxlQUFlO0VBQ3ZCQyxJQUFBQSxXQUFXLEVBQUUsd0RBQUE7S0FDZDtFQUNEYSxFQUFBQSxnQkFBZ0IsRUFBRTtFQUNoQmQsSUFBQUEsTUFBTSxFQUFFLGVBQWU7RUFDdkJDLElBQUFBLFdBQVcsRUFBRSx3REFBQTtLQUNkO0VBQ0RjLEVBQUFBLElBQUksRUFBRTtFQUNKZixJQUFBQSxNQUFNLEVBQUUseUZBQXlGO0VBQ2pHQyxJQUFBQSxXQUFXLEVBQUUsNENBQUE7S0FDZDtFQUNEZSxFQUFBQSxJQUFJLEVBQUU7RUFDSmhCLElBQUFBLE1BQU0sRUFBRSwyQkFBMkI7RUFDbkNDLElBQUFBLFdBQVcsRUFBRSwrQ0FBQTtLQUNkO0VBQ0RnQixFQUFBQSxJQUFJLEVBQUU7RUFDSmpCLElBQUFBLE1BQU0sRUFBRSxpQ0FBaUM7RUFDekNDLElBQUFBLFdBQVcsRUFBRSwrQ0FBQTtLQUNkO0VBQ0Q7RUFDQWlCLEVBQUFBLGNBQWMsRUFBRTtFQUNkbEIsSUFBQUEsTUFBTSxFQUFFLGlCQUFpQjtFQUN6QkMsSUFBQUEsV0FBVyxFQUFFLHdEQUFBO0tBQ2Q7RUFDRGtCLEVBQUFBLHlCQUF5QixFQUFFO0VBQ3pCO0VBQ0FuQixJQUFBQSxNQUFNLEVBQUUsNEtBQTRLO0VBQ3BMQyxJQUFBQSxXQUFXLEVBQUUsMFFBQTBRO01BQ3ZSbUIsZ0JBQWdCLEVBQUUsQ0FBQztFQUNyQixHQUFBO0VBQ0YsQ0FBQyxDQUFBOztFQUVEO0VBQ0E7RUFDQTtFQUNBLElBQUlDLGdCQUFnQixHQUFHLENBQ3JCO0VBQUVDLEVBQUFBLEtBQUssRUFBRSx5Q0FBeUM7RUFBRUMsRUFBQUEsR0FBRyxFQUFFLG1DQUFtQztFQUFFQyxFQUFBQSxhQUFhLEVBQUUsSUFBQTtFQUFLLENBQUMsRUFDbkg7RUFBRUYsRUFBQUEsS0FBSyxFQUFFLGFBQWE7RUFBRUMsRUFBQUEsR0FBRyxFQUFFLEtBQUs7RUFBRUMsRUFBQUEsYUFBYSxFQUFFLElBQUE7RUFBSyxDQUFDLEVBQ3pEO0VBQUVGLEVBQUFBLEtBQUssRUFBRSxZQUFZO0VBQUVDLEVBQUFBLEdBQUcsRUFBRSxLQUFLO0VBQUVDLEVBQUFBLGFBQWEsRUFBRSxJQUFBO0VBQUssQ0FBQyxFQUN4RDtFQUFFRixFQUFBQSxLQUFLLEVBQUUsZ0JBQWdCO0VBQUVDLEVBQUFBLEdBQUcsRUFBRSxHQUFHO0VBQUVDLEVBQUFBLGFBQWEsRUFBRyxJQUFBO0VBQUksQ0FBQyxFQUMxRDtFQUFFRixFQUFBQSxLQUFLLEVBQUUsb0JBQW9CO0VBQUVDLEVBQUFBLEdBQUcsRUFBRSxPQUFPO0VBQUVDLEVBQUFBLGFBQWEsRUFBRyxJQUFBO0VBQUksQ0FBQyxFQUNsRTtFQUFFRixFQUFBQSxLQUFLLEVBQUUsNkNBQTZDO0VBQUVDLEVBQUFBLEdBQUcsRUFBRSxLQUFLO0VBQUVDLEVBQUFBLGFBQWEsRUFBRSxJQUFBO0VBQUksQ0FBQyxFQUN4RjtFQUFFRixFQUFBQSxLQUFLLEVBQUUseUNBQXlDO0VBQUVDLEVBQUFBLEdBQUcsRUFBRSxLQUFLO0VBQUVDLEVBQUFBLGFBQWEsRUFBRSxLQUFBO0VBQUssQ0FBQyxDQUN0RixDQUFBOztFQUVEO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQSxJQUFJQyxhQUFhLEdBQUc7RUFDbEJDLEVBQUFBLE1BQU0sRUFBRztFQUNQMUIsSUFBQUEsTUFBTSxFQUFFLHVCQUF1QjtFQUMvQkMsSUFBQUEsV0FBVyxFQUFHLGtEQUFBO0tBQ2Y7RUFDRDlRLEVBQUFBLElBQUksRUFBRztFQUNMNlEsSUFBQUEsTUFBTSxFQUFFLHFDQUFxQztFQUM3Q0MsSUFBQUEsV0FBVyxFQUFHLHdIQUFBO0tBQ2Y7RUFDRDBCLEVBQUFBLFFBQVEsRUFBRztFQUNUM0IsSUFBQUEsTUFBTSxFQUFFLG9DQUFvQztFQUM1Q0MsSUFBQUEsV0FBVyxFQUFFLHdJQUFBO0tBQ2Q7RUFDRDJCLEVBQUFBLElBQUksRUFBRztFQUNMNUIsSUFBQUEsTUFBTSxFQUFFLGtHQUFrRztFQUMxR0MsSUFBQUEsV0FBVyxFQUFFLGdDQUFBO0tBQ2Q7RUFDRDRCLEVBQUFBLFFBQVEsRUFBRztFQUNUN0IsSUFBQUEsTUFBTSxFQUFFLEtBQUs7RUFDYkMsSUFBQUEsV0FBVyxFQUFFLEVBQUE7S0FDZDtFQUNENkIsRUFBQUEsU0FBUyxFQUFHO0VBQ1Y5QixJQUFBQSxNQUFNLEVBQUUsTUFBTTtFQUNkQyxJQUFBQSxXQUFXLEVBQUcsRUFBQTtLQUNmO0VBQ0Q4QixFQUFBQSxTQUFTLEVBQUc7RUFDVi9CLElBQUFBLE1BQU0sRUFBRSwwQkFBMEI7RUFDbENDLElBQUFBLFdBQVcsRUFBRSxFQUFFO0VBQ2ZtQixJQUFBQSxnQkFBZ0IsRUFBRSxDQUFBO0tBQ25CO0VBQ0RZLEVBQUFBLE9BQU8sRUFBRztFQUNSaEMsSUFBQUEsTUFBTSxFQUFFLDBCQUEwQjtFQUNsQ0MsSUFBQUEsV0FBVyxFQUFFLElBQUE7RUFDZixHQUFBO0VBQ0YsQ0FBQyxDQUFBOztFQUVEO0VBQ0EsTUFBTWdDLGlCQUFpQixHQUFHLElBQUk3RCxNQUFNLENBQUM5TCxNQUFNLENBQUNFLElBQUksQ0FBQ3FNLFlBQVksQ0FBQyxDQUFDMUwsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUE7O0VBRXpFO0VBQ0EsTUFBTStPLFdBQVcsR0FBRSxDQUFDLEdBQUc1UCxNQUFNLENBQUNFLElBQUksQ0FBQ2lQLGFBQWEsQ0FBQyxDQUFDLENBQUE7RUFDbEQsS0FBSyxJQUFJVSxJQUFJLElBQUlELFdBQVcsRUFBRTtJQUM1QixJQUFJRSxFQUFFLEdBQUdYLGFBQWEsQ0FBQ1UsSUFBSSxDQUFDLENBQUNuQyxNQUFNLENBQUN0RyxNQUFNLENBQUE7RUFDMUM7RUFDQSxFQUFBLE9BQU8wSSxFQUFFLENBQUNyUCxLQUFLLENBQUNrUCxpQkFBaUIsQ0FBQyxFQUFFO01BQ2xDRyxFQUFFLEdBQUdBLEVBQUUsQ0FBQ3BGLE9BQU8sQ0FBQ2lGLGlCQUFpQixFQUFHSSxNQUFNLElBQUs7RUFBRSxNQUFBLE9BQU94RCxZQUFZLENBQUN3RCxNQUFNLENBQUMsQ0FBQzNJLE1BQU0sQ0FBQTtFQUFFLEtBQUMsQ0FBQyxDQUFBO0VBQ3pGLEdBQUE7RUFDQStILEVBQUFBLGFBQWEsQ0FBQ1UsSUFBSSxDQUFDLENBQUNuQyxNQUFNLEdBQUcsSUFBSTVCLE1BQU0sQ0FBQ2dFLEVBQUUsRUFBRVgsYUFBYSxDQUFDVSxJQUFJLENBQUMsQ0FBQ25DLE1BQU0sQ0FBQ3NDLEtBQUssQ0FBQyxDQUFBO0VBQy9FLENBQUE7O0VBRUE7RUFDQSxLQUFLLElBQUlILElBQUksSUFBSWQsZ0JBQWdCLEVBQUU7RUFDakMsRUFBQSxJQUFJZSxFQUFFLEdBQUdELElBQUksQ0FBQ2IsS0FBSyxDQUFDNUgsTUFBTSxDQUFBO0VBQzFCO0VBQ0EsRUFBQSxPQUFPMEksRUFBRSxDQUFDclAsS0FBSyxDQUFDa1AsaUJBQWlCLENBQUMsRUFBRTtNQUNsQ0csRUFBRSxHQUFHQSxFQUFFLENBQUNwRixPQUFPLENBQUNpRixpQkFBaUIsRUFBR0ksTUFBTSxJQUFLO0VBQUUsTUFBQSxPQUFPeEQsWUFBWSxDQUFDd0QsTUFBTSxDQUFDLENBQUMzSSxNQUFNLENBQUE7RUFBRSxLQUFDLENBQUMsQ0FBQTtFQUN6RixHQUFBO0VBQ0F5SSxFQUFBQSxJQUFJLENBQUNiLEtBQUssR0FBRyxJQUFJbEQsTUFBTSxDQUFDZ0UsRUFBRSxFQUFFRCxJQUFJLENBQUNiLEtBQUssQ0FBQ2dCLEtBQUssQ0FBQyxDQUFBO0VBQy9DLENBQUE7O0VBRUE7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBLFNBQVNDLFVBQVVBLENBQUNGLE1BQU0sRUFBRTtJQUMxQixPQUFPLENBQUNBLE1BQU0sR0FBR0EsTUFBTSxHQUFHLEVBQUUsRUFDekJyRixPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUN0QkEsT0FBTyxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FDckJBLE9BQU8sQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUE7RUFDMUIsQ0FBQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBLE1BQU05TCxRQUFRLEdBQUc7RUFDZjtFQUNBakMsRUFBQUEsSUFBSSxFQUFFO0VBQ0pxTixJQUFBQSxJQUFJLEVBQUUsUUFBUTtFQUNkckssSUFBQUEsU0FBUyxFQUFFLFVBQVU7RUFDckJnSyxJQUFBQSxHQUFHLEVBQUU7RUFBQ3VHLE1BQUFBLEdBQUcsRUFBRSxJQUFJO0VBQUVDLE1BQUFBLElBQUksRUFBRSxJQUFBO09BQUs7RUFDNUJDLElBQUFBLEtBQUssRUFBRTtFQUFDQyxNQUFBQSxVQUFVLEVBQUUsY0FBYztFQUFFQyxNQUFBQSxXQUFXLEVBQUUsY0FBQTtFQUFjLEtBQUE7S0FDaEU7RUFDRHBULEVBQUFBLE1BQU0sRUFBRTtFQUNOOE0sSUFBQUEsSUFBSSxFQUFFLFFBQVE7RUFDZHJLLElBQUFBLFNBQVMsRUFBRSxNQUFNO0VBQ2pCZ0ssSUFBQUEsR0FBRyxFQUFFO0VBQUN1RyxNQUFBQSxHQUFHLEVBQUUsR0FBRztFQUFFQyxNQUFBQSxJQUFJLEVBQUUsR0FBQTtPQUFJO0VBQzFCQyxJQUFBQSxLQUFLLEVBQUU7RUFBQ0MsTUFBQUEsVUFBVSxFQUFFLFdBQVc7RUFBRUMsTUFBQUEsV0FBVyxFQUFFLFdBQUE7RUFBVyxLQUFBO0tBQzFEO0VBQ0R6VCxFQUFBQSxJQUFJLEVBQUU7RUFDSm1OLElBQUFBLElBQUksRUFBRSxRQUFRO0VBQ2RySyxJQUFBQSxTQUFTLEVBQUUsUUFBUTtFQUNuQmdLLElBQUFBLEdBQUcsRUFBRTtFQUFDdUcsTUFBQUEsR0FBRyxFQUFFLEdBQUc7RUFBRUMsTUFBQUEsSUFBSSxFQUFFLEdBQUE7T0FBSTtFQUMxQkMsSUFBQUEsS0FBSyxFQUFFO0VBQUNDLE1BQUFBLFVBQVUsRUFBRSxLQUFLO0VBQUVDLE1BQUFBLFdBQVcsRUFBRSxLQUFBO0VBQUssS0FBQztLQUMvQzs7RUFDRGpULEVBQUFBLGFBQWEsRUFBRTtFQUNiMk0sSUFBQUEsSUFBSSxFQUFFLFFBQVE7RUFDZHJLLElBQUFBLFNBQVMsRUFBRSxpQkFBaUI7RUFDNUJnSyxJQUFBQSxHQUFHLEVBQUU7RUFBQ3VHLE1BQUFBLEdBQUcsRUFBRSxJQUFJO0VBQUVDLE1BQUFBLElBQUksRUFBRSxJQUFBO09BQUs7RUFDNUJDLElBQUFBLEtBQUssRUFBRTtFQUFDQyxNQUFBQSxVQUFVLEVBQUMsS0FBSztFQUFFQyxNQUFBQSxXQUFXLEVBQUUsS0FBQTtFQUFNLEtBQUE7S0FDOUM7RUFDRHhULEVBQUFBLEVBQUUsRUFBRTtFQUNGa04sSUFBQUEsSUFBSSxFQUFFLE1BQU07RUFDWnJLLElBQUFBLFNBQVMsRUFBRSxNQUFNO0VBQ2pCZ0ssSUFBQUEsR0FBRyxFQUFFO0VBQUM0RyxNQUFBQSxPQUFPLEVBQUUscURBQXFEO0VBQUU1QyxNQUFBQSxXQUFXLEVBQUUsTUFBQTtPQUFPO0VBQzFGeUMsSUFBQUEsS0FBSyxFQUFFO0VBQUNHLE1BQUFBLE9BQU8sRUFBRSxvQ0FBb0M7RUFBRTVDLE1BQUFBLFdBQVcsRUFBRSxJQUFBO0VBQUksS0FBQTtLQUN6RTtFQUNENVEsRUFBQUEsRUFBRSxFQUFFO0VBQ0ZpTixJQUFBQSxJQUFJLEVBQUUsTUFBTTtFQUNackssSUFBQUEsU0FBUyxFQUFFLE1BQU07RUFDakJnSyxJQUFBQSxHQUFHLEVBQUU7RUFBQzRHLE1BQUFBLE9BQU8sRUFBRSxxREFBcUQ7RUFBRTVDLE1BQUFBLFdBQVcsRUFBRSxPQUFBO09BQVE7RUFDM0Z5QyxJQUFBQSxLQUFLLEVBQUU7RUFBQ0csTUFBQUEsT0FBTyxFQUFFLHFDQUFxQztFQUFFNUMsTUFBQUEsV0FBVyxFQUFFLElBQUE7RUFBSSxLQUFBO0tBQzFFO0VBQ0Q2QyxFQUFBQSxFQUFFLEVBQUU7RUFDRnhHLElBQUFBLElBQUksRUFBRSxNQUFNO0VBQ1pySyxJQUFBQSxTQUFTLEVBQUUsTUFBTTtFQUNqQmdLLElBQUFBLEdBQUcsRUFBRTtFQUFDNEcsTUFBQUEsT0FBTyxFQUFFLHFEQUFxRDtFQUFFNUMsTUFBQUEsV0FBVyxFQUFFLFFBQUE7T0FBUztFQUM1RnlDLElBQUFBLEtBQUssRUFBRTtFQUFDRyxNQUFBQSxPQUFPLEVBQUUsc0NBQXNDO0VBQUU1QyxNQUFBQSxXQUFXLEVBQUUsSUFBQTtFQUFJLEtBQUE7S0FDM0U7RUFDRDhDLEVBQUFBLEVBQUUsRUFBRTtFQUNGekcsSUFBQUEsSUFBSSxFQUFFLE1BQU07RUFDWnJLLElBQUFBLFNBQVMsRUFBRSxNQUFNO0VBQ2pCZ0ssSUFBQUEsR0FBRyxFQUFFO0VBQUM0RyxNQUFBQSxPQUFPLEVBQUUscURBQXFEO0VBQUU1QyxNQUFBQSxXQUFXLEVBQUUsU0FBQTtPQUFVO0VBQzdGeUMsSUFBQUEsS0FBSyxFQUFFO0VBQUNHLE1BQUFBLE9BQU8sRUFBRSx1Q0FBdUM7RUFBRTVDLE1BQUFBLFdBQVcsRUFBRSxJQUFBO0VBQUksS0FBQTtLQUM1RTtFQUNEK0MsRUFBQUEsRUFBRSxFQUFFO0VBQ0YxRyxJQUFBQSxJQUFJLEVBQUUsTUFBTTtFQUNackssSUFBQUEsU0FBUyxFQUFFLE1BQU07RUFDakJnSyxJQUFBQSxHQUFHLEVBQUU7RUFBQzRHLE1BQUFBLE9BQU8sRUFBRSxxREFBcUQ7RUFBRTVDLE1BQUFBLFdBQVcsRUFBRSxVQUFBO09BQVc7RUFDOUZ5QyxJQUFBQSxLQUFLLEVBQUU7RUFBQ0csTUFBQUEsT0FBTyxFQUFFLHdDQUF3QztFQUFFNUMsTUFBQUEsV0FBVyxFQUFFLElBQUE7RUFBSSxLQUFBO0tBQzdFO0VBQ0RnRCxFQUFBQSxFQUFFLEVBQUU7RUFDRjNHLElBQUFBLElBQUksRUFBRSxNQUFNO0VBQ1pySyxJQUFBQSxTQUFTLEVBQUUsTUFBTTtFQUNqQmdLLElBQUFBLEdBQUcsRUFBRTtFQUFDNEcsTUFBQUEsT0FBTyxFQUFFLHFEQUFxRDtFQUFFNUMsTUFBQUEsV0FBVyxFQUFFLFdBQUE7T0FBWTtFQUMvRnlDLElBQUFBLEtBQUssRUFBRTtFQUFDRyxNQUFBQSxPQUFPLEVBQUUseUNBQXlDO0VBQUU1QyxNQUFBQSxXQUFXLEVBQUUsSUFBQTtFQUFJLEtBQUE7S0FDOUU7RUFDRHJRLEVBQUFBLEVBQUUsRUFBRTtFQUNGME0sSUFBQUEsSUFBSSxFQUFFLE1BQU07RUFDWnJLLElBQUFBLFNBQVMsRUFBRSxNQUFNO0VBQ2pCZ0ssSUFBQUEsR0FBRyxFQUFFO0VBQUM0RyxNQUFBQSxPQUFPLEVBQUUscURBQXFEO0VBQUU1QyxNQUFBQSxXQUFXLEVBQUUsTUFBQTtPQUFPO0VBQzFGeUMsSUFBQUEsS0FBSyxFQUFFO0VBQUNHLE1BQUFBLE9BQU8sRUFBRSwyQkFBMkI7RUFBRTVDLE1BQUFBLFdBQVcsRUFBRSxJQUFBO0VBQUksS0FBQTtLQUNoRTtFQUNEdlEsRUFBQUEsRUFBRSxFQUFFO0VBQ0Y0TSxJQUFBQSxJQUFJLEVBQUUsTUFBTTtFQUNackssSUFBQUEsU0FBUyxFQUFFLE1BQU07RUFDakJnSyxJQUFBQSxHQUFHLEVBQUU7RUFBQzRHLE1BQUFBLE9BQU8sRUFBRSxxREFBcUQ7RUFBRTVDLE1BQUFBLFdBQVcsRUFBRSxRQUFBO09BQVM7RUFDNUZ5QyxJQUFBQSxLQUFLLEVBQUU7RUFBQ0csTUFBQUEsT0FBTyxFQUFFLGlDQUFpQztFQUFFNUMsTUFBQUEsV0FBVyxFQUFFLElBQUE7RUFBSSxLQUFBO0tBQ3RFO0VBQ0RqUixFQUFBQSxVQUFVLEVBQUU7RUFDVnNOLElBQUFBLElBQUksRUFBRSxNQUFNO0VBQ1pySyxJQUFBQSxTQUFTLEVBQUUsY0FBYztFQUN6QmdLLElBQUFBLEdBQUcsRUFBRTtFQUFDNEcsTUFBQUEsT0FBTyxFQUFFLHFEQUFxRDtFQUFFNUMsTUFBQUEsV0FBVyxFQUFFLE1BQUE7T0FBTztFQUMxRnlDLElBQUFBLEtBQUssRUFBRTtFQUFDRyxNQUFBQSxPQUFPLEVBQUUscUJBQXFCO0VBQUU1QyxNQUFBQSxXQUFXLEVBQUUsSUFBQTtFQUFJLEtBQUE7RUFDM0QsR0FBQTtFQUNGLENBQUM7O0VDL1JELE1BQU1pRCxNQUFNLENBQUM7RUFDWG5TLEVBQUFBLFdBQVdBLEdBQWE7RUFBQSxJQUFBLElBQVpDLEtBQUssR0FBQXNFLFNBQUEsQ0FBQXpDLE1BQUEsR0FBQSxDQUFBLElBQUF5QyxTQUFBLENBQUEsQ0FBQSxDQUFBLEtBQUE1QixTQUFBLEdBQUE0QixTQUFBLENBQUEsQ0FBQSxDQUFBLEdBQUcsRUFBRSxDQUFBO01BQ3BCLElBQUksQ0FBQ3JFLENBQUMsR0FBRyxJQUFJLENBQUE7TUFDYixJQUFJLENBQUNrUyxRQUFRLEdBQUcsSUFBSSxDQUFBO01BQ3BCLElBQUksQ0FBQ0MsS0FBSyxHQUFHLEVBQUUsQ0FBQTtNQUNmLElBQUksQ0FBQ0MsWUFBWSxHQUFHLEVBQUUsQ0FBQTtNQUN0QixJQUFJLENBQUNDLFNBQVMsR0FBRyxFQUFFLENBQUE7TUFDbkIsSUFBSSxDQUFDQyxZQUFZLEdBQUcsRUFBRSxDQUFBO01BQ3RCLElBQUksQ0FBQ0MsZ0JBQWdCLEdBQUcsRUFBRSxDQUFBO01BQzFCLElBQUksQ0FBQ0MsVUFBVSxHQUFHLEVBQUUsQ0FBQTtNQUNwQixJQUFJLENBQUNDLFNBQVMsR0FBRyxFQUFFLENBQUE7TUFDbkIsSUFBSSxDQUFDQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUE7TUFFNUIsSUFBSSxDQUFDQyxTQUFTLEdBQUc7RUFDZkMsTUFBQUEsTUFBTSxFQUFFLEVBQUU7RUFDVkMsTUFBQUEsU0FBUyxFQUFFLEVBQUU7RUFDYkMsTUFBQUEsSUFBSSxFQUFFLEVBQUE7T0FDUCxDQUFBO0VBRUQsSUFBQSxJQUFJelMsT0FBTyxHQUFHTixLQUFLLENBQUNNLE9BQU8sQ0FBQTtFQUMzQixJQUFBLElBQUksQ0FBQzZSLFFBQVEsR0FBR25TLEtBQUssQ0FBQ21TLFFBQVEsQ0FBQTtNQUU5QixJQUFJLElBQUksQ0FBQ0EsUUFBUSxFQUFFO0VBQ2pCLE1BQUEsSUFBSSxDQUFDLElBQUksQ0FBQ0EsUUFBUSxDQUFDNVIsT0FBTyxFQUFFO1VBQzFCLElBQUksQ0FBQzRSLFFBQVEsR0FBRzNSLFFBQVEsQ0FBQ0MsY0FBYyxDQUFDLElBQUksQ0FBQzBSLFFBQVEsQ0FBQyxDQUFBO0VBQ3hELE9BQUE7RUFDQSxNQUFBLElBQUksQ0FBQzdSLE9BQU8sRUFBRUEsT0FBTyxHQUFHLElBQUksQ0FBQzZSLFFBQVEsQ0FBQTtFQUN2QyxLQUFBO0VBRUEsSUFBQSxJQUFJN1IsT0FBTyxJQUFJLENBQUNBLE9BQU8sQ0FBQ0MsT0FBTyxFQUFFO1FBQy9CRCxPQUFPLEdBQUdFLFFBQVEsQ0FBQ0MsY0FBYyxDQUFDVCxLQUFLLENBQUNNLE9BQU8sQ0FBQyxDQUFBO0VBQ2xELEtBQUE7TUFDQSxJQUFJLENBQUNBLE9BQU8sRUFBRTtRQUNaQSxPQUFPLEdBQUdFLFFBQVEsQ0FBQ3dTLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO0VBQ3BELEtBQUE7RUFDQSxJQUFBLElBQUkxUyxPQUFPLENBQUNDLE9BQU8sSUFBSSxVQUFVLEVBQUU7UUFDakMsSUFBSSxDQUFDNFIsUUFBUSxHQUFHN1IsT0FBTyxDQUFBO0VBQ3ZCQSxNQUFBQSxPQUFPLEdBQUcsSUFBSSxDQUFDNlIsUUFBUSxDQUFDYyxVQUFVLENBQUE7RUFDcEMsS0FBQTtNQUVBLElBQUksSUFBSSxDQUFDZCxRQUFRLEVBQUU7RUFDakIsTUFBQSxJQUFJLENBQUNBLFFBQVEsQ0FBQ2UsS0FBSyxDQUFDQyxPQUFPLEdBQUcsTUFBTSxDQUFBO0VBQ3RDLEtBQUE7RUFFQSxJQUFBLElBQUksQ0FBQ0MsbUJBQW1CLENBQUM5UyxPQUFPLEVBQUVOLEtBQUssQ0FBQyxDQUFBO01BQ3hDLElBQUksQ0FBQ3FULFVBQVUsQ0FDYixPQUFPclQsS0FBSyxDQUFDc1QsT0FBTyxLQUFLLFFBQVEsR0FDN0J0VCxLQUFLLENBQUNzVCxPQUFPLEdBQ2IsSUFBSSxDQUFDbkIsUUFBUSxHQUNiLElBQUksQ0FBQ0EsUUFBUSxDQUFDdk0sS0FBSyxHQUNuQixpQ0FDTixDQUFDLENBQUE7RUFDSCxHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNBO0VBQ0E7RUFDRXdOLEVBQUFBLG1CQUFtQkEsQ0FBQzlTLE9BQU8sRUFBRU4sS0FBSyxFQUFFO0VBQ2xDLElBQUEsSUFBSUEsS0FBSyxJQUFJQSxLQUFLLENBQUNULE1BQU0sS0FBS21ELFNBQVMsRUFBRTtRQUN2QyxJQUFJMUMsS0FBSyxDQUFDVCxNQUFNLENBQUNnQixPQUFPLEVBQUUsSUFBSSxDQUFDTixDQUFDLEdBQUdELEtBQUssQ0FBQ1QsTUFBTSxDQUFDLEtBQzNDLElBQUksQ0FBQ1UsQ0FBQyxHQUFHTyxRQUFRLENBQUNDLGNBQWMsQ0FBQ1QsS0FBSyxDQUFDVCxNQUFNLENBQUMsQ0FBQTtPQUNwRCxNQUFNLElBQUksQ0FBQ1UsQ0FBQyxHQUFHTyxRQUFRLENBQUNRLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQTtNQUU3QyxJQUFJLENBQUNmLENBQUMsQ0FBQ3NULFNBQVMsQ0FBQ0MsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFBO0VBQy9CLElBQUEsSUFBSSxDQUFDdlQsQ0FBQyxDQUFDd1QsZUFBZSxHQUFHLElBQUksQ0FBQTtFQUM3QjtFQUNBO0VBQ0EsSUFBQSxJQUFJLENBQUN4VCxDQUFDLENBQUNpVCxLQUFLLENBQUNRLFVBQVUsR0FBRyxVQUFVLENBQUE7RUFDcEM7RUFDQSxJQUFBLElBQUksQ0FBQ3pULENBQUMsQ0FBQ2lULEtBQUssQ0FBQ1MsZ0JBQWdCLEdBQUcsMkJBQTJCLENBQUE7RUFFM0QsSUFBQSxJQUFJM1QsS0FBSyxDQUFDVCxNQUFNLEtBQUttRCxTQUFTLEVBQUU7RUFDOUIsTUFBQSxJQUNFLElBQUksQ0FBQ3lQLFFBQVEsSUFDYixJQUFJLENBQUNBLFFBQVEsQ0FBQ2MsVUFBVSxJQUFJM1MsT0FBTyxJQUNuQyxJQUFJLENBQUM2UixRQUFRLENBQUN5QixXQUFXLEVBQ3pCO0VBQ0F0VCxRQUFBQSxPQUFPLENBQUN1VCxZQUFZLENBQUMsSUFBSSxDQUFDNVQsQ0FBQyxFQUFFLElBQUksQ0FBQ2tTLFFBQVEsQ0FBQ3lCLFdBQVcsQ0FBQyxDQUFBO0VBQ3pELE9BQUMsTUFBTTtFQUNMdFQsUUFBQUEsT0FBTyxDQUFDYyxXQUFXLENBQUMsSUFBSSxDQUFDbkIsQ0FBQyxDQUFDLENBQUE7RUFDN0IsT0FBQTtFQUNGLEtBQUE7RUFFQSxJQUFBLElBQUksQ0FBQ0EsQ0FBQyxDQUFDVyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUdYLENBQUMsSUFBSyxJQUFJLENBQUM2VCxnQkFBZ0IsQ0FBQzdULENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDakUsSUFBQSxJQUFJLENBQUNBLENBQUMsQ0FBQ1csZ0JBQWdCLENBQUMsZ0JBQWdCLEVBQUdYLENBQUMsSUFBSyxJQUFJLENBQUM2VCxnQkFBZ0IsQ0FBQzdULENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDMUVPLElBQUFBLFFBQVEsQ0FBQ0ksZ0JBQWdCLENBQUMsaUJBQWlCLEVBQUdYLENBQUMsSUFDN0MsSUFBSSxDQUFDOFQsMEJBQTBCLENBQUM5VCxDQUFDLENBQ25DLENBQUMsQ0FBQTtFQUNELElBQUEsSUFBSSxDQUFDQSxDQUFDLENBQUNXLGdCQUFnQixDQUFDLE9BQU8sRUFBR1gsQ0FBQyxJQUFLLElBQUksQ0FBQytULFdBQVcsQ0FBQy9ULENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDNUQsSUFBQSxJQUFJLENBQUNBLENBQUMsQ0FBQ1csZ0JBQWdCLENBQUMsTUFBTSxFQUFHWCxDQUFDLElBQUssSUFBSSxDQUFDZ1UsVUFBVSxDQUFDaFUsQ0FBQyxDQUFDLENBQUMsQ0FBQTtNQUMxRCxJQUFJLENBQUNvUyxZQUFZLEdBQUcsSUFBSSxDQUFDcFMsQ0FBQyxDQUFDaVUsVUFBVSxDQUFDO0VBQ3hDLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7SUFDRWIsVUFBVUEsQ0FBQ0MsT0FBTyxFQUFFO0VBQ2xCO0VBQ0EsSUFBQSxPQUFPLElBQUksQ0FBQ3JULENBQUMsQ0FBQ2tVLFVBQVUsRUFBRTtRQUN4QixJQUFJLENBQUNsVSxDQUFDLENBQUNtVSxXQUFXLENBQUMsSUFBSSxDQUFDblUsQ0FBQyxDQUFDa1UsVUFBVSxDQUFDLENBQUE7RUFDdkMsS0FBQTtNQUNBLElBQUksQ0FBQy9CLEtBQUssR0FBR2tCLE9BQU8sQ0FBQzdSLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFBO01BQzVDLElBQUksQ0FBQ2lSLFNBQVMsR0FBRyxFQUFFLENBQUE7RUFDbkIsSUFBQSxLQUFLLElBQUkyQixPQUFPLEdBQUcsQ0FBQyxFQUFFQSxPQUFPLEdBQUcsSUFBSSxDQUFDakMsS0FBSyxDQUFDdlEsTUFBTSxFQUFFd1MsT0FBTyxFQUFFLEVBQUU7RUFDNUQsTUFBQSxJQUFJQyxFQUFFLEdBQUc5VCxRQUFRLENBQUNRLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQTtFQUN0QyxNQUFBLElBQUksQ0FBQ2YsQ0FBQyxDQUFDbUIsV0FBVyxDQUFDa1QsRUFBRSxDQUFDLENBQUE7RUFDdEIsTUFBQSxJQUFJLENBQUM1QixTQUFTLENBQUM1USxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUE7RUFDM0IsS0FBQTtNQUNBLElBQUksQ0FBQ3dRLFNBQVMsR0FBRyxJQUFJaUMsS0FBSyxDQUFDLElBQUksQ0FBQ25DLEtBQUssQ0FBQ3ZRLE1BQU0sQ0FBQyxDQUFBO01BQzdDLElBQUksQ0FBQzJTLGdCQUFnQixFQUFFLENBQUE7TUFDdkIsSUFBSSxDQUFDQyxVQUFVLEVBQUUsQ0FBQTtFQUNuQixHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNBO0VBQ0VDLEVBQUFBLFVBQVVBLEdBQUc7RUFDWCxJQUFBLE9BQU8sSUFBSSxDQUFDdEMsS0FBSyxDQUFDalEsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFBO0VBQzlCLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0VxUyxFQUFBQSxnQkFBZ0JBLEdBQUc7RUFDakI7RUFDQTtNQUNBLElBQUksQ0FBQ0csZUFBZSxFQUFFLENBQUE7RUFDdEI7TUFDQSxJQUFJLENBQUNDLGdCQUFnQixFQUFFLENBQUE7RUFDdkI7TUFDQSxJQUFJLENBQUNDLGNBQWMsRUFBRSxDQUFBO0VBQ3ZCLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0VELEVBQUFBLGdCQUFnQkEsR0FBRztNQUNqQixJQUFJLENBQUNuQyxVQUFVLEdBQUcsRUFBRSxDQUFBO0VBQ3BCLElBQUEsS0FBSyxJQUFJcUMsQ0FBQyxHQUFHLENBQUMsRUFBRUEsQ0FBQyxHQUFHLElBQUksQ0FBQzFDLEtBQUssQ0FBQ3ZRLE1BQU0sRUFBRWlULENBQUMsRUFBRSxFQUFFO1FBQzFDLElBQUksSUFBSSxDQUFDeEMsU0FBUyxDQUFDd0MsQ0FBQyxDQUFDLElBQUksMkJBQTJCLEVBQUU7RUFDcEQsUUFBQSxJQUFJLENBQUNyQyxVQUFVLENBQUMzUSxJQUFJLENBQ2xCLElBQUksQ0FBQ3lRLFlBQVksQ0FBQ3VDLENBQUMsQ0FBQyxDQUNsQmhHLFdBQVcsQ0FBQ3FCLHlCQUF5QixDQUFDQyxnQkFBZ0IsQ0FFMUQsQ0FBQyxDQUFBO0VBQ0gsT0FBQTtFQUNGLEtBQUE7RUFDRixHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0VwRSxFQUFBQSxPQUFPQSxDQUFDaUQsV0FBVyxFQUFFOEYsT0FBTyxFQUFFO0VBQzVCLElBQUEsT0FBTzlGLFdBQVcsQ0FBQ2pELE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxDQUFDZ0osR0FBRyxFQUFFQyxFQUFFLEVBQUVDLEVBQUUsS0FBSztRQUMvRCxJQUFJRCxFQUFFLElBQUksR0FBRyxFQUFFLE9BQU8xRCxVQUFVLENBQUN3RCxPQUFPLENBQUNHLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FFNUMsT0FBUSxDQUFrQyxnQ0FBQSxFQUFBLElBQUksQ0FBQ0MsbUJBQW1CLENBQ2hFSixPQUFPLENBQUNHLEVBQUUsQ0FDWixDQUFFLENBQVEsT0FBQSxDQUFBLENBQUE7RUFDZCxLQUFDLENBQUMsQ0FBQTtFQUNKLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7RUFDRUwsRUFBQUEsY0FBY0EsR0FBRztFQUNmLElBQUEsS0FBSyxJQUFJUixPQUFPLEdBQUcsQ0FBQyxFQUFFQSxPQUFPLEdBQUcsSUFBSSxDQUFDakMsS0FBSyxDQUFDdlEsTUFBTSxFQUFFd1MsT0FBTyxFQUFFLEVBQUU7RUFDNUQsTUFBQSxJQUFJLElBQUksQ0FBQzNCLFNBQVMsQ0FBQzJCLE9BQU8sQ0FBQyxFQUFFO0VBQzNCLFFBQUEsSUFBSWUsV0FBVyxHQUFHLElBQUksQ0FBQ3BKLE9BQU8sQ0FDNUIsSUFBSSxDQUFDd0csZ0JBQWdCLENBQUM2QixPQUFPLENBQUMsRUFDOUIsSUFBSSxDQUFDOUIsWUFBWSxDQUFDOEIsT0FBTyxDQUMzQixDQUFDLENBQUE7RUFDRDtFQUNBLFFBQUEsSUFBSSxDQUFDaEMsWUFBWSxDQUFDZ0MsT0FBTyxDQUFDLENBQUNwVCxTQUFTLEdBQUcsSUFBSSxDQUFDcVIsU0FBUyxDQUFDK0IsT0FBTyxDQUFDLENBQUE7VUFDOUQsSUFBSSxDQUFDaEMsWUFBWSxDQUFDZ0MsT0FBTyxDQUFDLENBQUNnQixlQUFlLENBQUMsT0FBTyxDQUFDLENBQUE7RUFDbkQsUUFBQSxJQUFJLENBQUNoRCxZQUFZLENBQUNnQyxPQUFPLENBQUMsQ0FBQ2pWLFNBQVMsR0FDbENnVyxXQUFXLElBQUksRUFBRSxHQUFHLFFBQVEsR0FBR0EsV0FBVyxDQUFDO0VBQy9DLE9BQUE7O1FBQ0EsSUFBSSxDQUFDL0MsWUFBWSxDQUFDZ0MsT0FBTyxDQUFDLENBQUNpQixPQUFPLENBQUNqQixPQUFPLEdBQUdBLE9BQU8sQ0FBQTtFQUN0RCxLQUFBO0VBQ0YsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0VNLEVBQUFBLGVBQWVBLEdBQUc7TUFDaEIsSUFBSVksYUFBYSxHQUFHLEtBQUssQ0FBQTtNQUN6QixJQUFJQyxrQkFBa0IsR0FBRyxDQUFDLENBQUE7TUFDMUIsSUFBSUMsU0FBUyxHQUFHLEtBQUssQ0FBQTtFQUVyQixJQUFBLEtBQUssSUFBSXBCLE9BQU8sR0FBRyxDQUFDLEVBQUVBLE9BQU8sR0FBRyxJQUFJLENBQUNqQyxLQUFLLENBQUN2USxNQUFNLEVBQUV3UyxPQUFPLEVBQUUsRUFBRTtRQUM1RCxJQUFJcUIsUUFBUSxHQUFHLFFBQVEsQ0FBQTtRQUN2QixJQUFJQyxXQUFXLEdBQUcsQ0FBQyxJQUFJLENBQUN2RCxLQUFLLENBQUNpQyxPQUFPLENBQUMsQ0FBQyxDQUFBO0VBQ3ZDLE1BQUEsSUFBSXVCLGVBQWUsR0FBRyxLQUFLLENBQUM7O0VBRTVCO0VBQ0E7UUFDQSxJQUFJTCxhQUFhLElBQUkseUJBQXlCLEVBQUU7RUFDOUM7RUFDQSxRQUFBLElBQUlSLE9BQU8sR0FBR2pHLFdBQVcsQ0FBQ1ksd0JBQXdCLENBQUNWLE1BQU0sQ0FBQ3pMLElBQUksQ0FDNUQsSUFBSSxDQUFDNk8sS0FBSyxDQUFDaUMsT0FBTyxDQUNwQixDQUFDLENBQUE7RUFDRCxRQUFBLElBQUlVLE9BQU8sSUFBSUEsT0FBTyxDQUFDYyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUNoVSxNQUFNLElBQUkyVCxrQkFBa0IsRUFBRTtFQUNqRUUsVUFBQUEsUUFBUSxHQUFHLDBCQUEwQixDQUFBO0VBQ3JDRSxVQUFBQSxlQUFlLEdBQUc5RyxXQUFXLENBQUNZLHdCQUF3QixDQUFDVCxXQUFXLENBQUE7RUFDbEUwRyxVQUFBQSxXQUFXLEdBQUdaLE9BQU8sQ0FBQTtFQUNyQlEsVUFBQUEsYUFBYSxHQUFHLEtBQUssQ0FBQTtFQUN2QixTQUFDLE1BQU07RUFDTEcsVUFBQUEsUUFBUSxHQUFHLHNCQUFzQixDQUFBO0VBQ2pDRSxVQUFBQSxlQUFlLEdBQUcsNENBQTRDLENBQUE7WUFDOURELFdBQVcsR0FBRyxDQUFDLElBQUksQ0FBQ3ZELEtBQUssQ0FBQ2lDLE9BQU8sQ0FBQyxDQUFDLENBQUE7RUFDckMsU0FBQTtFQUNGLE9BQUE7RUFDQTtFQUFBLFdBQ0ssSUFBSWtCLGFBQWEsSUFBSSxzQkFBc0IsRUFBRTtFQUNoRDtFQUNBLFFBQUEsSUFBSVIsT0FBTyxHQUFHakcsV0FBVyxDQUFDYSxxQkFBcUIsQ0FBQ1gsTUFBTSxDQUFDekwsSUFBSSxDQUN6RCxJQUFJLENBQUM2TyxLQUFLLENBQUNpQyxPQUFPLENBQ3BCLENBQUMsQ0FBQTtFQUNELFFBQUEsSUFBSVUsT0FBTyxJQUFJQSxPQUFPLENBQUNjLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQ2hVLE1BQU0sSUFBSTJULGtCQUFrQixFQUFFO0VBQ2pFRSxVQUFBQSxRQUFRLEdBQUcsdUJBQXVCLENBQUE7RUFDbENFLFVBQUFBLGVBQWUsR0FBRzlHLFdBQVcsQ0FBQ2EscUJBQXFCLENBQUNWLFdBQVcsQ0FBQTtFQUMvRDBHLFVBQUFBLFdBQVcsR0FBR1osT0FBTyxDQUFBO0VBQ3JCUSxVQUFBQSxhQUFhLEdBQUcsS0FBSyxDQUFBO0VBQ3ZCLFNBQUMsTUFBTTtFQUNMRyxVQUFBQSxRQUFRLEdBQUcsbUJBQW1CLENBQUE7RUFDOUJFLFVBQUFBLGVBQWUsR0FBRyw0Q0FBNEMsQ0FBQTtZQUM5REQsV0FBVyxHQUFHLENBQUMsSUFBSSxDQUFDdkQsS0FBSyxDQUFDaUMsT0FBTyxDQUFDLENBQUMsQ0FBQTtFQUNyQyxTQUFBO0VBQ0YsT0FBQTs7RUFFQTtFQUNBLE1BQUEsSUFBSXFCLFFBQVEsSUFBSSxRQUFRLElBQUlELFNBQVMsS0FBSyxLQUFLLEVBQUU7RUFDL0MsUUFBQSxLQUFLLElBQUlLLGFBQWEsSUFBSXpGLGdCQUFnQixFQUFFO0VBQzFDLFVBQUEsSUFBSSxJQUFJLENBQUMrQixLQUFLLENBQUNpQyxPQUFPLENBQUMsQ0FBQ3RTLEtBQUssQ0FBQytULGFBQWEsQ0FBQ3hGLEtBQUssQ0FBQyxFQUFFO0VBQ2xEO2NBQ0EsSUFDRXdGLGFBQWEsQ0FBQ3RGLGFBQWEsSUFDM0I2RCxPQUFPLElBQUksQ0FBQyxJQUNaLEVBQ0UsSUFBSSxDQUFDL0IsU0FBUyxDQUFDK0IsT0FBTyxHQUFHLENBQUMsQ0FBQyxJQUFJLFFBQVEsSUFDdkMsSUFBSSxDQUFDL0IsU0FBUyxDQUFDK0IsT0FBTyxHQUFHLENBQUMsQ0FBQyxJQUFJLE1BQU0sSUFDckMsSUFBSSxDQUFDL0IsU0FBUyxDQUFDK0IsT0FBTyxHQUFHLENBQUMsQ0FBQyxJQUFJLE1BQU0sSUFDckMsSUFBSSxDQUFDL0IsU0FBUyxDQUFDK0IsT0FBTyxHQUFHLENBQUMsQ0FBQyxJQUFJLGNBQWMsQ0FDOUMsRUFDRDtFQUNBb0IsY0FBQUEsU0FBUyxHQUFHSyxhQUFhLENBQUE7RUFDekIsY0FBQSxNQUFBO0VBQ0YsYUFBQTtFQUNGLFdBQUE7RUFDRixTQUFBO0VBQ0YsT0FBQTtRQUVBLElBQUlMLFNBQVMsS0FBSyxLQUFLLEVBQUU7RUFDdkJDLFFBQUFBLFFBQVEsR0FBRyxhQUFhLENBQUE7VUFDeEJFLGVBQWUsR0FBRyw2Q0FBNkMsQ0FBQztVQUNoRUQsV0FBVyxHQUFHLENBQUMsSUFBSSxDQUFDdkQsS0FBSyxDQUFDaUMsT0FBTyxDQUFDLENBQUMsQ0FBQzs7RUFFcEM7VUFDQSxJQUFJb0IsU0FBUyxDQUFDbEYsR0FBRyxFQUFFO0VBQ2pCO0VBQ0EsVUFBQSxJQUFJLElBQUksQ0FBQzZCLEtBQUssQ0FBQ2lDLE9BQU8sQ0FBQyxDQUFDdFMsS0FBSyxDQUFDMFQsU0FBUyxDQUFDbEYsR0FBRyxDQUFDLEVBQUU7RUFDNUNrRixZQUFBQSxTQUFTLEdBQUcsS0FBSyxDQUFBO0VBQ25CLFdBQUE7RUFDRixTQUFDLE1BQU07RUFDTDtZQUNBLElBQ0VwQixPQUFPLElBQUksSUFBSSxDQUFDakMsS0FBSyxDQUFDdlEsTUFBTSxHQUFHLENBQUMsSUFDaEMsSUFBSSxDQUFDdVEsS0FBSyxDQUFDaUMsT0FBTyxHQUFHLENBQUMsQ0FBQyxDQUFDdFMsS0FBSyxDQUFDK00sV0FBVyxDQUFDYyxXQUFXLENBQUNaLE1BQU0sQ0FBQyxFQUM3RDtFQUNBeUcsWUFBQUEsU0FBUyxHQUFHLEtBQUssQ0FBQTtFQUNuQixXQUFBO0VBQ0YsU0FBQTtFQUNGLE9BQUE7O0VBRUE7UUFDQSxJQUFJQyxRQUFRLElBQUksUUFBUSxFQUFFO0VBQ3hCLFFBQUEsS0FBSyxJQUFJcEssSUFBSSxJQUFJd0QsV0FBVyxFQUFFO0VBQzVCLFVBQUEsSUFBSUEsV0FBVyxDQUFDeEQsSUFBSSxDQUFDLENBQUMwRCxNQUFNLEVBQUU7RUFDNUIsWUFBQSxJQUFJK0YsT0FBTyxHQUFHakcsV0FBVyxDQUFDeEQsSUFBSSxDQUFDLENBQUMwRCxNQUFNLENBQUN6TCxJQUFJLENBQUMsSUFBSSxDQUFDNk8sS0FBSyxDQUFDaUMsT0FBTyxDQUFDLENBQUMsQ0FBQTtFQUNoRSxZQUFBLElBQUlVLE9BQU8sRUFBRTtFQUNYVyxjQUFBQSxRQUFRLEdBQUdwSyxJQUFJLENBQUE7RUFDZnNLLGNBQUFBLGVBQWUsR0FBRzlHLFdBQVcsQ0FBQ3hELElBQUksQ0FBQyxDQUFDMkQsV0FBVyxDQUFBO0VBQy9DMEcsY0FBQUEsV0FBVyxHQUFHWixPQUFPLENBQUE7RUFDckIsY0FBQSxNQUFBO0VBQ0YsYUFBQTtFQUNGLFdBQUE7RUFDRixTQUFBO0VBQ0YsT0FBQTs7RUFFQTtFQUNBLE1BQUEsSUFDRVcsUUFBUSxJQUFJLHlCQUF5QixJQUNyQ0EsUUFBUSxJQUFJLHNCQUFzQixFQUNsQztFQUNBSCxRQUFBQSxhQUFhLEdBQUdHLFFBQVEsQ0FBQTtVQUN4QkYsa0JBQWtCLEdBQUdHLFdBQVcsQ0FBQ0UsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDaFUsTUFBTSxDQUFBO0VBQ3ZELE9BQUE7O0VBRUE7UUFDQSxJQUNFLENBQUM2VCxRQUFRLElBQUksZ0JBQWdCLElBQzNCQSxRQUFRLElBQUksMkJBQTJCLEtBQ3pDckIsT0FBTyxHQUFHLENBQUMsS0FDVixJQUFJLENBQUMvQixTQUFTLENBQUMrQixPQUFPLEdBQUcsQ0FBQyxDQUFDLElBQUksUUFBUSxJQUN0QyxJQUFJLENBQUMvQixTQUFTLENBQUMrQixPQUFPLEdBQUcsQ0FBQyxDQUFDLElBQUksTUFBTSxJQUNyQyxJQUFJLENBQUMvQixTQUFTLENBQUMrQixPQUFPLEdBQUcsQ0FBQyxDQUFDLElBQUksTUFBTSxJQUNyQyxJQUFJLENBQUMvQixTQUFTLENBQUMrQixPQUFPLEdBQUcsQ0FBQyxDQUFDLElBQUksY0FBYyxDQUFDLEVBQ2hEO0VBQ0E7RUFDQXFCLFFBQUFBLFFBQVEsR0FBRyxRQUFRLENBQUE7VUFDbkJDLFdBQVcsR0FBRyxDQUFDLElBQUksQ0FBQ3ZELEtBQUssQ0FBQ2lDLE9BQU8sQ0FBQyxDQUFDLENBQUE7RUFDbkN1QixRQUFBQSxlQUFlLEdBQUcsS0FBSyxDQUFBO0VBQ3pCLE9BQUE7O0VBRUE7UUFDQSxJQUFJRixRQUFRLElBQUksa0JBQWtCLEVBQUU7RUFDbEMsUUFBQSxJQUFJWCxPQUFPLEdBQUdqRyxXQUFXLENBQUNrQixJQUFJLENBQUNoQixNQUFNLENBQUN6TCxJQUFJLENBQUMsSUFBSSxDQUFDNk8sS0FBSyxDQUFDaUMsT0FBTyxDQUFDLENBQUMsQ0FBQTtFQUMvRCxRQUFBLElBQUlVLE9BQU8sRUFBRTtFQUNYVyxVQUFBQSxRQUFRLEdBQUcsTUFBTSxDQUFBO0VBQ2pCRSxVQUFBQSxlQUFlLEdBQUc5RyxXQUFXLENBQUNrQixJQUFJLENBQUNmLFdBQVcsQ0FBQTtFQUM5QzBHLFVBQUFBLFdBQVcsR0FBR1osT0FBTyxDQUFBO0VBQ3ZCLFNBQUE7RUFDRixPQUFBOztFQUVBO0VBQ0EsTUFBQSxJQUFJVyxRQUFRLElBQUksa0JBQWtCLElBQUlBLFFBQVEsSUFBSSxrQkFBa0IsRUFBRTtFQUNwRSxRQUFBLElBQUlyQixPQUFPLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQy9CLFNBQVMsQ0FBQytCLE9BQU8sR0FBRyxDQUFDLENBQUMsSUFBSSxRQUFRLEVBQUU7RUFDM0Q7RUFDQSxVQUFBLElBQUlVLE9BQU8sR0FBR2pHLFdBQVcsQ0FBQ2lCLElBQUksQ0FBQ2YsTUFBTSxDQUFDekwsSUFBSSxDQUFDLElBQUksQ0FBQzZPLEtBQUssQ0FBQ2lDLE9BQU8sQ0FBQyxDQUFDLENBQUE7RUFDL0QsVUFBQSxJQUFJVSxPQUFPLEVBQUU7RUFDWDtFQUNBVyxZQUFBQSxRQUFRLEdBQUcsTUFBTSxDQUFBO0VBQ2pCQyxZQUFBQSxXQUFXLEdBQUdaLE9BQU8sQ0FBQTtFQUNyQmEsWUFBQUEsZUFBZSxHQUFHOUcsV0FBVyxDQUFDaUIsSUFBSSxDQUFDZCxXQUFXLENBQUE7RUFDaEQsV0FBQyxNQUFNO0VBQ0w7RUFDQXlHLFlBQUFBLFFBQVEsR0FBRyxRQUFRLENBQUE7Y0FDbkJDLFdBQVcsR0FBRyxDQUFDLElBQUksQ0FBQ3ZELEtBQUssQ0FBQ2lDLE9BQU8sQ0FBQyxDQUFDLENBQUE7RUFDbkN1QixZQUFBQSxlQUFlLEdBQUcsS0FBSyxDQUFBO0VBQ3pCLFdBQUE7RUFDRixTQUFDLE1BQU07RUFDTDtFQUNBLFVBQUEsSUFBSUcsV0FBVyxHQUFHMUIsT0FBTyxHQUFHLENBQUMsQ0FBQTtZQUM3QixNQUFNMkIsZUFBZSxHQUNuQk4sUUFBUSxJQUFJLGtCQUFrQixHQUFHLFlBQVksR0FBRyxZQUFZLENBQUE7WUFDOUQsR0FBRztjQUNELElBQUksSUFBSSxDQUFDcEQsU0FBUyxDQUFDMEQsZUFBZSxDQUFDLElBQUlBLGVBQWUsRUFBRTtFQUN0RCxjQUFBLElBQUksQ0FBQzFELFNBQVMsQ0FBQ3lELFdBQVcsQ0FBQyxHQUFHQyxlQUFlLENBQUE7RUFDN0MsY0FBQSxJQUFJLENBQUN0RCxTQUFTLENBQUNzRCxlQUFlLENBQUMsR0FBRyxJQUFJLENBQUE7RUFDeEMsYUFBQTtFQUNBLFlBQUEsSUFBSSxDQUFDeEQsZ0JBQWdCLENBQUN1RCxXQUFXLENBQUMsR0FBRyxLQUFLLENBQUE7RUFDMUMsWUFBQSxJQUFJLENBQUN4RCxZQUFZLENBQUN3RCxXQUFXLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQzNELEtBQUssQ0FBQzJELFdBQVcsQ0FBQyxDQUFDLENBQUE7RUFFMURBLFlBQUFBLFdBQVcsRUFBRSxDQUFBO0VBQ2YsV0FBQyxRQUFRQSxXQUFXLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQ3pELFNBQVMsQ0FBQ3lELFdBQVcsQ0FBQyxJQUFJLFFBQVEsRUFBQTtFQUN0RSxTQUFBO0VBQ0YsT0FBQTtFQUNBO1FBQ0EsSUFBSSxJQUFJLENBQUN6RCxTQUFTLENBQUMrQixPQUFPLENBQUMsSUFBSXFCLFFBQVEsRUFBRTtFQUN2QyxRQUFBLElBQUksQ0FBQ3BELFNBQVMsQ0FBQytCLE9BQU8sQ0FBQyxHQUFHcUIsUUFBUSxDQUFBO0VBQ2xDLFFBQUEsSUFBSSxDQUFDaEQsU0FBUyxDQUFDMkIsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFBO0VBQ2hDLE9BQUE7RUFDQSxNQUFBLElBQUksQ0FBQzdCLGdCQUFnQixDQUFDNkIsT0FBTyxDQUFDLEdBQUd1QixlQUFlLENBQUE7RUFDaEQsTUFBQSxJQUFJLENBQUNyRCxZQUFZLENBQUM4QixPQUFPLENBQUMsR0FBR3NCLFdBQVcsQ0FBQTtFQUMxQyxLQUFBO0VBQ0YsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDRU0sRUFBQUEsK0JBQStCQSxHQUFHO01BQ2hDLElBQUksQ0FBQ0MsY0FBYyxFQUFFLENBQUE7TUFDckIsSUFBSSxDQUFDQyxrQkFBa0IsRUFBRSxDQUFBO01BQ3pCLElBQUksQ0FBQzNCLGdCQUFnQixFQUFFLENBQUE7RUFDekIsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDRTRCLEVBQUFBLGdCQUFnQkEsQ0FBQ0MsY0FBYyxFQUFFQyxPQUFPLEVBQUU7RUFDeEM7RUFDQSxJQUFBLElBQUlDLFVBQVUsR0FBR0QsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUE7TUFDaEMsSUFBSUUsTUFBTSxHQUFHSCxjQUFjLENBQUNJLE1BQU0sQ0FBQyxDQUFDLEVBQUVGLFVBQVUsQ0FBQyxDQUFBO0VBQ2pELElBQUEsSUFBSWpMLElBQUksR0FBR2dMLE9BQU8sR0FBRyxTQUFTLEdBQUcsUUFBUSxDQUFBO01BQ3pDLElBQUlJLGFBQWEsR0FBR0gsVUFBVSxDQUFBO01BRTlCLElBQUlJLFlBQVksR0FBRyxDQUFDLENBQUE7TUFDcEIsSUFBSUMsUUFBUSxHQUFHLEtBQUssQ0FBQTtNQUNwQixJQUFJQyxPQUFPLEdBQUcsS0FBSyxDQUFBO01BQ25CLElBQUk5RixTQUFTLEdBQUcsRUFBRSxDQUFBO0VBQ2xCLElBQUEsSUFBSStGLFdBQVcsR0FBRyxFQUFFLENBQUM7O0VBRXJCQyxJQUFBQSxTQUFTLEVBQUUsT0FDVEwsYUFBYSxHQUFHTCxjQUFjLENBQUN4VSxNQUFNLElBQ3JDK1UsUUFBUSxLQUFLLEtBQUssNkJBQ2xCO0VBQ0EsTUFBQSxJQUFJdkYsTUFBTSxHQUFHZ0YsY0FBYyxDQUFDSSxNQUFNLENBQUNDLGFBQWEsQ0FBQyxDQUFBOztFQUVqRDtFQUNBO0VBQ0EsTUFBQSxLQUFLLElBQUl2RixJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxNQUFNLENBQUMsRUFBRTtFQUN2RCxRQUFBLElBQUk2RixHQUFHLEdBQUd2RyxhQUFhLENBQUNVLElBQUksQ0FBQyxDQUFDbkMsTUFBTSxDQUFDekwsSUFBSSxDQUFDOE4sTUFBTSxDQUFDLENBQUE7RUFDakQsUUFBQSxJQUFJMkYsR0FBRyxFQUFFO0VBQ1BOLFVBQUFBLGFBQWEsSUFBSU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFBO0VBQzlCLFVBQUEsU0FBU2tWLFNBQVMsQ0FBQTtFQUNwQixTQUFBO0VBQ0YsT0FBQTs7RUFFQTtRQUNBLElBQUkxRixNQUFNLENBQUN0UCxLQUFLLENBQUMwTyxhQUFhLENBQUNLLFNBQVMsQ0FBQzlCLE1BQU0sQ0FBQyxFQUFFO0VBQ2hEO0VBQ0EySCxRQUFBQSxZQUFZLEVBQUUsQ0FBQTtFQUNkRCxRQUFBQSxhQUFhLElBQUksQ0FBQyxDQUFBO0VBQ2xCLFFBQUEsU0FBU0ssU0FBUyxDQUFBO0VBQ3BCLE9BQUE7O0VBRUE7UUFDQSxJQUFJMUYsTUFBTSxDQUFDdFAsS0FBSyxDQUFDME8sYUFBYSxDQUFDSSxRQUFRLENBQUM3QixNQUFNLENBQUMsRUFBRTtFQUMvQztFQUNBO0VBQ0E7RUFDQTJILFFBQUFBLFlBQVksRUFBRSxDQUFBO0VBQ2Q7VUFDQSxJQUFJLENBQUNMLE9BQU8sRUFBRTtZQUNaLElBQUksSUFBSSxDQUFDRixnQkFBZ0IsQ0FBQy9FLE1BQU0sRUFBRSxLQUFLLENBQUMsRUFBRTtFQUN4QztFQUNBLFlBQUEsT0FBTyxLQUFLLENBQUE7RUFDZCxXQUFBO0VBQ0YsU0FBQTtFQUNBcUYsUUFBQUEsYUFBYSxJQUFJLENBQUMsQ0FBQTtFQUNsQixRQUFBLFNBQVNLLFNBQVMsQ0FBQTtFQUNwQixPQUFBOztFQUVBO0VBQ0EsTUFBQSxJQUFJMUYsTUFBTSxDQUFDdFAsS0FBSyxDQUFDLEtBQUssQ0FBQyxFQUFFO0VBQ3ZCNFUsUUFBQUEsWUFBWSxFQUFFLENBQUE7VUFDZCxJQUFJQSxZQUFZLElBQUksQ0FBQyxFQUFFO0VBQ3JCO1lBQ0FDLFFBQVEsR0FBR1AsY0FBYyxDQUFDSSxNQUFNLENBQzlCRixVQUFVLEVBQ1ZHLGFBQWEsR0FBR0gsVUFDbEIsQ0FBQyxDQUFBO0VBQ0RHLFVBQUFBLGFBQWEsRUFBRSxDQUFBO0VBQ2YsVUFBQSxTQUFTSyxTQUFTLENBQUE7RUFDcEIsU0FBQTtFQUNGLE9BQUE7O0VBRUE7RUFDQUwsTUFBQUEsYUFBYSxFQUFFLENBQUE7RUFDakIsS0FBQTs7RUFFQTtFQUNBLElBQUEsSUFBSUUsUUFBUSxLQUFLLEtBQUssRUFBRSxPQUFPLEtBQUssQ0FBQzs7RUFFckM7RUFDQSxJQUFBLElBQUlLLFFBQVEsR0FDVlAsYUFBYSxHQUFHTCxjQUFjLENBQUN4VSxNQUFNLEdBQ2pDd1UsY0FBYyxDQUFDSSxNQUFNLENBQUNDLGFBQWEsRUFBRSxDQUFDLENBQUMsR0FDdkMsRUFBRSxDQUFBOztFQUVSO01BQ0EsSUFBSU8sUUFBUSxJQUFJLEdBQUcsRUFBRTtFQUNuQixNQUFBLElBQUk1RixNQUFNLEdBQUdnRixjQUFjLENBQUNJLE1BQU0sQ0FBQ0MsYUFBYSxDQUFDLENBQUE7UUFDakQsSUFBSU0sR0FBRyxHQUFHdkcsYUFBYSxDQUFDTSxTQUFTLENBQUMvQixNQUFNLENBQUN6TCxJQUFJLENBQUM4TixNQUFNLENBQUMsQ0FBQTtFQUNyRCxNQUFBLElBQUkyRixHQUFHLEVBQUU7RUFDUDtFQUNBTixRQUFBQSxhQUFhLElBQUlNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQ25WLE1BQU0sQ0FBQTtFQUM5QmtQLFFBQUFBLFNBQVMsQ0FBQ2pQLElBQUksQ0FBQ2tWLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRUEsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFQSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtVQUN0QyxJQUFJQSxHQUFHLENBQUN2RyxhQUFhLENBQUNNLFNBQVMsQ0FBQ1gsZ0JBQWdCLENBQUMsRUFBRTtFQUNqRDtZQUNBeUcsT0FBTyxHQUFHRyxHQUFHLENBQUN2RyxhQUFhLENBQUNNLFNBQVMsQ0FBQ1gsZ0JBQWdCLENBQUMsQ0FBQTtFQUN6RCxTQUFDLE1BQU07RUFDTDtFQUNBeUcsVUFBQUEsT0FBTyxHQUFHRCxRQUFRLENBQUNNLElBQUksRUFBRSxDQUFBO0VBQzNCLFNBQUE7RUFDRixPQUFDLE1BQU07RUFDTDtFQUNBLFFBQUEsT0FBTyxLQUFLLENBQUE7RUFDZCxPQUFBO0VBQ0YsS0FBQyxNQUFNLElBQUlELFFBQVEsSUFBSSxHQUFHLEVBQUU7RUFDMUI7RUFDQUosTUFBQUEsT0FBTyxHQUFHRCxRQUFRLENBQUNNLElBQUksRUFBRSxDQUFBOztFQUV6QjtFQUNGLEtBQUMsTUFBTTtFQUNMOztFQUVBO0VBQ0FSLE1BQUFBLGFBQWEsRUFBRSxDQUFBO1FBRWYsSUFBSVMsZ0JBQWdCLEdBQUcsQ0FBQyxDQUFBO1FBQ3hCQyxXQUFXLEVBQUUsT0FDWFYsYUFBYSxHQUFHTCxjQUFjLENBQUN4VSxNQUFNLElBQ3JDc1YsZ0JBQWdCLEdBQUcsQ0FBQyxFQUNwQjtFQUNBLFFBQUEsSUFBSTlGLE1BQU0sR0FBR2dGLGNBQWMsQ0FBQ0ksTUFBTSxDQUFDQyxhQUFhLENBQUMsQ0FBQTs7RUFFakQ7RUFDQSxRQUFBLElBQUlNLEdBQUcsR0FBRyxNQUFNLENBQUN6VCxJQUFJLENBQUM4TixNQUFNLENBQUMsQ0FBQTtFQUM3QixRQUFBLElBQUkyRixHQUFHLEVBQUU7WUFDUCxRQUFRRixXQUFXLENBQUNqVixNQUFNO0VBQ3hCLFlBQUEsS0FBSyxDQUFDO0VBQ0ppVixjQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUNrVixHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUN4QixjQUFBLE1BQUE7RUFBTztFQUNULFlBQUEsS0FBSyxDQUFDO0VBQ0pGLGNBQUFBLFdBQVcsQ0FBQ2hWLElBQUksQ0FBQ2tWLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO0VBQ3hCLGNBQUEsTUFBQTtFQUFPO0VBQ1QsWUFBQSxLQUFLLENBQUM7RUFBRTtnQkFDTixJQUFJRixXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMvVSxLQUFLLENBQUMsR0FBRyxDQUFDLEVBQUU7RUFDN0IrVSxnQkFBQUEsV0FBVyxDQUFDLENBQUMsQ0FBQyxHQUFHQSxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM1VSxNQUFNLENBQUM4VSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUNoRCxlQUFDLE1BQU07RUFDTCxnQkFBQSxJQUFJRyxnQkFBZ0IsSUFBSSxDQUFDLEVBQUUsT0FBTyxLQUFLLENBQUM7RUFDeENMLGdCQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7a0JBQ3JCZ1YsV0FBVyxDQUFDaFYsSUFBSSxDQUFDa1YsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7RUFDM0IsZUFBQTs7RUFDQSxjQUFBLE1BQUE7RUFDRixZQUFBLEtBQUssQ0FBQztFQUNKRixjQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUNrVixHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUN4QixjQUFBLE1BQUE7RUFBTztFQUNULFlBQUEsS0FBSyxDQUFDO0VBQ0osY0FBQSxPQUFPLEtBQUssQ0FBQTtFQUFFO0VBQ2hCLFlBQUEsS0FBSyxDQUFDO0VBQ0pGLGNBQUFBLFdBQVcsQ0FBQ2hWLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQTtFQUFFO0VBQ3hCLFlBQUEsS0FBSyxDQUFDO0VBQ0pnVixjQUFBQSxXQUFXLENBQUMsQ0FBQyxDQUFDLEdBQUdBLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQzVVLE1BQU0sQ0FBQzhVLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO0VBQzlDLGNBQUEsTUFBQTtFQUFPO0VBQ1QsWUFBQSxLQUFLLENBQUM7RUFDSkYsY0FBQUEsV0FBVyxDQUFDLENBQUMsQ0FBQyxHQUFHQSxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM1VSxNQUFNLENBQUM4VSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUM5QyxjQUFBLE1BQUE7RUFBTztFQUNULFlBQUE7RUFDRSxjQUFBLE9BQU8sS0FBSyxDQUFBO0VBQUU7RUFDbEIsV0FBQTs7RUFDQU4sVUFBQUEsYUFBYSxJQUFJTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUNuVixNQUFNLENBQUE7RUFDOUIsVUFBQSxTQUFTdVYsV0FBVyxDQUFBO0VBQ3RCLFNBQUE7O0VBRUE7VUFDQUosR0FBRyxHQUFHdkcsYUFBYSxDQUFDQyxNQUFNLENBQUMxQixNQUFNLENBQUN6TCxJQUFJLENBQUM4TixNQUFNLENBQUMsQ0FBQTtFQUM5QyxRQUFBLElBQUkyRixHQUFHLEVBQUU7WUFDUCxRQUFRRixXQUFXLENBQUNqVixNQUFNO0VBQ3hCLFlBQUEsS0FBSyxDQUFDO0VBQ0ppVixjQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUE7RUFBRTtFQUN4QixZQUFBLEtBQUssQ0FBQztFQUNKZ1YsY0FBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDa1YsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDeEIsY0FBQSxNQUFBO0VBQU87RUFDVCxZQUFBLEtBQUssQ0FBQztFQUNKRixjQUFBQSxXQUFXLENBQUMsQ0FBQyxDQUFDLEdBQUdBLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQzVVLE1BQU0sQ0FBQzhVLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO0VBQzlDLGNBQUEsTUFBQTtFQUFPO0VBQ1QsWUFBQSxLQUFLLENBQUM7RUFDSixjQUFBLE9BQU8sS0FBSyxDQUFBO0VBQUU7RUFDaEIsWUFBQSxLQUFLLENBQUM7RUFDSixjQUFBLE9BQU8sS0FBSyxDQUFBO0VBQUU7RUFDaEIsWUFBQSxLQUFLLENBQUM7RUFDSkYsY0FBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBQUU7RUFDeEIsWUFBQSxLQUFLLENBQUM7RUFDSmdWLGNBQUFBLFdBQVcsQ0FBQyxDQUFDLENBQUMsR0FBR0EsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDNVUsTUFBTSxDQUFDOFUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDOUMsY0FBQSxNQUFBO0VBQU87RUFDVCxZQUFBO0VBQ0UsY0FBQSxPQUFPLEtBQUssQ0FBQTtFQUFFO0VBQ2xCLFdBQUE7O0VBQ0FOLFVBQUFBLGFBQWEsSUFBSU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFBO0VBQzlCLFVBQUEsU0FBU3VWLFdBQVcsQ0FBQTtFQUN0QixTQUFBOztFQUVBO0VBQ0EsUUFBQSxJQUFJTixXQUFXLENBQUNqVixNQUFNLEdBQUcsQ0FBQyxJQUFJd1AsTUFBTSxDQUFDdFAsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2hELElBQUkrVSxXQUFXLENBQUNqVixNQUFNLElBQUksQ0FBQyxFQUFFaVYsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBQ2pEZ1YsVUFBQUEsV0FBVyxDQUFDLENBQUMsQ0FBQyxHQUFHQSxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM1VSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUE7RUFDM0N3VSxVQUFBQSxhQUFhLEVBQUUsQ0FBQTtFQUNmLFVBQUEsU0FBU1UsV0FBVyxDQUFBO0VBQ3RCLFNBQUE7O0VBRUE7RUFDQSxRQUFBLElBQ0UsQ0FBQ04sV0FBVyxDQUFDalYsTUFBTSxJQUFJLENBQUMsSUFBSWlWLFdBQVcsQ0FBQ2pWLE1BQU0sSUFBSSxDQUFDLEtBQ25Ed1AsTUFBTSxDQUFDdFAsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUNsQjtFQUNBLFVBQUEsSUFBSStVLFdBQVcsQ0FBQ2pWLE1BQU0sSUFBSSxDQUFDLEVBQUVpVixXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7RUFDbERnVixVQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUE7RUFDckI0VSxVQUFBQSxhQUFhLEVBQUUsQ0FBQTtFQUNmLFVBQUEsU0FBU1UsV0FBVyxDQUFBO0VBQ3RCLFNBQUE7O0VBRUE7RUFDQUosUUFBQUEsR0FBRyxHQUFHLE9BQU8sQ0FBQ3pULElBQUksQ0FBQzhOLE1BQU0sQ0FBQyxDQUFBO0VBQzFCO0VBQ0E7VUFDQSxJQUNFMkYsR0FBRyxLQUNGRixXQUFXLENBQUNqVixNQUFNLElBQUksQ0FBQyxJQUN0QmlWLFdBQVcsQ0FBQ2pWLE1BQU0sSUFBSSxDQUFDLElBQ3ZCaVYsV0FBVyxDQUFDalYsTUFBTSxJQUFJLENBQUMsQ0FBQyxFQUMxQjtZQUNBLE9BQU9pVixXQUFXLENBQUNqVixNQUFNLEdBQUcsQ0FBQyxFQUFFaVYsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBQ25EZ1YsVUFBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDa1YsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDeEJOLFVBQUFBLGFBQWEsRUFBRSxDQUFBO0VBQ2YsVUFBQSxTQUFTVSxXQUFXLENBQUE7RUFDdEIsU0FBQTs7RUFFQTtVQUNBLElBQ0VKLEdBQUcsS0FDRkYsV0FBVyxDQUFDalYsTUFBTSxJQUFJLENBQUMsSUFBSWlWLFdBQVcsQ0FBQ2pWLE1BQU0sSUFBSSxDQUFDLENBQUMsSUFDcERpVixXQUFXLENBQUMsQ0FBQyxDQUFDLElBQUlFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDeEI7RUFDQSxVQUFBLElBQUlGLFdBQVcsQ0FBQ2pWLE1BQU0sSUFBSSxDQUFDLEVBQUVpVixXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7RUFDbERnVixVQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUNrVixHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQTtFQUN4Qk4sVUFBQUEsYUFBYSxFQUFFLENBQUE7RUFDZixVQUFBLFNBQVNVLFdBQVcsQ0FBQTtFQUN0QixTQUFBO0VBQ0E7O0VBRUE7RUFDQSxRQUFBLElBQUkvRixNQUFNLENBQUN0UCxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdkIsUUFBUStVLFdBQVcsQ0FBQ2pWLE1BQU07RUFDeEIsWUFBQSxLQUFLLENBQUM7RUFDSmlWLGNBQUFBLFdBQVcsQ0FBQ2hWLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQTtFQUFFO0VBQ3hCLFlBQUEsS0FBSyxDQUFDO0VBQ0pnVixjQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUE7RUFBRTtFQUN4QixZQUFBLEtBQUssQ0FBQztFQUFFO0VBQ05nVixjQUFBQSxXQUFXLENBQUMsQ0FBQyxDQUFDLEdBQUdBLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQzVVLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQTtFQUMzQyxjQUFBLElBQUksQ0FBQzRVLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQy9VLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRW9WLGdCQUFnQixFQUFFLENBQUE7RUFDbkQsY0FBQSxNQUFBO0VBQ0YsWUFBQSxLQUFLLENBQUM7RUFDSkwsY0FBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBQUU7RUFDeEIsWUFBQSxLQUFLLENBQUM7RUFDSmdWLGNBQUFBLFdBQVcsQ0FBQ2hWLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQTtFQUNyQixjQUFBLE1BQUE7RUFBTztFQUNULFlBQUEsS0FBSyxDQUFDO0VBQ0pnVixjQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUE7RUFBRTtFQUN4QixZQUFBLEtBQUssQ0FBQztFQUFFO2dCQUNOLElBQUlnVixXQUFXLENBQUMsQ0FBQyxDQUFDLElBQUksR0FBRyxFQUFFLE9BQU8sS0FBSyxDQUFBO0VBQ3ZDQSxjQUFBQSxXQUFXLENBQUMsQ0FBQyxDQUFDLEdBQUdBLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQzVVLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQTtFQUMzQyxjQUFBLE1BQUE7RUFDRixZQUFBO0VBQ0UsY0FBQSxPQUFPLEtBQUssQ0FBQTtFQUFFO0VBQ2xCLFdBQUE7O0VBQ0F3VSxVQUFBQSxhQUFhLEVBQUUsQ0FBQTtFQUNmLFVBQUEsU0FBU1UsV0FBVyxDQUFBO0VBQ3RCLFNBQUE7O0VBRUE7RUFDQSxRQUFBLElBQUkvRixNQUFNLENBQUN0UCxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUU7RUFDdkIsVUFBQSxJQUFJK1UsV0FBVyxDQUFDalYsTUFBTSxJQUFJLENBQUMsRUFBRTtFQUMzQjtjQUNBLE9BQU9pVixXQUFXLENBQUNqVixNQUFNLEdBQUcsQ0FBQyxFQUFFaVYsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBRW5ELFlBQUEsSUFBSSxDQUFDZ1YsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDL1UsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFb1YsZ0JBQWdCLEVBQUUsQ0FBQTtjQUVuRCxJQUFJQSxnQkFBZ0IsR0FBRyxDQUFDLEVBQUU7RUFDeEJMLGNBQUFBLFdBQVcsQ0FBQyxDQUFDLENBQUMsR0FBR0EsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDNVUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFBO0VBQzdDLGFBQUE7RUFDRixXQUFDLE1BQU0sSUFBSTRVLFdBQVcsQ0FBQ2pWLE1BQU0sSUFBSSxDQUFDLElBQUlpVixXQUFXLENBQUNqVixNQUFNLElBQUksQ0FBQyxFQUFFO0VBQzdEO0VBQ0EsWUFBQSxJQUFJaVYsV0FBVyxDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcsRUFBRTtFQUN6QjtnQkFDQSxJQUFJQSxXQUFXLENBQUNqVixNQUFNLElBQUksQ0FBQyxFQUFFaVYsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBQ2pEZ1YsY0FBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFBO0VBQ3ZCLGFBQUMsTUFBTTtFQUNMO2dCQUNBLElBQUlnVixXQUFXLENBQUNqVixNQUFNLElBQUksQ0FBQyxFQUFFaVYsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEtBQzlDZ1YsV0FBVyxDQUFDLENBQUMsQ0FBQyxHQUFHQSxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM1VSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUE7RUFDbEQsYUFBQTtFQUNGLFdBQUMsTUFBTTtjQUNMaVYsZ0JBQWdCLEVBQUUsQ0FBQztFQUNyQixXQUFBOztZQUVBLElBQUlBLGdCQUFnQixJQUFJLENBQUMsRUFBRTtFQUN6QjtjQUNBLE9BQU9MLFdBQVcsQ0FBQ2pWLE1BQU0sR0FBRyxDQUFDLEVBQUVpVixXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUE7RUFDckQsV0FBQTtFQUVBNFUsVUFBQUEsYUFBYSxFQUFFLENBQUE7RUFDZixVQUFBLFNBQVNVLFdBQVcsQ0FBQTtFQUN0QixTQUFBOztFQUVBO0VBQ0FKLFFBQUFBLEdBQUcsR0FBRyxJQUFJLENBQUN6VCxJQUFJLENBQUM4TixNQUFNLENBQUMsQ0FBQTtFQUN2QixRQUFBLElBQUkyRixHQUFHLEVBQUU7WUFDUCxRQUFRRixXQUFXLENBQUNqVixNQUFNO0VBQ3hCLFlBQUEsS0FBSyxDQUFDO0VBQ0ppVixjQUFBQSxXQUFXLENBQUNoVixJQUFJLENBQUMsRUFBRSxDQUFDLENBQUE7RUFBRTtFQUN4QixZQUFBLEtBQUssQ0FBQztFQUNKZ1YsY0FBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDa1YsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDeEIsY0FBQSxNQUFBO0VBQU87RUFDVCxZQUFBLEtBQUssQ0FBQztFQUNKRixjQUFBQSxXQUFXLENBQUMsQ0FBQyxDQUFDLEdBQUdBLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQzVVLE1BQU0sQ0FBQzhVLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO0VBQzlDLGNBQUEsTUFBQTtFQUFPO0VBQ1QsWUFBQSxLQUFLLENBQUM7RUFDSixjQUFBLE9BQU8sS0FBSyxDQUFBO0VBQUU7RUFDaEIsWUFBQSxLQUFLLENBQUM7RUFDSixjQUFBLE9BQU8sS0FBSyxDQUFBO0VBQUU7RUFDaEIsWUFBQSxLQUFLLENBQUM7RUFDSkYsY0FBQUEsV0FBVyxDQUFDaFYsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0VBQUU7RUFDeEIsWUFBQSxLQUFLLENBQUM7RUFDSmdWLGNBQUFBLFdBQVcsQ0FBQyxDQUFDLENBQUMsR0FBR0EsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDNVUsTUFBTSxDQUFDOFUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDOUMsY0FBQSxNQUFBO0VBQU87RUFDVCxZQUFBO0VBQ0UsY0FBQSxPQUFPLEtBQUssQ0FBQTtFQUFFO0VBQ2xCLFdBQUE7O0VBQ0FOLFVBQUFBLGFBQWEsSUFBSU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFBO0VBQzlCLFVBQUEsU0FBU3VWLFdBQVcsQ0FBQTtFQUN0QixTQUFBO1VBQ0EsTUFBTSxlQUFlLENBQUM7RUFDeEIsT0FBQTs7RUFDQSxNQUFBLElBQUlELGdCQUFnQixHQUFHLENBQUMsRUFBRSxPQUFPLEtBQUssQ0FBQztFQUN6QyxLQUFBOztNQUVBLElBQUlOLE9BQU8sS0FBSyxLQUFLLEVBQUU7RUFDckI7UUFDQSxJQUFJUSxLQUFLLEdBQUcsS0FBSyxDQUFBO0VBQ2pCLE1BQUEsS0FBSyxJQUFJQyxLQUFLLElBQUksSUFBSSxDQUFDN0UsVUFBVSxFQUFFO1VBQ2pDLElBQUk2RSxLQUFLLElBQUlULE9BQU8sRUFBRTtFQUNwQlEsVUFBQUEsS0FBSyxHQUFHLElBQUksQ0FBQTtFQUNaLFVBQUEsTUFBQTtFQUNGLFNBQUE7RUFDRixPQUFBO0VBQ0EsTUFBQSxJQUFJQyxLQUFLLEdBQUdELEtBQUssR0FDYiwrQkFBK0IsR0FDL0IsaUNBQWlDLENBQUE7RUFDckMsTUFBQSxJQUFJRSxNQUFNLEdBQUksQ0FBNkJqTSwyQkFBQUEsRUFBQUEsSUFBSyxLQUFJa0wsTUFBTyxDQUFBLG9CQUFBLEVBQXNCbEwsSUFBSyxDQUFBLENBQUEsRUFDcEZ5RixTQUFTLENBQUNsUCxNQUFNLEdBQUcsQ0FBQyxJQUFJLENBQUNrUCxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUd1RyxLQUFLLEdBQUcsRUFDakQsQ0FBQSxFQUFBLEVBQUksSUFBSSxDQUFDbkMsbUJBQW1CLENBQzNCeUIsUUFDRixDQUFFLENBQUEsa0NBQUEsRUFBb0N0TCxJQUFLLENBQVcsVUFBQSxDQUFBLENBQUE7RUFFdEQsTUFBQSxJQUFJeUYsU0FBUyxDQUFDbFAsTUFBTSxJQUFJLENBQUMsRUFBRTtFQUN6QjBWLFFBQUFBLE1BQU0sR0FBR0EsTUFBTSxDQUFDclYsTUFBTSxDQUNuQixDQUFBLDJCQUFBLEVBQTZCb0osSUFBSyxDQUFBLEVBQUEsRUFBSXlGLFNBQVMsQ0FBQyxDQUFDLENBQUUsU0FBUSxFQUMzRCxDQUFBLGFBQUEsRUFBZXVHLEtBQU0sQ0FBQSxFQUFBLEVBQUl2RyxTQUFTLENBQUMsQ0FBQyxDQUFFLFNBQVEsRUFDOUMsQ0FBQSwyQkFBQSxFQUE2QnpGLElBQUssQ0FBQSxFQUFBLEVBQUl5RixTQUFTLENBQUMsQ0FBQyxDQUFFLFNBQ3RELENBQUMsQ0FBQTtFQUNILE9BQUE7UUFDQSxPQUFPO0VBQ0x3RyxRQUFBQSxNQUFNLEVBQUVBLE1BQU07RUFDZEMsUUFBQUEsU0FBUyxFQUFFZCxhQUFBQTtTQUNaLENBQUE7T0FDRixNQUFNLElBQUlJLFdBQVcsRUFBRTtFQUN0Qjs7RUFFQTtFQUNBLE1BQUEsT0FBT0EsV0FBVyxDQUFDalYsTUFBTSxHQUFHLENBQUMsRUFBRTtFQUM3QmlWLFFBQUFBLFdBQVcsQ0FBQ2hWLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQTtFQUN0QixPQUFBO1FBRUEsT0FBTztVQUNMeVYsTUFBTSxFQUFHLENBQTZCak0sMkJBQUFBLEVBQUFBLElBQUssQ0FBSWtMLEVBQUFBLEVBQUFBLE1BQU8sQ0FBc0JsTCxvQkFBQUEsRUFBQUEsSUFBSyxDQUFJLEVBQUEsRUFBQSxJQUFJLENBQUM2SixtQkFBbUIsQ0FDM0d5QixRQUNGLENBQUUsQ0FBQSxrQ0FBQSxFQUFvQ3RMLElBQUssQ0FBQSxJQUFBLEVBQ3pDd0wsV0FBVyxDQUFDLENBQUMsQ0FDZCxDQUFzQnhMLG9CQUFBQSxFQUFBQSxJQUFLLENBQzFCd0wsYUFBQUEsRUFBQUEsV0FBVyxDQUFDLENBQUMsQ0FDZCxDQUFBLGtDQUFBLEVBQW9DeEwsSUFBSyxDQUFBLEVBQUEsRUFBSXdMLFdBQVcsQ0FBQyxDQUFDLENBQUUsQ0FDM0RBLEVBQUFBLFdBQVcsQ0FBQyxDQUFDLENBQ2QsQ0FBQSxFQUFFQSxXQUFXLENBQUMsQ0FBQyxDQUFFLENBQXNCeEwsb0JBQUFBLEVBQUFBLElBQUssQ0FDM0N3TCxPQUFBQSxFQUFBQSxXQUFXLENBQUMsQ0FBQyxDQUNkLENBQUEsa0NBQUEsRUFBb0N4TCxJQUFLLENBQUEsRUFBQSxFQUFJd0wsV0FBVyxDQUFDLENBQUMsQ0FBRSxDQUFTLFFBQUEsQ0FBQTtFQUN0RVUsUUFBQUEsU0FBUyxFQUFFZCxhQUFBQTtTQUNaLENBQUE7RUFDSCxLQUFBO0VBRUEsSUFBQSxPQUFPLEtBQUssQ0FBQTtFQUNkLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7RUFDQTtJQUNFdkIsbUJBQW1CQSxDQUFDa0IsY0FBYyxFQUFFO01BQ2xDLElBQUlvQixTQUFTLEdBQUcsRUFBRSxDQUFBO0VBQ2xCLElBQUEsSUFBSUMsS0FBSyxHQUFHLEVBQUUsQ0FBQztNQUNmLElBQUlDLE1BQU0sR0FBRyxDQUFDLENBQUE7TUFDZCxJQUFJdEcsTUFBTSxHQUFHZ0YsY0FBYyxDQUFBO01BRTNCMVQsS0FBSyxFQUFFLE9BQU8wTyxNQUFNLEVBQUU7RUFDcEI7RUFDQSxNQUFBLEtBQUssSUFBSUYsSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUUsTUFBTSxDQUFDLEVBQUU7RUFDdkQsUUFBQSxJQUFJNkYsR0FBRyxHQUFHdkcsYUFBYSxDQUFDVSxJQUFJLENBQUMsQ0FBQ25DLE1BQU0sQ0FBQ3pMLElBQUksQ0FBQzhOLE1BQU0sQ0FBQyxDQUFBO0VBQ2pELFFBQUEsSUFBSTJGLEdBQUcsRUFBRTtZQUNQM0YsTUFBTSxHQUFHQSxNQUFNLENBQUNvRixNQUFNLENBQUNPLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQ25WLE1BQU0sQ0FBQyxDQUFBO0VBQ3JDOFYsVUFBQUEsTUFBTSxJQUFJWCxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUNuVixNQUFNLENBQUE7RUFDdkI0VixVQUFBQSxTQUFTLElBQUloSCxhQUFhLENBQUNVLElBQUksQ0FBQyxDQUFDbEMsV0FBQUE7RUFDL0I7RUFBQSxXQUNDakQsT0FBTyxDQUFDLFlBQVksRUFBRSxDQUFDZ0osR0FBRyxFQUFFQyxFQUFFLEtBQUsxRCxVQUFVLENBQUN5RixHQUFHLENBQUMvQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUE7RUFDMUQsVUFBQSxTQUFTdFMsS0FBSyxDQUFBO0VBQ2hCLFNBQUE7RUFDRixPQUFBOztFQUVBO1FBQ0EsSUFBSWlWLGFBQWEsR0FBR3ZHLE1BQU0sQ0FBQ3RQLEtBQUssQ0FBQzBPLGFBQWEsQ0FBQ0ksUUFBUSxDQUFDN0IsTUFBTSxDQUFDLENBQUE7UUFDL0QsSUFBSTZJLGNBQWMsR0FBR3hHLE1BQU0sQ0FBQ3RQLEtBQUssQ0FBQzBPLGFBQWEsQ0FBQ0ssU0FBUyxDQUFDOUIsTUFBTSxDQUFDLENBQUE7UUFDakUsSUFBSTZJLGNBQWMsSUFBSUQsYUFBYSxFQUFFO1VBQ25DLElBQUluTyxNQUFNLEdBQUcsSUFBSSxDQUFDMk0sZ0JBQWdCLENBQUMvRSxNQUFNLEVBQUV3RyxjQUFjLENBQUMsQ0FBQTtFQUMxRCxRQUFBLElBQUlwTyxNQUFNLEVBQUU7RUFDVmdPLFVBQUFBLFNBQVMsR0FBSSxDQUFFQSxFQUFBQSxTQUFVLEdBQUVoTyxNQUFNLENBQUM4TixNQUFPLENBQUMsQ0FBQSxDQUFBO1lBQzFDbEcsTUFBTSxHQUFHQSxNQUFNLENBQUNvRixNQUFNLENBQUNoTixNQUFNLENBQUMrTixTQUFTLENBQUMsQ0FBQTtZQUN4Q0csTUFBTSxJQUFJbE8sTUFBTSxDQUFDK04sU0FBUyxDQUFBO0VBQzFCLFVBQUEsU0FBUzdVLEtBQUssQ0FBQTtFQUNoQixTQUFBO0VBQ0YsT0FBQTs7RUFFQTtFQUNBLE1BQUEsSUFBSXFVLEdBQUcsR0FBRyxjQUFjLENBQUN6VCxJQUFJLENBQUM4TixNQUFNLENBQUMsQ0FBQTtFQUNyQyxNQUFBLElBQUkyRixHQUFHLEVBQUU7RUFDUCxRQUFBLElBQUljLFVBQVUsR0FBR2QsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFBO0VBQzlCLFFBQUEsTUFBTWtXLFdBQVcsR0FBR2YsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFBO1VBQzFCLE1BQU1nQixnQkFBZ0IsR0FBR2hCLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQzs7VUFFbkMzRixNQUFNLEdBQUdBLE1BQU0sQ0FBQ29GLE1BQU0sQ0FBQ08sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFDLENBQUE7O0VBRXJDOztFQUVBLFFBQUEsTUFBTW9XLFNBQVMsR0FBR04sTUFBTSxHQUFHLENBQUMsR0FBR3RCLGNBQWMsQ0FBQ0ksTUFBTSxDQUFDLENBQUMsRUFBRWtCLE1BQU0sQ0FBQyxHQUFHLEdBQUcsQ0FBQztFQUN0RSxRQUFBLE1BQU1PLFNBQVMsR0FDYlAsTUFBTSxHQUFHWCxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUNuVixNQUFNLEdBQUd3VSxjQUFjLENBQUN4VSxNQUFNLEdBQUd3UCxNQUFNLEdBQUcsR0FBRyxDQUFBO0VBRS9ELFFBQUEsTUFBTThHLGtCQUFrQixHQUFHRCxTQUFTLENBQUNuVyxLQUFLLENBQUM2TSxrQkFBa0IsQ0FBQyxDQUFBO0VBQzlELFFBQUEsTUFBTXdKLG1CQUFtQixHQUFHSCxTQUFTLENBQUNsVyxLQUFLLENBQUM4TSxtQkFBbUIsQ0FBQyxDQUFBO0VBQ2hFLFFBQUEsTUFBTXdKLGlCQUFpQixHQUFHSCxTQUFTLENBQUNuVyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUE7RUFDaEQsUUFBQSxNQUFNdVcsa0JBQWtCLEdBQUdMLFNBQVMsQ0FBQ2xXLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQTs7RUFFakQ7RUFDQSxRQUFBLElBQUl3VyxPQUFPLEdBQ1QsQ0FBQ0YsaUJBQWlCLEtBQ2pCLENBQUNGLGtCQUFrQixJQUNsQixDQUFDLENBQUNHLGtCQUFrQixJQUNwQixDQUFDLENBQUNGLG1CQUFtQixDQUFDLENBQUE7RUFDMUIsUUFBQSxJQUFJSSxRQUFRLEdBQ1YsQ0FBQ0Ysa0JBQWtCLEtBQ2xCLENBQUNGLG1CQUFtQixJQUFJLENBQUMsQ0FBQ0MsaUJBQWlCLElBQUksQ0FBQyxDQUFDRixrQkFBa0IsQ0FBQyxDQUFBOztFQUV2RTtFQUNBLFFBQUEsSUFBSUgsZ0JBQWdCLElBQUksR0FBRyxJQUFJTyxPQUFPLElBQUlDLFFBQVEsRUFBRTtFQUNsREQsVUFBQUEsT0FBTyxHQUFHSCxtQkFBbUIsQ0FBQTtFQUM3QkksVUFBQUEsUUFBUSxHQUFHTCxrQkFBa0IsQ0FBQTtFQUMvQixTQUFBOztFQUVBO0VBQ0EsUUFBQSxJQUFJSyxRQUFRLEVBQUU7RUFDWixVQUFBLElBQUlDLFlBQVksR0FBR2YsS0FBSyxDQUFDN1YsTUFBTSxHQUFHLENBQUMsQ0FBQTtFQUNuQztFQUNBLFVBQUEsT0FBT2lXLFVBQVUsSUFBSVcsWUFBWSxJQUFJLENBQUMsRUFBRTtjQUN0QyxJQUFJZixLQUFLLENBQUNlLFlBQVksQ0FBQyxDQUFDQyxTQUFTLElBQUlWLGdCQUFnQixFQUFFO0VBQ3JEOztFQUVBO0VBQ0EsY0FBQSxPQUFPUyxZQUFZLEdBQUdmLEtBQUssQ0FBQzdWLE1BQU0sR0FBRyxDQUFDLEVBQUU7RUFDdEMsZ0JBQUEsTUFBTThXLEtBQUssR0FBR2pCLEtBQUssQ0FBQ2tCLEdBQUcsRUFBRSxDQUFBO2tCQUN6Qm5CLFNBQVMsR0FBSSxHQUFFa0IsS0FBSyxDQUFDcEIsTUFBTyxDQUFFb0IsRUFBQUEsS0FBSyxDQUFDWixXQUFXLENBQUN0QixNQUFNLENBQ3BELENBQUMsRUFDRGtDLEtBQUssQ0FBQ0UsS0FDUixDQUFFLENBQUEsRUFBRXBCLFNBQVUsQ0FBQyxDQUFBLENBQUE7RUFDakIsZUFBQTs7RUFFQTtFQUNBLGNBQUEsSUFBSUssVUFBVSxJQUFJLENBQUMsSUFBSUosS0FBSyxDQUFDZSxZQUFZLENBQUMsQ0FBQ0ksS0FBSyxJQUFJLENBQUMsRUFBRTtFQUNyRDtrQkFDQXBCLFNBQVMsR0FBSSxDQUF1Qk8scUJBQUFBLEVBQUFBLGdCQUFpQixDQUFFQSxFQUFBQSxnQkFBaUIsQ0FBa0NQLGdDQUFBQSxFQUFBQSxTQUFVLENBQWdDTyw4QkFBQUEsRUFBQUEsZ0JBQWlCLENBQUVBLEVBQUFBLGdCQUFpQixDQUFRLE9BQUEsQ0FBQSxDQUFBO0VBQ2hNRixnQkFBQUEsVUFBVSxJQUFJLENBQUMsQ0FBQTtFQUNmSixnQkFBQUEsS0FBSyxDQUFDZSxZQUFZLENBQUMsQ0FBQ0ksS0FBSyxJQUFJLENBQUMsQ0FBQTtFQUNoQyxlQUFDLE1BQU07RUFDTDtFQUNBcEIsZ0JBQUFBLFNBQVMsR0FBSSxDQUF1Qk8scUJBQUFBLEVBQUFBLGdCQUFpQiwyQkFBMEJQLFNBQVUsQ0FBQSwwQkFBQSxFQUE0Qk8sZ0JBQWlCLENBQVEsT0FBQSxDQUFBLENBQUE7RUFDOUlGLGdCQUFBQSxVQUFVLElBQUksQ0FBQyxDQUFBO0VBQ2ZKLGdCQUFBQSxLQUFLLENBQUNlLFlBQVksQ0FBQyxDQUFDSSxLQUFLLElBQUksQ0FBQyxDQUFBO0VBQ2hDLGVBQUE7O0VBRUE7Z0JBQ0EsSUFBSW5CLEtBQUssQ0FBQ2UsWUFBWSxDQUFDLENBQUNJLEtBQUssSUFBSSxDQUFDLEVBQUU7RUFDbEMsZ0JBQUEsSUFBSUYsS0FBSyxHQUFHakIsS0FBSyxDQUFDa0IsR0FBRyxFQUFFLENBQUE7RUFDdkJuQixnQkFBQUEsU0FBUyxHQUFJLENBQUVrQixFQUFBQSxLQUFLLENBQUNwQixNQUFPLENBQUEsRUFBRUUsU0FBVSxDQUFDLENBQUEsQ0FBQTtFQUN6Q2dCLGdCQUFBQSxZQUFZLEVBQUUsQ0FBQTtFQUNoQixlQUFBO0VBQ0YsYUFBQyxNQUFNO0VBQ0w7RUFDQTtFQUNBQSxjQUFBQSxZQUFZLEVBQUUsQ0FBQTtFQUNoQixhQUFBO0VBQ0YsV0FBQTtFQUNGLFNBQUE7RUFDQTtVQUNBLElBQUlYLFVBQVUsSUFBSVMsT0FBTyxFQUFFO1lBQ3pCYixLQUFLLENBQUM1VixJQUFJLENBQUM7RUFDVDRXLFlBQUFBLFNBQVMsRUFBRVYsZ0JBQWdCO0VBQzNCRCxZQUFBQSxXQUFXLEVBQUVBLFdBQVc7RUFDeEJjLFlBQUFBLEtBQUssRUFBRWYsVUFBVTtFQUNqQlAsWUFBQUEsTUFBTSxFQUFFRSxTQUFBQTtFQUNWLFdBQUMsQ0FBQyxDQUFBO1lBQ0ZBLFNBQVMsR0FBRyxFQUFFLENBQUM7RUFDZkssVUFBQUEsVUFBVSxHQUFHLENBQUMsQ0FBQTtFQUNoQixTQUFBOztFQUVBO0VBQ0EsUUFBQSxJQUFJQSxVQUFVLEVBQUU7RUFDZEwsVUFBQUEsU0FBUyxHQUFJLENBQUEsRUFBRUEsU0FBVSxDQUFBLEVBQUVNLFdBQVcsQ0FBQ3RCLE1BQU0sQ0FBQyxDQUFDLEVBQUVxQixVQUFVLENBQUUsQ0FBQyxDQUFBLENBQUE7RUFDaEUsU0FBQTtFQUVBSCxRQUFBQSxNQUFNLElBQUlYLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQ25WLE1BQU0sQ0FBQTtFQUN2QixRQUFBLFNBQVNjLEtBQUssQ0FBQTtFQUNoQixPQUFBOztFQUVBO0VBQ0FxVSxNQUFBQSxHQUFHLEdBQUcsS0FBSyxDQUFDelQsSUFBSSxDQUFDOE4sTUFBTSxDQUFDLENBQUE7RUFDeEIsTUFBQSxJQUFJMkYsR0FBRyxFQUFFO1VBQ1AsSUFBSThCLFFBQVEsR0FBRyxLQUFLLENBQUE7RUFDcEIsUUFBQSxJQUFJTCxZQUFZLEdBQUdmLEtBQUssQ0FBQzdWLE1BQU0sR0FBRyxDQUFDLENBQUE7RUFDbkM7RUFDQSxRQUFBLE9BQU8sQ0FBQ2lYLFFBQVEsSUFBSUwsWUFBWSxJQUFJLENBQUMsRUFBRTtZQUNyQyxJQUFJZixLQUFLLENBQUNlLFlBQVksQ0FBQyxDQUFDQyxTQUFTLElBQUksR0FBRyxFQUFFO0VBQ3hDOztFQUVBO0VBQ0EsWUFBQSxPQUFPRCxZQUFZLEdBQUdmLEtBQUssQ0FBQzdWLE1BQU0sR0FBRyxDQUFDLEVBQUU7RUFDdEMsY0FBQSxNQUFNOFcsS0FBSyxHQUFHakIsS0FBSyxDQUFDa0IsR0FBRyxFQUFFLENBQUE7Z0JBQ3pCbkIsU0FBUyxHQUFJLEdBQUVrQixLQUFLLENBQUNwQixNQUFPLENBQUVvQixFQUFBQSxLQUFLLENBQUNaLFdBQVcsQ0FBQ3RCLE1BQU0sQ0FDcEQsQ0FBQyxFQUNEa0MsS0FBSyxDQUFDRSxLQUNSLENBQUUsQ0FBQSxFQUFFcEIsU0FBVSxDQUFDLENBQUEsQ0FBQTtFQUNqQixhQUFBOztFQUVBO2NBQ0FBLFNBQVMsR0FBSSxDQUE2REEsMkRBQUFBLEVBQUFBLFNBQVUsQ0FBcUMsb0NBQUEsQ0FBQSxDQUFBO0VBQ3pILFlBQUEsSUFBSWtCLEtBQUssR0FBR2pCLEtBQUssQ0FBQ2tCLEdBQUcsRUFBRSxDQUFBO0VBQ3ZCbkIsWUFBQUEsU0FBUyxHQUFJLENBQUVrQixFQUFBQSxLQUFLLENBQUNwQixNQUFPLENBQUEsRUFBRUUsU0FBVSxDQUFDLENBQUEsQ0FBQTtFQUN6Q3FCLFlBQUFBLFFBQVEsR0FBRyxJQUFJLENBQUE7RUFDakIsV0FBQyxNQUFNO0VBQ0w7RUFDQTtFQUNBTCxZQUFBQSxZQUFZLEVBQUUsQ0FBQTtFQUNoQixXQUFBO0VBQ0YsU0FBQTs7RUFFQTtVQUNBLElBQUksQ0FBQ0ssUUFBUSxFQUFFO1lBQ2JwQixLQUFLLENBQUM1VixJQUFJLENBQUM7RUFDVDRXLFlBQUFBLFNBQVMsRUFBRSxHQUFHO0VBQ2RYLFlBQUFBLFdBQVcsRUFBRSxJQUFJO0VBQ2pCYyxZQUFBQSxLQUFLLEVBQUUsQ0FBQztFQUNSdEIsWUFBQUEsTUFBTSxFQUFFRSxTQUFBQTtFQUNWLFdBQUMsQ0FBQyxDQUFBO1lBQ0ZBLFNBQVMsR0FBRyxFQUFFLENBQUM7RUFDakIsU0FBQTs7RUFFQUUsUUFBQUEsTUFBTSxJQUFJWCxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUNuVixNQUFNLENBQUE7VUFDdkJ3UCxNQUFNLEdBQUdBLE1BQU0sQ0FBQ29GLE1BQU0sQ0FBQ08sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFDLENBQUE7RUFDckMsUUFBQSxTQUFTYyxLQUFLLENBQUE7RUFDaEIsT0FBQTs7RUFFQTtRQUNBcVUsR0FBRyxHQUFHdkcsYUFBYSxDQUFDTyxPQUFPLENBQUNoQyxNQUFNLENBQUN6TCxJQUFJLENBQUM4TixNQUFNLENBQUMsQ0FBQTtFQUMvQyxNQUFBLElBQUkyRixHQUFHLEVBQUU7VUFDUDNGLE1BQU0sR0FBR0EsTUFBTSxDQUFDb0YsTUFBTSxDQUFDTyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUNuVixNQUFNLENBQUMsQ0FBQTtFQUNyQzhWLFFBQUFBLE1BQU0sSUFBSVgsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDblYsTUFBTSxDQUFBO1VBQ3ZCNFYsU0FBUyxJQUFJaEgsYUFBYSxDQUFDTyxPQUFPLENBQUMvQixXQUFXLENBQUNqRCxPQUFPLENBQ3BELFlBQVksRUFDWixDQUFDZ0osR0FBRyxFQUFFQyxFQUFFLEtBQUsxRCxVQUFVLENBQUN5RixHQUFHLENBQUMvQixFQUFFLENBQUMsQ0FDakMsQ0FBQyxDQUFBO0VBQ0QsUUFBQSxTQUFTdFMsS0FBSyxDQUFBO0VBQ2hCLE9BQUE7RUFDQSxNQUFBLE1BQU0sZ0JBQWdCLENBQUE7RUFDeEIsS0FBQTs7RUFFQTtNQUNBLE9BQU8rVSxLQUFLLENBQUM3VixNQUFNLEVBQUU7RUFDbkIsTUFBQSxNQUFNOFcsS0FBSyxHQUFHakIsS0FBSyxDQUFDa0IsR0FBRyxFQUFFLENBQUE7UUFDekJuQixTQUFTLEdBQUksR0FBRWtCLEtBQUssQ0FBQ3BCLE1BQU8sQ0FBRW9CLEVBQUFBLEtBQUssQ0FBQ1osV0FBVyxDQUFDdEIsTUFBTSxDQUNwRCxDQUFDLEVBQ0RrQyxLQUFLLENBQUNFLEtBQ1IsQ0FBRSxDQUFBLEVBQUVwQixTQUFVLENBQUMsQ0FBQSxDQUFBO0VBQ2pCLEtBQUE7RUFFQSxJQUFBLE9BQU9BLFNBQVMsQ0FBQTtFQUNsQixHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNFdkIsRUFBQUEsY0FBY0EsR0FBRztNQUNmLElBQUksQ0FBQ3hELFNBQVMsR0FBRyxJQUFJNkIsS0FBSyxDQUFDLElBQUksQ0FBQ25DLEtBQUssQ0FBQ3ZRLE1BQU0sQ0FBQyxDQUFBO0VBQzdDLElBQUEsS0FBSyxJQUFJRCxDQUFDLEdBQUcsQ0FBQyxFQUFFQSxDQUFDLEdBQUcsSUFBSSxDQUFDOFEsU0FBUyxDQUFDN1EsTUFBTSxFQUFFRCxDQUFDLEVBQUUsRUFBRTtFQUM5QyxNQUFBLElBQUksQ0FBQzhRLFNBQVMsQ0FBQzlRLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQTtFQUMzQixLQUFBO0VBQ0YsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNFdVUsRUFBQUEsa0JBQWtCQSxHQUFHO0VBQ25CO0VBQ0E7RUFDQTtFQUNBLElBQUEsSUFBSTRDLFNBQVMsR0FBRyxJQUFJLENBQUM5WSxDQUFDLENBQUMrWSxpQkFBaUIsR0FBRyxJQUFJLENBQUM1RyxLQUFLLENBQUN2USxNQUFNLENBQUE7RUFDNUQsSUFBQSxJQUFJa1gsU0FBUyxFQUFFO0VBQ2I7RUFDQTtRQUNBLElBQUlFLGdCQUFnQixHQUFHLENBQUMsQ0FBQTtRQUN4QixPQUNFQSxnQkFBZ0IsSUFBSSxJQUFJLENBQUM3RyxLQUFLLENBQUN2USxNQUFNLElBQ3JDb1gsZ0JBQWdCLElBQUksSUFBSSxDQUFDNUcsWUFBWSxDQUFDeFEsTUFBTSxJQUM1QyxJQUFJLENBQUN3USxZQUFZLENBQUM0RyxnQkFBZ0IsQ0FBQztFQUFJO0VBQ3ZDLE1BQUEsSUFBSSxDQUFDN0csS0FBSyxDQUFDNkcsZ0JBQWdCLENBQUMsSUFDMUIsSUFBSSxDQUFDNUcsWUFBWSxDQUFDNEcsZ0JBQWdCLENBQUMsQ0FBQ0MsV0FBVyxJQUNqRCxJQUFJLENBQUM1RyxTQUFTLENBQUMyRyxnQkFBZ0IsQ0FBQyxJQUM5QixJQUFJLENBQUM1RyxZQUFZLENBQUM0RyxnQkFBZ0IsQ0FBQyxDQUFDaFksU0FBUyxFQUMvQztFQUNBZ1ksUUFBQUEsZ0JBQWdCLEVBQUUsQ0FBQTtFQUNwQixPQUFBOztFQUVBO1FBQ0EsSUFBSUUsZUFBZSxHQUFHLENBQUMsQ0FBQyxDQUFBO0VBQ3hCLE1BQUEsT0FDRSxDQUFDQSxlQUFlLEdBQUcsSUFBSSxDQUFDL0csS0FBSyxDQUFDdlEsTUFBTSxJQUNwQyxDQUFDc1gsZUFBZSxHQUFHLElBQUksQ0FBQzlHLFlBQVksQ0FBQ3hRLE1BQU0sSUFDM0MsSUFBSSxDQUFDdVEsS0FBSyxDQUFDLElBQUksQ0FBQ0EsS0FBSyxDQUFDdlEsTUFBTSxHQUFHc1gsZUFBZSxDQUFDLElBQzdDLElBQUksQ0FBQzlHLFlBQVksQ0FBQyxJQUFJLENBQUNBLFlBQVksQ0FBQ3hRLE1BQU0sR0FBR3NYLGVBQWUsQ0FBQyxDQUMxREQsV0FBVyxJQUNoQixJQUFJLENBQUM1RyxTQUFTLENBQUMsSUFBSSxDQUFDRixLQUFLLENBQUN2USxNQUFNLEdBQUdzWCxlQUFlLENBQUMsSUFDakQsSUFBSSxDQUFDOUcsWUFBWSxDQUFDLElBQUksQ0FBQ0EsWUFBWSxDQUFDeFEsTUFBTSxHQUFHc1gsZUFBZSxDQUFDLENBQzFEbFksU0FBUyxFQUNkO0VBQ0FrWSxRQUFBQSxlQUFlLEVBQUUsQ0FBQTtFQUNuQixPQUFBO0VBRUEsTUFBQSxJQUFJQyxhQUFhLEdBQ2YsSUFBSSxDQUFDaEgsS0FBSyxDQUFDdlEsTUFBTSxHQUFHc1gsZUFBZSxHQUFHLENBQUMsR0FBR0YsZ0JBQWdCLENBQUE7UUFDNUQsSUFBSUcsYUFBYSxHQUFHLENBQUNMLFNBQVMsRUFBRUssYUFBYSxHQUFHLENBQUNMLFNBQVMsQ0FBQTtFQUMxRCxNQUFBLElBQUlLLGFBQWEsR0FBRyxDQUFDLEVBQUVBLGFBQWEsR0FBRyxDQUFDLENBQUE7UUFFeEMsSUFBSUMsVUFBVSxHQUFHLEVBQUUsQ0FBQTtFQUNuQixNQUFBLEtBQUssSUFBSXZFLENBQUMsR0FBRyxDQUFDLEVBQUVBLENBQUMsR0FBR3NFLGFBQWEsR0FBR0wsU0FBUyxFQUFFakUsQ0FBQyxFQUFFLEVBQUU7RUFDbER1RSxRQUFBQSxVQUFVLENBQUN2WCxJQUFJLENBQUMsSUFBSSxDQUFDdVEsWUFBWSxDQUFDNEcsZ0JBQWdCLEdBQUduRSxDQUFDLENBQUMsQ0FBQ29FLFdBQVcsQ0FBQyxDQUFBO0VBQ3RFLE9BQUE7UUFDQSxJQUFJLENBQUNJLFdBQVcsQ0FBQ0wsZ0JBQWdCLEVBQUVHLGFBQWEsRUFBRUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFBO0VBQ3RFLEtBQUMsTUFBTTtFQUNMO0VBQ0EsTUFBQSxLQUFLLElBQUlFLElBQUksR0FBRyxDQUFDLEVBQUVBLElBQUksR0FBRyxJQUFJLENBQUNsSCxZQUFZLENBQUN4USxNQUFNLEVBQUUwWCxJQUFJLEVBQUUsRUFBRTtFQUMxRCxRQUFBLElBQUl0WixDQUFDLEdBQUcsSUFBSSxDQUFDb1MsWUFBWSxDQUFDa0gsSUFBSSxDQUFDLENBQUE7RUFDL0IsUUFBQSxJQUFJQyxFQUFFLEdBQUd2WixDQUFDLENBQUNpWixXQUFXLENBQUE7RUFDdEIsUUFBQSxJQUFJLElBQUksQ0FBQzlHLEtBQUssQ0FBQ21ILElBQUksQ0FBQyxLQUFLQyxFQUFFLElBQUksSUFBSSxDQUFDbEgsU0FBUyxDQUFDaUgsSUFBSSxDQUFDLEtBQUt0WixDQUFDLENBQUNnQixTQUFTLEVBQUU7RUFDbkU7RUFDQSxVQUFBLElBQUksQ0FBQ21SLEtBQUssQ0FBQ21ILElBQUksQ0FBQyxHQUFHQyxFQUFFLENBQUE7WUFDckIsSUFBSSxDQUFDbEgsU0FBUyxDQUFDaUgsSUFBSSxDQUFDLEdBQUd0WixDQUFDLENBQUNnQixTQUFTLENBQUE7RUFDbEMsVUFBQSxJQUFJLENBQUN5UixTQUFTLENBQUM2RyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUE7RUFDN0IsU0FBQTtFQUNGLE9BQUE7RUFDRixLQUFBO0VBQ0YsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtJQUNFRSxtQkFBbUJBLENBQUNDLEdBQUcsRUFBRTtNQUN2QixJQUFJLENBQUNBLEdBQUcsRUFBRSxPQUFBOztFQUVWO01BQ0EsSUFBSSxDQUFDdkQsa0JBQWtCLEVBQUUsQ0FBQTtNQUV6QixJQUFJd0QsZUFBZSxHQUFHLEtBQUssQ0FBQTtFQUMzQjs7RUFFQSxJQUFBLElBQUlDLFNBQVMsR0FBR0YsR0FBRyxDQUFDRyxHQUFHLEdBQUcsQ0FBQyxHQUFHSCxHQUFHLENBQUNJLEdBQUcsR0FBR0osR0FBRyxDQUFDSSxHQUFHLEdBQUcsQ0FBQyxDQUFBO0VBQ25ELElBQUEsUUFBUSxJQUFJLENBQUN4SCxTQUFTLENBQUNzSCxTQUFTLENBQUM7RUFDL0IsTUFBQSxLQUFLLE1BQU07RUFDVEQsUUFBQUEsZUFBZSxHQUFHLE1BQU0sQ0FBQTtFQUN4QixRQUFBLE1BQUE7RUFDRixNQUFBLEtBQUssTUFBTTtFQUNUQSxRQUFBQSxlQUFlLEdBQUcsTUFBTSxDQUFBO0VBQ3hCLFFBQUEsTUFBQTtFQUNGLE1BQUEsS0FBSyxnQkFBZ0I7RUFDbkJBLFFBQUFBLGVBQWUsR0FBRyxnQkFBZ0IsQ0FBQTtFQUNsQyxRQUFBLE1BQUE7RUFDSixLQUFBO01BRUEsSUFBSXZILEtBQUssR0FBRyxJQUFJLENBQUNBLEtBQUssQ0FBQ3NILEdBQUcsQ0FBQ0ksR0FBRyxDQUFDLENBQzVCOU4sT0FBTyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FDdEJ2SyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQTtFQUMxQixJQUFBLElBQUkyUSxLQUFLLENBQUN2USxNQUFNLEdBQUcsQ0FBQyxFQUFFO0VBQ3BCO0VBQ0EsTUFBQSxJQUFJLENBQUN5WCxXQUFXLENBQUNJLEdBQUcsQ0FBQ0ksR0FBRyxFQUFFLENBQUMsRUFBRTFILEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQTtRQUN6Q3NILEdBQUcsQ0FBQ0ksR0FBRyxFQUFFLENBQUE7UUFDVEosR0FBRyxDQUFDRyxHQUFHLEdBQUcsQ0FBQyxDQUFBO0VBQ2IsS0FBQTtFQUVBLElBQUEsSUFBSUYsZUFBZSxFQUFFO0VBQ25CO1FBQ0EsSUFBSTVFLE9BQU8sR0FBR2pHLFdBQVcsQ0FBQzZLLGVBQWUsQ0FBQyxDQUFDM0ssTUFBTSxDQUFDekwsSUFBSSxDQUNwRCxJQUFJLENBQUM2TyxLQUFLLENBQUNzSCxHQUFHLENBQUNJLEdBQUcsR0FBRyxDQUFDLENBQ3hCLENBQUMsQ0FBQTtFQUNELE1BQUEsSUFBSS9FLE9BQU8sRUFBRTtFQUNYO0VBQ0EsUUFBQSxJQUFJQSxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7RUFDZDs7RUFFQTtZQUNBLElBQUk0RSxlQUFlLElBQUksTUFBTSxFQUFFO0VBQzdCNUUsWUFBQUEsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHQSxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMvSSxPQUFPLENBQUMsU0FBUyxFQUFHdkMsTUFBTSxJQUFLO2dCQUNyRCxPQUFPc1EsUUFBUSxDQUFDdFEsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFBO0VBQ2hDLGFBQUMsQ0FBQyxDQUFBO0VBQ0osV0FBQTtZQUNBLElBQUksQ0FBQzJJLEtBQUssQ0FBQ3NILEdBQUcsQ0FBQ0ksR0FBRyxDQUFDLEdBQUksQ0FBQSxFQUFFL0UsT0FBTyxDQUFDLENBQUMsQ0FBRSxDQUFBLEVBQUUsSUFBSSxDQUFDM0MsS0FBSyxDQUFDc0gsR0FBRyxDQUFDSSxHQUFHLENBQUUsQ0FBQyxDQUFBLENBQUE7WUFDM0QsSUFBSSxDQUFDcEgsU0FBUyxDQUFDZ0gsR0FBRyxDQUFDSSxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUE7WUFDOUJKLEdBQUcsQ0FBQ0csR0FBRyxHQUFHOUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDbFQsTUFBTSxDQUFBO0VBQzdCLFNBQUMsTUFBTTtFQUNMO1lBQ0EsSUFBSSxDQUFDdVEsS0FBSyxDQUFDc0gsR0FBRyxDQUFDSSxHQUFHLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFBO1lBQzVCLElBQUksQ0FBQ3BILFNBQVMsQ0FBQ2dILEdBQUcsQ0FBQ0ksR0FBRyxHQUFHLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQTtFQUNwQyxTQUFBO0VBQ0YsT0FBQTtFQUNGLEtBQUE7TUFDQSxJQUFJLENBQUN0RixnQkFBZ0IsRUFBRSxDQUFBO0VBQ3pCLEdBQUE7O0VBRUE7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTs7RUFFQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDRXdGLEVBQUFBLFlBQVlBLEdBQW9CO0VBQUEsSUFBQSxJQUFuQkMsU0FBUyxHQUFBM1YsU0FBQSxDQUFBekMsTUFBQSxHQUFBLENBQUEsSUFBQXlDLFNBQUEsQ0FBQSxDQUFBLENBQUEsS0FBQTVCLFNBQUEsR0FBQTRCLFNBQUEsQ0FBQSxDQUFBLENBQUEsR0FBRyxLQUFLLENBQUE7RUFDNUIsSUFBQSxNQUFNd08sU0FBUyxHQUFHNVAsTUFBTSxDQUFDOFcsWUFBWSxFQUFFLENBQUE7TUFDdkMsSUFBSUUsU0FBUyxHQUFHRCxTQUFTLEdBQUduSCxTQUFTLENBQUNxSCxVQUFVLEdBQUdySCxTQUFTLENBQUNzSCxTQUFTLENBQUE7RUFDdEUsSUFBQSxJQUFJLENBQUNGLFNBQVMsRUFBRSxPQUFPLElBQUksQ0FBQTtNQUMzQixJQUFJdkMsTUFBTSxHQUFHc0MsU0FBUyxHQUFHbkgsU0FBUyxDQUFDdUgsWUFBWSxHQUFHdkgsU0FBUyxDQUFDd0gsV0FBVyxDQUFBO0VBQ3ZFLElBQUEsSUFBSUosU0FBUyxJQUFJLElBQUksQ0FBQ2phLENBQUMsRUFBRTtRQUN2QixJQUFJMFgsTUFBTSxHQUFHLElBQUksQ0FBQ3ZGLEtBQUssQ0FBQ3ZRLE1BQU0sRUFDNUIsT0FBTztFQUNMaVksUUFBQUEsR0FBRyxFQUFFbkMsTUFBTTtFQUNYa0MsUUFBQUEsR0FBRyxFQUFFLENBQUE7U0FDTixDQUFBO1FBQ0gsT0FBTztVQUNMQyxHQUFHLEVBQUVuQyxNQUFNLEdBQUcsQ0FBQztVQUNma0MsR0FBRyxFQUFFLElBQUksQ0FBQ3pILEtBQUssQ0FBQ3VGLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQzlWLE1BQUFBO1NBQzdCLENBQUE7RUFDSCxLQUFBO01BRUEsSUFBSWdZLEdBQUcsR0FBRyxJQUFJLENBQUNVLGFBQWEsQ0FBQ0wsU0FBUyxFQUFFdkMsTUFBTSxDQUFDLENBQUE7RUFDL0MsSUFBQSxJQUFJa0MsR0FBRyxLQUFLLElBQUksRUFBRSxPQUFPLElBQUksQ0FBQzs7RUFFOUI7TUFDQSxJQUFJVyxJQUFJLEdBQUdOLFNBQVMsQ0FBQTtFQUNwQixJQUFBLE9BQU9NLElBQUksQ0FBQ3paLGFBQWEsSUFBSSxJQUFJLENBQUNkLENBQUMsRUFBRTtRQUNuQ3VhLElBQUksR0FBR0EsSUFBSSxDQUFDelosYUFBYSxDQUFBO0VBQzNCLEtBQUE7TUFFQSxJQUFJK1ksR0FBRyxHQUFHLENBQUMsQ0FBQTtFQUNYO0VBQ0E7RUFDQSxJQUFBLElBQ0VVLElBQUksQ0FBQ2xGLE9BQU8sSUFDWmtGLElBQUksQ0FBQ2xGLE9BQU8sQ0FBQ2pCLE9BQU8sS0FDbkIsQ0FBQ21HLElBQUksQ0FBQ0MsZUFBZSxJQUNwQkQsSUFBSSxDQUFDQyxlQUFlLENBQUNuRixPQUFPLENBQUNqQixPQUFPLElBQUltRyxJQUFJLENBQUNsRixPQUFPLENBQUNqQixPQUFPLENBQUMsRUFDL0Q7UUFDQXlGLEdBQUcsR0FBR0MsUUFBUSxDQUFDUyxJQUFJLENBQUNsRixPQUFPLENBQUNqQixPQUFPLENBQUMsQ0FBQTtFQUN0QyxLQUFDLE1BQU07UUFDTCxPQUFPbUcsSUFBSSxDQUFDQyxlQUFlLEVBQUU7RUFDM0JYLFFBQUFBLEdBQUcsRUFBRSxDQUFBO1VBQ0xVLElBQUksR0FBR0EsSUFBSSxDQUFDQyxlQUFlLENBQUE7RUFDN0IsT0FBQTtFQUNGLEtBQUE7TUFDQSxPQUFPO0VBQUVYLE1BQUFBLEdBQUcsRUFBRUEsR0FBRztFQUFFRCxNQUFBQSxHQUFHLEVBQUVBLEdBQUc7RUFBRVcsTUFBQUEsSUFBSSxFQUFFTixTQUFBQTtPQUFXLENBQUE7RUFDaEQsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDRUssRUFBQUEsYUFBYUEsQ0FBQ0wsU0FBUyxFQUFFdkMsTUFBTSxFQUFFO01BQy9CLElBQUk2QyxJQUFJLEdBQUdOLFNBQVMsQ0FBQTtFQUNwQixJQUFBLElBQUlMLEdBQUcsQ0FBQTtFQUNQO01BQ0EsT0FBT1csSUFBSSxJQUFJQSxJQUFJLENBQUN2SCxVQUFVLElBQUksSUFBSSxDQUFDaFQsQ0FBQyxFQUFFO1FBQ3hDdWEsSUFBSSxHQUFHQSxJQUFJLENBQUN2SCxVQUFVLENBQUE7RUFDeEIsS0FBQTtFQUNBLElBQUEsSUFBSXVILElBQUksSUFBSSxJQUFJLEVBQUUsT0FBTyxJQUFJLENBQUE7O0VBRTdCO0VBQ0E7RUFDQTtFQUNBO01BQ0EsSUFBSU4sU0FBUyxDQUFDUSxRQUFRLEtBQUtDLElBQUksQ0FBQ0MsU0FBUyxJQUFJakQsTUFBTSxLQUFLLENBQUMsRUFBRTtFQUN6RDtFQUNBO0VBQ0E7RUFDQTtFQUNBa0MsTUFBQUEsR0FBRyxHQUFHbEMsTUFBTSxDQUFBO0VBQ1o2QyxNQUFBQSxJQUFJLEdBQUdOLFNBQVMsQ0FBQTtFQUNsQixLQUFDLE1BQU0sSUFBSXZDLE1BQU0sR0FBRyxDQUFDLEVBQUU7UUFDckI2QyxJQUFJLEdBQUdOLFNBQVMsQ0FBQ2hHLFVBQVUsQ0FBQ3lELE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQTtFQUN2Q2tDLE1BQUFBLEdBQUcsR0FBR1csSUFBSSxDQUFDdEIsV0FBVyxDQUFDclgsTUFBTSxDQUFBO0VBQy9CLEtBQUE7RUFDQSxJQUFBLE9BQU8yWSxJQUFJLENBQUN2SCxVQUFVLElBQUksSUFBSSxDQUFDaFQsQ0FBQyxFQUFFO1FBQ2hDLElBQUl1YSxJQUFJLENBQUNDLGVBQWUsRUFBRTtVQUN4QkQsSUFBSSxHQUFHQSxJQUFJLENBQUNDLGVBQWUsQ0FBQTtFQUMzQlosUUFBQUEsR0FBRyxJQUFJVyxJQUFJLENBQUN0QixXQUFXLENBQUNyWCxNQUFNLENBQUE7RUFDaEMsT0FBQyxNQUFNO1VBQ0wyWSxJQUFJLEdBQUdBLElBQUksQ0FBQ3ZILFVBQVUsQ0FBQTtFQUN4QixPQUFBO0VBQ0YsS0FBQTtFQUNBLElBQUEsT0FBTzRHLEdBQUcsQ0FBQTtFQUNaLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0VnQixFQUFBQSxvQkFBb0JBLENBQUNmLEdBQUcsRUFBRUQsR0FBRyxFQUFxQjtFQUFBLElBQUEsSUFBbkJpQixTQUFTLEdBQUF4VyxTQUFBLENBQUF6QyxNQUFBLEdBQUEsQ0FBQSxJQUFBeUMsU0FBQSxDQUFBLENBQUEsQ0FBQSxLQUFBNUIsU0FBQSxHQUFBNEIsU0FBQSxDQUFBLENBQUEsQ0FBQSxHQUFHLEtBQUssQ0FBQTtFQUM5QyxJQUFBLElBQUl3VixHQUFHLElBQUksSUFBSSxDQUFDekgsWUFBWSxDQUFDeFEsTUFBTSxFQUFFO0VBQ25DO0VBQ0FpWSxNQUFBQSxHQUFHLEdBQUcsSUFBSSxDQUFDekgsWUFBWSxDQUFDeFEsTUFBTSxHQUFHLENBQUMsQ0FBQTtRQUNsQ2dZLEdBQUcsR0FBRyxJQUFJLENBQUN6SCxLQUFLLENBQUMwSCxHQUFHLENBQUMsQ0FBQ2pZLE1BQU0sQ0FBQTtFQUM5QixLQUFBO01BQ0EsSUFBSWdZLEdBQUcsR0FBRyxJQUFJLENBQUN6SCxLQUFLLENBQUMwSCxHQUFHLENBQUMsQ0FBQ2pZLE1BQU0sRUFBRTtRQUNoQ2dZLEdBQUcsR0FBRyxJQUFJLENBQUN6SCxLQUFLLENBQUMwSCxHQUFHLENBQUMsQ0FBQ2pZLE1BQU0sQ0FBQTtFQUM5QixLQUFBO0VBQ0EsSUFBQSxNQUFNb1IsVUFBVSxHQUFHLElBQUksQ0FBQ1osWUFBWSxDQUFDeUgsR0FBRyxDQUFDLENBQUE7RUFDekMsSUFBQSxJQUFJVSxJQUFJLEdBQUd2SCxVQUFVLENBQUNrQixVQUFVLENBQUE7TUFFaEMsSUFBSTRHLGdCQUFnQixHQUFHLEtBQUssQ0FBQTtFQUM1QjtFQUNBLElBQUEsSUFBSUMsRUFBRSxHQUFHO1FBQ1BSLElBQUksRUFBRXZILFVBQVUsQ0FBQ2tCLFVBQVUsR0FBR2xCLFVBQVUsQ0FBQ2tCLFVBQVUsR0FBR2xCLFVBQVU7RUFDaEUwRSxNQUFBQSxNQUFNLEVBQUUsQ0FBQTtPQUNULENBQUE7TUFFRCxPQUFPNkMsSUFBSSxJQUFJdkgsVUFBVSxFQUFFO1FBQ3pCLElBQUksQ0FBQzhILGdCQUFnQixJQUFJUCxJQUFJLENBQUNFLFFBQVEsS0FBS0MsSUFBSSxDQUFDQyxTQUFTLEVBQUU7RUFDekQsUUFBQSxJQUFJSixJQUFJLENBQUNTLFNBQVMsQ0FBQ3BaLE1BQU0sSUFBSWdZLEdBQUcsRUFBRTtZQUNoQyxJQUFJaUIsU0FBUyxJQUFJTixJQUFJLENBQUNTLFNBQVMsQ0FBQ3BaLE1BQU0sSUFBSWdZLEdBQUcsRUFBRTtFQUM3QztFQUNBO0VBQ0FtQixZQUFBQSxFQUFFLEdBQUc7RUFBRVIsY0FBQUEsSUFBSSxFQUFFQSxJQUFJO0VBQUU3QyxjQUFBQSxNQUFNLEVBQUVrQyxHQUFBQTtlQUFLLENBQUE7RUFDaENBLFlBQUFBLEdBQUcsR0FBRyxDQUFDLENBQUE7RUFDVCxXQUFDLE1BQU07Y0FDTCxPQUFPO0VBQUVXLGNBQUFBLElBQUksRUFBRUEsSUFBSTtFQUFFN0MsY0FBQUEsTUFBTSxFQUFFa0MsR0FBQUE7ZUFBSyxDQUFBO0VBQ3BDLFdBQUE7RUFDRixTQUFDLE1BQU07RUFDTEEsVUFBQUEsR0FBRyxJQUFJVyxJQUFJLENBQUNTLFNBQVMsQ0FBQ3BaLE1BQU0sQ0FBQTtFQUM5QixTQUFBO0VBQ0YsT0FBQTtFQUNBLE1BQUEsSUFBSSxDQUFDa1osZ0JBQWdCLElBQUlQLElBQUksQ0FBQ3JHLFVBQVUsRUFBRTtVQUN4Q3FHLElBQUksR0FBR0EsSUFBSSxDQUFDckcsVUFBVSxDQUFBO0VBQ3hCLE9BQUMsTUFBTSxJQUFJcUcsSUFBSSxDQUFDNUcsV0FBVyxFQUFFO0VBQzNCbUgsUUFBQUEsZ0JBQWdCLEdBQUcsS0FBSyxDQUFBO1VBQ3hCUCxJQUFJLEdBQUdBLElBQUksQ0FBQzVHLFdBQVcsQ0FBQTtFQUN6QixPQUFDLE1BQU07RUFDTG1ILFFBQUFBLGdCQUFnQixHQUFHLElBQUksQ0FBQTtVQUN2QlAsSUFBSSxHQUFHQSxJQUFJLENBQUN2SCxVQUFVLENBQUE7RUFDeEIsT0FBQTtFQUNGLEtBQUE7O0VBRUE7RUFDQTtFQUNBLElBQUEsT0FBTytILEVBQUUsQ0FBQTtFQUNYLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7RUFDQTtJQUNFRSxZQUFZQSxDQUFDdmIsS0FBSyxFQUFpQjtFQUFBLElBQUEsSUFBZkMsTUFBTSxHQUFBMEUsU0FBQSxDQUFBekMsTUFBQSxHQUFBLENBQUEsSUFBQXlDLFNBQUEsQ0FBQSxDQUFBLENBQUEsS0FBQTVCLFNBQUEsR0FBQTRCLFNBQUEsQ0FBQSxDQUFBLENBQUEsR0FBRyxJQUFJLENBQUE7TUFDL0IsSUFBSSxDQUFDM0UsS0FBSyxFQUFFLE9BQUE7RUFFWixJQUFBLElBQUl3YixLQUFLLEdBQUczYSxRQUFRLENBQUM0YSxXQUFXLEVBQUUsQ0FBQTtNQUVsQyxJQUFJO0VBQUVaLE1BQUFBLElBQUksRUFBRUosU0FBUztFQUFFekMsTUFBQUEsTUFBTSxFQUFFMkMsV0FBQUE7RUFBWSxLQUFDLEdBQUcsSUFBSSxDQUFDTyxvQkFBb0IsQ0FDdEVsYixLQUFLLENBQUNtYSxHQUFHLEVBQ1RuYSxLQUFLLENBQUNrYSxHQUFHLEVBQ1RqYSxNQUFNLElBQUlBLE1BQU0sQ0FBQ2thLEdBQUcsSUFBSW5hLEtBQUssQ0FBQ21hLEdBQUcsSUFBSWxhLE1BQU0sQ0FBQ2lhLEdBQUcsR0FBR2xhLEtBQUssQ0FBQ2thLEdBQzFELENBQUMsQ0FBQztNQUNGLElBQUlNLFVBQVUsR0FBRyxJQUFJO0VBQ25CRSxNQUFBQSxZQUFZLEdBQUcsSUFBSSxDQUFBO0VBQ3JCLElBQUEsSUFBSXphLE1BQU0sS0FBS0EsTUFBTSxDQUFDa2EsR0FBRyxJQUFJbmEsS0FBSyxDQUFDbWEsR0FBRyxJQUFJbGEsTUFBTSxDQUFDaWEsR0FBRyxJQUFJbGEsS0FBSyxDQUFDa2EsR0FBRyxDQUFDLEVBQUU7UUFDbEUsSUFBSTtVQUFFVyxJQUFJO0VBQUU3QyxRQUFBQSxNQUFBQTtFQUFPLE9BQUMsR0FBRyxJQUFJLENBQUNrRCxvQkFBb0IsQ0FDOUNqYixNQUFNLENBQUNrYSxHQUFHLEVBQ1ZsYSxNQUFNLENBQUNpYSxHQUFHLEVBQ1ZsYSxLQUFLLENBQUNtYSxHQUFHLElBQUlsYSxNQUFNLENBQUNrYSxHQUFHLElBQUluYSxLQUFLLENBQUNrYSxHQUFHLEdBQUdqYSxNQUFNLENBQUNpYSxHQUNoRCxDQUFDLENBQUE7RUFDRE0sTUFBQUEsVUFBVSxHQUFHSyxJQUFJLENBQUE7RUFDakJILE1BQUFBLFlBQVksR0FBRzFDLE1BQU0sQ0FBQTtFQUN2QixLQUFBO0VBRUEsSUFBQSxJQUFJd0MsVUFBVSxFQUFFZ0IsS0FBSyxDQUFDRSxRQUFRLENBQUNsQixVQUFVLEVBQUVFLFlBQVksQ0FBQyxDQUFDLEtBQ3BEYyxLQUFLLENBQUNFLFFBQVEsQ0FBQ2pCLFNBQVMsRUFBRUUsV0FBVyxDQUFDLENBQUE7RUFDM0NhLElBQUFBLEtBQUssQ0FBQ0csTUFBTSxDQUFDbEIsU0FBUyxFQUFFRSxXQUFXLENBQUMsQ0FBQTtFQUVwQyxJQUFBLElBQUlpQixlQUFlLEdBQUdyWSxNQUFNLENBQUM4VyxZQUFZLEVBQUUsQ0FBQTtNQUMzQ3VCLGVBQWUsQ0FBQ0MsZUFBZSxFQUFFLENBQUE7RUFDakNELElBQUFBLGVBQWUsQ0FBQ0UsUUFBUSxDQUFDTixLQUFLLENBQUMsQ0FBQTtFQUNqQyxHQUFBOztFQUVBO0VBQ0Y7RUFDQTtJQUNFckgsZ0JBQWdCQSxDQUFDelIsS0FBSyxFQUFFO0VBQ3RCO0VBQ0E7RUFDQTtFQUNBLElBQUEsSUFBSUEsS0FBSyxDQUFDcVosU0FBUyxJQUFJLHVCQUF1QixFQUFFLE9BQUE7RUFFaEQsSUFBQSxJQUFJL2IsS0FBSyxHQUFHLElBQUksQ0FBQ3FhLFlBQVksRUFBRSxDQUFBO0VBRS9CLElBQUEsSUFDRSxDQUFDM1gsS0FBSyxDQUFDcVosU0FBUyxJQUFJLGlCQUFpQixJQUNuQ3JaLEtBQUssQ0FBQ3FaLFNBQVMsSUFBSSxpQkFBaUIsS0FDdEMvYixLQUFLLEVBQ0w7UUFDQSxJQUFJLENBQUN1VyxjQUFjLEVBQUUsQ0FBQTtFQUNyQixNQUFBLElBQUksQ0FBQ3VELG1CQUFtQixDQUFDOVosS0FBSyxDQUFDLENBQUE7RUFDakMsS0FBQyxNQUFNO0VBQ0wsTUFBQSxJQUFJLENBQUMsSUFBSSxDQUFDTSxDQUFDLENBQUNrVSxVQUFVLEVBQUU7RUFDdEIsUUFBQSxJQUFJLENBQUNsVSxDQUFDLENBQUNiLFNBQVMsR0FBRyxxQ0FBcUMsQ0FBQTtFQUMxRCxPQUFDLE1BQU07VUFDTCxJQUFJLENBQUN1YyxnQkFBZ0IsRUFBRSxDQUFBO0VBQ3pCLE9BQUE7UUFDQSxJQUFJLENBQUMxRiwrQkFBK0IsRUFBRSxDQUFBO0VBQ3hDLEtBQUE7RUFDQSxJQUFBLElBQUl0VyxLQUFLLEVBQUU7RUFDVCxNQUFBLElBQUksQ0FBQ3ViLFlBQVksQ0FBQ3ZiLEtBQUssQ0FBQyxDQUFBO0VBQzFCLEtBQUE7TUFFQSxJQUFJLENBQUM4VSxVQUFVLEVBQUUsQ0FBQTtFQUNuQixHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNFa0gsRUFBQUEsZ0JBQWdCQSxHQUFHO01BQ2pCLE1BQU1DLGdCQUFnQixHQUFHckgsS0FBSyxDQUFDc0gsSUFBSSxDQUFDLElBQUksQ0FBQzViLENBQUMsQ0FBQ2lVLFVBQVUsQ0FBQyxDQUFBO0VBRXRELElBQUEsTUFBTTRILFlBQVksR0FBRyxVQUFDQyxLQUFLLEVBQXFCO0VBQzlDLE1BQUEsTUFBTUMsTUFBTSxHQUFHRCxLQUFLLENBQUNoYixhQUFhLENBQUE7RUFDbEMsTUFBQSxNQUFNNlMsV0FBVyxHQUFHbUksS0FBSyxDQUFDbkksV0FBVyxDQUFBO0VBQ3JDb0ksTUFBQUEsTUFBTSxDQUFDNUgsV0FBVyxDQUFDMkgsS0FBSyxDQUFDLENBQUE7UUFBQyxLQUFBRSxJQUFBQSxJQUFBLEdBQUEzWCxTQUFBLENBQUF6QyxNQUFBLEVBSElxYSxXQUFXLE9BQUEzSCxLQUFBLENBQUEwSCxJQUFBLEdBQUFBLENBQUFBLEdBQUFBLElBQUEsV0FBQUUsSUFBQSxHQUFBLENBQUEsRUFBQUEsSUFBQSxHQUFBRixJQUFBLEVBQUFFLElBQUEsRUFBQSxFQUFBO0VBQVhELFFBQUFBLFdBQVcsQ0FBQUMsSUFBQSxHQUFBN1gsQ0FBQUEsQ0FBQUEsR0FBQUEsU0FBQSxDQUFBNlgsSUFBQSxDQUFBLENBQUE7RUFBQSxPQUFBO1FBSXpDRCxXQUFXLENBQUNFLE9BQU8sQ0FBRUMsUUFBUSxJQUMzQnpJLFdBQVcsR0FDUG9JLE1BQU0sQ0FBQ25JLFlBQVksQ0FBQ3dJLFFBQVEsRUFBRXpJLFdBQVcsQ0FBQyxHQUMxQ29JLE1BQU0sQ0FBQzVhLFdBQVcsQ0FBQ2liLFFBQVEsQ0FDakMsQ0FBQyxDQUFBO09BQ0YsQ0FBQTtFQUVEVCxJQUFBQSxnQkFBZ0IsQ0FBQ1EsT0FBTyxDQUFFTCxLQUFLLElBQUs7RUFDbEMsTUFBQSxJQUFJQSxLQUFLLENBQUNyQixRQUFRLEtBQUtDLElBQUksQ0FBQzJCLFlBQVksSUFBSVAsS0FBSyxDQUFDeGIsT0FBTyxLQUFLLEtBQUssRUFBRTtFQUNuRTtFQUNBLFFBQUEsTUFBTWdjLFVBQVUsR0FBRy9iLFFBQVEsQ0FBQ1EsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFBO0VBQ2hEOGEsUUFBQUEsWUFBWSxDQUFDQyxLQUFLLEVBQUVRLFVBQVUsQ0FBQyxDQUFBO0VBQy9CQSxRQUFBQSxVQUFVLENBQUNuYixXQUFXLENBQUMyYSxLQUFLLENBQUMsQ0FBQTtTQUM5QixNQUFNLElBQUlBLEtBQUssQ0FBQzdILFVBQVUsQ0FBQ3JTLE1BQU0sSUFBSSxDQUFDLEVBQUU7RUFDdkM7VUFDQWthLEtBQUssQ0FBQzNhLFdBQVcsQ0FBQ1osUUFBUSxDQUFDUSxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQTtFQUNqRCxPQUFDLE1BQU07VUFDTCxNQUFNd2IsYUFBYSxHQUFHakksS0FBSyxDQUFDc0gsSUFBSSxDQUFDRSxLQUFLLENBQUM3SCxVQUFVLENBQUMsQ0FBQTtVQUNsRCxJQUNFc0ksYUFBYSxDQUFDQyxJQUFJLENBQ2ZDLFVBQVUsSUFDVEEsVUFBVSxDQUFDaEMsUUFBUSxLQUFLQyxJQUFJLENBQUMyQixZQUFZLElBQ3pDSSxVQUFVLENBQUNuYyxPQUFPLEtBQUssS0FDM0IsQ0FBQyxFQUNEO0VBQ0EsVUFBQSxPQUFPdWIsWUFBWSxDQUFDQyxLQUFLLEVBQUVTLGFBQWEsQ0FBQyxDQUFBO0VBQzNDLFNBQUE7RUFDRixPQUFBO0VBQ0YsS0FBQyxDQUFDLENBQUE7RUFDSixHQUFBOztFQUVBO0VBQ0Y7RUFDQTtJQUNFdkksVUFBVUEsQ0FBQzVSLEtBQUssRUFBRTtNQUNoQkEsS0FBSyxDQUFDQyxjQUFjLEVBQUUsQ0FBQTtFQUV0QixJQUFBLElBQUksQ0FBQ3FhLFFBQVEsQ0FBQ3RhLEtBQUssQ0FBQ3VhLFlBQVksQ0FBQyxDQUFBO0VBQ25DLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0U3SSxFQUFBQSwwQkFBMEJBLEdBQUc7TUFDM0IsSUFBSSxDQUFDOEksYUFBYSxFQUFFLENBQUE7RUFDdEIsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0lBQ0V2RCxXQUFXQSxDQUNUd0QsU0FBUyxFQUlUO0VBQUEsSUFBQSxJQUhBMUQsYUFBYSxHQUFBOVUsU0FBQSxDQUFBekMsTUFBQSxHQUFBLENBQUEsSUFBQXlDLFNBQUEsQ0FBQSxDQUFBLENBQUEsS0FBQTVCLFNBQUEsR0FBQTRCLFNBQUEsQ0FBQSxDQUFBLENBQUEsR0FBRyxDQUFDLENBQUE7RUFBQSxJQUFBLElBQ2pCeVksYUFBYSxHQUFBelksU0FBQSxDQUFBekMsTUFBQSxHQUFBLENBQUEsSUFBQXlDLFNBQUEsQ0FBQSxDQUFBLENBQUEsS0FBQTVCLFNBQUEsR0FBQTRCLFNBQUEsQ0FBQSxDQUFBLENBQUEsR0FBRyxFQUFFLENBQUE7RUFBQSxJQUFBLElBQ2xCMFksa0JBQWtCLEdBQUExWSxTQUFBLENBQUF6QyxNQUFBLEdBQUEsQ0FBQSxJQUFBeUMsU0FBQSxDQUFBLENBQUEsQ0FBQSxLQUFBNUIsU0FBQSxHQUFBNEIsU0FBQSxDQUFBLENBQUEsQ0FBQSxHQUFHLElBQUksQ0FBQTtFQUV6QixJQUFBLElBQUkwWSxrQkFBa0IsRUFBRTtRQUN0QixLQUFLLElBQUlwYixDQUFDLEdBQUcsQ0FBQyxFQUFFQSxDQUFDLEdBQUd3WCxhQUFhLEVBQUV4WCxDQUFDLEVBQUUsRUFBRTtFQUN0QyxRQUFBLElBQUksQ0FBQzNCLENBQUMsQ0FBQ21VLFdBQVcsQ0FBQyxJQUFJLENBQUNuVSxDQUFDLENBQUNpVSxVQUFVLENBQUM0SSxTQUFTLENBQUMsQ0FBQyxDQUFBO0VBQ2xELE9BQUE7RUFDRixLQUFBO01BRUEsSUFBSUcsYUFBYSxHQUFHLEVBQUUsQ0FBQTtNQUN0QixJQUFJQyxhQUFhLEdBQUcsRUFBRSxDQUFBO0VBRXRCLElBQUEsS0FBSyxJQUFJdGIsQ0FBQyxHQUFHLENBQUMsRUFBRUEsQ0FBQyxHQUFHbWIsYUFBYSxDQUFDbGIsTUFBTSxFQUFFRCxDQUFDLEVBQUUsRUFBRTtFQUM3Q3FiLE1BQUFBLGFBQWEsQ0FBQ25iLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQTtFQUN0Qm9iLE1BQUFBLGFBQWEsQ0FBQ3BiLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQTtFQUN4QixNQUFBLElBQUlrYixrQkFBa0IsRUFBRTtVQUN0QixJQUFJLElBQUksQ0FBQy9jLENBQUMsQ0FBQ2lVLFVBQVUsQ0FBQzRJLFNBQVMsQ0FBQyxFQUM5QixJQUFJLENBQUM3YyxDQUFDLENBQUM0VCxZQUFZLENBQ2pCclQsUUFBUSxDQUFDUSxhQUFhLENBQUMsS0FBSyxDQUFDLEVBQzdCLElBQUksQ0FBQ2YsQ0FBQyxDQUFDaVUsVUFBVSxDQUFDNEksU0FBUyxDQUM3QixDQUFDLENBQUMsS0FDQyxJQUFJLENBQUM3YyxDQUFDLENBQUNtQixXQUFXLENBQUNaLFFBQVEsQ0FBQ1EsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUE7RUFDeEQsT0FBQTtFQUNGLEtBQUE7TUFFQSxJQUFJLENBQUNvUixLQUFLLENBQUMrSyxNQUFNLENBQUNMLFNBQVMsRUFBRTFELGFBQWEsRUFBRSxHQUFHMkQsYUFBYSxDQUFDLENBQUE7TUFDN0QsSUFBSSxDQUFDekssU0FBUyxDQUFDNkssTUFBTSxDQUFDTCxTQUFTLEVBQUUxRCxhQUFhLEVBQUUsR0FBRzZELGFBQWEsQ0FBQyxDQUFBO01BQ2pFLElBQUksQ0FBQ3ZLLFNBQVMsQ0FBQ3lLLE1BQU0sQ0FBQ0wsU0FBUyxFQUFFMUQsYUFBYSxFQUFFLEdBQUc4RCxhQUFhLENBQUMsQ0FBQTtFQUNuRSxHQUFBOztFQUVBO0VBQ0Y7RUFDQTtJQUNFbEosV0FBV0EsQ0FBQzNSLEtBQUssRUFBRTtNQUNqQkEsS0FBSyxDQUFDQyxjQUFjLEVBQUUsQ0FBQTs7RUFFdEI7RUFDQSxJQUFBLElBQUk4YSxJQUFJLEdBQUcsQ0FBQy9hLEtBQUssQ0FBQ2diLGFBQWEsSUFBSWhiLEtBQUssRUFBRWliLGFBQWEsQ0FBQ0MsT0FBTyxDQUM3RCxZQUNGLENBQUMsQ0FBQTs7RUFFRDtFQUNBLElBQUEsSUFBSSxDQUFDMWQsS0FBSyxDQUFDdWQsSUFBSSxDQUFDLENBQUE7RUFDbEIsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtJQUNFdmQsS0FBS0EsQ0FBQ3VkLElBQUksRUFBK0I7RUFBQSxJQUFBLElBQTdCeGQsTUFBTSxHQUFBMEUsU0FBQSxDQUFBekMsTUFBQSxHQUFBLENBQUEsSUFBQXlDLFNBQUEsQ0FBQSxDQUFBLENBQUEsS0FBQTVCLFNBQUEsR0FBQTRCLFNBQUEsQ0FBQSxDQUFBLENBQUEsR0FBRyxJQUFJLENBQUE7RUFBQSxJQUFBLElBQUUzRSxLQUFLLEdBQUEyRSxTQUFBLENBQUF6QyxNQUFBLEdBQUEsQ0FBQSxJQUFBeUMsU0FBQSxDQUFBLENBQUEsQ0FBQSxLQUFBNUIsU0FBQSxHQUFBNEIsU0FBQSxDQUFBLENBQUEsQ0FBQSxHQUFHLElBQUksQ0FBQTtNQUNyQyxJQUFJLENBQUMxRSxNQUFNLEVBQUVBLE1BQU0sR0FBRyxJQUFJLENBQUNvYSxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUE7TUFDN0MsSUFBSSxDQUFDcmEsS0FBSyxFQUFFQSxLQUFLLEdBQUcsSUFBSSxDQUFDcWEsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFBO01BQzVDLElBQUl3RCxTQUFTLEVBQUVqTixHQUFHLENBQUE7TUFDbEIsSUFBSSxDQUFDNVEsS0FBSyxFQUFFO0VBQ1ZBLE1BQUFBLEtBQUssR0FBRztFQUNObWEsUUFBQUEsR0FBRyxFQUFFLElBQUksQ0FBQzFILEtBQUssQ0FBQ3ZRLE1BQU0sR0FBRyxDQUFDO0VBQzFCZ1ksUUFBQUEsR0FBRyxFQUFFLElBQUksQ0FBQ3pILEtBQUssQ0FBQyxJQUFJLENBQUNBLEtBQUssQ0FBQ3ZRLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQ0EsTUFBQUE7RUFDekMsT0FBQyxDQUFDO0VBQ0osS0FBQTs7TUFDQSxJQUFJLENBQUNqQyxNQUFNLEVBQUU7RUFDWEEsTUFBQUEsTUFBTSxHQUFHRCxLQUFLLENBQUE7RUFDaEIsS0FBQTtNQUNBLElBQ0VDLE1BQU0sQ0FBQ2thLEdBQUcsR0FBR25hLEtBQUssQ0FBQ21hLEdBQUcsSUFDckJsYSxNQUFNLENBQUNrYSxHQUFHLElBQUluYSxLQUFLLENBQUNtYSxHQUFHLElBQUlsYSxNQUFNLENBQUNpYSxHQUFHLElBQUlsYSxLQUFLLENBQUNrYSxHQUFJLEVBQ3BEO0VBQ0EyRCxNQUFBQSxTQUFTLEdBQUc1ZCxNQUFNLENBQUE7RUFDbEIyUSxNQUFBQSxHQUFHLEdBQUc1USxLQUFLLENBQUE7RUFDYixLQUFDLE1BQU07RUFDTDZkLE1BQUFBLFNBQVMsR0FBRzdkLEtBQUssQ0FBQTtFQUNqQjRRLE1BQUFBLEdBQUcsR0FBRzNRLE1BQU0sQ0FBQTtFQUNkLEtBQUE7RUFDQSxJQUFBLElBQUk2ZCxhQUFhLEdBQUdMLElBQUksQ0FBQzNiLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFBO0VBQ2hELElBQUEsSUFBSWljLFVBQVUsR0FBRyxJQUFJLENBQUN0TCxLQUFLLENBQUNvTCxTQUFTLENBQUMxRCxHQUFHLENBQUMsQ0FBQ3JELE1BQU0sQ0FBQyxDQUFDLEVBQUUrRyxTQUFTLENBQUMzRCxHQUFHLENBQUMsQ0FBQTtFQUNuRSxJQUFBLElBQUk4RCxPQUFPLEdBQUcsSUFBSSxDQUFDdkwsS0FBSyxDQUFDN0IsR0FBRyxDQUFDdUosR0FBRyxDQUFDLENBQUNyRCxNQUFNLENBQUNsRyxHQUFHLENBQUNzSixHQUFHLENBQUMsQ0FBQTtFQUNqRDRELElBQUFBLGFBQWEsQ0FBQyxDQUFDLENBQUMsR0FBR0MsVUFBVSxDQUFDeGIsTUFBTSxDQUFDdWIsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7TUFDdEQsSUFBSUcsU0FBUyxHQUFHSCxhQUFhLENBQUNBLGFBQWEsQ0FBQzViLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQ0EsTUFBTSxDQUFBO01BQzlENGIsYUFBYSxDQUFDQSxhQUFhLENBQUM1YixNQUFNLEdBQUcsQ0FBQyxDQUFDLEdBQ3JDNGIsYUFBYSxDQUFDQSxhQUFhLENBQUM1YixNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUNLLE1BQU0sQ0FBQ3liLE9BQU8sQ0FBQyxDQUFBO0VBQ3pELElBQUEsSUFBSSxDQUFDckUsV0FBVyxDQUFDa0UsU0FBUyxDQUFDMUQsR0FBRyxFQUFFLENBQUMsR0FBR3ZKLEdBQUcsQ0FBQ3VKLEdBQUcsR0FBRzBELFNBQVMsQ0FBQzFELEdBQUcsRUFBRTJELGFBQWEsQ0FBQyxDQUFBO01BQzNFOWQsS0FBSyxDQUFDbWEsR0FBRyxHQUFHMEQsU0FBUyxDQUFDMUQsR0FBRyxHQUFHMkQsYUFBYSxDQUFDNWIsTUFBTSxHQUFHLENBQUMsQ0FBQTtNQUNwRGxDLEtBQUssQ0FBQ2thLEdBQUcsR0FBRytELFNBQVMsQ0FBQTtNQUNyQixJQUFJLENBQUNwSixnQkFBZ0IsRUFBRSxDQUFBO0VBQ3ZCLElBQUEsSUFBSSxDQUFDMEcsWUFBWSxDQUFDdmIsS0FBSyxDQUFDLENBQUE7TUFDeEIsSUFBSSxDQUFDOFUsVUFBVSxFQUFFLENBQUE7RUFDbkIsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDRW9KLEVBQUFBLHFCQUFxQkEsQ0FBQ0MsS0FBSyxFQUFFQyxLQUFLLEVBQUU7RUFDbEMsSUFBQSxJQUFJLENBQUNELEtBQUssSUFBSSxDQUFDQyxLQUFLLEVBQUUsT0FBTyxJQUFJLENBQUE7RUFDakMsSUFBQSxJQUFJRCxLQUFLLElBQUlDLEtBQUssRUFBRSxPQUFPRCxLQUFLLENBQUE7TUFDaEMsTUFBTUUsUUFBUSxHQUFJeEQsSUFBSSxJQUFLO1FBQ3pCLElBQUl3RCxRQUFRLEdBQUcsRUFBRSxDQUFBO0VBQ2pCLE1BQUEsT0FBT3hELElBQUksRUFBRTtFQUNYd0QsUUFBQUEsUUFBUSxDQUFDQyxPQUFPLENBQUN6RCxJQUFJLENBQUMsQ0FBQTtVQUN0QkEsSUFBSSxHQUFHQSxJQUFJLENBQUN2SCxVQUFVLENBQUE7RUFDeEIsT0FBQTtFQUNBLE1BQUEsT0FBTytLLFFBQVEsQ0FBQTtPQUNoQixDQUFBO0VBRUQsSUFBQSxNQUFNRSxTQUFTLEdBQUdGLFFBQVEsQ0FBQ0YsS0FBSyxDQUFDLENBQUE7RUFDakMsSUFBQSxNQUFNSyxTQUFTLEdBQUdILFFBQVEsQ0FBQ0QsS0FBSyxDQUFDLENBQUE7TUFFakMsSUFBSUcsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsT0FBTyxJQUFJLENBQUE7RUFDN0MsSUFBQSxJQUFJdmMsQ0FBQyxDQUFBO0VBQ0wsSUFBQSxLQUFLQSxDQUFDLEdBQUcsQ0FBQyxFQUFFc2MsU0FBUyxDQUFDdGMsQ0FBQyxDQUFDLElBQUl1YyxTQUFTLENBQUN2YyxDQUFDLENBQUMsRUFBRUEsQ0FBQyxFQUFFLENBQUMsQ0FBQTtFQUM5QyxJQUFBLE9BQU9zYyxTQUFTLENBQUN0YyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUE7RUFDekIsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNFd2MsRUFBQUEsMEJBQTBCQSxDQUFDemUsS0FBSyxFQUFFQyxNQUFNLEVBQUVxQixTQUFTLEVBQUU7TUFDbkQsSUFBSXVaLElBQUksR0FBRyxJQUFJLENBQUE7RUFDZixJQUFBLElBQUksQ0FBQzdhLEtBQUssRUFBRSxPQUFPLElBQUksQ0FBQTtNQUN2QixJQUFJLENBQUNDLE1BQU0sRUFBRTtRQUNYNGEsSUFBSSxHQUFHN2EsS0FBSyxDQUFDNmEsSUFBSSxDQUFBO0VBQ25CLEtBQUMsTUFBTTtRQUNMLElBQUk3YSxLQUFLLENBQUNtYSxHQUFHLElBQUlsYSxNQUFNLENBQUNrYSxHQUFHLEVBQUUsT0FBTyxJQUFJLENBQUE7RUFDeENVLE1BQUFBLElBQUksR0FBRyxJQUFJLENBQUNxRCxxQkFBcUIsQ0FBQ2xlLEtBQUssQ0FBQzZhLElBQUksRUFBRTVhLE1BQU0sQ0FBQzRhLElBQUksQ0FBQyxDQUFBO0VBQzVELEtBQUE7RUFDQSxJQUFBLElBQUksQ0FBQ0EsSUFBSSxFQUFFLE9BQU8sSUFBSSxDQUFBO0VBQ3RCLElBQUEsT0FBT0EsSUFBSSxJQUFJLElBQUksQ0FBQ3ZhLENBQUMsRUFBRTtFQUNyQixNQUFBLElBQUl1YSxJQUFJLENBQUN2WixTQUFTLElBQUl1WixJQUFJLENBQUN2WixTQUFTLENBQUNvZCxRQUFRLENBQUNwZCxTQUFTLENBQUMsRUFBRSxPQUFPdVosSUFBSSxDQUFBO1FBQ3JFQSxJQUFJLEdBQUdBLElBQUksQ0FBQ3ZILFVBQVUsQ0FBQTtFQUN4QixLQUFBO0VBQ0E7RUFDQSxJQUFBLE9BQU8sSUFBSSxDQUFBO0VBQ2IsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0VxTCxFQUFBQSxlQUFlQSxHQUE4QjtFQUFBLElBQUEsSUFBN0IzZSxLQUFLLEdBQUEyRSxTQUFBLENBQUF6QyxNQUFBLEdBQUEsQ0FBQSxJQUFBeUMsU0FBQSxDQUFBLENBQUEsQ0FBQSxLQUFBNUIsU0FBQSxHQUFBNEIsU0FBQSxDQUFBLENBQUEsQ0FBQSxHQUFHLElBQUksQ0FBQTtFQUFBLElBQUEsSUFBRTFFLE1BQU0sR0FBQTBFLFNBQUEsQ0FBQXpDLE1BQUEsR0FBQSxDQUFBLElBQUF5QyxTQUFBLENBQUEsQ0FBQSxDQUFBLEtBQUE1QixTQUFBLEdBQUE0QixTQUFBLENBQUEsQ0FBQSxDQUFBLEdBQUcsSUFBSSxDQUFBO01BQ3pDLElBQUk3QixZQUFZLEdBQUcsRUFBRSxDQUFBO01BQ3JCLElBQUksQ0FBQzlDLEtBQUssRUFBRUEsS0FBSyxHQUFHLElBQUksQ0FBQ3FhLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQTtNQUM1QyxJQUFJLENBQUNwYSxNQUFNLEVBQUVBLE1BQU0sR0FBRyxJQUFJLENBQUNvYSxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUE7TUFDN0MsSUFBSSxDQUFDcmEsS0FBSyxFQUFFO0VBQ1YsTUFBQSxLQUFLLElBQUk0ZSxHQUFHLElBQUlyZSxRQUFRLEVBQUU7RUFDeEJ1QyxRQUFBQSxZQUFZLENBQUM4YixHQUFHLENBQUMsR0FBRyxJQUFJLENBQUE7RUFDMUIsT0FBQTtFQUNBLE1BQUEsT0FBTzliLFlBQVksQ0FBQTtFQUNyQixLQUFBO0VBQ0EsSUFBQSxJQUFJLENBQUM3QyxNQUFNLEVBQUVBLE1BQU0sR0FBR0QsS0FBSyxDQUFBO01BRTNCLElBQUkyUSxLQUFLLEVBQUVDLEdBQUcsQ0FBQTtNQUNkLElBQ0UzUSxNQUFNLENBQUNrYSxHQUFHLEdBQUduYSxLQUFLLENBQUNtYSxHQUFHLElBQ3JCbGEsTUFBTSxDQUFDa2EsR0FBRyxJQUFJbmEsS0FBSyxDQUFDbWEsR0FBRyxJQUFJbGEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHbGEsS0FBSyxDQUFDa2EsR0FBSSxFQUNuRDtFQUNBdkosTUFBQUEsS0FBSyxHQUFHMVEsTUFBTSxDQUFBO0VBQ2QyUSxNQUFBQSxHQUFHLEdBQUc1USxLQUFLLENBQUE7RUFDYixLQUFDLE1BQU07RUFDTDJRLE1BQUFBLEtBQUssR0FBRzNRLEtBQUssQ0FBQTtFQUNiNFEsTUFBQUEsR0FBRyxHQUFHM1EsTUFBTSxDQUFBO0VBQ2QsS0FBQTtFQUNBLElBQUEsSUFBSTJRLEdBQUcsQ0FBQ3VKLEdBQUcsR0FBR3hKLEtBQUssQ0FBQ3dKLEdBQUcsSUFBSXZKLEdBQUcsQ0FBQ3NKLEdBQUcsSUFBSSxDQUFDLEVBQUU7UUFDdkN0SixHQUFHLENBQUN1SixHQUFHLEVBQUUsQ0FBQTtFQUNUdkosTUFBQUEsR0FBRyxDQUFDc0osR0FBRyxHQUFHLElBQUksQ0FBQ3pILEtBQUssQ0FBQzdCLEdBQUcsQ0FBQ3VKLEdBQUcsQ0FBQyxDQUFDalksTUFBTSxDQUFDO0VBQ3ZDLEtBQUE7O0VBRUEsSUFBQSxLQUFLLElBQUkwYyxHQUFHLElBQUlyZSxRQUFRLEVBQUU7UUFDeEIsSUFBSUEsUUFBUSxDQUFDcWUsR0FBRyxDQUFDLENBQUNqVCxJQUFJLElBQUksUUFBUSxFQUFFO1VBQ2xDLElBQ0UsQ0FBQzNMLEtBQUssSUFDTkEsS0FBSyxDQUFDbWEsR0FBRyxJQUFJbGEsTUFBTSxDQUFDa2EsR0FBRyxJQUN2QixDQUFDLElBQUksQ0FBQ3RhLHlCQUF5QixDQUFDRyxLQUFLLEVBQUVDLE1BQU0sQ0FBQyxFQUM5QztFQUNBNkMsVUFBQUEsWUFBWSxDQUFDOGIsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFBO0VBQzFCLFNBQUMsTUFBTTtFQUNMO1lBQ0E5YixZQUFZLENBQUM4YixHQUFHLENBQUMsR0FDZixDQUFDLENBQUMsSUFBSSxDQUFDSCwwQkFBMEIsQ0FDL0J6ZSxLQUFLLEVBQ0xDLE1BQU0sRUFDTk0sUUFBUSxDQUFDcWUsR0FBRyxDQUFDLENBQUN0ZCxTQUNoQixDQUFDO0VBQ0Q7RUFDQ3RCLFVBQUFBLEtBQUssQ0FBQ2thLEdBQUcsSUFBSWphLE1BQU0sQ0FBQ2lhLEdBQUcsSUFDdEIsQ0FBQyxDQUFDLElBQUksQ0FBQ3pILEtBQUssQ0FBQ3pTLEtBQUssQ0FBQ21hLEdBQUcsQ0FBQyxDQUNwQnJELE1BQU0sQ0FBQyxDQUFDLEVBQUU5VyxLQUFLLENBQUNrYSxHQUFHLENBQUMsQ0FDcEI5WCxLQUFLLENBQUM3QixRQUFRLENBQUNxZSxHQUFHLENBQUMsQ0FBQzdNLEtBQUssQ0FBQ0MsVUFBVSxDQUFDLElBQ3hDLENBQUMsQ0FBQyxJQUFJLENBQUNTLEtBQUssQ0FBQ3pTLEtBQUssQ0FBQ21hLEdBQUcsQ0FBQyxDQUNwQnJELE1BQU0sQ0FBQzlXLEtBQUssQ0FBQ2thLEdBQUcsQ0FBQyxDQUNqQjlYLEtBQUssQ0FBQzdCLFFBQVEsQ0FBQ3FlLEdBQUcsQ0FBQyxDQUFDN00sS0FBSyxDQUFDRSxXQUFXLENBQUUsQ0FBQTtFQUNoRCxTQUFBO0VBQ0YsT0FBQTtRQUNBLElBQUkxUixRQUFRLENBQUNxZSxHQUFHLENBQUMsQ0FBQ2pULElBQUksSUFBSSxNQUFNLEVBQUU7VUFDaEMsSUFBSSxDQUFDM0wsS0FBSyxFQUFFO0VBQ1Y4QyxVQUFBQSxZQUFZLENBQUM4YixHQUFHLENBQUMsR0FBRyxJQUFJLENBQUE7RUFDMUIsU0FBQyxNQUFNO0VBQ0wsVUFBQSxJQUFJbmUsS0FBSyxHQUFHLElBQUksQ0FBQ2tTLFNBQVMsQ0FBQ2hDLEtBQUssQ0FBQ3dKLEdBQUcsQ0FBQyxJQUFJNVosUUFBUSxDQUFDcWUsR0FBRyxDQUFDLENBQUN0ZCxTQUFTLENBQUE7RUFFaEUsVUFBQSxLQUFLLElBQUlzWSxJQUFJLEdBQUdqSixLQUFLLENBQUN3SixHQUFHLEVBQUVQLElBQUksSUFBSWhKLEdBQUcsQ0FBQ3VKLEdBQUcsRUFBRVAsSUFBSSxFQUFFLEVBQUU7RUFDbEQsWUFBQSxJQUFLLElBQUksQ0FBQ2pILFNBQVMsQ0FBQ2lILElBQUksQ0FBQyxJQUFJclosUUFBUSxDQUFDcWUsR0FBRyxDQUFDLENBQUN0ZCxTQUFTLElBQUtiLEtBQUssRUFBRTtFQUM5REEsY0FBQUEsS0FBSyxHQUFHLElBQUksQ0FBQTtFQUNaLGNBQUEsTUFBQTtFQUNGLGFBQUE7RUFDRixXQUFBO0VBQ0FxQyxVQUFBQSxZQUFZLENBQUM4YixHQUFHLENBQUMsR0FBR25lLEtBQUssQ0FBQTtFQUMzQixTQUFBO0VBQ0YsT0FBQTtFQUNGLEtBQUE7RUFDQSxJQUFBLE9BQU9xQyxZQUFZLENBQUE7RUFDckIsR0FBQTs7RUFFQTtFQUNGO0VBQ0E7RUFDQTtFQUNBO0VBQ0VGLEVBQUFBLGVBQWVBLENBQUNyQixPQUFPLEVBQUVkLEtBQUssRUFBRTtNQUM5QixJQUFJRixRQUFRLENBQUNnQixPQUFPLENBQUMsQ0FBQ29LLElBQUksSUFBSSxRQUFRLEVBQUU7RUFDdEMsTUFBQSxJQUFJMUwsTUFBTSxHQUFHLElBQUksQ0FBQ29hLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQTtFQUNwQyxNQUFBLElBQUlyYSxLQUFLLEdBQUcsSUFBSSxDQUFDcWEsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFBO0VBQ3BDLE1BQUEsSUFBSSxDQUFDcGEsTUFBTSxFQUFFQSxNQUFNLEdBQUdELEtBQUssQ0FBQTtRQUMzQixJQUFJLENBQUNDLE1BQU0sRUFBRSxPQUFBO0VBQ2IsTUFBQSxJQUFJQSxNQUFNLENBQUNrYSxHQUFHLElBQUluYSxLQUFLLENBQUNtYSxHQUFHLEVBQUUsT0FBQTtRQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDdGEseUJBQXlCLENBQUNHLEtBQUssRUFBRUMsTUFBTSxDQUFDLEVBQUUsT0FBQTtFQUNwRCxNQUFBLElBQUk0ZSxVQUFVLEdBQUcsSUFBSSxDQUFDSiwwQkFBMEIsQ0FDOUN6ZSxLQUFLLEVBQ0xDLE1BQU0sRUFDTk0sUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUNELFNBQ3BCLENBQUMsQ0FBQTtRQUNELElBQUksQ0FBQ2lWLGNBQWMsRUFBRSxDQUFBOztFQUVyQjtFQUNBLE1BQUEsSUFBSXNJLFVBQVUsRUFBRTtVQUNkLElBQUksQ0FBQzlMLFNBQVMsQ0FBQy9TLEtBQUssQ0FBQ21hLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQTtVQUNoQyxNQUFNMkUsUUFBUSxHQUFHLElBQUksQ0FBQ2xFLGFBQWEsQ0FBQ2lFLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQTtFQUNsRCxRQUFBLE1BQU1FLEdBQUcsR0FBR0YsVUFBVSxDQUFDdEYsV0FBVyxDQUFDclgsTUFBTSxDQUFBO0VBQ3pDLFFBQUEsTUFBTThjLElBQUksR0FBRyxJQUFJLENBQUN2TSxLQUFLLENBQUN6UyxLQUFLLENBQUNtYSxHQUFHLENBQUMsQ0FDL0JyRCxNQUFNLENBQUMsQ0FBQyxFQUFFZ0ksUUFBUSxDQUFDLENBQ25CelMsT0FBTyxDQUFDOUwsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN3USxLQUFLLENBQUNDLFVBQVUsRUFBRSxFQUFFLENBQUMsQ0FBQTtFQUNsRCxRQUFBLE1BQU1pTixHQUFHLEdBQUcsSUFBSSxDQUFDeE0sS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQUNyRCxNQUFNLENBQUNnSSxRQUFRLEVBQUVDLEdBQUcsQ0FBQyxDQUFBO0VBQ3ZELFFBQUEsTUFBTUcsS0FBSyxHQUFHLElBQUksQ0FBQ3pNLEtBQUssQ0FBQ3pTLEtBQUssQ0FBQ21hLEdBQUcsQ0FBQyxDQUNoQ3JELE1BQU0sQ0FBQ2dJLFFBQVEsR0FBR0MsR0FBRyxDQUFDLENBQ3RCMVMsT0FBTyxDQUFDOUwsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN3USxLQUFLLENBQUNFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQTtFQUNuRCxRQUFBLElBQUksQ0FBQ1EsS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLEdBQUc2RSxJQUFJLENBQUN6YyxNQUFNLENBQUMwYyxHQUFHLEVBQUVDLEtBQUssQ0FBQyxDQUFBO0VBQy9DamYsUUFBQUEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHOEUsSUFBSSxDQUFDOWMsTUFBTSxDQUFBO0VBQ3hCbEMsUUFBQUEsS0FBSyxDQUFDa2EsR0FBRyxHQUFHamEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHNkUsR0FBRyxDQUFBO1VBQzVCLElBQUksQ0FBQ2xLLGdCQUFnQixFQUFFLENBQUE7RUFDdkIsUUFBQSxJQUFJLENBQUMwRyxZQUFZLENBQUN2YixLQUFLLEVBQUVDLE1BQU0sQ0FBQyxDQUFBO1VBQ2hDLElBQUksQ0FBQzZVLFVBQVUsRUFBRSxDQUFBOztFQUVqQjtFQUNGLE9BQUMsTUFBTSxJQUNMOVUsS0FBSyxDQUFDa2EsR0FBRyxJQUFJamEsTUFBTSxDQUFDaWEsR0FBRyxJQUN2QixDQUFDLENBQUMsSUFBSSxDQUFDekgsS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQ3BCckQsTUFBTSxDQUFDLENBQUMsRUFBRTlXLEtBQUssQ0FBQ2thLEdBQUcsQ0FBQyxDQUNwQjlYLEtBQUssQ0FBQzdCLFFBQVEsQ0FBQ2dCLE9BQU8sQ0FBQyxDQUFDd1EsS0FBSyxDQUFDQyxVQUFVLENBQUMsSUFDNUMsQ0FBQyxDQUFDLElBQUksQ0FBQ1MsS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQ3BCckQsTUFBTSxDQUFDOVcsS0FBSyxDQUFDa2EsR0FBRyxDQUFDLENBQ2pCOVgsS0FBSyxDQUFDN0IsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN3USxLQUFLLENBQUNFLFdBQVcsQ0FBQyxFQUM3QztVQUNBLElBQUksQ0FBQ2MsU0FBUyxDQUFDL1MsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFBO0VBQ2hDLFFBQUEsTUFBTTZFLElBQUksR0FBRyxJQUFJLENBQUN2TSxLQUFLLENBQUN6UyxLQUFLLENBQUNtYSxHQUFHLENBQUMsQ0FDL0JyRCxNQUFNLENBQUMsQ0FBQyxFQUFFOVcsS0FBSyxDQUFDa2EsR0FBRyxDQUFDLENBQ3BCN04sT0FBTyxDQUFDOUwsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN3USxLQUFLLENBQUNDLFVBQVUsRUFBRSxFQUFFLENBQUMsQ0FBQTtFQUNsRCxRQUFBLE1BQU1rTixLQUFLLEdBQUcsSUFBSSxDQUFDek0sS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQ2hDckQsTUFBTSxDQUFDOVcsS0FBSyxDQUFDa2EsR0FBRyxDQUFDLENBQ2pCN04sT0FBTyxDQUFDOUwsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN3USxLQUFLLENBQUNFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQTtFQUNuRCxRQUFBLElBQUksQ0FBQ1EsS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLEdBQUc2RSxJQUFJLENBQUN6YyxNQUFNLENBQUMyYyxLQUFLLENBQUMsQ0FBQTtVQUMxQ2xmLEtBQUssQ0FBQ2thLEdBQUcsR0FBR2phLE1BQU0sQ0FBQ2lhLEdBQUcsR0FBRzhFLElBQUksQ0FBQzljLE1BQU0sQ0FBQTtVQUNwQyxJQUFJLENBQUMyUyxnQkFBZ0IsRUFBRSxDQUFBO0VBQ3ZCLFFBQUEsSUFBSSxDQUFDMEcsWUFBWSxDQUFDdmIsS0FBSyxFQUFFQyxNQUFNLENBQUMsQ0FBQTtVQUNoQyxJQUFJLENBQUM2VSxVQUFVLEVBQUUsQ0FBQTs7RUFFakI7RUFDRixPQUFDLE1BQU07RUFDTDtVQUNBLElBQUk7WUFBRWdLLFFBQVE7RUFBRUssVUFBQUEsTUFBQUE7V0FBUSxHQUN0Qm5mLEtBQUssQ0FBQ2thLEdBQUcsR0FBR2phLE1BQU0sQ0FBQ2lhLEdBQUcsR0FDbEI7WUFBRTRFLFFBQVEsRUFBRTllLEtBQUssQ0FBQ2thLEdBQUc7WUFBRWlGLE1BQU0sRUFBRWxmLE1BQU0sQ0FBQ2lhLEdBQUFBO0VBQUksU0FBQyxHQUMzQztZQUFFNEUsUUFBUSxFQUFFN2UsTUFBTSxDQUFDaWEsR0FBRztZQUFFaUYsTUFBTSxFQUFFbmYsS0FBSyxDQUFDa2EsR0FBQUE7V0FBSyxDQUFBO1VBRWpELElBQUk5WCxLQUFLLEdBQUcsSUFBSSxDQUFDcVEsS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQzlCckQsTUFBTSxDQUFDZ0ksUUFBUSxFQUFFSyxNQUFNLEdBQUdMLFFBQVEsQ0FBQyxDQUNuQzFjLEtBQUssQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFBO0VBQ2pELFFBQUEsSUFBSUEsS0FBSyxFQUFFO0VBQ1QwYyxVQUFBQSxRQUFRLElBQUkxYyxLQUFLLENBQUM4VCxNQUFNLENBQUNrSixPQUFPLENBQUNsZCxNQUFNLENBQUE7RUFDdkNpZCxVQUFBQSxNQUFNLElBQUkvYyxLQUFLLENBQUM4VCxNQUFNLENBQUNtSixRQUFRLENBQUNuZCxNQUFNLENBQUE7RUFDeEMsU0FBQTtVQUVBbEMsS0FBSyxDQUFDa2EsR0FBRyxHQUFHNEUsUUFBUSxDQUFBO1VBQ3BCN2UsTUFBTSxDQUFDaWEsR0FBRyxHQUFHaUYsTUFBTSxDQUFBOztFQUVuQjtVQUNBLElBQUksQ0FBQ3JmLGFBQWEsQ0FDaEJTLFFBQVEsQ0FBQ2dCLE9BQU8sQ0FBQyxDQUFDK0osR0FBRyxDQUFDdUcsR0FBRyxFQUN6QnRSLFFBQVEsQ0FBQ2dCLE9BQU8sQ0FBQyxDQUFDK0osR0FBRyxDQUFDd0csSUFBSSxFQUMxQjlSLEtBQUssRUFDTEMsTUFDRixDQUFDLENBQUE7VUFDRCxJQUFJLENBQUM2VSxVQUFVLEVBQUUsQ0FBQTtFQUNqQjtFQUNGLE9BQUE7T0FDRCxNQUFNLElBQUl2VSxRQUFRLENBQUNnQixPQUFPLENBQUMsQ0FBQ29LLElBQUksSUFBSSxNQUFNLEVBQUU7RUFDM0MsTUFBQSxJQUFJMUwsTUFBTSxHQUFHLElBQUksQ0FBQ29hLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQTtFQUNwQyxNQUFBLElBQUlyYSxLQUFLLEdBQUcsSUFBSSxDQUFDcWEsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFBO0VBQ3BDLE1BQUEsSUFBSSxDQUFDcGEsTUFBTSxFQUFFQSxNQUFNLEdBQUdELEtBQUssQ0FBQTtRQUMzQixJQUFJLENBQUNBLEtBQUssRUFBRSxPQUFBO1FBQ1osSUFBSSxDQUFDdVcsY0FBYyxFQUFFLENBQUE7RUFDckIsTUFBQSxJQUFJNUYsS0FBSyxHQUFHMVEsTUFBTSxDQUFDa2EsR0FBRyxHQUFHbmEsS0FBSyxDQUFDbWEsR0FBRyxHQUFHbmEsS0FBSyxHQUFHQyxNQUFNLENBQUE7RUFDbkQsTUFBQSxJQUFJMlEsR0FBRyxHQUFHM1EsTUFBTSxDQUFDa2EsR0FBRyxHQUFHbmEsS0FBSyxDQUFDbWEsR0FBRyxHQUFHbGEsTUFBTSxHQUFHRCxLQUFLLENBQUE7RUFDakQsTUFBQSxJQUFJNFEsR0FBRyxDQUFDdUosR0FBRyxHQUFHeEosS0FBSyxDQUFDd0osR0FBRyxJQUFJdkosR0FBRyxDQUFDc0osR0FBRyxJQUFJLENBQUMsRUFBRTtVQUN2Q3RKLEdBQUcsQ0FBQ3VKLEdBQUcsRUFBRSxDQUFBO0VBQ1gsT0FBQTtFQUVBLE1BQUEsS0FBSyxJQUFJUCxJQUFJLEdBQUdqSixLQUFLLENBQUN3SixHQUFHLEVBQUVQLElBQUksSUFBSWhKLEdBQUcsQ0FBQ3VKLEdBQUcsRUFBRVAsSUFBSSxFQUFFLEVBQUU7RUFDbEQsUUFBQSxJQUFJblosS0FBSyxJQUFJLElBQUksQ0FBQ2tTLFNBQVMsQ0FBQ2lILElBQUksQ0FBQyxJQUFJclosUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUNELFNBQVMsRUFBRTtZQUNoRSxJQUFJLENBQUNtUixLQUFLLENBQUNtSCxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUNuSCxLQUFLLENBQUNtSCxJQUFJLENBQUMsQ0FBQ3ZOLE9BQU8sQ0FDekM5TCxRQUFRLENBQUNnQixPQUFPLENBQUMsQ0FBQytKLEdBQUcsQ0FBQzRHLE9BQU8sRUFDN0IzUixRQUFRLENBQUNnQixPQUFPLENBQUMsQ0FBQytKLEdBQUcsQ0FBQ2dFLFdBQVcsQ0FBQ2pELE9BQU8sQ0FDdkMsSUFBSSxFQUNKdU4sSUFBSSxHQUFHakosS0FBSyxDQUFDd0osR0FBRyxHQUFHLENBQ3JCLENBQ0YsQ0FBQyxDQUFBO0VBQ0QsVUFBQSxJQUFJLENBQUNwSCxTQUFTLENBQUM2RyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUE7RUFDN0IsU0FBQTtFQUNBLFFBQUEsSUFBSSxDQUFDblosS0FBSyxJQUFJLElBQUksQ0FBQ2tTLFNBQVMsQ0FBQ2lILElBQUksQ0FBQyxJQUFJclosUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUNELFNBQVMsRUFBRTtFQUNqRSxVQUFBLElBQUksQ0FBQ21SLEtBQUssQ0FBQ21ILElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQ25ILEtBQUssQ0FBQ21ILElBQUksQ0FBQyxDQUFDdk4sT0FBTyxDQUN6QzlMLFFBQVEsQ0FBQ2dCLE9BQU8sQ0FBQyxDQUFDd1EsS0FBSyxDQUFDRyxPQUFPLEVBQy9CM1IsUUFBUSxDQUFDZ0IsT0FBTyxDQUFDLENBQUN3USxLQUFLLENBQUN6QyxXQUMxQixDQUFDLENBQUE7RUFDRCxVQUFBLElBQUksQ0FBQ3lELFNBQVMsQ0FBQzZHLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQTtFQUM3QixTQUFBO0VBQ0YsT0FBQTtRQUNBLElBQUksQ0FBQy9FLGdCQUFnQixFQUFFLENBQUE7UUFDdkIsSUFBSSxDQUFDMEcsWUFBWSxDQUNmO1VBQUVwQixHQUFHLEVBQUV2SixHQUFHLENBQUN1SixHQUFHO1VBQUVELEdBQUcsRUFBRSxJQUFJLENBQUN6SCxLQUFLLENBQUM3QixHQUFHLENBQUN1SixHQUFHLENBQUMsQ0FBQ2pZLE1BQUFBO0VBQU8sT0FBQyxFQUNqRDtVQUFFaVksR0FBRyxFQUFFeEosS0FBSyxDQUFDd0osR0FBRztFQUFFRCxRQUFBQSxHQUFHLEVBQUUsQ0FBQTtFQUFFLE9BQzNCLENBQUMsQ0FBQTtRQUNELElBQUksQ0FBQ3BGLFVBQVUsRUFBRSxDQUFBO0VBQ25CLEtBQUE7RUFDRixHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNBO0VBQ0VqVixFQUFBQSx5QkFBeUJBLEdBQUc7RUFDMUI7RUFDQSxJQUFBLE1BQU1rYSxHQUFHLEdBQUd4VyxNQUFNLENBQUM4VyxZQUFZLEVBQUUsQ0FBQTtFQUNqQyxJQUFBLElBQUksQ0FBQ04sR0FBRyxJQUFJLENBQUNBLEdBQUcsQ0FBQ1UsU0FBUyxJQUFJLENBQUNWLEdBQUcsQ0FBQ1MsVUFBVSxFQUFFLE9BQU8sS0FBSyxDQUFBOztFQUUzRDs7RUFFQTtNQUNBLElBQ0VULEdBQUcsQ0FBQ3VGLFdBQVcsSUFDZnZGLEdBQUcsQ0FBQ1UsU0FBUyxDQUFDTSxRQUFRLElBQUksQ0FBQyxJQUMzQmhCLEdBQUcsQ0FBQ1ksV0FBVyxJQUFJWixHQUFHLENBQUNVLFNBQVMsQ0FBQ2EsU0FBUyxDQUFDcFosTUFBTSxFQUNqRDtFQUNBLE1BQUEsSUFBSTJZLElBQUksQ0FBQTtFQUNSLE1BQUEsS0FDRUEsSUFBSSxHQUFHZCxHQUFHLENBQUNVLFNBQVMsRUFDcEJJLElBQUksSUFBSUEsSUFBSSxDQUFDNUcsV0FBVyxJQUFJLElBQUksRUFDaEM0RyxJQUFJLEdBQUdBLElBQUksQ0FBQ3ZILFVBQVUsQ0FDdkIsQ0FBQTtRQUNELElBQ0V1SCxJQUFJLElBQ0pBLElBQUksQ0FBQzVHLFdBQVcsQ0FBQzNTLFNBQVMsSUFDMUJ1WixJQUFJLENBQUM1RyxXQUFXLENBQUMzUyxTQUFTLENBQUNvZCxRQUFRLENBQUMsbUJBQW1CLENBQUMsRUFFeEQsT0FBTyxJQUFJLENBQUE7RUFDZixLQUFBOztFQUVBO0VBQ0EsSUFBQSxJQUFJYSxRQUFRLEdBQUcsSUFBSSxDQUFDckIscUJBQXFCLENBQUNuRSxHQUFHLENBQUNVLFNBQVMsRUFBRVYsR0FBRyxDQUFDUyxVQUFVLENBQUMsQ0FBQTtFQUN4RSxJQUFBLElBQUksQ0FBQytFLFFBQVEsRUFBRSxPQUFPLEtBQUssQ0FBQTs7RUFFM0I7RUFDQSxJQUFBLE9BQU9BLFFBQVEsSUFBSUEsUUFBUSxJQUFJLElBQUksQ0FBQ2pmLENBQUMsRUFBRTtFQUNyQyxNQUFBLElBQ0VpZixRQUFRLENBQUNqZSxTQUFTLElBQ2xCLE9BQU9pZSxRQUFRLENBQUNqZSxTQUFTLENBQUNvZCxRQUFRLElBQUksVUFBVSxLQUMvQ2EsUUFBUSxDQUFDamUsU0FBUyxDQUFDb2QsUUFBUSxDQUFDLG1CQUFtQixDQUFDLElBQy9DYSxRQUFRLENBQUNqZSxTQUFTLENBQUNvZCxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUMsRUFFN0MsT0FBTyxJQUFJLENBQUE7UUFDYmEsUUFBUSxHQUFHQSxRQUFRLENBQUNqTSxVQUFVLENBQUE7RUFDaEMsS0FBQTtFQUVBLElBQUEsT0FBTyxLQUFLLENBQUE7RUFDZCxHQUFBOztFQUVBO0VBQ0Y7RUFDQTtFQUNBO0VBQ0E7RUFDQTtFQUNBO0VBQ0V4VCxFQUFBQSxhQUFhQSxDQUFDK1IsR0FBRyxFQUFFQyxJQUFJLEVBQStCO0VBQUEsSUFBQSxJQUE3QjlSLEtBQUssR0FBQTJFLFNBQUEsQ0FBQXpDLE1BQUEsR0FBQSxDQUFBLElBQUF5QyxTQUFBLENBQUEsQ0FBQSxDQUFBLEtBQUE1QixTQUFBLEdBQUE0QixTQUFBLENBQUEsQ0FBQSxDQUFBLEdBQUcsSUFBSSxDQUFBO0VBQUEsSUFBQSxJQUFFMUUsTUFBTSxHQUFBMEUsU0FBQSxDQUFBekMsTUFBQSxHQUFBLENBQUEsSUFBQXlDLFNBQUEsQ0FBQSxDQUFBLENBQUEsS0FBQTVCLFNBQUEsR0FBQTRCLFNBQUEsQ0FBQSxDQUFBLENBQUEsR0FBRyxJQUFJLENBQUE7TUFDbEQsSUFBSSxDQUFDM0UsS0FBSyxFQUFFQSxLQUFLLEdBQUcsSUFBSSxDQUFDcWEsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFBO01BQzVDLElBQUksQ0FBQ3BhLE1BQU0sRUFBRUEsTUFBTSxHQUFHLElBQUksQ0FBQ29hLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQTtFQUM3QyxJQUFBLElBQUksQ0FBQ3JhLEtBQUssSUFBSSxDQUFDQyxNQUFNLElBQUlELEtBQUssQ0FBQ21hLEdBQUcsSUFBSWxhLE1BQU0sQ0FBQ2thLEdBQUcsRUFBRSxPQUFBO01BQ2xELElBQUksQ0FBQ3BILFNBQVMsQ0FBQy9TLEtBQUssQ0FBQ21hLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQTtFQUVoQyxJQUFBLE1BQU0yRSxRQUFRLEdBQUc5ZSxLQUFLLENBQUNrYSxHQUFHLEdBQUdqYSxNQUFNLENBQUNpYSxHQUFHLEdBQUdsYSxLQUFLLENBQUNrYSxHQUFHLEdBQUdqYSxNQUFNLENBQUNpYSxHQUFHLENBQUE7RUFDaEUsSUFBQSxNQUFNaUYsTUFBTSxHQUFHbmYsS0FBSyxDQUFDa2EsR0FBRyxHQUFHamEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHamEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHbGEsS0FBSyxDQUFDa2EsR0FBRyxDQUFBO01BQzlELE1BQU04RSxJQUFJLEdBQUcsSUFBSSxDQUFDdk0sS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQUNyRCxNQUFNLENBQUMsQ0FBQyxFQUFFZ0ksUUFBUSxDQUFDLENBQUN2YyxNQUFNLENBQUNzUCxHQUFHLENBQUMsQ0FBQTtNQUNsRSxNQUFNb04sR0FBRyxHQUNQRSxNQUFNLElBQUlMLFFBQVEsR0FDZCxFQUFFLEdBQ0YsSUFBSSxDQUFDck0sS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQUNyRCxNQUFNLENBQUNnSSxRQUFRLEVBQUVLLE1BQU0sR0FBR0wsUUFBUSxDQUFDLENBQUE7RUFDL0QsSUFBQSxNQUFNSSxLQUFLLEdBQUdwTixJQUFJLENBQUN2UCxNQUFNLENBQUMsSUFBSSxDQUFDa1EsS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLENBQUNyRCxNQUFNLENBQUNxSSxNQUFNLENBQUMsQ0FBQyxDQUFBO0VBQy9ELElBQUEsSUFBSSxDQUFDMU0sS0FBSyxDQUFDelMsS0FBSyxDQUFDbWEsR0FBRyxDQUFDLEdBQUc2RSxJQUFJLENBQUN6YyxNQUFNLENBQUMwYyxHQUFHLEVBQUVDLEtBQUssQ0FBQyxDQUFBO0VBQy9DamYsSUFBQUEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHOEUsSUFBSSxDQUFDOWMsTUFBTSxDQUFBO01BQ3hCbEMsS0FBSyxDQUFDa2EsR0FBRyxHQUFHamEsTUFBTSxDQUFDaWEsR0FBRyxHQUFHK0UsR0FBRyxDQUFDL2MsTUFBTSxDQUFBO01BRW5DLElBQUksQ0FBQzJTLGdCQUFnQixFQUFFLENBQUE7RUFDdkIsSUFBQSxJQUFJLENBQUMwRyxZQUFZLENBQUN2YixLQUFLLEVBQUVDLE1BQU0sQ0FBQyxDQUFBO0VBQ2xDLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7SUFDRXVmLGtCQUFrQkEsQ0FBQ2plLE9BQU8sRUFBRTtFQUMxQixJQUFBLElBQUksQ0FBQyxJQUFJLENBQUN5UixnQkFBZ0IsRUFBRSxJQUFJLENBQUNBLGdCQUFnQixHQUFHLElBQUksQ0FBQzJMLGVBQWUsRUFBRSxDQUFBO0VBQzFFLElBQUEsSUFBSSxDQUFDL2IsZUFBZSxDQUFDckIsT0FBTyxFQUFFLENBQUMsSUFBSSxDQUFDeVIsZ0JBQWdCLENBQUN6UixPQUFPLENBQUMsQ0FBQyxDQUFBO0VBQ2hFLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0V1VCxFQUFBQSxVQUFVQSxHQUFHO0VBQ1gsSUFBQSxJQUFJLENBQUMsSUFBSSxDQUFDdEMsUUFBUSxJQUFJLENBQUMsSUFBSSxDQUFDUyxTQUFTLENBQUNDLE1BQU0sQ0FBQ2hSLE1BQU0sRUFBRSxPQUFBO0VBQ3JELElBQUEsTUFBTXlSLE9BQU8sR0FBRyxJQUFJLENBQUNvQixVQUFVLEVBQUUsQ0FBQTtNQUNqQyxJQUFJLElBQUksQ0FBQ3ZDLFFBQVEsRUFBRSxJQUFJLENBQUNBLFFBQVEsQ0FBQ3ZNLEtBQUssR0FBRzBOLE9BQU8sQ0FBQTtNQUNoRCxLQUFLLElBQUk4TCxRQUFRLElBQUksSUFBSSxDQUFDeE0sU0FBUyxDQUFDQyxNQUFNLEVBQUU7RUFDMUN1TSxNQUFBQSxRQUFRLENBQUM7RUFDUDlMLFFBQUFBLE9BQU8sRUFBRUEsT0FBTztVQUNoQitMLFVBQVUsRUFBRSxJQUFJLENBQUNBLFVBQUFBO0VBQ25CLE9BQUMsQ0FBQyxDQUFBO0VBQ0osS0FBQTtFQUNGLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0V4QyxFQUFBQSxhQUFhQSxHQUFHO0VBQ2QsSUFBQSxJQUFJLElBQUksQ0FBQ2pLLFNBQVMsQ0FBQ0UsU0FBUyxJQUFJLElBQUksQ0FBQ0YsU0FBUyxDQUFDRSxTQUFTLENBQUNqUixNQUFNLEVBQUU7RUFDL0QsTUFBQSxJQUFJbEMsS0FBSyxHQUFHLElBQUksQ0FBQ3FhLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQTtFQUNwQyxNQUFBLElBQUlwYSxNQUFNLEdBQUcsSUFBSSxDQUFDb2EsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFBO1FBQ3BDLElBQUl2WCxZQUFZLEdBQUcsSUFBSSxDQUFDNmIsZUFBZSxDQUFDM2UsS0FBSyxFQUFFQyxNQUFNLENBQUMsQ0FBQTtRQUN0RCxJQUFJLElBQUksQ0FBQytTLGdCQUFnQixFQUFFO1VBQ3pCclIsTUFBTSxDQUFDQyxNQUFNLENBQUMsSUFBSSxDQUFDb1IsZ0JBQWdCLEVBQUVsUSxZQUFZLENBQUMsQ0FBQTtFQUNwRCxPQUFDLE1BQU07VUFDTCxJQUFJLENBQUNrUSxnQkFBZ0IsR0FBR3JSLE1BQU0sQ0FBQ0MsTUFBTSxDQUFDLEVBQUUsRUFBRWtCLFlBQVksQ0FBQyxDQUFBO0VBQ3pELE9BQUE7UUFDQSxLQUFLLElBQUkyYyxRQUFRLElBQUksSUFBSSxDQUFDeE0sU0FBUyxDQUFDRSxTQUFTLEVBQUU7RUFDN0NzTSxRQUFBQSxRQUFRLENBQUM7RUFDUHpmLFVBQUFBLEtBQUssRUFBRUEsS0FBSztFQUNaQyxVQUFBQSxNQUFNLEVBQUVBLE1BQU07WUFDZDZDLFlBQVksRUFBRSxJQUFJLENBQUNrUSxnQkFBQUE7RUFDckIsU0FBQyxDQUFDLENBQUE7RUFDSixPQUFBO0VBQ0YsS0FBQTtFQUNGLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0lBQ0VnSyxRQUFRQSxDQUFDQyxZQUFZLEVBQUU7TUFDckIsS0FBSyxJQUFJd0MsUUFBUSxJQUFJLElBQUksQ0FBQ3hNLFNBQVMsQ0FBQ0csSUFBSSxFQUFFO0VBQ3hDcU0sTUFBQUEsUUFBUSxDQUFDO0VBQ1B4QyxRQUFBQSxZQUFBQTtFQUNGLE9BQUMsQ0FBQyxDQUFBO0VBQ0osS0FBQTtFQUNGLEdBQUE7O0VBRUE7RUFDRjtFQUNBO0VBQ0E7RUFDQTtFQUNFaGMsRUFBQUEsZ0JBQWdCQSxDQUFDMEssSUFBSSxFQUFFOFQsUUFBUSxFQUFFO0VBQy9CLElBQUEsSUFBSTlULElBQUksQ0FBQ3ZKLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFO1FBQ3JDLElBQUksQ0FBQzZRLFNBQVMsQ0FBQ0MsTUFBTSxDQUFDL1EsSUFBSSxDQUFDc2QsUUFBUSxDQUFDLENBQUE7RUFDdEMsS0FBQTtFQUNBLElBQUEsSUFBSTlULElBQUksQ0FBQ3ZKLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxFQUFFO1FBQ2xELElBQUksQ0FBQzZRLFNBQVMsQ0FBQ0UsU0FBUyxDQUFDaFIsSUFBSSxDQUFDc2QsUUFBUSxDQUFDLENBQUE7RUFDekMsS0FBQTtFQUNBLElBQUEsSUFBSTlULElBQUksQ0FBQ3ZKLEtBQUssQ0FBQyxhQUFhLENBQUMsRUFBRTtRQUM3QixJQUFJLENBQUM2USxTQUFTLENBQUNHLElBQUksQ0FBQ2pSLElBQUksQ0FBQ3NkLFFBQVEsQ0FBQyxDQUFBO0VBQ3BDLEtBQUE7RUFDRixHQUFBO0VBQ0Y7Ozs7Ozs7OzsiLCJ4X2dvb2dsZV9pZ25vcmVMaXN0IjpbMiwzLDQsNSw2LDcsOCw5LDEwLDExLDEyLDEzLDE0LDE1LDE2LDE3LDE4LDE5LDIwLDIxLDIyLDIzLDI0LDI1LDI2LDI3LDI4LDI5LDMwLDMxLDMyLDMzLDM0LDM1LDM2LDM3LDM4LDM5LDQwLDQxLDQyLDQzLDQ0LDQ1LDQ2LDQ3LDQ4XX0={"version":3,"file":"index.js","sources":["src/svg/svg.js","src/TinyMDECommandBar.js","node_modules/core-js/internals/global.js","node_modules/core-js/internals/fails.js","node_modules/core-js/internals/descriptors.js","node_modules/core-js/internals/function-bind-native.js","node_modules/core-js/internals/function-uncurry-this.js","node_modules/core-js/internals/document-all.js","node_modules/core-js/internals/is-callable.js","node_modules/core-js/internals/is-null-or-undefined.js","node_modules/core-js/internals/require-object-coercible.js","node_modules/core-js/internals/to-object.js","node_modules/core-js/internals/has-own-property.js","node_modules/core-js/internals/function-name.js","node_modules/core-js/internals/define-global-property.js","node_modules/core-js/internals/shared-store.js","node_modules/core-js/internals/inspect-source.js","node_modules/core-js/internals/weak-map-basic-detection.js","node_modules/core-js/internals/is-object.js","node_modules/core-js/internals/document-create-element.js","node_modules/core-js/internals/ie8-dom-define.js","node_modules/core-js/internals/v8-prototype-define-bug.js","node_modules/core-js/internals/an-object.js","node_modules/core-js/internals/function-call.js","node_modules/core-js/internals/get-built-in.js","node_modules/core-js/internals/object-is-prototype-of.js","node_modules/core-js/internals/engine-user-agent.js","node_modules/core-js/internals/engine-v8-version.js","node_modules/core-js/internals/symbol-constructor-detection.js","node_modules/core-js/internals/use-symbol-as-uid.js","node_modules/core-js/internals/is-symbol.js","node_modules/core-js/internals/try-to-string.js","node_modules/core-js/internals/a-callable.js","node_modules/core-js/internals/get-method.js","node_modules/core-js/internals/ordinary-to-primitive.js","node_modules/core-js/internals/shared.js","node_modules/core-js/internals/uid.js","node_modules/core-js/internals/well-known-symbol.js","node_modules/core-js/internals/to-primitive.js","node_modules/core-js/internals/to-property-key.js","node_modules/core-js/internals/object-define-property.js","node_modules/core-js/internals/create-property-descriptor.js","node_modules/core-js/internals/create-non-enumerable-property.js","node_modules/core-js/internals/shared-key.js","node_modules/core-js/internals/internal-state.js","node_modules/core-js/internals/make-built-in.js","node_modules/core-js/internals/define-built-in-accessor.js","node_modules/core-js/internals/regexp-flags.js","node_modules/core-js/modules/es.regexp.flags.js","src/grammar.js","src/TinyMDE.js"],"sourcesContent":["const svg = {\n  blockquote: `<svg height=\"18\" width=\"18\"><rect width=\"5\" height=\"5\" x=\"3\" y=\"4\" ry=\"1\"/><rect ry=\"1\" y=\"4\" x=\"10\" height=\"5\" width=\"5\"/><path d=\"M8 6.999v3c0 1-1 3-1 3s-.331 1-1.331 1h-1c-1 0-.669-1-.669-1s1-2 1-3v-3zm7 0v3c0 1-1 3-1 3s-.331 1-1.331 1h-1c-1 0-.669-1-.669-1s1-2 1-3v-3z\"/></svg>`,\n  bold: `<svg height=\"18\" width=\"18\"><path d=\"M4 2a1 1 0 00-1 1v12a1 1 0 001 1h6c4 0 5-2 5-4 0-1.322-.434-2.636-1.885-3.381C13.772 7.885 14 6.945 14 6c0-2-1-4-5-4zm1 2h4c1.668 0 2.32.393 2.6.672.28.279.4.662.4 1.328s-.12 1.057-.4 1.338c-.275.274-1.014.646-2.6.662H5zm5 6c1.666.005 2.318.388 2.596.666.277.278.404.667.404 1.334s-.127 1.06-.404 1.338c-.278.278-.93.66-2.596.662l-4.992.004L5 10z\"/></svg>`,\n  clear_formatting: `<svg height=\"18\" width=\"18\"><path d=\"M11.03 1a1 1 0 00-.74.3l-9 9a1 1 0 000 1.4l4 4A1 1 0 006 16h2a1 1 0 00.7-.3l8-8a1 1 0 000-1.4l-5-5a1 1 0 00-.67-.3zM8.7 5.7l3.58 3.6L7.6 14H6.4l-3-3 5.3-5.3z\"/><rect ry=\".8\" rx=\".8\" y=\"14\" x=\"16\" height=\"2\" width=\"2\"/><rect width=\"2\" height=\"2\" x=\"13\" y=\"14\" rx=\".8\" ry=\".8\"/><rect ry=\".8\" rx=\".8\" y=\"14\" x=\"10\" height=\"2\" width=\"2\"/></svg>`,\n  code: `<svg height=\"18\" width=\"18\"><path d=\"M13.5 2.994a.5.5 0 00-.5.5.5.5 0 000 .034V4.53a5.993 5.993 0 00-7.451-.445A6 6 0 1012.45 13.9a5.99 5.99 0 001.346-1.334.5.5 0 00.096-.101.5.5 0 00-.12-.698.5.5 0 00-.697.12l-.004-.005a5 5 0 01-1.197 1.2 5 5 0 111.215-6.965.5.5 0 00.697.12.5.5 0 00.211-.44V4.745H14V3.493a.5.5 0 00-.5-.5z\"/></svg>`,\n  h1: `<svg height=\"18\" width=\"18\"><path d=\"M3 2s0-1 1-1h1c1 0 1 1 1 1v6h6V2s0-1 1-1h1c1 0 1 1 1 1v14s0 1-1 1h-1c-1 0-1-1-1-1v-6H6v6s0 1-1 1H4c-1 0-1-1-1-1z\"/></svg>`,\n  h2: `<svg height=\"18\" width=\"18\"><path d=\"M4 2s0-1 1-1 1 1 1 1v6c1-1 2-1 4-1 3 0 4 2 4 4v5s0 1-1 1-1-1-1-1v-5c0-1.094-1-2-2-2-2 0-3 0-4 2v5s0 1-1 1-1-1-1-1z\"/></svg>`,\n  hr: `<svg height=\"18\" width=\"18\"><rect ry=\"1\" y=\"8\" height=\"2\" width=\"18\" style=\"font-variation-settings:normal;marker:none\"/></svg>`,\n  image: `<svg height=\"18\" width=\"18\"><path d=\"M1 2v14h16V2H1zm2 2h12v7l-3-3-4 4-2-2-3 3V4z\"/><circle r=\"1.5\" cy=\"6.5\" cx=\"5.5\"/></svg>`,\n  italic: `<svg height=\"18\" width=\"18\"><path d=\"M9 2a1 1 0 000 2L7 14a1 1 0 100 2h2a1 1 0 000-2l2-10a1 1 0 100-2z\"/></svg>`,\n  link: `<svg height=\"18\" width=\"18\"><path d=\"M9.07 5.18a3.9 3.9 0 00-1.52.43C6.31 6.22 5.3 7.29 4.3 8.29c-1 1-2.07 2.02-2.68 3.26-.31.62-.5 1.33-.41 2.07.09.75.48 1.47 1.1 2.09.61.61 1.33 1 2.08 1.1.74.09 1.45-.1 2.07-.42 1.24-.61 2.26-1.68 3.26-2.68.46-.47.94-.94 1.39-1.44l-.43.26c-.68.34-1.49.56-2.36.46-.2-.03-.4-.08-.6-.14-.79.76-1.55 1.45-2.16 1.76-.38.19-.67.24-.92.21-.26-.03-.54-.14-.92-.53-.39-.38-.5-.66-.53-.91-.03-.26.02-.55.21-.93.39-.76 1.32-1.74 2.32-2.74 1-1 1.98-1.93 2.74-2.32.38-.19.67-.24.93-.21.25.03.53.14.91.53.39.38.5.66.53.92v.06l1.12-1.06.44-.47c-.18-.3-.4-.6-.67-.87-.62-.61-1.34-1-2.09-1.1a3.08 3.08 0 00-.55-.01z\"/><path d=\"M13.07.86a3.9 3.9 0 00-1.52.43c-1.24.62-2.26 1.69-3.26 2.69-.46.47-.94.94-1.39 1.43.15-.08.28-.18.43-.25a4.4 4.4 0 012.36-.46c.2.02.4.07.6.14.79-.77 1.55-1.46 2.16-1.76.38-.19.67-.25.93-.21.25.03.53.14.91.52.39.38.5.66.53.92.03.26-.02.55-.21.93-.39.76-1.32 1.74-2.32 2.74-1 1-1.98 1.93-2.74 2.31-.38.2-.67.25-.93.22-.25-.04-.53-.15-.91-.53-.39-.38-.5-.66-.53-.92V9c-.36.33-.73.67-1.12 1.06l-.43.46c.17.3.4.6.66.87.62.62 1.34 1 2.08 1.1.75.1 1.46-.1 2.08-.41 1.24-.62 2.26-1.69 3.26-2.69s2.07-2.02 2.68-3.26c.31-.62.5-1.32.41-2.07a3.63 3.63 0 00-1.1-2.08c-.61-.62-1.33-1-2.07-1.1a3.08 3.08 0 00-.56-.02z\"/></svg>`,\n  ol: `<svg height=\"18\" width=\"18\"><path d=\"M1.5 7a.5.5 0 100 1h1a.5.5 0 01.5.5c0 .164-.08.31-.14.355l-1.655 1.25A.492.492 0 001 10.5a.5.5 0 00.5.5h2a.5.5 0 000-1H3l.398-.299A1.5 1.5 0 002.5 7z\"/><path d=\"M1.5 12c-.667 0-.667 1 0 1h1.25c.333 0 .333.5 0 .5H2.5c-.667 0-.667 1 0 1h.25c.333 0 .333.5 0 .5H1.5c-.667 0-.667 1 0 1h1c.398 0 .78-.131 1.06-.365.282-.235.44-.554.44-.885a1.121 1.121 0 00-.303-.75c.191-.204.3-.47.303-.75 0-.332-.158-.651-.44-.885-.3-.241-.675-.37-1.06-.365z\"/><rect y=\"13\" x=\"6\" height=\"2\" width=\"12\" ry=\"1\"/><rect ry=\"1\" width=\"12\" height=\"2\" x=\"6\" y=\"8\"/><rect y=\"3\" x=\"6\" height=\"2\" width=\"12\" ry=\"1\"/><path d=\"M1.5 2a.5.5 0 100 1H2v2h-.5a.5.5 0 100 1h2a.5.5 0 100-1H3V2.5c0-.277-.223-.5-.5-.5z\"/></svg>`,\n  strikethrough: `<svg width=\"18\" height=\"18\"><path d=\"M10 2C6 2 4 4 4 6c0 .338.08.672.193 1h2.34C6.113 6.629 6 6.295 6 6c0-.334.117-.725.691-1.154C7.265 4.416 8.331 4 10 4h3a1 1 0 000-2zm1.477 9c.413.368.523.706.523 1 0 .334-.127.712-.701 1.143-.575.43-1.632.85-3.299.857l-1.006.007V14H5a1 1 0 000 2h3c4 0 6-2 6-4 0-.338-.081-.672-.195-1z\"/><rect ry=\"1\" y=\"8\" x=\"1\" height=\"2\" width=\"16\"/></svg>`,\n  ul: `<svg height=\"18\" width=\"18\"><circle cx=\"2\" cy=\"9\" r=\"2\"/><circle cy=\"4\" cx=\"2\" r=\"2\"/><rect y=\"3\" x=\"6\" height=\"2\" width=\"12\" ry=\"1\"/><circle cx=\"2\" cy=\"14\" r=\"2\"/><rect ry=\"1\" width=\"12\" height=\"2\" x=\"6\" y=\"8\"/><rect y=\"13\" x=\"6\" height=\"2\" width=\"12\" ry=\"1\"/></svg>`\n};\n\nexport default svg;","import svg from './svg/svg';\n\nconst isMacLike = /(Mac|iPhone|iPod|iPad)/i.test(typeof navigator !== \"undefined\" ? navigator.platform : \"\");\n\nconst DefaultCommands = {\n  'bold': {\n    name: 'bold',\n    action: 'bold',\n    innerHTML: svg.bold,\n    title: 'Bold',\n    hotkey: 'Mod-B',\n  },\n  'italic': {\n    name: 'italic',\n    action: 'italic',\n    innerHTML: svg.italic,\n    title: 'Italic',\n    hotkey: 'Mod-I',\n  },\n  'strikethrough': {\n    name: 'strikethrough',\n    action: 'strikethrough',\n    innerHTML: svg.strikethrough,\n    title: 'Strikethrough',\n    hotkey: 'Mod2-Shift-5',\n  },\n  'code': {\n    name: 'code',\n    action: 'code',\n    innerHTML: svg.code,\n    title: 'Format as code',\n  },\n  'h1': {\n    name: 'h1',\n    action: 'h1',\n    innerHTML: svg.h1,\n    title: 'Level 1 heading',\n    hotkey: 'Mod-Shift-1',\n  },\n  'h2': {\n    name: 'h2',\n    action: 'h2',\n    innerHTML: svg.h2,\n    title: 'Level 2 heading',\n    hotkey: 'Mod-Shift-2',\n  },\n  'ul': {\n    name: 'ul',\n    action: 'ul',\n    innerHTML: svg.ul,\n    title: 'Bulleted list',\n  },\n  'ol': {\n    name: 'ol',\n    action: 'ol',\n    innerHTML: svg.ol,\n    title: 'Numbered list',\n  },\n  'blockquote': {\n    name: 'blockquote',\n    action: 'blockquote',\n    innerHTML: svg.blockquote,\n    title: 'Quote',\n    hotkey: 'Mod2-Shift-Q',\n  },\n  'insertLink': {\n    name: 'insertLink',\n    action: (editor) => {if (editor.isInlineFormattingAllowed()) editor.wrapSelection('[', ']()')},\n    enabled: (editor, focus, anchor) => editor.isInlineFormattingAllowed(focus, anchor) ? false : null,\n    innerHTML: svg.link,\n    title: 'Insert link',\n    hotkey: 'Mod-K',\n  },\n  'insertImage': {\n    name: 'insertImage',\n    action: (editor) => {if (editor.isInlineFormattingAllowed()) editor.wrapSelection('![', ']()')},\n    enabled: (editor, focus, anchor) => editor.isInlineFormattingAllowed(focus, anchor) ? false : null,\n    innerHTML: svg.image,\n    title: 'Insert image',\n    hotkey: 'Mod2-Shift-I',\n  },\n  'hr': {\n    name: 'hr',\n    action: (editor) => editor.paste('\\n***\\n'),\n    enabled: () => false,\n    innerHTML: svg.hr,\n    title: 'Insert horizontal line',\n    hotkey: 'Mod2-Shift-L'\n  }\n}\n\n\nclass CommandBar {\n  constructor(props) {\n    this.e = null;\n    this.editor = null;\n    this.commands = [];\n    this.buttons = {};\n    this.state = {};\n    this.hotkeys = [];\n\n    let element = props.element;\n    if (element && !element.tagName) {\n      element = document.getElementById(props.element);\n    }\n    if (!element) {\n      element = document.body; \n    }\n    this.createCommandBarElement(element, props.commands || ['bold', 'italic', 'strikethrough', '|', 'code', '|', 'h1', 'h2', '|', 'ul', 'ol', '|', 'blockquote', 'hr', '|', 'insertLink', 'insertImage']);\n    document.addEventListener('keydown', (e) => this.handleKeydown(e));\n    if (props.editor) this.setEditor(props.editor);\n  }\n\n  createCommandBarElement(parentElement, commands) {\n    this.e = document.createElement('div');\n    this.e.className = 'TMCommandBar';\n\n    for (let command of commands) {\n      if (command == '|') {\n        let el = document.createElement('div');\n        el.className = 'TMCommandDivider';\n        this.e.appendChild(el);\n      } else {\n        let commandName;\n        if (typeof command == \"string\") {\n          // Reference to default command\n\n          if (DefaultCommands[command]) {\n            commandName = command;\n            this.commands[commandName] = DefaultCommands[commandName];\n\n            \n          } else {\n            continue;\n          }\n          \n        } else if (typeof command == \"object\" && command.name) {\n          commandName = command.name;\n          this.commands[commandName] = {}; \n          if (DefaultCommands[commandName]) Object.assign(this.commands[commandName], DefaultCommands[commandName]);\n          Object.assign(this.commands[commandName], command);\n        \n\n        } else {\n          continue;\n        }\n\n        let title = this.commands[commandName].title || commandName;\n\n        if (this.commands[commandName].hotkey) {\n          const keys = this.commands[commandName].hotkey.split('-');\n          // construct modifiers\n          let modifiers = [];\n          let modifierexplanation = [];\n          for (let i = 0; i < keys.length - 1; i++) {\n            switch (keys[i]) {\n              case 'Ctrl': modifiers.push('ctrlKey'); modifierexplanation.push('Ctrl'); break;\n              case 'Cmd': modifiers.push('metaKey'); modifierexplanation.push('⌘'); break;\n              case 'Alt': modifiers.push('altKey'); modifierexplanation.push('Alt'); break;\n              case 'Option': modifiers.push('altKey'); modifierexplanation.push('⌥'); break;\n              case 'Win': modifiers.push('metaKey'); modifierexplanation.push('⊞ Win'); break;\n\n              case 'Shift':  modifiers.push('shiftKey'); modifierexplanation.push('⇧'); break;\n\n              case 'Mod': // Mod is a convenience mechanism: Ctrl on Windows, Cmd on Mac\n                if (isMacLike) {modifiers.push('metaKey'); modifierexplanation.push('⌘');} \n                else {modifiers.push('ctrlKey'); modifierexplanation.push('Ctrl');} \n                break; \n              case 'Mod2': \n                modifiers.push('altKey'); \n                if (isMacLike) modifierexplanation.push('⌥');\n                else modifierexplanation.push('Alt');\n                break; // Mod2 is a convenience mechanism: Alt on Windows, Option on Mac\n            }\n          }\n          modifierexplanation.push(keys[keys.length - 1]);\n          let hotkey = {\n            \n            modifiers: modifiers,\n            command: commandName,\n          };\n          // TODO Right now this is working only for letters and numbers\n          if (keys[keys.length - 1].match(/^[0-9]$/)) {\n            hotkey.code = `Digit${keys[keys.length - 1]}`;\n          } else {\n            hotkey.key = keys[keys.length - 1].toLowerCase();\n          }\n          this.hotkeys.push(hotkey);\n          title = title.concat(` (${modifierexplanation.join('+')})`);\n        }\n\n        this.buttons[commandName] = document.createElement('div');\n        this.buttons[commandName].className = 'TMCommandButton TMCommandButton_Disabled';\n        this.buttons[commandName].title = title;\n        this.buttons[commandName].innerHTML = this.commands[commandName].innerHTML;\n\n        this.buttons[commandName].addEventListener('mousedown', (e) => this.handleClick(commandName, e));\n        this.e.appendChild(this.buttons[commandName]);\n      }\n    }\n    parentElement.appendChild(this.e);\n  }\n\n  handleClick(commandName, event) {\n    if (!this.editor) return;\n    event.preventDefault();\n    if (typeof this.commands[commandName].action == \"string\") {\n      if (this.state[commandName] === false) this.editor.setCommandState(commandName, true);\n      else this.editor.setCommandState(commandName, false);  \n    } else if (typeof this.commands[commandName].action == \"function\") {\n      this.commands[commandName].action(this.editor);\n    }\n  }\n\n  setEditor(editor) {\n    this.editor = editor;\n    editor.addEventListener('selection', (e) => this.handleSelection(e));\n  }\n\n  handleSelection(event) {\n    if (event.commandState) {\n      for (let command in this.commands) {\n        if (event.commandState[command] === undefined) {\n          if (this.commands[command].enabled) this.state[command] = this.commands[command].enabled(this.editor, event.focus, event.anchor);\n          else this.state[command] = event.focus ? false : null;\n        } else {\n          this.state[command] = event.commandState[command];\n        }\n\n        if (this.state[command] === true) {\n          this.buttons[command].className = 'TMCommandButton TMCommandButton_Active';\n        } else if (this.state[command] === false) {\n          this.buttons[command].className = 'TMCommandButton TMCommandButton_Inactive';\n        } else {\n          this.buttons[command].className =  'TMCommandButton TMCommandButton_Disabled';\n        }\n      }\n    }\n  }\n\n  handleKeydown(event) {\n    outer: for (let hotkey of this.hotkeys) {\n      if ((hotkey.key && event.key.toLowerCase() == hotkey.key) || (hotkey.code && event.code == hotkey.code)) {\n        // Key matches hotkey. Look for any required modifier that wasn't pressed\n        for (let modifier of hotkey.modifiers) {\n          if (!event[modifier]) continue outer;\n        }\n        // Everything matches.\n        this.handleClick(hotkey.command, event);\n        return;\n      }\n    }\n  }\n}\n\nexport default CommandBar;","'use strict';\nvar check = function (it) {\n  return it && it.Math === Math && it;\n};\n\n// https://github.com/zloirock/core-js/issues/86#issuecomment-115759028\nmodule.exports =\n  // eslint-disable-next-line es/no-global-this -- safe\n  check(typeof globalThis == 'object' && globalThis) ||\n  check(typeof window == 'object' && window) ||\n  // eslint-disable-next-line no-restricted-globals -- safe\n  check(typeof self == 'object' && self) ||\n  check(typeof global == 'object' && global) ||\n  // eslint-disable-next-line no-new-func -- fallback\n  (function () { return this; })() || this || Function('return this')();\n","'use strict';\nmodule.exports = function (exec) {\n  try {\n    return !!exec();\n  } catch (error) {\n    return true;\n  }\n};\n","'use strict';\nvar fails = require('../internals/fails');\n\n// Detect IE8's incomplete defineProperty implementation\nmodule.exports = !fails(function () {\n  // eslint-disable-next-line es/no-object-defineproperty -- required for testing\n  return Object.defineProperty({}, 1, { get: function () { return 7; } })[1] !== 7;\n});\n","'use strict';\nvar fails = require('../internals/fails');\n\nmodule.exports = !fails(function () {\n  // eslint-disable-next-line es/no-function-prototype-bind -- safe\n  var test = (function () { /* empty */ }).bind();\n  // eslint-disable-next-line no-prototype-builtins -- safe\n  return typeof test != 'function' || test.hasOwnProperty('prototype');\n});\n","'use strict';\nvar NATIVE_BIND = require('../internals/function-bind-native');\n\nvar FunctionPrototype = Function.prototype;\nvar call = FunctionPrototype.call;\nvar uncurryThisWithBind = NATIVE_BIND && FunctionPrototype.bind.bind(call, call);\n\nmodule.exports = NATIVE_BIND ? uncurryThisWithBind : function (fn) {\n  return function () {\n    return call.apply(fn, arguments);\n  };\n};\n","'use strict';\nvar documentAll = typeof document == 'object' && document.all;\n\n// https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot\n// eslint-disable-next-line unicorn/no-typeof-undefined -- required for testing\nvar IS_HTMLDDA = typeof documentAll == 'undefined' && documentAll !== undefined;\n\nmodule.exports = {\n  all: documentAll,\n  IS_HTMLDDA: IS_HTMLDDA\n};\n","'use strict';\nvar $documentAll = require('../internals/document-all');\n\nvar documentAll = $documentAll.all;\n\n// `IsCallable` abstract operation\n// https://tc39.es/ecma262/#sec-iscallable\nmodule.exports = $documentAll.IS_HTMLDDA ? function (argument) {\n  return typeof argument == 'function' || argument === documentAll;\n} : function (argument) {\n  return typeof argument == 'function';\n};\n","'use strict';\n// we can't use just `it == null` since of `document.all` special case\n// https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot-aec\nmodule.exports = function (it) {\n  return it === null || it === undefined;\n};\n","'use strict';\nvar isNullOrUndefined = require('../internals/is-null-or-undefined');\n\nvar $TypeError = TypeError;\n\n// `RequireObjectCoercible` abstract operation\n// https://tc39.es/ecma262/#sec-requireobjectcoercible\nmodule.exports = function (it) {\n  if (isNullOrUndefined(it)) throw new $TypeError(\"Can't call method on \" + it);\n  return it;\n};\n","'use strict';\nvar requireObjectCoercible = require('../internals/require-object-coercible');\n\nvar $Object = Object;\n\n// `ToObject` abstract operation\n// https://tc39.es/ecma262/#sec-toobject\nmodule.exports = function (argument) {\n  return $Object(requireObjectCoercible(argument));\n};\n","'use strict';\nvar uncurryThis = require('../internals/function-uncurry-this');\nvar toObject = require('../internals/to-object');\n\nvar hasOwnProperty = uncurryThis({}.hasOwnProperty);\n\n// `HasOwnProperty` abstract operation\n// https://tc39.es/ecma262/#sec-hasownproperty\n// eslint-disable-next-line es/no-object-hasown -- safe\nmodule.exports = Object.hasOwn || function hasOwn(it, key) {\n  return hasOwnProperty(toObject(it), key);\n};\n","'use strict';\nvar DESCRIPTORS = require('../internals/descriptors');\nvar hasOwn = require('../internals/has-own-property');\n\nvar FunctionPrototype = Function.prototype;\n// eslint-disable-next-line es/no-object-getownpropertydescriptor -- safe\nvar getDescriptor = DESCRIPTORS && Object.getOwnPropertyDescriptor;\n\nvar EXISTS = hasOwn(FunctionPrototype, 'name');\n// additional protection from minified / mangled / dropped function names\nvar PROPER = EXISTS && (function something() { /* empty */ }).name === 'something';\nvar CONFIGURABLE = EXISTS && (!DESCRIPTORS || (DESCRIPTORS && getDescriptor(FunctionPrototype, 'name').configurable));\n\nmodule.exports = {\n  EXISTS: EXISTS,\n  PROPER: PROPER,\n  CONFIGURABLE: CONFIGURABLE\n};\n","'use strict';\nvar global = require('../internals/global');\n\n// eslint-disable-next-line es/no-object-defineproperty -- safe\nvar defineProperty = Object.defineProperty;\n\nmodule.exports = function (key, value) {\n  try {\n    defineProperty(global, key, { value: value, configurable: true, writable: true });\n  } catch (error) {\n    global[key] = value;\n  } return value;\n};\n","'use strict';\nvar global = require('../internals/global');\nvar defineGlobalProperty = require('../internals/define-global-property');\n\nvar SHARED = '__core-js_shared__';\nvar store = global[SHARED] || defineGlobalProperty(SHARED, {});\n\nmodule.exports = store;\n","'use strict';\nvar uncurryThis = require('../internals/function-uncurry-this');\nvar isCallable = require('../internals/is-callable');\nvar store = require('../internals/shared-store');\n\nvar functionToString = uncurryThis(Function.toString);\n\n// this helper broken in `core-js@3.4.1-3.4.4`, so we can't use `shared` helper\nif (!isCallable(store.inspectSource)) {\n  store.inspectSource = function (it) {\n    return functionToString(it);\n  };\n}\n\nmodule.exports = store.inspectSource;\n","'use strict';\nvar global = require('../internals/global');\nvar isCallable = require('../internals/is-callable');\n\nvar WeakMap = global.WeakMap;\n\nmodule.exports = isCallable(WeakMap) && /native code/.test(String(WeakMap));\n","'use strict';\nvar isCallable = require('../internals/is-callable');\nvar $documentAll = require('../internals/document-all');\n\nvar documentAll = $documentAll.all;\n\nmodule.exports = $documentAll.IS_HTMLDDA ? function (it) {\n  return typeof it == 'object' ? it !== null : isCallable(it) || it === documentAll;\n} : function (it) {\n  return typeof it == 'object' ? it !== null : isCallable(it);\n};\n","'use strict';\nvar global = require('../internals/global');\nvar isObject = require('../internals/is-object');\n\nvar document = global.document;\n// typeof document.createElement is 'object' in old IE\nvar EXISTS = isObject(document) && isObject(document.createElement);\n\nmodule.exports = function (it) {\n  return EXISTS ? document.createElement(it) : {};\n};\n","'use strict';\nvar DESCRIPTORS = require('../internals/descriptors');\nvar fails = require('../internals/fails');\nvar createElement = require('../internals/document-create-element');\n\n// Thanks to IE8 for its funny defineProperty\nmodule.exports = !DESCRIPTORS && !fails(function () {\n  // eslint-disable-next-line es/no-object-defineproperty -- required for testing\n  return Object.defineProperty(createElement('div'), 'a', {\n    get: function () { return 7; }\n  }).a !== 7;\n});\n","'use strict';\nvar DESCRIPTORS = require('../internals/descriptors');\nvar fails = require('../internals/fails');\n\n// V8 ~ Chrome 36-\n// https://bugs.chromium.org/p/v8/issues/detail?id=3334\nmodule.exports = DESCRIPTORS && fails(function () {\n  // eslint-disable-next-line es/no-object-defineproperty -- required for testing\n  return Object.defineProperty(function () { /* empty */ }, 'prototype', {\n    value: 42,\n    writable: false\n  }).prototype !== 42;\n});\n","'use strict';\nvar isObject = require('../internals/is-object');\n\nvar $String = String;\nvar $TypeError = TypeError;\n\n// `Assert: Type(argument) is Object`\nmodule.exports = function (argument) {\n  if (isObject(argument)) return argument;\n  throw new $TypeError($String(argument) + ' is not an object');\n};\n","'use strict';\nvar NATIVE_BIND = require('../internals/function-bind-native');\n\nvar call = Function.prototype.call;\n\nmodule.exports = NATIVE_BIND ? call.bind(call) : function () {\n  return call.apply(call, arguments);\n};\n","'use strict';\nvar global = require('../internals/global');\nvar isCallable = require('../internals/is-callable');\n\nvar aFunction = function (argument) {\n  return isCallable(argument) ? argument : undefined;\n};\n\nmodule.exports = function (namespace, method) {\n  return arguments.length < 2 ? aFunction(global[namespace]) : global[namespace] && global[namespace][method];\n};\n","'use strict';\nvar uncurryThis = require('../internals/function-uncurry-this');\n\nmodule.exports = uncurryThis({}.isPrototypeOf);\n","'use strict';\nmodule.exports = typeof navigator != 'undefined' && String(navigator.userAgent) || '';\n","'use strict';\nvar global = require('../internals/global');\nvar userAgent = require('../internals/engine-user-agent');\n\nvar process = global.process;\nvar Deno = global.Deno;\nvar versions = process && process.versions || Deno && Deno.version;\nvar v8 = versions && versions.v8;\nvar match, version;\n\nif (v8) {\n  match = v8.split('.');\n  // in old Chrome, versions of V8 isn't V8 = Chrome / 10\n  // but their correct versions are not interesting for us\n  version = match[0] > 0 && match[0] < 4 ? 1 : +(match[0] + match[1]);\n}\n\n// BrowserFS NodeJS `process` polyfill incorrectly set `.v8` to `0.0`\n// so check `userAgent` even if `.v8` exists, but 0\nif (!version && userAgent) {\n  match = userAgent.match(/Edge\\/(\\d+)/);\n  if (!match || match[1] >= 74) {\n    match = userAgent.match(/Chrome\\/(\\d+)/);\n    if (match) version = +match[1];\n  }\n}\n\nmodule.exports = version;\n","'use strict';\n/* eslint-disable es/no-symbol -- required for testing */\nvar V8_VERSION = require('../internals/engine-v8-version');\nvar fails = require('../internals/fails');\nvar global = require('../internals/global');\n\nvar $String = global.String;\n\n// eslint-disable-next-line es/no-object-getownpropertysymbols -- required for testing\nmodule.exports = !!Object.getOwnPropertySymbols && !fails(function () {\n  var symbol = Symbol('symbol detection');\n  // Chrome 38 Symbol has incorrect toString conversion\n  // `get-own-property-symbols` polyfill symbols converted to object are not Symbol instances\n  // nb: Do not call `String` directly to avoid this being optimized out to `symbol+''` which will,\n  // of course, fail.\n  return !$String(symbol) || !(Object(symbol) instanceof Symbol) ||\n    // Chrome 38-40 symbols are not inherited from DOM collections prototypes to instances\n    !Symbol.sham && V8_VERSION && V8_VERSION < 41;\n});\n","'use strict';\n/* eslint-disable es/no-symbol -- required for testing */\nvar NATIVE_SYMBOL = require('../internals/symbol-constructor-detection');\n\nmodule.exports = NATIVE_SYMBOL\n  && !Symbol.sham\n  && typeof Symbol.iterator == 'symbol';\n","'use strict';\nvar getBuiltIn = require('../internals/get-built-in');\nvar isCallable = require('../internals/is-callable');\nvar isPrototypeOf = require('../internals/object-is-prototype-of');\nvar USE_SYMBOL_AS_UID = require('../internals/use-symbol-as-uid');\n\nvar $Object = Object;\n\nmodule.exports = USE_SYMBOL_AS_UID ? function (it) {\n  return typeof it == 'symbol';\n} : function (it) {\n  var $Symbol = getBuiltIn('Symbol');\n  return isCallable($Symbol) && isPrototypeOf($Symbol.prototype, $Object(it));\n};\n","'use strict';\nvar $String = String;\n\nmodule.exports = function (argument) {\n  try {\n    return $String(argument);\n  } catch (error) {\n    return 'Object';\n  }\n};\n","'use strict';\nvar isCallable = require('../internals/is-callable');\nvar tryToString = require('../internals/try-to-string');\n\nvar $TypeError = TypeError;\n\n// `Assert: IsCallable(argument) is true`\nmodule.exports = function (argument) {\n  if (isCallable(argument)) return argument;\n  throw new $TypeError(tryToString(argument) + ' is not a function');\n};\n","'use strict';\nvar aCallable = require('../internals/a-callable');\nvar isNullOrUndefined = require('../internals/is-null-or-undefined');\n\n// `GetMethod` abstract operation\n// https://tc39.es/ecma262/#sec-getmethod\nmodule.exports = function (V, P) {\n  var func = V[P];\n  return isNullOrUndefined(func) ? undefined : aCallable(func);\n};\n","'use strict';\nvar call = require('../internals/function-call');\nvar isCallable = require('../internals/is-callable');\nvar isObject = require('../internals/is-object');\n\nvar $TypeError = TypeError;\n\n// `OrdinaryToPrimitive` abstract operation\n// https://tc39.es/ecma262/#sec-ordinarytoprimitive\nmodule.exports = function (input, pref) {\n  var fn, val;\n  if (pref === 'string' && isCallable(fn = input.toString) && !isObject(val = call(fn, input))) return val;\n  if (isCallable(fn = input.valueOf) && !isObject(val = call(fn, input))) return val;\n  if (pref !== 'string' && isCallable(fn = input.toString) && !isObject(val = call(fn, input))) return val;\n  throw new $TypeError(\"Can't convert object to primitive value\");\n};\n","'use strict';\nvar IS_PURE = require('../internals/is-pure');\nvar store = require('../internals/shared-store');\n\n(module.exports = function (key, value) {\n  return store[key] || (store[key] = value !== undefined ? value : {});\n})('versions', []).push({\n  version: '3.33.0',\n  mode: IS_PURE ? 'pure' : 'global',\n  copyright: '© 2014-2023 Denis Pushkarev (zloirock.ru)',\n  license: 'https://github.com/zloirock/core-js/blob/v3.33.0/LICENSE',\n  source: 'https://github.com/zloirock/core-js'\n});\n","'use strict';\nvar uncurryThis = require('../internals/function-uncurry-this');\n\nvar id = 0;\nvar postfix = Math.random();\nvar toString = uncurryThis(1.0.toString);\n\nmodule.exports = function (key) {\n  return 'Symbol(' + (key === undefined ? '' : key) + ')_' + toString(++id + postfix, 36);\n};\n","'use strict';\nvar global = require('../internals/global');\nvar shared = require('../internals/shared');\nvar hasOwn = require('../internals/has-own-property');\nvar uid = require('../internals/uid');\nvar NATIVE_SYMBOL = require('../internals/symbol-constructor-detection');\nvar USE_SYMBOL_AS_UID = require('../internals/use-symbol-as-uid');\n\nvar Symbol = global.Symbol;\nvar WellKnownSymbolsStore = shared('wks');\nvar createWellKnownSymbol = USE_SYMBOL_AS_UID ? Symbol['for'] || Symbol : Symbol && Symbol.withoutSetter || uid;\n\nmodule.exports = function (name) {\n  if (!hasOwn(WellKnownSymbolsStore, name)) {\n    WellKnownSymbolsStore[name] = NATIVE_SYMBOL && hasOwn(Symbol, name)\n      ? Symbol[name]\n      : createWellKnownSymbol('Symbol.' + name);\n  } return WellKnownSymbolsStore[name];\n};\n","'use strict';\nvar call = require('../internals/function-call');\nvar isObject = require('../internals/is-object');\nvar isSymbol = require('../internals/is-symbol');\nvar getMethod = require('../internals/get-method');\nvar ordinaryToPrimitive = require('../internals/ordinary-to-primitive');\nvar wellKnownSymbol = require('../internals/well-known-symbol');\n\nvar $TypeError = TypeError;\nvar TO_PRIMITIVE = wellKnownSymbol('toPrimitive');\n\n// `ToPrimitive` abstract operation\n// https://tc39.es/ecma262/#sec-toprimitive\nmodule.exports = function (input, pref) {\n  if (!isObject(input) || isSymbol(input)) return input;\n  var exoticToPrim = getMethod(input, TO_PRIMITIVE);\n  var result;\n  if (exoticToPrim) {\n    if (pref === undefined) pref = 'default';\n    result = call(exoticToPrim, input, pref);\n    if (!isObject(result) || isSymbol(result)) return result;\n    throw new $TypeError(\"Can't convert object to primitive value\");\n  }\n  if (pref === undefined) pref = 'number';\n  return ordinaryToPrimitive(input, pref);\n};\n","'use strict';\nvar toPrimitive = require('../internals/to-primitive');\nvar isSymbol = require('../internals/is-symbol');\n\n// `ToPropertyKey` abstract operation\n// https://tc39.es/ecma262/#sec-topropertykey\nmodule.exports = function (argument) {\n  var key = toPrimitive(argument, 'string');\n  return isSymbol(key) ? key : key + '';\n};\n","'use strict';\nvar DESCRIPTORS = require('../internals/descriptors');\nvar IE8_DOM_DEFINE = require('../internals/ie8-dom-define');\nvar V8_PROTOTYPE_DEFINE_BUG = require('../internals/v8-prototype-define-bug');\nvar anObject = require('../internals/an-object');\nvar toPropertyKey = require('../internals/to-property-key');\n\nvar $TypeError = TypeError;\n// eslint-disable-next-line es/no-object-defineproperty -- safe\nvar $defineProperty = Object.defineProperty;\n// eslint-disable-next-line es/no-object-getownpropertydescriptor -- safe\nvar $getOwnPropertyDescriptor = Object.getOwnPropertyDescriptor;\nvar ENUMERABLE = 'enumerable';\nvar CONFIGURABLE = 'configurable';\nvar WRITABLE = 'writable';\n\n// `Object.defineProperty` method\n// https://tc39.es/ecma262/#sec-object.defineproperty\nexports.f = DESCRIPTORS ? V8_PROTOTYPE_DEFINE_BUG ? function defineProperty(O, P, Attributes) {\n  anObject(O);\n  P = toPropertyKey(P);\n  anObject(Attributes);\n  if (typeof O === 'function' && P === 'prototype' && 'value' in Attributes && WRITABLE in Attributes && !Attributes[WRITABLE]) {\n    var current = $getOwnPropertyDescriptor(O, P);\n    if (current && current[WRITABLE]) {\n      O[P] = Attributes.value;\n      Attributes = {\n        configurable: CONFIGURABLE in Attributes ? Attributes[CONFIGURABLE] : current[CONFIGURABLE],\n        enumerable: ENUMERABLE in Attributes ? Attributes[ENUMERABLE] : current[ENUMERABLE],\n        writable: false\n      };\n    }\n  } return $defineProperty(O, P, Attributes);\n} : $defineProperty : function defineProperty(O, P, Attributes) {\n  anObject(O);\n  P = toPropertyKey(P);\n  anObject(Attributes);\n  if (IE8_DOM_DEFINE) try {\n    return $defineProperty(O, P, Attributes);\n  } catch (error) { /* empty */ }\n  if ('get' in Attributes || 'set' in Attributes) throw new $TypeError('Accessors not supported');\n  if ('value' in Attributes) O[P] = Attributes.value;\n  return O;\n};\n","'use strict';\nmodule.exports = function (bitmap, value) {\n  return {\n    enumerable: !(bitmap & 1),\n    configurable: !(bitmap & 2),\n    writable: !(bitmap & 4),\n    value: value\n  };\n};\n","'use strict';\nvar DESCRIPTORS = require('../internals/descriptors');\nvar definePropertyModule = require('../internals/object-define-property');\nvar createPropertyDescriptor = require('../internals/create-property-descriptor');\n\nmodule.exports = DESCRIPTORS ? function (object, key, value) {\n  return definePropertyModule.f(object, key, createPropertyDescriptor(1, value));\n} : function (object, key, value) {\n  object[key] = value;\n  return object;\n};\n","'use strict';\nvar shared = require('../internals/shared');\nvar uid = require('../internals/uid');\n\nvar keys = shared('keys');\n\nmodule.exports = function (key) {\n  return keys[key] || (keys[key] = uid(key));\n};\n","'use strict';\nvar NATIVE_WEAK_MAP = require('../internals/weak-map-basic-detection');\nvar global = require('../internals/global');\nvar isObject = require('../internals/is-object');\nvar createNonEnumerableProperty = require('../internals/create-non-enumerable-property');\nvar hasOwn = require('../internals/has-own-property');\nvar shared = require('../internals/shared-store');\nvar sharedKey = require('../internals/shared-key');\nvar hiddenKeys = require('../internals/hidden-keys');\n\nvar OBJECT_ALREADY_INITIALIZED = 'Object already initialized';\nvar TypeError = global.TypeError;\nvar WeakMap = global.WeakMap;\nvar set, get, has;\n\nvar enforce = function (it) {\n  return has(it) ? get(it) : set(it, {});\n};\n\nvar getterFor = function (TYPE) {\n  return function (it) {\n    var state;\n    if (!isObject(it) || (state = get(it)).type !== TYPE) {\n      throw new TypeError('Incompatible receiver, ' + TYPE + ' required');\n    } return state;\n  };\n};\n\nif (NATIVE_WEAK_MAP || shared.state) {\n  var store = shared.state || (shared.state = new WeakMap());\n  /* eslint-disable no-self-assign -- prototype methods protection */\n  store.get = store.get;\n  store.has = store.has;\n  store.set = store.set;\n  /* eslint-enable no-self-assign -- prototype methods protection */\n  set = function (it, metadata) {\n    if (store.has(it)) throw new TypeError(OBJECT_ALREADY_INITIALIZED);\n    metadata.facade = it;\n    store.set(it, metadata);\n    return metadata;\n  };\n  get = function (it) {\n    return store.get(it) || {};\n  };\n  has = function (it) {\n    return store.has(it);\n  };\n} else {\n  var STATE = sharedKey('state');\n  hiddenKeys[STATE] = true;\n  set = function (it, metadata) {\n    if (hasOwn(it, STATE)) throw new TypeError(OBJECT_ALREADY_INITIALIZED);\n    metadata.facade = it;\n    createNonEnumerableProperty(it, STATE, metadata);\n    return metadata;\n  };\n  get = function (it) {\n    return hasOwn(it, STATE) ? it[STATE] : {};\n  };\n  has = function (it) {\n    return hasOwn(it, STATE);\n  };\n}\n\nmodule.exports = {\n  set: set,\n  get: get,\n  has: has,\n  enforce: enforce,\n  getterFor: getterFor\n};\n","'use strict';\nvar uncurryThis = require('../internals/function-uncurry-this');\nvar fails = require('../internals/fails');\nvar isCallable = require('../internals/is-callable');\nvar hasOwn = require('../internals/has-own-property');\nvar DESCRIPTORS = require('../internals/descriptors');\nvar CONFIGURABLE_FUNCTION_NAME = require('../internals/function-name').CONFIGURABLE;\nvar inspectSource = require('../internals/inspect-source');\nvar InternalStateModule = require('../internals/internal-state');\n\nvar enforceInternalState = InternalStateModule.enforce;\nvar getInternalState = InternalStateModule.get;\nvar $String = String;\n// eslint-disable-next-line es/no-object-defineproperty -- safe\nvar defineProperty = Object.defineProperty;\nvar stringSlice = uncurryThis(''.slice);\nvar replace = uncurryThis(''.replace);\nvar join = uncurryThis([].join);\n\nvar CONFIGURABLE_LENGTH = DESCRIPTORS && !fails(function () {\n  return defineProperty(function () { /* empty */ }, 'length', { value: 8 }).length !== 8;\n});\n\nvar TEMPLATE = String(String).split('String');\n\nvar makeBuiltIn = module.exports = function (value, name, options) {\n  if (stringSlice($String(name), 0, 7) === 'Symbol(') {\n    name = '[' + replace($String(name), /^Symbol\\(([^)]*)\\)/, '$1') + ']';\n  }\n  if (options && options.getter) name = 'get ' + name;\n  if (options && options.setter) name = 'set ' + name;\n  if (!hasOwn(value, 'name') || (CONFIGURABLE_FUNCTION_NAME && value.name !== name)) {\n    if (DESCRIPTORS) defineProperty(value, 'name', { value: name, configurable: true });\n    else value.name = name;\n  }\n  if (CONFIGURABLE_LENGTH && options && hasOwn(options, 'arity') && value.length !== options.arity) {\n    defineProperty(value, 'length', { value: options.arity });\n  }\n  try {\n    if (options && hasOwn(options, 'constructor') && options.constructor) {\n      if (DESCRIPTORS) defineProperty(value, 'prototype', { writable: false });\n    // in V8 ~ Chrome 53, prototypes of some methods, like `Array.prototype.values`, are non-writable\n    } else if (value.prototype) value.prototype = undefined;\n  } catch (error) { /* empty */ }\n  var state = enforceInternalState(value);\n  if (!hasOwn(state, 'source')) {\n    state.source = join(TEMPLATE, typeof name == 'string' ? name : '');\n  } return value;\n};\n\n// add fake Function#toString for correct work wrapped methods / constructors with methods like LoDash isNative\n// eslint-disable-next-line no-extend-native -- required\nFunction.prototype.toString = makeBuiltIn(function toString() {\n  return isCallable(this) && getInternalState(this).source || inspectSource(this);\n}, 'toString');\n","'use strict';\nvar makeBuiltIn = require('../internals/make-built-in');\nvar defineProperty = require('../internals/object-define-property');\n\nmodule.exports = function (target, name, descriptor) {\n  if (descriptor.get) makeBuiltIn(descriptor.get, name, { getter: true });\n  if (descriptor.set) makeBuiltIn(descriptor.set, name, { setter: true });\n  return defineProperty.f(target, name, descriptor);\n};\n","'use strict';\nvar anObject = require('../internals/an-object');\n\n// `RegExp.prototype.flags` getter implementation\n// https://tc39.es/ecma262/#sec-get-regexp.prototype.flags\nmodule.exports = function () {\n  var that = anObject(this);\n  var result = '';\n  if (that.hasIndices) result += 'd';\n  if (that.global) result += 'g';\n  if (that.ignoreCase) result += 'i';\n  if (that.multiline) result += 'm';\n  if (that.dotAll) result += 's';\n  if (that.unicode) result += 'u';\n  if (that.unicodeSets) result += 'v';\n  if (that.sticky) result += 'y';\n  return result;\n};\n","'use strict';\nvar global = require('../internals/global');\nvar DESCRIPTORS = require('../internals/descriptors');\nvar defineBuiltInAccessor = require('../internals/define-built-in-accessor');\nvar regExpFlags = require('../internals/regexp-flags');\nvar fails = require('../internals/fails');\n\n// babel-minify and Closure Compiler transpiles RegExp('.', 'd') -> /./d and it causes SyntaxError\nvar RegExp = global.RegExp;\nvar RegExpPrototype = RegExp.prototype;\n\nvar FORCED = DESCRIPTORS && fails(function () {\n  var INDICES_SUPPORT = true;\n  try {\n    RegExp('.', 'd');\n  } catch (error) {\n    INDICES_SUPPORT = false;\n  }\n\n  var O = {};\n  // modern V8 bug\n  var calls = '';\n  var expected = INDICES_SUPPORT ? 'dgimsy' : 'gimsy';\n\n  var addGetter = function (key, chr) {\n    // eslint-disable-next-line es/no-object-defineproperty -- safe\n    Object.defineProperty(O, key, { get: function () {\n      calls += chr;\n      return true;\n    } });\n  };\n\n  var pairs = {\n    dotAll: 's',\n    global: 'g',\n    ignoreCase: 'i',\n    multiline: 'm',\n    sticky: 'y'\n  };\n\n  if (INDICES_SUPPORT) pairs.hasIndices = 'd';\n\n  for (var key in pairs) addGetter(key, pairs[key]);\n\n  // eslint-disable-next-line es/no-object-getownpropertydescriptor -- safe\n  var result = Object.getOwnPropertyDescriptor(RegExpPrototype, 'flags').get.call(O);\n\n  return result !== expected || calls !== expected;\n});\n\n// `RegExp.prototype.flags` getter\n// https://tc39.es/ecma262/#sec-get-regexp.prototype.flags\nif (FORCED) defineBuiltInAccessor(RegExpPrototype, 'flags', {\n  configurable: true,\n  get: regExpFlags\n});\n","// const replacements = {\n//   ASCIIPunctuation: '!\"#$%&\\'()*+,\\\\-./:;<=>?@\\\\[\\\\]^_`{|}~',\n//   TriggerChars: '`_\\*\\[\\]\\(\\)',\n//   Scheme: `[A-Za-z][A-Za-z0-9\\+\\.\\-]{1,31}`,\n//   Email: `[a-zA-Z0-9.!#$%&'*+/=?^_\\`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*`, // From CommonMark spec\n\n// }\nconst replacements = {\n  ASCIIPunctuation: /[!\"#$%&'()*+,\\-./:;<=>?@[\\]^_`{|}~\\\\]/,  \n  NotTriggerChar: /[^`_*[\\]()<>!~]/,\n  Scheme: /[A-Za-z][A-Za-z0-9+.-]{1,31}/,\n  Email: /[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*/, // From CommonMark spec\n  HTMLOpenTag: /<HTMLTagName(?:HTMLAttribute)*\\s*\\/?>/,\n  HTMLCloseTag: /<\\/HTMLTagName\\s*>/,\n  HTMLTagName: /[A-Za-z][A-Za-z0-9-]*/, \n  HTMLComment: /<!--(?:[^>-]|(?:[^>-](?:[^-]|-[^-])*[^-]))-->/,\n  HTMLPI: /<\\?(?:|.|(?:[^?]|\\?[^>])*)\\?>/,\n  HTMLDeclaration: /<![A-Z]+\\s[^>]*>/,\n  HTMLCDATA: /<!\\[CDATA\\[.*?\\]\\]>/,\n  HTMLAttribute: /\\s+[A-Za-z_:][A-Za-z0-9_.:-]*(?:HTMLAttValue)?/,\n  HTMLAttValue: /\\s*=\\s*(?:(?:'[^']*')|(?:\"[^\"]*\")|(?:[^\\s\"'=<>`]+))/,\n  KnownTag: /address|article|aside|base|basefont|blockquote|body|caption|center|col|colgroup|dd|details|dialog|dir|div|dl|dt|fieldset|figcaption|figure|footer|form|frame|frameset|h1|h2|h3|h4|h5|h6|head|header|hr|html|iframe|legend|li|link|main|menu|menuitem|nav|noframes|ol|optgroup|option|p|param|section|source|summary|table|tbody|td|tfoot|th|thead|title|tr|track|ul/\n}\n\n// From CommonMark.js. \nconst punctuationLeading = new RegExp(/^(?:[!\"#$%&'()*+,\\-./:;<=>?@[\\]\\\\^_`{|}~\\xA1\\xA7\\xAB\\xB6\\xB7\\xBB\\xBF\\u037E\\u0387\\u055A-\\u055F\\u0589\\u058A\\u05BE\\u05C0\\u05C3\\u05C6\\u05F3\\u05F4\\u0609\\u060A\\u060C\\u060D\\u061B\\u061E\\u061F\\u066A-\\u066D\\u06D4\\u0700-\\u070D\\u07F7-\\u07F9\\u0830-\\u083E\\u085E\\u0964\\u0965\\u0970\\u0AF0\\u0DF4\\u0E4F\\u0E5A\\u0E5B\\u0F04-\\u0F12\\u0F14\\u0F3A-\\u0F3D\\u0F85\\u0FD0-\\u0FD4\\u0FD9\\u0FDA\\u104A-\\u104F\\u10FB\\u1360-\\u1368\\u1400\\u166D\\u166E\\u169B\\u169C\\u16EB-\\u16ED\\u1735\\u1736\\u17D4-\\u17D6\\u17D8-\\u17DA\\u1800-\\u180A\\u1944\\u1945\\u1A1E\\u1A1F\\u1AA0-\\u1AA6\\u1AA8-\\u1AAD\\u1B5A-\\u1B60\\u1BFC-\\u1BFF\\u1C3B-\\u1C3F\\u1C7E\\u1C7F\\u1CC0-\\u1CC7\\u1CD3\\u2010-\\u2027\\u2030-\\u2043\\u2045-\\u2051\\u2053-\\u205E\\u207D\\u207E\\u208D\\u208E\\u2308-\\u230B\\u2329\\u232A\\u2768-\\u2775\\u27C5\\u27C6\\u27E6-\\u27EF\\u2983-\\u2998\\u29D8-\\u29DB\\u29FC\\u29FD\\u2CF9-\\u2CFC\\u2CFE\\u2CFF\\u2D70\\u2E00-\\u2E2E\\u2E30-\\u2E42\\u3001-\\u3003\\u3008-\\u3011\\u3014-\\u301F\\u3030\\u303D\\u30A0\\u30FB\\uA4FE\\uA4FF\\uA60D-\\uA60F\\uA673\\uA67E\\uA6F2-\\uA6F7\\uA874-\\uA877\\uA8CE\\uA8CF\\uA8F8-\\uA8FA\\uA8FC\\uA92E\\uA92F\\uA95F\\uA9C1-\\uA9CD\\uA9DE\\uA9DF\\uAA5C-\\uAA5F\\uAADE\\uAADF\\uAAF0\\uAAF1\\uABEB\\uFD3E\\uFD3F\\uFE10-\\uFE19\\uFE30-\\uFE52\\uFE54-\\uFE61\\uFE63\\uFE68\\uFE6A\\uFE6B\\uFF01-\\uFF03\\uFF05-\\uFF0A\\uFF0C-\\uFF0F\\uFF1A\\uFF1B\\uFF1F\\uFF20\\uFF3B-\\uFF3D\\uFF3F\\uFF5B\\uFF5D\\uFF5F-\\uFF65]|\\uD800[\\uDD00-\\uDD02\\uDF9F\\uDFD0]|\\uD801\\uDD6F|\\uD802[\\uDC57\\uDD1F\\uDD3F\\uDE50-\\uDE58\\uDE7F\\uDEF0-\\uDEF6\\uDF39-\\uDF3F\\uDF99-\\uDF9C]|\\uD804[\\uDC47-\\uDC4D\\uDCBB\\uDCBC\\uDCBE-\\uDCC1\\uDD40-\\uDD43\\uDD74\\uDD75\\uDDC5-\\uDDC9\\uDDCD\\uDDDB\\uDDDD-\\uDDDF\\uDE38-\\uDE3D\\uDEA9]|\\uD805[\\uDCC6\\uDDC1-\\uDDD7\\uDE41-\\uDE43\\uDF3C-\\uDF3E]|\\uD809[\\uDC70-\\uDC74]|\\uD81A[\\uDE6E\\uDE6F\\uDEF5\\uDF37-\\uDF3B\\uDF44]|\\uD82F\\uDC9F|\\uD836[\\uDE87-\\uDE8B])/);\n\nconst punctuationTrailing = new RegExp(/(?:[!\"#$%&'()*+,\\-./:;<=>?@[\\]\\\\^_`{|}~\\xA1\\xA7\\xAB\\xB6\\xB7\\xBB\\xBF\\u037E\\u0387\\u055A-\\u055F\\u0589\\u058A\\u05BE\\u05C0\\u05C3\\u05C6\\u05F3\\u05F4\\u0609\\u060A\\u060C\\u060D\\u061B\\u061E\\u061F\\u066A-\\u066D\\u06D4\\u0700-\\u070D\\u07F7-\\u07F9\\u0830-\\u083E\\u085E\\u0964\\u0965\\u0970\\u0AF0\\u0DF4\\u0E4F\\u0E5A\\u0E5B\\u0F04-\\u0F12\\u0F14\\u0F3A-\\u0F3D\\u0F85\\u0FD0-\\u0FD4\\u0FD9\\u0FDA\\u104A-\\u104F\\u10FB\\u1360-\\u1368\\u1400\\u166D\\u166E\\u169B\\u169C\\u16EB-\\u16ED\\u1735\\u1736\\u17D4-\\u17D6\\u17D8-\\u17DA\\u1800-\\u180A\\u1944\\u1945\\u1A1E\\u1A1F\\u1AA0-\\u1AA6\\u1AA8-\\u1AAD\\u1B5A-\\u1B60\\u1BFC-\\u1BFF\\u1C3B-\\u1C3F\\u1C7E\\u1C7F\\u1CC0-\\u1CC7\\u1CD3\\u2010-\\u2027\\u2030-\\u2043\\u2045-\\u2051\\u2053-\\u205E\\u207D\\u207E\\u208D\\u208E\\u2308-\\u230B\\u2329\\u232A\\u2768-\\u2775\\u27C5\\u27C6\\u27E6-\\u27EF\\u2983-\\u2998\\u29D8-\\u29DB\\u29FC\\u29FD\\u2CF9-\\u2CFC\\u2CFE\\u2CFF\\u2D70\\u2E00-\\u2E2E\\u2E30-\\u2E42\\u3001-\\u3003\\u3008-\\u3011\\u3014-\\u301F\\u3030\\u303D\\u30A0\\u30FB\\uA4FE\\uA4FF\\uA60D-\\uA60F\\uA673\\uA67E\\uA6F2-\\uA6F7\\uA874-\\uA877\\uA8CE\\uA8CF\\uA8F8-\\uA8FA\\uA8FC\\uA92E\\uA92F\\uA95F\\uA9C1-\\uA9CD\\uA9DE\\uA9DF\\uAA5C-\\uAA5F\\uAADE\\uAADF\\uAAF0\\uAAF1\\uABEB\\uFD3E\\uFD3F\\uFE10-\\uFE19\\uFE30-\\uFE52\\uFE54-\\uFE61\\uFE63\\uFE68\\uFE6A\\uFE6B\\uFF01-\\uFF03\\uFF05-\\uFF0A\\uFF0C-\\uFF0F\\uFF1A\\uFF1B\\uFF1F\\uFF20\\uFF3B-\\uFF3D\\uFF3F\\uFF5B\\uFF5D\\uFF5F-\\uFF65]|\\uD800[\\uDD00-\\uDD02\\uDF9F\\uDFD0]|\\uD801\\uDD6F|\\uD802[\\uDC57\\uDD1F\\uDD3F\\uDE50-\\uDE58\\uDE7F\\uDEF0-\\uDEF6\\uDF39-\\uDF3F\\uDF99-\\uDF9C]|\\uD804[\\uDC47-\\uDC4D\\uDCBB\\uDCBC\\uDCBE-\\uDCC1\\uDD40-\\uDD43\\uDD74\\uDD75\\uDDC5-\\uDDC9\\uDDCD\\uDDDB\\uDDDD-\\uDDDF\\uDE38-\\uDE3D\\uDEA9]|\\uD805[\\uDCC6\\uDDC1-\\uDDD7\\uDE41-\\uDE43\\uDF3C-\\uDF3E]|\\uD809[\\uDC70-\\uDC74]|\\uD81A[\\uDE6E\\uDE6F\\uDEF5\\uDF37-\\uDF3B\\uDF44]|\\uD82F\\uDC9F|\\uD836[\\uDE87-\\uDE8B])$/);\n\n// export const inlineTriggerChars = new RegExp(`[${replacements.TriggerChars}]`);\n\n/**\n * This is CommonMark's block grammar, but we're ignoring nested blocks here.  \n */ \nconst lineGrammar = { \n  TMH1: { \n    regexp: /^( {0,3}#\\s)(.*?)((?:\\s+#+\\s*)?)$/, \n    replacement: '<span class=\"TMMark TMMark_TMH1\">$1</span>$$2<span class=\"TMMark TMMark_TMH1\">$3</span>'\n  },\n  TMH2: { \n    regexp: /^( {0,3}##\\s)(.*?)((?:\\s+#+\\s*)?)$/, \n    replacement: '<span class=\"TMMark TMMark_TMH2\">$1</span>$$2<span class=\"TMMark TMMark_TMH2\">$3</span>'\n  },\n  TMH3: { \n    regexp: /^( {0,3}###\\s)(.*?)((?:\\s+#+\\s*)?)$/, \n    replacement: '<span class=\"TMMark TMMark_TMH3\">$1</span>$$2<span class=\"TMMark TMMark_TMH3\">$3</span>'\n  },\n  TMH4: { \n    regexp: /^( {0,3}####\\s)(.*?)((?:\\s+#+\\s*)?)$/, \n    replacement: '<span class=\"TMMark TMMark_TMH4\">$1</span>$$2<span class=\"TMMark TMMark_TMH4\">$3</span>'\n  },\n  TMH5: { \n    regexp: /^( {0,3}#####\\s)(.*?)((?:\\s+#+\\s*)?)$/, \n    replacement: '<span class=\"TMMark TMMark_TMH5\">$1</span>$$2<span class=\"TMMark TMMark_TMH5\">$3</span>'\n  },\n  TMH6: { \n    regexp: /^( {0,3}######\\s)(.*?)((?:\\s+#+\\s*)?)$/, \n    replacement: '<span class=\"TMMark TMMark_TMH6\">$1</span>$$2<span class=\"TMMark TMMark_TMH6\">$3</span>'\n  },\n  TMBlockquote: { \n    regexp: /^( {0,3}>[ ]?)(.*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMBlockquote\">$1</span>$$2'\n  },\n  TMCodeFenceBacktickOpen: { \n    regexp: /^( {0,3}(?<seq>````*)\\s*)([^`]*?)(\\s*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMCodeFenceBacktick\">$1</span><span class=\"TMInfoString\">$3</span>$4'\n  },\n  TMCodeFenceTildeOpen: { \n    regexp: /^( {0,3}(?<seq>~~~~*)\\s*)(.*?)(\\s*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMCodeFenceTilde\">$1</span><span class=\"TMInfoString\">$3</span>$4'\n  },\n  TMCodeFenceBacktickClose: { \n    regexp: /^( {0,3}(?<seq>````*))(\\s*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMCodeFenceBacktick\">$1</span>$3'\n  },\n  TMCodeFenceTildeClose: { \n    regexp: /^( {0,3}(?<seq>~~~~*))(\\s*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMCodeFenceTilde\">$1</span>$3'\n  },\n  TMBlankLine: { \n    regexp: /^([ \\t]*)$/, \n    replacement: '$0'\n  },\n  TMSetextH1Marker: { \n    regexp: /^ {0,3}=+\\s*$/, \n    replacement: '<span class=\"TMMark TMMark_TMSetextH1Marker\">$0</span>'\n  },\n  TMSetextH2Marker: { \n    regexp: /^ {0,3}-+\\s*$/, \n    replacement: '<span class=\"TMMark TMMark_TMSetextH1Marker\">$0</span>'\n  },\n  TMHR: { \n    regexp: /^( {0,3}(\\*[ \\t]*\\*[ \\t]*\\*[ \\t*]*)|(-[ \\t]*-[ \\t]*-[ \\t-]*)|(_[ \\t]*_[ \\t]*_[ \\t_]*))$/, \n    replacement: '<span class=\"TMMark TMMark_TMHR\">$0</span>'\n  },\n  TMUL: { \n    regexp: /^( {0,3}[+*-] {1,4})(.*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMUL\">$1</span>$$2'\n  },\n  TMOL: { \n    regexp: /^( {0,3}\\d{1,9}[.)] {1,4})(.*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMOL\">$1</span>$$2'\n  },\n  // TODO: This is currently preventing sublists (and any content within list items, really) from working\n  TMIndentedCode: { \n    regexp: /^( {4}|\\t)(.*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMIndentedCode\">$1</span>$2'\n  },\n  TMLinkReferenceDefinition: {\n    // TODO: Link destination can't include unbalanced parantheses, but we just ignore that here \n    regexp: /^( {0,3}\\[\\s*)([^\\s\\]](?:[^\\]]|\\\\\\])*?)(\\s*\\]:\\s*)((?:[^\\s<>]+)|(?:<(?:[^<>\\\\]|\\\\.)*>))?(\\s*)((?:\\((?:[^()\\\\]|\\\\.)*\\))|(?:\"(?:[^\"\\\\]|\\\\.)*\")|(?:'(?:[^'\\\\]|\\\\.)*'))?(\\s*)$/, \n    replacement: '<span class=\"TMMark TMMark_TMLinkReferenceDefinition\">$1</span><span class=\"TMLinkLabel TMLinkLabel_Definition\">$2</span><span class=\"TMMark TMMark_TMLinkReferenceDefinition\">$3</span><span class=\"TMLinkDestination\">$4</span>$5<span class=\"TMLinkTitle\">$6</span>$7',\n    labelPlaceholder: 2, // this defines which placeholder in the above regex is the link \"label\"\n  },\n};\n\n/**\n * HTML blocks have multiple different classes of opener and closer. This array defines all the cases\n */\nvar htmlBlockGrammar = [\n  { start: /^ {0,3}<(?:script|pre|style)(?:\\s|>|$)/i, end: /(?:<\\/script>|<\\/pre>|<\\/style>)/i, paraInterrupt: true },\n  { start: /^ {0,3}<!--/, end: /-->/, paraInterrupt: true },\n  { start: /^ {0,3}<\\?/, end: /\\?>/, paraInterrupt: true },\n  { start: /^ {0,3}<![A-Z]/, end: />/, paraInterrupt : true},\n  { start: /^ {0,3}<!\\[CDATA\\[/, end: /\\]\\]>/, paraInterrupt : true},\n  { start: /^ {0,3}(?:<|<\\/)(?:KnownTag)(?:\\s|>|\\/>|$)/i, end: false, paraInterrupt: true},\n  { start: /^ {0,3}(?:HTMLOpenTag|HTMLCloseTag)\\s*$/, end: false, paraInterrupt: false},\n];\n\n/**\n * Structure of the object:\n * Top level entries are rules, each consisting of a regular expressions (in string format) as well as a replacement.\n * In the regular expressions, replacements from the object 'replacements' will be processed before compiling into the property regexp.\n */\nvar inlineGrammar = {\n  escape : {\n    regexp: /^\\\\(ASCIIPunctuation)/,\n    replacement : '<span class=\"TMMark TMMark_TMEscape\">\\\\</span>$1'\n  },\n  code : {\n    regexp: /^(`+)((?:[^`])|(?:[^`].*?[^`]))(\\1)/,\n    replacement : '<span class=\"TMMark TMMark_TMCode\">$1</span><code class=\"TMCode\">$2</code><span class=\"TMMark TMMark_TMCode\">$3</span>' \n  },\n  autolink : {\n    regexp: /^<((?:Scheme:[^\\s<>]*)|(?:Email))>/,\n    replacement: '<span class=\"TMMark TMMark_TMAutolink\">&lt;</span><span class=\"TMAutolink\">$1</span><span class=\"TMMark TMMark_TMAutolink\">&gt;</span>'\n  },\n  html : {\n    regexp: /^((?:HTMLOpenTag)|(?:HTMLCloseTag)|(?:HTMLComment)|(?:HTMLPI)|(?:HTMLDeclaration)|(?:HTMLCDATA))/,\n    replacement: '<span class=\"TMHTML\">$1</span>',\n  },\n  linkOpen : {\n    regexp: /^\\[/,\n    replacement: ''\n  },\n  imageOpen : {\n    regexp: /^!\\[/,\n    replacement : ''\n  },\n  linkLabel : {\n    regexp: /^(\\[\\s*)([^\\]]*?)(\\s*\\])/,\n    replacement: '',\n    labelPlaceholder: 2\n  },\n  default : {\n    regexp: /^(.|(?:NotTriggerChar+))/,\n    replacement: '$1'\n  }\n};\n\n// Process replacements in regexps\nconst replacementRegexp = new RegExp(Object.keys(replacements).join('|'));\n\n// Inline\nconst inlineRules =[...Object.keys(inlineGrammar)];\nfor (let rule of inlineRules) {\n  let re = inlineGrammar[rule].regexp.source;\n  // Replace while there is something to replace. This means it also works over multiple levels (replacements containing replacements)\n  while (re.match(replacementRegexp)) {\n    re = re.replace(replacementRegexp, (string) => { return replacements[string].source; });\n  }\n  inlineGrammar[rule].regexp = new RegExp(re, inlineGrammar[rule].regexp.flags);\n}\n\n// HTML Block (only opening rule is processed currently)\nfor (let rule of htmlBlockGrammar) {\n  let re = rule.start.source;\n  // Replace while there is something to replace. This means it also works over multiple levels (replacements containing replacements)\n  while (re.match(replacementRegexp)) {\n    re = re.replace(replacementRegexp, (string) => { return replacements[string].source; });\n  }\n  rule.start = new RegExp(re, rule.start.flags);\n}\n\n/**\n * Escapes HTML special characters (<, >, and &) in the string.\n * @param {string} string The raw string to be escaped\n * @returns {string} The string, ready to be used in HTML\n */\nfunction htmlescape(string) {\n  return (string ? string : '')\n    .replace(/&/g, '&amp;')\n    .replace(/</g, '&lt;')\n    .replace(/>/g, '&gt;');\n}\n/**\n * Contains the commands that can be sent to the editor. Contains objects with a name representing the name of the command.\n * Each of the objects contains the following keys:\n * \n *   - type: Can be either inline (for inline formatting) or line (for block / line formatting).\n *   - className: Used to determine whether the command is active at a given position. \n *     For line formatting, this looks at the class of the line element. For inline elements, tries to find an enclosing element with that class.\n *   - set / unset: Contain instructions how to set and unset the command. For line type commands, both consist of a pattern and replacement that \n *     will be applied to each line (using String.replace). For inline type commands, the set object contains a pre and post string which will\n *     be inserted before and after the selection. The unset object contains a prePattern and a postPattern. Both should be regular expressions and \n *     they will be applied to the portion of the line before and after the selection (using String.replace, with an empty replacement string).\n */\nconst commands = {\n  // Replacements for unset for inline commands are '' by default\n  bold: {\n    type: 'inline', \n    className: 'TMStrong', \n    set: {pre: '**', post: '**'}, \n    unset: {prePattern: /(?:\\*\\*|__)$/, postPattern: /^(?:\\*\\*|__)/}\n  }, \n  italic: {\n    type: 'inline', \n    className: 'TMEm', \n    set: {pre: '*', post: '*'}, \n    unset: {prePattern: /(?:\\*|_)$/, postPattern: /^(?:\\*|_)/}\n  },\n  code: {\n    type: 'inline', \n    className: 'TMCode', \n    set: {pre: '`', post: '`'}, \n    unset: {prePattern: /`+$/, postPattern: /^`+/} // FIXME this doesn't ensure balanced backticks right now\n  }, \n  strikethrough: {\n    type: 'inline', \n    className: 'TMStrikethrough', \n    set: {pre: '~~', post: '~~'}, \n    unset: {prePattern:/~~$/, postPattern: /^~~/ }\n  },\n  h1: {\n    type: 'line', \n    className: 'TMH1', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '# $2'}, \n    unset: {pattern: /^( {0,3}#\\s+)(.*?)((?:\\s+#+\\s*)?)$/, replacement: '$2'}\n  },\n  h2: {\n    type: 'line', \n    className: 'TMH2', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '## $2'}, \n    unset: {pattern: /^( {0,3}##\\s+)(.*?)((?:\\s+#+\\s*)?)$/, replacement: '$2'}\n  },\n  h3: {\n    type: 'line', \n    className: 'TMH3', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '### $2'}, \n    unset: {pattern: /^( {0,3}###\\s+)(.*?)((?:\\s+#+\\s*)?)$/, replacement: '$2'}\n  },\n  h4: {\n    type: 'line', \n    className: 'TMH4', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '#### $2'}, \n    unset: {pattern: /^( {0,3}####\\s+)(.*?)((?:\\s+#+\\s*)?)$/, replacement: '$2'}\n  },\n  h5: {\n    type: 'line', \n    className: 'TMH5', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '##### $2'}, \n    unset: {pattern: /^( {0,3}#####\\s+)(.*?)((?:\\s+#+\\s*)?)$/, replacement: '$2'}\n  },\n  h6: {\n    type: 'line', \n    className: 'TMH6', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '###### $2'}, \n    unset: {pattern: /^( {0,3}######\\s+)(.*?)((?:\\s+#+\\s*)?)$/, replacement: '$2'}\n  },\n  ul: {\n    type: 'line', \n    className: 'TMUL', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '- $2'}, \n    unset: {pattern: /^( {0,3}[+*-] {1,4})(.*)$/, replacement: '$2'}\n  },\n  ol: {\n    type: 'line', \n    className: 'TMOL', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '$#. $2'}, \n    unset: {pattern: /^( {0,3}\\d{1,9}[.)] {1,4})(.*)$/, replacement: '$2'}\n  }, \n  blockquote: {\n    type: 'line', \n    className: 'TMBlockquote', \n    set: {pattern: /^( {0,3}(?:(?:#+|[0-9]{1,9}[).]|[>\\-*+])\\s+)?)(.*)$/, replacement: '> $2'}, \n    unset: {pattern: /^( {0,3}>[ ]?)(.*)$/, replacement: '$2'}\n  },\n};\n\nexport { lineGrammar, inlineGrammar, punctuationLeading, punctuationTrailing, htmlescape, htmlBlockGrammar, commands };","import {\n  inlineGrammar,\n  lineGrammar,\n  punctuationLeading,\n  punctuationTrailing,\n  htmlescape,\n  htmlBlockGrammar,\n  commands,\n} from \"./grammar\";\n\nclass Editor {\n  constructor(props = {}) {\n    this.e = null;\n    this.textarea = null;\n    this.lines = [];\n    this.lineElements = [];\n    this.lineTypes = [];\n    this.lineCaptures = [];\n    this.lineReplacements = [];\n    this.linkLabels = [];\n    this.lineDirty = [];\n    this.lastCommandState = null;\n\n    this.listeners = {\n      change: [],\n      selection: [],\n      drop: [],\n    };\n\n    let element = props.element;\n    this.textarea = props.textarea;\n\n    if (this.textarea) {\n      if (!this.textarea.tagName) {\n        this.textarea = document.getElementById(this.textarea);\n      }\n      if (!element) element = this.textarea;\n    }\n\n    if (element && !element.tagName) {\n      element = document.getElementById(props.element);\n    }\n    if (!element) {\n      element = document.getElementsByTagName(\"body\")[0];\n    }\n    if (element.tagName == \"TEXTAREA\") {\n      this.textarea = element;\n      element = this.textarea.parentNode;\n    }\n\n    if (this.textarea) {\n      this.textarea.style.display = \"none\";\n    }\n\n    this.createEditorElement(element, props);\n    this.setContent(\n      typeof props.content === \"string\"\n        ? props.content\n        : this.textarea\n        ? this.textarea.value\n        : \"# Hello TinyMDE!\\nEdit **here**\"\n    );\n  }\n\n  /**\n   * Creates the editor element inside the target element of the DOM tree\n   * @param element The target element of the DOM tree\n   * @param props options, passed from constructor's props\n   */\n  createEditorElement(element, props) {\n    if (props && props.editor !== undefined) {\n      if (props.editor.tagName) this.e = props.editor;\n      else this.e = document.getElementById(props.editor);\n    } else this.e = document.createElement(\"div\");\n\n    this.e.classList.add(\"TinyMDE\");\n    this.e.contentEditable = true;\n    // The following is important for formatting purposes, but also since otherwise the browser replaces subsequent spaces with  &nbsp; &nbsp;\n    // That breaks a lot of stuff, so we do this here and not in CSS—therefore, you don't have to remember to put this in the CSS file\n    this.e.style.whiteSpace = \"pre-wrap\";\n    // Avoid formatting (B / I / U) popping up on iOS\n    this.e.style.webkitUserModify = \"read-write-plaintext-only\";\n\n    if (props.editor === undefined) {\n      if (\n        this.textarea &&\n        this.textarea.parentNode == element &&\n        this.textarea.nextSibling\n      ) {\n        element.insertBefore(this.e, this.textarea.nextSibling);\n      } else {\n        element.appendChild(this.e);\n      }\n    }\n\n    this.e.addEventListener(\"input\", (e) => this.handleInputEvent(e));\n    this.e.addEventListener(\"compositionend\", (e) => this.handleInputEvent(e));\n    document.addEventListener(\"selectionchange\", (e) =>\n      this.handleSelectionChangeEvent(e)\n    );\n    this.e.addEventListener(\"paste\", (e) => this.handlePaste(e));\n    this.e.addEventListener(\"drop\", (e) => this.handleDrop(e));\n    this.lineElements = this.e.childNodes; // this will automatically update\n  }\n\n  /**\n   * Sets the editor content.\n   * @param {string} content The new Markdown content\n   */\n  setContent(content) {\n    // Delete any existing content\n    while (this.e.firstChild) {\n      this.e.removeChild(this.e.firstChild);\n    }\n    this.lines = content.split(/(?:\\r\\n|\\r|\\n)/);\n    this.lineDirty = [];\n    for (let lineNum = 0; lineNum < this.lines.length; lineNum++) {\n      let le = document.createElement(\"div\");\n      this.e.appendChild(le);\n      this.lineDirty.push(true);\n    }\n    this.lineTypes = new Array(this.lines.length);\n    this.updateFormatting();\n    this.fireChange();\n  }\n\n  /**\n   * Gets the editor content as a Markdown string.\n   * @returns {string} The editor content as a markdown string\n   */\n  getContent() {\n    return this.lines.join(\"\\n\");\n  }\n\n  /**\n   * This is the main method to update the formatting (from this.lines to HTML output)\n   */\n  updateFormatting() {\n    // First, parse line types. This will update this.lineTypes, this.lineReplacements, and this.lineCaptures\n    // We don't apply the formatting yet\n    this.updateLineTypes();\n    // Collect any valid link labels from link reference definitions—we need that for formatting to determine what's a valid link\n    this.updateLinkLabels();\n    // Now, apply the formatting\n    this.applyLineTypes();\n  }\n\n  /**\n   * Updates this.linkLabels: For every link reference definition (line type TMLinkReferenceDefinition), we collect the label\n   */\n  updateLinkLabels() {\n    this.linkLabels = [];\n    for (let l = 0; l < this.lines.length; l++) {\n      if (this.lineTypes[l] == \"TMLinkReferenceDefinition\") {\n        this.linkLabels.push(\n          this.lineCaptures[l][\n            lineGrammar.TMLinkReferenceDefinition.labelPlaceholder\n          ]\n        );\n      }\n    }\n  }\n\n  /**\n   * Helper function to replace placeholders from a RegExp capture. The replacement string can contain regular dollar placeholders (e.g., $1),\n   * which are interpreted like in String.replace(), but also double dollar placeholders ($$1). In the case of double dollar placeholders,\n   * Markdown inline grammar is applied on the content of the captured subgroup, i.e., $$1 processes inline Markdown grammar in the content of the\n   * first captured subgroup, and replaces `$$1` with the result.\n   *\n   * @param {string} replacement The replacement string, including placeholders.\n   * @param  capture The result of a RegExp.exec() call\n   * @returns The replacement string, with placeholders replaced from the capture result.\n   */\n  replace(replacement, capture) {\n    return replacement.replace(/(\\${1,2})([0-9])/g, (str, p1, p2) => {\n      if (p1 == \"$\") return htmlescape(capture[p2]);\n      else\n        return `<span class=\"TMInlineFormatted\">${this.processInlineStyles(\n          capture[p2]\n        )}</span>`;\n    });\n  }\n\n  /**\n   * Applies the line types (from this.lineTypes as well as the capture result in this.lineReplacements and this.lineCaptures)\n   * and processes inline formatting for all lines.\n   */\n  applyLineTypes() {\n    for (let lineNum = 0; lineNum < this.lines.length; lineNum++) {\n      if (this.lineDirty[lineNum]) {\n        let contentHTML = this.replace(\n          this.lineReplacements[lineNum],\n          this.lineCaptures[lineNum]\n        );\n        // this.lineHTML[lineNum] = (contentHTML == '' ? '<br />' : contentHTML); // Prevent empty elements which can't be selected etc.\n        this.lineElements[lineNum].className = this.lineTypes[lineNum];\n        this.lineElements[lineNum].removeAttribute(\"style\");\n        this.lineElements[lineNum].innerHTML =\n          contentHTML == \"\" ? \"<br />\" : contentHTML; // Prevent empty elements which can't be selected etc.\n      }\n      this.lineElements[lineNum].dataset.lineNum = lineNum;\n    }\n  }\n\n  /**\n   * Determines line types for all lines based on the line / block grammar. Captures the results of the respective line\n   * grammar regular expressions.\n   * Updates this.lineTypes, this.lineCaptures, and this.lineReplacements, as well as this.lineDirty.\n   */\n  updateLineTypes() {\n    let codeBlockType = false;\n    let codeBlockSeqLength = 0;\n    let htmlBlock = false;\n\n    for (let lineNum = 0; lineNum < this.lines.length; lineNum++) {\n      let lineType = \"TMPara\";\n      let lineCapture = [this.lines[lineNum]];\n      let lineReplacement = \"$$0\"; // Default replacement for paragraph: Inline format the entire line\n\n      // Check ongoing code blocks\n      // if (lineNum > 0 && (this.lineTypes[lineNum - 1] == 'TMCodeFenceBacktickOpen' || this.lineTypes[lineNum - 1] == 'TMFencedCodeBacktick')) {\n      if (codeBlockType == \"TMCodeFenceBacktickOpen\") {\n        // We're in a backtick-fenced code block, check if the current line closes it\n        let capture = lineGrammar.TMCodeFenceBacktickClose.regexp.exec(\n          this.lines[lineNum]\n        );\n        if (capture && capture.groups[\"seq\"].length >= codeBlockSeqLength) {\n          lineType = \"TMCodeFenceBacktickClose\";\n          lineReplacement = lineGrammar.TMCodeFenceBacktickClose.replacement;\n          lineCapture = capture;\n          codeBlockType = false;\n        } else {\n          lineType = \"TMFencedCodeBacktick\";\n          lineReplacement = '<span class=\"TMFencedCode\">$0<br /></span>';\n          lineCapture = [this.lines[lineNum]];\n        }\n      }\n      // if (lineNum > 0 && (this.lineTypes[lineNum - 1] == 'TMCodeFenceTildeOpen' || this.lineTypes[lineNum - 1] == 'TMFencedCodeTilde')) {\n      else if (codeBlockType == \"TMCodeFenceTildeOpen\") {\n        // We're in a tilde-fenced code block\n        let capture = lineGrammar.TMCodeFenceTildeClose.regexp.exec(\n          this.lines[lineNum]\n        );\n        if (capture && capture.groups[\"seq\"].length >= codeBlockSeqLength) {\n          lineType = \"TMCodeFenceTildeClose\";\n          lineReplacement = lineGrammar.TMCodeFenceTildeClose.replacement;\n          lineCapture = capture;\n          codeBlockType = false;\n        } else {\n          lineType = \"TMFencedCodeTilde\";\n          lineReplacement = '<span class=\"TMFencedCode\">$0<br /></span>';\n          lineCapture = [this.lines[lineNum]];\n        }\n      }\n\n      // Check HTML block types\n      if (lineType == \"TMPara\" && htmlBlock === false) {\n        for (let htmlBlockType of htmlBlockGrammar) {\n          if (this.lines[lineNum].match(htmlBlockType.start)) {\n            // Matching start condition. Check if this tag can start here (not all start conditions allow breaking a paragraph).\n            if (\n              htmlBlockType.paraInterrupt ||\n              lineNum == 0 ||\n              !(\n                this.lineTypes[lineNum - 1] == \"TMPara\" ||\n                this.lineTypes[lineNum - 1] == \"TMUL\" ||\n                this.lineTypes[lineNum - 1] == \"TMOL\" ||\n                this.lineTypes[lineNum - 1] == \"TMBlockquote\"\n              )\n            ) {\n              htmlBlock = htmlBlockType;\n              break;\n            }\n          }\n        }\n      }\n\n      if (htmlBlock !== false) {\n        lineType = \"TMHTMLBlock\";\n        lineReplacement = '<span class=\"TMHTMLContent\">$0<br /></span>'; // No formatting in TMHTMLBlock\n        lineCapture = [this.lines[lineNum]]; // This should already be set but better safe than sorry\n\n        // Check if HTML block should be closed\n        if (htmlBlock.end) {\n          // Specific end condition\n          if (this.lines[lineNum].match(htmlBlock.end)) {\n            htmlBlock = false;\n          }\n        } else {\n          // No specific end condition, ends with blank line\n          if (\n            lineNum == this.lines.length - 1 ||\n            this.lines[lineNum + 1].match(lineGrammar.TMBlankLine.regexp)\n          ) {\n            htmlBlock = false;\n          }\n        }\n      }\n\n      // Check all regexps if we haven't applied one of the code block types\n      if (lineType == \"TMPara\") {\n        for (let type in lineGrammar) {\n          if (lineGrammar[type].regexp) {\n            let capture = lineGrammar[type].regexp.exec(this.lines[lineNum]);\n            if (capture) {\n              lineType = type;\n              lineReplacement = lineGrammar[type].replacement;\n              lineCapture = capture;\n              break;\n            }\n          }\n        }\n      }\n\n      // If we've opened a code block, remember that\n      if (\n        lineType == \"TMCodeFenceBacktickOpen\" ||\n        lineType == \"TMCodeFenceTildeOpen\"\n      ) {\n        codeBlockType = lineType;\n        codeBlockSeqLength = lineCapture.groups[\"seq\"].length;\n      }\n\n      // Link reference definition and indented code can't interrupt a paragraph\n      if (\n        (lineType == \"TMIndentedCode\" ||\n          lineType == \"TMLinkReferenceDefinition\") &&\n        lineNum > 0 &&\n        (this.lineTypes[lineNum - 1] == \"TMPara\" ||\n          this.lineTypes[lineNum - 1] == \"TMUL\" ||\n          this.lineTypes[lineNum - 1] == \"TMOL\" ||\n          this.lineTypes[lineNum - 1] == \"TMBlockquote\")\n      ) {\n        // Fall back to TMPara\n        lineType = \"TMPara\";\n        lineCapture = [this.lines[lineNum]];\n        lineReplacement = \"$$0\";\n      }\n\n      // Setext H2 markers that can also be interpreted as an empty list item should be regarded as such (as per CommonMark spec)\n      if (lineType == \"TMSetextH2Marker\") {\n        let capture = lineGrammar.TMUL.regexp.exec(this.lines[lineNum]);\n        if (capture) {\n          lineType = \"TMUL\";\n          lineReplacement = lineGrammar.TMUL.replacement;\n          lineCapture = capture;\n        }\n      }\n\n      // Setext headings are only valid if preceded by a paragraph (and if so, they change the type of the previous paragraph)\n      if (lineType == \"TMSetextH1Marker\" || lineType == \"TMSetextH2Marker\") {\n        if (lineNum == 0 || this.lineTypes[lineNum - 1] != \"TMPara\") {\n          // Setext marker is invalid. However, a H2 marker might still be a valid HR, so let's check that\n          let capture = lineGrammar.TMHR.regexp.exec(this.lines[lineNum]);\n          if (capture) {\n            // Valid HR\n            lineType = \"TMHR\";\n            lineCapture = capture;\n            lineReplacement = lineGrammar.TMHR.replacement;\n          } else {\n            // Not valid HR, format as TMPara\n            lineType = \"TMPara\";\n            lineCapture = [this.lines[lineNum]];\n            lineReplacement = \"$$0\";\n          }\n        } else {\n          // Valid setext marker. Change types of preceding para lines\n          let headingLine = lineNum - 1;\n          const headingLineType =\n            lineType == \"TMSetextH1Marker\" ? \"TMSetextH1\" : \"TMSetextH2\";\n          do {\n            if (this.lineTypes[headingLineType] != headingLineType) {\n              this.lineTypes[headingLine] = headingLineType;\n              this.lineDirty[headingLineType] = true;\n            }\n            this.lineReplacements[headingLine] = \"$$0\";\n            this.lineCaptures[headingLine] = [this.lines[headingLine]];\n\n            headingLine--;\n          } while (headingLine >= 0 && this.lineTypes[headingLine] == \"TMPara\");\n        }\n      }\n      // Lastly, save the line style to be applied later\n      if (this.lineTypes[lineNum] != lineType) {\n        this.lineTypes[lineNum] = lineType;\n        this.lineDirty[lineNum] = true;\n      }\n      this.lineReplacements[lineNum] = lineReplacement;\n      this.lineCaptures[lineNum] = lineCapture;\n    }\n  }\n\n  /**\n   * Updates all line contents from the HTML, then re-applies formatting.\n   */\n  updateLineContentsAndFormatting() {\n    this.clearDirtyFlag();\n    this.updateLineContents();\n    this.updateFormatting();\n  }\n\n  /**\n   * Attempts to parse a link or image at the current position. This assumes that the opening [ or ![ has already been matched.\n   * Returns false if this is not a valid link, image. See below for more information\n   * @param {string} originalString The original string, starting at the opening marker ([ or ![)\n   * @param {boolean} isImage Whether or not this is an image (opener == ![)\n   * @returns false if not a valid link / image.\n   * Otherwise returns an object with two properties: output is the string to be included in the processed output,\n   * charCount is the number of input characters (from originalString) consumed.\n   */\n  parseLinkOrImage(originalString, isImage) {\n    // Skip the opening bracket\n    let textOffset = isImage ? 2 : 1;\n    let opener = originalString.substr(0, textOffset);\n    let type = isImage ? \"TMImage\" : \"TMLink\";\n    let currentOffset = textOffset;\n\n    let bracketLevel = 1;\n    let linkText = false;\n    let linkRef = false;\n    let linkLabel = [];\n    let linkDetails = []; // If matched, this will be an array: [whitespace + link destination delimiter, link destination, link destination delimiter, whitespace, link title delimiter, link title, link title delimiter + whitespace]. All can be empty strings.\n\n    textOuter: while (\n      currentOffset < originalString.length &&\n      linkText === false /* empty string is okay */\n    ) {\n      let string = originalString.substr(currentOffset);\n\n      // Capture any escapes and code blocks at current position, they bind more strongly than links\n      // We don't have to actually process them here, that'll be done later in case the link / image is valid, but we need to skip over them.\n      for (let rule of [\"escape\", \"code\", \"autolink\", \"html\"]) {\n        let cap = inlineGrammar[rule].regexp.exec(string);\n        if (cap) {\n          currentOffset += cap[0].length;\n          continue textOuter;\n        }\n      }\n\n      // Check for image. It's okay for an image to be included in a link or image\n      if (string.match(inlineGrammar.imageOpen.regexp)) {\n        // Opening image. It's okay if this is a matching pair of brackets\n        bracketLevel++;\n        currentOffset += 2;\n        continue textOuter;\n      }\n\n      // Check for link (not an image because that would have been captured and skipped over above)\n      if (string.match(inlineGrammar.linkOpen.regexp)) {\n        // Opening bracket. Two things to do:\n        // 1) it's okay if this part of a pair of brackets.\n        // 2) If we are currently trying to parse a link, this nested bracket musn't start a valid link (no nested links allowed)\n        bracketLevel++;\n        // if (bracketLevel >= 2) return false; // Nested unescaped brackets, this doesn't qualify as a link / image\n        if (!isImage) {\n          if (this.parseLinkOrImage(string, false)) {\n            // Valid link inside this possible link, which makes this link invalid (inner links beat outer ones)\n            return false;\n          }\n        }\n        currentOffset += 1;\n        continue textOuter;\n      }\n\n      // Check for closing bracket\n      if (string.match(/^\\]/)) {\n        bracketLevel--;\n        if (bracketLevel == 0) {\n          // Found matching bracket and haven't found anything disqualifying this as link / image.\n          linkText = originalString.substr(\n            textOffset,\n            currentOffset - textOffset\n          );\n          currentOffset++;\n          continue textOuter;\n        }\n      }\n\n      // Nothing matches, proceed to next char\n      currentOffset++;\n    }\n\n    // Did we find a link text (i.e., find a matching closing bracket?)\n    if (linkText === false) return false; // Nope\n\n    // So far, so good. We've got a valid link text. Let's see what type of link this is\n    let nextChar =\n      currentOffset < originalString.length\n        ? originalString.substr(currentOffset, 1)\n        : \"\";\n\n    // REFERENCE LINKS\n    if (nextChar == \"[\") {\n      let string = originalString.substr(currentOffset);\n      let cap = inlineGrammar.linkLabel.regexp.exec(string);\n      if (cap) {\n        // Valid link label\n        currentOffset += cap[0].length;\n        linkLabel.push(cap[1], cap[2], cap[3]);\n        if (cap[inlineGrammar.linkLabel.labelPlaceholder]) {\n          // Full reference link\n          linkRef = cap[inlineGrammar.linkLabel.labelPlaceholder];\n        } else {\n          // Collapsed reference link\n          linkRef = linkText.trim();\n        }\n      } else {\n        // Not a valid link label\n        return false;\n      }\n    } else if (nextChar != \"(\") {\n      // Shortcut ref link\n      linkRef = linkText.trim();\n\n      // INLINE LINKS\n    } else {\n      // nextChar == '('\n\n      // Potential inline link\n      currentOffset++;\n\n      let parenthesisLevel = 1;\n      inlineOuter: while (\n        currentOffset < originalString.length &&\n        parenthesisLevel > 0\n      ) {\n        let string = originalString.substr(currentOffset);\n\n        // Process whitespace\n        let cap = /^\\s+/.exec(string);\n        if (cap) {\n          switch (linkDetails.length) {\n            case 0:\n              linkDetails.push(cap[0]);\n              break; // Opening whitespace\n            case 1:\n              linkDetails.push(cap[0]);\n              break; // Open destination, but not a destination yet; desination opened with <\n            case 2: // Open destination with content in it. Whitespace only allowed if opened by angle bracket, otherwise this closes the destination\n              if (linkDetails[0].match(/</)) {\n                linkDetails[1] = linkDetails[1].concat(cap[0]);\n              } else {\n                if (parenthesisLevel != 1) return false; // Unbalanced parenthesis\n                linkDetails.push(\"\"); // Empty end delimiter for destination\n                linkDetails.push(cap[0]); // Whitespace in between destination and title\n              }\n              break;\n            case 3:\n              linkDetails.push(cap[0]);\n              break; // Whitespace between destination and title\n            case 4:\n              return false; // This should never happen (no opener for title yet, but more whitespace to capture)\n            case 5:\n              linkDetails.push(\"\"); // Whitespace at beginning of title, push empty title and continue\n            case 6:\n              linkDetails[5] = linkDetails[5].concat(cap[0]);\n              break; // Whitespace in title\n            case 7:\n              linkDetails[6] = linkDetails[6].concat(cap[0]);\n              break; // Whitespace after closing delimiter\n            default:\n              return false; // We should never get here\n          }\n          currentOffset += cap[0].length;\n          continue inlineOuter;\n        }\n\n        // Process backslash escapes\n        cap = inlineGrammar.escape.regexp.exec(string);\n        if (cap) {\n          switch (linkDetails.length) {\n            case 0:\n              linkDetails.push(\"\"); // this opens the link destination, add empty opening delimiter and proceed to next case\n            case 1:\n              linkDetails.push(cap[0]);\n              break; // This opens the link destination, append it\n            case 2:\n              linkDetails[1] = linkDetails[1].concat(cap[0]);\n              break; // Part of the link destination\n            case 3:\n              return false; // Lacking opening delimiter for link title\n            case 4:\n              return false; // Lcaking opening delimiter for link title\n            case 5:\n              linkDetails.push(\"\"); // This opens the link title\n            case 6:\n              linkDetails[5] = linkDetails[5].concat(cap[0]);\n              break; // Part of the link title\n            default:\n              return false; // After link title was closed, without closing parenthesis\n          }\n          currentOffset += cap[0].length;\n          continue inlineOuter;\n        }\n\n        // Process opening angle bracket as deilimiter of destination\n        if (linkDetails.length < 2 && string.match(/^</)) {\n          if (linkDetails.length == 0) linkDetails.push(\"\");\n          linkDetails[0] = linkDetails[0].concat(\"<\");\n          currentOffset++;\n          continue inlineOuter;\n        }\n\n        // Process closing angle bracket as delimiter of destination\n        if (\n          (linkDetails.length == 1 || linkDetails.length == 2) &&\n          string.match(/^>/)\n        ) {\n          if (linkDetails.length == 1) linkDetails.push(\"\"); // Empty link destination\n          linkDetails.push(\">\");\n          currentOffset++;\n          continue inlineOuter;\n        }\n\n        // Process  non-parenthesis delimiter for title.\n        cap = /^[\"']/.exec(string);\n        // For this to be a valid opener, we have to either have no destination, only whitespace so far,\n        // or a destination with trailing whitespace.\n        if (\n          cap &&\n          (linkDetails.length == 0 ||\n            linkDetails.length == 1 ||\n            linkDetails.length == 4)\n        ) {\n          while (linkDetails.length < 4) linkDetails.push(\"\");\n          linkDetails.push(cap[0]);\n          currentOffset++;\n          continue inlineOuter;\n        }\n\n        // For this to be a valid closer, we have to have an opener and some or no title, and this has to match the opener\n        if (\n          cap &&\n          (linkDetails.length == 5 || linkDetails.length == 6) &&\n          linkDetails[4] == cap[0]\n        ) {\n          if (linkDetails.length == 5) linkDetails.push(\"\"); // Empty link title\n          linkDetails.push(cap[0]);\n          currentOffset++;\n          continue inlineOuter;\n        }\n        // Other cases (linkDetails.length == 2, 3, 7) will be handled with the \"default\" case below.\n\n        // Process opening parenthesis\n        if (string.match(/^\\(/)) {\n          switch (linkDetails.length) {\n            case 0:\n              linkDetails.push(\"\"); // this opens the link destination, add empty opening delimiter and proceed to next case\n            case 1:\n              linkDetails.push(\"\"); // This opens the link destination\n            case 2: // Part of the link destination\n              linkDetails[1] = linkDetails[1].concat(\"(\");\n              if (!linkDetails[0].match(/<$/)) parenthesisLevel++;\n              break;\n            case 3:\n              linkDetails.push(\"\"); //  opening delimiter for link title\n            case 4:\n              linkDetails.push(\"(\");\n              break; // opening delimiter for link title\n            case 5:\n              linkDetails.push(\"\"); // opens the link title, add empty title content and proceed to next case\n            case 6: // Part of the link title. Un-escaped parenthesis only allowed in \" or ' delimited title\n              if (linkDetails[4] == \"(\") return false;\n              linkDetails[5] = linkDetails[5].concat(\"(\");\n              break;\n            default:\n              return false; // After link title was closed, without closing parenthesis\n          }\n          currentOffset++;\n          continue inlineOuter;\n        }\n\n        // Process closing parenthesis\n        if (string.match(/^\\)/)) {\n          if (linkDetails.length <= 2) {\n            // We are inside the link destination. Parentheses have to be matched if not in angle brackets\n            while (linkDetails.length < 2) linkDetails.push(\"\");\n\n            if (!linkDetails[0].match(/<$/)) parenthesisLevel--;\n\n            if (parenthesisLevel > 0) {\n              linkDetails[1] = linkDetails[1].concat(\")\");\n            }\n          } else if (linkDetails.length == 5 || linkDetails.length == 6) {\n            // We are inside the link title.\n            if (linkDetails[4] == \"(\") {\n              // This closes the link title\n              if (linkDetails.length == 5) linkDetails.push(\"\");\n              linkDetails.push(\")\");\n            } else {\n              // Just regular ol' content\n              if (linkDetails.length == 5) linkDetails.push(\")\");\n              else linkDetails[5] = linkDetails[5].concat(\")\");\n            }\n          } else {\n            parenthesisLevel--; // This should decrease it from 1 to 0...\n          }\n\n          if (parenthesisLevel == 0) {\n            // No invalid condition, let's make sure the linkDetails array is complete\n            while (linkDetails.length < 7) linkDetails.push(\"\");\n          }\n\n          currentOffset++;\n          continue inlineOuter;\n        }\n\n        // Any old character\n        cap = /^./.exec(string);\n        if (cap) {\n          switch (linkDetails.length) {\n            case 0:\n              linkDetails.push(\"\"); // this opens the link destination, add empty opening delimiter and proceed to next case\n            case 1:\n              linkDetails.push(cap[0]);\n              break; // This opens the link destination, append it\n            case 2:\n              linkDetails[1] = linkDetails[1].concat(cap[0]);\n              break; // Part of the link destination\n            case 3:\n              return false; // Lacking opening delimiter for link title\n            case 4:\n              return false; // Lcaking opening delimiter for link title\n            case 5:\n              linkDetails.push(\"\"); // This opens the link title\n            case 6:\n              linkDetails[5] = linkDetails[5].concat(cap[0]);\n              break; // Part of the link title\n            default:\n              return false; // After link title was closed, without closing parenthesis\n          }\n          currentOffset += cap[0].length;\n          continue inlineOuter;\n        }\n        throw \"Infinite loop\"; // we should never get here since the last test matches any character\n      }\n      if (parenthesisLevel > 0) return false; // Parenthes(es) not closed\n    }\n\n    if (linkRef !== false) {\n      // Ref link; check that linkRef is valid\n      let valid = false;\n      for (let label of this.linkLabels) {\n        if (label == linkRef) {\n          valid = true;\n          break;\n        }\n      }\n      let label = valid\n        ? \"TMLinkLabel TMLinkLabel_Valid\"\n        : \"TMLinkLabel TMLinkLabel_Invalid\";\n      let output = `<span class=\"TMMark TMMark_${type}\">${opener}</span><span class=\"${type} ${\n        linkLabel.length < 3 || !linkLabel[1] ? label : \"\"\n      }\">${this.processInlineStyles(\n        linkText\n      )}</span><span class=\"TMMark TMMark_${type}\">]</span>`;\n\n      if (linkLabel.length >= 3) {\n        output = output.concat(\n          `<span class=\"TMMark TMMark_${type}\">${linkLabel[0]}</span>`,\n          `<span class=\"${label}\">${linkLabel[1]}</span>`,\n          `<span class=\"TMMark TMMark_${type}\">${linkLabel[2]}</span>`\n        );\n      }\n      return {\n        output: output,\n        charCount: currentOffset,\n      };\n    } else if (linkDetails) {\n      // Inline link\n\n      // This should never happen, but better safe than sorry.\n      while (linkDetails.length < 7) {\n        linkDetails.push(\"\");\n      }\n\n      return {\n        output: `<span class=\"TMMark TMMark_${type}\">${opener}</span><span class=\"${type}\">${this.processInlineStyles(\n          linkText\n        )}</span><span class=\"TMMark TMMark_${type}\">](${\n          linkDetails[0]\n        }</span><span class=\"${type}Destination\">${\n          linkDetails[1]\n        }</span><span class=\"TMMark TMMark_${type}\">${linkDetails[2]}${\n          linkDetails[3]\n        }${linkDetails[4]}</span><span class=\"${type}Title\">${\n          linkDetails[5]\n        }</span><span class=\"TMMark TMMark_${type}\">${linkDetails[6]})</span>`,\n        charCount: currentOffset,\n      };\n    }\n\n    return false;\n  }\n\n  /**\n   * Formats a markdown string as HTML, using Markdown inline formatting.\n   * @param {string} originalString The input (markdown inline formatted) string\n   * @returns {string} The HTML formatted output\n   */\n  processInlineStyles(originalString) {\n    let processed = \"\";\n    let stack = []; // Stack is an array of objects of the format: {delimiter, delimString, count, output}\n    let offset = 0;\n    let string = originalString;\n\n    outer: while (string) {\n      // Process simple rules (non-delimiter)\n      for (let rule of [\"escape\", \"code\", \"autolink\", \"html\"]) {\n        let cap = inlineGrammar[rule].regexp.exec(string);\n        if (cap) {\n          string = string.substr(cap[0].length);\n          offset += cap[0].length;\n          processed += inlineGrammar[rule].replacement\n            // .replace(/\\$\\$([1-9])/g, (str, p1) => processInlineStyles(cap[p1])) // todo recursive calling\n            .replace(/\\$([1-9])/g, (str, p1) => htmlescape(cap[p1]));\n          continue outer;\n        }\n      }\n\n      // Check for links / images\n      let potentialLink = string.match(inlineGrammar.linkOpen.regexp);\n      let potentialImage = string.match(inlineGrammar.imageOpen.regexp);\n      if (potentialImage || potentialLink) {\n        let result = this.parseLinkOrImage(string, potentialImage);\n        if (result) {\n          processed = `${processed}${result.output}`;\n          string = string.substr(result.charCount);\n          offset += result.charCount;\n          continue outer;\n        }\n      }\n\n      // Check for em / strong delimiters\n      let cap = /(^\\*+)|(^_+)/.exec(string);\n      if (cap) {\n        let delimCount = cap[0].length;\n        const delimString = cap[0];\n        const currentDelimiter = cap[0][0]; // This should be * or _\n\n        string = string.substr(cap[0].length);\n\n        // We have a delimiter run. Let's check if it can open or close an emphasis.\n\n        const preceding = offset > 0 ? originalString.substr(0, offset) : \" \"; // beginning and end of line count as whitespace\n        const following =\n          offset + cap[0].length < originalString.length ? string : \" \";\n\n        const punctuationFollows = following.match(punctuationLeading);\n        const punctuationPrecedes = preceding.match(punctuationTrailing);\n        const whitespaceFollows = following.match(/^\\s/);\n        const whitespacePrecedes = preceding.match(/\\s$/);\n\n        // These are the rules for right-flanking and left-flanking delimiter runs as per CommonMark spec\n        let canOpen =\n          !whitespaceFollows &&\n          (!punctuationFollows ||\n            !!whitespacePrecedes ||\n            !!punctuationPrecedes);\n        let canClose =\n          !whitespacePrecedes &&\n          (!punctuationPrecedes || !!whitespaceFollows || !!punctuationFollows);\n\n        // Underscores have more detailed rules than just being part of left- or right-flanking run:\n        if (currentDelimiter == \"_\" && canOpen && canClose) {\n          canOpen = punctuationPrecedes;\n          canClose = punctuationFollows;\n        }\n\n        // If the delimiter can close, check the stack if there's something it can close\n        if (canClose) {\n          let stackPointer = stack.length - 1;\n          // See if we can find a matching opening delimiter, move down through the stack\n          while (delimCount && stackPointer >= 0) {\n            if (stack[stackPointer].delimiter == currentDelimiter) {\n              // We found a matching delimiter, let's construct the formatted string\n\n              // Firstly, if we skipped any stack levels, pop them immediately (non-matching delimiters)\n              while (stackPointer < stack.length - 1) {\n                const entry = stack.pop();\n                processed = `${entry.output}${entry.delimString.substr(\n                  0,\n                  entry.count\n                )}${processed}`;\n              }\n\n              // Then, format the string\n              if (delimCount >= 2 && stack[stackPointer].count >= 2) {\n                // Strong\n                processed = `<span class=\"TMMark\">${currentDelimiter}${currentDelimiter}</span><strong class=\"TMStrong\">${processed}</strong><span class=\"TMMark\">${currentDelimiter}${currentDelimiter}</span>`;\n                delimCount -= 2;\n                stack[stackPointer].count -= 2;\n              } else {\n                // Em\n                processed = `<span class=\"TMMark\">${currentDelimiter}</span><em class=\"TMEm\">${processed}</em><span class=\"TMMark\">${currentDelimiter}</span>`;\n                delimCount -= 1;\n                stack[stackPointer].count -= 1;\n              }\n\n              // If that stack level is empty now, pop it\n              if (stack[stackPointer].count == 0) {\n                let entry = stack.pop();\n                processed = `${entry.output}${processed}`;\n                stackPointer--;\n              }\n            } else {\n              // This stack level's delimiter type doesn't match the current delimiter type\n              // Go down one level in the stack\n              stackPointer--;\n            }\n          }\n        }\n        // If there are still delimiters left, and the delimiter run can open, push it on the stack\n        if (delimCount && canOpen) {\n          stack.push({\n            delimiter: currentDelimiter,\n            delimString: delimString,\n            count: delimCount,\n            output: processed,\n          });\n          processed = \"\"; // Current formatted output has been pushed on the stack and will be prepended when the stack gets popped\n          delimCount = 0;\n        }\n\n        // Any delimiters that are left (closing unmatched) are appended to the output.\n        if (delimCount) {\n          processed = `${processed}${delimString.substr(0, delimCount)}`;\n        }\n\n        offset += cap[0].length;\n        continue outer;\n      }\n\n      // Check for strikethrough delimiter\n      cap = /^~~/.exec(string);\n      if (cap) {\n        let consumed = false;\n        let stackPointer = stack.length - 1;\n        // See if we can find a matching opening delimiter, move down through the stack\n        while (!consumed && stackPointer >= 0) {\n          if (stack[stackPointer].delimiter == \"~\") {\n            // We found a matching delimiter, let's construct the formatted string\n\n            // Firstly, if we skipped any stack levels, pop them immediately (non-matching delimiters)\n            while (stackPointer < stack.length - 1) {\n              const entry = stack.pop();\n              processed = `${entry.output}${entry.delimString.substr(\n                0,\n                entry.count\n              )}${processed}`;\n            }\n\n            // Then, format the string\n            processed = `<span class=\"TMMark\">~~</span><del class=\"TMStrikethrough\">${processed}</del><span class=\"TMMark\">~~</span>`;\n            let entry = stack.pop();\n            processed = `${entry.output}${processed}`;\n            consumed = true;\n          } else {\n            // This stack level's delimiter type doesn't match the current delimiter type\n            // Go down one level in the stack\n            stackPointer--;\n          }\n        }\n\n        // If there are still delimiters left, and the delimiter run can open, push it on the stack\n        if (!consumed) {\n          stack.push({\n            delimiter: \"~\",\n            delimString: \"~~\",\n            count: 2,\n            output: processed,\n          });\n          processed = \"\"; // Current formatted output has been pushed on the stack and will be prepended when the stack gets popped\n        }\n\n        offset += cap[0].length;\n        string = string.substr(cap[0].length);\n        continue outer;\n      }\n\n      // Process 'default' rule\n      cap = inlineGrammar.default.regexp.exec(string);\n      if (cap) {\n        string = string.substr(cap[0].length);\n        offset += cap[0].length;\n        processed += inlineGrammar.default.replacement.replace(\n          /\\$([1-9])/g,\n          (str, p1) => htmlescape(cap[p1])\n        );\n        continue outer;\n      }\n      throw \"Infinite loop!\";\n    }\n\n    // Empty the stack, any opening delimiters are unused\n    while (stack.length) {\n      const entry = stack.pop();\n      processed = `${entry.output}${entry.delimString.substr(\n        0,\n        entry.count\n      )}${processed}`;\n    }\n\n    return processed;\n  }\n\n  /**\n   * Clears the line dirty flag (resets it to an array of false)\n   */\n  clearDirtyFlag() {\n    this.lineDirty = new Array(this.lines.length);\n    for (let i = 0; i < this.lineDirty.length; i++) {\n      this.lineDirty[i] = false;\n    }\n  }\n\n  /**\n   * Updates the class properties (lines, lineElements) from the DOM.\n   * @returns true if contents changed\n   */\n  updateLineContents() {\n    // this.lineDirty = [];\n    // Check if we have changed anything about the number of lines (inserted or deleted a paragraph)\n    // < 0 means line(s) removed; > 0 means line(s) added\n    let lineDelta = this.e.childElementCount - this.lines.length;\n    if (lineDelta) {\n      // yup. Let's try how much we can salvage (find out which lines from beginning and end were unchanged)\n      // Find lines from the beginning that haven't changed...\n      let firstChangedLine = 0;\n      while (\n        firstChangedLine <= this.lines.length &&\n        firstChangedLine <= this.lineElements.length &&\n        this.lineElements[firstChangedLine] && // Check that the line element hasn't been deleted\n        this.lines[firstChangedLine] ==\n          this.lineElements[firstChangedLine].textContent &&\n        this.lineTypes[firstChangedLine] ==\n          this.lineElements[firstChangedLine].className\n      ) {\n        firstChangedLine++;\n      }\n\n      // End also from the end\n      let lastChangedLine = -1;\n      while (\n        -lastChangedLine < this.lines.length &&\n        -lastChangedLine < this.lineElements.length &&\n        this.lines[this.lines.length + lastChangedLine] ==\n          this.lineElements[this.lineElements.length + lastChangedLine]\n            .textContent &&\n        this.lineTypes[this.lines.length + lastChangedLine] ==\n          this.lineElements[this.lineElements.length + lastChangedLine]\n            .className\n      ) {\n        lastChangedLine--;\n      }\n\n      let linesToDelete =\n        this.lines.length + lastChangedLine + 1 - firstChangedLine;\n      if (linesToDelete < -lineDelta) linesToDelete = -lineDelta;\n      if (linesToDelete < 0) linesToDelete = 0;\n\n      let linesToAdd = [];\n      for (let l = 0; l < linesToDelete + lineDelta; l++) {\n        linesToAdd.push(this.lineElements[firstChangedLine + l].textContent);\n      }\n      this.spliceLines(firstChangedLine, linesToDelete, linesToAdd, false);\n    } else {\n      // No lines added or removed\n      for (let line = 0; line < this.lineElements.length; line++) {\n        let e = this.lineElements[line];\n        let ct = e.textContent;\n        if (this.lines[line] !== ct || this.lineTypes[line] !== e.className) {\n          // Line changed, update it\n          this.lines[line] = ct;\n          this.lineTypes[line] = e.className;\n          this.lineDirty[line] = true;\n        }\n      }\n    }\n  }\n\n  /**\n   * Processes a new paragraph.\n   * @param sel The current selection\n   */\n  processNewParagraph(sel) {\n    if (!sel) return;\n\n    // Update lines from content\n    this.updateLineContents();\n\n    let continuableType = false;\n    // Let's see if we need to continue a list\n\n    let checkLine = sel.col > 0 ? sel.row : sel.row - 1;\n    switch (this.lineTypes[checkLine]) {\n      case \"TMUL\":\n        continuableType = \"TMUL\";\n        break;\n      case \"TMOL\":\n        continuableType = \"TMOL\";\n        break;\n      case \"TMIndentedCode\":\n        continuableType = \"TMIndentedCode\";\n        break;\n    }\n\n    let lines = this.lines[sel.row]\n      .replace(/\\n\\n$/, \"\\n\")\n      .split(/(?:\\r\\n|\\n|\\r)/);\n    if (lines.length > 1) {\n      // New line\n      this.spliceLines(sel.row, 1, lines, true);\n      sel.row++;\n      sel.col = 0;\n    }\n\n    if (continuableType) {\n      // Check if the previous line was non-empty\n      let capture = lineGrammar[continuableType].regexp.exec(\n        this.lines[sel.row - 1]\n      );\n      if (capture) {\n        // Convention: capture[1] is the line type marker, capture[2] is the content\n        if (capture[2]) {\n          // Previous line has content, continue the continuable type\n\n          // Hack for OL: increment number\n          if (continuableType == \"TMOL\") {\n            capture[1] = capture[1].replace(/\\d{1,9}/, (result) => {\n              return parseInt(result[0]) + 1;\n            });\n          }\n          this.lines[sel.row] = `${capture[1]}${this.lines[sel.row]}`;\n          this.lineDirty[sel.row] = true;\n          sel.col = capture[1].length;\n        } else {\n          // Previous line has no content, remove the continuable type from the previous row\n          this.lines[sel.row - 1] = \"\";\n          this.lineDirty[sel.row - 1] = true;\n        }\n      }\n    }\n    this.updateFormatting();\n  }\n\n  // /**\n  //  * Processes a \"delete\" input action.\n  //  * @param {object} focus The selection\n  //  * @param {boolean} forward If true, performs a forward delete, otherwise performs a backward delete\n  //  */\n  // processDelete(focus, forward) {\n  //   if (!focus) return;\n  //   let anchor = this.getSelection(true);\n  //   // Do we have a non-empty selection?\n  //   if (focus.col != anchor.col || focus.row != anchor.row) {\n  //     // non-empty. direction doesn't matter.\n  //     this.paste('', anchor, focus);\n  //   } else {\n  //     if (forward) {\n  //       if (focus.col < this.lines[focus.row].length) this.paste('', {row: focus.row, col: focus.col + 1}, focus);\n  //       else if (focus.col < this.lines.length) this.paste('', {row: focus.row + 1, col: 0}, focus);\n  //       // Otherwise, we're at the very end and can't delete forward\n  //     } else {\n  //       if (focus.col > 0) this.paste('', {row: focus.row, col: focus.col - 1}, focus);\n  //       else if (focus.row > 0) this.paste('', {row: focus.row - 1, col: this.lines[focus.row - 1].length - 1}, focus);\n  //       // Otherwise, we're at the very beginning and can't delete backwards\n  //     }\n  //   }\n\n  // }\n\n  /**\n   * Gets the current position of the selection counted by row and column of the editor Markdown content (as opposed to the position in the DOM).\n   *\n   * @param {boolean} getAnchor if set to true, gets the selection anchor (start point of the selection), otherwise gets the focus (end point).\n   * @return {object} An object representing the selection, with properties col and row.\n   */\n  getSelection(getAnchor = false) {\n    const selection = window.getSelection();\n    let startNode = getAnchor ? selection.anchorNode : selection.focusNode;\n    if (!startNode) return null;\n    let offset = getAnchor ? selection.anchorOffset : selection.focusOffset;\n    if (startNode == this.e) {\n      if (offset < this.lines.length)\n        return {\n          row: offset,\n          col: 0,\n        };\n      return {\n        row: offset - 1,\n        col: this.lines[offset - 1].length,\n      };\n    }\n\n    let col = this.computeColumn(startNode, offset);\n    if (col === null) return null; // We are outside of the editor\n\n    // Find the row node\n    let node = startNode;\n    while (node.parentElement != this.e) {\n      node = node.parentElement;\n    }\n\n    let row = 0;\n    // Check if we can read a line number from the data-line-num attribute.\n    // The last condition is a security measure since inserting a new paragraph copies the previous rows' line number\n    if (\n      node.dataset &&\n      node.dataset.lineNum &&\n      (!node.previousSibling ||\n        node.previousSibling.dataset.lineNum != node.dataset.lineNum)\n    ) {\n      row = parseInt(node.dataset.lineNum);\n    } else {\n      while (node.previousSibling) {\n        row++;\n        node = node.previousSibling;\n      }\n    }\n    return { row: row, col: col, node: startNode };\n  }\n\n  /**\n   * Computes a column within an editor line from a node and offset within that node.\n   * @param {Node} startNode The node\n   * @param {int} offset THe selection\n   * @returns {int} the column, or null if the node is not inside the editor\n   */\n  computeColumn(startNode, offset) {\n    let node = startNode;\n    let col;\n    // First, make sure we're actually in the editor.\n    while (node && node.parentNode != this.e) {\n      node = node.parentNode;\n    }\n    if (node == null) return null;\n\n    // There are two ways that offset can be defined:\n    // - Either, the node is a text node, in which case it is the offset within the text\n    // - Or, the node is an element with child notes, in which case the offset refers to the\n    //   child node after which the selection is located\n    if (startNode.nodeType === Node.TEXT_NODE || offset === 0) {\n      // In the case that the node is non-text node but the offset is 0,\n      // The selection is at the beginning of that element so we\n      // can simply use the same approach as if it were at the beginning\n      // of a text node.\n      col = offset;\n      node = startNode;\n    } else if (offset > 0) {\n      node = startNode.childNodes[offset - 1];\n      col = node.textContent.length;\n    }\n    while (node.parentNode != this.e) {\n      if (node.previousSibling) {\n        node = node.previousSibling;\n        col += node.textContent.length;\n      } else {\n        node = node.parentNode;\n      }\n    }\n    return col;\n  }\n\n  /**\n   * Computes DOM node and offset within that node from a position expressed as row and column.\n   * @param {int} row Row (line number)\n   * @param {int} col Column\n   * @returns An object with two properties: node and offset. offset may be null;\n   */\n  computeNodeAndOffset(row, col, bindRight = false) {\n    if (row >= this.lineElements.length) {\n      // Selection past the end of text, set selection to end of text\n      row = this.lineElements.length - 1;\n      col = this.lines[row].length;\n    }\n    if (col > this.lines[row].length) {\n      col = this.lines[row].length;\n    }\n    const parentNode = this.lineElements[row];\n    let node = parentNode.firstChild;\n\n    let childrenComplete = false;\n    // default return value\n    let rv = {\n      node: parentNode.firstChild ? parentNode.firstChild : parentNode,\n      offset: 0,\n    };\n\n    while (node != parentNode) {\n      if (!childrenComplete && node.nodeType === Node.TEXT_NODE) {\n        if (node.nodeValue.length >= col) {\n          if (bindRight && node.nodeValue.length == col) {\n            // Selection is at the end of this text node, but we are binding right (prefer an offset of 0 in the next text node)\n            // Remember return value in case we don't find another text node\n            rv = { node: node, offset: col };\n            col = 0;\n          } else {\n            return { node: node, offset: col };\n          }\n        } else {\n          col -= node.nodeValue.length;\n        }\n      }\n      if (!childrenComplete && node.firstChild) {\n        node = node.firstChild;\n      } else if (node.nextSibling) {\n        childrenComplete = false;\n        node = node.nextSibling;\n      } else {\n        childrenComplete = true;\n        node = node.parentNode;\n      }\n    }\n\n    // Either, the position was invalid and we just return the default return value\n    // Or we are binding right and the selection is at the end of the line\n    return rv;\n  }\n\n  /**\n   * Sets the selection based on rows and columns within the editor Markdown content.\n   * @param {object} focus Object representing the selection, needs to have properties row and col.\n   * @param anchor Anchor of the selection. If not given, assumes the current anchor.\n   */\n  setSelection(focus, anchor = null) {\n    if (!focus) return;\n\n    let range = document.createRange();\n\n    let { node: focusNode, offset: focusOffset } = this.computeNodeAndOffset(\n      focus.row,\n      focus.col,\n      anchor && anchor.row == focus.row && anchor.col > focus.col\n    ); // Bind selection right if anchor is in the same row and behind the focus\n    let anchorNode = null,\n      anchorOffset = null;\n    if (anchor && (anchor.row != focus.row || anchor.col != focus.col)) {\n      let { node, offset } = this.computeNodeAndOffset(\n        anchor.row,\n        anchor.col,\n        focus.row == anchor.row && focus.col > anchor.col\n      );\n      anchorNode = node;\n      anchorOffset = offset;\n    }\n\n    if (anchorNode) range.setStart(anchorNode, anchorOffset);\n    else range.setStart(focusNode, focusOffset);\n    range.setEnd(focusNode, focusOffset);\n\n    let windowSelection = window.getSelection();\n    windowSelection.removeAllRanges();\n    windowSelection.addRange(range);\n  }\n\n  /**\n   * Event handler for input events\n   */\n  handleInputEvent(event) {\n    // For composition input, we are only updating the text after we have received\n    // a compositionend event, so we return upon insertCompositionText.\n    // Otherwise, the DOM changes break the text input.\n    if (event.inputType == \"insertCompositionText\") return;\n\n    let focus = this.getSelection();\n\n    if (\n      (event.inputType == \"insertParagraph\" ||\n        event.inputType == \"insertLineBreak\") &&\n      focus\n    ) {\n      this.clearDirtyFlag();\n      this.processNewParagraph(focus);\n    } else {\n      if (!this.e.firstChild) {\n        this.e.innerHTML = '<div class=\"TMBlankLine\"><br></div>';\n      } else {\n        this.fixNodeHierarchy();\n      }\n      this.updateLineContentsAndFormatting();\n    }\n    if (focus) {\n      this.setSelection(focus);\n    }\n\n    this.fireChange();\n  }\n\n  /**\n   * Fixes the node hierarchy – makes sure that each line is in a div, and there are no nested divs\n   */\n  fixNodeHierarchy() {\n    const originalChildren = Array.from(this.e.childNodes);\n\n    const replaceChild = (child, ...newChildren) => {\n      const parent = child.parentElement;\n      const nextSibling = child.nextSibling;\n      parent.removeChild(child);\n      newChildren.forEach((newChild) =>\n        nextSibling\n          ? parent.insertBefore(newChild, nextSibling)\n          : parent.appendChild(newChild)\n      );\n    };\n\n    originalChildren.forEach((child) => {\n      if (child.nodeType !== Node.ELEMENT_NODE || child.tagName !== \"DIV\") {\n        // Found a child node that's either not an element or not a div. Wrap it in a div.\n        const divWrapper = document.createElement(\"div\");\n        replaceChild(child, divWrapper);\n        divWrapper.appendChild(child);\n      } else if (child.childNodes.length == 0) {\n        // Empty div child node, include at least a <br />\n        child.appendChild(document.createElement(\"br\"));\n      } else {\n        const grandChildren = Array.from(child.childNodes);\n        if (\n          grandChildren.some(\n            (grandChild) =>\n              grandChild.nodeType === Node.ELEMENT_NODE &&\n              grandChild.tagName === \"DIV\"\n          )\n        ) {\n          return replaceChild(child, grandChildren);\n        }\n      }\n    });\n  }\n\n  /**\n   * Event handler for the \"drop\" event\n   */\n  handleDrop(event) {\n    event.preventDefault();\n\n    this.fireDrop(event.dataTransfer);\n  }\n\n  /**\n   * Event handler for \"selectionchange\" events.\n   */\n  handleSelectionChangeEvent() {\n    this.fireSelection();\n  }\n\n  /**\n   * Convenience function to \"splice\" new lines into the arrays this.lines, this.lineDirty, this.lineTypes, and the DOM elements\n   * underneath the editor element.\n   * This method is essentially Array.splice, only that the third parameter takes an un-spread array (and the forth parameter)\n   * determines whether the DOM should also be adjusted.\n   *\n   * @param {int} startLine Position at which to start changing the array of lines\n   * @param {int} linesToDelete Number of lines to delete\n   * @param {array} linesToInsert Array of strings representing the lines to be inserted\n   * @param {boolean} adjustLineElements If true, then <div> elements are also inserted in the DOM at the respective position\n   */\n  spliceLines(\n    startLine,\n    linesToDelete = 0,\n    linesToInsert = [],\n    adjustLineElements = true\n  ) {\n    if (adjustLineElements) {\n      for (let i = 0; i < linesToDelete; i++) {\n        this.e.removeChild(this.e.childNodes[startLine]);\n      }\n    }\n\n    let insertedBlank = [];\n    let insertedDirty = [];\n\n    for (let i = 0; i < linesToInsert.length; i++) {\n      insertedBlank.push(\"\");\n      insertedDirty.push(true);\n      if (adjustLineElements) {\n        if (this.e.childNodes[startLine])\n          this.e.insertBefore(\n            document.createElement(\"div\"),\n            this.e.childNodes[startLine]\n          );\n        else this.e.appendChild(document.createElement(\"div\"));\n      }\n    }\n\n    this.lines.splice(startLine, linesToDelete, ...linesToInsert);\n    this.lineTypes.splice(startLine, linesToDelete, ...insertedBlank);\n    this.lineDirty.splice(startLine, linesToDelete, ...insertedDirty);\n  }\n\n  /**\n   * Event handler for the \"paste\" event\n   */\n  handlePaste(event) {\n    event.preventDefault();\n\n    // get text representation of clipboard\n    let text = (event.originalEvent || event).clipboardData.getData(\n      \"text/plain\"\n    );\n\n    // insert text manually\n    this.paste(text);\n  }\n\n  /**\n   * Pastes the text at the current selection (or at the end, if no current selection)\n   * @param {string} text\n   */\n  paste(text, anchor = null, focus = null) {\n    if (!anchor) anchor = this.getSelection(true);\n    if (!focus) focus = this.getSelection(false);\n    let beginning, end;\n    if (!focus) {\n      focus = {\n        row: this.lines.length - 1,\n        col: this.lines[this.lines.length - 1].length,\n      }; // Insert at end\n    }\n    if (!anchor) {\n      anchor = focus;\n    }\n    if (\n      anchor.row < focus.row ||\n      (anchor.row == focus.row && anchor.col <= focus.col)\n    ) {\n      beginning = anchor;\n      end = focus;\n    } else {\n      beginning = focus;\n      end = anchor;\n    }\n    let insertedLines = text.split(/(?:\\r\\n|\\r|\\n)/);\n    let lineBefore = this.lines[beginning.row].substr(0, beginning.col);\n    let lineEnd = this.lines[end.row].substr(end.col);\n    insertedLines[0] = lineBefore.concat(insertedLines[0]);\n    let endColPos = insertedLines[insertedLines.length - 1].length;\n    insertedLines[insertedLines.length - 1] =\n      insertedLines[insertedLines.length - 1].concat(lineEnd);\n    this.spliceLines(beginning.row, 1 + end.row - beginning.row, insertedLines);\n    focus.row = beginning.row + insertedLines.length - 1;\n    focus.col = endColPos;\n    this.updateFormatting();\n    this.setSelection(focus);\n    this.fireChange();\n  }\n\n  /**\n   * Computes the (lowest in the DOM tree) common ancestor of two DOM nodes.\n   * @param {Node} node1 the first node\n   * @param {Node} node2 the second node\n   * @returns {Node} The commen ancestor node, or null if there is no common ancestor\n   */\n  computeCommonAncestor(node1, node2) {\n    if (!node1 || !node2) return null;\n    if (node1 == node2) return node1;\n    const ancestry = (node) => {\n      let ancestry = [];\n      while (node) {\n        ancestry.unshift(node);\n        node = node.parentNode;\n      }\n      return ancestry;\n    };\n\n    const ancestry1 = ancestry(node1);\n    const ancestry2 = ancestry(node2);\n\n    if (ancestry1[0] != ancestry2[0]) return null;\n    let i;\n    for (i = 0; ancestry1[i] == ancestry2[i]; i++);\n    return ancestry1[i - 1];\n  }\n\n  /**\n   * Finds the (lowest in the DOM tree) enclosing DOM node with a given class.\n   * @param {object} focus The focus selection object\n   * @param {object} anchor The anchor selection object\n   * @param {string} className The class name to find\n   * @returns {Node} The enclosing DOM node with the respective class (inside the editor), if there is one; null otherwise.\n   */\n  computeEnclosingMarkupNode(focus, anchor, className) {\n    let node = null;\n    if (!focus) return null;\n    if (!anchor) {\n      node = focus.node;\n    } else {\n      if (focus.row != anchor.row) return null;\n      node = this.computeCommonAncestor(focus.node, anchor.node);\n    }\n    if (!node) return null;\n    while (node != this.e) {\n      if (node.className && node.className.includes(className)) return node;\n      node = node.parentNode;\n    }\n    // Ascended all the way to the editor element\n    return null;\n  }\n\n  /**\n   * Returns the state (true / false) of all commands.\n   * @param focus Focus of the selection. If not given, assumes the current focus.\n   * @param anchor Anchor of the selection. If not given, assumes the current anchor.\n   */\n  getCommandState(focus = null, anchor = null) {\n    let commandState = {};\n    if (!focus) focus = this.getSelection(false);\n    if (!anchor) anchor = this.getSelection(true);\n    if (!focus) {\n      for (let cmd in commands) {\n        commandState[cmd] = null;\n      }\n      return commandState;\n    }\n    if (!anchor) anchor = focus;\n\n    let start, end;\n    if (\n      anchor.row < focus.row ||\n      (anchor.row == focus.row && anchor.col < focus.col)\n    ) {\n      start = anchor;\n      end = focus;\n    } else {\n      start = focus;\n      end = anchor;\n    }\n    if (end.row > start.row && end.col == 0) {\n      end.row--;\n      end.col = this.lines[end.row].length; // Selection to beginning of next line is said to end at the beginning of the last line\n    }\n\n    for (let cmd in commands) {\n      if (commands[cmd].type == \"inline\") {\n        if (\n          !focus ||\n          focus.row != anchor.row ||\n          !this.isInlineFormattingAllowed(focus, anchor)\n        ) {\n          commandState[cmd] = null;\n        } else {\n          // The command state is true if there is a respective enclosing markup node (e.g., the selection is enclosed in a <b>..</b>) ...\n          commandState[cmd] =\n            !!this.computeEnclosingMarkupNode(\n              focus,\n              anchor,\n              commands[cmd].className\n            ) ||\n            // ... or if it's an empty string preceded by and followed by formatting markers, e.g. **|** where | is the cursor\n            (focus.col == anchor.col &&\n              !!this.lines[focus.row]\n                .substr(0, focus.col)\n                .match(commands[cmd].unset.prePattern) &&\n              !!this.lines[focus.row]\n                .substr(focus.col)\n                .match(commands[cmd].unset.postPattern));\n        }\n      }\n      if (commands[cmd].type == \"line\") {\n        if (!focus) {\n          commandState[cmd] = null;\n        } else {\n          let state = this.lineTypes[start.row] == commands[cmd].className;\n\n          for (let line = start.row; line <= end.row; line++) {\n            if ((this.lineTypes[line] == commands[cmd].className) != state) {\n              state = null;\n              break;\n            }\n          }\n          commandState[cmd] = state;\n        }\n      }\n    }\n    return commandState;\n  }\n\n  /**\n   * Sets a command state\n   * @param {string} command\n   * @param {boolean} state\n   */\n  setCommandState(command, state) {\n    if (commands[command].type == \"inline\") {\n      let anchor = this.getSelection(true);\n      let focus = this.getSelection(false);\n      if (!anchor) anchor = focus;\n      if (!anchor) return;\n      if (anchor.row != focus.row) return;\n      if (!this.isInlineFormattingAllowed(focus, anchor)) return;\n      let markupNode = this.computeEnclosingMarkupNode(\n        focus,\n        anchor,\n        commands[command].className\n      );\n      this.clearDirtyFlag();\n\n      // First case: There's an enclosing markup node, remove the markers around that markup node\n      if (markupNode) {\n        this.lineDirty[focus.row] = true;\n        const startCol = this.computeColumn(markupNode, 0);\n        const len = markupNode.textContent.length;\n        const left = this.lines[focus.row]\n          .substr(0, startCol)\n          .replace(commands[command].unset.prePattern, \"\");\n        const mid = this.lines[focus.row].substr(startCol, len);\n        const right = this.lines[focus.row]\n          .substr(startCol + len)\n          .replace(commands[command].unset.postPattern, \"\");\n        this.lines[focus.row] = left.concat(mid, right);\n        anchor.col = left.length;\n        focus.col = anchor.col + len;\n        this.updateFormatting();\n        this.setSelection(focus, anchor);\n        this.fireChange();\n\n        // Second case: Empty selection with surrounding formatting markers, remove those\n      } else if (\n        focus.col == anchor.col &&\n        !!this.lines[focus.row]\n          .substr(0, focus.col)\n          .match(commands[command].unset.prePattern) &&\n        !!this.lines[focus.row]\n          .substr(focus.col)\n          .match(commands[command].unset.postPattern)\n      ) {\n        this.lineDirty[focus.row] = true;\n        const left = this.lines[focus.row]\n          .substr(0, focus.col)\n          .replace(commands[command].unset.prePattern, \"\");\n        const right = this.lines[focus.row]\n          .substr(focus.col)\n          .replace(commands[command].unset.postPattern, \"\");\n        this.lines[focus.row] = left.concat(right);\n        focus.col = anchor.col = left.length;\n        this.updateFormatting();\n        this.setSelection(focus, anchor);\n        this.fireChange();\n\n        // Not currently formatted, insert formatting markers\n      } else {\n        // Trim any spaces from the selection\n        let { startCol, endCol } =\n          focus.col < anchor.col\n            ? { startCol: focus.col, endCol: anchor.col }\n            : { startCol: anchor.col, endCol: focus.col };\n\n        let match = this.lines[focus.row]\n          .substr(startCol, endCol - startCol)\n          .match(/^(?<leading>\\s*).*\\S(?<trailing>\\s*)$/);\n        if (match) {\n          startCol += match.groups.leading.length;\n          endCol -= match.groups.trailing.length;\n        }\n\n        focus.col = startCol;\n        anchor.col = endCol;\n\n        // Just insert markup before and after and hope for the best.\n        this.wrapSelection(\n          commands[command].set.pre,\n          commands[command].set.post,\n          focus,\n          anchor\n        );\n        this.fireChange();\n        // TODO clean this up so that markup remains properly nested\n      }\n    } else if (commands[command].type == \"line\") {\n      let anchor = this.getSelection(true);\n      let focus = this.getSelection(false);\n      if (!anchor) anchor = focus;\n      if (!focus) return;\n      this.clearDirtyFlag();\n      let start = anchor.row > focus.row ? focus : anchor;\n      let end = anchor.row > focus.row ? anchor : focus;\n      if (end.row > start.row && end.col == 0) {\n        end.row--;\n      }\n\n      for (let line = start.row; line <= end.row; line++) {\n        if (state && this.lineTypes[line] != commands[command].className) {\n          this.lines[line] = this.lines[line].replace(\n            commands[command].set.pattern,\n            commands[command].set.replacement.replace(\n              \"$#\",\n              line - start.row + 1\n            )\n          );\n          this.lineDirty[line] = true;\n        }\n        if (!state && this.lineTypes[line] == commands[command].className) {\n          this.lines[line] = this.lines[line].replace(\n            commands[command].unset.pattern,\n            commands[command].unset.replacement\n          );\n          this.lineDirty[line] = true;\n        }\n      }\n      this.updateFormatting();\n      this.setSelection(\n        { row: end.row, col: this.lines[end.row].length },\n        { row: start.row, col: 0 }\n      );\n      this.fireChange();\n    }\n  }\n\n  /**\n   * Returns whether or not inline formatting is allowed at the current focus\n   * @param {object} focus The current focus\n   */\n  isInlineFormattingAllowed() {\n    // TODO Remove parameters from all calls\n    const sel = window.getSelection();\n    if (!sel || !sel.focusNode || !sel.anchorNode) return false;\n\n    // Check if we can find a common ancestor with the class `TMInlineFormatted`\n\n    // Special case: Empty selection right before `TMInlineFormatted`\n    if (\n      sel.isCollapsed &&\n      sel.focusNode.nodeType == 3 &&\n      sel.focusOffset == sel.focusNode.nodeValue.length\n    ) {\n      let node;\n      for (\n        node = sel.focusNode;\n        node && node.nextSibling == null;\n        node = node.parentNode\n      );\n      if (\n        node &&\n        node.nextSibling.className &&\n        node.nextSibling.className.includes(\"TMInlineFormatted\")\n      )\n        return true;\n    }\n\n    // Look for a common ancestor\n    let ancestor = this.computeCommonAncestor(sel.focusNode, sel.anchorNode);\n    if (!ancestor) return false;\n\n    // Check if there's an ancestor of class 'TMInlineFormatted' or 'TMBlankLine'\n    while (ancestor && ancestor != this.e) {\n      if (\n        ancestor.className &&\n        typeof ancestor.className.includes == \"function\" &&\n        (ancestor.className.includes(\"TMInlineFormatted\") ||\n          ancestor.className.includes(\"TMBlankLine\"))\n      )\n        return true;\n      ancestor = ancestor.parentNode;\n    }\n\n    return false;\n  }\n\n  /**\n   * Wraps the current selection in the strings pre and post. If the selection is not on one line, returns.\n   * @param {string} pre The string to insert before the selection.\n   * @param {string} post The string to insert after the selection.\n   * @param {object} focus The current selection focus. If null, selection will be computed.\n   * @param {object} anchor The current selection focus. If null, selection will be computed.\n   */\n  wrapSelection(pre, post, focus = null, anchor = null) {\n    if (!focus) focus = this.getSelection(false);\n    if (!anchor) anchor = this.getSelection(true);\n    if (!focus || !anchor || focus.row != anchor.row) return;\n    this.lineDirty[focus.row] = true;\n\n    const startCol = focus.col < anchor.col ? focus.col : anchor.col;\n    const endCol = focus.col < anchor.col ? anchor.col : focus.col;\n    const left = this.lines[focus.row].substr(0, startCol).concat(pre);\n    const mid =\n      endCol == startCol\n        ? \"\"\n        : this.lines[focus.row].substr(startCol, endCol - startCol);\n    const right = post.concat(this.lines[focus.row].substr(endCol));\n    this.lines[focus.row] = left.concat(mid, right);\n    anchor.col = left.length;\n    focus.col = anchor.col + mid.length;\n\n    this.updateFormatting();\n    this.setSelection(focus, anchor);\n  }\n\n  /**\n   * Toggles the command state for a command (true <-> false)\n   * @param {string} command The editor command\n   */\n  toggleCommandState(command) {\n    if (!this.lastCommandState) this.lastCommandState = this.getCommandState();\n    this.setCommandState(command, !this.lastCommandState[command]);\n  }\n\n  /**\n   * Fires a change event. Updates the linked textarea and notifies any event listeners.\n   */\n  fireChange() {\n    if (!this.textarea && !this.listeners.change.length) return;\n    const content = this.getContent();\n    if (this.textarea) this.textarea.value = content;\n    for (let listener of this.listeners.change) {\n      listener({\n        content: content,\n        linesDirty: this.linesDirty,\n      });\n    }\n  }\n\n  /**\n   * Fires a \"selection changed\" event.\n   */\n  fireSelection() {\n    if (this.listeners.selection && this.listeners.selection.length) {\n      let focus = this.getSelection(false);\n      let anchor = this.getSelection(true);\n      let commandState = this.getCommandState(focus, anchor);\n      if (this.lastCommandState) {\n        Object.assign(this.lastCommandState, commandState);\n      } else {\n        this.lastCommandState = Object.assign({}, commandState);\n      }\n      for (let listener of this.listeners.selection) {\n        listener({\n          focus: focus,\n          anchor: anchor,\n          commandState: this.lastCommandState,\n        });\n      }\n    }\n  }\n\n  /**\n   * Fires a drop event.\n   */\n  fireDrop(dataTransfer) {\n    for (let listener of this.listeners.drop) {\n      listener({\n        dataTransfer,\n      });\n    }\n  }\n\n  /**\n   * Adds an event listener.\n   * @param {string} type The type of event to listen to. Can be 'change', 'selection' or 'drop'.\n   * @param {*} listener Function of the type (event) => {} to be called when the event occurs.\n   */\n  addEventListener(type, listener) {\n    if (type.match(/^(?:change|input)$/i)) {\n      this.listeners.change.push(listener);\n    }\n    if (type.match(/^(?:selection|selectionchange)$/i)) {\n      this.listeners.selection.push(listener);\n    }\n    if (type.match(/^(?:drop)$/i)) {\n      this.listeners.drop.push(listener);\n    }\n  }\n}\n\nexport default Editor;\n"],"names":["svg","blockquote","bold","clear_formatting","code","h1","h2","hr","image","italic","link","ol","strikethrough","ul","isMacLike","test","navigator","platform","DefaultCommands","name","action","innerHTML","title","hotkey","editor","isInlineFormattingAllowed","wrapSelection","enabled","focus","anchor","paste","CommandBar","constructor","props","e","commands","buttons","state","hotkeys","element","tagName","document","getElementById","body","createCommandBarElement","addEventListener","handleKeydown","setEditor","parentElement","createElement","className","command","el","appendChild","commandName","Object","assign","keys","split","modifiers","modifierexplanation","i","length","push","match","key","toLowerCase","concat","join","handleClick","event","preventDefault","setCommandState","handleSelection","commandState","undefined","outer","modifier","check","it","Math","module","globalThis","window","self","global","this","Function","exec","error","fails","require","defineProperty","get","bind","hasOwnProperty","NATIVE_BIND","FunctionPrototype","prototype","call","uncurryThisWithBind","fn","apply","arguments","documentAll","all","IS_HTMLDDA","$documentAll","argument","isNullOrUndefined","$TypeError","TypeError","requireObjectCoercible","$Object","uncurryThis","toObject","hasOwn","DESCRIPTORS","getDescriptor","getOwnPropertyDescriptor","EXISTS","PROPER","something","CONFIGURABLE","configurable","value","writable","defineGlobalProperty","SHARED","store","isCallable","functionToString","toString","inspectSource","WeakMap","String","isObject","a","$String","aFunction","namespace","method","isPrototypeOf","userAgent","process","Deno","versions","version","v8","V8_VERSION","getOwnPropertySymbols","symbol","Symbol","sham","NATIVE_SYMBOL","iterator","getBuiltIn","USE_SYMBOL_AS_UID","$Symbol","tryToString","aCallable","V","P","func","input","pref","val","valueOf","mode","copyright","license","source","id","postfix","random","shared","uid","WellKnownSymbolsStore","createWellKnownSymbol","withoutSetter","isSymbol","getMethod","ordinaryToPrimitive","wellKnownSymbol","TO_PRIMITIVE","exoticToPrim","result","toPrimitive","IE8_DOM_DEFINE","V8_PROTOTYPE_DEFINE_BUG","anObject","toPropertyKey","$defineProperty","$getOwnPropertyDescriptor","ENUMERABLE","WRITABLE","exports","O","Attributes","current","enumerable","bitmap","definePropertyModule","createPropertyDescriptor","object","f","NATIVE_WEAK_MAP","createNonEnumerableProperty","sharedKey","OBJECT_ALREADY_INITIALIZED","set","has","enforce","getterFor","TYPE","type","metadata","facade","STATE","CONFIGURABLE_FUNCTION_NAME","InternalStateModule","enforceInternalState","getInternalState","stringSlice","slice","replace","CONFIGURABLE_LENGTH","TEMPLATE","makeBuiltIn","options","getter","setter","arity","target","descriptor","that","hasIndices","ignoreCase","multiline","dotAll","unicode","unicodeSets","sticky","defineBuiltInAccessor","regExpFlags","RegExp","RegExpPrototype","FORCED","INDICES_SUPPORT","calls","expected","addGetter","chr","pairs","replacements","ASCIIPunctuation","NotTriggerChar","Scheme","Email","HTMLOpenTag","HTMLCloseTag","HTMLTagName","HTMLComment","HTMLPI","HTMLDeclaration","HTMLCDATA","HTMLAttribute","HTMLAttValue","KnownTag","punctuationLeading","punctuationTrailing","lineGrammar","TMH1","regexp","replacement","TMH2","TMH3","TMH4","TMH5","TMH6","TMBlockquote","TMCodeFenceBacktickOpen","TMCodeFenceTildeOpen","TMCodeFenceBacktickClose","TMCodeFenceTildeClose","TMBlankLine","TMSetextH1Marker","TMSetextH2Marker","TMHR","TMUL","TMOL","TMIndentedCode","TMLinkReferenceDefinition","labelPlaceholder","htmlBlockGrammar","start","end","paraInterrupt","inlineGrammar","escape","autolink","html","linkOpen","imageOpen","linkLabel","default","replacementRegexp","inlineRules","rule","re","string","flags","htmlescape","pre","post","unset","prePattern","postPattern","pattern","h3","h4","h5","h6","Editor","textarea","lines","lineElements","lineTypes","lineCaptures","lineReplacements","linkLabels","lineDirty","lastCommandState","listeners","change","selection","drop","getElementsByTagName","parentNode","style","display","createEditorElement","setContent","content","classList","add","contentEditable","whiteSpace","webkitUserModify","nextSibling","insertBefore","handleInputEvent","handleSelectionChangeEvent","handlePaste","handleDrop","childNodes","firstChild","removeChild","lineNum","le","Array","updateFormatting","fireChange","getContent","updateLineTypes","updateLinkLabels","applyLineTypes","l","capture","str","p1","p2","processInlineStyles","contentHTML","removeAttribute","dataset","codeBlockType","codeBlockSeqLength","htmlBlock","lineType","lineCapture","lineReplacement","groups","htmlBlockType","headingLine","headingLineType","updateLineContentsAndFormatting","clearDirtyFlag","updateLineContents","parseLinkOrImage","originalString","isImage","textOffset","opener","substr","currentOffset","bracketLevel","linkText","linkRef","linkDetails","textOuter","cap","nextChar","trim","parenthesisLevel","inlineOuter","valid","label","output","charCount","processed","stack","offset","potentialLink","potentialImage","delimCount","delimString","currentDelimiter","preceding","following","punctuationFollows","punctuationPrecedes","whitespaceFollows","whitespacePrecedes","canOpen","canClose","stackPointer","delimiter","entry","pop","count","consumed","lineDelta","childElementCount","firstChangedLine","textContent","lastChangedLine","linesToDelete","linesToAdd","spliceLines","line","ct","processNewParagraph","sel","continuableType","checkLine","col","row","parseInt","getSelection","getAnchor","startNode","anchorNode","focusNode","anchorOffset","focusOffset","computeColumn","node","previousSibling","nodeType","Node","TEXT_NODE","computeNodeAndOffset","bindRight","childrenComplete","rv","nodeValue","setSelection","range","createRange","setStart","setEnd","windowSelection","removeAllRanges","addRange","inputType","fixNodeHierarchy","originalChildren","from","replaceChild","child","parent","_len","newChildren","_key","forEach","newChild","ELEMENT_NODE","divWrapper","grandChildren","some","grandChild","fireDrop","dataTransfer","fireSelection","startLine","linesToInsert","adjustLineElements","insertedBlank","insertedDirty","splice","text","originalEvent","clipboardData","getData","beginning","insertedLines","lineBefore","lineEnd","endColPos","computeCommonAncestor","node1","node2","ancestry","unshift","ancestry1","ancestry2","computeEnclosingMarkupNode","includes","getCommandState","cmd","markupNode","startCol","len","left","mid","right","endCol","leading","trailing","isCollapsed","ancestor","toggleCommandState","listener","linesDirty"],"mappings":";;;;;;EAAA,MAAMA,GAAG,GAAG;EACVC,EAAAA,UAAU,EAAG,CAA0R,yRAAA,CAAA;EACvSC,EAAAA,IAAI,EAAG,CAAyY,wYAAA,CAAA;EAChZC,EAAAA,gBAAgB,EAAG,CAA0X,yXAAA,CAAA;EAC7YC,EAAAA,IAAI,EAAG,CAA8U,6UAAA,CAAA;EACrVC,EAAAA,EAAE,EAAG,CAA+J,8JAAA,CAAA;EACpKC,EAAAA,EAAE,EAAG,CAAiK,gKAAA,CAAA;EACtKC,EAAAA,EAAE,EAAG,CAAgI,+HAAA,CAAA;EACrIC,EAAAA,KAAK,EAAG,CAA8H,6HAAA,CAAA;EACtIC,EAAAA,MAAM,EAAG,CAAgH,+GAAA,CAAA;EACzHC,EAAAA,IAAI,EAAG,CAA0uC,yuCAAA,CAAA;EACjvCC,EAAAA,EAAE,EAAG,CAAotB,mtBAAA,CAAA;EACztBC,EAAAA,aAAa,EAAG,CAA2X,0XAAA,CAAA;EAC3YC,EAAAA,EAAE,EAAG,CAAA,2QAAA,CAAA;EACP,CAAC;;ECZD,MAAMC,SAAS,GAAG,yBAAyB,CAACC,IAAI,CAAC,OAAOC,SAAS,KAAK,WAAW,GAAGA,SAAS,CAACC,QAAQ,GAAG,EAAE,CAAC,CAAA;EAE5G,MAAMC,eAAe,GAAG;EACtB,EAAA,MAAM,EAAE;EACNC,IAAAA,IAAI,EAAE,MAAM;EACZC,IAAAA,MAAM,EAAE,MAAM;MACdC,SAAS,EAAErB,GAAG,CAACE,IAAI;EACnBoB,IAAAA,KAAK,EAAE,MAAM;EACbC,IAAAA,MAAM,EAAE,OAAA;KACT;EACD,EAAA,QAAQ,EAAE;EACRJ,IAAAA,IAAI,EAAE,QAAQ;EACdC,IAAAA,MAAM,EAAE,QAAQ;MAChBC,SAAS,EAAErB,GAAG,CAACS,MAAM;EACrBa,IAAAA,KAAK,EAAE,QAAQ;EACfC,IAAAA,MAAM,EAAE,OAAA;KACT;EACD,EAAA,eAAe,EAAE;EACfJ,IAAAA,IAAI,EAAE,eAAe;EACrBC,IAAAA,MAAM,EAAE,eAAe;MACvBC,SAAS,EAAErB,GAAG,CAACY,aAAa;EAC5BU,IAAAA,KAAK,EAAE,eAAe;EACtBC,IAAAA,MAAM,EAAE,cAAA;KACT;EACD,EAAA,MAAM,EAAE;EACNJ,IAAAA,IAAI,EAAE,MAAM;EACZC,IAAAA,MAAM,EAAE,MAAM;MACdC,SAAS,EAAErB,GAAG,CAACI,IAAI;EACnBkB,IAAAA,KAAK,EAAE,gBAAA;KACR;EACD,EAAA,IAAI,EAAE;EACJH,IAAAA,IAAI,EAAE,IAAI;EACVC,IAAAA,MAAM,EAAE,IAAI;MACZC,SAAS,EAAErB,GAAG,CAACK,EAAE;EACjBiB,IAAAA,KAAK,EAAE,iBAAiB;EACxBC,IAAAA,MAAM,EAAE,aAAA;KACT;EACD,EAAA,IAAI,EAAE;EACJJ,IAAAA,IAAI,EAAE,IAAI;EACVC,IAAAA,MAAM,EAAE,IAAI;MACZC,SAAS,EAAErB,GAAG,CAACM,EAAE;EACjBgB,IAAAA,KAAK,EAAE,iBAAiB;EACxBC,IAAAA,MAAM,EAAE,aAAA;KACT;EACD,EAAA,IAAI,EAAE;EACJJ,IAAAA,IAAI,EAAE,IAAI;EACVC,IAAAA,MAAM,EAAE,IAAI;MACZC,SAAS,EAAErB,GAAG,CAACa,EAAE;EACjBS,IAAAA,KAAK,EAAE,eAAA;KACR;EACD,EAAA,IAAI,EAAE;EACJH,IAAAA,IAAI,EAAE,IAAI;EACVC,IAAAA,MAAM,EAAE,IAAI;MACZC,SAAS,EAAErB,GAAG,CAACW,EAAE;EACjBW,IAAAA,KAAK,EAAE,eAAA;KACR;EACD,EAAA,YAAY,EAAE;EACZH,IAAAA,IAAI,EAAE,YAAY;EAClBC,IAAAA,MAAM,EAAE,YAAY;MACpBC,SAAS,EAAErB,GAAG,CAACC,UAAU;EACzBqB,IAAAA,KAAK,EAAE,OAAO;EACdC,IAAAA,MAAM,EAAE,cAAA;KACT;EACD,EAAA,YAAY,EAAE;EACZJ,IAAAA,IAAI,EAAE,YAAY;MAClBC,MAAM,EAAGI,MAAM,IAAK;EAAC,MAAA,IAAIA,MAAM,CAACC,yBAAyB,EAAE,EAAED,MAAM,CAACE,aAAa,CAAC,GAAG,EAAE,KAAK,CAAC,CAAA;OAAC;EAC9FC,IAAAA,OAAO,EAAEA,CAACH,MAAM,EAAEI,KAAK,EAAEC,MAAM,KAAKL,MAAM,CAACC,yBAAyB,CAACG,KAAK,EAAEC,MAAM,CAAC,GAAG,KAAK,GAAG,IAAI;MAClGR,SAAS,EAAErB,GAAG,CAACU,IAAI;EACnBY,IAAAA,KAAK,EAAE,aAAa;EACpBC,IAAAA,MAAM,EAAE,OAAA;KACT;EACD,EAAA,aAAa,EAAE;EACbJ,IAAAA,IAAI,EAAE,aAAa;MACnBC,MAAM,EAAGI,MAAM,IAAK;EAAC,MAAA,IAAIA,MAAM,CAACC,yBAAyB,EAAE,EAAED,MAAM,CAACE,aAAa,CAAC,IAAI,EAAE,KAAK,CAAC,CAAA;OAAC;EAC/FC,IAAAA,OAAO,EAAEA,CAACH,MAAM,EAAEI,KAAK,EAAEC,MAAM,KAAKL,MAAM,CAACC,yBAAyB,CAACG,KAAK,EAAEC,MAAM,CAAC,GAAG,KAAK,GAAG,IAAI;MAClGR,SAAS,EAAErB,GAAG,CAACQ,KAAK;EACpBc,IAAAA,KAAK,EAAE,cAAc;EACrBC,IAAAA,MAAM,EAAE,cAAA;KACT;EACD,EAAA,IAAI,EAAE;EACJJ,IAAAA,IAAI,EAAE,IAAI;MACVC,MAAM,EAAGI,MAAM,IAAKA,MAAM,CAACM,KAAK,CAAC,SAAS,CAAC;MAC3CH,OAAO,EAAEA,MAAM,KAAK;MACpBN,SAAS,EAAErB,GAAG,CAACO,EAAE;EACjBe,IAAAA,KAAK,EAAE,wBAAwB;EAC/BC,IAAAA,MAAM,EAAE,cAAA;EACV,GAAA;EACF,CAAC,CAAA;EAGD,MAAMQ,UAAU,CAAC;IACfC,WAAWA,CAACC,KAAK,EAAE;MACjB,IAAI,CAACC,CAAC,GAAG,IAAI,CAAA;MACb,IAAI,CAACV,MAAM,GAAG,IAAI,CAAA;MAClB,IAAI,CAACW,QAAQ,GAAG,EAAE,CAAA;EAClB,IAAA,IAAI,CAACC,OAAO,GAAG,EAAE,CAAA;EACjB,IAAA,IAAI,CAACC,KAAK,GAAG,EAAE,CAAA;MACf,IAAI,CAACC,OAAO,GAAG,EAAE,CAAA;EAEjB,IAAA,IAAIC,OAAO,GAAGN,KAAK,CAACM,OAAO,CAAA;EAC3B,IAAA,IAAIA,OAAO,IAAI,CAACA,OAAO,CAACC,OAAO,EAAE;QAC/BD,OAAO,GAAGE,QAAQ,CAACC,cAAc,CAACT,KAAK,CAACM,OAAO,CAAC,CAAA;EAClD,KAAA;MACA,IAAI,CAACA,OAAO,EAAE;QACZA,OAAO,GAAGE,QAAQ,CAACE,IAAI,CAAA;EACzB,KAAA;MACA,IAAI,CAACC,uBAAuB,CAACL,OAAO,EAAEN,KAAK,CAACE,QAAQ,IAAI,CAAC,MAAM,EAAE,QAAQ,EAAE,eAAe,EAAE,GAAG,EAAE,MAAM,EAAE,GAAG,EAAE,IAAI,EAAE,IAAI,EAAE,GAAG,EAAE,IAAI,EAAE,IAAI,EAAE,GAAG,EAAE,YAAY,EAAE,IAAI,EAAE,GAAG,EAAE,YAAY,EAAE,aAAa,CAAC,CAAC,CAAA;EACtMM,IAAAA,QAAQ,CAACI,gBAAgB,CAAC,SAAS,EAAGX,CAAC,IAAK,IAAI,CAACY,aAAa,CAACZ,CAAC,CAAC,CAAC,CAAA;MAClE,IAAID,KAAK,CAACT,MAAM,EAAE,IAAI,CAACuB,SAAS,CAACd,KAAK,CAACT,MAAM,CAAC,CAAA;EAChD,GAAA;EAEAoB,EAAAA,uBAAuBA,CAACI,aAAa,EAAEb,QAAQ,EAAE;MAC/C,IAAI,CAACD,CAAC,GAAGO,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAA;EACtC,IAAA,IAAI,CAACf,CAAC,CAACgB,SAAS,GAAG,cAAc,CAAA;EAEjC,IAAA,KAAK,IAAIC,OAAO,IAAIhB,QAAQ,EAAE;QAC5B,IAAIgB,OAAO,IAAI,GAAG,EAAE;EAClB,QAAA,IAAIC,EAAE,GAAGX,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAA;UACtCG,EAAE,CAACF,SAAS,GAAG,kBAAkB,CAAA;EACjC,QAAA,IAAI,CAAChB,CAAC,CAACmB,WAAW,CAACD,EAAE,CAAC,CAAA;EACxB,OAAC,MAAM;EACL,QAAA,IAAIE,WAAW,CAAA;EACf,QAAA,IAAI,OAAOH,OAAO,IAAI,QAAQ,EAAE;EAC9B;;EAEA,UAAA,IAAIjC,eAAe,CAACiC,OAAO,CAAC,EAAE;EAC5BG,YAAAA,WAAW,GAAGH,OAAO,CAAA;cACrB,IAAI,CAAChB,QAAQ,CAACmB,WAAW,CAAC,GAAGpC,eAAe,CAACoC,WAAW,CAAC,CAAA;EAG3D,WAAC,MAAM;EACL,YAAA,SAAA;EACF,WAAA;WAED,MAAM,IAAI,OAAOH,OAAO,IAAI,QAAQ,IAAIA,OAAO,CAAChC,IAAI,EAAE;YACrDmC,WAAW,GAAGH,OAAO,CAAChC,IAAI,CAAA;EAC1B,UAAA,IAAI,CAACgB,QAAQ,CAACmB,WAAW,CAAC,GAAG,EAAE,CAAA;YAC/B,IAAIpC,eAAe,CAACoC,WAAW,CAAC,EAAEC,MAAM,CAACC,MAAM,CAAC,IAAI,CAACrB,QAAQ,CAACmB,WAAW,CAAC,EAAEpC,eAAe,CAACoC,WAAW,CAAC,CAAC,CAAA;YACzGC,MAAM,CAACC,MAAM,CAAC,IAAI,CAACrB,QAAQ,CAACmB,WAAW,CAAC,EAAEH,OAAO,CAAC,CAAA;EAGpD,SAAC,MAAM;EACL,UAAA,SAAA;EACF,SAAA;UAEA,IAAI7B,KAAK,GAAG,IAAI,CAACa,QAAQ,CAACmB,WAAW,CAAC,CAAChC,KAAK,IAAIgC,WAAW,CAAA;UAE3D,IAAI,IAAI,CAACnB,QAAQ,CAACmB,WAAW,CAAC,CAAC/B,MAAM,EAAE;EACrC,UAAA,MAAMkC,IAAI,GAAG,IAAI,CAACtB,QAAQ,CAACmB,WAAW,CAAC,CAAC/B,MAAM,CAACmC,KAAK,CAAC,GAAG,CAAC,CAAA;EACzD;YACA,IAAIC,SAAS,GAAG,EAAE,CAAA;YAClB,IAAIC,mBAAmB,GAAG,EAAE,CAAA;EAC5B,UAAA,KAAK,IAAIC,CAAC,GAAG,CAAC,EAAEA,CAAC,GAAGJ,IAAI,CAACK,MAAM,GAAG,CAAC,EAAED,CAAC,EAAE,EAAE;cACxC,QAAQJ,IAAI,CAACI,CAAC,CAAC;EACb,cAAA,KAAK,MAAM;EAAEF,gBAAAA,SAAS,CAACI,IAAI,CAAC,SAAS,CAAC,CAAA;EAAEH,gBAAAA,mBAAmB,CAACG,IAAI,CAAC,MAAM,CAAC,CAAA;EAAE,gBAAA,MAAA;EAC1E,cAAA,KAAK,KAAK;EAAEJ,gBAAAA,SAAS,CAACI,IAAI,CAAC,SAAS,CAAC,CAAA;EAAEH,gBAAAA,mBAAmB,CAACG,IAAI,CAAC,GAAG,CAAC,CAAA;EAAE,gBAAA,MAAA;EACtE,cAAA,KAAK,KAAK;EAAEJ,gBAAAA,SAAS,CAACI,IAAI,CAAC,QAAQ,CAAC,CAAA;EAAEH,gBAAAA,mBAAmB,CAACG,IAAI,CAAC,KAAK,CAAC,CAAA;EAAE,gBAAA,MAAA;EACvE,cAAA,KAAK,QAAQ;EAAEJ,gBAAAA,SAAS,CAACI,IAAI,CAAC,QAAQ,CAAC,CAAA;EAAEH,gBAAAA,mBAAmB,CAACG,IAAI,CAAC,GAAG,CAAC,CAAA;EAAE,gBAAA,MAAA;EACxE,cAAA,KAAK,KAAK;EAAEJ,gBAAAA,SAAS,CAACI,IAAI,CAAC,SAAS,CAAC,CAAA;EAAEH,gBAAAA,mBAAmB,CAACG,IAAI,CAAC,OAAO,CAAC,CAAA;EAAE,gBAAA,MAAA;EAE1E,cAAA,KAAK,OAAO;EAAGJ,gBAAAA,SAAS,CAACI,IAAI,CAAC,UAAU,CAAC,CAAA;EAAEH,gBAAAA,mBAAmB,CAACG,IAAI,CAAC,GAAG,CAAC,CAAA;EAAE,gBAAA,MAAA;EAE1E,cAAA,KAAK,KAAK;EAAE;EACV,gBAAA,IAAIjD,SAAS,EAAE;EAAC6C,kBAAAA,SAAS,CAACI,IAAI,CAAC,SAAS,CAAC,CAAA;EAAEH,kBAAAA,mBAAmB,CAACG,IAAI,CAAC,GAAG,CAAC,CAAA;EAAC,iBAAC,MACrE;EAACJ,kBAAAA,SAAS,CAACI,IAAI,CAAC,SAAS,CAAC,CAAA;EAAEH,kBAAAA,mBAAmB,CAACG,IAAI,CAAC,MAAM,CAAC,CAAA;EAAC,iBAAA;EAClE,gBAAA,MAAA;EACF,cAAA,KAAK,MAAM;EACTJ,gBAAAA,SAAS,CAACI,IAAI,CAAC,QAAQ,CAAC,CAAA;EACxB,gBAAA,IAAIjD,SAAS,EAAE8C,mBAAmB,CAACG,IAAI,CAAC,GAAG,CAAC,CAAC,KACxCH,mBAAmB,CAACG,IAAI,CAAC,KAAK,CAAC,CAAA;EACpC,gBAAA,MAAA;EAAO;EACX,aAAA;EACF,WAAA;;YACAH,mBAAmB,CAACG,IAAI,CAACN,IAAI,CAACA,IAAI,CAACK,MAAM,GAAG,CAAC,CAAC,CAAC,CAAA;EAC/C,UAAA,IAAIvC,MAAM,GAAG;EAEXoC,YAAAA,SAAS,EAAEA,SAAS;EACpBR,YAAAA,OAAO,EAAEG,WAAAA;aACV,CAAA;EACD;EACA,UAAA,IAAIG,IAAI,CAACA,IAAI,CAACK,MAAM,GAAG,CAAC,CAAC,CAACE,KAAK,CAAC,SAAS,CAAC,EAAE;EAC1CzC,YAAAA,MAAM,CAACnB,IAAI,GAAI,CAAA,KAAA,EAAOqD,IAAI,CAACA,IAAI,CAACK,MAAM,GAAG,CAAC,CAAE,CAAC,CAAA,CAAA;EAC/C,WAAC,MAAM;EACLvC,YAAAA,MAAM,CAAC0C,GAAG,GAAGR,IAAI,CAACA,IAAI,CAACK,MAAM,GAAG,CAAC,CAAC,CAACI,WAAW,EAAE,CAAA;EAClD,WAAA;EACA,UAAA,IAAI,CAAC5B,OAAO,CAACyB,IAAI,CAACxC,MAAM,CAAC,CAAA;EACzBD,UAAAA,KAAK,GAAGA,KAAK,CAAC6C,MAAM,CAAE,CAAA,EAAA,EAAIP,mBAAmB,CAACQ,IAAI,CAAC,GAAG,CAAE,GAAE,CAAC,CAAA;EAC7D,SAAA;UAEA,IAAI,CAAChC,OAAO,CAACkB,WAAW,CAAC,GAAGb,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAA;UACzD,IAAI,CAACb,OAAO,CAACkB,WAAW,CAAC,CAACJ,SAAS,GAAG,0CAA0C,CAAA;UAChF,IAAI,CAACd,OAAO,CAACkB,WAAW,CAAC,CAAChC,KAAK,GAAGA,KAAK,CAAA;EACvC,QAAA,IAAI,CAACc,OAAO,CAACkB,WAAW,CAAC,CAACjC,SAAS,GAAG,IAAI,CAACc,QAAQ,CAACmB,WAAW,CAAC,CAACjC,SAAS,CAAA;UAE1E,IAAI,CAACe,OAAO,CAACkB,WAAW,CAAC,CAACT,gBAAgB,CAAC,WAAW,EAAGX,CAAC,IAAK,IAAI,CAACmC,WAAW,CAACf,WAAW,EAAEpB,CAAC,CAAC,CAAC,CAAA;UAChG,IAAI,CAACA,CAAC,CAACmB,WAAW,CAAC,IAAI,CAACjB,OAAO,CAACkB,WAAW,CAAC,CAAC,CAAA;EAC/C,OAAA;EACF,KAAA;EACAN,IAAAA,aAAa,CAACK,WAAW,CAAC,IAAI,CAACnB,CAAC,CAAC,CAAA;EACnC,GAAA;EAEAmC,EAAAA,WAAWA,CAACf,WAAW,EAAEgB,KAAK,EAAE;EAC9B,IAAA,IAAI,CAAC,IAAI,CAAC9C,MAAM,EAAE,OAAA;MAClB8C,KAAK,CAACC,cAAc,EAAE,CAAA;MACtB,IAAI,OAAO,IAAI,CAACpC,QAAQ,CAACmB,WAAW,CAAC,CAAClC,MAAM,IAAI,QAAQ,EAAE;EACxD,MAAA,IAAI,IAAI,CAACiB,KAAK,CAACiB,WAAW,CAAC,KAAK,KAAK,EAAE,IAAI,CAAC9B,MAAM,CAACgD,eAAe,CAAClB,WAAW,EAAE,IAAI,CAAC,CAAC,KACjF,IAAI,CAAC9B,MAAM,CAACgD,eAAe,CAAClB,WAAW,EAAE,KAAK,CAAC,CAAA;EACtD,KAAC,MAAM,IAAI,OAAO,IAAI,CAACnB,QAAQ,CAACmB,WAAW,CAAC,CAAClC,MAAM,IAAI,UAAU,EAAE;QACjE,IAAI,CAACe,QAAQ,CAACmB,WAAW,CAAC,CAAClC,MAAM,CAAC,IAAI,CAACI,MAAM,CAAC,CAAA;EAChD,KAAA;EACF,GAAA;IAEAuB,SAASA,CAACvB,MAAM,EAAE;MAChB,IAAI,CAACA,MAAM,GAAGA,MAAM,CAAA;EACpBA,IAAAA,MAAM,CAACqB,gBAAgB,CAAC,WAAW,EAAGX,CAAC,IAAK,IAAI,CAACuC,eAAe,CAACvC,CAAC,CAAC,CAAC,CAAA;EACtE,GAAA;IAEAuC,eAAeA,CAACH,KAAK,EAAE;MACrB,IAAIA,KAAK,CAACI,YAAY,EAAE;EACtB,MAAA,KAAK,IAAIvB,OAAO,IAAI,IAAI,CAAChB,QAAQ,EAAE;UACjC,IAAImC,KAAK,CAACI,YAAY,CAACvB,OAAO,CAAC,KAAKwB,SAAS,EAAE;YAC7C,IAAI,IAAI,CAACxC,QAAQ,CAACgB,OAAO,CAAC,CAACxB,OAAO,EAAE,IAAI,CAACU,KAAK,CAACc,OAAO,CAAC,GAAG,IAAI,CAAChB,QAAQ,CAACgB,OAAO,CAAC,CAACxB,OAAO,CAAC,IAAI,CAACH,MAAM,EAAE8C,KAAK,CAAC1C,KAAK,EAAE0C,KAAK,CAACzC,MAAM,CAAC,CAAC,KAC5H,IAAI,CAACQ,KAAK,CAACc,OAAO,CAAC,GAAGmB,KAAK,CAAC1C,KAAK,GAAG,KAAK,GAAG,IAAI,CAAA;EACvD,SAAC,MAAM;YACL,IAAI,CAACS,KAAK,CAACc,OAAO,CAAC,GAAGmB,KAAK,CAACI,YAAY,CAACvB,OAAO,CAAC,CAAA;EACnD,SAAA;UAEA,IAAI,IAAI,CAACd,KAAK,CAACc,OAAO,CAAC,KAAK,IAAI,EAAE;YAChC,IAAI,CAACf,OAAO,CAACe,OAAO,CAAC,CAACD,SAAS,GAAG,wCAAwC,CAAA;WAC3E,MAAM,IAAI,IAAI,CAACb,KAAK,CAACc,OAAO,CAAC,KAAK,KAAK,EAAE;YACxC,IAAI,CAACf,OAAO,CAACe,OAAO,CAAC,CAACD,SAAS,GAAG,0CAA0C,CAAA;EAC9E,SAAC,MAAM;YACL,IAAI,CAACd,OAAO,CAACe,OAAO,CAAC,CAACD,SAAS,GAAI,0CAA0C,CAAA;EAC/E,SAAA;EACF,OAAA;EACF,KAAA;EACF,GAAA;IAEAJ,aAAaA,CAACwB,KAAK,EAAE;MACnBM,KAAK,EAAE,KAAK,IAAIrD,MAAM,IAAI,IAAI,CAACe,OAAO,EAAE;QACtC,IAAKf,MAAM,CAAC0C,GAAG,IAAIK,KAAK,CAACL,GAAG,CAACC,WAAW,EAAE,IAAI3C,MAAM,CAAC0C,GAAG,IAAM1C,MAAM,CAACnB,IAAI,IAAIkE,KAAK,CAAClE,IAAI,IAAImB,MAAM,CAACnB,IAAK,EAAE;EACvG;EACA,QAAA,KAAK,IAAIyE,QAAQ,IAAItD,MAAM,CAACoC,SAAS,EAAE;EACrC,UAAA,IAAI,CAACW,KAAK,CAACO,QAAQ,CAAC,EAAE,SAASD,KAAK,CAAA;EACtC,SAAA;EACA;UACA,IAAI,CAACP,WAAW,CAAC9C,MAAM,CAAC4B,OAAO,EAAEmB,KAAK,CAAC,CAAA;EACvC,QAAA,OAAA;EACF,OAAA;EACF,KAAA;EACF,GAAA;EACF;;;;EC5PA,IAAIQ,KAAK,GAAG,UAAUC,EAAE,EAAE;IACxB,OAAOA,EAAE,IAAIA,EAAE,CAACC,IAAI,KAAKA,IAAI,IAAID,EAAE,CAAA;EACrC,CAAC,CAAA;;EAED;MACAE,QAAc;EACZ;EACAH,KAAK,CAAC,OAAOI,UAAU,IAAI,QAAQ,IAAIA,UAAU,CAAC,IAClDJ,KAAK,CAAC,OAAOK,MAAM,IAAI,QAAQ,IAAIA,MAAM,CAAC;EAC1C;EACAL,KAAK,CAAC,OAAOM,IAAI,IAAI,QAAQ,IAAIA,IAAI,CAAC,IACtCN,KAAK,CAAC,OAAOO,cAAM,IAAI,QAAQ,IAAIA,cAAM,CAAC;EAC1C;EACC,YAAY;EAAE,EAAA,OAAO,IAAI,CAAA;EAAE,CAAC,EAAG,IAAIC,cAAI,IAAIC,QAAQ,CAAC,aAAa,CAAC,EAAE;;MCbvEN,OAAc,GAAG,UAAUO,IAAI,EAAE;IAC/B,IAAI;EACF,IAAA,OAAO,CAAC,CAACA,IAAI,EAAE,CAAA;KAChB,CAAC,OAAOC,KAAK,EAAE;EACd,IAAA,OAAO,IAAI,CAAA;EACb,GAAA;EACF,CAAC;;ECND,IAAIC,OAAK,GAAGC,OAA6B,CAAA;;EAEzC;EACAV,IAAAA,WAAc,GAAG,CAACS,OAAK,CAAC,YAAY;EAClC;IACA,OAAOnC,MAAM,CAACqC,cAAc,CAAC,EAAE,EAAE,CAAC,EAAE;MAAEC,GAAG,EAAE,YAAY;EAAE,MAAA,OAAO,CAAC,CAAA;EAAE,KAAA;EAAE,GAAC,CAAC,CAAC,CAAC,CAAC,KAAK,CAAC,CAAA;EAClF,CAAC,CAAC;;;;;;;;ECNF,IAAIH,OAAK,GAAGC,OAA6B,CAAA;EAEzCV,IAAAA,kBAAc,GAAG,CAACS,OAAK,CAAC,YAAY;EAClC;IACA,IAAI3E,IAAI,GAAI,YAAY,aAAe,CAAE+E,IAAI,EAAE,CAAA;EAC/C;IACA,OAAO,OAAO/E,IAAI,IAAI,UAAU,IAAIA,IAAI,CAACgF,cAAc,CAAC,WAAW,CAAC,CAAA;EACtE,CAAC,CAAC;;ECPF,IAAIC,aAAW,GAAGL,kBAA4C,CAAA;EAE9D,IAAIM,mBAAiB,GAAGV,QAAQ,CAACW,SAAS,CAAA;EAC1C,IAAIC,MAAI,GAAGF,mBAAiB,CAACE,IAAI,CAAA;EACjC,IAAIC,mBAAmB,GAAGJ,aAAW,IAAIC,mBAAiB,CAACH,IAAI,CAACA,IAAI,CAACK,MAAI,EAAEA,MAAI,CAAC,CAAA;EAEhFlB,IAAAA,mBAAc,GAAGe,aAAW,GAAGI,mBAAmB,GAAG,UAAUC,EAAE,EAAE;EACjE,EAAA,OAAO,YAAY;EACjB,IAAA,OAAOF,MAAI,CAACG,KAAK,CAACD,EAAE,EAAEE,SAAS,CAAC,CAAA;KACjC,CAAA;EACH,CAAC;;ECVD,IAAIC,aAAW,GAAG,OAAO/D,QAAQ,IAAI,QAAQ,IAAIA,QAAQ,CAACgE,GAAG,CAAA;;EAE7D;EACA;EACA,IAAIC,UAAU,GAAG,OAAOF,aAAW,IAAI,WAAW,IAAIA,aAAW,KAAK7B,SAAS,CAAA;EAE/EM,IAAAA,aAAc,GAAG;EACfwB,EAAAA,GAAG,EAAED,aAAW;EAChBE,EAAAA,UAAU,EAAEA,UAAAA;EACd,CAAC;;ECTD,IAAIC,cAAY,GAAGhB,aAAoC,CAAA;EAEvD,IAAIa,aAAW,GAAGG,cAAY,CAACF,GAAG,CAAA;;EAElC;EACA;EACAxB,IAAAA,YAAc,GAAG0B,cAAY,CAACD,UAAU,GAAG,UAAUE,QAAQ,EAAE;EAC7D,EAAA,OAAO,OAAOA,QAAQ,IAAI,UAAU,IAAIA,QAAQ,KAAKJ,aAAW,CAAA;EAClE,CAAC,GAAG,UAAUI,QAAQ,EAAE;IACtB,OAAO,OAAOA,QAAQ,IAAI,UAAU,CAAA;EACtC,CAAC;;ECVD;EACA;MACA3B,mBAAc,GAAG,UAAUF,EAAE,EAAE;EAC7B,EAAA,OAAOA,EAAE,KAAK,IAAI,IAAIA,EAAE,KAAKJ,SAAS,CAAA;EACxC,CAAC;;ECJD,IAAIkC,mBAAiB,GAAGlB,mBAA4C,CAAA;EAEpE,IAAImB,YAAU,GAAGC,SAAS,CAAA;;EAE1B;EACA;MACA9B,wBAAc,GAAG,UAAUF,EAAE,EAAE;EAC7B,EAAA,IAAI8B,mBAAiB,CAAC9B,EAAE,CAAC,EAAE,MAAM,IAAI+B,YAAU,CAAC,uBAAuB,GAAG/B,EAAE,CAAC,CAAA;EAC7E,EAAA,OAAOA,EAAE,CAAA;EACX,CAAC;;ECTD,IAAIiC,sBAAsB,GAAGrB,wBAAgD,CAAA;EAE7E,IAAIsB,SAAO,GAAG1D,MAAM,CAAA;;EAEpB;EACA;MACA0B,UAAc,GAAG,UAAU2B,QAAQ,EAAE;EACnC,EAAA,OAAOK,SAAO,CAACD,sBAAsB,CAACJ,QAAQ,CAAC,CAAC,CAAA;EAClD,CAAC;;ECRD,IAAIM,aAAW,GAAGvB,mBAA6C,CAAA;EAC/D,IAAIwB,QAAQ,GAAGxB,UAAiC,CAAA;EAEhD,IAAII,cAAc,GAAGmB,aAAW,CAAC,EAAE,CAACnB,cAAc,CAAC,CAAA;;EAEnD;EACA;EACA;MACAd,gBAAc,GAAG1B,MAAM,CAAC6D,MAAM,IAAI,SAASA,MAAMA,CAACrC,EAAE,EAAEd,GAAG,EAAE;IACzD,OAAO8B,cAAc,CAACoB,QAAQ,CAACpC,EAAE,CAAC,EAAEd,GAAG,CAAC,CAAA;EAC1C,CAAC;;ECVD,IAAIoD,aAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAIyB,QAAM,GAAGzB,gBAAwC,CAAA;EAErD,IAAIM,iBAAiB,GAAGV,QAAQ,CAACW,SAAS,CAAA;EAC1C;EACA,IAAIoB,aAAa,GAAGD,aAAW,IAAI9D,MAAM,CAACgE,wBAAwB,CAAA;EAElE,IAAIC,QAAM,GAAGJ,QAAM,CAACnB,iBAAiB,EAAE,MAAM,CAAC,CAAA;EAC9C;EACA,IAAIwB,MAAM,GAAGD,QAAM,IAAK,SAASE,SAASA,GAAG,aAAe,CAAEvG,IAAI,KAAK,WAAW,CAAA;EAClF,IAAIwG,cAAY,GAAGH,QAAM,KAAK,CAACH,aAAW,IAAKA,aAAW,IAAIC,aAAa,CAACrB,iBAAiB,EAAE,MAAM,CAAC,CAAC2B,YAAa,CAAC,CAAA;EAErH3C,IAAAA,YAAc,GAAG;EACfuC,EAAAA,MAAM,EAAEA,QAAM;EACdC,EAAAA,MAAM,EAAEA,MAAM;EACdE,EAAAA,YAAY,EAAEA,cAAAA;EAChB,CAAC;;EChBD,IAAItC,QAAM,GAAGM,QAA8B,CAAA;;EAE3C;EACA,IAAIC,gBAAc,GAAGrC,MAAM,CAACqC,cAAc,CAAA;EAE1CX,IAAAA,sBAAc,GAAG,UAAUhB,GAAG,EAAE4D,KAAK,EAAE;IACrC,IAAI;EACFjC,IAAAA,gBAAc,CAACP,QAAM,EAAEpB,GAAG,EAAE;EAAE4D,MAAAA,KAAK,EAAEA,KAAK;EAAED,MAAAA,YAAY,EAAE,IAAI;EAAEE,MAAAA,QAAQ,EAAE,IAAA;EAAK,KAAC,CAAC,CAAA;KAClF,CAAC,OAAOrC,KAAK,EAAE;EACdJ,IAAAA,QAAM,CAACpB,GAAG,CAAC,GAAG4D,KAAK,CAAA;EACrB,GAAA;EAAE,EAAA,OAAOA,KAAK,CAAA;EAChB,CAAC;;ECXD,IAAIxC,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAIoC,oBAAoB,GAAGpC,sBAA8C,CAAA;EAEzE,IAAIqC,MAAM,GAAG,oBAAoB,CAAA;EACjC,IAAIC,OAAK,GAAG5C,QAAM,CAAC2C,MAAM,CAAC,IAAID,oBAAoB,CAACC,MAAM,EAAE,EAAE,CAAC,CAAA;EAE9D/C,IAAAA,WAAc,GAAGgD,OAAK;;ECNtB,IAAIf,aAAW,GAAGvB,mBAA6C,CAAA;EAC/D,IAAIuC,YAAU,GAAGvC,YAAmC,CAAA;EACpD,IAAIsC,OAAK,GAAGtC,WAAoC,CAAA;EAEhD,IAAIwC,gBAAgB,GAAGjB,aAAW,CAAC3B,QAAQ,CAAC6C,QAAQ,CAAC,CAAA;;EAErD;EACA,IAAI,CAACF,YAAU,CAACD,OAAK,CAACI,aAAa,CAAC,EAAE;EACpCJ,EAAAA,OAAK,CAACI,aAAa,GAAG,UAAUtD,EAAE,EAAE;MAClC,OAAOoD,gBAAgB,CAACpD,EAAE,CAAC,CAAA;KAC5B,CAAA;EACH,CAAA;MAEAE,eAAc,GAAGgD,OAAK,CAACI,aAAa;;ECbpC,IAAIhD,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAIuC,YAAU,GAAGvC,YAAmC,CAAA;EAEpD,IAAI2C,SAAO,GAAGjD,QAAM,CAACiD,OAAO,CAAA;EAE5BrD,IAAAA,qBAAc,GAAGiD,YAAU,CAACI,SAAO,CAAC,IAAI,aAAa,CAACvH,IAAI,CAACwH,MAAM,CAACD,SAAO,CAAC,CAAC;;ECL3E,IAAIJ,YAAU,GAAGvC,YAAmC,CAAA;EACpD,IAAIgB,YAAY,GAAGhB,aAAoC,CAAA;EAEvD,IAAIa,WAAW,GAAGG,YAAY,CAACF,GAAG,CAAA;EAElCxB,IAAAA,UAAc,GAAG0B,YAAY,CAACD,UAAU,GAAG,UAAU3B,EAAE,EAAE;EACvD,EAAA,OAAO,OAAOA,EAAE,IAAI,QAAQ,GAAGA,EAAE,KAAK,IAAI,GAAGmD,YAAU,CAACnD,EAAE,CAAC,IAAIA,EAAE,KAAKyB,WAAW,CAAA;EACnF,CAAC,GAAG,UAAUzB,EAAE,EAAE;EAChB,EAAA,OAAO,OAAOA,EAAE,IAAI,QAAQ,GAAGA,EAAE,KAAK,IAAI,GAAGmD,YAAU,CAACnD,EAAE,CAAC,CAAA;EAC7D,CAAC;;;;ECTD,IAAIM,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAI6C,UAAQ,GAAG7C,UAAiC,CAAA;EAEhD,IAAIlD,UAAQ,GAAG4C,QAAM,CAAC5C,QAAQ,CAAA;EAC9B;EACA,IAAI+E,MAAM,GAAGgB,UAAQ,CAAC/F,UAAQ,CAAC,IAAI+F,UAAQ,CAAC/F,UAAQ,CAACQ,aAAa,CAAC,CAAA;MAEnEgC,qBAAc,GAAG,UAAUF,EAAE,EAAE;IAC7B,OAAOyC,MAAM,GAAG/E,UAAQ,CAACQ,aAAa,CAAC8B,EAAE,CAAC,GAAG,EAAE,CAAA;EACjD,CAAC;;ECTD,IAAIsC,aAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAID,OAAK,GAAGC,OAA6B,CAAA;EACzC,IAAI1C,aAAa,GAAG0C,qBAA+C,CAAA;;EAEnE;EACAV,IAAAA,YAAc,GAAG,CAACoC,aAAW,IAAI,CAAC3B,OAAK,CAAC,YAAY;EAClD;IACA,OAAOnC,MAAM,CAACqC,cAAc,CAAC3C,aAAa,CAAC,KAAK,CAAC,EAAE,GAAG,EAAE;MACtD4C,GAAG,EAAE,YAAY;EAAE,MAAA,OAAO,CAAC,CAAA;EAAE,KAAA;EAC/B,GAAC,CAAC,CAAC4C,CAAC,KAAK,CAAC,CAAA;EACZ,CAAC,CAAC;;ECVF,IAAIpB,aAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAID,OAAK,GAAGC,OAA6B,CAAA;;EAEzC;EACA;EACAV,IAAAA,oBAAc,GAAGoC,aAAW,IAAI3B,OAAK,CAAC,YAAY;EAChD;IACA,OAAOnC,MAAM,CAACqC,cAAc,CAAC,YAAY,aAAe,EAAE,WAAW,EAAE;EACrEiC,IAAAA,KAAK,EAAE,EAAE;EACTC,IAAAA,QAAQ,EAAE,KAAA;EACZ,GAAC,CAAC,CAAC5B,SAAS,KAAK,EAAE,CAAA;EACrB,CAAC,CAAC;;ECXF,IAAIsC,UAAQ,GAAG7C,UAAiC,CAAA;EAEhD,IAAI+C,SAAO,GAAGH,MAAM,CAAA;EACpB,IAAIzB,YAAU,GAAGC,SAAS,CAAA;;EAE1B;MACA9B,UAAc,GAAG,UAAU2B,QAAQ,EAAE;EACnC,EAAA,IAAI4B,UAAQ,CAAC5B,QAAQ,CAAC,EAAE,OAAOA,QAAQ,CAAA;IACvC,MAAM,IAAIE,YAAU,CAAC4B,SAAO,CAAC9B,QAAQ,CAAC,GAAG,mBAAmB,CAAC,CAAA;EAC/D,CAAC;;ECTD,IAAIZ,WAAW,GAAGL,kBAA4C,CAAA;EAE9D,IAAIQ,MAAI,GAAGZ,QAAQ,CAACW,SAAS,CAACC,IAAI,CAAA;MAElClB,YAAc,GAAGe,WAAW,GAAGG,MAAI,CAACL,IAAI,CAACK,MAAI,CAAC,GAAG,YAAY;EAC3D,EAAA,OAAOA,MAAI,CAACG,KAAK,CAACH,MAAI,EAAEI,SAAS,CAAC,CAAA;EACpC,CAAC;;ECND,IAAIlB,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAIuC,YAAU,GAAGvC,YAAmC,CAAA;EAEpD,IAAIgD,SAAS,GAAG,UAAU/B,QAAQ,EAAE;EAClC,EAAA,OAAOsB,YAAU,CAACtB,QAAQ,CAAC,GAAGA,QAAQ,GAAGjC,SAAS,CAAA;EACpD,CAAC,CAAA;EAEDM,IAAAA,YAAc,GAAG,UAAU2D,SAAS,EAAEC,MAAM,EAAE;IAC5C,OAAOtC,SAAS,CAACzC,MAAM,GAAG,CAAC,GAAG6E,SAAS,CAACtD,QAAM,CAACuD,SAAS,CAAC,CAAC,GAAGvD,QAAM,CAACuD,SAAS,CAAC,IAAIvD,QAAM,CAACuD,SAAS,CAAC,CAACC,MAAM,CAAC,CAAA;EAC7G,CAAC;;ECTD,IAAI3B,aAAW,GAAGvB,mBAA6C,CAAA;EAE/DV,IAAAA,mBAAc,GAAGiC,aAAW,CAAC,EAAE,CAAC4B,aAAa,CAAC;;ECF9C7D,IAAAA,eAAc,GAAG,OAAOjE,SAAS,IAAI,WAAW,IAAIuH,MAAM,CAACvH,SAAS,CAAC+H,SAAS,CAAC,IAAI,EAAE;;ECArF,IAAI1D,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAIoD,SAAS,GAAGpD,eAAyC,CAAA;EAEzD,IAAIqD,OAAO,GAAG3D,QAAM,CAAC2D,OAAO,CAAA;EAC5B,IAAIC,IAAI,GAAG5D,QAAM,CAAC4D,IAAI,CAAA;EACtB,IAAIC,QAAQ,GAAGF,OAAO,IAAIA,OAAO,CAACE,QAAQ,IAAID,IAAI,IAAIA,IAAI,CAACE,OAAO,CAAA;EAClE,IAAIC,EAAE,GAAGF,QAAQ,IAAIA,QAAQ,CAACE,EAAE,CAAA;EAChC,IAAIpF,KAAK,EAAEmF,OAAO,CAAA;EAElB,IAAIC,EAAE,EAAE;EACNpF,EAAAA,KAAK,GAAGoF,EAAE,CAAC1F,KAAK,CAAC,GAAG,CAAC,CAAA;EACrB;EACA;IACAyF,OAAO,GAAGnF,KAAK,CAAC,CAAC,CAAC,GAAG,CAAC,IAAIA,KAAK,CAAC,CAAC,CAAC,GAAG,CAAC,GAAG,CAAC,GAAG,EAAEA,KAAK,CAAC,CAAC,CAAC,GAAGA,KAAK,CAAC,CAAC,CAAC,CAAC,CAAA;EACrE,CAAA;;EAEA;EACA;EACA,IAAI,CAACmF,OAAO,IAAIJ,SAAS,EAAE;EACzB/E,EAAAA,KAAK,GAAG+E,SAAS,CAAC/E,KAAK,CAAC,aAAa,CAAC,CAAA;IACtC,IAAI,CAACA,KAAK,IAAIA,KAAK,CAAC,CAAC,CAAC,IAAI,EAAE,EAAE;EAC5BA,IAAAA,KAAK,GAAG+E,SAAS,CAAC/E,KAAK,CAAC,eAAe,CAAC,CAAA;MACxC,IAAIA,KAAK,EAAEmF,OAAO,GAAG,CAACnF,KAAK,CAAC,CAAC,CAAC,CAAA;EAChC,GAAA;EACF,CAAA;EAEAiB,IAAAA,eAAc,GAAGkE,OAAO;;EC1BxB;EACA,IAAIE,UAAU,GAAG1D,eAAyC,CAAA;EAC1D,IAAID,OAAK,GAAGC,OAA6B,CAAA;EACzC,IAAIN,QAAM,GAAGM,QAA8B,CAAA;EAE3C,IAAI+C,SAAO,GAAGrD,QAAM,CAACkD,MAAM,CAAA;;EAE3B;MACAtD,0BAAc,GAAG,CAAC,CAAC1B,MAAM,CAAC+F,qBAAqB,IAAI,CAAC5D,OAAK,CAAC,YAAY;EACpE,EAAA,IAAI6D,MAAM,GAAGC,MAAM,CAAC,kBAAkB,CAAC,CAAA;EACvC;EACA;EACA;EACA;EACA,EAAA,OAAO,CAACd,SAAO,CAACa,MAAM,CAAC,IAAI,EAAEhG,MAAM,CAACgG,MAAM,CAAC,YAAYC,MAAM,CAAC;EAC5D;IACA,CAACA,MAAM,CAACC,IAAI,IAAIJ,UAAU,IAAIA,UAAU,GAAG,EAAE,CAAA;EACjD,CAAC,CAAC;;ECjBF;EACA,IAAIK,eAAa,GAAG/D,0BAAoD,CAAA;EAExEV,IAAAA,cAAc,GAAGyE,eAAa,IACzB,CAACF,MAAM,CAACC,IAAI,IACZ,OAAOD,MAAM,CAACG,QAAQ,IAAI,QAAQ;;ECLvC,IAAIC,UAAU,GAAGjE,YAAoC,CAAA;EACrD,IAAIuC,YAAU,GAAGvC,YAAmC,CAAA;EACpD,IAAImD,aAAa,GAAGnD,mBAA8C,CAAA;EAClE,IAAIkE,mBAAiB,GAAGlE,cAAyC,CAAA;EAEjE,IAAIsB,OAAO,GAAG1D,MAAM,CAAA;EAEpB0B,IAAAA,UAAc,GAAG4E,mBAAiB,GAAG,UAAU9E,EAAE,EAAE;IACjD,OAAO,OAAOA,EAAE,IAAI,QAAQ,CAAA;EAC9B,CAAC,GAAG,UAAUA,EAAE,EAAE;EAChB,EAAA,IAAI+E,OAAO,GAAGF,UAAU,CAAC,QAAQ,CAAC,CAAA;EAClC,EAAA,OAAO1B,YAAU,CAAC4B,OAAO,CAAC,IAAIhB,aAAa,CAACgB,OAAO,CAAC5D,SAAS,EAAEe,OAAO,CAAClC,EAAE,CAAC,CAAC,CAAA;EAC7E,CAAC;;ECZD,IAAI2D,SAAO,GAAGH,MAAM,CAAA;MAEpBtD,aAAc,GAAG,UAAU2B,QAAQ,EAAE;IACnC,IAAI;MACF,OAAO8B,SAAO,CAAC9B,QAAQ,CAAC,CAAA;KACzB,CAAC,OAAOnB,KAAK,EAAE;EACd,IAAA,OAAO,QAAQ,CAAA;EACjB,GAAA;EACF,CAAC;;ECRD,IAAIyC,YAAU,GAAGvC,YAAmC,CAAA;EACpD,IAAIoE,WAAW,GAAGpE,aAAqC,CAAA;EAEvD,IAAImB,YAAU,GAAGC,SAAS,CAAA;;EAE1B;MACA9B,WAAc,GAAG,UAAU2B,QAAQ,EAAE;EACnC,EAAA,IAAIsB,YAAU,CAACtB,QAAQ,CAAC,EAAE,OAAOA,QAAQ,CAAA;IACzC,MAAM,IAAIE,YAAU,CAACiD,WAAW,CAACnD,QAAQ,CAAC,GAAG,oBAAoB,CAAC,CAAA;EACpE,CAAC;;ECTD,IAAIoD,SAAS,GAAGrE,WAAkC,CAAA;EAClD,IAAIkB,iBAAiB,GAAGlB,mBAA4C,CAAA;;EAEpE;EACA;EACAV,IAAAA,WAAc,GAAG,UAAUgF,CAAC,EAAEC,CAAC,EAAE;EAC/B,EAAA,IAAIC,IAAI,GAAGF,CAAC,CAACC,CAAC,CAAC,CAAA;IACf,OAAOrD,iBAAiB,CAACsD,IAAI,CAAC,GAAGxF,SAAS,GAAGqF,SAAS,CAACG,IAAI,CAAC,CAAA;EAC9D,CAAC;;ECRD,IAAIhE,MAAI,GAAGR,YAAqC,CAAA;EAChD,IAAIuC,YAAU,GAAGvC,YAAmC,CAAA;EACpD,IAAI6C,UAAQ,GAAG7C,UAAiC,CAAA;EAEhD,IAAImB,YAAU,GAAGC,SAAS,CAAA;;EAE1B;EACA;EACA9B,IAAAA,qBAAc,GAAG,UAAUmF,KAAK,EAAEC,IAAI,EAAE;IACtC,IAAIhE,EAAE,EAAEiE,GAAG,CAAA;IACX,IAAID,IAAI,KAAK,QAAQ,IAAInC,YAAU,CAAC7B,EAAE,GAAG+D,KAAK,CAAChC,QAAQ,CAAC,IAAI,CAACI,UAAQ,CAAC8B,GAAG,GAAGnE,MAAI,CAACE,EAAE,EAAE+D,KAAK,CAAC,CAAC,EAAE,OAAOE,GAAG,CAAA;IACxG,IAAIpC,YAAU,CAAC7B,EAAE,GAAG+D,KAAK,CAACG,OAAO,CAAC,IAAI,CAAC/B,UAAQ,CAAC8B,GAAG,GAAGnE,MAAI,CAACE,EAAE,EAAE+D,KAAK,CAAC,CAAC,EAAE,OAAOE,GAAG,CAAA;IAClF,IAAID,IAAI,KAAK,QAAQ,IAAInC,YAAU,CAAC7B,EAAE,GAAG+D,KAAK,CAAChC,QAAQ,CAAC,IAAI,CAACI,UAAQ,CAAC8B,GAAG,GAAGnE,MAAI,CAACE,EAAE,EAAE+D,KAAK,CAAC,CAAC,EAAE,OAAOE,GAAG,CAAA;EACxG,EAAA,MAAM,IAAIxD,YAAU,CAAC,yCAAyC,CAAC,CAAA;EACjE,CAAC;;;;;;;;ECbD,IAAImB,OAAK,GAAGtC,WAAoC,CAAA;EAEhD,CAACV,gBAAc,GAAG,UAAUhB,GAAG,EAAE4D,KAAK,EAAE;EACtC,EAAA,OAAOI,OAAK,CAAChE,GAAG,CAAC,KAAKgE,OAAK,CAAChE,GAAG,CAAC,GAAG4D,KAAK,KAAKlD,SAAS,GAAGkD,KAAK,GAAG,EAAE,CAAC,CAAA;EACtE,CAAC,EAAE,UAAU,EAAE,EAAE,CAAC,CAAC9D,IAAI,CAAC;EACtBoF,EAAAA,OAAO,EAAE,QAAQ;EACjBqB,EAAAA,IAAI,EAAqB,QAAQ;EACjCC,EAAAA,SAAS,EAAE,2CAA2C;EACtDC,EAAAA,OAAO,EAAE,0DAA0D;EACnEC,EAAAA,MAAM,EAAE,qCAAA;EACV,CAAC,CAAC;;ECXF,IAAIzD,aAAW,GAAGvB,mBAA6C,CAAA;EAE/D,IAAIiF,EAAE,GAAG,CAAC,CAAA;EACV,IAAIC,OAAO,GAAG7F,IAAI,CAAC8F,MAAM,EAAE,CAAA;EAC3B,IAAI1C,QAAQ,GAAGlB,aAAW,CAAC,GAAG,CAACkB,QAAQ,CAAC,CAAA;MAExCnD,KAAc,GAAG,UAAUhB,GAAG,EAAE;IAC9B,OAAO,SAAS,IAAIA,GAAG,KAAKU,SAAS,GAAG,EAAE,GAAGV,GAAG,CAAC,GAAG,IAAI,GAAGmE,QAAQ,CAAC,EAAEwC,EAAE,GAAGC,OAAO,EAAE,EAAE,CAAC,CAAA;EACzF,CAAC;;ECRD,IAAIxF,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAIoF,QAAM,GAAGpF,aAA8B,CAAA;EAC3C,IAAIyB,QAAM,GAAGzB,gBAAwC,CAAA;EACrD,IAAIqF,KAAG,GAAGrF,KAA2B,CAAA;EACrC,IAAI+D,aAAa,GAAG/D,0BAAoD,CAAA;EACxE,IAAIkE,iBAAiB,GAAGlE,cAAyC,CAAA;EAEjE,IAAI6D,QAAM,GAAGnE,QAAM,CAACmE,MAAM,CAAA;EAC1B,IAAIyB,qBAAqB,GAAGF,QAAM,CAAC,KAAK,CAAC,CAAA;EACzC,IAAIG,qBAAqB,GAAGrB,iBAAiB,GAAGL,QAAM,CAAC,KAAK,CAAC,IAAIA,QAAM,GAAGA,QAAM,IAAIA,QAAM,CAAC2B,aAAa,IAAIH,KAAG,CAAA;MAE/G/F,iBAAc,GAAG,UAAU9D,IAAI,EAAE;EAC/B,EAAA,IAAI,CAACiG,QAAM,CAAC6D,qBAAqB,EAAE9J,IAAI,CAAC,EAAE;MACxC8J,qBAAqB,CAAC9J,IAAI,CAAC,GAAGuI,aAAa,IAAItC,QAAM,CAACoC,QAAM,EAAErI,IAAI,CAAC,GAC/DqI,QAAM,CAACrI,IAAI,CAAC,GACZ+J,qBAAqB,CAAC,SAAS,GAAG/J,IAAI,CAAC,CAAA;EAC7C,GAAA;IAAE,OAAO8J,qBAAqB,CAAC9J,IAAI,CAAC,CAAA;EACtC,CAAC;;ECjBD,IAAIgF,IAAI,GAAGR,YAAqC,CAAA;EAChD,IAAI6C,UAAQ,GAAG7C,UAAiC,CAAA;EAChD,IAAIyF,UAAQ,GAAGzF,UAAiC,CAAA;EAChD,IAAI0F,SAAS,GAAG1F,WAAkC,CAAA;EAClD,IAAI2F,mBAAmB,GAAG3F,qBAA6C,CAAA;EACvE,IAAI4F,eAAe,GAAG5F,iBAAyC,CAAA;EAE/D,IAAImB,YAAU,GAAGC,SAAS,CAAA;EAC1B,IAAIyE,YAAY,GAAGD,eAAe,CAAC,aAAa,CAAC,CAAA;;EAEjD;EACA;EACAtG,IAAAA,aAAc,GAAG,UAAUmF,KAAK,EAAEC,IAAI,EAAE;EACtC,EAAA,IAAI,CAAC7B,UAAQ,CAAC4B,KAAK,CAAC,IAAIgB,UAAQ,CAAChB,KAAK,CAAC,EAAE,OAAOA,KAAK,CAAA;EACrD,EAAA,IAAIqB,YAAY,GAAGJ,SAAS,CAACjB,KAAK,EAAEoB,YAAY,CAAC,CAAA;EACjD,EAAA,IAAIE,MAAM,CAAA;EACV,EAAA,IAAID,YAAY,EAAE;EAChB,IAAA,IAAIpB,IAAI,KAAK1F,SAAS,EAAE0F,IAAI,GAAG,SAAS,CAAA;MACxCqB,MAAM,GAAGvF,IAAI,CAACsF,YAAY,EAAErB,KAAK,EAAEC,IAAI,CAAC,CAAA;EACxC,IAAA,IAAI,CAAC7B,UAAQ,CAACkD,MAAM,CAAC,IAAIN,UAAQ,CAACM,MAAM,CAAC,EAAE,OAAOA,MAAM,CAAA;EACxD,IAAA,MAAM,IAAI5E,YAAU,CAAC,yCAAyC,CAAC,CAAA;EACjE,GAAA;EACA,EAAA,IAAIuD,IAAI,KAAK1F,SAAS,EAAE0F,IAAI,GAAG,QAAQ,CAAA;EACvC,EAAA,OAAOiB,mBAAmB,CAAClB,KAAK,EAAEC,IAAI,CAAC,CAAA;EACzC,CAAC;;ECxBD,IAAIsB,WAAW,GAAGhG,aAAoC,CAAA;EACtD,IAAIyF,QAAQ,GAAGzF,UAAiC,CAAA;;EAEhD;EACA;MACAV,eAAc,GAAG,UAAU2B,QAAQ,EAAE;EACnC,EAAA,IAAI3C,GAAG,GAAG0H,WAAW,CAAC/E,QAAQ,EAAE,QAAQ,CAAC,CAAA;IACzC,OAAOwE,QAAQ,CAACnH,GAAG,CAAC,GAAGA,GAAG,GAAGA,GAAG,GAAG,EAAE,CAAA;EACvC,CAAC;;ECRD,IAAIoD,aAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAIiG,cAAc,GAAGjG,YAAsC,CAAA;EAC3D,IAAIkG,uBAAuB,GAAGlG,oBAA+C,CAAA;EAC7E,IAAImG,UAAQ,GAAGnG,UAAiC,CAAA;EAChD,IAAIoG,aAAa,GAAGpG,eAAuC,CAAA;EAE3D,IAAImB,UAAU,GAAGC,SAAS,CAAA;EAC1B;EACA,IAAIiF,eAAe,GAAGzI,MAAM,CAACqC,cAAc,CAAA;EAC3C;EACA,IAAIqG,yBAAyB,GAAG1I,MAAM,CAACgE,wBAAwB,CAAA;EAC/D,IAAI2E,UAAU,GAAG,YAAY,CAAA;EAC7B,IAAIvE,YAAY,GAAG,cAAc,CAAA;EACjC,IAAIwE,QAAQ,GAAG,UAAU,CAAA;;EAEzB;EACA;EACAC,oBAAAA,CAAAA,CAAS,GAAG/E,aAAW,GAAGwE,uBAAuB,GAAG,SAASjG,cAAcA,CAACyG,CAAC,EAAEnC,CAAC,EAAEoC,UAAU,EAAE;IAC5FR,UAAQ,CAACO,CAAC,CAAC,CAAA;EACXnC,EAAAA,CAAC,GAAG6B,aAAa,CAAC7B,CAAC,CAAC,CAAA;IACpB4B,UAAQ,CAACQ,UAAU,CAAC,CAAA;IACpB,IAAI,OAAOD,CAAC,KAAK,UAAU,IAAInC,CAAC,KAAK,WAAW,IAAI,OAAO,IAAIoC,UAAU,IAAIH,QAAQ,IAAIG,UAAU,IAAI,CAACA,UAAU,CAACH,QAAQ,CAAC,EAAE;EAC5H,IAAA,IAAII,OAAO,GAAGN,yBAAyB,CAACI,CAAC,EAAEnC,CAAC,CAAC,CAAA;EAC7C,IAAA,IAAIqC,OAAO,IAAIA,OAAO,CAACJ,QAAQ,CAAC,EAAE;EAChCE,MAAAA,CAAC,CAACnC,CAAC,CAAC,GAAGoC,UAAU,CAACzE,KAAK,CAAA;EACvByE,MAAAA,UAAU,GAAG;EACX1E,QAAAA,YAAY,EAAED,YAAY,IAAI2E,UAAU,GAAGA,UAAU,CAAC3E,YAAY,CAAC,GAAG4E,OAAO,CAAC5E,YAAY,CAAC;EAC3F6E,QAAAA,UAAU,EAAEN,UAAU,IAAII,UAAU,GAAGA,UAAU,CAACJ,UAAU,CAAC,GAAGK,OAAO,CAACL,UAAU,CAAC;EACnFpE,QAAAA,QAAQ,EAAE,KAAA;SACX,CAAA;EACH,KAAA;EACF,GAAA;EAAE,EAAA,OAAOkE,eAAe,CAACK,CAAC,EAAEnC,CAAC,EAAEoC,UAAU,CAAC,CAAA;EAC5C,CAAC,GAAGN,eAAe,GAAG,SAASpG,cAAcA,CAACyG,CAAC,EAAEnC,CAAC,EAAEoC,UAAU,EAAE;IAC9DR,UAAQ,CAACO,CAAC,CAAC,CAAA;EACXnC,EAAAA,CAAC,GAAG6B,aAAa,CAAC7B,CAAC,CAAC,CAAA;IACpB4B,UAAQ,CAACQ,UAAU,CAAC,CAAA;IACpB,IAAIV,cAAc,EAAE,IAAI;EACtB,IAAA,OAAOI,eAAe,CAACK,CAAC,EAAEnC,CAAC,EAAEoC,UAAU,CAAC,CAAA;EAC1C,GAAC,CAAC,OAAO7G,KAAK,EAAE,aAAE;EAClB,EAAA,IAAI,KAAK,IAAI6G,UAAU,IAAI,KAAK,IAAIA,UAAU,EAAE,MAAM,IAAIxF,UAAU,CAAC,yBAAyB,CAAC,CAAA;IAC/F,IAAI,OAAO,IAAIwF,UAAU,EAAED,CAAC,CAACnC,CAAC,CAAC,GAAGoC,UAAU,CAACzE,KAAK,CAAA;EAClD,EAAA,OAAOwE,CAAC,CAAA;EACV;;EC1CApH,IAAAA,0BAAc,GAAG,UAAUwH,MAAM,EAAE5E,KAAK,EAAE;IACxC,OAAO;EACL2E,IAAAA,UAAU,EAAE,EAAEC,MAAM,GAAG,CAAC,CAAC;EACzB7E,IAAAA,YAAY,EAAE,EAAE6E,MAAM,GAAG,CAAC,CAAC;EAC3B3E,IAAAA,QAAQ,EAAE,EAAE2E,MAAM,GAAG,CAAC,CAAC;EACvB5E,IAAAA,KAAK,EAAEA,KAAAA;KACR,CAAA;EACH,CAAC;;ECPD,IAAIR,aAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAI+G,oBAAoB,GAAG/G,oBAA8C,CAAA;EACzE,IAAIgH,wBAAwB,GAAGhH,0BAAkD,CAAA;MAEjFV,6BAAc,GAAGoC,aAAW,GAAG,UAAUuF,MAAM,EAAE3I,GAAG,EAAE4D,KAAK,EAAE;EAC3D,EAAA,OAAO6E,oBAAoB,CAACG,CAAC,CAACD,MAAM,EAAE3I,GAAG,EAAE0I,wBAAwB,CAAC,CAAC,EAAE9E,KAAK,CAAC,CAAC,CAAA;EAChF,CAAC,GAAG,UAAU+E,MAAM,EAAE3I,GAAG,EAAE4D,KAAK,EAAE;EAChC+E,EAAAA,MAAM,CAAC3I,GAAG,CAAC,GAAG4D,KAAK,CAAA;EACnB,EAAA,OAAO+E,MAAM,CAAA;EACf,CAAC;;ECTD,IAAI7B,QAAM,GAAGpF,aAA8B,CAAA;EAC3C,IAAIqF,GAAG,GAAGrF,KAA2B,CAAA;EAErC,IAAIlC,IAAI,GAAGsH,QAAM,CAAC,MAAM,CAAC,CAAA;MAEzB9F,WAAc,GAAG,UAAUhB,GAAG,EAAE;EAC9B,EAAA,OAAOR,IAAI,CAACQ,GAAG,CAAC,KAAKR,IAAI,CAACQ,GAAG,CAAC,GAAG+G,GAAG,CAAC/G,GAAG,CAAC,CAAC,CAAA;EAC5C,CAAC;;ECPD,IAAI6I,eAAe,GAAGnH,qBAAgD,CAAA;EACtE,IAAIN,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAI6C,QAAQ,GAAG7C,UAAiC,CAAA;EAChD,IAAIoH,2BAA2B,GAAGpH,6BAAsD,CAAA;EACxF,IAAIyB,QAAM,GAAGzB,gBAAwC,CAAA;EACrD,IAAIoF,MAAM,GAAGpF,WAAoC,CAAA;EACjD,IAAIqH,SAAS,GAAGrH,WAAkC,CAAA;EAGlD,IAAIsH,0BAA0B,GAAG,4BAA4B,CAAA;EAC7D,IAAIlG,WAAS,GAAG1B,QAAM,CAAC0B,SAAS,CAAA;EAChC,IAAIuB,OAAO,GAAGjD,QAAM,CAACiD,OAAO,CAAA;EAC5B,IAAI4E,GAAG,EAAErH,GAAG,EAAEsH,GAAG,CAAA;EAEjB,IAAIC,OAAO,GAAG,UAAUrI,EAAE,EAAE;EAC1B,EAAA,OAAOoI,GAAG,CAACpI,EAAE,CAAC,GAAGc,GAAG,CAACd,EAAE,CAAC,GAAGmI,GAAG,CAACnI,EAAE,EAAE,EAAE,CAAC,CAAA;EACxC,CAAC,CAAA;EAED,IAAIsI,SAAS,GAAG,UAAUC,IAAI,EAAE;IAC9B,OAAO,UAAUvI,EAAE,EAAE;EACnB,IAAA,IAAI1C,KAAK,CAAA;EACT,IAAA,IAAI,CAACmG,QAAQ,CAACzD,EAAE,CAAC,IAAI,CAAC1C,KAAK,GAAGwD,GAAG,CAACd,EAAE,CAAC,EAAEwI,IAAI,KAAKD,IAAI,EAAE;QACpD,MAAM,IAAIvG,WAAS,CAAC,yBAAyB,GAAGuG,IAAI,GAAG,WAAW,CAAC,CAAA;EACrE,KAAA;EAAE,IAAA,OAAOjL,KAAK,CAAA;KACf,CAAA;EACH,CAAC,CAAA;EAED,IAAIyK,eAAe,IAAI/B,MAAM,CAAC1I,KAAK,EAAE;EACnC,EAAA,IAAI4F,KAAK,GAAG8C,MAAM,CAAC1I,KAAK,KAAK0I,MAAM,CAAC1I,KAAK,GAAG,IAAIiG,OAAO,EAAE,CAAC,CAAA;EAC1D;EACAL,EAAAA,KAAK,CAACpC,GAAG,GAAGoC,KAAK,CAACpC,GAAG,CAAA;EACrBoC,EAAAA,KAAK,CAACkF,GAAG,GAAGlF,KAAK,CAACkF,GAAG,CAAA;EACrBlF,EAAAA,KAAK,CAACiF,GAAG,GAAGjF,KAAK,CAACiF,GAAG,CAAA;EACrB;EACAA,EAAAA,GAAG,GAAG,UAAUnI,EAAE,EAAEyI,QAAQ,EAAE;EAC5B,IAAA,IAAIvF,KAAK,CAACkF,GAAG,CAACpI,EAAE,CAAC,EAAE,MAAM,IAAIgC,WAAS,CAACkG,0BAA0B,CAAC,CAAA;MAClEO,QAAQ,CAACC,MAAM,GAAG1I,EAAE,CAAA;EACpBkD,IAAAA,KAAK,CAACiF,GAAG,CAACnI,EAAE,EAAEyI,QAAQ,CAAC,CAAA;EACvB,IAAA,OAAOA,QAAQ,CAAA;KAChB,CAAA;EACD3H,EAAAA,GAAG,GAAG,UAAUd,EAAE,EAAE;MAClB,OAAOkD,KAAK,CAACpC,GAAG,CAACd,EAAE,CAAC,IAAI,EAAE,CAAA;KAC3B,CAAA;EACDoI,EAAAA,GAAG,GAAG,UAAUpI,EAAE,EAAE;EAClB,IAAA,OAAOkD,KAAK,CAACkF,GAAG,CAACpI,EAAE,CAAC,CAAA;KACrB,CAAA;EACH,CAAC,MAAM;EACL,EAAA,IAAI2I,KAAK,GAAGV,SAAS,CAAC,OAAO,CAAC,CAAA;EAE9BE,EAAAA,GAAG,GAAG,UAAUnI,EAAE,EAAEyI,QAAQ,EAAE;EAC5B,IAAA,IAAIpG,QAAM,CAACrC,EAAE,EAAE2I,KAAK,CAAC,EAAE,MAAM,IAAI3G,WAAS,CAACkG,0BAA0B,CAAC,CAAA;MACtEO,QAAQ,CAACC,MAAM,GAAG1I,EAAE,CAAA;EACpBgI,IAAAA,2BAA2B,CAAChI,EAAE,EAAE2I,KAAK,EAAEF,QAAQ,CAAC,CAAA;EAChD,IAAA,OAAOA,QAAQ,CAAA;KAChB,CAAA;EACD3H,EAAAA,GAAG,GAAG,UAAUd,EAAE,EAAE;EAClB,IAAA,OAAOqC,QAAM,CAACrC,EAAE,EAAE2I,KAAK,CAAC,GAAG3I,EAAE,CAAC2I,KAAK,CAAC,GAAG,EAAE,CAAA;KAC1C,CAAA;EACDP,EAAAA,GAAG,GAAG,UAAUpI,EAAE,EAAE;EAClB,IAAA,OAAOqC,QAAM,CAACrC,EAAE,EAAE2I,KAAK,CAAC,CAAA;KACzB,CAAA;EACH,CAAA;EAEAzI,IAAAA,aAAc,GAAG;EACfiI,EAAAA,GAAG,EAAEA,GAAG;EACRrH,EAAAA,GAAG,EAAEA,GAAG;EACRsH,EAAAA,GAAG,EAAEA,GAAG;EACRC,EAAAA,OAAO,EAAEA,OAAO;EAChBC,EAAAA,SAAS,EAAEA,SAAAA;EACb,CAAC;;ECrED,IAAInG,WAAW,GAAGvB,mBAA6C,CAAA;EAC/D,IAAID,OAAK,GAAGC,OAA6B,CAAA;EACzC,IAAIuC,UAAU,GAAGvC,YAAmC,CAAA;EACpD,IAAIyB,MAAM,GAAGzB,gBAAwC,CAAA;EACrD,IAAI0B,aAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAIgI,0BAA0B,GAAGhI,YAAqC,CAACgC,YAAY,CAAA;EACnF,IAAIU,aAAa,GAAG1C,eAAsC,CAAA;EAC1D,IAAIiI,mBAAmB,GAAGjI,aAAsC,CAAA;EAEhE,IAAIkI,oBAAoB,GAAGD,mBAAmB,CAACR,OAAO,CAAA;EACtD,IAAIU,gBAAgB,GAAGF,mBAAmB,CAAC/H,GAAG,CAAA;EAC9C,IAAI6C,OAAO,GAAGH,MAAM,CAAA;EACpB;EACA,IAAI3C,gBAAc,GAAGrC,MAAM,CAACqC,cAAc,CAAA;EAC1C,IAAImI,WAAW,GAAG7G,WAAW,CAAC,EAAE,CAAC8G,KAAK,CAAC,CAAA;EACvC,IAAIC,OAAO,GAAG/G,WAAW,CAAC,EAAE,CAAC+G,OAAO,CAAC,CAAA;EACrC,IAAI7J,IAAI,GAAG8C,WAAW,CAAC,EAAE,CAAC9C,IAAI,CAAC,CAAA;EAE/B,IAAI8J,mBAAmB,GAAG7G,aAAW,IAAI,CAAC3B,OAAK,CAAC,YAAY;EAC1D,EAAA,OAAOE,gBAAc,CAAC,YAAY,aAAe,EAAE,QAAQ,EAAE;EAAEiC,IAAAA,KAAK,EAAE,CAAA;EAAE,GAAC,CAAC,CAAC/D,MAAM,KAAK,CAAC,CAAA;EACzF,CAAC,CAAC,CAAA;EAEF,IAAIqK,QAAQ,GAAG5F,MAAM,CAACA,MAAM,CAAC,CAAC7E,KAAK,CAAC,QAAQ,CAAC,CAAA;EAE7C,IAAI0K,aAAW,GAAGnJ,aAAAA,CAAAA,OAAc,GAAG,UAAU4C,KAAK,EAAE1G,IAAI,EAAEkN,OAAO,EAAE;EACjE,EAAA,IAAIN,WAAW,CAACrF,OAAO,CAACvH,IAAI,CAAC,EAAE,CAAC,EAAE,CAAC,CAAC,KAAK,SAAS,EAAE;EAClDA,IAAAA,IAAI,GAAG,GAAG,GAAG8M,OAAO,CAACvF,OAAO,CAACvH,IAAI,CAAC,EAAE,oBAAoB,EAAE,IAAI,CAAC,GAAG,GAAG,CAAA;EACvE,GAAA;IACA,IAAIkN,OAAO,IAAIA,OAAO,CAACC,MAAM,EAAEnN,IAAI,GAAG,MAAM,GAAGA,IAAI,CAAA;IACnD,IAAIkN,OAAO,IAAIA,OAAO,CAACE,MAAM,EAAEpN,IAAI,GAAG,MAAM,GAAGA,IAAI,CAAA;EACnD,EAAA,IAAI,CAACiG,MAAM,CAACS,KAAK,EAAE,MAAM,CAAC,IAAK8F,0BAA0B,IAAI9F,KAAK,CAAC1G,IAAI,KAAKA,IAAK,EAAE;EACjF,IAAA,IAAIkG,aAAW,EAAEzB,gBAAc,CAACiC,KAAK,EAAE,MAAM,EAAE;EAAEA,MAAAA,KAAK,EAAE1G,IAAI;EAAEyG,MAAAA,YAAY,EAAE,IAAA;EAAK,KAAC,CAAC,CAAC,KAC/EC,KAAK,CAAC1G,IAAI,GAAGA,IAAI,CAAA;EACxB,GAAA;EACA,EAAA,IAAI+M,mBAAmB,IAAIG,OAAO,IAAIjH,MAAM,CAACiH,OAAO,EAAE,OAAO,CAAC,IAAIxG,KAAK,CAAC/D,MAAM,KAAKuK,OAAO,CAACG,KAAK,EAAE;EAChG5I,IAAAA,gBAAc,CAACiC,KAAK,EAAE,QAAQ,EAAE;QAAEA,KAAK,EAAEwG,OAAO,CAACG,KAAAA;EAAM,KAAC,CAAC,CAAA;EAC3D,GAAA;IACA,IAAI;EACF,IAAA,IAAIH,OAAO,IAAIjH,MAAM,CAACiH,OAAO,EAAE,aAAa,CAAC,IAAIA,OAAO,CAACrM,WAAW,EAAE;EACpE,MAAA,IAAIqF,aAAW,EAAEzB,gBAAc,CAACiC,KAAK,EAAE,WAAW,EAAE;EAAEC,QAAAA,QAAQ,EAAE,KAAA;EAAM,OAAC,CAAC,CAAA;EAC1E;OACC,MAAM,IAAID,KAAK,CAAC3B,SAAS,EAAE2B,KAAK,CAAC3B,SAAS,GAAGvB,SAAS,CAAA;EACzD,GAAC,CAAC,OAAOc,KAAK,EAAE,aAAE;EAClB,EAAA,IAAIpD,KAAK,GAAGwL,oBAAoB,CAAChG,KAAK,CAAC,CAAA;EACvC,EAAA,IAAI,CAACT,MAAM,CAAC/E,KAAK,EAAE,QAAQ,CAAC,EAAE;EAC5BA,IAAAA,KAAK,CAACsI,MAAM,GAAGvG,IAAI,CAAC+J,QAAQ,EAAE,OAAOhN,IAAI,IAAI,QAAQ,GAAGA,IAAI,GAAG,EAAE,CAAC,CAAA;EACpE,GAAA;EAAE,EAAA,OAAO0G,KAAK,CAAA;EAChB,CAAC,CAAA;;EAED;EACA;EACAtC,QAAQ,CAACW,SAAS,CAACkC,QAAQ,GAAGgG,aAAW,CAAC,SAAShG,QAAQA,GAAG;EAC5D,EAAA,OAAOF,UAAU,CAAC,IAAI,CAAC,IAAI4F,gBAAgB,CAAC,IAAI,CAAC,CAACnD,MAAM,IAAItC,aAAa,CAAC,IAAI,CAAC,CAAA;EACjF,CAAC,EAAE,UAAU,CAAC;;ECrDd,IAAI+F,WAAW,GAAGzI,kBAAqC,CAAA;EACvD,IAAIC,cAAc,GAAGD,oBAA8C,CAAA;EAEnEV,IAAAA,uBAAc,GAAG,UAAUwJ,MAAM,EAAEtN,IAAI,EAAEuN,UAAU,EAAE;IACnD,IAAIA,UAAU,CAAC7I,GAAG,EAAEuI,WAAW,CAACM,UAAU,CAAC7I,GAAG,EAAE1E,IAAI,EAAE;EAAEmN,IAAAA,MAAM,EAAE,IAAA;EAAK,GAAC,CAAC,CAAA;IACvE,IAAII,UAAU,CAACxB,GAAG,EAAEkB,WAAW,CAACM,UAAU,CAACxB,GAAG,EAAE/L,IAAI,EAAE;EAAEoN,IAAAA,MAAM,EAAE,IAAA;EAAK,GAAC,CAAC,CAAA;IACvE,OAAO3I,cAAc,CAACiH,CAAC,CAAC4B,MAAM,EAAEtN,IAAI,EAAEuN,UAAU,CAAC,CAAA;EACnD,CAAC;;ECPD,IAAI5C,QAAQ,GAAGnG,UAAiC,CAAA;;EAEhD;EACA;EACAV,IAAAA,WAAc,GAAG,YAAY;EAC3B,EAAA,IAAI0J,IAAI,GAAG7C,QAAQ,CAAC,IAAI,CAAC,CAAA;IACzB,IAAIJ,MAAM,GAAG,EAAE,CAAA;EACf,EAAA,IAAIiD,IAAI,CAACC,UAAU,EAAElD,MAAM,IAAI,GAAG,CAAA;EAClC,EAAA,IAAIiD,IAAI,CAACtJ,MAAM,EAAEqG,MAAM,IAAI,GAAG,CAAA;EAC9B,EAAA,IAAIiD,IAAI,CAACE,UAAU,EAAEnD,MAAM,IAAI,GAAG,CAAA;EAClC,EAAA,IAAIiD,IAAI,CAACG,SAAS,EAAEpD,MAAM,IAAI,GAAG,CAAA;EACjC,EAAA,IAAIiD,IAAI,CAACI,MAAM,EAAErD,MAAM,IAAI,GAAG,CAAA;EAC9B,EAAA,IAAIiD,IAAI,CAACK,OAAO,EAAEtD,MAAM,IAAI,GAAG,CAAA;EAC/B,EAAA,IAAIiD,IAAI,CAACM,WAAW,EAAEvD,MAAM,IAAI,GAAG,CAAA;EACnC,EAAA,IAAIiD,IAAI,CAACO,MAAM,EAAExD,MAAM,IAAI,GAAG,CAAA;EAC9B,EAAA,OAAOA,MAAM,CAAA;EACf,CAAC;;EChBD,IAAIrG,QAAM,GAAGM,QAA8B,CAAA;EAC3C,IAAI0B,WAAW,GAAG1B,WAAmC,CAAA;EACrD,IAAIwJ,qBAAqB,GAAGxJ,uBAAgD,CAAA;EAC5E,IAAIyJ,WAAW,GAAGzJ,WAAoC,CAAA;EACtD,IAAID,KAAK,GAAGC,OAA6B,CAAA;;EAEzC;EACA,IAAI0J,QAAM,GAAGhK,QAAM,CAACgK,MAAM,CAAA;EAC1B,IAAIC,eAAe,GAAGD,QAAM,CAACnJ,SAAS,CAAA;EAEtC,IAAIqJ,MAAM,GAAGlI,WAAW,IAAI3B,KAAK,CAAC,YAAY;IAC5C,IAAI8J,eAAe,GAAG,IAAI,CAAA;IAC1B,IAAI;EACFH,IAAAA,QAAM,CAAC,GAAG,EAAE,GAAG,CAAC,CAAA;KACjB,CAAC,OAAO5J,KAAK,EAAE;EACd+J,IAAAA,eAAe,GAAG,KAAK,CAAA;EACzB,GAAA;IAEA,IAAInD,CAAC,GAAG,EAAE,CAAA;EACV;IACA,IAAIoD,KAAK,GAAG,EAAE,CAAA;EACd,EAAA,IAAIC,QAAQ,GAAGF,eAAe,GAAG,QAAQ,GAAG,OAAO,CAAA;EAEnD,EAAA,IAAIG,SAAS,GAAG,UAAU1L,GAAG,EAAE2L,GAAG,EAAE;EAClC;EACArM,IAAAA,MAAM,CAACqC,cAAc,CAACyG,CAAC,EAAEpI,GAAG,EAAE;QAAE4B,GAAG,EAAE,YAAY;EAC/C4J,QAAAA,KAAK,IAAIG,GAAG,CAAA;EACZ,QAAA,OAAO,IAAI,CAAA;EACb,OAAA;EAAE,KAAC,CAAC,CAAA;KACL,CAAA;EAED,EAAA,IAAIC,KAAK,GAAG;EACVd,IAAAA,MAAM,EAAE,GAAG;EACX1J,IAAAA,MAAM,EAAE,GAAG;EACXwJ,IAAAA,UAAU,EAAE,GAAG;EACfC,IAAAA,SAAS,EAAE,GAAG;EACdI,IAAAA,MAAM,EAAE,GAAA;KACT,CAAA;EAED,EAAA,IAAIM,eAAe,EAAEK,KAAK,CAACjB,UAAU,GAAG,GAAG,CAAA;EAE3C,EAAA,KAAK,IAAI3K,GAAG,IAAI4L,KAAK,EAAEF,SAAS,CAAC1L,GAAG,EAAE4L,KAAK,CAAC5L,GAAG,CAAC,CAAC,CAAA;;EAEjD;EACA,EAAA,IAAIyH,MAAM,GAAGnI,MAAM,CAACgE,wBAAwB,CAAC+H,eAAe,EAAE,OAAO,CAAC,CAACzJ,GAAG,CAACM,IAAI,CAACkG,CAAC,CAAC,CAAA;EAElF,EAAA,OAAOX,MAAM,KAAKgE,QAAQ,IAAID,KAAK,KAAKC,QAAQ,CAAA;EAClD,CAAC,CAAC,CAAA;;EAEF;EACA;EACA,IAAIH,MAAM,EAAEJ,qBAAqB,CAACG,eAAe,EAAE,OAAO,EAAE;EAC1D1H,EAAAA,YAAY,EAAE,IAAI;EAClB/B,EAAAA,GAAG,EAAEuJ,WAAAA;EACP,CAAC,CAAC;;ECvDF;EACA;EACA;EACA;EACA;;EAEA;EACA,MAAMU,YAAY,GAAG;EACnBC,EAAAA,gBAAgB,EAAE,uCAAuC;EACzDC,EAAAA,cAAc,EAAE,iBAAiB;EACjCC,EAAAA,MAAM,EAAE,8BAA8B;EACtCC,EAAAA,KAAK,EAAE,oIAAoI;EAAE;EAC7IC,EAAAA,WAAW,EAAE,uCAAuC;EACpDC,EAAAA,YAAY,EAAE,oBAAoB;EAClCC,EAAAA,WAAW,EAAE,uBAAuB;EACpCC,EAAAA,WAAW,EAAE,+CAA+C;EAC5DC,EAAAA,MAAM,EAAE,+BAA+B;EACvCC,EAAAA,eAAe,EAAE,kBAAkB;EACnCC,EAAAA,SAAS,EAAE,qBAAqB;EAChCC,EAAAA,aAAa,EAAE,gDAAgD;EAC/DC,EAAAA,YAAY,EAAE,qDAAqD;EACnEC,EAAAA,QAAQ,EAAE,qWAAA;EACZ,CAAC,CAAA;;EAED;EACA,MAAMC,kBAAkB,GAAG,IAAIxB,MAAM,CAAC,qpDAAqpD,CAAC,CAAA;EAE5rD,MAAMyB,mBAAmB,GAAG,IAAIzB,MAAM,CAAC,qpDAAqpD,CAAC,CAAA;;EAE7rD;;EAEA;EACA;EACA;EACA,MAAM0B,WAAW,GAAG;EAClBC,EAAAA,IAAI,EAAE;EACJC,IAAAA,MAAM,EAAE,mCAAmC;EAC3CC,IAAAA,WAAW,EAAE,yFAAA;KACd;EACDC,EAAAA,IAAI,EAAE;EACJF,IAAAA,MAAM,EAAE,oCAAoC;EAC5CC,IAAAA,WAAW,EAAE,yFAAA;KACd;EACDE,EAAAA,IAAI,EAAE;EACJH,IAAAA,MAAM,EAAE,qCAAqC;EAC7CC,IAAAA,WAAW,EAAE,yFAAA;KACd;EACDG,EAAAA,IAAI,EAAE;EACJJ,IAAAA,MAAM,EAAE,sCAAsC;EAC9CC,IAAAA,WAAW,EAAE,yFAAA;KACd;EACDI,EAAAA,IAAI,EAAE;EACJL,IAAAA,MAAM,EAAE,uCAAuC;EAC/CC,IAAAA,WAAW,EAAE,yFAAA;KACd;EACDK,EAAAA,IAAI,EAAE;EACJN,IAAAA,MAAM,EAAE,wCAAwC;EAChDC,IAAAA,WAAW,EAAE,yFAAA;KACd;EACDM,EAAAA,YAAY,EAAE;EACZP,IAAAA,MAAM,EAAE,qBAAqB;EAC7BC,IAAAA,WAAW,EAAE,uDAAA;KACd;EACDO,EAAAA,uBAAuB,EAAE;EACvBR,IAAAA,MAAM,EAAE,yCAAyC;EACjDC,IAAAA,WAAW,EAAE,iGAAA;KACd;EACDQ,EAAAA,oBAAoB,EAAE;EACpBT,IAAAA,MAAM,EAAE,sCAAsC;EAC9CC,IAAAA,WAAW,EAAE,8FAAA;KACd;EACDS,EAAAA,wBAAwB,EAAE;EACxBV,IAAAA,MAAM,EAAE,8BAA8B;EACtCC,IAAAA,WAAW,EAAE,6DAAA;KACd;EACDU,EAAAA,qBAAqB,EAAE;EACrBX,IAAAA,MAAM,EAAE,8BAA8B;EACtCC,IAAAA,WAAW,EAAE,0DAAA;KACd;EACDW,EAAAA,WAAW,EAAE;EACXZ,IAAAA,MAAM,EAAE,YAAY;EACpBC,IAAAA,WAAW,EAAE,IAAA;KACd;EACDY,EAAAA,gBAAgB,EAAE;EAChBb,IAAAA,MAAM,EAAE,eAAe;EACvBC,IAAAA,WAAW,EAAE,wDAAA;KACd;EACDa,EAAAA,gBAAgB,EAAE;EAChBd,IAAAA,MAAM,EAAE,eAAe;EACvBC,IAAAA,WAAW,EAAE,wDAAA;KACd;EACDc,EAAAA,IAAI,EAAE;EACJf,IAAAA,MAAM,EAAE,yFAAyF;EACjGC,IAAAA,WAAW,EAAE,4CAAA;KACd;EACDe,EAAAA,IAAI,EAAE;EACJhB,IAAAA,MAAM,EAAE,2BAA2B;EACnCC,IAAAA,WAAW,EAAE,+CAAA;KACd;EACDgB,EAAAA,IAAI,EAAE;EACJjB,IAAAA,MAAM,EAAE,iCAAiC;EACzCC,IAAAA,WAAW,EAAE,+CAAA;KACd;EACD;EACAiB,EAAAA,cAAc,EAAE;EACdlB,IAAAA,MAAM,EAAE,iBAAiB;EACzBC,IAAAA,WAAW,EAAE,wDAAA;KACd;EACDkB,EAAAA,yBAAyB,EAAE;EACzB;EACAnB,IAAAA,MAAM,EAAE,4KAA4K;EACpLC,IAAAA,WAAW,EAAE,0QAA0Q;MACvRmB,gBAAgB,EAAE,CAAC;EACrB,GAAA;EACF,CAAC,CAAA;;EAED;EACA;EACA;EACA,IAAIC,gBAAgB,GAAG,CACrB;EAAEC,EAAAA,KAAK,EAAE,yCAAyC;EAAEC,EAAAA,GAAG,EAAE,mCAAmC;EAAEC,EAAAA,aAAa,EAAE,IAAA;EAAK,CAAC,EACnH;EAAEF,EAAAA,KAAK,EAAE,aAAa;EAAEC,EAAAA,GAAG,EAAE,KAAK;EAAEC,EAAAA,aAAa,EAAE,IAAA;EAAK,CAAC,EACzD;EAAEF,EAAAA,KAAK,EAAE,YAAY;EAAEC,EAAAA,GAAG,EAAE,KAAK;EAAEC,EAAAA,aAAa,EAAE,IAAA;EAAK,CAAC,EACxD;EAAEF,EAAAA,KAAK,EAAE,gBAAgB;EAAEC,EAAAA,GAAG,EAAE,GAAG;EAAEC,EAAAA,aAAa,EAAG,IAAA;EAAI,CAAC,EAC1D;EAAEF,EAAAA,KAAK,EAAE,oBAAoB;EAAEC,EAAAA,GAAG,EAAE,OAAO;EAAEC,EAAAA,aAAa,EAAG,IAAA;EAAI,CAAC,EAClE;EAAEF,EAAAA,KAAK,EAAE,6CAA6C;EAAEC,EAAAA,GAAG,EAAE,KAAK;EAAEC,EAAAA,aAAa,EAAE,IAAA;EAAI,CAAC,EACxF;EAAEF,EAAAA,KAAK,EAAE,yCAAyC;EAAEC,EAAAA,GAAG,EAAE,KAAK;EAAEC,EAAAA,aAAa,EAAE,KAAA;EAAK,CAAC,CACtF,CAAA;;EAED;EACA;EACA;EACA;EACA;EACA,IAAIC,aAAa,GAAG;EAClBC,EAAAA,MAAM,EAAG;EACP1B,IAAAA,MAAM,EAAE,uBAAuB;EAC/BC,IAAAA,WAAW,EAAG,kDAAA;KACf;EACD9Q,EAAAA,IAAI,EAAG;EACL6Q,IAAAA,MAAM,EAAE,qCAAqC;EAC7CC,IAAAA,WAAW,EAAG,wHAAA;KACf;EACD0B,EAAAA,QAAQ,EAAG;EACT3B,IAAAA,MAAM,EAAE,oCAAoC;EAC5CC,IAAAA,WAAW,EAAE,wIAAA;KACd;EACD2B,EAAAA,IAAI,EAAG;EACL5B,IAAAA,MAAM,EAAE,kGAAkG;EAC1GC,IAAAA,WAAW,EAAE,gCAAA;KACd;EACD4B,EAAAA,QAAQ,EAAG;EACT7B,IAAAA,MAAM,EAAE,KAAK;EACbC,IAAAA,WAAW,EAAE,EAAA;KACd;EACD6B,EAAAA,SAAS,EAAG;EACV9B,IAAAA,MAAM,EAAE,MAAM;EACdC,IAAAA,WAAW,EAAG,EAAA;KACf;EACD8B,EAAAA,SAAS,EAAG;EACV/B,IAAAA,MAAM,EAAE,0BAA0B;EAClCC,IAAAA,WAAW,EAAE,EAAE;EACfmB,IAAAA,gBAAgB,EAAE,CAAA;KACnB;EACDY,EAAAA,OAAO,EAAG;EACRhC,IAAAA,MAAM,EAAE,0BAA0B;EAClCC,IAAAA,WAAW,EAAE,IAAA;EACf,GAAA;EACF,CAAC,CAAA;;EAED;EACA,MAAMgC,iBAAiB,GAAG,IAAI7D,MAAM,CAAC9L,MAAM,CAACE,IAAI,CAACqM,YAAY,CAAC,CAAC1L,IAAI,CAAC,GAAG,CAAC,CAAC,CAAA;;EAEzE;EACA,MAAM+O,WAAW,GAAE,CAAC,GAAG5P,MAAM,CAACE,IAAI,CAACiP,aAAa,CAAC,CAAC,CAAA;EAClD,KAAK,IAAIU,IAAI,IAAID,WAAW,EAAE;IAC5B,IAAIE,EAAE,GAAGX,aAAa,CAACU,IAAI,CAAC,CAACnC,MAAM,CAACtG,MAAM,CAAA;EAC1C;EACA,EAAA,OAAO0I,EAAE,CAACrP,KAAK,CAACkP,iBAAiB,CAAC,EAAE;MAClCG,EAAE,GAAGA,EAAE,CAACpF,OAAO,CAACiF,iBAAiB,EAAGI,MAAM,IAAK;EAAE,MAAA,OAAOxD,YAAY,CAACwD,MAAM,CAAC,CAAC3I,MAAM,CAAA;EAAE,KAAC,CAAC,CAAA;EACzF,GAAA;EACA+H,EAAAA,aAAa,CAACU,IAAI,CAAC,CAACnC,MAAM,GAAG,IAAI5B,MAAM,CAACgE,EAAE,EAAEX,aAAa,CAACU,IAAI,CAAC,CAACnC,MAAM,CAACsC,KAAK,CAAC,CAAA;EAC/E,CAAA;;EAEA;EACA,KAAK,IAAIH,IAAI,IAAId,gBAAgB,EAAE;EACjC,EAAA,IAAIe,EAAE,GAAGD,IAAI,CAACb,KAAK,CAAC5H,MAAM,CAAA;EAC1B;EACA,EAAA,OAAO0I,EAAE,CAACrP,KAAK,CAACkP,iBAAiB,CAAC,EAAE;MAClCG,EAAE,GAAGA,EAAE,CAACpF,OAAO,CAACiF,iBAAiB,EAAGI,MAAM,IAAK;EAAE,MAAA,OAAOxD,YAAY,CAACwD,MAAM,CAAC,CAAC3I,MAAM,CAAA;EAAE,KAAC,CAAC,CAAA;EACzF,GAAA;EACAyI,EAAAA,IAAI,CAACb,KAAK,GAAG,IAAIlD,MAAM,CAACgE,EAAE,EAAED,IAAI,CAACb,KAAK,CAACgB,KAAK,CAAC,CAAA;EAC/C,CAAA;;EAEA;EACA;EACA;EACA;EACA;EACA,SAASC,UAAUA,CAACF,MAAM,EAAE;IAC1B,OAAO,CAACA,MAAM,GAAGA,MAAM,GAAG,EAAE,EACzBrF,OAAO,CAAC,IAAI,EAAE,OAAO,CAAC,CACtBA,OAAO,CAAC,IAAI,EAAE,MAAM,CAAC,CACrBA,OAAO,CAAC,IAAI,EAAE,MAAM,CAAC,CAAA;EAC1B,CAAA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA,MAAM9L,QAAQ,GAAG;EACf;EACAjC,EAAAA,IAAI,EAAE;EACJqN,IAAAA,IAAI,EAAE,QAAQ;EACdrK,IAAAA,SAAS,EAAE,UAAU;EACrBgK,IAAAA,GAAG,EAAE;EAACuG,MAAAA,GAAG,EAAE,IAAI;EAAEC,MAAAA,IAAI,EAAE,IAAA;OAAK;EAC5BC,IAAAA,KAAK,EAAE;EAACC,MAAAA,UAAU,EAAE,cAAc;EAAEC,MAAAA,WAAW,EAAE,cAAA;EAAc,KAAA;KAChE;EACDpT,EAAAA,MAAM,EAAE;EACN8M,IAAAA,IAAI,EAAE,QAAQ;EACdrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAACuG,MAAAA,GAAG,EAAE,GAAG;EAAEC,MAAAA,IAAI,EAAE,GAAA;OAAI;EAC1BC,IAAAA,KAAK,EAAE;EAACC,MAAAA,UAAU,EAAE,WAAW;EAAEC,MAAAA,WAAW,EAAE,WAAA;EAAW,KAAA;KAC1D;EACDzT,EAAAA,IAAI,EAAE;EACJmN,IAAAA,IAAI,EAAE,QAAQ;EACdrK,IAAAA,SAAS,EAAE,QAAQ;EACnBgK,IAAAA,GAAG,EAAE;EAACuG,MAAAA,GAAG,EAAE,GAAG;EAAEC,MAAAA,IAAI,EAAE,GAAA;OAAI;EAC1BC,IAAAA,KAAK,EAAE;EAACC,MAAAA,UAAU,EAAE,KAAK;EAAEC,MAAAA,WAAW,EAAE,KAAA;EAAK,KAAC;KAC/C;;EACDjT,EAAAA,aAAa,EAAE;EACb2M,IAAAA,IAAI,EAAE,QAAQ;EACdrK,IAAAA,SAAS,EAAE,iBAAiB;EAC5BgK,IAAAA,GAAG,EAAE;EAACuG,MAAAA,GAAG,EAAE,IAAI;EAAEC,MAAAA,IAAI,EAAE,IAAA;OAAK;EAC5BC,IAAAA,KAAK,EAAE;EAACC,MAAAA,UAAU,EAAC,KAAK;EAAEC,MAAAA,WAAW,EAAE,KAAA;EAAM,KAAA;KAC9C;EACDxT,EAAAA,EAAE,EAAE;EACFkN,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,MAAA;OAAO;EAC1FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,oCAAoC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KACzE;EACD5Q,EAAAA,EAAE,EAAE;EACFiN,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,OAAA;OAAQ;EAC3FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,qCAAqC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KAC1E;EACD6C,EAAAA,EAAE,EAAE;EACFxG,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,QAAA;OAAS;EAC5FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,sCAAsC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KAC3E;EACD8C,EAAAA,EAAE,EAAE;EACFzG,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,SAAA;OAAU;EAC7FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,uCAAuC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KAC5E;EACD+C,EAAAA,EAAE,EAAE;EACF1G,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,UAAA;OAAW;EAC9FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,wCAAwC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KAC7E;EACDgD,EAAAA,EAAE,EAAE;EACF3G,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,WAAA;OAAY;EAC/FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,yCAAyC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KAC9E;EACDrQ,EAAAA,EAAE,EAAE;EACF0M,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,MAAA;OAAO;EAC1FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,2BAA2B;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KAChE;EACDvQ,EAAAA,EAAE,EAAE;EACF4M,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,MAAM;EACjBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,QAAA;OAAS;EAC5FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,iCAAiC;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;KACtE;EACDjR,EAAAA,UAAU,EAAE;EACVsN,IAAAA,IAAI,EAAE,MAAM;EACZrK,IAAAA,SAAS,EAAE,cAAc;EACzBgK,IAAAA,GAAG,EAAE;EAAC4G,MAAAA,OAAO,EAAE,qDAAqD;EAAE5C,MAAAA,WAAW,EAAE,MAAA;OAAO;EAC1FyC,IAAAA,KAAK,EAAE;EAACG,MAAAA,OAAO,EAAE,qBAAqB;EAAE5C,MAAAA,WAAW,EAAE,IAAA;EAAI,KAAA;EAC3D,GAAA;EACF,CAAC;;EC/RD,MAAMiD,MAAM,CAAC;EACXnS,EAAAA,WAAWA,GAAa;EAAA,IAAA,IAAZC,KAAK,GAAAsE,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,EAAE,CAAA;MACpB,IAAI,CAACrE,CAAC,GAAG,IAAI,CAAA;MACb,IAAI,CAACkS,QAAQ,GAAG,IAAI,CAAA;MACpB,IAAI,CAACC,KAAK,GAAG,EAAE,CAAA;MACf,IAAI,CAACC,YAAY,GAAG,EAAE,CAAA;MACtB,IAAI,CAACC,SAAS,GAAG,EAAE,CAAA;MACnB,IAAI,CAACC,YAAY,GAAG,EAAE,CAAA;MACtB,IAAI,CAACC,gBAAgB,GAAG,EAAE,CAAA;MAC1B,IAAI,CAACC,UAAU,GAAG,EAAE,CAAA;MACpB,IAAI,CAACC,SAAS,GAAG,EAAE,CAAA;MACnB,IAAI,CAACC,gBAAgB,GAAG,IAAI,CAAA;MAE5B,IAAI,CAACC,SAAS,GAAG;EACfC,MAAAA,MAAM,EAAE,EAAE;EACVC,MAAAA,SAAS,EAAE,EAAE;EACbC,MAAAA,IAAI,EAAE,EAAA;OACP,CAAA;EAED,IAAA,IAAIzS,OAAO,GAAGN,KAAK,CAACM,OAAO,CAAA;EAC3B,IAAA,IAAI,CAAC6R,QAAQ,GAAGnS,KAAK,CAACmS,QAAQ,CAAA;MAE9B,IAAI,IAAI,CAACA,QAAQ,EAAE;EACjB,MAAA,IAAI,CAAC,IAAI,CAACA,QAAQ,CAAC5R,OAAO,EAAE;UAC1B,IAAI,CAAC4R,QAAQ,GAAG3R,QAAQ,CAACC,cAAc,CAAC,IAAI,CAAC0R,QAAQ,CAAC,CAAA;EACxD,OAAA;EACA,MAAA,IAAI,CAAC7R,OAAO,EAAEA,OAAO,GAAG,IAAI,CAAC6R,QAAQ,CAAA;EACvC,KAAA;EAEA,IAAA,IAAI7R,OAAO,IAAI,CAACA,OAAO,CAACC,OAAO,EAAE;QAC/BD,OAAO,GAAGE,QAAQ,CAACC,cAAc,CAACT,KAAK,CAACM,OAAO,CAAC,CAAA;EAClD,KAAA;MACA,IAAI,CAACA,OAAO,EAAE;QACZA,OAAO,GAAGE,QAAQ,CAACwS,oBAAoB,CAAC,MAAM,CAAC,CAAC,CAAC,CAAC,CAAA;EACpD,KAAA;EACA,IAAA,IAAI1S,OAAO,CAACC,OAAO,IAAI,UAAU,EAAE;QACjC,IAAI,CAAC4R,QAAQ,GAAG7R,OAAO,CAAA;EACvBA,MAAAA,OAAO,GAAG,IAAI,CAAC6R,QAAQ,CAACc,UAAU,CAAA;EACpC,KAAA;MAEA,IAAI,IAAI,CAACd,QAAQ,EAAE;EACjB,MAAA,IAAI,CAACA,QAAQ,CAACe,KAAK,CAACC,OAAO,GAAG,MAAM,CAAA;EACtC,KAAA;EAEA,IAAA,IAAI,CAACC,mBAAmB,CAAC9S,OAAO,EAAEN,KAAK,CAAC,CAAA;MACxC,IAAI,CAACqT,UAAU,CACb,OAAOrT,KAAK,CAACsT,OAAO,KAAK,QAAQ,GAC7BtT,KAAK,CAACsT,OAAO,GACb,IAAI,CAACnB,QAAQ,GACb,IAAI,CAACA,QAAQ,CAACvM,KAAK,GACnB,iCACN,CAAC,CAAA;EACH,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACEwN,EAAAA,mBAAmBA,CAAC9S,OAAO,EAAEN,KAAK,EAAE;EAClC,IAAA,IAAIA,KAAK,IAAIA,KAAK,CAACT,MAAM,KAAKmD,SAAS,EAAE;QACvC,IAAI1C,KAAK,CAACT,MAAM,CAACgB,OAAO,EAAE,IAAI,CAACN,CAAC,GAAGD,KAAK,CAACT,MAAM,CAAC,KAC3C,IAAI,CAACU,CAAC,GAAGO,QAAQ,CAACC,cAAc,CAACT,KAAK,CAACT,MAAM,CAAC,CAAA;OACpD,MAAM,IAAI,CAACU,CAAC,GAAGO,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAA;MAE7C,IAAI,CAACf,CAAC,CAACsT,SAAS,CAACC,GAAG,CAAC,SAAS,CAAC,CAAA;EAC/B,IAAA,IAAI,CAACvT,CAAC,CAACwT,eAAe,GAAG,IAAI,CAAA;EAC7B;EACA;EACA,IAAA,IAAI,CAACxT,CAAC,CAACiT,KAAK,CAACQ,UAAU,GAAG,UAAU,CAAA;EACpC;EACA,IAAA,IAAI,CAACzT,CAAC,CAACiT,KAAK,CAACS,gBAAgB,GAAG,2BAA2B,CAAA;EAE3D,IAAA,IAAI3T,KAAK,CAACT,MAAM,KAAKmD,SAAS,EAAE;EAC9B,MAAA,IACE,IAAI,CAACyP,QAAQ,IACb,IAAI,CAACA,QAAQ,CAACc,UAAU,IAAI3S,OAAO,IACnC,IAAI,CAAC6R,QAAQ,CAACyB,WAAW,EACzB;EACAtT,QAAAA,OAAO,CAACuT,YAAY,CAAC,IAAI,CAAC5T,CAAC,EAAE,IAAI,CAACkS,QAAQ,CAACyB,WAAW,CAAC,CAAA;EACzD,OAAC,MAAM;EACLtT,QAAAA,OAAO,CAACc,WAAW,CAAC,IAAI,CAACnB,CAAC,CAAC,CAAA;EAC7B,OAAA;EACF,KAAA;EAEA,IAAA,IAAI,CAACA,CAAC,CAACW,gBAAgB,CAAC,OAAO,EAAGX,CAAC,IAAK,IAAI,CAAC6T,gBAAgB,CAAC7T,CAAC,CAAC,CAAC,CAAA;EACjE,IAAA,IAAI,CAACA,CAAC,CAACW,gBAAgB,CAAC,gBAAgB,EAAGX,CAAC,IAAK,IAAI,CAAC6T,gBAAgB,CAAC7T,CAAC,CAAC,CAAC,CAAA;EAC1EO,IAAAA,QAAQ,CAACI,gBAAgB,CAAC,iBAAiB,EAAGX,CAAC,IAC7C,IAAI,CAAC8T,0BAA0B,CAAC9T,CAAC,CACnC,CAAC,CAAA;EACD,IAAA,IAAI,CAACA,CAAC,CAACW,gBAAgB,CAAC,OAAO,EAAGX,CAAC,IAAK,IAAI,CAAC+T,WAAW,CAAC/T,CAAC,CAAC,CAAC,CAAA;EAC5D,IAAA,IAAI,CAACA,CAAC,CAACW,gBAAgB,CAAC,MAAM,EAAGX,CAAC,IAAK,IAAI,CAACgU,UAAU,CAAChU,CAAC,CAAC,CAAC,CAAA;MAC1D,IAAI,CAACoS,YAAY,GAAG,IAAI,CAACpS,CAAC,CAACiU,UAAU,CAAC;EACxC,GAAA;;EAEA;EACF;EACA;EACA;IACEb,UAAUA,CAACC,OAAO,EAAE;EAClB;EACA,IAAA,OAAO,IAAI,CAACrT,CAAC,CAACkU,UAAU,EAAE;QACxB,IAAI,CAAClU,CAAC,CAACmU,WAAW,CAAC,IAAI,CAACnU,CAAC,CAACkU,UAAU,CAAC,CAAA;EACvC,KAAA;MACA,IAAI,CAAC/B,KAAK,GAAGkB,OAAO,CAAC7R,KAAK,CAAC,gBAAgB,CAAC,CAAA;MAC5C,IAAI,CAACiR,SAAS,GAAG,EAAE,CAAA;EACnB,IAAA,KAAK,IAAI2B,OAAO,GAAG,CAAC,EAAEA,OAAO,GAAG,IAAI,CAACjC,KAAK,CAACvQ,MAAM,EAAEwS,OAAO,EAAE,EAAE;EAC5D,MAAA,IAAIC,EAAE,GAAG9T,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAA;EACtC,MAAA,IAAI,CAACf,CAAC,CAACmB,WAAW,CAACkT,EAAE,CAAC,CAAA;EACtB,MAAA,IAAI,CAAC5B,SAAS,CAAC5Q,IAAI,CAAC,IAAI,CAAC,CAAA;EAC3B,KAAA;MACA,IAAI,CAACwQ,SAAS,GAAG,IAAIiC,KAAK,CAAC,IAAI,CAACnC,KAAK,CAACvQ,MAAM,CAAC,CAAA;MAC7C,IAAI,CAAC2S,gBAAgB,EAAE,CAAA;MACvB,IAAI,CAACC,UAAU,EAAE,CAAA;EACnB,GAAA;;EAEA;EACF;EACA;EACA;EACEC,EAAAA,UAAUA,GAAG;EACX,IAAA,OAAO,IAAI,CAACtC,KAAK,CAACjQ,IAAI,CAAC,IAAI,CAAC,CAAA;EAC9B,GAAA;;EAEA;EACF;EACA;EACEqS,EAAAA,gBAAgBA,GAAG;EACjB;EACA;MACA,IAAI,CAACG,eAAe,EAAE,CAAA;EACtB;MACA,IAAI,CAACC,gBAAgB,EAAE,CAAA;EACvB;MACA,IAAI,CAACC,cAAc,EAAE,CAAA;EACvB,GAAA;;EAEA;EACF;EACA;EACED,EAAAA,gBAAgBA,GAAG;MACjB,IAAI,CAACnC,UAAU,GAAG,EAAE,CAAA;EACpB,IAAA,KAAK,IAAIqC,CAAC,GAAG,CAAC,EAAEA,CAAC,GAAG,IAAI,CAAC1C,KAAK,CAACvQ,MAAM,EAAEiT,CAAC,EAAE,EAAE;QAC1C,IAAI,IAAI,CAACxC,SAAS,CAACwC,CAAC,CAAC,IAAI,2BAA2B,EAAE;EACpD,QAAA,IAAI,CAACrC,UAAU,CAAC3Q,IAAI,CAClB,IAAI,CAACyQ,YAAY,CAACuC,CAAC,CAAC,CAClBhG,WAAW,CAACqB,yBAAyB,CAACC,gBAAgB,CAE1D,CAAC,CAAA;EACH,OAAA;EACF,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACEpE,EAAAA,OAAOA,CAACiD,WAAW,EAAE8F,OAAO,EAAE;EAC5B,IAAA,OAAO9F,WAAW,CAACjD,OAAO,CAAC,mBAAmB,EAAE,CAACgJ,GAAG,EAAEC,EAAE,EAAEC,EAAE,KAAK;QAC/D,IAAID,EAAE,IAAI,GAAG,EAAE,OAAO1D,UAAU,CAACwD,OAAO,CAACG,EAAE,CAAC,CAAC,CAAC,KAE5C,OAAQ,CAAkC,gCAAA,EAAA,IAAI,CAACC,mBAAmB,CAChEJ,OAAO,CAACG,EAAE,CACZ,CAAE,CAAQ,OAAA,CAAA,CAAA;EACd,KAAC,CAAC,CAAA;EACJ,GAAA;;EAEA;EACF;EACA;EACA;EACEL,EAAAA,cAAcA,GAAG;EACf,IAAA,KAAK,IAAIR,OAAO,GAAG,CAAC,EAAEA,OAAO,GAAG,IAAI,CAACjC,KAAK,CAACvQ,MAAM,EAAEwS,OAAO,EAAE,EAAE;EAC5D,MAAA,IAAI,IAAI,CAAC3B,SAAS,CAAC2B,OAAO,CAAC,EAAE;EAC3B,QAAA,IAAIe,WAAW,GAAG,IAAI,CAACpJ,OAAO,CAC5B,IAAI,CAACwG,gBAAgB,CAAC6B,OAAO,CAAC,EAC9B,IAAI,CAAC9B,YAAY,CAAC8B,OAAO,CAC3B,CAAC,CAAA;EACD;EACA,QAAA,IAAI,CAAChC,YAAY,CAACgC,OAAO,CAAC,CAACpT,SAAS,GAAG,IAAI,CAACqR,SAAS,CAAC+B,OAAO,CAAC,CAAA;UAC9D,IAAI,CAAChC,YAAY,CAACgC,OAAO,CAAC,CAACgB,eAAe,CAAC,OAAO,CAAC,CAAA;EACnD,QAAA,IAAI,CAAChD,YAAY,CAACgC,OAAO,CAAC,CAACjV,SAAS,GAClCgW,WAAW,IAAI,EAAE,GAAG,QAAQ,GAAGA,WAAW,CAAC;EAC/C,OAAA;;QACA,IAAI,CAAC/C,YAAY,CAACgC,OAAO,CAAC,CAACiB,OAAO,CAACjB,OAAO,GAAGA,OAAO,CAAA;EACtD,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACEM,EAAAA,eAAeA,GAAG;MAChB,IAAIY,aAAa,GAAG,KAAK,CAAA;MACzB,IAAIC,kBAAkB,GAAG,CAAC,CAAA;MAC1B,IAAIC,SAAS,GAAG,KAAK,CAAA;EAErB,IAAA,KAAK,IAAIpB,OAAO,GAAG,CAAC,EAAEA,OAAO,GAAG,IAAI,CAACjC,KAAK,CAACvQ,MAAM,EAAEwS,OAAO,EAAE,EAAE;QAC5D,IAAIqB,QAAQ,GAAG,QAAQ,CAAA;QACvB,IAAIC,WAAW,GAAG,CAAC,IAAI,CAACvD,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EACvC,MAAA,IAAIuB,eAAe,GAAG,KAAK,CAAC;;EAE5B;EACA;QACA,IAAIL,aAAa,IAAI,yBAAyB,EAAE;EAC9C;EACA,QAAA,IAAIR,OAAO,GAAGjG,WAAW,CAACY,wBAAwB,CAACV,MAAM,CAACzL,IAAI,CAC5D,IAAI,CAAC6O,KAAK,CAACiC,OAAO,CACpB,CAAC,CAAA;EACD,QAAA,IAAIU,OAAO,IAAIA,OAAO,CAACc,MAAM,CAAC,KAAK,CAAC,CAAChU,MAAM,IAAI2T,kBAAkB,EAAE;EACjEE,UAAAA,QAAQ,GAAG,0BAA0B,CAAA;EACrCE,UAAAA,eAAe,GAAG9G,WAAW,CAACY,wBAAwB,CAACT,WAAW,CAAA;EAClE0G,UAAAA,WAAW,GAAGZ,OAAO,CAAA;EACrBQ,UAAAA,aAAa,GAAG,KAAK,CAAA;EACvB,SAAC,MAAM;EACLG,UAAAA,QAAQ,GAAG,sBAAsB,CAAA;EACjCE,UAAAA,eAAe,GAAG,4CAA4C,CAAA;YAC9DD,WAAW,GAAG,CAAC,IAAI,CAACvD,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EACrC,SAAA;EACF,OAAA;EACA;EAAA,WACK,IAAIkB,aAAa,IAAI,sBAAsB,EAAE;EAChD;EACA,QAAA,IAAIR,OAAO,GAAGjG,WAAW,CAACa,qBAAqB,CAACX,MAAM,CAACzL,IAAI,CACzD,IAAI,CAAC6O,KAAK,CAACiC,OAAO,CACpB,CAAC,CAAA;EACD,QAAA,IAAIU,OAAO,IAAIA,OAAO,CAACc,MAAM,CAAC,KAAK,CAAC,CAAChU,MAAM,IAAI2T,kBAAkB,EAAE;EACjEE,UAAAA,QAAQ,GAAG,uBAAuB,CAAA;EAClCE,UAAAA,eAAe,GAAG9G,WAAW,CAACa,qBAAqB,CAACV,WAAW,CAAA;EAC/D0G,UAAAA,WAAW,GAAGZ,OAAO,CAAA;EACrBQ,UAAAA,aAAa,GAAG,KAAK,CAAA;EACvB,SAAC,MAAM;EACLG,UAAAA,QAAQ,GAAG,mBAAmB,CAAA;EAC9BE,UAAAA,eAAe,GAAG,4CAA4C,CAAA;YAC9DD,WAAW,GAAG,CAAC,IAAI,CAACvD,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EACrC,SAAA;EACF,OAAA;;EAEA;EACA,MAAA,IAAIqB,QAAQ,IAAI,QAAQ,IAAID,SAAS,KAAK,KAAK,EAAE;EAC/C,QAAA,KAAK,IAAIK,aAAa,IAAIzF,gBAAgB,EAAE;EAC1C,UAAA,IAAI,IAAI,CAAC+B,KAAK,CAACiC,OAAO,CAAC,CAACtS,KAAK,CAAC+T,aAAa,CAACxF,KAAK,CAAC,EAAE;EAClD;cACA,IACEwF,aAAa,CAACtF,aAAa,IAC3B6D,OAAO,IAAI,CAAC,IACZ,EACE,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,QAAQ,IACvC,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,MAAM,IACrC,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,MAAM,IACrC,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,cAAc,CAC9C,EACD;EACAoB,cAAAA,SAAS,GAAGK,aAAa,CAAA;EACzB,cAAA,MAAA;EACF,aAAA;EACF,WAAA;EACF,SAAA;EACF,OAAA;QAEA,IAAIL,SAAS,KAAK,KAAK,EAAE;EACvBC,QAAAA,QAAQ,GAAG,aAAa,CAAA;UACxBE,eAAe,GAAG,6CAA6C,CAAC;UAChED,WAAW,GAAG,CAAC,IAAI,CAACvD,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAC;;EAEpC;UACA,IAAIoB,SAAS,CAAClF,GAAG,EAAE;EACjB;EACA,UAAA,IAAI,IAAI,CAAC6B,KAAK,CAACiC,OAAO,CAAC,CAACtS,KAAK,CAAC0T,SAAS,CAAClF,GAAG,CAAC,EAAE;EAC5CkF,YAAAA,SAAS,GAAG,KAAK,CAAA;EACnB,WAAA;EACF,SAAC,MAAM;EACL;YACA,IACEpB,OAAO,IAAI,IAAI,CAACjC,KAAK,CAACvQ,MAAM,GAAG,CAAC,IAChC,IAAI,CAACuQ,KAAK,CAACiC,OAAO,GAAG,CAAC,CAAC,CAACtS,KAAK,CAAC+M,WAAW,CAACc,WAAW,CAACZ,MAAM,CAAC,EAC7D;EACAyG,YAAAA,SAAS,GAAG,KAAK,CAAA;EACnB,WAAA;EACF,SAAA;EACF,OAAA;;EAEA;QACA,IAAIC,QAAQ,IAAI,QAAQ,EAAE;EACxB,QAAA,KAAK,IAAIpK,IAAI,IAAIwD,WAAW,EAAE;EAC5B,UAAA,IAAIA,WAAW,CAACxD,IAAI,CAAC,CAAC0D,MAAM,EAAE;EAC5B,YAAA,IAAI+F,OAAO,GAAGjG,WAAW,CAACxD,IAAI,CAAC,CAAC0D,MAAM,CAACzL,IAAI,CAAC,IAAI,CAAC6O,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EAChE,YAAA,IAAIU,OAAO,EAAE;EACXW,cAAAA,QAAQ,GAAGpK,IAAI,CAAA;EACfsK,cAAAA,eAAe,GAAG9G,WAAW,CAACxD,IAAI,CAAC,CAAC2D,WAAW,CAAA;EAC/C0G,cAAAA,WAAW,GAAGZ,OAAO,CAAA;EACrB,cAAA,MAAA;EACF,aAAA;EACF,WAAA;EACF,SAAA;EACF,OAAA;;EAEA;EACA,MAAA,IACEW,QAAQ,IAAI,yBAAyB,IACrCA,QAAQ,IAAI,sBAAsB,EAClC;EACAH,QAAAA,aAAa,GAAGG,QAAQ,CAAA;UACxBF,kBAAkB,GAAGG,WAAW,CAACE,MAAM,CAAC,KAAK,CAAC,CAAChU,MAAM,CAAA;EACvD,OAAA;;EAEA;QACA,IACE,CAAC6T,QAAQ,IAAI,gBAAgB,IAC3BA,QAAQ,IAAI,2BAA2B,KACzCrB,OAAO,GAAG,CAAC,KACV,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,QAAQ,IACtC,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,MAAM,IACrC,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,MAAM,IACrC,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,cAAc,CAAC,EAChD;EACA;EACAqB,QAAAA,QAAQ,GAAG,QAAQ,CAAA;UACnBC,WAAW,GAAG,CAAC,IAAI,CAACvD,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EACnCuB,QAAAA,eAAe,GAAG,KAAK,CAAA;EACzB,OAAA;;EAEA;QACA,IAAIF,QAAQ,IAAI,kBAAkB,EAAE;EAClC,QAAA,IAAIX,OAAO,GAAGjG,WAAW,CAACkB,IAAI,CAAChB,MAAM,CAACzL,IAAI,CAAC,IAAI,CAAC6O,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EAC/D,QAAA,IAAIU,OAAO,EAAE;EACXW,UAAAA,QAAQ,GAAG,MAAM,CAAA;EACjBE,UAAAA,eAAe,GAAG9G,WAAW,CAACkB,IAAI,CAACf,WAAW,CAAA;EAC9C0G,UAAAA,WAAW,GAAGZ,OAAO,CAAA;EACvB,SAAA;EACF,OAAA;;EAEA;EACA,MAAA,IAAIW,QAAQ,IAAI,kBAAkB,IAAIA,QAAQ,IAAI,kBAAkB,EAAE;EACpE,QAAA,IAAIrB,OAAO,IAAI,CAAC,IAAI,IAAI,CAAC/B,SAAS,CAAC+B,OAAO,GAAG,CAAC,CAAC,IAAI,QAAQ,EAAE;EAC3D;EACA,UAAA,IAAIU,OAAO,GAAGjG,WAAW,CAACiB,IAAI,CAACf,MAAM,CAACzL,IAAI,CAAC,IAAI,CAAC6O,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EAC/D,UAAA,IAAIU,OAAO,EAAE;EACX;EACAW,YAAAA,QAAQ,GAAG,MAAM,CAAA;EACjBC,YAAAA,WAAW,GAAGZ,OAAO,CAAA;EACrBa,YAAAA,eAAe,GAAG9G,WAAW,CAACiB,IAAI,CAACd,WAAW,CAAA;EAChD,WAAC,MAAM;EACL;EACAyG,YAAAA,QAAQ,GAAG,QAAQ,CAAA;cACnBC,WAAW,GAAG,CAAC,IAAI,CAACvD,KAAK,CAACiC,OAAO,CAAC,CAAC,CAAA;EACnCuB,YAAAA,eAAe,GAAG,KAAK,CAAA;EACzB,WAAA;EACF,SAAC,MAAM;EACL;EACA,UAAA,IAAIG,WAAW,GAAG1B,OAAO,GAAG,CAAC,CAAA;YAC7B,MAAM2B,eAAe,GACnBN,QAAQ,IAAI,kBAAkB,GAAG,YAAY,GAAG,YAAY,CAAA;YAC9D,GAAG;cACD,IAAI,IAAI,CAACpD,SAAS,CAAC0D,eAAe,CAAC,IAAIA,eAAe,EAAE;EACtD,cAAA,IAAI,CAAC1D,SAAS,CAACyD,WAAW,CAAC,GAAGC,eAAe,CAAA;EAC7C,cAAA,IAAI,CAACtD,SAAS,CAACsD,eAAe,CAAC,GAAG,IAAI,CAAA;EACxC,aAAA;EACA,YAAA,IAAI,CAACxD,gBAAgB,CAACuD,WAAW,CAAC,GAAG,KAAK,CAAA;EAC1C,YAAA,IAAI,CAACxD,YAAY,CAACwD,WAAW,CAAC,GAAG,CAAC,IAAI,CAAC3D,KAAK,CAAC2D,WAAW,CAAC,CAAC,CAAA;EAE1DA,YAAAA,WAAW,EAAE,CAAA;EACf,WAAC,QAAQA,WAAW,IAAI,CAAC,IAAI,IAAI,CAACzD,SAAS,CAACyD,WAAW,CAAC,IAAI,QAAQ,EAAA;EACtE,SAAA;EACF,OAAA;EACA;QACA,IAAI,IAAI,CAACzD,SAAS,CAAC+B,OAAO,CAAC,IAAIqB,QAAQ,EAAE;EACvC,QAAA,IAAI,CAACpD,SAAS,CAAC+B,OAAO,CAAC,GAAGqB,QAAQ,CAAA;EAClC,QAAA,IAAI,CAAChD,SAAS,CAAC2B,OAAO,CAAC,GAAG,IAAI,CAAA;EAChC,OAAA;EACA,MAAA,IAAI,CAAC7B,gBAAgB,CAAC6B,OAAO,CAAC,GAAGuB,eAAe,CAAA;EAChD,MAAA,IAAI,CAACrD,YAAY,CAAC8B,OAAO,CAAC,GAAGsB,WAAW,CAAA;EAC1C,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACEM,EAAAA,+BAA+BA,GAAG;MAChC,IAAI,CAACC,cAAc,EAAE,CAAA;MACrB,IAAI,CAACC,kBAAkB,EAAE,CAAA;MACzB,IAAI,CAAC3B,gBAAgB,EAAE,CAAA;EACzB,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACE4B,EAAAA,gBAAgBA,CAACC,cAAc,EAAEC,OAAO,EAAE;EACxC;EACA,IAAA,IAAIC,UAAU,GAAGD,OAAO,GAAG,CAAC,GAAG,CAAC,CAAA;MAChC,IAAIE,MAAM,GAAGH,cAAc,CAACI,MAAM,CAAC,CAAC,EAAEF,UAAU,CAAC,CAAA;EACjD,IAAA,IAAIjL,IAAI,GAAGgL,OAAO,GAAG,SAAS,GAAG,QAAQ,CAAA;MACzC,IAAII,aAAa,GAAGH,UAAU,CAAA;MAE9B,IAAII,YAAY,GAAG,CAAC,CAAA;MACpB,IAAIC,QAAQ,GAAG,KAAK,CAAA;MACpB,IAAIC,OAAO,GAAG,KAAK,CAAA;MACnB,IAAI9F,SAAS,GAAG,EAAE,CAAA;EAClB,IAAA,IAAI+F,WAAW,GAAG,EAAE,CAAC;;EAErBC,IAAAA,SAAS,EAAE,OACTL,aAAa,GAAGL,cAAc,CAACxU,MAAM,IACrC+U,QAAQ,KAAK,KAAK,6BAClB;EACA,MAAA,IAAIvF,MAAM,GAAGgF,cAAc,CAACI,MAAM,CAACC,aAAa,CAAC,CAAA;;EAEjD;EACA;EACA,MAAA,KAAK,IAAIvF,IAAI,IAAI,CAAC,QAAQ,EAAE,MAAM,EAAE,UAAU,EAAE,MAAM,CAAC,EAAE;EACvD,QAAA,IAAI6F,GAAG,GAAGvG,aAAa,CAACU,IAAI,CAAC,CAACnC,MAAM,CAACzL,IAAI,CAAC8N,MAAM,CAAC,CAAA;EACjD,QAAA,IAAI2F,GAAG,EAAE;EACPN,UAAAA,aAAa,IAAIM,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EAC9B,UAAA,SAASkV,SAAS,CAAA;EACpB,SAAA;EACF,OAAA;;EAEA;QACA,IAAI1F,MAAM,CAACtP,KAAK,CAAC0O,aAAa,CAACK,SAAS,CAAC9B,MAAM,CAAC,EAAE;EAChD;EACA2H,QAAAA,YAAY,EAAE,CAAA;EACdD,QAAAA,aAAa,IAAI,CAAC,CAAA;EAClB,QAAA,SAASK,SAAS,CAAA;EACpB,OAAA;;EAEA;QACA,IAAI1F,MAAM,CAACtP,KAAK,CAAC0O,aAAa,CAACI,QAAQ,CAAC7B,MAAM,CAAC,EAAE;EAC/C;EACA;EACA;EACA2H,QAAAA,YAAY,EAAE,CAAA;EACd;UACA,IAAI,CAACL,OAAO,EAAE;YACZ,IAAI,IAAI,CAACF,gBAAgB,CAAC/E,MAAM,EAAE,KAAK,CAAC,EAAE;EACxC;EACA,YAAA,OAAO,KAAK,CAAA;EACd,WAAA;EACF,SAAA;EACAqF,QAAAA,aAAa,IAAI,CAAC,CAAA;EAClB,QAAA,SAASK,SAAS,CAAA;EACpB,OAAA;;EAEA;EACA,MAAA,IAAI1F,MAAM,CAACtP,KAAK,CAAC,KAAK,CAAC,EAAE;EACvB4U,QAAAA,YAAY,EAAE,CAAA;UACd,IAAIA,YAAY,IAAI,CAAC,EAAE;EACrB;YACAC,QAAQ,GAAGP,cAAc,CAACI,MAAM,CAC9BF,UAAU,EACVG,aAAa,GAAGH,UAClB,CAAC,CAAA;EACDG,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASK,SAAS,CAAA;EACpB,SAAA;EACF,OAAA;;EAEA;EACAL,MAAAA,aAAa,EAAE,CAAA;EACjB,KAAA;;EAEA;EACA,IAAA,IAAIE,QAAQ,KAAK,KAAK,EAAE,OAAO,KAAK,CAAC;;EAErC;EACA,IAAA,IAAIK,QAAQ,GACVP,aAAa,GAAGL,cAAc,CAACxU,MAAM,GACjCwU,cAAc,CAACI,MAAM,CAACC,aAAa,EAAE,CAAC,CAAC,GACvC,EAAE,CAAA;;EAER;MACA,IAAIO,QAAQ,IAAI,GAAG,EAAE;EACnB,MAAA,IAAI5F,MAAM,GAAGgF,cAAc,CAACI,MAAM,CAACC,aAAa,CAAC,CAAA;QACjD,IAAIM,GAAG,GAAGvG,aAAa,CAACM,SAAS,CAAC/B,MAAM,CAACzL,IAAI,CAAC8N,MAAM,CAAC,CAAA;EACrD,MAAA,IAAI2F,GAAG,EAAE;EACP;EACAN,QAAAA,aAAa,IAAIM,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EAC9BkP,QAAAA,SAAS,CAACjP,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,EAAEA,GAAG,CAAC,CAAC,CAAC,EAAEA,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;UACtC,IAAIA,GAAG,CAACvG,aAAa,CAACM,SAAS,CAACX,gBAAgB,CAAC,EAAE;EACjD;YACAyG,OAAO,GAAGG,GAAG,CAACvG,aAAa,CAACM,SAAS,CAACX,gBAAgB,CAAC,CAAA;EACzD,SAAC,MAAM;EACL;EACAyG,UAAAA,OAAO,GAAGD,QAAQ,CAACM,IAAI,EAAE,CAAA;EAC3B,SAAA;EACF,OAAC,MAAM;EACL;EACA,QAAA,OAAO,KAAK,CAAA;EACd,OAAA;EACF,KAAC,MAAM,IAAID,QAAQ,IAAI,GAAG,EAAE;EAC1B;EACAJ,MAAAA,OAAO,GAAGD,QAAQ,CAACM,IAAI,EAAE,CAAA;;EAEzB;EACF,KAAC,MAAM;EACL;;EAEA;EACAR,MAAAA,aAAa,EAAE,CAAA;QAEf,IAAIS,gBAAgB,GAAG,CAAC,CAAA;QACxBC,WAAW,EAAE,OACXV,aAAa,GAAGL,cAAc,CAACxU,MAAM,IACrCsV,gBAAgB,GAAG,CAAC,EACpB;EACA,QAAA,IAAI9F,MAAM,GAAGgF,cAAc,CAACI,MAAM,CAACC,aAAa,CAAC,CAAA;;EAEjD;EACA,QAAA,IAAIM,GAAG,GAAG,MAAM,CAACzT,IAAI,CAAC8N,MAAM,CAAC,CAAA;EAC7B,QAAA,IAAI2F,GAAG,EAAE;YACP,QAAQF,WAAW,CAACjV,MAAM;EACxB,YAAA,KAAK,CAAC;EACJiV,cAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxB,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxB,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EAAE;gBACN,IAAIF,WAAW,CAAC,CAAC,CAAC,CAAC/U,KAAK,CAAC,GAAG,CAAC,EAAE;EAC7B+U,gBAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAChD,eAAC,MAAM;EACL,gBAAA,IAAIG,gBAAgB,IAAI,CAAC,EAAE,OAAO,KAAK,CAAC;EACxCL,gBAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAC;kBACrBgV,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAC;EAC3B,eAAA;;EACA,cAAA,MAAA;EACF,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxB,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJ,cAAA,OAAO,KAAK,CAAA;EAAE;EAChB,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAC9C,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAC9C,cAAA,MAAA;EAAO;EACT,YAAA;EACE,cAAA,OAAO,KAAK,CAAA;EAAE;EAClB,WAAA;;EACAN,UAAAA,aAAa,IAAIM,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EAC9B,UAAA,SAASuV,WAAW,CAAA;EACtB,SAAA;;EAEA;UACAJ,GAAG,GAAGvG,aAAa,CAACC,MAAM,CAAC1B,MAAM,CAACzL,IAAI,CAAC8N,MAAM,CAAC,CAAA;EAC9C,QAAA,IAAI2F,GAAG,EAAE;YACP,QAAQF,WAAW,CAACjV,MAAM;EACxB,YAAA,KAAK,CAAC;EACJiV,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxB,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAC9C,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJ,cAAA,OAAO,KAAK,CAAA;EAAE;EAChB,YAAA,KAAK,CAAC;EACJ,cAAA,OAAO,KAAK,CAAA;EAAE;EAChB,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAC9C,cAAA,MAAA;EAAO;EACT,YAAA;EACE,cAAA,OAAO,KAAK,CAAA;EAAE;EAClB,WAAA;;EACAN,UAAAA,aAAa,IAAIM,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EAC9B,UAAA,SAASuV,WAAW,CAAA;EACtB,SAAA;;EAEA;EACA,QAAA,IAAIN,WAAW,CAACjV,MAAM,GAAG,CAAC,IAAIwP,MAAM,CAACtP,KAAK,CAAC,IAAI,CAAC,EAAE;YAChD,IAAI+U,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EACjDgV,UAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC,GAAG,CAAC,CAAA;EAC3CwU,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASU,WAAW,CAAA;EACtB,SAAA;;EAEA;EACA,QAAA,IACE,CAACN,WAAW,CAACjV,MAAM,IAAI,CAAC,IAAIiV,WAAW,CAACjV,MAAM,IAAI,CAAC,KACnDwP,MAAM,CAACtP,KAAK,CAAC,IAAI,CAAC,EAClB;EACA,UAAA,IAAI+U,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAC;EAClDgV,UAAAA,WAAW,CAAChV,IAAI,CAAC,GAAG,CAAC,CAAA;EACrB4U,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASU,WAAW,CAAA;EACtB,SAAA;;EAEA;EACAJ,QAAAA,GAAG,GAAG,OAAO,CAACzT,IAAI,CAAC8N,MAAM,CAAC,CAAA;EAC1B;EACA;UACA,IACE2F,GAAG,KACFF,WAAW,CAACjV,MAAM,IAAI,CAAC,IACtBiV,WAAW,CAACjV,MAAM,IAAI,CAAC,IACvBiV,WAAW,CAACjV,MAAM,IAAI,CAAC,CAAC,EAC1B;YACA,OAAOiV,WAAW,CAACjV,MAAM,GAAG,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EACnDgV,UAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxBN,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASU,WAAW,CAAA;EACtB,SAAA;;EAEA;UACA,IACEJ,GAAG,KACFF,WAAW,CAACjV,MAAM,IAAI,CAAC,IAAIiV,WAAW,CAACjV,MAAM,IAAI,CAAC,CAAC,IACpDiV,WAAW,CAAC,CAAC,CAAC,IAAIE,GAAG,CAAC,CAAC,CAAC,EACxB;EACA,UAAA,IAAIF,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAC;EAClDgV,UAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxBN,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASU,WAAW,CAAA;EACtB,SAAA;EACA;;EAEA;EACA,QAAA,IAAI/F,MAAM,CAACtP,KAAK,CAAC,KAAK,CAAC,EAAE;YACvB,QAAQ+U,WAAW,CAACjV,MAAM;EACxB,YAAA,KAAK,CAAC;EACJiV,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EAAE;EACNgV,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC,GAAG,CAAC,CAAA;EAC3C,cAAA,IAAI,CAAC4U,WAAW,CAAC,CAAC,CAAC,CAAC/U,KAAK,CAAC,IAAI,CAAC,EAAEoV,gBAAgB,EAAE,CAAA;EACnD,cAAA,MAAA;EACF,YAAA,KAAK,CAAC;EACJL,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAChV,IAAI,CAAC,GAAG,CAAC,CAAA;EACrB,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EAAE;gBACN,IAAIgV,WAAW,CAAC,CAAC,CAAC,IAAI,GAAG,EAAE,OAAO,KAAK,CAAA;EACvCA,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC,GAAG,CAAC,CAAA;EAC3C,cAAA,MAAA;EACF,YAAA;EACE,cAAA,OAAO,KAAK,CAAA;EAAE;EAClB,WAAA;;EACAwU,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASU,WAAW,CAAA;EACtB,SAAA;;EAEA;EACA,QAAA,IAAI/F,MAAM,CAACtP,KAAK,CAAC,KAAK,CAAC,EAAE;EACvB,UAAA,IAAI+U,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAE;EAC3B;cACA,OAAOiV,WAAW,CAACjV,MAAM,GAAG,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAEnD,YAAA,IAAI,CAACgV,WAAW,CAAC,CAAC,CAAC,CAAC/U,KAAK,CAAC,IAAI,CAAC,EAAEoV,gBAAgB,EAAE,CAAA;cAEnD,IAAIA,gBAAgB,GAAG,CAAC,EAAE;EACxBL,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC,GAAG,CAAC,CAAA;EAC7C,aAAA;EACF,WAAC,MAAM,IAAI4U,WAAW,CAACjV,MAAM,IAAI,CAAC,IAAIiV,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAE;EAC7D;EACA,YAAA,IAAIiV,WAAW,CAAC,CAAC,CAAC,IAAI,GAAG,EAAE;EACzB;gBACA,IAAIA,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EACjDgV,cAAAA,WAAW,CAAChV,IAAI,CAAC,GAAG,CAAC,CAAA;EACvB,aAAC,MAAM;EACL;gBACA,IAAIgV,WAAW,CAACjV,MAAM,IAAI,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,GAAG,CAAC,CAAC,KAC9CgV,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC,GAAG,CAAC,CAAA;EAClD,aAAA;EACF,WAAC,MAAM;cACLiV,gBAAgB,EAAE,CAAC;EACrB,WAAA;;YAEA,IAAIA,gBAAgB,IAAI,CAAC,EAAE;EACzB;cACA,OAAOL,WAAW,CAACjV,MAAM,GAAG,CAAC,EAAEiV,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EACrD,WAAA;EAEA4U,UAAAA,aAAa,EAAE,CAAA;EACf,UAAA,SAASU,WAAW,CAAA;EACtB,SAAA;;EAEA;EACAJ,QAAAA,GAAG,GAAG,IAAI,CAACzT,IAAI,CAAC8N,MAAM,CAAC,CAAA;EACvB,QAAA,IAAI2F,GAAG,EAAE;YACP,QAAQF,WAAW,CAACjV,MAAM;EACxB,YAAA,KAAK,CAAC;EACJiV,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAChV,IAAI,CAACkV,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EACxB,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAC9C,cAAA,MAAA;EAAO;EACT,YAAA,KAAK,CAAC;EACJ,cAAA,OAAO,KAAK,CAAA;EAAE;EAChB,YAAA,KAAK,CAAC;EACJ,cAAA,OAAO,KAAK,CAAA;EAAE;EAChB,YAAA,KAAK,CAAC;EACJF,cAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EAAE;EACxB,YAAA,KAAK,CAAC;EACJgV,cAAAA,WAAW,CAAC,CAAC,CAAC,GAAGA,WAAW,CAAC,CAAC,CAAC,CAAC5U,MAAM,CAAC8U,GAAG,CAAC,CAAC,CAAC,CAAC,CAAA;EAC9C,cAAA,MAAA;EAAO;EACT,YAAA;EACE,cAAA,OAAO,KAAK,CAAA;EAAE;EAClB,WAAA;;EACAN,UAAAA,aAAa,IAAIM,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EAC9B,UAAA,SAASuV,WAAW,CAAA;EACtB,SAAA;UACA,MAAM,eAAe,CAAC;EACxB,OAAA;;EACA,MAAA,IAAID,gBAAgB,GAAG,CAAC,EAAE,OAAO,KAAK,CAAC;EACzC,KAAA;;MAEA,IAAIN,OAAO,KAAK,KAAK,EAAE;EACrB;QACA,IAAIQ,KAAK,GAAG,KAAK,CAAA;EACjB,MAAA,KAAK,IAAIC,KAAK,IAAI,IAAI,CAAC7E,UAAU,EAAE;UACjC,IAAI6E,KAAK,IAAIT,OAAO,EAAE;EACpBQ,UAAAA,KAAK,GAAG,IAAI,CAAA;EACZ,UAAA,MAAA;EACF,SAAA;EACF,OAAA;EACA,MAAA,IAAIC,KAAK,GAAGD,KAAK,GACb,+BAA+B,GAC/B,iCAAiC,CAAA;EACrC,MAAA,IAAIE,MAAM,GAAI,CAA6BjM,2BAAAA,EAAAA,IAAK,KAAIkL,MAAO,CAAA,oBAAA,EAAsBlL,IAAK,CAAA,CAAA,EACpFyF,SAAS,CAAClP,MAAM,GAAG,CAAC,IAAI,CAACkP,SAAS,CAAC,CAAC,CAAC,GAAGuG,KAAK,GAAG,EACjD,CAAA,EAAA,EAAI,IAAI,CAACnC,mBAAmB,CAC3ByB,QACF,CAAE,CAAA,kCAAA,EAAoCtL,IAAK,CAAW,UAAA,CAAA,CAAA;EAEtD,MAAA,IAAIyF,SAAS,CAAClP,MAAM,IAAI,CAAC,EAAE;EACzB0V,QAAAA,MAAM,GAAGA,MAAM,CAACrV,MAAM,CACnB,CAAA,2BAAA,EAA6BoJ,IAAK,CAAA,EAAA,EAAIyF,SAAS,CAAC,CAAC,CAAE,SAAQ,EAC3D,CAAA,aAAA,EAAeuG,KAAM,CAAA,EAAA,EAAIvG,SAAS,CAAC,CAAC,CAAE,SAAQ,EAC9C,CAAA,2BAAA,EAA6BzF,IAAK,CAAA,EAAA,EAAIyF,SAAS,CAAC,CAAC,CAAE,SACtD,CAAC,CAAA;EACH,OAAA;QACA,OAAO;EACLwG,QAAAA,MAAM,EAAEA,MAAM;EACdC,QAAAA,SAAS,EAAEd,aAAAA;SACZ,CAAA;OACF,MAAM,IAAII,WAAW,EAAE;EACtB;;EAEA;EACA,MAAA,OAAOA,WAAW,CAACjV,MAAM,GAAG,CAAC,EAAE;EAC7BiV,QAAAA,WAAW,CAAChV,IAAI,CAAC,EAAE,CAAC,CAAA;EACtB,OAAA;QAEA,OAAO;UACLyV,MAAM,EAAG,CAA6BjM,2BAAAA,EAAAA,IAAK,CAAIkL,EAAAA,EAAAA,MAAO,CAAsBlL,oBAAAA,EAAAA,IAAK,CAAI,EAAA,EAAA,IAAI,CAAC6J,mBAAmB,CAC3GyB,QACF,CAAE,CAAA,kCAAA,EAAoCtL,IAAK,CAAA,IAAA,EACzCwL,WAAW,CAAC,CAAC,CACd,CAAsBxL,oBAAAA,EAAAA,IAAK,CAC1BwL,aAAAA,EAAAA,WAAW,CAAC,CAAC,CACd,CAAA,kCAAA,EAAoCxL,IAAK,CAAA,EAAA,EAAIwL,WAAW,CAAC,CAAC,CAAE,CAC3DA,EAAAA,WAAW,CAAC,CAAC,CACd,CAAA,EAAEA,WAAW,CAAC,CAAC,CAAE,CAAsBxL,oBAAAA,EAAAA,IAAK,CAC3CwL,OAAAA,EAAAA,WAAW,CAAC,CAAC,CACd,CAAA,kCAAA,EAAoCxL,IAAK,CAAA,EAAA,EAAIwL,WAAW,CAAC,CAAC,CAAE,CAAS,QAAA,CAAA;EACtEU,QAAAA,SAAS,EAAEd,aAAAA;SACZ,CAAA;EACH,KAAA;EAEA,IAAA,OAAO,KAAK,CAAA;EACd,GAAA;;EAEA;EACF;EACA;EACA;EACA;IACEvB,mBAAmBA,CAACkB,cAAc,EAAE;MAClC,IAAIoB,SAAS,GAAG,EAAE,CAAA;EAClB,IAAA,IAAIC,KAAK,GAAG,EAAE,CAAC;MACf,IAAIC,MAAM,GAAG,CAAC,CAAA;MACd,IAAItG,MAAM,GAAGgF,cAAc,CAAA;MAE3B1T,KAAK,EAAE,OAAO0O,MAAM,EAAE;EACpB;EACA,MAAA,KAAK,IAAIF,IAAI,IAAI,CAAC,QAAQ,EAAE,MAAM,EAAE,UAAU,EAAE,MAAM,CAAC,EAAE;EACvD,QAAA,IAAI6F,GAAG,GAAGvG,aAAa,CAACU,IAAI,CAAC,CAACnC,MAAM,CAACzL,IAAI,CAAC8N,MAAM,CAAC,CAAA;EACjD,QAAA,IAAI2F,GAAG,EAAE;YACP3F,MAAM,GAAGA,MAAM,CAACoF,MAAM,CAACO,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAC,CAAA;EACrC8V,UAAAA,MAAM,IAAIX,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EACvB4V,UAAAA,SAAS,IAAIhH,aAAa,CAACU,IAAI,CAAC,CAAClC,WAAAA;EAC/B;EAAA,WACCjD,OAAO,CAAC,YAAY,EAAE,CAACgJ,GAAG,EAAEC,EAAE,KAAK1D,UAAU,CAACyF,GAAG,CAAC/B,EAAE,CAAC,CAAC,CAAC,CAAA;EAC1D,UAAA,SAAStS,KAAK,CAAA;EAChB,SAAA;EACF,OAAA;;EAEA;QACA,IAAIiV,aAAa,GAAGvG,MAAM,CAACtP,KAAK,CAAC0O,aAAa,CAACI,QAAQ,CAAC7B,MAAM,CAAC,CAAA;QAC/D,IAAI6I,cAAc,GAAGxG,MAAM,CAACtP,KAAK,CAAC0O,aAAa,CAACK,SAAS,CAAC9B,MAAM,CAAC,CAAA;QACjE,IAAI6I,cAAc,IAAID,aAAa,EAAE;UACnC,IAAInO,MAAM,GAAG,IAAI,CAAC2M,gBAAgB,CAAC/E,MAAM,EAAEwG,cAAc,CAAC,CAAA;EAC1D,QAAA,IAAIpO,MAAM,EAAE;EACVgO,UAAAA,SAAS,GAAI,CAAEA,EAAAA,SAAU,GAAEhO,MAAM,CAAC8N,MAAO,CAAC,CAAA,CAAA;YAC1ClG,MAAM,GAAGA,MAAM,CAACoF,MAAM,CAAChN,MAAM,CAAC+N,SAAS,CAAC,CAAA;YACxCG,MAAM,IAAIlO,MAAM,CAAC+N,SAAS,CAAA;EAC1B,UAAA,SAAS7U,KAAK,CAAA;EAChB,SAAA;EACF,OAAA;;EAEA;EACA,MAAA,IAAIqU,GAAG,GAAG,cAAc,CAACzT,IAAI,CAAC8N,MAAM,CAAC,CAAA;EACrC,MAAA,IAAI2F,GAAG,EAAE;EACP,QAAA,IAAIc,UAAU,GAAGd,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EAC9B,QAAA,MAAMkW,WAAW,GAAGf,GAAG,CAAC,CAAC,CAAC,CAAA;UAC1B,MAAMgB,gBAAgB,GAAGhB,GAAG,CAAC,CAAC,CAAC,CAAC,CAAC,CAAC,CAAC;;UAEnC3F,MAAM,GAAGA,MAAM,CAACoF,MAAM,CAACO,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAC,CAAA;;EAErC;;EAEA,QAAA,MAAMoW,SAAS,GAAGN,MAAM,GAAG,CAAC,GAAGtB,cAAc,CAACI,MAAM,CAAC,CAAC,EAAEkB,MAAM,CAAC,GAAG,GAAG,CAAC;EACtE,QAAA,MAAMO,SAAS,GACbP,MAAM,GAAGX,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,GAAGwU,cAAc,CAACxU,MAAM,GAAGwP,MAAM,GAAG,GAAG,CAAA;EAE/D,QAAA,MAAM8G,kBAAkB,GAAGD,SAAS,CAACnW,KAAK,CAAC6M,kBAAkB,CAAC,CAAA;EAC9D,QAAA,MAAMwJ,mBAAmB,GAAGH,SAAS,CAAClW,KAAK,CAAC8M,mBAAmB,CAAC,CAAA;EAChE,QAAA,MAAMwJ,iBAAiB,GAAGH,SAAS,CAACnW,KAAK,CAAC,KAAK,CAAC,CAAA;EAChD,QAAA,MAAMuW,kBAAkB,GAAGL,SAAS,CAAClW,KAAK,CAAC,KAAK,CAAC,CAAA;;EAEjD;EACA,QAAA,IAAIwW,OAAO,GACT,CAACF,iBAAiB,KACjB,CAACF,kBAAkB,IAClB,CAAC,CAACG,kBAAkB,IACpB,CAAC,CAACF,mBAAmB,CAAC,CAAA;EAC1B,QAAA,IAAII,QAAQ,GACV,CAACF,kBAAkB,KAClB,CAACF,mBAAmB,IAAI,CAAC,CAACC,iBAAiB,IAAI,CAAC,CAACF,kBAAkB,CAAC,CAAA;;EAEvE;EACA,QAAA,IAAIH,gBAAgB,IAAI,GAAG,IAAIO,OAAO,IAAIC,QAAQ,EAAE;EAClDD,UAAAA,OAAO,GAAGH,mBAAmB,CAAA;EAC7BI,UAAAA,QAAQ,GAAGL,kBAAkB,CAAA;EAC/B,SAAA;;EAEA;EACA,QAAA,IAAIK,QAAQ,EAAE;EACZ,UAAA,IAAIC,YAAY,GAAGf,KAAK,CAAC7V,MAAM,GAAG,CAAC,CAAA;EACnC;EACA,UAAA,OAAOiW,UAAU,IAAIW,YAAY,IAAI,CAAC,EAAE;cACtC,IAAIf,KAAK,CAACe,YAAY,CAAC,CAACC,SAAS,IAAIV,gBAAgB,EAAE;EACrD;;EAEA;EACA,cAAA,OAAOS,YAAY,GAAGf,KAAK,CAAC7V,MAAM,GAAG,CAAC,EAAE;EACtC,gBAAA,MAAM8W,KAAK,GAAGjB,KAAK,CAACkB,GAAG,EAAE,CAAA;kBACzBnB,SAAS,GAAI,GAAEkB,KAAK,CAACpB,MAAO,CAAEoB,EAAAA,KAAK,CAACZ,WAAW,CAACtB,MAAM,CACpD,CAAC,EACDkC,KAAK,CAACE,KACR,CAAE,CAAA,EAAEpB,SAAU,CAAC,CAAA,CAAA;EACjB,eAAA;;EAEA;EACA,cAAA,IAAIK,UAAU,IAAI,CAAC,IAAIJ,KAAK,CAACe,YAAY,CAAC,CAACI,KAAK,IAAI,CAAC,EAAE;EACrD;kBACApB,SAAS,GAAI,CAAuBO,qBAAAA,EAAAA,gBAAiB,CAAEA,EAAAA,gBAAiB,CAAkCP,gCAAAA,EAAAA,SAAU,CAAgCO,8BAAAA,EAAAA,gBAAiB,CAAEA,EAAAA,gBAAiB,CAAQ,OAAA,CAAA,CAAA;EAChMF,gBAAAA,UAAU,IAAI,CAAC,CAAA;EACfJ,gBAAAA,KAAK,CAACe,YAAY,CAAC,CAACI,KAAK,IAAI,CAAC,CAAA;EAChC,eAAC,MAAM;EACL;EACApB,gBAAAA,SAAS,GAAI,CAAuBO,qBAAAA,EAAAA,gBAAiB,2BAA0BP,SAAU,CAAA,0BAAA,EAA4BO,gBAAiB,CAAQ,OAAA,CAAA,CAAA;EAC9IF,gBAAAA,UAAU,IAAI,CAAC,CAAA;EACfJ,gBAAAA,KAAK,CAACe,YAAY,CAAC,CAACI,KAAK,IAAI,CAAC,CAAA;EAChC,eAAA;;EAEA;gBACA,IAAInB,KAAK,CAACe,YAAY,CAAC,CAACI,KAAK,IAAI,CAAC,EAAE;EAClC,gBAAA,IAAIF,KAAK,GAAGjB,KAAK,CAACkB,GAAG,EAAE,CAAA;EACvBnB,gBAAAA,SAAS,GAAI,CAAEkB,EAAAA,KAAK,CAACpB,MAAO,CAAA,EAAEE,SAAU,CAAC,CAAA,CAAA;EACzCgB,gBAAAA,YAAY,EAAE,CAAA;EAChB,eAAA;EACF,aAAC,MAAM;EACL;EACA;EACAA,cAAAA,YAAY,EAAE,CAAA;EAChB,aAAA;EACF,WAAA;EACF,SAAA;EACA;UACA,IAAIX,UAAU,IAAIS,OAAO,EAAE;YACzBb,KAAK,CAAC5V,IAAI,CAAC;EACT4W,YAAAA,SAAS,EAAEV,gBAAgB;EAC3BD,YAAAA,WAAW,EAAEA,WAAW;EACxBc,YAAAA,KAAK,EAAEf,UAAU;EACjBP,YAAAA,MAAM,EAAEE,SAAAA;EACV,WAAC,CAAC,CAAA;YACFA,SAAS,GAAG,EAAE,CAAC;EACfK,UAAAA,UAAU,GAAG,CAAC,CAAA;EAChB,SAAA;;EAEA;EACA,QAAA,IAAIA,UAAU,EAAE;EACdL,UAAAA,SAAS,GAAI,CAAA,EAAEA,SAAU,CAAA,EAAEM,WAAW,CAACtB,MAAM,CAAC,CAAC,EAAEqB,UAAU,CAAE,CAAC,CAAA,CAAA;EAChE,SAAA;EAEAH,QAAAA,MAAM,IAAIX,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;EACvB,QAAA,SAASc,KAAK,CAAA;EAChB,OAAA;;EAEA;EACAqU,MAAAA,GAAG,GAAG,KAAK,CAACzT,IAAI,CAAC8N,MAAM,CAAC,CAAA;EACxB,MAAA,IAAI2F,GAAG,EAAE;UACP,IAAI8B,QAAQ,GAAG,KAAK,CAAA;EACpB,QAAA,IAAIL,YAAY,GAAGf,KAAK,CAAC7V,MAAM,GAAG,CAAC,CAAA;EACnC;EACA,QAAA,OAAO,CAACiX,QAAQ,IAAIL,YAAY,IAAI,CAAC,EAAE;YACrC,IAAIf,KAAK,CAACe,YAAY,CAAC,CAACC,SAAS,IAAI,GAAG,EAAE;EACxC;;EAEA;EACA,YAAA,OAAOD,YAAY,GAAGf,KAAK,CAAC7V,MAAM,GAAG,CAAC,EAAE;EACtC,cAAA,MAAM8W,KAAK,GAAGjB,KAAK,CAACkB,GAAG,EAAE,CAAA;gBACzBnB,SAAS,GAAI,GAAEkB,KAAK,CAACpB,MAAO,CAAEoB,EAAAA,KAAK,CAACZ,WAAW,CAACtB,MAAM,CACpD,CAAC,EACDkC,KAAK,CAACE,KACR,CAAE,CAAA,EAAEpB,SAAU,CAAC,CAAA,CAAA;EACjB,aAAA;;EAEA;cACAA,SAAS,GAAI,CAA6DA,2DAAAA,EAAAA,SAAU,CAAqC,oCAAA,CAAA,CAAA;EACzH,YAAA,IAAIkB,KAAK,GAAGjB,KAAK,CAACkB,GAAG,EAAE,CAAA;EACvBnB,YAAAA,SAAS,GAAI,CAAEkB,EAAAA,KAAK,CAACpB,MAAO,CAAA,EAAEE,SAAU,CAAC,CAAA,CAAA;EACzCqB,YAAAA,QAAQ,GAAG,IAAI,CAAA;EACjB,WAAC,MAAM;EACL;EACA;EACAL,YAAAA,YAAY,EAAE,CAAA;EAChB,WAAA;EACF,SAAA;;EAEA;UACA,IAAI,CAACK,QAAQ,EAAE;YACbpB,KAAK,CAAC5V,IAAI,CAAC;EACT4W,YAAAA,SAAS,EAAE,GAAG;EACdX,YAAAA,WAAW,EAAE,IAAI;EACjBc,YAAAA,KAAK,EAAE,CAAC;EACRtB,YAAAA,MAAM,EAAEE,SAAAA;EACV,WAAC,CAAC,CAAA;YACFA,SAAS,GAAG,EAAE,CAAC;EACjB,SAAA;;EAEAE,QAAAA,MAAM,IAAIX,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;UACvBwP,MAAM,GAAGA,MAAM,CAACoF,MAAM,CAACO,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAC,CAAA;EACrC,QAAA,SAASc,KAAK,CAAA;EAChB,OAAA;;EAEA;QACAqU,GAAG,GAAGvG,aAAa,CAACO,OAAO,CAAChC,MAAM,CAACzL,IAAI,CAAC8N,MAAM,CAAC,CAAA;EAC/C,MAAA,IAAI2F,GAAG,EAAE;UACP3F,MAAM,GAAGA,MAAM,CAACoF,MAAM,CAACO,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAC,CAAA;EACrC8V,QAAAA,MAAM,IAAIX,GAAG,CAAC,CAAC,CAAC,CAACnV,MAAM,CAAA;UACvB4V,SAAS,IAAIhH,aAAa,CAACO,OAAO,CAAC/B,WAAW,CAACjD,OAAO,CACpD,YAAY,EACZ,CAACgJ,GAAG,EAAEC,EAAE,KAAK1D,UAAU,CAACyF,GAAG,CAAC/B,EAAE,CAAC,CACjC,CAAC,CAAA;EACD,QAAA,SAAStS,KAAK,CAAA;EAChB,OAAA;EACA,MAAA,MAAM,gBAAgB,CAAA;EACxB,KAAA;;EAEA;MACA,OAAO+U,KAAK,CAAC7V,MAAM,EAAE;EACnB,MAAA,MAAM8W,KAAK,GAAGjB,KAAK,CAACkB,GAAG,EAAE,CAAA;QACzBnB,SAAS,GAAI,GAAEkB,KAAK,CAACpB,MAAO,CAAEoB,EAAAA,KAAK,CAACZ,WAAW,CAACtB,MAAM,CACpD,CAAC,EACDkC,KAAK,CAACE,KACR,CAAE,CAAA,EAAEpB,SAAU,CAAC,CAAA,CAAA;EACjB,KAAA;EAEA,IAAA,OAAOA,SAAS,CAAA;EAClB,GAAA;;EAEA;EACF;EACA;EACEvB,EAAAA,cAAcA,GAAG;MACf,IAAI,CAACxD,SAAS,GAAG,IAAI6B,KAAK,CAAC,IAAI,CAACnC,KAAK,CAACvQ,MAAM,CAAC,CAAA;EAC7C,IAAA,KAAK,IAAID,CAAC,GAAG,CAAC,EAAEA,CAAC,GAAG,IAAI,CAAC8Q,SAAS,CAAC7Q,MAAM,EAAED,CAAC,EAAE,EAAE;EAC9C,MAAA,IAAI,CAAC8Q,SAAS,CAAC9Q,CAAC,CAAC,GAAG,KAAK,CAAA;EAC3B,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACA;EACEuU,EAAAA,kBAAkBA,GAAG;EACnB;EACA;EACA;EACA,IAAA,IAAI4C,SAAS,GAAG,IAAI,CAAC9Y,CAAC,CAAC+Y,iBAAiB,GAAG,IAAI,CAAC5G,KAAK,CAACvQ,MAAM,CAAA;EAC5D,IAAA,IAAIkX,SAAS,EAAE;EACb;EACA;QACA,IAAIE,gBAAgB,GAAG,CAAC,CAAA;QACxB,OACEA,gBAAgB,IAAI,IAAI,CAAC7G,KAAK,CAACvQ,MAAM,IACrCoX,gBAAgB,IAAI,IAAI,CAAC5G,YAAY,CAACxQ,MAAM,IAC5C,IAAI,CAACwQ,YAAY,CAAC4G,gBAAgB,CAAC;EAAI;EACvC,MAAA,IAAI,CAAC7G,KAAK,CAAC6G,gBAAgB,CAAC,IAC1B,IAAI,CAAC5G,YAAY,CAAC4G,gBAAgB,CAAC,CAACC,WAAW,IACjD,IAAI,CAAC5G,SAAS,CAAC2G,gBAAgB,CAAC,IAC9B,IAAI,CAAC5G,YAAY,CAAC4G,gBAAgB,CAAC,CAAChY,SAAS,EAC/C;EACAgY,QAAAA,gBAAgB,EAAE,CAAA;EACpB,OAAA;;EAEA;QACA,IAAIE,eAAe,GAAG,CAAC,CAAC,CAAA;EACxB,MAAA,OACE,CAACA,eAAe,GAAG,IAAI,CAAC/G,KAAK,CAACvQ,MAAM,IACpC,CAACsX,eAAe,GAAG,IAAI,CAAC9G,YAAY,CAACxQ,MAAM,IAC3C,IAAI,CAACuQ,KAAK,CAAC,IAAI,CAACA,KAAK,CAACvQ,MAAM,GAAGsX,eAAe,CAAC,IAC7C,IAAI,CAAC9G,YAAY,CAAC,IAAI,CAACA,YAAY,CAACxQ,MAAM,GAAGsX,eAAe,CAAC,CAC1DD,WAAW,IAChB,IAAI,CAAC5G,SAAS,CAAC,IAAI,CAACF,KAAK,CAACvQ,MAAM,GAAGsX,eAAe,CAAC,IACjD,IAAI,CAAC9G,YAAY,CAAC,IAAI,CAACA,YAAY,CAACxQ,MAAM,GAAGsX,eAAe,CAAC,CAC1DlY,SAAS,EACd;EACAkY,QAAAA,eAAe,EAAE,CAAA;EACnB,OAAA;EAEA,MAAA,IAAIC,aAAa,GACf,IAAI,CAAChH,KAAK,CAACvQ,MAAM,GAAGsX,eAAe,GAAG,CAAC,GAAGF,gBAAgB,CAAA;QAC5D,IAAIG,aAAa,GAAG,CAACL,SAAS,EAAEK,aAAa,GAAG,CAACL,SAAS,CAAA;EAC1D,MAAA,IAAIK,aAAa,GAAG,CAAC,EAAEA,aAAa,GAAG,CAAC,CAAA;QAExC,IAAIC,UAAU,GAAG,EAAE,CAAA;EACnB,MAAA,KAAK,IAAIvE,CAAC,GAAG,CAAC,EAAEA,CAAC,GAAGsE,aAAa,GAAGL,SAAS,EAAEjE,CAAC,EAAE,EAAE;EAClDuE,QAAAA,UAAU,CAACvX,IAAI,CAAC,IAAI,CAACuQ,YAAY,CAAC4G,gBAAgB,GAAGnE,CAAC,CAAC,CAACoE,WAAW,CAAC,CAAA;EACtE,OAAA;QACA,IAAI,CAACI,WAAW,CAACL,gBAAgB,EAAEG,aAAa,EAAEC,UAAU,EAAE,KAAK,CAAC,CAAA;EACtE,KAAC,MAAM;EACL;EACA,MAAA,KAAK,IAAIE,IAAI,GAAG,CAAC,EAAEA,IAAI,GAAG,IAAI,CAAClH,YAAY,CAACxQ,MAAM,EAAE0X,IAAI,EAAE,EAAE;EAC1D,QAAA,IAAItZ,CAAC,GAAG,IAAI,CAACoS,YAAY,CAACkH,IAAI,CAAC,CAAA;EAC/B,QAAA,IAAIC,EAAE,GAAGvZ,CAAC,CAACiZ,WAAW,CAAA;EACtB,QAAA,IAAI,IAAI,CAAC9G,KAAK,CAACmH,IAAI,CAAC,KAAKC,EAAE,IAAI,IAAI,CAAClH,SAAS,CAACiH,IAAI,CAAC,KAAKtZ,CAAC,CAACgB,SAAS,EAAE;EACnE;EACA,UAAA,IAAI,CAACmR,KAAK,CAACmH,IAAI,CAAC,GAAGC,EAAE,CAAA;YACrB,IAAI,CAAClH,SAAS,CAACiH,IAAI,CAAC,GAAGtZ,CAAC,CAACgB,SAAS,CAAA;EAClC,UAAA,IAAI,CAACyR,SAAS,CAAC6G,IAAI,CAAC,GAAG,IAAI,CAAA;EAC7B,SAAA;EACF,OAAA;EACF,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACA;IACEE,mBAAmBA,CAACC,GAAG,EAAE;MACvB,IAAI,CAACA,GAAG,EAAE,OAAA;;EAEV;MACA,IAAI,CAACvD,kBAAkB,EAAE,CAAA;MAEzB,IAAIwD,eAAe,GAAG,KAAK,CAAA;EAC3B;;EAEA,IAAA,IAAIC,SAAS,GAAGF,GAAG,CAACG,GAAG,GAAG,CAAC,GAAGH,GAAG,CAACI,GAAG,GAAGJ,GAAG,CAACI,GAAG,GAAG,CAAC,CAAA;EACnD,IAAA,QAAQ,IAAI,CAACxH,SAAS,CAACsH,SAAS,CAAC;EAC/B,MAAA,KAAK,MAAM;EACTD,QAAAA,eAAe,GAAG,MAAM,CAAA;EACxB,QAAA,MAAA;EACF,MAAA,KAAK,MAAM;EACTA,QAAAA,eAAe,GAAG,MAAM,CAAA;EACxB,QAAA,MAAA;EACF,MAAA,KAAK,gBAAgB;EACnBA,QAAAA,eAAe,GAAG,gBAAgB,CAAA;EAClC,QAAA,MAAA;EACJ,KAAA;MAEA,IAAIvH,KAAK,GAAG,IAAI,CAACA,KAAK,CAACsH,GAAG,CAACI,GAAG,CAAC,CAC5B9N,OAAO,CAAC,OAAO,EAAE,IAAI,CAAC,CACtBvK,KAAK,CAAC,gBAAgB,CAAC,CAAA;EAC1B,IAAA,IAAI2Q,KAAK,CAACvQ,MAAM,GAAG,CAAC,EAAE;EACpB;EACA,MAAA,IAAI,CAACyX,WAAW,CAACI,GAAG,CAACI,GAAG,EAAE,CAAC,EAAE1H,KAAK,EAAE,IAAI,CAAC,CAAA;QACzCsH,GAAG,CAACI,GAAG,EAAE,CAAA;QACTJ,GAAG,CAACG,GAAG,GAAG,CAAC,CAAA;EACb,KAAA;EAEA,IAAA,IAAIF,eAAe,EAAE;EACnB;QACA,IAAI5E,OAAO,GAAGjG,WAAW,CAAC6K,eAAe,CAAC,CAAC3K,MAAM,CAACzL,IAAI,CACpD,IAAI,CAAC6O,KAAK,CAACsH,GAAG,CAACI,GAAG,GAAG,CAAC,CACxB,CAAC,CAAA;EACD,MAAA,IAAI/E,OAAO,EAAE;EACX;EACA,QAAA,IAAIA,OAAO,CAAC,CAAC,CAAC,EAAE;EACd;;EAEA;YACA,IAAI4E,eAAe,IAAI,MAAM,EAAE;EAC7B5E,YAAAA,OAAO,CAAC,CAAC,CAAC,GAAGA,OAAO,CAAC,CAAC,CAAC,CAAC/I,OAAO,CAAC,SAAS,EAAGvC,MAAM,IAAK;gBACrD,OAAOsQ,QAAQ,CAACtQ,MAAM,CAAC,CAAC,CAAC,CAAC,GAAG,CAAC,CAAA;EAChC,aAAC,CAAC,CAAA;EACJ,WAAA;YACA,IAAI,CAAC2I,KAAK,CAACsH,GAAG,CAACI,GAAG,CAAC,GAAI,CAAA,EAAE/E,OAAO,CAAC,CAAC,CAAE,CAAA,EAAE,IAAI,CAAC3C,KAAK,CAACsH,GAAG,CAACI,GAAG,CAAE,CAAC,CAAA,CAAA;YAC3D,IAAI,CAACpH,SAAS,CAACgH,GAAG,CAACI,GAAG,CAAC,GAAG,IAAI,CAAA;YAC9BJ,GAAG,CAACG,GAAG,GAAG9E,OAAO,CAAC,CAAC,CAAC,CAAClT,MAAM,CAAA;EAC7B,SAAC,MAAM;EACL;YACA,IAAI,CAACuQ,KAAK,CAACsH,GAAG,CAACI,GAAG,GAAG,CAAC,CAAC,GAAG,EAAE,CAAA;YAC5B,IAAI,CAACpH,SAAS,CAACgH,GAAG,CAACI,GAAG,GAAG,CAAC,CAAC,GAAG,IAAI,CAAA;EACpC,SAAA;EACF,OAAA;EACF,KAAA;MACA,IAAI,CAACtF,gBAAgB,EAAE,CAAA;EACzB,GAAA;;EAEA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;;EAEA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACEwF,EAAAA,YAAYA,GAAoB;EAAA,IAAA,IAAnBC,SAAS,GAAA3V,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,KAAK,CAAA;EAC5B,IAAA,MAAMwO,SAAS,GAAG5P,MAAM,CAAC8W,YAAY,EAAE,CAAA;MACvC,IAAIE,SAAS,GAAGD,SAAS,GAAGnH,SAAS,CAACqH,UAAU,GAAGrH,SAAS,CAACsH,SAAS,CAAA;EACtE,IAAA,IAAI,CAACF,SAAS,EAAE,OAAO,IAAI,CAAA;MAC3B,IAAIvC,MAAM,GAAGsC,SAAS,GAAGnH,SAAS,CAACuH,YAAY,GAAGvH,SAAS,CAACwH,WAAW,CAAA;EACvE,IAAA,IAAIJ,SAAS,IAAI,IAAI,CAACja,CAAC,EAAE;QACvB,IAAI0X,MAAM,GAAG,IAAI,CAACvF,KAAK,CAACvQ,MAAM,EAC5B,OAAO;EACLiY,QAAAA,GAAG,EAAEnC,MAAM;EACXkC,QAAAA,GAAG,EAAE,CAAA;SACN,CAAA;QACH,OAAO;UACLC,GAAG,EAAEnC,MAAM,GAAG,CAAC;UACfkC,GAAG,EAAE,IAAI,CAACzH,KAAK,CAACuF,MAAM,GAAG,CAAC,CAAC,CAAC9V,MAAAA;SAC7B,CAAA;EACH,KAAA;MAEA,IAAIgY,GAAG,GAAG,IAAI,CAACU,aAAa,CAACL,SAAS,EAAEvC,MAAM,CAAC,CAAA;EAC/C,IAAA,IAAIkC,GAAG,KAAK,IAAI,EAAE,OAAO,IAAI,CAAC;;EAE9B;MACA,IAAIW,IAAI,GAAGN,SAAS,CAAA;EACpB,IAAA,OAAOM,IAAI,CAACzZ,aAAa,IAAI,IAAI,CAACd,CAAC,EAAE;QACnCua,IAAI,GAAGA,IAAI,CAACzZ,aAAa,CAAA;EAC3B,KAAA;MAEA,IAAI+Y,GAAG,GAAG,CAAC,CAAA;EACX;EACA;EACA,IAAA,IACEU,IAAI,CAAClF,OAAO,IACZkF,IAAI,CAAClF,OAAO,CAACjB,OAAO,KACnB,CAACmG,IAAI,CAACC,eAAe,IACpBD,IAAI,CAACC,eAAe,CAACnF,OAAO,CAACjB,OAAO,IAAImG,IAAI,CAAClF,OAAO,CAACjB,OAAO,CAAC,EAC/D;QACAyF,GAAG,GAAGC,QAAQ,CAACS,IAAI,CAAClF,OAAO,CAACjB,OAAO,CAAC,CAAA;EACtC,KAAC,MAAM;QACL,OAAOmG,IAAI,CAACC,eAAe,EAAE;EAC3BX,QAAAA,GAAG,EAAE,CAAA;UACLU,IAAI,GAAGA,IAAI,CAACC,eAAe,CAAA;EAC7B,OAAA;EACF,KAAA;MACA,OAAO;EAAEX,MAAAA,GAAG,EAAEA,GAAG;EAAED,MAAAA,GAAG,EAAEA,GAAG;EAAEW,MAAAA,IAAI,EAAEN,SAAAA;OAAW,CAAA;EAChD,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACEK,EAAAA,aAAaA,CAACL,SAAS,EAAEvC,MAAM,EAAE;MAC/B,IAAI6C,IAAI,GAAGN,SAAS,CAAA;EACpB,IAAA,IAAIL,GAAG,CAAA;EACP;MACA,OAAOW,IAAI,IAAIA,IAAI,CAACvH,UAAU,IAAI,IAAI,CAAChT,CAAC,EAAE;QACxCua,IAAI,GAAGA,IAAI,CAACvH,UAAU,CAAA;EACxB,KAAA;EACA,IAAA,IAAIuH,IAAI,IAAI,IAAI,EAAE,OAAO,IAAI,CAAA;;EAE7B;EACA;EACA;EACA;MACA,IAAIN,SAAS,CAACQ,QAAQ,KAAKC,IAAI,CAACC,SAAS,IAAIjD,MAAM,KAAK,CAAC,EAAE;EACzD;EACA;EACA;EACA;EACAkC,MAAAA,GAAG,GAAGlC,MAAM,CAAA;EACZ6C,MAAAA,IAAI,GAAGN,SAAS,CAAA;EAClB,KAAC,MAAM,IAAIvC,MAAM,GAAG,CAAC,EAAE;QACrB6C,IAAI,GAAGN,SAAS,CAAChG,UAAU,CAACyD,MAAM,GAAG,CAAC,CAAC,CAAA;EACvCkC,MAAAA,GAAG,GAAGW,IAAI,CAACtB,WAAW,CAACrX,MAAM,CAAA;EAC/B,KAAA;EACA,IAAA,OAAO2Y,IAAI,CAACvH,UAAU,IAAI,IAAI,CAAChT,CAAC,EAAE;QAChC,IAAIua,IAAI,CAACC,eAAe,EAAE;UACxBD,IAAI,GAAGA,IAAI,CAACC,eAAe,CAAA;EAC3BZ,QAAAA,GAAG,IAAIW,IAAI,CAACtB,WAAW,CAACrX,MAAM,CAAA;EAChC,OAAC,MAAM;UACL2Y,IAAI,GAAGA,IAAI,CAACvH,UAAU,CAAA;EACxB,OAAA;EACF,KAAA;EACA,IAAA,OAAO4G,GAAG,CAAA;EACZ,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACEgB,EAAAA,oBAAoBA,CAACf,GAAG,EAAED,GAAG,EAAqB;EAAA,IAAA,IAAnBiB,SAAS,GAAAxW,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,KAAK,CAAA;EAC9C,IAAA,IAAIwV,GAAG,IAAI,IAAI,CAACzH,YAAY,CAACxQ,MAAM,EAAE;EACnC;EACAiY,MAAAA,GAAG,GAAG,IAAI,CAACzH,YAAY,CAACxQ,MAAM,GAAG,CAAC,CAAA;QAClCgY,GAAG,GAAG,IAAI,CAACzH,KAAK,CAAC0H,GAAG,CAAC,CAACjY,MAAM,CAAA;EAC9B,KAAA;MACA,IAAIgY,GAAG,GAAG,IAAI,CAACzH,KAAK,CAAC0H,GAAG,CAAC,CAACjY,MAAM,EAAE;QAChCgY,GAAG,GAAG,IAAI,CAACzH,KAAK,CAAC0H,GAAG,CAAC,CAACjY,MAAM,CAAA;EAC9B,KAAA;EACA,IAAA,MAAMoR,UAAU,GAAG,IAAI,CAACZ,YAAY,CAACyH,GAAG,CAAC,CAAA;EACzC,IAAA,IAAIU,IAAI,GAAGvH,UAAU,CAACkB,UAAU,CAAA;MAEhC,IAAI4G,gBAAgB,GAAG,KAAK,CAAA;EAC5B;EACA,IAAA,IAAIC,EAAE,GAAG;QACPR,IAAI,EAAEvH,UAAU,CAACkB,UAAU,GAAGlB,UAAU,CAACkB,UAAU,GAAGlB,UAAU;EAChE0E,MAAAA,MAAM,EAAE,CAAA;OACT,CAAA;MAED,OAAO6C,IAAI,IAAIvH,UAAU,EAAE;QACzB,IAAI,CAAC8H,gBAAgB,IAAIP,IAAI,CAACE,QAAQ,KAAKC,IAAI,CAACC,SAAS,EAAE;EACzD,QAAA,IAAIJ,IAAI,CAACS,SAAS,CAACpZ,MAAM,IAAIgY,GAAG,EAAE;YAChC,IAAIiB,SAAS,IAAIN,IAAI,CAACS,SAAS,CAACpZ,MAAM,IAAIgY,GAAG,EAAE;EAC7C;EACA;EACAmB,YAAAA,EAAE,GAAG;EAAER,cAAAA,IAAI,EAAEA,IAAI;EAAE7C,cAAAA,MAAM,EAAEkC,GAAAA;eAAK,CAAA;EAChCA,YAAAA,GAAG,GAAG,CAAC,CAAA;EACT,WAAC,MAAM;cACL,OAAO;EAAEW,cAAAA,IAAI,EAAEA,IAAI;EAAE7C,cAAAA,MAAM,EAAEkC,GAAAA;eAAK,CAAA;EACpC,WAAA;EACF,SAAC,MAAM;EACLA,UAAAA,GAAG,IAAIW,IAAI,CAACS,SAAS,CAACpZ,MAAM,CAAA;EAC9B,SAAA;EACF,OAAA;EACA,MAAA,IAAI,CAACkZ,gBAAgB,IAAIP,IAAI,CAACrG,UAAU,EAAE;UACxCqG,IAAI,GAAGA,IAAI,CAACrG,UAAU,CAAA;EACxB,OAAC,MAAM,IAAIqG,IAAI,CAAC5G,WAAW,EAAE;EAC3BmH,QAAAA,gBAAgB,GAAG,KAAK,CAAA;UACxBP,IAAI,GAAGA,IAAI,CAAC5G,WAAW,CAAA;EACzB,OAAC,MAAM;EACLmH,QAAAA,gBAAgB,GAAG,IAAI,CAAA;UACvBP,IAAI,GAAGA,IAAI,CAACvH,UAAU,CAAA;EACxB,OAAA;EACF,KAAA;;EAEA;EACA;EACA,IAAA,OAAO+H,EAAE,CAAA;EACX,GAAA;;EAEA;EACF;EACA;EACA;EACA;IACEE,YAAYA,CAACvb,KAAK,EAAiB;EAAA,IAAA,IAAfC,MAAM,GAAA0E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;MAC/B,IAAI,CAAC3E,KAAK,EAAE,OAAA;EAEZ,IAAA,IAAIwb,KAAK,GAAG3a,QAAQ,CAAC4a,WAAW,EAAE,CAAA;MAElC,IAAI;EAAEZ,MAAAA,IAAI,EAAEJ,SAAS;EAAEzC,MAAAA,MAAM,EAAE2C,WAAAA;EAAY,KAAC,GAAG,IAAI,CAACO,oBAAoB,CACtElb,KAAK,CAACma,GAAG,EACTna,KAAK,CAACka,GAAG,EACTja,MAAM,IAAIA,MAAM,CAACka,GAAG,IAAIna,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACia,GAAG,GAAGla,KAAK,CAACka,GAC1D,CAAC,CAAC;MACF,IAAIM,UAAU,GAAG,IAAI;EACnBE,MAAAA,YAAY,GAAG,IAAI,CAAA;EACrB,IAAA,IAAIza,MAAM,KAAKA,MAAM,CAACka,GAAG,IAAIna,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACia,GAAG,IAAIla,KAAK,CAACka,GAAG,CAAC,EAAE;QAClE,IAAI;UAAEW,IAAI;EAAE7C,QAAAA,MAAAA;EAAO,OAAC,GAAG,IAAI,CAACkD,oBAAoB,CAC9Cjb,MAAM,CAACka,GAAG,EACVla,MAAM,CAACia,GAAG,EACVla,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACka,GAAG,IAAIna,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAChD,CAAC,CAAA;EACDM,MAAAA,UAAU,GAAGK,IAAI,CAAA;EACjBH,MAAAA,YAAY,GAAG1C,MAAM,CAAA;EACvB,KAAA;EAEA,IAAA,IAAIwC,UAAU,EAAEgB,KAAK,CAACE,QAAQ,CAAClB,UAAU,EAAEE,YAAY,CAAC,CAAC,KACpDc,KAAK,CAACE,QAAQ,CAACjB,SAAS,EAAEE,WAAW,CAAC,CAAA;EAC3Ca,IAAAA,KAAK,CAACG,MAAM,CAAClB,SAAS,EAAEE,WAAW,CAAC,CAAA;EAEpC,IAAA,IAAIiB,eAAe,GAAGrY,MAAM,CAAC8W,YAAY,EAAE,CAAA;MAC3CuB,eAAe,CAACC,eAAe,EAAE,CAAA;EACjCD,IAAAA,eAAe,CAACE,QAAQ,CAACN,KAAK,CAAC,CAAA;EACjC,GAAA;;EAEA;EACF;EACA;IACErH,gBAAgBA,CAACzR,KAAK,EAAE;EACtB;EACA;EACA;EACA,IAAA,IAAIA,KAAK,CAACqZ,SAAS,IAAI,uBAAuB,EAAE,OAAA;EAEhD,IAAA,IAAI/b,KAAK,GAAG,IAAI,CAACqa,YAAY,EAAE,CAAA;EAE/B,IAAA,IACE,CAAC3X,KAAK,CAACqZ,SAAS,IAAI,iBAAiB,IACnCrZ,KAAK,CAACqZ,SAAS,IAAI,iBAAiB,KACtC/b,KAAK,EACL;QACA,IAAI,CAACuW,cAAc,EAAE,CAAA;EACrB,MAAA,IAAI,CAACuD,mBAAmB,CAAC9Z,KAAK,CAAC,CAAA;EACjC,KAAC,MAAM;EACL,MAAA,IAAI,CAAC,IAAI,CAACM,CAAC,CAACkU,UAAU,EAAE;EACtB,QAAA,IAAI,CAAClU,CAAC,CAACb,SAAS,GAAG,qCAAqC,CAAA;EAC1D,OAAC,MAAM;UACL,IAAI,CAACuc,gBAAgB,EAAE,CAAA;EACzB,OAAA;QACA,IAAI,CAAC1F,+BAA+B,EAAE,CAAA;EACxC,KAAA;EACA,IAAA,IAAItW,KAAK,EAAE;EACT,MAAA,IAAI,CAACub,YAAY,CAACvb,KAAK,CAAC,CAAA;EAC1B,KAAA;MAEA,IAAI,CAAC8U,UAAU,EAAE,CAAA;EACnB,GAAA;;EAEA;EACF;EACA;EACEkH,EAAAA,gBAAgBA,GAAG;MACjB,MAAMC,gBAAgB,GAAGrH,KAAK,CAACsH,IAAI,CAAC,IAAI,CAAC5b,CAAC,CAACiU,UAAU,CAAC,CAAA;EAEtD,IAAA,MAAM4H,YAAY,GAAG,UAACC,KAAK,EAAqB;EAC9C,MAAA,MAAMC,MAAM,GAAGD,KAAK,CAAChb,aAAa,CAAA;EAClC,MAAA,MAAM6S,WAAW,GAAGmI,KAAK,CAACnI,WAAW,CAAA;EACrCoI,MAAAA,MAAM,CAAC5H,WAAW,CAAC2H,KAAK,CAAC,CAAA;QAAC,KAAAE,IAAAA,IAAA,GAAA3X,SAAA,CAAAzC,MAAA,EAHIqa,WAAW,OAAA3H,KAAA,CAAA0H,IAAA,GAAAA,CAAAA,GAAAA,IAAA,WAAAE,IAAA,GAAA,CAAA,EAAAA,IAAA,GAAAF,IAAA,EAAAE,IAAA,EAAA,EAAA;EAAXD,QAAAA,WAAW,CAAAC,IAAA,GAAA7X,CAAAA,CAAAA,GAAAA,SAAA,CAAA6X,IAAA,CAAA,CAAA;EAAA,OAAA;QAIzCD,WAAW,CAACE,OAAO,CAAEC,QAAQ,IAC3BzI,WAAW,GACPoI,MAAM,CAACnI,YAAY,CAACwI,QAAQ,EAAEzI,WAAW,CAAC,GAC1CoI,MAAM,CAAC5a,WAAW,CAACib,QAAQ,CACjC,CAAC,CAAA;OACF,CAAA;EAEDT,IAAAA,gBAAgB,CAACQ,OAAO,CAAEL,KAAK,IAAK;EAClC,MAAA,IAAIA,KAAK,CAACrB,QAAQ,KAAKC,IAAI,CAAC2B,YAAY,IAAIP,KAAK,CAACxb,OAAO,KAAK,KAAK,EAAE;EACnE;EACA,QAAA,MAAMgc,UAAU,GAAG/b,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAA;EAChD8a,QAAAA,YAAY,CAACC,KAAK,EAAEQ,UAAU,CAAC,CAAA;EAC/BA,QAAAA,UAAU,CAACnb,WAAW,CAAC2a,KAAK,CAAC,CAAA;SAC9B,MAAM,IAAIA,KAAK,CAAC7H,UAAU,CAACrS,MAAM,IAAI,CAAC,EAAE;EACvC;UACAka,KAAK,CAAC3a,WAAW,CAACZ,QAAQ,CAACQ,aAAa,CAAC,IAAI,CAAC,CAAC,CAAA;EACjD,OAAC,MAAM;UACL,MAAMwb,aAAa,GAAGjI,KAAK,CAACsH,IAAI,CAACE,KAAK,CAAC7H,UAAU,CAAC,CAAA;UAClD,IACEsI,aAAa,CAACC,IAAI,CACfC,UAAU,IACTA,UAAU,CAAChC,QAAQ,KAAKC,IAAI,CAAC2B,YAAY,IACzCI,UAAU,CAACnc,OAAO,KAAK,KAC3B,CAAC,EACD;EACA,UAAA,OAAOub,YAAY,CAACC,KAAK,EAAES,aAAa,CAAC,CAAA;EAC3C,SAAA;EACF,OAAA;EACF,KAAC,CAAC,CAAA;EACJ,GAAA;;EAEA;EACF;EACA;IACEvI,UAAUA,CAAC5R,KAAK,EAAE;MAChBA,KAAK,CAACC,cAAc,EAAE,CAAA;EAEtB,IAAA,IAAI,CAACqa,QAAQ,CAACta,KAAK,CAACua,YAAY,CAAC,CAAA;EACnC,GAAA;;EAEA;EACF;EACA;EACE7I,EAAAA,0BAA0BA,GAAG;MAC3B,IAAI,CAAC8I,aAAa,EAAE,CAAA;EACtB,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;EACA;IACEvD,WAAWA,CACTwD,SAAS,EAIT;EAAA,IAAA,IAHA1D,aAAa,GAAA9U,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,CAAC,CAAA;EAAA,IAAA,IACjByY,aAAa,GAAAzY,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,EAAE,CAAA;EAAA,IAAA,IAClB0Y,kBAAkB,GAAA1Y,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;EAEzB,IAAA,IAAI0Y,kBAAkB,EAAE;QACtB,KAAK,IAAIpb,CAAC,GAAG,CAAC,EAAEA,CAAC,GAAGwX,aAAa,EAAExX,CAAC,EAAE,EAAE;EACtC,QAAA,IAAI,CAAC3B,CAAC,CAACmU,WAAW,CAAC,IAAI,CAACnU,CAAC,CAACiU,UAAU,CAAC4I,SAAS,CAAC,CAAC,CAAA;EAClD,OAAA;EACF,KAAA;MAEA,IAAIG,aAAa,GAAG,EAAE,CAAA;MACtB,IAAIC,aAAa,GAAG,EAAE,CAAA;EAEtB,IAAA,KAAK,IAAItb,CAAC,GAAG,CAAC,EAAEA,CAAC,GAAGmb,aAAa,CAAClb,MAAM,EAAED,CAAC,EAAE,EAAE;EAC7Cqb,MAAAA,aAAa,CAACnb,IAAI,CAAC,EAAE,CAAC,CAAA;EACtBob,MAAAA,aAAa,CAACpb,IAAI,CAAC,IAAI,CAAC,CAAA;EACxB,MAAA,IAAIkb,kBAAkB,EAAE;UACtB,IAAI,IAAI,CAAC/c,CAAC,CAACiU,UAAU,CAAC4I,SAAS,CAAC,EAC9B,IAAI,CAAC7c,CAAC,CAAC4T,YAAY,CACjBrT,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,EAC7B,IAAI,CAACf,CAAC,CAACiU,UAAU,CAAC4I,SAAS,CAC7B,CAAC,CAAC,KACC,IAAI,CAAC7c,CAAC,CAACmB,WAAW,CAACZ,QAAQ,CAACQ,aAAa,CAAC,KAAK,CAAC,CAAC,CAAA;EACxD,OAAA;EACF,KAAA;MAEA,IAAI,CAACoR,KAAK,CAAC+K,MAAM,CAACL,SAAS,EAAE1D,aAAa,EAAE,GAAG2D,aAAa,CAAC,CAAA;MAC7D,IAAI,CAACzK,SAAS,CAAC6K,MAAM,CAACL,SAAS,EAAE1D,aAAa,EAAE,GAAG6D,aAAa,CAAC,CAAA;MACjE,IAAI,CAACvK,SAAS,CAACyK,MAAM,CAACL,SAAS,EAAE1D,aAAa,EAAE,GAAG8D,aAAa,CAAC,CAAA;EACnE,GAAA;;EAEA;EACF;EACA;IACElJ,WAAWA,CAAC3R,KAAK,EAAE;MACjBA,KAAK,CAACC,cAAc,EAAE,CAAA;;EAEtB;EACA,IAAA,IAAI8a,IAAI,GAAG,CAAC/a,KAAK,CAACgb,aAAa,IAAIhb,KAAK,EAAEib,aAAa,CAACC,OAAO,CAC7D,YACF,CAAC,CAAA;;EAED;EACA,IAAA,IAAI,CAAC1d,KAAK,CAACud,IAAI,CAAC,CAAA;EAClB,GAAA;;EAEA;EACF;EACA;EACA;IACEvd,KAAKA,CAACud,IAAI,EAA+B;EAAA,IAAA,IAA7Bxd,MAAM,GAAA0E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;EAAA,IAAA,IAAE3E,KAAK,GAAA2E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;MACrC,IAAI,CAAC1E,MAAM,EAAEA,MAAM,GAAG,IAAI,CAACoa,YAAY,CAAC,IAAI,CAAC,CAAA;MAC7C,IAAI,CAACra,KAAK,EAAEA,KAAK,GAAG,IAAI,CAACqa,YAAY,CAAC,KAAK,CAAC,CAAA;MAC5C,IAAIwD,SAAS,EAAEjN,GAAG,CAAA;MAClB,IAAI,CAAC5Q,KAAK,EAAE;EACVA,MAAAA,KAAK,GAAG;EACNma,QAAAA,GAAG,EAAE,IAAI,CAAC1H,KAAK,CAACvQ,MAAM,GAAG,CAAC;EAC1BgY,QAAAA,GAAG,EAAE,IAAI,CAACzH,KAAK,CAAC,IAAI,CAACA,KAAK,CAACvQ,MAAM,GAAG,CAAC,CAAC,CAACA,MAAAA;EACzC,OAAC,CAAC;EACJ,KAAA;;MACA,IAAI,CAACjC,MAAM,EAAE;EACXA,MAAAA,MAAM,GAAGD,KAAK,CAAA;EAChB,KAAA;MACA,IACEC,MAAM,CAACka,GAAG,GAAGna,KAAK,CAACma,GAAG,IACrBla,MAAM,CAACka,GAAG,IAAIna,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACia,GAAG,IAAIla,KAAK,CAACka,GAAI,EACpD;EACA2D,MAAAA,SAAS,GAAG5d,MAAM,CAAA;EAClB2Q,MAAAA,GAAG,GAAG5Q,KAAK,CAAA;EACb,KAAC,MAAM;EACL6d,MAAAA,SAAS,GAAG7d,KAAK,CAAA;EACjB4Q,MAAAA,GAAG,GAAG3Q,MAAM,CAAA;EACd,KAAA;EACA,IAAA,IAAI6d,aAAa,GAAGL,IAAI,CAAC3b,KAAK,CAAC,gBAAgB,CAAC,CAAA;EAChD,IAAA,IAAIic,UAAU,GAAG,IAAI,CAACtL,KAAK,CAACoL,SAAS,CAAC1D,GAAG,CAAC,CAACrD,MAAM,CAAC,CAAC,EAAE+G,SAAS,CAAC3D,GAAG,CAAC,CAAA;EACnE,IAAA,IAAI8D,OAAO,GAAG,IAAI,CAACvL,KAAK,CAAC7B,GAAG,CAACuJ,GAAG,CAAC,CAACrD,MAAM,CAAClG,GAAG,CAACsJ,GAAG,CAAC,CAAA;EACjD4D,IAAAA,aAAa,CAAC,CAAC,CAAC,GAAGC,UAAU,CAACxb,MAAM,CAACub,aAAa,CAAC,CAAC,CAAC,CAAC,CAAA;MACtD,IAAIG,SAAS,GAAGH,aAAa,CAACA,aAAa,CAAC5b,MAAM,GAAG,CAAC,CAAC,CAACA,MAAM,CAAA;MAC9D4b,aAAa,CAACA,aAAa,CAAC5b,MAAM,GAAG,CAAC,CAAC,GACrC4b,aAAa,CAACA,aAAa,CAAC5b,MAAM,GAAG,CAAC,CAAC,CAACK,MAAM,CAACyb,OAAO,CAAC,CAAA;EACzD,IAAA,IAAI,CAACrE,WAAW,CAACkE,SAAS,CAAC1D,GAAG,EAAE,CAAC,GAAGvJ,GAAG,CAACuJ,GAAG,GAAG0D,SAAS,CAAC1D,GAAG,EAAE2D,aAAa,CAAC,CAAA;MAC3E9d,KAAK,CAACma,GAAG,GAAG0D,SAAS,CAAC1D,GAAG,GAAG2D,aAAa,CAAC5b,MAAM,GAAG,CAAC,CAAA;MACpDlC,KAAK,CAACka,GAAG,GAAG+D,SAAS,CAAA;MACrB,IAAI,CAACpJ,gBAAgB,EAAE,CAAA;EACvB,IAAA,IAAI,CAAC0G,YAAY,CAACvb,KAAK,CAAC,CAAA;MACxB,IAAI,CAAC8U,UAAU,EAAE,CAAA;EACnB,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACEoJ,EAAAA,qBAAqBA,CAACC,KAAK,EAAEC,KAAK,EAAE;EAClC,IAAA,IAAI,CAACD,KAAK,IAAI,CAACC,KAAK,EAAE,OAAO,IAAI,CAAA;EACjC,IAAA,IAAID,KAAK,IAAIC,KAAK,EAAE,OAAOD,KAAK,CAAA;MAChC,MAAME,QAAQ,GAAIxD,IAAI,IAAK;QACzB,IAAIwD,QAAQ,GAAG,EAAE,CAAA;EACjB,MAAA,OAAOxD,IAAI,EAAE;EACXwD,QAAAA,QAAQ,CAACC,OAAO,CAACzD,IAAI,CAAC,CAAA;UACtBA,IAAI,GAAGA,IAAI,CAACvH,UAAU,CAAA;EACxB,OAAA;EACA,MAAA,OAAO+K,QAAQ,CAAA;OAChB,CAAA;EAED,IAAA,MAAME,SAAS,GAAGF,QAAQ,CAACF,KAAK,CAAC,CAAA;EACjC,IAAA,MAAMK,SAAS,GAAGH,QAAQ,CAACD,KAAK,CAAC,CAAA;MAEjC,IAAIG,SAAS,CAAC,CAAC,CAAC,IAAIC,SAAS,CAAC,CAAC,CAAC,EAAE,OAAO,IAAI,CAAA;EAC7C,IAAA,IAAIvc,CAAC,CAAA;EACL,IAAA,KAAKA,CAAC,GAAG,CAAC,EAAEsc,SAAS,CAACtc,CAAC,CAAC,IAAIuc,SAAS,CAACvc,CAAC,CAAC,EAAEA,CAAC,EAAE,CAAC,CAAA;EAC9C,IAAA,OAAOsc,SAAS,CAACtc,CAAC,GAAG,CAAC,CAAC,CAAA;EACzB,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACA;EACEwc,EAAAA,0BAA0BA,CAACze,KAAK,EAAEC,MAAM,EAAEqB,SAAS,EAAE;MACnD,IAAIuZ,IAAI,GAAG,IAAI,CAAA;EACf,IAAA,IAAI,CAAC7a,KAAK,EAAE,OAAO,IAAI,CAAA;MACvB,IAAI,CAACC,MAAM,EAAE;QACX4a,IAAI,GAAG7a,KAAK,CAAC6a,IAAI,CAAA;EACnB,KAAC,MAAM;QACL,IAAI7a,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACka,GAAG,EAAE,OAAO,IAAI,CAAA;EACxCU,MAAAA,IAAI,GAAG,IAAI,CAACqD,qBAAqB,CAACle,KAAK,CAAC6a,IAAI,EAAE5a,MAAM,CAAC4a,IAAI,CAAC,CAAA;EAC5D,KAAA;EACA,IAAA,IAAI,CAACA,IAAI,EAAE,OAAO,IAAI,CAAA;EACtB,IAAA,OAAOA,IAAI,IAAI,IAAI,CAACva,CAAC,EAAE;EACrB,MAAA,IAAIua,IAAI,CAACvZ,SAAS,IAAIuZ,IAAI,CAACvZ,SAAS,CAACod,QAAQ,CAACpd,SAAS,CAAC,EAAE,OAAOuZ,IAAI,CAAA;QACrEA,IAAI,GAAGA,IAAI,CAACvH,UAAU,CAAA;EACxB,KAAA;EACA;EACA,IAAA,OAAO,IAAI,CAAA;EACb,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACEqL,EAAAA,eAAeA,GAA8B;EAAA,IAAA,IAA7B3e,KAAK,GAAA2E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;EAAA,IAAA,IAAE1E,MAAM,GAAA0E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;MACzC,IAAI7B,YAAY,GAAG,EAAE,CAAA;MACrB,IAAI,CAAC9C,KAAK,EAAEA,KAAK,GAAG,IAAI,CAACqa,YAAY,CAAC,KAAK,CAAC,CAAA;MAC5C,IAAI,CAACpa,MAAM,EAAEA,MAAM,GAAG,IAAI,CAACoa,YAAY,CAAC,IAAI,CAAC,CAAA;MAC7C,IAAI,CAACra,KAAK,EAAE;EACV,MAAA,KAAK,IAAI4e,GAAG,IAAIre,QAAQ,EAAE;EACxBuC,QAAAA,YAAY,CAAC8b,GAAG,CAAC,GAAG,IAAI,CAAA;EAC1B,OAAA;EACA,MAAA,OAAO9b,YAAY,CAAA;EACrB,KAAA;EACA,IAAA,IAAI,CAAC7C,MAAM,EAAEA,MAAM,GAAGD,KAAK,CAAA;MAE3B,IAAI2Q,KAAK,EAAEC,GAAG,CAAA;MACd,IACE3Q,MAAM,CAACka,GAAG,GAAGna,KAAK,CAACma,GAAG,IACrBla,MAAM,CAACka,GAAG,IAAIna,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACia,GAAG,GAAGla,KAAK,CAACka,GAAI,EACnD;EACAvJ,MAAAA,KAAK,GAAG1Q,MAAM,CAAA;EACd2Q,MAAAA,GAAG,GAAG5Q,KAAK,CAAA;EACb,KAAC,MAAM;EACL2Q,MAAAA,KAAK,GAAG3Q,KAAK,CAAA;EACb4Q,MAAAA,GAAG,GAAG3Q,MAAM,CAAA;EACd,KAAA;EACA,IAAA,IAAI2Q,GAAG,CAACuJ,GAAG,GAAGxJ,KAAK,CAACwJ,GAAG,IAAIvJ,GAAG,CAACsJ,GAAG,IAAI,CAAC,EAAE;QACvCtJ,GAAG,CAACuJ,GAAG,EAAE,CAAA;EACTvJ,MAAAA,GAAG,CAACsJ,GAAG,GAAG,IAAI,CAACzH,KAAK,CAAC7B,GAAG,CAACuJ,GAAG,CAAC,CAACjY,MAAM,CAAC;EACvC,KAAA;;EAEA,IAAA,KAAK,IAAI0c,GAAG,IAAIre,QAAQ,EAAE;QACxB,IAAIA,QAAQ,CAACqe,GAAG,CAAC,CAACjT,IAAI,IAAI,QAAQ,EAAE;UAClC,IACE,CAAC3L,KAAK,IACNA,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACka,GAAG,IACvB,CAAC,IAAI,CAACta,yBAAyB,CAACG,KAAK,EAAEC,MAAM,CAAC,EAC9C;EACA6C,UAAAA,YAAY,CAAC8b,GAAG,CAAC,GAAG,IAAI,CAAA;EAC1B,SAAC,MAAM;EACL;YACA9b,YAAY,CAAC8b,GAAG,CAAC,GACf,CAAC,CAAC,IAAI,CAACH,0BAA0B,CAC/Bze,KAAK,EACLC,MAAM,EACNM,QAAQ,CAACqe,GAAG,CAAC,CAACtd,SAChB,CAAC;EACD;EACCtB,UAAAA,KAAK,CAACka,GAAG,IAAIja,MAAM,CAACia,GAAG,IACtB,CAAC,CAAC,IAAI,CAACzH,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CACpBrD,MAAM,CAAC,CAAC,EAAE9W,KAAK,CAACka,GAAG,CAAC,CACpB9X,KAAK,CAAC7B,QAAQ,CAACqe,GAAG,CAAC,CAAC7M,KAAK,CAACC,UAAU,CAAC,IACxC,CAAC,CAAC,IAAI,CAACS,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CACpBrD,MAAM,CAAC9W,KAAK,CAACka,GAAG,CAAC,CACjB9X,KAAK,CAAC7B,QAAQ,CAACqe,GAAG,CAAC,CAAC7M,KAAK,CAACE,WAAW,CAAE,CAAA;EAChD,SAAA;EACF,OAAA;QACA,IAAI1R,QAAQ,CAACqe,GAAG,CAAC,CAACjT,IAAI,IAAI,MAAM,EAAE;UAChC,IAAI,CAAC3L,KAAK,EAAE;EACV8C,UAAAA,YAAY,CAAC8b,GAAG,CAAC,GAAG,IAAI,CAAA;EAC1B,SAAC,MAAM;EACL,UAAA,IAAIne,KAAK,GAAG,IAAI,CAACkS,SAAS,CAAChC,KAAK,CAACwJ,GAAG,CAAC,IAAI5Z,QAAQ,CAACqe,GAAG,CAAC,CAACtd,SAAS,CAAA;EAEhE,UAAA,KAAK,IAAIsY,IAAI,GAAGjJ,KAAK,CAACwJ,GAAG,EAAEP,IAAI,IAAIhJ,GAAG,CAACuJ,GAAG,EAAEP,IAAI,EAAE,EAAE;EAClD,YAAA,IAAK,IAAI,CAACjH,SAAS,CAACiH,IAAI,CAAC,IAAIrZ,QAAQ,CAACqe,GAAG,CAAC,CAACtd,SAAS,IAAKb,KAAK,EAAE;EAC9DA,cAAAA,KAAK,GAAG,IAAI,CAAA;EACZ,cAAA,MAAA;EACF,aAAA;EACF,WAAA;EACAqC,UAAAA,YAAY,CAAC8b,GAAG,CAAC,GAAGne,KAAK,CAAA;EAC3B,SAAA;EACF,OAAA;EACF,KAAA;EACA,IAAA,OAAOqC,YAAY,CAAA;EACrB,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACEF,EAAAA,eAAeA,CAACrB,OAAO,EAAEd,KAAK,EAAE;MAC9B,IAAIF,QAAQ,CAACgB,OAAO,CAAC,CAACoK,IAAI,IAAI,QAAQ,EAAE;EACtC,MAAA,IAAI1L,MAAM,GAAG,IAAI,CAACoa,YAAY,CAAC,IAAI,CAAC,CAAA;EACpC,MAAA,IAAIra,KAAK,GAAG,IAAI,CAACqa,YAAY,CAAC,KAAK,CAAC,CAAA;EACpC,MAAA,IAAI,CAACpa,MAAM,EAAEA,MAAM,GAAGD,KAAK,CAAA;QAC3B,IAAI,CAACC,MAAM,EAAE,OAAA;EACb,MAAA,IAAIA,MAAM,CAACka,GAAG,IAAIna,KAAK,CAACma,GAAG,EAAE,OAAA;QAC7B,IAAI,CAAC,IAAI,CAACta,yBAAyB,CAACG,KAAK,EAAEC,MAAM,CAAC,EAAE,OAAA;EACpD,MAAA,IAAI4e,UAAU,GAAG,IAAI,CAACJ,0BAA0B,CAC9Cze,KAAK,EACLC,MAAM,EACNM,QAAQ,CAACgB,OAAO,CAAC,CAACD,SACpB,CAAC,CAAA;QACD,IAAI,CAACiV,cAAc,EAAE,CAAA;;EAErB;EACA,MAAA,IAAIsI,UAAU,EAAE;UACd,IAAI,CAAC9L,SAAS,CAAC/S,KAAK,CAACma,GAAG,CAAC,GAAG,IAAI,CAAA;UAChC,MAAM2E,QAAQ,GAAG,IAAI,CAAClE,aAAa,CAACiE,UAAU,EAAE,CAAC,CAAC,CAAA;EAClD,QAAA,MAAME,GAAG,GAAGF,UAAU,CAACtF,WAAW,CAACrX,MAAM,CAAA;EACzC,QAAA,MAAM8c,IAAI,GAAG,IAAI,CAACvM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAC/BrD,MAAM,CAAC,CAAC,EAAEgI,QAAQ,CAAC,CACnBzS,OAAO,CAAC9L,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACC,UAAU,EAAE,EAAE,CAAC,CAAA;EAClD,QAAA,MAAMiN,GAAG,GAAG,IAAI,CAACxM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAACrD,MAAM,CAACgI,QAAQ,EAAEC,GAAG,CAAC,CAAA;EACvD,QAAA,MAAMG,KAAK,GAAG,IAAI,CAACzM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAChCrD,MAAM,CAACgI,QAAQ,GAAGC,GAAG,CAAC,CACtB1S,OAAO,CAAC9L,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACE,WAAW,EAAE,EAAE,CAAC,CAAA;EACnD,QAAA,IAAI,CAACQ,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,GAAG6E,IAAI,CAACzc,MAAM,CAAC0c,GAAG,EAAEC,KAAK,CAAC,CAAA;EAC/Cjf,QAAAA,MAAM,CAACia,GAAG,GAAG8E,IAAI,CAAC9c,MAAM,CAAA;EACxBlC,QAAAA,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,GAAG6E,GAAG,CAAA;UAC5B,IAAI,CAAClK,gBAAgB,EAAE,CAAA;EACvB,QAAA,IAAI,CAAC0G,YAAY,CAACvb,KAAK,EAAEC,MAAM,CAAC,CAAA;UAChC,IAAI,CAAC6U,UAAU,EAAE,CAAA;;EAEjB;EACF,OAAC,MAAM,IACL9U,KAAK,CAACka,GAAG,IAAIja,MAAM,CAACia,GAAG,IACvB,CAAC,CAAC,IAAI,CAACzH,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CACpBrD,MAAM,CAAC,CAAC,EAAE9W,KAAK,CAACka,GAAG,CAAC,CACpB9X,KAAK,CAAC7B,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACC,UAAU,CAAC,IAC5C,CAAC,CAAC,IAAI,CAACS,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CACpBrD,MAAM,CAAC9W,KAAK,CAACka,GAAG,CAAC,CACjB9X,KAAK,CAAC7B,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACE,WAAW,CAAC,EAC7C;UACA,IAAI,CAACc,SAAS,CAAC/S,KAAK,CAACma,GAAG,CAAC,GAAG,IAAI,CAAA;EAChC,QAAA,MAAM6E,IAAI,GAAG,IAAI,CAACvM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAC/BrD,MAAM,CAAC,CAAC,EAAE9W,KAAK,CAACka,GAAG,CAAC,CACpB7N,OAAO,CAAC9L,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACC,UAAU,EAAE,EAAE,CAAC,CAAA;EAClD,QAAA,MAAMkN,KAAK,GAAG,IAAI,CAACzM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAChCrD,MAAM,CAAC9W,KAAK,CAACka,GAAG,CAAC,CACjB7N,OAAO,CAAC9L,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACE,WAAW,EAAE,EAAE,CAAC,CAAA;EACnD,QAAA,IAAI,CAACQ,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,GAAG6E,IAAI,CAACzc,MAAM,CAAC2c,KAAK,CAAC,CAAA;UAC1Clf,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,GAAG8E,IAAI,CAAC9c,MAAM,CAAA;UACpC,IAAI,CAAC2S,gBAAgB,EAAE,CAAA;EACvB,QAAA,IAAI,CAAC0G,YAAY,CAACvb,KAAK,EAAEC,MAAM,CAAC,CAAA;UAChC,IAAI,CAAC6U,UAAU,EAAE,CAAA;;EAEjB;EACF,OAAC,MAAM;EACL;UACA,IAAI;YAAEgK,QAAQ;EAAEK,UAAAA,MAAAA;WAAQ,GACtBnf,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,GAClB;YAAE4E,QAAQ,EAAE9e,KAAK,CAACka,GAAG;YAAEiF,MAAM,EAAElf,MAAM,CAACia,GAAAA;EAAI,SAAC,GAC3C;YAAE4E,QAAQ,EAAE7e,MAAM,CAACia,GAAG;YAAEiF,MAAM,EAAEnf,KAAK,CAACka,GAAAA;WAAK,CAAA;UAEjD,IAAI9X,KAAK,GAAG,IAAI,CAACqQ,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAC9BrD,MAAM,CAACgI,QAAQ,EAAEK,MAAM,GAAGL,QAAQ,CAAC,CACnC1c,KAAK,CAAC,uCAAuC,CAAC,CAAA;EACjD,QAAA,IAAIA,KAAK,EAAE;EACT0c,UAAAA,QAAQ,IAAI1c,KAAK,CAAC8T,MAAM,CAACkJ,OAAO,CAACld,MAAM,CAAA;EACvCid,UAAAA,MAAM,IAAI/c,KAAK,CAAC8T,MAAM,CAACmJ,QAAQ,CAACnd,MAAM,CAAA;EACxC,SAAA;UAEAlC,KAAK,CAACka,GAAG,GAAG4E,QAAQ,CAAA;UACpB7e,MAAM,CAACia,GAAG,GAAGiF,MAAM,CAAA;;EAEnB;UACA,IAAI,CAACrf,aAAa,CAChBS,QAAQ,CAACgB,OAAO,CAAC,CAAC+J,GAAG,CAACuG,GAAG,EACzBtR,QAAQ,CAACgB,OAAO,CAAC,CAAC+J,GAAG,CAACwG,IAAI,EAC1B9R,KAAK,EACLC,MACF,CAAC,CAAA;UACD,IAAI,CAAC6U,UAAU,EAAE,CAAA;EACjB;EACF,OAAA;OACD,MAAM,IAAIvU,QAAQ,CAACgB,OAAO,CAAC,CAACoK,IAAI,IAAI,MAAM,EAAE;EAC3C,MAAA,IAAI1L,MAAM,GAAG,IAAI,CAACoa,YAAY,CAAC,IAAI,CAAC,CAAA;EACpC,MAAA,IAAIra,KAAK,GAAG,IAAI,CAACqa,YAAY,CAAC,KAAK,CAAC,CAAA;EACpC,MAAA,IAAI,CAACpa,MAAM,EAAEA,MAAM,GAAGD,KAAK,CAAA;QAC3B,IAAI,CAACA,KAAK,EAAE,OAAA;QACZ,IAAI,CAACuW,cAAc,EAAE,CAAA;EACrB,MAAA,IAAI5F,KAAK,GAAG1Q,MAAM,CAACka,GAAG,GAAGna,KAAK,CAACma,GAAG,GAAGna,KAAK,GAAGC,MAAM,CAAA;EACnD,MAAA,IAAI2Q,GAAG,GAAG3Q,MAAM,CAACka,GAAG,GAAGna,KAAK,CAACma,GAAG,GAAGla,MAAM,GAAGD,KAAK,CAAA;EACjD,MAAA,IAAI4Q,GAAG,CAACuJ,GAAG,GAAGxJ,KAAK,CAACwJ,GAAG,IAAIvJ,GAAG,CAACsJ,GAAG,IAAI,CAAC,EAAE;UACvCtJ,GAAG,CAACuJ,GAAG,EAAE,CAAA;EACX,OAAA;EAEA,MAAA,KAAK,IAAIP,IAAI,GAAGjJ,KAAK,CAACwJ,GAAG,EAAEP,IAAI,IAAIhJ,GAAG,CAACuJ,GAAG,EAAEP,IAAI,EAAE,EAAE;EAClD,QAAA,IAAInZ,KAAK,IAAI,IAAI,CAACkS,SAAS,CAACiH,IAAI,CAAC,IAAIrZ,QAAQ,CAACgB,OAAO,CAAC,CAACD,SAAS,EAAE;YAChE,IAAI,CAACmR,KAAK,CAACmH,IAAI,CAAC,GAAG,IAAI,CAACnH,KAAK,CAACmH,IAAI,CAAC,CAACvN,OAAO,CACzC9L,QAAQ,CAACgB,OAAO,CAAC,CAAC+J,GAAG,CAAC4G,OAAO,EAC7B3R,QAAQ,CAACgB,OAAO,CAAC,CAAC+J,GAAG,CAACgE,WAAW,CAACjD,OAAO,CACvC,IAAI,EACJuN,IAAI,GAAGjJ,KAAK,CAACwJ,GAAG,GAAG,CACrB,CACF,CAAC,CAAA;EACD,UAAA,IAAI,CAACpH,SAAS,CAAC6G,IAAI,CAAC,GAAG,IAAI,CAAA;EAC7B,SAAA;EACA,QAAA,IAAI,CAACnZ,KAAK,IAAI,IAAI,CAACkS,SAAS,CAACiH,IAAI,CAAC,IAAIrZ,QAAQ,CAACgB,OAAO,CAAC,CAACD,SAAS,EAAE;EACjE,UAAA,IAAI,CAACmR,KAAK,CAACmH,IAAI,CAAC,GAAG,IAAI,CAACnH,KAAK,CAACmH,IAAI,CAAC,CAACvN,OAAO,CACzC9L,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACG,OAAO,EAC/B3R,QAAQ,CAACgB,OAAO,CAAC,CAACwQ,KAAK,CAACzC,WAC1B,CAAC,CAAA;EACD,UAAA,IAAI,CAACyD,SAAS,CAAC6G,IAAI,CAAC,GAAG,IAAI,CAAA;EAC7B,SAAA;EACF,OAAA;QACA,IAAI,CAAC/E,gBAAgB,EAAE,CAAA;QACvB,IAAI,CAAC0G,YAAY,CACf;UAAEpB,GAAG,EAAEvJ,GAAG,CAACuJ,GAAG;UAAED,GAAG,EAAE,IAAI,CAACzH,KAAK,CAAC7B,GAAG,CAACuJ,GAAG,CAAC,CAACjY,MAAAA;EAAO,OAAC,EACjD;UAAEiY,GAAG,EAAExJ,KAAK,CAACwJ,GAAG;EAAED,QAAAA,GAAG,EAAE,CAAA;EAAE,OAC3B,CAAC,CAAA;QACD,IAAI,CAACpF,UAAU,EAAE,CAAA;EACnB,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACA;EACEjV,EAAAA,yBAAyBA,GAAG;EAC1B;EACA,IAAA,MAAMka,GAAG,GAAGxW,MAAM,CAAC8W,YAAY,EAAE,CAAA;EACjC,IAAA,IAAI,CAACN,GAAG,IAAI,CAACA,GAAG,CAACU,SAAS,IAAI,CAACV,GAAG,CAACS,UAAU,EAAE,OAAO,KAAK,CAAA;;EAE3D;;EAEA;MACA,IACET,GAAG,CAACuF,WAAW,IACfvF,GAAG,CAACU,SAAS,CAACM,QAAQ,IAAI,CAAC,IAC3BhB,GAAG,CAACY,WAAW,IAAIZ,GAAG,CAACU,SAAS,CAACa,SAAS,CAACpZ,MAAM,EACjD;EACA,MAAA,IAAI2Y,IAAI,CAAA;EACR,MAAA,KACEA,IAAI,GAAGd,GAAG,CAACU,SAAS,EACpBI,IAAI,IAAIA,IAAI,CAAC5G,WAAW,IAAI,IAAI,EAChC4G,IAAI,GAAGA,IAAI,CAACvH,UAAU,CACvB,CAAA;QACD,IACEuH,IAAI,IACJA,IAAI,CAAC5G,WAAW,CAAC3S,SAAS,IAC1BuZ,IAAI,CAAC5G,WAAW,CAAC3S,SAAS,CAACod,QAAQ,CAAC,mBAAmB,CAAC,EAExD,OAAO,IAAI,CAAA;EACf,KAAA;;EAEA;EACA,IAAA,IAAIa,QAAQ,GAAG,IAAI,CAACrB,qBAAqB,CAACnE,GAAG,CAACU,SAAS,EAAEV,GAAG,CAACS,UAAU,CAAC,CAAA;EACxE,IAAA,IAAI,CAAC+E,QAAQ,EAAE,OAAO,KAAK,CAAA;;EAE3B;EACA,IAAA,OAAOA,QAAQ,IAAIA,QAAQ,IAAI,IAAI,CAACjf,CAAC,EAAE;EACrC,MAAA,IACEif,QAAQ,CAACje,SAAS,IAClB,OAAOie,QAAQ,CAACje,SAAS,CAACod,QAAQ,IAAI,UAAU,KAC/Ca,QAAQ,CAACje,SAAS,CAACod,QAAQ,CAAC,mBAAmB,CAAC,IAC/Ca,QAAQ,CAACje,SAAS,CAACod,QAAQ,CAAC,aAAa,CAAC,CAAC,EAE7C,OAAO,IAAI,CAAA;QACba,QAAQ,GAAGA,QAAQ,CAACjM,UAAU,CAAA;EAChC,KAAA;EAEA,IAAA,OAAO,KAAK,CAAA;EACd,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACA;EACA;EACExT,EAAAA,aAAaA,CAAC+R,GAAG,EAAEC,IAAI,EAA+B;EAAA,IAAA,IAA7B9R,KAAK,GAAA2E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;EAAA,IAAA,IAAE1E,MAAM,GAAA0E,SAAA,CAAAzC,MAAA,GAAA,CAAA,IAAAyC,SAAA,CAAA,CAAA,CAAA,KAAA5B,SAAA,GAAA4B,SAAA,CAAA,CAAA,CAAA,GAAG,IAAI,CAAA;MAClD,IAAI,CAAC3E,KAAK,EAAEA,KAAK,GAAG,IAAI,CAACqa,YAAY,CAAC,KAAK,CAAC,CAAA;MAC5C,IAAI,CAACpa,MAAM,EAAEA,MAAM,GAAG,IAAI,CAACoa,YAAY,CAAC,IAAI,CAAC,CAAA;EAC7C,IAAA,IAAI,CAACra,KAAK,IAAI,CAACC,MAAM,IAAID,KAAK,CAACma,GAAG,IAAIla,MAAM,CAACka,GAAG,EAAE,OAAA;MAClD,IAAI,CAACpH,SAAS,CAAC/S,KAAK,CAACma,GAAG,CAAC,GAAG,IAAI,CAAA;EAEhC,IAAA,MAAM2E,QAAQ,GAAG9e,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,GAAGla,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,CAAA;EAChE,IAAA,MAAMiF,MAAM,GAAGnf,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,GAAGja,MAAM,CAACia,GAAG,GAAGla,KAAK,CAACka,GAAG,CAAA;MAC9D,MAAM8E,IAAI,GAAG,IAAI,CAACvM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAACrD,MAAM,CAAC,CAAC,EAAEgI,QAAQ,CAAC,CAACvc,MAAM,CAACsP,GAAG,CAAC,CAAA;MAClE,MAAMoN,GAAG,GACPE,MAAM,IAAIL,QAAQ,GACd,EAAE,GACF,IAAI,CAACrM,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAACrD,MAAM,CAACgI,QAAQ,EAAEK,MAAM,GAAGL,QAAQ,CAAC,CAAA;EAC/D,IAAA,MAAMI,KAAK,GAAGpN,IAAI,CAACvP,MAAM,CAAC,IAAI,CAACkQ,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,CAACrD,MAAM,CAACqI,MAAM,CAAC,CAAC,CAAA;EAC/D,IAAA,IAAI,CAAC1M,KAAK,CAACzS,KAAK,CAACma,GAAG,CAAC,GAAG6E,IAAI,CAACzc,MAAM,CAAC0c,GAAG,EAAEC,KAAK,CAAC,CAAA;EAC/Cjf,IAAAA,MAAM,CAACia,GAAG,GAAG8E,IAAI,CAAC9c,MAAM,CAAA;MACxBlC,KAAK,CAACka,GAAG,GAAGja,MAAM,CAACia,GAAG,GAAG+E,GAAG,CAAC/c,MAAM,CAAA;MAEnC,IAAI,CAAC2S,gBAAgB,EAAE,CAAA;EACvB,IAAA,IAAI,CAAC0G,YAAY,CAACvb,KAAK,EAAEC,MAAM,CAAC,CAAA;EAClC,GAAA;;EAEA;EACF;EACA;EACA;IACEuf,kBAAkBA,CAACje,OAAO,EAAE;EAC1B,IAAA,IAAI,CAAC,IAAI,CAACyR,gBAAgB,EAAE,IAAI,CAACA,gBAAgB,GAAG,IAAI,CAAC2L,eAAe,EAAE,CAAA;EAC1E,IAAA,IAAI,CAAC/b,eAAe,CAACrB,OAAO,EAAE,CAAC,IAAI,CAACyR,gBAAgB,CAACzR,OAAO,CAAC,CAAC,CAAA;EAChE,GAAA;;EAEA;EACF;EACA;EACEuT,EAAAA,UAAUA,GAAG;EACX,IAAA,IAAI,CAAC,IAAI,CAACtC,QAAQ,IAAI,CAAC,IAAI,CAACS,SAAS,CAACC,MAAM,CAAChR,MAAM,EAAE,OAAA;EACrD,IAAA,MAAMyR,OAAO,GAAG,IAAI,CAACoB,UAAU,EAAE,CAAA;MACjC,IAAI,IAAI,CAACvC,QAAQ,EAAE,IAAI,CAACA,QAAQ,CAACvM,KAAK,GAAG0N,OAAO,CAAA;MAChD,KAAK,IAAI8L,QAAQ,IAAI,IAAI,CAACxM,SAAS,CAACC,MAAM,EAAE;EAC1CuM,MAAAA,QAAQ,CAAC;EACP9L,QAAAA,OAAO,EAAEA,OAAO;UAChB+L,UAAU,EAAE,IAAI,CAACA,UAAAA;EACnB,OAAC,CAAC,CAAA;EACJ,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACExC,EAAAA,aAAaA,GAAG;EACd,IAAA,IAAI,IAAI,CAACjK,SAAS,CAACE,SAAS,IAAI,IAAI,CAACF,SAAS,CAACE,SAAS,CAACjR,MAAM,EAAE;EAC/D,MAAA,IAAIlC,KAAK,GAAG,IAAI,CAACqa,YAAY,CAAC,KAAK,CAAC,CAAA;EACpC,MAAA,IAAIpa,MAAM,GAAG,IAAI,CAACoa,YAAY,CAAC,IAAI,CAAC,CAAA;QACpC,IAAIvX,YAAY,GAAG,IAAI,CAAC6b,eAAe,CAAC3e,KAAK,EAAEC,MAAM,CAAC,CAAA;QACtD,IAAI,IAAI,CAAC+S,gBAAgB,EAAE;UACzBrR,MAAM,CAACC,MAAM,CAAC,IAAI,CAACoR,gBAAgB,EAAElQ,YAAY,CAAC,CAAA;EACpD,OAAC,MAAM;UACL,IAAI,CAACkQ,gBAAgB,GAAGrR,MAAM,CAACC,MAAM,CAAC,EAAE,EAAEkB,YAAY,CAAC,CAAA;EACzD,OAAA;QACA,KAAK,IAAI2c,QAAQ,IAAI,IAAI,CAACxM,SAAS,CAACE,SAAS,EAAE;EAC7CsM,QAAAA,QAAQ,CAAC;EACPzf,UAAAA,KAAK,EAAEA,KAAK;EACZC,UAAAA,MAAM,EAAEA,MAAM;YACd6C,YAAY,EAAE,IAAI,CAACkQ,gBAAAA;EACrB,SAAC,CAAC,CAAA;EACJ,OAAA;EACF,KAAA;EACF,GAAA;;EAEA;EACF;EACA;IACEgK,QAAQA,CAACC,YAAY,EAAE;MACrB,KAAK,IAAIwC,QAAQ,IAAI,IAAI,CAACxM,SAAS,CAACG,IAAI,EAAE;EACxCqM,MAAAA,QAAQ,CAAC;EACPxC,QAAAA,YAAAA;EACF,OAAC,CAAC,CAAA;EACJ,KAAA;EACF,GAAA;;EAEA;EACF;EACA;EACA;EACA;EACEhc,EAAAA,gBAAgBA,CAAC0K,IAAI,EAAE8T,QAAQ,EAAE;EAC/B,IAAA,IAAI9T,IAAI,CAACvJ,KAAK,CAAC,qBAAqB,CAAC,EAAE;QACrC,IAAI,CAAC6Q,SAAS,CAACC,MAAM,CAAC/Q,IAAI,CAACsd,QAAQ,CAAC,CAAA;EACtC,KAAA;EACA,IAAA,IAAI9T,IAAI,CAACvJ,KAAK,CAAC,kCAAkC,CAAC,EAAE;QAClD,IAAI,CAAC6Q,SAAS,CAACE,SAAS,CAAChR,IAAI,CAACsd,QAAQ,CAAC,CAAA;EACzC,KAAA;EACA,IAAA,IAAI9T,IAAI,CAACvJ,KAAK,CAAC,aAAa,CAAC,EAAE;QAC7B,IAAI,CAAC6Q,SAAS,CAACG,IAAI,CAACjR,IAAI,CAACsd,QAAQ,CAAC,CAAA;EACpC,KAAA;EACF,GAAA;EACF;;;;;;;;;","x_google_ignoreList":[2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48]}