(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(factory((global.tippex = global.tippex || {})));
}(this, (function (exports) { 'use strict';

function getLocation ( source, charIndex ) {
	var lines = source.split( '\n' );
	var len = lines.length;

	for ( var i = 0, lineStart = 0; i < len; i += 1 ) {
		var line = lines[i];
		var lineEnd =  lineStart + line.length + 1; // +1 for newline

		if ( lineEnd > charIndex ) {
			return { line: i + 1, column: charIndex - lineStart };
		}

		lineStart = lineEnd;
	}

	throw new Error( ("Could not determine location of character " + charIndex) );
}

var keywords = /(case|default|delete|do|else|in|instanceof|new|return|throw|typeof|void)\s*$/;
var punctuators = /(^|\{|\(|\[\.|;|,|<|>|<=|>=|==|!=|===|!==|\+|-|\*\%|<<|>>|>>>|&|\||\^|!|~|&&|\|\||\?|:|=|\+=|-=|\*=|%=|<<=|>>=|>>>=|&=|\|=|\^=|\/=|\/)\s*$/;
var ambiguous = /(\}|\)|\+\+|--)\s*$/;
var beforeJsx = /^$|[=:;,\(\{\}\[|&+]\s*$/;

function find ( str ) {
	var quote;
	var escapedFrom;
	var regexEnabled = true;
	var pfixOp = false;
	var jsxTagDepth = 0;
	var stack = [];

	var start;
	var found = [];
	var state = base;

	function base ( char, i ) {
		if ( char === '/' ) {
			// could be start of regex literal OR division punctuator. Solution via
			// http://stackoverflow.com/questions/5519596/when-parsing-javascript-what-determines-the-meaning-of-a-slash/27120110#27120110
			var substr = str.substr( 0, i );
			if ( keywords.test( substr ) || punctuators.test( substr ) ) { regexEnabled = true; }
			else if ( ambiguous.test( substr ) && !tokenClosesExpression( substr, found ) ) { regexEnabled = true; } // TODO save this determination for when it's necessary?
			else { regexEnabled = false; }

			return start = i, slash;
		}

		if ( char === '"' || char === "'" ) { return start = i, quote = char, stack.push( base ), string; }
		if ( char === '`' ) { return start = i, templateString; }

		if ( char === '{' ) { return stack.push( base ), base; }
		if ( char === '}' ) { return start = i, stack.pop(); }

		if ( !( pfixOp && /\W/.test( char ) ) ) {
			pfixOp = ( char === '+' && str[ i - 1 ] === '+' ) || ( char === '-' && str[ i - 1 ] === '-' );
		}

		if ( char === '<' ) {
			var substr$1 = str.substr( 0, i );
			substr$1 = _erase( substr$1, found ).trim();
			if ( beforeJsx.test( substr$1 ) ) { return stack.push( base ), jsxTagStart; }
		}

		return base;
	}

	function slash ( char ) {
		if ( char === '/' ) { return lineComment; }
		if ( char === '*' ) { return blockComment; }
		if ( char === '[' ) { return regexEnabled ? regexCharacter : base; }
		if ( char === '\\' ) { return escapedFrom = regex, escaped; }
		return regexEnabled && !pfixOp ? regex : base;
	}

	function regex ( char, i ) {
		if ( char === '[' ) { return regexCharacter; }
		if ( char === '\\' ) { return escapedFrom = regex, escaped; }

		if ( char === '/' ) {
			var end = i + 1;
			var outer = str.slice( start, end );
			var inner = outer.slice( 1, -1 );

			found.push({ start: start, end: end, inner: inner, outer: outer, type: 'regex' });

			return base;
		}

		return regex;
	}

	function regexCharacter ( char ) {
		if ( char === ']' ) { return regex; }
		if ( char === '\\' ) { return escapedFrom = regexCharacter, escaped; }
		return regexCharacter;
	}

	function string ( char, i ) {
		if ( char === '\\' ) { return escapedFrom = string, escaped; }
		if ( char === quote ) {
			var end = i + 1;
			var outer = str.slice( start, end );
			var inner = outer.slice( 1, -1 );

			found.push({ start: start, end: end, inner: inner, outer: outer, type: 'string' });

			return stack.pop();
		}

		return string;
	}

	function escaped () {
		return escapedFrom;
	}

	function templateString ( char, i ) {
		if ( char === '$' ) { return templateStringDollar; }
		if ( char === '\\' ) { return escapedFrom = templateString, escaped; }

		if ( char === '`' ) {
			var end = i + 1;
			var outer = str.slice( start, end );
			var inner = outer.slice( 1, -1 );

			found.push({ start: start, end: end, inner: inner, outer: outer, type: 'templateEnd' });

			return base;
		}

		return templateString;
	}

	function templateStringDollar ( char, i ) {
		if ( char === '{' ) {
			var end = i + 1;
			var outer = str.slice( start, end );
			var inner = outer.slice( 1, -2 );

			found.push({ start: start, end: end, inner: inner, outer: outer, type: 'templateChunk' });

			stack.push( templateString );
			return base;
		}
		return templateString( char, i );
	}

	// JSX is an XML-like extension to ECMAScript
	// https://facebook.github.io/jsx/

	function jsxTagStart ( char ) {
		if ( char === '/' ) { return jsxTagDepth--, jsxTag; }
		return jsxTagDepth++, jsxTag;
	}

	function jsxTag ( char, i ) {
		if ( char === '"' || char === "'" ) { return start = i, quote = char, stack.push( jsxTag ), string; }
		if ( char === '{' ) { return stack.push( jsxTag ), base; }
		if ( char === '>' ) {
			if ( jsxTagDepth <= 0 ) { return base; }
			return jsx;
		}
		if ( char === '/' ) { return jsxTagSelfClosing; }

		return jsxTag;
	}

	function jsxTagSelfClosing ( char ) {
		if ( char === '>' ) {
			jsxTagDepth--;
			if ( jsxTagDepth <= 0 ) { return base; }
			return jsx;
		}

		return jsxTag;
	}

	function jsx ( char ) {
		if ( char === '{' ) { return stack.push( jsx ), base; }
		if ( char === '<' ) { return jsxTagStart; }

		return jsx;
	}

	function lineComment ( char, end ) {
		if ( char === '\n' ) {
			var outer = str.slice( start, end );
			var inner = outer.slice( 2 );

			found.push({ start: start, end: end, inner: inner, outer: outer, type: 'line' });

			return base;
		}

		return lineComment;
	}

	function blockComment ( char ) {
		if ( char === '*' ) { return blockCommentEnding; }
		return blockComment;
	}

	function blockCommentEnding ( char, i ) {
		if ( char === '/' ) {
			var end = i + 1;
			var outer = str.slice( start, end );
			var inner = outer.slice( 2, -2 );

			found.push({ start: start, end: end, inner: inner, outer: outer, type: 'block' });

			return base;
		}

		return blockComment( char );
	}

	for ( var i = 0; i < str.length; i += 1 ) {
		if ( !state ) {
			var ref = getLocation( str, i );
			var line = ref.line;
			var column = ref.column;
			var before = str.slice( 0, i );
			var beforeLine = /(^|\n).+$/.exec( before )[0];
			var after = str.slice( i );
			var afterLine = /.+(\n|$)/.exec( after )[0];

			var snippet = "" + beforeLine + afterLine + "\n" + (Array( beforeLine.length + 1 ).join( ' ' )) + "^";

			throw new Error( ("Unexpected character (" + line + ":" + column + "). If this is valid JavaScript, it's probably a bug in tippex. Please raise an issue at https://github.com/Rich-Harris/tippex/issues – thanks!\n\n" + snippet) );
		}

		state = state( str[i], i );
	}

	return found;
}

function tokenClosesExpression ( substr, found ) {
	substr = _erase( substr, found );

	var token = ambiguous.exec( substr );
	if ( token ) { token = token[1]; }

	if ( token === ')' ) {
		var count = 0;
		var i = substr.length;
		while ( i-- ) {
			if ( substr[i] === ')' ) {
				count += 1;
			}

			if ( substr[i] === '(' ) {
				count -= 1;
				if ( count === 0 ) {
					i -= 1;
					break;
				}
			}
		}

		// if parenthesized expression is immediately preceded by `if`/`while`, it's not closing an expression
		while ( /\s/.test( substr[i - 1] ) ) { i -= 1; }
		if ( substr.slice( i - 2, i ) === 'if' || substr.slice( i - 5, i ) === 'while' ) { return false; }
	}

	// TODO handle }, ++ and -- tokens immediately followed by / character
	return true;
}

function spaces ( count ) {
	var spaces = '';
	while ( count-- ) { spaces += ' '; }
	return spaces;
}

var erasers = {
	string: function (chunk) { return chunk.outer[0] + spaces( chunk.inner.length ) + chunk.outer[0]; },
	line: function (chunk) { return spaces( chunk.outer.length ); },
	block: function (chunk) { return chunk.outer.split( '\n' ).map( function (line) { return spaces( line.length ); } ).join( '\n' ); },
	regex: function (chunk) { return '/' + spaces( chunk.inner.length ) + '/'; },
	templateChunk: function (chunk) { return chunk.outer[0] + spaces( chunk.inner.length ) + '${'; },
	templateEnd: function (chunk) { return chunk.outer[0] + spaces( chunk.inner.length ) + '`'; }
};

function erase ( str ) {
	var found = find( str );
	return _erase( str, found );
}

function _erase ( str, found ) {
	var erased = '';
	var charIndex = 0;

	for ( var i = 0; i < found.length; i += 1 ) {
		var chunk = found[i];
		erased += str.slice( charIndex, chunk.start );
		erased += erasers[ chunk.type ]( chunk );

		charIndex = chunk.end;
	}

	erased += str.slice( charIndex );
	return erased;
}

function makeGlobalRegExp ( original ) {
	var flags = 'g';

	if ( original.multiline ) { flags += 'm'; }
	if ( original.ignoreCase ) { flags += 'i'; }
	if ( original.sticky ) { flags += 'y'; }
	if ( original.unicode ) { flags += 'u'; }

	return new RegExp( original.source, flags );
}

function match ( str, pattern, callback ) {
	var g = pattern.global;
	if ( !g ) { pattern = makeGlobalRegExp( pattern ); }

	var found = find( str );

	var match;
	var chunkIndex = 0;

	while ( match = pattern.exec( str ) ) {
		var chunk = (void 0);

		do {
			chunk = found[ chunkIndex ];

			if ( chunk && chunk.end < match.index ) {
				chunkIndex += 1;
			} else {
				break;
			}
		} while ( chunk );

		if ( !chunk || chunk.start > match.index ) {
			var args = [].slice.call( match ).concat( match.index, str );
			callback.apply( null, args );
			if ( !g ) { break; }
		}
	}
}

function replace ( str, pattern, callback ) {
	var replacements = [];

	match( str, pattern, function ( match ) {
		var start = arguments[ arguments.length - 2 ];
		var end = start + match.length;
		var content = callback.apply( null, arguments );

		replacements.push({ start: start, end: end, content: content });
	});

	var replaced = '';
	var lastIndex = 0;

	for ( var i = 0; i < replacements.length; i += 1 ) {
		var ref = replacements[i];
		var start = ref.start;
		var end = ref.end;
		var content = ref.content;
		replaced += str.slice( lastIndex, start ) + content;

		lastIndex = end;
	}

	replaced += str.slice( lastIndex );

	return replaced;
}

exports.find = find;
exports.erase = erase;
exports.match = match;
exports.replace = replace;

Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=tippex.umd.js.map
