"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var fs = require("fs");
var jest_config_1 = require("jest-config");
var path = require("path");
var tsc = require("typescript");
var setFromArgv = require('jest-config/build/setFromArgv');
function parseConfig(argv) {
    if (argv.config && typeof argv.config === 'string') {
        if (argv.config[0] === '{' && argv.config[argv.config.length - 1] === '}') {
            return JSON.parse(argv.config);
        }
    }
    return argv.config;
}
function loadJestConfigFromFile(filePath, argv) {
    var config = JSON.parse(fs.readFileSync(filePath, 'utf-8'));
    config.rootDir = config.rootDir ?
        path.resolve(path.dirname(filePath), config.rootDir) :
        process.cwd();
    return jest_config_1.normalize(config, argv);
}
function loadJestConfigFromPackage(filePath, argv) {
    var R_OK = fs.constants && fs.constants.R_OK || fs['R_OK'];
    try {
        fs.accessSync(filePath, R_OK);
    }
    catch (e) {
        return null;
    }
    var packageData = require(filePath);
    var config = packageData.jest || {};
    var root = path.dirname(filePath);
    config.rootDir = config.rootDir ? path.resolve(root, config.rootDir) : root;
    return jest_config_1.normalize(config, argv);
}
function readRawConfig(argv, root) {
    var rawConfig = parseConfig(argv);
    if (typeof rawConfig === 'string') {
        return loadJestConfigFromFile(path.resolve(process.cwd(), rawConfig), argv);
    }
    if (typeof rawConfig === 'object') {
        var config = Object.assign({}, rawConfig);
        config.rootDir = config.rootDir || root;
        return jest_config_1.normalize(config, argv);
    }
    var packageConfig = loadJestConfigFromPackage(path.join(root, 'package.json'), argv);
    return packageConfig || jest_config_1.normalize({ rootDir: root }, argv);
}
function getJestConfig(root) {
    var yargs = require('yargs');
    var argv = yargs(process.argv.slice(2)).argv;
    var rawConfig = readRawConfig(argv, root);
    return Object.freeze(setFromArgv(rawConfig, argv));
}
exports.getJestConfig = getJestConfig;
function getTSJestConfig(globals) {
    return (globals && globals['ts-jest']) ? globals['ts-jest'] : {};
}
exports.getTSJestConfig = getTSJestConfig;
function formatTscParserErrors(errors) {
    return errors.map(function (s) { return JSON.stringify(s, null, 4); }).join('\n');
}
function readCompilerOptions(configPath) {
    var loaded = tsc.readConfigFile(configPath, function (file) {
        var read = tsc.sys.readFile(file);
        if (!read) {
            throw new Error("ENOENT: no such file or directory, open '" + configPath + "'");
        }
        return read;
    });
    if (loaded.error) {
        throw new Error(JSON.stringify(loaded.error, null, 4));
    }
    var basePath = path.dirname(configPath);
    var parsedConfig = tsc.parseJsonConfigFileContent(loaded.config, tsc.sys, basePath);
    if (parsedConfig.errors.length > 0) {
        var formattedErrors = formatTscParserErrors(parsedConfig.errors);
        throw new Error("Some errors occurred while attempting to read from " + configPath + ": " + formattedErrors);
    }
    return parsedConfig.options;
}
function getTSConfigOptionFromConfig(globals) {
    if (!globals) {
        return 'tsconfig.json';
    }
    var tsJestConfig = getTSJestConfig(globals);
    if (globals.__TS_CONFIG__) {
        console.warn("Using globals > __TS_CONFIG__ option for setting TS config is deprecated.\nPlease set config using this option:\nglobals > ts-jest > tsConfigFile (string).\nMore information at https://github.com/kulshekhar/ts-jest#tsconfig");
        return globals.__TS_CONFIG__;
    }
    else if (tsJestConfig.tsConfigFile) {
        return tsJestConfig.tsConfigFile;
    }
    return 'tsconfig.json';
}
exports.getTSConfigOptionFromConfig = getTSConfigOptionFromConfig;
function mockGlobalTSConfigSchema(globals) {
    var config = getTSConfigOptionFromConfig(globals);
    return (typeof config === 'string') ?
        { 'ts-jest': { tsConfigFile: config } } :
        { __TS_CONFIG__: config };
}
exports.mockGlobalTSConfigSchema = mockGlobalTSConfigSchema;
function getTSConfig(globals, collectCoverage) {
    if (collectCoverage === void 0) { collectCoverage = false; }
    var config = getTSConfigOptionFromConfig(globals);
    var skipBabel = getTSJestConfig(globals).skipBabel;
    var isReferencedExternalFile = typeof config === 'string';
    if (isReferencedExternalFile) {
        var configFileName = config;
        var configPath = path.resolve(config);
        config = readCompilerOptions(configPath);
        if (configFileName === 'tsconfig.json') {
            config.module = tsc.ModuleKind.CommonJS;
        }
    }
    if (config.inlineSourceMap !== false) {
        config.inlineSourceMap = true;
    }
    if (collectCoverage) {
        delete config.sourceMap;
        config.inlineSourceMap = true;
        config.inlineSources = true;
        delete config.outDir;
    }
    if (isReferencedExternalFile) {
        config.jsx = config.jsx || tsc.JsxEmit.React;
        config.module = config.module || tsc.ModuleKind.CommonJS;
        if (config.allowSyntheticDefaultImports && !skipBabel) {
            config.module = tsc.ModuleKind.ES2015;
        }
        return config;
    }
    else {
        config.jsx = config.jsx || 'react';
        config.module = config.module || 'commmonjs';
        if (config.allowSyntheticDefaultImports && !skipBabel) {
            config.module = 'es2015';
        }
        var converted = tsc.convertCompilerOptionsFromJson(config, undefined);
        if (converted.errors && converted.errors.length > 0) {
            var formattedErrors = formatTscParserErrors(converted.errors);
            throw new Error("Some errors occurred while attempting to convert " + JSON.stringify(config) + ": " + formattedErrors);
        }
        return converted.options;
    }
}
exports.getTSConfig = getTSConfig;
//# sourceMappingURL=data:application/json;base64,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