/**
 * Copyright (c) 2016 shogogg <shogo@studofly.net>
 *
 * This software is released under the MIA License.
 * http://opensource.org/licenses/mit-license.php
 */
export declare abstract class Option<A> {
    /**
     * Returns true if the option is non-empty and the predicate p returns true when applied to the option's value.
     */
    abstract exists(p: (_: A) => boolean): boolean;
    /**
     * Returns the option if it is non-empty and applying the predicate p to the option's value returns true.
     */
    abstract filter(p: (_: A) => boolean): Option<A>;
    /**
     * Returns the option if it is non-empty and applying the predicate p to the option's value returns false.
     */
    abstract filterNot(p: (_: A) => boolean): Option<A>;
    /**
     * Returns the result of applying f to the option's value if the option is non-empty.
     */
    abstract flatMap<B>(f: (_: A) => Option<B>): Option<B>;
    /**
     * Returns the result of applying f to the option's value if the option is non-empty.
     * Otherwise, evaluates expression ifEmpty.
     */
    abstract fold<B>(ifEmpty: () => B): (f: (_: A) => B) => B;
    /**
     * Tests whether a predicate holds for all elements of the option.
     */
    abstract forAll(p: (_: A) => boolean): boolean;
    /**
     * Apply the given procedure f to the option's value, if it is non-empty.
     */
    abstract forEach(f: (_: A) => any): void;
    /**
     * Returns the option's value if the option is non-empty, otherwise throws an error.
     */
    get: A;
    /**
     * Returns the option's value if the option is non-empty, otherwise return the result of evaluating default.
     */
    abstract getOrElse(defaultValue: A | (() => A)): A;
    /**
     * Returns true if the option's value is non-empty, false otherwise.
     */
    isDefined: boolean;
    /**
     * Returns true if the option's value is empty, false otherwise.
     */
    isEmpty: boolean;
    /**
     * Builds a new option by applying a function to all elements of this option.
     */
    abstract map<B>(f: (_: A) => B): Option<B>;
    /**
     * Pattern match signature.
     */
    abstract match<B>(matcher: Matcher<A, B>): B;
    /**
     * Returns true if the option's value is non-empty, false otherwise.
     */
    nonEmpty: boolean;
    /**
     * Returns the option itself if it is non-empty, otherwise return the result of evaluating alternative.
     */
    abstract orElse(alternative: () => Option<A>): Option<A>;
    /**
     * Returns the option's value if it is non-empty, or null if it is empty.
     */
    orNull: A;
    /**
     * Converts the option to an array.
     */
    toArray: Array<A>;
    /**
     * Performs a for-comprehension like flatMap and map operation using the given functions
     */
    abstract forComprehension(...fns: ((x: any) => Option<any>)[]): Option<any>;
}
export interface Matcher<A, B> {
    some: (_: A) => B;
    none: () => B;
}
export declare class Some<A> extends Option<A> {
    private _value;
    constructor(value: A);
    exists(p: (_: A) => boolean): boolean;
    filter(p: (_: A) => boolean): Option<A>;
    filterNot(p: (_: A) => boolean): Option<A>;
    flatMap<B>(f: (_: A) => Option<B>): Option<B>;
    fold<B>(ifEmpty: () => B): (f: (_: A) => B) => B;
    forAll(p: (_: A) => boolean): boolean;
    forEach(f: (_: A) => any): void;
    get: A;
    getOrElse(defaultValue: A | (() => A)): A;
    isDefined: boolean;
    isEmpty: boolean;
    map<B>(f: (_: A) => B): Option<B>;
    match<B>(matcher: Matcher<A, B>): B;
    nonEmpty: boolean;
    orElse(alternative: () => Option<any>): Option<A>;
    orNull: A;
    toArray: Array<A>;
    forComprehension(...fns: ((x: any) => Option<any>)[]): Option<any>;
}
export declare class None extends Option<any> {
    exists(p: (_: any) => boolean): boolean;
    filter(p: (_: any) => boolean): Option<any>;
    filterNot(p: (_: any) => boolean): Option<any>;
    flatMap<B>(f: (_: any) => Option<B>): Option<B>;
    fold<B>(ifEmpty: () => B): (f: (_: any) => B) => B;
    forAll(p: (_: any) => boolean): boolean;
    forEach(f: (_: any) => any): void;
    get: any;
    getOrElse(defaultValue: any | (() => any)): any;
    isDefined: boolean;
    isEmpty: boolean;
    map<B>(f: (_: any) => B): Option<B>;
    match<B>(matcher: Matcher<any, B>): B;
    nonEmpty: boolean;
    orElse(alternative: () => Option<any>): Option<any>;
    orNull: any;
    toArray: Array<any>;
    forComprehension<B>(...fns: ((x: any) => Option<B>)[]): Option<B>;
}
export declare function some<A>(value: A): Some<A>;
export declare const none: None;
export declare function option<A>(value?: A): Option<A>;
