/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/cli_support", ["require", "exports", "tsickle/src/path"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const path = require("tsickle/src/path");
    /**
     * asserts that the given fileName is an absolute path.
     *
     * The TypeScript API works in absolute paths, so we must be careful to resolve
     * paths before handing them over to TypeScript.
     */
    function assertAbsolute(fileName) {
        if (!path.isAbsolute(fileName)) {
            throw new Error(`expected ${JSON.stringify(fileName)} to be absolute`);
        }
    }
    exports.assertAbsolute = assertAbsolute;
    /**
     * Takes a context (ts.SourceFile.fileName of the current file) and the import URL of an ES6
     * import and generates a googmodule module name for the imported module.
     */
    function pathToModuleName(rootModulePath, context, fileName) {
        fileName = fileName.replace(/(\.d)?\.[tj]s$/, '');
        if (fileName[0] === '.') {
            // './foo' or '../foo'.
            // Resolve the path against the dirname of the current module.
            fileName = path.join(path.dirname(context), fileName);
        }
        // TODO(evanm): various tests assume they can import relative paths like
        // 'foo/bar' and have them interpreted as root-relative; preserve that here.
        // Fix this by removing the next line.
        if (!path.isAbsolute(fileName))
            fileName = path.join(rootModulePath, fileName);
        // TODO(evanm): various tests assume they can pass in a 'fileName' like
        // 'goog:foo.bar' and have this function do something reasonable.
        // For correctness, the above must have produced an absolute path.
        // assertAbsolute(fileName);
        if (rootModulePath) {
            fileName = path.relative(rootModulePath, fileName);
        }
        // Replace characters not supported by goog.module.
        const moduleName = fileName.replace(/\/|\\/g, '.').replace(/^[^a-zA-Z_$]/, '_').replace(/[^a-zA-Z0-9._$]/g, '_');
        return moduleName;
    }
    exports.pathToModuleName = pathToModuleName;
});
//# sourceMappingURL=data:application/json;base64,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