/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/decorators", ["require", "exports", "typescript", "tsickle/src/transformer_util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    /**
     * Returns the declarations for the given decorator.
     */
    function getDecoratorDeclarations(decorator, typeChecker) {
        // Walk down the expression to find the identifier of the decorator function.
        let node = decorator;
        while (node.kind !== ts.SyntaxKind.Identifier) {
            if (node.kind === ts.SyntaxKind.Decorator || node.kind === ts.SyntaxKind.CallExpression) {
                node = node.expression;
            }
            else {
                // We do not know how to handle this type of decorator.
                return [];
            }
        }
        let decSym = typeChecker.getSymbolAtLocation(node);
        if (!decSym)
            return [];
        if (decSym.flags & ts.SymbolFlags.Alias) {
            decSym = typeChecker.getAliasedSymbol(decSym);
        }
        return decSym.getDeclarations() || [];
    }
    exports.getDecoratorDeclarations = getDecoratorDeclarations;
    /**
     * Returns true if node has an exporting decorator  (i.e., a decorator with @ExportDecoratedItems
     * in its JSDoc).
     */
    function hasExportingDecorator(node, typeChecker) {
        return node.decorators &&
            node.decorators.some(decorator => isExportingDecorator(decorator, typeChecker));
    }
    exports.hasExportingDecorator = hasExportingDecorator;
    /**
     * Returns true if the given decorator has an @ExportDecoratedItems directive in its JSDoc.
     */
    function isExportingDecorator(decorator, typeChecker) {
        return getDecoratorDeclarations(decorator, typeChecker).some(declaration => {
            const range = transformer_util_1.getAllLeadingComments(declaration);
            if (!range) {
                return false;
            }
            for (const { text } of range) {
                if (/@ExportDecoratedItems\b/.test(text)) {
                    return true;
                }
            }
            return false;
        });
    }
    /**
     * A transform pass that adds goog.reflect.objectProperty calls to the property
     * name string literals that are emitted as part of TypeScript's default
     * decorator output.
     *
     * goog.reflect.objectProperty is a special function that is recognized by
     * Closure Compiler. It is called like goog.reflect.objectProperty('prop', obj)
     * and it is compiled to a string literal that's the property named 'prop' on
     * the obj value.
     *
     * This way, runtime decorators can use the property names (e.g. to register
     * the property as a getter/setter pair) while still being compatible with
     * Closure Compiler's property renaming.
     *
     * Transforms:
     *
     *     tslib_1.__decorate([
     *       decorator,
     *       tslib_1.__metadata("design:type", Object)
     *     ], Foo.prototype, "prop", void 0);
     *
     * Into:
     *
     *     tslib_1.__decorate([
     *           decorator,
     *           tslib_1.__metadata("design:type", Object)
     *         ], Foo.prototype,
     *         __googReflect.objectProperty("prop", Foo.prototype), void 0);
     */
    function transformDecoratorsOutputForClosurePropertyRenaming(diagnostics) {
        return (context) => {
            const result = (sourceFile) => {
                let nodeNeedingGoogReflect = undefined;
                const visitor = (node) => {
                    const replacementNode = rewriteDecorator(node, context);
                    if (replacementNode) {
                        nodeNeedingGoogReflect = node;
                        return replacementNode;
                    }
                    return ts.visitEachChild(node, visitor, context);
                };
                const updatedSourceFile = ts.visitNode(sourceFile, visitor);
                if (nodeNeedingGoogReflect !== undefined) {
                    const statements = [...updatedSourceFile.statements];
                    const googModuleIndex = statements.findIndex(isGoogModuleStatement);
                    if (googModuleIndex === -1) {
                        transformer_util_1.reportDiagnostic(diagnostics, nodeNeedingGoogReflect, 'Internal tsickle error: could not find goog.module statement to import __tsickle_googReflect for decorator compilation.');
                        return sourceFile;
                    }
                    const googRequireReflectObjectProperty = ts.createVariableStatement(undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration('__tsickle_googReflect', undefined, ts.createCall(ts.createPropertyAccess(ts.createIdentifier('goog'), 'require'), undefined, [ts.createStringLiteral('goog.reflect')]))], ts.NodeFlags.Const));
                    // The boilerplate we produce has a goog.module line, then two related
                    // lines dealing with the `module` variable. Insert our goog.require
                    // after that to avoid visually breaking up the module info, and to be
                    // with the rest of the goog.require statements.
                    statements.splice(googModuleIndex + 3, 0, googRequireReflectObjectProperty);
                    updatedSourceFile.statements =
                        ts.setTextRange(ts.createNodeArray(statements), updatedSourceFile.statements);
                }
                return updatedSourceFile;
            };
            return result;
        };
    }
    exports.transformDecoratorsOutputForClosurePropertyRenaming = transformDecoratorsOutputForClosurePropertyRenaming;
    /**
     * If `node` is a call to the tslib __decorate function, this returns a modified
     * call with the string argument replaced with
     * `__tsickle_googReflect.objectProperty('prop', TheClass.prototype)`.
     *
     * Returns undefined if no modification is necessary.
     */
    function rewriteDecorator(node, context) {
        if (!ts.isCallExpression(node)) {
            return;
        }
        const identifier = node.expression;
        if (!ts.isIdentifier(identifier) || identifier.text !== '__decorate') {
            return;
        }
        const args = [...node.arguments];
        if (args.length !== 4) {
            // Some decorators, like class decorators, have fewer arguments, and don't
            // need help to be renaming-safe.
            return;
        }
        const untypedFieldNameLiteral = args[2];
        if (!ts.isStringLiteral(untypedFieldNameLiteral)) {
            // This is allowed, for example:
            //
            //     const prop = Symbol();
            //     class Foo {
            //       @decorate [prop] = 'val';
            //     }
            //
            // Nothing for us to do in that case.
            return;
        }
        const fieldNameLiteral = untypedFieldNameLiteral;
        args[2] = ts.createCall(ts.createPropertyAccess(ts.createIdentifier('__tsickle_googReflect'), 'objectProperty'), undefined, [ts.createStringLiteral(fieldNameLiteral.text), ts.getMutableClone(args[1])]);
        return ts.updateCall(node, node.expression, node.typeArguments, args);
    }
    function isGoogModuleStatement(statement) {
        if (!ts.isExpressionStatement(statement)) {
            return false;
        }
        const expr = statement.expression;
        if (!ts.isCallExpression(expr)) {
            return false;
        }
        if (!ts.isPropertyAccessExpression(expr.expression)) {
            return false;
        }
        const goog = expr.expression.expression;
        if (!ts.isIdentifier(goog)) {
            return false;
        }
        return goog.text === 'goog' && expr.expression.name.text === 'module';
    }
});
//# sourceMappingURL=data:application/json;base64,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