/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/enum_transformer", ["require", "exports", "typescript", "tsickle/src/transformer_util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    /**
     * @fileoverview Transforms TypeScript enum declarations to Closure enum declarations, which
     * look like:
     *
     *     /.. @enum {number} ./
     *     const Foo = {BAR: 0, BAZ: 1, ...};
     *     export {Foo};  // even if originally exported on one line.
     *
     * This declares an enum type for Closure Compiler (and Closure JS users of this TS code).
     * Splitting the enum into declaration and export is required so that local references to the
     * type resolve ("@type {Foo}").
     */
    const ts = require("typescript");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    /** isInNamespace returns true if any of node's ancestors is a namespace (ModuleDeclaration). */
    function isInNamespace(node) {
        // Must use the original node because node might have already been transformed, with node.parent
        // no longer being set.
        let parent = ts.getOriginalNode(node).parent;
        while (parent) {
            if (parent.kind === ts.SyntaxKind.ModuleDeclaration) {
                return true;
            }
            parent = parent.parent;
        }
        return false;
    }
    /**
     * getEnumMemberType computes the type of an enum member by inspecting its initializer expression.
     */
    function getEnumMemberType(typeChecker, member) {
        // Enum members without initialization have type 'number'
        if (!member.initializer) {
            return 'number';
        }
        const type = typeChecker.getTypeAtLocation(member.initializer);
        // Note: checking against 'NumberLike' instead of just 'Number' means this code
        // handles both
        //   MEMBER = 3,  // TypeFlags.NumberLiteral
        // and
        //   MEMBER = someFunction(),  // TypeFlags.Number
        if (type.flags & ts.TypeFlags.NumberLike) {
            return 'number';
        }
        // If the value is not a number, it must be a string.
        // TypeScript does not allow enum members to have any other type.
        return 'string';
    }
    /**
     * getEnumType computes the Closure type of an enum, by iterating through the members and gathering
     * their types.
     */
    function getEnumType(typeChecker, enumDecl) {
        let hasNumber = false;
        let hasString = false;
        for (const member of enumDecl.members) {
            const type = getEnumMemberType(typeChecker, member);
            if (type === 'string') {
                hasString = true;
            }
            else if (type === 'number') {
                hasNumber = true;
            }
        }
        if (hasNumber && hasString) {
            return '?'; // Closure's new type inference doesn't support enums of unions.
        }
        else if (hasNumber) {
            return 'number';
        }
        else if (hasString) {
            return 'string';
        }
        else {
            // Perhaps an empty enum?
            return '?';
        }
    }
    exports.getEnumType = getEnumType;
    /**
     * Transformer factory for the enum transformer. See fileoverview for details.
     */
    function enumTransformer(typeChecker, diagnostics) {
        return (context) => {
            function visitor(node) {
                if (!ts.isEnumDeclaration(node))
                    return ts.visitEachChild(node, visitor, context);
                // TODO(martinprobst): The enum transformer does not work for enums embedded in namespaces,
                // because TS does not support splitting export and declaration ("export {Foo};") in
                // namespaces. tsickle's emit for namespaces is unintelligible for Closure in any case, so
                // this is left to fix for another day.
                if (isInNamespace(node))
                    return ts.visitEachChild(node, visitor, context);
                // TypeScript does not emit any code for ambient enums, so early exit here to prevent the code
                // below from producing runtime values for an ambient structure.
                if (transformer_util_1.isAmbient(node))
                    return ts.visitEachChild(node, visitor, context);
                const isExported = transformer_util_1.hasModifierFlag(node, ts.ModifierFlags.Export);
                const enumType = getEnumType(typeChecker, node);
                const values = [];
                let enumIndex = 0;
                for (const member of node.members) {
                    let enumValue;
                    if (member.initializer) {
                        const enumConstValue = typeChecker.getConstantValue(member);
                        if (typeof enumConstValue === 'number') {
                            enumIndex = enumConstValue + 1;
                            enumValue = ts.createLiteral(enumConstValue);
                        }
                        else if (typeof enumConstValue === 'string') {
                            // tsickle does not care about string enum values. However TypeScript expects compile
                            // time constant enum values to be replaced with their constant expression, and e.g.
                            // doesn't emit imports for modules referenced in them. Because tsickle replaces the
                            // enum with an object literal, i.e. handles the enum transform, it must thus also do
                            // the const value substitution for strings.
                            enumValue = ts.createLiteral(enumConstValue);
                        }
                        else {
                            // Non-numeric enum value (string or an expression).
                            // Emit this initializer expression as-is.
                            // Note: if the member's initializer expression refers to another
                            // value within the enum (e.g. something like
                            //   enum Foo {
                            //     Field1,
                            //     Field2 = Field1 + something(),
                            //   }
                            // Then when we emit the initializer we produce invalid code because
                            // on the Closure side the reference to Field1 has to be namespaced,
                            // e.g. written "Foo.Field1 + something()".
                            // Hopefully this doesn't come up often -- if the enum instead has
                            // something like
                            //     Field2 = Field1 + 3,
                            // then it's still a constant expression and we inline the constant
                            // value in the above branch of this "if" statement.
                            enumValue = visitor(member.initializer);
                        }
                    }
                    else {
                        enumValue = ts.createLiteral(enumIndex);
                        enumIndex++;
                    }
                    values.push(ts.setOriginalNode(ts.setTextRange(ts.createPropertyAssignment(member.name, enumValue), member), member));
                }
                const varDecl = ts.createVariableDeclaration(node.name, /* type */ undefined, ts.createObjectLiteral(ts.setTextRange(ts.createNodeArray(values, true), node.members), true));
                const varDeclStmt = ts.setOriginalNode(ts.setTextRange(ts.createVariableStatement(
                /* modifiers */ undefined, ts.createVariableDeclarationList([varDecl], 
                /* create a const var */ ts.NodeFlags.Const)), node), node);
                const comment = {
                    kind: ts.SyntaxKind.MultiLineCommentTrivia,
                    text: `* @enum {${enumType}} `,
                    hasTrailingNewLine: true,
                    pos: -1,
                    end: -1
                };
                ts.setSyntheticLeadingComments(varDeclStmt, [comment]);
                const name = node.name.getText();
                const resultNodes = [varDeclStmt];
                if (isExported) {
                    // Create a separate export {...} statement, so that the enum name can be used in local
                    // type annotations within the file.
                    resultNodes.push(ts.createExportDeclaration(undefined, undefined, ts.createNamedExports([ts.createExportSpecifier(undefined, name)])));
                }
                if (transformer_util_1.hasModifierFlag(node, ts.ModifierFlags.Const)) {
                    // By TypeScript semantics, const enums disappear after TS compilation.
                    // We still need to generate the runtime value above to make Closure Compiler's type system
                    // happy and allow refering to enums from JS code, but we should at least not emit string
                    // value mappings.
                    return resultNodes;
                }
                // Emit the reverse mapping of foo[foo.BAR] = 'BAR'; lines for number enum members
                for (const member of node.members) {
                    const memberName = member.name;
                    const memberType = getEnumMemberType(typeChecker, member);
                    // Enum members cannot be named with a private identifier, although it
                    // is technically valid in the AST.
                    if (memberType !== 'number' || ts.isPrivateIdentifier(memberName)) {
                        continue;
                    }
                    // TypeScript enum members can have Identifier names or String names.
                    // We need to emit slightly different code to support these two syntaxes:
                    let nameExpr;
                    let memberAccess;
                    if (ts.isIdentifier(memberName)) {
                        // Foo[Foo.ABC] = "ABC";
                        nameExpr = transformer_util_1.createSingleQuoteStringLiteral(memberName.text);
                        // Make sure to create a clean, new identifier, so comments do not get emitted twice.
                        const ident = ts.createIdentifier(transformer_util_1.getIdentifierText(memberName));
                        memberAccess = ts.createPropertyAccess(ts.createIdentifier(name), ident);
                    }
                    else {
                        // Foo[Foo["A B C"]] = "A B C"; or Foo[Foo[expression]] = expression;
                        nameExpr = ts.isComputedPropertyName(memberName) ? memberName.expression : memberName;
                        memberAccess = ts.createElementAccess(ts.createIdentifier(name), nameExpr);
                    }
                    resultNodes.push(ts.createStatement(ts.createAssignment(ts.createElementAccess(ts.createIdentifier(name), memberAccess), nameExpr)));
                }
                return resultNodes;
            }
            return (sf) => visitor(sf);
        };
    }
    exports.enumTransformer = enumTransformer;
});
//# sourceMappingURL=data:application/json;base64,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