/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/externs", ["require", "exports", "typescript", "tsickle/src/annotator_host", "tsickle/src/enum_transformer", "tsickle/src/googmodule", "tsickle/src/jsdoc", "tsickle/src/jsdoc_transformer", "tsickle/src/module_type_translator", "tsickle/src/path", "tsickle/src/transformer_util", "tsickle/src/type_translator"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    /**
     * @fileoverview Externs creates Closure Compiler #externs definitions from the
     * ambient declarations in a TypeScript file.
     *
     * (Note that we cannot write the "@" form of the externs tag, even in comments,
     * because the compiler greps for it in source files(!).  So we write #externs
     * instead.)
     *
     * For example, a
     *   declare interface Foo { bar: string; }
     *
     * Would generate a
     *   /.. #externs ./
     *   /.. @record ./
     *   var Foo = function() {};
     *   /.. @type {string} ./
     *   Foo.prototype.bar;
     *
     * The generated externs indicate to Closure Compiler that symbols are external
     * to the optimization process, i.e. they are provided by outside code. That
     * most importantly means they must not be renamed or removed.
     *
     * A major difficulty here is that TypeScript supports module-scoped external
     * symbols; `.d.ts` files can contain `export`s and `import` other files.
     * Closure Compiler does not have such a concept, so tsickle must emulate the
     * behaviour. It does so by following this scheme:
     *
     * 1. non-module .d.ts produces global symbols
     * 2. module .d.ts produce symbols namespaced to the module, by creating a
     *    mangled name matching the current file's path. tsickle expects outside
     *    code (e.g. build system integration or manually written code) to contain a
     *    goog.module/provide that references the mangled path.
     * 3. declarations in `.ts` files produce types that can be separately emitted
     *    in e.g. an `externs.js`, using `getGeneratedExterns` below.
     *    1. non-exported symbols produce global types, because that's what users
     *       expect and it matches TypeScripts emit, which just references `Foo` for
     *       a locally declared symbol `Foo` in a module. Arguably these should be
     *       wrapped in `declare global { ... }`.
     *    2. exported symbols are scoped to the `.ts` file by prefixing them with a
     *       mangled name. Exported types are re-exported from the JavaScript
     *       `goog.module`, allowing downstream code to reference them. This has the
     *       same problem regarding ambient values as above, it is unclear where the
     *       value symbol would be defined, so for the time being this is
     *       unsupported.
     *
     * The effect of this is that:
     * - symbols in a module (i.e. not globals) are generally scoped to the local
     *   module using a mangled name, preventing symbol collisions on the Closure
     *   side.
     * - importing code can unconditionally refer to and import any symbol defined
     *   in a module `X` as `path.to.module.X`, regardless of whether the defining
     *   location is a `.d.ts` file or a `.ts` file, and regardless whether the
     *   symbol is ambient (assuming there's an appropriate shim).
     * - if there is a shim present, tsickle avoids emitting the Closure namespace
     *   itself, expecting the shim to provide the namespace and initialize it to a
     *   symbol that provides the right value at runtime (i.e. the implementation of
     *   whatever third party library the .d.ts describes).
     */
    const ts = require("typescript");
    const annotator_host_1 = require("tsickle/src/annotator_host");
    const enum_transformer_1 = require("tsickle/src/enum_transformer");
    const googmodule_1 = require("tsickle/src/googmodule");
    const jsdoc = require("tsickle/src/jsdoc");
    const jsdoc_transformer_1 = require("tsickle/src/jsdoc_transformer");
    const module_type_translator_1 = require("tsickle/src/module_type_translator");
    const path = require("tsickle/src/path");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    const type_translator_1 = require("tsickle/src/type_translator");
    /**
     * Symbols that are already declared as externs in Closure, that should
     * be avoided by tsickle's "declare ..." => externs.js conversion.
     */
    const CLOSURE_EXTERNS_BLACKLIST = [
        'exports',
        'global',
        'module',
        // ErrorConstructor is the interface of the Error object itself.
        // tsickle detects that this is part of the TypeScript standard library
        // and assumes it's part of the Closure standard library, but this
        // assumption is wrong for ErrorConstructor.  To properly handle this
        // we'd somehow need to map methods defined on the ErrorConstructor
        // interface into properties on Closure's Error object, but for now it's
        // simpler to just blacklist it.
        'ErrorConstructor',
        'Symbol',
        'WorkerGlobalScope',
    ];
    /**
     * The header to be used in generated externs.  This is not included in the output of
     * generateExterns() because generateExterns() works one file at a time, and typically you create
     * one externs file from the entire compilation unit.
     *
     * Suppressions:
     * - duplicate: because externs might duplicate re-opened definitions from other JS files.
     * - checkTypes: Closure's type system does not match TS'.
     * - undefinedNames: code below tries to be careful not to overwrite previously emitted definitions,
     *   but on the flip side might accidentally miss definitions.
     */
    const EXTERNS_HEADER = `/**
 * @${''}externs
 * @suppress {duplicate,checkTypes}
 */
// NOTE: generated by tsickle, do not edit.
`;
    /**
     * Concatenate all generated externs definitions together into a string, including a file comment
     * header.
     *
     * @param rootDir Project root.  Emitted comments will reference paths relative to this root.
     *    This param is effectively required, but made optional here until Angular is fixed.
     */
    function getGeneratedExterns(externs, rootDir) {
        let allExterns = EXTERNS_HEADER;
        for (const fileName of Object.keys(externs)) {
            const srcPath = rootDir ? path.relative(rootDir, fileName) :
                'ERROR: getGeneratedExterns called without rootDir';
            allExterns += `// externs from ${srcPath}:\n`;
            allExterns += externs[fileName];
        }
        return allExterns;
    }
    exports.getGeneratedExterns = getGeneratedExterns;
    /**
     * isInGlobalAugmentation returns true if declaration is the immediate child of a 'declare global'
     * block.
     */
    function isInGlobalAugmentation(declaration) {
        // declare global { ... } creates a ModuleDeclaration containing a ModuleBlock containing the
        // declaration, with the ModuleDeclaration having the GlobalAugmentation flag set.
        if (!declaration.parent || !declaration.parent.parent)
            return false;
        return (declaration.parent.parent.flags & ts.NodeFlags.GlobalAugmentation) !== 0;
    }
    /**
     * generateExterns generates extern definitions for all ambient declarations in the given source
     * file. It returns a string representation of the Closure JavaScript, not including the initial
     * comment with \@fileoverview and #externs (see above for that).
     */
    function generateExterns(typeChecker, sourceFile, host, moduleResolutionHost, options) {
        let output = '';
        const diagnostics = [];
        const isDts = transformer_util_1.isDtsFileName(sourceFile.fileName);
        const isExternalModule = ts.isExternalModule(sourceFile);
        const mtt = new module_type_translator_1.ModuleTypeTranslator(sourceFile, typeChecker, host, diagnostics, /*isForExterns*/ true);
        // .d.ts files declare symbols. The code below translates these into a form understood by Closure
        // Compiler, converting the type syntax, but also converting symbol names into a form accessible
        // to Closure Compiler.
        // Like regular .ts files, .d.ts can be either scripts or modules. Scripts declare symbols in the
        // global namespace, which has the same semantics in Closure and TypeScript, so the code below
        // emits those with the same name.
        // Modules however declare symbols scoped to the module that can be exported. Closure has no
        // concept of externs that are non-global, so tsickle needs to mangle the symbol names, both at
        // their declaration and at their use site.
        // This mangling happens by wrapping all declared symbols in a namespace based on the file name.
        // This namespace is then essentially the exports object for the ambient module (externs in
        // Closure terms). This namespace is called `moduleNamespace` below:
        let moduleNamespace = '';
        if (isExternalModule) {
            moduleNamespace = annotator_host_1.moduleNameAsIdentifier(host, sourceFile.fileName);
        }
        // Symbols are generated starting in rootNamespace. For script .d.ts with global symbols, this is
        // the empty string. For most module `.d.ts` files, this is the mangled namespace object. The
        // remaining special case are `.d.ts` files containing an `export = something;` statement. In
        // these, the effective exports object, i.e. the object containing the symbols that importing code
        // receives, is different from the main module scope.
        // tsickle handles the `export =` case by generating symbols in a different namespace (escaped
        // with a `_`) below, and then assigning whatever is actually exported into the `moduleNamespace`
        // below.
        let rootNamespace = moduleNamespace;
        // There can only be one export =, and if there is one, there cannot be any other exports.
        const exportAssignment = sourceFile.statements.find(ts.isExportAssignment);
        const hasExportEquals = exportAssignment && exportAssignment.isExportEquals;
        if (hasExportEquals) {
            // If so, move all generated symbols into a different sub-namespace, so that later on we can
            // control what exactly goes on the actual exported namespace.
            rootNamespace = rootNamespace + '_';
        }
        for (const stmt of sourceFile.statements) {
            if (!isDts && !transformer_util_1.hasModifierFlag(stmt, ts.ModifierFlags.Ambient)) {
                continue;
            }
            visitor(stmt, []);
        }
        /**
         * Convert a qualified name from a .d.ts file or declaration context into a mangled identifier.
         * E.g. for a qualified name in `export = someName;` or `import = someName;`.
         * If `someName` is `declare global { namespace someName {...} }`, tsickle must not qualify access
         * to it with the mangled module namespace as it is emitted in the global namespace. Similarly, if
         * the symbol is declared in a non-module context, it must not be mangled.
         */
        function qualifiedNameToMangledIdentifier(name) {
            const entityName = transformer_util_1.getEntityNameText(name);
            let symbol = typeChecker.getSymbolAtLocation(name);
            if (symbol) {
                // If this is an aliased name (e.g. from an import), use the alias to refer to it.
                if (symbol.flags & ts.SymbolFlags.Alias) {
                    symbol = typeChecker.getAliasedSymbol(symbol);
                }
                const alias = mtt.symbolsToAliasedNames.get(symbol);
                if (alias)
                    return alias;
                const isGlobalSymbol = symbol && symbol.declarations && symbol.declarations.some(d => {
                    if (isInGlobalAugmentation(d))
                        return true;
                    // If the declaration's source file is not a module, it must be global.
                    // If it is a module, the identifier must be local to this file, or handled above via the
                    // alias.
                    return !ts.isExternalModule(d.getSourceFile());
                });
                if (isGlobalSymbol)
                    return entityName;
            }
            return rootNamespace + '.' + entityName;
        }
        if (output && isExternalModule) {
            // If tsickle generated any externs and this is an external module, prepend the namespace
            // declaration for it.
            output = `/** @const */\nvar ${rootNamespace} = {};\n` + output;
            let exportedNamespace = rootNamespace;
            if (exportAssignment && hasExportEquals) {
                if (ts.isIdentifier(exportAssignment.expression) ||
                    ts.isQualifiedName(exportAssignment.expression)) {
                    // E.g. export = someName;
                    // If someName is "declare global { namespace someName {...} }", tsickle must not qualify
                    // access to it with module namespace as it is emitted in the global namespace.
                    exportedNamespace = qualifiedNameToMangledIdentifier(exportAssignment.expression);
                }
                else {
                    transformer_util_1.reportDiagnostic(diagnostics, exportAssignment.expression, `export = expression must be a qualified name, got ${ts.SyntaxKind[exportAssignment.expression.kind]}.`);
                }
                // Assign the actually exported namespace object (which lives somewhere under rootNamespace)
                // into the module's namespace.
                emit(`/**\n * export = ${exportAssignment.expression.getText()}\n * @const\n */\n`);
                emit(`var ${moduleNamespace} = ${exportedNamespace};\n`);
            }
            if (isDts && host.provideExternalModuleDtsNamespace) {
                // In a non-shimmed module, create a global namespace. This exists purely for backwards
                // compatiblity, in the medium term all code using tsickle should always use `goog.module`s,
                // so global names should not be neccessary.
                for (const nsExport of sourceFile.statements.filter(ts.isNamespaceExportDeclaration)) {
                    const namespaceName = transformer_util_1.getIdentifierText(nsExport.name);
                    emit(`// export as namespace ${namespaceName}\n`);
                    writeVariableStatement(namespaceName, [], exportedNamespace);
                }
            }
        }
        return { output, diagnostics };
        function emit(str) {
            output += str;
        }
        /**
         * isFirstDeclaration returns true if decl is the first declaration
         * of its symbol.  E.g. imagine
         *   interface Foo { x: number; }
         *   interface Foo { y: number; }
         * we only want to emit the "\@record" for Foo on the first one.
         *
         * The exception are variable declarations, which - in externs - do not assign a value:
         *   /.. \@type {...} ./
         *   var someVariable;
         *   /.. \@type {...} ./
         *   someNamespace.someVariable;
         * If a later declaration wants to add additional properties on someVariable, tsickle must still
         * emit an assignment into the object, as it's otherwise absent.
         */
        function isFirstValueDeclaration(decl) {
            if (!decl.name)
                return true;
            const sym = typeChecker.getSymbolAtLocation(decl.name);
            if (!sym.declarations || sym.declarations.length < 2)
                return true;
            const earlierDecls = sym.declarations.slice(0, sym.declarations.indexOf(decl));
            // Either there are no earlier declarations, or all of them are variables (see above). tsickle
            // emits a value for all other declaration kinds (function for functions, classes, interfaces,
            // {} object for namespaces).
            return earlierDecls.length === 0 || earlierDecls.every(ts.isVariableDeclaration);
        }
        /** Writes the actual variable statement of a Closure variable declaration. */
        function writeVariableStatement(name, namespace, value) {
            const qualifiedName = namespace.concat([name]).join('.');
            if (namespace.length === 0)
                emit(`var `);
            emit(qualifiedName);
            if (value)
                emit(` = ${value}`);
            emit(';\n');
        }
        /**
         * Writes a Closure variable declaration, i.e. the variable statement with a leading JSDoc
         * comment making it a declaration.
         */
        function writeVariableDeclaration(decl, namespace) {
            if (decl.name.kind === ts.SyntaxKind.Identifier) {
                const name = transformer_util_1.getIdentifierText(decl.name);
                if (CLOSURE_EXTERNS_BLACKLIST.indexOf(name) >= 0)
                    return;
                emit(jsdoc.toString([{ tagName: 'type', type: mtt.typeToClosure(decl) }]));
                emit('\n');
                writeVariableStatement(name, namespace);
            }
            else {
                errorUnimplementedKind(decl.name, 'externs for variable');
            }
        }
        /**
         * Emits a JSDoc declaration that merges the signatures of the given function declaration (for
         * overloads), and returns the parameter names chosen.
         */
        function emitFunctionType(decls, extraTags = []) {
            const { tags, parameterNames } = mtt.getFunctionTypeJSDoc(decls, extraTags);
            emit('\n');
            emit(jsdoc.toString(tags));
            return parameterNames;
        }
        function writeFunction(name, params, namespace) {
            const paramsStr = params.join(', ');
            if (namespace.length > 0) {
                let fqn = namespace.join('.');
                if (name.kind === ts.SyntaxKind.Identifier) {
                    fqn += '.'; // computed names include [ ] in their getText() representation.
                }
                fqn += name.getText();
                emit(`${fqn} = function(${paramsStr}) {};\n`);
            }
            else {
                if (name.kind !== ts.SyntaxKind.Identifier) {
                    transformer_util_1.reportDiagnostic(diagnostics, name, 'Non-namespaced computed name in externs');
                }
                emit(`function ${name.getText()}(${paramsStr}) {}\n`);
            }
        }
        function writeEnum(decl, namespace) {
            // E.g. /** @enum {number} */ var COUNTRY = {US: 1, CA: 1};
            const name = transformer_util_1.getIdentifierText(decl.name);
            let members = '';
            const enumType = enum_transformer_1.getEnumType(typeChecker, decl);
            // Closure enums members must have a value of the correct type, but the actual value does not
            // matter in externs.
            const initializer = enumType === 'string' ? `''` : 1;
            for (const member of decl.members) {
                let memberName;
                switch (member.name.kind) {
                    case ts.SyntaxKind.Identifier:
                        memberName = transformer_util_1.getIdentifierText(member.name);
                        break;
                    case ts.SyntaxKind.StringLiteral:
                        const text = member.name.text;
                        if (type_translator_1.isValidClosurePropertyName(text))
                            memberName = text;
                        break;
                    default:
                        break;
                }
                if (!memberName) {
                    members += `  /* TODO: ${ts.SyntaxKind[member.name.kind]}: ${jsdoc_transformer_1.escapeForComment(member.name.getText())} */\n`;
                    continue;
                }
                members += `  ${memberName}: ${initializer},\n`;
            }
            emit(`\n/** @enum {${enumType}} */\n`);
            writeVariableStatement(name, namespace, `{\n${members}}`);
        }
        function writeTypeAlias(decl, namespace) {
            const typeStr = mtt.typeToClosure(decl, undefined);
            emit(`\n/** @typedef {${typeStr}} */\n`);
            writeVariableStatement(transformer_util_1.getIdentifierText(decl.name), namespace);
        }
        function writeType(decl, namespace) {
            const name = decl.name;
            if (!name) {
                transformer_util_1.reportDiagnostic(diagnostics, decl, 'anonymous type in externs');
                return;
            }
            const typeName = namespace.concat([name.getText()]).join('.');
            if (CLOSURE_EXTERNS_BLACKLIST.indexOf(typeName) >= 0)
                return;
            if (isFirstValueDeclaration(decl)) {
                // Emit the 'function' that is actually the declaration of the interface
                // itself.  If it's a class, this function also must include the type
                // annotations of the constructor.
                let paramNames = [];
                const jsdocTags = [];
                let wroteJsDoc = false;
                jsdoc_transformer_1.maybeAddHeritageClauses(jsdocTags, mtt, decl);
                jsdoc_transformer_1.maybeAddTemplateClause(jsdocTags, decl);
                if (decl.kind === ts.SyntaxKind.ClassDeclaration) {
                    // TODO: it appears you can just write 'class Foo { ...' in externs.
                    // This code instead tries to translate it to a function.
                    jsdocTags.push({ tagName: 'constructor' }, { tagName: 'struct' });
                    const ctors = decl
                        .members.filter((m) => m.kind === ts.SyntaxKind.Constructor);
                    if (ctors.length) {
                        const firstCtor = ctors[0];
                        if (ctors.length > 1) {
                            paramNames = emitFunctionType(ctors, jsdocTags);
                        }
                        else {
                            paramNames = emitFunctionType([firstCtor], jsdocTags);
                        }
                        wroteJsDoc = true;
                    }
                }
                else {
                    // Otherwise it's an interface; tag it as structurally typed.
                    jsdocTags.push({ tagName: 'record' }, { tagName: 'struct' });
                }
                if (!wroteJsDoc)
                    emit(jsdoc.toString(jsdocTags));
                writeFunction(name, paramNames, namespace);
            }
            // Process everything except (MethodSignature|MethodDeclaration|Constructor)
            const methods = new Map();
            for (const member of decl.members) {
                switch (member.kind) {
                    case ts.SyntaxKind.PropertySignature:
                    case ts.SyntaxKind.PropertyDeclaration:
                        const prop = member;
                        if (prop.name.kind === ts.SyntaxKind.Identifier) {
                            let type = mtt.typeToClosure(prop);
                            if (prop.questionToken && type === '?') {
                                // An optional 'any' type translates to '?|undefined' in Closure.
                                type = '?|undefined';
                            }
                            emit(jsdoc.toString([{ tagName: 'type', type }]));
                            if (transformer_util_1.hasModifierFlag(prop, ts.ModifierFlags.Static)) {
                                emit(`\n${typeName}.${prop.name.getText()};\n`);
                            }
                            else {
                                emit(`\n${typeName}.prototype.${prop.name.getText()};\n`);
                            }
                            continue;
                        }
                        // TODO: For now property names other than Identifiers are not handled; e.g.
                        //    interface Foo { "123bar": number }
                        break;
                    case ts.SyntaxKind.MethodSignature:
                    case ts.SyntaxKind.MethodDeclaration:
                        const method = member;
                        const isStatic = transformer_util_1.hasModifierFlag(method, ts.ModifierFlags.Static);
                        const methodSignature = `${method.name.getText()}$$$${isStatic ? 'static' : 'instance'}`;
                        if (methods.has(methodSignature)) {
                            methods.get(methodSignature).push(method);
                        }
                        else {
                            methods.set(methodSignature, [method]);
                        }
                        continue;
                    case ts.SyntaxKind.Constructor:
                        continue; // Handled above.
                    default:
                        // Members can include things like index signatures, for e.g.
                        //   interface Foo { [key: string]: number; }
                        // For now, just skip it.
                        break;
                }
                // If we get here, the member wasn't handled in the switch statement.
                let memberName = namespace;
                if (member.name) {
                    memberName = memberName.concat([member.name.getText()]);
                }
                emit(`\n/* TODO: ${ts.SyntaxKind[member.kind]}: ${memberName.join('.')} */\n`);
            }
            // Handle method declarations/signatures separately, since we need to deal with overloads.
            for (const methodVariants of Array.from(methods.values())) {
                const firstMethodVariant = methodVariants[0];
                let parameterNames;
                if (methodVariants.length > 1) {
                    parameterNames = emitFunctionType(methodVariants);
                }
                else {
                    parameterNames = emitFunctionType([firstMethodVariant]);
                }
                const methodNamespace = namespace.concat([name.getText()]);
                // If the method is static, don't add the prototype.
                if (!transformer_util_1.hasModifierFlag(firstMethodVariant, ts.ModifierFlags.Static)) {
                    methodNamespace.push('prototype');
                }
                writeFunction(firstMethodVariant.name, parameterNames, methodNamespace);
            }
        }
        function writeExportDeclaration(exportDeclaration, namespace) {
            if (!exportDeclaration.exportClause) {
                emit(`\n// TODO(tsickle): export * declaration in ${debugLocationStr(exportDeclaration, namespace)}\n`);
                return;
            }
            if (ts.isNamespaceExport(exportDeclaration.exportClause)) {
                // TODO(#1135): Support generating externs using this syntax.
                emit(`\n// TODO(tsickle): export * as declaration in ${debugLocationStr(exportDeclaration, namespace)}\n`);
                return;
            }
            for (const exportSpecifier of exportDeclaration.exportClause.elements) {
                // No need to do anything for properties exported under their original name.
                if (!exportSpecifier.propertyName)
                    continue;
                emit('/** @const */\n');
                writeVariableStatement(exportSpecifier.name.text, namespace, namespace.join('.') + '.' + exportSpecifier.propertyName.text);
            }
        }
        /**
         * Adds aliases for the symbols imported in the given declaration, so that their types get
         * printed as the fully qualified name, and not just as a reference to the local import alias.
         *
         * tsickle generates .js files that (at most) contain a `goog.provide`, but are not
         * `goog.module`s. These files cannot express an aliased import. However Closure Compiler allows
         * referencing types using fully qualified names in such files, so tsickle can resolve the
         * imported module URI and produce `path.to.module.Symbol` as an alias, and use that when
         * referencing the type.
         */
        function addImportAliases(decl) {
            let moduleUri;
            if (ts.isImportDeclaration(decl)) {
                moduleUri = decl.moduleSpecifier.text;
            }
            else if (ts.isExternalModuleReference(decl.moduleReference)) {
                // import foo = require('./bar');
                moduleUri = decl.moduleReference.expression.text;
            }
            else {
                // import foo = bar.baz.bam;
                // handled at call site.
                return;
            }
            if (ts.isImportEqualsDeclaration(decl)) {
                // import foo = require('./bar');
                addImportAlias(decl.name, moduleUri, undefined);
                return;
            }
            // Side effect import, like "import 'somepath';" declares no local aliases.
            if (!decl.importClause)
                return;
            if (decl.importClause.name) {
                // import name from ... -> map to .default on the module.name.
                addImportAlias(decl.importClause.name, moduleUri, 'default');
            }
            const namedBindings = decl.importClause.namedBindings;
            if (!namedBindings)
                return;
            if (ts.isNamespaceImport(namedBindings)) {
                // import * as name -> map directly to the module.name.
                addImportAlias(namedBindings.name, moduleUri, undefined);
            }
            if (ts.isNamedImports(namedBindings)) {
                // import {A as B}, map to module.name.A
                for (const namedBinding of namedBindings.elements) {
                    addImportAlias(namedBinding.name, moduleUri, namedBinding.name);
                }
            }
        }
        /**
         * Adds an import alias for the symbol defined at the given node. Creates an alias name based on
         * the given moduleName and (optionally) the name.
         */
        function addImportAlias(node, moduleUri, name) {
            let symbol = typeChecker.getSymbolAtLocation(node);
            if (!symbol) {
                transformer_util_1.reportDiagnostic(diagnostics, node, `named import has no symbol`);
                return;
            }
            if (symbol.flags & ts.SymbolFlags.Alias) {
                symbol = typeChecker.getAliasedSymbol(symbol);
            }
            const googNamespace = googmodule_1.extractGoogNamespaceImport(moduleUri);
            let aliasName;
            if (googNamespace) {
                aliasName = googNamespace;
            }
            else {
                // While type_translator does add the mangled prefix for ambient declarations, it only does so
                // for non-aliased (i.e. not imported) symbols. That's correct for its use in regular modules,
                // which will have a local symbol for the imported ambient symbol. However within an externs
                // file, there are no imports, so we need to make sure the alias already contains the correct
                // module name, which means the mangled module name in case of imports symbols.
                // This only applies to non-Closure ('goog:') imports.
                const isAmbientModuleDeclaration = symbol.declarations && symbol.declarations.some(d => transformer_util_1.isAmbient(d));
                const fullUri = googmodule_1.resolveModuleName(host, sourceFile.fileName, moduleUri);
                if (isAmbientModuleDeclaration) {
                    aliasName = annotator_host_1.moduleNameAsIdentifier(host, fullUri);
                }
                else {
                    aliasName = host.pathToModuleName(sourceFile.fileName, googmodule_1.resolveModuleName(host, sourceFile.fileName, fullUri));
                }
                if (typeof name === 'string') {
                    aliasName += '.' + name;
                }
                else if (name) {
                    aliasName += '.' + transformer_util_1.getIdentifierText(name);
                }
            }
            mtt.symbolsToAliasedNames.set(symbol, aliasName);
        }
        /**
         * Produces a compiler error that references the Node's kind. This is useful for the "else"
         * branch of code that is attempting to handle all possible input Node types, to ensure all cases
         * covered.
         */
        function errorUnimplementedKind(node, where) {
            transformer_util_1.reportDiagnostic(diagnostics, node, `${ts.SyntaxKind[node.kind]} not implemented in ${where}`);
        }
        /**
         * getNamespaceForLocalDeclaration returns the namespace that should be used for the given
         * declaration, deciding whether to namespace the symbol to the file or whether to create a
         * global name.
         *
         * The function covers these cases:
         * 1) a declaration in a .d.ts
         * 1a) where the .d.ts is an external module     --> namespace
         * 1b) where the .d.ts is not an external module --> global
         * 2) a declaration in a .ts file (all are treated as modules)
         * 2a) that is exported                          --> namespace
         * 2b) that is unexported                        --> global
         *
         * For 1), all symbols in .d.ts should generally be namespaced to the file to avoid collisions.
         * However .d.ts files that are not external modules do declare global names (1b).
         *
         * For 2), ambient declarations in .ts files must be namespaced, for the same collision reasons.
         * The exception is 2b), where in TypeScript, an unexported local "declare const x: string;"
         * creates a symbol that, when used locally, is emitted as just "x". That is, it behaves
         * like a variable declared in a 'declare global' block. Closure Compiler would fail the build if
         * there is no declaration for "x", so tsickle must generate a global external symbol, i.e.
         * without the namespace wrapper.
         */
        function getNamespaceForTopLevelDeclaration(declaration, namespace) {
            // Only use rootNamespace for top level symbols, any other namespacing (global names, nested
            // namespaces) is always kept.
            if (namespace.length !== 0)
                return namespace;
            // All names in a module (external) .d.ts file can only be accessed locally, so they always get
            // namespace prefixed.
            if (isDts && isExternalModule)
                return [rootNamespace];
            // Same for exported declarations in regular .ts files.
            if (transformer_util_1.hasModifierFlag(declaration, ts.ModifierFlags.Export))
                return [rootNamespace];
            // But local declarations in .ts files or .d.ts files (1b, 2b) are global, too.
            return [];
        }
        /**
         * Returns a string representation for the location: either the namespace, or, if empty, the
         * current source file name. This is intended to be included in the emit for warnings, so that
         * users can more easily find where a problematic definition is from.
         *
         * The code below does not use diagnostics to avoid breaking the build for harmless unhandled
         * cases.
         */
        function debugLocationStr(node, namespace) {
            // Use a regex to grab the filename without a path, to make the output stable
            // under bazel where sandboxes use different paths.
            return namespace.join('.') || node.getSourceFile().fileName.replace(/.*[/\\]/, '');
        }
        function visitor(node, namespace) {
            if (node.parent === sourceFile) {
                namespace = getNamespaceForTopLevelDeclaration(node, namespace);
            }
            switch (node.kind) {
                case ts.SyntaxKind.ModuleDeclaration:
                    const decl = node;
                    switch (decl.name.kind) {
                        case ts.SyntaxKind.Identifier:
                            if (decl.flags & ts.NodeFlags.GlobalAugmentation) {
                                // E.g. "declare global { ... }".  Reset to the outer namespace.
                                namespace = [];
                            }
                            else {
                                // E.g. "declare namespace foo {"
                                const name = transformer_util_1.getIdentifierText(decl.name);
                                if (isFirstValueDeclaration(decl)) {
                                    emit('/** @const */\n');
                                    writeVariableStatement(name, namespace, '{}');
                                }
                                namespace = namespace.concat(name);
                            }
                            if (decl.body)
                                visitor(decl.body, namespace);
                            break;
                        case ts.SyntaxKind.StringLiteral:
                            // E.g. "declare module 'foo' {" (note the quotes).
                            // We still want to emit externs for this module, but Closure doesn't provide a
                            // mechanism for module-scoped externs. Instead, we emit in a mangled namespace.
                            // The mangled namespace (after resolving files) matches the emit for an original module
                            // file, so effectively this augments any existing module.
                            const importName = decl.name.text;
                            const importedModuleName = googmodule_1.resolveModuleName({ moduleResolutionHost, options }, sourceFile.fileName, importName);
                            const mangled = annotator_host_1.moduleNameAsIdentifier(host, importedModuleName);
                            emit(`// Derived from: declare module "${importName}"\n`);
                            namespace = [mangled];
                            // Declare "mangled$name" if it's not declared already elsewhere.
                            if (isFirstValueDeclaration(decl)) {
                                emit('/** @const */\n');
                                writeVariableStatement(mangled, [], '{}');
                            }
                            // Declare the contents inside the "mangled$name".
                            if (decl.body)
                                visitor(decl.body, [mangled]);
                            break;
                        default:
                            errorUnimplementedKind(decl.name, 'externs generation of namespace');
                            break;
                    }
                    break;
                case ts.SyntaxKind.ModuleBlock:
                    const block = node;
                    for (const stmt of block.statements) {
                        visitor(stmt, namespace);
                    }
                    break;
                case ts.SyntaxKind.ImportEqualsDeclaration:
                    const importEquals = node;
                    const localName = transformer_util_1.getIdentifierText(importEquals.name);
                    if (importEquals.moduleReference.kind === ts.SyntaxKind.ExternalModuleReference) {
                        addImportAliases(importEquals);
                        break;
                    }
                    const qn = qualifiedNameToMangledIdentifier(importEquals.moduleReference);
                    // @const so that Closure Compiler understands this is an alias.
                    emit('/** @const */\n');
                    writeVariableStatement(localName, namespace, qn);
                    break;
                case ts.SyntaxKind.ClassDeclaration:
                case ts.SyntaxKind.InterfaceDeclaration:
                    writeType(node, namespace);
                    break;
                case ts.SyntaxKind.FunctionDeclaration:
                    const fnDecl = node;
                    const name = fnDecl.name;
                    if (!name) {
                        transformer_util_1.reportDiagnostic(diagnostics, fnDecl, 'anonymous function in externs');
                        break;
                    }
                    // Gather up all overloads of this function.
                    const sym = typeChecker.getSymbolAtLocation(name);
                    const decls = sym.declarations.filter(ts.isFunctionDeclaration);
                    // Only emit the first declaration of each overloaded function.
                    if (fnDecl !== decls[0])
                        break;
                    const params = emitFunctionType(decls);
                    writeFunction(name, params, namespace);
                    break;
                case ts.SyntaxKind.VariableStatement:
                    for (const decl of node.declarationList.declarations) {
                        writeVariableDeclaration(decl, namespace);
                    }
                    break;
                case ts.SyntaxKind.EnumDeclaration:
                    writeEnum(node, namespace);
                    break;
                case ts.SyntaxKind.TypeAliasDeclaration:
                    writeTypeAlias(node, namespace);
                    break;
                case ts.SyntaxKind.ImportDeclaration:
                    addImportAliases(node);
                    break;
                case ts.SyntaxKind.NamespaceExportDeclaration:
                case ts.SyntaxKind.ExportAssignment:
                    // Handled on the file level.
                    break;
                case ts.SyntaxKind.ExportDeclaration:
                    const exportDeclaration = node;
                    writeExportDeclaration(exportDeclaration, namespace);
                    break;
                default:
                    emit(`\n// TODO(tsickle): ${ts.SyntaxKind[node.kind]} in ${debugLocationStr(node, namespace)}\n`);
                    break;
            }
        }
    }
    exports.generateExterns = generateExterns;
});
//# sourceMappingURL=data:application/json;base64,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