/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/fileoverview_comment_transformer", ["require", "exports", "typescript", "tsickle/src/jsdoc", "tsickle/src/path", "tsickle/src/transformer_util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    const jsdoc = require("tsickle/src/jsdoc");
    const path = require("tsickle/src/path");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    /**
     * A set of JSDoc tags that mark a comment as a fileoverview comment. These are recognized by other
     * pieces of infrastructure (Closure Compiler, module system, ...).
     */
    const FILEOVERVIEW_COMMENT_MARKERS = new Set(['fileoverview', 'externs', 'modName', 'mods', 'pintomodule']);
    /**
     * Given a parsed \@fileoverview comment, ensures it has all the attributes we need.
     * This function can be called to modify an existing comment or to make a new one.
     *
     * @param source Original TS source file. Its path is added in \@fileoverview.
     * @param tags Comment as parsed list of tags; modified in-place.
     */
    function augmentFileoverviewComments(options, source, tags) {
        // Ensure we start with a @fileoverview.
        let fileOverview = tags.find(t => t.tagName === 'fileoverview');
        if (!fileOverview) {
            fileOverview = { tagName: 'fileoverview', text: 'added by tsickle' };
            tags.splice(0, 0, fileOverview);
        }
        if (options.rootDir != null) {
            // This comment is read by other tools so it's important that its format
            // doesn't change.
            fileOverview.text += `\nGenerated from: ${path.relative(options.rootDir, source.fileName)}`;
        }
        // Find or create a @suppress tag.
        // Closure compiler barfs if there's a duplicated @suppress tag in a file, so the tag must
        // only appear once and be merged.
        let suppressTag = tags.find(t => t.tagName === 'suppress');
        let suppressions;
        if (suppressTag) {
            suppressions = new Set((suppressTag.type || '').split(',').map(s => s.trim()));
        }
        else {
            suppressTag = { tagName: 'suppress', text: 'checked by tsc' };
            // Special case the @license tag because all text following this tag is
            // treated by the compiler as part of the license, so we need to place the
            // new @suppress tag before @license.
            const licenseTagIndex = tags.findIndex(t => t.tagName === 'license');
            if (licenseTagIndex !== -1) {
                tags.splice(licenseTagIndex, 0, suppressTag);
            }
            else {
                tags.push(suppressTag);
            }
            suppressions = new Set();
        }
        // Ensure our suppressions are included in the @suppress tag:
        // 1) Suppress checkTypes.  We believe the code has already been type-checked by TypeScript,
        // and we cannot model all the TypeScript type decisions in Closure syntax.
        suppressions.add('checkTypes');
        // 2) Suppress extraRequire.  We remove extra requires at the TypeScript level, so any require
        // that gets to the JS level is a load-bearing require.
        suppressions.add('extraRequire');
        // 3) Suppress uselessCode.  We emit an "if (false)" around type declarations,
        // which is flagged as unused code unless we suppress it.
        suppressions.add('uselessCode');
        // 4) Suppress some checks for user errors that TS already checks.
        suppressions.add('missingReturn');
        suppressions.add('unusedPrivateMembers');
        // 5) Suppress checking for @override, because TS doesn't model it.
        suppressions.add('missingOverride');
        // 6) Suppress constantProperty checking, which errors when a namespace is
        // reopened. Namespace reopening happens when one writes namespace foo {}
        // or namespace foo.* {} more than once.
        suppressions.add('constantProperty');
        suppressTag.type = Array.from(suppressions.values()).sort().join(',');
        return tags;
    }
    /**
     * A transformer that ensures the emitted JS file has an \@fileoverview comment that contains an
     * \@suppress {checkTypes} annotation by either adding or updating an existing comment.
     */
    function transformFileoverviewCommentFactory(options, diagnostics) {
        return () => {
            function checkNoFileoverviewComments(context, comments, message) {
                for (const comment of comments) {
                    const parse = jsdoc.parse(comment);
                    if (parse !== null && parse.tags.some(t => FILEOVERVIEW_COMMENT_MARKERS.has(t.tagName))) {
                        // Report a warning; this should not break compilation in third party code.
                        transformer_util_1.reportDiagnostic(diagnostics, context, message, comment.originalRange, ts.DiagnosticCategory.Warning);
                    }
                }
            }
            return (sourceFile) => {
                const text = sourceFile.getFullText();
                let fileComments = [];
                const firstStatement = sourceFile.statements.length && sourceFile.statements[0] || null;
                const originalComments = ts.getLeadingCommentRanges(text, 0) || [];
                if (!firstStatement) {
                    // In an empty source file, all comments are file-level comments.
                    fileComments = transformer_util_1.synthesizeCommentRanges(sourceFile, originalComments);
                }
                else {
                    // Search for the last comment split from the file with a \n\n. All comments before that are
                    // considered fileoverview comments, all comments after that belong to the next
                    // statement(s). If none found, comments remains empty, and the code below will insert a new
                    // fileoverview comment.
                    for (let i = originalComments.length - 1; i >= 0; i--) {
                        const end = originalComments[i].end;
                        if (!text.substring(end).startsWith('\n\n') &&
                            !text.substring(end).startsWith('\r\n\r\n')) {
                            continue;
                        }
                        // This comment is separated from the source file with a double break, marking it (and any
                        // preceding comments) as a file-level comment. Split them off and attach them onto a
                        // NotEmittedStatement, so that they do not get lost later on.
                        const synthesizedComments = jsdoc.synthesizeLeadingComments(firstStatement);
                        const notEmitted = ts.createNotEmittedStatement(sourceFile);
                        // Modify the comments on the firstStatement in place by removing the file-level comments.
                        fileComments = synthesizedComments.splice(0, i + 1);
                        // Move the fileComments onto notEmitted.
                        ts.setSyntheticLeadingComments(notEmitted, fileComments);
                        sourceFile = transformer_util_1.updateSourceFileNode(sourceFile, ts.createNodeArray([notEmitted, firstStatement, ...sourceFile.statements.slice(1)]));
                        break;
                    }
                    // Now walk every top level statement and escape/drop any @fileoverview comments found.
                    // Closure ignores all @fileoverview comments but the last, so tsickle must make sure not to
                    // emit duplicated ones.
                    for (let i = 0; i < sourceFile.statements.length; i++) {
                        const stmt = sourceFile.statements[i];
                        // Accept the NotEmittedStatement inserted above.
                        if (i === 0 && stmt.kind === ts.SyntaxKind.NotEmittedStatement)
                            continue;
                        const comments = jsdoc.synthesizeLeadingComments(stmt);
                        checkNoFileoverviewComments(stmt, comments, `file comments must be at the top of the file, ` +
                            `separated from the file body by an empty line.`);
                    }
                }
                // Closure Compiler considers the *last* comment with @fileoverview (or #externs or
                // @nocompile) that has not been attached to some other tree node to be the file overview
                // comment, and only applies @suppress tags from it. Google-internal tooling considers *any*
                // comment mentioning @fileoverview.
                let fileoverviewIdx = -1;
                let tags = [];
                for (let i = fileComments.length - 1; i >= 0; i--) {
                    const parse = jsdoc.parseContents(fileComments[i].text);
                    if (parse !== null && parse.tags.some(t => FILEOVERVIEW_COMMENT_MARKERS.has(t.tagName))) {
                        fileoverviewIdx = i;
                        tags = parse.tags;
                        break;
                    }
                }
                if (fileoverviewIdx !== -1) {
                    checkNoFileoverviewComments(firstStatement || sourceFile, fileComments.slice(0, fileoverviewIdx), `duplicate file level comment`);
                }
                augmentFileoverviewComments(options, sourceFile, tags);
                const commentText = jsdoc.toStringWithoutStartEnd(tags);
                if (fileoverviewIdx < 0) {
                    // No existing comment to merge with, just emit a new one.
                    return addNewFileoverviewComment(sourceFile, commentText);
                }
                fileComments[fileoverviewIdx].text = commentText;
                // sf does not need to be updated, synthesized comments are mutable.
                return sourceFile;
            };
        };
    }
    exports.transformFileoverviewCommentFactory = transformFileoverviewCommentFactory;
    function addNewFileoverviewComment(sf, commentText) {
        let syntheticFirstStatement = transformer_util_1.createNotEmittedStatement(sf);
        syntheticFirstStatement = ts.addSyntheticTrailingComment(syntheticFirstStatement, ts.SyntaxKind.MultiLineCommentTrivia, commentText, true);
        return transformer_util_1.updateSourceFileNode(sf, ts.createNodeArray([syntheticFirstStatement, ...sf.statements]));
    }
});
//# sourceMappingURL=data:application/json;base64,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