/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/googmodule", ["require", "exports", "typescript", "tsickle/src/path", "tsickle/src/transformer_util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    const path = require("tsickle/src/path");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    /**
     * Returns true if node is a property access of `child` on the identifier `parent`.
     */
    function isPropertyAccess(node, parent, child) {
        if (!ts.isPropertyAccessExpression(node))
            return false;
        return ts.isIdentifier(node.expression) && node.expression.escapedText === parent &&
            node.name.escapedText === child;
    }
    /** isUseStrict returns true if node is a "use strict"; statement. */
    function isUseStrict(node) {
        if (node.kind !== ts.SyntaxKind.ExpressionStatement)
            return false;
        const exprStmt = node;
        const expr = exprStmt.expression;
        if (expr.kind !== ts.SyntaxKind.StringLiteral)
            return false;
        const literal = expr;
        return literal.text === 'use strict';
    }
    /**
     * TypeScript inserts the following code to mark ES moduels in CommonJS:
     *   Object.defineProperty(exports, "__esModule", { value: true });
     * This matches that code snippet.
     */
    function isEsModuleProperty(stmt) {
        // We're matching the explicit source text generated by the TS compiler.
        // Object.defineProperty(exports, "__esModule", { value: true });
        const expr = stmt.expression;
        if (!ts.isCallExpression(expr))
            return false;
        if (!isPropertyAccess(expr.expression, 'Object', 'defineProperty'))
            return false;
        if (expr.arguments.length !== 3)
            return false;
        const [exp, esM, val] = expr.arguments;
        if (!ts.isIdentifier(exp) || exp.escapedText !== 'exports')
            return false;
        if (!ts.isStringLiteral(esM) || esM.text !== '__esModule')
            return false;
        if (!ts.isObjectLiteralExpression(val) || val.properties.length !== 1)
            return false;
        const prop = val.properties[0];
        if (!ts.isPropertyAssignment(prop))
            return false;
        const ident = prop.name;
        if (!ident || !ts.isIdentifier(ident) || ident.text !== 'value')
            return false;
        return prop.initializer.kind === ts.SyntaxKind.TrueKeyword;
    }
    /**
     * Returns the string argument if call is of the form
     *   require('foo')
     */
    function extractRequire(call) {
        // Verify that the call is a call to require(...).
        if (call.expression.kind !== ts.SyntaxKind.Identifier)
            return null;
        const ident = call.expression;
        if (ident.escapedText !== 'require')
            return null;
        // Verify the call takes a single string argument and grab it.
        if (call.arguments.length !== 1)
            return null;
        const arg = call.arguments[0];
        if (arg.kind !== ts.SyntaxKind.StringLiteral)
            return null;
        return arg.text;
    }
    /** Creates a call expression corresponding to `goog.${methodName}(${literal})`. */
    function createGoogCall(methodName, literal) {
        return ts.createCall(ts.createPropertyAccess(ts.createIdentifier('goog'), methodName), undefined, [literal]);
    }
    /**
     * Extracts the namespace part of a goog: import URL, or returns null if the given import URL is not
     * a goog: import.
     *
     * For example, for `import 'goog:foo.Bar';`, returns `foo.Bar`.
     */
    function extractGoogNamespaceImport(tsImport) {
        if (tsImport.match(/^goog:/))
            return tsImport.substring('goog:'.length);
        return null;
    }
    exports.extractGoogNamespaceImport = extractGoogNamespaceImport;
    /**
     * Convert from implicit `import {} from 'pkg'` to a full resolved file name, including any `/index`
     * suffix and also resolving path mappings. TypeScript and many module loaders support the
     * shorthand, but `goog.module` does not, so tsickle needs to resolve the module name shorthand
     * before generating `goog.module` names.
     */
    function resolveModuleName({ options, moduleResolutionHost }, pathOfImportingFile, imported) {
        // The strategy taken here is to use ts.resolveModuleName() to resolve the import to
        // a specific path, which resolves any /index and path mappings.
        const resolved = ts.resolveModuleName(imported, pathOfImportingFile, options, moduleResolutionHost);
        if (!resolved || !resolved.resolvedModule)
            return imported;
        const resolvedModule = resolved.resolvedModule.resolvedFileName;
        // Check if the resolution went into node_modules.
        // Note that the ResolvedModule returned by resolveModuleName() has an
        // attribute isExternalLibraryImport that is documented with
        // "True if resolvedFileName comes from node_modules", but actually it is just
        // true if the absolute path includes node_modules, and is always true when
        // tsickle itself is under a directory named node_modules.
        const relativeResolved = path.relative(options.rootDir || '', resolvedModule);
        if (relativeResolved.indexOf('node_modules') !== -1) {
            // Imports into node_modules resolve through package.json and must be
            // specially handled by the loader anyway.  Return the input.
            return imported;
        }
        // Otherwise return the full resolved file name. This path will be turned into a module name using
        // AnnotatorHost#pathToModuleName, which also takes care of baseUrl and rootDirs.
        return resolved.resolvedModule.resolvedFileName;
    }
    exports.resolveModuleName = resolveModuleName;
    /**
     * importPathToGoogNamespace converts a TS/ES module './import/path' into a goog.module compatible
     * namespace, handling regular imports and `goog:` namespace imports.
     */
    function importPathToGoogNamespace(host, file, tsImport) {
        let modName;
        const nsImport = extractGoogNamespaceImport(tsImport);
        if (nsImport !== null) {
            // This is a namespace import, of the form "goog:foo.bar".
            // Fix it to just "foo.bar".
            modName = nsImport;
        }
        else {
            if (host.convertIndexImportShorthand) {
                tsImport = resolveModuleName(host, file.fileName, tsImport);
            }
            modName = host.pathToModuleName(file.fileName, tsImport);
        }
        return transformer_util_1.createSingleQuoteStringLiteral(modName);
    }
    /**
     * Replace "module.exports = ..." with just "exports = ...". Returns null if `expr` is not an
     * exports assignment.
     */
    function rewriteModuleExportsAssignment(expr) {
        if (!ts.isBinaryExpression(expr.expression))
            return null;
        if (expr.expression.operatorToken.kind !== ts.SyntaxKind.EqualsToken)
            return null;
        if (!isPropertyAccess(expr.expression.left, 'module', 'exports'))
            return null;
        return ts.setOriginalNode(ts.setTextRange(ts.createStatement(ts.createAssignment(ts.createIdentifier('exports'), expr.expression.right)), expr), expr);
    }
    /**
     * Convert a series of comma-separated expressions
     *   x = foo, y(), z.bar();
     * with statements
     *   x = foo; y(); z.bar();
     * This is for handling in particular the case where
     *   exports.x = ..., exports.y = ...;
     * which Closure rejects.
     *
     * @return An array of statements if it converted, or null otherwise.
     */
    function rewriteCommaExpressions(expr) {
        // There are two representation for comma expressions:
        // 1) a tree of "binary expressions" whose contents are comma operators
        const isBinaryCommaExpression = (expr) => ts.isBinaryExpression(expr) && expr.operatorToken.kind === ts.SyntaxKind.CommaToken;
        // or,
        // 2) a "comma list" expression, where the subexpressions are in one array
        const isCommaList = (expr) => expr.kind === ts.SyntaxKind.CommaListExpression;
        if (!isBinaryCommaExpression(expr) && !isCommaList(expr)) {
            return null;
        }
        // Recursively visit comma-separated subexpressions, and collect them all as
        // separate expression statements.
        return visit(expr);
        function visit(expr) {
            if (isBinaryCommaExpression(expr)) {
                return visit(expr.left).concat(visit(expr.right));
            }
            if (isCommaList(expr)) {
                // TODO(blickly): Simplify using flatMap once node 11 available
                return [].concat(...expr.elements.map(visit));
            }
            return [ts.setOriginalNode(ts.createExpressionStatement(expr), expr)];
        }
    }
    /**
     * commonJsToGoogmoduleTransformer returns a transformer factory that converts TypeScript's CommonJS
     * module emit to Closure Compiler compatible goog.module and goog.require statements.
     */
    function commonJsToGoogmoduleTransformer(host, modulesManifest, typeChecker, diagnostics) {
        return (context) => {
            // TS' CommonJS processing uses onSubstituteNode to, at the very end of processing, substitute
            // "modulename.default" for default imports. We intercept the substitution here, check if it's a
            // .default access, then check if the original node (and thus original import) was from a goog:
            // module, and if so, replace with just the module name.
            const previousOnSubstituteNode = context.onSubstituteNode;
            context.enableSubstitution(ts.SyntaxKind.PropertyAccessExpression);
            context.onSubstituteNode = (hint, node) => {
                node = previousOnSubstituteNode(hint, node);
                if (!ts.isPropertyAccessExpression(node))
                    return node;
                if (node.name.text !== 'default')
                    return node;
                if (!ts.isIdentifier(node.expression))
                    return node;
                // Find the import declaration this node comes from.
                // This may be the original node, if the identifier was transformed from it.
                const orig = ts.getOriginalNode(node.expression);
                let importExportDecl;
                if (ts.isImportDeclaration(orig) || ts.isExportDeclaration(orig)) {
                    importExportDecl = orig;
                }
                else {
                    // Alternatively, we can try to find the declaration of the symbol. This only works for
                    // user-written .default accesses, the generated ones do not have a symbol associated as
                    // they are only produced in the CommonJS transformation, after type checking.
                    const sym = typeChecker.getSymbolAtLocation(node.expression);
                    if (!sym)
                        return node;
                    const decls = sym.getDeclarations();
                    if (!decls || !decls.length)
                        return node;
                    const decl = decls[0];
                    if (decl.parent && decl.parent.parent && ts.isImportDeclaration(decl.parent.parent)) {
                        importExportDecl = decl.parent.parent;
                    }
                    else {
                        return node;
                    }
                }
                // If the import declaration's URL is a "goog:..." style namespace, then all ".default"
                // accesses on it should be replaced with the symbol itself.
                // This allows referring to the module-level export of a "goog.module" or "goog.provide" as if
                // it was an ES6 default export.
                if (extractGoogNamespaceImport(importExportDecl.moduleSpecifier.text)) {
                    // Substitute "foo.default" with just "foo".
                    return node.expression;
                }
                return node;
            };
            return (sf) => {
                // In TS2.9, transformers can receive Bundle objects, which this code cannot handle (given
                // that a bundle by definition cannot be a goog.module()). The cast through any is necessary
                // to remain compatible with earlier TS versions.
                // tslint:disable-next-line:no-any
                if (sf['kind'] !== ts.SyntaxKind.SourceFile)
                    return sf;
                // JS scripts (as opposed to modules), must not be rewritten to
                // goog.modules.
                if (host.isJsTranspilation && !isModule(sf)) {
                    return sf;
                }
                let moduleVarCounter = 1;
                /**
                 * Creates a new unique variable to assign side effect imports into. This allows us to re-use
                 * the variable later on for other imports of the same namespace.
                 */
                function nextModuleVar() {
                    return `tsickle_module_${moduleVarCounter++}_`;
                }
                /**
                 * Maps goog.require namespaces to the variable name they are assigned into. E.g.:
                 *     var $varName = goog.require('$namespace'));
                 */
                const namespaceToModuleVarName = new Map();
                /**
                 * maybeCreateGoogRequire returns a `goog.require()` call for the given CommonJS `require`
                 * call. Returns null if `call` is not a CommonJS require.
                 */
                function maybeCreateGoogRequire(original, call, newIdent) {
                    const importedUrl = extractRequire(call);
                    if (!importedUrl)
                        return null;
                    const imp = importPathToGoogNamespace(host, sf, importedUrl);
                    modulesManifest.addReferencedModule(sf.fileName, imp.text);
                    const ident = namespaceToModuleVarName.get(imp.text);
                    let initializer;
                    if (!ident) {
                        namespaceToModuleVarName.set(imp.text, newIdent);
                        initializer = createGoogCall('require', imp);
                    }
                    else {
                        initializer = ident;
                    }
                    // In JS modules it's recommended that users get a handle on the
                    // goog namespace via:
                    //
                    //    import * as goog from 'google3/javascript/closure/goog.js';
                    //
                    // In a goog.module we just want to access the global `goog` value,
                    // so we skip emitting that import as a goog.require.
                    // We check the goog module name so that we also catch relative imports.
                    if (newIdent.escapedText === 'goog' && imp.text === 'google3.javascript.closure.goog') {
                        return transformer_util_1.createNotEmittedStatementWithComments(sf, original);
                    }
                    const varDecl = ts.createVariableDeclaration(newIdent, /* type */ undefined, initializer);
                    const newStmt = ts.createVariableStatement(
                    /* modifiers */ undefined, ts.createVariableDeclarationList([varDecl], 
                    // Use 'const' in ES6 mode so Closure properly forwards type aliases.
                    host.es5Mode ? undefined : ts.NodeFlags.Const));
                    return ts.setOriginalNode(ts.setTextRange(newStmt, original), original);
                }
                /**
                 * Rewrite goog.declareModuleId to something that works in a goog.module.
                 *
                 * goog.declareModuleId exposes a JS module as a goog.module. After we
                 * convert the JS module to a goog.module, what we really want is to
                 * expose the current goog.module at two different module ids. This isn't
                 * possible with the public APIs, but we can make it work at runtime
                 * by writing a record to goog.loadedModules_.
                 *
                 * This only works at runtime, and would fail if compiled by closure
                 * compiler, but that's ok because we only transpile JS in development
                 * mode.
                 */
                function maybeRewriteDeclareModuleId(original, call) {
                    // Verify that the call is a call to goog.declareModuleId(...).
                    if (!ts.isPropertyAccessExpression(call.expression)) {
                        return null;
                    }
                    const propAccess = call.expression;
                    if (propAccess.name.escapedText !== 'declareModuleId') {
                        return null;
                    }
                    if (!ts.isIdentifier(propAccess.expression) ||
                        propAccess.expression.escapedText !== 'goog') {
                        return null;
                    }
                    // Verify the call takes a single string argument and grab it.
                    if (call.arguments.length !== 1) {
                        return null;
                    }
                    const arg = call.arguments[0];
                    if (!ts.isStringLiteral(arg)) {
                        return null;
                    }
                    const moduleId = arg.text;
                    // replace goog.declareModuleId['foo.bar'] with:
                    // goog.loadedModules_['foo.bar'] = {
                    //   exports: exports,
                    //   type: goog.ModuleType.GOOG,
                    //   moduleId: 'foo.bar'
                    // };
                    //
                    // For more info, see `goog.loadModule` in
                    // https://github.com/google/closure-library/blob/master/closure/goog/base.js
                    const newStmt = ts.createStatement(ts.createAssignment(ts.createElementAccess(ts.createPropertyAccess(ts.createIdentifier('goog'), ts.createIdentifier('loadedModules_')), transformer_util_1.createSingleQuoteStringLiteral(moduleId)), ts.createObjectLiteral([
                        ts.createPropertyAssignment('exports', ts.createIdentifier('exports')),
                        ts.createPropertyAssignment('type', ts.createPropertyAccess(ts.createPropertyAccess(ts.createIdentifier('goog'), ts.createIdentifier('ModuleType')), ts.createIdentifier('GOOG'))),
                        ts.createPropertyAssignment('moduleId', transformer_util_1.createSingleQuoteStringLiteral(moduleId)),
                    ])));
                    return ts.setOriginalNode(ts.setTextRange(newStmt, original), original);
                }
                /**
                 * maybeRewriteRequireTslib rewrites a require('tslib') calls to goog.require('tslib'). It
                 * returns the input statement untouched if it does not match.
                 */
                function maybeRewriteRequireTslib(stmt) {
                    if (!ts.isExpressionStatement(stmt))
                        return null;
                    if (!ts.isCallExpression(stmt.expression))
                        return null;
                    const callExpr = stmt.expression;
                    if (!ts.isIdentifier(callExpr.expression) || callExpr.expression.text !== 'require') {
                        return null;
                    }
                    if (callExpr.arguments.length !== 1)
                        return stmt;
                    const arg = callExpr.arguments[0];
                    if (!ts.isStringLiteral(arg) || arg.text !== 'tslib')
                        return null;
                    return ts.setOriginalNode(ts.setTextRange(ts.createStatement(createGoogCall('require', arg)), stmt), stmt);
                }
                /**
                 * Rewrites code generated by `export * as ns from 'ns'` to something like:
                 *
                 * ```
                 * const tsickle_module_n_ = goog.require('ns');
                 * exports.ns = tsickle_module_n_;
                 * ```
                 *
                 * Separating the `goog.require` and `exports.ns` assignment is required by Closure to
                 * correctly infer the type of the exported namespace.
                 */
                function maybeRewriteExportStarAsNs(stmt) {
                    // Ensure this looks something like `exports.ns = require('ns);`.
                    if (!ts.isExpressionStatement(stmt))
                        return null;
                    if (!ts.isBinaryExpression(stmt.expression))
                        return null;
                    if (stmt.expression.operatorToken.kind !== ts.SyntaxKind.EqualsToken)
                        return null;
                    // Ensure the left side of the expression is an access on `exports`.
                    if (!ts.isPropertyAccessExpression(stmt.expression.left))
                        return null;
                    if (!ts.isIdentifier(stmt.expression.left.expression))
                        return null;
                    if (stmt.expression.left.expression.escapedText !== 'exports')
                        return null;
                    // Grab the call to `require`, and exit early if not calling `require`.
                    if (!ts.isCallExpression(stmt.expression.right))
                        return null;
                    const ident = ts.createIdentifier(nextModuleVar());
                    const require = maybeCreateGoogRequire(stmt, stmt.expression.right, ident);
                    if (!require)
                        return null;
                    const exportedName = stmt.expression.left.name;
                    const exportStmt = ts.setOriginalNode(ts.setTextRange(ts.createStatement(ts.createAssignment(ts.createPropertyAccess(ts.createIdentifier('exports'), exportedName), ident)), stmt), stmt);
                    return [require, exportStmt];
                }
                /**
                 * visitTopLevelStatement implements the main CommonJS to goog.module conversion. It visits a
                 * SourceFile level statement and adds a (possibly) transformed representation of it into
                 * statements. It adds at least one node per statement to statements.
                 *
                 * visitTopLevelStatement:
                 * - converts require() calls to goog.require() calls, with or w/o var assignment
                 * - removes "use strict"; and "Object.defineProperty(__esModule)" statements
                 * - converts module.exports assignments to just exports assignments
                 * - splits __exportStar() calls into require and export (this needs two statements)
                 * - makes sure to only import each namespace exactly once, and use variables later on
                 */
                function visitTopLevelStatement(statements, sf, node) {
                    // Handle each particular case by adding node to statements, then return.
                    // For unhandled cases, break to jump to the default handling below.
                    // In JS transpilation mode, always rewrite `require('tslib')` to
                    // goog.require('tslib'), ignoring normal module resolution.
                    if (host.isJsTranspilation) {
                        const rewrittenTsLib = maybeRewriteRequireTslib(node);
                        if (rewrittenTsLib) {
                            statements.push(rewrittenTsLib);
                            return;
                        }
                    }
                    switch (node.kind) {
                        case ts.SyntaxKind.ExpressionStatement: {
                            const exprStmt = node;
                            // Check for "use strict" and certain Object.defineProperty and skip it if necessary.
                            if (isUseStrict(exprStmt) || isEsModuleProperty(exprStmt)) {
                                stmts.push(transformer_util_1.createNotEmittedStatementWithComments(sf, exprStmt));
                                return;
                            }
                            // Check for:
                            //   module.exports = ...;
                            const modExports = rewriteModuleExportsAssignment(exprStmt);
                            if (modExports) {
                                stmts.push(modExports);
                                return;
                            }
                            // Check for use of the comma operator.
                            // This occurs in code like
                            //   exports.a = ..., exports.b = ...;
                            // which we want to change into multiple statements.
                            const commaExpanded = rewriteCommaExpressions(exprStmt.expression);
                            if (commaExpanded) {
                                stmts.push(...commaExpanded);
                                return;
                            }
                            // Check for:
                            //   exports.ns = require('...');
                            // which is generated by the `export * as ns from` syntax.
                            const exportStarAsNs = maybeRewriteExportStarAsNs(exprStmt);
                            if (exportStarAsNs) {
                                stmts.push(...exportStarAsNs);
                                return;
                            }
                            // Check for:
                            //   "require('foo');" (a require for its side effects)
                            const expr = exprStmt.expression;
                            if (!ts.isCallExpression(expr))
                                break;
                            let callExpr = expr;
                            const declaredModuleId = maybeRewriteDeclareModuleId(exprStmt, callExpr);
                            if (declaredModuleId) {
                                statements.push(declaredModuleId);
                                return;
                            }
                            // Handle export * in ES5 mode (in ES6 mode, export * is dereferenced already).
                            // export * creates either a pure top-level '__export(require(...))' or the imported
                            // version, 'tslib.__exportStar(require(...))'. The imported version is only substituted
                            // later on though, so appears as a plain "__exportStar" on the top level here.
                            const isExportStar = (ts.isIdentifier(expr.expression) && expr.expression.text === '__exportStar') ||
                                (ts.isIdentifier(expr.expression) && expr.expression.text === '__export');
                            if (isExportStar)
                                callExpr = expr.arguments[0];
                            const ident = ts.createIdentifier(nextModuleVar());
                            const require = maybeCreateGoogRequire(exprStmt, callExpr, ident);
                            if (!require)
                                break;
                            statements.push(require);
                            // If this is an export star, split it up into the import (created by the maybe call
                            // above), and the export operation. This avoids a Closure complaint about non-top-level
                            // requires.
                            if (isExportStar) {
                                const args = [ident];
                                if (expr.arguments.length > 1)
                                    args.push(expr.arguments[1]);
                                statements.push(ts.createStatement(ts.createCall(expr.expression, undefined, args)));
                            }
                            return;
                        }
                        case ts.SyntaxKind.VariableStatement: {
                            // It's possibly of the form "var x = require(...);".
                            const varStmt = node;
                            // Verify it's a single decl (and not "var x = ..., y = ...;").
                            if (varStmt.declarationList.declarations.length !== 1)
                                break;
                            const decl = varStmt.declarationList.declarations[0];
                            // Grab the variable name (avoiding things like destructuring binds).
                            if (decl.name.kind !== ts.SyntaxKind.Identifier)
                                break;
                            if (!decl.initializer || !ts.isCallExpression(decl.initializer)) {
                                break;
                            }
                            const require = maybeCreateGoogRequire(varStmt, decl.initializer, decl.name);
                            if (!require)
                                break;
                            statements.push(require);
                            return;
                        }
                        default:
                            break;
                    }
                    statements.push(node);
                }
                const moduleName = host.pathToModuleName('', sf.fileName);
                // Register the namespace this file provides.
                modulesManifest.addModule(sf.fileName, moduleName);
                // Convert each top level statement to goog.module.
                const stmts = [];
                for (const stmt of sf.statements) {
                    visitTopLevelStatement(stmts, sf, stmt);
                }
                // Additional statements that will be prepended (goog.module call etc).
                const headerStmts = [];
                // Emit: goog.module('moduleName');
                const googModule = ts.createStatement(createGoogCall('module', transformer_util_1.createSingleQuoteStringLiteral(moduleName)));
                headerStmts.push(googModule);
                // Allow code to use `module.id` to discover its module URL, e.g. to resolve a template URL
                // against. Uses 'var', as this code is inserted in ES6 and ES5 modes. The following pattern
                // ensures closure doesn't throw an error in advanced optimizations mode.
                // var module = module || {id: 'path/to/module.ts'};
                const moduleId = host.fileNameToModuleId(sf.fileName);
                const moduleVarInitializer = ts.createBinary(ts.createIdentifier('module'), ts.SyntaxKind.BarBarToken, ts.createObjectLiteral([ts.createPropertyAssignment('id', transformer_util_1.createSingleQuoteStringLiteral(moduleId))]));
                const modAssign = ts.createVariableStatement(
                /* modifiers */ undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration('module', /* type */ undefined, moduleVarInitializer)]));
                headerStmts.push(modAssign);
                if (!host.es5Mode) {
                    // The module=module assignment suppresses an unused variable warning which may trigger
                    // depending on the project's compilation flags.
                    headerStmts.push(ts.createStatement(ts.createAssignment(ts.createIdentifier('module'), ts.createIdentifier('module'))));
                }
                // Add `goog.require('tslib');` if not JS transpilation, and it hasn't already been required.
                // Rationale:
                // TS gets compiled to Development mode (ES5) and Closure mode (~ES6)
                // sources. Tooling generates module manifests from the Closure version.
                // These manifests are used both with the Closure version and the
                // Development mode version. 'tslib' is sometimes required by the
                // development version but not the Closure version. Inserting the import
                // below unconditionally makes sure that the module manifests are
                // identical between Closure and Development mode, avoiding breakages
                // caused by missing module dependencies.
                if (!host.isJsTranspilation) {
                    // Get a copy of the already resolved module names before calling
                    // resolveModuleName on 'tslib'. Otherwise, resolveModuleName will
                    // add 'tslib' to namespaceToModuleVarName and prevent checking whether
                    // 'tslib' has already been required.
                    const resolvedModuleNames = [...namespaceToModuleVarName.keys()];
                    const tslibModuleName = host.pathToModuleName(sf.fileName, resolveModuleName(host, sf.fileName, 'tslib'));
                    // Only add the extra require if it hasn't already been required
                    if (resolvedModuleNames.indexOf(tslibModuleName) === -1) {
                        const tslibImport = ts.createExpressionStatement(createGoogCall('require', transformer_util_1.createSingleQuoteStringLiteral(tslibModuleName)));
                        // Place the goog.require('tslib') statement right after the goog.module statements
                        headerStmts.push(tslibImport);
                    }
                }
                // Insert goog.module() etc after any leading comments in the source file. The comments have
                // been converted to NotEmittedStatements by transformer_util, which this depends on.
                const insertionIdx = stmts.findIndex(s => s.kind !== ts.SyntaxKind.NotEmittedStatement);
                if (insertionIdx === -1) {
                    stmts.push(...headerStmts);
                }
                else {
                    stmts.splice(insertionIdx, 0, ...headerStmts);
                }
                return ts.updateSourceFileNode(sf, ts.setTextRange(ts.createNodeArray(stmts), sf.statements));
            };
        };
    }
    exports.commonJsToGoogmoduleTransformer = commonJsToGoogmoduleTransformer;
    function isModule(sourceFile) {
        return Boolean(sourceFile.externalModuleIndicator);
    }
});
//# sourceMappingURL=data:application/json;base64,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