/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/jsdoc", ["require", "exports", "typescript"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    /**
     * A list of all JSDoc tags allowed by the Closure compiler.
     * All tags other than these are escaped before emitting.
     *
     * Note that some of these tags are also rejected by tsickle when seen in
     * the user-provided source, but also that tsickle itself may generate some of these.
     * This whitelist is just used for controlling the output.
     *
     * The public Closure docs don't list all the tags it allows; this list comes
     * from the compiler source itself.
     * https://github.com/google/closure-compiler/blob/master/src/com/google/javascript/jscomp/parsing/Annotation.java
     * https://github.com/google/closure-compiler/blob/master/src/com/google/javascript/jscomp/parsing/ParserConfig.properties
     */
    const JSDOC_TAGS_OUTPUT_WHITELIST = new Set([
        'abstract',
        'argument',
        'author',
        'const',
        'constant',
        'constructor',
        'copyright',
        'define',
        'deprecated',
        'desc',
        'dict',
        'disposes',
        'enhance',
        'enhanceable',
        'enum',
        'export',
        'expose',
        'extends',
        'externs',
        'fileoverview',
        'final',
        'hassoydelcall',
        'hassoydeltemplate',
        'hidden',
        'id',
        'idGenerator',
        'ignore',
        'implements',
        'implicitCast',
        'inheritDoc',
        'interface',
        'jaggerInject',
        'jaggerModule',
        'jaggerProvide',
        'jaggerProvidePromise',
        'lends',
        'license',
        'link',
        'meaning',
        'modifies',
        'modName',
        'mods',
        'ngInject',
        'noalias',
        'nocollapse',
        'nocompile',
        'nosideeffects',
        'override',
        'owner',
        'package',
        'param',
        'pintomodule',
        'polymer',
        'polymerBehavior',
        'preserve',
        'preserveTry',
        'private',
        'protected',
        'public',
        'record',
        'requirecss',
        'requires',
        'return',
        'returns',
        'see',
        'struct',
        'suppress',
        'template',
        'this',
        'throws',
        'type',
        'typedef',
        'unrestricted',
        'version',
        'wizaction',
        'wizmodule',
    ]);
    /**
     * A list of JSDoc @tags that are never allowed in TypeScript source. These are Closure tags that
     * can be expressed in the TypeScript surface syntax. As tsickle's emit will mangle type names,
     * these will cause Closure Compiler issues and should not be used.
     * Note: 'template' is special-cased below; see where this set is queried.
     */
    const JSDOC_TAGS_INPUT_BLACKLIST = new Set([
        'augments', 'class', 'constructs', 'constructor', 'enum', 'extends', 'field',
        'function', 'implements', 'interface', 'lends', 'namespace', 'private', 'public',
        'record', 'static', 'template', 'this', 'type', 'typedef',
    ]);
    /**
     * JSDoc \@tags that might include a {type} after them. Specifying a type is forbidden, since it
     * would collide with TypeScript's type information. If a type *is* given, the entire tag will be
     * ignored.
     */
    const JSDOC_TAGS_WITH_TYPES = new Set([
        'const',
        'define',
        'export',
        'param',
        'return',
    ]);
    /**
     * parse parses JSDoc out of a comment string.
     * Returns null if comment is not JSDoc.
     */
    // TODO(martinprobst): representing JSDoc as a list of tags is too simplistic. We need functionality
    // such as merging (below), de-duplicating certain tags (@deprecated), and special treatment for
    // others (e.g. @suppress). We should introduce a proper model class with a more suitable data
    // strucure (e.g. a Map<TagName, Values[]>).
    function parse(comment) {
        // TODO(evanm): this is a pile of hacky regexes for now, because we
        // would rather use the better TypeScript implementation of JSDoc
        // parsing.  https://github.com/Microsoft/TypeScript/issues/7393
        if (comment.kind !== ts.SyntaxKind.MultiLineCommentTrivia)
            return null;
        // comment.text does not include /* and */, so must start with '*' for JSDoc.
        if (comment.text[0] !== '*')
            return null;
        const text = comment.text.substring(1).trim();
        return parseContents(text);
    }
    exports.parse = parse;
    /**
     * Returns the input string with line endings normalized to '\n'.
     */
    function normalizeLineEndings(input) {
        return input.replace(/\r\n/g, '\n');
    }
    exports.normalizeLineEndings = normalizeLineEndings;
    /**
     * parseContents parses JSDoc out of a comment text.
     * Returns null if comment is not JSDoc.
     *
     * @param commentText a comment's text content, i.e. the comment w/o /* and * /.
     */
    function parseContents(commentText) {
        // Make sure we have proper line endings before parsing on Windows.
        commentText = normalizeLineEndings(commentText);
        // Strip all the " * " bits from the front of each line.
        commentText = commentText.replace(/^\s*\*? ?/gm, '');
        const lines = commentText.split('\n');
        const tags = [];
        const warnings = [];
        for (const line of lines) {
            let match = line.match(/^\s*@(\S+) *(.*)/);
            if (match) {
                let [_, tagName, text] = match;
                if (tagName === 'returns') {
                    // A synonym for 'return'.
                    tagName = 'return';
                }
                let type;
                if (JSDOC_TAGS_INPUT_BLACKLIST.has(tagName)) {
                    if (tagName !== 'template') {
                        // Tell the user to not write blacklisted tags, because there is TS
                        // syntax available for them.
                        warnings.push(`@${tagName} annotations are redundant with TypeScript equivalents`);
                        continue; // Drop the tag so Closure won't process it.
                    }
                    else {
                        // But @template in particular is special: it's ok for the user to
                        // write it for documentation purposes, but we don't want the
                        // user-written one making it into the output because Closure interprets
                        // it as well.
                        // Drop it without any warning.  (We also don't ensure its correctness.)
                        continue;
                    }
                }
                else if (JSDOC_TAGS_WITH_TYPES.has(tagName)) {
                    if (text[0] === '{') {
                        warnings.push(`the type annotation on @${tagName} is redundant with its TypeScript type, ` +
                            `remove the {...} part`);
                        continue;
                    }
                }
                else if (tagName === 'suppress') {
                    const typeMatch = text.match(/^\{(.*)\}(.*)$/);
                    if (typeMatch) {
                        [, type, text] = typeMatch;
                    }
                    else {
                        warnings.push(`malformed @${tagName} tag: "${text}"`);
                    }
                }
                else if (tagName === 'dict') {
                    warnings.push('use index signatures (`[k: string]: type`) instead of @dict');
                    continue;
                }
                // Grab the parameter name from @param tags.
                let parameterName;
                if (tagName === 'param') {
                    match = text.match(/^(\S+) ?(.*)/);
                    if (match)
                        [_, parameterName, text] = match;
                }
                const tag = { tagName };
                if (parameterName)
                    tag.parameterName = parameterName;
                if (text)
                    tag.text = text;
                if (type)
                    tag.type = type;
                tags.push(tag);
            }
            else {
                // Text without a preceding @tag on it is either the plain text
                // documentation or a continuation of a previous tag.
                if (tags.length === 0) {
                    tags.push({ tagName: '', text: line });
                }
                else {
                    const lastTag = tags[tags.length - 1];
                    lastTag.text = (lastTag.text || '') + '\n' + line;
                }
            }
        }
        if (warnings.length > 0) {
            return { tags, warnings };
        }
        return { tags };
    }
    exports.parseContents = parseContents;
    /**
     * Serializes a Tag into a string usable in a comment.
     * Returns a string like " @foo {bar} baz" (note the whitespace).
     */
    function tagToString(tag, escapeExtraTags = new Set()) {
        let out = '';
        if (tag.tagName) {
            if (!JSDOC_TAGS_OUTPUT_WHITELIST.has(tag.tagName) || escapeExtraTags.has(tag.tagName)) {
                // Escape tags we don't understand.  This is a subtle
                // compromise between multiple issues.
                // 1) If we pass through these non-Closure tags, the user will
                //    get a warning from Closure, and the point of tsickle is
                //    to insulate the user from Closure.
                // 2) The output of tsickle is for Closure but also may be read
                //    by humans, for example non-TypeScript users of Angular.
                // 3) Finally, we don't want to warn because users should be
                //    free to add whichever JSDoc they feel like.  If the user
                //    wants help ensuring they didn't typo a tag, that is the
                //    responsibility of a linter.
                out += ` \\@${tag.tagName}`;
            }
            else {
                out += ` @${tag.tagName}`;
            }
        }
        if (tag.type) {
            out += ' {';
            if (tag.restParam) {
                out += '...';
            }
            out += tag.type;
            if (tag.optional) {
                out += '=';
            }
            out += '}';
        }
        if (tag.parameterName) {
            out += ' ' + tag.parameterName;
        }
        if (tag.text) {
            out += ' ' + tag.text.replace(/@/g, '\\@');
        }
        return out;
    }
    /** Tags that must only occur onces in a comment (filtered below). */
    const SINGLETON_TAGS = new Set(['deprecated']);
    /** Tags that conflict with \@type in Closure Compiler (e.g. \@param). */
    exports.TAGS_CONFLICTING_WITH_TYPE = new Set(['param', 'return']);
    /**
     * synthesizeLeadingComments parses the leading comments of node, converts them
     * to synthetic comments, and makes sure the original text comments do not get
     * emitted by TypeScript.
     */
    function synthesizeLeadingComments(node) {
        const existing = ts.getSyntheticLeadingComments(node);
        if (existing)
            return existing;
        const text = node.getFullText();
        const synthComments = getLeadingCommentRangesSynthesized(text, node.getFullStart());
        if (synthComments.length) {
            ts.setSyntheticLeadingComments(node, synthComments);
            suppressLeadingCommentsRecursively(node);
        }
        return synthComments;
    }
    exports.synthesizeLeadingComments = synthesizeLeadingComments;
    /**
     * parseLeadingCommentRangesSynthesized parses the leading comment ranges out of the given text and
     * converts them to SynthesizedComments.
     * @param offset the offset of text in the source file, e.g. node.getFullStart().
     */
    // VisibleForTesting
    function getLeadingCommentRangesSynthesized(text, offset = 0) {
        const comments = ts.getLeadingCommentRanges(text, 0) || [];
        return comments.map((cr) => {
            // Confusingly, CommentRange in TypeScript includes start and end markers, but
            // SynthesizedComments do not.
            const commentText = cr.kind === ts.SyntaxKind.SingleLineCommentTrivia ?
                text.substring(cr.pos + 2, cr.end) :
                text.substring(cr.pos + 2, cr.end - 2);
            return Object.assign(Object.assign({}, cr), { text: commentText, pos: -1, end: -1, originalRange: { pos: cr.pos + offset, end: cr.end + offset } });
        });
    }
    exports.getLeadingCommentRangesSynthesized = getLeadingCommentRangesSynthesized;
    /**
     * suppressCommentsRecursively prevents emit of leading comments on node, and any recursive nodes
     * underneath it that start at the same offset.
     */
    function suppressLeadingCommentsRecursively(node) {
        // TypeScript emits leading comments on a node, unless:
        // - the comment was emitted by the parent node
        // - the node has the NoLeadingComments emit flag.
        // However, transformation steps sometimes copy nodes without keeping their emit flags, so just
        // setting NoLeadingComments recursively is not enough, we must also set the text range to avoid
        // the copied node to have comments emitted.
        const originalStart = node.getFullStart();
        const actualStart = node.getStart();
        function suppressCommentsInternal(node) {
            ts.setEmitFlags(node, ts.EmitFlags.NoLeadingComments);
            return !!ts.forEachChild(node, (child) => {
                if (child.pos !== originalStart)
                    return true;
                return suppressCommentsInternal(child);
            });
        }
        suppressCommentsInternal(node);
    }
    exports.suppressLeadingCommentsRecursively = suppressLeadingCommentsRecursively;
    function toSynthesizedComment(tags, escapeExtraTags) {
        return {
            kind: ts.SyntaxKind.MultiLineCommentTrivia,
            text: toStringWithoutStartEnd(tags, escapeExtraTags),
            pos: -1,
            end: -1,
            hasTrailingNewLine: true,
        };
    }
    exports.toSynthesizedComment = toSynthesizedComment;
    /** Serializes a Comment out to a string, but does not include the start and end comment tokens. */
    function toStringWithoutStartEnd(tags, escapeExtraTags = new Set()) {
        return serialize(tags, false, escapeExtraTags);
    }
    exports.toStringWithoutStartEnd = toStringWithoutStartEnd;
    /** Serializes a Comment out to a string usable in source code. */
    function toString(tags, escapeExtraTags = new Set()) {
        return serialize(tags, true, escapeExtraTags);
    }
    exports.toString = toString;
    function serialize(tags, includeStartEnd, escapeExtraTags = new Set()) {
        if (tags.length === 0)
            return '';
        if (tags.length === 1) {
            const tag = tags[0];
            if ((tag.tagName === 'type' || tag.tagName === 'typedef' || tag.tagName === 'nocollapse') &&
                (!tag.text || !tag.text.match('\n'))) {
                // Special-case one-liner "type" and "nocollapse" tags to fit on one line, e.g.
                //   /** @type {foo} */
                const text = tagToString(tag, escapeExtraTags);
                return includeStartEnd ? `/**${text} */` : `*${text} `;
            }
            // Otherwise, fall through to the multi-line output.
        }
        let out = includeStartEnd ? '/**\n' : '*\n';
        const emitted = new Set();
        for (const tag of tags) {
            if (emitted.has(tag.tagName) && SINGLETON_TAGS.has(tag.tagName)) {
                continue;
            }
            emitted.add(tag.tagName);
            out += ' *';
            // If the tagToString is multi-line, insert " * " prefixes on subsequent lines.
            out += tagToString(tag, escapeExtraTags).split('\n').join('\n * ');
            out += '\n';
        }
        out += includeStartEnd ? ' */\n' : ' ';
        return out;
    }
    /** Merges multiple tags (of the same tagName type) into a single unified tag. */
    function merge(tags) {
        const tagNames = new Set();
        const parameterNames = new Set();
        const types = new Set();
        const texts = new Set();
        // If any of the tags are optional/rest, then the merged output is optional/rest.
        let optional = false;
        let restParam = false;
        for (const tag of tags) {
            tagNames.add(tag.tagName);
            if (tag.parameterName !== undefined)
                parameterNames.add(tag.parameterName);
            if (tag.type !== undefined)
                types.add(tag.type);
            if (tag.text !== undefined)
                texts.add(tag.text);
            if (tag.optional)
                optional = true;
            if (tag.restParam)
                restParam = true;
        }
        if (tagNames.size !== 1) {
            throw new Error(`cannot merge differing tags: ${JSON.stringify(tags)}`);
        }
        const tagName = tagNames.values().next().value;
        const parameterName = parameterNames.size > 0 ? Array.from(parameterNames).join('_or_') : undefined;
        const type = types.size > 0 ? Array.from(types).join('|') : undefined;
        // @template uses text (not type!) to declare its type parameters, with ','-separated text.
        const isTemplateTag = tagName === 'template';
        const text = texts.size > 0 ? Array.from(texts).join(isTemplateTag ? ',' : ' / ') : undefined;
        const tag = { tagName, parameterName, type, text };
        // Note: a param can either be optional or a rest param; if we merged an
        // optional and rest param together, prefer marking it as a rest param.
        if (restParam) {
            tag.restParam = true;
        }
        else if (optional) {
            tag.optional = true;
        }
        return tag;
    }
    exports.merge = merge;
});
//# sourceMappingURL=data:application/json;base64,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