/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/jsdoc_transformer", ["require", "exports", "typescript", "tsickle/src/annotator_host", "tsickle/src/decorators", "tsickle/src/googmodule", "tsickle/src/jsdoc", "tsickle/src/module_type_translator", "tsickle/src/transformer_util", "tsickle/src/transformer_util", "tsickle/src/type_translator"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    /**
     * @fileoverview jsdoc_transformer contains the logic to add JSDoc comments to TypeScript code.
     *
     * One of tsickle's features is to add Closure Compiler compatible JSDoc comments containing type
     * annotations, inheritance information, etc., onto TypeScript code. This allows Closure Compiler to
     * make better optimization decisions compared to an untyped code base.
     *
     * The entry point to the annotation operation is jsdocTransformer below. It adds synthetic comments
     * to existing TypeScript constructs, for example:
     *     const x: number = 1;
     * Might get transformed to:
     *     /.. \@type {number} ./
     *     const x: number = 1;
     * Later TypeScript phases then remove the type annotation, and the final emit is JavaScript that
     * only contains the JSDoc comment.
     *
     * To handle certain constructs, this transformer also performs AST transformations, e.g. by adding
     * CommonJS-style exports for type constructs, expanding `export *`, parenthesizing casts, etc.
     */
    const ts = require("typescript");
    const annotator_host_1 = require("tsickle/src/annotator_host");
    const decorators_1 = require("tsickle/src/decorators");
    const googmodule = require("tsickle/src/googmodule");
    const jsdoc = require("tsickle/src/jsdoc");
    const module_type_translator_1 = require("tsickle/src/module_type_translator");
    const transformerUtil = require("tsickle/src/transformer_util");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    const type_translator_1 = require("tsickle/src/type_translator");
    function addCommentOn(node, tags, escapeExtraTags) {
        const comment = jsdoc.toSynthesizedComment(tags, escapeExtraTags);
        const comments = ts.getSyntheticLeadingComments(node) || [];
        comments.push(comment);
        ts.setSyntheticLeadingComments(node, comments);
        return comment;
    }
    /** Adds an \@template clause to docTags if decl has type parameters. */
    function maybeAddTemplateClause(docTags, decl) {
        if (!decl.typeParameters)
            return;
        // Closure does not support template constraints (T extends X), these are ignored below.
        docTags.push({
            tagName: 'template',
            text: decl.typeParameters.map(tp => transformerUtil.getIdentifierText(tp.name)).join(', ')
        });
    }
    exports.maybeAddTemplateClause = maybeAddTemplateClause;
    /**
     * Adds heritage clauses (\@extends, \@implements) to the given docTags for decl. Used by
     * jsdoc_transformer and externs generation.
     */
    function maybeAddHeritageClauses(docTags, mtt, decl) {
        if (!decl.heritageClauses)
            return;
        const isClass = decl.kind === ts.SyntaxKind.ClassDeclaration;
        const hasExtends = decl.heritageClauses.some(c => c.token === ts.SyntaxKind.ExtendsKeyword);
        for (const heritage of decl.heritageClauses) {
            const isExtends = heritage.token === ts.SyntaxKind.ExtendsKeyword;
            if (isClass && isExtends) {
                // If a class has an "extends", that is preserved in the ES6 output
                // and we don't need to emit any additional jsdoc.
                //
                // However for ambient declarations, we only emit externs, and in those we do need to
                // add "@extends {Foo}" as they use ES5 syntax.
                if (!transformerUtil.isAmbient(decl))
                    continue;
            }
            // Otherwise, if we get here, we need to emit some jsdoc.
            for (const expr of heritage.types) {
                addHeritage(isExtends ? 'extends' : 'implements', hasExtends, expr);
            }
        }
        /**
         * Adds the relevant Closure JSdoc tags for an expression occurring in a heritage clause,
         * e.g. "implements FooBar" => "@implements {FooBar}".
         *
         * Will return null if the expression is inexpressible in Closure semantics.
         *
         * Note that we don't need to consider all possible combinations of types/values and
         * extends/implements because our input is already verified to be valid TypeScript.  See
         * test_files/class/ for the full cartesian product of test cases.
         *
         * @param hasAnyExtends True if there are any 'extends' clauses present at all.
         */
        function addHeritage(relation, hasAnyExtends, expr) {
            let tagName = relation;
            let sym = mtt.typeChecker.getSymbolAtLocation(expr.expression);
            if (!sym) {
                // It's possible for a class declaration to extend an expression that
                // does not have have a symbol, for example when a mixin function is
                // used to build a base class, as in `declare MyClass extends
                // MyMixin(MyBaseClass)`.
                //
                // Handling this correctly is tricky. Closure throws on this
                // `extends <expression>` syntax (see
                // https://github.com/google/closure-compiler/issues/2182). We would
                // probably need to generate an intermediate class declaration and
                // extend that.
                warn(expr, `dropped ${relation} of non-symbol supertype: ${expr.getText()}`);
                return;
            }
            // Resolve any aliases to the underlying type.
            if (sym.flags & ts.SymbolFlags.TypeAlias) {
                // It's implementing a type alias.  Follow the type alias back
                // to the original symbol to check whether it's a type or a value.
                const type = mtt.typeChecker.getDeclaredTypeOfSymbol(sym);
                if (!type.symbol) {
                    // It's not clear when this can happen.
                    warn(decl, `could not get type of symbol: ${expr.getText()}`);
                    return;
                }
                sym = type.symbol;
            }
            if (sym.flags & ts.SymbolFlags.Alias) {
                sym = mtt.typeChecker.getAliasedSymbol(sym);
            }
            const typeTranslator = mtt.newTypeTranslator(expr.expression);
            if (typeTranslator.isBlackListed(sym)) {
                // Don't emit references to blacklisted types.
                warn(decl, `dropped ${relation} of blacklisted type: ${expr.getText()}`);
                return;
            }
            if (sym.flags & ts.SymbolFlags.Class) {
                if (!isClass) {
                    // Closure interfaces cannot extend or implements classes.
                    warn(decl, `dropped interface ${relation} class: ${expr.getText()}`);
                    return;
                }
                if (relation !== 'extends') {
                    if (!hasAnyExtends) {
                        // A special case: for a class that has no existing 'extends' clause but does
                        // have an 'implements' clause that refers to another class, we change it to
                        // instead be an 'extends'.  This was a poorly-thought-out hack that may
                        // actually cause compiler bugs:
                        //   https://github.com/google/closure-compiler/issues/3126
                        // but we have code that now relies on it, ugh.
                        tagName = 'extends';
                    }
                    else {
                        // Closure can only @implements an interface, not a class.
                        warn(decl, `dropped implements of class: ${expr.getText()}`);
                        return;
                    }
                }
            }
            else if (sym.flags & ts.SymbolFlags.Value) {
                // If the symbol came from tsickle emit and it's something other than a class in the value
                // namespace, then tsickle may not have emitted the type.
                if (!type_translator_1.typeValueConflictHandled(sym)) {
                    warn(decl, `dropped ${relation} of a type/value conflict: ${expr.getText()}`);
                    return;
                }
            }
            else if (sym.flags & ts.SymbolFlags.TypeLiteral) {
                // A type literal is a type like `{foo: string}`.
                // These can come up as the output of a mapped type.
                warn(decl, `dropped ${relation} of a type literal: ${expr.getText()}`);
                return;
            }
            // typeToClosure includes nullability modifiers, so call symbolToString directly here.
            const parentName = typeTranslator.symbolToString(sym);
            if (!parentName) {
                warn(decl, `dropped ${relation} of unnameable type: ${expr.getText()}`);
                return;
            }
            docTags.push({
                tagName,
                type: parentName,
            });
        }
        /** Records a warning, both in the source text and in the emit host. */
        function warn(context, message) {
            docTags.push({ tagName: '', text: `tsickle: ${message}` });
            mtt.debugWarn(context, message);
        }
    }
    exports.maybeAddHeritageClauses = maybeAddHeritageClauses;
    /**
     * createMemberTypeDeclaration emits the type annotations for members of a class. It's necessary in
     * the case where TypeScript syntax specifies there are additional properties on the class, because
     * to declare these in Closure you must declare these separately from the class.
     *
     * createMemberTypeDeclaration produces an if (false) statement containing property declarations, or
     * null if no declarations could or needed to be generated (e.g. no members, or an unnamed type).
     * The if statement is used to make sure the code is not executed, otherwise property accesses could
     * trigger getters on a superclass. See test_files/fields/fields.ts:BaseThatThrows.
     */
    function createMemberTypeDeclaration(mtt, typeDecl) {
        // Gather parameter properties from the constructor, if it exists.
        const ctors = [];
        let paramProps = [];
        const nonStaticProps = [];
        const staticProps = [];
        const unhandled = [];
        const abstractMethods = [];
        for (const member of typeDecl.members) {
            if (member.kind === ts.SyntaxKind.Constructor) {
                ctors.push(member);
            }
            else if (ts.isPropertyDeclaration(member) || ts.isPropertySignature(member)) {
                const isStatic = transformerUtil.hasModifierFlag(member, ts.ModifierFlags.Static);
                if (isStatic) {
                    staticProps.push(member);
                }
                else {
                    nonStaticProps.push(member);
                }
            }
            else if (member.kind === ts.SyntaxKind.MethodDeclaration ||
                member.kind === ts.SyntaxKind.MethodSignature ||
                member.kind === ts.SyntaxKind.GetAccessor || member.kind === ts.SyntaxKind.SetAccessor) {
                if (transformerUtil.hasModifierFlag(member, ts.ModifierFlags.Abstract) ||
                    ts.isInterfaceDeclaration(typeDecl)) {
                    abstractMethods.push(member);
                }
                // Non-abstract methods only exist on classes, and are handled in regular emit.
            }
            else {
                unhandled.push(member);
            }
        }
        if (ctors.length > 0) {
            // Only the actual constructor implementation, which must be last in a potential sequence of
            // overloaded constructors, may contain parameter properties.
            const ctor = ctors[ctors.length - 1];
            paramProps = ctor.parameters.filter(p => transformerUtil.hasModifierFlag(p, ts.ModifierFlags.ParameterPropertyModifier));
        }
        if (nonStaticProps.length === 0 && paramProps.length === 0 && staticProps.length === 0 &&
            abstractMethods.length === 0) {
            // There are no members so we don't need to emit any type
            // annotations helper.
            return null;
        }
        if (!typeDecl.name) {
            mtt.debugWarn(typeDecl, 'cannot add types on unnamed declarations');
            return null;
        }
        const className = transformerUtil.getIdentifierText(typeDecl.name);
        const staticPropAccess = ts.createIdentifier(className);
        const instancePropAccess = ts.createPropertyAccess(staticPropAccess, 'prototype');
        // Closure Compiler will report conformance errors about this being unknown type when emitting
        // class properties as {?|undefined}, instead of just {?}. So make sure to only emit {?|undefined}
        // on interfaces.
        const isInterface = ts.isInterfaceDeclaration(typeDecl);
        const propertyDecls = staticProps.map(p => createClosurePropertyDeclaration(mtt, staticPropAccess, p, isInterface && !!p.questionToken));
        propertyDecls.push(...[...nonStaticProps, ...paramProps].map(p => createClosurePropertyDeclaration(mtt, instancePropAccess, p, isInterface && !!p.questionToken)));
        propertyDecls.push(...unhandled.map(p => transformerUtil.createMultiLineComment(p, `Skipping unhandled member: ${escapeForComment(p.getText())}`)));
        for (const fnDecl of abstractMethods) {
            // If the function declaration is computed, its name is the computed expression; otherwise, its
            // name can be resolved to a string.
            const name = fnDecl.name && ts.isComputedPropertyName(fnDecl.name) ? fnDecl.name.expression :
                propertyName(fnDecl);
            if (!name) {
                mtt.error(fnDecl, 'anonymous abstract function');
                continue;
            }
            const { tags, parameterNames } = mtt.getFunctionTypeJSDoc([fnDecl], []);
            if (decorators_1.hasExportingDecorator(fnDecl, mtt.typeChecker))
                tags.push({ tagName: 'export' });
            // Use element access instead of property access for compued names.
            const lhs = typeof name === 'string' ? ts.createPropertyAccess(instancePropAccess, name) :
                ts.createElementAccess(instancePropAccess, name);
            // memberNamespace because abstract methods cannot be static in TypeScript.
            const abstractFnDecl = ts.createStatement(ts.createAssignment(lhs, ts.createFunctionExpression(
            /* modifiers */ undefined, 
            /* asterisk */ undefined, 
            /* name */ undefined, 
            /* typeParameters */ undefined, parameterNames.map(n => ts.createParameter(
            /* decorators */ undefined, /* modifiers */ undefined, 
            /* dotDotDot */ undefined, n)), undefined, ts.createBlock([]))));
            ts.setSyntheticLeadingComments(abstractFnDecl, [jsdoc.toSynthesizedComment(tags)]);
            propertyDecls.push(ts.setSourceMapRange(abstractFnDecl, fnDecl));
        }
        // See test_files/fields/fields.ts:BaseThatThrows for a note on this wrapper.
        return ts.createIf(ts.createLiteral(false), ts.createBlock(propertyDecls, true));
    }
    function propertyName(prop) {
        if (!prop.name)
            return null;
        switch (prop.name.kind) {
            case ts.SyntaxKind.Identifier:
                return transformerUtil.getIdentifierText(prop.name);
            case ts.SyntaxKind.StringLiteral:
                // E.g. interface Foo { 'bar': number; }
                // If 'bar' is a name that is not valid in Closure then there's nothing we can do.
                const text = prop.name.text;
                if (!type_translator_1.isValidClosurePropertyName(text))
                    return null;
                return text;
            default:
                return null;
        }
    }
    /** Removes comment metacharacters from a string, to make it safe to embed in a comment. */
    function escapeForComment(str) {
        return str.replace(/\/\*/g, '__').replace(/\*\//g, '__');
    }
    exports.escapeForComment = escapeForComment;
    function createClosurePropertyDeclaration(mtt, expr, prop, optional) {
        const name = propertyName(prop);
        if (!name) {
            // Skip warning for private identifiers because it is expected they are skipped in the
            // Closure output.
            // TODO(rdel): Once Closure Compiler determines how private properties should be represented,
            // adjust this output accordingly.
            if (ts.isPrivateIdentifier(prop.name)) {
                return transformerUtil.createMultiLineComment(prop, `Skipping private member:\n${escapeForComment(prop.getText())}`);
            }
            else {
                mtt.debugWarn(prop, `handle unnamed member:\n${escapeForComment(prop.getText())}`);
                return transformerUtil.createMultiLineComment(prop, `Skipping unnamed member:\n${escapeForComment(prop.getText())}`);
            }
        }
        if (name === 'prototype') {
            // Code that declares a property named 'prototype' typically is doing something
            // funny with the TS type system, and isn't actually interested in naming a
            // a field 'prototype', as prototype has special meaning in JS.
            return transformerUtil.createMultiLineComment(prop, `Skipping illegal member name:\n${escapeForComment(prop.getText())}`);
        }
        let type = mtt.typeToClosure(prop);
        // When a property is optional, e.g.
        //   foo?: string;
        // Then the TypeScript type of the property is string|undefined, the
        // typeToClosure translation handles it correctly, and string|undefined is
        // how you write an optional property in Closure.
        //
        // But in the special case of an optional property with type any:
        //   foo?: any;
        // The TypeScript type of the property is just "any" (because any includes
        // undefined as well) so our default translation of the type is just "?".
        // To mark the property as optional in Closure it must have "|undefined",
        // so the Closure type must be ?|undefined.
        if (optional && type === '?')
            type += '|undefined';
        const tags = mtt.getJSDoc(prop, /* reportWarnings */ true);
        tags.push({ tagName: 'type', type });
        const flags = ts.getCombinedModifierFlags(prop);
        if (flags & ts.ModifierFlags.Protected) {
            tags.push({ tagName: 'protected' });
        }
        else if (flags & ts.ModifierFlags.Private) {
            tags.push({ tagName: 'private' });
        }
        if (decorators_1.hasExportingDecorator(prop, mtt.typeChecker)) {
            tags.push({ tagName: 'export' });
        }
        const declStmt = ts.setSourceMapRange(ts.createStatement(ts.createPropertyAccess(expr, name)), prop);
        // Avoid printing annotations that can conflict with @type
        // This avoids Closure's error "type annotation incompatible with other annotations"
        addCommentOn(declStmt, tags, jsdoc.TAGS_CONFLICTING_WITH_TYPE);
        return declStmt;
    }
    /**
     * Removes any type assertions and non-null expressions from the AST before TypeScript processing.
     *
     * Ideally, the code in jsdoc_transformer below should just remove the cast expression and
     * replace it with the Closure equivalent. However Angular's compiler is fragile to AST
     * nodes being removed or changing type, so the code must retain the type assertion
     * expression, see: https://github.com/angular/angular/issues/24895.
     *
     * tsickle also cannot just generate and keep a `(/.. @type {SomeType} ./ (expr as SomeType))`
     * because TypeScript removes the parenthesized expressions in that syntax, (reasonably) believing
     * they were only added for the TS cast.
     *
     * The final workaround is then to keep the TypeScript type assertions, and have a post-Angular
     * processing step that removes the assertions before TypeScript sees them.
     *
     * TODO(martinprobst): remove once the Angular issue is fixed.
     */
    function removeTypeAssertions() {
        return (context) => {
            return (sourceFile) => {
                function visitor(node) {
                    switch (node.kind) {
                        case ts.SyntaxKind.TypeAssertionExpression:
                        case ts.SyntaxKind.AsExpression:
                            return ts.visitNode(node.expression, visitor);
                        case ts.SyntaxKind.NonNullExpression:
                            return ts.visitNode(node.expression, visitor);
                        default:
                            break;
                    }
                    return ts.visitEachChild(node, visitor, context);
                }
                return visitor(sourceFile);
            };
        };
    }
    exports.removeTypeAssertions = removeTypeAssertions;
    /**
     * Returns true if node lexically (recursively) contains an 'async' function.
     */
    function containsAsync(node) {
        if (ts.isFunctionLike(node) && transformerUtil.hasModifierFlag(node, ts.ModifierFlags.Async)) {
            return true;
        }
        return ts.forEachChild(node, containsAsync) || false;
    }
    /**
     * jsdocTransformer returns a transformer factory that converts TypeScript types into the equivalent
     * JSDoc annotations.
     */
    function jsdocTransformer(host, tsOptions, typeChecker, diagnostics) {
        return (context) => {
            return (sourceFile) => {
                const moduleTypeTranslator = new module_type_translator_1.ModuleTypeTranslator(sourceFile, typeChecker, host, diagnostics, /*isForExterns*/ false);
                /**
                 * The set of all names exported from an export * in the current module. Used to prevent
                 * emitting duplicated exports. The first export * takes precedence in ES6.
                 */
                const expandedStarImports = new Set();
                /**
                 * While Closure compiler supports parameterized types, including parameterized `this` on
                 * methods, it does not support constraints on them. That means that an `\@template`d type is
                 * always considered to be `unknown` within the method, including `THIS`.
                 *
                 * To help Closure Compiler, we keep track of any templated this return type, and substitute
                 * explicit casts to the templated type.
                 *
                 * This is an incomplete solution and works around a specific problem with warnings on unknown
                 * this accesses. More generally, Closure also cannot infer constraints for any other
                 * templated types, but that might require a more general solution in Closure Compiler.
                 */
                let contextThisType = null;
                function visitClassDeclaration(classDecl) {
                    const contextThisTypeBackup = contextThisType;
                    const mjsdoc = moduleTypeTranslator.getMutableJSDoc(classDecl);
                    if (transformerUtil.hasModifierFlag(classDecl, ts.ModifierFlags.Abstract)) {
                        mjsdoc.tags.push({ tagName: 'abstract' });
                    }
                    maybeAddTemplateClause(mjsdoc.tags, classDecl);
                    if (!host.untyped) {
                        maybeAddHeritageClauses(mjsdoc.tags, moduleTypeTranslator, classDecl);
                    }
                    mjsdoc.updateComment();
                    const decls = [];
                    const memberDecl = createMemberTypeDeclaration(moduleTypeTranslator, classDecl);
                    // WARNING: order is significant; we must create the member decl before transforming away
                    // parameter property comments when visiting the constructor.
                    decls.push(ts.visitEachChild(classDecl, visitor, context));
                    if (memberDecl)
                        decls.push(memberDecl);
                    contextThisType = contextThisTypeBackup;
                    return decls;
                }
                /**
                 * visitHeritageClause works around a Closure Compiler issue, where the expression in an
                 * "extends" clause must be a simple identifier, and in particular must not be a parenthesized
                 * expression.
                 *
                 * This is triggered when TS code writes "class X extends (Foo as Bar) { ... }", commonly done
                 * to support mixins. For extends clauses in classes, the code below drops the cast and any
                 * parentheticals, leaving just the original expression.
                 *
                 * This is an incomplete workaround, as Closure will still bail on other super expressions,
                 * but retains compatibility with the previous emit that (accidentally) dropped the cast
                 * expression.
                 *
                 * TODO(martinprobst): remove this once the Closure side issue has been resolved.
                 */
                function visitHeritageClause(heritageClause) {
                    if (heritageClause.token !== ts.SyntaxKind.ExtendsKeyword || !heritageClause.parent ||
                        heritageClause.parent.kind === ts.SyntaxKind.InterfaceDeclaration) {
                        return ts.visitEachChild(heritageClause, visitor, context);
                    }
                    if (heritageClause.types.length !== 1) {
                        moduleTypeTranslator.error(heritageClause, `expected exactly one type in class extension clause`);
                    }
                    const type = heritageClause.types[0];
                    let expr = type.expression;
                    while (ts.isParenthesizedExpression(expr) || ts.isNonNullExpression(expr) ||
                        ts.isAssertionExpression(expr)) {
                        expr = expr.expression;
                    }
                    return ts.updateHeritageClause(heritageClause, [ts.updateExpressionWithTypeArguments(type, type.typeArguments || [], expr)]);
                }
                function visitInterfaceDeclaration(iface) {
                    const sym = typeChecker.getSymbolAtLocation(iface.name);
                    if (!sym) {
                        moduleTypeTranslator.error(iface, 'interface with no symbol');
                        return [];
                    }
                    // If this symbol is both a type and a value, we cannot emit both into Closure's
                    // single namespace.
                    if (transformer_util_1.symbolIsValue(typeChecker, sym)) {
                        moduleTypeTranslator.debugWarn(iface, `type/symbol conflict for ${sym.name}, using {?} for now`);
                        return [transformerUtil.createSingleLineComment(iface, 'WARNING: interface has both a type and a value, skipping emit')];
                    }
                    const tags = moduleTypeTranslator.getJSDoc(iface, /* reportWarnings */ true) || [];
                    tags.push({ tagName: 'record' });
                    maybeAddTemplateClause(tags, iface);
                    if (!host.untyped) {
                        maybeAddHeritageClauses(tags, moduleTypeTranslator, iface);
                    }
                    const name = transformerUtil.getIdentifierText(iface.name);
                    const modifiers = transformerUtil.hasModifierFlag(iface, ts.ModifierFlags.Export) ?
                        [ts.createToken(ts.SyntaxKind.ExportKeyword)] :
                        undefined;
                    const decl = ts.setSourceMapRange(ts.createFunctionDeclaration(
                    /* decorators */ undefined, modifiers, 
                    /* asterisk */ undefined, name, 
                    /* typeParameters */ undefined, 
                    /* parameters */ [], 
                    /* type */ undefined, 
                    /* body */ ts.createBlock([])), iface);
                    addCommentOn(decl, tags);
                    const memberDecl = createMemberTypeDeclaration(moduleTypeTranslator, iface);
                    return memberDecl ? [decl, memberDecl] : [decl];
                }
                /** Returns the `this` type in this context, or undefined if none. */
                function getContextThisType(node) {
                    while (node) {
                        if (ts.isClassDeclaration(node) && node.name) {
                            return typeChecker.getTypeAtLocation(node.name);
                        }
                        if (ts.isFunctionDeclaration(node) && node.parameters.length > 0) {
                            const firstParam = node.parameters[0];
                            // ts.Signature does not expose a `this` type, so comparing identifier names to 'this'
                            // is the only way to find a this type declaration.
                            if (ts.isIdentifier(firstParam.name) &&
                                transformerUtil.getIdentifierText(firstParam.name) === 'this' && firstParam.type) {
                                return typeChecker.getTypeAtLocation(firstParam.type);
                            }
                        }
                        node = node.parent;
                    }
                    return undefined;
                }
                /** Function declarations are emitted as they are, with only JSDoc added. */
                function visitFunctionLikeDeclaration(fnDecl) {
                    if (!fnDecl.body) {
                        // Two cases: abstract methods and overloaded methods/functions.
                        // Abstract methods are handled in emitTypeAnnotationsHandler.
                        // Overloads are union-ized into the shared type in FunctionType.
                        return ts.visitEachChild(fnDecl, visitor, context);
                    }
                    const extraTags = [];
                    if (decorators_1.hasExportingDecorator(fnDecl, typeChecker))
                        extraTags.push({ tagName: 'export' });
                    const { tags, thisReturnType } = moduleTypeTranslator.getFunctionTypeJSDoc([fnDecl], extraTags);
                    // async functions when down-leveled access `this` to pass it to
                    // tslib.__awaiter.  Closure wants to know the type of 'this' for that.
                    // The type is known in many contexts (e.g. methods, arrow functions)
                    // per the normal rules (e.g. looking at parent nodes and @this tags)
                    // but if the search bottoms out at a function scope, then Closure
                    // warns that 'this' is unknown.
                    // Because we have already checked the type of 'this', we are ok to just
                    // suppress in that case.  We do so by stuffing a @this on any function
                    // where it might be needed; it's harmless to overapproximate.
                    const isDownlevellingAsync = tsOptions.target !== undefined && tsOptions.target <= ts.ScriptTarget.ES2015;
                    const isFunction = fnDecl.kind === ts.SyntaxKind.FunctionDeclaration;
                    const hasExistingThisTag = tags.some(t => t.tagName === 'this');
                    if (isDownlevellingAsync && isFunction && !hasExistingThisTag && containsAsync(fnDecl)) {
                        tags.push({ tagName: 'this', type: '*' });
                    }
                    const mjsdoc = moduleTypeTranslator.getMutableJSDoc(fnDecl);
                    mjsdoc.tags = tags;
                    mjsdoc.updateComment();
                    const contextThisTypeBackup = contextThisType;
                    // Arrow functions retain their context `this` type. All others reset the this type to
                    // either none (if not specified) or the type given in a fn(this: T, ...) declaration.
                    if (!ts.isArrowFunction(fnDecl))
                        contextThisType = thisReturnType;
                    const result = ts.visitEachChild(fnDecl, visitor, context);
                    contextThisType = contextThisTypeBackup;
                    return result;
                }
                /**
                 * In methods with a templated this type, adds explicit casts to accesses on this.
                 *
                 * @see contextThisType
                 */
                function visitThisExpression(node) {
                    if (!contextThisType)
                        return ts.visitEachChild(node, visitor, context);
                    return createClosureCast(node, node, contextThisType);
                }
                /**
                 * visitVariableStatement flattens variable declaration lists (`var a, b;` to `var a; var
                 * b;`), and attaches JSDoc comments to each variable. JSDoc comments preceding the
                 * original variable are attached to the first newly created one.
                 */
                function visitVariableStatement(varStmt) {
                    const stmts = [];
                    // "const", "let", etc are stored in node flags on the declarationList.
                    const flags = ts.getCombinedNodeFlags(varStmt.declarationList);
                    let tags = moduleTypeTranslator.getJSDoc(varStmt, /* reportWarnings */ true);
                    const leading = ts.getSyntheticLeadingComments(varStmt);
                    if (leading) {
                        // Attach non-JSDoc comments to a not emitted statement.
                        const commentHolder = ts.createNotEmittedStatement(varStmt);
                        ts.setSyntheticLeadingComments(commentHolder, leading.filter(c => c.text[0] !== '*'));
                        stmts.push(commentHolder);
                    }
                    for (const decl of varStmt.declarationList.declarations) {
                        const localTags = [];
                        if (tags) {
                            // Add any tags and docs preceding the entire statement to the first variable.
                            localTags.push(...tags);
                            tags = null;
                        }
                        // Add an @type for plain identifiers, but not for bindings patterns (i.e. object or array
                        // destructuring - those do not have a syntax in Closure) or @defines, which already
                        // declare their type.
                        if (ts.isIdentifier(decl.name)) {
                            // For variables that are initialized and use a blacklisted type, do not emit a type at
                            // all. Closure Compiler might be able to infer a better type from the initializer than
                            // the `?` the code below would emit.
                            // TODO(martinprobst): consider doing this for all types that get emitted as ?, not just
                            // for blacklisted ones.
                            const blackListedInitialized = !!decl.initializer && moduleTypeTranslator.isBlackListed(decl);
                            if (!blackListedInitialized) {
                                // getOriginalNode(decl) is required because the type checker cannot type check
                                // synthesized nodes.
                                const typeStr = moduleTypeTranslator.typeToClosure(ts.getOriginalNode(decl));
                                // If @define is present then add the type to it, rather than adding a normal @type.
                                const defineTag = localTags.find(({ tagName }) => tagName === 'define');
                                if (defineTag) {
                                    defineTag.type = typeStr;
                                }
                                else {
                                    localTags.push({ tagName: 'type', type: typeStr });
                                }
                            }
                        }
                        const newDecl = ts.visitNode(decl, visitor);
                        const newStmt = ts.createVariableStatement(varStmt.modifiers, ts.createVariableDeclarationList([newDecl], flags));
                        if (localTags.length)
                            addCommentOn(newStmt, localTags, jsdoc.TAGS_CONFLICTING_WITH_TYPE);
                        stmts.push(newStmt);
                    }
                    return stmts;
                }
                /**
                 * shouldEmitExportsAssignments returns true if tsickle should emit `exports.Foo = ...` style
                 * export statements.
                 *
                 * TypeScript modules can export types. Because types are pure design-time constructs in
                 * TypeScript, it does not emit any actual exported symbols for these. But tsickle has to emit
                 * an export, so that downstream Closure code (including tsickle-converted Closure code) can
                 * import upstream types. tsickle has to pick a module format for that, because the pure ES6
                 * export would get stripped by TypeScript.
                 *
                 * tsickle uses CommonJS to emit googmodule, and code not using googmodule doesn't care about
                 * the Closure annotations anyway, so tsickle skips emitting exports if the module target
                 * isn't commonjs.
                 */
                function shouldEmitExportsAssignments() {
                    return tsOptions.module === ts.ModuleKind.CommonJS;
                }
                function visitTypeAliasDeclaration(typeAlias) {
                    const sym = moduleTypeTranslator.mustGetSymbolAtLocation(typeAlias.name);
                    // If the type is also defined as a value, skip emitting it. Closure collapses type & value
                    // namespaces, the two emits would conflict if tsickle emitted both.
                    if (transformer_util_1.symbolIsValue(typeChecker, sym))
                        return [];
                    if (!shouldEmitExportsAssignments())
                        return [];
                    const typeName = typeAlias.name.getText();
                    // Blacklist any type parameters, Closure does not support type aliases with type
                    // parameters.
                    moduleTypeTranslator.newTypeTranslator(typeAlias).blacklistTypeParameters(moduleTypeTranslator.symbolsToAliasedNames, typeAlias.typeParameters);
                    const typeStr = host.untyped ? '?' : moduleTypeTranslator.typeToClosure(typeAlias, undefined);
                    // We want to emit a @typedef.  They are a bit weird because they are 'var' statements
                    // that have no value.
                    const tags = moduleTypeTranslator.getJSDoc(typeAlias, /* reportWarnings */ true);
                    tags.push({ tagName: 'typedef', type: typeStr });
                    const isExported = transformerUtil.hasModifierFlag(typeAlias, ts.ModifierFlags.Export);
                    let decl;
                    if (isExported) {
                        // Given: export type T = ...;
                        // We cannot emit `export var foo;` and let TS generate from there because TypeScript
                        // drops exports that are never assigned values, and Closure requires us to not assign
                        // values to typedef exports. Introducing a new local variable and exporting it can cause
                        // bugs due to name shadowing and confusing TypeScript's logic on what symbols and types
                        // vs values are exported. Mangling the name to avoid the conflicts would be reasonably
                        // clean, but would require a two pass emit to first find all type alias names, mangle
                        // them, and emit the use sites only later.
                        // So we produce: exports.T;
                        decl = ts.createStatement(ts.createPropertyAccess(ts.createIdentifier('exports'), ts.createIdentifier(typeName)));
                    }
                    else {
                        // Given: type T = ...;
                        // We produce: var T;
                        // Note: not const, because 'const Foo;' is illegal;
                        // not let, because we want hoisting behavior for types.
                        decl = ts.createVariableStatement(
                        /* modifiers */ undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration(ts.createIdentifier(typeName))]));
                    }
                    decl = ts.setSourceMapRange(decl, typeAlias);
                    addCommentOn(decl, tags, jsdoc.TAGS_CONFLICTING_WITH_TYPE);
                    return [decl];
                }
                /** Emits a parenthesized Closure cast: `(/** \@type ... * / (expr))`. */
                function createClosureCast(context, expression, type) {
                    const inner = ts.createParen(expression);
                    const comment = addCommentOn(inner, [{ tagName: 'type', type: moduleTypeTranslator.typeToClosure(context, type) }]);
                    comment.hasTrailingNewLine = false;
                    return ts.setSourceMapRange(ts.createParen(inner), context);
                }
                /** Converts a TypeScript type assertion into a Closure Cast. */
                function visitAssertionExpression(assertion) {
                    const type = typeChecker.getTypeAtLocation(assertion.type);
                    return createClosureCast(assertion, ts.visitEachChild(assertion, visitor, context), type);
                }
                /**
                 * Converts a TypeScript non-null assertion into a Closure Cast, by stripping |null and
                 * |undefined from a union type.
                 */
                function visitNonNullExpression(nonNull) {
                    const type = typeChecker.getTypeAtLocation(nonNull.expression);
                    const nonNullType = typeChecker.getNonNullableType(type);
                    return createClosureCast(nonNull, ts.visitEachChild(nonNull, visitor, context), nonNullType);
                }
                function visitImportDeclaration(importDecl) {
                    // For each import, insert a goog.requireType for the module, so that if TypeScript does not
                    // emit the module because it's only used in type positions, the JSDoc comments still
                    // reference a valid Closure level symbol.
                    // No need to requireType side effect imports.
                    if (!importDecl.importClause)
                        return importDecl;
                    const sym = typeChecker.getSymbolAtLocation(importDecl.moduleSpecifier);
                    // Scripts do not have a symbol, and neither do unused modules. Scripts can still be
                    // imported, either as side effect imports or with an empty import set ("{}"). TypeScript
                    // does not emit a runtime load for an import with an empty list of symbols, but the import
                    // forces any global declarations from the library to be visible, which is what users use
                    // this for. No symbols from the script need requireType, so just return.
                    // TODO(evmar): revisit this.  If TS needs to see the module import, it's likely Closure
                    // does too.
                    if (!sym)
                        return importDecl;
                    const importPath = googmodule.resolveModuleName({ options: tsOptions, moduleResolutionHost: host.moduleResolutionHost }, sourceFile.fileName, importDecl.moduleSpecifier.text);
                    moduleTypeTranslator.requireType(importPath, sym, 
                    /* default import? */ !!importDecl.importClause.name);
                    return importDecl;
                }
                /**
                 * Closure Compiler will fail when it finds incorrect JSDoc tags on nodes. This function
                 * parses and then re-serializes JSDoc comments, escaping or removing illegal tags.
                 */
                function escapeIllegalJSDoc(node) {
                    const mjsdoc = moduleTypeTranslator.getMutableJSDoc(node);
                    mjsdoc.updateComment();
                }
                /** Returns true if a value export should be emitted for the given symbol in export *. */
                function shouldEmitValueExportForSymbol(sym) {
                    if (sym.flags & ts.SymbolFlags.Alias) {
                        sym = typeChecker.getAliasedSymbol(sym);
                    }
                    if ((sym.flags & ts.SymbolFlags.Value) === 0) {
                        // Note: We create explicit exports of type symbols for closure in visitExportDeclaration.
                        return false;
                    }
                    if (!tsOptions.preserveConstEnums && sym.flags & ts.SymbolFlags.ConstEnum) {
                        return false;
                    }
                    return true;
                }
                /**
                 * visitExportDeclaration requireTypes exported modules and emits explicit exports for
                 * types (which normally do not get emitted by TypeScript).
                 */
                function visitExportDeclaration(exportDecl) {
                    const importedModuleSymbol = exportDecl.moduleSpecifier &&
                        typeChecker.getSymbolAtLocation(exportDecl.moduleSpecifier);
                    if (importedModuleSymbol) {
                        // requireType all explicitly imported modules, so that symbols can be referenced and
                        // type only modules are usable from type declarations.
                        moduleTypeTranslator.requireType(exportDecl.moduleSpecifier.text, importedModuleSymbol, 
                        /* default import? */ false);
                    }
                    const typesToExport = [];
                    if (!exportDecl.exportClause) {
                        // export * from '...'
                        // Resolve the * into all value symbols exported, and update the export declaration.
                        // Explicitly spelled out exports (i.e. the exports of the current module) take precedence
                        // over implicit ones from export *. Use the current module's exports to filter.
                        const currentModuleSymbol = typeChecker.getSymbolAtLocation(sourceFile);
                        const currentModuleExports = currentModuleSymbol && currentModuleSymbol.exports;
                        if (!importedModuleSymbol) {
                            moduleTypeTranslator.error(exportDecl, `export * without module symbol`);
                            return exportDecl;
                        }
                        const exportedSymbols = typeChecker.getExportsOfModule(importedModuleSymbol);
                        const exportSpecifiers = [];
                        for (const sym of exportedSymbols) {
                            if (currentModuleExports && currentModuleExports.has(sym.escapedName))
                                continue;
                            // We might have already generated an export for the given symbol.
                            if (expandedStarImports.has(sym.name))
                                continue;
                            expandedStarImports.add(sym.name);
                            // Only create an export specifier for values that are exported. For types, the code
                            // below creates specific export statements that match Closure's expectations.
                            if (shouldEmitValueExportForSymbol(sym)) {
                                exportSpecifiers.push(ts.createExportSpecifier(undefined, sym.name));
                            }
                            else {
                                typesToExport.push([sym.name, sym]);
                            }
                        }
                        const isTypeOnlyExport = false;
                        exportDecl = ts.updateExportDeclaration(exportDecl, exportDecl.decorators, exportDecl.modifiers, ts.createNamedExports(exportSpecifiers), exportDecl.moduleSpecifier, isTypeOnlyExport);
                    }
                    else if (ts.isNamedExports(exportDecl.exportClause)) {
                        // export {a, b, c} from 'abc';
                        for (const exp of exportDecl.exportClause.elements) {
                            const exportedName = transformerUtil.getIdentifierText(exp.name);
                            typesToExport.push([exportedName, moduleTypeTranslator.mustGetSymbolAtLocation(exp.name)]);
                        }
                    }
                    // Do not emit typedef re-exports in untyped mode.
                    if (host.untyped)
                        return exportDecl;
                    const result = [exportDecl];
                    for (const [exportedName, sym] of typesToExport) {
                        let aliasedSymbol = sym;
                        if (sym.flags & ts.SymbolFlags.Alias) {
                            aliasedSymbol = typeChecker.getAliasedSymbol(sym);
                        }
                        const isTypeAlias = (aliasedSymbol.flags & ts.SymbolFlags.Value) === 0 &&
                            (aliasedSymbol.flags & (ts.SymbolFlags.TypeAlias | ts.SymbolFlags.Interface)) !== 0;
                        if (!isTypeAlias)
                            continue;
                        const typeName = moduleTypeTranslator.symbolsToAliasedNames.get(aliasedSymbol) || aliasedSymbol.name;
                        const stmt = ts.createStatement(ts.createPropertyAccess(ts.createIdentifier('exports'), exportedName));
                        addCommentOn(stmt, [{ tagName: 'typedef', type: '!' + typeName }]);
                        ts.addSyntheticTrailingComment(stmt, ts.SyntaxKind.SingleLineCommentTrivia, ' re-export typedef', true);
                        result.push(stmt);
                    }
                    return result;
                }
                /**
                 * Returns the identifiers exported in a single exported statement - typically just one
                 * identifier (e.g. for `export function foo()`), but multiple for `export declare var a, b`.
                 */
                function getExportDeclarationNames(node) {
                    switch (node.kind) {
                        case ts.SyntaxKind.VariableStatement:
                            const varDecl = node;
                            return varDecl.declarationList.declarations.map((d) => getExportDeclarationNames(d)[0]);
                        case ts.SyntaxKind.VariableDeclaration:
                        case ts.SyntaxKind.FunctionDeclaration:
                        case ts.SyntaxKind.InterfaceDeclaration:
                        case ts.SyntaxKind.ClassDeclaration:
                        case ts.SyntaxKind.ModuleDeclaration:
                        case ts.SyntaxKind.EnumDeclaration:
                            const decl = node;
                            if (!decl.name || decl.name.kind !== ts.SyntaxKind.Identifier) {
                                break;
                            }
                            return [decl.name];
                        case ts.SyntaxKind.TypeAliasDeclaration:
                            const typeAlias = node;
                            return [typeAlias.name];
                        default:
                            break;
                    }
                    moduleTypeTranslator.error(node, `unsupported export declaration ${ts.SyntaxKind[node.kind]}: ${node.getText()}`);
                    return [];
                }
                /**
                 * Ambient declarations declare types for TypeScript's benefit, and will be removed by
                 * TypeScript during its emit phase. Downstream Closure code however might be importing
                 * symbols from this module, so tsickle must emit a Closure-compatible exports declaration.
                 */
                function visitExportedAmbient(node) {
                    if (host.untyped || !shouldEmitExportsAssignments())
                        return [node];
                    const declNames = getExportDeclarationNames(node);
                    const result = [node];
                    for (const decl of declNames) {
                        const sym = typeChecker.getSymbolAtLocation(decl);
                        // Non-value objects do not exist at runtime, so we cannot access the symbol (it only
                        // exists in externs). Export them as a typedef, which forwards to the type in externs.
                        // Note: TypeScript emits odd code for exported ambients (exports.x for variables, just x
                        // for everything else). That seems buggy, and in either case this code should not attempt
                        // to fix it.
                        // See also https://github.com/Microsoft/TypeScript/issues/8015.
                        if (!transformer_util_1.symbolIsValue(typeChecker, sym)) {
                            // Do not emit re-exports for ModuleDeclarations.
                            // Ambient ModuleDeclarations are always referenced as global symbols, so they don't
                            // need to be exported.
                            if (node.kind === ts.SyntaxKind.ModuleDeclaration)
                                continue;
                            const mangledName = annotator_host_1.moduleNameAsIdentifier(host, sourceFile.fileName);
                            const declName = transformerUtil.getIdentifierText(decl);
                            const stmt = ts.createStatement(ts.createPropertyAccess(ts.createIdentifier('exports'), declName));
                            addCommentOn(stmt, [{ tagName: 'typedef', type: `!${mangledName}.${declName}` }]);
                            result.push(stmt);
                        }
                    }
                    return result;
                }
                /**
                 * Visits enum declarations to check for validity of JSDoc comments without transforming the
                 * node at all.
                 */
                function visitEnumDeclaration(node) {
                    // Calling `getJSDoc` will validate and report any errors, but this code
                    // doesn't really care about the return value.
                    moduleTypeTranslator.getJSDoc(node, /* reportWarnings */ true);
                }
                function visitor(node) {
                    if (transformerUtil.isAmbient(node)) {
                        if (!transformerUtil.hasModifierFlag(node, ts.ModifierFlags.Export)) {
                            return node;
                        }
                        return visitExportedAmbient(node);
                    }
                    switch (node.kind) {
                        case ts.SyntaxKind.ImportDeclaration:
                            return visitImportDeclaration(node);
                        case ts.SyntaxKind.ExportDeclaration:
                            return visitExportDeclaration(node);
                        case ts.SyntaxKind.ClassDeclaration:
                            return visitClassDeclaration(node);
                        case ts.SyntaxKind.InterfaceDeclaration:
                            return visitInterfaceDeclaration(node);
                        case ts.SyntaxKind.HeritageClause:
                            return visitHeritageClause(node);
                        case ts.SyntaxKind.ArrowFunction:
                        case ts.SyntaxKind.FunctionExpression:
                            // Inserting a comment before an expression can trigger automatic semicolon insertion,
                            // e.g. if the function below is the expression in a `return` statement. Parenthesizing
                            // prevents ASI, as long as the opening paren remains on the same line (which it does).
                            return ts.createParen(visitFunctionLikeDeclaration(node));
                        case ts.SyntaxKind.Constructor:
                        case ts.SyntaxKind.FunctionDeclaration:
                        case ts.SyntaxKind.MethodDeclaration:
                        case ts.SyntaxKind.GetAccessor:
                        case ts.SyntaxKind.SetAccessor:
                            return visitFunctionLikeDeclaration(node);
                        case ts.SyntaxKind.ThisKeyword:
                            return visitThisExpression(node);
                        case ts.SyntaxKind.VariableStatement:
                            return visitVariableStatement(node);
                        case ts.SyntaxKind.PropertyDeclaration:
                        case ts.SyntaxKind.PropertyAssignment:
                            escapeIllegalJSDoc(node);
                            break;
                        case ts.SyntaxKind.Parameter:
                            // Parameter properties (e.g. `constructor(/** docs */ private foo: string)`) might have
                            // JSDoc comments, including JSDoc tags recognized by Closure Compiler. Prevent emitting
                            // any comments on them, so that Closure doesn't error on them.
                            // See test_files/parameter_properties.ts.
                            const paramDecl = node;
                            if (transformerUtil.hasModifierFlag(paramDecl, ts.ModifierFlags.ParameterPropertyModifier)) {
                                ts.setSyntheticLeadingComments(paramDecl, []);
                                jsdoc.suppressLeadingCommentsRecursively(paramDecl);
                            }
                            break;
                        case ts.SyntaxKind.TypeAliasDeclaration:
                            return visitTypeAliasDeclaration(node);
                        case ts.SyntaxKind.AsExpression:
                        case ts.SyntaxKind.TypeAssertionExpression:
                            return visitAssertionExpression(node);
                        case ts.SyntaxKind.NonNullExpression:
                            return visitNonNullExpression(node);
                        case ts.SyntaxKind.EnumDeclaration:
                            visitEnumDeclaration(node);
                            break;
                        default:
                            break;
                    }
                    return ts.visitEachChild(node, visitor, context);
                }
                sourceFile = ts.visitEachChild(sourceFile, visitor, context);
                return moduleTypeTranslator.insertAdditionalImports(sourceFile);
            };
        };
    }
    exports.jsdocTransformer = jsdocTransformer;
});
//# sourceMappingURL=data:application/json;base64,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