/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/module_type_translator", ["require", "exports", "typescript", "tsickle/src/googmodule", "tsickle/src/jsdoc", "tsickle/src/transformer_util", "tsickle/src/type_translator"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    /**
     * @fileoverview module_type_translator builds on top of type_translator, adding functionality to
     * translate types within the scope of a single module. The main entry point is
     * ModuleTypeTranslator.
     */
    const ts = require("typescript");
    const googmodule = require("tsickle/src/googmodule");
    const jsdoc = require("tsickle/src/jsdoc");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    const typeTranslator = require("tsickle/src/type_translator");
    /**
     * MutableJSDoc encapsulates a (potential) JSDoc comment on a specific node, and allows code to
     * modify (including delete) it.
     */
    class MutableJSDoc {
        constructor(node, sourceComment, tags) {
            this.node = node;
            this.sourceComment = sourceComment;
            this.tags = tags;
        }
        updateComment(escapeExtraTags) {
            const text = jsdoc.toStringWithoutStartEnd(this.tags, escapeExtraTags);
            if (this.sourceComment) {
                if (!text) {
                    // Delete the (now empty) comment.
                    const comments = ts.getSyntheticLeadingComments(this.node);
                    const idx = comments.indexOf(this.sourceComment);
                    comments.splice(idx, 1);
                    this.sourceComment = null;
                    return;
                }
                this.sourceComment.text = text;
                return;
            }
            // Don't add an empty comment.
            if (!text)
                return;
            const comment = {
                kind: ts.SyntaxKind.MultiLineCommentTrivia,
                text,
                hasTrailingNewLine: true,
                pos: -1,
                end: -1,
            };
            const comments = ts.getSyntheticLeadingComments(this.node) || [];
            comments.push(comment);
            ts.setSyntheticLeadingComments(this.node, comments);
        }
    }
    exports.MutableJSDoc = MutableJSDoc;
    /** Returns the Closure name of a function parameter, special-casing destructuring. */
    function getParameterName(param, index) {
        switch (param.name.kind) {
            case ts.SyntaxKind.Identifier:
                let name = transformer_util_1.getIdentifierText(param.name);
                // TypeScript allows parameters named "arguments", but Closure
                // disallows this, even in externs.
                if (name === 'arguments')
                    name = 'tsickle_arguments';
                return name;
            case ts.SyntaxKind.ArrayBindingPattern:
            case ts.SyntaxKind.ObjectBindingPattern:
                // Closure crashes if you put a binding pattern in the externs.
                // Avoid this by just generating an unused name; the name is
                // ignored anyway.
                return `__${index}`;
            default:
                // The above list of kinds is exhaustive.  param.name is 'never' at this point.
                const paramName = param.name;
                throw new Error(`unhandled function parameter kind: ${ts.SyntaxKind[paramName.kind]}`);
        }
    }
    /**
     * ModuleTypeTranslator encapsulates knowledge and helper functions to translate types in the scope
     * of a specific module. This includes managing Closure requireType statements and any symbol
     * aliases in scope for a whole file.
     */
    class ModuleTypeTranslator {
        constructor(sourceFile, typeChecker, host, diagnostics, isForExterns) {
            this.sourceFile = sourceFile;
            this.typeChecker = typeChecker;
            this.host = host;
            this.diagnostics = diagnostics;
            this.isForExterns = isForExterns;
            /**
             * A mapping of aliases for symbols in the current file, used when emitting types. TypeScript
             * emits imported symbols with unpredictable prefixes. To generate correct type annotations,
             * tsickle creates its own aliases for types, and registers them in this map (see
             * `emitImportDeclaration` and `requireType()` below). The aliases are then used when emitting
             * types.
             */
            this.symbolsToAliasedNames = new Map();
            /**
             * A cache for expensive symbol lookups, see TypeTranslator.symbolToString. Maps symbols to their
             * Closure name in this file scope.
             */
            this.symbolToNameCache = new Map();
            /**
             * The set of module symbols requireTyped in the local namespace.  This tracks which imported
             * modules we've already added to additionalImports below.
             */
            this.requireTypeModules = new Set();
            /**
             * The list of generated goog.requireType statements for this module. These are inserted into
             * the module's body statements after translation.
             */
            this.additionalImports = [];
        }
        debugWarn(context, messageText) {
            transformer_util_1.reportDebugWarning(this.host, context, messageText);
        }
        error(node, messageText) {
            transformer_util_1.reportDiagnostic(this.diagnostics, node, messageText);
        }
        /**
         * Convert a TypeScript ts.Type into the equivalent Closure type.
         *
         * @param context The ts.Node containing the type reference; used for resolving symbols
         *     in context.
         * @param type The type to translate; if not provided, the Node's type will be used.
         * @param resolveAlias If true, do not emit aliases as their symbol, but rather as the resolved
         *     type underlying the alias. This should be true only when emitting the typedef itself.
         */
        typeToClosure(context, type) {
            if (this.host.untyped) {
                return '?';
            }
            const typeChecker = this.typeChecker;
            if (!type) {
                type = typeChecker.getTypeAtLocation(context);
            }
            return this.newTypeTranslator(context).translate(type);
        }
        newTypeTranslator(context) {
            // In externs, there is no local scope, so all types must be relative to the file level scope.
            const translationContext = this.isForExterns ? this.sourceFile : context;
            const translator = new typeTranslator.TypeTranslator(this.host, this.typeChecker, translationContext, this.host.typeBlackListPaths || new Set(), this.symbolsToAliasedNames, this.symbolToNameCache, (sym) => this.ensureSymbolDeclared(sym));
            translator.isForExterns = this.isForExterns;
            translator.warn = msg => this.debugWarn(context, msg);
            return translator;
        }
        isBlackListed(context) {
            const type = this.typeChecker.getTypeAtLocation(context);
            let sym = type.symbol;
            if (!sym)
                return false;
            if (sym.flags & ts.SymbolFlags.Alias) {
                sym = this.typeChecker.getAliasedSymbol(sym);
            }
            return this.newTypeTranslator(context).isBlackListed(sym);
        }
        /**
         * Get the ts.Symbol at a location or throw.
         * The TypeScript API can return undefined when fetching a symbol, but in many contexts we know it
         * won't (e.g. our input is already type-checked).
         */
        mustGetSymbolAtLocation(node) {
            const sym = this.typeChecker.getSymbolAtLocation(node);
            if (!sym)
                throw new Error('no symbol');
            return sym;
        }
        /** Finds an exported (i.e. not global) declaration for the given symbol. */
        findExportedDeclaration(sym) {
            // TODO(martinprobst): it's unclear when a symbol wouldn't have a declaration, maybe just for
            // some builtins (e.g. Symbol)?
            if (!sym.declarations || sym.declarations.length === 0)
                return undefined;
            // A symbol declared in this file does not need to be imported.
            if (sym.declarations.some(d => d.getSourceFile() === this.sourceFile))
                return undefined;
            // Find an exported declaration.
            // Because tsickle runs with the --declaration flag, all types referenced from exported types
            // must be exported, too, so there must either be some declaration that is exported, or the
            // symbol is actually a global declaration (declared in a script file, not a module).
            const decl = sym.declarations.find(d => {
                // Check for Export | Default (default being a default export).
                if (!transformer_util_1.hasModifierFlag(d, ts.ModifierFlags.ExportDefault))
                    return false;
                // Exclude symbols declared in `declare global {...}` blocks, they are global and don't need
                // imports.
                let current = d;
                while (current) {
                    if (current.flags & ts.NodeFlags.GlobalAugmentation)
                        return false;
                    current = current.parent;
                }
                return true;
            });
            return decl;
        }
        /**
         * Generates a somewhat human-readable module prefix for the given import context, to make
         * debugging the emitted Closure types a bit easier.
         */
        generateModulePrefix(importPath) {
            const modulePrefix = importPath.replace(/(\/index)?(\.d)?\.[tj]sx?$/, '')
                .replace(/^.*[/.](.+?)/, '$1')
                .replace(/\W/g, '_');
            return `tsickle_${modulePrefix || 'reqType'}_`;
        }
        /**
         * Records that we we want a `const x = goog.requireType...` import of the given `importPath`,
         * which will be inserted when we emit.
         * This also registers aliases for symbols from the module that map to this requireType.
         *
         * @param isDefaultImport True if the import statement is a default import, e.g.
         *     `import Foo from ...;`, which matters for adjusting whether we emit a `.default`.
         */
        requireType(importPath, moduleSymbol, isDefaultImport = false) {
            if (this.host.untyped)
                return;
            // Already imported? Do not emit a duplicate requireType.
            if (this.requireTypeModules.has(moduleSymbol))
                return;
            if (typeTranslator.isBlacklisted(this.host.typeBlackListPaths, moduleSymbol)) {
                return; // Do not emit goog.requireType for blacklisted paths.
            }
            const nsImport = googmodule.extractGoogNamespaceImport(importPath);
            const requireTypePrefix = this.generateModulePrefix(importPath) + String(this.requireTypeModules.size + 1);
            const moduleNamespace = nsImport !== null ?
                nsImport :
                this.host.pathToModuleName(this.sourceFile.fileName, importPath);
            // In TypeScript, importing a module for use in a type annotation does not cause a runtime load.
            // In Closure Compiler, goog.require'ing a module causes a runtime load, so emitting requires
            // here would cause a change in load order, which is observable (and can lead to errors).
            // Instead, goog.requireType types, which allows using them in type annotations without
            // causing a load.
            //   const requireTypePrefix = goog.requireType(moduleNamespace)
            this.additionalImports.push(ts.createVariableStatement(undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration(requireTypePrefix, undefined, ts.createCall(ts.createPropertyAccess(ts.createIdentifier('goog'), 'requireType'), undefined, [ts.createLiteral(moduleNamespace)]))], ts.NodeFlags.Const)));
            this.requireTypeModules.add(moduleSymbol);
            for (let sym of this.typeChecker.getExportsOfModule(moduleSymbol)) {
                // Some users import {default as SomeAlias} from 'goog:...';
                // The code below must recognize this as a default import to alias the symbol to just the
                // blank module name.
                const namedDefaultImport = sym.name === 'default';
                if (sym.flags & ts.SymbolFlags.Alias) {
                    sym = this.typeChecker.getAliasedSymbol(sym);
                }
                // goog: imports don't actually use the .default property that TS thinks they have.
                const qualifiedName = nsImport && (isDefaultImport || namedDefaultImport) ?
                    requireTypePrefix :
                    requireTypePrefix + '.' + sym.name;
                this.symbolsToAliasedNames.set(sym, qualifiedName);
            }
        }
        ensureSymbolDeclared(sym) {
            const decl = this.findExportedDeclaration(sym);
            if (!decl)
                return;
            if (this.isForExterns) {
                this.error(decl, `declaration from module used in ambient type: ${sym.name}`);
                return;
            }
            // Actually import the symbol.
            const sourceFile = decl.getSourceFile();
            if (sourceFile === ts.getOriginalNode(this.sourceFile))
                return;
            const moduleSymbol = this.typeChecker.getSymbolAtLocation(sourceFile);
            // A source file might not have a symbol if it's not a module (no ES6 im/exports).
            if (!moduleSymbol)
                return;
            // TODO(martinprobst): this should possibly use fileNameToModuleId.
            this.requireType(sourceFile.fileName, moduleSymbol);
        }
        insertAdditionalImports(sourceFile) {
            let insertion = 0;
            // Skip over a leading file comment holder.
            if (sourceFile.statements.length &&
                sourceFile.statements[0].kind === ts.SyntaxKind.NotEmittedStatement) {
                insertion++;
            }
            return ts.updateSourceFileNode(sourceFile, [
                ...sourceFile.statements.slice(0, insertion),
                ...this.additionalImports,
                ...sourceFile.statements.slice(insertion),
            ]);
        }
        /**
         * Parses and synthesizes comments on node, and returns the JSDoc from it, if any.
         * @param reportWarnings if true, will report warnings from parsing the JSDoc. Set to false if
         *     this is not the "main" location dealing with a node to avoid duplicated warnings.
         */
        getJSDoc(node, reportWarnings) {
            const [tags,] = this.parseJSDoc(node, reportWarnings);
            return tags;
        }
        getMutableJSDoc(node) {
            const [tags, comment] = this.parseJSDoc(node, /* reportWarnings */ true);
            return new MutableJSDoc(node, comment, tags);
        }
        parseJSDoc(node, reportWarnings) {
            // synthesizeLeadingComments below changes text locations for node, so extract the location here
            // in case it is needed later to report diagnostics.
            const start = node.getFullStart();
            const length = node.getLeadingTriviaWidth(this.sourceFile);
            const comments = jsdoc.synthesizeLeadingComments(node);
            if (!comments || comments.length === 0)
                return [[], null];
            for (let i = comments.length - 1; i >= 0; i--) {
                const comment = comments[i];
                const parsed = jsdoc.parse(comment);
                if (parsed) {
                    if (reportWarnings && parsed.warnings) {
                        const range = comment.originalRange || { pos: start, end: start + length };
                        transformer_util_1.reportDiagnostic(this.diagnostics, node, parsed.warnings.join('\n'), range, ts.DiagnosticCategory.Warning);
                    }
                    return [parsed.tags, comment];
                }
            }
            return [[], null];
        }
        /**
         * resolveRestParameterType resolves the array member type for a rest parameter ("...").
         * In TypeScript you write "...x: number[]", but in Closure you don't write the array:
         * `@param {...number} x`. The code below unwraps the Array<> wrapper.
         */
        resolveRestParameterType(newTag, fnDecl, paramNode) {
            let type = this.typeChecker.getTypeAtLocation(paramNode);
            newTag.restParam = true;
            if ((type.flags & ts.TypeFlags.Object) === 0 && type.flags & ts.TypeFlags.TypeParameter) {
                // function f<T extends string[]>(...ts: T) has the Array type on the type parameter
                // constraint, not on the parameter itself. Resolve it.
                const baseConstraint = this.typeChecker.getBaseConstraintOfType(type);
                if (baseConstraint)
                    type = baseConstraint;
            }
            if ((type.flags & ts.TypeFlags.Object) !== 0 &&
                type.objectFlags & ts.ObjectFlags.Reference) {
                const typeRef = type;
                const typeArgs = this.typeChecker.getTypeArguments(typeRef);
                if (!typeArgs) {
                    throw new Error('rest parameter does not resolve to a reference type');
                }
                newTag.type = this.typeToClosure(fnDecl, typeArgs[0]);
                return;
            }
            // If we fail to unwrap the Array<> type, emit an unknown type.
            this.debugWarn(paramNode, 'failed to resolve rest parameter type, emitting ?');
            newTag.type = '?';
        }
        /**
         * Creates the jsdoc for methods, including overloads.
         * If overloaded, merges the signatures in the list of SignatureDeclarations into a single jsdoc.
         * - Total number of parameters will be the maximum count found across all variants.
         * - Different names at the same parameter index will be joined with "_or_"
         * - Variable args (...type[] in TypeScript) will be output as "...type",
         *    except if found at the same index as another argument.
         * @param fnDecls Pass > 1 declaration for overloads of same name
         * @return The list of parameter names that should be used to emit the actual
         *    function statement; for overloads, name will have been merged.
         */
        getFunctionTypeJSDoc(fnDecls, extraTags = []) {
            const typeChecker = this.typeChecker;
            // De-duplicate tags and docs found for the fnDecls.
            const tagsByName = new Map();
            function addTag(tag) {
                const existing = tagsByName.get(tag.tagName);
                tagsByName.set(tag.tagName, existing ? jsdoc.merge([existing, tag]) : tag);
            }
            for (const extraTag of extraTags)
                addTag(extraTag);
            const isConstructor = fnDecls.find(d => d.kind === ts.SyntaxKind.Constructor) !== undefined;
            // For each parameter index i, paramTags[i] is an array of parameters
            // that can be found at index i.  E.g.
            //    function foo(x: string)
            //    function foo(y: number, z: string)
            // then paramTags[0] = [info about x, info about y].
            const paramTags = [];
            const returnTags = [];
            const thisTags = [];
            const typeParameterNames = new Set();
            const argCounts = [];
            let thisReturnType = null;
            for (const fnDecl of fnDecls) {
                // Construct the JSDoc comment by reading the existing JSDoc, if
                // any, and merging it with the known types of the function
                // parameters and return type.
                const tags = this.getJSDoc(fnDecl, /* reportWarnings */ false);
                // Copy all the tags other than @param/@return into the new
                // JSDoc without any change; @param/@return are handled specially.
                // TODO: there may be problems if an annotation doesn't apply to all overloads;
                // is it worth checking for this and erroring?
                for (const tag of tags) {
                    if (tag.tagName === 'param' || tag.tagName === 'return')
                        continue;
                    addTag(tag);
                }
                const flags = ts.getCombinedModifierFlags(fnDecl);
                // Add @abstract on "abstract" declarations.
                if (flags & ts.ModifierFlags.Abstract) {
                    addTag({ tagName: 'abstract' });
                }
                // Add @protected/@private if present.
                if (flags & ts.ModifierFlags.Protected) {
                    addTag({ tagName: 'protected' });
                }
                else if (flags & ts.ModifierFlags.Private) {
                    addTag({ tagName: 'private' });
                }
                // Add any @template tags.
                // Multiple declarations with the same template variable names should work:
                // the declarations get turned into union types, and Closure Compiler will need
                // to find a union where all type arguments are satisfied.
                if (fnDecl.typeParameters) {
                    for (const tp of fnDecl.typeParameters) {
                        typeParameterNames.add(transformer_util_1.getIdentifierText(tp.name));
                    }
                }
                // Merge the parameters into a single list of merged names and list of types
                const sig = typeChecker.getSignatureFromDeclaration(fnDecl);
                if (!sig || !sig.declaration)
                    throw new Error(`invalid signature ${fnDecl.name}`);
                if (sig.declaration.kind === ts.SyntaxKind.JSDocSignature) {
                    throw new Error(`JSDoc signature ${fnDecl.name}`);
                }
                let hasThisParam = false;
                for (let i = 0; i < sig.declaration.parameters.length; i++) {
                    const paramNode = sig.declaration.parameters[i];
                    const name = getParameterName(paramNode, i);
                    const isThisParam = name === 'this';
                    if (isThisParam)
                        hasThisParam = true;
                    const newTag = {
                        tagName: isThisParam ? 'this' : 'param',
                        optional: paramNode.initializer !== undefined || paramNode.questionToken !== undefined,
                        parameterName: isThisParam ? undefined : name,
                    };
                    if (paramNode.dotDotDotToken === undefined) {
                        // The simple case: a plain parameter type.
                        newTag.type = this.typeToClosure(fnDecl, this.typeChecker.getTypeAtLocation(paramNode));
                    }
                    else {
                        // The complex case: resolve the array member type in ...foo[].
                        this.resolveRestParameterType(newTag, fnDecl, paramNode);
                    }
                    for (const { tagName, parameterName, text } of tags) {
                        if (tagName === 'param' && parameterName === newTag.parameterName) {
                            newTag.text = text;
                            break;
                        }
                    }
                    if (!isThisParam) {
                        const paramIdx = hasThisParam ? i - 1 : i;
                        if (!paramTags[paramIdx])
                            paramTags.push([]);
                        paramTags[paramIdx].push(newTag);
                    }
                    else {
                        thisTags.push(newTag);
                    }
                }
                argCounts.push(hasThisParam ? sig.declaration.parameters.length - 1 : sig.declaration.parameters.length);
                // Return type.
                if (!isConstructor) {
                    const returnTag = {
                        tagName: 'return',
                    };
                    const retType = typeChecker.getReturnTypeOfSignature(sig);
                    // Generate a templated `@this` tag for TypeScript `foo(): this` return type specification.
                    // Make sure not to do that if the function already has used `@this` due to a this
                    // parameter. It's not clear how to resolve the two conflicting this types best, the current
                    // solution prefers the explicitly given `this` parameter.
                    // tslint:disable-next-line:no-any accessing TS internal field.
                    if (retType['isThisType'] && !hasThisParam) {
                        // foo(): this
                        thisReturnType = retType;
                        addTag({ tagName: 'template', text: 'THIS' });
                        addTag({ tagName: 'this', type: 'THIS' });
                        returnTag.type = 'THIS';
                    }
                    else {
                        returnTag.type = this.typeToClosure(fnDecl, retType);
                        for (const { tagName, text } of tags) {
                            if (tagName === 'return') {
                                returnTag.text = text;
                                break;
                            }
                        }
                    }
                    returnTags.push(returnTag);
                }
            }
            if (typeParameterNames.size > 0) {
                addTag({ tagName: 'template', text: Array.from(typeParameterNames.values()).join(', ') });
            }
            const newDoc = Array.from(tagsByName.values());
            if (thisTags.length > 0) {
                newDoc.push(jsdoc.merge(thisTags));
            }
            const minArgsCount = Math.min(...argCounts);
            const maxArgsCount = Math.max(...argCounts);
            // Merge the JSDoc tags for each overloaded parameter.
            // Ensure each parameter has a unique name; the merging process can otherwise
            // accidentally generate the same parameter name twice.
            const paramNames = new Set();
            let foundOptional = false;
            for (let i = 0; i < maxArgsCount; i++) {
                const paramTag = jsdoc.merge(paramTags[i]);
                if (paramTag.parameterName) {
                    if (paramNames.has(paramTag.parameterName)) {
                        paramTag.parameterName += i.toString();
                    }
                    paramNames.add(paramTag.parameterName);
                }
                // If the tag is optional, mark parameters following optional as optional,
                // even if they are not, since Closure restricts this, see
                // https://github.com/google/closure-compiler/issues/2314
                if (!paramTag.restParam && (paramTag.optional || foundOptional || i >= minArgsCount)) {
                    foundOptional = true;
                    paramTag.optional = true;
                }
                newDoc.push(paramTag);
                if (paramTag.restParam) {
                    // Cannot have any parameters after a rest param.
                    // Just dump the remaining parameters.
                    break;
                }
            }
            // Merge the JSDoc tags for each overloaded return.
            if (!isConstructor) {
                newDoc.push(jsdoc.merge(returnTags));
            }
            return {
                tags: newDoc,
                parameterNames: newDoc.filter(t => t.tagName === 'param').map(t => t.parameterName),
                thisReturnType,
            };
        }
    }
    exports.ModuleTypeTranslator = ModuleTypeTranslator;
});
//# sourceMappingURL=data:application/json;base64,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